/*
 * Function calls
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <ctype.h>
#include <stdlib.h>

#include "runtime.h"

/*
 * Exception flag pointer
 */
static int *except = NULL;

/*
 * Set address of exception flag
 */
void call_exception_pointer(int *location)
{
  except = location;
}

/*
 * Call function by signature
 */
value *call_function(const char *name, signature *sig, unsigned int argc,
  value **argv)
{
  value *result = NULL;
  char rettype;
  
  sanity(name && sig && (argc == 0 || argv));

  if (!except || *except == 0) {
    call_check(name, sig, argc, argv);

    switch (sig->type) {
      case FUNCTION_TYPE_BUILTIN:
        sanity(sig->call_u.builtin_vector);
        result = sig->call_u.builtin_vector(argc, argv);
        break;
      case FUNCTION_TYPE_USERDEF:
        sanity(sig->call_u.userdef_vector);
        result = sig->call_u.userdef_vector(sig->data, sig->def, sig->args,
                                            argc, argv);
        break;
    }
  } else {
    result = value_make_void();
  }
 
  if (except && *except != 0) {
    rettype = '?';
  } else {
    rettype = sig->rettype;
  }
 
  if (rettype != '?' && call_typechar(result) != tolower(rettype)) {
    if (islower(rettype)) {
      fatal("function `%s': return type mismatch (`%s' instead of `%s')",
        name, call_typename(result->type),
        call_typename(call_chartype(rettype))
      );
    } else {
      value_cast_inplace(&result, call_chartype(tolower(rettype)));
    }
  }
  return result;
}

/*
 * Call function by name
 */
value *call_named_function(const char *name, unsigned int argc, value **argv)
{
  symtab_entry *entry;
  
  sanity(name && (argc == 0 || argv));
  
  entry = symtab_stack_lookup(name);
  if (!entry || entry->type != SYMTAB_ENTRY_FUNCTION) {
    fatal("call to undefined function `%s'", name);
  }
  return call_function(name, entry->entry_u.sig, argc, argv);
}
