/*
 * Error printing
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <stdarg.h>
#include <stdio.h> 
#include <stdlib.h>
 
#include "misc.h"

/*
 * Current source file for printing
 */
char *source_file = NULL;

/*
 * Current source line for printing
 */
int source_line = 1;

/*
 * Current source column for printing
 */
int source_col = 0;

/*
 * Write error message
 */
static void errmsg(int abort, const char *msg, va_list ap)
{
  if (!source_file) source_file = "arena";

  if (source_col > 0) {
    fprintf(stderr, "%s:%i:%i: ", source_file, source_line, source_col);
  } else if (source_line > 0) {
    fprintf(stderr, "%s:%i: ", source_file, source_line);
  } else {
    fprintf(stderr, "%s: ", source_file);
  }

  vfprintf(stderr, msg, ap);
  fprintf(stderr, "\n");

  if (abort) exit(1);
}

/*
 * Write fatal error message
 *
 * This function prints a fatal error message. Execution of the
 * running program is aborted.
 */
void fatal(const char *msg, ...)
{
  va_list ap;
  
  va_start(ap, msg);
  errmsg(1, msg, ap);
  va_end(ap);
}

/*
 * Write nonfatal error message
 *
 * This function prints an error message. Execution of the
 * running program continues as normal.
 */
void nonfatal(const char *msg, ...)
{
  va_list ap;
  
  va_start(ap, msg);
  errmsg(1, msg, ap);
  va_end(ap);
}

/*
 * Report an internal error
 *
 * This function prints an internal error message and is invoked
 * whenever an interpreter core function is passed impossible
 * values.
 */
void internal(const char *file, int line)
{
  fatal("internal error at %s:%i", file, line);
}

/*
 * Report out-of-memory conditions
 *
 * This function prints a message about an out-of-memory condition
 * and aborts the running program if the given pointer is NULL.
 */
void *oom(void *ptr)
{
  if (!ptr) {
    fatal("out of memory");
  }
  return ptr;
}
