/*
 * Statements evaluation
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <stdlib.h>
#include <string.h>

#include "eval.h"

/*
 * Flag for return statement
 */
static int return_flag = 0;

/*
 * Flag for continue statement
 */
static int continue_flag = 0;

/*
 * Flag for try block
 */
static int try_flag = 0;

/*
 * Flag for exception in progress
 */
int except_flag = 0;

/*
 * Exception value
 */
static value *except_value = NULL;

/*
 * Flag for loop in progress
 */
static int loop_flag = 0;

/*
 * Flag for user-defined function in progress
 */
static int func_flag = 0;

/*
 * Evaluate statement list
 *
 * This function evaluates the contents of a statement list. If
 * a continue statment or an error occurs, processing is halted.
 * On successful evaluation, 0 is returned. On error, -1 is
 * returned.
 */
void eval_stmt_list(stmt_list *list, int cookie)
{
  unsigned int i;
  
  sanity(list);
  
  if (return_flag) return;
  
  for (i = 0; i < list->len; i++) {
    eval_stmt(list->list[i], cookie);
    if (continue_flag || return_flag || except_flag) break;
  }
}

/*
 * Flag for break statement
 */
static int break_flag = 0;

/*
 * Evaluate boolean test expression
 *
 * This function tries to evaluate the test expression from an if,
 * while, do, or for statement. It returns 0 is the test returned
 * false, 1 if the test returned true, and -1 if an error occured.
 */
static int runtest(expr *ex)
{
  value *val;
  int res;
  
  val = eval_expr(ex);
  value_cast_inplace(&val, VALUE_TYPE_BOOL);
  res = val->value_u.bool_val;
  value_free(val);

  return res;
}

/*
 * Value from last return statement
 */
static value *retval = NULL;

/*
 * Cookie from when retval was allocated
 */
static int retval_cookie = 0;

/*
 * Current retval cookie
 */
static int global_cookie = 0;

/*
 * Put function arguments into symbol table
 *
 * This function adds variables named "argc" and "argv" to the
 * current function's symbol table.
 */
static void varargs(unsigned int argc, value **argv)
{
  value *count, *vector;
  unsigned int i;

  count = value_make_int(argc);
  vector = value_make_array();
  for (i = 0; i < argc; i++) {
    value_add_to_array(vector, argv[i]);
  }
  symtab_stack_add_variable("argc", count);
  symtab_stack_add_variable("argv", vector);
  value_free(count);
  value_free(vector);
}

/*
 * Run user-defined function
 *
 * This function runs a user-defined function by creating a new
 * symbol table, adding the function parameters as variables, and
 * then executing the function body. This function is implicitly
 * called from eval_call.c::eval_call(), via call_func() from
 * libruntime.
 */
static value *run_func(void *data, void *def, unsigned int args,
                unsigned int argc, value **argv)
{
  char **names = (char **) data;
  stmt *st = (stmt *) def;
  unsigned int i;
  int cookie;

  sanity(def && argc >= args && (argc == 0 || argv));

  if (++func_flag < 1) {
    fatal("too deep function call nesting");
  }

  varargs(argc, argv);

  for (i = 0; i < args; i++) {
    symtab_stack_add_variable(names[i], argv[i]);
  }
  
  retval = NULL;
  cookie = ++global_cookie;

  eval_stmt(st, cookie);

  return_flag = continue_flag = break_flag = 0;

  /*
   * Check whether retval was allocated on the level of this
   * function call -- if not, allocate new void value
   */
  if (retval_cookie != cookie || !retval) {
    retval = value_make_void();
  }

  --func_flag;
  return retval;
}

/*
 * Create user-defined function symtab entry
 *
 * This function creates a symbol table entry for a user defined
 * function and adds it to the current symbol table.
 */
void define_func(stmt *st)
{
  signature *sig;
  char *proto_copy;
  
  sanity(st && st->name);
  
  sig = call_sig_alloc();
  
  if (st->proto) {
    proto_copy = oom(malloc(strlen(st->proto) + 1));
    strcpy(proto_copy, st->proto);
  } else {
    proto_copy = NULL;
  }
  
  sig->type    = FUNCTION_TYPE_USERDEF;
  sig->args    = st->args;
  sig->proto   = proto_copy;
  sig->rettype = st->rettype;
  sig->data    = st->names;
  sig->def     = st->true_case;
  sig->call_u.userdef_vector = run_func;

  symtab_stack_add_function(st->name, sig);
  call_sig_free(sig);
}

/*
 * Evaluate single statement
 *
 * This function evaluates a single statment. On success, 0 is returned.
 * If an error occurs during processing, -1 is returned.
 */
void eval_stmt(stmt *st, int cookie)
{
  value *val;
  int res = 0;
  
  sanity(st);

  if (return_flag || except_flag) return;

  switch (st->type) {
    case STMT_SWITCH:
      eval_stmt_switch(st, cookie);
      break;
    case STMT_NOP:
      /* null statement */
      break;
    case STMT_BLOCK:
      /* block of multiple statements */
      eval_stmt_list((stmt_list *) st->block, cookie);
      break;
    case STMT_IF:
      /* if without else block */
      res = runtest(st->expr);
      if (res == 1) {
        eval_stmt(st->true_case, cookie);
      }
      break;
    case STMT_IF_ELSE:
      /* if with else block */
      res = runtest(st->expr);
      if (res == 1) {
        eval_stmt(st->true_case, cookie);
      } else if (res == 0) {
        eval_stmt(st->false_case, cookie);
      }
      break;
    case STMT_WHILE:
      /* while loop */
      if (++loop_flag < 1) {
        fatal("too deep loop nesting");
      }
      while ((res = runtest(st->expr)) == 1) {
        eval_stmt(st->true_case, cookie);
        continue_flag = 0;
        if (break_flag) break;
      }
      break_flag = 0;
      --loop_flag;
      break;
    case STMT_DO:
      /* do loop */
      if (++loop_flag < 1) {
        fatal("too deep loop nesting");
      }
      do {
        eval_stmt(st->true_case, cookie);
        if (!break_flag) {
          res = runtest(st->expr);
        }
        continue_flag = 0;
      } while (res == 1 && !break_flag);
      break_flag = 0;
      --loop_flag;
      break;
    case STMT_FOR:
      /* for loop */
      if (++loop_flag < 1) {
        fatal("too deep loop nesting");
      }
      val = eval_expr(st->init);
      value_free(val);
      while ((res = runtest(st->expr)) == 1) {
        eval_stmt(st->true_case, cookie);
        continue_flag = 0;
        if (break_flag) break;
        val = eval_expr(st->guard);
        value_free(val);
      }
      break_flag = 0;
      --loop_flag;
      break;
    case STMT_CONTINUE:
      /* continue statement */
      if (loop_flag) {
        continue_flag = 1;
      }
      break;
    case STMT_BREAK:
      /* break statement */
      if (loop_flag) {
        continue_flag = break_flag = 1;
      }
      break;
    case STMT_RETURN:
      /* return statement */
      if (func_flag) {
        if (st->expr) {
          val = eval_expr(st->expr);
        } else {
          val = value_make_void();
        }
        /* store result, for function returns */
        retval = val;
        retval_cookie = cookie;
        return_flag = continue_flag = break_flag = 1;
      }
      break;
    case STMT_EXPR:
      /* expression statment */
      val = eval_expr(st->expr);
      value_free(val);
      break;
    case STMT_FUNC:
      define_func(st);
      break;
    case STMT_TRY:
      ++try_flag;
      if (try_flag < 1) {
        fatal("too deep try block nesting");
      }
      eval_stmt(st->true_case, cookie);
      if (except_flag) {
        symtab_stack_add_variable(st->name, except_value);
        value_free(except_value);
        except_value = NULL;
        except_flag = 0;
        eval_stmt(st->false_case, cookie);
      }
      --try_flag;
      break;
    case STMT_THROW:
      if (!try_flag) {
        fatal("uncaught exception");
      }
      except_value = eval_expr(st->expr);
      except_flag = 1;
      break;
    case STMT_TMPL:
      symtab_stack_add_template(st->name, st->proto, st->block);
      break;
    case STMT_CASE:
    case STMT_DEFAULT:
      sanity(0);
      break;
  }
}
