/*
 * Expression evaluation
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file ../doc/LICENSE for license
 */

#include <stdlib.h>

#include "eval.h"

/*
 * Evaluate lambda expression
 */
value *eval_lambda(expr *ex)
{
  symtab_entry *entry;
  value *res;

  sanity(ex && ex->lambda);

  symtab_stack_enter();
  eval_stmt(ex->lambda, 0);
  
  entry = symtab_stack_lookup("\\lambda");
  if (!entry || entry->type != SYMTAB_ENTRY_FUNCTION) {
    fatal("lambda expression did not yield function");
  }
  res = value_make_fn(entry->entry_u.sig);
  symtab_stack_leave();
  
  return res;
}

/*
 * Evaluate expression
 *
 * This function returns the value that results from evaluating the
 * given expression. Recursion on subexpressions is used as needed.
 */
value *eval_expr(expr *ex)
{
  value *res = NULL;
  
  sanity(ex);

  /*
   * record source position for error printing
   */
  source_line = ex->line;
  source_file = ex->file;
  
  switch (ex->type) {
    case EXPR_NOP:
      res = value_make_void();
      break;
    case EXPR_ASSIGN:
      res = eval_assign(ex);
      break;
    case EXPR_ASSIGN_ARRAY:
      res = eval_assign_array(ex);
      break;
    case EXPR_CAST:
      res = eval_cast(ex);
      break;
    case EXPR_CALL:
      res = eval_call(ex);
      break;
    case EXPR_NEW:
      res = eval_new(ex);
      break;
    case EXPR_STATIC:
      res = eval_static(ex);
      break;
    case EXPR_METHOD:
      res = eval_method(ex);
      break;
    case EXPR_REF:
      res = eval_ref(ex);
      break;
    case EXPR_REF_ARRAY:
      res = eval_ref_array(ex);
      break;
    case EXPR_INDEX:
      res = eval_ref_index(ex);
      break;
    case EXPR_CONST_VOID:
      res = eval_const_void();
      break;
    case EXPR_CONST_BOOL:
      res = eval_const_bool(ex);
      break;
    case EXPR_CONST_INT:
      res = eval_const_int(ex);
      break;
    case EXPR_CONST_FLOAT:
      res = eval_const_float(ex);
      break;
    case EXPR_CONST_STRING:
      res = eval_const_string(ex);
      break;
    case EXPR_INFIX:
      res = eval_infix(ex);
      break;
    case EXPR_POSTFIX:
      res = eval_postfix(ex);
      break;
    case EXPR_PREFIX:
      res = eval_prefix(ex);
      break;
    case EXPR_FIELD:
      res = eval_const_string(ex);
      break;
    case EXPR_STATIC_REF:
      res = eval_static_ref(ex);
      break;
    case EXPR_LAMBDA:
      res = eval_lambda(ex);
      break;
    case EXPR_IF:
      res = eval_expr_if(ex);
      break;
    case EXPR_PASS_REF:
      res = eval_expr(ex->inner);
      break;
    case EXPR_FILE:
      res = value_make_string(ex->file);
      break;
    case EXPR_LINE:
      res = value_make_int(ex->line);
      break;
  }
  
  return res;
}
