/*
 * Interpreter main program
 * (C) 2006, Pascal Schmidt <arena-language@ewetel.net>
 * see file doc/LICENSE for license
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libmisc/misc.h"
#include "libruntime/runtime.h"
#include "libparser/parser.h"
#include "libeval/eval.h"
#include "libstdlib/stdlib.h"

#define VERSION 	"0.9.2"

#define MODE_NORMAL	0x0
#define MODE_CHECK	0x1
#define MODE_DUMP	0x2

/*
 * Interpreter mode
 */
static int mode = MODE_NORMAL;

/*
 * First input file and position in command line
 */
static char *filename = NULL;
static int filearg = 0;

/*
 * Current script data
 */
static stmt_list *script = NULL;

/*
 * Print usage message
 */
static void usage(char *name)
{
  printf("arena v" VERSION " C-like scripting interpreter\n"
         "(C) 2006, Pascal Schmidt <arena-language@ewetel.net>\n\n"
         "Usage: %s [options] script [script options]\n"
         "Options:\n"
         "\t-h, --help\tdisplay this help message\n"
         "\t-V, --version\tdisplay interpreter version and exit\n"
         "\t-c, --check\tsyntax check only\n"
         "\t-d, --dump\tparse script and dump parse tree\n\n"
         "The script is not executed when -c or -d are in effect.\n"
         "You can use the special option -- to terminate option\n"
         "processing; the next argument will then be used as the\n"
         "name of the script to execute.\n"
         , name);
}

/*
 * Evaluate command line options
 */
static void options(int argc, char **argv)
{
  int i, in_opts = 1;
  char *opt;
  
  for (i = 1; i < argc; i++) {
    opt = argv[i];
    if (in_opts) {
      if (strcmp(opt, "-d") == 0 || strcmp(opt, "--dump") == 0) {
        mode = MODE_DUMP;
        continue;
      } else if (strcmp(opt, "-c") == 0 || strcmp(opt, "--check") == 0) {
        mode = MODE_CHECK;
        continue;
      } else if (strcmp(opt, "-h") == 0 || strcmp(opt, "--help") == 0) {
        usage(argv[0]);
        exit(0);
      } else if (strcmp(opt, "-V") == 0 || strcmp(opt, "--version") == 0) {
        printf("%s\n", VERSION);
        exit(0);
      } else if (strcmp(opt, "--") == 0) {
        in_opts = 0;
        continue;
      } else if (*opt == '-') {
        fatal("unknown option `%s'", opt);
      }
    }
    if (!filename) {
      filename = argv[i];
      filearg  = i;
      break;
    }
  }
}

/*
 * Free script data
 */
static void script_teardown(void)
{
  if (script) {
    stmt_list_free(script);
    script = NULL;
  }
}

/*
 * Main function
 *
 * This function will attempt to parse the first command line
 * argument as a script and will then execute that script if
 * it was parsed successfully.
 */
int main(int argc, char **argv)
{
  setvbuf(stdin,  NULL, _IONBF, 0);
  setvbuf(stdout, NULL, _IONBF, 0);
  setvbuf(stderr, NULL, _IONBF, 0);

  /* disable source line printing for option errors */
  source_line = 0;

  options(argc, argv);
  
  if (!filename) {
    fatal("missing script name on command line");
  }

  /* enable source line printing for syntax errors */
  source_line = 1;

  /* initialize state */
  atexit(script_teardown);
  symtab_stack_init();
  stdlib_register(argc - filearg, argv + filearg);

  /* parse file */  
  script = parse_file(filename);
  source_line = 1;
  source_col  = 0;

  /* execute operation */
  if (mode == MODE_NORMAL) {  
    call_exception_pointer(&except_flag);
    eval_stmt_list(script, 0);
  } else if (mode == MODE_DUMP) {
    stmt_list_dump(script, 0);
  }

  return 0;
}
