/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 * =========================================================================
 *
 * ftext, the text widget used by eboard, based on xtext, the text widget
 * used by X-Chat
 *
 * By  Felipe Bergo  <bergo@seul.org>
 * and Peter Zelezny <zed@linux.com>.
 *
 */


#define REFRESH_TIMEOUT 20
#define WORDWRAP_LIMIT 24
#define MOTION_MONITOR 1      /* URL hilights. */
#define MARGIN 2	      /* dont touch. */
#define FTEXT_MB_FIX 1
#undef SMOOTH_SCROLL

#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkselection.h>

#include "ftext.h"

#undef GTK_WIDGET
#define GTK_WIDGET(n) ((GtkWidget*)n)
#undef GTK_OBJECT
#define GTK_OBJECT(n) ((GtkObject*)n)
#undef GTK_OBJECT_CLASS
#define GTK_OBJECT_CLASS(n) ((GtkObjectClass*)n)

static GtkWidgetClass *parent_class = NULL;

enum
{
  WORD_CLICK,
  CHANGED,
  LAST_SIGNAL
};
#ifdef FTEXT_MB_FIX
/* values for selection info */
enum
{
  TARGET_STRING,
  TARGET_TEXT,
  TARGET_COMPOUND_TEXT
};
#endif
static guint ftext_signals[LAST_SIGNAL] = { 0 , 0 };

static void gtk_ftext_set_mark_scheme(GtkFText *ftext);
static void gtk_ftext_set_scheme(GtkFText *ftext, int fore);

static void gtk_ftext_render_page (GtkFText * ftext);
static void gtk_ftext_calc_lines (GtkFText * ftext, int);
static textentry *gtk_ftext_nth (GtkFText * ftext, textentry * start_ent,
				 int line, int width, int *subline);
static gint gtk_ftext_selection_kill (GtkWidget * widget,
				      GdkEventSelection * event);
static void gtk_ftext_selection_get (GtkWidget * widget,
				     GtkSelectionData * selection_data_ptr,
				     guint info, guint time);
static int gtk_ftext_text_width (GtkFText * ftext, unsigned char *text,int len);
static void gtk_ftext_adjustment_changed (GtkAdjustment * adj,GtkFText * ftext);
static void gtk_ftext_render_ents (GtkFText * ftext, textentry *, textentry *,int);
static void gtk_ftext_recalc_widths (GtkFText * ftext, int);
static void gtk_ftext_fix_indent (GtkFText * ftext);

/* some utility functions first */

static char *
nocasestrstr (char *s, char *wanted)
{
  register const size_t len = strlen (wanted);

  if (len == 0)
    return (char *)s;
  while (toupper(*s) != toupper(*wanted) || strncasecmp (s, wanted, len))
    if (*s++ == '\0')
      return (char *)NULL;
  return (char *)s;   
}

static int
is_del (char c)
{
  switch (c)
    {
    case ' ':
    case 0:
    case '\n':
      /*case '[':
	case ']': */
    case ')':
    case '(':
    case '>':
    case '<':
      return 1;
    }
  return 0;
}

static void
ftext_set_fg (GdkGC *gc, gulong pixel)
{
  gdk_rgb_gc_set_foreground (gc, pixel);
}

static void
ftext_set_bg (GdkGC *gc, gulong pixel)
{
  gdk_rgb_gc_set_background (gc, pixel);
}

static void
gtk_ftext_init (GtkFText * ftext)
{
  ftext->old_value = -1;
  ftext->text_first = NULL;
  ftext->text_last = NULL;
  ftext->last_ent_start = NULL;
  ftext->last_ent_end = NULL;
  ftext->io_tag = -1;
  ftext->add_io_tag = -1;
  ftext->scroll_tag = -1;
  /*   ftext->frozen = 0;*/
  ftext->num_lines = 0;
  ftext->max_lines = 0;
  ftext->col_back = 0x000000;
  ftext->col_fore = 0xffffff;
  ftext->pixel_offset = 0;
  ftext->scrollbar_down = TRUE;
  ftext->bold = FALSE;
  ftext->underline = FALSE;
  ftext->reverse = FALSE;
  ftext->time_stamp = FALSE;
  ftext->font = NULL;
  ftext->error_function = NULL;
  ftext->urlcheck_function = NULL;
  ftext->color_paste = FALSE;
  ftext->skip_fills = FALSE;
  ftext->skip_border_fills = FALSE;
  ftext->skip_stamp = FALSE;
  ftext->do_underline_fills_only = FALSE;

#ifdef SMOOTH_SCROLL
  ftext->adj = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 0.4, 0, 0);
#else
  ftext->adj = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 1, 0, 0);
#endif
  gtk_object_ref ((GtkObject *) ftext->adj);
  gtk_object_sink ((GtkObject *) ftext->adj);

  gtk_signal_connect (GTK_OBJECT (ftext->adj), "value_changed",
		      GTK_SIGNAL_FUNC (gtk_ftext_adjustment_changed), ftext);
  gtk_signal_connect (GTK_OBJECT (ftext), "selection_clear_event",
		      GTK_SIGNAL_FUNC (gtk_ftext_selection_kill), ftext);
#ifdef FTEXT_MB_FIX
  {
    static const GtkTargetEntry targets[] = {
      { "STRING", 0, TARGET_STRING },
      { "TEXT",   0, TARGET_TEXT }, 
      { "COMPOUND_TEXT", 0, TARGET_COMPOUND_TEXT }
    };
    static const gint n_targets = sizeof (targets) / sizeof (targets[0]);

    gtk_selection_add_targets (GTK_WIDGET (ftext),
			       GDK_SELECTION_PRIMARY,
			       targets, n_targets);
  }
#else
  gtk_selection_add_target (GTK_WIDGET (ftext),
			    GDK_SELECTION_PRIMARY,
			    GDK_SELECTION_TYPE_STRING, 1);
#endif
  gtk_signal_connect (GTK_OBJECT (ftext), "selection_get",
		      GTK_SIGNAL_FUNC (gtk_ftext_selection_get), ftext);
}

static void
gtk_ftext_adjustment_set (GtkFText * ftext, int fire_signal)
{
  GtkAdjustment *adj = ftext->adj;

  adj->lower = 0;
  adj->upper = ftext->num_lines;

  adj->page_size =
    (GTK_WIDGET (ftext)->allocation.height -
     ftext->font->descent) / ftext->fontsize;
  adj->page_increment = adj->page_size;

  if (adj->value > adj->upper - adj->page_size)
    adj->value = adj->upper - adj->page_size;

  if (fire_signal)
    gtk_adjustment_changed (adj);
}

static gint
gtk_ftext_adjustment_timeout (GtkFText * ftext)
{
  gtk_ftext_render_page (ftext);
  ftext->io_tag = -1;
  return 0;
}

static void
gtk_ftext_adjustment_changed (GtkAdjustment * adj, GtkFText * ftext)
{
  /*   if (ftext->frozen)
       return;*/

#ifdef SMOOTH_SCROLL
  if (ftext->old_value != ftext->adj->value)
#else
    if ((int) ftext->old_value != (int) ftext->adj->value)
#endif
      {
	if (ftext->adj->value >= ftext->adj->upper - ftext->adj->page_size)
	  ftext->scrollbar_down = TRUE;
	else
	  ftext->scrollbar_down = FALSE;

	if (ftext->adj->value + 1 == ftext->old_value ||
	    ftext->adj->value - 1 == ftext->old_value)	/* clicked an arrow? */
	  {
	    if (ftext->io_tag != -1)
	      {
		gtk_timeout_remove (ftext->io_tag);
		ftext->io_tag = -1;
	      }
	    gtk_ftext_render_page (ftext);
	  } else
	    {
	      if (ftext->io_tag == -1)
		ftext->io_tag = gtk_timeout_add (REFRESH_TIMEOUT,
						 (GtkFunction)
						 gtk_ftext_adjustment_timeout,
						 ftext);
	    }
      }
  ftext->old_value = adj->value;
}

GtkWidget *
gtk_ftext_new (int indent, int separator)
{
  GtkFText *ftext;

  ftext = gtk_type_new (gtk_ftext_get_type ());
  ftext->indent = indent;
  ftext->separator = separator;
  ftext->wordwrap = FALSE;

  return GTK_WIDGET (ftext);
}

static void
gtk_ftext_destroy (GtkObject * object)
{
  GtkFText *ftext = GTK_FTEXT (object);
  textentry *ent, *next;

  if (ftext->add_io_tag != -1)
    {
      gtk_timeout_remove (ftext->add_io_tag);
      ftext->add_io_tag = -1;
    }

  if (ftext->scroll_tag != -1)
    {
      gtk_timeout_remove (ftext->scroll_tag);
      ftext->scroll_tag = -1;
    }

  if (ftext->io_tag != -1)
    {
      gtk_timeout_remove (ftext->io_tag);
      ftext->io_tag = -1;
    }

  if (ftext->font)
    {
      gdk_font_unref (ftext->font);
      ftext->font = NULL;
    }

  if (ftext->adj)
    {
      gtk_signal_disconnect_by_data (GTK_OBJECT (ftext->adj), ftext);
      gtk_object_unref (GTK_OBJECT (ftext->adj));
      ftext->adj = NULL;
    }

  if (ftext->bgc)
    {
      gdk_gc_destroy (ftext->bgc);
      ftext->bgc = NULL;
    }

  if (ftext->fgc)
    {
      gdk_gc_destroy (ftext->fgc);
      ftext->fgc = NULL;
    }

  if (ftext->hand_cursor)
    {
      gdk_cursor_destroy (ftext->hand_cursor);
      ftext->hand_cursor = NULL;
    }

  ent = ftext->text_first;
  while (ent)
    {
      next = ent->next;
      free (ent);
      ent = next;
    }
  ftext->text_first = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_ftext_realize (GtkWidget * widget)
{
  GtkFText *ftext;
  GdkWindowAttr attributes;
  GdkGCValues val;
  GdkColor col;
  GdkColormap *cmap;

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  ftext = GTK_FTEXT (widget);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget) |
    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
#ifdef MOTION_MONITOR
    | GDK_POINTER_MOTION_MASK | GDK_LEAVE_NOTIFY_MASK;
#else
  | GDK_POINTER_MOTION_MASK;
#endif

  cmap = gtk_widget_get_colormap (widget);
  attributes.colormap = cmap;
  attributes.visual = gtk_widget_get_visual (widget);

  widget->window = gdk_window_new (widget->parent->window, &attributes,
				   GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL |
				   GDK_WA_COLORMAP);

  gdk_window_set_user_data (widget->window, widget);

  ftext->depth = gdk_window_get_visual (widget->window)->depth;

  val.subwindow_mode = GDK_INCLUDE_INFERIORS;
  val.graphics_exposures = 0;

  ftext->bgc = gdk_gc_new_with_values (widget->window, &val,
				       GDK_GC_EXPOSURES | GDK_GC_SUBWINDOW);
  ftext->fgc = gdk_gc_new_with_values (widget->window, &val,
				       GDK_GC_EXPOSURES | GDK_GC_SUBWINDOW);

  if (ftext->fonttype != FONT_SET && ftext->font != NULL)
    gdk_gc_set_font (ftext->fgc, ftext->font);
  
  ftext_set_fg (ftext->fgc, 0xffffff);
  ftext_set_bg (ftext->fgc, 0x000000);
  ftext_set_fg (ftext->bgc, 0x000000);

  ftext->hand_cursor = gdk_cursor_new (GDK_HAND1);

  gdk_window_set_back_pixmap (widget->window, NULL, FALSE);

  /* draw directly to window */
  ftext->draw_buf = widget->window;

  if (ftext->auto_indent)
    ftext->indent = 1;
}

static void
gtk_ftext_size_request (GtkWidget * widget, GtkRequisition * requisition)
{
  requisition->width = GTK_FTEXT (widget)->fontwidth['Z'] * 20;
  requisition->height = (GTK_FTEXT (widget)->fontsize * 10) + 3;
}

static void
gtk_ftext_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
  GtkFText *ftext = GTK_FTEXT (widget);

  if (allocation->width == widget->allocation.width &&
      allocation->height == widget->allocation.height &&
      allocation->x == widget->allocation.x &&
      allocation->y == widget->allocation.y)
    return;

  widget->allocation = *allocation;
  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);
      gtk_ftext_calc_lines (ftext, FALSE);
    }
}

static void
gtk_ftext_draw (GtkWidget * widget, GdkRectangle * area)
{
  GtkFText *ftext = GTK_FTEXT (widget);

  if (ftext->scrollbar_down)
    gtk_adjustment_set_value (ftext->adj,
			      ftext->adj->upper - ftext->adj->page_size);
  gtk_ftext_render_page (ftext);
}

static int
gtk_ftext_selection_clear (GtkFText * ftext)
{
  textentry *ent;
  int ret = 0;

  ent = ftext->last_ent_start;
  while (ent)
    {
      if (ent->mark_start != -1)
	ret = 1;
      ent->mark_start = -1;
      ent->mark_end = -1;
      if (ent == ftext->last_ent_end)
	break;
      ent = ent->next;
    }

  return ret;
}

static int
find_x_8bit (GtkFText *ftext, textentry *ent, char *text, int x, int indent)
{
  int xx = indent;
  int i = 0;
  char *orig = text;
  int a;

  while (*text) {
    a = *((unsigned char *)text);
    xx += ftext->fontwidth[a];
    if (xx >= x)
      return i + (orig - ent->str);
    text++;
    i++;
    if (text - orig >= ent->str_len)
      return ent->str_len;
  }
  return ent->str_len;
}

#ifdef FTEXT_MB_FIX
static int
find_x_mb (GtkFText * ftext, textentry * ent, char *str, int x, int indent)
{
  int str_width;

  int len;
  int mbl = mblen (str, MB_CUR_MAX);

  if (0 >= mbl) mbl = 1;
  len = mbl;

  while (1)
    {
      str_width = gtk_ftext_text_width (ftext, str, len);
      if (str_width + indent >= x)
	return (str + len) - ent->str - mbl;
      mbl = mblen (str + len, MB_CUR_MAX);
      if (0 >= mbl) mbl = 1;
      len += mbl;
      if (len + (str - ent->str) > ent->str_len)
	return ent->str_len;
    }
}
#endif

static int
find_x_16bit (GtkFText * ftext, textentry * ent, char *str, int x, int indent)
{
  int str_width;
  int len = 1;

  while (1)
    {
      str_width = gtk_ftext_text_width (ftext, str, len);
      if (str_width + indent >= x)
	return (str + len) - ent->str;
#ifdef FTEXT_MB_FIX
      len += 2;
#else
      len++;
#endif
      if (len + (str - ent->str) > ent->str_len)
	return ent->str_len;
      /*		if (str_width + indent + 40 < x)
			len += 2;*/
    }
}

static int
find_x (GtkFText * ftext, textentry * ent, char *str, int x, int indent)
{
  if (ftext->fonttype == FONT_1BYTE)
    return find_x_8bit (ftext, ent, str, x, indent);

#ifdef FTEXT_MB_FIX
  if (ftext->fonttype == FONT_SET)
    return find_x_mb (ftext, ent, str, x, indent);
#endif

  return find_x_16bit (ftext, ent, str, x, indent);
}

static int
gtk_ftext_find_x (GtkFText * ftext, int x, textentry * ent, int offset,
		  int line, int win_width, int *out_of_bounds)
{
  int indent;
  char *str;

  if (offset < 1)
    indent = ent->indent;
  else
    indent = ftext->indent;

  if (line > ftext->adj->page_size || line < 0)
    return 0;

  if (ftext->grid_offset[line] > ent->str_len)
    return 0;

  if (ftext->grid_offset[line] < 0)
    return 0;

  str = ent->str + ftext->grid_offset[line];

  if (x < indent)
    {
      *out_of_bounds = 1;
      return (str - ent->str);
    }

  *out_of_bounds = 0;

  return find_x (ftext, ent, str, x, indent);
}

static textentry *
gtk_ftext_find_char (GtkFText * ftext, int x, int y, int *off,
		     int *out_of_bounds)
{
  textentry *ent;
  int line;
  int subline;
  int win_width;

  gdk_window_get_size (GTK_WIDGET (ftext)->window, &win_width, 0);
  win_width -= MARGIN;

  line = (y - ftext->font->descent + ftext->pixel_offset) / ftext->fontsize;

  subline = ftext->pagetop_subline;
  ent = gtk_ftext_nth (ftext, ftext->pagetop_ent, line, win_width, &subline);
  if (!ent)
    return 0;

  if (off)
    *off = gtk_ftext_find_x (ftext, x, ent, subline, line, win_width,
			     out_of_bounds);

  return ent;
}

static gint
gtk_ftext_expose (GtkWidget * widget, GdkEventExpose * event)
{
  GtkFText *ftext = GTK_FTEXT (widget);
  textentry *ent_start, *ent_end;

  gdk_draw_rectangle (ftext->draw_buf, ftext->bgc, 1,
		      event->area.x, event->area.y,
		      event->area.width, event->area.height);

  ent_start = gtk_ftext_find_char (ftext, event->area.x, event->area.y,
				   NULL, NULL);
  ent_end = gtk_ftext_find_char (ftext, event->area.x + event->area.width,
				 event->area.y + event->area.height + ftext->font->descent,
				 NULL, NULL);

  ftext->skip_fills = TRUE;
  ftext->skip_border_fills = TRUE;

  gtk_ftext_render_ents (ftext, ent_start, ent_end, TRUE);

  ftext->skip_fills = FALSE;
  ftext->skip_border_fills = FALSE;

  return FALSE;
}

static void
gtk_ftext_selection_render (GtkFText *ftext, textentry *start_ent, int start_offset,
			    textentry *end_ent, int end_offset)
{
  textentry *ent, *ent2;

  ftext->skip_border_fills = TRUE;
  ftext->skip_stamp = TRUE;

  /* marking downward? */
  if (ftext->last_ent_start == start_ent &&
      ftext->last_offset_start == start_offset)
    {
      ent = start_ent;
      while (ent)
	{
	  if (ent == ftext->last_ent_end)
	    {
	      gtk_ftext_render_ents (ftext, ent, end_ent, TRUE);
	      break;
	    }
	  if (ent == end_ent)
	    {
	      gtk_ftext_render_ents (ftext, ent, ftext->last_ent_end, TRUE);
	      break;
	    }
	  ent = ent->next;
	}
    }
  /* marking upward? */
  else if (ftext->last_ent_end == end_ent &&
	   ftext->last_offset_end == end_offset)
    {
      ent = start_ent;
      ent2 = ftext->last_ent_start;
      do
	{
	  if (ent == ftext->last_ent_start)
	    {
	      gtk_ftext_render_ents (ftext, start_ent, ent, TRUE);
	      break;
	    }
	  if (ent2 == start_ent)
	    {
	      gtk_ftext_render_ents (ftext, ftext->last_ent_start, start_ent, TRUE);
	      break;
	    }
	  if (ent)
	    ent = ent->next;
	  if (ent2)
	    ent2 = ent2->next;
	}
      while (ent || ent2);
    }
  else	/* cross-over mark */
    {
      gtk_ftext_render_ents (ftext, ftext->last_ent_start, ftext->last_ent_end, TRUE);
      gtk_ftext_render_ents (ftext, start_ent, end_ent, TRUE);
    }

  ftext->last_ent_start = start_ent;
  ftext->last_ent_end = end_ent;
  ftext->last_offset_start = start_offset;
  ftext->last_offset_end = end_offset;

  ftext->skip_border_fills = FALSE;
  ftext->skip_stamp = FALSE;
}

static void
gtk_ftext_selection_draw (GtkFText * ftext, GdkEventMotion * event)
{
  textentry *ent;
  textentry *ent_end;
  textentry *ent_start;
  int offset_start;
  int offset_end;
  int low_x;
  int low_y;
  int high_x;
  int high_y;
  int tmp;

  if (ftext->select_start_y > ftext->select_end_y)
    {
      low_x = ftext->select_end_x;
      low_y = ftext->select_end_y;
      high_x = ftext->select_start_x;
      high_y = ftext->select_start_y;
    } else
      {
	low_x = ftext->select_start_x;
	low_y = ftext->select_start_y;
	high_x = ftext->select_end_x;
	high_y = ftext->select_end_y;
      }

  ent_start = gtk_ftext_find_char (ftext, low_x, low_y, &offset_start, &tmp);
  ent_end = gtk_ftext_find_char (ftext, high_x, high_y, &offset_end, &tmp);

  if (ent_start && !ent_end)
    {
      ent_end = ftext->text_last;
      offset_end = ent_end->str_len;
    }

  if (!ent_start || !ent_end)
    {
      if (ftext->adj->value != ftext->old_value)
	gtk_ftext_render_page (ftext);
      return;
    }

  gtk_ftext_selection_clear (ftext);

  /* marking less than a complete line? */
  if (ent_start == ent_end)
    {
      ent_start->mark_start = MIN (offset_start, offset_end);
      ent_start->mark_end = MAX (offset_end, offset_start);
      if (offset_start == offset_end)
#ifdef FTEXT_MB_FIX
	{
	  int mbl;

	  mbl = mblen(ent_start->str + offset_start, MB_CUR_MAX);
	  if (0 < mbl)
	    ent_start->mark_end += mbl;
	}
#else
      ent_start->mark_end++;
#endif
    } else
      {
	ent_start->mark_start = offset_start;
	ent_start->mark_end = ent_start->str_len;

	if (offset_end != 0)
	  {
	    ent_end->mark_start = 0;
	    ent_end->mark_end = offset_end;
	  }
      }

  if (ent_start != ent_end)
    {
      ent = ent_start->next;
      while (ent && ent != ent_end)
	{
	  ent->mark_start = 0;
	  ent->mark_end = ent->str_len;
	  ent = ent->next;
	}
    }

  /* has the selection changed? Dont render unless necessary */
  if (ftext->last_ent_start == ent_start &&
      ftext->last_ent_end == ent_end &&
      ftext->last_offset_start == offset_start &&
      ftext->last_offset_end == offset_end)
    return;

  gtk_selection_owner_set (GTK_WIDGET (ftext), GDK_SELECTION_PRIMARY,
			   event->time);

  gtk_ftext_selection_render (ftext, ent_start, offset_start, ent_end, offset_end);
}

static gint
gtk_ftext_scrolldown_timeout (GtkFText * ftext)
{
  int p_y, win_height;

  gdk_window_get_pointer (GTK_WIDGET (ftext)->window, 0, &p_y, 0);
  gdk_window_get_size (GTK_WIDGET (ftext)->window, 0, &win_height);

  if (p_y > win_height &&
      ftext->adj->value < (ftext->adj->upper - ftext->adj->page_size))
    {
      ftext->adj->value++;
      gtk_adjustment_changed (ftext->adj);
      gtk_ftext_render_page (ftext);
      return 1;
    }

  ftext->scroll_tag = -1;
  return 0;
}

static gint
gtk_ftext_scrollup_timeout (GtkFText * ftext)
{
  int p_y;

  gdk_window_get_pointer (GTK_WIDGET (ftext)->window, 0, &p_y, 0);

  if (p_y < 0 && ftext->adj->value > 0.0)
    {
      ftext->adj->value--;
      gtk_adjustment_changed (ftext->adj);
      gtk_ftext_render_page (ftext);
      return 1;
    }

  ftext->scroll_tag = -1;
  return 0;
}

static void
gtk_ftext_selection_update (GtkFText * ftext, GdkEventMotion * event, int p_y)
{
  int win_height;
  int moved;

  gdk_window_get_size (GTK_WIDGET (ftext)->window, 0, &win_height);

  /* selecting past top of window, scroll up! */
  if (p_y < 0 && ftext->adj->value >= 0)
    {
      if (ftext->scroll_tag == -1)
	ftext->scroll_tag = gtk_timeout_add (100,
					     (GtkFunction)
					     gtk_ftext_scrollup_timeout,
					     ftext);
      return;
    }

  /* selecting past bottom of window, scroll down! */
  if (p_y > win_height &&
      ftext->adj->value < (ftext->adj->upper - ftext->adj->page_size))
    {
      if (ftext->scroll_tag == -1)
	ftext->scroll_tag = gtk_timeout_add (100,
					     (GtkFunction)
					     gtk_ftext_scrolldown_timeout,
					     ftext);
      return;
    }

  moved = ftext->adj->value - ftext->select_start_adj;
  ftext->select_start_y -= (moved * ftext->fontsize);
  ftext->select_start_adj = ftext->adj->value;
  gtk_ftext_selection_draw (ftext, event);
}

static char *
gtk_ftext_get_word (GtkFText * ftext, int x, int y, textentry ** ret_ent,
		    int *ret_off, int *ret_len)
{
  textentry *ent;
  int offset;
  char *str;
  char *word,*bfr;
  int len;
  int out_of_bounds;

  ent = gtk_ftext_find_char (ftext, x, y, &offset, &out_of_bounds);
  if (!ent)
    return 0;

  if (out_of_bounds)
    return 0;

  if (offset == ent->str_len)
    return 0;

  if (offset < 1)
    return 0;

  offset--;

  str = ent->str + offset;

  while (!is_del (*str) && str != ent->str)
    str--;
  word = str + 1;

  len = 0;
  str = word;
  while (!is_del (*str) && len != ent->str_len)
    {
      str++;
      len++;
    }

  if (ret_ent)
    *ret_ent = ent;
  if (ret_off)
    *ret_off = word - ent->str;
  if (ret_len)
    *ret_len = str - word;

  bfr=malloc(strlen(word)+1);
  strcpy(bfr,word);

  return bfr;
}

static gint
gtk_ftext_leave_notify (GtkWidget * widget, GdkEventCrossing * event)
{
#ifdef MOTION_MONITOR
  GtkFText *ftext = GTK_FTEXT (widget);

  if (ftext->cursor_hand)
    {
      ftext->hilight_start = -1;
      ftext->hilight_end = -1;
      ftext->cursor_hand = FALSE;
      gdk_window_set_cursor (widget->window, 0);
      ftext->skip_border_fills = TRUE;
      ftext->do_underline_fills_only = TRUE;
      gtk_ftext_render_ents (ftext, ftext->hilight_ent, NULL, FALSE);
      ftext->skip_border_fills = FALSE;
      ftext->do_underline_fills_only = FALSE;
      ftext->hilight_ent = NULL;
    }
#endif
  return FALSE;
}

static gint
gtk_ftext_motion_notify (GtkWidget * widget, GdkEventMotion * event)
{
  GtkFText *ftext = GTK_FTEXT (widget);
  int tmp, x, y, offset, len;
  char *word;
  textentry *word_ent, *old_ent;

  gdk_window_get_pointer (widget->window, &x, &y, 0);

  if (ftext->moving_separator)
    {
      if (x < (3 * widget->allocation.width) / 5 && x > 15)
	{
	  tmp = ftext->indent;
	  ftext->indent = x;
	  gtk_ftext_fix_indent (ftext);
	  if (tmp != ftext->indent)
	    {
	      gtk_ftext_recalc_widths (ftext, FALSE);
	      if (ftext->scrollbar_down)
		gtk_adjustment_set_value (ftext->adj, ftext->adj->upper -
					  ftext->adj->page_size);
	      if (ftext->io_tag == -1)
		ftext->io_tag = gtk_timeout_add (REFRESH_TIMEOUT,
						 (GtkFunction)
						 gtk_ftext_adjustment_timeout,
						 ftext);
	    }
	}
      return FALSE;
    }

  if (ftext->button_down)
    {
      gtk_grab_add (widget);
      /*gdk_pointer_grab (widget->window, TRUE,
	GDK_BUTTON_RELEASE_MASK |
	GDK_BUTTON_MOTION_MASK, NULL, NULL, 0);*/
      ftext->select_end_x = x;
      ftext->select_end_y = y;
      gtk_ftext_selection_update (ftext, event, y);
      return FALSE;
    }
#ifdef MOTION_MONITOR

  if (ftext->urlcheck_function == NULL)
    return FALSE;

  word = gtk_ftext_get_word (ftext, x, y, &word_ent, &offset, &len);
  if (word)
    {
      if (ftext->urlcheck_function (ftext, word) > 0)
	{
	  free (word);
	  if (!ftext->cursor_hand ||
	      ftext->hilight_ent != word_ent ||
	      ftext->hilight_start != offset ||
	      ftext->hilight_end != offset + len)
	    {
	      if (!ftext->cursor_hand)
		{
		  gdk_window_set_cursor (GTK_WIDGET (ftext)->window,
					 ftext->hand_cursor);
		  ftext->cursor_hand = TRUE;
		}
	      old_ent = ftext->hilight_ent;
	      ftext->hilight_ent = word_ent;
	      ftext->hilight_start = offset;
	      ftext->hilight_end = offset + len;
	      ftext->skip_border_fills = TRUE;
	      ftext->do_underline_fills_only = TRUE;
	      gtk_ftext_render_ents (ftext, old_ent, word_ent, FALSE);
	      ftext->skip_border_fills = FALSE;
	      ftext->do_underline_fills_only = FALSE;
	    }
	  return FALSE;
	}
      free (word);
    }

  gtk_ftext_leave_notify (widget, NULL);

#endif

  return FALSE;
}

static gint
gtk_ftext_button_release (GtkWidget * widget, GdkEventButton * event)
{
  GtkFText *ftext = GTK_FTEXT (widget);
  char *word;

  if (ftext->moving_separator)
    {
      ftext->moving_separator = FALSE;
      if (event->x < (4 * widget->allocation.width) / 5 && event->x > 15)
	{
	  ftext->indent = event->x;
	}
      gtk_ftext_fix_indent (ftext);
      gtk_ftext_recalc_widths (ftext, FALSE);
      gtk_ftext_adjustment_set (ftext, TRUE);
      gtk_ftext_render_page (ftext);
      return FALSE;
    }

  if (ftext->word_or_line_select)
    {
      ftext->word_or_line_select = FALSE;
      ftext->button_down = FALSE;
      return FALSE;
    }

  if (event->button == 1)
    {
      ftext->button_down = FALSE;

      gtk_grab_remove (widget);
      /*gdk_pointer_ungrab (0);*/

      if (ftext->select_start_x == event->x &&
	  ftext->select_start_y == event->y)
	{
	  if (gtk_ftext_selection_clear (ftext))
	    {
	      ftext->skip_border_fills = TRUE;
	      gtk_ftext_render_ents (ftext, ftext->last_ent_start, ftext->last_ent_end, TRUE);
	      ftext->skip_border_fills = FALSE;

	      ftext->last_ent_start = NULL;
	      ftext->last_ent_end = NULL;
	    }
	} else
	  {
	    word = gtk_ftext_get_word (ftext, event->x, event->y, 0, 0, 0);
	    if (word)
	      {
		gtk_signal_emit (GTK_OBJECT (ftext), ftext_signals[WORD_CLICK],
				 word, event);
		free (word);
		return FALSE;
	      }
	  }
    }

  return FALSE;
}

static gint
gtk_ftext_button_press (GtkWidget * widget, GdkEventButton * event)
{
  GtkFText *ftext = GTK_FTEXT (widget);
  textentry *ent;
  char *word;
  int line_x, x, y, offset, len;
  gfloat new_value;

  gdk_window_get_pointer (widget->window, &x, &y, 0);

  if (event->button == 3)		  /* right click */
    {
      word = gtk_ftext_get_word (ftext, x, y, 0, 0, 0);
      if (word)
	{
	  gtk_signal_emit (GTK_OBJECT (ftext), ftext_signals[WORD_CLICK],
			   word, event);
	  free (word);
	} else
	  gtk_signal_emit (GTK_OBJECT (ftext), ftext_signals[WORD_CLICK],
			   "", event);
      return FALSE;
    }

  if (event->button == 4)		  /* mouse wheel pageUp */
    {
      new_value = ftext->adj->value - ftext->adj->page_increment;
      if (new_value < ftext->adj->lower)
	new_value = ftext->adj->lower;
      gtk_adjustment_set_value (ftext->adj, new_value);
      return FALSE;
    }

  if (event->button == 5)		  /* mouse wheel pageDn */
    {
      new_value = ftext->adj->value + ftext->adj->page_increment;
      if (new_value > (ftext->adj->upper - ftext->adj->page_size))
	new_value = ftext->adj->upper - ftext->adj->page_size;
      gtk_adjustment_set_value (ftext->adj, new_value);
      return FALSE;
    }

  if (event->button == 2)
    {
      gtk_signal_emit (GTK_OBJECT (ftext), ftext_signals[WORD_CLICK], "", event);
      return FALSE;
    }

  if (event->button != 1)		  /* we only want left button */
    return FALSE;

  if (event->type == GDK_2BUTTON_PRESS)	/* WORD select */
    {
      word = gtk_ftext_get_word (ftext, x, y, &ent, &offset, &len);
      if (word)
	{
	  free (word);
	  if (len == 0)
	    return FALSE;
	  gtk_ftext_selection_clear (ftext);
	  ent->mark_start = offset;
	  ent->mark_end = offset + len;
	  /* clear the old mark, if any */
	  ftext->skip_border_fills = TRUE;
	  gtk_ftext_render_ents (ftext, ftext->last_ent_start, ftext->last_ent_end, TRUE);
	  /* render the word select */
	  ftext->last_ent_start = ent;
	  ftext->last_ent_end = ent;
	  gtk_ftext_render_ents (ftext, ent, NULL, FALSE);
	  ftext->skip_border_fills = FALSE;
	  ftext->word_or_line_select = TRUE;
	  gtk_selection_owner_set (widget, GDK_SELECTION_PRIMARY, event->time);
	}

      return FALSE;
    }

  if (event->type == GDK_3BUTTON_PRESS)	/* LINE select */
    {
      word = gtk_ftext_get_word (ftext, x, y, &ent, 0, 0);
      if (word)
	{
	  free (word);
	  gtk_ftext_selection_clear (ftext);
	  ent->mark_start = 0;
	  ent->mark_end = ent->str_len;
	  /* clear the old mark, if any */
	  ftext->skip_border_fills = TRUE;
	  gtk_ftext_render_ents (ftext, ftext->last_ent_start, ftext->last_ent_end, TRUE);
	  /* render the line select */
	  ftext->last_ent_start = ent;
	  ftext->last_ent_end = ent;
	  gtk_ftext_render_ents (ftext, ent, NULL, FALSE);
	  ftext->skip_border_fills = FALSE;
	  ftext->word_or_line_select = TRUE;
	  gtk_selection_owner_set (widget, GDK_SELECTION_PRIMARY, event->time);
	}

      return FALSE;
    }

  /* check if it was a separator-bar click */
  if (ftext->separator && ftext->indent)
    {
      line_x = ftext->indent - ((ftext->space_width + 1) / 2);
      if (line_x == x || line_x == x + 1 || line_x == x - 1)
	{
	  ftext->moving_separator = TRUE;
	  gtk_ftext_render_page (ftext);
	  return FALSE;
	}
    }

  ftext->button_down = TRUE;

  ftext->select_start_x = x;
  ftext->select_start_y = y;

  ftext->select_start_adj = ftext->adj->value;

  return FALSE;
}

/* another program has claimed the selection */

static gint
gtk_ftext_selection_kill (GtkWidget * widget, GdkEventSelection * event)
{
  if (gtk_ftext_selection_clear (GTK_FTEXT (widget)))
    {
      GTK_FTEXT (widget)->last_ent_start = NULL;
      GTK_FTEXT (widget)->last_ent_end = NULL;
      gtk_ftext_render_page (GTK_FTEXT (widget));
    }
  return TRUE;
}

/* another program is asking for our selection */

static void
gtk_ftext_selection_get (GtkWidget * widget,
			 GtkSelectionData * selection_data_ptr,
			 guint info, guint time)
{
  GtkFText *ftext = GTK_FTEXT (widget);
  textentry *ent;
  char *txt;
  char *pos;
  char *stripped;
  int len;
  int first = TRUE;

  /* first find out how much we need to malloc ... */
  len = 0;
  ent = ftext->last_ent_start;
  while (ent)
    {
      if (ent->mark_start != -1)
	{
	  if (ent->mark_end - ent->mark_start > 0)
	    len += (ent->mark_end - ent->mark_start) + 1;
	  else
	    len++;
	}
      if (ent == ftext->last_ent_end)
	break;
      ent = ent->next;
    }

  if (len < 1)
    return;

  /* now allocate mem and copy buffer */
  pos = txt = malloc (len);
  ent = ftext->last_ent_start;
  while (ent)
    {
      if (ent->mark_start != -1)
	{
	  if (!first)
	    {
	      *pos = '\n';
	      pos++;
	    }
	  first = FALSE;
	  if (ent->mark_end - ent->mark_start > 0)
	    {
	      memcpy (pos, ent->str + ent->mark_start,
		      ent->mark_end - ent->mark_start);
	      pos += ent->mark_end - ent->mark_start;
	    }
	}
      if (ent == ftext->last_ent_end)
	break;
      ent = ent->next;
    }
  *pos = 0;

  if (ftext->color_paste)
    {
#ifdef FTEXT_MB_FIX
      if ((info == TARGET_TEXT) || (info == TARGET_COMPOUND_TEXT))
	{
	  guchar *new_text;
	  GdkAtom encoding;
	  gint format;
	  gint new_length;

	  gdk_string_to_compound_text (txt, &encoding, &format, &new_text,
				       &new_length);
	  gtk_selection_data_set (selection_data_ptr, encoding, format,
				  new_text, new_length);
	  gdk_free_compound_text (new_text);
	} else
	  gtk_selection_data_set (selection_data_ptr,
				  GDK_SELECTION_TYPE_STRING,
				  8, txt, strlen (txt));
#else
      gtk_selection_data_set (selection_data_ptr, GDK_SELECTION_TYPE_STRING,
			      8, txt, strlen (txt));
#endif
    } else
      {
	stripped = txt;
#ifdef FTEXT_MB_FIX
	if ((info == TARGET_TEXT) || (info == TARGET_COMPOUND_TEXT))
	  {
	    guchar *new_text;
	    GdkAtom encoding;
	    gint format;
	    gint new_length;

	    gdk_string_to_compound_text (stripped, &encoding, &format,
					 &new_text, &new_length);
	    gtk_selection_data_set (selection_data_ptr, encoding, format,
				    new_text, new_length);
	    gdk_free_compound_text (new_text);
	  } else
	    gtk_selection_data_set (selection_data_ptr,
				    GDK_SELECTION_TYPE_STRING,
				    8, stripped, strlen (stripped));
#else
	gtk_selection_data_set (selection_data_ptr, GDK_SELECTION_TYPE_STRING,
				8, stripped, strlen (stripped));
#endif
      }

  free (txt);
}

static void
gtk_ftext_class_init (GtkFTextClass * class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkFTextClass *ftext_class;

  object_class = (GtkObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;
  ftext_class = (GtkFTextClass *) class;

  parent_class = gtk_type_class (gtk_widget_get_type ());

  ftext_signals[WORD_CLICK] =
    gtk_signal_new (/*name*/"word_click",
		    /*GtkSignalRunType*/GTK_RUN_FIRST,
		    /*GtkType*/object_class->type,
		    /*funcoffset*/GTK_SIGNAL_OFFSET (GtkFTextClass, word_click),
		    /*GtkSignalMarshaller*/gtk_marshal_NONE__POINTER_POINTER,
		    /*returnval*/GTK_TYPE_NONE,
		    /*num args*/2, /*args*/GTK_TYPE_POINTER, GTK_TYPE_POINTER);

  ftext_signals[CHANGED] =
    gtk_signal_new (/*name*/"changed",
		    /*GtkSignalRunType*/GTK_RUN_FIRST,
		    /*GtkType*/object_class->type,
		    /*funcoffset*/GTK_SIGNAL_OFFSET (GtkFTextClass, changed),
		    /*GtkSignalMarshaller*/gtk_signal_default_marshaller,
		    /*returnval*/GTK_TYPE_NONE,
		    /*num args*/0);

  gtk_object_class_add_signals (object_class, ftext_signals, LAST_SIGNAL);

  object_class->destroy = gtk_ftext_destroy;

  widget_class->realize = gtk_ftext_realize;
  widget_class->size_request = gtk_ftext_size_request;
  widget_class->size_allocate = gtk_ftext_size_allocate;
  widget_class->button_press_event = gtk_ftext_button_press;
  widget_class->button_release_event = gtk_ftext_button_release;
  widget_class->motion_notify_event = gtk_ftext_motion_notify;
  widget_class->leave_notify_event = gtk_ftext_leave_notify;
  widget_class->draw = gtk_ftext_draw;
  widget_class->expose_event = gtk_ftext_expose;

  ftext_class->word_click = NULL;
  ftext_class->changed    = NULL;
}

guint gtk_ftext_get_type ()
{
  static guint ftext_type = 0;

  if (!ftext_type)
    {
      GtkTypeInfo ftext_info = {
	"GtkFText",
	sizeof (GtkFText),
	sizeof (GtkFTextClass),
	(GtkClassInitFunc) gtk_ftext_class_init,
	(GtkObjectInitFunc) gtk_ftext_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      ftext_type = gtk_type_unique (gtk_widget_get_type (), &ftext_info);
    }

  return ftext_type;
}

/*void
gtk_ftext_thaw (GtkFText *ftext)
{
   if (ftext->frozen > 0)
      ftext->frozen--;

   if (ftext->frozen == 0)
      gtk_ftext_render_page (ftext);
}

void
gtk_ftext_freeze (GtkFText *ftext)
{
   ftext->frozen++;
}*/

/* gives width of a string */

static int
gtk_ftext_text_width (GtkFText * ftext, unsigned char *str, int len)
{
  int width = 0;

  if (ftext->fixed_width_font)
    return (ftext->space_width * len);

  while (len)
    {
      width += ftext->fontwidth[*str];
      str++;
      len--;
    }

  return width;
}

/* actually draw text to screen */

static int
gtk_ftext_render_flush (GtkFText * ftext, int x, int y, char *str, int len,
			GdkGC *gc)
{
  int str_width;

  if (ftext->dont_render || len < 1)
    return 0;

  if (ftext->fonttype == FONT_1BYTE)
    str_width = gtk_ftext_text_width (ftext, str, len);
  else
    str_width = gdk_text_width (ftext->font, str, len);

  gdk_draw_rectangle (ftext->draw_buf, ftext->bgc, 1,
		      x, y - ftext->font->ascent, str_width,
		      ftext->fontsize);

  gdk_draw_text (ftext->draw_buf, ftext->font, gc, x, y, str, len);

  if (ftext->bold)
    gdk_draw_text (ftext->draw_buf, ftext->font, gc, x + 1, y, str, len);

  if (ftext->underline)
    gdk_draw_line (ftext->draw_buf, gc, x, y+1, x+str_width-1, y+1);

  return str_width;
}

static void
gtk_ftext_set_mark_scheme(GtkFText *ftext) {
  ftext_set_fg (ftext->fgc, 0x000000);
  ftext_set_fg (ftext->bgc, 0xc0c0c0);
}

static void
gtk_ftext_set_scheme(GtkFText *ftext,int fore) {
  if (fore<0) fore=0xffffff;
  ftext_set_fg (ftext->fgc, fore);
  ftext_set_fg (ftext->bgc, ftext->col_back);
}

static void
gtk_ftext_reset (GtkFText * ftext, int mark, int attribs)
{
  if (attribs)
    {
      ftext->underline = FALSE;
      ftext->bold = FALSE;
    }
  if (!mark)
    gtk_ftext_set_scheme(ftext,ftext->col_fore);
}

/* render a single line, which WONT wrap */

static void
gtk_ftext_render_str (GtkFText * ftext, int y, textentry * ent, char *str,
		      int len, int win_width, int indent, int line)
{
  GdkGC *gc;
  int i = 0, x = indent, j = 0;
  char *pstr = str;
  int col_num, tmp;
  int offset;
  int mark = FALSE;
  int hilight = FALSE;

  offset = str - ent->str;

  if (line < 255 && line >= 0)
    ftext->grid_offset[line] = offset;

  gc = ftext->fgc;	 /* our foreground GC */
  gtk_ftext_set_scheme(ftext,ent->color);

  if (ent->mark_start != -1 &&
      ent->mark_start <= i + offset && ent->mark_end > i + offset)
    {
      gtk_ftext_set_mark_scheme(ftext);
      mark = TRUE;
    }
#ifdef MOTION_MONITOR
  if (ftext->hilight_ent == ent &&
      ftext->hilight_start <= i + offset && ftext->hilight_end > i + offset)
    {
      ftext->underline = TRUE;
      hilight = TRUE;
    }
#endif

  if (!ftext->skip_border_fills) {
    /* draw background to the left of the text */
    if (str == ent->str && indent && ftext->time_stamp) {
      /* don't overwrite the timestamp */
      if (indent > ftext->stamp_width) {
	gdk_draw_rectangle (ftext->draw_buf, ftext->bgc, 1, 
			    ftext->stamp_width, y - ftext->font->ascent,
			    indent - ftext->stamp_width, ftext->fontsize);
      }
    } else {
      /* fill the indent area with background gc */
      gdk_draw_rectangle (ftext->draw_buf, ftext->bgc, 1, 0,
			  y - ftext->font->ascent, indent, ftext->fontsize);
    }
  }
  
  while (i < len) {
    
#ifdef MOTION_MONITOR
    if (ftext->hilight_ent == ent && ftext->hilight_start == (i + offset))
      {
	  x += gtk_ftext_render_flush (ftext, x, y, pstr, j, gc);
	  pstr += j;
	  j = 0;
	  ftext->underline = TRUE;
	  hilight = TRUE;
      }
#endif
    
    if (!mark && ent->mark_start == (i + offset))
      {
	x += gtk_ftext_render_flush (ftext, x, y, pstr, j, gc);
	pstr += j;
	j = 0;
	gtk_ftext_set_mark_scheme(ftext);
	mark = TRUE;
      }

    switch (str[i]) {
    case '\t':
      str[i] = ' ';
      j++;
      break;
    case '\n':
      break;
    default:
      j++;
    }
    i++;
    
#ifdef MOTION_MONITOR
    if (ftext->hilight_ent == ent && ftext->hilight_end == (i + offset))
      {
	x += gtk_ftext_render_flush (ftext, x, y, pstr, j, gc);
	pstr += j;
	j = 0;
	ftext->underline = FALSE;
	hilight = FALSE;
      }
#endif
    
    if (mark && ent->mark_end == (i + offset)) {
      x += gtk_ftext_render_flush (ftext, x, y, pstr, j, gc);
      pstr += j;
      j = 0;      
      gtk_ftext_set_scheme(ftext,ent->color);
      mark = FALSE;
    }
  } /* while i < len */
  
  if (j)
    x += gtk_ftext_render_flush (ftext, x, y, pstr, j, gc);
  
  /* draw background to the right of the text */
  if (!ftext->skip_border_fills)
    gdk_draw_rectangle (ftext->draw_buf, ftext->bgc, 1, 
			x, y - ftext->font->ascent, (win_width + MARGIN) - x,
			ftext->fontsize);
}

/* render a single line, which may wrap to more lines */

static int
gtk_ftext_render_line (GtkFText * ftext, textentry * ent, int line,
		       int lines_max, int subline, int win_width)
{
  char *str = ent->str;
  char *time_str;
  int orig_len;
  int len;
  int y;
  int ret = 1;
  int tmp;
  int str_width = ent->str_width;
  int indent = ent->indent;

  orig_len = len = ent->str_len;

  if (ftext->time_stamp && !ftext->skip_stamp)
    {
      time_str = ctime (&ent->stamp) + 10;
      time_str[0] = '[';
      time_str[9] = ']';
      time_str[10] = 0;
      y = (ftext->fontsize * line) + ftext->font->ascent - ftext->pixel_offset;
      gtk_ftext_render_str (ftext, y, ent, time_str, 10, win_width, 2, line);
    }

  do {
    y = (ftext->fontsize * line) + ftext->font->ascent - ftext->pixel_offset;
    str_width += indent;
    tmp = 0;
    while (str_width > win_width || (!is_del (str[len]) && ftext->wordwrap)) {
      if (str_width <= win_width && !tmp)
	tmp = len;
      len--;
      if (ftext->wordwrap && tmp - len > WORDWRAP_LIMIT) {
	len = tmp;
	str_width = gtk_ftext_text_width (ftext, str, len) + indent;
	break;
      }
      if (len < 1)
	return 1;

      /* this is quite a HACK but it speeds things up!
      if (str_width > win_width + 256)
	len -= 10;
      */

      str_width = gtk_ftext_text_width (ftext, str, len) + indent;
    }

    if (!subline)
      gtk_ftext_render_str (ftext, y, ent, str, len, win_width, indent, line);
    else {
      ftext->dont_render = TRUE;
      gtk_ftext_render_str (ftext, y, ent, str, len, win_width, indent, line);
      ftext->dont_render = FALSE;
      subline--;
      line--;
      ret--;
    }
    
    if (ftext->wordwrap && str[len] == ' ')
      len++;
    
    if (len == orig_len || lines_max <= line + 1)
      return ret;
    
    ret++;
    str += len;
    len = orig_len = ent->str_len - (str - ent->str);
    line++;
    indent = ftext->indent;
    str_width = gtk_ftext_text_width (ftext, str, len);
  } while (1);
}

static void
gtk_ftext_fix_indent (GtkFText * ftext)
{
  int j;

  /* make indent a multiple of the space width */
  if (ftext->indent && ftext->space_width)
    {
      j = 0;
      while (j < ftext->indent)
	{
	  j += ftext->space_width;
	}
      ftext->indent = j;
    }
}

static void
gtk_ftext_recalc_widths (GtkFText * ftext, int do_str_width)
{
  textentry *ent;

  /* since we have a new font, we have to recalc the text widths */
  ent = ftext->text_first;
  while (ent)
    {
      if (do_str_width)
	{
	  ent->str_width =
	    gtk_ftext_text_width (ftext, ent->str, ent->str_len);
	}
      if (ent->left_len != -1)
	{
	  ent->indent =
	    (ftext->indent -
	     gtk_ftext_text_width (ftext, ent->str,
				   ent->left_len)) - ftext->space_width;
	  if (ent->indent < MARGIN)
	    ent->indent = MARGIN;
	}
      ent = ent->next;
    }

  gtk_ftext_calc_lines (ftext, FALSE);
}

void
gtk_ftext_set_font (GtkFText * ftext, GdkFont * font, char *name)
{
  if (ftext->font)
    gdk_font_unref (ftext->font);

  if (font)
    {
      ftext->font = font;
      gdk_font_ref (font);
    } else
      font = ftext->font = gdk_font_load (name);

  if (!font)
    font = ftext->font = gdk_font_load ("fixed");

  switch (font->type)
    {
    case GDK_FONT_FONT:
      ftext->fontsize = font->ascent + font->descent;
				/* without X11 pretend they are all 2BYTE- This is ok, just
				   a bit slower. */
      ftext->fonttype = FONT_2BYTE;
      ftext->space_width = gdk_char_width (font, ' ');
      break;

    case GDK_FONT_FONTSET:
      ftext->fontsize = gdk_text_height (font, " ", 1);
      ftext->fonttype = FONT_SET;
      ftext->space_width = gdk_char_width (font, ' ');
      break;
    }

  /* kudgy fixed-width font checking */
  if (ftext->space_width == gdk_char_width (ftext->font, 'Z'))
    ftext->fixed_width_font = TRUE;
  else
    ftext->fixed_width_font = FALSE;

  ftext->stamp_width =
    gtk_ftext_text_width (ftext, "[88:88:88]", 10) + MARGIN;

  gtk_ftext_fix_indent (ftext);

  if (GTK_WIDGET_REALIZED (ftext))
    {
      if (ftext->fonttype != FONT_SET)
	gdk_gc_set_font (ftext->fgc, ftext->font);

      gtk_ftext_recalc_widths (ftext, TRUE);
    }
}

gchar *
gtk_ftext_get_chars (GtkFText * ftext)
{
  int lenght = 0;
  gchar *chars;
  textentry *tentry = ftext->text_first;
  while (tentry != NULL)
    {
      lenght += tentry->str_len + 1;
      tentry = tentry->next;
    }
  if (lenght == 0)
    return NULL;
  chars = g_malloc (lenght + 1);
  *chars = 0;

  tentry = ftext->text_first;
  while (tentry != NULL)
    {
      strcat (chars, tentry->str);
      strcat (chars, "\n");
      tentry = tentry->next;
    }

  return chars;
}

/* High speed version of gtk_ftext_lines_taken, only for FONT_1BYTE. */
/* This function is 5-10X faster than gtk_ftext_lines_taken_general. */

static int
gtk_ftext_lines_taken_8bit (GtkFText * ftext, textentry * ent)
{
  int win_width, str_width, len, lines, pos, last_space;
  unsigned char *str;

  win_width = GTK_WIDGET (ftext)->allocation.width - MARGIN;
  str_width = ent->str_width + ent->indent;

  if (str_width < win_width)
    return 1;

  str = ent->str;

  lines = 1;
  str_width = ent->indent;
  pos = 0;
  last_space = -1;
  do
    {
      str_width += ftext->fontwidth[(int)str[pos]];
      if (str_width > win_width)
	{
	  lines++;
	  if (ftext->wordwrap)
	    {
	      if (str[pos] == ' ')
		pos++;
	      else if (last_space != -1 && pos - last_space <= WORDWRAP_LIMIT)
		pos = last_space + 1;
	    }
	  str_width = ftext->fontwidth[(int)str[pos]] + ftext->indent;
	  last_space = -1;
	}
      if (is_del (str[pos]))
	last_space = pos;
      pos++;
    } while (pos < len);

  free(str);
  return lines;
}

static int
gtk_ftext_lines_taken_general (GtkFText * ftext, textentry * ent)
{
  int tmp, orig_len, indent, len, win_width, str_width, lines = 0;
  char *str;

  str = ent->str;
  len = orig_len = ent->str_len;
  indent = ent->indent;

  if (len < 2)
    return 1;

  win_width = GTK_WIDGET (ftext)->allocation.width - MARGIN;
  str_width = ent->str_width + indent;

  while (1) {
    lines++;
    if (str_width <= win_width)
      break;
    tmp = 0;
    while (str_width > win_width || (!is_del (str[len]) && ftext->wordwrap))
      {
	if (str_width <= win_width && !tmp)
	  tmp = len;
	len--;
	if (ftext->wordwrap && tmp - len > WORDWRAP_LIMIT)
	    {
	      len = tmp;
	      str_width = gtk_ftext_text_width (ftext, str, len) + indent;
	      break;
	    }
	if (len < 1)
	  return 1;
	/*
	  if (str_width > win_width + 256)
	  len -= 10;
	*/
	str_width = gtk_ftext_text_width (ftext, str, len) + indent;
      }
    
    if (len == orig_len)
      break;
    
    if (ftext->wordwrap && str[len] == ' ')
      len++;
    
    str += len;
    len = ent->str_len - (str - ent->str);
    indent = ftext->indent;
    str_width = gtk_ftext_text_width (ftext, str, len) + indent;
  }
  return lines;
}

static int
gtk_ftext_lines_taken (GtkFText * ftext, textentry * ent)
{
  if (ftext->fonttype == FONT_1BYTE)
    return gtk_ftext_lines_taken_8bit (ftext, ent);

  return gtk_ftext_lines_taken_general (ftext, ent);
}

/* Calculate number of actual lines (with wraps), to set adj->lower. *
 * This should only be called when the window resizes.               */

static void
gtk_ftext_calc_lines (GtkFText * ftext, int fire_signal)
{
  textentry *ent;
  int width;
  int height;
  int lines;

  width = GTK_WIDGET (ftext)->allocation.width - MARGIN;
  height = GTK_WIDGET (ftext)->allocation.height;

  if (width < 30 || height < ftext->fontsize || width < ftext->indent + 30)
    return;

  lines = 0;
  ent = ftext->text_first;
  while (ent)
    {
      ent->lines_taken = gtk_ftext_lines_taken (ftext, ent);
      lines += ent->lines_taken;
      ent = ent->next;
    }

  ftext->pagetop_ent = NULL;
  ftext->num_lines = lines;
  gtk_ftext_adjustment_set (ftext, fire_signal);
}

/* find the n-th line in the linked list, this includes wrap calculations */

static textentry *
gtk_ftext_nth (GtkFText * ftext, textentry * ent, int line, int width,
	       int *subline)
{
  int lines = 0;

  if (ent == NULL)
    {
      ent = ftext->text_first;
      line += ftext->adj->value;
    } else
      {
	lines -= *subline;
      }

  while (ent)
    {
      lines += ent->lines_taken;
      if (lines > line)
	{
	  *subline = ent->lines_taken - (lines - line);
	  return ent;
	}
      ent = ent->next;
    }
  return 0;
}

/* render 2 ents (or an inclusive range) */

static void
gtk_ftext_render_ents (GtkFText * ftext, textentry * enta, textentry * entb,
		       int inclusive)
{
  textentry *ent, *orig_ent, *tmp_ent;
  int line;
  int lines_taken;
  int lines_max;
  int width;
  int height;
  int subline;
  int drawing = FALSE;

  if (ftext->indent < MARGIN)
    ftext->indent = MARGIN;	  /* 2 pixels is our left margin */

  gdk_window_get_size (GTK_WIDGET (ftext)->window, &width, &height);
  width -= MARGIN;

  if (width < 32 || height < ftext->fontsize || width < ftext->indent + 30)
    return;

#ifdef SMOOTH_SCROLL
  lines_max = (height - ftext->font->descent + ftext->pixel_offset) / ftext->fontsize + 1;
#else
  lines_max = (height - ftext->font->descent) / ftext->fontsize;
#endif
  line = 0;
  orig_ent = ftext->pagetop_ent;
  subline = ftext->pagetop_subline;

  /* check if enta is before the start of this page */
  if (inclusive)
    {
      tmp_ent = orig_ent;
      while (tmp_ent)
	{
	  if (tmp_ent == enta)
	    break;
	  if (tmp_ent == entb)
	    {
	      drawing = TRUE;
	      break;
	    }
	  tmp_ent = tmp_ent->next;
	}
    }

  line = 0;

  ent = orig_ent;
  while (ent)
    {
      if (inclusive && ent == enta)
	drawing = TRUE;

      if (drawing || ent == entb || ent == enta)
	{
	  gtk_ftext_reset (ftext, FALSE, TRUE);
	  lines_taken = gtk_ftext_render_line (ftext, ent,
					       line, lines_max,
					       subline, width);
	  line += ent->lines_taken;
	  line -= subline;
	  if (ent == orig_ent)
	    subline = 0;
	} else
	  {
	    if (ent == orig_ent)
	      {
		line -= subline;
		subline = 0;
	      }
	    line += ent->lines_taken;
	  }

      if (inclusive && ent == entb)
	break;

      if (line >= lines_max)
	break;

      ent = ent->next;
    }
}

/* render a whole page/window, starting from 'startline' */

static void
gtk_ftext_render_page (GtkFText * ftext)
{
  textentry *ent;
  int line;
  int lines_max;
  int width;
  int height;
  int subline;
  int startline = ftext->adj->value;

  if (ftext->indent < MARGIN)
    ftext->indent = MARGIN;	  /* 2 pixels is our left margin */

  gdk_window_get_size (GTK_WIDGET (ftext)->window, &width, &height);
  width -= MARGIN;

  if (width < 32 || height < ftext->fontsize || width < ftext->indent + 30)
    return;

#ifdef SMOOTH_SCROLL
  ftext->pixel_offset = ((float)((float)ftext->adj->value - (float)startline) * ftext->fontsize);
  lines_max = (height - ftext->font->descent + ftext->pixel_offset) / ftext->fontsize + 1;
#else
  ftext->pixel_offset = 0;
  lines_max = (height - ftext->font->descent) / ftext->fontsize;
#endif

  subline = line = 0;
  ent = ftext->text_first;

  if (startline > 0)
    ent = gtk_ftext_nth (ftext, ent, startline, width, &subline);

  ftext->pagetop_ent = ent;
  ftext->pagetop_subline = subline;

  while (ent)
    {
      gtk_ftext_reset (ftext, FALSE, TRUE);
      line +=
	gtk_ftext_render_line (ftext, ent, line, lines_max, subline, width);

      subline = 0;

      if (line >= lines_max)
	break;

      ent = ent->next;
    }

  line = (ftext->fontsize * line) - ftext->pixel_offset;
  /* fill any space below the last line with our background GC */
  gdk_draw_rectangle (ftext->draw_buf, ftext->bgc, 1,
		      0, line, width + MARGIN, height - line);

}

void
gtk_ftext_refresh (GtkFText * ftext)
{
  if (GTK_WIDGET_REALIZED (GTK_WIDGET (ftext)))
      gtk_ftext_render_page (ftext);
}

/* remove the topline from the list */

static void
gtk_ftext_remove_top (GtkFText * ftext)
{
  textentry *ent;

  ent = ftext->text_first;
  if (!ent)
    return;
  ftext->num_lines -= ent->lines_taken;
  ftext->text_first = ent->next;

  if (ent == ftext->pagetop_ent)
    ftext->pagetop_ent = NULL;

  if (ent == ftext->last_ent_start)
    ftext->last_ent_start = ent->next;

  if (ent == ftext->last_ent_end)
    {
      ftext->last_ent_start = NULL;
      ftext->last_ent_end = NULL;
    }

  free (ent);
}

void
gtk_ftext_clear (GtkFText * ftext)
{
  textentry *next;

  ftext->last_ent_start = NULL;
  ftext->last_ent_end = NULL;

  while (ftext->text_first)
    {
      next = ftext->text_first->next;
      free (ftext->text_first);
      ftext->text_first = next;
    }
  ftext->text_last = NULL;

  gtk_ftext_calc_lines (ftext, TRUE);
  gtk_ftext_refresh (ftext);
}

void *
gtk_ftext_search (GtkFText * ftext, char *text, void *start)
{
  textentry *ent, *fent;
  char *str;
  int line;

  gtk_ftext_selection_clear (ftext);
  ftext->last_ent_start = NULL;
  ftext->last_ent_end = NULL;

  if (start)
    ent = ((textentry *) start)->next;
  else
    ent = ftext->text_first;
  while (ent)
    {
      if ((str = nocasestrstr (ent->str, text)))
	{
	  ent->mark_start = str - ent->str;
	  ent->mark_end = ent->mark_start + strlen (text);
	  break;
	}
      ent = ent->next;
    }

  fent = ent;
  ent = ftext->text_first;
  line = 0;
  while (ent)
    {
      line += ent->lines_taken;
      ent = ent->next;
      if (ent == fent)
	break;
    }
  while (line > ftext->adj->upper - ftext->adj->page_size)
    line--;

  if (fent != 0)
    {
      ftext->adj->value = line;
      ftext->scrollbar_down = FALSE;
      gtk_adjustment_changed (ftext->adj);
    }
  gtk_ftext_render_page (ftext);

  return fent;
}

static int
gtk_ftext_render_page_timeout (GtkFText * ftext)
{
  GtkAdjustment *adj = ftext->adj;
  gfloat val;

  if (!GTK_WIDGET_REALIZED(ftext))
    goto dontrender;

  if (ftext->scrollbar_down)
    {
      gtk_ftext_adjustment_set (ftext, FALSE);
      gtk_adjustment_set_value (adj, adj->upper - adj->page_size);
    } else
      {
	val = adj->value;
	gtk_ftext_adjustment_set (ftext, TRUE);
	gtk_adjustment_set_value (adj, val);
      }

  if (adj->value >= adj->upper - adj->page_size || adj->value < 1)
    gtk_ftext_render_page (ftext);

 dontrender:

  ftext->add_io_tag = -1;
  return 0;
}

/* append a textentry to our linked list */

static void
gtk_ftext_append_entry (GtkFText * ftext, textentry * ent)
{
  ent->stamp = time (0);
  ent->str_width = gtk_ftext_text_width (ftext, ent->str, ent->str_len);
  ent->mark_start = -1;
  ent->mark_end = -1;
  ent->next = NULL;

  if (ent->indent < MARGIN)
    ent->indent = MARGIN;	  /* 2 pixels is the left margin */

  /* append to our linked list */
  if (ftext->text_last)
    ftext->text_last->next = ent;
  else
    ftext->text_first = ent;
  ftext->text_last = ent;

  ent->lines_taken = gtk_ftext_lines_taken (ftext, ent);
  ftext->num_lines += ent->lines_taken;

  if (ftext->max_lines > 2 && ftext->max_lines < ftext->num_lines)
    gtk_ftext_remove_top (ftext);

  /*   if (ftext->frozen == 0 && ftext->add_io_tag == -1)*/
  if (ftext->add_io_tag == -1)
    {
      ftext->add_io_tag = gtk_timeout_add (REFRESH_TIMEOUT * 2,
					   (GtkFunction)
					   gtk_ftext_render_page_timeout,
					   ftext);
    }
}

/* the main two public functions */

void
gtk_ftext_append (GtkFText * ftext, char *text, int len, int color)
{
  textentry *ent;

  if (len == -1)
    len = strlen (text);

  ent = malloc (len + 1 + sizeof (textentry));
  ent->str = (char *) ent + sizeof (textentry);
  ent->str_len = len;
  if (len)
    memcpy (ent->str, text, len);
  ent->str[len] = 0;
  ent->indent = 0;
  ent->left_len = -1;
  ent->color=color;

  gtk_ftext_append_entry (ftext, ent);
  gtk_signal_emit (GTK_OBJECT (ftext), ftext_signals[CHANGED]);
}

