/* $Id: chess.cc,v 1.41 2001/08/05 03:33:01 bergo Exp $ */

/*

    eboard - chess client
    http://eboard.sourceforge.net
    Copyright (C) 2000-2001 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include "chess.h"
#include "movelist.h"
#include "tstring.h"
#include "util.h"
#include "global.h"

ChessGame::ChessGame() {
  int i;

  for(i=0;i<8;i++)
    protodata[i]=0;

  GameNumber=-1;
  STime=0;
  Inc=0;
  Rated=0;
  Variant=REGULAR;
  memset(PlayerName[0],0,64);
  memset(PlayerName[1],0,64);
  Rating[0][0]=0;
  Rating[1][0]=0;
  last_half_move=-1;
  myboard=0;
  clock_regressive=1;
  info0[0]=0;
  cursor=moves.end();
  mymovelist=0;
  over=0;
  MyColor=WHITE;
  StopClock=0;
}

ChessGame::ChessGame(int _number,int _tyme,int _inc, int _rated,
		     variant _variant,
		     char *p1,char *p2) {
  GameNumber=_number;
  STime=_tyme;
  Inc=_inc;
  Rated=_rated;
  Variant=_variant;
  PlayerName[0][63]=PlayerName[1][63]=0;
  strncpy(PlayerName[0],p1,64);
  strncpy(PlayerName[1],p2,64);
  last_half_move=-1;
  myboard=0;
  cursor=moves.end();
  mymovelist=0;
  over=0;
  MyColor=WHITE;
  StopClock=0;
}

int ChessGame::operator==(int gnum) {
  return(gnum==GameNumber);
}

void ChessGame::setFree() {
  if (myboard)
    myboard->FreeMove=true;
}

void ChessGame::acknowledgeInfo() {
  char s[64],t[64];

  if (!myboard)
    return;
  myboard->freeze();

  if ((!STime)&&(GameNumber > 7000)) {
    strcpy(s,info0);
  } else
    if ((STime)||(Inc)) {
      sprintf(s,"Game #%d - %d %d %s",
	      GameNumber,
	      STime/60,Inc,
	      Rated?"rated":"unrated");
    } else {
      sprintf(s,"Game #%d - untimed %s",
	      GameNumber,
	      Rated?"rated":"unrated");
    }

  
  myboard->setInfo(0,s);

  strcpy(s,variantName(Variant));
  myboard->setInfo(1,s);

  if (over)
    showResult();

  myboard->thaw();
}

void ChessGame::showResult() {
  char str[256];
  if ((!over)||(!myboard))
    return;
  switch(result) {
  case WHITE_WIN: strcpy(str,"1-0 "); break;
  case BLACK_WIN: strcpy(str,"0-1 "); break;
  case DRAW: strcpy(str,"1/2-1/2 "); break;
  default: strcpy(str,"(*) "); break;
  }
  if (strlen(ereason)>2)
    strcat(str,ereason);
  myboard->setInfo(3,str);
}

void ChessGame::updateStock() {
  if (over)
    return;
  if (!myboard)
    return;
  if (!moves.empty())
    myboard->setInfo(5,moves.back()->getHouseString());
}

bool ChessGame::isFresh() {
  return((bool)(moves.empty()));
}

// usually called to remove an illegal move from the movelist
void ChessGame::retreat(int nmoves) {
  for(;nmoves;nmoves--)
    if (!moves.empty()) {
      delete(moves.back());
      moves.pop_back();
    }
  cursor=moves.end();
  cursor--;
}

void ChessGame::updatePosition(Position &p,int movenum,int blacktomove,
			       int wclock,int bclock,char *infoline, bool sndflag=false) {
  Position *np;
  char z[32];

  global.debug("ChessGame","updatePosition");

  if (over)
    return;

  if (last_half_move>=0)
    if (! (moves.empty()) )
      if (p == *(moves.back())) {
	myboard->update(sndflag);
	return;
      }

  np=new Position();
  *np=p;
  np->setLastMove(infoline);
  //  cerr << "LastMove = [" << infoline << "]\n";
  moves.push_back(np);
  cursor=moves.end();
  cursor--;
  last_half_move=(movenum*2)+(blacktomove?0:1);

  if (myboard) {
    myboard->freeze();

    if (StopClock)
      myboard->setClock(wclock,bclock,0,clock_regressive);
    else
      myboard->setClock(wclock,bclock,blacktomove?1:-1,clock_regressive);
    
    if (infoline)
      myboard->setInfo(2,infoline);
    myboard->setInfo(4,np->getMaterialString(Variant));
    myboard->setInfo(5,np->getHouseString());
    myboard->update(sndflag);
    myboard->thaw();
    myboard->contentUpdated();
  }

  if (mymovelist)
    mymovelist->updateList(moves);
}

void ChessGame::fireWhiteClock(int wval,int bval) {
  myboard->setClock(wval,bval,-1,clock_regressive);
}

void ChessGame::setBoard(Board *b) {
  myboard=b;
  if (myboard)
    myboard->setGame(this);
}

Board * ChessGame::getBoard() {
  return(myboard);
}

Position * ChessGame::getLastPosition() {
  if (moves.empty())
    return(&startpos);
  return(moves.back());
}

Position * ChessGame::getCurrentPosition() {
  if (moves.empty())
    return(&startpos);
  if (cursor!=moves.end())
    return(*cursor);
  else
    return(moves.back());
}

Position * ChessGame::getPreviousPosition() {
  list<Position *>::iterator pv;
  if (moves.empty())
    return(&startpos);
  if (cursor!=moves.end()) {
    pv=cursor;
    if (pv!=moves.begin())
      pv--;    
    return(*pv);
  } else {
    pv=moves.end();
    pv--;
    if (pv!=moves.begin())
      pv--;    
    return(*pv);
  }
}

void ChessGame::goBack1() {
  if (moves.empty())
    return;
  if (cursor!=moves.begin())
    cursor--;
}

void ChessGame::goBackAll() {
  if (moves.empty())
    return;
  cursor=moves.begin();
}

void ChessGame::goForward1() {
  if (moves.empty())
    return;
  cursor++;
  if (cursor==moves.end())
    cursor--;
}

void ChessGame::goForwardAll() {
  if (moves.empty())
    return;
  cursor=moves.end();
  cursor--;
}

void ChessGame::openMoveList() {
  if (mymovelist) {
    gtk_window_activate_focus(GTK_WINDOW(mymovelist->widget));
  } else {
    mymovelist=new MoveListWindow(PlayerName[0],PlayerName[1],
				  GameNumber,moves,over,result,ereason);
    mymovelist->setListener(this);
    mymovelist->show();
  }
}

void ChessGame::moveListClosed() {
  delete mymovelist;
  mymovelist=0;
}

void ChessGame::sendDrop(piece p, int x, int y) {
  global.protocol->sendDrop(p,x,y);
}

void ChessGame::sendMove(int x1,int y1,int x2,int y2) {
  Position *p;
  int promote=0;

  if (!moves.empty()) {
    p=moves.back();
    if ((p->getPiece(x1,y1)==(PAWN|WHITE))&&
	(y1==6)&&(y2==7))
      promote=1;
    if ((p->getPiece(x1,y1)==(PAWN|BLACK))&&
	(y1==1)&&(y2==0))
      promote=1;
  }
  
  global.protocol->sendMove(x1,y1,x2,y2,promote);
}

void ChessGame::flipHint(int flip) {
  if (myboard)
    myboard->setFlipped(flip);
}

void ChessGame::enableMoving(int flag) {
  if (!myboard)
    return;
  myboard->setCanMove(flag);
}

void ChessGame::endGame(char *reason,GameResult _result) {
  if (over)
    return; // can't end twice
  result=_result;

  if (myboard)
    myboard->stopClock();

  over=1;
  strncpy(ereason,reason,128);
  ereason[127]=0;

  showResult();
  if (myboard)
    myboard->setCanMove(0);

  if (pgn.empty())
    guessPGNFromInfo();

  if (mymovelist)
    mymovelist->updateList(moves,1,result,ereason);
}

int ChessGame::isOver() {
  return(over);
}

void ChessGame::updateGame(list<Position *> &gamedata) {
  list<Position *>::iterator li;
  Position *np;
  int hmn=-1;

  for(li=moves.begin();li!=moves.end();li++)
    delete(*li);
  moves.clear();

  for(li=gamedata.begin();li!=gamedata.end();li++) {
    np=new Position();
    (*np)=(*(*li));
    moves.push_back(np);
    ++hmn;
  }

  cursor=moves.end();
  cursor--;
  last_half_move=hmn;
  if (mymovelist)
    mymovelist->updateList(moves);
}

void ChessGame::dump() {
  cerr.setf(ios::hex,ios::basefield);
  cerr.setf(ios::showbase);

  cerr << "[game " << ((unsigned int)this) << "] ";

  cerr.setf(ios::dec,ios::basefield);

  cerr << "game#=" << GameNumber << " ";
  cerr << "time=(" << STime << "," << Inc << ") ";
  cerr << "rated=" << Rated << " ";
  cerr << "white=" << PlayerName[0] << " ";
  cerr << "black=" << PlayerName[1] << " ";
  cerr << "over=" << over << " ";

  cerr.setf(ios::hex,ios::basefield);
  cerr.setf(ios::showbase);

  cerr << "board=[" << ((unsigned int)myboard) << "]" << endl;
}

char * ChessGame::getPlayerString(int index) {
  index%=2;
  PrivateString[0]=0;
  if ((global.ShowRating)&&(strlen(Rating[index])))
    sprintf(PrivateString,"%s (%s)",PlayerName[index],Rating[index]);
  else
    strcpy(PrivateString,PlayerName[index]);
  return(PrivateString);
}

// ---- PGN

void ChessGame::guessInfoFromPGN() {
  const char *cp;

  if (cp=pgn.get("White"))
    strcpy(PlayerName[0],cp);

  if (cp=pgn.get("Black"))
    strcpy(PlayerName[1],cp);

  cp=pgn.get("Result");
  if (cp) {
    if (cp[0]=='0') { result=BLACK_WIN; return; }
    if (cp[1]=='/') { result=DRAW; return; }
    if (cp[0]=='1') { result=WHITE_WIN; return; }
    result=UNDEF;
  }
}

void ChessGame::guessPGNFromInfo() {
  time_t now;
  struct tm *snow;
  char z[64];
  Position startpos;

  char rz[64];
  int i,j;
  
  now=time(0);
  snow=localtime(&now);

  pgn.set("Event","?");
  pgn.set("Site","?");
  
  sprintf(z,"%d.%0.2d.%0.2d",1900+snow->tm_year,
	  1+snow->tm_mon,snow->tm_mday);
  
  pgn.set("Date",z);
  pgn.set("Round","?");

  if (strlen(PlayerName[0])) pgn.set("White",PlayerName[0]);
  if (strlen(PlayerName[1])) pgn.set("Black",PlayerName[1]);

  // get rating, but prevent strings like (1967), (----), (UNR)...
  memset(rz,0,64);
  j=strlen(Rating[0]);
  for(i=0;i<j;i++)
    if (isdigit(Rating[0][i]))
      rz[strlen(rz)]=Rating[0][i];
  if (strlen(rz)) pgn.set("WhiteElo",rz);

  memset(rz,0,64);
  j=strlen(Rating[1]);
  for(i=0;i<j;i++)
    if (isdigit(Rating[1][i]))
      rz[strlen(rz)]=Rating[1][i];
  if (strlen(rz)) pgn.set("BlackElo",rz);

  if (over) {
    switch(result) {
    case WHITE_WIN: pgn.set("Result","1-0"); break;
    case BLACK_WIN: pgn.set("Result","0-1"); break;
    case DRAW:      pgn.set("Result","1/2-1/2"); break;
    case UNDEF:     pgn.set("Result","*"); break;
    }
  }

  if ( startpos != *moves.front() )
    pgn.set("FEN",(char *)(moves.front()->getFEN()));
}

bool ChessGame::savePGN(char *filename, bool append=false) {
  FILE *f;
  char z[512],zz[512],*p;
  const char *cp;
  list<Position *>::iterator li;
  int xp,lm,mn;
  TString t(64);

  if (moves.size() < 4) {
    global.status->setText("savePGN failed: Won't save game with less than 2 moves");
    return false;
  }

  // FIXME ~username won't work, only ~/something
  if (filename[0]=='~') {
    sprintf(zz,"%s/%s",getenv("HOME"),&filename[2]);
    filename=zz;
  }

  f=fopen(filename,append?"a":"w");
  if (!f) {
    sprintf(z,"savePGN failed: %s",filename);
    global.status->setText(z);
    return false;
  }

  if (pgn.empty())
    guessPGNFromInfo();
  pgn.write(f);

  if (Variant!=REGULAR)
    fprintf(f,"%%eboard:variant:%s\n",variantName(Variant));
  if (STime > 0)
    fprintf(f,"%%eboard:clock/r:%d:%d:%d\n",STime,Inc,Rated);
  if (strlen(ereason)>5)
    fprintf(f,"%%eboard:clue:%s\n",ereason);

  fprintf(f,"\n");

  lm=0; xp=0;
  for(li=moves.begin();li!=moves.end();li++) {
    p=(*li)->getLastMove();
    if (!strlen(p))
      continue;
    t.set(p);
    mn=atoi(t.token("."));
    if (!mn)
      continue;
    if (mn>lm) {
      fprintf(f,"%d. ",mn);
      lm=mn;
      xp+=(lm>9)?4:3;
    }
    p=t.token("(). \t");
    fprintf(f,"%s ",p);
    xp+=1+strlen(p);

    if (xp>70) {
      xp=0;
      fprintf(f,"\n");
    }
  }
  cp=pgn.get("Result");
  if (cp)
    fprintf(f,"%s\n\n",cp);
  else
    fprintf(f,"*\n\n");
  fclose(f);

  sprintf(z,"--- %s game to PGN file %s",
	  append?"Appended":"Wrote",filename);
  global.output->append(z,0xc0ff00);
  return true;
}

void ChessGame::LoadPGN(char *filename) {
  FILE *f;
  ChessGame *game;
  char z[512],w[64],mn[32],tmp[64],reas[128];
  static char *sep=" \t\n\r";

  ExtPatternMatcher PGNHeader, MoveNumber, Result, 
    VariantHdr, TCHdr, ClueHdr, BMoveNumber;
  PatternBinder Binder;

  TString t(512);
  list<Position *> gl;
  Position *p;
  char *c,*v,*zzz;
  piece color;
  GameResult res;
  long oops;
  int inComment=0;
  int baseid=10000;

  UnboundedProgressWindow *pw;
  int count=0;

  f=fopen(filename,"r");
  if (!f)
    return;

  //  cerr << "reading " << filename << endl;

  PGNHeader.set("[%s \"*\"]*");
  MoveNumber.set("%n.");
  BMoveNumber.set("%n...");
  Result.set("*%N-%N*");
  VariantHdr.set("%%eboard:variant:%s*");
  TCHdr.set("%%eboard:clock/r:%n:%n:%n*");  
  ClueHdr.set("%%eboard:clue:*");

  Binder.add(&PGNHeader,&VariantHdr,&TCHdr,&ClueHdr,0);

  game=new ChessGame();
  game->Variant=REGULAR;
  p=new Position();
  gl.push_back(p);
  color=WHITE;
  oops=ftell(f);
  reas[0]=0;

  pw=new UnboundedProgressWindow("%d games read");

  while(fgets(z,512,f)) {
    //    cerr << z;

    Binder.prepare(z);

    if (TCHdr.match()) {
      game->STime=atoi(TCHdr.getNToken(0));
      game->Inc=atoi(TCHdr.getNToken(1));
      game->Rated=atoi(TCHdr.getNToken(2));
      continue;
    }

    if (ClueHdr.match()) {
      strncpy(reas,ClueHdr.getStarToken(0),128);
      if (reas[strlen(reas)-1]=='\n')
	reas[strlen(reas)-1]=0;
      continue;
    }

    if (VariantHdr.match()) {
      variant vr;
      vr=REGULAR;
      c=VariantHdr.getSToken(0);
      if (!strcmp(c,variantName(CRAZYHOUSE))) vr=CRAZYHOUSE;
      else
	if (!strcmp(c,variantName(BUGHOUSE)))   vr=BUGHOUSE;
	else
	  if (!strcmp(c,variantName(WILD)))       vr=WILD;
	  else
	    if (!strcmp(c,variantName(SUICIDE)))    vr=SUICIDE;
	    else
	      if (!strcmp(c,variantName(LOSERS)))     vr=LOSERS;
	      else
		if (!strcmp(c,variantName(GIVEAWAY)))   vr=GIVEAWAY;
      if (vr!=REGULAR)
	game->Variant=vr;
      continue;
    }

    if (z[0]=='%')
      continue;
    
    if (PGNHeader.match()) {
      strncpy(tmp,PGNHeader.getSToken(0),64);
      game->pgn.set((const char *)tmp, PGNHeader.getStarToken(0));

      if (!strcmp(tmp,"FEN"))
	p->setFEN(game->pgn.get("FEN"));

      continue;
    }

    inComment=StripPGNComments(z,inComment);

    t.set(z);
    while( ( c=t.token(sep) )!= 0 ) {
      //      cerr << "token(" << c << ")\n";
      if ((c[0]>='0')&&(c[0]<='9')&&(MoveNumber.match(c))) { strcpy(mn,c); continue; }
      if (strchr("*01",c[0])||c[1]=='*')
	if ((c[0]=='*')||(c[1]=='*')||(Result.match(c))) {
	  game->updateGame(gl);
	  if ((c[0]=='*')||(c[1]=='*')) {
	    res=UNDEF;
	  } else {
	    v=Result.getNToken(0);
	    switch(v[0]) {
	    case '0': res=BLACK_WIN; break;
	    case '1': res=WHITE_WIN; break;
	    case '2': res=DRAW; break;
	    default: res=UNDEF;
	    }
	  }
	  if (!strlen(reas)) strcpy(reas," ");
	  game->endGame(reas,res);
	  game->guessInfoFromPGN();
	  game->GameNumber=baseid=global.nextFreeGameId(baseid);
	  ++baseid;
	  global.appendGame(game,false);
	  
	  // prepare for next game
	  while(!gl.empty()) {
	    delete(gl.front());
	    gl.erase(gl.begin());
	  }
	  game=new ChessGame();
	  game->Variant=REGULAR;
	  p=new Position();
	  gl.push_back(p);
	  color=WHITE;
	  
	  ++count;
	  if (! (count%5) )
	    pw->setProgress(count);
	  oops=ftell(f);
	  reas[0]=0;
	  break;
	}

      if (BMoveNumber.match(c)) // skip any 14... tokens
	continue;

      p=new Position();
      (*p)=*(gl.back());
      
      // crazyhouse detection hack
      if ((c[1]=='@')&&(game->Variant==REGULAR)) {
	// oh-oh, this is crazyhouse, all game data
	// is oopsed, dump everything and rewind file

	while(!gl.empty()) {
	  delete(gl.front());
	  gl.erase(gl.begin());
	}
	delete game;

	game=new ChessGame();
	game->Variant=CRAZYHOUSE;
	p=new Position();
	gl.push_back(p);
	color=WHITE;
	fseek(f,oops,SEEK_SET);
	reas[0]=0;
	break; // process line
      }

      p->moveAnyNotation(c,color,game->Variant);
      sprintf(w,"%s%s%s",mn,(color==WHITE)?" ":" ... ",c);
      p->setLastMove(w);
      gl.push_back(p);
      color^=COLOR_MASK;      
    } // process line

  } // get more lines

  fclose(f);

  // leftovers
  while(!gl.empty()) {
    delete(gl.front());
    gl.erase(gl.begin());
  }
  delete game;
  delete pw;
}

int ChessGame::StripPGNComments(char *z, int incomment) {
  char aux[512];
  char *p,*q;
  int ic;

  ic=incomment;
  strcpy(aux,z);
  memset(z,0,512);
 
  p=aux;
  q=z;

  for(;*p;p++) {
    if (ic) { // we're inside a comment

      if (*p == '}')
	ic=0;

    } else { // we're not inside a comment

      if (*p == '{') {
	ic=1;
      } else {
	*(q++)=*p;
      }

    }
  }

  return ic;
}

GameResult ChessGame::getResult() {
  return(over?result:UNDEF);
}

const char * ChessGame::variantName(variant v) {
  switch(v) {
  case REGULAR:    return("chess");
  case CRAZYHOUSE: return("crazyhouse");
  case SUICIDE:    return("suicide");
  case BUGHOUSE:   return("bughouse");
  case WILD:       return("wild");
  case LOSERS:     return("losers");
  case GIVEAWAY:   return("giveaway");
  default: return("unknown");
  }
}

char *ChessGame::getEndReason() {
  if (strlen(ereason)>2)
    return(ereason);
  else
    return 0;
}

// --------------- PGN classes

PGNpair::PGNpair() {
  name="none";
  value="empty";
}

PGNpair::PGNpair(const char *n, char *v) {
  name=n;
  value=v;
}
  
void PGNpair::write(FILE *f) {
  fprintf(f,"[%s \"%s\"]\n",name.c_str(),value.c_str());
}

PGNheader::~PGNheader() {
  int i,j;
  j=size();
  for(i=0;i<j;i++)
    delete header[i];
  header.clear();
}

void PGNheader::set(const char *n,char *v) {
  vector<PGNpair *>::iterator li;
  for(li=header.begin();li!=header.end();li++)
    if ( (*li)->name == n ) {
      (*li)->value = v;
      return;
    }
  header.push_back(new PGNpair(n,v));
}

void PGNheader::setIfAbsent(const char *n,char *v) {
  vector<PGNpair *>::iterator li;
  for(li=header.begin();li!=header.end();li++)
    if ( (*li)->name == n )
      return;
  header.push_back(new PGNpair(n,v));
}

void PGNheader::remove(const char *n) {
  vector<PGNpair *>::iterator li;
  for(li=header.begin();li!=header.end();li++)
    if ( (*li)->name == n ) {
      delete(*li);
      header.erase(li);
      break;
    }
}

void PGNheader::write(FILE *f) {
  vector<PGNpair *>::iterator li;
  for(li=header.begin();li!=header.end();li++)
    (*li)->write(f);
}

int PGNheader::empty() {
  return(header.empty());
}

int PGNheader::size() {
  return(header.size());
}

PGNpair * PGNheader::get(int index) {
  return(header[index]);
}

const char * PGNheader::get(const char *n) {
  vector<PGNpair *>::iterator li;
  for(li=header.begin();li!=header.end();li++)
    if ( (*li)->name == n )
      return((*li)->value.c_str());
  return 0;
}

// -- PGN info edit dialog ------

PGNEditInfoDialog::PGNEditInfoDialog(ChessGame *src) :
  ModalDialog("PGN Headers")
{
  GtkWidget *v,*sw,*hb,*hb2,*l[2],*setb,*hs,*bb,*closeb;
  obj=src;

  gtk_window_set_default_size(GTK_WINDOW(widget),270,300);
  v=gtk_vbox_new(FALSE,4);  
  gtk_container_add(GTK_CONTAINER(widget),v);

  sw=gtk_scrolled_window_new(0,0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);

  clist=gtk_clist_new(2);
  gtk_clist_set_shadow_type(GTK_CLIST(clist),GTK_SHADOW_IN);
  gtk_clist_set_selection_mode(GTK_CLIST(clist),GTK_SELECTION_SINGLE);
  gtk_clist_set_column_title(GTK_CLIST(clist),0,"Key");
  gtk_clist_set_column_title(GTK_CLIST(clist),1,"Value");
  gtk_clist_column_titles_passive(GTK_CLIST(clist));
  gtk_clist_column_titles_show(GTK_CLIST(clist));
  gtk_clist_set_column_min_width(GTK_CLIST(clist),0,64);
  gtk_clist_set_column_min_width(GTK_CLIST(clist),1,64);

  gtk_box_pack_start(GTK_BOX(v),sw,TRUE,TRUE,0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(sw),clist);

  hb=gtk_hbox_new(FALSE,0);
  del=gtk_button_new_with_label(" Remove Field ");
  gtk_box_pack_end(GTK_BOX(hb),del,FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(v),hb,FALSE,FALSE,4);

  hb2=gtk_hbox_new(FALSE,0);
  l[0]=gtk_label_new("Key:");
  l[1]=gtk_label_new("Value:");
  en[0]=gtk_entry_new();
  en[1]=gtk_entry_new();
  
  setb=gtk_button_new_with_label(" Set ");

  gtk_box_pack_start(GTK_BOX(hb2),l[0],FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hb2),en[0],TRUE,TRUE,4);
  gtk_box_pack_start(GTK_BOX(hb2),l[1],FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hb2),en[1],TRUE,TRUE,4);
  gtk_box_pack_start(GTK_BOX(hb2),setb,FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(v),hb2,FALSE,FALSE,4);

  hs=gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(v),hs,FALSE,FALSE,4);

  bb=gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(bb), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(bb), 5);
  gtk_box_pack_start(GTK_BOX(v),bb,FALSE,FALSE,2);  

  closeb=gtk_button_new_with_label("Close");
  GTK_WIDGET_SET_FLAGS(closeb,GTK_CAN_DEFAULT);
  gtk_box_pack_start(GTK_BOX(bb),closeb,TRUE,TRUE,0);
  gtk_widget_grab_default(closeb);

  gtk_widget_show(closeb);
  gtk_widget_show(bb);
  gtk_widget_show(hs);
  gtk_widget_show(l[0]);
  gtk_widget_show(l[1]);
  gtk_widget_show(en[0]);
  gtk_widget_show(en[1]);
  gtk_widget_show(setb);
  gtk_widget_show(hb2);

  gtk_widget_show(hb);
  gtk_widget_show(del);
  gtk_widget_show(sw);
  gtk_widget_show(clist);
  gtk_widget_show(v);
  setDismiss(GTK_OBJECT(closeb),"clicked");
  
  populate();

  gtk_widget_set_sensitive(del,FALSE);
  Selection=-1;

  gtk_signal_connect(GTK_OBJECT(setb),"clicked",
		     GTK_SIGNAL_FUNC(pgnedit_set),(gpointer)this);
  gtk_signal_connect(GTK_OBJECT(del),"clicked",
		     GTK_SIGNAL_FUNC(pgnedit_del),(gpointer)this);
  gtk_signal_connect(GTK_OBJECT(clist),"select_row",
		     GTK_SIGNAL_FUNC(pgnedit_rowsel),(gpointer)this);
  gtk_signal_connect(GTK_OBJECT(clist),"unselect_row",
		     GTK_SIGNAL_FUNC(pgnedit_rowunsel),(gpointer)this);
}

void PGNEditInfoDialog::populate() {
  int i,j;
  const char *p[2];
  PGNpair *pp;

  gtk_clist_freeze(GTK_CLIST(clist));
  gtk_clist_clear(GTK_CLIST(clist));

  j=obj->pgn.size();
  for(i=0;i<j;i++) {
    pp=obj->pgn.get(i);
    p[0]=pp->name.c_str();
    p[1]=pp->value.c_str();
    gtk_clist_append(GTK_CLIST(clist),(gchar **)p);
  }
  gtk_clist_thaw(GTK_CLIST(clist));
  Selection=-1;
  gtk_widget_set_sensitive(del,FALSE);
}

void pgnedit_set(GtkWidget *w, gpointer data) {
  PGNEditInfoDialog *me;
  me=(PGNEditInfoDialog *)data;
  char a[64],b[64];
  strncpy(a,gtk_entry_get_text(GTK_ENTRY(me->en[0])),64);
  strncpy(b,gtk_entry_get_text(GTK_ENTRY(me->en[1])),64);
  if (strlen(a) && strlen(b)) {
    me->obj->pgn.set((const char *)a,b);
    me->populate();
  }
}

void pgnedit_del(GtkWidget *w, gpointer data) {
  PGNEditInfoDialog *me;
  const char *k;
  me=(PGNEditInfoDialog *)data;
  if (me->Selection >= 0) {
    k=(me->obj->pgn.get(me->Selection))->name.c_str();
    me->obj->pgn.remove(k);
    me->populate();
  }
}

void pgnedit_rowsel(GtkCList *w, gint row, gint col,
		    GdkEventButton *eb,gpointer data) {
  PGNEditInfoDialog *me;
  PGNpair *pp;
  me=(PGNEditInfoDialog *)data;
  me->Selection=row;
  gtk_widget_set_sensitive(me->del,TRUE);

  pp=me->obj->pgn.get(row);
  if (pp) {
    gtk_entry_set_text(GTK_ENTRY(me->en[0]),pp->name.c_str());
    gtk_entry_set_text(GTK_ENTRY(me->en[1]),pp->value.c_str());
  }
}

void pgnedit_rowunsel(GtkCList *w, gint row, gint col,
		      GdkEventButton *eb,gpointer data) {
  PGNEditInfoDialog *me;
  me=(PGNEditInfoDialog *)data;
  me->Selection=-1;
  gtk_widget_set_sensitive(me->del,FALSE);
}
