<?php
	/**
	* General operations
	*
	* All mapi operations, like create, change and delete, are set in this class.
	* A module calls one of these methods. 
	*
	* Note: All entryids in this class are binary
	*
	* @todo This class is bloated. It also returns data in various arbitrary formats
	* that other functions depend on, making lots of code almost completely unreadable.
	* @package core
	*/

	include_once("server/core/class.filter.php");
	include_once("mapi/class.recurrence.php");
	include_once("mapi/class.meetingrequest.php");
	include_once("mapi/class.freebusypublish.php");

	class Operations 
	{
		function Operations()
		{
		}
		
		/**
		* Gets the hierarchy list of all required stores.
		*
		* getHierarchyList builds an entire hierarchy list of all folders that should be shown in various places. Most importantly,
		* it generates the list of folders to be show in the hierarchylistmodule (left-hand folder browser) on the client.
		*
		* It is also used to generate smaller hierarchy lists, for example for the 'create folder' dialog.
		*
		* The returned array is a flat array of folders, so if the caller wishes to build a tree, it is up to the caller to correlate
		* the entryids and the parent_entryids of all the folders to build the tree.
		*
		* The return value is an associated array with the following keys:
		* - attributes: array("type" => "list")
		* - store: array of stores
		*
		* Each store contains:
        * - attributes: array("id" => entryid of store, name => name of store, subtree => entryid of viewable root, type => default|public|other, foldertype => "all")
		* - folder: array of folders with each an array of properties (see Operations::setFolder() for properties)
		*
		* @param array $properties MAPI property mapping for folders
		* @param int $type Which stores to fetch (HIERARCHY_GET_ALL | HIERARCHY_GET_DEFAULT | HIERARCHY_GET_ONE)
		* @param object $store Only when $type == HIERARCHY_GET_ONE
		*
		* @return array Return structure
		*/
		function getHierarchyList($properties, $type = HIERARCHY_GET_ALL, $store = null)
		{
			
			switch($type) 
			{
				case HIERARCHY_GET_ALL:
					$storelist = $GLOBALS["mapisession"]->getAllMessageStores();
					break;
				
				case HIERARCHY_GET_DEFAULT: 
					$storelist = array($GLOBALS["mapisession"]->getDefaultMessageStore());
					break;
					
				case HIERARCHY_GET_ONE:
					$storelist = array($store);
					break;
			}
						
			$data = array();
			$data["attributes"] = array("type" => "list");
			$data["store"] = array();
			
			foreach($storelist as $store)
			{
				$msgstore_props = mapi_getprops($store, array(PR_ENTRYID, PR_DISPLAY_NAME, PR_IPM_SUBTREE_ENTRYID, PR_IPM_OUTBOX_ENTRYID, PR_IPM_SENTMAIL_ENTRYID, PR_IPM_WASTEBASKET_ENTRYID, PR_MDB_PROVIDER, PR_IPM_PUBLIC_FOLDERS_ENTRYID, PR_IPM_FAVORITES_ENTRYID));
				
				$inboxProps = array();

				switch ($msgstore_props[PR_MDB_PROVIDER])
				{
					case ZARAFA_SERVICE_GUID:
						$storeType = "default";
						break;
					case ZARAFA_STORE_PUBLIC_GUID:
						$storeType = "public";
						break;
					case ZARAFA_STORE_DELEGATE_GUID:
						$storeType = "other";
						break;
				}

				$storeData = array();
				$storeData["attributes"] = array(	"id" => bin2hex($msgstore_props[PR_ENTRYID]), 
													"name" => windows1252_to_utf8($msgstore_props[PR_DISPLAY_NAME]), 
													"subtree" => bin2hex($msgstore_props[PR_IPM_SUBTREE_ENTRYID]),
													"type" => $storeType,
													"foldertype" => "all"
												);
				
				// save username if other store	
				if ($storeType == "other"){
					$storeData["attributes"]["username"] = $GLOBALS["mapisession"]->getUserNameOfStore($msgstore_props[PR_ENTRYID]);
				}
				
				// public store doesn't have an inbox
				if ($storeType != "public"){
					$inbox = mapi_msgstore_getreceivefolder($store);
					
					if(!mapi_last_hresult()) {
						$inboxProps = mapi_getprops($inbox, array(PR_ENTRYID));
					}
				}

				$root = mapi_msgstore_openentry($store, null);
				$rootProps = mapi_getprops($root, array(PR_IPM_APPOINTMENT_ENTRYID, PR_IPM_CONTACT_ENTRYID, PR_IPM_DRAFTS_ENTRYID, PR_IPM_JOURNAL_ENTRYID, PR_IPM_NOTE_ENTRYID, PR_IPM_TASK_ENTRYID, PR_ADDITIONAL_REN_ENTRYIDS));

				$additional_ren_entryids = array();
				if(isset($rootProps[PR_ADDITIONAL_REN_ENTRYIDS])) $additional_ren_entryids = $rootProps[PR_ADDITIONAL_REN_ENTRYIDS];

				$defaultfolders = array(
						"inbox"			=>	array("inbox"=>PR_ENTRYID),
						"outbox"		=>	array("store"=>PR_IPM_OUTBOX_ENTRYID), 
						"sent"			=>	array("store"=>PR_IPM_SENTMAIL_ENTRYID),
						"wastebasket"	=>	array("store"=>PR_IPM_WASTEBASKET_ENTRYID),
						"favorites"		=>	array("store"=>PR_IPM_FAVORITES_ENTRYID),
						"publicfolders"	=>	array("store"=>PR_IPM_PUBLIC_FOLDERS_ENTRYID),
						"calendar"		=>	array("root" =>PR_IPM_APPOINTMENT_ENTRYID),
						"contact"		=>	array("root" =>PR_IPM_CONTACT_ENTRYID),
						"drafts"		=>	array("root" =>PR_IPM_DRAFTS_ENTRYID),
						"journal"		=>	array("root" =>PR_IPM_JOURNAL_ENTRYID),
						"note"			=>	array("root" =>PR_IPM_NOTE_ENTRYID),
						"task"			=>	array("root" =>PR_IPM_TASK_ENTRYID),
						"junk"			=>	array("additional" =>4),
						"syncissues"	=>	array("additional" =>1),
						"conflicts"		=>	array("additional" =>0),
						"localfailures"	=>	array("additional" =>2),
						"serverfailures"=>	array("additional" =>3),
				);

				$storeData["defaultfolders"] = array();
				foreach($defaultfolders as $key=>$prop){
					$tag = reset($prop);
					$from = key($prop);
					switch($from){
						case "inbox":
							if(isset($inboxProps[$tag])) $storeData["defaultfolders"][$key] = bin2hex($inboxProps[$tag]);
							break;
						case "store":
							if(isset($msgstore_props[$tag])) $storeData["defaultfolders"][$key] = bin2hex($msgstore_props[$tag]);
							break;
						case "root":
							if(isset($rootProps[$tag])) $storeData["defaultfolders"][$key] = bin2hex($rootProps[$tag]);
							break;
						case "additional":
							if(isset($additional_ren_entryids[$tag])) $storeData["defaultfolders"][$key] = bin2hex($additional_ren_entryids[$tag]);
					}
				}
				
				$storeData["folder"] = array();

				// check if we just want a single folder or a whole store
				$singleFolder = false;
				if ($storeType == "other"){
					$otherusers = $GLOBALS["mapisession"]->retrieveOtherUsersFromSettings();
			
					if(is_array($otherusers)) {
						foreach($otherusers as $username=>$foldertypes) {
							if ($username == $storeData["attributes"]["username"]){
								if (isset($foldertypes["all"])){
									break; // when foldertype is "all" we only need the whole store, so break the for-loop
								}
								
								// add single folder
								$singleFolder = true;
								foreach($foldertypes as $type){
									// duplicate storeData and remove defaultfolder data
									$folderData = $storeData;
									unset($folderData["defaultfolders"]);
									if (!isset($storeData["defaultfolders"][$type])){
										continue; // TODO: no rights? give error to the user
									}
									$folderEntryID = hex2bin($storeData["defaultfolders"][$type]);

									// load folder props
									$folder = mapi_msgstore_openentry($store, $folderEntryID);
									if (mapi_last_hresult()!=NOERROR){
										continue; // TODO: no rights? give error to the user
									}
									$folderProps = $this->setFolder(mapi_getprops($folder, $properties));
									$folderProps["subfolders"] = -1; // we don't want subfolders here
									array_push($folderData['folder'], $folderProps);
									
									// change it from storeData to folderData
									$folderData["attributes"]["name"] = $folderProps["display_name"]." "._("of")." ".$username;
									$folderData["attributes"]["subtree"] = bin2hex($folderEntryID);
									$folderData["attributes"]["foldertype"] = $type;
									$folderData["attributes"]["id"] .= "_".$type;
									$folderData["defaultfolders"] = array($type=>bin2hex($folderEntryID));

									// add the folder to the list
									array_push($data["store"], $folderData);
								}
							}
						}
					}
				}
				
				// add entire store
				if(!$singleFolder && isset($msgstore_props[PR_IPM_SUBTREE_ENTRYID])) {
					$folder = mapi_msgstore_openentry($store, $msgstore_props[PR_IPM_SUBTREE_ENTRYID]);
					
					if($folder) {
						// Add root folder
						array_push($storeData['folder'], $this->setFolder(mapi_getprops($folder, $properties)));

						if($storeType != "public") {
							// Recursively add all subfolders
							$this->getSubFolders($folder, $store, $properties, $storeData);
						} else {
							$this->getSubFoldersPublic($folder, $store, $properties, $storeData);
						}
					}
					array_push($data["store"], $storeData);
					
				
				}
			}

			return $data;
		}
		
		/**
		 * Helper function to get the subfolders of a folder
		 *
		 * @access private
		 * @param object $folder Mapi Folder Object.
		 * @param object $store Message Store Object
		 * @param array $properties MAPI property mappings for folders
		 * @param array $storeData Reference to an array. The folder properties are added to this array.
		 */
		function getSubFolders($folder, $store, $properties, &$storeData)
		{
			$hierarchyTable = mapi_folder_gethierarchytable($folder, CONVENIENT_DEPTH);

			// Make sure our folders are sorted by name. The ordering from the depth is not important
			// because we send the parent in the folder information anyway.
			mapi_table_sort($hierarchyTable, array(PR_DISPLAY_NAME => TABLE_SORT_ASCEND));
			$subfolders = mapi_table_queryallrows($hierarchyTable, array_merge(Array(PR_ENTRYID,PR_SUBFOLDERS), $properties));

			if (is_array($subfolders)) {
				foreach($subfolders as $subfolder)
				{
					array_push($storeData['folder'], $this->setFolder($subfolder));
				} 
			} 
		}
		
		/**
		 * Helper to loop through the hierarchy list.in the public and replace the parent entry id
		 *
		 * @access private
		 * @param object $folder Mapi Folder Object
		 * @param object $store Message Store Object
		 * @param array $properties MAPI property mappings for folders
		 * @param array $storeData Reference to an array. The folder properties are added to this array
		*/
		function getSubFoldersPublic($folder, $store, $properties, &$storeData)
		{
			
			$hierarchyTable = mapi_folder_gethierarchytable($folder);
			
			if (mapi_table_getrowcount($hierarchyTable) == 0) {
				return false;
			}
			
			mapi_table_sort($hierarchyTable, array(PR_DISPLAY_NAME => TABLE_SORT_ASCEND));

			$subfolders = mapi_table_queryallrows($hierarchyTable, array_merge(Array(PR_ENTRYID,PR_SUBFOLDERS,PR_PARENT_ENTRYID), $properties));

			$folderEntryid = mapi_getprops($folder, array(PR_ENTRYID));
			if (is_array($subfolders)) {
				foreach($subfolders as $subfolder)
				{
					$subfolder[PR_PARENT_ENTRYID] = $folderEntryid[PR_ENTRYID];

					if($subfolder[PR_SUBFOLDERS]) {
						$folderObject = mapi_msgstore_openentry($store, $subfolder[PR_ENTRYID]);

						if ($folderObject) {
							if($this->getSubFoldersPublic($folderObject, $store, $properties, $storeData) == false)
								$subfolder[PR_SUBFOLDERS] = false;
						}
					}
					array_push($storeData['folder'], $this->setFolder($subfolder));
				}
			}
			
			return true;
		}

		/**
		 * Convert MAPI properties into useful XML properties for a folder
		 *
		 * @access private
		 * @param array $folderProps Properties of a folder
		 * @return array List of properties of a folder
		 * @todo The name of this function is misleading because it doesn't 'set' anything, it just reads some properties.
		 */ 		 		
		function setFolder($folderProps)
		{
			$props = array();
			$props["entryid"] = bin2hex($folderProps[PR_ENTRYID]);
			$props["parent_entryid"] = bin2hex($folderProps[PR_PARENT_ENTRYID]);
			$props["store_entryid"] = bin2hex($folderProps[PR_STORE_ENTRYID]);
			$props["display_name"] = windows1252_to_utf8($folderProps[PR_DISPLAY_NAME]);
			
			if(isset($folderProps[PR_CONTAINER_CLASS])) {
				$props["container_class"] = $folderProps[PR_CONTAINER_CLASS];
			} else {
				$props["container_class"] = "IPF.Note";
			}
			
			//Access levels
			$props["access"] = array();
			$props["access"]["modify"]          = $folderProps[PR_ACCESS] & MAPI_ACCESS_MODIFY;
			$props["access"]["read"]            = $folderProps[PR_ACCESS] & MAPI_ACCESS_READ;
			$props["access"]["delete"]          = $folderProps[PR_ACCESS] & MAPI_ACCESS_DELETE;
			$props["access"]["create_hierarchy"]= $folderProps[PR_ACCESS] & MAPI_ACCESS_CREATE_HIERARCHY;
			$props["access"]["create_contents"] = $folderProps[PR_ACCESS] & MAPI_ACCESS_CREATE_CONTENTS;
			
			//Rights levels
			$props["rights"] = array();
			$props["rights"]["deleteowned"]     = $folderProps[PR_RIGHTS] & ecRightsDeleteOwned;
			$props["rights"]["deleteany"]       = $folderProps[PR_RIGHTS] & ecRightsDeleteAny;
			
			$props["content_count"] = $folderProps[PR_CONTENT_COUNT];
			$props["content_unread"] = $folderProps[PR_CONTENT_UNREAD];
			$props["subfolders"] = $folderProps[PR_SUBFOLDERS]?1:-1;
			$props["store_support_mask"] = $folderProps[PR_STORE_SUPPORT_MASK];
			
			return $props;
		}
		
		/**
		 * Create a MAPI folder
		 *
		 * This function simply creates a MAPI folder at a specific location with a specific folder
		 * type.
		 *
		 * @param object $store MAPI Message Store Object in which the folder lives
		 * @param string $parententryid The parent entryid in which the new folder should be created
		 * @param string $name The name of the new folder
		 * @param string $type The type of the folder (PR_CONTAINER_CLASS, so value should be 'IPM.Appointment', etc)
		 * @param array $folderProps reference to an array which will be filled with PR_ENTRYID and PR_STORE_ENTRYID of new folder
		 * @return boolean true if action succeeded, false if not
		 */
		function createFolder($store, $parententryid, $name, $type, &$folderProps)
		{
			$result = false;
			$folder = mapi_msgstore_openentry($store, $parententryid);
			$name = utf8_to_windows1252($name);

			if($folder) {
				$foldername = $this->checkFolderNameConflict($store, $folder, $name);
				$new_folder = mapi_folder_createfolder($folder, $foldername);
				
				if($new_folder) {
					mapi_setprops($new_folder, array(PR_CONTAINER_CLASS => $type));
					$result = true;
					
					$folderProps = mapi_getprops($new_folder, array(PR_ENTRYID, PR_STORE_ENTRYID));
				}
			}
			
			return $result;
		}
		
		/**
		 * Rename a folder
		 *
		 * This function renames the specified folder. However, a conflict situation can arise
		 * if the specified folder name already exists. In this case, the folder name is postfixed with
		 * an ever-higher integer to create a unique folder name.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid The entryid of the folder to rename
		 * @param string $name The new name of the folder
		 * @param array $folderProps reference to an array which will be filled with PR_ENTRYID and PR_STORE_ENTRYID
		 * @return boolean true if action succeeded, false if not
		 * @todo Function name should be renameFolder()
		 */
		function modifyFolder($store, $entryid, $name, &$folderProps)
		{
			$result = false;
			$folder = mapi_msgstore_openentry($store, $entryid);
			$name = utf8_to_windows1252($name);
			
			if($folder && !$this->isSpecialFolder($store, $entryid)) {
				$foldername = $this->checkFolderNameConflict($store, $folder, $name);
				$props = array(PR_DISPLAY_NAME => $foldername);
				
				mapi_setprops($folder, $props);
				mapi_savechanges($folder);
				$result = true;
				
				$folderProps = mapi_getprops($folder, array(PR_ENTRYID, PR_STORE_ENTRYID));
			}
			return $result;
		}
		
		/**
		 * Check if a folder is 'special'
		 *
		 * All default MAPI folders such as 'inbox', 'outbox', etc have special permissions; you can not rename them for example. This
		 * function returns TRUE if the specified folder is 'special'.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid The entryid of the folder
		 * @return boolean true if folder is a special folder, false if not		 		 		 		 		 
		 */
		function isSpecialFolder($store, $entryid)
		{
			$result = true;
			
			$msgstore_props = mapi_getprops($store, array(PR_IPM_SUBTREE_ENTRYID, PR_IPM_OUTBOX_ENTRYID, PR_IPM_SENTMAIL_ENTRYID, PR_IPM_WASTEBASKET_ENTRYID, PR_MDB_PROVIDER, PR_IPM_PUBLIC_FOLDERS_ENTRYID, PR_IPM_FAVORITES_ENTRYID));
			
			// "special" folders don't exists in public store
			if ($msgstore_props[PR_MDB_PROVIDER]==ZARAFA_STORE_PUBLIC_GUID){
				$result = false;
			}else if (array_search($entryid, $msgstore_props)===false){
				$inbox = mapi_msgstore_getreceivefolder($store);
				if($inbox) {
					$inboxProps = mapi_getprops($inbox, array(PR_ENTRYID, PR_IPM_APPOINTMENT_ENTRYID, PR_IPM_CONTACT_ENTRYID, PR_IPM_DRAFTS_ENTRYID, PR_IPM_JOURNAL_ENTRYID, PR_IPM_NOTE_ENTRYID, PR_IPM_TASK_ENTRYID, PR_ADDITIONAL_REN_ENTRYIDS));
					if (array_search($entryid, $inboxProps)===false){
						if (isset($inboxProps[PR_ADDITIONAL_REN_ENTRYIDS]) && is_array($inboxProps[PR_ADDITIONAL_REN_ENTRYIDS])){
							if (array_search($entryid, $inboxProps[PR_ADDITIONAL_REN_ENTRYIDS])===false) {
								$result = false;
							}
						} else {
							$result = false;
						}
					}
				}
			}
			
			return $result;
		}

		/**
		 * Delete a folder
		 *
		 * Deleting a folder normally just moves the folder to the wastebasket, which is what this function does. However,
		 * if the folder was already in the wastebasket, then the folder is really deleted.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $parententryid The parent in which the folder should be deleted
		 * @param string $entryid The entryid of the folder which will be deleted
		 * @param array $folderProps reference to an array which will be filled with PR_ENTRYID, PR_STORE_ENTRYID of the deleted object
		 * @return boolean true if action succeeded, false if not
		 * @todo subfolders of folders in the wastebasket should also be hard-deleted
		 */
		function deleteFolder($store, $parententryid, $entryid, &$folderProps)
		{
			$result = false;
			$msgprops = mapi_getprops($store, array(PR_IPM_WASTEBASKET_ENTRYID));
			$folder = mapi_msgstore_openentry($store, $parententryid);

			if($folder && !$this->isSpecialFolder($store, $entryid)) {
				if(isset($msgprops[PR_IPM_WASTEBASKET_ENTRYID])) {
					// TODO: check if not only $parententryid=wastebasket, but also the parents of that parent...
					if($msgprops[PR_IPM_WASTEBASKET_ENTRYID] == $parententryid) {
						if(mapi_folder_deletefolder($folder, $entryid, DEL_MESSAGES | DEL_FOLDERS)) {
							$result = true;
							
							// if exists, also delete settings made for this folder (client don't need an update for this)
							$GLOBALS["settings"]->delete("folders/entryid_".bin2hex($entryid));
						}
					} else {
						$wastebasket = mapi_msgstore_openentry($store, $msgprops[PR_IPM_WASTEBASKET_ENTRYID]);

						$deleted_folder = mapi_msgstore_openentry($store, $entryid);
						$props = mapi_getprops($deleted_folder, array(PR_DISPLAY_NAME));
						$foldername = $this->checkFolderNameConflict($store, $wastebasket, $props[PR_DISPLAY_NAME]);

						if(mapi_folder_copyfolder($folder, $entryid, $wastebasket, $foldername, FOLDER_MOVE)) {
							$result = true;
							
							$folderProps = mapi_getprops($deleted_folder, array(PR_ENTRYID, PR_STORE_ENTRYID));
						}
					}
				} else {
					if(mapi_folder_deletefolder($folder, $entryid, DEL_MESSAGES | DEL_FOLDERS)) {
						$result = true;
						
						// if exists, also delete settings made for this folder (client don't need an update for this)
						$GLOBALS["settings"]->delete("folders/entryid_".bin2hex($entryid));
					}
				}
			}
			
			return $result;
		}
		
		/**
		 * Empty folder
		 *
		 * Removes all items from a folder. This is a real delete, not a move.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid The entryid of the folder which will be emptied
		 * @param array $folderProps reference to an array which will be filled with PR_ENTRYID and PR_STORE_ENTRYID of the emptied folder
		 * @return boolean true if action succeeded, false if not
		 */
		function emptyFolder($store, $entryid, &$folderProps)
		{
			$result = false;
			$folder = mapi_msgstore_openentry($store, $entryid);
			
			if($folder) {
				$result = mapi_folder_emptyfolder($folder, DEL_ASSOCIATED);

				$hresult = mapi_last_hresult();
				if ($hresult == NOERROR){
					$folderProps = mapi_getprops($folder, array(PR_ENTRYID, PR_STORE_ENTRYID));
					$result = true;
				}
			}
			
			return $result;
		}
		
		/**
		 * Copy or move a folder
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $parentfolderentryid The parent entryid of the folder which will be copied or moved
		 * @param string $sourcefolderentryid The entryid of the folder which will be copied or moved
		 * @param string $destfolderentryid The entryid of the folder which the folder will be copied or moved to
		 * @param boolean $moveFolder true - move folder, false - copy folder
		 * @param array $folderProps reference to an array which will be filled with entryids
		 * @return boolean true if action succeeded, false if not
		 */
		function copyFolder($store, $parentfolderentryid, $sourcefolderentryid, $destfolderentryid, $deststore, $moveFolder, &$folderProps)
		{
			$result = false;
			$sourceparentfolder = mapi_msgstore_openentry($store, $parentfolderentryid);
			$destfolder = mapi_msgstore_openentry($deststore, $destfolderentryid);
			if(!$this->isSpecialFolder($store, $sourcefolderentryid) && $sourceparentfolder && $destfolder && $deststore) {
				$folder = mapi_msgstore_openentry($store, $sourcefolderentryid);
				$props = mapi_getprops($folder, array(PR_DISPLAY_NAME));
				$foldername = $this->checkFolderNameConflict($deststore, $destfolder, $props[PR_DISPLAY_NAME]);
				if($moveFolder) {
					if(mapi_folder_copyfolder($sourceparentfolder, $sourcefolderentryid, $destfolder, $foldername, FOLDER_MOVE)) {
						$result = true;
						$folderProps = mapi_getprops($folder, array(PR_ENTRYID, PR_STORE_ENTRYID));
					}
				} else {
					if(mapi_folder_copyfolder($sourceparentfolder, $sourcefolderentryid, $destfolder, $foldername, COPY_SUBFOLDERS)) {
						$result = true;
					}
				}
			}
			return $result;
		}
		
		/**
		 * Set the readflags of all messages in a folder to 'read'
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid The entryid of the folder
		 * @param array $folderProps reference to an array which will be filled with PR_ENTRYID and PR_STORE_ENTRYID of the folder
		 * @return boolean true if action succeeded, false if not
		 * @todo This function is message a 'set unread' option
		 */
		function setReadFlags($store, $entryid, &$folderProps)
		{
			$result = false;
			$folder = mapi_msgstore_openentry($store, $entryid);
			
			if($folder) {
				if(mapi_folder_setreadflags($folder, array(), SUPPRESS_RECEIPT)) {
					$result = true;
					
					$folderProps = mapi_getprops($folder, array(PR_ENTRYID, PR_STORE_ENTRYID));
				}
			}
			
			return $result;
		}
		
		/**
		 * Read MAPI table
		 *
		 * This function performs various operations to open, setup, and read all rows from a MAPI table.
		 *
		 * The output from this function is an XML array structure which can be sent directly to XML serialisation.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid The entryid of the folder to read the table from
		 * @param array $properties The set of properties which will be read
		 * @param array $sort The set properties which the table will be sort on (formatted as a MAPI sort order)
		 * @param integer $start Starting row at which to start reading rows
		 * @param integer $rowcount Number of rows which should be read
		 * @param array $restriction Table restriction to apply to the table (formatted as MAPI restriction)
		 * @param array $folderProps reference to an array which will be filled with PR_ENTRYID and PR_STORE_ENTRYID of the folder
		 * @return array XML array structure with row data
		 */
		function getTable($store, $entryid, $properties, $sort, $start, $rowcount = false, $restriction = false)
		{
			$data = array();
			$folder = mapi_msgstore_openentry($store, $entryid);
			
			if($folder) {
				$table = mapi_folder_getcontentstable($folder);

				if(!$rowcount) {
					$rowcount = $GLOBALS["settings"]->get("global/rowcount", 50);
				}

				$data["page"] = array();
				$data["page"]["start"] = $start;
				$data["page"]["rowcount"] = $rowcount;
				
				if(is_array($restriction)) {
					mapi_table_restrict($table, $restriction);
				}

				if (is_array($sort) && count($sort)>0)
					mapi_table_sort($table, $sort);

				$data["page"]["totalrowcount"] = mapi_table_getrowcount($table); 
				
				$rows = array();				
	  			$rows = mapi_table_queryrows($table, $properties, $start, $rowcount);

				$data["item"] = array();
				foreach($rows as $row)
				{
					array_push($data["item"], Conversion::mapMAPI2XML($properties, $row));
				}
			}
			
			return $data;
		}
		
		/**
		* Returns TRUE of the MAPI message only has inline attachments
		*
		* @param mapimessage $message The MAPI message object to check
		* @return boolean TRUE if the item contains only inline attachments, FALSE otherwise
		* @deprecated This function is not used, because it is much too slow to run on all messages in your inbox
		*/
		function hasInlineAttachmentsOnly($message)
		{
			$attachmentTable = mapi_message_getattachmenttable($message);
			if($attachmentTable) {
				$attachments = mapi_table_queryallrows($attachmentTable, array(PR_ATTACHMENT_HIDDEN));
				foreach($attachments as $attachmentRow)	{
					if(!isset($attachmentRow[PR_ATTACHMENT_HIDDEN]) || !$attachmentRow[PR_ATTACHMENT_HIDDEN]) {
						return false;
					}
				}
			}
			return true;
		}

		/**
		 * Read message properties
		 *
		 * Reads a message and returns the data as an XML array structure with all data from the message that is needed
		 * to show a message (for example in the preview pane)
		 *
		 * @param object $store MAPI Message Store Object
		 * @param object $message The MAPI Message Object
		 * @param array $properties Mapping of properties that should be read
		 * @param boolean $html2text true - body will be converted from html to text, false - html body will be returned
		 * @return array item properties
		 * @todo Function name is misleading as it doesn't just get message properties
		 */
		function getMessageProps($store, $message, $properties, $html2text = false)
		{
			$props = array();
			
			if($message) {
				if (!isset($properties[PR_ENTRYID]))
					$properties["entryid"] = PR_ENTRYID;

				$messageprops = mapi_getprops($message, $properties);
				$props = Conversion::mapMAPI2XML($properties, $messageprops);
				
				// Get actual SMTP address for sent_representing_email_address and received_by_email_address
				$smtpprops = mapi_getprops($message, array(PR_SENT_REPRESENTING_ENTRYID, PR_RECEIVED_BY_ENTRYID, PR_SENDER_ENTRYID));
				
				if(isset($smtpprops[PR_SENT_REPRESENTING_ENTRYID])) 
    				$props["sent_representing_email_address"] = $this->getEmailAddressFromEntryID($smtpprops[PR_SENT_REPRESENTING_ENTRYID]);
                
				if(isset($smtpprops[PR_SENDER_ENTRYID])) 
    				$props["sender_email_address"] = $this->getEmailAddressFromEntryID($smtpprops[PR_SENDER_ENTRYID]);
                
                if(isset($smtpprops[PR_RECEIVED_BY_ENTRYID]))
                    $props["received_by_email_address"] = $this->getEmailAddressFromEntryID($smtpprops[PR_RECEIVED_BY_ENTRYID]);
                
				
				// Get body content
				$plaintext = $this->isPlainText($message);

				$content = false;
				if (!$plaintext){
					$content = mapi_message_openproperty($message, PR_HTML);
				
					if($content) {
						$filter = new filter();
		
						if(!$html2text) {
							$msgstore_props = mapi_getprops($store, array(PR_ENTRYID));
								
							$content = $filter->safeHTML($content, bin2hex($msgstore_props[PR_ENTRYID]), bin2hex($messageprops[PR_ENTRYID]));
							$props["isHTML"] = true;
						} else {
							$content = false;
						}
					}
				}
				
				if(!$content) {
					$content = mapi_message_openproperty($message, PR_BODY);
					$props["isHTML"] = false;
				}

				$props["body"] = trim(windows1252_to_utf8($content), "\0");
				
				// Get reply-to information
				$messageprops = mapi_getprops($message, array(PR_REPLY_RECIPIENT_ENTRIES));
				if(isset($messageprops[PR_REPLY_RECIPIENT_ENTRIES])) {
					$props["reply-to"] = $this->readReplyRecipientEntry($messageprops[PR_REPLY_RECIPIENT_ENTRIES]);
				}
				
				// Get recipients
				$recipientTable = mapi_message_getrecipienttable($message);
				if($recipientTable) {
					$recipients = mapi_table_queryrows($recipientTable, array(PR_ADDRTYPE, PR_ENTRYID, PR_DISPLAY_NAME, PR_EMAIL_ADDRESS, PR_SMTP_ADDRESS, PR_RECIPIENT_TYPE, PR_RECIPIENT_FLAGS, PR_PROPOSEDNEWTIME, PR_PROPOSENEWTIME_START, PR_PROPOSENEWTIME_END, PR_RECIPIENT_TRACKSTATUS), 0, 99999999);

					$props["recipients"] = array();
					$props["recipients"]["recipient"] = array();
					
					foreach($recipients as $recipientRow)
					{
						$recipient = array();
						$recipient["display_name"] = windows1252_to_utf8(isset($recipientRow[PR_DISPLAY_NAME])?$recipientRow[PR_DISPLAY_NAME]:'');
						$recipient["email_address"] = isset($recipientRow[PR_SMTP_ADDRESS])?$recipientRow[PR_SMTP_ADDRESS]:'';
						$recipient["email_address"] = windows1252_to_utf8($recipient["email_address"]);
						if(isset($recipientRow[PR_RECIPIENT_FLAGS])){
							$recipient["recipient_flags"] = $recipientRow[PR_RECIPIENT_FLAGS];
						}

						if ($recipientRow[PR_ADDRTYPE]=="ZARAFA"){
							$recipient["entryid"] = bin2hex($recipientRow[PR_ENTRYID]);
							// Get the SMTP address from the addressbook if no address is found
							if($recipient["email_address"] == ""){
								$recipient["email_address"] = windows1252_to_utf8($GLOBALS['operations']->getEmailAddressFromEntryID($recipientRow[PR_ENTRYID]));
							}
						}
						if($recipient["email_address"] == ""){
							$recipient["email_address"] = isset($recipientRow[PR_EMAIL_ADDRESS])?$recipientRow[PR_EMAIL_ADDRESS]:'';
						}

						switch($recipientRow[PR_RECIPIENT_TYPE])
						{
							case MAPI_ORIG:
								$recipient["type"] = "orig";
								$recipient["recipient_attendees"] = _("Organizor");
								break;
							case MAPI_TO:
								$recipient["type"] = "to";
								$recipient["recipient_attendees"] = _("Required");
								break;
							case MAPI_CC:
								$recipient["type"] = "cc";
								$recipient["recipient_attendees"] = _("Optional");
								break;
							case MAPI_BCC:
								$recipient["type"] = "bcc";
								$recipient["recipient_attendees"] = _("Resource");
								break;
						}

						// Set propose new time properties
						if(isset($recipientRow[PR_PROPOSEDNEWTIME]) && isset($recipientRow[PR_PROPOSENEWTIME_START]) && isset($recipientRow[PR_PROPOSENEWTIME_END])){
							$recipient["proposenewstarttime"] = $recipientRow[PR_PROPOSENEWTIME_START];
							$recipient["proposenewendtime"] = $recipientRow[PR_PROPOSENEWTIME_END];
							$recipient["proposednewtime"] = $recipientRow[PR_PROPOSEDNEWTIME];
						}else{
							$recipient["proposednewtime"] = 0;
						}
						
						$recipientTrackingStatus = (isset($recipientRow[PR_RECIPIENT_TRACKSTATUS]))?$recipientRow[PR_RECIPIENT_TRACKSTATUS]:0;
						//Switch case for setting the track status of each recipient
						switch($recipientTrackingStatus)
						{
							case 0:
								$recipient["recipient_status"] = _("No Response");
								break;
							case 1:
								$recipient["recipient_status"] = _("Organizor");
								break;
							case 2:
								$recipient["recipient_status"] = _("Tentative");
								break;
							case 3:
								$recipient["recipient_status"] = _("Accepted");
								break;
							case 4:
								$recipient["recipient_status"] = _("Declined");
								break;
							case 5:
								$recipient["recipient_status"] = _("Not Responded");
								break;
						}
						$recipient["recipient_status_num"] = $recipientTrackingStatus;
						array_push($props["recipients"]["recipient"], $recipient);
					}
				}
				
				// Get attachments
				$hasattachProp = mapi_getprops($message, array(PR_HASATTACH));
				if (isset($hasattachProp[PR_HASATTACH])&& $hasattachProp[PR_HASATTACH]){
					$attachmentTable = mapi_message_getattachmenttable($message);
					$attachments = mapi_table_queryallrows($attachmentTable, array(PR_ATTACH_NUM, PR_ATTACH_SIZE, PR_ATTACH_LONG_FILENAME, PR_ATTACH_FILENAME, PR_ATTACHMENT_HIDDEN, PR_DISPLAY_NAME, PR_ATTACH_METHOD));
					$props["attachments"] = array();
					$props["attachments"]["attachment"] = array();

					foreach($attachments as $attachmentRow)
					{
						if(!isset($attachmentRow[PR_ATTACHMENT_HIDDEN]) || !$attachmentRow[PR_ATTACHMENT_HIDDEN]) {
							$attachment = array();
							$attachment["attach_num"] = $attachmentRow[PR_ATTACH_NUM];
							$attachment["attach_method"] = $attachmentRow[PR_ATTACH_METHOD];
							$attachment["size"] = $attachmentRow[PR_ATTACH_SIZE];
							
							if(isset($attachmentRow[PR_ATTACH_LONG_FILENAME]))
								$attachment["name"] = $attachmentRow[PR_ATTACH_LONG_FILENAME];
							else if(isset($attachmentRow[PR_ATTACH_FILENAME]))
								$attachment["name"] = $attachmentRow[PR_ATTACH_FILENAME];
							else if(isset($attachmentRow[PR_DISPLAY_NAME]))
								$attachment["name"] = $attachmentRow[PR_DISPLAY_NAME];
							else
								$attcahment["name"] = "untitled";

							if (isset($attachment["name"])){
								$attachment["name"] = windows1252_to_utf8($attachment["name"]);
							}

							if ($attachment["attach_method"] == ATTACH_EMBEDDED_MSG){
								// open attachment to get the message class
								$attach = mapi_message_openattach($message, $attachment["attach_num"]);
								$embMessage = mapi_attach_openobj($attach);
								$embProps = mapi_getprops($embMessage, array(PR_MESSAGE_CLASS));
								if (isset($embProps[PR_MESSAGE_CLASS]))
									$attachment["attach_message_class"] = $embProps[PR_MESSAGE_CLASS];
							}

							array_push($props["attachments"]["attachment"], $attachment);
						}
					}
				}
			}
			
			return $props;
		}
		
		/**
		 * Get and convert properties of a message into an XML array structure
		 *
		 * @param object $store MAPI Message Store Object
		 * @param object $item The MAPI Object
		 * @param array $properties Mapping of properties that should be read
		 * @return array XML array structure
		 * @todo Function name is misleading, especially compared to getMessageProps()
		 */
		function getProps($store, $item, $properties)
		{
			$props = array();
			
			if($item) {
				$itemprops = mapi_getprops($item, $properties);
				$props = Conversion::mapMAPI2XML($properties, $itemprops);
			}
					
			return $props;
		}

		/**
		 * Get embedded message data
		 *
		 * Returns the same data as getMessageProps, but then for a specific sub/sub/sub message
		 * of a MAPI message.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid entryid of the root message
		 * @param array $properties a set of properties which will be selected
		 * @param array $attach_num a list of attachment numbers (aka 2,1 means 'attachment nr 1 of attachment nr 2')
		 * @return array item XML array structure of the embedded message
		 */
		function getEmbeddedMessage($store, $entryid, $properties, $attach_num)
		{
			$message = mapi_msgstore_openentry($store, $entryid);
			$props = array();
			
			if($message) {
				foreach($attach_num as $num)
				{
					$attachment = mapi_message_openattach($message, $num);
					
					if($attachment) {
  						$message = mapi_attach_openobj($attachment);
  					} else {
  						return $props;
  					}
  				}
  				
				$msgprops = mapi_getprops($message, Array(PR_MESSAGE_CLASS));
				switch($msgprops[PR_MESSAGE_CLASS]){
					case 'IPM.Note':
						$html2text = false;
						break;
					default:
						$html2text = true;
				}
  				$props = $this->getMessageProps($store, $message, $properties, $html2text);
			}
			
			return $props;
		}
		
		/**
		 * Create a MAPI message
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $parententryid The entryid of the folder in which the new message is to be created
		 * @return mapimessage Created MAPI message resource
		 */
		function createMessage($store, $parententryid)
		{
			$folder = mapi_msgstore_openentry($store, $parententryid); 
			return mapi_folder_createmessage($folder); 
		}
		
		/**
		 * Open a MAPI message
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid entryid of the message
		 * @return object MAPI Message
		 */
		function openMessage($store, $entryid)
		{
			return mapi_msgstore_openentry($store, $entryid); 
		}
		
		/**
		 * Save a MAPI message
		 *
		 * The to-be-saved message can be of any type, including e-mail items, appointments, contacts, etc. The message may be pre-existing
		 * or it may be a new message.
		 *
		 * The dialog_attachments parameter represents a unique ID which for the dialog in the client for which this function was called; This
		 * is used as follows; Whenever a user uploads an attachment, the attachment is stored in a temporary place on the server. At the same time,
		 * the temporary server location of the attachment is saved in the session information, accompanied by the $dialog_attachments unique ID. This
		 * way, when we save the message into MAPI, we know which attachment was previously uploaded ready for this message, because when the user saves
		 * the message, we pass the same $dialog_attachments ID as when we uploaded the file.
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $parententryid Parent entryid of the message
		 * @param array $props The MAPI properties to be saved
		 * @param array $recipients XML array structure of recipients for the recipient table
		 * @param string $dialog_attachments Unique check number which checks if attachments should be added ($_SESSION)
		 * @param array $messageProps reference to an array which will be filled with PR_ENTRYID and PR_STORE_ENTRYID of the saved message
		 * @param mixed $copyAttachments If set we copy all attachments from this entryid to this item
		 * @param mixed $copyAttachmentsStore If set we use this store to copy the attachments from
		 * @param array $propertiesToDelete Properties specified in this array are deleted from the MAPI message
		 * @return mapimessage Saved MAPI message resource
		 */
		function saveMessage($store, $parententryid, $props, $recipients, $dialog_attachments, &$messageProps, $copyAttachments = false, $copyAttachmentsStore = false, $propertiesToDelete = array())
		{
			$message = false;

			// Check if an entryid is set, otherwise create a new message
			if(isset($props[PR_ENTRYID]) && !empty($props[PR_ENTRYID])) {
				$message = $this->openMessage($store, $props[PR_ENTRYID]);
			} else {
				$message = $this->createMessage($store, $parententryid);
			}

			if($message) {
				$property = false;
				$body = "";

				// Check if the body is set.
				if (isset($props[PR_BODY])) {
					$body = $props[PR_BODY];
					$property = PR_BODY;
					$bodyPropertiesToDelete = array(PR_HTML, PR_RTF_COMPRESSED);
					
					if(isset($props[PR_HTML])) {
						$body = $this->fckEditor2html($props[PR_BODY],$props[PR_SUBJECT]);
						$property = PR_HTML;
						$bodyPropertiesToDelete = array(PR_BODY, PR_RTF_COMPRESSED);
						unset($props[PR_HTML]);
					}
					unset($props[PR_BODY]);

					$propertiesToDelete = array_unique(array_merge($propertiesToDelete, $bodyPropertiesToDelete));
				}

				if(isset($props[PR_SENT_REPRESENTING_EMAIL_ADDRESS]) && strlen($props[PR_SENT_REPRESENTING_EMAIL_ADDRESS]) > 0 && !isset($props[PR_SENT_REPRESENTING_ENTRYID])){
					// Set FROM field properties
					$props[PR_SENT_REPRESENTING_ENTRYID] = mapi_createoneoff($props[PR_SENT_REPRESENTING_NAME], $props[PR_SENT_REPRESENTING_ADDRTYPE], $props[PR_SENT_REPRESENTING_EMAIL_ADDRESS]);
				}

				// remove mv properties when needed
				foreach($props as $propTag=>$propVal){
					if((mapi_prop_type($propTag) & MV_FLAG) == MV_FLAG) {
						if(is_array($propVal) && count($propVal) == 0) {
							// if empty array then also remove that property
							$propertiesToDelete[] = $propTag;
						} else if(is_null($propVal)) {
							$propertiesToDelete[] = $propTag;
						}
					}
				}
				
				foreach($propertiesToDelete as $prop){
					unset($props[$prop]);
				}

				// Set the properties
				mapi_setprops($message, $props);
				
				// Delete the properties we don't need anymore
				mapi_deleteprops($message, $propertiesToDelete);
				
				if ($property != false){
					// Stream the body to the PR_BODY or PR_HTML property
					$stream = mapi_openpropertytostream($message, $property, MAPI_CREATE | MAPI_MODIFY);
					mapi_stream_setsize($stream, strlen($body));
					mapi_stream_write($stream, $body);
					mapi_stream_commit($stream);
				} 
				
				// Check if $recipients is set
				if(is_array($recipients)){
					// Save the recipients
					$this->setRecipients($message, $recipients);
				}

				// Save the attachments with the $dialog_attachments
				$this->setAttachments($copyAttachmentsStore, $message, $dialog_attachments, $copyAttachments);
				
				// Save changes
				mapi_savechanges($message);
				
				// Get the PR_ENTRYID, PR_PARENT_ENTRYID and PR_STORE_ENTRYID properties of this message
				$messageProps = mapi_getprops($message, array(PR_ENTRYID, PR_PARENT_ENTRYID, PR_STORE_ENTRYID));
			}
			
			return $message;
		}
		
		/**
		 * Save an appointment item.
		 *
		 * This is basically the same as saving any other type of message with the added complexity that
		 * we support saving exceptions to recurrence here. This means that if the client sends a basedate
		 * in the action, that we will attempt to open an existing exception and change that, and if that
		 * fails, create a new exception with the specified data. 
		 *
		 * @param mapistore $store MAPI store of the message
		 * @param string $parententryid Parent entryid of the message (folder entryid, NOT message entryid)
		 * @param array $action Action array containing XML request
		 * @return array of PR_ENTRYID, PR_PARENT_ENTRYID and PR_STORE_ENTRYID properties of modified item
		 */
		function saveAppointment($store, $parententryid, $action)
		{
			$newitem = false;
			$exception = false;
			$messageProps = false;
			$properties = $GLOBALS["properties"]->getAppointmentProperties();

			if($store && $parententryid) {
				if(isset($action["props"])) {
					$messageProps = array();

					if(isset($action["props"]["entryid"]) && !empty($action["props"]["entryid"]) ) {
						// Modify existing or add/change exception
						$message = mapi_msgstore_openentry($store, hex2bin($action["props"]["entryid"]));

						if($message) {
							$props = mapi_getprops($message, $properties);

							// Check if appointment is an exception to a recurring item
							if(isset($action["props"]["basedate"]) && $action["props"]["basedate"] > 0) {
								$exception = true;

								// Create recurrence object
								$recurrence = new Recurrence($store, $message);

								$basedate = (int) $action["props"]["basedate"];

								if(isset($action["delete"]) && $action["delete"]) {
									$recurrence->createException(array(), $basedate, true);
								} else {
									// Retrieve the items on the given start and enddate 
									$recuritems = $recurrence->getItems((int) $action["props"]["startdate"], (int) $action["props"]["duedate"]);
									
									
									// Loop through the items
									$items = count($recuritems);
									foreach($recuritems as $recuritem) 
									{
										if($recurrence->isSameDay($recuritem["basedate"],$basedate)) {
											$items--;
										}
									}

									// A recurring item can take place only once on a day. So
									// no items should be left on that given day --> items 
									// should be 0 (zero), before saving it
									if($items == 0) {
										if($recurrence->isException($basedate)) {
											$recurrence->modifyException(Conversion::mapXML2MAPI($properties, $action["props"]), $basedate);
										} else {
											$recurrence->createException(Conversion::mapXML2MAPI($properties, $action["props"]), $basedate);
										}
									}
								}
								mapi_savechanges($message);
							} else {								
								// Modifying non-exception (the series) or normal appointment item
								if(isset($action["recipients"]["recipient"])) {
									$recips = $action["recipients"]["recipient"];
								} else {
									$recips = false;
								}

								$message = $GLOBALS["operations"]->saveMessage($store, $parententryid, Conversion::mapXML2MAPI($properties, $action["props"]), $recips, $action["dialog_attachments"], $messageProps);

								// Only save recurrence if it has been changed by the user (because otherwise we'll reset
								// the exceptions)								
								if(isset($action['props']['recurring_reset']) && $action['props']['recurring_reset'] == 1) {
									$recur = new Recurrence($store, $message);
									
									if(isset($action['props']['timezone'])) {
										$tzprops = Array('timezone','timezonedst','dststartmonth','dststartweek','dststartday','dststarthour','dstendmonth','dstendweek','dstendday','dstendhour');
										
										// Get timezone info
										$tz = Array();
										foreach($tzprops as $tzprop) {
											$tz[$tzprop] = $action['props'][$tzprop];
										}
									}
										
									// Act like the 'props' are the recurrence pattern; it has more information but that
									// is ignored
									$recur->setRecurrence($tz, $action['props']);
									
									mapi_savechanges($message);
								}
							}								 
							// Get the properties of the main object of which the exception was changed, and post
							// that message as being modified. This will cause the update function to update all
							// occurrences of the item to the client
							$messageProps = mapi_getprops($message, array(PR_ENTRYID, PR_PARENT_ENTRYID, PR_STORE_ENTRYID));
						}
					} else {
						$newitem = true;
						// New item
						$recips = array();
						
						if(isset($action["recipients"]["recipient"]))
							$recips = $action["recipients"]["recipient"];
							
						//Set sender of new Appointment.
						$this->setSenderAddress($store, $action);	
						
						$message = $GLOBALS["operations"]->saveMessage($store, $parententryid, Conversion::mapXML2MAPI($properties, $action["props"]), $recips, $action["dialog_attachments"], $messageProps);
					
						if(isset($action['props']['timezone'])) {
							$tzprops = Array('timezone','timezonedst','dststartmonth','dststartweek','dststartday','dststarthour','dstendmonth','dstendweek','dstendday','dstendhour');
							
							// Get timezone info
							$tz = Array();
							foreach($tzprops as $tzprop) {
								$tz[$tzprop] = $action['props'][$tzprop];
							}
						}
							
						// Set recurrence
						if(isset($action['props']['recurring']) && $action['props']['recurring'] == 1) {
							$recur = new Recurrence($store, $message);
							$recur->setRecurrence($tz, $action['props']);
							mapi_savechanges($message);
						}		
					}
				}
				
				// Publish updated free/busy information
				$GLOBALS["operations"]->publishFreeBusy($store, $parententryid);
			}
			
			// Check to see if it should be sent as a meeting request
			if(isset($action["send"]) && $action["send"]) {
				$request = new Meetingrequest($store, $message, $GLOBALS["mapisession"]->getSession());
				$request->updateMeetingRequest();

				mapi_savechanges($message);
				$sendMeetingRequestResult = $request->sendMeetingRequest(0, $newitem ? false : _("Updated:") . " ");

				if($sendMeetingRequestResult === true){
					// Return message properties that can be sent to the bus to notify changes
					return $messageProps;
				}else{
					return $sendMeetingRequestResult;
				}
			}else{
				// Return message properties that can be sent to the bus to notify changes
				return $messageProps;
			}
			
		}
		
		/**
		 * Set sent_representing_email_address property of Appointment.
		 *
		 * Before saving any new appointment, sent_representing_email_address property of appointment
		 * should contain email_address of user, who is the owner of store(in which the appointment 
		 * is created).
		 *
		 * @param mapistore $store  MAPI store of the message
		 * @param array     $action reference to action array containing XML request
		 */
		function setSenderAddress($store, &$action)
		{
			$storeProps = mapi_getprops($store, array(PR_MAILBOX_OWNER_ENTRYID));
			$mailuser = mapi_ab_openentry($GLOBALS["mapisession"]->getAddressbook(), $storeProps[PR_MAILBOX_OWNER_ENTRYID]);
			if($mailuser){
				$userprops = mapi_getprops($mailuser, array(PR_ADDRTYPE, PR_DISPLAY_NAME, PR_EMAIL_ADDRESS, PR_SMTP_ADDRESS));
				$action["props"]["sent_representing_entryid"]       = bin2hex($storeProps[PR_MAILBOX_OWNER_ENTRYID]);
				$action["props"]["sent_representing_name"]          = $userprops[PR_DISPLAY_NAME];
				$action["props"]["sent_representing_addrtype"]      = $userprops[PR_ADDRTYPE];
				if($userprops[PR_ADDRTYPE] == 'SMTP'){
					$emailAddress = $userprops[PR_EMAIL_ADDRESS];
				}else{
					$emailAddress = $userprops[PR_SMTP_ADDRESS];
				}
				$action["props"]["sent_representing_email_address"] = $emailAddress;
				$action["props"]["sent_representing_search_key"]    = strtoupper($userprops[PR_ADDRTYPE]) . ":" .strtoupper($emailAddress);
			}
		}
		
		/**
		 * Submit a message for sending
		 *
		 * This function is an extension of the saveMessage() function, with the extra functionality
		 * that the item is actually sent and queued for moving to 'Sent Items'. Also, the e-mail addresses
		 * used in the message are processed for later auto-suggestion.
		 *
		 * @see Operations::saveMessage() for more information on the parameters, which are identical.
		 *
		 * @param mapistore $store MAPI Message Store Object
		 * @param array $props The properties to be saved
		 * @param array $recipients XML array structure of recipients for the recipient table
		 * @param string $dialog_attachments Unique ID of attachments for this message
		 * @param array $messageProps reference to an array which will be filled with PR_ENTRYID, PR_PARENT_ENTRYID and PR_STORE_ENTRYID
		 * @param array $copyAttachments if set this holds the entryid of the message which holds the attachments
		 * @param array $copyAttachmentsStore if set we this this store to copy the attachments from
		 * @return boolean true if action succeeded, false if not
		 *
		 */
		function submitMessage($store, $props, $recipients, $dialog_attachments, &$messageProps, $copyAttachments = false, $copyAttachmentsStore = false)
		{
			$result = false;
			$origStore = $store;

			// Get the outbox and sent mail entryid, ignore the given $store, use the default store for submitting messages
			$store = $GLOBALS["mapisession"]->getDefaultMessageStore();
			$storeprops = mapi_getprops($store, array(PR_IPM_OUTBOX_ENTRYID, PR_IPM_SENTMAIL_ENTRYID));

			if(isset($storeprops[PR_IPM_OUTBOX_ENTRYID])) {
				if(isset($storeprops[PR_IPM_SENTMAIL_ENTRYID])) {
					$props[PR_SENTMAIL_ENTRYID] = $storeprops[PR_IPM_SENTMAIL_ENTRYID];
				}
				
				// Set reply to settings
				$replyTo = $GLOBALS['settings']->get("createmail/reply_to",null);
				if (!empty($replyTo)){
					$props[PR_REPLY_RECIPIENT_NAMES] = $replyTo;
					$replyRecip = array(array(PR_EMAIL_ADDRESS=>$replyTo, PR_ADDRTYPE=>"SMTP"));
					$props[PR_REPLY_RECIPIENT_ENTRIES] = $this->writeFlatEntryList($replyRecip);
				}

				if($origStore !== $store) {
					// set properties for "on behalf of" mails
					$origStoreProps = mapi_getprops($origStore, array(PR_MAILBOX_OWNER_ENTRYID));

					// get email address of user from entryid
					$ab = $GLOBALS['mapisession']->getAddressbook();
					$abitem = mapi_ab_openentry($ab, $origStoreProps[PR_MAILBOX_OWNER_ENTRYID]);
					$abitemprops = mapi_getprops($abitem, array(PR_DISPLAY_NAME, PR_EMAIL_ADDRESS));

					$props[PR_SENT_REPRESENTING_ENTRYID] = $origStoreProps[PR_MAILBOX_OWNER_ENTRYID];
					$props[PR_SENT_REPRESENTING_NAME] = $abitemprops[PR_DISPLAY_NAME];
					$props[PR_SENT_REPRESENTING_EMAIL_ADDRESS] = $abitemprops[PR_EMAIL_ADDRESS];
					$props[PR_SENT_REPRESENTING_ADDRTYPE] = "ZARAFA";

					$abitem = mapi_ab_openentry($ab, $GLOBALS["mapisession"]->getUserEntryID());
					$abitemprops = mapi_getprops($abitem, array(PR_DISPLAY_NAME, PR_EMAIL_ADDRESS));

					$props[PR_SENDER_ENTRYID] = $GLOBALS["mapisession"]->getUserEntryID();
					$props[PR_SENDER_NAME] = $abitemprops[PR_DISPLAY_NAME];
					$props[PR_SENDER_EMAIL_ADDRESS] = $abitemprops[PR_EMAIL_ADDRESS];
					$props[PR_SENDER_ADDRTYPE] = "ZARAFA";

					// FIXME: currently message is deleted from original store and new message is created
					// in current user's store, but message should be moved

					// delete message from it's original location
					$folder = mapi_msgstore_openentry($origStore, $props[PR_PARENT_ENTRYID]);
					mapi_folder_deletemessages($folder, array($props[PR_ENTRYID]), DELETE_HARD_DELETE);

					// remove pr_entryid property to create new message in default message store of logged member
					if(isset($props[PR_ENTRYID])) {
						unset($props[PR_ENTRYID]);
					}
				}else{
					// When the message is in your own store, just move it to your outbox. We move it manually so we know the new entryid after it has been moved.
					$outbox = mapi_msgstore_openentry($store, $storeprops[PR_IPM_OUTBOX_ENTRYID]);
					
					// Open the old and the new message
					$oldentryid = $props[PR_ENTRYID];
					$newmessage = mapi_folder_createmessage($outbox);
					$message = mapi_msgstore_openentry($store, $oldentryid);

					// Copy the entire message
					mapi_copyto($message, array(), array(), $newmessage);
					mapi_savechanges($newmessage);
					
					// Remember the new entryid
					$newprops = mapi_getprops($newmessage, array(PR_ENTRYID));
					$props[PR_ENTRYID] = $newprops[PR_ENTRYID]; 
					
					// Delete the old message
					mapi_folder_deletemessages($outbox, array($oldentryid));
				}

				// Save the new message properties
				$message = $this->saveMessage($store, $storeprops[PR_IPM_OUTBOX_ENTRYID], $props, $recipients, $dialog_attachments, $messageProps, $copyAttachments, $copyAttachmentsStore);
				
				if($message) {
					// Submit the message (send)
					mapi_message_submitmessage($message);
					$tmp_props = mapi_getprops($message, array(PR_PARENT_ENTRYID));
					$messageProps[PR_PARENT_ENTRYID] = $tmp_props[PR_PARENT_ENTRYID];
					$result = true;

					$this->addEmailsToRecipientHistory($recipients);
				}
			}
			
			return $result;
		}
		
		/**
		 * Delete messages
		 *
		 * This function does what is needed when a user presses 'delete' on a MAPI message. This means that:
		 *
		 * - Items in the own store are moved to the wastebasket
		 * - Items in the wastebasket are deleted
		 * - Items in other users stores are moved to our own wastebasket
		 * - Items in the public store are deleted
		 *
		 * @param mapistore $store MAPI Message Store Object
		 * @param string $parententryid parent entryid of the messages to be deleted
		 * @param array $entryids a list of entryids which will be deleted
		 * @return boolean true if action succeeded, false if not
		 */
		function deleteMessages($store, $parententryid, $entryids)
		{
			$result = false;
			if(!is_array($entryids)) {
				$entryids = array($entryids);
			}

			$folder = mapi_msgstore_openentry($store, $parententryid);

			$msgprops = mapi_getprops($store, array(PR_IPM_WASTEBASKET_ENTRYID, PR_MDB_PROVIDER));
			
			switch($msgprops[PR_MDB_PROVIDER]){
				case ZARAFA_STORE_DELEGATE_GUID:
					// with a store from an other user we need our own waste basket...
					if(isset($msgprops[PR_IPM_WASTEBASKET_ENTRYID]) && $msgprops[PR_IPM_WASTEBASKET_ENTRYID] == $parententryid) {
						// except when it is the waste basket itself
						$result = mapi_folder_deletemessages($folder, $entryids);
					}else{
						$defaultstore = $GLOBALS["mapisession"]->getDefaultMessageStore();
						$msgprops = mapi_getprops($defaultstore, array(PR_IPM_WASTEBASKET_ENTRYID, PR_MDB_PROVIDER));

						if(isset($msgprops[PR_IPM_WASTEBASKET_ENTRYID]) && $msgprops[PR_IPM_WASTEBASKET_ENTRYID] != $parententryid) {
							$result = $this->copyMessages($store, $parententryid, $defaultstore, $msgprops[PR_IPM_WASTEBASKET_ENTRYID], $entryids, true);
							if (!$result){ // if moving fails, try normal delete
								$result = mapi_folder_deletemessages($folder, $entryids);
							}
						}else{
							$result = mapi_folder_deletemessages($folder, $entryids);
						}
					}
					break;

				case ZARAFA_STORE_PUBLIC_GUID:
					// always delete in public store
					$result = mapi_folder_deletemessages($folder, $entryids);
					break;

				case ZARAFA_SERVICE_GUID:
					// delete message when in your own waste basket, else move it to the waste basket
					if(isset($msgprops[PR_IPM_WASTEBASKET_ENTRYID]) && $msgprops[PR_IPM_WASTEBASKET_ENTRYID] == $parententryid) {
						$result = mapi_folder_deletemessages($folder, $entryids);
					}else{
						$result = $this->copyMessages($store, $parententryid, $store, $msgprops[PR_IPM_WASTEBASKET_ENTRYID], $entryids, true);
						if (!$result){ // if moving fails, try normal delete
							$result = mapi_folder_deletemessages($folder, $entryids);
						}
					}
					break;
			}

			return $result;
		}
		
		/**
		 * Copy or move messages
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $parententryid parent entryid of the messages		 
		 * @param string $destentryid destination folder		 
		 * @param array $entryids a list of entryids which will be copied or moved
		 * @param boolean $moveMessages true - move messages, false - copy messages
		 * @return boolean true if action succeeded, false if not		  
		 * @todo Duplicate code here which is easily fixed
		 */
		function copyMessages($store, $parententryid, $destStore, $destentryid, $entryids, $moveMessages)
		{
			$result = false;
			$sourcefolder = mapi_msgstore_openentry($store, $parententryid); 
			$destfolder = mapi_msgstore_openentry($destStore, $destentryid); 
			
			if(!is_array($entryids)) {
				$entryids = array($entryids);
			}
			
			if($moveMessages) {
				mapi_folder_copymessages($sourcefolder, $entryids, $destfolder, MESSAGE_MOVE);
				$result = (mapi_last_hresult()==NOERROR);
			} else {
				mapi_folder_copymessages($sourcefolder, $entryids, $destfolder);
				$result = (mapi_last_hresult()==NOERROR);
			}
			
			return $result;
		}
		
		/**
		 * Set message read flag
		 *
		 * @param object $store MAPI Message Store Object
		 * @param string $entryid entryid of the message		 
		 * @param array $flags Array of options, may contain "unread" and "noreceipt". The absence of these flags indicate the opposite ("read" and "sendreceipt").		 
		 * @param array $messageProps reference to an array which will be filled with PR_ENTRYID, PR_STORE_ENTRYID and PR_PARENT_ENTRYID of the message
		 * @return boolean true if action succeeded, false if not
		 */
		function setMessageFlag($store, $entryid, $flags, &$messageProps)
		{
			$result = false;
			$message = $this->openMessage($store, $entryid);

			if($message) {
				$flags = explode(",",$flags);
				$flag = 0; // set read and sent receipt if required
				foreach($flags as $value){
					switch($value){
						case "unread":
							$flag |= CLEAR_READ_FLAG;
							break;
						case "noreceipt":
							$flag |= SUPPRESS_RECEIPT;
							break;
					}
				}
				$result = mapi_message_setreadflag($message, $flag);

				$messageProps = mapi_getprops($message, array(PR_ENTRYID, PR_PARENT_ENTRYID, PR_STORE_ENTRYID, PR_MESSAGE_FLAGS));
			} 

			return $result;
		}
		
		/**
		 * Create a unique folder name based on a provided new folder name
		 *
		 * checkFolderNameConflict() checks if a folder name conflict is caused by the given $foldername.
		 * This function is used for copying of moving a folder to another folder. It returns
		 * a unique foldername.		 		  
		 *
		 * @access private
		 * @param object $store MAPI Message Store Object
		 * @param object $folder MAPI Folder Object
		 * @param string $foldername the folder name		 
		 * @return string correct foldername		  
		 */
		function checkFolderNameConflict($store, $folder, $foldername)
		{
			$folderNames = array();
			
			$hierarchyTable = mapi_folder_gethierarchytable($folder);
			mapi_table_sort($hierarchyTable, array(PR_DISPLAY_NAME => TABLE_SORT_ASCEND)); 
			
			$subfolders = mapi_table_queryallrows($hierarchyTable, array(PR_ENTRYID));

			if (is_array($subfolders)) {
				foreach($subfolders as $subfolder)
				{
					$folderObject = mapi_msgstore_openentry($store, $subfolder[PR_ENTRYID]); 
					$folderProps = mapi_folder_getprops($folderObject, array(PR_DISPLAY_NAME));
					
					array_push($folderNames, strtolower($folderProps[PR_DISPLAY_NAME]));
				} 
			}

			if(array_search(strtolower($foldername), $folderNames) !== false) {
				$i = 1;
				
				while(array_search((strtolower($foldername) . $i), $folderNames) !== false)
				{
					$i++;
				}
				
				$foldername .= $i;
			}

			return $foldername;
		}
	
		/**
		 * Set the recipients of a MAPI message
		 *
		 * @access private
		 * @param object $message MAPI Message Object
		 * @param array $recipients XML array structure of recipients
		 */
		function setRecipients($message, $recipients)
		{
			$recipients = $this->createRecipientList($recipients);
			
			$recipientTable = mapi_message_getrecipienttable($message);
			$recipientRows = mapi_table_queryallrows($recipientTable, array(PR_ROWID));
			foreach($recipientRows as $recipient)
			{
				mapi_message_modifyrecipients($message, MODRECIP_REMOVE, array($recipient));
			} 
			
			if(count($recipients) > 0){
				mapi_message_modifyrecipients($message, MODRECIP_ADD, $recipients);
			}
		}
		
		/**
		 * Delete properties in a message
		 *
		 * @todo Why does this function call savechange while most other functions
		 *       here do not? (for example setRecipients)
		 * @param object $store MAPI Message Store Object		 
		 * @param string $entryid Entryid of the message in which to delete the properties
		 * @param array $props array of property tags which to be deleted
		 */
		function deleteProps($store, $entryid, $props)
		{
			$message = $this->openMessage($store, $entryid);
			mapi_deleteprops($message, $props);
			mapi_savechanges($message);
		}
		
		/**
		 * Set attachments in a MAPI message
		 *
		 * This function reads any attachments that have been previously uploaded and copies them into
         * the passed MAPI message resource. For a description of the dialog_attachments variable and
         * generally how attachments work when uploading, see Operations::saveMessage()
		 *
		 * @see Operations::saveMessage()
		 * @param object $message MAPI Message Object
		 * @param string $dialog_attachments unique dialog_attachments, which is set in the $_SESSION variable
		 * @param boolean $copyfromEntryid if set, copy the attachments from this entryid in addition to the uploaded attachments
		 */
		function setAttachments($store, $message, $dialog_attachments, $copyfromEntryid=false)
		{
			// check if we need to copy attachments from a forwared message first
			if ($copyfromEntryid != false){
				$copyfrom = mapi_msgstore_openentry($store, $copyfromEntryid);

				$attachmentTable = mapi_message_getattachmenttable($copyfrom);
				if($attachmentTable) {
					$attachments = mapi_table_queryallrows($attachmentTable, array(PR_ATTACH_NUM, PR_ATTACH_SIZE, PR_ATTACH_LONG_FILENAME, PR_ATTACHMENT_HIDDEN, PR_DISPLAY_NAME, PR_ATTACH_METHOD));

					foreach($attachments as $props){
						// check if this attachment is "deleted"
						if(isset($_SESSION["deleteattachment"]) && isset($_SESSION["deleteattachment"][$dialog_attachments])) {
							if (in_array($props[PR_ATTACH_NUM],$_SESSION["deleteattachment"][$dialog_attachments])) {
								// skip attachment
								continue;
							}
						}

						$old = mapi_message_openattach($copyfrom, (int) $props[PR_ATTACH_NUM]);
						$new = mapi_message_createattach($message);
						mapi_copyto($old, array(), array(), $new, 0);
						mapi_savechanges($new);
					}
				}
			} else {
	
				// Check if attachments should be deleted. This is set in the "upload_attachment.php" file
				if(isset($_SESSION["deleteattachment"]) && isset($_SESSION["deleteattachment"][$dialog_attachments])) {
					foreach($_SESSION["deleteattachment"][$dialog_attachments] as $attach_num) {
						mapi_message_deleteattach($message, (int) $attach_num);
					}	
				}
			}

			// Check if ["files"] and ["files"][$dialog_attachments] is set
			if(isset($_SESSION["files"])) {
				if(isset($_SESSION["files"][$dialog_attachments])) {
					// Loop through teh uploaded attachments
					foreach ($_SESSION["files"][$dialog_attachments] as $tmpname => $fileinfo)
					{
						if (is_file(TMP_PATH."/".$tmpname)) {
							// Construct file name
							$filenameshort = "";
							if (strpos($fileinfo["name"], ".")) {
								$filenameparts = explode(".", $fileinfo['name']);
								for ($i = 0;$i < count($filenameparts);$i++)
								{
									if ($i == count($filenameparts)-1) {
										$filenameshort = substr($filenameshort, 0, 8) . "." . substr($filenameparts[$i], 0, 3);
									} else {
										$filenameshort .= $filenameparts[0];
									} 
								} 
							} else {
								$filenameshort = substr($fileinfo["name"], 0, 8);
							} 
							
							// Set attachment properties
							$props = Array(PR_ATTACH_LONG_FILENAME => $fileinfo["name"],
										   PR_DISPLAY_NAME => $fileinfo["name"],
										   PR_ATTACH_FILENAME => $filenameshort,
										   PR_ATTACH_METHOD => ATTACH_BY_VALUE,
										   PR_ATTACH_DATA_BIN => "");

							// Create attachment and set props
							$attachment = mapi_message_createattach($message);
							mapi_setprops($attachment, $props);
							
							// Stream the file to the PR_ATTACH_DATA_BIN property 
							$stream = mapi_openpropertytostream($attachment, PR_ATTACH_DATA_BIN, MAPI_CREATE | MAPI_MODIFY);
							$handle = fopen(TMP_PATH."/".$tmpname, "r");
							while (!feof($handle))
							{
								$contents = fread($handle, BLOCK_SIZE);
								mapi_stream_write($stream, $contents);
							}
							
							// Commit the stream and save changes
							mapi_stream_commit($stream);
							mapi_savechanges($attachment);
							fclose($handle);
							unlink(TMP_PATH."/".$tmpname);
						} 
					} 
					
					// Delete all the files in the $_SESSION["files"][$dialog_attachments].
					unset($_SESSION["files"][$dialog_attachments]);
				}
			}
		}
		
		/**
		 * Create a MAPI recipient list from an XML array structure
		 *
		 * This functions is used for setting the recipient table of a message.		 
		 * @param array $recipientList a list of recipients as XML array structure
		 * @return array list of recipients with the correct MAPI properties ready for mapi_message_modifyrecipients()
		 */
		function createRecipientList($recipientList)
		{
			$recipients = array();
			
			foreach($recipientList as $recipientItem)
			{
				// TODO: check this code, this looks like some sort of resolving names feature...
				if(!empty($recipientItem["address"]) && !empty($recipientItem["name"]) && !eregi("^[_a-z0-9-]+(\.[_a-z0-9-]+)*@[a-z0-9-]+(\.[a-z0-9-]+)*(\.[a-z]{2,3})$", $recipientItem["address"])) {
					$users = mapi_zarafa_getuserlist($GLOBALS["mapisession"]->getDefaultMessageStore());
					foreach($users as $user)
					{
						if(strpos(strtolower($user["username"]), $recipientItem["name"]) !== false ||
            			   strpos(strtolower($user["fullname"]), $recipientItem["name"]) !== false ||
            			   strpos(strtolower($user["emailaddress"]), $recipientItem["name"]) !== false) {
						   $recipientItem["name"] = $user["fullname"];
						   $recipientItem["address"] = $user["emailaddress"];
            			}
					}
				}

				if(!empty($recipientItem["address"]) && !empty($recipientItem["name"])) {
					$recipient = array();
					$recipient[PR_DISPLAY_NAME] = utf8_to_windows1252($recipientItem["name"]);
					$recipient[PR_EMAIL_ADDRESS] = utf8_to_windows1252($recipientItem["address"]);
					$recipient[PR_SMTP_ADDRESS] = utf8_to_windows1252($recipientItem["address"]);
					$recipient[PR_SEARCH_KEY] = utf8_to_windows1252($recipientItem["address"]);
					$recipient[PR_ADDRTYPE] = "SMTP";
					
					if (isset($recipientItem["recipient_status_num"])){
						$recipient[PR_RECIPIENT_TRACKSTATUS] = $recipientItem["recipient_status_num"];
					}
					
					switch(strtolower($recipientItem["type"]))
					{
						case "mapi_to":
							$recipient[PR_RECIPIENT_TYPE] = MAPI_TO;
							break;
						case "mapi_cc":
							$recipient[PR_RECIPIENT_TYPE] = MAPI_CC;
							break;
						case "mapi_bcc":
							$recipient[PR_RECIPIENT_TYPE] = MAPI_BCC;
							break;
					}
					
					$recipient[PR_OBJECT_TYPE] = MAPI_MAILUSER;

					if(isset($recipientItem["recipient_flags"])){
						$recipient[PR_RECIPIENT_FLAGS] = $recipientItem["recipient_flags"];
					}else{
						$recipient[PR_RECIPIENT_FLAGS] = 1;
					}

					if(isset($recipientItem["proposednewtime"]) && isset($recipientItem["proposenewstarttime"]) && isset($recipientItem["proposenewendtime"])){
						$recipient[PR_PROPOSEDNEWTIME] = true;
						$recipient[PR_PROPOSENEWTIME_START] = intval($recipientItem["proposenewstarttime"]);
						$recipient[PR_PROPOSENEWTIME_END] = intval($recipientItem["proposenewendtime"]);
					}

					$recipient[PR_ENTRYID] = mapi_createoneoff($recipient[PR_DISPLAY_NAME], $recipient[PR_ADDRTYPE], $recipient[PR_EMAIL_ADDRESS]);
					array_push($recipients, $recipient);
				}
			}
			
			return $recipients;
		}
		
		/**
		 * Create a reply-to property value
		 *
		 * @param array $recipients list of recipients
		 * @return string the value for the PR_REPLY_RECIPIENT_ENTRIES property
		 * @todo remove it, it is unused
		 */
		function createReplyRecipientEntry($replyTo)
		{
			// TODO: implement. Is used for the setting: Reply To
			// for now, use the old function:
			return $this->createReplyRecipientEntry_old($replyTo);
		}
		
		/**
		 * Parse reply-to value from PR_REPLY_RECIPIENT_ENTRIES property
		 * @param string $flatEntryList the PR_REPLY_RECIPIENT_ENTRIES value
		 * @return array list of recipients in XML array structure
		 */
		function readReplyRecipientEntry($flatEntryList)
		{
			// Unpack number of entries, the byte count and the entries
			$unpacked = unpack("V1cEntries/V1cbEntries/a*abEntries", $flatEntryList);
			
			$abEntries = Array();
			$stream = $unpacked['abEntries'];
			$pos = 8;
			
			for ($i=0; $i<$unpacked['cEntries']; $i++)
			{
				$findEntry = unpack("a".$pos."before/V1cb/a*after", $flatEntryList);
				// Go to after the unsigned int
				$pos += 4;
				$entry = unpack("a".$pos."before/a".$findEntry['cb']."abEntry/a*after", $flatEntryList);
				// Move to after the entry
				$pos += $findEntry['cb'];
				// Move to next 4-byte boundary
				$pos += $pos%4;
				// One one-off entry id
				$abEntries[] = $entry['abEntry'];
			}
			
			$recipients = Array();
			foreach ($abEntries as $abEntry)
			{
				// Unpack the one-off entry identifier
				$findID = unpack("V1version/a16mapiuid/v1flags/v1abFlags/a*abEntry", $abEntry);
				$tempArray = Array();
				// Split the entry in its three fields
				
				// Workaround (if Unicode then strip \0's)
				if (($findID['abFlags'] & 0x8000)) {
					$idParts = explode("\0\0", $findID['abEntry']);
					foreach ($idParts as $idPart)
					{
						// Remove null characters from the field contents
						$tempArray[] = str_replace("\x00", "", $idPart);
					}
				} else {
					// Not Unicode. Just split by \0.
					$tempArray = explode("\0", $findID['abEntry']);
				}
				
				// Put data in recipient array
				$recipients[] = Array("display_name" => windows1252_to_utf8($tempArray[0]),
				    "email_address" => windows1252_to_utf8($tempArray[2]));
			}
			
			return $recipients;
		}
		
		/**
		 * Build full-page HTML from the FCK Editor fragment
		 *
		 * This function basically takes the generated HTML from FCK editor and embeds it in
		 * a standonline HTML page (including header and CSS) to form.
		 *
		 * @param string $fck_html This is the HTML created by the FCK Editor
		 * @param string $title  Optional, this string is placed in the <title>
		 * @return string full HTML message		 
		 *
		 */
		function fckEditor2html($fck_html, $title = "Zarafa WebAccess"){
			$html = "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">"
					."<html>\n"
					."<head>\n"
					."  <meta name=\"Generator\" content=\"Zarafa WebAccess v".phpversion("mapi")."\">\n"
					."  <meta http-equiv=\"Content-Type\" content=\"text/html; charset=Windows-1252\">\n"
					."  <title>".htmlspecialchars($title)."</title>\n"
					."  <style type=\"text/css\">\n";
			
			// embed CSS file in e-mail
			$css_file = FCKEDITOR_PATH."/fck_editorarea.css";
			$html .= file_get_contents($css_file);
			
			$html .="  </style>\n"
					."</head>\n"
					."<body>\n"
					. $fck_html . "\n"
					."</body>\n"
					."</html>";
			
			return $html;
		}

		function calcFolderMessageSize($folder, $recursive=false)
		{
			$size = 0;
			if ($recursive){
				trigger_error("Recursive calculation of folder messageSize not yet implemented", E_USER_NOTICE);
			}

			$table = mapi_folder_getcontentstable($folder);
				
			$item_count = mapi_table_getrowcount($table);
			for($i = 0; $i<$item_count; $i+=50){
				$messages = mapi_table_queryrows($table, array(PR_MESSAGE_SIZE), $i, 50);
				foreach($messages as $message){
					if (isset($message[PR_MESSAGE_SIZE])){
						$size += $message[PR_MESSAGE_SIZE];
					}
				}
			}
			return $size;
		}


////////////////////////////////////////////
// FIXME: OLD FUNCTIONS!!!                //
////////////////////////////////////////////
	
	
	
		/**
		* Parse string of recipients into individual recipients
		*
		* @author Robbert Monna
		* @param string $inputText The string from which to create recipient array (eg. 'john doe <john@doe.org>; jane@domain.com')
		* @param integer $rType An optional PR_RECIPIENT_TYPE
		* @return array Array of arrays (table) with message recipients, false if no valid recipients found
		*/
		function createRecipientList_old($inputText, $rType="")
		{
			$parsedRcpts = Array();
			// Data is separated by , or ; ; so we split by this first.
			$splitArray = preg_split("/[;]/", $inputText);
			$errors = 0;
			for ($i=0; $i<count($splitArray); $i++)
			{
				if($splitArray[$i])
				{
					$rcptMatches = Array();
					// Check format of the address
					if(preg_match('/([^<]*)<([^>]*)>/', $splitArray[$i], $rcptMatches)==0)
					{
						// Address only
						$parsedRcpts[$i][PR_DISPLAY_NAME] = trim($splitArray[$i]);
						$parsedRcpts[$i][PR_EMAIL_ADDRESS] = trim($splitArray[$i]);
					} else
					{
						// Address with name (a <b@c.nl> format)
						if (trim($rcptMatches[1]))
						{
							// Name available, so use it as name
							$parsedRcpts[$i][PR_DISPLAY_NAME] = trim($rcptMatches[1]);
						} else
						{
							// Name not available, use mail address as display name
							$parsedRcpts[$i][PR_DISPLAY_NAME] = trim($rcptMatches[2]);
						}
						$parsedRcpts[$i][PR_EMAIL_ADDRESS] = trim($rcptMatches[2]);
					}
					// Other properties, we only support SMTP here
					$parsedRcpts[$i][PR_ADDRTYPE] = "SMTP";
					if ($rType!="")
						$parsedRcpts[$i][PR_RECIPIENT_TYPE] = $rType;
					$parsedRcpts[$i][PR_OBJECT_TYPE] = MAPI_MAILUSER;
		
					// One-Off entry identifier; needed for Zarafa
					$parsedRcpts[$i][PR_ENTRYID] = mapi_createoneoff($parsedRcpts[$i][PR_DISPLAY_NAME], $parsedRcpts[$i][PR_ADDRTYPE], $parsedRcpts[$i][PR_EMAIL_ADDRESS]);
				}
			}
			if (count($parsedRcpts)>0&&$errors==0)
				return $parsedRcpts;
			else
				return false;
		}
		
		/**
		* Create a flat entrylist (used for PR_REPLY_RECIPIENT_ENTRIES) from a list of recipients
		*
		* These flatentrylists are used in PR_REPLY_RECIPIENT_ENTRIES, remember to
		* keep this property synchronized with PR_REPLY_RECIPIENT_NAMES.
		*
		* @author Robbert Monna
		* @param String $recipientArray The array with recipients to convert
		* @return boolean Returns the resulting flatentrylist
		*/
		function writeFlatEntryList($recipientArray)
		{
			$oneOffs = Array();
			foreach ($recipientArray as $recipient)
			{
				// Add display name if it doesn't exist
				if (!array_key_exists(PR_DISPLAY_NAME, $recipient)||empty($recipient[PR_DISPLAY_NAME]))
					$recipient[PR_DISPLAY_NAME] = $recipient[PR_EMAIL_ADDRESS];
				$oneOffs[] = mapi_createoneoff($recipient[PR_DISPLAY_NAME], $recipient[PR_ADDRTYPE], $recipient[PR_EMAIL_ADDRESS]);
			}
		
			// Construct string from array with (padded) One-Off entry identifiers
			//
			// Remember, if you want to take the createOneOff part above out: that code
			// produces a padded OneOff and we add the right amount of null characters
			// below.
			//
			// So below is a wrong method for composing a flatentrylist from oneoffs and
			// above is a wrong method form composing a oneoff.
			$flatEntryString = "";
			for ($i=0; $i<count($oneOffs); $i++)
			{
				$flatEntryString .= pack("Va*", strlen($oneOffs[$i]), $oneOffs[$i]);
				// Fill to 4-byte boundary
				$rest = strlen($oneOffs[$i])%4;
				for ($j=0;$j<$rest;$j++)
					$flatEntryString .= "\0";
			}
			// Pack the string with the number of flatentries and the stringlength
			return pack("V2a*", count($oneOffs), strlen($flatEntryString), $flatEntryString);
		}
		
		/**
		* Get a text body for a Non-Delivery report
		*
		* This function reads the necessary properties from the passed message and constructs
		* a user-readable NDR message from those properties
		*
		* @param mapimessage $message The NDR message to read the information from
		* @return string NDR body message as plaintext message.
		*/
		function getNDRbody($message)
		{
			$message_props  = mapi_getprops($message, array(PR_ORIGINAL_SUBJECT,PR_ORIGINAL_SUBMIT_TIME));	
			$body = _("Your message did not reach some or all of the intended recipients")."\n\n";
			$body .= "\t"._("Subject").": ".$message_props[PR_ORIGINAL_SUBJECT]."\n";
			$body .= "\t"._("Sent").":    ".strftime("%a %x %X",$message_props[PR_ORIGINAL_SUBMIT_TIME])."\n\n";
			$body .= _("The following recipient(s) could not be reached").":\n\n";

			$recipienttable = mapi_message_getrecipienttable($message);
			$recipientrows = mapi_table_queryallrows($recipienttable,array(PR_DISPLAY_NAME,PR_REPORT_TIME,PR_REPORT_TEXT));
			foreach ($recipientrows as $recipient){
				$body .= "\t".$recipient[PR_DISPLAY_NAME]." on ".strftime("%a %x %X",$recipient[PR_REPORT_TIME])."\n";
				$body .= "\t\t".$recipient[PR_REPORT_TEXT]."\n";
			}
			
			// Converting the body to UTF-8 before it leaves
			$body = w2u($body);

			// Bon voyage!
			return $body;
		}

		/**
		* Detect plaintext body type of message
		*
		* @param mapimessage $message MAPI message resource to check
		* @return boolean TRUE if the message is a plaintext message, FALSE if otherwise
		*/
		function isPlainText($message)
		{
			$rtf = mapi_message_openproperty($message, PR_RTF_COMPRESSED);
			
			if (!$rtf)
				return true; // no RTF is found, so we use plain text

			// get first line of the RTF (removing all other lines after opening/decompressing)
			$rtf = preg_replace("/(\n.*)/m","",mapi_decompressrtf($rtf));

			// check if "\fromtext" exists, if so, it was plain text
			return strpos($rtf,"\\fromtext") !== false;
		}

		/**
		* Parse email recipient list and add all e-mail addresses to the recipient history
		*
		* The recipient history is used for auto-suggestion when writing e-mails. This function
		* opens the recipient history property (PR_EC_RECIPIENT_HISTORY) and updates or appends
		* it with the passed email addresses.
		*
		* @param emailAddresses XML array structure with recipients
		*/
		function addEmailsToRecipientHistory($emailAddresses){
			if(is_array($emailAddresses) && count($emailAddresses) > 0){
				// Retrieve the recipient history
				$stream = mapi_openpropertytostream($GLOBALS["mapisession"]->getDefaultMessageStore(), PR_EC_RECIPIENT_HISTORY);
				$hresult = mapi_last_hresult();

				if($hresult == NOERROR){
					$stat = mapi_stream_stat($stream);
					mapi_stream_seek($stream, 0, STREAM_SEEK_SET);
					$xmlstring = '';
					for($i=0;$i<$stat['cb'];$i+=1024){
						$xmlstring .= mapi_stream_read($stream, 1024);
					}
					$xml = new XMLParser();
					$recipient_history = $xml->getData($xmlstring);
				}

				/**
				 * Check to make sure the recipient history is returned in array format 
				 * and not a PEAR error object.
				 */
				if(!isset($recipient_history) || !is_array($recipient_history) || !isset($recipient_history['recipients']['recipient'])){
					$recipient_history = Array(
						'recipients' => Array(
							'recipient' => Array()
						)
					);
				}else{
					/**
					 * When only one recipient is found in the XML it is saved as a single dimensional array
					 * in $recipient_history['recipients']['recipient'][RECIPDATA]. When multiple recipients
					 * are found, a multi-dimensional array is used in the format 
					 * $recipient_history['recipients']['recipient'][0][RECIPDATA].
					 */
					if($recipient_history['recipients']['recipient']){
						if(!is_numeric(key($recipient_history['recipients']['recipient']))){
							$recipient_history['recipients']['recipient'] = Array(
								0 => $recipient_history['recipients']['recipient']
							);
						}
					}
				}

				$l_aNewHistoryItems = Array();
				// Loop through all new recipients
				for($i=0;$i<count($emailAddresses);$i++){
					// Convert UTF8 input to windows1252
					$emailAddresses[$i]['name'] = u2w($emailAddresses[$i]['name']);
					$emailAddresses[$i]['address'] = u2w($emailAddresses[$i]['address']);

					$l_bFoundInHistory = false;
					// Loop through all the recipients in history
					if(is_array($recipient_history['recipients']['recipient'])) {
                        for($j=0;$j<count($recipient_history['recipients']['recipient']);$j++){
                            // Email address already found in history
                            if($emailAddresses[$i]['address'] == $recipient_history['recipients']['recipient'][$j]['email']){
                                $l_bFoundInHistory = true;
                                // Check if a name has been supplied.
                                if(strlen(trim($emailAddresses[$i]['name'])) > 0){
                                    // Check if the name is not the same as the email address
                                    if(trim($emailAddresses[$i]['name']) != $emailAddresses[$i]['address']){
                                        $recipient_history['recipients']['recipient'][$j]['name'] = trim($emailAddresses[$i]['name']);
                                    // Check if the recipient history has no name for this email
                                    }elseif(strlen(trim($recipient_history['recipients']['recipient'][$j]['name'])) == 0){
                                        $recipient_history['recipients']['recipient'][$j]['name'] = trim($emailAddresses[$i]['name']);
                                    }
                                }
                                $recipient_history['recipients']['recipient'][$j]['count']++;
                                $recipient_history['recipients']['recipient'][$j]['last_used'] = mktime();
                                break;
                            }
                        }
                    }
					if(!$l_bFoundInHistory && !isset($l_aNewHistoryItems[$emailAddresses[$i]['address']])){
						$l_aNewHistoryItems[$emailAddresses[$i]['address']] = Array(
							'name' => $emailAddresses[$i]['name'],
							'email' => $emailAddresses[$i]['address'],
							'count' => 1,
							'last_used' => mktime()
							);
					}
				}
				if(count($l_aNewHistoryItems) > 0){
					foreach($l_aNewHistoryItems as $l_aValue){
						$recipient_history['recipients']['recipient'][] = $l_aValue;
					}
				}

				$xml = new XMLBuilder();
				$l_sNewRecipientHistoryXML = $xml->build($recipient_history);

				$stream = mapi_openpropertytostream($GLOBALS["mapisession"]->getDefaultMessageStore(), PR_EC_RECIPIENT_HISTORY, MAPI_CREATE | MAPI_MODIFY);
				mapi_stream_write($stream, $l_sNewRecipientHistoryXML);
				mapi_stream_commit($stream);
				mapi_savechanges($GLOBALS["mapisession"]->getDefaultMessageStore());
			}
		}

		/**
		* Extract all email addresses from a list of recipients
		*
		* @param string $p_sRecipients String containing e-mail addresses, as typed by user (eg. '<john doe> john@doe.org; jane@doe.org')
		* @return array Array of e-mail address parts (eg. 'john@doe.org', 'jane@doe.org')
		*/
		function extractEmailAddresses($p_sRecipients){
			$l_aRecipients = explode(';', $p_sRecipients);
			$l_aReturn = Array();
			for($i=0;$i<count($l_aRecipients);$i++){
				$l_aRecipients[$i] = trim($l_aRecipients[$i]);
				$l_sRegex = '/[_a-zA-Z0-9-]+(\.[_a-zA-Z0-9-]+)*@[a-zA-Z0-9-]+(\.[a-zA-Z0-9-]+)*(\.[a-zA-Z]+)/';
				preg_match($l_sRegex, $l_aRecipients[$i], $l_aMatches);
				$l_aReturn[] = $l_aMatches[0];
			}
			return $l_aReturn;
		}
		
		/**
		* Get the SMTP e-mail of an addressbook entry
		*
		* @param string $entryid Addressbook entryid of object
		* @return string SMTP e-mail address of that entry or FALSE on error
		*/
		function getEmailAddressFromEntryID($entryid) {
		    $mailuser = mapi_ab_openentry($GLOBALS["mapisession"]->getAddressbook(), $entryid);
		    if(!$mailuser)
		        return false;
		        
            $abprops = mapi_getprops($mailuser, array(PR_SMTP_ADDRESS, PR_EMAIL_ADDRESS));

            if(isset($abprops[PR_SMTP_ADDRESS]))
                return $abprops[PR_SMTP_ADDRESS];
            else if(isset($abprops[PR_EMAIL_ADDRESS]))
				return $abprops[PR_EMAIL_ADDRESS];
			else
                return false;
		}

		/**
		* Get all rules of a store
		*
		* This function opens the rules table for the specified store, and reads
		* all rules with PR_RULE_PROVIDER equal to 'RuleOrganizer'. These are the rules
		* that the user sees when managing rules from Outlook.
		*
		* @param mapistore $store Store in which rules reside
		* @param array $properties Property mappings for rules properties
		* @return array XML array structure of rules from the store
		*/
		function getRules($store, $properties) {
			$inbox = mapi_msgstore_getreceivefolder($store);
			
			if(!$inbox)
				return false;
				
			$modifyrulestable = mapi_folder_openmodifytable($inbox);
			$rulestable = mapi_rules_gettable($modifyrulestable);

			mapi_table_restrict($rulestable, array(RES_CONTENT,
												array(
													FUZZYLEVEL	=>	FL_FULLSTRING | FL_IGNORECASE,
													ULPROPTAG	=>	PR_RULE_PROVIDER,
													VALUE		=>	array(
														PR_RULE_PROVIDER	=>	"RuleOrganizer"
													)
												)
											)
			);


			mapi_table_sort($rulestable, array(PR_RULE_SEQUENCE => TABLE_SORT_ASCEND));			
			$rows = mapi_table_queryrows($rulestable, $properties, 0, 0x7fffffff);
	
			$xmlrows = array();
		
			foreach($rows as $row) {
				$xmlrow = Conversion::mapMAPI2XML($properties, $row);
				array_push($xmlrows, $xmlrow);
			}
			
			return $xmlrows;
		}

		/**
		* Update rules in a store
		* 
		* This function replaces all rules in a store with the passed new ruleset.
		*
		* @param mapistore $store The store whose rules should be modified
		* @param array $rules The XML array structure with the new rules
		* @param array $properties The property mapping for rules properties
		* @return boolean True on success, false on error
		*/
		function updateRules($store, $rules, $properties)
		{
			if (!is_array($rules))
				return false;

			$inbox = mapi_msgstore_getreceivefolder($store);
			
			if(!$inbox)
				return false;
				
			$modifyrulestable = mapi_folder_openmodifytable($inbox);
			
			// get provider data id's from all rules
			$rulestable = mapi_rules_gettable($modifyrulestable);
			mapi_table_restrict($rulestable, array(RES_CONTENT,
												array(
													FUZZYLEVEL	=>	FL_FULLSTRING | FL_IGNORECASE,
													ULPROPTAG	=>	PR_RULE_PROVIDER,
													VALUE		=>	array(
														PR_RULE_PROVIDER	=>	"RuleOrganizer"
													)
												)
											)
			);
			mapi_table_sort($rulestable, array(PR_RULE_SEQUENCE => TABLE_SORT_ASCEND));			
			$current_rules = mapi_table_queryrows($rulestable, $properties, 0, 0x7fffffff);
			$provider_data = array();
			foreach($current_rules as $row){
				$data = unpack("Vnum/Vid/a*datetime", $row[PR_RULE_PROVIDER_DATA]);
				$provider_data[$row[PR_RULE_ID]] = $data["id"];
			}

			// delete all rules from the rules table
			$deleteRules = array();
			foreach($current_rules as $delRow){
				$deleteRules[] = array(
						"rowflags"	=>	ROW_REMOVE,
						"properties"=>	$delRow
				);
			}
			mapi_rules_modifytable($modifyrulestable, $deleteRules);

			if (!isset($rules[0]))
				$rules = array($rules);

			// find rules to delete
			$deleteRules = array();
			$deletedRowIDs = array();
			foreach($rules as $key=>$rule){
				if (isset($rule["deleted"])){
					$deleteRules[] = $rule;
					$convertedRule = Conversion::mapXML2MAPI($properties, $rule);
					$deletedRowIDs[] = $convertedRule[PR_RULE_ID];
					unset($rules[$key]);
				}
			}

			// add/modify new rules
			foreach($rules as $k=>$rule){
				$rules[$k] = Conversion::mapXML2MAPI($properties, $rule);
			}

			$rows = array();

			$modified = array();

			// add/update rules from the client
			foreach($rules as $rule){

				if (!isset($rule[PR_RULE_ID])){
					$row_flags = ROW_ADD;
					if (is_array($provider_data)){
						$dataId = max($provider_data)+1;
					}else{
						$dataId = 1;
					}
					$provider_data[] = $dataId;
				}else{
					$row_flags = ROW_ADD;
					$dataId = $provider_data[$rule[PR_RULE_ID]];
					$modified[] = $rule[PR_RULE_ID]; 
				}
			
				if (!isset($rule[PR_RULE_ACTIONS])){
					$rule[PR_RULE_ACTIONS] = array(array("action"=>OP_DEFER_ACTION, "dam"=>hex2bin("E0C810000120000100000000000000010000000000000001000000360000000200FFFF00000C004352756C65456C656D656E7490010000010000000000000001000000018064000000010000000000000001000000")));
				}
				if (!isset($rule[PR_RULE_CONDITION])){
					$rule[PR_RULE_CONDITION] = array(RES_EXIST, array(ULPROPTAG=>PR_MESSAGE_CLASS));
				}
				if (!isset($rule[PR_RULE_SEQUENCE])){
					$rule[PR_RULE_SEQUENCE] = 10;
				}

				if (!isset($rule[PR_RULE_STATE])){
					$rule[PR_RULE_STATE] = ST_DISABLED;
				}

				if (!isset($rule[PR_RULE_NAME]) || $rule[PR_RULE_NAME]==""){
					$rule[PR_RULE_NAME] = _("Untitled rule");
				}

				$rule[PR_RULE_LEVEL] = 0;
				$rule[PR_RULE_PROVIDER] = "RuleOrganizer";
				$rule[PR_RULE_PROVIDER_DATA] = pack("VVa*", 1, $dataId, Conversion::UnixTimeToCOleDateTime(time()));

				$rows[] = array(
						"rowflags"	=>	$row_flags,
						"properties"=>	$rule
				);
			}
			// update PR_RULE_PROVIDER_DATA for every other rule
			foreach($current_rules as $rule){
				if (!in_array($rule[PR_RULE_ID], $modified) && !in_array($rule[PR_RULE_ID], $deletedRowIDs)){

					$dataId = $provider_data[$rule[PR_RULE_ID]];
					$rule[PR_RULE_PROVIDER_DATA] = pack("VVa*", 1, $dataId, Conversion::UnixTimeToCOleDateTime(time()));

					$rows[] = array("properties"=>$rule, "rowflags"=> ROW_MODIFY);
				}
			}

			// sort rules on PR_RULE_SEQUENCE to fix the order
			usort($rows, array($this, "cmpRuleSequence"));
			$seq = 10;
			foreach($rows as $k=>$row){
				$rows[$k]["properties"][PR_RULE_SEQUENCE] = $seq;
				$seq++;
				unset($rows[$k][PR_RULE_ID]); // remove RULE_ID because this is the property Outlook does its sort when displaying rules instead of RULE_SEQUENCE!
			}
			
			// sort the rules, so they will get a RULE_ID in the right order when adding the rules
			uasort($rows, create_function('$a,$b','return $a["properties"][PR_RULE_SEQUENCE]>$b["properties"][PR_RULE_SEQUENCE];'));

			mapi_rules_modifytable($modifyrulestable, $rows);
			$result = mapi_last_hresult();

			// delete (outlook) client rules
			$assocTable = mapi_folder_getcontentstable($inbox, MAPI_ASSOCIATED);
			mapi_table_restrict($assocTable, array(RES_CONTENT,
												array(
													FUZZYLEVEL	=>	FL_FULLSTRING | FL_IGNORECASE,
													ULPROPTAG	=>	PR_MESSAGE_CLASS,
													VALUE		=>	array(
														PR_MESSAGE_CLASS	=>	"IPM.RuleOrganizer"
													)
												)
											)
			);
			$messages = mapi_table_queryallrows($assocTable, array(PR_ENTRYID, PR_MESSAGE_CLASS));
			$delete = array();
			for($i=0;$i<count($messages);$i++){
				if ($messages[$i][PR_MESSAGE_CLASS] == "IPM.RuleOrganizer"){ // just to be sure that the restriction worked ;)
					array_push($delete, $messages[$i][PR_ENTRYID]);
				}
			}
			if (count($delete)>0){
				mapi_folder_deletemessages($inbox, $delete);
			}


			return ($result==NOERROR);
		}
		
		/**
		* @access private
		*/
		function cmpRuleSequence($a, $b)
		{
			if ($a["properties"][PR_RULE_SEQUENCE]==$b["properties"][PR_RULE_SEQUENCE])
				return 0;
			return ($a["properties"][PR_RULE_SEQUENCE]<$b["properties"][PR_RULE_SEQUENCE])?-1:1;
		}

		/**
		* Get folder name
		*
		* @param $storeid Store entryid of store in which folder resides
		* @param $folderid Folder entryid of folder
		* @return string Folder name of specified folder
		*/
		function getFolderName($storeid, $folderid) {
			$store = $GLOBALS["mapisession"]->openMessageStore($storeid);
			if(!$store)
				return false;
			
			$folder = mapi_msgstore_openentry($store, $folderid);
			if(!$folder)
				return false;
				
			$folderprops = mapi_getprops($folder, array(PR_DISPLAY_NAME));
			
			return $folderprops[PR_DISPLAY_NAME];
		}

		/**
		* Send a meeting cancellation
		*
		* This function sends a meeting cancellation for the meeting references by the passed entryid. It
		* will send the meeting cancellation and move the item itself to the waste basket.
		*
		* @param mapistore $store The store in which the meeting request resides
		* @param string $entryid Entryid of the appointment for which the cancellation should be sent.
		*/
		function cancelInvitation($store, $entryid, $action = false) {
			$message = $GLOBALS["operations"]->openMessage($store, $entryid);

			$req = new Meetingrequest($store, $message, $GLOBALS["mapisession"]->getSession());

			if(isset($action["exception"]) && $action["exception"] == true) {
				$recurrence = new Recurrence($store, $message);
				$basedate = (int) $action["basedate"];

				if(isset($action["delete"]) && $action["delete"] == true) {
					$recurrence->createException(array(), $basedate, true);
				}

				// if recurrence pattern of meeting request is changed then need to send full update
				// Send the meeting request
				$req->updateMeetingRequest();
				$req->sendMeetingRequest(0);

				// Notify the bus that the message has been deleted
				$messageProps = mapi_getprops($message, array(PR_ENTRYID, PR_STORE_ENTRYID, PR_PARENT_ENTRYID));
				$GLOBALS["bus"]->notify(bin2hex($messageProps[PR_PARENT_ENTRYID]), TABLE_SAVE, $messageProps);
			} else {
				// Send the cancellation
				$req->updateMeetingRequest();
				$req->sendMeetingRequest(true, _("Canceled: "));

				$root = mapi_msgstore_openentry($store);
				$storeprops = mapi_getprops($store, array(PR_IPM_WASTEBASKET_ENTRYID));
				$wastebasket = mapi_msgstore_openentry($store, $storeprops[PR_IPM_WASTEBASKET_ENTRYID]);

				// Move the message to the deleted items
				mapi_folder_copymessages($root, array($entryid), $wastebasket, MESSAGE_MOVE);

				// Notify the bus that the message has been deleted			
				$messageProps = mapi_getprops($message, array(PR_ENTRYID, PR_STORE_ENTRYID, PR_PARENT_ENTRYID));
				$GLOBALS["bus"]->notify(bin2hex($messageProps[PR_PARENT_ENTRYID]), TABLE_DELETE, $messageProps);
			}
		}
		
		/**
		* Remove all appointments for a certain meeting request
		*
		* This function searches the default calendar for all meeting requests for the specified
		* meeting. All those appointments are then removed.
		*
		* @param mapistore $store Mapi store in which the meeting request and the calendar reside
		* @param string $entryid Entryid of the meeting request or appointment for which all items should be deleted
		*/
		function removeFromCalendar($store, $entryid) {
			$message = $GLOBALS["operations"]->openMessage($store, $entryid);
			
			$req = new Meetingrequest($store, $message, $GLOBALS["mapisession"]->getSession());
			
			$req->doRemoveFromCalendar();
			
			// Notify the bus that the message has been deleted			
			$messageProps = mapi_getprops($message, array(PR_ENTRYID, PR_STORE_ENTRYID, PR_PARENT_ENTRYID));
			$GLOBALS["bus"]->notify(bin2hex($messageProps[PR_PARENT_ENTRYID]), TABLE_DELETE, $messageProps);
		}
		
		/**
		* Get addressbook hierarchy
		*
		* This function returns the entire hierarchy of the addressbook, with global addressbooks, and contacts
		* folders.
		*
		* The input parameter $storelist is an associative array with the format:
		* $storelist["store"] = array of store objects to traverse for all contacts folders
		* $storelist["folder"] = array of store objects to traverse for only the default contacts folder
		* 
		* The output array contains an associative array for each found contact folder. Each entry contains
		* "display_name" => Name of the folder, "entryid" => entryid of the folder, "parent_entryid" => parent entryid
		* "storeid" => store entryid, "type" => gab | contacts
		* 
		* @param array Associative array with store information
		* @return array Array of associative arrays with addressbook container information
		* @todo Fix bizarre input parameter format
		*/
		function getAddressbookHierarchy($storelist=false)
		{
			$ab = $GLOBALS["mapisession"]->getAddressbook();
			$dir = mapi_ab_openentry($ab);
			$table = mapi_folder_gethierarchytable($dir, CONVENIENT_DEPTH);
			$items = mapi_table_queryallrows($table, array(PR_DISPLAY_NAME, PR_ENTRYID, PR_PARENT_ENTRYID, PR_DEPTH));
			
			$folders = array();

			$parent = false;
			foreach($items as $item){
				// TODO: fix for missing PR_PARENT_ENTRYID, see #2190
				if ($item[PR_DEPTH]==0)
					$parent = $item[PR_ENTRYID];

				$item[PR_PARENT_ENTRYID] = $parent;
				
				$folders[] = array(
								"display_name"	=> w2u($item[PR_DISPLAY_NAME]),
								"entryid"		=> array("attributes"=>array("type"=>"binary"), "_content"=> bin2hex($item[PR_ENTRYID])),
								"parent_entryid"=> array("attributes"=>array("type"=>"binary"), "_content"=> bin2hex($item[PR_PARENT_ENTRYID])),
								"type"			=> "gab"
							);
			}

			if ($storelist){

				foreach($storelist["store"] as $store){
					if (empty($store))
						continue;

					$cur_store = $GLOBALS["mapisession"]->openMessageStore(hex2bin($store));
					$store_props = mapi_getprops($cur_store, array(PR_IPM_SUBTREE_ENTRYID, PR_IPM_PUBLIC_FOLDERS_ENTRYID, PR_MDB_PROVIDER, PR_DISPLAY_NAME, PR_MAILBOX_OWNER_NAME, PR_IPM_WASTEBASKET_ENTRYID));
					/**
					 * Check to see if we are dealing with a public store or not. The first level below
					 * the public folder contains folders that are linked. The CONVENIENT_DEPTH property
					 * will not work there. The PR_IPM_PUBLIC_FOLDERS_ENTRYID holds the entryID of the
					 * folder that we can use as the $subtree.
					*/
					// Public store
					if($store_props[PR_MDB_PROVIDER] == ZARAFA_STORE_PUBLIC_GUID){
							$subtree = mapi_msgstore_openentry($cur_store, $store_props[PR_IPM_PUBLIC_FOLDERS_ENTRYID]);
							if (mapi_last_hresult()) continue;
					// Private store
					}else{
							$subtree = mapi_msgstore_openentry($cur_store, $store_props[PR_IPM_SUBTREE_ENTRYID]);
							if (mapi_last_hresult()) continue;
					}

					//Get "Deleted Items" folder
					if (isset($store_props[PR_IPM_WASTEBASKET_ENTRYID])) {
						$hierarchy = mapi_folder_gethierarchytable($subtree);
						mapi_table_restrict($hierarchy, array(RES_PROPERTY,
															array(	RELOP		=>	RELOP_EQ,
																	ULPROPTAG	=>	PR_ENTRYID,
																	VALUE		=>	array(PR_ENTRYID=>$store_props[PR_IPM_WASTEBASKET_ENTRYID])
															)
														)
						);
						$deleted_folder = mapi_table_queryallrows($hierarchy, array(PR_DISPLAY_NAME, PR_ENTRYID, PR_PARENT_ENTRYID, PR_DEPTH));						
					}
					
					//Get all contact folders from hierarchylist
					$hierarchy = mapi_folder_gethierarchytable($subtree, CONVENIENT_DEPTH);
					mapi_table_restrict($hierarchy, array(RES_CONTENT, 
														array(	FUZZYLEVEL=>FL_PREFIX|FL_IGNORECASE,
																ULPROPTAG=>PR_CONTAINER_CLASS,
																VALUE=>array(PR_CONTAINER_CLASS=>"IPF.Contact")
															)
													)
					);
					
					//Remove folder from hierarchylist that are also in "Deleted Items" folder
					if (isset($deleted_folder[0][PR_DISPLAY_NAME])) {
						
						//Get all contact folders from "Deleted Items" folder
						$cur_folder = mapi_msgstore_openentry($cur_store, $deleted_folder[0][PR_ENTRYID]);
						$hie = mapi_folder_gethierarchytable($cur_folder, CONVENIENT_DEPTH);
						mapi_table_restrict($hie, array(RES_CONTENT, 
															array(	FUZZYLEVEL=>FL_PREFIX|FL_IGNORECASE,
																	ULPROPTAG=>PR_CONTAINER_CLASS,
																	VALUE=>array(PR_CONTAINER_CLASS=>"IPF.Contact")
																)
														)
						);
						$deleted_folder_items = mapi_table_queryallrows($hie, array(PR_DISPLAY_NAME, PR_ENTRYID));
						
						/**
						 * Prepare restriction that removes all folders 
						 * which are also in "Deleted Items" folder.
						 */
						$folderRestriction = array();
						foreach($deleted_folder_items as $folder){
							$folderRestriction[] = 	array(RES_PROPERTY,
														array(
																RELOP		=>	RELOP_NE,
																ULPROPTAG	=>	PR_ENTRYID,
																VALUE		=>	array(PR_ENTRYID	=>	$folder[PR_ENTRYID])
														)
													);
						}
			
						mapi_table_restrict($hierarchy, array(RES_AND,
															array(
																array(RES_AND,
																		$folderRestriction,
																),
																array(RES_CONTENT, 
																	array(	FUZZYLEVEL=>FL_PREFIX|FL_IGNORECASE,
																			ULPROPTAG=>PR_CONTAINER_CLASS,
																			VALUE=>array(PR_CONTAINER_CLASS=>"IPF.Contact")
																	)
																)
															)
														)
						);
					}
					
					switch($store_props[PR_MDB_PROVIDER]){
						case ZARAFA_STORE_PUBLIC_GUID:
							$store_owner = " - " . w2u($store_props[PR_DISPLAY_NAME]);
							break;
						case ZARAFA_STORE_DELEGATE_GUID:
						default:
							$store_owner = " - " . w2u($store_props[PR_MAILBOX_OWNER_NAME]);
							break;
					}

					$store_folders = mapi_table_queryallrows($hierarchy, array(PR_DISPLAY_NAME, PR_ENTRYID));
					foreach($store_folders as $item){
						$folders[] = array(
										"display_name"	=> w2u($item[PR_DISPLAY_NAME]) . $store_owner,
										"entryid"		=> array("attributes"=>array("type"=>"binary"), "_content"=> bin2hex($item[PR_ENTRYID])),
										"parent_entryid"=> array("attributes"=>array("type"=>"binary"), "_content"=> bin2hex($item[PR_ENTRYID])),
										"storeid"		=> array("attributes"=>array("type"=>"binary"), "_content"=> $store),
										"type"			=> "contacts"
									);
					}
				}

				foreach($storelist["folder"] as $store){
					if (empty($store))
						continue;
					
					$cur_store = $GLOBALS["mapisession"]->openMessageStore(hex2bin($store));
					$store_props = mapi_getprops($cur_store, array(PR_MDB_PROVIDER, PR_DISPLAY_NAME, PR_MAILBOX_OWNER_NAME));
					$root_folder = mapi_msgstore_openentry($cur_store, null);
					$root_props = mapi_getprops($root_folder, array(PR_IPM_CONTACT_ENTRYID));
					$contact_folder = mapi_msgstore_openentry($cur_store, $root_props[PR_IPM_CONTACT_ENTRYID]);
					if (mapi_last_hresult()) continue;

					switch($store_props[PR_MDB_PROVIDER]){
						case ZARAFA_STORE_PUBLIC_GUID:
							$store_owner = " - " . w2u($store_props[PR_DISPLAY_NAME]);
							break;
						case ZARAFA_STORE_DELEGATE_GUID:
						default:
							$store_owner = " - " . w2u($store_props[PR_MAILBOX_OWNER_NAME]);
							break;
					}

					$contact_props = mapi_getprops($contact_folder, array(PR_DISPLAY_NAME, PR_ENTRYID));
					$folders[] = array(
									"display_name"	=> w2u($item[PR_DISPLAY_NAME]) . $store_owner,
									"entryid"		=> array("attributes"=>array("type"=>"binary"), "_content"=> bin2hex($contact_props[PR_ENTRYID])),
									"parent_entryid"=> array("attributes"=>array("type"=>"binary"), "_content"=> bin2hex($contact_props[PR_ENTRYID])),
									"storeid"		=> array("attributes"=>array("type"=>"binary"), "_content"=> $store),
									"type"			=> "contacts"
								);
				}
			}
			
			return $folders;
		}

		/**
		 * Publishing the FreeBusy information of the default calendar. The 
		 * folderentryid argument is used to check if the default calendar 
		 * should be updated or not.
		 * 
		 * @param $store MAPIobject Store object of the store that needs publishing
		 * @param $folderentryid binary entryid of the folder that needs to be updated.
		 */
		function publishFreeBusy($store, $folderentryid=false){
			// Publish updated free/busy information
			// First get default calendar from the root folder
			$rootFolder = mapi_msgstore_openentry($store, null);
			$rootFolderProps = mapi_getprops($rootFolder, array(PR_IPM_APPOINTMENT_ENTRYID));

			// If no folderentryid supplied or if the supplied entryid matches the default calendar.
			if(!$folderentryid || $rootFolderProps[PR_IPM_APPOINTMENT_ENTRYID] == $folderentryid){
				// Get the calendar and owner entryID
				$calendar = mapi_msgstore_openentry($store, $rootFolderProps[PR_IPM_APPOINTMENT_ENTRYID]);
				$storeProps = mapi_msgstore_getprops($store, array(PR_MAILBOX_OWNER_ENTRYID));
				if (isset($storeProps[PR_MAILBOX_OWNER_ENTRYID])){
					// Lets share!
					$pub = new FreeBusyPublish($GLOBALS["mapisession"]->getSession(), $store, $calendar, $storeProps[PR_MAILBOX_OWNER_ENTRYID]);
					$pub->publishFB(time() - (7 * 24 * 60 * 60), 6 * 30 * 24 * 60 * 60); // publish from one week ago, 6 months ahead
				}
			}
		}
	}
?>
