/**
 * --TimePicker widget--  
 * @author Frans van Veen <f.vanveen@zarafa.com>
 * @type	Widget
 * @classDescription	This widget will create 1 time picker
 *
 * HOWTO BUILD:
 * - make one div with unique id like "startpick" without undersquare
 * - make a object: this.tp = new DateTimePicker(dhtml.getElementById("startpick"),"Start time");
 * HOWTO USE:
 * - to change the time: this.tp.setValue(1159164000);//use timestamps
 * - to get the time: this.tp.getValue();//will return a timestamp
 * HOWTO REACT:
 * - it is possible to overwrite the onchange function
 *   this function will be exectute when the time or date
 *   are changed: this.tp.onchange = tpOnchange;
 * DEPENDS ON:
 * |------> timepicker.js
 * |------> dhtml.js
 * |------> date.js
 */

timePicker.prototype = new Widget;
timePicker.prototype.constructor = timePicker;
timePicker.superclass = Widget.prototype;
 
//PUBLIC
/**
 * @constructor	This widget will create 1 time picker
 * @param {HtmlElement} element
 * @param {String} picTitle
 * @param {Int} timeInSec
 */
function timePicker(element,picTitle,timeInSec)
{
	this.element = element;
	this.changed = false;
	this.visible = true;
	if(picTitle){
		this.picTitle = picTitle;
	}
	else{
		this.picTitle = "";
	}
	if(timeInSec){
		this.value = timeInSec;
	}
	else{
		this.value = 32400;//32400 = 9:00
	}
	this.timeInput = null;
	this.onchange = null;

	this.render();
	this.setValue(this.value);
}

/**
 * Function will resturn the value "timestamp" of "this.value"
 * @return {int}
 */ 
timePicker.prototype.getValue = function()
{
	var result = 0;
	if(this.visible == true){
		result = timeToSeconds(this.timeInput.value);
	}
	return parseInt(result,10);
}

/**
 * Function will set "this.value" time
 * @param {Int} unixtime timestamp 
 */ 
timePicker.prototype.setValue = function(timeInSec)
{
	var newValue = parseInt(timeInSec,10);
	var oldValue = this.value;
	var dayValue = 0;
	
	if(newValue>=86400){
		newValue=newValue-86400;//86400 = ONE_DAY
		dayValue = 1;
	}
	if(newValue<0){
		newValue=86400+newValue;//86400 = ONE_DAY
		dayValue = -1; 
	}
	if(oldValue != newValue){
		this.value = newValue;
	}

	this.dayValue = dayValue;
	this.timeInput.value = secondsToTime(this.value);
}

timePicker.prototype.hide = function()
{
	this.timeInput.parentNode.parentNode.parentNode.parentNode.style.visibility = "hidden";
	this.visible = false;
}

timePicker.prototype.show = function()
{
	this.timeInput.parentNode.parentNode.parentNode.parentNode.style.visibility = "visible";
	this.visible = true;
}

//PRIVATE
timePicker.prototype.render = function()
{
	//drawing elements
	var container = dhtml.addElement(this.element,"table");
	container.setAttribute("border","0");
	container.setAttribute("cellpadding","0");
	container.setAttribute("cellspacing","0");

	var container = dhtml.addElement(container, "tbody");

	var row1 = dhtml.addElement(container,"tr");
	if(this.picTitle.length > 0){
		var col1 = dhtml.addElement(row1,"td","propertynormal propertywidth",null,this.picTitle+":");
	}
	
	var col4 = dhtml.addElement(row1,"td");
	this.timeInput = dhtml.addElement(col4,"input","fieldsize");
	this.timeInput.setAttribute("type","text");
	this.timeInput.setAttribute("size","4");
	dhtml.addEvent(-1,this.timeInput,"change",eventTimePickerInputChange);
	this.timeInput.timePickerObject = this;
	
	var col5 = dhtml.addElement(row1,"td");
	col5.style.width = 40+"px";
	var col5spinUp = dhtml.addElement(col5,"div","spinner_up","","\u00a0");
	dhtml.addEvent(-1,col5spinUp,"click",eventTimePickerSpinnerUp);
	col5spinUp.timePickerObject = this;
	var col5spinDown = dhtml.addElement(col5,"div","spinner_down","","\u00a0");
	dhtml.addEvent(-1,col5spinDown,"click",eventTimePickerSpinnerDown);
	col5spinDown.timePickerObject = this;
}

/**
 * @param {Object} moduleObject
 * @param {HtmlElement} element
 * @param {Object} event
 */
function eventTimePickerInputChange(moduleObject, element, event)
{
	var dtpObj = element.timePickerObject;
	var oldValue = dtpObj.value;
	var newValue = element.value;

	var regTime = /^[0-2]?[0-9][:][0-6]?[0-9]$/; //time
	if(!regTime.test(newValue)){
		if(newValue.length != 0){
			alert(_("You must specify a valid date and/or time. Check your entries in this dialog box to make sure they represent a valid date and/or time."));
			dtpObj.setValue(32400);//32400 = 9:00
			if(dtpObj.onchange)
				dtpObj.onchange(dtpObj, oldValue);
		}
	}
	else{
		dtpObj.setValue(timeToSeconds(newValue));
		if(dtpObj.onchange)
			dtpObj.onchange(dtpObj, oldValue);
	}
}

/**
 * @param {Object} moduleObject
 * @param {HtmlElement} element
 * @param {Object} event
 */
function eventTimePickerSpinnerUp(moduleObject, element, event)
{
	var dtpObj = element.timePickerObject;
	var oldValue = dtpObj.value;
	dtpObj.setValue(oldValue+HALF_HOUR/1000);
	if(dtpObj.onchange)
		dtpObj.onchange(dtpObj, oldValue, dtpObj.dayValue);
}

/**
 * @param {Object} moduleObject
 * @param {HtmlElement} element
 * @param {Object} event
 */
function eventTimePickerSpinnerDown(moduleObject, element, event)
{
	var dtpObj = element.timePickerObject;
	var oldValue = dtpObj.value;
	dtpObj.setValue(oldValue-HALF_HOUR/1000);
	if(dtpObj.onchange)
		dtpObj.onchange(dtpObj, oldValue, dtpObj.dayValue);
}
