/* reverse.c
 *
 * handle the reversal of ip addressen when
 * querying for the reverse
 *
 * See the file LICENSE for the license
 *
 * (c) Copyright NLnet Labs, 2004
 */

#include "common.h"

/**
 * Returns the in-addr.arpa address
 * of addr
 * Can handle both ipv4/ipv6
 */
char *
in_addr(char *addr)
{
	char *new;
	char *rev;
	assert(addr != NULL);
	if (is_ipv6_addr(addr)) {
		new = xmalloc(74);
		/*new = xmalloc(strlen(addr)*2 + strlen(IPV6_ARPA) + 2);*/
		rev = reverse_ipv6(addr);
		snprintf(new, 74, "%s.%s", rev, IPV6_ARPA);
	} else {
		new = xmalloc(strlen(addr) + strlen(IN_ADDR_ARPA) + 2);
		rev = reverse_ipv4(addr);
		snprintf(new, 74, "%s.%s", rev, IN_ADDR_ARPA);
	}
	xfree(rev);
	return new;
}

/* reverse an IPv4 addr:
 * 192.168.11.12 becomes
 * 12.11.168.192
 */
char *
reverse_ipv4(char *addr) 
{
	char *ca = xstrdup(addr);
	char *rev = xmalloc(strlen(addr) + 1);
	char *cp; char *d;
	size_t i = 0, l;
	
	for(cp = ca; (d = strrchr(cp, '.')); *d = '\0' ) {
		l = strlen(d + 1);
		memcpy(rev + i, d + 1, l);

		/* add dot */
		*(rev + i + l) = '.';
		i += l + 1;
	}
	memcpy(rev + i, cp, strlen(cp));
	*(rev + i + strlen(cp)) = '\0';
	xfree(ca);
	return rev;
}

/* reverse an ipv6 addr:
 * 2001:7b8:32a:0:280:48ff:feb3:26eb
 * becomes
 * b.e.6.2.3.b.e.f.f.f.8.4.0.8.2.0.0.0.0.0.a.2.3.0.8.b.7.0.1.0.0.2
 *
 * This function expects a normalized ipv6 address! - it does not 
 * do that by itself!
 */
char * 
reverse_ipv6(char *addr)
{
	size_t i, j;

	char *normalized = normalize_ipv6(addr);
	char *rev = xmalloc(strlen(normalized) * 2);
	
	for (i = strlen(normalized) - 1, j = 0; i > 0; i--) {
		if (*(normalized + i) == ':')
			continue;
		
		rev[j] = normalized[i];
	rev[j+1] = '.';
	rev[j+2] = '\0';
		j += 2;
	}
	rev[j] = normalized[i];
	rev[j+1] = '\0';
	xfree(normalized);
	return rev;
}

/* normalize an ipv6 addr:
 * :: -> :0000:
 * :280: -> :0280:
 */
char *
normalize_ipv6(char *addr)
{
	/* print hex with prefix zero */
	uint8_t *bin;
	char *result = NULL;
	bin = xmalloc(16);
	if (ipv62bin(addr, bin) == RET_SUC) {
		result = bin2ipv6(bin);
	} else {
		error("in reading ipv6 address: %s", addr);
	}
	xfree(bin);
	return result;
}
