/**
 * print.c
 *
 * conversion function for printing the
 * internals to screen
 *
 * (c) 2004 NLnet Labs
 *
 * See the file LICENSE for the license
 *
 */

#include <stdio.h>
#include <assert.h>

#include "common.h"

/**
 * Returns the string representation of the given rdata
 * If there is a known name, return that (for instance RSASHA1 instead of 5
 * s algorithm)
 */
const char *
rdata2nicestr(struct t_rdata *rdata, int rdata_type)
{
	const char *val;
	char *res;
	size_t i, j, i_res;
	int c;

	switch (rdata_type) {
		case RDATA_TYPE_TYPE:
			res = typebyint(rdata2uint16(rdata), ztypes);
			break;
		case RDATA_TYPE_CLASS:
			val = classbyint(rdata2uint16(rdata), zclasses);
			res = xmalloc(strlen(val)+1);
			strncpy(res, val, strlen(val)+1);
			break;
		case RDATA_TYPE_ALGO:
			val = namebyint(rdata2uint8(rdata), dnssec_algos);
			res = xmalloc(strlen(val)+1);
			strncpy(res, val, strlen(val)+1);
			break;
		case RDATA_TYPE_CERTTYPE:
			val = namebyint(rdata2uint8(rdata), cert_types);
			res = xmalloc(strlen(val)+1);
			strncpy(res, val, strlen(val)+1);
			break;
		case RDATA_TYPE_STR:
			val = rdata2str(rdata);
			res = xmalloc(rdata->length + 3);
			snprintf(res, rdata->length + 3, "\"%s\"", val);
			xfree((char *)val);
			break;
		case RDATA_TYPE_DNAME:
			val = rdata2str(rdata);
			/* shortcut for root label */
			if (rdata->length == 0) {
				res = xstrdup(".");
				break;
			}
			
			res = xmalloc(strlen(val)+1);
			memset(res, 0, strlen(val)+1);
			/* find multiple occurences of \XXX and replace with
			 * 1 \XXX{times}
			 */
			i_res = 0;
			for (i=0; i<strlen(val); i++) {
				if (val[i] == '\\' &&
				    i < strlen(val) - 3 &&
				    isdigit(val[i+1]) &&
				    isdigit(val[i+2]) &&
				    isdigit(val[i+3])
				) {
					c = 0;
					j = i;
					while (j < strlen(val) &&
					       val[j] == '\\' &&
					       j < strlen(val) - 3 &&
					       isdigit(val[j+1]) &&
					       isdigit(val[j+2]) &&
					       isdigit(val[j+3])
					) {
						c++;
						j += 4;
					}
					if (c > 1) {
						memcpy(&res[i_res], &val[i], 4);
						i_res += 4;
						res[i_res] = '{';
						i_res++;
						snprintf(&res[i_res], 3, "%d", c);
						if (c > 99) {
							i_res += 3;
						} else if (c > 9) {
							i_res += 2;
						} else {
							i_res++;
						}
						res[i_res] = '}';
						i_res++;
						i = j;
					}
				}
					res[i_res] = val[i];
					i_res++;
			}
			res[i_res] = '\0';
			xfree((char *) val);
			break;
		default:
			/* most are directly printable */
			res = rdata2str(rdata);
	}

	return res;
}

/**
 * Prints an rr to stdout
 * follow = FOLLOW; follow the linked list
 */
void
print_rr(struct t_rr *rr, int follow)
{
	uint16_t c;
	char *str;
	
	if (!rr)
		return;
	/* miek.nl	IN 	MX	10 mx.miek.nl. */
	print_rd(rr->name); /* owner name */
	if (rr->section != SEC_QUESTION) {
		str = typebyint(rr->type, ztypes);
		printf("\t%ld\t%s\t%s\t", (long int) rr->ttl,
				classbyint(rr->class, zclasses),
				str
		      );
		xfree(str);
	} else {
		/* skip the ttl */
		str = typebyint(rr->type,  ztypes);
		printf("\t%s\t%s\t", classbyint(rr->class, zclasses), str);
		xfree(str);
	}
	/* TODO better/smarter printing */

	if (rr->section != SEC_QUESTION) {
		for(c = 0; c < rr->rdcount; c++) {
			str = (char *) rdata2nicestr(rr->rdata[c], rr_rdata_types[rr->type][c]);
			printf("%s", str);
			xfree(str);
			
			if (c < (uint16_t) rr->rdcount - 1) {
				printf(" ");
			}
		}
		/* print the key id */
		if (rr->type == TYPE_DNSKEY) {
			printf(" ; key id = %d", (int) keytag(rr));
			if (drill_opt->ds == 1) {
				/* also print the DS */
				printf("\n; ");
				print_rr(key2ds(rr), NO_FOLLOW);
				return;
			}
		}
	}
		
	printf("\n");
	if (follow == FOLLOW)
		print_rr(rr->next, FOLLOW);
}

void
prettyprint_packet(struct t_dpacket *packet)
{
	assert(packet != NULL);
	
	print_packet_header(packet);
	
	prettyprint_packet_body(packet);

	print_packet_footer(packet);
}

/* pretty print a packet by using
 * prettyprint_rr 
 */
void
prettyprint_packet_body(struct t_dpacket *p) 
{
	unsigned int longest = get_longest_dname(p->rrs, FOLLOW);
    
	if (p->count[SEC_QUESTION] > 0) {
                printf(";; QUESTION SECTION:\n");
                printf(";");	/* be more like dig */
                prettyprint_rr_sec(p->rrs, SEC_QUESTION, NO_COMMENT, longest);
                fflush(stdout);
                printf("\n");
        }
        if (p->count[SEC_ANSWER] > 0) {
                printf(";; ANSWER SECTION:\n");
                prettyprint_rr_sec(p->rrs, SEC_ANSWER, NO_COMMENT, longest);
                printf("\n");
        }
        if (p->count[SEC_AUTH] > 0) {
                printf(";; AUTHORITY SECTION:\n");
                prettyprint_rr_sec(p->rrs, SEC_AUTH, NO_COMMENT, longest);
                printf("\n");
        }
        if (p->count[SEC_ADD] > 0) {
                printf(";; ADDITIONAL SECTION:\n");
                prettyprint_rr_sec(p->rrs, SEC_ADD, NO_COMMENT, longest);
                printf("\n");
        }
}

/**
 * Prints an formatted rr to stdout
 * follow = FOLLOW; follow the linked list
 * if longest_dname is zero it calculates it by itself given the rr(set)
 *
 * This function is ugly - don't really now how to better that XXX
 */
void
prettyprint_rr(struct t_rr *rr, int follow, int comment, unsigned int longest_dname)
{
	uint16_t screen_width = 70; /* constant */
	uint16_t spacer = 2;
	uint16_t dname_length;
	uint16_t rdata_start;
	uint16_t width;
	uint16_t i,j;
	
	uint16_t linebreak = 0;
	uint16_t char_cnt = 0;
	uint16_t offset = 0;
	const char *rd;
	char *tmpstr;
	
	if (!rr)
		return;

	/* miek.nl	IN 	MX	10 mx.miek.nl. */
        /* |-----| longest_dname
	 *        |-----| spacer
         * |---------------------------| rdata_start
	 * |-----------| offset
         *                              |---------------|
	 *                                 width
	 * |--------------------------------------------|
	 *   screen_width (=80)
	 */

	if (longest_dname == 0) 
		longest_dname = get_longest_dname(rr, follow);

	dname_length = (unsigned int) rr->name->length; /* lenght of the current ownername */

	if (comment == COMMENT) {
		printf(";");
		tmpstr = (char *) rdata2nicestr(rr->name, RDATA_TYPE_DNAME);
		printf("%s", tmpstr);
		xfree(tmpstr);
		/*print_rd(rr->name);*/ /* owner name */
		print_spaces(longest_dname - dname_length + spacer - 1);
	} else {
		tmpstr = (char *) rdata2nicestr(rr->name, RDATA_TYPE_DNAME);
		printf("%s", tmpstr);
		xfree(tmpstr);
		/*print_rd(rr->name);*/ /* owner name */
		print_spaces(longest_dname - dname_length + spacer);
	}

	/* now print the common stuff */
	if (rr->section == SEC_QUESTION) {
		/* skip the ttl */
		tmpstr = typebyint(rr->type,  ztypes);
		printf("    %2s  %-7s  ", classbyint(rr->class, zclasses), tmpstr);
		xfree(tmpstr);
	} else {
		tmpstr = typebyint(rr->type,  ztypes);
		printf("%-5ld  %2s  %-7s  ", (long int) rr->ttl, classbyint(rr->class, zclasses), tmpstr);
		xfree(tmpstr);
		
		rdata_start = longest_dname + (3 * spacer) + 14  ; /* start of the rdata */
		char_cnt    = rdata_start;
		offset	    = rdata_start - longest_dname;
		width       = screen_width - offset ; /* how many pos do we have left */

		/* print the rdata specific for this RR */
		for(i = 0; i < rr->rdcount; i++) {
			if (char_cnt > screen_width) {
				if (linebreak)
					printf("\n");
				else
					printf("( \n");

				linebreak = 1;
				if (comment == COMMENT) {
					printf(";");
					print_spaces(offset - 1);
				} else 
					print_spaces(offset);

				char_cnt = offset;
			}
			
			switch(rr_rdata_types[rr->type][i]) {
				/* print each of the elements of the rdata */
				case RDATA_TYPE_DNAME:
				case RDATA_TYPE_INT48:
				case RDATA_TYPE_INT32:
				case RDATA_TYPE_INT16:
				case RDATA_TYPE_TYPE:
				case RDATA_TYPE_CLASS:
				case RDATA_TYPE_INT8:
				case RDATA_TYPE_ALGO:
				case RDATA_TYPE_A:
				case RDATA_TYPE_AAAA:
				case RDATA_TYPE_ANY:
				case RDATA_TYPE_STR:
				case RDATA_TYPE_HEX:
				case RDATA_TYPE_NSEC:
				default:
					rd = rdata2nicestr(rr->rdata[i], 
							rr_rdata_types[rr->type][i]);
					printf("%s ", rd);
					char_cnt = (unsigned int) char_cnt + strlen(rd) + 1;
					xfree((char *)rd);
					break;
				case RDATA_TYPE_BASE64:
					if (linebreak == 0) {
						linebreak = 1;
						printf("(");
						print_spaces(offset);
						char_cnt = offset;
					}

					
					if (comment != COMMENT) {
						/* this always the last in a RR, I think... */
						for(j = 0 ; j < rr->rdata[i]->length; j = j + width) {
							if (!j)
								printf("\n");

							if (linebreak)
								print_spaces(offset);

							if (j + width > rr->rdata[i]->length) {
								/* guard against overflow */
								print_piece(rr->rdata[i]->data, j, 
										rr->rdata[i]->length - j);
							} else {
								print_piece(rr->rdata[i]->data, j, width);
							}

							if ( j + width < rr->rdata[i]->length)
								printf("\n");
						}
				 	} else  {
						/* this always the last in a RR */
						for(j = 0 ; j < rr->rdata[i]->length; j = j + width) {
							if (!j)
								printf("\n");

							if (linebreak) {
								printf(";");
								print_spaces(offset - 1);
							} 
							
							if (j + width > rr->rdata[i]->length) {
								/* guard against overflow */
								print_piece(rr->rdata[i]->data, j, 
										rr->rdata[i]->length - j);
							} else {
								print_piece(rr->rdata[i]->data, j, width);
							}
							if ( j + width < rr->rdata[i]->length)
								printf("\n");
						}
					}
					break;
			}
		}
		if (linebreak) {
			if (comment) {
				printf(";");
		                print_spaces(offset - 1);
			}
			printf(" )");
		}
		
		/* for dnskeys we need to do some more */
		if (rr->type == TYPE_DNSKEY) {
			printf(" ; {id = %d}", (int) keytag(rr));
			tmpstr = rdata2str(rr->rdata[0]);
			if (atoi(tmpstr) % 2 == 1)
				printf(" {sep}");
			xfree(tmpstr);

			if (drill_opt->ds == 1) {
				/* also print the DS */
				printf("\n");
				prettyprint_rr(key2ds(rr), NO_FOLLOW, COMMENT, longest_dname);
				return;
			}
		}
	}
	printf("\n");
	if (follow == FOLLOW)
		prettyprint_rr(rr->next, FOLLOW, comment, longest_dname);
}

/* print only specfic sections */
void
prettyprint_rr_sec(struct t_rr *rr, t_section s, int comment, unsigned int longest)
{
	if (!rr) 
		return;
	if (rr->section == s)
		prettyprint_rr(rr, NO_FOLLOW, comment, longest);
	prettyprint_rr_sec(rr->next, s, comment, longest);
}

/**
 * Prints the rr to the screen, not showing the rdata part
 */
void
print_rr_short(struct t_rr *rr, int follow)
{
	char *str;
	
        if (!rr)
                return;
        /* miek.nl      IN      MX      10 mx.miek.nl. */
        print_rd(rr->name); /* owner name */
        if (rr->section != SEC_QUESTION) {  
                str = typebyint(rr->type,  ztypes);
                printf("\t%ld\t%s\t%s\t", (long int) rr->ttl,
                                classbyint(rr->class, zclasses), str
		);
		xfree(str);
        } else {
                /* skip the ttl */
                str = typebyint(rr->type,  ztypes);
                printf("\t%s\t%s\t", classbyint(rr->class, zclasses), str);
                xfree(str);
        }
        printf("\n");
        if (follow == FOLLOW)
                print_rr_short(rr->next, FOLLOW);
}

/**
 * Prints all the rrs in the given section (SEC_QUESTION, SEC_ANSWER,
 * SEC_AUTH, SEC_ADD, SEC_MAX or SEC_ALL)
 */
void
print_rr_sec(struct t_rr *rr, t_section s)
{
	if (!rr) 
		return;
	if (rr->section == s)
		print_rr(rr, NO_FOLLOW);
	print_rr_sec(rr->next, s);
}


/**
 * Prints the given packet in a dense form (dig +trace) to stdout
 */
void
print_packet_dense(struct t_dpacket *p)
{
	assert(p != NULL);

	if (p->count[SEC_ANSWER] > 0)
		print_rr_sec(p->rrs, SEC_ANSWER);

	if (p->count[SEC_AUTH] > 0)
		print_rr_sec(p->rrs, SEC_AUTH);
	
	if (p->flags.qr) {
		printf(";; Received %d bytes from %s in %ld ms\n",
				(int) p->udppacketsize,
				p->serverip,
				p->querytime);
	}
}

/**
 * Prints the given packet to stdout
 */
void
print_packet_header(struct t_dpacket *packet)
{
	assert(packet != NULL);
	
	printf(";; ->>HEADER<<- opcode: %u, status %s, id %u\n", packet->flags.opcode,
			namebyint(packet->flags.rcode + 1, rcodes),
			(unsigned int) packet->id);
	print_flags(packet);
	printf(" QUERY: %u, ANSWER: %u, AUTHORITY: %u, ADDITIONAL: %u\n",
			(unsigned int) packet->count[SEC_QUESTION],
			(unsigned int) packet->count[SEC_ANSWER],
			(unsigned int) packet->count[SEC_AUTH],
			(unsigned int) packet->count[SEC_ADD]);
	
	if (packet_is_opt(packet)) {
		printf(";; OPT PSEUDOSECTION:\n");
		printf("; EDNS: version %u, flags:", GET_VERSION(packet));
		if (GET_DNSSEC(packet)) {
			printf(" do");
		} 
		printf("; udp: %u\n\n", (unsigned int) GET_UDPSIZE(packet));
	}
}

void
print_packet_body(struct t_dpacket *packet)
{
	if (packet->count[SEC_QUESTION] > 0) {
		printf(";; QUESTION SECTION:\n");
		printf(";; ");
		print_rr_sec(packet->rrs, SEC_QUESTION);
		fflush(stdout);
		printf("\n");
	}
	if (packet->count[SEC_ANSWER] > 0) {
		printf(";; ANSWER SECTION:\n");
		print_rr_sec(packet->rrs, SEC_ANSWER);
		printf("\n");
	}
	if (packet->count[SEC_AUTH] > 0) {
		printf(";; AUTHORITY SECTION:\n");
		print_rr_sec(packet->rrs, SEC_AUTH);
		printf("\n");
	}
	if (packet->count[SEC_ADD] > 0) {
		printf(";; ADDITIONAL SECTION:\n");
		print_rr_sec(packet->rrs, SEC_ADD);
		printf("\n");
	}
}

void
print_packet_footer(struct t_dpacket *packet)
{
        if (packet->flags.qr) {
                printf(";; Query time: %ld msec\n", packet->querytime);
                printf(";; SERVER: %s\n", packet->serverip);
		/* TODO strange behaviour on some machines, TODO, check, find, fix and return */
                printf(";; WHEN: %s", asctime(localtime(&packet->date))); 
        }

	printf(";; MSG SIZE");
	if (packet->flags.qr) {
		printf(" rcvd");
	}
	printf(": %u\n", (unsigned int) packet->udppacketsize);
}

void
print_packet(struct t_dpacket *packet)
{
	assert(packet != NULL);
	
	print_packet_header(packet);
	
	print_packet_body(packet);
	
	print_packet_footer(packet);
}

/**
 * Prints the flags part of the given packet to stdout
 */
void
print_flags(struct t_dpacket *p)
{
	printf(";; flags:");
	if (p->flags.qr == 1)
		printf(" qr");
	if (p->flags.aa == 1)
		printf(" aa");
	if (p->flags.rd == 1)
		printf(" rd");
	if (p->flags.ra == 1)
		printf(" ra");
	if (p->flags.tc == 1)
		printf(" tc");
	if (p->flags.z == 1)
		printf(" z");
	if (p->flags.ad == 1)
		printf(" ad");
	if (p->flags.cd == 1)
		printf(" cd");
	printf(";");
}

/**
 * Prints a nameserver struct t_rr
 */
void
print_ns(struct t_rr *n, int follow)
{
	print_rr(n, follow);
}

/**
 * Prints the raw data of the given rdata as a string
 */
void
print_rd(struct t_rdata *r)
{
	char * s;
	if (!r)
		return;
	
	if (r->length == 0) {
		/* root label */
		printf(".");
		return;
	}
	s = xmalloc(r->length + 1);
	memcpy(s, r->data, r->length);
	*(s + r->length) = '\0';
	printf("%s", s);
	xfree(s);
}

/**
 * return the lenght of the longest owner name in the packet
 */
unsigned int
get_longest_dname(struct t_rr *rr, int follow)
{
	size_t max = 0;
	char *tmp;
	
	if (rr) {
		while(rr != NULL) {
			tmp = (char *) rdata2nicestr(rr->name, RDATA_TYPE_DNAME);
			if (strlen(tmp) > max) {
				max = strlen(tmp);
			}
			xfree(tmp);
			/*
				if (rr->name->length > max) {
				max = rr->name->length;
			}
			*/
			if (follow) {
				rr = rr->next;
			} else {
				rr = NULL;
			}
		}
	}
	return (unsigned int) max;
}

/**
 * print x spaces
 */
void
print_spaces(uint16_t x)
{
	/* 71 spaces */
	const char *space = "                                                                      ";

	if (x > 71)
		return;

	printf("%.*s", (int) x, space); /* print some spaces */
}

/**
 * print a subsection of a string 
 *
 * beware: no length stuff is checked 
 */
void
print_piece(uint8_t * str, uint16_t offset, uint16_t length) 
{
	char *newstr;

	newstr = xmalloc(length + 1);
	memcpy(newstr, str + offset, length);
	newstr[length]  = '\0'; /* our closing friend */
	printf("%s", newstr);
	xfree(newstr);
}
