/**
 * keyparser.c
 *
 * (c) NLnet Labs, 2004
 *
 * See the file LICENSE for the license
 *
 * contains helper function to parse a key record from
 * the command line
 */

#include <ctype.h>

#include "common.h"

/**
 * Gets the next token from a file handle (delimited by spaces)
 */
int
get_next_token(FILE *f, char *token)
{
        char *t;
        int c;
        int p = 0; /* for '(' and ')' */

        t = token;

        while ((c = getc(f)) != EOF) {
                if (c == ')' && p != 1)
                        return RET_FAIL;
                else {
                        p = 0; 
                }

                if (c == '\n' && p == 0)
                        *t++ = c;

                if (c == '(') {
                        p = 1;
                }

                if (isspace(c)) {
                        /* ready token */
                        *t = '\0'; return RET_SUC;
                }
                *t++ = c; 
        }
        *t = '\0';
        return RET_SUC;
}

/**
 * Gets the last token from a file handle (not delimited, skips over spaces!)
 */
int
get_last_token(FILE *f, char *token)
{
        char *t;
        int c;
        int p = 0; /* for '(' and ')' */

        t = token;

        while ((c = getc(f)) != EOF) {
                if (c == ')' && p != 1)
                        return RET_FAIL;
                else {
                        p = 0; 
                }

                if (c == '\n' && p == 0)
                        *t++ = c;

                if (c == '(') {
                        p = 1;
                }

                if (!isspace(c)) {
                	*t++ = c; 
		}
        }
        *t = '\0';
        return RET_SUC;
}

/**
 * Returns the next rdata
 */
struct t_rdata *
get_next_rdata(FILE *f)
{
	char *tok;
	struct t_rdata *r = NULL;

	tok = xmalloc(MAX_PACKET); 

	if (!get_next_token(f, tok))
		r = rdata_create((uint8_t*)tok, strlen(tok));
	
	xfree(tok);
	return r;
}

struct t_rdata *
get_last_rdata(FILE *f)
{
	char *tok;
	struct t_rdata *r = NULL;

	tok = xmalloc(MAX_PACKET); 

	if (!get_last_token(f, tok))
		r = rdata_create((uint8_t*)tok, strlen(tok));
	
	xfree(tok);
	return r;
}

/**
 * Stores the next token and returns the length
 */
size_t
get_next_token_string(char *buf, char *token, size_t len)
{
	char *t;
	int c;
	size_t i;
	int p = 0; /* for '(' and ')' */
	t = token;
	
	for (i=0; i < len && i < strlen(buf); i++) {
		if (buf[i] == ')' && p != 1) {
			return -1;
		} else {
			p = 0;
		}
		
		if (c == '\n' && p == 0) {
			*t++ = buf[i];
		}
		
		if (isspace(buf[i])) {
			/* ready token */
			*t++ = '\0';
			return (i + 1);
		}
		*t++ = buf[i];
	}
	*t = '\0';
	return i;
}

/**
 * Stores the next rdata and return the length
 */
size_t
get_next_rdata_string(char *buf, struct t_rdata **rd)
{
	char *tok;
	size_t l;
	tok = xmalloc(MAX_PACKET); 
	
	l = get_next_token_string(buf, tok, strlen(buf));
	if (l >= 0) {
		*rd = rdata_create((uint8_t*)tok, strlen(tok));
	}

	xfree(tok);
	return l;
}
