/* 
 * cache.c
 *
 * Implements a RRset cache 
 * This is not intended to be used as a base for a caching 
 * forwarder! That kind of caching is more complex then what
 * is attempted here
 *
 * See the file LICENSE for the license
 *
 * (c) NLnet Labs, 2004
 * 
 */

#include "common.h"

/* we should be implemented here
 * - push, put something in the cache
 * - pop,  retrieve latest pushed item
 * - search, look for an item
 * - rm, remove an specific item
 *
 * Cache is like a mathematical set, no duplicates are allowed
 * the internal implemention right now is linked list.
 * This will change in future
 */

/* TODO duplicates in push? */

/**
 * create a new cache structure and return 
 * it
 */
struct t_cache *
cache_new(void)
{
	struct t_cache *c;

	c = xmalloc(sizeof(struct t_cache));
	c->prev = c; /* or NULL? */
	c->next = NULL;
	c->rr   = NULL;
	return c;
}

/**
 * create a new cache structure from an RR 
 * and return it
 */
struct t_cache *
cache_new_from_rr(struct t_rr *r)
{
	struct t_cache *c;

	c = xmalloc(sizeof(struct t_cache));
	c->prev = c; /* or NULL? */
	c->next = NULL;
	c->rr   = r;
	return c;
}
 

/**
 * push an rr to the cache
 * it will be added to the head
 */
struct t_cache *
cache_push(struct t_cache *c, struct t_rr *r)
{
	struct t_cache *c0;

	assert(c != NULL);
	assert(r != NULL);
	
	c0 = cache_new_from_rr(r);

	/* hang it in front */
	c0->next = c;
	c->prev = c0;
	return c0;
}

/**
 * pop an rr from the cache
 * it will be popped to the head
 */
struct t_cache *
cache_pop(struct t_cache *c, struct t_rr *result)
{
	struct t_cache *c0; /* the first item */
	struct t_cache *c1;

	assert(c != NULL); 

	/* we assert C, so we have at least one item to pop */

	c0 = c;	/* the first item */
	c1 = c->next; 
	if (!c1)
		c1->prev = c1;

	result = c0->rr;
	return c1;
}

/**
 * search through a cache using an ownername and return the rr 
 * or NULL if not found. Return ALL matching RR in a single
 * RRset
 */
struct t_rr *
cache_search_name(struct t_cache *c, struct t_rdata *name)
{
	struct t_cache *t;
	struct t_rr *gem;

	gem = NULL;
	t = c;
	while(t->next != NULL) {
		if (1 == rdata_cmp(t->rr->name, name))  {
			gem = rr_add_rr(gem, t->rr);
		}
		t = t->next;
	}
	return gem;
}

/**
 * search through a cache using type and return the rr 
 * or NULL if not found
 * Return ALL matching RR's in a RRset (which isn't really
 * a set anymore, because the ownername is not guarenteed to be 
 * the same
 */
struct t_rr *
cache_search_type(struct t_cache *c, uint16_t type)
{
	struct t_cache *t;
	struct t_rr *gem;

	gem = NULL;
	t = c;
	while(t->next != NULL) {
		if (t->rr->type == type)  {
			gem = rr_add_rr(gem, t->rr);
		}
		t = t->next;
	}
	return gem;
}

/**
 * search through a cache using an ownername and remove the rr 
 */
struct t_cache *
cache_rm_name(struct t_cache *c, struct t_rdata *name)
{
	warning("cache_rm_name: Not implemented");
	return NULL;
}

/**
 * search through a cache using type and remove the rr 
 */
struct t_cache *
cache_rm_type(struct t_cache *c, uint16_t type)
{
	warning("cache_rm_type: Not implemented");
	return NULL;
}

/**
 * print a cache
 */
void
cache_print(struct t_cache *c)
{
	struct t_cache *t;
	t = c;
	while(t->next != NULL) {
		printf("%p %p -- ", t->prev, t->next);
		print_rr(t->rr, NO_FOLLOW);
		t = t->next;
	}
}
