/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: recon.c,v 6.17.2.1 2002/10/09 19:49:59 brbarret Exp $
 * 
 *	Function:	- LAM reconnaissance tool
 *			- tests if Trollius is bootable on the LAN
 *			- uses "tkill -N"
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <lamnet.h>
#include <net.h>
#include <terror.h>
#include <boot.h>
#include <etc_misc.h>
#include <sfh.h>

/*
 * static variables
 */
static int fl_debug;		/* debug mode */
static int fl_verbose;		/* verbose mode */

/* 
 * static functions
 */
static char *findbhost(int cmdc, char **cmdv, char **env);

/*
 *	main
 */
int
main(int argc, char *argv[], char *env[])
{
	FILE		*fp;		/* boot schema file pointer */
	char		*bhost;		/* boot schema filename */
	char		**cmdv;		/* test command argv */
	int		cmdn;		/* test command argc */
	int		i, j, success = 1;
	int		fl_fast;        /* look for .profile */
	int		iorigin;	/* index of origin node */
	int		r;		/* child return error code */
	int		global_ret;	/* global return code */
	int		badhost;	/* bad host index */
	int		nlamnet;	/* lamnet size */
	struct lamnode	*lamnet;	/* node description array */

	/* It's ok to run recon as root, but with a short warning */

	if (getuid() == 0 || geteuid() == 0) {
	  show_help(NULL, "root-warning", NULL);
	}

/*
 * Parse the command line.
 */
	validopts("abdhv");

	if (do_args(&argc, argv)) {
	  show_help("recon", "usage", NULL);
	  exit(EUSAGE);
	}

	if ((errno = (argc <= 2) ? 0 : EUSAGE)) {
	  show_help("recon", "usage", NULL);
	  exit(errno);
	}
 
	if (opt_taken('h')) {
	  show_help("recon", "usage", NULL);
	  exit(0);
	}

	fl_fast = !opt_taken('b');
	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;
/* 
 * Locate the system file.
 */
	bhost = findbhost(argc, argv, env);
/*
 * Open the system file.
 */
	DBUG("recon: opening hostfile %s\n", bhost);
	fp = fopen(bhost, "r");

	if (fp == 0) {
	  show_help("boot", "open-hostfile", "recon", bhost, NULL);
	  exit(errno);
	}
/*
 * Parse the system file.
 */
	if (bhostparse(fp, &lamnet, &nlamnet)) {
	  show_help("boot", "bhostparse", "recon", bhost, NULL);
	  exit(errno);
	}
	if (fl_debug) {
	  printf("recon: found the following hosts:\n");
	  for (j = 0; j < nlamnet; j++)
	    printf("recon:   n%d %s\n", j, lamnet[j].lnd_hname);
	}
/*
 * Close the system file.
 */
	if (fclose(fp)) {
	  show_help("boot", "close-hostfile", "recon", bhost, NULL);
	  exit(errno);
	}
/*
 * Locate the host nodes.
 */
	if (lamnet_findhosts(lamnet, nlamnet, &badhost)) {

		if (errno == EBADHOST) {
		  show_help("boot", "resolv-hostname", "recon",
			    lamnet[badhost].lnd_hname, bhost, NULL);
		} else {
		  char buffer[16];
		  sprintf(buffer, "%d", errno);
		  show_help("boot", "resolv-unknown", "recon", bhost,
			    buffer, NULL);
		}

		exit(errno);
	}
	DBUG("recon: found addresses for all hosts\n");
/*
 * Find the origin node.
 */
	iorigin = lamnet_findorig(lamnet, nlamnet);

	if (iorigin < 0) {
	  show_help("boot", "no-localhost", "recon", bhost, NULL);
	  exit(EINVAL);
	}

	lamnet[iorigin].lnd_type |= NT_ORIGIN | NT_ME;

	DBUG("recon: found %d host node(s)\n", nlamnet);
	DBUG("recon: origin node is n%d (%s)\n", iorigin, 
	     lamnet[iorigin].lnd_hname);
/*
 * Build the reconnaissance command.
 * We use "tkill -N" as a test.
 */
	cmdn = 0;
	cmdv = 0;
	argvadd(&cmdn, &cmdv, DEFTTKILL);
	argvadd(&cmdn, &cmdv, "-N");
/*
 * Loop over all host nodes.
 */
	global_ret = 0;

	for (i = 0; i < nlamnet; ++i) {
		VERBOSE("recon: -- testing n%d (%s)\n",
				lamnet[i].lnd_nodeid, lamnet[i].lnd_hname);
                if (fl_debug) {
		  printf("recon: attempting to launch \"");
		  for (j = 0; j < cmdn; j++) {
		    if (j > 0)
		      printf(" ");
		    printf("%s", cmdv[j]);
		  }
		  printf("\" ");
		}

		if (lamnet[i].lnd_type & NT_ORIGIN) {
 		        DBUG("(local execution)\n");
			r = _lam_few(cmdv);

			if (r) {
				errno = r;
			}
		} else {
                        DBUG("(remote execution)\n");
			r = inetexec(lamnet[i].lnd_hname,
				     lamnet[i].lnd_uname, cmdv, 
				     (fl_debug ? "recon" : NULL), 
				     fl_fast);
		}

		if (r) {
			global_ret = errno;
			fprintf(stderr, "recon: \"%s\" cannot be booted.\n",
					lamnet[i].lnd_hname);

			if (errno != EUNKNOWN) {
				terror("recon");
			} else
			  show_help(NULL, "unknown", NULL);

			if (! opt_taken('a')) {
			  success = 0;
			  break;
			}
		} 
		DBUG("recon: launch successful\n");
		  
	}
	
	if (success)
	  show_help("recon", "happiness", NULL);
	else
	  show_help("recon", "unhappiness", NULL);

	return(global_ret);
}

/*
 *	findbhost
 *
 *	Function:	- locates boot schema file
 *	Accepts:	- argc of remaining command line
 *			- argv of remaining command line
 *	Returns:	- full pathname of boot schema file
 */
static char *
findbhost(int cmdc, char **cmdv, char **env)
{
	char		*bhost;
	char		*full;		/* temporary for full pathname */
	char		**pathv;
	int		pathc;
/*
 * Set the directories for the boot schema file search.
 */
	pathc = 0;
	pathv = 0;
	argvadd(&pathc, &pathv, "");
	argvadd(&pathc, &pathv, "$TROLLIUSHOME/etc");
	argvadd(&pathc, &pathv, "$LAMHOME/etc");
	argvadd(&pathc, &pathv, LAM_SYSCONFDIR);
/*
 * Set the boot schema file names.
 */
	if (cmdc == 2) {
		bhost = cmdv[1];
	} else if ((bhost = getenv("LAMBHOST"))) {
	} else if ((bhost = getenv("TROLLIUSBHOST"))) {
	} else {
		bhost = DEFFBHOST;
	}
/*
 * Search the directories.
 */
	full = sfh_path_findv(bhost, pathv, R_OK, env);

	if (full == 0) {
	  show_help("boot", "open-hostfile", "recon", bhost, NULL);
	  exit(errno);
	}

	return(full);
}

