/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	JRV/RBD
 *
 *	$Id: allgatherv.c,v 6.4.2.1 2002/10/09 19:49:05 brbarret Exp $
 *
 *	Function:	- gather varying length buffers at all process ranks
 *	Accepts:	- send buffer
 *			- send count
 *			- send datatype
 *			- recv buffer
 *			- recv counts
 *			- displacements
 *			- recv datatype
 *			- communicator
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <lam_config.h>

/*@

MPI_Allgatherv - Gathers data from all tasks and deliver it to all

Input Parameters:
+ sbuf - starting address of send buffer (choice) 
. scount - number of elements in send buffer (integer) 
. sdtype - data type of send buffer elements (handle) 
. rcounts - integer array (of length group size) 
containing the number of elements that are received from each process 
. disps - integer array (of length group size). Entry 
 'i'  specifies the displacement (relative to recvbuf ) at
which to place the incoming data from process  'i'  
. rdtype - data type of receive buffer elements (handle) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - address of receive buffer (choice) 

Notes:
 The MPI standard (1.0 and 1.1) says that 

 "The jth block of data sent from each proess is received by every
 process and placed in the jth block of the buffer recvbuf."

 This is misleading; a better description is

 "The block of data sent from the jth process is received by every
 process and placed in the jth block of the buffer recvbuf."

 This text was suggested by Rajeev Thakur.

.N IMPI

.N fortran

.N Errors
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_BUFFER
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE

.N ACK
@*/
int MPI_Allgatherv(void *sbuf, int scount, MPI_Datatype sdtype,
		   void * rbuf, int *rcounts, int *disps, 
		   MPI_Datatype rdtype, MPI_Comm comm)
{
	int		i;			/* favourite index */
	int		size;			/* group size */
	int		err;			/* error code */

	lam_initerr();
	lam_setfunc(BLKMPIALLGATHERV);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPIALLGATHERV,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if ((sdtype == MPI_DATATYPE_NULL) || (rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPIALLGATHERV,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (disps == 0) {
		return(lam_errfunc(comm, BLKMPIALLGATHERV,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((scount < 0) || (rcounts == 0)) {
		return(lam_errfunc(comm, BLKMPIALLGATHERV,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

#if LAM_WANT_IMPI

	/* Remove this when IMPI collectives are implemented */

        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}
#endif

	LAM_TRACE(lam_tr_cffstart(BLKMPIALLGATHERV));
/*
 * Collect all values at each process, one at a time.
 */
	MPI_Comm_size(comm, &size);

	for (i = 0; i < size; ++i) {

		err = MPI_Gatherv(sbuf, scount, sdtype, rbuf,
					rcounts, disps, rdtype, i, comm);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIALLGATHERV, err));
		}
	}

	LAM_TRACE(lam_tr_cffend(BLKMPIALLGATHERV, -1, comm, sdtype, scount));

	lam_resetfunc(BLKMPIALLGATHERV);
	return(MPI_SUCCESS);
}
