/*
 * log.cc - log files handling functions
 * $Id: log.cc,v 1.9 2003/01/26 09:09:08 rdenisc Exp $
 */

/***********************************************************************
 *  Copyright (C) 2002-2003 Rmi Denis-Courmont.                       *
 *  This program is free software; you can redistribute and/or modify  *
 *  it under the terms of the GNU General Public License as published  *
 *  by the Free Software Foundation; version 2 of the license.         *
 *                                                                     *
 *  This program is distributed in the hope that it will be useful,    *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.               *
 *  See the GNU General Public License for more details.               *
 *                                                                     *
 *  You should have received a copy of the GNU General Public License  *
 *  along with this program; if not, you can get it from:              *
 *  http://www.gnu.org/copyleft/gpl.html                               *
 ***********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include "log.h"
#include "safopen.h"
#include <string.h>
#include <time.h>


/*** DataLog class implementation ***/
DataLog::~DataLog (void)
{
	if ((out != NULL) && close_out)
		fclose (out);
}


void
DataLog::Connect (const char *, const char *)
{
} 


int
DataLog::WriteServerData (const void *, int length, int)
{
	return length;
}


int
DataLog::WriteClientData (const void *, int length, int)
{
	return length;
}


void
DataLog::ShutdownServer (void)
{
}


void
DataLog::ShutdownClient (void)
{
}


/*** DataLogList implementation ***/
int
DataLogList::AddLog (DataLog *log)
{
	if (log == NULL)
		return -1;
	
	struct listnode *node;

	try
	{
		node = new listnode;
	}
	catch (...)
	{
		return -1;
	}

	node->log = log;
	node->next = head;
	head = node;
	return 0;
}


DataLogList::~DataLogList (void)
{
	for (struct listnode *node = head; node != NULL; node = head)
	{
		head = node->next;
		delete node->log;
		delete node;
	}
}


void
DataLogList::WriteServerData (const void *data, int length, int oob)
{
	for (struct listnode *node = head; node != NULL; node = node->next)
		/* FIXME: errors not handled */
		node->log->WriteServerData (data, length, oob);
}


void
DataLogList::WriteClientData (const void *data, int length, int oob)
{
	for (struct listnode *node = head; node != NULL; node = node->next)
		/* FIXME: errors not handled */
		node->log->WriteClientData (data, length, oob);
}


void
DataLogList::ShutdownServer (void)
{
	for (struct listnode *node = head; node != NULL; node = node->next)
		node->log->ShutdownServer ();
}


void
DataLogList::ShutdownClient (void)
{
	for (struct listnode *node = head; node != NULL; node = node->next)
		node->log->ShutdownClient ();
}


void
DataLogList::Connect (const char *server, const char *client)
{
	for (struct listnode *node = head; node != NULL; node = node->next)
		node->log->Connect (server, client);
}


/*** DataLogListMaker implementation ***/
int
DataLogListMaker::AddLogMaker (DataLog *(*maker) (void), FILE *stream)
{
	struct listnode *node;

	try
	{
		node = new struct listnode;
	}
	catch (...)
	{
		return -1;
	}

	node->maker = maker;
	node->next = head;
	node->basename = NULL;
	node->stream = stream;
	head = node;
	return 0;
}


int
DataLogListMaker::AddLogMaker (DataLog *(*maker) (void), const char *prefix)
{
	struct listnode *node;

	try
	{
		node = new struct listnode;
	}
	catch (...)
	{
		return -1;
	}

	node->maker = maker;
	node->next = head;

	/* FIXME: could be improved */
	if (prefix != NULL)
	{
		int len = strlen (prefix) + 1;
		char *buf;
		try
		{
			buf = new char[len];
		}
		catch (...)
		{
			delete node;
			return -1;
		}
		memcpy (buf, prefix, len);
		
		node->basename = buf;
	}
	
	node->stream = NULL;
	head = node;
	return 0;
}


DataLogListMaker::~DataLogListMaker (void)
{
	for (struct listnode *node = head; node != NULL; node = head)
	{
		head = node->next;
		if (node->basename != NULL)
			delete node->basename;
		delete node;
	}
}


static
char *valid_chars (char *str)
{
	char c;
	
	for (char *ptr = str; (c = *ptr) != 0; ptr++)
		switch (c)
		{
			case '[':
			case ']':
			case ':':
				*ptr = '-';
		}
	return str;
}


DataLogList *
DataLogListMaker::MakeLogList (const char *suffix)
{
	DataLogList *list;

	try
	{
		list = new DataLogList;
	}
	catch (...)
	{
		return NULL;
	}

	// Creates a log for each maker in the list
	for (struct listnode *node = head; node != NULL; node = node->next)
	{
		DataLog *log;
		
		try
		{
			log = node->maker ();
		}
		catch (...)
		{
			delete list;
			return NULL;
		}
		
		if (log != NULL)
		{
			// Adds a log in the list
			if (list->AddLog (log))
			{
				delete log;
				delete list;
				return NULL;
			}

			// Attaches a stream to the new log
			if (node->stream != NULL)
				log->SetStream (node->stream, 0);
			else
			{	// Do we need more abstraction?
				const char *prefix = node->basename;
				if (prefix == NULL)
					prefix = "";
				if (suffix == NULL)
					suffix = "";
				
				char *filename;
				
				int len = strlen (prefix) + strlen (suffix)+1;

				try
				{
					filename = new char[len];
				}
				catch (...)
				{
					delete list;
					return NULL;
				}
				sprintf (filename, "%s%s", prefix, suffix);
				valid_chars (filename);
				
				FILE *stream = secure_append_fopen (filename);
				delete filename;

				if (stream == NULL)
				{
					delete list;
					return NULL;
				}
				log->SetStream (stream);
			}	
		}
	}

	return list;
}

