/* victronups.c - model specific routines for Victron-UPS units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
   Copyright (C) 2000  Radek Benedikt <benedikt@lphard.cz>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX 16
#define UPS_DELAY 150000
#define UPS_LONG_DELAY 450000
#define ENDCHAR 13
#define IGNCHARS ""
#define CR 13

	int	shmok = 1, maxbattvolts = 0, cap_upstemp = 0;
	char	statefn[256];
	itype	*info;
extern	int	flag_timeoutfailure;
	float	lowvolt = 0, voltrange;
extern	int	upsfd;
	int	infomax = INFOMAX;
	int	sdwdelay = 30;	/* shutdown aftar 30 seconds */
	int	sddelay = 60;	/* wait 60 seconds for shutdown */

void initinfo (void)
{
	int	i;

	info = create_info (INFOMAX, shmok);

	/* clear out everything first */
	for (i = 0; i < INFOMAX; i++)
		info[i].type = INFO_UNUSED;

	/* number of variables - max possible */
	info[0].type = INFO_MEMBERS;
	snprintf (info[0].value, sizeof(info[0].value), "%i", INFOMAX);

	/* now set up room for all future variables that are supported */
	info[1].type = INFO_MFR;
	info[2].type = INFO_MODEL;
	info[3].type = INFO_STATUS;
	info[4].type = INFO_UTILITY;
	info[5].type = INFO_LOADPCT;
	info[6].type = INFO_UPSTEMP;
	info[7].type = INFO_BATTPCT;
	info[8].type = INFO_LOWXFER;
	info[9].type = INFO_HIGHXFER;
	info[10].type = INFO_ACFREQ;
	info[1].flags = 0;
	info[2].flags = 0;
	info[3].flags = 0;
	info[4].flags = 0;
	info[5].flags = 0;
	info[6].flags = 0;
	info[7].flags = 0;
	info[8].flags = 0;
	info[9].flags = 0;
	info[10].flags = 0;
	info[1].auxdata = 0;
	info[2].auxdata = 0;
	info[3].auxdata = 0;
	info[4].auxdata = 0;
	info[5].auxdata = 0;
	info[6].auxdata = 0;
	info[7].auxdata = 0;
	info[8].auxdata = 0;
	info[9].auxdata = 0;
	info[10].auxdata = 0;
}

char *rtrim (char *in)
{
	int	i;
	char	tmp[256];

	strncpy (tmp, in, sizeof(tmp) - 1);

	for (i = 0; i < strlen(tmp); i++)
		if (tmp [i] == ' ')
			tmp[i] = 0;

	return (strdup(tmp));
}

void openser(char *port)
{
	struct	termios	tio;
	int	fd, flags;

	fd = open(port, O_RDWR | O_NONBLOCK);

	if (fd < 0) {
		printf ("Port not available.\n");
		exit (-1);
	}

	tio.c_cflag = 0 | CREAD | CS8 | HUPCL | CLOCAL;
	tio.c_iflag = 0 | IXANY | IMAXBEL | IXOFF;
	tio.c_oflag = 0 | ONLCR;
	tio.c_lflag = 0 | ECHOE | ECHOKE | ECHOCTL | PENDIN;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, B1200);
	cfsetospeed (&tio, B1200);
#endif

	tcsetattr (fd, TCSANOW, &tio);


        ioctl (fd, TIOCMGET, &flags);
        flags &= ~(TIOCM_DTR | TIOCM_RTS) ;
        ioctl (fd, TIOCMSET, &flags);

	upsfd = fd;
}

void getbaseinfo (char *port)
{
/*	char	temp[256], mch[8], mtemp[8]; */
	int	flags;
/*	int	flags, sl; */
	char	temp[256];

	/* kill DTR and RTS */

        ioctl (upsfd, TIOCMGET, &flags);
        flags &= ~(TIOCM_DTR | TIOCM_RTS);
        ioctl (upsfd, TIOCMSET, &flags);


/* flush buffers */
	upssendchar (CR);
	usleep (UPS_LONG_DELAY);
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_LONG_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_LONG_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
/* hardcoded MFR */
	snprintf (info[1].value, sizeof(info[1].value), "Victron");
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	snprintf (info[2].value, sizeof(info[2].value), "%s", temp);
/*	snprintf (info[2].value, sizeof(info[2].value), "Match"); */
	snprintf (info[8].value, sizeof(info[8].value), "200");
	snprintf (info[9].value, sizeof(info[9].value), "300");
	return;
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	char	temp[256];
	int	tval;

	upssendchar ('v');
	upssendchar ('A');
	upssendchar ('a');
	upssendchar ('?');
	upssendchar (CR);

	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	tval = atoi (temp+3);

	strcpy (info[3].value, "");

	if (tval & 128)
		strcat (info[3].value, "OVER ");	/* overload */
	if (tval & 1)
		strcat (info[3].value, "RB ");		/* replace batt */
	if (tval & 2)
		strcat (info[3].value, "OB ");		/* on battery */
	else
		strcat (info[3].value, "OL ");		/* on line */
	if (tval & 4)
		strcat (info[3].value, "LB ");		/* low battery */

	if (info[3].value[strlen(info[3].value)-1] == ' ')
		info[3].value[strlen(info[3].value)-1] = 0;

	/* input voltage */
	upssendchar ('v');
	upssendchar ('I');
	upssendchar ('0');
	upssendchar ('U');
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	snprintf (info[4].value, sizeof(info[4].value), "%s", temp+4);

	/* ups load percent */
	upssendchar ('v');
	upssendchar ('O');
	upssendchar ('0');
	upssendchar ('L');
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	snprintf (info[5].value, sizeof(info[5].value), "%s", temp+4);

	/* ups temp */
	upssendchar ('v');
	upssendchar ('B');
	upssendchar ('T');
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	snprintf (info[6].value, sizeof(info[6].value), "%s", temp+3);

	/* battery charge */
	upssendchar ('v');
	upssendchar ('B');
	upssendchar ('C');
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	snprintf (info[7].value, sizeof(info[7].value), "%s", temp+3);

	/* input freq */
	upssendchar ('v');
	upssendchar ('I');
	upssendchar ('0');
	upssendchar ('f');
	upssendchar ('?');
	upssendchar (CR);
	usleep (UPS_DELAY);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	snprintf (info[10].value, sizeof(info[10].value), "%2.1f",
	          atof(temp+4) / 10.0);

	writeinfo(info);
}

void forceshutdown(char *port)
{
    char	temp[10];
    int		i, j;

    syslog (LOG_INFO, "Initiating UPS shutdown\n");
    printf ("Initiating forced UPS shutdown!\n");

    openser (port);

    i = sprintf (temp, "%i", sdwdelay);

    upssendchar ('v');
    upssendchar ('C');
    upssendchar ('c');
    upssendchar ('0');
    upssendchar ('!');
    upssendchar (CR);
    usleep (UPS_DELAY);

    upssendchar ('v');
    upssendchar ('C');
    upssendchar ('b');
    for (j = 0; j<i; j++)
	upssendchar (temp[j]);
    upssendchar ('!');
    upssendchar (CR);

    if (sddelay > 0) {
	sleep (sddelay+sdwdelay);
	printf ("Hmm, did the shutdown fail?  Oh well...\n");
	exit (1);
    }
    exit (0);
}

void help (char *prog)
{
    printf ("usage: %s [-h] [-s <num>] [-d <num>] [-k] <device>\n", prog);
    printf ("\n");
    printf ("-s <num> - shutdown after <num> second (default 30)\n");
    printf ("-d <num> - wait <num> seconds after sending shutdown command (default 60)\n");
    printf ("-h       - display this help\n");
    printf ("-k       - force shutdown\n");
    printf ("<device> - /dev entry corresponding to UPS port\n");
}

void usage(char *prog)
{
    printf ("usage: %s <portname>\n", prog);
    printf ("Example: %s /dev/ttyS0\n", prog);
    exit (1);
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	i;

	printf ("Network UPS Tools - Victron-UPS driver 0.10 (%s)\n", UPS_VERSION);
	openlog ("victronups", LOG_PID, LOG_FACILITY);

	prog = argv[0];

	while ((i = getopt(argc, argv, "+s:+d:hk:")) != EOF) {
		switch (i) {
			case 's':
				sdwdelay = atoi(optarg);
				break;
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				forceshutdown(optarg);
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/victronups-%s", STATEPATH,
	          portname);

	openser(argv[0]);

	initinfo();

	createmsgq();		/* try to create IPC message queue */

	getbaseinfo(argv[0]);

 	background();

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			syslog (LOG_INFO, "Received a message from upsd\n");
	}
}
