/* upscommon.c - functions used in more than one model support module

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/termios.h>
#include <sys/types.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"

#ifdef HAVE_SYS_SHM_H
#include <sys/ipc.h>
#include <sys/shm.h>
#else
#define shmget(a,b,c) (-1)
#define shmctl(a,b,c) /* nop */
struct shmid_ds {
        void *junk;
};
#endif

#ifdef HAVE_UU_LOCK
#include <libutil.h>
#endif

#ifdef HAVE_MSGGET
#include <sys/ipc.h>
#include <sys/msg.h>
#endif

	int	upsfd, shmid = -1, msgid = -1, upsc_debug = 0;
	char	statefn[256], *upsport;
extern	itype	*info;
struct	ups_handler	upsh;
extern	int	infomax;
	
	int	flag_timeoutfailure = 0;

/* signal handler for SIGALRM when opening a serial port */
void openfail(int sig)
{
	printf ("Fatal error: serial port open timed out\n");
	exit (1);
}

/* try whatever method(s) are available to lock the port for our use */
void lockport (int upsfd, char *port)
{
	int	res, i;
	char	*portname;

	/* keep gcc quiet */
	res = i = 0;
	portname = NULL;

#ifdef HAVE_UU_LOCK
	if (upsport)		/* already locked? */
		return;

	for (i = strlen (port); i >= 0; i--)
		if (port[i] == '/') {
			portname = &port[i+1];
			break;
		}
		
	/* save for later in case we need it at shutdown */
	upsport = strdup (portname);

	res = uu_lock(upsport);

	if (res != 0) {
		printf ("Can't uu_lock %s: %s\n", upsport, uu_lockerr(res));
		exit (1);
	}
#else			/* no uu_lock */
#ifdef HAVE_FLOCK
	if (flock(upsfd, LOCK_EX | LOCK_NB) != 0) {
		printf ("%s is locked by another process\n", port);
		exit (1);
	}
#endif /* HAVE_FLOCK */
#endif /* HAVE_UU_LOCK */
}

/* open a serial port and setup the flags properly at a given speed */
void open_serial(char *port, speed_t speed)
{
	struct	termios	tio;

	signal (SIGALRM, openfail);
	alarm (3);

	/* there's a 'double open' here since this unsticks the port for us */

	upsfd = open (port, O_RDWR | O_NOCTTY | O_NONBLOCK | O_EXCL);
	alarm (0);

	if (upsfd < 0) {
		printf ("Unable to open (1) %s: %s\n", port, strerror(errno));
		exit (1);
	}

	upsport = NULL;
	lockport (upsfd, port);

	tcgetattr (upsfd, &tio);
	tio.c_lflag = 0 | ECHOE | ECHOKE | ECHOCTL | PENDIN;
	tio.c_iflag = 0 | IXANY | IMAXBEL | IXOFF;
	tio.c_oflag = 0 | ONLCR;
	tio.c_cflag = 0 | CREAD | CS8 | HUPCL | CLOCAL;

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, speed);
	cfsetospeed (&tio, speed);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;
	tcflush (upsfd, TCIFLUSH);
	tcsetattr(upsfd, TCSANOW, &tio);
	close (upsfd);

	/* reopen for normal use */

	signal (SIGALRM, openfail);
	alarm (3);

	upsfd = open (port, O_RDWR | O_NOCTTY | O_EXCL);
	alarm (0);

	if (upsfd < 0) {
		printf ("Unable to open (2) %s: %s\n", port, strerror(errno));
		exit (1);
	}

	signal (SIGALRM, SIG_IGN);

	lockport (upsfd, port);

	tcgetattr (upsfd, &tio);
	tio.c_cflag = CS8 | CLOCAL | CREAD;
	tio.c_iflag = IGNPAR;
	tio.c_oflag = 0;
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, speed);
	cfsetospeed (&tio, speed);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	tcflush (upsfd, TCIFLUSH);
	tcsetattr (upsfd, TCSANOW, &tio);
}

/* put a notice in the syslog */
void notice (char *msg)
{
	char	logbuf[256];

	snprintf (logbuf, sizeof(logbuf), "Notice: %s", msg);
	syslog (LOG_NOTICE, msg);
}

/* put a fatal error in the syslog and exit */
void fatal (char *msg)
{
	char	logbuf[256];

	snprintf (logbuf, sizeof(logbuf), "Fatal error: %s", msg);
	syslog (LOG_ERR, msg);
	exit (1);
}

/* function for erasing "timeout"-conditions */
void nolongertimeout(void)
{
	static int flag_firstcall = 0;

	/* if override enabled, then return without changing anything */
	if (flag_timeoutfailure == -1)
		return;

	if (flag_firstcall == 0)
		flag_firstcall = 1;
	
	if (flag_timeoutfailure == 1)
		notice ("Serial port read ok again");
	
	flag_timeoutfailure = 0;
	return;
}

/* signal handler for SIGALRM when trying to read */
void timeout(int sig)
{
	struct sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = SIG_DFL;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	/* if override enabled, then return without changing anything */
	if (flag_timeoutfailure == -1)
		return;

	if (flag_timeoutfailure == 0)
		notice ("Serial port read timed out");
	
	flag_timeoutfailure = 1;
	return;
}

/* wait for an answer in the form <data><endchar> and store in buf */
int upsrecv (char *buf, int buflen, char endchar, char *ignchars)
{
	char	recvbuf[512], *ptr, in;
	int	ret, cnt;
	struct 	sigaction sa;
	sigset_t sigmask;

	strcpy (recvbuf, "");

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (3);

	ptr = recvbuf;
	*ptr = 0;
	cnt = 0;
	for (;;) {
		ret = read (upsfd, &in, 1);
		if (ret > 0) {

			if (in == endchar) {
				alarm (0);
				signal (SIGALRM, SIG_IGN);
				strncpy (buf, recvbuf, buflen);
				buf[buflen - 1] = 0;
				if (upsc_debug > 0) {
					int i;
					printf ("upsrecv: read %d bytes [", cnt);
					for (i = 0; i < cnt; ++i)
						printf(isprint(buf[i]) ? "%c" : "\\%03o",
							   buf[i]);
					printf ("]\n");
				}
				return (buflen);
			}

			if (strchr(ignchars, in) == NULL) {
				*ptr++ = in;
				*ptr = 0; /* tie off end */
				cnt++;
			}
			
			nolongertimeout();
		}
		else {
			alarm (0);
			signal (SIGALRM, SIG_IGN);
			strncpy (buf, recvbuf, buflen);
			return (-1);
		}

		/* keep from overrunning the buffer - lame hack */
		if (cnt > (sizeof(recvbuf) - 4)) {
			notice ("UPS is spewing wildly");
			return(-1);
		}
	}

	return (-1);	/* not reached */
}

/* send a single byte to the UPS */
int upssendchar (char data)
{
	if (upsc_debug > 0) {
		printf ("upssendchar: sending [");
		printf (isprint(data) ? "%c" : "\\%03o", data);
		printf ("]\n");
	}

	tcflush (upsfd, TCIFLUSH);
	return (write (upsfd, &data, 1));
}

/* get data from the UPS and install it in the data array */
void installinfo (int infotype, char reqchar, char endchar, char *ignchars)
{
	int	i, pos = 0;

	for (i = 0; i < infomax; i++)
		if (info[i].type == infotype) {
			pos = i;
			break;
		}

	if (pos == 0)		/* not found, probably not supported */
		return;

	upssendchar (reqchar);
	upsrecv (info[pos].value, sizeof(info[pos].value), endchar, ignchars);
}

/* store data into the array */
void setinfo (int infotype, char *data)
{
	int	i, pos = 0;

	for (i = 0; i < infomax; i++)
		if (info[i].type == infotype) {
			pos = i;
			break;
		}

	if (pos == 0) {	/* not found, probably debugging? */
		syslog (LOG_ERR, "setinfo: can't find type %i\n", infotype);
		return;
	}

	snprintf (info[pos].value, sizeof(info[pos].value), "%s", data);
}

/* set a flag on an existing member of the array */
void setflag (int infotype, int newflags)
{
	int	i, pos = 0;

	for (i = 0; i < infomax; i++)
		if (info[i].type == infotype) {
			pos = i;
			break;
		}

	if (pos == 0) {	/* not found, probably debugging? */
		syslog (LOG_ERR, "setflag: can't find type %i\n", infotype);
		return;
	}

	info[pos].flags = newflags;
}

/* find data of a given type in the info array */
char *getdata (int infotype)
{
	int	i;

	for (i = 0; i < infomax; i++)
		if (info[i].type == infotype)
			return (info[i].value);
	
	return (NULL);
}

/* write data to state file - effectively a no-op for shared memory mode */
void writeinfo (itype *iwrite)
{
	int	sffd, ret;
	struct	shmid_ds shmbuf;

	/* if data is stale, don't write it so the information ages */
	if (flag_timeoutfailure == 1)
		return;

	/* if in shm mode and not writing shm struct, exit immediately */
	if ((iwrite[0].type != INFO_SHMID) && (shmid >= 0)) {

		/* this updates the ctime that upsd wants to see */
		shmctl (shmid, IPC_STAT, &shmbuf);
		shmctl (shmid, IPC_SET, &shmbuf);
		return;
	}

	sffd = open (statefn, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP);
	if (sffd >= 0) {
		int	writelen;

		/* only write 1 record when doing the SHM pointer file */
		if (iwrite[0].type == INFO_SHMID)
			writelen = sizeof(itype);
		else
			writelen = sizeof(itype) * infomax;

		ret = write (sffd, iwrite, writelen);
		if (ret >= 0) {
			close (sffd);
			return;
		}
	}
	/* ssfd < 0 || ret < 0 */
	{
		char buf[512];
		snprintf (buf, sizeof(buf),
			  "Can't open %s: %s\n", statefn, strerror(errno));
		buf[sizeof(buf)-1] = '\0';
		fatal(buf);
	}
}

#ifdef HAVE_SHMAT
static void test_writeinfo(void) 
{
        int sffd = open (statefn, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP);
        int tmperr;

        /* tests a write to the file only so the user
	   can see if something is wrong before it enters the background */

	if (sffd < 0) {
		tmperr = errno;
		printf ("Can't open %s: %s\n", statefn, strerror(tmperr));
		if (tmperr == EACCES)
			printf ("This file has to be accessible by user %s\n",
                    RUN_AS_USER);
		exit (1);
	}
	close (sffd);
}

void cleanup_shm (void)
{
	if (shmid == -1)
		return;

	/* mark for deletion after total detachment */
	shmctl (shmid, IPC_RMID, NULL);                

	shmdt ((char *) info);	/* detach */

	/* clean up msg queue if necessary */
#ifdef HAVE_MSGGET
	if (msgid != -1)
		msgctl (msgid, IPC_RMID, NULL);
#endif	/* HAVE_MSGGET */

}
#endif	/* HAVE_SHMAT */

/* handler for SIGTERM */
void sigterm(int sig)
{
	syslog (LOG_INFO, "SIGTERM: Shutting down");
#ifdef HAVE_UU_LOCK
	uu_unlock (upsport);
#endif
	exit (0);
}

/* install handler for sigterm */
void catch_sigterm(void)
{
	struct sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = sigterm;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGTERM, &sa, NULL);
}

/* add cleanup handlers for SIGTERM and the like */
void addcleanup(void)
{
	catch_sigterm();

#ifdef HAVE_SHMAT
	/* safe to add this now... */
	atexit (cleanup_shm);
#endif
}

itype *create_info (int numinfo, int shmok)
{
	itype	*tmpinfo = NULL;
	int	i;

#ifdef HAVE_SHMAT
	key_t	shmkey = IPC_PRIVATE;
	itype	shminfo[1];

        test_writeinfo();

/* some systems (NetBSD) don't define SHM_R and SHM_W - they default to it */
#ifdef SHM_R
#define IPC_MODE IPC_CREAT|SHM_R|SHM_W|S_IRGRP|S_IWGRP
#else
#define IPC_MODE IPC_CREAT
#endif
        
	if (shmok == 1) {
		/* get some shared memory for the array */
		shmid = shmget (shmkey, sizeof(itype) * numinfo, 
		                IPC_MODE);

		if (shmid == -1) {
			perror ("shmget");
			shmid = -1;
		}
		else {		/* got a good ID, now attach to it */
			tmpinfo = (itype *) shmat (shmid, 0, 0);

			if (tmpinfo == (itype *) (-1)) {
				perror ("shmat");
				shmid = -1;
			}
		}

		if (shmid != -1) {
			/* create state file with SHM id */
			shminfo[0].type = INFO_SHMID;
			snprintf (shminfo[0].value, sizeof(shminfo[0].value), 
			          "%i", shmid);
		
			writeinfo (shminfo);
		}
	}	/* if shmok == 1 */

#endif	/* HAVE_SHMAT */

	if (shmid == -1) {	/* shm failed or is disabled - fallback */
		tmpinfo = calloc (numinfo, sizeof(itype));
	}

	/* initialize the array */
	for (i = 0; i < infomax; i++)
		tmpinfo[i].type = INFO_UNUSED;

	tmpinfo[0].type = INFO_MEMBERS;
	snprintf (tmpinfo[0].value, VALSIZE, "%i", infomax);

	return (tmpinfo);
}

/* add a member to the info struct */
void addinfo (int type, char *value, int flags, int auxdata)
{
	int	i;

	for (i = 0; i < infomax; i++) {
		if (info[i].type == INFO_UNUSED) {	/* found open spot */
			info[i].type = type;
			snprintf (info[i].value, sizeof(info[i].value), "%s", 
			          value);
			info[i].flags = flags;
			info[i].auxdata = auxdata;
			return;
		}
	}

	syslog (LOG_ERR, "Unable to add new member to info array!\n");

	return;
}

/* add a new ENUM info entry and do other related housekeeping */
void addenum (int basetype, char *value)
{
	int	i;

	/* first find the basetype in the struct */
	for (i = 0; i < infomax; i++)
		if (info[i].type == basetype) {

			/* add the enum value, then increment the enum count */
			addinfo (INFO_ENUM, value, 0, basetype);
			info[i].auxdata++;
			return;
		}

	syslog (LOG_ERR, "Unable to add enumerated type for base 0x%04x\n",
	        basetype);

	return;
}

#ifndef HAVE_MSGGET

/* stubs */
void createmsgq(void) {}
int getupsmsg(int wait) { return 0; }
void sendupsmsg(msgtype *buf) {}

#else

/* create a SysV IPC message queue */

void msgget_fail(int signal)
{
	syslog (LOG_ERR, "Can't create message queue! (SIGSYS)\n");
	return;
}

void createmsgq(void)
{
	char	tempid[16];

	/* init handlers early */
	memset (&upsh, '\0', sizeof(upsh));

	/* FreeBSD throws SIGSYS if sysv messaging isn't in the kernel */
#ifdef SIGSYS
	signal (SIGSYS, msgget_fail);
#endif
	msgid = msgget (IPC_PRIVATE, IPC_CREAT | 0660);
#ifdef SIGSYS
	signal (SIGSYS, SIG_IGN);
#endif

	if (msgid == -1) {
		perror ("msgget");
		return;
	}

	snprintf (tempid, sizeof(tempid), "%i", msgid);
	addinfo (INFO_MSGID, tempid, 0, 0);
}

/* handler for when msgrcv times out */
void nomsg(int sig)
{
	return;
}

/* make our own struct since some systems don't have struct msgbuf */
typedef struct {
	long    mtype;
	msgtype	msg;
}       mbuf;           

/* get a message if one's waiting in the queue */
int getupsmsg (int wait)
{
	mbuf	*mb;
	int	ret;

	if (msgid == -1)	/* no queue was created */
		return (0);	/* no msg received */

	mb = malloc (sizeof(mbuf) + UPSMSG_MAXLEN);

	/* set up alarm handler and schedule an alarm */
	signal (SIGALRM, nomsg);
	alarm (wait);

	ret = msgrcv (msgid, (struct msgbuf *) mb, UPSMSG_MAXLEN, UPSMSG_TOMODEL, 0);

	/* cancel the alarm */
	alarm (0);
	signal (SIGALRM, SIG_IGN);

	if (ret == -1) {
		free (mb);
		return (0);	/* no msg received */
	}

	/* now parse the message and deal with it */
	switch (mb->msg.cmdtype) {
		case UPSMSG_CMDSET:
			if (upsh.setvar)
				upsh.setvar(mb->msg.auxcmd, mb->msg.dlen,
				            &mb->msg.data);
			break;
		case UPSMSG_INSTCMD:
			if (upsh.instcmd)
				upsh.instcmd(mb->msg.auxcmd, mb->msg.dlen,
				             &mb->msg.data);
			break;
		default: 
			syslog (LOG_INFO, "Unknown msgcmd 0x%04x\n",
			        mb->msg.cmdtype);
	}		

	free (mb);
	return (1);	/* msg received */
}

/* send a message to the upsd process */
void sendupsmsg (msgtype *buf)
{
	mbuf	*mb;
	int	ret;

	if (msgid == -1)	/* no queue was created */
		return;

	mb = malloc (sizeof(mbuf) + UPSMSG_MAXLEN);

	mb->mtype = UPSMSG_TOUPSD;
	memcpy (&mb->msg, buf, UPSMSG_MAXLEN);

	ret = msgsnd (msgid, (struct msgbuf *) mb, UPSMSG_MAXLEN, 0);
	free (mb);
}

#endif	/* HAVE_MSGGET */

void msgreply (int reptype)
{
	msgtype	msg;

	msg.cmdtype = reptype;
	msg.auxcmd = 0;
	msg.dlen = 0;
	msg.data = '\0';

	sendupsmsg (&msg);
}
