/* upsmon - monitor power status over the 'net (talks to upsd via UDP)

   Copyright (C) 1998  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
 */

#include <errno.h>
#include <fcntl.h>                            
#include <stdio.h>
#include <signal.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>

#include "config.h"
#include "proto.h"
#include "upsfetch.h"
#include "version.h"
#include "common.h"
#include "upsmon.h"

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

	char	*shutdowncmd = NULL, *notifycmd = NULL;
	char	*powerdownflag = NULL;

	int	minsupplies = 1, upscount = 0, sleepval = 5, deadtime = 15;

	/* default polling interval = 5 sec */
	int	pollfreq = 5, pollfreqalert = 5;

	/* default alert time = 30 sec */
	int	alerttime = 30;

	/* slave hosts are given 15 sec by default to logout from upsd */
	int	hostsync = 15;  

	/* sum of all power values from config file */
	int	totalpv = 0;

	/* should we go into an infinite loop upon shutdown? */
	int	playdead = 0;

	/* default replace battery warning interval (seconds) */
	int	rbwarntime = 43200;

	int	debuglevel = 0;

	utype	*firstups = NULL;

	/* signal handling things */
	struct sigaction sa;
	sigset_t sigmask;

void debug (char *format, ...)
{
#ifdef HAVE_STDARG_H
	va_list	args;

	if (debuglevel < 1)
		return;

	va_start (args, format);
	vprintf (format, args);
	va_end (args);
#endif

	return;
}	

void setflag (int *val, int flag)
{
	*val = (*val |= flag);
}
        
void clearflag (int *val, int flag)  
{
	*val = (*val ^= (*val & flag));
}

int isset (int num, int flag)
{
	return ((num & flag) == flag);
}

void wall(char *text)
{
	char	exec[LARGEBUF];

	snprintf (exec, sizeof(exec), "echo \"%s\" | wall", text);
	system (exec);
} 

void notify (char *notice, int flags, char *ntype, char *upsname)
{
	char	exec[LARGEBUF];

	if (isset(flags, NOTIFY_SYSLOG))
		syslog (LOG_NOTICE, notice);

	if (isset(flags, NOTIFY_WALL))
		wall (notice);

	if (isset(flags, NOTIFY_EXEC)) {
		if (notifycmd != NULL) {
			snprintf (exec, sizeof(exec), "%s \"%s\"", notifycmd, notice);

			if (upsname)
				setenv ("UPSNAME", upsname, 1);
			else
				setenv ("UPSNAME", "", 1);

			setenv ("NOTIFYTYPE", ntype, 1);
			system (exec);
		}
	}
}

void do_notify (utype *ups, int ntype)
{
	int	i;
	char	msg[SMALLBUF], *upsname = NULL;

	/* grab this for later */
	if (ups)
		upsname = ups->sys;

	for (i = 0; notifylist[i].name != NULL; i++) {
		if (notifylist[i].type == ntype) {

			if (ups)
				snprintf (msg, sizeof(msg), notifylist[i].msg,
				          ups->sys);
			else
				snprintf (msg, sizeof(msg), notifylist[i].msg);

			notify (msg, notifylist[i].flags, notifylist[i].name,
			        upsname);
			return;
		}
	}

	/* not found ?! */
}

/* check for master permissions on the server for this ups */
void checkmaster(utype *ups)
{
	char	buf[SMALLBUF];

	/* don't bother if we're not configured as a master for this ups */
	if (!isset(ups->status, ST_MASTER))
		return;

	snprintf (buf, sizeof(buf), "MASTER %s\n", ups->upsname);
	upssendraw (ups->fd, buf);

	if (upsreadraw (ups->fd, buf, sizeof(buf)) > 0) {
		if (!strncmp(buf, "OK", 2))
			return;

		/* not ERR, but not caught by readraw either? */

		syslog (LOG_ALERT, "Master privileges unavailable on UPS [%s]",
		        ups->sys);
		syslog (LOG_ALERT, "Response: [%s]", buf);
	}
	else {	/* something caught by readraw's parsing call */
		syslog (LOG_ALERT, "Master privileges unavailable on UPS [%s]\n",
		        ups->sys);
		syslog (LOG_ALERT, "Reason: %s\n", upsstrerror(upserror));
	}
}

/* set flags and make announcements when a UPS has been checked successfully */
void heardups(utype *ups)
{
	char	logbuf[SMALLBUF];
	time_t	now;

	time (&now);
	ups->lastpoll = now;

	if (isset(ups->status, ST_ALIVE))
		return;

	setflag (&ups->status, ST_ALIVE);

	/* only notify on subsequent reconnects */
	if (!isset(ups->status, ST_FIRST))
		setflag (&ups->status, ST_FIRST);
	else
		do_notify (ups, NOTIFY_COMMOK);

	if (ups->pv == 0)	/* monitor only, no need to login */
		return;

	/* not logged in?  then do it */
	if (!isset(ups->status, ST_LOGIN)) {
		if (upslogin (ups->fd, ups->upsname, ups->pw) == 0) {
			debug ("Logged into UPS %s\n", ups->sys);
			setflag (&ups->status, ST_LOGIN);

			checkmaster (ups);
		}
		else {
			snprintf (logbuf, sizeof(logbuf), "Unable to login to %s: %s\n",
			          ups->sys, upsstrerror(upserror));
			syslog (LOG_ERR, logbuf);
			return;
		}
	}
}

void upsgone(utype *ups)
{
	if (isset(ups->status, ST_ALIVE)) {	/* ups is gone, first time */
		syslog (LOG_INFO, "Communication with UPS %s lost", ups->sys);
		close (ups->fd);
		ups->fd = -1;

		/* don't clear status flags since we may use them later */
		clearflag (&ups->status, ST_ALIVE);
		clearflag (&ups->status, ST_LOGIN);

		do_notify (ups, NOTIFY_COMMBAD);
	}
}

void upsonbatt(utype *ups)
{
	sleepval = pollfreqalert;	/* bump up polling frequency */

	if (!isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag (&ups->status, ST_ONBATT);
		clearflag (&ups->status, ST_ONLINE);
		return;
	}

	heardups(ups);

	if (isset(ups->status, ST_ONBATT)) 	/* no change */
		return;

	/* must have changed from OL to OB, so notify */

	do_notify (ups, NOTIFY_ONBATT);
	setflag (&ups->status, ST_ONBATT);
	clearflag (&ups->status, ST_ONLINE);
}

void upsonline(utype *ups)
{
	if (!isset(ups->status, ST_ALIVE)) {
		debug ("upsonline: %s (alive out)\n", ups->sys);
		heardups(ups);
		setflag (&ups->status, ST_ONLINE);
		clearflag (&ups->status, ST_ONBATT);
		return;
	}

	debug ("upsonline: %s (normal)\n", ups->sys);

	heardups(ups);

	if (isset(ups->status, ST_ONLINE)) 	/* no change */
		return;

	/* must have changed from OB to OL, so notify */

	do_notify (ups, NOTIFY_ONLINE);
	setflag (&ups->status, ST_ONLINE);
	clearflag (&ups->status, ST_ONBATT);
}

/* the actual shutdown procedure */
void doshutdown()
{
	int	ret;

	syslog (LOG_CRIT, "Executing automatic power-fail shutdown");
	wall ("Executing automatic power-fail shutdown\n");

	if (getuid() != 0) {
		wall ("Not root, unable to shutdown system\n");
		syslog (LOG_ALERT, "Not root, unable to shutdown system");
		exit (1);
	}

	do_notify (NULL, NOTIFY_SHUTDOWN);

	sleep (15);	/* TODO: allow configuring this too */

	ret = system (shutdowncmd);

	if (ret != 0)
		syslog (LOG_ERR, "Unable to call shutdown command: %s\n",
		        shutdowncmd);

	/* if instructed to go into the infinite loop, then do so */
	if (playdead == 1)
		for (;;)
			sleep (100);

	/* hopefully not reached */
	exit (1);
}

/* create the flag file if necessary */
void setpdflag()
{
	FILE	*pdf;

	if (powerdownflag != NULL) {
		pdf = fopen (powerdownflag, "w");
		if (!pdf) {
			syslog (LOG_ERR, "Failed to create power down flag!");
			return;
		}
		fprintf (pdf, SDMAGIC);
		fclose (pdf);
	}
}

/* set forced shutdown flag so other upsmons know what's going on here */
void setfsd (utype *ups)
{
	char	buf[SMALLBUF];

	debug ("Setting FSD on UPS %s (fd %d)\n", ups->sys, ups->fd);

	snprintf (buf, sizeof(buf), "FSD %s\n", ups->upsname);
	upssendraw (ups->fd, buf);

	upsreadraw (ups->fd, buf, sizeof(buf));
}

void slavesync (void)
{
	utype	*ups;
	char	temp[SMALLBUF];
	time_t	start, now;
	int	maxlogins, logins;

	time (&start);

	for (;;) {
		maxlogins = 0;

		for (ups = firstups; ups != NULL; ups = ups->next) {
			if (getupsvarfd(ups->fd, ups->upsname, "numlogins", 
			                temp, sizeof(temp)) >= 0) {
				logins = strtol(temp, (char **)NULL, 10);

				if (logins > maxlogins)
					maxlogins = logins;
			}
		}

		/* if no UPS has more than 1 login (us), then slaves are gone */
		if (maxlogins <= 1)
			return;

		/* after HOSTSYNC seconds, assume slaves are stuck and bail */
		time (&now);

		if ((now - start) > hostsync) {
			syslog (LOG_INFO, "Host sync timer expired, forcing shutdown");
			return;
		}

		usleep (250000);
	}
}

void forceshutdown (void)
{
	utype	*ups;
	int	isamaster = 0;

	debug ("Shutting down any UPSes in MASTER mode...\n");

	/* set FSD on any "master" UPS entries (forced shutdown in progress) */
	for (ups = firstups; ups != NULL; ups = ups->next)
		if (isset(ups->status, ST_MASTER)) {
			isamaster = 1;
			setfsd (ups);
		}

	/* if we're not a master on anything, we should shut down now */
	if (!isamaster)
		doshutdown();

	/* must be the master now */
	debug ("This system is a master... waiting for slave logout...\n");

	/* wait up to HOSTSYNC seconds for slaves to logout */
	slavesync();

	/* set power down flag for the shutdown script to call <model> -k */
	setpdflag();

	/* time expired or all the slaves are gone, so shutdown */
	doshutdown();
}

/* recalculate the online power value and see if things are still OK */
void recalc (void)
{
	utype	*ups;
	int	val_ol = 0;
	time_t	now;

	time (&now);
	ups = firstups;
	while (ups != NULL) {
		/* promote dead UPSes that were last known OB to OB+LB */
		if ((now - ups->lastpoll) > deadtime)
			if (isset(ups->status, ST_ONBATT)) {
				debug ("Promoting dead UPS: %s\n", ups->sys);
				setflag (&ups->status, ST_LOWBATT);
			}

		/* note: we assume that a UPS that isn't critical must be OK *
		 *							     *
		 * this means a UPS we've never heard from is assumed OL     *
		 * whether this is really the best thing to do is undecided  */

		/* crit = (OB & LB) || (FSD) */
		if (((isset(ups->status, ST_ONBATT)) && 
		     (isset(ups->status, ST_LOWBATT))) ||
		     (isset(ups->status, ST_FSD)))
			debug ("Critical UPS: %s\n", ups->sys);
		else
			val_ol += ups->pv;

		ups = ups->next;
	}

	/* debug ("Current power value: %d\n", val_ol);
	debug ("Minimum power value: %d\n", minsupplies); */

	if (val_ol < minsupplies)
		forceshutdown();
}		

void upslowbatt (utype *ups)
{
	if (!isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag (&ups->status, ST_LOWBATT);
		return;
	}

	heardups(ups);

	if (isset(ups->status, ST_LOWBATT)) 	/* no change */
		return;

	/* must have changed from !LB to LB, so notify */

	do_notify (ups, NOTIFY_LOWBATT);
	setflag (&ups->status, ST_LOWBATT);
}

void upsreplbatt (utype *ups)
{
	time_t	now;

	time (&now);

	if ((now - ups->lastrbwarn) > rbwarntime) {
		do_notify (ups, NOTIFY_REPLBATT);
		ups->lastrbwarn = now;
	}
}

void upsfsd (utype *ups)
{
	if (!isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag (&ups->status, ST_FSD);
		return;
	}

	heardups(ups);

	if (isset(ups->status, ST_FSD)) 	/* no change */
		return;

	/* must have changed from !FSD to FSD, so notify */

	do_notify (ups, NOTIFY_FSD);
	setflag (&ups->status, ST_FSD);

	/* TODO: if we're a master, log something about it */
}

void addups (char *sys, char *pv, char *pw, char *master)
{
	utype	*tmp, *last;
	char	*upsname, *ptr;

	if ((!sys) || (!pv) || (!pw) || (!master)) {
		printf ("Ignoring invalid MONITOR line in upsmon.conf!\n");
		printf ("MONITOR configuration directives require four arguments.\n");
		return;
	}

	last = tmp = firstups;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof(utype));
	tmp->fd = -1;
	tmp->sys = strdup (sys);

	/* parse out <upsname>@<host> for later */
	upsname = strdup (sys);
	ptr = strchr (upsname, '@');
	if (ptr) {
		*ptr = '\0';
		tmp->upsname = strdup (upsname);
		tmp->host = strdup (++ptr);
	}
	else {
		tmp->upsname = NULL;
		tmp->host = strdup (sys);
	}

	free (upsname);

	tmp->pv = atoi (pv);
	totalpv += tmp->pv;

	tmp->pw = strdup (pw);

	tmp->status = 0;
	tmp->lastpoll = 0;
	tmp->lastrbwarn = 0;

	if (!strcasecmp(master, "master"))
		setflag (&tmp->status, ST_MASTER);

	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		firstups = tmp;

	if (tmp->pv)
		printf ("UPS: %s (%s) (power value %d)\n", tmp->sys, 
		        isset(tmp->status, ST_MASTER) ? "master" : "slave",
			tmp->pv);
	else
		printf ("UPS: %s (monitoring only)\n", tmp->sys);
	upscount++;

	tmp->fd = upsconnect (tmp->host);
}		

void set_notifymsg (char *name, char *msg)
{
	int	i;

	for (i = 0; notifylist[i].name != NULL; i++) {
		if (!strcasecmp(notifylist[i].name, name)) {
			notifylist[i].msg = strdup (msg);
			return;
		}
	}

	printf ("'%s' is not a valid notify event name\n", name);
}

void set_notifyflag (char *ntype, char *flags)
{
	int	i, pos;
	char	*ptr, *tmp;

	/* find ntype */

	pos = -1;
	for (i = 0; notifylist[i].name != NULL; i++) {
		if (!strcasecmp(notifylist[i].name, ntype)) {
			pos = i;
			break;
		}
	}

	if (pos == -1) {
		printf ("Error: invalid notify type [%s]\n", ntype);
		return;
	}

	ptr = flags;

	/* zero existing flags */
	notifylist[pos].flags = 0;

	while (ptr) {
		int	newflag;

		tmp = strchr (ptr, '+');
		if (tmp)
			*tmp++ = '\0';

		newflag = 0;

		if (!strcmp(ptr, "SYSLOG"))
			newflag = NOTIFY_SYSLOG;
		if (!strcmp(ptr, "WALL"))
			newflag = NOTIFY_WALL;
		if (!strcmp(ptr, "EXEC"))
			newflag = NOTIFY_EXEC;
		if (!strcmp(ptr, "IGNORE"))
			newflag = NOTIFY_IGNORE;

		if (newflag)
			notifylist[i].flags |= newflag;
		else
			printf ("Invalid notify flag: [%s]\n", ptr);

		ptr = tmp;
	}
}

void loadconfig (void)
{
	char	cfn[SMALLBUF], buf[SMALLBUF], *arg[5];
	FILE	*conf;
	int	i, ln;

	snprintf (cfn, sizeof(cfn), "%s/upsmon.conf", CONFPATH);

	conf = fopen(cfn, "r");
	if (conf == NULL) {
		printf ("Can't open %s/upsmon.conf: %s\n", CONFPATH, 
		        strerror(errno));
		exit (1);
	}

	ln = 0;
	while (fgets(buf, sizeof(buf), conf)) {
		buf[strlen(buf) - 1] = '\0';
		ln++;

		i = parseconf ("upsmon.conf", ln, buf, arg, 5);

		if (i == 0)
			continue;

		if (!strcmp(arg[0], "SHUTDOWNCMD"))
			shutdowncmd = strdup (arg[1]);

		if (!strcmp(arg[0], "NOTIFYCMD"))
			notifycmd = strdup (arg[1]);

		if (!strcmp(arg[0], "POWERDOWNFLAG")) {
			powerdownflag = strdup (arg[1]);
			printf ("Using power down flag file %s\n", powerdownflag);
		}

		if (!strcmp(arg[0], "POLLFREQ "))
			pollfreq = atoi(arg[1]);

		if (!strcmp(arg[0], "POLLFREQALERT"))
			pollfreqalert = atoi(arg[1]);

		if (!strcmp(arg[0], "ALERTTIME"))
			alerttime = atoi(arg[1]);

		if (!strcmp(arg[0], "HOSTSYNC"))
			hostsync = atoi(arg[1]);

		/* MONITOR <system> <powervalue> <password> ("master"|"slave") */
		if (!strcmp(arg[0], "MONITOR"))
			addups (arg[1], arg[2], arg[3], arg[4]);

		if (!strcmp(arg[0], "MINSUPPLIES"))
			minsupplies = atoi (arg[1]);

		if (!strcmp(arg[0], "DEADTIME"))
			deadtime = atoi(arg[1]);

		/* NOTIFYMSG <msgname> "<replacement message>" */
		if (!strcmp(arg[0], "NOTIFYMSG"))
			set_notifymsg (arg[1], arg[2]);

		/* NOTIFYFLAG <notify type> <flags> */
		if (!strcmp(arg[0], "NOTIFYFLAG"))
			set_notifyflag (arg[1], arg[2]);

		/* RBWARNTIME <secs> */
		if (!strcmp(arg[0], "RBWARNTIME"))
			rbwarntime = atoi(arg[1]);

	}
	
	fclose (conf);
}

/* SIGPIPE handler */
void sigpipe (int sig)
{
	debug ("SIGPIPE: dazed and confused, but continuing...\n");
}

/* SIGQUIT, SIGTERM handler */
void shut_myself_down (int sig)
{
	utype	*ups;

	syslog (LOG_INFO, "Shutting down\n") ;

	/* close all fds */
	for (ups = firstups; ups != NULL; ups = ups->next)
		if (ups->fd != -1)
			close (ups->fd);

	exit (0);
}

/* basic signal setup to handle SIGPIPE and SIGQUIT */
void setupsignals()
{
	sa.sa_handler = sigpipe;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGPIPE, &sa, NULL);

	sa.sa_handler = shut_myself_down;
	sigaction (SIGQUIT, &sa, NULL);
	sigaction (SIGTERM, &sa, NULL);
}

/* handler for alarm when getupsvarfd times out */
void timeout (int sig)
{
	/* don't do anything here, just return */
}

/* see what the status of the UPS is and handle any changes */
void pollups (utype *ups)
{
	char	status[SMALLBUF], *stat, *ptr;
	struct	sigaction sa;
	sigset_t sigmask;

	debug ("pollups: %s (fd %d)\n", ups->sys, ups->fd);

	if (ups->fd < 0)
		ups->fd = upsconnect (ups->host);

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (10);

	if (getupsvarfd(ups->fd, ups->upsname, "status", status, sizeof(status)) >= 0) {
		signal (SIGALRM, SIG_IGN);
		alarm (0);

		debug (" - status [%s] : ", status);

		/* clear these out early if they disappear */
		if (!strstr(status, "LB"))
			clearflag (&ups->status, ST_LOWBATT);
		if (!strstr(status, "FSD"))
			clearflag (&ups->status, ST_FSD);

		stat = status;

		while (stat != NULL) {
			ptr = strchr (stat, ' ');
			if (ptr)
				*ptr++ = '\0';

			debug ("{%s} ", stat);

			if (!strcasecmp(stat, "OL"))
				upsonline(ups);
			if (!strcasecmp(stat, "OB"))
				upsonbatt(ups);
			if (!strcasecmp(stat, "LB"))
				upslowbatt(ups);
			if (!strcasecmp(stat, "RB"))
				upsreplbatt(ups);

			/* do it last to override any possible OL */
			if (!strcasecmp(stat, "FSD"))
				upsfsd(ups);

			stat = ptr;
		} 

		debug ("\n");
	}
	else {					/* fetch failed */
		signal (SIGALRM, SIG_IGN);
		alarm (0);
		upsgone(ups);
	}
}

/* remove the power down flag if it exists and is the proper form */
void clearpdf ()
{
	FILE	*pdf;
	char	buf[SMALLBUF];

	pdf = fopen (powerdownflag, "r");

	if (pdf == NULL)	/* no such file, nothing to do */
		return;

	/* if it exists, see if it has the right text in it */

	fgets (buf, sizeof(buf), pdf);
	fclose (pdf);

	/* reasoning: say upsmon.conf is world-writable (!) and some nasty
	 * user puts something "important" as the power flag file.  This 
	 * keeps upsmon from utterly trashing it when starting up or powering
	 * down at the expense of not shutting down the UPS.
	 *
	 * solution: don't let mere mortals edit that configuration file.
	 */

	if (!strncmp (buf, SDMAGIC, strlen(SDMAGIC)))	/* ok, it's from us */
		unlink (powerdownflag);
	else {
	  	snprintf (buf, sizeof(buf), "%s doesn't seem to be a proper flag file.  Disabling.\n",
			  powerdownflag);
		syslog (LOG_INFO, buf);
		printf ("%s doesn't seem to be a proper flag file.  Disabling.\n",
		        powerdownflag);
		powerdownflag = NULL;
	}
}

void help (char *progname)
{
	printf ("usage: %s [-h] [-i]\n", progname);
	printf ("\n");	
	printf ("-i - go into an infinite loop after calling shutdown\n");
	exit (0);
}

/* set all the notify values to a default */
void initnotify(void)
{
	int	i;

	for (i = 0; notifylist[i].name != NULL; i++)
		notifylist[i].flags = NOTIFY_SYSLOG | NOTIFY_WALL;
}

int main(int argc, char *argv[])  
{
	int	i;

	printf ("Network UPS Tools upsmon %s\n", UPS_VERSION);

	while ((i = getopt(argc, argv, "+dhi")) != EOF) {
		switch (i) {
			case 'd':
				debuglevel = 1;
				break;
			case 'h':
				help(argv[0]);
				break;
			case 'i':
				playdead = 1;
				break;
			default:
				help(argv[0]);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	openlog ("upsmon", LOG_PID, LOG_FACILITY);

	initnotify();
	loadconfig();

	if (shutdowncmd == NULL)
		printf ("Warning: no shutdown command defined!\n");

	/* we may need to get rid of a flag from a previous shutdown */
	if (powerdownflag != NULL)
		clearpdf();

	/* ignore sigpipes before the connection opens */
	setupsignals();

	if (totalpv < minsupplies) {
		printf ("\nFatal error: insufficient power configured!\n\n");

		printf ("Sum of power values........: %d\n", totalpv);
		printf ("Minimum value (MINSUPPLIES): %d\n", minsupplies);

		printf ("\nEdit your upsmon.conf and change the values.\n");
		exit (1);
	}

	if (debuglevel < 1)
		background();

	/* reopen the log for the child process */
	closelog();
	openlog ("upsmon", LOG_PID, LOG_FACILITY);

	for (;;) {
		utype	*ups;		

		sleepval = pollfreq;
		for (ups = firstups; ups != NULL; ups = ups->next)
			pollups (ups);

		recalc();
		sleep (sleepval);
	}

	return (1);
}
