/*
 * tabstops.c
 * Copyright (C) 1999-2001 A.J. van Os; Released under GPL
 *
 * Description:
 * Read the tab stop information from a MS Word file
 */

#include <stdio.h>
#include "antiword.h"

static long	lDefaultTabWidth = 36000;	/* In millipoints */


/*
 * vSet6DefaultTabWidth -
 */
static void
vSet6DefaultTabWidth(FILE *pFile, ULONG ulStartBlock,
	const ULONG *aulBBD, size_t tBBDLen, const UCHAR *aucHeader)
{
	UCHAR	*aucBuffer;
	ULONG	ulBeginDocpInfo;
	size_t	tDocpInfoLen;

	ulBeginDocpInfo = ulGetLong(0x150, aucHeader);
	DBG_HEX(ulBeginDocpInfo);
	tDocpInfoLen = (size_t)ulGetLong(0x154, aucHeader);
	DBG_DEC(tDocpInfoLen);

	aucBuffer = xmalloc(tDocpInfoLen);
	if (!bReadBuffer(pFile, ulStartBlock,
			aulBBD, tBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, ulBeginDocpInfo, tDocpInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	lDefaultTabWidth = lTwips2MilliPoints(usGetWord(0x0a, aucBuffer));
	DBG_DEC(lDefaultTabWidth);
	aucBuffer = xfree(aucBuffer);
} /* end of vSet6DefaultTabWidth */

/*
 * vSet8DefaultTabWidth -
 */
static void
vSet8DefaultTabWidth(FILE *pFile, const pps_info_type *pPPS,
	const ULONG *aulBBD, size_t tBBDLen,
	const ULONG *aulSBD, size_t tSBDLen,
	const UCHAR *aucHeader)
{
        const ULONG	*aulBlockDepot;
	UCHAR	*aucBuffer;
	ULONG	ulBeginDocpInfo;
	ULONG	ulTableSize, ulTableStartBlock;
	size_t	tDocpInfoLen, tBlockDepotLen, tBlockSize;
	USHORT	usDocStatus;

	ulBeginDocpInfo = ulGetLong(0x192, aucHeader);
	DBG_HEX(ulBeginDocpInfo);
	tDocpInfoLen = (size_t)ulGetLong(0x196, aucHeader);
	DBG_DEC(tDocpInfoLen);

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		ulTableStartBlock = pPPS->t1Table.ulSB;
		ulTableSize = pPPS->t1Table.ulSize;
	} else {
		ulTableStartBlock = pPPS->t0Table.ulSB;
		ulTableSize = pPPS->t0Table.ulSize;
	}
	DBG_DEC(ulTableStartBlock);
	if (ulTableStartBlock == 0) {
		DBG_DEC(ulTableStartBlock);
		DBG_MSG("No TAB information");
		return;
	}
	DBG_HEX(ulTableSize);
	if (ulTableSize < MIN_SIZE_FOR_BBD_USE) {
		/* Use the Small Block Depot */
		aulBlockDepot = aulSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
		/* Use the Big Block Depot */
		aulBlockDepot = aulBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tDocpInfoLen);
	if (!bReadBuffer(pFile, ulTableStartBlock,
			aulBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, ulBeginDocpInfo, tDocpInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	lDefaultTabWidth = lTwips2MilliPoints(usGetWord(0x0a, aucBuffer));
	DBG_DEC(lDefaultTabWidth);
	aucBuffer = xfree(aucBuffer);
} /* end of vSet8DefaultTabWidth */

/*
 * vSetDefaultTabWidth -
 */
void
vSetDefaultTabWidth(FILE *pFile, const pps_info_type *pPPS,
	const ULONG *aulBBD, size_t tBBDLen,
	const ULONG *aulSBD, size_t tSBDLen,
	const UCHAR *aucHeader, int iWordVersion)
{
	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(iWordVersion < 6 || iWordVersion > 8);
	fail(aulBBD == NULL || aulSBD == NULL);

	switch (iWordVersion) {
	case 6:
	case 7:
		vSet6DefaultTabWidth(pFile, pPPS->tWordDocument.ulSB,
				aulBBD, tBBDLen, aucHeader);
		break;
	case 8:
		vSet8DefaultTabWidth(pFile, pPPS,
				aulBBD, tBBDLen, aulSBD, tSBDLen, aucHeader);
		break;
	default:
		werr(0, "Sorry, no TAB information");
		break;
	}
} /* end of vSetDefaultTabWidth */

/*
 * lGetDefaultTabWidth - Get the default tabwidth in millipoints
 */
long
lGetDefaultTabWidth(void)
{
	if (lDefaultTabWidth <= 0) {
		DBG_DEC(lDefaultTabWidth);
		return lTwips2MilliPoints(1);
	}
	return lDefaultTabWidth;
} /* end of lGetDefaultTabWidth */
