/*
 * prop8.c
 * Copyright (C) 1998-2002 A.J. van Os; Released under GPL
 *
 * Description:
 * Read the property information from a MS Word 8, 9 or 10 file
 *
 * Word 8 is better known as Word 97
 * Word 9 is better known as Word 2000
 * Word 10 is better known as Word 2002 or as Word XP
 */

#include <stdlib.h>
#include <string.h>
#include "antiword.h"


/*
 * iGet8InfoLength - the length of the information for Word 8/9/10 files
 */
static int
iGet8InfoLength(int iByteNbr, const UCHAR *aucGrpprl)
{
	int	iTmp, iDel, iAdd;
	USHORT	usOpCode;

	usOpCode = usGetWord(iByteNbr, aucGrpprl);

	switch (usOpCode & 0xe000) {
	case 0x0000: case 0x2000:
		return 3;
	case 0x4000: case 0x8000: case 0xa000:
		return 4;
	case 0xe000:
		return 5;
	case 0x6000:
		return 6;
	case 0xc000:
		iTmp = (int)ucGetByte(iByteNbr + 2, aucGrpprl);
		if (usOpCode == 0xc615 && iTmp == 255) {
			iDel = (int)ucGetByte(iByteNbr + 3, aucGrpprl);
			iAdd = (int)ucGetByte(
					iByteNbr + 4 + iDel * 4, aucGrpprl);
			iTmp = 2 + iDel * 4 + iAdd * 3;
		}
		return 3 + iTmp;
	default:
		DBG_HEX(usOpCode);
		DBG_FIXME();
		return 1;
	}
} /* end of iGet8InfoLength */

/*
 * Fill the section information block with information
 * from a Word 8/9/10 file.
 */
static void
vGet8SectionInfo(const UCHAR *aucGrpprl, size_t tBytes,
		section_block_type *pSection)
{
	int	iFodoOff, iInfoLen, iSize, iIndex, iTmp;
	USHORT	usCcol;
	UCHAR	ucTmp;

	fail(aucGrpprl == NULL || pSection == NULL);

	iFodoOff = 0;
	while (tBytes >= (size_t)iFodoOff + 2) {
		iInfoLen = 0;
		switch (usGetWord(iFodoOff, aucGrpprl)) {
		case 0x3009:	/* bkc */
			ucTmp = ucGetByte(iFodoOff + 2, aucGrpprl);
			DBG_DEC(ucTmp);
			pSection->bNewPage = ucTmp != 0 && ucTmp != 1;
			break;
		case 0x500b:	/* ccolM1 */
			usCcol = 1 + usGetWord(iFodoOff + 2, aucGrpprl);
			DBG_DEC(usCcol);
			break;
		case 0xd202:	/* olstAnm */
			iSize = (int)ucGetByte(iFodoOff + 2, aucGrpprl);
			DBG_DEC_C(iSize != 212, iSize);
			for (iIndex = 0, iTmp = iFodoOff + 3;
			     iIndex < 9 && iTmp < iFodoOff + 3 + iSize - 15;
			     iIndex++, iTmp += 16) {
				pSection->aucNFC[iIndex] =
						ucGetByte(iTmp, aucGrpprl);
				DBG_DEC(pSection->aucNFC[iIndex]);
				ucTmp = ucGetByte(iTmp + 3, aucGrpprl);
				DBG_HEX(ucTmp);
				if ((ucTmp & BIT(2)) != 0) {
					pSection->usNeedPrevLvl |=
							(USHORT)BIT(iIndex);
				}
				if ((ucTmp & BIT(3)) != 0) {
					pSection->usHangingIndent |=
							(USHORT)BIT(iIndex);
				}
			}
			DBG_HEX(pSection->usNeedPrevLvl);
			DBG_HEX(pSection->usHangingIndent);
			break;
		default:
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen = iGet8InfoLength(iFodoOff, aucGrpprl);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
} /* end of vGet8SectionInfo */

/*
 * Build the lists with Section Property Information for Word 8/9/10 files
 */
void
vGet8SepInfo(FILE *pFile, const pps_info_type *pPPS,
	const ULONG *aulBBD, size_t tBBDLen,
	const ULONG *aulSBD, size_t tSBDLen,
	const UCHAR *aucHeader)
{
	section_block_type	tSection;
	ULONG		*aulSectPage, *aulTextOffset;
	const ULONG	*aulBlockDepot;
	UCHAR	*aucBuffer, *aucFpage;
	ULONG	ulBeginSectInfo;
	ULONG	ulTableSize, ulTableStartBlock;
	size_t	tSectInfoLen, tBlockDepotLen;
	size_t	tBlockSize, tOffset, tSize, tLen, tBytes;
	int	iIndex;
	USHORT	usDocStatus;
	UCHAR	aucTmp[2];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(aulBBD == NULL || aulSBD == NULL);

	ulBeginSectInfo = ulGetLong(0xca, aucHeader); /* fcPlcfsed */
	NO_DBG_HEX(ulBeginSectInfo);
	tSectInfoLen = (size_t)ulGetLong(0xce, aucHeader); /* lcbPlcfsed */
	NO_DBG_DEC(tSectInfoLen);
	if (tSectInfoLen < 4) {
		DBG_DEC(tSectInfoLen);
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		ulTableStartBlock = pPPS->t1Table.ulSB;
		ulTableSize = pPPS->t1Table.ulSize;
	} else {
		ulTableStartBlock = pPPS->t0Table.ulSB;
		ulTableSize = pPPS->t0Table.ulSize;
	}
	DBG_DEC(ulTableStartBlock);
	if (ulTableStartBlock == 0) {
		DBG_MSG("No section information");
		return;
	}
	DBG_HEX(ulTableSize);
	if (ulTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aulBlockDepot = aulSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aulBlockDepot = aulBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tSectInfoLen);
	if (!bReadBuffer(pFile, ulTableStartBlock,
			aulBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, ulBeginSectInfo, tSectInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tSectInfoLen);

	/* Read the Section Descriptors */
	tLen = (tSectInfoLen - 4) / 16;
	tSize = tLen * sizeof(ULONG);
	/* Save the section offsets */
	aulTextOffset = xmalloc(tSize);
	for (iIndex = 0, tOffset = 0;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 4) {
		aulTextOffset[iIndex] = ulGetLong(tOffset, aucBuffer);
	}
	/* Save the Sepx offsets */
	aulSectPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 12) {
		 aulSectPage[iIndex] = ulGetLong(tOffset + 2, aucBuffer);
		 NO_DBG_HEX(aulSectPage[iIndex]); /* fcSepx */
	}
	aucBuffer = xfree(aucBuffer);

	/* Read the Section Properties */
	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		if (aulSectPage[iIndex] == FC_INVALID) {
			vDefault2SectionInfoList(aulTextOffset[iIndex]);
			continue;
		}
		/* Get the number of bytes to read */
		if (!bReadBuffer(pFile, pPPS->tWordDocument.ulSB,
				aulBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucTmp, aulSectPage[iIndex], 2)) {
			continue;
		}
		tBytes = 2 + (size_t)usGetWord(0, aucTmp);
		NO_DBG_DEC(tBytes);
		/* Read the bytes */
		aucFpage = xmalloc(tBytes);
		if (!bReadBuffer(pFile, pPPS->tWordDocument.ulSB,
				aulBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage, aulSectPage[iIndex], tBytes)) {
			aucFpage = xfree(aucFpage);
			continue;
		}
		NO_DBG_PRINT_BLOCK(aucFpage, tBytes);
		/* Process the bytes */
		vGetDefaultSection(&tSection);
		vGet8SectionInfo(aucFpage + 2, tBytes - 2, &tSection);
		vAdd2SectionInfoList(&tSection, aulTextOffset[iIndex]);
		aucFpage = xfree(aucFpage);
	}
	aulTextOffset = xfree(aulTextOffset);
	aulSectPage = xfree(aulSectPage);
} /* end of vGet8SepInfo */

/*
 * Translate the rowinfo to a member of the row_info enumeration
 */
row_info_enum
eGet8RowInfo(int iFodo,
	const UCHAR *aucGrpprl, int iBytes, row_block_type *pRow)
{
	int	iFodoOff, iInfoLen;
	int	iIndex, iSize, iCol;
	int	iPosCurr, iPosPrev;
	BOOL	bFound2416_0, bFound2416_1, bFound2417_0, bFound2417_1;
	BOOL	bFoundd608;

	fail(iFodo < 0 || aucGrpprl == NULL || pRow == NULL);

	iFodoOff = 0;
	bFound2416_0 = FALSE;
	bFound2416_1 = FALSE;
	bFound2417_0 = FALSE;
	bFound2417_1 = FALSE;
	bFoundd608 = FALSE;
	while (iBytes >= iFodoOff + 2) {
		iInfoLen = 0;
		switch (usGetWord(iFodo + iFodoOff, aucGrpprl)) {
		case 0x2416:	/* fIntable */
			if (odd(ucGetByte(iFodo + iFodoOff + 2, aucGrpprl))) {
				bFound2416_1 = TRUE;
			} else {
				bFound2416_0 = TRUE;
			}
			break;
		case 0x2417:	/* fTtp */
			if (odd(ucGetByte(iFodo + iFodoOff + 2, aucGrpprl))) {
				bFound2417_1 = TRUE;
			} else {
				bFound2417_0 = TRUE;
			}
			break;
		case 0xd608:	/* cDefTable */
			iSize = (int)usGetWord(iFodo + iFodoOff + 2, aucGrpprl);
			if (iSize < 6 || iBytes < iFodoOff + 8) {
				DBG_DEC(iSize);
				DBG_DEC(iFodoOff);
				iInfoLen = 2;
				break;
			}
			iCol = (int)ucGetByte(iFodo + iFodoOff + 4, aucGrpprl);
			if (iCol < 1 ||
			    iBytes < iFodoOff + 4 + (iCol + 1) * 2) {
				DBG_DEC(iCol);
				DBG_DEC(iFodoOff);
				iInfoLen = 2;
				break;
			}
			if (iCol >= (int)elementsof(pRow->asColumnWidth)) {
				DBG_DEC(iCol);
				werr(1, "The number of columns is corrupt");
			}
			pRow->ucNumberOfColumns = (UCHAR)iCol;
			pRow->iColumnWidthSum = 0;
			iPosPrev = (int)(short)usGetWord(
					iFodo + iFodoOff + 5,
					aucGrpprl);
			for (iIndex = 0; iIndex < iCol; iIndex++) {
				iPosCurr = (int)(short)usGetWord(
					iFodo + iFodoOff + 7 + iIndex * 2,
					aucGrpprl);
				pRow->asColumnWidth[iIndex] =
						(short)(iPosCurr - iPosPrev);
				pRow->iColumnWidthSum +=
					pRow->asColumnWidth[iIndex];
				iPosPrev = iPosCurr;
			}
			bFoundd608 = TRUE;
			break;
		default:
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen =
				iGet8InfoLength(iFodo + iFodoOff, aucGrpprl);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
	if (bFound2416_1 && bFound2417_1 && bFoundd608) {
		return found_end_of_row;
	}
	if (bFound2416_0 && bFound2417_0 && !bFoundd608) {
		return found_not_end_of_row;
	}
	if (bFound2416_1) {
		return found_a_cell;
	}
	if (bFound2416_0) {
		return found_not_a_cell;
	}
	return found_nothing;
} /* end of eGet8RowInfo */

/*
 * Fill the style information block with information
 * from a Word 8/9/10 file.
 */
void
vGet8StyleInfo(int iFodo,
	const UCHAR *aucGrpprl, int iBytes, style_block_type *pStyle)
{
	int	iFodoOff, iInfoLen;
	int	iTmp, iDel, iAdd, iBefore;
	USHORT	usOpCode, usIlfo, usTmp;
	short	sTmp;

	fail(iFodo < 0 || aucGrpprl == NULL || pStyle == NULL);

	NO_DBG_DEC(pStyle->usIstd);

	usIlfo = 0;
	iFodoOff = 0;
	while (iBytes >= iFodoOff + 2) {
		iInfoLen = 0;
		usOpCode = usGetWord(iFodo + iFodoOff, aucGrpprl);
		switch (usOpCode) {
		case 0x2403:	/* jc */
			pStyle->ucAlignment = ucGetByte(
					iFodo + iFodoOff + 2, aucGrpprl);
			break;
		case 0x260a:	/* ilvl */
			pStyle->bInList = TRUE;
			pStyle->ucListLevel =
				ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(pStyle->ucListLevel);
			break;
		case 0x4600:	/* istd */
			usTmp = usGetWord(iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(usTmp);
			break;
		case 0x460b:	/* ilfo */
			pStyle->bInList = TRUE;
			usIlfo = usGetWord(iFodo + iFodoOff + 2, aucGrpprl);
			pStyle->bNumPause = usIlfo == 0;
			break;
		case 0x4610: /* Nest dxaLeft */
			sTmp = (short)usGetWord(
					iFodo + iFodoOff + 2, aucGrpprl);
			pStyle->sLeftIndent += sTmp;
			if (pStyle->sLeftIndent < 0) {
				pStyle->sLeftIndent = 0;
			}
			DBG_DEC(sTmp);
			DBG_DEC(pStyle->sLeftIndent);
			break;
		case 0x6c0d:	/* ChgTabsPapx */
			iTmp = (int)ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (iTmp < 2) {
				iInfoLen = 1;
				break;
			}
			DBG_DEC(iTmp);
			iDel = (int)ucGetByte(iFodo + iFodoOff + 3, aucGrpprl);
			if (iTmp < 2 + 2 * iDel) {
				iInfoLen = 1;
				break;
			}
			DBG_DEC(iDel);
			iAdd = (int)ucGetByte(
				iFodo + iFodoOff + 4 + 2 * iDel, aucGrpprl);
			if (iTmp < 2 + 2 * iDel + 2 * iAdd) {
				iInfoLen = 1;
				break;
			}
			DBG_DEC(iAdd);
			break;
		case 0x840e:	/* dxaRight */
			pStyle->sRightIndent = (short)usGetWord(
					iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(pStyle->sRightIndent);
			break;
		case 0x840f:	/* dxaLeft */
			pStyle->sLeftIndent = (short)usGetWord(
					iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(pStyle->sLeftIndent);
			break;
		case 0xa413:	/* dyaBefore */
			pStyle->usBeforeIndent = usGetWord(
					iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(pStyle->usBeforeIndent);
			break;
		case 0xa414:	/* dyaAfter */
			pStyle->usAfterIndent = usGetWord(
					iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(pStyle->usAfterIndent);
			break;
		case 0xc63e:	/* anld */
#if 0
			if (usIlfo != 2047) {
				break;
			}
#endif
			iTmp = (int)ucGetByte(
					iFodo + iFodoOff + 2, aucGrpprl);
			DBG_DEC_C(iTmp < 84, iTmp);
			pStyle->bInList = TRUE;
			if (iTmp >= 1) {
				pStyle->ucNFC = ucGetByte(
					iFodo + iFodoOff + 3, aucGrpprl);
			}
			if (pStyle->ucNFC != LIST_BULLETS && iTmp >= 2) {
				iBefore = (int)ucGetByte(
					iFodo + iFodoOff + 4, aucGrpprl);
			} else {
				iBefore = 0;
			}
			if (iTmp >= 12) {
				pStyle->usStartAt = usGetWord(
					iFodo + iFodoOff + 13, aucGrpprl);
			}
			if (iTmp >= iBefore + 21) {
				pStyle->ucListCharacter = ucGetByte(
					iFodo + iFodoOff + iBefore + 23,
					aucGrpprl);
				NO_DBG_CHR(pStyle->ucListCharacter);
			}
			break;
		default:
			NO_DBG_HEX(usOpCode);
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen =
				iGet8InfoLength(iFodo + iFodoOff, aucGrpprl);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
} /* end of vGet8StyleInfo */

/*
 * Build the lists with Paragraph Information for Word 8/9/10 files
 */
void
vGet8PapInfo(FILE *pFile, const pps_info_type *pPPS,
	const ULONG *aulBBD, size_t tBBDLen,
	const ULONG *aulSBD, size_t tSBDLen,
	const UCHAR *aucHeader)
{
	row_block_type		tRow;
	style_block_type	tStyle;
	ULONG		*aulParfPage;
	const ULONG	*aulBlockDepot;
	UCHAR	*aucBuffer;
	ULONG	ulCharPos, ulCharPosFirst, ulCharPosLast;
	ULONG	ulBeginParfInfo;
	ULONG	ulTableSize, ulTableStartBlock;
	size_t	tParfInfoLen, tBlockDepotLen;
	size_t	tBlockSize, tOffset, tSize, tLen;
	int	iIndex, iIndex2, iRun, iFodo, iLen;
	row_info_enum	eRowInfo;
	USHORT	usDocStatus, usIstd;
	UCHAR	aucFpage[BIG_BLOCK_SIZE];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(aulBBD == NULL || aulSBD == NULL);

	ulBeginParfInfo = ulGetLong(0x102, aucHeader); /* fcPlcfbtePapx */
	NO_DBG_HEX(ulBeginParfInfo);
	tParfInfoLen = (size_t)ulGetLong(0x106, aucHeader); /* lcbPlcfbtePapx */
	NO_DBG_DEC(tParfInfoLen);
	if (tParfInfoLen < 4) {
		DBG_DEC(tParfInfoLen);
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		ulTableStartBlock = pPPS->t1Table.ulSB;
		ulTableSize = pPPS->t1Table.ulSize;
	} else {
		ulTableStartBlock = pPPS->t0Table.ulSB;
		ulTableSize = pPPS->t0Table.ulSize;
	}
	DBG_DEC(ulTableStartBlock);
	if (ulTableStartBlock == 0) {
		DBG_MSG("No paragraph information");
		return;
	}
	DBG_HEX(ulTableSize);
	if (ulTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aulBlockDepot = aulSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aulBlockDepot = aulBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tParfInfoLen);
	if (!bReadBuffer(pFile, ulTableStartBlock,
			aulBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, ulBeginParfInfo, tParfInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tParfInfoLen);

	tLen = (tParfInfoLen / 4 - 1) / 2;
	tSize = tLen * sizeof(ULONG);
	aulParfPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 4) {
		 aulParfPage[iIndex] = ulGetLong(tOffset, aucBuffer);
		 NO_DBG_DEC(aulParfPage[iIndex]);
	}
	DBG_HEX(ulGetLong(0, aucBuffer));
	aucBuffer = xfree(aucBuffer);
	NO_DBG_PRINT_BLOCK(aucHeader, HEADER_SIZE);

	(void)memset(&tRow, 0, sizeof(tRow));
	ulCharPosFirst = CP_INVALID;
	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		fail(aulParfPage[iIndex] > ULONG_MAX / BIG_BLOCK_SIZE);
		if (!bReadBuffer(pFile, pPPS->tWordDocument.ulSB,
				aulBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage,
				aulParfPage[iIndex] * BIG_BLOCK_SIZE,
				BIG_BLOCK_SIZE)) {
			break;
		}
		NO_DBG_PRINT_BLOCK(aucFpage, BIG_BLOCK_SIZE);
		iRun = (int)ucGetByte(0x1ff, aucFpage);
		NO_DBG_DEC(iRun);
		for (iIndex2 = 0; iIndex2 < iRun; iIndex2++) {
			NO_DBG_HEX(ulGetLong(iIndex2 * 4, aucFpage));
			iFodo = 2 * (int)ucGetByte(
				(iRun + 1) * 4 + iIndex2 * 13, aucFpage);
			if (iFodo <= 0) {
				continue;
			}

			iLen = 2 * (int)ucGetByte(iFodo, aucFpage);
			if (iLen == 0) {
				iFodo++;
				iLen = 2 * (int)ucGetByte(iFodo, aucFpage);
			}

			usIstd = usGetWord(iFodo + 1, aucFpage);
			vFillStyleFromStylesheet(usIstd, &tStyle);
			vGet8StyleInfo(iFodo, aucFpage + 3, iLen - 3, &tStyle);
			ulCharPos = ulGetLong(iIndex2 * 4, aucFpage);
			NO_DBG_HEX(ulCharPos);
			tStyle.ulFileOffset = ulCharPos2FileOffset(ulCharPos);
			vAdd2StyleInfoList(&tStyle);

			eRowInfo = eGet8RowInfo(iFodo,
					aucFpage + 3, iLen - 3, &tRow);
			switch (eRowInfo) {
			case found_a_cell:
				if (ulCharPosFirst != CP_INVALID) {
					break;
				}
				ulCharPosFirst = ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(ulCharPosFirst);
				tRow.ulCharPosStart = ulCharPosFirst;
				tRow.ulFileOffsetStart =
					ulCharPos2FileOffset(ulCharPosFirst);
				NO_DBG_HEX_C(
					tRow.ulFileOffsetStart == FC_INVALID,
					ulCharPosFirst);
				break;
			case found_end_of_row:
				ulCharPosLast = ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(ulCharPosLast);
				tRow.ulCharPosEnd = ulCharPosLast;
				tRow.ulFileOffsetEnd =
					ulCharPos2FileOffset(ulCharPosLast);
				NO_DBG_HEX_C(tRow.ulFileOffsetEnd == FC_INVALID,
							ulCharPosLast);
				vAdd2RowInfoList(&tRow);
				(void)memset(&tRow, 0, sizeof(tRow));
				ulCharPosFirst = CP_INVALID;
				break;
			case found_nothing:
				break;
			default:
				DBG_DEC(eRowInfo);
				break;
			}
		}
	}
	aulParfPage = xfree(aulParfPage);
} /* end of vGet8PapInfo */

/*
 * Fill the font information block with information
 * from a Word 8/9/10 file.
 */
void
vGet8FontInfo(int iFodo, USHORT usIstd,
	const UCHAR *aucGrpprl, int iBytes, font_block_type *pFont)
{
	long	lTmp;
	int	iFodoOff, iInfoLen;
	USHORT	usTmp;
	short	sTmp;
	UCHAR	ucTmp;

	fail(iFodo < 0 || aucGrpprl == NULL || pFont == NULL);

	iFodoOff = 0;
	while (iBytes >= iFodoOff + 2) {
		switch (usGetWord(iFodo + iFodoOff, aucGrpprl)) {
		case 0x0800:	/* fRMarkDel */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (ucTmp == 0) {
				pFont->ucFontstyle &= ~FONT_MARKDEL;
			} else {
				pFont->ucFontstyle |= FONT_MARKDEL;
			}
			break;
		case 0x0835:	/* fBold */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_BOLD;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_BOLD;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_BOLD;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x0836:	/* fItalic */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_ITALIC;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_ITALIC;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_ITALIC;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x0837:	/* fStrike */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_STRIKE;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_STRIKE;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_STRIKE;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x083a:	/* fSmallCaps */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_SMALL_CAPITALS;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_SMALL_CAPITALS;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_SMALL_CAPITALS;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x083b:	/* fCaps */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_CAPITALS;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_CAPITALS;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_CAPITALS;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x083c:	/* fVanish */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_HIDDEN;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_HIDDEN;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_HIDDEN;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x2a32:	/* cDefault */
			pFont->ucFontstyle &= FONT_HIDDEN;
			pFont->ucFontcolor = FONT_COLOR_DEFAULT;
			break;
		case 0x2a33:	/* cPlain */
			DBG_MSG("2a33: cPlain");
			vFillFontFromStylesheet(usIstd, pFont);
			break;
		case 0x2a3e:	/* cKul */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (ucTmp == 0 || ucTmp == 5) {
				pFont->ucFontstyle &= ~FONT_UNDERLINE;
			} else {
				NO_DBG_MSG("Underline text");
				pFont->ucFontstyle |= FONT_UNDERLINE;
				if (ucTmp == 6) {
					DBG_MSG("Bold text");
					pFont->ucFontstyle |= FONT_BOLD;
				}
			}
			break;
		case 0x2a42:	/* cIco */
			pFont->ucFontcolor =
				ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(pFont->ucFontcolor);
			break;
		case 0x2a48:	/* cIss */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			ucTmp &= 0x07;
			DBG_MSG_C(ucTmp == 1, "Superscript");
			DBG_MSG_C(ucTmp == 2, "Subscript");
			break;
		case 0x4a30:	/* cIstd */
			usTmp = usGetWord(iFodo + iFodoOff + 2, aucGrpprl);
			NO_DBG_DEC(usTmp);
			break;
		case 0x4a43:	/* cHps */
			usTmp = usGetWord(iFodo + iFodoOff + 2, aucGrpprl);
			if (usTmp > (USHORT)SHRT_MAX) {
				pFont->sFontsize = SHRT_MAX;
			} else {
				pFont->sFontsize = (short)usTmp;
			}
			NO_DBG_DEC(pFont->sFontsize);
			break;
		case 0x4a51:	/* cFtc */
			usTmp = usGetWord(iFodo + iFodoOff + 2, aucGrpprl);
			if (usTmp <= (USHORT)UCHAR_MAX) {
				pFont->ucFontnumber = (UCHAR)usTmp;
			} else {
				pFont->ucFontnumber = 0;
			}
			break;
		case 0xca4a:	/* cHps */
			sTmp = (short)usGetWord(
					iFodo + iFodoOff + 2, aucGrpprl);
			lTmp = (long)pFont->sFontsize + (long)sTmp;
			if (lTmp < 8) {
				pFont->sFontsize = 8;
			} else if (lTmp > 32766) {
				pFont->sFontsize = 32766;
			} else {
				pFont->sFontsize = (short)lTmp;
			}
			break;
		case 0xea3f:	/* cHps, cHpsPos */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (ucTmp != 0) {
				pFont->sFontsize = (short)ucTmp;
			}
			DBG_DEC(ucTmp);
			break;
		default:
			break;
		}
		iInfoLen = iGet8InfoLength(iFodo + iFodoOff, aucGrpprl);
		fail(iInfoLen <= 0);
		iFodoOff += iInfoLen;
	}
} /* end of vGet8FontInfo */

/*
 * Fill the picture information block with information
 * from a Word 8/9/10 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet8PicInfo(int iFodo,
	const UCHAR *aucGrpprl, int iBytes, picture_block_type *pPicture)
{
	int	iFodoOff, iInfoLen;
	BOOL	bFound;
	UCHAR	ucTmp;

	fail(iFodo <= 0 || aucGrpprl == NULL || pPicture == NULL);

	iFodoOff = 0;
	bFound = FALSE;
	while (iBytes >= iFodoOff + 2) {
		switch (usGetWord(iFodo + iFodoOff, aucGrpprl)) {
		case 0x0806:	/* fData */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (ucTmp == 0x01) {
				/* Not a picture, but a form field */
				return FALSE;
			}
			DBG_DEC_C(ucTmp != 0, ucTmp);
			break;
		case 0x080a:	/* fOle2 */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (ucTmp == 0x01) {
				/* Not a picture, but an OLE object */
				return FALSE;
			}
			DBG_DEC_C(ucTmp != 0, ucTmp);
			break;
		case 0x6a03:	/* fcPic */
			pPicture->ulPictureOffset = ulGetLong(
					iFodo + iFodoOff + 2, aucGrpprl);
			bFound = TRUE;
			break;
		default:
			break;
		}
		iInfoLen = iGet8InfoLength(iFodo + iFodoOff, aucGrpprl);
		fail(iInfoLen <= 0);
		iFodoOff += iInfoLen;
	}
	return bFound;
} /* end of bGet8PicInfo */

/*
 * Build the lists with Character Information for Word 8/9/10 files
 */
void
vGet8ChrInfo(FILE *pFile, const pps_info_type *pPPS,
	const ULONG *aulBBD, size_t tBBDLen,
	const ULONG *aulSBD, size_t tSBDLen,
	const UCHAR *aucHeader)
{
	font_block_type		tFont;
	picture_block_type	tPicture;
	ULONG		*aulCharPage;
	const ULONG	*aulBlockDepot;
	UCHAR	*aucBuffer;
	ULONG	ulFileOffset, ulCharPos, ulBeginCharInfo;
	ULONG	ulTableSize, ulTableStartBlock;
	size_t	tCharInfoLen, tBlockDepotLen;
	size_t	tOffset, tBlockSize, tSize, tLen;
	int	iIndex, iIndex2, iRun, iFodo, iLen;
	USHORT	usDocStatus, usIstd;
	UCHAR	aucFpage[BIG_BLOCK_SIZE];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(aulBBD == NULL || aulSBD == NULL);

	ulBeginCharInfo = ulGetLong(0xfa, aucHeader); /* fcPlcfbteChpx */
	NO_DBG_HEX(ulBeginCharInfo);
	tCharInfoLen = (size_t)ulGetLong(0xfe, aucHeader); /* lcbPlcfbteChpx */
	NO_DBG_DEC(tCharInfoLen);
	if (tCharInfoLen < 4) {
		DBG_DEC(tCharInfoLen);
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		ulTableStartBlock = pPPS->t1Table.ulSB;
		ulTableSize = pPPS->t1Table.ulSize;
	} else {
		ulTableStartBlock = pPPS->t0Table.ulSB;
		ulTableSize = pPPS->t0Table.ulSize;
	}
	DBG_DEC(ulTableStartBlock);
	if (ulTableStartBlock == 0) {
		DBG_MSG("No character information");
		return;
	}
	DBG_HEX(ulTableSize);
	if (ulTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aulBlockDepot = aulSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aulBlockDepot = aulBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tCharInfoLen);
	if (!bReadBuffer(pFile, ulTableStartBlock,
			aulBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, ulBeginCharInfo, tCharInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tCharInfoLen);

	tLen = (tCharInfoLen / 4 - 1) / 2;
	tSize = tLen * sizeof(ULONG);
	aulCharPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 4) {
		 aulCharPage[iIndex] = ulGetLong(tOffset, aucBuffer);
		 NO_DBG_DEC(aulCharPage[iIndex]);
	}
	DBG_HEX(ulGetLong(0, aucBuffer));
	aucBuffer = xfree(aucBuffer);
	NO_DBG_PRINT_BLOCK(aucHeader, HEADER_SIZE);

	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		fail(aulCharPage[iIndex] > ULONG_MAX / BIG_BLOCK_SIZE);
		if (!bReadBuffer(pFile, pPPS->tWordDocument.ulSB,
				aulBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage,
				aulCharPage[iIndex] * BIG_BLOCK_SIZE,
				BIG_BLOCK_SIZE)) {
			break;
		}
		NO_DBG_PRINT_BLOCK(aucFpage, BIG_BLOCK_SIZE);
		iRun = (int)ucGetByte(0x1ff, aucFpage);
		NO_DBG_DEC(iRun);
		for (iIndex2 = 0; iIndex2 < iRun; iIndex2++) {
			ulCharPos = ulGetLong(iIndex2 * 4, aucFpage);
			ulFileOffset = ulCharPos2FileOffset(ulCharPos);
			iFodo = 2 * (int)ucGetByte(
				(iRun + 1) * 4 + iIndex2, aucFpage);

			iLen = (int)ucGetByte(iFodo, aucFpage);

			usIstd = usGetIstd(ulFileOffset);
			vFillFontFromStylesheet(usIstd, &tFont);
			if (iFodo != 0) {
				vGet8FontInfo(iFodo, usIstd,
					aucFpage + 1, iLen - 1, &tFont);
			}
			tFont.ulFileOffset = ulFileOffset;
			vAdd2FontInfoList(&tFont);

			if (iFodo <= 0) {
				continue;
			}

			(void)memset(&tPicture, 0, sizeof(tPicture));
			if (bGet8PicInfo(iFodo, aucFpage + 1,
						iLen - 1, &tPicture)) {
				tPicture.ulFileOffset = ulFileOffset;
				tPicture.ulFileOffsetPicture =
					ulDataPos2FileOffset(
						tPicture.ulPictureOffset);
				vAdd2PictInfoList(&tPicture);
			}
		}
	}
	aulCharPage = xfree(aulCharPage);
} /* end of vGet8ChrInfo */
