/*
 * blocklist.c
 * Copyright (C) 1998-2002 A.J. van Os; Released under GPL
 *
 * Description:
 * Build, read and destroy a list of Word text blocks
 */

#include <stdlib.h>
#include "antiword.h"


/*
 * Private structure to hide the way the information
 * is stored from the rest of the program
 */
typedef struct list_mem_tag {
	text_block_type		tInfo;
	struct list_mem_tag	*pNext;
} list_mem_type;

/* Variables to describe the start of the five block lists */
static list_mem_type	*pTextAnchor = NULL;
static list_mem_type	*pFootAnchor = NULL;
static list_mem_type	*pUnused1Anchor = NULL;
static list_mem_type	*pEndAnchor = NULL;
static list_mem_type	*pUnused2Anchor = NULL;
/* Variable needed to build the block list */
static list_mem_type	*pBlockLast = NULL;
/* Variable needed to read the text block list */
static list_mem_type	*pTextBlockCurrent = NULL;
/* Variable needed to read the footnote block list */
static list_mem_type	*pFootBlockCurrent = NULL;
/* Variable needed to read the endnote block list */
static list_mem_type	*pEndBlockCurrent = NULL;
/* Last block read */
static UCHAR		aucBlock[BIG_BLOCK_SIZE];


/*
 * vDestroyTextBlockList - destroy the text block list
 */
void
vDestroyTextBlockList(void)
{
	list_mem_type	*apAnchor[5];
	list_mem_type	*pCurr, *pNext;
	int	iIndex;

	DBG_MSG("vDestroyTextBlockList");

	apAnchor[0] = pTextAnchor;
	apAnchor[1] = pFootAnchor;
	apAnchor[2] = pUnused1Anchor;
	apAnchor[3] = pEndAnchor;
	apAnchor[4] = pUnused2Anchor;

	for (iIndex = 0; iIndex < 5; iIndex++) {
		pCurr = apAnchor[iIndex];
		while (pCurr != NULL) {
			pNext = pCurr->pNext;
			pCurr = xfree(pCurr);
			pCurr = pNext;
		}
	}
	/* Show that there are no lists any more */
	pTextAnchor = NULL;
	pFootAnchor = NULL;
	pUnused1Anchor = NULL;
	pEndAnchor = NULL;
	pUnused2Anchor = NULL;
	/* Reset all the controle variables */
	pBlockLast = NULL;
	pTextBlockCurrent = NULL;
	pFootBlockCurrent = NULL;
	pEndBlockCurrent = NULL;
} /* end of vDestroyTextBlockList */

/*
 * bAdd2TextBlockList - add an element to the text block list
 *
 * returns: TRUE when successful, otherwise FALSE
 */
BOOL
bAdd2TextBlockList(text_block_type *pTextBlock)
{
	list_mem_type	*pListMember;

	fail(pTextBlock == NULL);
	fail(pTextBlock->ulFileOffset == FC_INVALID);
	fail(pTextBlock->ulCharPos == CP_INVALID);
	fail(pTextBlock->ulLength == 0);
	fail(pTextBlock->bUsesUnicode && odd(pTextBlock->ulLength));

	NO_DBG_MSG("bAdd2TextBlockList");
	NO_DBG_HEX(pTextBlock->ulFileOffset);
	NO_DBG_HEX(pTextBlock->ulCharPos);
	NO_DBG_HEX(pTextBlock->ulLength);
	NO_DBG_DEC(pTextBlock->bUsesUnicode);
	NO_DBG_DEC(pTextBlock->usPropMod);

	if (pTextBlock->ulFileOffset == FC_INVALID ||
	    pTextBlock->ulCharPos == CP_INVALID ||
	    pTextBlock->ulLength == 0 ||
	    (pTextBlock->bUsesUnicode && odd(pTextBlock->ulLength))) {
		werr(0, "Software (textblock) error");
		return FALSE;
	}
	/*
	 * Check for continuous blocks of the same character size and
	 * the same properties modifier
	 */
	if (pBlockLast != NULL &&
	    pBlockLast->tInfo.ulFileOffset +
	     pBlockLast->tInfo.ulLength == pTextBlock->ulFileOffset &&
	    pBlockLast->tInfo.ulCharPos +
	     pBlockLast->tInfo.ulLength == pTextBlock->ulCharPos &&
	    pBlockLast->tInfo.bUsesUnicode == pTextBlock->bUsesUnicode &&
	    pBlockLast->tInfo.usPropMod == pTextBlock->usPropMod) {
		/* These are continous blocks */
		pBlockLast->tInfo.ulLength += pTextBlock->ulLength;
		return TRUE;
	}
	/* Make a new block */
	pListMember = xmalloc(sizeof(list_mem_type));
	/* Add the block to the list */
	pListMember->tInfo = *pTextBlock;
	pListMember->pNext = NULL;
	if (pTextAnchor == NULL) {
		pTextAnchor = pListMember;
	} else {
		fail(pBlockLast == NULL);
		pBlockLast->pNext = pListMember;
	}
	pBlockLast = pListMember;
	return TRUE;
} /* end of bAdd2TextBlockList */


/*
 * vSplitBlockList - split the block list in five parts
 *
 * Split the blocklist in a Text block list, a Footnote block list, a
 * Endnote block list and two Unused lists.
 *
 * NOTE:
 * The various l*Len input parameters are given in characters, but the
 * length of the blocks are in bytes.
 */
void
vSplitBlockList(ULONG ulTextLen, ULONG ulFootnoteLen, ULONG ulUnused1Len,
	ULONG ulEndnoteLen, ULONG ulUnused2Len, BOOL bMustExtend)
{
	list_mem_type	*apAnchors[5];
	list_mem_type	*pGarbageAnchor, *pCurr, *pNext;
	long		lCharsToGo, lBytesTooFar;
	int		iIndex;
#if defined(DEBUG)
	ULONG		ulTotal;
#endif /* DEBUG */

	DBG_MSG("vSplitBlockList");

	fail(ulTextLen > (ULONG)LONG_MAX);
	fail(ulFootnoteLen > (ULONG)LONG_MAX);
	fail(ulUnused1Len > (ULONG)LONG_MAX);
	fail(ulEndnoteLen > (ULONG)LONG_MAX);
	fail(ulUnused2Len > (ULONG)LONG_MAX);

/* Text block list */
	pCurr = NULL;
	lCharsToGo = (long)ulTextLen;
	lBytesTooFar = -1;
	if (ulTextLen != 0) {
		DBG_MSG("Text block list");
		DBG_DEC(ulTextLen);
		for (pCurr = pTextAnchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			NO_DBG_DEC(pCurr->tInfo.ulLength);
			fail(pCurr->tInfo.ulLength == 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.ulLength));
				lCharsToGo -= pCurr->tInfo.ulLength / 2;
				if (lCharsToGo < 0) {
					lBytesTooFar = -2 * lCharsToGo;
				}
			} else {
				lCharsToGo -= pCurr->tInfo.ulLength;
				if (lCharsToGo < 0) {
					lBytesTooFar = -lCharsToGo;
				}
			}
			if (lCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (ulTextLen == 0) {
		/* Empty text blocks list */
		pFootAnchor = pTextAnchor;
		pTextAnchor = NULL;
	} else if (pCurr == NULL) {
		/* No footnote blocks */
		pFootAnchor = NULL;
	} else if (lCharsToGo == 0) {
		/* Move the integral number of footnote blocks */
		pFootAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-text block, part-footnote block */
		DBG_DEC(lBytesTooFar);
		fail(lBytesTooFar <= 0);
		pFootAnchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		pFootAnchor->tInfo.ulFileOffset =
				pCurr->tInfo.ulFileOffset +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pFootAnchor->tInfo.ulFileOffset);
		DBG_HEX(pCurr->tInfo.ulCharPos);
		pFootAnchor->tInfo.ulCharPos =
				pCurr->tInfo.ulCharPos +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pFootAnchor->tInfo.ulCharPos);
		pFootAnchor->tInfo.ulLength = (ULONG)lBytesTooFar;
		pCurr->tInfo.ulLength -= lBytesTooFar;
		pFootAnchor->tInfo.bUsesUnicode = pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of footnote blocks */
		pFootAnchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Footnote block list */
	pCurr = NULL;
	lCharsToGo = (long)ulFootnoteLen;
	lBytesTooFar = -1;
	if (ulFootnoteLen != 0) {
		DBG_MSG("Footnote block list");
		DBG_DEC(ulFootnoteLen);
		for (pCurr = pFootAnchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.ulLength);
			fail(pCurr->tInfo.ulLength == 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.ulLength));
				lCharsToGo -= pCurr->tInfo.ulLength / 2;
				if (lCharsToGo < 0) {
					lBytesTooFar = -2 * lCharsToGo;
				}
			} else {
				lCharsToGo -= pCurr->tInfo.ulLength;
				if (lCharsToGo < 0) {
					lBytesTooFar = -lCharsToGo;
				}
			}
			if (lCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (ulFootnoteLen == 0) {
		/* Empty footnote list */
		pUnused1Anchor = pFootAnchor;
		pFootAnchor = NULL;
	} else if (pCurr == NULL) {
		/* No unused1 blocks */
		pUnused1Anchor = NULL;
	} else if (lCharsToGo == 0) {
		/* Move the integral number of unused1-list blocks */
		pUnused1Anchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-footnote block, part-unused1 block */
		DBG_DEC(lBytesTooFar);
		fail(lBytesTooFar <= 0);
		pUnused1Anchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		pUnused1Anchor->tInfo.ulFileOffset =
				pCurr->tInfo.ulFileOffset +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pUnused1Anchor->tInfo.ulFileOffset);
		DBG_HEX(pCurr->tInfo.ulCharPos);
		pUnused1Anchor->tInfo.ulCharPos =
				pCurr->tInfo.ulCharPos +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pUnused1Anchor->tInfo.ulCharPos);
		pUnused1Anchor->tInfo.ulLength = (ULONG)lBytesTooFar;
		pCurr->tInfo.ulLength -= lBytesTooFar;
		pUnused1Anchor->tInfo.bUsesUnicode =
				pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of unused1 blocks */
		pUnused1Anchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Unused1 block list */
	pCurr = NULL;
	lCharsToGo = (long)ulUnused1Len;
	lBytesTooFar = -1;
	if (ulUnused1Len != 0) {
		DBG_MSG("Unused1 block list");
		DBG_DEC(ulUnused1Len);
		for (pCurr = pUnused1Anchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.ulLength);
			fail(pCurr->tInfo.ulLength == 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.ulLength));
				lCharsToGo -= pCurr->tInfo.ulLength / 2;
				if (lCharsToGo < 0) {
					lBytesTooFar = -2 * lCharsToGo;
				}
			} else {
				lCharsToGo -= pCurr->tInfo.ulLength;
				if (lCharsToGo < 0) {
					lBytesTooFar = -lCharsToGo;
				}
			}
			if (lCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (ulUnused1Len == 0) {
		/* Empty unused1 list */
		pEndAnchor = pUnused1Anchor;
		pUnused1Anchor = NULL;
	} else if (pCurr == NULL) {
		/* No endnote blocks */
		pEndAnchor = NULL;
	} else if (lCharsToGo == 0) {
		/* Move the intergral number of endnote blocks */
		pEndAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-unused1-list block, part-endnote block */
		DBG_DEC(lBytesTooFar);
		fail(lBytesTooFar <= 0);
		pEndAnchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		pEndAnchor->tInfo.ulFileOffset =
				pCurr->tInfo.ulFileOffset +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pEndAnchor->tInfo.ulFileOffset);
		DBG_HEX(pCurr->tInfo.ulCharPos);
		pEndAnchor->tInfo.ulCharPos =
				pCurr->tInfo.ulCharPos +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pEndAnchor->tInfo.ulCharPos);
		pEndAnchor->tInfo.ulLength = (ULONG)lBytesTooFar;
		pCurr->tInfo.ulLength -= lBytesTooFar;
		pEndAnchor->tInfo.bUsesUnicode = pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of endnote blocks */
		pEndAnchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Endnote block list */
	pCurr = NULL;
	lCharsToGo = (long)ulEndnoteLen;
	lBytesTooFar = -1;
	if (ulEndnoteLen != 0) {
		DBG_MSG("Endnote block list");
		DBG_DEC(ulEndnoteLen);
		for (pCurr = pEndAnchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.ulLength);
			fail(pCurr->tInfo.ulLength == 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.ulLength));
				lCharsToGo -= pCurr->tInfo.ulLength / 2;
				if (lCharsToGo <= 0) {
					lBytesTooFar = -2 * lCharsToGo;
				}
			} else {
				lCharsToGo -= pCurr->tInfo.ulLength;
				if (lCharsToGo <= 0) {
					lBytesTooFar = -lCharsToGo;
				}
			}
			if (lCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (ulEndnoteLen == 0) {
		/* Empty endnote list */
		pUnused2Anchor = pEndAnchor;
		pEndAnchor = NULL;
	} else if (pCurr == NULL) {
		/* No unused2 blocks */
		pUnused2Anchor = NULL;
	} else if (lCharsToGo == 0) {
		/* Move the intergral number of unused2 blocks */
		pUnused2Anchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Split the part-endnote block, part-unused2 block */
		DBG_DEC(lBytesTooFar);
		fail(lBytesTooFar <= 0);
		pUnused2Anchor = xmalloc(sizeof(list_mem_type));
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		pUnused2Anchor->tInfo.ulFileOffset =
				pCurr->tInfo.ulFileOffset +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pUnused2Anchor->tInfo.ulFileOffset);
		DBG_HEX(pCurr->tInfo.ulCharPos);
		pUnused2Anchor->tInfo.ulCharPos =
				pCurr->tInfo.ulCharPos +
				pCurr->tInfo.ulLength -
				lBytesTooFar;
		DBG_HEX(pUnused2Anchor->tInfo.ulCharPos);
		pUnused2Anchor->tInfo.ulLength = (ULONG)lBytesTooFar;
		pCurr->tInfo.ulLength -= lBytesTooFar;
		pUnused2Anchor->tInfo.bUsesUnicode =
				pCurr->tInfo.bUsesUnicode;
		/* Move the integral number of unused2 blocks */
		pUnused2Anchor->pNext = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Unused2 block list */
	pCurr = NULL;
	lCharsToGo = (long)ulUnused2Len;
	lBytesTooFar = -1;
	if (ulUnused2Len != 0) {
		DBG_MSG("Unused2 block list");
		DBG_DEC(ulUnused2Len);
		for (pCurr = pUnused2Anchor;
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			DBG_DEC(pCurr->tInfo.ulLength);
			fail(pCurr->tInfo.ulLength == 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.ulLength));
				lCharsToGo -= pCurr->tInfo.ulLength / 2;
				if (lCharsToGo < 0) {
					lBytesTooFar = -2 * lCharsToGo;
				}
			} else {
				lCharsToGo -= pCurr->tInfo.ulLength;
				if (lCharsToGo < 0) {
					lBytesTooFar = -lCharsToGo;
				}
			}
			if (lCharsToGo <= 0) {
				break;
			}
		}
	}
/* Split the list */
	if (ulUnused2Len == 0) {
		/* Empty unused2 list */
		pGarbageAnchor = pUnused2Anchor;
		pUnused2Anchor = NULL;
	} else if (pCurr == NULL) {
		/* No garbage block list */
		pGarbageAnchor = NULL;
	} else if (lCharsToGo == 0) {
		/* Move the intergral number of garbage blocks */
		pGarbageAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	} else {
		/* Reduce the part-unused2 block */
		DBG_DEC(lBytesTooFar);
		fail(lBytesTooFar <= 0);
		pCurr->tInfo.ulLength -= lBytesTooFar;
		/* Move the integral number of garbage blocks */
		pGarbageAnchor = pCurr->pNext;
		pCurr->pNext = NULL;
	}
/* Free the garbage block list, this should never be needed */
	pCurr = pGarbageAnchor;
	while (pCurr != NULL) {
		DBG_FIXME();
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		DBG_DEC(pCurr->tInfo.ulLength);
		pNext = pCurr->pNext;
		pCurr = xfree(pCurr);
		pCurr = pNext;
	}

#if defined(DEBUG)
	/* Check the number of bytes in the block lists */
	ulTotal = 0;
	for (pCurr = pTextAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		NO_DBG_HEX(pCurr->tInfo.ulFileOffset);
		NO_DBG_HEX(pCurr->tInfo.ulCharPos);
		NO_DBG_DEC(pCurr->tInfo.ulLength);
		fail(pCurr->tInfo.ulLength == 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.ulLength));
			ulTotal += pCurr->tInfo.ulLength / 2;
		} else {
			ulTotal += pCurr->tInfo.ulLength;
		}
	}
	DBG_DEC(ulTotal);
	if (ulTotal != ulTextLen) {
		DBG_DEC(ulTextLen);
		werr(1, "Software error (Text)");
	}
	ulTotal = 0;
	for (pCurr = pFootAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		NO_DBG_HEX(pCurr->tInfo.ulCharPos);
		DBG_DEC(pCurr->tInfo.ulLength);
		fail(pCurr->tInfo.ulLength == 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.ulLength));
			ulTotal += pCurr->tInfo.ulLength / 2;
		} else {
			ulTotal += pCurr->tInfo.ulLength;
		}
	}
	DBG_DEC(ulTotal);
	if (ulTotal != ulFootnoteLen) {
		DBG_DEC(ulFootnoteLen);
		werr(1, "Software error (Footnotes)");
	}
	ulTotal = 0;
	for (pCurr = pUnused1Anchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		NO_DBG_HEX(pCurr->tInfo.ulCharPos);
		DBG_DEC(pCurr->tInfo.ulLength);
		fail(pCurr->tInfo.ulLength == 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.ulLength));
			ulTotal += pCurr->tInfo.ulLength / 2;
		} else {
			ulTotal += pCurr->tInfo.ulLength;
		}
	}
	DBG_DEC(ulTotal);
	if (ulTotal != ulUnused1Len) {
		DBG_DEC(ulUnused1Len);
		werr(1, "Software error (Unused1-list)");
	}
	ulTotal = 0;
	for (pCurr = pEndAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		NO_DBG_HEX(pCurr->tInfo.ulCharPos);
		DBG_DEC(pCurr->tInfo.ulLength);
		fail(pCurr->tInfo.ulLength == 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.ulLength));
			ulTotal += pCurr->tInfo.ulLength / 2;
		} else {
			ulTotal += pCurr->tInfo.ulLength;
		}
	}
	DBG_DEC(ulTotal);
	if (ulTotal != ulEndnoteLen) {
		DBG_DEC(ulEndnoteLen);
		werr(1, "Software error (Endnotes)");
	}
	ulTotal = 0;
	for (pCurr = pUnused2Anchor; pCurr != NULL; pCurr = pCurr->pNext) {
		DBG_HEX(pCurr->tInfo.ulFileOffset);
		NO_DBG_HEX(pCurr->tInfo.ulCharPos);
		DBG_DEC(pCurr->tInfo.ulLength);
		fail(pCurr->tInfo.ulLength == 0);
		if (pCurr->tInfo.bUsesUnicode) {
			fail(odd(pCurr->tInfo.ulLength));
			ulTotal += pCurr->tInfo.ulLength / 2;
		} else {
			ulTotal += pCurr->tInfo.ulLength;
		}
	}
	DBG_DEC(ulTotal);
	if (ulTotal != ulUnused2Len) {
		DBG_DEC(ulUnused2Len);
		werr(1, "Software error (Unused2-list)");
	}
#endif /* DEBUG */

	if (!bMustExtend) {
		return;
	}
	/*
	 * All blocks (except the last one) must have a length that
	 * is a multiple of the Big Block Size
	 */

	apAnchors[0] = pTextAnchor;
	apAnchors[1] = pFootAnchor;
	apAnchors[2] = pUnused1Anchor;
	apAnchors[3] = pEndAnchor;
	apAnchors[4] = pUnused2Anchor;

	for (iIndex = 0; iIndex < 5; iIndex++) {
		for (pCurr = apAnchors[iIndex];
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			if (pCurr->pNext != NULL &&
			    pCurr->tInfo.ulLength % BIG_BLOCK_SIZE != 0) {
				DBG_HEX(pCurr->tInfo.ulFileOffset);
				DBG_HEX(pCurr->tInfo.ulCharPos);
				DBG_DEC(pCurr->tInfo.ulLength);
				pCurr->tInfo.ulLength /= BIG_BLOCK_SIZE;
				pCurr->tInfo.ulLength++;
				pCurr->tInfo.ulLength *= BIG_BLOCK_SIZE;
				DBG_DEC(pCurr->tInfo.ulLength);
			}
		}
	}
} /* end of vSplitBlockList */

#if defined(__riscos)
/*
 * lGetDocumentLength - get the combined character length of the three lists
 *
 * returns: The total number of characters
 */
ULONG
ulGetDocumentLength(void)
{
	list_mem_type	*apAnchors[3];
	list_mem_type	*pCurr;
	long		ulTotal;
	int		iIndex;

	DBG_MSG("ulGetDocumentLength");

	apAnchors[0] = pTextAnchor;
	apAnchors[1] = pFootAnchor;
	apAnchors[2] = pEndAnchor;

	ulTotal = 0;
	for (iIndex = 0; iIndex < 3; iIndex++) {
		for (pCurr = apAnchors[iIndex];
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			fail(pCurr->tInfo.ulLength == 0);
			if (pCurr->tInfo.bUsesUnicode) {
				fail(odd(pCurr->tInfo.ulLength));
				ulTotal += pCurr->tInfo.ulLength / 2;
			} else {
				ulTotal += pCurr->tInfo.ulLength;
			}
		}
	}
	DBG_DEC(ulTotal);
	return ulTotal;
} /* end of ulGetDocumentLength */
#endif /* __riscos */

/*
 * usNextTextByte - get the next byte from the text block list
 */
static USHORT
usNextTextByte(FILE *pFile,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	static ULONG	ulBlockOffset = 0;
	static size_t	tByteNext = 0;
	ULONG	ulReadOff;
	size_t	tReadLen;

	if (pTextBlockCurrent == NULL ||
	    tByteNext >= sizeof(aucBlock) ||
	    ulBlockOffset + tByteNext >= pTextBlockCurrent->tInfo.ulLength) {
		if (pTextBlockCurrent == NULL) {
			/* First block, first part */
			pTextBlockCurrent = pTextAnchor;
			ulBlockOffset = 0;
		} else if (ulBlockOffset + sizeof(aucBlock) <
				pTextBlockCurrent->tInfo.ulLength) {
			/* Same block, next part */
			ulBlockOffset += sizeof(aucBlock);
		} else {
			/* Next block, first part */
			pTextBlockCurrent = pTextBlockCurrent->pNext;
			ulBlockOffset = 0;
		}
		if (pTextBlockCurrent == NULL) {
			/* Past the last part of the last block */
			return (USHORT)EOF;
		}
		tReadLen = (size_t)
			(pTextBlockCurrent->tInfo.ulLength - ulBlockOffset);
		if (tReadLen > sizeof(aucBlock)) {
			tReadLen = sizeof(aucBlock);
		}
		ulReadOff = pTextBlockCurrent->tInfo.ulFileOffset +
				ulBlockOffset;
		if (!bReadBytes(aucBlock, tReadLen, ulReadOff, pFile)) {
			return (USHORT)EOF;
		}
		tByteNext = 0;
	}
	if (pulFileOffset != NULL) {
		*pulFileOffset = pTextBlockCurrent->tInfo.ulFileOffset +
			ulBlockOffset + tByteNext;
	}
	if (pulCharPos != NULL) {
		*pulCharPos = pTextBlockCurrent->tInfo.ulCharPos +
			ulBlockOffset + tByteNext;
	}
	if (pusPropMod != NULL) {
		*pusPropMod = pTextBlockCurrent->tInfo.usPropMod;
	}
	return (USHORT)aucBlock[tByteNext++];
} /* end of usNextTextByte */

/*
 * usNextFootByte - get the next byte from the footnote block list
 */
static USHORT
usNextFootByte(FILE *pFile,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	static ULONG	ulBlockOffset = 0;
	static size_t	tByteNext = 0;
	ULONG	ulReadOff;
	size_t	tReadLen;

	if (pFootBlockCurrent == NULL ||
	    tByteNext >= sizeof(aucBlock) ||
	    ulBlockOffset + tByteNext >= pFootBlockCurrent->tInfo.ulLength) {
		if (pFootBlockCurrent == NULL) {
			/* First block, first part */
			pFootBlockCurrent = pFootAnchor;
			ulBlockOffset = 0;
		} else if (ulBlockOffset + sizeof(aucBlock) <
				pFootBlockCurrent->tInfo.ulLength) {
			/* Same block, next part */
			ulBlockOffset += sizeof(aucBlock);
		} else {
			/* Next block, first part */
			pFootBlockCurrent = pFootBlockCurrent->pNext;
			ulBlockOffset = 0;
		}
		if (pFootBlockCurrent == NULL) {
			/* Past the last part of the last block */
			return (USHORT)EOF;
		}
		tReadLen = (size_t)
			(pFootBlockCurrent->tInfo.ulLength - ulBlockOffset);
		if (tReadLen > sizeof(aucBlock)) {
			tReadLen = sizeof(aucBlock);
		}
		ulReadOff = pFootBlockCurrent->tInfo.ulFileOffset +
				ulBlockOffset;
		if (!bReadBytes(aucBlock, tReadLen, ulReadOff, pFile)) {
			return (USHORT)EOF;
		}
		tByteNext = 0;
	}
	if (pulFileOffset != NULL) {
		*pulFileOffset = pFootBlockCurrent->tInfo.ulFileOffset +
			ulBlockOffset + tByteNext;
	}
	if (pulCharPos != NULL) {
		*pulCharPos = pFootBlockCurrent->tInfo.ulCharPos +
			ulBlockOffset + tByteNext;
	}
	if (pusPropMod != NULL) {
		*pusPropMod = pFootBlockCurrent->tInfo.usPropMod;
	}
	return (USHORT)aucBlock[tByteNext++];
} /* end of usNextFootByte */

/*
 * usNextEndByte - get the next byte from the endnote block list
 */
static USHORT
usNextEndByte(FILE *pFile,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	static ULONG	ulBlockOffset = 0;
	static size_t	tByteNext = 0;
	ULONG	ulReadOff;
	size_t	tReadLen;

	if (pEndBlockCurrent == NULL ||
	    tByteNext >= sizeof(aucBlock) ||
	    ulBlockOffset + tByteNext >= pEndBlockCurrent->tInfo.ulLength) {
		if (pEndBlockCurrent == NULL) {
			/* First block, first part */
			pEndBlockCurrent = pEndAnchor;
			ulBlockOffset = 0;
		} else if (ulBlockOffset + sizeof(aucBlock) <
				pEndBlockCurrent->tInfo.ulLength) {
			/* Same block, next part */
			ulBlockOffset += sizeof(aucBlock);
		} else {
			/* Next block, first part */
			pEndBlockCurrent = pEndBlockCurrent->pNext;
			ulBlockOffset = 0;
		}
		if (pEndBlockCurrent == NULL) {
			/* Past the last part of the last block */
			return (USHORT)EOF;
		}
		tReadLen = (size_t)
			(pEndBlockCurrent->tInfo.ulLength - ulBlockOffset);
		if (tReadLen > sizeof(aucBlock)) {
			tReadLen = sizeof(aucBlock);
		}
		ulReadOff = pEndBlockCurrent->tInfo.ulFileOffset +
				ulBlockOffset;
		if (!bReadBytes(aucBlock, tReadLen, ulReadOff, pFile)) {
			return (USHORT)EOF;
		}
		tByteNext = 0;
	}
	if (pulFileOffset != NULL) {
		*pulFileOffset = pEndBlockCurrent->tInfo.ulFileOffset +
			ulBlockOffset + tByteNext;
	}
	if (pulCharPos != NULL) {
		*pulCharPos = pEndBlockCurrent->tInfo.ulCharPos +
			ulBlockOffset + tByteNext;
	}
	if (pusPropMod != NULL) {
		*pusPropMod = pEndBlockCurrent->tInfo.usPropMod;
	}
	return (USHORT)aucBlock[tByteNext++];
} /* end of usNextEndByte */

/*
 * usNextTextChar - get the next character from the text block list
 */
static USHORT
usNextTextChar(FILE *pFile,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	USHORT	usLSB, usMSB;

	usLSB = usNextTextByte(pFile, pulFileOffset, pulCharPos, pusPropMod);
	if (usLSB == (USHORT)EOF) {
		return (USHORT)EOF;
	}
	if (pTextBlockCurrent->tInfo.bUsesUnicode) {
		usMSB = usNextTextByte(pFile, NULL, NULL, NULL);
	} else {
		usMSB = 0x00;
	}
	if (usMSB == (USHORT)EOF) {
		DBG_MSG("usNextTextChar: Unexpected EOF");
		DBG_HEX_C(pulFileOffset != NULL, *pulFileOffset);
		DBG_HEX_C(pulCharPos != NULL, *pulCharPos);
		return (USHORT)EOF;
	}
	return (usMSB << 8) | usLSB;
} /* end of usNextTextChar */

/*
 * usNextFootChar - get the next character from the footnote block list
 */
static USHORT
usNextFootChar(FILE *pFile,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	USHORT	usLSB, usMSB;

	usLSB = usNextFootByte(pFile, pulFileOffset, pulCharPos, pusPropMod);
	if (usLSB == (USHORT)EOF) {
		return (USHORT)EOF;
	}
	if (pFootBlockCurrent->tInfo.bUsesUnicode) {
		usMSB = usNextFootByte(pFile, NULL, NULL, NULL);
	} else {
		usMSB = 0x00;
	}
	if (usMSB == (USHORT)EOF) {
		DBG_MSG("usNextFootChar: Unexpected EOF");
		DBG_HEX_C(pulFileOffset != NULL, *pulFileOffset);
		DBG_HEX_C(pulCharPos != NULL, *pulCharPos);
		return (USHORT)EOF;
	}
	return (usMSB << 8) | usLSB;
} /* end of usNextFootChar */

/*
 * usNextEndChar - get the next character from the endnote block list
 */
static USHORT
usNextEndChar(FILE *pFile,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	USHORT	usLSB, usMSB;

	usLSB = usNextEndByte(pFile, pulFileOffset, pulCharPos, pusPropMod);
	if (usLSB == (USHORT)EOF) {
		return (USHORT)EOF;
	}
	if (pEndBlockCurrent->tInfo.bUsesUnicode) {
		usMSB = usNextEndByte(pFile, NULL, NULL, NULL);
	} else {
		usMSB = 0x00;
	}
	if (usMSB == (USHORT)EOF) {
		DBG_MSG("usNextEndChar: Unexpected EOF");
		DBG_HEX_C(pulFileOffset != NULL, *pulFileOffset);
		DBG_HEX_C(pulCharPos != NULL, *pulCharPos);
		return (USHORT)EOF;
	}
	return (usMSB << 8) | usLSB;
} /* end of usNextEndChar */

/*
 * usNextChar - get the next character from the given block list
 */
USHORT
usNextChar(FILE *pFile, list_id_enum eListID,
	ULONG *pulFileOffset, ULONG *pulCharPos, USHORT *pusPropMod)
{
	fail(pFile == NULL);

	switch (eListID) {
	case text_list:
		return usNextTextChar(pFile,
				pulFileOffset, pulCharPos, pusPropMod);
	case footnote_list:
		return usNextFootChar(pFile,
				pulFileOffset, pulCharPos, pusPropMod);
	case endnote_list:
		return usNextEndChar(pFile,
				pulFileOffset, pulCharPos, pusPropMod);
	default:
		if (pulFileOffset != NULL) {
			*pulFileOffset = FC_INVALID;
		}
		if (pulCharPos != NULL) {
			*pulCharPos = CP_INVALID;
		}
		if (pusPropMod != NULL) {
			*pusPropMod = IGNORE_PROPMOD;
		}
		return (USHORT)EOF;
	}
} /* end of usNextChar */

/*
 * Translate a character position to an offset in the file.
 * Logical to physical offset.
 *
 * Returns:	FC_INVALID: in case of error
 *		otherwise: the computed file offset
 */
ULONG
ulCharPos2FileOffset(ULONG ulCharPos)
{
	list_mem_type	*apAnchors[3];
	list_mem_type	*pCurr;
	ULONG		ulBestGuess;
	int		iIndex;

	apAnchors[0] = pTextAnchor;
	apAnchors[1] = pFootAnchor;
	apAnchors[2] = pEndAnchor;

	ulBestGuess = FC_INVALID; /* Best guess is "fileoffset not found" */

	for (iIndex = 0; iIndex < 3; iIndex++) {
		for (pCurr = apAnchors[iIndex];
		     pCurr != NULL;
		     pCurr = pCurr->pNext) {
			if (ulCharPos == pCurr->tInfo.ulCharPos +
			     pCurr->tInfo.ulLength &&
			    pCurr->pNext != NULL) {
				/*
				 * The textoffset is one beyond this block,
				 * so we guess it's the first byte of the next
				 * block (if there is a next block)
				 */
				ulBestGuess = pCurr->pNext->tInfo.ulFileOffset;
			}

			if (ulCharPos < pCurr->tInfo.ulCharPos ||
			    ulCharPos >= pCurr->tInfo.ulCharPos +
			     pCurr->tInfo.ulLength) {
				/* The textoffset is not in this block */
				continue;
			}

			/* The textoffset is in the current block */
			return pCurr->tInfo.ulFileOffset +
				ulCharPos - pCurr->tInfo.ulCharPos;
		}
	}
	/* Passed beyond the end of the last list */
	NO_DBG_HEX(ulCharPos);
	NO_DBG_HEX(ulBestGuess);
	return ulBestGuess;
} /* end of ulCharPos2FileOffset */

/*
 * Get the sequence number beloning to the given file offset
 *
 * Returns the sequence number
 */
ULONG
ulGetSeqNumber(ULONG ulFileOffset)
{
	list_mem_type	*pCurr;
	ULONG		ulSeq;

	if (ulFileOffset == FC_INVALID) {
		return FC_INVALID;
	}

	ulSeq = 0;
	for (pCurr = pTextAnchor; pCurr != NULL; pCurr = pCurr->pNext) {
		if (ulFileOffset >= pCurr->tInfo.ulFileOffset &&
		    ulFileOffset < pCurr->tInfo.ulFileOffset +
		     pCurr->tInfo.ulLength) {
			/* The file offset is within the current textblock */
			return ulSeq + ulFileOffset - pCurr->tInfo.ulFileOffset;
		}
		ulSeq += pCurr->tInfo.ulLength;
	}
	return FC_INVALID;
} /* end of ulGetSeqNumber */
