/* AbiSource Program Utilities
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

#ifndef _ODI_ELEMENTSTACK_H_
#define _ODI_ELEMENTSTACK_H_

// AbiWord includes
#include <ut_types.h>
#include <ut_vector.h>

// Internal classes
class ODi_StartTag;


/**
 * This class stores info about the parent elements of the current element being
 * parsed.
 * 
 * Example:
 * 
 * If you are parsing the <style:footnote-sep/> element, this class will hold
 * info about its parents. On this case: <style:page-layout-properties>,
 * <style:page-layout> and <office:automatic-styles>.
 * 
 * <office:automatic-styles>
 *    <style:page-layout>
 *        <style:page-layout-properties>
 *            <style:footnote-sep/>
 *        </style:page-layout-properties>
 *    </style:page-layout>
 * </office:automatic-styles>
 * 
 * 
 * When parsing a start element tag, level 0 is the parent element.
 * 
 * When parsing a start element tag, level 0 is its own element, with level 1
 * being the parent and so on.
 * 
 */
class ODi_ElementStack {
    
public:

    ODi_ElementStack();
    ~ODi_ElementStack();

    /**
     * @param level 0 is the immediate parent, 1 is the parent of the parent
     *              and so on.
     * 
     * On the startElement method, level 0 is the parent start tag.
     * On the endElement method, level 0 is the corresponding start tag.
     */
    const ODi_StartTag* getStartTag(UT_uint32 level);
    
    bool hasElement(const XML_Char* pName) const;
    
    /**
     * Returns the level of the closest element with the given name.
     */
    UT_uint32 getElementLevel(const XML_Char* pName) const;
    
    /**
     * Returns the closest parent with the given name. It returns NULL if there
     * is no parent with the given name.
     * 
     * @param pName Element name.
     * @param fromLevel The level from which the search begins.
     */
    const ODi_StartTag* getClosestElement(const XML_Char* pName,
                                          UT_uint32 fromLevel = 0) const;
        
    
    void startElement (const XML_Char* pName, const XML_Char** ppAtts);
    void endElement (const XML_Char* pName);
    bool isEmpty() const {return m_stackSize==0;}
    void clear() {m_stackSize = 0;}
    
    UT_uint32 getStackSize() const {return m_stackSize;}

    
private:
    
    UT_GenericVector <ODi_StartTag*>* m_pStartTags;
    UT_uint32 m_stackSize;
};

#endif //_ODI_ELEMENTSTACK_H_
