/*

Copyright (C) 2000 - 2006 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_debug.h>
#include <libnd_globals.h>
#include <libnd_packet.h>
#include <libnd_prefs.h>
#include <libnd_packet_recycler.h>

/* The number of slots in the recycler array */
#define LND_PREC_SLOTS                 21

/* A threshold for the payload size, starting from which
 * (inclusive) the "huge" slot is used.
 */
#define LND_PREC_HUGE_THRESH           ((LND_PREC_SLOTS - 1) * 100)

/* The index of the "huge" slot.
 */
#define LND_PREC_HUGE_SLOT             (LND_PREC_SLOTS - 1)

static GList **recycler = NULL;
static guint   recycler_size = 0;
static guint   recycler_size_max = 0;

static guint
prec_findslot(guint mem_size)
{
  if (mem_size >= LND_PREC_HUGE_THRESH)
    return LND_PREC_HUGE_SLOT;

  return mem_size / 100;
}


static guint
prec_padmem(guint mem_req)
{
  /* Round up memory requirement to next 100. */
  return mem_req + (100 - (mem_req % 100));
}


void         
libnd_prec_init()
{
  if (recycler)
    return;

  recycler = g_new0(GList*, LND_PREC_SLOTS);
  
  if (! libnd_prefs_get_int_item(LND_DOM_NETDUDE, "num_recycled_packets",
				 (int *) &recycler_size_max))
    recycler_size_max = 1000;  

  D(("Using maximum recycler size of %i packets.\n", recycler_size_max));
}


gboolean
libnd_prec_put(LND_Packet *packet)
{
  guint   slot;

  if (!packet)
    return FALSE;

  if (recycler_size >= recycler_size_max)
    return FALSE;

  slot = prec_findslot(packet->ph.caplen);
  recycler[slot] = g_list_prepend(recycler[slot], packet);
  recycler_size++;

  return TRUE;
}


LND_Packet   *
libnd_prec_get(guint mem_needed)
{
  guchar *data;
  guint      slot;
  LND_Packet *result;

  slot = prec_findslot(mem_needed);
  
  if (!recycler[slot])
    {
      result = g_new0(LND_Packet, 1);
      result->data_size = prec_padmem(mem_needed);
      result->data = malloc(sizeof(guchar) * result->data_size);
      
      return result;
    }

  result = (LND_Packet *) recycler[slot]->data;
  recycler[slot] = g_list_remove_link(recycler[slot], recycler[slot]);
  recycler_size--;

  data = result->data;
  memset(result, 0, sizeof(LND_Packet));
  result->data = data;

  return result;
}

guint
libnd_prec_calc_mem(guint mem_req)
{
  return prec_padmem(mem_req);
}
