/* $Id: input.c,v 1.3 1998/11/26 00:05:16 marcus Exp $
******************************************************************************

   Standalone X inputlib

   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <X11/Xlib.h>

#include <ggi/internal/gii-dl.h>
#include "xev.h"

typedef struct {
	int      fd;
	Display *disp;
	Window   win;
	Cursor   cursor;
	int      width, height;
	int      oldx, oldy;
} x_priv;

#define X_PRIV(inp)	((x_priv*)((inp)->priv))

static Cursor make_cursor(Display *disp, Window win)
{
	char emptybm[] = {0};
	Pixmap crsrpix;
	XColor nocol;
	Cursor mycrsr;

	crsrpix = XCreateBitmapFromData(disp, win, emptybm, 1, 1);
	mycrsr = XCreatePixmapCursor(disp, crsrpix, crsrpix,
				     &nocol, &nocol, 0, 0);
	XFreePixmap(disp, crsrpix);

	return mycrsr;
}

static int do_grab(Display *disp, Window win, Cursor crsr)
{
	if (XGrabKeyboard(disp, win, True, GrabModeAsync, GrabModeAsync,
			  CurrentTime)
	    != GrabSuccess ||
	    XGrabPointer(disp, win, True,
			 PointerMotionMask|ButtonPressMask|ButtonReleaseMask,
			 GrabModeAsync, GrabModeAsync, None, None, CurrentTime)
	    != GrabSuccess) {
		DPRINT_LIBS("input-X: Unable to grab input\n");
		return -1;
	}

	return 0;
}

static inline void center_pointer(x_priv *priv)
{
	XEvent event;
	event.type = MotionNotify;
	event.xmotion.display = priv->disp;
	event.xmotion.window = priv->win;
	event.xmotion.x = priv->width/2;
	event.xmotion.y = priv->height/2;
	XSendEvent(priv->disp, priv->win, False, PointerMotionMask, &event);
	XWarpPointer(priv->disp, None, priv->win, 0, 0, 0, 0,
		     priv->width/2, priv->height/2);
}

static gii_event_mask GII_x_eventpoll(gii_input *inp)
{
	x_priv  *priv = X_PRIV(inp);
	int n;
	int rc = 0;
	gii_event releasecache;
	Time      releasetime = 0;
	int       havecached = 0;

	n = XPending(priv->disp);

	while (n) {
		XEvent xev;
		gii_event giiev;
		
		n--;

		XNextEvent(priv->disp, &xev);
		_giiEventBlank(&giiev);

		switch(xev.type) { 
		case KeyPress:
			giiev.any.size = sizeof(gii_key_event);
			giiev.any.type = evKeyPress;
			giiev.key.button = xev.xkey.keycode;

			if (havecached) {
				if (xev.xkey.time == releasetime) {
					giiev.any.type = evKeyRepeat;
					rc |= emKeyRepeat;
				} else {
					_giiEvQueueAdd(inp, &releasecache);
					rc |= emKeyRelease;
					rc |= emKeyPress;
				}
				havecached = 0;
			} else {
				rc |= emKeyPress;
			}
			giiev.key.sym = _gii_xev_ev2sym((XKeyEvent*)&xev);
			giiev.key.label = _gii_xev_ev2label((XKeyEvent*)&xev);
			giiev.key.effect =
				_gii_xev_state2effect(xev.xkey.state);
			DPRINT_EVENTS("GII_x_eventpoll: KeyPress\n");
			break;

		case KeyRelease: 
			if (havecached) {
				_giiEvQueueAdd(inp, &releasecache);
				rc |= emKeyRelease;
			}
			_giiEventBlank(&releasecache);
			releasecache.any.origin = inp->origin;
			releasecache.any.type = evKeyRelease;
			releasecache.any.size = sizeof(gii_key_event);
			releasecache.key.sym
				= _gii_xev_ev2sym((XKeyEvent*)&xev);
			releasecache.key.label
				= _gii_xev_ev2label((XKeyEvent*)&xev);
			releasecache.key.button = xev.xkey.keycode;
			releasecache.key.effect =
				_gii_xev_state2effect(xev.xkey.state);

			havecached = 1;
			releasetime = xev.xkey.time;
			DPRINT_EVENTS("GII_x_eventpoll: KeyRelease\n");
			break;

		case ButtonPress:
			giiev.any.type = evPtrButtonPress;
			rc |= emPtrButtonPress;
			DPRINT_EVENTS("GII_x_eventpoll: ButtonPress %x\n",
				      xev.xbutton.state); 
			break;

		case ButtonRelease:
			giiev.any.type = evPtrButtonRelease;
			rc |= emPtrButtonRelease;
			DPRINT_EVENTS("GII_x_eventpoll: ButtonRelease %x\n",
				      xev.xbutton.state); 
			break;

		case MotionNotify:
		{
			int realmove = 0;

			if (!xev.xmotion.send_event) {
				giiev.pmove.x =
					xev.xmotion.x - priv->oldx;
				giiev.pmove.y =
					xev.xmotion.y - priv->oldy;
				realmove = 1;
#undef ABS
#define ABS(a) (((a)<0) ? -(a) : (a) )
				if (ABS(priv->width/2 - xev.xmotion.x)
				    > priv->width / 4
				    || ABS(priv->height/2 - xev.xmotion.y)
				    > priv->height / 4) {
#undef ABS	
					center_pointer(priv);
				}
			}
			priv->oldx = xev.xmotion.x;
			priv->oldy = xev.xmotion.y;
			if (!realmove) goto dont_queue_this_event;
			giiev.any.size = sizeof(gii_pmove_event);
			giiev.any.type = evPtrRelative;
			DPRINT_EVENTS("GII_x_eventpoll: MouseMove %d,%d\n",
				      giiev.pmove.x, giiev.pmove.y);
			break;
		}
		}
		switch(giiev.any.type) {
		case evPtrButtonPress:
		case evPtrButtonRelease:
			giiev.any.size = sizeof(gii_pbutton_event);
			giiev.pbutton.button =
				_gii_xev_buttontrans(xev.xbutton.button);
			break;
		}
		if (giiev.any.type) {
			giiev.any.origin = inp->origin;
			_giiEvQueueAdd(inp, &giiev);
		}
		dont_queue_this_event:
		/* "ANSI C forbids label at end of compound statement"
		   Well, this makes GCC happy at least... */
		while(0){};
	}
	if (havecached) {
		_giiEvQueueAdd(inp, &releasecache);
		rc |= emKeyRelease;
	}

	return rc;
}

static int GII_x_close(gii_input *inp)
{
	x_priv  *priv = X_PRIV(inp);
	
	XDestroyWindow(priv->disp, priv->win);
	XCloseDisplay(priv->disp);
	free(priv);
	
	return 0;
}


int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	Display *disp;
	Window   win;
	Screen  *sc;
	XSetWindowAttributes attr;
	int      scr;
	Cursor   crsr;
	x_priv  *priv;

	if ((disp = XOpenDisplay(NULL)) == NULL) {
		DPRINT_LIBS("input-X: Unable to open display\n");
		return GGI_ENODEVICE;
	}
 
	scr = XScreenNumberOfScreen(sc=DefaultScreenOfDisplay(disp));

	attr.event_mask =
		KeyPressMask | KeyReleaseMask |
		ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
	win = XCreateWindow(disp, RootWindow(disp, scr), 0, 0,
			    WidthOfScreen(sc),  HeightOfScreen(sc), 0, 0,
			    InputOnly, DefaultVisual(disp, scr),
			    CWEventMask , &attr);

	XMapRaised(disp, win);
	XSync(disp, 0);

	crsr = make_cursor(disp, win);

	if (do_grab(disp, win, crsr) != 0) {
		XDestroyWindow(disp, win);
		XCloseDisplay(disp);
		return GGI_ENODEVICE;
	}
	
	if ((priv = malloc(sizeof(x_priv))) == NULL) {
		XDestroyWindow(disp, win);
		XCloseDisplay(disp);
		return GGI_ENOMEM;
	}
	
	priv->fd = ConnectionNumber(disp);
	priv->disp = disp;
	priv->win = win;
	priv->cursor = crsr;
	
	{
		Window dummywin;
		unsigned int w, h;
		int dummy;
		
		XGetGeometry(disp, win, &dummywin, &dummy, &dummy,
			     &w, &h, &dummy, &dummy);
		priv->width = w;
		priv->height = h;
		priv->oldx = w/2;
		priv->oldy = h/2;
		center_pointer(priv);
	}

	inp->priv = priv;
	
	inp->GIIeventpoll = GII_x_eventpoll;
	inp->GIIclose = GII_x_close;

	inp->targetcan = emKey | emPointer;
	inp->curreventmask = emKey | emPointer;

	inp->maxfd = priv->fd + 1;
	FD_SET(priv->fd, &inp->fdset);

	return 0;
}
