/* $Id: input.c,v 1.17 1998/12/17 04:42:40 ajapted Exp $
******************************************************************************

   Linux_kbd: input

   Copyright (C) 1998 Andrew Apted     [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <unistd.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <linux/kd.h>
#include <linux/vt.h>
#include <linux/keyboard.h>

#include <ggi/internal/gii.h>
#include "linkey.h"

typedef struct keyboard_hook
{
	int fd;
	int eof;

	int old_mode;
	struct termios old_termios;
	
	int effect;

	unsigned char keydown_buf[128];

} KeyboardHook;

#define KEYBOARD_HOOK(inp)	((KeyboardHook *) inp->priv)


/* ---------------------------------------------------------------------- */


static int GII_keyboard_init(gii_input *inp, char *filename)
{
	int fd;
	struct termios new;
	KeyboardHook *kk;

	/* open the tty */

	fd = open(filename, O_RDWR);

	if (fd < 0) {
		perror("Linux_kbd: Couldn't open TTY");
		return GGI_ENODEVICE;;
	}

	/* allocate keyboard hook */

	if ((kk = malloc(sizeof(KeyboardHook))) == NULL) {
		close(fd);
		return GGI_ENOMEM;
	}

	/* put tty into "straight through" mode.
	 */

	if (tcgetattr(fd, &kk->old_termios) < 0) {
		perror("Linux_kbd: tcgetattr failed");
	}

	new = kk->old_termios;

	new.c_lflag &= ~(ICANON | ECHO  | ISIG);
	new.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
	new.c_iflag |= IGNBRK;
	new.c_cc[VMIN]  = 0;
	new.c_cc[VTIME] = 0;

	if (tcsetattr(fd, TCSANOW, &new) < 0) {
		perror("Linux_kbd: tcsetattr failed");
	}

	/* Put the keyboard into mediumraw mode.  Despite the name, this
	 * is really "mostly raw", with the kernel just folding long
	 * scancode sequences (e.g. E0 XX) onto single keycodes.
	 */

	if (ioctl(fd, KDGKBMODE, &kk->old_mode) < 0) {
		perror("Linux_kbd: couldn't get mode");
		kk->old_mode = K_XLATE;
	}

	if (ioctl(fd, KDSKBMODE, K_MEDIUMRAW) < 0) {
		perror("Linux_kbd: couldn't set raw mode");
		tcsetattr(fd, TCSANOW, &(kk->old_termios));
		close(fd);
		free(kk);
		return GGI_ENODEVICE;
	}

	kk->fd = fd;
	kk->eof = 0;
	inp->priv = kk;

	DPRINT("Linux_kbd: init OK.\n");

	return 0;
}


static gii_event_mask GII_keyboard_handle_data(gii_input *inp)
{
	KeyboardHook *kk = KEYBOARD_HOOK(inp);
	gii_event ev;
	struct kbentry entry;
	unsigned char buf[32];
	int code, shift;
	int read_len;

	/* calculate kernel-like shift value */

	shift = ((kk->effect & (1<<GII_KM_SHIFT)) ? (1<<KG_SHIFT) : 0) |
		((kk->effect & (1<<GII_KM_CTRL))  ? (1<<KG_CTRL)  : 0) |
		((kk->effect & (1<<GII_KM_ALT))   ? (1<<KG_ALT)   : 0) |
		((kk->effect & (1<<GII_KM_META))  ? (1<<KG_ALT)   : 0);


	_giiEventBlank(&ev);

	/* first get the keycode */

	read_len = read(kk->fd, buf, 1);
	if (read_len == 0) {
		/* End-of-file occured */
		kk->eof = 1;
		DPRINT_EVENTS("Linux-kbd: EOF occured on fd: %d\n", kk->fd);
		return 0;
	} else if (read_len < 0) {
		/* Error, we try again next time */
		perror("Linux_kbd: Error reading keyboard");
		return 0;
	}

	code = buf[0];

	if (code & 0x80) {
		code &= 0x7f;
		ev.key.type = evKeyRelease;
		kk->keydown_buf[code] = 0;

	} else if (kk->keydown_buf[code] == 0) {
		ev.key.type = evKeyPress;
		kk->keydown_buf[code] = 1;

	} else {
		ev.key.type = evKeyRepeat;
	}

	ev.key.button = code;
	
	
	/* Look up the keysym without modifiers, which will give us
	 * the key label (more or less).
	 */

	entry.kb_table = 0;
	entry.kb_index = code;
	
	if (ioctl(kk->fd, KDGKBENT, &entry) < 0) {
		DPRINT("Linux_kbd: ioctl(KDGKBENT) failed.\n");
		return 0;
	}

	ev.key.label = _gii_linkey_labeltrans(entry.kb_value, shift);


	/* Now look up the full keysym in the kernel's table */

	entry.kb_table = shift;
	entry.kb_index = code;
	
	if (ioctl(kk->fd, KDGKBENT, &entry) < 0) {
		DPRINT("Linux_kbd: ioctl(KDGKBENT) failed.\n");
		return 0;
	}

	switch (entry.kb_value) {
		case K_NOSUCHMAP: case K_HOLE:
			return 0;
	}

	ev.key.sym = _gii_linkey_symtrans(entry.kb_value, shift);


	/* keep track of shift state */

	if (GII_KTYP(ev.key.sym) == GII_KT_SHIFT) {
		
		int mask = 1 << (ev.key.sym & GII_KM_MASK);
		
		if (ev.key.type == evKeyRelease) {
			kk->effect &= ~mask;
		} else {
			kk->effect |= mask;
		}
	}
	
	ev.key.effect = kk->effect;


	DPRINT_EVENTS("KEY-%s button=0x%02x effect=0x%02x "
		"sym=0x%04x label=0x%04x\n",
		(ev.key.type == evKeyRelease) ? "UP" : "DN",
		ev.key.button, ev.key.effect, 
		ev.key.sym,  ev.key.label);


	/* Check for console switch.  Unfortunately, the kernel doesn't
	 * recognize KT_CONS when the keyboard is in RAW or MEDIUMRAW
	 * mode, so _we_ have to.  Sigh.
	 */
	
	if (KTYP(entry.kb_value) == KT_CONS) {

		int new_cons = 1+KVAL(entry.kb_value);

		DPRINT_MISC("Switching to console %d.\n", new_cons);
		
		if (ioctl(kk->fd, VT_ACTIVATE, new_cons) < 0) {
			  perror("ioctl(VT_ACTIVATE)");
		}
		
		/* No need to wait till it's active, the vt_process
		 * signal handler does that for us.
		 */
		 
		return 0;
	}

	
	/* finally queue the key event */

	ev.any.size   = sizeof(gii_key_event);
	ev.any.origin = inp->origin;
	ev.any.target = GII_EV_TARGET_ALL;

	_giiEvQueueAdd(inp, &ev);

	return (1 << ev.any.type);
}


static gii_event_mask GII_keyboard_poll(gii_input *inp)
{
	KeyboardHook *kk = KEYBOARD_HOOK(inp);
	gii_event_mask result = 0;
		
	if (kk->eof) {
		/* End-of-file, don't do any polling */
		return 0;
	}

	DPRINT_MISC("linux_kbd: poll(%p)\n", inp);
	
	for (;;) {
		fd_set readset;
		struct timeval t = {0,0};
		int rc;

		FD_ZERO(&readset);
		FD_SET(kk->fd, &readset);

		rc = select(inp->maxfd, &readset, NULL, NULL, &t);

		if (rc <= 0) break;

		result |= GII_keyboard_handle_data(inp);
		
		if (kk->eof) break;
	}

	return result;
}


/* ---------------------------------------------------------------------- */


static gii_cmddata_getdevinfo linux_kbd_devinfo =
{
	"Linux Keyboard",		/* long device name */
	"lkbd",				/* shorthand */
	emKey,				/* can_generate */
	128,				/* num_buttons */
	0				/* num_axes */
};

static int GII_linux_kbd_getdevinfo(gii_input *inp, gii_event *ev)
{
	gii_cmddata_getdevinfo *DI;
	

	_giiEventBlank(ev);
	
	ev->any.size   = sizeof(gii_cmd_nodata_event) +
		         sizeof(gii_cmddata_getdevinfo);
	ev->any.type   = evCommand;
	ev->any.origin = inp->origin;
	ev->cmd.code   = GII_CMDCODE_GETDEVINFO;

	DI = (gii_cmddata_getdevinfo *) &ev->cmd.data;

	*DI = linux_kbd_devinfo;
	
	return 0;
}


/* ---------------------------------------------------------------------- */


static int GII_linux_kbd_close(gii_input *inp)
{
	KeyboardHook *kk = KEYBOARD_HOOK(inp);

	DPRINT_MISC("Linux_kbd cleanup\n");

	if (kk == NULL) {
		return -1;
	}

	if (ioctl(kk->fd, KDSKBMODE, kk->old_mode) < 0) {
		perror("Linux_kbd: couldn't set mode");
	}

	if (tcsetattr(kk->fd, TCSANOW, &kk->old_termios) < 0) {
		perror("Linux_kbd: tcsetattr failed");
	}

	close(kk->fd);
	kk->fd = -1;

	free(kk);
	inp->priv = NULL;

	ggUnregisterCleanup((ggcleanup_func *)GII_linux_kbd_close, (void *)inp);

	DPRINT("Linux_kbd: exit OK.\n");

	return 0;
}

int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	char *filename = "/dev/tty";
	gii_event ev;

	
	DPRINT_MISC("linux_kbd starting.(args=\"%s\",argptr=%p)\n",
		    args, argptr);

	/* Initialize */

	if (args && *args) {
		filename = (char *) args;
	}

	if (GII_keyboard_init(inp, filename) < 0) {
		return -1;
	}
	
	/* Make sure the keyboard is reset when the app terminates */
	ggRegisterCleanup((ggcleanup_func *)GII_linux_kbd_close, (void *)inp);

	inp->GIIeventpoll = GII_keyboard_poll;
	inp->GIIclose = GII_linux_kbd_close;

	inp->targetcan = emKey;
	inp->GIIseteventmask(inp, emKey);

	inp->maxfd = KEYBOARD_HOOK(inp)->fd + 1;
	FD_SET(KEYBOARD_HOOK(inp)->fd, &inp->fdset);

	/* Send initial cmdDevInfo event */
	GII_linux_kbd_getdevinfo(inp, &ev);
	_giiEvQueueAdd(inp, &ev);
	
	DPRINT_MISC("linux_kbd fully up\n");

	return 0;
}
