/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
*/

#ifndef __KPLAYERLIST_H 
#define __KPLAYERLIST_H 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

#include <klistview.h>
#include <qpopupmenu.h>
#include <qtabdialog.h>
#include <kstatusbar.h>
#include <qcheckbox.h>
#include <kaction.h>


/**
 *
 * Simple extension to the QListViewItem class that has a custom key 
 * function. The key function can deal with the different items of the
 * player list.
 *
 */
class KFibsPlayerListLVI : public QListViewItem
{
public:
	
	/**
	 * Constructor, just calls QListViewItem()
	 */
	KFibsPlayerListLVI(QListView * parent);

	/**
	 * Destructor is empty
	 */
	virtual ~KFibsPlayerListLVI();
	
	/**
	 * Overloaded key for sorting
	 */
	virtual QString key(int column, bool ascending) const;
};


/**
 *
 * Simple container for information on columns of the list view
 *
 */
struct columnInfo 
{
	int index;
	int width;
	bool show;
	QCheckBox *cb;
};


/**
 *
 * A class that keeps track of players on the server. The server is flooding 
 * us with user information. At any given time we are able to have an current
 * list of all loged-in players and their status.
 *
 */
class KFibsPlayerList : public KListView
{
	Q_OBJECT
	
public:

	/**
	 * Enumerate the different columns of the list 
	 */
	enum {Player, Opponent, Watches, Status, Rating, Experience, Idle, Time, Host, Client, Email, LVEnd};

	/**
	 * Constructor 
	 */
	KFibsPlayerList(QWidget *parent = 0, const char *name = 0);
	
	/**
	 * Destructor
	 */
	virtual ~KFibsPlayerList();

	/**
	 * Clear the list and reset the client counter
	 */
	virtual void clear();

public slots:

        /**
	 * Remove the player with the name given by the first word
	 */
        void deletePlayer(const QString &player);
	
	/**
	 * Change/Add the entry for the given player
	 */
	void changePlayer(const QString &line);
	
	/**
	 * Enables list redraws after an update
	 */
	void stopUpdate();
	
	/**
	 * Read the UI settings from disk
	 */
	void readConfig();
	
	/**
	 * Restore settings from previously stored settings
	 */
	void saveConfig();

	/**
	 * Change the status of a player
	 */
	void changePlayerStatus(const QString &player, int stat, bool flag);
	
	/**
	 * Fills the playerlist page into the notebook
	 */
	virtual void getSetupPages(QTabDialog *nb);
	
protected slots:

        /**
	 * Double click handler, requests information on a player
	 */
        void getPlayerInfo(QListViewItem *i, const QPoint &p, int col);

	/**
	 * Display a popup menu for the current player
	 */
	void showContextMenu(KListView *, QListViewItem *, const QPoint &);
	
	/**
	 * Request an invitation of opPlayer
	 */
	void invitePlayer();
	
	/**
	 * Request information on opPlayer
	 */
	void infoPlayer();
	
	/**
	 * Look at opPlayer
	 */
	void lookPlayer();
	
	/**
	 * Send an email to opPlayer
	 */
	void mailPlayer();
	
	/**
	 * Talk to opPlayer
	 */
	void talkPlayer();
	
	/**
	 * Watch opPlayer
	 */
	void watchPlayer();
	
	/**
	 * Stop watching
	 */
	void unwatchPlayer();
	
	/**
	 * Blind opPlayer
	 */
	void blindPlayer();
	
	/**
	 * Update line of opPlayer
	 */
	void updatePlayer();
	
	/**
	 * Reload the whole list
	 */
	void reloadList();

	/**
	 * Upate the caption
	 */
	void updateCaption();

	/**
	 * Save setting changes
	 */
	void setupOk();

signals:

	/**
	 * Send a command to the server
	 */
	void fibsCommand(const QString &);
	
	/**
	 * Initiate an invitation of a player
	 */
	void fibsInvite(const QString &);

	/**
	 * Request talking to player opPlayer
	 */
	void fibsTalk(const QString &);

	/**
	 * Allow the engine's menu to be updated
	 */
	void windowVisible(bool);

protected:
	
	/**
	 * Catch show events, so the engine's menu can be update. 
	 */
	virtual void showEvent(QShowEvent *e);

	/**
	 * Catch hide events, so the engine's menu can be update. 
	 */
	virtual void hideEvent(QHideEvent *e);

	/**
	 * Context menu for player related commands
	 */
	QPopupMenu *menu;
	
	/**
	 * Name of the last selected player - for internal purposes
	 */
	QString opPlayer;

	/**
	 * Email address of the last selected player - for internal purposes
	 */
	QString opEmail;

	/**
	 * count similar clients - KFibs & kbackgammon
	 */
	int cCount[2];

	/**
	 * All relevant information on the columns 
	 */
	struct columnInfo *column[LVEnd];

	/**
	 * Column titles for the UI
	 */
	static QString columnTitle[LVEnd];

	/**
	 * Internal names for the columns for the config files
	 */
	static QString columnKey[LVEnd];

	/**
	 * Named constants for the popup menu actions
	 */
	enum MenuID {Info, Talk, Mail, Invite, Look, Watch, Unwatch, Blind, Update, Reload, ActionEnd}; 

	/**
	 * Various actions for the context menu
	 */
	KAction *act[ActionEnd];
};

#endif // __KPLAYERLIST_H 
