/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
*/

/*
  TODO
  ----

  - do not offer invite if the player is not ready
  - make invite shortcuts for 1,3.5

*/

#include "kplayerlist.moc"
#include "kplayerlist.h"

#include <kapp.h>
#include <string.h>
#include <qregexp.h>
#include <klocale.h>
#include <stdlib.h>
#include <stdio.h>
#include <kconfig.h>
#include <qlayout.h>
#include <iostream.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qwhatsthis.h>
#include <qdatetime.h>

#include "version.h"
#include "clip.h"
#include "kbgfibs.h"

#include <iostream.h>


// == constructor, destructor and setup ========================================

/*
 * The titles of the column headers
 */
QString KFibsPlayerList::columnTitle[LVEnd] = {
	
	i18n("Player"),
	i18n("Opponent"),
	i18n("Watching"),
	i18n("Status"),
	i18n("Rating"),
	i18n("Exp."),
	i18n("Idle"),
	i18n("Time"),
	i18n("Hostname"),
	i18n("Client"),
	i18n("Email")
};

/*
 * These strings shouldn't be translated!!
 */
QString KFibsPlayerList::columnKey[LVEnd] = {
	
	"player",
	"opponent",
	"watching",
	"status",
	"rating",
	"experience",
	"idle",
	"time",
	"hostname",
	"client",
	"email"
};

/*
 * Construct the playerlist and do some initial setup
 */
KFibsPlayerList::KFibsPlayerList(QWidget *parent, const char *name)
	: KListView(parent, name)
{
	cCount[0] = cCount[1] = 0;

	updateCaption();

	/*
	 * Create space for the column information
	 */
	for (int i = 0; i < LVEnd; i++)
		column[i] = new columnInfo;

	/*
	 * Rather early, since we have to know which columns to show
	 */
	readConfig();

	/*
	 * Put the columns into the list view
	 */
	for (int i = 0; i < LVEnd; i++) {
		if (column[i]->show) {
		 	column[i]->index = addColumn(columnTitle[i], column[i]->width);
			if (i == Experience || i == Rating || i == Time || i == Idle)
				setColumnAlignment(column[i]->index, AlignRight);
		}
	}
	setAllColumnsShowFocus(true);

	/* 
	 * Create and initialize a context menu, the labels will be changed
	 * when the menu pops up
	 */
	menu = new QPopupMenu();
	 
	act[Info]    = new KAction(i18n("Info   "), 0, this, SLOT(infoPlayer()),   this);
	act[Talk]    = new KAction(i18n("Talk   "), 0, this, SLOT(talkPlayer()),   this);
	act[Mail]    = new KAction(i18n("Mail   "), 0, this, SLOT(mailPlayer()),   this);
	act[Invite]  = new KAction(i18n("Invite "), 0, this, SLOT(invitePlayer()), this);
	act[Look]    = new KAction(i18n("Look   "), 0, this, SLOT(lookPlayer()),   this);
	act[Watch]   = new KAction(i18n("Watch  "), 0, this, SLOT(watchPlayer()),  this);
	act[Unwatch] = new KAction(i18n("Unwatch"), 0, this, SLOT(unwatchPlayer()),this);
	act[Blind]   = new KAction(i18n("Blind  "), 0, this, SLOT(blindPlayer()),  this);
	act[Update]  = new KAction(i18n("Update "), 0, this, SLOT(updatePlayer()), this);
	act[Reload]  = new KAction(i18n("Reload "), 0, this, SLOT(reloadList()),   this);

	act[Info]->plug(menu); 
	act[Talk]->plug(menu);  
	act[Mail]->plug(menu);    
	menu->insertSeparator();
	act[Invite]->plug(menu); 
	act[Look]->plug(menu); 
	act[Watch]->plug(menu); 
	act[Unwatch]->plug(menu); 
	act[Blind]->plug(menu);
	menu->insertSeparator();
	act[Update]->plug(menu); 
	act[Reload]->plug(menu);

	/* 
	 * Right mouse button gets a context menu 
	 */
	connect(this, SIGNAL(contextMenu(KListView *, QListViewItem *, const QPoint &)), 
		this, SLOT(showContextMenu(KListView *, QListViewItem *, const QPoint &)));

	/*
	 * some eye candy :)
	 */
	setIcon(kapp->miniIcon());

	QWhatsThis::add(this, i18n("This window contains the player list. It shows "
				   "all players that are currently logged into FIBS."
				   "You can use the configuration option to select "
				   "the columns you would like to see."));
}

/*
 * Destructor saves current UI settings and deletes the menu
 */
KFibsPlayerList::~KFibsPlayerList()
{
	saveConfig();
	for (int i = 0; i < LVEnd; i++)
		delete column[i];
	delete menu;
}

/*
 * Update the caption of this window by including the current 
 * client count
 */
void KFibsPlayerList::updateCaption()
{
	setCaption(i18n("Player List - %1 - %2/%3").arg(childCount()).arg(cCount[0]).arg(cCount[1]));
}

/*
 * Catch hide events, so the engine's menu can be update. 
 */
void KFibsPlayerList::showEvent(QShowEvent *e)
{
	KListView::showEvent(e);
	emit windowVisible(true);
}

/*
 * Catch hide events, so the engine's menu can be update. 
 */
void KFibsPlayerList::hideEvent(QHideEvent *e)
{
	emit windowVisible(false);
	KListView::hideEvent(e);
}


// == settings and config ======================================================

/*
 * Restore the saved settings
 */
void KFibsPlayerList::readConfig()
{
	KConfig* config = kapp->config();
	config->setGroup(name());
	
	QPoint pos, defpos(10, 10);
	pos = config->readPointEntry("ori", &defpos);
	setGeometry(pos.x(), pos.y(), config->readNumEntry("wdt",460), config->readNumEntry("hgt",190));
	
	(config->readBoolEntry("vis", false)) ? show() : hide();

	for (int i = 0; i < LVEnd; i++) {
		column[i]->show  = config->readBoolEntry("col-" + columnKey[i], true);
		column[i]->width =  config->readNumEntry("col-w-" + columnKey[i], -1);
	}
}

/*
 * Save current settings
 */
void KFibsPlayerList::saveConfig()
{
	KConfig* config = kapp->config();
	config->setGroup(name());

	config->writeEntry("ori", pos());
	config->writeEntry("hgt", height());
	config->writeEntry("wdt", width());

	config->writeEntry("vis", isVisible());

	for (int i = 0; i < LVEnd; i++) {
		config->writeEntry("col-" + columnKey[i], column[i]->show);
		if (column[i]->show)
			config->writeEntry("col-w-" + columnKey[i], columnWidth(column[i]->index));
	}
}

/*
 * Setup dialog page of the player list - allow the user to select the 
 * columns to show
 */
void KFibsPlayerList::getSetupPages(QTabDialog *nb)
{
	/*
	 * Main Widget
	 * ===========
	 */
	QWidget *w = new QWidget(nb);
	QGridLayout *gl = new QGridLayout(w, 1, 1, 20);
	
	/*
	 * Label
	 * =====
	 */
	QGroupBox *gbl = new QGroupBox(w);
	gbl->setTitle(i18n("Column Selection:"));

	gl->addWidget(gbl, 0, 0, AlignLeft);

	QGridLayout *gll = new QGridLayout(gbl, LVEnd, 1, 20);

	/*
	 * Note that the first column (Player == 0) is always there
	 */
	for (int i = 1; i < LVEnd; i++) {
		column[i]->cb = new QCheckBox(columnTitle[i], gbl);
		column[i]->cb->setChecked(column[i]->show);
		column[i]->cb->adjustSize();
		column[i]->cb->setMinimumSize(column[i]->cb->size());
		gll->addRowSpacing(0, column[i]->cb->height());
		gll->addWidget(column[i]->cb, i, 0, AlignLeft);
	}

	/*
	 * put in the page and connect
	 * ===========================
	 */
	gl->activate();
	w->adjustSize();
	w->setMinimumSize(w->size());
	nb->addTab(w, i18n("&Player List"));

	connect(nb, SIGNAL(applyButtonPressed()), this, SLOT(setupOk()));
}

/*
 * Called when the setup dialog is positively closed
 */
void KFibsPlayerList::setupOk()
{
	int i;
	bool change = false;
	
	for (i = 1; i < LVEnd; i++) 
		change |= (column[i]->cb->isChecked() != column[i]->show);

	/*
	 * Only juggle with the columns if something changed
	 */
	if (change) {

		/*
		 * It's important to remove the columns in reverse order
		 */
		for (i = LVEnd-1; i > 0; i--)
			if (column[i]->show) 
				removeColumn(column[i]->index);
		
		/*
		 * Now add all columns that are selected
		 */
		for (i = 1; i < LVEnd; i++) {
			column[i]->show = column[i]->cb->isChecked();
			if (column[i]->show) {
				column[i]->index = addColumn(columnTitle[i], column[i]->width);
				if (i == Experience || i == Rating || i == Time || i == Idle)
					setColumnAlignment(column[i]->index, AlignRight);
			}
		}
		
		/*
		 * Reload the list
		 */
		reloadList();
	}
		
	/*
	 * store the new settings
	 */
	saveConfig();
}


// == inserting and updating the list ==========================================

/*
 * Add or change the entry of player with the corresponding string
 * from the server - rawwho
 */
void KFibsPlayerList::changePlayer(const QString &line)
{
	char entry[LVEnd][100];
	char ready[2], away[2];
	QDateTime fromEpoch;
	QString tmp;
	int j;

	// the line comes from FIBS and is 7 bit ASCII
	sscanf(line.latin1(), "%s %s %s %s %s %s %s %s %s %s %s %s", entry[Player], entry[Opponent], 
	       entry[Watches], ready, away, entry[Rating], entry[Experience], entry[Idle], entry[Time], 
	       entry[Host], entry[Client], entry[Email]);
	
	// convert time
	tmp = entry[Time];
	fromEpoch.setTime_t(tmp.toUInt());
	strcpy(entry[Time], fromEpoch.toString().latin1());
	
	// create status string
	entry[Status][0] = ready[0] == '0' ? '-' : 'R';
	entry[Status][1] = away[0]  == '0' ? '-' : 'A';
	entry[Status][2] = '-';
	entry[Status][3] = '\0';

	// clear empty strings
	for (j = 1; j < LVEnd; j++)
		if (entry[j][0] == '-' && strlen(entry[j]) == 1) 
			entry[j][0] = '\0';

	// disable drawing until the end of update
	setUpdatesEnabled(false); 

	QListViewItem *i;

	QListViewItemIterator it(this);
	for ( ; it.current(); ++it) {
		if (strcmp(it.current()->text(0).latin1(), entry[Player]) == 0) {		//lukas: FIXME
			i = it.current();
			goto found;
		}
	}
	// get here means we have to create a new entry
	i = new KFibsPlayerListLVI(this);
	
	// detect special clients :)
	tmp = entry[Client]; 
	if (tmp.contains("KFibs"))
		cCount[0]++;
	else if (tmp.contains(PROG_NAME)) 
		cCount[1]++;

	// new entry requires an update to the player count
	updateCaption();
	
	goto update;

 found:
	
	// get here only if the player has already been in the list
	entry[Status][2] = (i->text(Status).contains('B')) ? 'B' : '-';

update:

	// update entries
        for (j = 0; j < LVEnd; j++) 
		if (column[j]->show) 
			i->setText(column[j]->index, entry[j]);
}

/*
 * Remove player from the list
 */
void KFibsPlayerList::deletePlayer(const QString &player)
{
	QListViewItemIterator it(this);
	for ( ; it.current(); ++it) {
		if (strcmp(it.current()->text(0).latin1(), player.latin1()) == 0) {	//lukas: FIXME
			if (it.current()->text(Client).contains(PROG_NAME))
				--cCount[1];
			else if (it.current()->text(Client).contains("KFibs"))
				--cCount[0];
			delete it.current();
			updateCaption();
			return;
		}
	}
}

/*
 * Set/Unset the status stat in the corresponding column of the list
 */
void KFibsPlayerList::changePlayerStatus(const QString &player, int stat, bool flag)
{
	QListViewItem *i = 0;

	/*
	 * Find the correct line
	 */
	QListViewItemIterator it(this);
	for ( ; it.current(); ++it) {
		if (strcmp(it.current()->text(Player).latin1(), player.latin1()) == 0) {		//lukas: FIXME
			i = it.current();
			goto found;
		}
	}
	return;

 found:

	QString curr;
	
	/*
	 * Set the status flag
	 */
	switch (stat) {
	case KBgEngineFIBS::Blind:
		curr = i->text(Status);
		if (flag)
			curr.replace(2, 1, "B");
		else
			curr.replace(2, 1, "-");
		i->setText(Status, curr);
		break;
	case KBgEngineFIBS::Away:
		curr = i->text(Status);
		if (flag)
			curr.replace(1, 1, "A");
		else
			curr.replace(1, 1, "-");
		i->setText(Status, curr);
		break;
	}
}

/*
 * Called at the end of updates to re-enable the UI
 */
void KFibsPlayerList::stopUpdate()
{
	setUpdatesEnabled(true);
	triggerUpdate();
}

/*
 * Request information about the user selected user
 */
void KFibsPlayerList::getPlayerInfo(QListViewItem *i, const QPoint &p, int col)
{
	if (col < 0 || col > 2 || strcmp(i->text(col).latin1(), "-") == 0)	//lukas: FIXME
		col = 0*p.x(); // avoids compiler warning :)
	emit fibsCommand("whois " + i->text(col));
}

/*
 * Clear the list and reset the client counters
 */
void KFibsPlayerList::clear()
{
	cCount[0] = 0;
	cCount[1] = 0;
	QListView::clear();
}


// == popup menu slots and functions ===========================================

/*
 * Save selected player, update the menu entries and show the popup menu
 */
void KFibsPlayerList::showContextMenu(KListView *l, QListViewItem *i, const QPoint &p) 
{
	if (l && i) { // l is this - avoids compiler warning
		/*
		 * Player is always there
		 */
		opPlayer = i->text(Player);
		
		act[Info  ]->setText(i18n("Info on " ) + opPlayer);
		act[Talk  ]->setText(i18n("Talk to " ) + opPlayer);
		act[Mail  ]->setText(i18n("Email to ") + opPlayer);
		act[Invite]->setText(i18n("Invite "  ) + opPlayer);
		act[Look  ]->setText(i18n("Look at " ) + opPlayer);
		act[Watch ]->setText(i18n("Watch "   ) + opPlayer);
		act[Update]->setText(i18n("Update "  ) + opPlayer);
		
		if (column[Status]->show)
			act[Blind]->setText((i->text(column[Status]->index).contains("B")) ? 
					    i18n("Unblind ") : i18n("Blind ") + opPlayer);

		if (column[Email]->show) {
			opEmail  = i->text(column[Email]->index);
			act[Mail]->setEnabled(!opEmail.isEmpty());
		}

		menu->popup(p);
	}
}

/*
 * Request information on opPlayer
 */
void KFibsPlayerList::infoPlayer()
{
	emit fibsCommand("whois " + opPlayer);
}

/*
 * Initiate an invitation of opPlayer
 */
void KFibsPlayerList::invitePlayer()
{
	emit fibsInvite(opPlayer);
}

/*
 * Look at opPlayer
 */
void KFibsPlayerList::lookPlayer()
{
	emit fibsCommand("look " + opPlayer);
}

/*
 * Send an email to player opPlayer at address opEmail
 */
void KFibsPlayerList::mailPlayer()
{
	kapp->invokeMailer(opEmail, QString::null);
}

/*
 * Watch opPlayer and get an updated board
 */
void KFibsPlayerList::watchPlayer()
{
	emit fibsCommand("watch " + opPlayer);
	emit fibsCommand("board");
}

/*
 * Unwatch player
 */
void KFibsPlayerList::unwatchPlayer()
{
	emit fibsCommand("unwatch");
}

/*
 * Blind/Unblind player opPlayer
 */
void KFibsPlayerList::blindPlayer()
{
	emit fibsCommand("blind " + opPlayer);
}

/*
 * Start talking to player opPlayer
 */
void KFibsPlayerList::talkPlayer()
{
	emit fibsTalk(opPlayer);
}

/*
 * Request a new entry for opPlayer
 */
void KFibsPlayerList::updatePlayer()
{
	emit fibsCommand("rawwho " + opPlayer);
}

/*
 * Reload the entire list
 */
void KFibsPlayerList::reloadList()
{
	clear();
	emit fibsCommand("rawwho");
}


// == list view item class =====================================================

/*
 * Constructor of the QListViewItem extension
 */
KFibsPlayerListLVI::KFibsPlayerListLVI(QListView *parent)
	: QListViewItem(parent)
{
	// empty
}

/*
 * Destructor of the QListViewItem extension
 */
KFibsPlayerListLVI::~KFibsPlayerListLVI()
{
	// empty
}


/*
 * Overloaded key() function for sorting
 */
QString KFibsPlayerListLVI::key(int col, bool asc) const
{
	QString s = text(col);

	switch (col) {
	case KFibsPlayerList::Player:
	case KFibsPlayerList::Opponent:
	case KFibsPlayerList::Watches:
		s = s.lower();
		break;
	case KFibsPlayerList::Experience:
		s.sprintf("%08d", s.toInt());
		break;
	case KFibsPlayerList::Rating:
		s.sprintf("%08d", (int)(1000*s.toDouble()));
		break;
	case KFibsPlayerList::Status:
	case KFibsPlayerList::Idle:
	case KFibsPlayerList::Time:
	case KFibsPlayerList::Host:
	case KFibsPlayerList::Client:
	case KFibsPlayerList::Email:
		break;
	}
	if (asc) // avoid compiler warning
		return s;
	return s;
}

// EOF
