/*
 * Copyright (C), 2000-2003 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */


%{
  
/*
 * DESCRIPTION
 *   Lexical grammar for tokenizing the control file. 
 *
 *  AUTHOR
 *   Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *   Olivier Beyssac, <ob@r14.freenix.org>
 *   Christian Hopp, <chopp@iei.tu-clausthal.de>
 *   Martin Pala, <martin.pala@iol.cz>
 *
 *  CVS INFO
 *     $Id: l.l,v 1.66 2003/10/25 19:17:10 hauk Exp $
 */

#include <config.h>
  
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#include "monitor.h"
#include "y.tab.h"

  int lineno= 1;
  
  /* Prototypes */
  void yyerror(const char*);
  static char *handle_quoted_string(char *);
  void steplinenobycr(char *);
      
%}

ws          [ \t]+
wws         [ \t=;,()]+
number      [0-9]+
real        [0-9]+([.][0-9]+)?
str         [^\000-\041@:{}"';(),]+
hostname    {str}(\.{str})*
greater     ("greater"|"gt"|">")
less        ("less"|"lt"|"<")
equal       ("equal"|"eq"|"==")
notequal    ("notequal"|"ne"|"!=")
loadavg1    load(avg)[ ]*(\([ ]*1[ ]*(m|min)?[ ]*\))?
loadavg5    load(avg)[ ]*\([ ]*5[ ]*(m|min)?[ ]*\)
loadavg15   load(avg)[ ]*\([ ]*15[ ]*(m|min)?[ ]*\)
startarg    start{wws}?(program)?{wws}?["]
stoparg     stop{wws}?(program)?{wws}?["]
execarg     exec(ute)?{wws}?["]
percent     ([ ]*"percent"|[ ]*"%")
byte        ("byte"|"b")
kilobyte    ("kilobyte"|"kb")
megabyte    ("megabyte"|"mb")
gigabyte    ("gigabyte"|"gb")

%x ARGUMENT_COND DEPEND_COND SERVICE_COND

%%

{wws}             { /* Wide white space */ }
(#.*)?\\?\n       { lineno++; } 

is                {/* EMPTY */}
as                {/* EMPTY */}
are               {/* EMPTY */}
for               {/* EMPTY */}
on(ly)?           {/* EMPTY */}
with(in)?         {/* EMPTY */}
program(s)?       {/* EMPTY */}
and               {/* EMPTY */}
has               {/* EMPTY */}
using             {/* EMPTY */}
use               {/* EMPTY */}
the               {/* EMPTY */}
sum               {/* EMPTY */}
than              {/* EMPTY */}
usage             {/* EMPTY */}
was               {/* EMPTY */}

{startarg}        { BEGIN(ARGUMENT_COND); return START; }
{stoparg}         { BEGIN(ARGUMENT_COND); return STOP; }
{execarg}         { BEGIN(ARGUMENT_COND); return EXEC; }

if                { return IF; }
then              { return THEN; }
else              { return ELSE; }
failed            { return FAILED; }
ssl               { return HTTPDSSL; }
enable            { return ENABLE; }
disable           { return DISABLE; }
set               { return SET; }
daemon            { return DAEMON; }
logfile           { return LOGFILE; }
syslog            { return SYSLOG; }
facility          { return FACILITY; }
httpd             { return HTTPD; }
address           { return ADDRESS; }
clientpemfile     { return CLIENTPEMFILE; }
allowselfcertification  { return ALLOWSELFCERTIFICATION; }
certmd5           { return CERTMD5; }
pemfile           { return PEMFILE; }
init              { return INIT; }
allow             { return ALLOW; }
read[-]?only      { return READONLY; }
pidfile           { return PIDFILE; }
statefile         { return STATEFILE; }
path              { return PATHTOK; }
start             { return START; }
stop              { return STOP; }
port(number)?     { return PORT; }
unix(socket)?     { return UNIXSOCKET; }
type              { return TYPE; }
proto(col)?       { return PROTOCOL; }
tcp               { return TCP; }
tcpssl            { return TCPSSL; }
udp               { return UDP; }
alert             { return ALERT; }
mail-format       { return MAILFORMAT; }
resource          { return RESOURCE; }
restart(s)?       { return RESTART; }
cycle(s)?         { return CYCLE;}
timeout           { return TIMEOUT; }
checksum          { return CHECKSUM; }
mailserver        { return MAILSERVER; }
every             { return EVERY; }
host              { return HOST; }
default           { return DEFAULT; }
http              { return HTTP; }
ftp               { return FTP; }
smtp              { return SMTP; }
pop               { return POP; }
imap              { return IMAP; }
nntp              { return NNTP; }
ssh               { return SSH; }
dwp               { return DWP; }
ldap2             { return LDAP2; }
ldap3             { return LDAP3; }
rdate             { return RDATE; }
rsync             { return RSYNC; }
mode              { return MODE; }
active            { return ACTIVE; }
passive           { return PASSIVE; }
manual            { return MANUAL; }
group             { return GROUP; }
uid               { return UID; }
gid               { return GID; }
env(ironment)?    { return ENVIRONMENT; }
request           { return REQUEST; }
cpuusage          { return CPUUSAGE; }
memusage          { return MEMUSAGE; }
memkbyte          { return MEMKBYTE; }
mem(ory)?         { return MEMORY; }
totalmem(ory)?    { return TOTALMEMORY; }
cpu               { return CPU; }
child(ren)        { return CHILDREN; }
totalmemkbyte     { return TOTALMEMKBYTE; }
totalmemusage     { return TOTALMEMUSAGE; }
timestamp         { return TIMESTAMP; }
changed           { return CHANGED; }
second(s)?        { return SECOND; }
minute(s)?        { return MINUTE; }
hour(s)?          { return HOUR; }
day(s)?           { return DAY; }
sslv2             { return SSLV2; }
sslv3             { return SSLV3; }
tlsv1             { return TLSV1; }
sslauto           { return SSLAUTO; }
inode(s)?         { return INODE; }
space             { return SPACE; }
perm(ission)?     { return PERMISSION; }
exec(ute)?        { return EXEC; }
size              { return SIZE; }
connection        { return CONNECTION; }
unmonitor         { return UNMONITOR; }
icmp              { return ICMP; }
echo              { return ICMPECHO; }
send              { return SEND; }
expect            { return EXPECT; }
cleartext         { return CLEARTEXT; }
md5               { return MD5TOKEN; }
crypt             { return CRYPT; }


{byte}            { return BYTE; }
{kilobyte}        { return KILOBYTE; }
{megabyte}        { return MEGABYTE; }
{gigabyte}        { return GIGABYTE; }

{loadavg1}        { return LOADAVG1; }
{loadavg5}        { return LOADAVG5; }
{loadavg15}       { return LOADAVG15; }

{greater}         { return GREATER; }
{less}            { return LESS; }
{equal}           { return EQUAL; }
{notequal}        { return NOTEQUAL; }

depend(s)?[ \t]+(on[ \t]*)?  {
                    BEGIN(DEPEND_COND);
                    return DEPENDS;
                  } 

check[ \t]+(process[ \t])? {
                    BEGIN(SERVICE_COND);
                    return CHECKPROC;
                  }

check[ \t]+device {
                    BEGIN(SERVICE_COND);
                    return CHECKDEV;
                  }

check[ \t]+file   {
                    BEGIN(SERVICE_COND);
                    return CHECKFILE;
                  }

check[ \t]+directory {
                    BEGIN(SERVICE_COND);
                    return CHECKDIR;
                  }

check[ \t]+host   {
                    BEGIN(SERVICE_COND);
                    return CHECKHOST;
                  }

{number}          {
                    yylval.number= atoi(yytext); return NUMBER;
                  }

{real}            {
                    yylval.real= atof(yytext); return REAL;
                  }

{real}{percent}   {
                    if(sscanf(yytext, "%f%*s", &yylval.real) != 1) {
                      log("%s: Internal lexer error, parsing '%s'\n",
                          prog, yytext);
                      exit(1);
                    }
                    return PERCENT;
                  }

[a-zA-Z0-9]{str}  {
                    yylval.string= xstrdup(yytext);
                    return STRING;
                  }

\"[/][^\"\n]*\"   {
                    yylval.string= handle_quoted_string(yytext);
                    return PATH;
                  }

\'[/][^\'\n]*\'   {
                    yylval.string= handle_quoted_string(yytext);
                    return PATH;
                  }

\"[^\"]*\"        {
                    steplinenobycr(yytext);
                    yylval.string= handle_quoted_string(yytext);
                    return STRING;
                  }

\'[^\']*\'        {
                    steplinenobycr(yytext);
                    yylval.string= handle_quoted_string(yytext);
                    return STRING;
                  }

{str}[@]{str}     {
                    yylval.string= xstrdup(yytext);
                    return MAILADDR;
                  }

[/]{str}          {
                     yylval.string= xstrdup(yytext);
                     return PATH;
                  }

"/"               {
                     yylval.string= xstrdup(yytext);
                     return PATH;
                  }

"from:"[ \t]*{str}[@]{str} {
                      char *p= yytext+strlen("from:");
                      yylval.string = trim(xstrdup(p));
                      return MAILFROM;
                  }
                     
"subject:"[^}\n]* {
                      char *p= yytext+strlen("subject:");
                      yylval.string = trim(xstrdup(p));
                      return MAILSUBJECT;
                  }

"message:"[^}]*   {
                      char *p= yytext+strlen("message:");
                      steplinenobycr(yytext);
                      yylval.string = trim(xstrdup(p));
                      return MAILBODY;
                  }

{hostname}        {
                      yylval.string = xstrdup(yytext);
                      return STRING;
                  }

[\"\']            {

                      yyerror("Unbalanced quotes");

                  }


<SERVICE_COND>{

  {ws}            ;

  [\n]            {
                    lineno++;
                  }

  {str}           {
                    yylval.string= xstrdup(yytext);
                    BEGIN(INITIAL);
                    return SERVICENAME;
                  }

  \"{str}\"       {
                    yylval.string= handle_quoted_string(yytext);
                    BEGIN(INITIAL);
                    return SERVICENAME;
                  }

  \'{str}\'       {
                    yylval.string= handle_quoted_string(yytext);
                    BEGIN(INITIAL);
                    return SERVICENAME;
                  }

  [\"\']          {
                      yyerror("Unbalanced quotes");
                  }

}

<DEPEND_COND>{

  {wws}           ;

  {wws}?[\n]{wws}? {
                    lineno++;
                  }

  {str}           {
                    yylval.string= xstrdup(yytext);
                    return SERVICENAME;
                  }

  [ \n\t]+[^,]    {
                    steplinenobycr(yytext);
                    unput(yytext[strlen(yytext)-1]);
                    BEGIN(INITIAL);
                  }

}

<ARGUMENT_COND>{

  {ws}            ;

  [\n]            {
                    lineno++;
                  }

  \"              {
                      BEGIN(INITIAL);
                  }

  \'[^\']*\'      {
                      steplinenobycr(yytext);
                      yylval.string= handle_quoted_string(yytext);
                      return STRING;
                  }

  \'              {
                      yyerror("Unbalanced quotes");
                  }

  [^ \t\n\"]+     {
                      yylval.string= xstrdup(yytext);
                      return STRING;
                  }

}

<INITIAL,ARGUMENT_COND,SERVICE_COND,DEPEND_COND>. {
                      return yytext[0];
                  }  


%%


/*
 * Do lineno++ for every occurrence of '\n' in a string.  This is
 * necessary whenever a yytext has an unknown number of CRs.
 */

void steplinenobycr(char *string) {

  char *pos= string;

  while(*pos)
    if('\n'==*pos++) {
      lineno++;
    }

}


static char *handle_quoted_string(char *string) {

  char *buf= xstrdup(string);

  trim_quotes(buf);
  handle_string_escapes(buf);

  return buf;

}
