/****************************************************************************
** $Id: qt/src/tools/qjpunicode.cpp   2.2.0   edited 2000-08-28 $
**
** Implementation of QJpUnicodeConv class
**
** Created : 990225
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of the tools module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

/*! \class QJpUnicodeConv qjpunicode.h

  \brief The QJpUnicodeConv class provides implementation support for
  QJisCodec, QSjisCodec, and QEucJpCodec.

  You should not need to use this class directly.  If you do, keep in
  mind that the function names will change in Qt 3.0: Every function
  name will get a lower-case first letter: Jisx0212ToUnicode becomes
  jisx0212ToUnicode, and so on.  We apologize for the upheaval.

  The environment variable \c UNICODEMAP_JP can be used to fine-tune how
  QJpUnicodeConv, QEucJpCodec, QJisCodec and QSjisCodec do their work.
  The mapping names are as for the Japanese XML working group's <a
  href="http://www.y-adagio.com/public/standards/tr_xml_jpf/toc.htm">
  XML Japanese Profile</a>, as it names and explains all the widely
  used mappings. Here are brief descriptions, written by Serika
  Kurusugawa: <ul>

  <li> "unicode-0.9" or "unicode-0201" for Unicode style. This assume
  JISX0201 for 0x00-0x7f. (0.9 is a table version of jisx02xx mapping
  used for Uniocde spec version 1.1)

  <li> "unicode-ascii" This assume US-ASCII for 0x00-0x7f, and some
  chars (JISX0208 0x2140 and JISX0212 0x2237) are different from
  Unicode 1.1 to avoid conflict.

  <li> "open-19970715-0201" ("open-0201" for convenience) or
  "jisx0221-1995" for JISX0221-JISX0201 style. JIS X 0221 is JIS
  version of Unicode, but a few chars (0x5c, 0x7e, 0x2140, 0x216f,
  0x2131) are different from Unicode 1.1. This is used when 0x5c is
  treated as YEN SIGN.

  <li> "open-19970715-ascii" ("open-ascii" for convenience) for
  JISX0221-ASCII style. This is used when 0x5c is treated as REVERSE
  SOLIDUS.

  <li> "open-19970715-ms" ("open-ms" for convenience) or "cp932" for
  Microsoft Windows style. Windows Code Page 932. Some chars (0x2140,
  0x2141, 0x2142, 0x215d, 0x2171, 0x2172) are different from Unicode
  1.1.

  <li> "jdk1.1.7" for Sun's JDK style Same as Unicode 1.1, except that
  JIS 0x2140 is mapped to UFF3C. Either ASCII or JISX0201 can be used
  for 0x00-0x7f.

  </ul> In addition, QJpUnicodeConv supports some extensions:
  "nec-vdc", "ibm-vdc" or "udc".

  For example, if you want to use Unicode style conversion, but with
  NEC's extension, set \c UNICODEMAP_JP to
  <nobr><code>unicode-0.9, nec-vdc</code>.</nobr> (You will probably
  need to quote that in the shell command.)

  Most of the code here was originally written by Serika Kurusugawa
  a.k.a. Junji Takagi, and is includes in Qt with the author's
  permission, and the grateful thanks of the Trolltech team. Here is
  the copyright statement for that code:

  \mustquote

  Copyright (c) 1999 Serika Kurusugawa, All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met: <ol>
  <li> Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  <li> Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  </ol>

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.

*/

#include "qcstring.h"

#ifndef QT_NO_CODECS

#include <stdlib.h>
#include <ctype.h>

#include "qjpunicode.h"

#define	USE_JISX0212

#define	STRICT

#define	IsLatin(c)	(/*((c) >= 0x00) &&*/ ((c) <= 0x7f))
#define	IsKana(c)	(((c) >= 0xa1) && ((c) <= 0xdf))
#define	IsJisChar(c)	(((c) >= 0x21) && ((c) <= 0x7e))
#define	IsSjisChar1(c)	((((c) >= 0x81) && ((c) <= 0x9f)) ||	\
			 (((c) >= 0xe0) && ((c) <= 0xfc)))
#define	IsSjisUDC1(c)	(((c) >= 0xf0) && ((c) <= 0xfc))
#define	IsSjisChar2(c)	(((c) >= 0x40) && ((c) != 0x7f) && ((c) <= 0xfc))

static uint Jisx0208ToSjis(uint h, uint l);
static
inline uint Jisx0208ToSjis(uint jis)
{
    return Jisx0208ToSjis((jis & 0xff00) >> 8, (jis & 0x00ff));
}

static uint SjisToJisx0208(uint h, uint l);
static
inline uint SjisToJisx0208(uint sjis)
{
    return SjisToJisx0208((sjis & 0xff00) >> 8, (sjis & 0x00ff));
}

static uint Jisx0201ToUnicode11(uint h, uint l);
static uint Jisx0208ToUnicode11(uint h, uint l);
static uint Jisx0212ToUnicode11(uint h, uint l);

static uint Unicode11ToJisx0201(uint h, uint l);
static uint Unicode11ToJisx0208(uint h, uint l);
static uint Unicode11ToJisx0212(uint h, uint l);

/*
 * Unicode 1.1 conversion.
 */

/*!
  \fn QJpUnicodeConv::QJpUnicodeConv (int r)
  \internal
*/


/*! \internal */
uint QJpUnicodeConv::AsciiToUnicode(uint h, uint l) const
{
    if ((h == 0) && (l < 0x80)) {
	return l;
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::Jisx0201ToUnicode(uint h, uint l) const
{
    if (h == 0) {
	if (IsLatin(l)) {
	    return Jisx0201LatinToUnicode(h, l);
	} else if (IsKana(l)) {
	    return Jisx0201KanaToUnicode(h, l);
	}
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::Jisx0201LatinToUnicode(uint h, uint l) const
{
    if ((h == 0) && IsLatin(l)) {
	return Jisx0201ToUnicode11(h, l);
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::Jisx0201KanaToUnicode(uint h, uint l) const
{
    if ((h == 0) && IsKana(l)) {
	return Jisx0201ToUnicode11(h, l);
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::Jisx0208ToUnicode(uint h, uint l) const
{
    if (rule & JU_UDC){
	if ((0x75 <= h) && (h <= 0x7e) && IsJisChar(l)/*0x21 - 0x7e*/) {
	    // User Defined Char (mapped to Private Use Area)
	    return 0xe000 + (h - 0x75) * 0x5e + (l - 0x21);
	}
    }
    if ((rule & JU_NEC_VDC) == 0) {
	if ((h == 0x2d) && (IsJisChar(l)/*0x21 - 0x7c*/)) {
	    // NEC Vendor Defined Char
	    return 0x0000;
	}
    }
    return Jisx0208ToUnicode11(h, l);
}

/*! \internal */
uint QJpUnicodeConv::Jisx0212ToUnicode(uint h, uint l) const
{
    if (rule & JU_UDC){
	if ((0x75 <= h) && (h <= 0x7e) && IsJisChar(l)/*0x21 - 0x7e*/) {
	    // User Defined Char (mapped to Private Use Area)
	    return 0xe3ac + (h - 0x75) * 0x5e + (l - 0x21);
	}
    }
    if ((rule & JU_IBM_VDC) == 0){
	if (((h == 0x73) && (0x73 <= l) && (l <= 0x7e)) ||
	    ((h == 0x74) && (IsJisChar(l)/*0x21 - 0x7e*/))) {
	    // IBM Vendor Defined Char
	    return 0x0000;
	}
    }
    return Jisx0212ToUnicode11(h, l);
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToAscii(uint h, uint l) const
{
    if ((h == 0) && (l < 0x80)) {
	return l;
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToJisx0201(uint h, uint l) const
{
    uint jis;
    if ((jis = UnicodeToJisx0201Latin(h, l)) != 0) {
	return jis;
    } else if ((jis = UnicodeToJisx0201Kana(h, l)) != 0) {
	return jis;
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToJisx0201Latin(uint h, uint l) const
{
    uint jis = Unicode11ToJisx0201(h, l);
    if (IsLatin(jis)) {
	return jis;
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToJisx0201Kana(uint h, uint l) const
{
    uint jis = Unicode11ToJisx0201(h, l);
    if (IsKana(jis)) {
	return jis;
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToJisx0208(uint h, uint l) const
{
    if (rule & JU_UDC){
	uint unicode = (h << 8) | l;
	if ((0xe000 <= unicode) && (unicode <= 0xe3ab)) {
	    // User Defined Char (mapped to Private Use Area)
	    unicode -= 0xe000;
	    return ((0x75 + unicode / 0x5e ) << 8) | (0x21 + unicode % 0x5e);
	}
    }
    uint jis = Unicode11ToJisx0208(h, l);
    if ((rule & JU_NEC_VDC) == 0) {
	if ((0x2d21 <= jis) && (jis <= 0x2d7c)) {
	    // NEC Vendor Defined Char
	    return 0x0000;
	}
    }
    return jis;
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToJisx0212(uint h, uint l) const
{
    if (rule & JU_UDC){
	uint unicode = (h << 8) | l;
	if ((0xe3ac <= unicode) && (unicode <= 0xe757)) {
	    // User Defined Char (mapped to Private Use Area)
	    unicode -= 0xe3ac;
	    return ((0x75 + unicode / 0x5e ) << 8) | (0x21 + unicode % 0x5e);
	}
    }
    uint jis = Unicode11ToJisx0212(h, l);
    if ((rule & JU_IBM_VDC) == 0){
	if (((0x7373 <= jis) && (jis <= 0x737e)) ||
	    ((0x7421 <= jis) && (jis <= 0x747e))) {
	    // IBM Vendor Defined Char
	    return 0x0000;
	}
    }
    return jis;
}

/*! \internal */
uint QJpUnicodeConv::SjisToUnicode(uint h, uint l) const
{
    if (h == 0) {
	return Jisx0201ToUnicode(h, l);
    } else if (IsSjisChar1(h) && IsSjisChar2(l)) {
	return Jisx0208ToUnicode(SjisToJisx0208(h, l));
    }
    return 0x0000;
}

/*! \internal */
uint QJpUnicodeConv::UnicodeToSjis(uint h, uint l) const
{
    uint jis;
    if ((jis = UnicodeToJisx0201(h, l)) != 0x0000) {
	return jis;
    } else if ((jis = UnicodeToJisx0208(h, l)) != 0x0000) {
	return Jisx0208ToSjis(jis);
    } else if ((jis = UnicodeToJisx0212(h, l)) != 0x0000) {
	return 0x0000;
    }
    return 0x0000;
}

/*
 * Unicode 1.1 with JISX0201 conversion.
 */

class QJpUnicodeConv_Unicode_JISX0201 : public QJpUnicodeConv {
public:
    QJpUnicodeConv_Unicode_JISX0201(int r) : QJpUnicodeConv(r) {}

//    uint AsciiToUnicode(uint h, uint l) const;
//    uint Jisx0201ToUnicode(uint h, uint l) const;
//    uint Jisx0201LatinToUnicode(uint h, uint l) const;
//    uint Jisx0201KanaToUnicode(uint h, uint l) const;
//    uint Jisx0208ToUnicode(uint h, uint l) const;
//    uint Jisx0212ToUnicode(uint h, uint l) const;

//    uint UnicodeToAscii(uint h, uint l) const;
//    uint UnicodeToJisx0201(uint h, uint l) const;
//    uint UnicodeToJisx0201Latin(uint h, uint l) const;
//    uint UnicodeToJisx0201Kana(uint h, uint l) const;
//    uint UnicodeToJisx0208(uint h, uint l) const;
//    uint UnicodeToJisx0212(uint h, uint l) const;
};

/*
 * Unicode 1.1 with ASCII conversion.
 */

class QJpUnicodeConv_Unicode_ASCII : public QJpUnicodeConv {
public:
    QJpUnicodeConv_Unicode_ASCII(int r) : QJpUnicodeConv(r) {}

//    uint AsciiToUnicode(uint h, uint l) const;
//    uint Jisx0201ToUnicode(uint h, uint l) const;
//    uint Jisx0201LatinToUnicode(uint h, uint l) const;
//    uint Jisx0201KanaToUnicode(uint h, uint l) const;
    uint Jisx0208ToUnicode(uint h, uint l) const;
    uint Jisx0212ToUnicode(uint h, uint l) const;

//    uint UnicodeToAscii(uint h, uint l) const;
//    uint UnicodeToJisx0201(uint h, uint l) const;
//    uint UnicodeToJisx0201Latin(uint h, uint l) const;
//    uint UnicodeToJisx0201Kana(uint h, uint l) const;
    uint UnicodeToJisx0208(uint h, uint l) const;
    uint UnicodeToJisx0212(uint h, uint l) const;
};

uint QJpUnicodeConv_Unicode_ASCII::Jisx0208ToUnicode(uint h, uint l) const
{
    if ((h == 0x21) && (l == 0x40)) {
	return 0xff3c;
    }
    return QJpUnicodeConv::Jisx0208ToUnicode(h, l);
}

uint QJpUnicodeConv_Unicode_ASCII::Jisx0212ToUnicode(uint h, uint l) const
{
    if ((h == 0x22) && (l == 0x37)) {
	return 0xff5e;
    }
    return QJpUnicodeConv::Jisx0212ToUnicode(h, l);
}

uint QJpUnicodeConv_Unicode_ASCII::UnicodeToJisx0208(uint h, uint l) const
{
    if ((h == 0x00) && (l == 0x5c)) {
	return 0x0000;
    }
    if ((h == 0xff) && (l == 0x3c)) {
	return 0x2140;
    }
    return QJpUnicodeConv::UnicodeToJisx0208(h, l);
}

/*! \internal */
uint QJpUnicodeConv_Unicode_ASCII::UnicodeToJisx0212(uint h, uint l) const
{
    if ((h == 0x00) && (l == 0x7e)) {
	return 0x0000;
    }
    if ((h == 0xff) && (l == 0x5e)) {
	return 0x2237;
    }
    return QJpUnicodeConv::UnicodeToJisx0208(h, l);
}

/*
 * JISX0221 with JISX0201 conversion.
 */

class QJpUnicodeConv_JISX0221_JISX0201 : public QJpUnicodeConv {
public:
    QJpUnicodeConv_JISX0221_JISX0201(int r) : QJpUnicodeConv(r) {}

    uint AsciiToUnicode(uint h, uint l) const;
//    uint Jisx0201ToUnicode(uint h, uint l) const;
//    uint Jisx0201LatinToUnicode(uint h, uint l) const;
//    uint Jisx0201KanaToUnicode(uint h, uint l) const;
    uint Jisx0208ToUnicode(uint h, uint l) const;
//    uint Jisx0212ToUnicode(uint h, uint l) const;

    uint UnicodeToAscii(uint h, uint l) const;
//    uint UnicodeToJisx0201(uint h, uint l) const;
//    uint UnicodeToJisx0201Latin(uint h, uint l) const;
//    uint UnicodeToJisx0201Kana(uint h, uint l) const;
    uint UnicodeToJisx0208(uint h, uint l) const;
//    uint UnicodeToJisx0212(uint h, uint l) const;
};

uint QJpUnicodeConv_JISX0221_JISX0201::AsciiToUnicode(uint h, uint l) const
{
    return Jisx0201LatinToUnicode(h, l);
}

uint QJpUnicodeConv_JISX0221_JISX0201::Jisx0208ToUnicode(uint h, uint l) const
{
    if ((h == 0x21) && (l == 0x3d)) {
	return 0x2014;
    }
    return QJpUnicodeConv::Jisx0208ToUnicode(h, l);
}

uint QJpUnicodeConv_JISX0221_JISX0201::UnicodeToAscii(uint h, uint l) const
{
    return UnicodeToJisx0201Latin(h, l);
}

uint QJpUnicodeConv_JISX0221_JISX0201::UnicodeToJisx0208(uint h, uint l) const
{
#ifdef STRICT
    if ((h == 0x20) && (l == 0x15)) {
	return 0x0000;
    }
#endif
    if ((h == 0x20) && (l == 0x14)) {
	return 0x213d;
    }
    return QJpUnicodeConv::UnicodeToJisx0208(h, l);
}

/*
 * JISX0221 with ASCII conversion.
 */

class QJpUnicodeConv_JISX0221_ASCII : public QJpUnicodeConv {
public:
    QJpUnicodeConv_JISX0221_ASCII(int r) : QJpUnicodeConv(r) {}

//    uint AsciiToUnicode(uint h, uint l) const;
//    uint Jisx0201ToUnicode(uint h, uint l) const;
    uint Jisx0201LatinToUnicode(uint h, uint l) const;
//    uint Jisx0201KanaToUnicode(uint h, uint l) const;
    uint Jisx0208ToUnicode(uint h, uint l) const;
    uint Jisx0212ToUnicode(uint h, uint l) const;

//    uint UnicodeToAscii(uint h, uint l) const;
//    uint UnicodeToJisx0201(uint h, uint l) const;
    uint UnicodeToJisx0201Latin(uint h, uint l) const;
//    uint UnicodeToJisx0201Kana(uint h, uint l) const;
    uint UnicodeToJisx0208(uint h, uint l) const;
    uint UnicodeToJisx0212(uint h, uint l) const;
};

uint QJpUnicodeConv_JISX0221_ASCII::Jisx0201LatinToUnicode(uint h, uint l) const
{
    return AsciiToUnicode(h, l);
}

uint QJpUnicodeConv_JISX0221_ASCII::Jisx0208ToUnicode(uint h, uint l) const
{
    if (h == 0x21) {
	if (l == 0x31) {
	    return 0x203e;
	} else if (l == 0x3d) {
	    return 0x02014;
	} else if (l == 0x40) {
	    return 0xff3c;
	} else if (l == 0x6f) {
	    return 0x00a5;
	}
    }
    return QJpUnicodeConv::Jisx0208ToUnicode(h, l);
}

uint QJpUnicodeConv_JISX0221_ASCII::Jisx0212ToUnicode(uint h, uint l) const
{
    if ((h == 0x22) && (l == 0x37)) {
	return 0xff5e;
    }
    return QJpUnicodeConv::Jisx0212ToUnicode(h, l);
}

uint QJpUnicodeConv_JISX0221_ASCII::UnicodeToJisx0201Latin(uint h, uint l) const
{
    return QJpUnicodeConv::UnicodeToAscii(h, l);
}

uint QJpUnicodeConv_JISX0221_ASCII::UnicodeToJisx0208(uint h, uint l) const
{
#ifdef STRICT
    if (((h == 0x00) && (l == 0x5c)) ||
	((h == 0x20) && (l == 0x15)) ||
	((h == 0xff) && ((l == 0xe3) || (l == 0xe5)))) {
	return 0x0000;
    }
#else
    if ((h == 0x00) && (l == 0x5c)) {
	return 0x0000;
    }
#endif
    if ((h == 0x00) && (l == 0xa5)) {
	return 0x216f;
    } else if (h == 0x20) {
	if (l == 0x14) {
	    return 0x213d;
	} else if (l == 0x3e) {
	    return 0x2131;
	}
    } else if ((h == 0xff) && (l == 0x3c)) {
	return 0x2140;
    }
    return QJpUnicodeConv::UnicodeToJisx0208(h, l);
}

/*! \internal */
uint QJpUnicodeConv_JISX0221_ASCII::UnicodeToJisx0212(uint h, uint l) const
{
    if ((h == 0x00) && (l == 0x7e)) {
	return 0x0000;
    }
    if ((h == 0xff) && (l == 0x5e)) {
	return 0x2237;
    }
    return QJpUnicodeConv::UnicodeToJisx0212(h, l);
}

/*
 * Sun Microsystems conversion.
 */

class QJpUnicodeConv_Sun : public QJpUnicodeConv {
public:
    QJpUnicodeConv_Sun(int r) : QJpUnicodeConv(r) {}

//    uint AsciiToUnicode(uint h, uint l) const;
//    uint Jisx0201ToUnicode(uint h, uint l) const;
    uint Jisx0201LatinToUnicode(uint h, uint l) const;
//    uint Jisx0201KanaToUnicode(uint h, uint l) const;
    uint Jisx0208ToUnicode(uint h, uint l) const;
    uint Jisx0212ToUnicode(uint h, uint l) const;

    uint UnicodeToAscii(uint h, uint l) const;
//    uint UnicodeToJisx0201(uint h, uint l) const;
    uint UnicodeToJisx0201Latin(uint h, uint l) const;
//    uint UnicodeToJisx0201Kana(uint h, uint l) const;
    uint UnicodeToJisx0208(uint h, uint l) const;
    uint UnicodeToJisx0212(uint h, uint l) const;
};

uint QJpUnicodeConv_Sun::Jisx0201LatinToUnicode(uint h, uint l) const
{
    return AsciiToUnicode(h, l);
}

uint QJpUnicodeConv_Sun::Jisx0208ToUnicode(uint h, uint l) const
{
    if ((h == 0x21) && (l == 0x40)) {
	return 0xff3c;
    }
    return QJpUnicodeConv::Jisx0208ToUnicode(h, l);
}

uint QJpUnicodeConv_Sun::Jisx0212ToUnicode(uint h, uint l) const
{
#if 1
    // Added by Serika Kususugawa to avoid conflict on U+007c.
    if ((h == 0x22) && (l == 0x37)) {
	return 0xff5e;
    }
#endif
    return QJpUnicodeConv::Jisx0212ToUnicode(h, l);
}

uint QJpUnicodeConv_Sun::UnicodeToAscii(uint h, uint l) const
{
    if ((h == 0x00) && (l == 0xa5)) {
	return 0x005c;
    } else if ((h == 0x20) && (l == 0x3e)) {
	return 0x007e;
    }
    return QJpUnicodeConv::UnicodeToAscii(h, l);
}

uint QJpUnicodeConv_Sun::UnicodeToJisx0201Latin(uint h, uint l) const
{
    return QJpUnicodeConv::UnicodeToAscii(h, l);
}

uint QJpUnicodeConv_Sun::UnicodeToJisx0208(uint h, uint l) const
{
    if ((h == 0x00) && (l == 0xa5)) {
	return 0x0000;
    } else if ((h == 0x20) && (l == 0x3e)) {
	return 0x0000;
    } else if ((h == 0xff) && (l == 0x3c)) {
	return 0x2140;
    }
    return QJpUnicodeConv::UnicodeToJisx0208(h, l);
}

/*! \internal */
uint QJpUnicodeConv_Sun::UnicodeToJisx0212(uint h, uint l) const
{
#if 1
    // Added by Serika Kususugawa to avoid conflict on U+007c.
    if ((h == 0x00) && (l == 0x7e)) {
	return 0x0000;
    }
    if ((h == 0xff) && (l == 0x5e)) {
	return 0x2237;
    }
#endif
    return QJpUnicodeConv::UnicodeToJisx0212(h, l);
}

/*
 * Microsoft conversion.
 */

class QJpUnicodeConv_Microsoft : public QJpUnicodeConv {
public:
    QJpUnicodeConv_Microsoft(int r) : QJpUnicodeConv(r) {}

//    uint AsciiToUnicode(uint h, uint l) const;
//    uint Jisx0201ToUnicode(uint h, uint l) const;
    uint Jisx0201LatinToUnicode(uint h, uint l) const;
//    uint Jisx0201KanaToUnicode(uint h, uint l) const;
    uint Jisx0208ToUnicode(uint h, uint l) const;
    uint Jisx0212ToUnicode(uint h, uint l) const;

//    uint UnicodeToAscii(uint h, uint l) const;
//    uint UnicodeToJisx0201(uint h, uint l) const;
    uint UnicodeToJisx0201Latin(uint h, uint l) const;
//    uint UnicodeToJisx0201Kana(uint h, uint l) const;
    uint UnicodeToJisx0208(uint h, uint l) const;
    uint UnicodeToJisx0212(uint h, uint l) const;
};

uint QJpUnicodeConv_Microsoft::Jisx0201LatinToUnicode(uint h, uint l) const
{
    return AsciiToUnicode(h, l);
}

uint QJpUnicodeConv_Microsoft::Jisx0208ToUnicode(uint h, uint l) const
{
    if (h == 0x21) {
	if (l == 0x40) {
	    return 0xff3c;
	} else if (l == 0x41) {
	    return 0xff5e;
	} else if (l == 0x42) {
	    return 0x2225;
	} else if (l == 0x5d) {
	    return 0xff0d;
	} else if (l == 0x71) {
	    return 0xffe0;
	} else if (l == 0x72) {
	    return 0xffe1;
	}
    } else if (h == 0x22) {
	if (l == 0x4c) {
	    return 0xffe2;
	}
    }
    return QJpUnicodeConv::Jisx0208ToUnicode(h, l);
}

uint QJpUnicodeConv_Microsoft::Jisx0212ToUnicode(uint h, uint l) const
{
    if (h == 0x22) {
	if (l == 0x37) {
	    return 0xff5e;
	} else if (l == 0x43) {
	    return 0xffe4;
	}
    }
    return QJpUnicodeConv::Jisx0212ToUnicode(h, l);
}

uint QJpUnicodeConv_Microsoft::UnicodeToJisx0201Latin(uint h, uint l) const
{
    return QJpUnicodeConv::UnicodeToAscii(h, l);
}

uint QJpUnicodeConv_Microsoft::UnicodeToJisx0208(uint h, uint l) const
{
#ifdef STRICT
    if (((h == 0x00) && ((l == 0x5c) || (l == 0xa2) || (l == 0xa3) || (l == 0xac))) ||
	((h == 0x20) && (l == 0x16)) ||
	((h == 0x22) && (l == 0x12)) ||
	((h == 0x30) && (l == 0x1c))) {
	return 0x0000;
    }
#else
    if ((h == 0x00) && (l == 0x5c)) {
	return 0x0000;
    }
#endif
    if ((h == 0x22) && (l == 0x25)) {
	return 0x2142;
    } else if (h == 0xff) {
	if (l == 0x0d) {
	    return 0x215d;
	} else if (l == 0x3c) {
	    return 0x2140;
	} else if (l == 0x5e) {
	    return 0x2141;
	} else if (l == 0xe0) {
	    return 0x2171;
	} else if (l == 0xe1) {
	    return 0x2172;
	} else if (l == 0xe2) {
	    return 0x224c;
	}
    }
    return QJpUnicodeConv::UnicodeToJisx0208(h, l);
}

uint QJpUnicodeConv_Microsoft::UnicodeToJisx0212(uint h, uint l) const
{
#ifdef STRICT
    if ((h == 0x00) && ((l == 0x7e) || (l == 0xa6))) {
	return 0x0000;
    }
#else
    if ((h == 0x00) && (l == 0x7e)) {
	return 0x0000;
    }
#endif
    if (h == 0xff) {
	if (l == 0x5e) {
	    return 0x2237;
	} else if (l == 0xe4) {
	    return 0x2243;
	}
    }
    return QJpUnicodeConv::UnicodeToJisx0212(h, l);
}


/*! \internal */
const QJpUnicodeConv *QJpUnicodeConv::newConverter(int rule)
{
    const char * e = 0;
    if ( rule == JU_Default && (e=getenv("UNICODEMAP_JP")) != 0 ) {
	QCString env( e );
	for (int i = 0; i < (int)env.length(); ) {
	    int j = env.find(',', i);
	    QCString s;
	    if (j < 0) {
		s = env.mid(i).stripWhiteSpace();
		i = env.length();
	    } else {
		s = env.mid(i, j - i).stripWhiteSpace();
		i = j + 1;
	    }
	    if (qstricmp(s, "unicode-0.9") == 0) {
		rule = (rule & 0xff00) | JU_Unicode;
	    } else if (qstricmp(s, "unicode-0201") == 0) {
		rule = (rule & 0xff00) | JU_Unicode_JISX0201;
	    } else if (qstricmp(s, "unicode-ascii") == 0) {
		rule = (rule & 0xff00) | JU_Unicode_ASCII;
	    } else if (qstricmp(s, "jisx0221-1995") == 0) {
		rule = (rule & 0xff00) | JU_JISX0221_JISX0201;
	    } else if ((qstricmp(s, "open-0201") == 0) ||
		       (qstricmp(s, "open-19970715-0201") == 0)) {
		rule = (rule & 0xff00) | JU_JISX0221_JISX0201;
	    } else if ((qstricmp(s, "open-ascii") == 0) ||
		       (qstricmp(s, "open-19970715-ascii") == 0)) {
		rule = (rule & 0xff00) | JU_JISX0221_ASCII;
	    } else if ((qstricmp(s, "open-ms") == 0) ||
		       (qstricmp(s, "open-19970715-ms") == 0)) {
		rule = (rule & 0xff00) | JU_Microsoft_CP932;
	    } else if (qstricmp(s, "cp932") == 0) {
		rule = (rule & 0xff00) | JU_Microsoft_CP932;
	    } else if (qstricmp(s, "jdk1.1.7") == 0) {
		rule = (rule & 0xff00) | JU_Sun_JDK117;
	    } else if (qstricmp(s, "nec-vdc") == 0) {
		rule = rule | JU_NEC_VDC;
	    } else if (qstricmp(s, "ibm-vdc") == 0) {
		rule = rule | JU_IBM_VDC;
	    } else if (qstricmp(s, "udc") == 0) {
		rule = rule | JU_UDC;
	    }
	}
    }

    switch (rule & 0x00ff) {
    case JU_Unicode_JISX0201:
	return new QJpUnicodeConv_Unicode_JISX0201(rule);
    case JU_Unicode_ASCII:
	return new QJpUnicodeConv_Unicode_ASCII(rule);
    case JU_JISX0221_JISX0201:
	return new QJpUnicodeConv_JISX0221_JISX0201(rule);
    case JU_JISX0221_ASCII:
	return new QJpUnicodeConv_JISX0221_ASCII(rule);
    case JU_Sun_JDK117:
	return new QJpUnicodeConv_Sun(rule);
    case JU_Microsoft_CP932:
	return new QJpUnicodeConv_Microsoft(rule);
    default:
	return new QJpUnicodeConv_Unicode_ASCII(rule);
    }
}


/*
 * JISX0208 <-> ShiftJIS conversion.
 */

static uint Jisx0208ToSjis(uint h, uint l)
{
    if ((0x0021 <= h) && (h <= 0x007e) && (0x0021 <= l) && (l <= 0x007e)) {
	return ((((h - 1) >> 1) + ((h <= 0x5e) ? 0x71 : 0xb1)) << 8) |
	       (l + ((h & 1) ? ((l < 0x60) ? 0x1f : 0x20) : 0x7e));
    }
    return 0x0000;
}

static uint SjisToJisx0208(uint h, uint l)
{
    if ((((0x81 <= h) && (h <= 0x9f)) || ((0xe0 <= h) && (h <= 0xef))) &&
	((0x40 <= l) && (l != 0x7f) && (l <= 0xfc))) {
	if (l < 0x9f) {
	    return (((h << 1) - ((h <= 0x9f) ? 0x00e1 : 0x0161)) << 8) |
		   (l - ((l <= 0x7f) ? 0x1f : 0x20));
	} else {
	    return (((h << 1) - ((h <= 0x9f) ? 0x00e1 : 0x0161) + 1) << 8) |
		   (l - 0x7e);
	}
    }
    return 0x0000;
}

/*
 * This function is derived from Unicode 1.1,
 * JIS X 0201 (1976) to Unicode mapping table version 0.9 .
 */

#define	JISX0201_YEN_SIGN	0x005c
#define	UNICODE_YEN_SIGN	0x00a5

#define	JISX0201_OVERLINE	0x007e
#define	UNICODE_OVERLINE	0x203e

static uint Jisx0201ToUnicode11(uint h, uint l)
{
    if (h == 0x00) {
	if (l < 0x80) {
	    if (l == JISX0201_YEN_SIGN) {
		return UNICODE_YEN_SIGN;
	    } else if (l == JISX0201_OVERLINE) {
		return UNICODE_OVERLINE;
	    } else {
		return l;
	    }
	} else if ((0xa1 <= l) && (l <= 0x00df)) {
	    return 0xff61 + l - 0x00a1;
	}
    }
    return 0x0000;
}

/*
 * This function is derived from Unicode 1.1,
 * JIS X 0201 (1976) to Unicode mapping table version 0.9 .
 */

static uint Unicode11ToJisx0201(uint h, uint l)
{
    if ((h == 0x00) && (l < 0x80)) {
	if ((l == JISX0201_YEN_SIGN) ||
	    (l == JISX0201_OVERLINE)) {
	    return 0x0000;
	}
	return l;
    } else if ((h == 0x00) && (l == 0xa5)) {
	return JISX0201_YEN_SIGN;
    } else if ((h == 0x20) && (l == 0x3e)) {
	return JISX0201_OVERLINE;
    } else if ((h == 0xff) && (0x61 <= l) && (l <= 0x9f)) {
	return 0x00a1 + l - 0x61;
    }
    return 0x0000;
}

/*
 * This data is derived from Unicode 1.1,
 * JIS X 0208 (1990) to Unicode mapping table version 0.9 .
 * (In addition NEC Vender Defined Char included)
 */
static unsigned short const jisx0208_to_unicode[] = {
	/* 0x2121 - 0x217e */
		0x3000, 0x3001, 0x3002, 0xff0c, 0xff0e, 0x30fb, 0xff1a,
	0xff1b, 0xff1f, 0xff01, 0x309b, 0x309c, 0x00b4, 0xff40, 0x00a8,
	0xff3e, 0xffe3, 0xff3f, 0x30fd, 0x30fe, 0x309d, 0x309e, 0x3003,
	0x4edd, 0x3005, 0x3006, 0x3007, 0x30fc, 0x2015, 0x2010, 0xff0f,
	0x005c, 0x301c, 0x2016, 0xff5c, 0x2026, 0x2025, 0x2018, 0x2019,
	0x201c, 0x201d, 0xff08, 0xff09, 0x3014, 0x3015, 0xff3b, 0xff3d,
	0xff5b, 0xff5d, 0x3008, 0x3009, 0x300a, 0x300b, 0x300c, 0x300d,
	0x300e, 0x300f, 0x3010, 0x3011, 0xff0b, 0x2212, 0x00b1, 0x00d7,
	0x00f7, 0xff1d, 0x2260, 0xff1c, 0xff1e, 0x2266, 0x2267, 0x221e,
	0x2234, 0x2642, 0x2640, 0x00b0, 0x2032, 0x2033, 0x2103, 0xffe5,
	0xff04, 0x00a2, 0x00a3, 0xff05, 0xff03, 0xff06, 0xff0a, 0xff20,
	0x00a7, 0x2606, 0x2605, 0x25cb, 0x25cf, 0x25ce, 0x25c7,
	/* 0x2221 - 0x227e */
		0x25c6, 0x25a1, 0x25a0, 0x25b3, 0x25b2, 0x25bd, 0x25bc,
	0x203b, 0x3012, 0x2192, 0x2190, 0x2191, 0x2193, 0x3013, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2208, 0x220b, 0x2286, 0x2287, 0x2282, 0x2283,
	0x222a, 0x2229, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2227, 0x2228, 0x00ac, 0x21d2, 0x21d4, 0x2200,
	0x2203, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2220, 0x22a5, 0x2312, 0x2202,
	0x2207, 0x2261, 0x2252, 0x226a, 0x226b, 0x221a, 0x223d, 0x221d,
	0x2235, 0x222b, 0x222c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x212b, 0x2030, 0x266f, 0x266d, 0x266a, 0x2020,
	0x2021, 0x00b6, 0x0000, 0x0000, 0x0000, 0x0000, 0x25ef,
	/* 0x2321 - 0x237e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17,
	0xff18, 0xff19, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27,
	0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f,
	0xff30, 0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37,
	0xff38, 0xff39, 0xff3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47,
	0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f,
	0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57,
	0xff58, 0xff59, 0xff5a, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2421 - 0x247e */
		0x3041, 0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047,
	0x3048, 0x3049, 0x304a, 0x304b, 0x304c, 0x304d, 0x304e, 0x304f,
	0x3050, 0x3051, 0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057,
	0x3058, 0x3059, 0x305a, 0x305b, 0x305c, 0x305d, 0x305e, 0x305f,
	0x3060, 0x3061, 0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 0x3067,
	0x3068, 0x3069, 0x306a, 0x306b, 0x306c, 0x306d, 0x306e, 0x306f,
	0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077,
	0x3078, 0x3079, 0x307a, 0x307b, 0x307c, 0x307d, 0x307e, 0x307f,
	0x3080, 0x3081, 0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087,
	0x3088, 0x3089, 0x308a, 0x308b, 0x308c, 0x308d, 0x308e, 0x308f,
	0x3090, 0x3091, 0x3092, 0x3093, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2521 - 0x257e */
		0x30a1, 0x30a2, 0x30a3, 0x30a4, 0x30a5, 0x30a6, 0x30a7,
	0x30a8, 0x30a9, 0x30aa, 0x30ab, 0x30ac, 0x30ad, 0x30ae, 0x30af,
	0x30b0, 0x30b1, 0x30b2, 0x30b3, 0x30b4, 0x30b5, 0x30b6, 0x30b7,
	0x30b8, 0x30b9, 0x30ba, 0x30bb, 0x30bc, 0x30bd, 0x30be, 0x30bf,
	0x30c0, 0x30c1, 0x30c2, 0x30c3, 0x30c4, 0x30c5, 0x30c6, 0x30c7,
	0x30c8, 0x30c9, 0x30ca, 0x30cb, 0x30cc, 0x30cd, 0x30ce, 0x30cf,
	0x30d0, 0x30d1, 0x30d2, 0x30d3, 0x30d4, 0x30d5, 0x30d6, 0x30d7,
	0x30d8, 0x30d9, 0x30da, 0x30db, 0x30dc, 0x30dd, 0x30de, 0x30df,
	0x30e0, 0x30e1, 0x30e2, 0x30e3, 0x30e4, 0x30e5, 0x30e6, 0x30e7,
	0x30e8, 0x30e9, 0x30ea, 0x30eb, 0x30ec, 0x30ed, 0x30ee, 0x30ef,
	0x30f0, 0x30f1, 0x30f2, 0x30f3, 0x30f4, 0x30f5, 0x30f6, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2621 - 0x267e */
		0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397,
	0x0398, 0x0399, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f,
	0x03a0, 0x03a1, 0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8,
	0x03a9, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7,
	0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf,
	0x03c0, 0x03c1, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7, 0x03c8,
	0x03c9, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2721 - 0x277e */
		0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401,
	0x0416, 0x0417, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d,
	0x041e, 0x041f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425,
	0x0426, 0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d,
	0x042e, 0x042f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451,
	0x0436, 0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d,
	0x043e, 0x043f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445,
	0x0446, 0x0447, 0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d,
	0x044e, 0x044f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2821 - 0x287e */
		0x2500, 0x2502, 0x250c, 0x2510, 0x2518, 0x2514, 0x251c,
	0x252c, 0x2524, 0x2534, 0x253c, 0x2501, 0x2503, 0x250f, 0x2513,
	0x251b, 0x2517, 0x2523, 0x2533, 0x252b, 0x253b, 0x254b, 0x2520,
	0x252f, 0x2528, 0x2537, 0x253f, 0x251d, 0x2530, 0x2525, 0x2538,
	0x2542, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2921 - 0x297e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2a21 - 0x2a7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2b21 - 0x2b7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2c21 - 0x2c7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2d21 - 0x2d7e */
		0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466,
	0x2467, 0x2468, 0x2469, 0x246a, 0x246b, 0x246c, 0x246d, 0x246e,
	0x246f, 0x2470, 0x2471, 0x2472, 0x2473, 0x2160, 0x2161, 0x2162,
	0x2163, 0x2164, 0x2165, 0x2166, 0x2167, 0x2168, 0x2169, 0x0000,
	0x3349, 0x3314, 0x3322, 0x334d, 0x3318, 0x3327, 0x3303, 0x3336,
	0x3351, 0x3357, 0x330d, 0x3326, 0x3323, 0x332b, 0x334a, 0x333b,
	0x339c, 0x339d, 0x339e, 0x338e, 0x338f, 0x33c4, 0x33a1, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x337b,
	0x301d, 0x301f, 0x2116, 0x33cd, 0x2121, 0x32a4, 0x32a5, 0x32a6,
	0x32a7, 0x32a8, 0x3231, 0x3232, 0x3239, 0x337e, 0x337d, 0x337c,
	0x2252, 0x2261, 0x222b, 0x222e, 0x2211, 0x221a, 0x22a5, 0x2220,
	0x221f, 0x22bf, 0x2235, 0x2229, 0x222a, 0x0000, 0x0000,
	/* 0x2e21 - 0x2e7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2f21 - 0x2f7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x3021 - 0x307e */
		0x4e9c, 0x5516, 0x5a03, 0x963f, 0x54c0, 0x611b, 0x6328,
	0x59f6, 0x9022, 0x8475, 0x831c, 0x7a50, 0x60aa, 0x63e1, 0x6e25,
	0x65ed, 0x8466, 0x82a6, 0x9bf5, 0x6893, 0x5727, 0x65a1, 0x6271,
	0x5b9b, 0x59d0, 0x867b, 0x98f4, 0x7d62, 0x7dbe, 0x9b8e, 0x6216,
	0x7c9f, 0x88b7, 0x5b89, 0x5eb5, 0x6309, 0x6697, 0x6848, 0x95c7,
	0x978d, 0x674f, 0x4ee5, 0x4f0a, 0x4f4d, 0x4f9d, 0x5049, 0x56f2,
	0x5937, 0x59d4, 0x5a01, 0x5c09, 0x60df, 0x610f, 0x6170, 0x6613,
	0x6905, 0x70ba, 0x754f, 0x7570, 0x79fb, 0x7dad, 0x7def, 0x80c3,
	0x840e, 0x8863, 0x8b02, 0x9055, 0x907a, 0x533b, 0x4e95, 0x4ea5,
	0x57df, 0x80b2, 0x90c1, 0x78ef, 0x4e00, 0x58f1, 0x6ea2, 0x9038,
	0x7a32, 0x8328, 0x828b, 0x9c2f, 0x5141, 0x5370, 0x54bd, 0x54e1,
	0x56e0, 0x59fb, 0x5f15, 0x98f2, 0x6deb, 0x80e4, 0x852d,
	/* 0x3121 - 0x317e */
		0x9662, 0x9670, 0x96a0, 0x97fb, 0x540b, 0x53f3, 0x5b87,
	0x70cf, 0x7fbd, 0x8fc2, 0x96e8, 0x536f, 0x9d5c, 0x7aba, 0x4e11,
	0x7893, 0x81fc, 0x6e26, 0x5618, 0x5504, 0x6b1d, 0x851a, 0x9c3b,
	0x59e5, 0x53a9, 0x6d66, 0x74dc, 0x958f, 0x5642, 0x4e91, 0x904b,
	0x96f2, 0x834f, 0x990c, 0x53e1, 0x55b6, 0x5b30, 0x5f71, 0x6620,
	0x66f3, 0x6804, 0x6c38, 0x6cf3, 0x6d29, 0x745b, 0x76c8, 0x7a4e,
	0x9834, 0x82f1, 0x885b, 0x8a60, 0x92ed, 0x6db2, 0x75ab, 0x76ca,
	0x99c5, 0x60a6, 0x8b01, 0x8d8a, 0x95b2, 0x698e, 0x53ad, 0x5186,
	0x5712, 0x5830, 0x5944, 0x5bb4, 0x5ef6, 0x6028, 0x63a9, 0x63f4,
	0x6cbf, 0x6f14, 0x708e, 0x7114, 0x7159, 0x71d5, 0x733f, 0x7e01,
	0x8276, 0x82d1, 0x8597, 0x9060, 0x925b, 0x9d1b, 0x5869, 0x65bc,
	0x6c5a, 0x7525, 0x51f9, 0x592e, 0x5965, 0x5f80, 0x5fdc,
	/* 0x3221 - 0x327e */
		0x62bc, 0x65fa, 0x6a2a, 0x6b27, 0x6bb4, 0x738b, 0x7fc1,
	0x8956, 0x9d2c, 0x9d0e, 0x9ec4, 0x5ca1, 0x6c96, 0x837b, 0x5104,
	0x5c4b, 0x61b6, 0x81c6, 0x6876, 0x7261, 0x4e59, 0x4ffa, 0x5378,
	0x6069, 0x6e29, 0x7a4f, 0x97f3, 0x4e0b, 0x5316, 0x4eee, 0x4f55,
	0x4f3d, 0x4fa1, 0x4f73, 0x52a0, 0x53ef, 0x5609, 0x590f, 0x5ac1,
	0x5bb6, 0x5be1, 0x79d1, 0x6687, 0x679c, 0x67b6, 0x6b4c, 0x6cb3,
	0x706b, 0x73c2, 0x798d, 0x79be, 0x7a3c, 0x7b87, 0x82b1, 0x82db,
	0x8304, 0x8377, 0x83ef, 0x83d3, 0x8766, 0x8ab2, 0x5629, 0x8ca8,
	0x8fe6, 0x904e, 0x971e, 0x868a, 0x4fc4, 0x5ce8, 0x6211, 0x7259,
	0x753b, 0x81e5, 0x82bd, 0x86fe, 0x8cc0, 0x96c5, 0x9913, 0x99d5,
	0x4ecb, 0x4f1a, 0x89e3, 0x56de, 0x584a, 0x58ca, 0x5efb, 0x5feb,
	0x602a, 0x6094, 0x6062, 0x61d0, 0x6212, 0x62d0, 0x6539,
	/* 0x3321 - 0x337e */
		0x9b41, 0x6666, 0x68b0, 0x6d77, 0x7070, 0x754c, 0x7686,
	0x7d75, 0x82a5, 0x87f9, 0x958b, 0x968e, 0x8c9d, 0x51f1, 0x52be,
	0x5916, 0x54b3, 0x5bb3, 0x5d16, 0x6168, 0x6982, 0x6daf, 0x788d,
	0x84cb, 0x8857, 0x8a72, 0x93a7, 0x9ab8, 0x6d6c, 0x99a8, 0x86d9,
	0x57a3, 0x67ff, 0x86ce, 0x920e, 0x5283, 0x5687, 0x5404, 0x5ed3,
	0x62e1, 0x64b9, 0x683c, 0x6838, 0x6bbb, 0x7372, 0x78ba, 0x7a6b,
	0x899a, 0x89d2, 0x8d6b, 0x8f03, 0x90ed, 0x95a3, 0x9694, 0x9769,
	0x5b66, 0x5cb3, 0x697d, 0x984d, 0x984e, 0x639b, 0x7b20, 0x6a2b,
	0x6a7f, 0x68b6, 0x9c0d, 0x6f5f, 0x5272, 0x559d, 0x6070, 0x62ec,
	0x6d3b, 0x6e07, 0x6ed1, 0x845b, 0x8910, 0x8f44, 0x4e14, 0x9c39,
	0x53f6, 0x691b, 0x6a3a, 0x9784, 0x682a, 0x515c, 0x7ac3, 0x84b2,
	0x91dc, 0x938c, 0x565b, 0x9d28, 0x6822, 0x8305, 0x8431,
	/* 0x3421 - 0x347e */
		0x7ca5, 0x5208, 0x82c5, 0x74e6, 0x4e7e, 0x4f83, 0x51a0,
	0x5bd2, 0x520a, 0x52d8, 0x52e7, 0x5dfb, 0x559a, 0x582a, 0x59e6,
	0x5b8c, 0x5b98, 0x5bdb, 0x5e72, 0x5e79, 0x60a3, 0x611f, 0x6163,
	0x61be, 0x63db, 0x6562, 0x67d1, 0x6853, 0x68fa, 0x6b3e, 0x6b53,
	0x6c57, 0x6f22, 0x6f97, 0x6f45, 0x74b0, 0x7518, 0x76e3, 0x770b,
	0x7aff, 0x7ba1, 0x7c21, 0x7de9, 0x7f36, 0x7ff0, 0x809d, 0x8266,
	0x839e, 0x89b3, 0x8acc, 0x8cab, 0x9084, 0x9451, 0x9593, 0x9591,
	0x95a2, 0x9665, 0x97d3, 0x9928, 0x8218, 0x4e38, 0x542b, 0x5cb8,
	0x5dcc, 0x73a9, 0x764c, 0x773c, 0x5ca9, 0x7feb, 0x8d0b, 0x96c1,
	0x9811, 0x9854, 0x9858, 0x4f01, 0x4f0e, 0x5371, 0x559c, 0x5668,
	0x57fa, 0x5947, 0x5b09, 0x5bc4, 0x5c90, 0x5e0c, 0x5e7e, 0x5fcc,
	0x63ee, 0x673a, 0x65d7, 0x65e2, 0x671f, 0x68cb, 0x68c4,
	/* 0x3521 - 0x357e */
		0x6a5f, 0x5e30, 0x6bc5, 0x6c17, 0x6c7d, 0x757f, 0x7948,
	0x5b63, 0x7a00, 0x7d00, 0x5fbd, 0x898f, 0x8a18, 0x8cb4, 0x8d77,
	0x8ecc, 0x8f1d, 0x98e2, 0x9a0e, 0x9b3c, 0x4e80, 0x507d, 0x5100,
	0x5993, 0x5b9c, 0x622f, 0x6280, 0x64ec, 0x6b3a, 0x72a0, 0x7591,
	0x7947, 0x7fa9, 0x87fb, 0x8abc, 0x8b70, 0x63ac, 0x83ca, 0x97a0,
	0x5409, 0x5403, 0x55ab, 0x6854, 0x6a58, 0x8a70, 0x7827, 0x6775,
	0x9ecd, 0x5374, 0x5ba2, 0x811a, 0x8650, 0x9006, 0x4e18, 0x4e45,
	0x4ec7, 0x4f11, 0x53ca, 0x5438, 0x5bae, 0x5f13, 0x6025, 0x6551,
	0x673d, 0x6c42, 0x6c72, 0x6ce3, 0x7078, 0x7403, 0x7a76, 0x7aae,
	0x7b08, 0x7d1a, 0x7cfe, 0x7d66, 0x65e7, 0x725b, 0x53bb, 0x5c45,
	0x5de8, 0x62d2, 0x62e0, 0x6319, 0x6e20, 0x865a, 0x8a31, 0x8ddd,
	0x92f8, 0x6f01, 0x79a6, 0x9b5a, 0x4ea8, 0x4eab, 0x4eac,
	/* 0x3621 - 0x367e */
		0x4f9b, 0x4fa0, 0x50d1, 0x5147, 0x7af6, 0x5171, 0x51f6,
	0x5354, 0x5321, 0x537f, 0x53eb, 0x55ac, 0x5883, 0x5ce1, 0x5f37,
	0x5f4a, 0x602f, 0x6050, 0x606d, 0x631f, 0x6559, 0x6a4b, 0x6cc1,
	0x72c2, 0x72ed, 0x77ef, 0x80f8, 0x8105, 0x8208, 0x854e, 0x90f7,
	0x93e1, 0x97ff, 0x9957, 0x9a5a, 0x4ef0, 0x51dd, 0x5c2d, 0x6681,
	0x696d, 0x5c40, 0x66f2, 0x6975, 0x7389, 0x6850, 0x7c81, 0x50c5,
	0x52e4, 0x5747, 0x5dfe, 0x9326, 0x65a4, 0x6b23, 0x6b3d, 0x7434,
	0x7981, 0x79bd, 0x7b4b, 0x7dca, 0x82b9, 0x83cc, 0x887f, 0x895f,
	0x8b39, 0x8fd1, 0x91d1, 0x541f, 0x9280, 0x4e5d, 0x5036, 0x53e5,
	0x533a, 0x72d7, 0x7396, 0x77e9, 0x82e6, 0x8eaf, 0x99c6, 0x99c8,
	0x99d2, 0x5177, 0x611a, 0x865e, 0x55b0, 0x7a7a, 0x5076, 0x5bd3,
	0x9047, 0x9685, 0x4e32, 0x6adb, 0x91e7, 0x5c51, 0x5c48,
	/* 0x3721 - 0x377e */
		0x6398, 0x7a9f, 0x6c93, 0x9774, 0x8f61, 0x7aaa, 0x718a,
	0x9688, 0x7c82, 0x6817, 0x7e70, 0x6851, 0x936c, 0x52f2, 0x541b,
	0x85ab, 0x8a13, 0x7fa4, 0x8ecd, 0x90e1, 0x5366, 0x8888, 0x7941,
	0x4fc2, 0x50be, 0x5211, 0x5144, 0x5553, 0x572d, 0x73ea, 0x578b,
	0x5951, 0x5f62, 0x5f84, 0x6075, 0x6176, 0x6167, 0x61a9, 0x63b2,
	0x643a, 0x656c, 0x666f, 0x6842, 0x6e13, 0x7566, 0x7a3d, 0x7cfb,
	0x7d4c, 0x7d99, 0x7e4b, 0x7f6b, 0x830e, 0x834a, 0x86cd, 0x8a08,
	0x8a63, 0x8b66, 0x8efd, 0x981a, 0x9d8f, 0x82b8, 0x8fce, 0x9be8,
	0x5287, 0x621f, 0x6483, 0x6fc0, 0x9699, 0x6841, 0x5091, 0x6b20,
	0x6c7a, 0x6f54, 0x7a74, 0x7d50, 0x8840, 0x8a23, 0x6708, 0x4ef6,
	0x5039, 0x5026, 0x5065, 0x517c, 0x5238, 0x5263, 0x55a7, 0x570f,
	0x5805, 0x5acc, 0x5efa, 0x61b2, 0x61f8, 0x62f3, 0x6372,
	/* 0x3821 - 0x387e */
		0x691c, 0x6a29, 0x727d, 0x72ac, 0x732e, 0x7814, 0x786f,
	0x7d79, 0x770c, 0x80a9, 0x898b, 0x8b19, 0x8ce2, 0x8ed2, 0x9063,
	0x9375, 0x967a, 0x9855, 0x9a13, 0x9e78, 0x5143, 0x539f, 0x53b3,
	0x5e7b, 0x5f26, 0x6e1b, 0x6e90, 0x7384, 0x73fe, 0x7d43, 0x8237,
	0x8a00, 0x8afa, 0x9650, 0x4e4e, 0x500b, 0x53e4, 0x547c, 0x56fa,
	0x59d1, 0x5b64, 0x5df1, 0x5eab, 0x5f27, 0x6238, 0x6545, 0x67af,
	0x6e56, 0x72d0, 0x7cca, 0x88b4, 0x80a1, 0x80e1, 0x83f0, 0x864e,
	0x8a87, 0x8de8, 0x9237, 0x96c7, 0x9867, 0x9f13, 0x4e94, 0x4e92,
	0x4f0d, 0x5348, 0x5449, 0x543e, 0x5a2f, 0x5f8c, 0x5fa1, 0x609f,
	0x68a7, 0x6a8e, 0x745a, 0x7881, 0x8a9e, 0x8aa4, 0x8b77, 0x9190,
	0x4e5e, 0x9bc9, 0x4ea4, 0x4f7c, 0x4faf, 0x5019, 0x5016, 0x5149,
	0x516c, 0x529f, 0x52b9, 0x52fe, 0x539a, 0x53e3, 0x5411,
	/* 0x3921 - 0x397e */
		0x540e, 0x5589, 0x5751, 0x57a2, 0x597d, 0x5b54, 0x5b5d,
	0x5b8f, 0x5de5, 0x5de7, 0x5df7, 0x5e78, 0x5e83, 0x5e9a, 0x5eb7,
	0x5f18, 0x6052, 0x614c, 0x6297, 0x62d8, 0x63a7, 0x653b, 0x6602,
	0x6643, 0x66f4, 0x676d, 0x6821, 0x6897, 0x69cb, 0x6c5f, 0x6d2a,
	0x6d69, 0x6e2f, 0x6e9d, 0x7532, 0x7687, 0x786c, 0x7a3f, 0x7ce0,
	0x7d05, 0x7d18, 0x7d5e, 0x7db1, 0x8015, 0x8003, 0x80af, 0x80b1,
	0x8154, 0x818f, 0x822a, 0x8352, 0x884c, 0x8861, 0x8b1b, 0x8ca2,
	0x8cfc, 0x90ca, 0x9175, 0x9271, 0x783f, 0x92fc, 0x95a4, 0x964d,
	0x9805, 0x9999, 0x9ad8, 0x9d3b, 0x525b, 0x52ab, 0x53f7, 0x5408,
	0x58d5, 0x62f7, 0x6fe0, 0x8c6a, 0x8f5f, 0x9eb9, 0x514b, 0x523b,
	0x544a, 0x56fd, 0x7a40, 0x9177, 0x9d60, 0x9ed2, 0x7344, 0x6f09,
	0x8170, 0x7511, 0x5ffd, 0x60da, 0x9aa8, 0x72db, 0x8fbc,
	/* 0x3a21 - 0x3a7e */
		0x6b64, 0x9803, 0x4eca, 0x56f0, 0x5764, 0x58be, 0x5a5a,
	0x6068, 0x61c7, 0x660f, 0x6606, 0x6839, 0x68b1, 0x6df7, 0x75d5,
	0x7d3a, 0x826e, 0x9b42, 0x4e9b, 0x4f50, 0x53c9, 0x5506, 0x5d6f,
	0x5de6, 0x5dee, 0x67fb, 0x6c99, 0x7473, 0x7802, 0x8a50, 0x9396,
	0x88df, 0x5750, 0x5ea7, 0x632b, 0x50b5, 0x50ac, 0x518d, 0x6700,
	0x54c9, 0x585e, 0x59bb, 0x5bb0, 0x5f69, 0x624d, 0x63a1, 0x683d,
	0x6b73, 0x6e08, 0x707d, 0x91c7, 0x7280, 0x7815, 0x7826, 0x796d,
	0x658e, 0x7d30, 0x83dc, 0x88c1, 0x8f09, 0x969b, 0x5264, 0x5728,
	0x6750, 0x7f6a, 0x8ca1, 0x51b4, 0x5742, 0x962a, 0x583a, 0x698a,
	0x80b4, 0x54b2, 0x5d0e, 0x57fc, 0x7895, 0x9dfa, 0x4f5c, 0x524a,
	0x548b, 0x643e, 0x6628, 0x6714, 0x67f5, 0x7a84, 0x7b56, 0x7d22,
	0x932f, 0x685c, 0x9bad, 0x7b39, 0x5319, 0x518a, 0x5237,
	/* 0x3b21 - 0x3b7e */
		0x5bdf, 0x62f6, 0x64ae, 0x64e6, 0x672d, 0x6bba, 0x85a9,
	0x96d1, 0x7690, 0x9bd6, 0x634c, 0x9306, 0x9bab, 0x76bf, 0x6652,
	0x4e09, 0x5098, 0x53c2, 0x5c71, 0x60e8, 0x6492, 0x6563, 0x685f,
	0x71e6, 0x73ca, 0x7523, 0x7b97, 0x7e82, 0x8695, 0x8b83, 0x8cdb,
	0x9178, 0x9910, 0x65ac, 0x66ab, 0x6b8b, 0x4ed5, 0x4ed4, 0x4f3a,
	0x4f7f, 0x523a, 0x53f8, 0x53f2, 0x55e3, 0x56db, 0x58eb, 0x59cb,
	0x59c9, 0x59ff, 0x5b50, 0x5c4d, 0x5e02, 0x5e2b, 0x5fd7, 0x601d,
	0x6307, 0x652f, 0x5b5c, 0x65af, 0x65bd, 0x65e8, 0x679d, 0x6b62,
	0x6b7b, 0x6c0f, 0x7345, 0x7949, 0x79c1, 0x7cf8, 0x7d19, 0x7d2b,
	0x80a2, 0x8102, 0x81f3, 0x8996, 0x8a5e, 0x8a69, 0x8a66, 0x8a8c,
	0x8aee, 0x8cc7, 0x8cdc, 0x96cc, 0x98fc, 0x6b6f, 0x4e8b, 0x4f3c,
	0x4f8d, 0x5150, 0x5b57, 0x5bfa, 0x6148, 0x6301, 0x6642,
	/* 0x3c21 - 0x3c7e */
		0x6b21, 0x6ecb, 0x6cbb, 0x723e, 0x74bd, 0x75d4, 0x78c1,
	0x793a, 0x800c, 0x8033, 0x81ea, 0x8494, 0x8f9e, 0x6c50, 0x9e7f,
	0x5f0f, 0x8b58, 0x9d2b, 0x7afa, 0x8ef8, 0x5b8d, 0x96eb, 0x4e03,
	0x53f1, 0x57f7, 0x5931, 0x5ac9, 0x5ba4, 0x6089, 0x6e7f, 0x6f06,
	0x75be, 0x8cea, 0x5b9f, 0x8500, 0x7be0, 0x5072, 0x67f4, 0x829d,
	0x5c61, 0x854a, 0x7e1e, 0x820e, 0x5199, 0x5c04, 0x6368, 0x8d66,
	0x659c, 0x716e, 0x793e, 0x7d17, 0x8005, 0x8b1d, 0x8eca, 0x906e,
	0x86c7, 0x90aa, 0x501f, 0x52fa, 0x5c3a, 0x6753, 0x707c, 0x7235,
	0x914c, 0x91c8, 0x932b, 0x82e5, 0x5bc2, 0x5f31, 0x60f9, 0x4e3b,
	0x53d6, 0x5b88, 0x624b, 0x6731, 0x6b8a, 0x72e9, 0x73e0, 0x7a2e,
	0x816b, 0x8da3, 0x9152, 0x9996, 0x5112, 0x53d7, 0x546a, 0x5bff,
	0x6388, 0x6a39, 0x7dac, 0x9700, 0x56da, 0x53ce, 0x5468,
	/* 0x3d21 - 0x3d7e */
		0x5b97, 0x5c31, 0x5dde, 0x4fee, 0x6101, 0x62fe, 0x6d32,
	0x79c0, 0x79cb, 0x7d42, 0x7e4d, 0x7fd2, 0x81ed, 0x821f, 0x8490,
	0x8846, 0x8972, 0x8b90, 0x8e74, 0x8f2f, 0x9031, 0x914b, 0x916c,
	0x96c6, 0x919c, 0x4ec0, 0x4f4f, 0x5145, 0x5341, 0x5f93, 0x620e,
	0x67d4, 0x6c41, 0x6e0b, 0x7363, 0x7e26, 0x91cd, 0x9283, 0x53d4,
	0x5919, 0x5bbf, 0x6dd1, 0x795d, 0x7e2e, 0x7c9b, 0x587e, 0x719f,
	0x51fa, 0x8853, 0x8ff0, 0x4fca, 0x5cfb, 0x6625, 0x77ac, 0x7ae3,
	0x821c, 0x99ff, 0x51c6, 0x5faa, 0x65ec, 0x696f, 0x6b89, 0x6df3,
	0x6e96, 0x6f64, 0x76fe, 0x7d14, 0x5de1, 0x9075, 0x9187, 0x9806,
	0x51e6, 0x521d, 0x6240, 0x6691, 0x66d9, 0x6e1a, 0x5eb6, 0x7dd2,
	0x7f72, 0x66f8, 0x85af, 0x85f7, 0x8af8, 0x52a9, 0x53d9, 0x5973,
	0x5e8f, 0x5f90, 0x6055, 0x92e4, 0x9664, 0x50b7, 0x511f,
	/* 0x3e21 - 0x3e7e */
		0x52dd, 0x5320, 0x5347, 0x53ec, 0x54e8, 0x5546, 0x5531,
	0x5617, 0x5968, 0x59be, 0x5a3c, 0x5bb5, 0x5c06, 0x5c0f, 0x5c11,
	0x5c1a, 0x5e84, 0x5e8a, 0x5ee0, 0x5f70, 0x627f, 0x6284, 0x62db,
	0x638c, 0x6377, 0x6607, 0x660c, 0x662d, 0x6676, 0x677e, 0x68a2,
	0x6a1f, 0x6a35, 0x6cbc, 0x6d88, 0x6e09, 0x6e58, 0x713c, 0x7126,
	0x7167, 0x75c7, 0x7701, 0x785d, 0x7901, 0x7965, 0x79f0, 0x7ae0,
	0x7b11, 0x7ca7, 0x7d39, 0x8096, 0x83d6, 0x848b, 0x8549, 0x885d,
	0x88f3, 0x8a1f, 0x8a3c, 0x8a54, 0x8a73, 0x8c61, 0x8cde, 0x91a4,
	0x9266, 0x937e, 0x9418, 0x969c, 0x9798, 0x4e0a, 0x4e08, 0x4e1e,
	0x4e57, 0x5197, 0x5270, 0x57ce, 0x5834, 0x58cc, 0x5b22, 0x5e38,
	0x60c5, 0x64fe, 0x6761, 0x6756, 0x6d44, 0x72b6, 0x7573, 0x7a63,
	0x84b8, 0x8b72, 0x91b8, 0x9320, 0x5631, 0x57f4, 0x98fe,
	/* 0x3f21 - 0x3f7e */
		0x62ed, 0x690d, 0x6b96, 0x71ed, 0x7e54, 0x8077, 0x8272,
	0x89e6, 0x98df, 0x8755, 0x8fb1, 0x5c3b, 0x4f38, 0x4fe1, 0x4fb5,
	0x5507, 0x5a20, 0x5bdd, 0x5be9, 0x5fc3, 0x614e, 0x632f, 0x65b0,
	0x664b, 0x68ee, 0x699b, 0x6d78, 0x6df1, 0x7533, 0x75b9, 0x771f,
	0x795e, 0x79e6, 0x7d33, 0x81e3, 0x82af, 0x85aa, 0x89aa, 0x8a3a,
	0x8eab, 0x8f9b, 0x9032, 0x91dd, 0x9707, 0x4eba, 0x4ec1, 0x5203,
	0x5875, 0x58ec, 0x5c0b, 0x751a, 0x5c3d, 0x814e, 0x8a0a, 0x8fc5,
	0x9663, 0x976d, 0x7b25, 0x8acf, 0x9808, 0x9162, 0x56f3, 0x53a8,
	0x9017, 0x5439, 0x5782, 0x5e25, 0x63a8, 0x6c34, 0x708a, 0x7761,
	0x7c8b, 0x7fe0, 0x8870, 0x9042, 0x9154, 0x9310, 0x9318, 0x968f,
	0x745e, 0x9ac4, 0x5d07, 0x5d69, 0x6570, 0x67a2, 0x8da8, 0x96db,
	0x636e, 0x6749, 0x6919, 0x83c5, 0x9817, 0x96c0, 0x88fe,
	/* 0x4021 - 0x407e */
		0x6f84, 0x647a, 0x5bf8, 0x4e16, 0x702c, 0x755d, 0x662f,
	0x51c4, 0x5236, 0x52e2, 0x59d3, 0x5f81, 0x6027, 0x6210, 0x653f,
	0x6574, 0x661f, 0x6674, 0x68f2, 0x6816, 0x6b63, 0x6e05, 0x7272,
	0x751f, 0x76db, 0x7cbe, 0x8056, 0x58f0, 0x88fd, 0x897f, 0x8aa0,
	0x8a93, 0x8acb, 0x901d, 0x9192, 0x9752, 0x9759, 0x6589, 0x7a0e,
	0x8106, 0x96bb, 0x5e2d, 0x60dc, 0x621a, 0x65a5, 0x6614, 0x6790,
	0x77f3, 0x7a4d, 0x7c4d, 0x7e3e, 0x810a, 0x8cac, 0x8d64, 0x8de1,
	0x8e5f, 0x78a9, 0x5207, 0x62d9, 0x63a5, 0x6442, 0x6298, 0x8a2d,
	0x7a83, 0x7bc0, 0x8aac, 0x96ea, 0x7d76, 0x820c, 0x8749, 0x4ed9,
	0x5148, 0x5343, 0x5360, 0x5ba3, 0x5c02, 0x5c16, 0x5ddd, 0x6226,
	0x6247, 0x64b0, 0x6813, 0x6834, 0x6cc9, 0x6d45, 0x6d17, 0x67d3,
	0x6f5c, 0x714e, 0x717d, 0x65cb, 0x7a7f, 0x7bad, 0x7dda,
	/* 0x4121 - 0x417e */
		0x7e4a, 0x7fa8, 0x817a, 0x821b, 0x8239, 0x85a6, 0x8a6e,
	0x8cce, 0x8df5, 0x9078, 0x9077, 0x92ad, 0x9291, 0x9583, 0x9bae,
	0x524d, 0x5584, 0x6f38, 0x7136, 0x5168, 0x7985, 0x7e55, 0x81b3,
	0x7cce, 0x564c, 0x5851, 0x5ca8, 0x63aa, 0x66fe, 0x66fd, 0x695a,
	0x72d9, 0x758f, 0x758e, 0x790e, 0x7956, 0x79df, 0x7c97, 0x7d20,
	0x7d44, 0x8607, 0x8a34, 0x963b, 0x9061, 0x9f20, 0x50e7, 0x5275,
	0x53cc, 0x53e2, 0x5009, 0x55aa, 0x58ee, 0x594f, 0x723d, 0x5b8b,
	0x5c64, 0x531d, 0x60e3, 0x60f3, 0x635c, 0x6383, 0x633f, 0x63bb,
	0x64cd, 0x65e9, 0x66f9, 0x5de3, 0x69cd, 0x69fd, 0x6f15, 0x71e5,
	0x4e89, 0x75e9, 0x76f8, 0x7a93, 0x7cdf, 0x7dcf, 0x7d9c, 0x8061,
	0x8349, 0x8358, 0x846c, 0x84bc, 0x85fb, 0x88c5, 0x8d70, 0x9001,
	0x906d, 0x9397, 0x971c, 0x9a12, 0x50cf, 0x5897, 0x618e,
	/* 0x4221 - 0x427e */
		0x81d3, 0x8535, 0x8d08, 0x9020, 0x4fc3, 0x5074, 0x5247,
	0x5373, 0x606f, 0x6349, 0x675f, 0x6e2c, 0x8db3, 0x901f, 0x4fd7,
	0x5c5e, 0x8cca, 0x65cf, 0x7d9a, 0x5352, 0x8896, 0x5176, 0x63c3,
	0x5b58, 0x5b6b, 0x5c0a, 0x640d, 0x6751, 0x905c, 0x4ed6, 0x591a,
	0x592a, 0x6c70, 0x8a51, 0x553e, 0x5815, 0x59a5, 0x60f0, 0x6253,
	0x67c1, 0x8235, 0x6955, 0x9640, 0x99c4, 0x9a28, 0x4f53, 0x5806,
	0x5bfe, 0x8010, 0x5cb1, 0x5e2f, 0x5f85, 0x6020, 0x614b, 0x6234,
	0x66ff, 0x6cf0, 0x6ede, 0x80ce, 0x817f, 0x82d4, 0x888b, 0x8cb8,
	0x9000, 0x902e, 0x968a, 0x9edb, 0x9bdb, 0x4ee3, 0x53f0, 0x5927,
	0x7b2c, 0x918d, 0x984c, 0x9df9, 0x6edd, 0x7027, 0x5353, 0x5544,
	0x5b85, 0x6258, 0x629e, 0x62d3, 0x6ca2, 0x6fef, 0x7422, 0x8a17,
	0x9438, 0x6fc1, 0x8afe, 0x8338, 0x51e7, 0x86f8, 0x53ea,
	/* 0x4321 - 0x437e */
		0x53e9, 0x4f46, 0x9054, 0x8fb0, 0x596a, 0x8131, 0x5dfd,
	0x7aea, 0x8fbf, 0x68da, 0x8c37, 0x72f8, 0x9c48, 0x6a3d, 0x8ab0,
	0x4e39, 0x5358, 0x5606, 0x5766, 0x62c5, 0x63a2, 0x65e6, 0x6b4e,
	0x6de1, 0x6e5b, 0x70ad, 0x77ed, 0x7aef, 0x7baa, 0x7dbb, 0x803d,
	0x80c6, 0x86cb, 0x8a95, 0x935b, 0x56e3, 0x58c7, 0x5f3e, 0x65ad,
	0x6696, 0x6a80, 0x6bb5, 0x7537, 0x8ac7, 0x5024, 0x77e5, 0x5730,
	0x5f1b, 0x6065, 0x667a, 0x6c60, 0x75f4, 0x7a1a, 0x7f6e, 0x81f4,
	0x8718, 0x9045, 0x99b3, 0x7bc9, 0x755c, 0x7af9, 0x7b51, 0x84c4,
	0x9010, 0x79e9, 0x7a92, 0x8336, 0x5ae1, 0x7740, 0x4e2d, 0x4ef2,
	0x5b99, 0x5fe0, 0x62bd, 0x663c, 0x67f1, 0x6ce8, 0x866b, 0x8877,
	0x8a3b, 0x914e, 0x92f3, 0x99d0, 0x6a17, 0x7026, 0x732a, 0x82e7,
	0x8457, 0x8caf, 0x4e01, 0x5146, 0x51cb, 0x558b, 0x5bf5,
	/* 0x4421 - 0x447e */
		0x5e16, 0x5e33, 0x5e81, 0x5f14, 0x5f35, 0x5f6b, 0x5fb4,
	0x61f2, 0x6311, 0x66a2, 0x671d, 0x6f6e, 0x7252, 0x753a, 0x773a,
	0x8074, 0x8139, 0x8178, 0x8776, 0x8abf, 0x8adc, 0x8d85, 0x8df3,
	0x929a, 0x9577, 0x9802, 0x9ce5, 0x52c5, 0x6357, 0x76f4, 0x6715,
	0x6c88, 0x73cd, 0x8cc3, 0x93ae, 0x9673, 0x6d25, 0x589c, 0x690e,
	0x69cc, 0x8ffd, 0x939a, 0x75db, 0x901a, 0x585a, 0x6802, 0x63b4,
	0x69fb, 0x4f43, 0x6f2c, 0x67d8, 0x8fbb, 0x8526, 0x7db4, 0x9354,
	0x693f, 0x6f70, 0x576a, 0x58f7, 0x5b2c, 0x7d2c, 0x722a, 0x540a,
	0x91e3, 0x9db4, 0x4ead, 0x4f4e, 0x505c, 0x5075, 0x5243, 0x8c9e,
	0x5448, 0x5824, 0x5b9a, 0x5e1d, 0x5e95, 0x5ead, 0x5ef7, 0x5f1f,
	0x608c, 0x62b5, 0x633a, 0x63d0, 0x68af, 0x6c40, 0x7887, 0x798e,
	0x7a0b, 0x7de0, 0x8247, 0x8a02, 0x8ae6, 0x8e44, 0x9013,
	/* 0x4521 - 0x457e */
		0x90b8, 0x912d, 0x91d8, 0x9f0e, 0x6ce5, 0x6458, 0x64e2,
	0x6575, 0x6ef4, 0x7684, 0x7b1b, 0x9069, 0x93d1, 0x6eba, 0x54f2,
	0x5fb9, 0x64a4, 0x8f4d, 0x8fed, 0x9244, 0x5178, 0x586b, 0x5929,
	0x5c55, 0x5e97, 0x6dfb, 0x7e8f, 0x751c, 0x8cbc, 0x8ee2, 0x985b,
	0x70b9, 0x4f1d, 0x6bbf, 0x6fb1, 0x7530, 0x96fb, 0x514e, 0x5410,
	0x5835, 0x5857, 0x59ac, 0x5c60, 0x5f92, 0x6597, 0x675c, 0x6e21,
	0x767b, 0x83df, 0x8ced, 0x9014, 0x90fd, 0x934d, 0x7825, 0x783a,
	0x52aa, 0x5ea6, 0x571f, 0x5974, 0x6012, 0x5012, 0x515a, 0x51ac,
	0x51cd, 0x5200, 0x5510, 0x5854, 0x5858, 0x5957, 0x5b95, 0x5cf6,
	0x5d8b, 0x60bc, 0x6295, 0x642d, 0x6771, 0x6843, 0x68bc, 0x68df,
	0x76d7, 0x6dd8, 0x6e6f, 0x6d9b, 0x706f, 0x71c8, 0x5f53, 0x75d8,
	0x7977, 0x7b49, 0x7b54, 0x7b52, 0x7cd6, 0x7d71, 0x5230,
	/* 0x4621 - 0x467e */
		0x8463, 0x8569, 0x85e4, 0x8a0e, 0x8b04, 0x8c46, 0x8e0f,
	0x9003, 0x900f, 0x9419, 0x9676, 0x982d, 0x9a30, 0x95d8, 0x50cd,
	0x52d5, 0x540c, 0x5802, 0x5c0e, 0x61a7, 0x649e, 0x6d1e, 0x77b3,
	0x7ae5, 0x80f4, 0x8404, 0x9053, 0x9285, 0x5ce0, 0x9d07, 0x533f,
	0x5f97, 0x5fb3, 0x6d9c, 0x7279, 0x7763, 0x79bf, 0x7be4, 0x6bd2,
	0x72ec, 0x8aad, 0x6803, 0x6a61, 0x51f8, 0x7a81, 0x6934, 0x5c4a,
	0x9cf6, 0x82eb, 0x5bc5, 0x9149, 0x701e, 0x5678, 0x5c6f, 0x60c7,
	0x6566, 0x6c8c, 0x8c5a, 0x9041, 0x9813, 0x5451, 0x66c7, 0x920d,
	0x5948, 0x90a3, 0x5185, 0x4e4d, 0x51ea, 0x8599, 0x8b0e, 0x7058,
	0x637a, 0x934b, 0x6962, 0x99b4, 0x7e04, 0x7577, 0x5357, 0x6960,
	0x8edf, 0x96e3, 0x6c5d, 0x4e8c, 0x5c3c, 0x5f10, 0x8fe9, 0x5302,
	0x8cd1, 0x8089, 0x8679, 0x5eff, 0x65e5, 0x4e73, 0x5165,
	/* 0x4721 - 0x477e */
		0x5982, 0x5c3f, 0x97ee, 0x4efb, 0x598a, 0x5fcd, 0x8a8d,
	0x6fe1, 0x79b0, 0x7962, 0x5be7, 0x8471, 0x732b, 0x71b1, 0x5e74,
	0x5ff5, 0x637b, 0x649a, 0x71c3, 0x7c98, 0x4e43, 0x5efc, 0x4e4b,
	0x57dc, 0x56a2, 0x60a9, 0x6fc3, 0x7d0d, 0x80fd, 0x8133, 0x81bf,
	0x8fb2, 0x8997, 0x86a4, 0x5df4, 0x628a, 0x64ad, 0x8987, 0x6777,
	0x6ce2, 0x6d3e, 0x7436, 0x7834, 0x5a46, 0x7f75, 0x82ad, 0x99ac,
	0x4ff3, 0x5ec3, 0x62dd, 0x6392, 0x6557, 0x676f, 0x76c3, 0x724c,
	0x80cc, 0x80ba, 0x8f29, 0x914d, 0x500d, 0x57f9, 0x5a92, 0x6885,
	0x6973, 0x7164, 0x72fd, 0x8cb7, 0x58f2, 0x8ce0, 0x966a, 0x9019,
	0x877f, 0x79e4, 0x77e7, 0x8429, 0x4f2f, 0x5265, 0x535a, 0x62cd,
	0x67cf, 0x6cca, 0x767d, 0x7b94, 0x7c95, 0x8236, 0x8584, 0x8feb,
	0x66dd, 0x6f20, 0x7206, 0x7e1b, 0x83ab, 0x99c1, 0x9ea6,
	/* 0x4821 - 0x487e */
		0x51fd, 0x7bb1, 0x7872, 0x7bb8, 0x8087, 0x7b48, 0x6ae8,
	0x5e61, 0x808c, 0x7551, 0x7560, 0x516b, 0x9262, 0x6e8c, 0x767a,
	0x9197, 0x9aea, 0x4f10, 0x7f70, 0x629c, 0x7b4f, 0x95a5, 0x9ce9,
	0x567a, 0x5859, 0x86e4, 0x96bc, 0x4f34, 0x5224, 0x534a, 0x53cd,
	0x53db, 0x5e06, 0x642c, 0x6591, 0x677f, 0x6c3e, 0x6c4e, 0x7248,
	0x72af, 0x73ed, 0x7554, 0x7e41, 0x822c, 0x85e9, 0x8ca9, 0x7bc4,
	0x91c6, 0x7169, 0x9812, 0x98ef, 0x633d, 0x6669, 0x756a, 0x76e4,
	0x78d0, 0x8543, 0x86ee, 0x532a, 0x5351, 0x5426, 0x5983, 0x5e87,
	0x5f7c, 0x60b2, 0x6249, 0x6279, 0x62ab, 0x6590, 0x6bd4, 0x6ccc,
	0x75b2, 0x76ae, 0x7891, 0x79d8, 0x7dcb, 0x7f77, 0x80a5, 0x88ab,
	0x8ab9, 0x8cbb, 0x907f, 0x975e, 0x98db, 0x6a0b, 0x7c38, 0x5099,
	0x5c3e, 0x5fae, 0x6787, 0x6bd8, 0x7435, 0x7709, 0x7f8e,
	/* 0x4921 - 0x497e */
		0x9f3b, 0x67ca, 0x7a17, 0x5339, 0x758b, 0x9aed, 0x5f66,
	0x819d, 0x83f1, 0x8098, 0x5f3c, 0x5fc5, 0x7562, 0x7b46, 0x903c,
	0x6867, 0x59eb, 0x5a9b, 0x7d10, 0x767e, 0x8b2c, 0x4ff5, 0x5f6a,
	0x6a19, 0x6c37, 0x6f02, 0x74e2, 0x7968, 0x8868, 0x8a55, 0x8c79,
	0x5edf, 0x63cf, 0x75c5, 0x79d2, 0x82d7, 0x9328, 0x92f2, 0x849c,
	0x86ed, 0x9c2d, 0x54c1, 0x5f6c, 0x658c, 0x6d5c, 0x7015, 0x8ca7,
	0x8cd3, 0x983b, 0x654f, 0x74f6, 0x4e0d, 0x4ed8, 0x57e0, 0x592b,
	0x5a66, 0x5bcc, 0x51a8, 0x5e03, 0x5e9c, 0x6016, 0x6276, 0x6577,
	0x65a7, 0x666e, 0x6d6e, 0x7236, 0x7b26, 0x8150, 0x819a, 0x8299,
	0x8b5c, 0x8ca0, 0x8ce6, 0x8d74, 0x961c, 0x9644, 0x4fae, 0x64ab,
	0x6b66, 0x821e, 0x8461, 0x856a, 0x90e8, 0x5c01, 0x6953, 0x98a8,
	0x847a, 0x8557, 0x4f0f, 0x526f, 0x5fa9, 0x5e45, 0x670d,
	/* 0x4a21 - 0x4a7e */
		0x798f, 0x8179, 0x8907, 0x8986, 0x6df5, 0x5f17, 0x6255,
	0x6cb8, 0x4ecf, 0x7269, 0x9b92, 0x5206, 0x543b, 0x5674, 0x58b3,
	0x61a4, 0x626e, 0x711a, 0x596e, 0x7c89, 0x7cde, 0x7d1b, 0x96f0,
	0x6587, 0x805e, 0x4e19, 0x4f75, 0x5175, 0x5840, 0x5e63, 0x5e73,
	0x5f0a, 0x67c4, 0x4e26, 0x853d, 0x9589, 0x965b, 0x7c73, 0x9801,
	0x50fb, 0x58c1, 0x7656, 0x78a7, 0x5225, 0x77a5, 0x8511, 0x7b86,
	0x504f, 0x5909, 0x7247, 0x7bc7, 0x7de8, 0x8fba, 0x8fd4, 0x904d,
	0x4fbf, 0x52c9, 0x5a29, 0x5f01, 0x97ad, 0x4fdd, 0x8217, 0x92ea,
	0x5703, 0x6355, 0x6b69, 0x752b, 0x88dc, 0x8f14, 0x7a42, 0x52df,
	0x5893, 0x6155, 0x620a, 0x66ae, 0x6bcd, 0x7c3f, 0x83e9, 0x5023,
	0x4ff8, 0x5305, 0x5446, 0x5831, 0x5949, 0x5b9d, 0x5cf0, 0x5cef,
	0x5d29, 0x5e96, 0x62b1, 0x6367, 0x653e, 0x65b9, 0x670b,
	/* 0x4b21 - 0x4b7e */
		0x6cd5, 0x6ce1, 0x70f9, 0x7832, 0x7e2b, 0x80de, 0x82b3,
	0x840c, 0x84ec, 0x8702, 0x8912, 0x8a2a, 0x8c4a, 0x90a6, 0x92d2,
	0x98fd, 0x9cf3, 0x9d6c, 0x4e4f, 0x4ea1, 0x508d, 0x5256, 0x574a,
	0x59a8, 0x5e3d, 0x5fd8, 0x5fd9, 0x623f, 0x66b4, 0x671b, 0x67d0,
	0x68d2, 0x5192, 0x7d21, 0x80aa, 0x81a8, 0x8b00, 0x8c8c, 0x8cbf,
	0x927e, 0x9632, 0x5420, 0x982c, 0x5317, 0x50d5, 0x535c, 0x58a8,
	0x64b2, 0x6734, 0x7267, 0x7766, 0x7a46, 0x91e6, 0x52c3, 0x6ca1,
	0x6b86, 0x5800, 0x5e4c, 0x5954, 0x672c, 0x7ffb, 0x51e1, 0x76c6,
	0x6469, 0x78e8, 0x9b54, 0x9ebb, 0x57cb, 0x59b9, 0x6627, 0x679a,
	0x6bce, 0x54e9, 0x69d9, 0x5e55, 0x819c, 0x6795, 0x9baa, 0x67fe,
	0x9c52, 0x685d, 0x4ea6, 0x4fe3, 0x53c8, 0x62b9, 0x672b, 0x6cab,
	0x8fc4, 0x4fad, 0x7e6d, 0x9ebf, 0x4e07, 0x6162, 0x6e80,
	/* 0x4c21 - 0x4c7e */
		0x6f2b, 0x8513, 0x5473, 0x672a, 0x9b45, 0x5df3, 0x7b95,
	0x5cac, 0x5bc6, 0x871c, 0x6e4a, 0x84d1, 0x7a14, 0x8108, 0x5999,
	0x7c8d, 0x6c11, 0x7720, 0x52d9, 0x5922, 0x7121, 0x725f, 0x77db,
	0x9727, 0x9d61, 0x690b, 0x5a7f, 0x5a18, 0x51a5, 0x540d, 0x547d,
	0x660e, 0x76df, 0x8ff7, 0x9298, 0x9cf4, 0x59ea, 0x725d, 0x6ec5,
	0x514d, 0x68c9, 0x7dbf, 0x7dec, 0x9762, 0x9eba, 0x6478, 0x6a21,
	0x8302, 0x5984, 0x5b5f, 0x6bdb, 0x731b, 0x76f2, 0x7db2, 0x8017,
	0x8499, 0x5132, 0x6728, 0x9ed9, 0x76ee, 0x6762, 0x52ff, 0x9905,
	0x5c24, 0x623b, 0x7c7e, 0x8cb0, 0x554f, 0x60b6, 0x7d0b, 0x9580,
	0x5301, 0x4e5f, 0x51b6, 0x591c, 0x723a, 0x8036, 0x91ce, 0x5f25,
	0x77e2, 0x5384, 0x5f79, 0x7d04, 0x85ac, 0x8a33, 0x8e8d, 0x9756,
	0x67f3, 0x85ae, 0x9453, 0x6109, 0x6108, 0x6cb9, 0x7652,
	/* 0x4d21 - 0x4d7e */
		0x8aed, 0x8f38, 0x552f, 0x4f51, 0x512a, 0x52c7, 0x53cb,
	0x5ba5, 0x5e7d, 0x60a0, 0x6182, 0x63d6, 0x6709, 0x67da, 0x6e67,
	0x6d8c, 0x7336, 0x7337, 0x7531, 0x7950, 0x88d5, 0x8a98, 0x904a,
	0x9091, 0x90f5, 0x96c4, 0x878d, 0x5915, 0x4e88, 0x4f59, 0x4e0e,
	0x8a89, 0x8f3f, 0x9810, 0x50ad, 0x5e7c, 0x5996, 0x5bb9, 0x5eb8,
	0x63da, 0x63fa, 0x64c1, 0x66dc, 0x694a, 0x69d8, 0x6d0b, 0x6eb6,
	0x7194, 0x7528, 0x7aaf, 0x7f8a, 0x8000, 0x8449, 0x84c9, 0x8981,
	0x8b21, 0x8e0a, 0x9065, 0x967d, 0x990a, 0x617e, 0x6291, 0x6b32,
	0x6c83, 0x6d74, 0x7fcc, 0x7ffc, 0x6dc0, 0x7f85, 0x87ba, 0x88f8,
	0x6765, 0x83b1, 0x983c, 0x96f7, 0x6d1b, 0x7d61, 0x843d, 0x916a,
	0x4e71, 0x5375, 0x5d50, 0x6b04, 0x6feb, 0x85cd, 0x862d, 0x89a7,
	0x5229, 0x540f, 0x5c65, 0x674e, 0x68a8, 0x7406, 0x7483,
	/* 0x4e21 - 0x4e7e */
		0x75e2, 0x88cf, 0x88e1, 0x91cc, 0x96e2, 0x9678, 0x5f8b,
	0x7387, 0x7acb, 0x844e, 0x63a0, 0x7565, 0x5289, 0x6d41, 0x6e9c,
	0x7409, 0x7559, 0x786b, 0x7c92, 0x9686, 0x7adc, 0x9f8d, 0x4fb6,
	0x616e, 0x65c5, 0x865c, 0x4e86, 0x4eae, 0x50da, 0x4e21, 0x51cc,
	0x5bee, 0x6599, 0x6881, 0x6dbc, 0x731f, 0x7642, 0x77ad, 0x7a1c,
	0x7ce7, 0x826f, 0x8ad2, 0x907c, 0x91cf, 0x9675, 0x9818, 0x529b,
	0x7dd1, 0x502b, 0x5398, 0x6797, 0x6dcb, 0x71d0, 0x7433, 0x81e8,
	0x8f2a, 0x96a3, 0x9c57, 0x9e9f, 0x7460, 0x5841, 0x6d99, 0x7d2f,
	0x985e, 0x4ee4, 0x4f36, 0x4f8b, 0x51b7, 0x52b1, 0x5dba, 0x601c,
	0x73b2, 0x793c, 0x82d3, 0x9234, 0x96b7, 0x96f6, 0x970a, 0x9e97,
	0x9f62, 0x66a6, 0x6b74, 0x5217, 0x52a3, 0x70c8, 0x88c2, 0x5ec9,
	0x604b, 0x6190, 0x6f23, 0x7149, 0x7c3e, 0x7df4, 0x806f,
	/* 0x4f21 - 0x4f7e */
		0x84ee, 0x9023, 0x932c, 0x5442, 0x9b6f, 0x6ad3, 0x7089,
	0x8cc2, 0x8def, 0x9732, 0x52b4, 0x5a41, 0x5eca, 0x5f04, 0x6717,
	0x697c, 0x6994, 0x6d6a, 0x6f0f, 0x7262, 0x72fc, 0x7bed, 0x8001,
	0x807e, 0x874b, 0x90ce, 0x516d, 0x9e93, 0x7984, 0x808b, 0x9332,
	0x8ad6, 0x502d, 0x548c, 0x8a71, 0x6b6a, 0x8cc4, 0x8107, 0x60d1,
	0x67a0, 0x9df2, 0x4e99, 0x4e98, 0x9c10, 0x8a6b, 0x85c1, 0x8568,
	0x6900, 0x6e7e, 0x7897, 0x8155, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x5021 - 0x507e */
		0x5f0c, 0x4e10, 0x4e15, 0x4e2a, 0x4e31, 0x4e36, 0x4e3c,
	0x4e3f, 0x4e42, 0x4e56, 0x4e58, 0x4e82, 0x4e85, 0x8c6b, 0x4e8a,
	0x8212, 0x5f0d, 0x4e8e, 0x4e9e, 0x4e9f, 0x4ea0, 0x4ea2, 0x4eb0,
	0x4eb3, 0x4eb6, 0x4ece, 0x4ecd, 0x4ec4, 0x4ec6, 0x4ec2, 0x4ed7,
	0x4ede, 0x4eed, 0x4edf, 0x4ef7, 0x4f09, 0x4f5a, 0x4f30, 0x4f5b,
	0x4f5d, 0x4f57, 0x4f47, 0x4f76, 0x4f88, 0x4f8f, 0x4f98, 0x4f7b,
	0x4f69, 0x4f70, 0x4f91, 0x4f6f, 0x4f86, 0x4f96, 0x5118, 0x4fd4,
	0x4fdf, 0x4fce, 0x4fd8, 0x4fdb, 0x4fd1, 0x4fda, 0x4fd0, 0x4fe4,
	0x4fe5, 0x501a, 0x5028, 0x5014, 0x502a, 0x5025, 0x5005, 0x4f1c,
	0x4ff6, 0x5021, 0x5029, 0x502c, 0x4ffe, 0x4fef, 0x5011, 0x5006,
	0x5043, 0x5047, 0x6703, 0x5055, 0x5050, 0x5048, 0x505a, 0x5056,
	0x506c, 0x5078, 0x5080, 0x509a, 0x5085, 0x50b4, 0x50b2,
	/* 0x5121 - 0x517e */
		0x50c9, 0x50ca, 0x50b3, 0x50c2, 0x50d6, 0x50de, 0x50e5,
	0x50ed, 0x50e3, 0x50ee, 0x50f9, 0x50f5, 0x5109, 0x5101, 0x5102,
	0x5116, 0x5115, 0x5114, 0x511a, 0x5121, 0x513a, 0x5137, 0x513c,
	0x513b, 0x513f, 0x5140, 0x5152, 0x514c, 0x5154, 0x5162, 0x7af8,
	0x5169, 0x516a, 0x516e, 0x5180, 0x5182, 0x56d8, 0x518c, 0x5189,
	0x518f, 0x5191, 0x5193, 0x5195, 0x5196, 0x51a4, 0x51a6, 0x51a2,
	0x51a9, 0x51aa, 0x51ab, 0x51b3, 0x51b1, 0x51b2, 0x51b0, 0x51b5,
	0x51bd, 0x51c5, 0x51c9, 0x51db, 0x51e0, 0x8655, 0x51e9, 0x51ed,
	0x51f0, 0x51f5, 0x51fe, 0x5204, 0x520b, 0x5214, 0x520e, 0x5227,
	0x522a, 0x522e, 0x5233, 0x5239, 0x524f, 0x5244, 0x524b, 0x524c,
	0x525e, 0x5254, 0x526a, 0x5274, 0x5269, 0x5273, 0x527f, 0x527d,
	0x528d, 0x5294, 0x5292, 0x5271, 0x5288, 0x5291, 0x8fa8,
	/* 0x5221 - 0x527e */
		0x8fa7, 0x52ac, 0x52ad, 0x52bc, 0x52b5, 0x52c1, 0x52cd,
	0x52d7, 0x52de, 0x52e3, 0x52e6, 0x98ed, 0x52e0, 0x52f3, 0x52f5,
	0x52f8, 0x52f9, 0x5306, 0x5308, 0x7538, 0x530d, 0x5310, 0x530f,
	0x5315, 0x531a, 0x5323, 0x532f, 0x5331, 0x5333, 0x5338, 0x5340,
	0x5346, 0x5345, 0x4e17, 0x5349, 0x534d, 0x51d6, 0x535e, 0x5369,
	0x536e, 0x5918, 0x537b, 0x5377, 0x5382, 0x5396, 0x53a0, 0x53a6,
	0x53a5, 0x53ae, 0x53b0, 0x53b6, 0x53c3, 0x7c12, 0x96d9, 0x53df,
	0x66fc, 0x71ee, 0x53ee, 0x53e8, 0x53ed, 0x53fa, 0x5401, 0x543d,
	0x5440, 0x542c, 0x542d, 0x543c, 0x542e, 0x5436, 0x5429, 0x541d,
	0x544e, 0x548f, 0x5475, 0x548e, 0x545f, 0x5471, 0x5477, 0x5470,
	0x5492, 0x547b, 0x5480, 0x5476, 0x5484, 0x5490, 0x5486, 0x54c7,
	0x54a2, 0x54b8, 0x54a5, 0x54ac, 0x54c4, 0x54c8, 0x54a8,
	/* 0x5321 - 0x537e */
		0x54ab, 0x54c2, 0x54a4, 0x54be, 0x54bc, 0x54d8, 0x54e5,
	0x54e6, 0x550f, 0x5514, 0x54fd, 0x54ee, 0x54ed, 0x54fa, 0x54e2,
	0x5539, 0x5540, 0x5563, 0x554c, 0x552e, 0x555c, 0x5545, 0x5556,
	0x5557, 0x5538, 0x5533, 0x555d, 0x5599, 0x5580, 0x54af, 0x558a,
	0x559f, 0x557b, 0x557e, 0x5598, 0x559e, 0x55ae, 0x557c, 0x5583,
	0x55a9, 0x5587, 0x55a8, 0x55da, 0x55c5, 0x55df, 0x55c4, 0x55dc,
	0x55e4, 0x55d4, 0x5614, 0x55f7, 0x5616, 0x55fe, 0x55fd, 0x561b,
	0x55f9, 0x564e, 0x5650, 0x71df, 0x5634, 0x5636, 0x5632, 0x5638,
	0x566b, 0x5664, 0x562f, 0x566c, 0x566a, 0x5686, 0x5680, 0x568a,
	0x56a0, 0x5694, 0x568f, 0x56a5, 0x56ae, 0x56b6, 0x56b4, 0x56c2,
	0x56bc, 0x56c1, 0x56c3, 0x56c0, 0x56c8, 0x56ce, 0x56d1, 0x56d3,
	0x56d7, 0x56ee, 0x56f9, 0x5700, 0x56ff, 0x5704, 0x5709,
	/* 0x5421 - 0x547e */
		0x5708, 0x570b, 0x570d, 0x5713, 0x5718, 0x5716, 0x55c7,
	0x571c, 0x5726, 0x5737, 0x5738, 0x574e, 0x573b, 0x5740, 0x574f,
	0x5769, 0x57c0, 0x5788, 0x5761, 0x577f, 0x5789, 0x5793, 0x57a0,
	0x57b3, 0x57a4, 0x57aa, 0x57b0, 0x57c3, 0x57c6, 0x57d4, 0x57d2,
	0x57d3, 0x580a, 0x57d6, 0x57e3, 0x580b, 0x5819, 0x581d, 0x5872,
	0x5821, 0x5862, 0x584b, 0x5870, 0x6bc0, 0x5852, 0x583d, 0x5879,
	0x5885, 0x58b9, 0x589f, 0x58ab, 0x58ba, 0x58de, 0x58bb, 0x58b8,
	0x58ae, 0x58c5, 0x58d3, 0x58d1, 0x58d7, 0x58d9, 0x58d8, 0x58e5,
	0x58dc, 0x58e4, 0x58df, 0x58ef, 0x58fa, 0x58f9, 0x58fb, 0x58fc,
	0x58fd, 0x5902, 0x590a, 0x5910, 0x591b, 0x68a6, 0x5925, 0x592c,
	0x592d, 0x5932, 0x5938, 0x593e, 0x7ad2, 0x5955, 0x5950, 0x594e,
	0x595a, 0x5958, 0x5962, 0x5960, 0x5967, 0x596c, 0x5969,
	/* 0x5521 - 0x557e */
		0x5978, 0x5981, 0x599d, 0x4f5e, 0x4fab, 0x59a3, 0x59b2,
	0x59c6, 0x59e8, 0x59dc, 0x598d, 0x59d9, 0x59da, 0x5a25, 0x5a1f,
	0x5a11, 0x5a1c, 0x5a09, 0x5a1a, 0x5a40, 0x5a6c, 0x5a49, 0x5a35,
	0x5a36, 0x5a62, 0x5a6a, 0x5a9a, 0x5abc, 0x5abe, 0x5acb, 0x5ac2,
	0x5abd, 0x5ae3, 0x5ad7, 0x5ae6, 0x5ae9, 0x5ad6, 0x5afa, 0x5afb,
	0x5b0c, 0x5b0b, 0x5b16, 0x5b32, 0x5ad0, 0x5b2a, 0x5b36, 0x5b3e,
	0x5b43, 0x5b45, 0x5b40, 0x5b51, 0x5b55, 0x5b5a, 0x5b5b, 0x5b65,
	0x5b69, 0x5b70, 0x5b73, 0x5b75, 0x5b78, 0x6588, 0x5b7a, 0x5b80,
	0x5b83, 0x5ba6, 0x5bb8, 0x5bc3, 0x5bc7, 0x5bc9, 0x5bd4, 0x5bd0,
	0x5be4, 0x5be6, 0x5be2, 0x5bde, 0x5be5, 0x5beb, 0x5bf0, 0x5bf6,
	0x5bf3, 0x5c05, 0x5c07, 0x5c08, 0x5c0d, 0x5c13, 0x5c20, 0x5c22,
	0x5c28, 0x5c38, 0x5c39, 0x5c41, 0x5c46, 0x5c4e, 0x5c53,
	/* 0x5621 - 0x567e */
		0x5c50, 0x5c4f, 0x5b71, 0x5c6c, 0x5c6e, 0x4e62, 0x5c76,
	0x5c79, 0x5c8c, 0x5c91, 0x5c94, 0x599b, 0x5cab, 0x5cbb, 0x5cb6,
	0x5cbc, 0x5cb7, 0x5cc5, 0x5cbe, 0x5cc7, 0x5cd9, 0x5ce9, 0x5cfd,
	0x5cfa, 0x5ced, 0x5d8c, 0x5cea, 0x5d0b, 0x5d15, 0x5d17, 0x5d5c,
	0x5d1f, 0x5d1b, 0x5d11, 0x5d14, 0x5d22, 0x5d1a, 0x5d19, 0x5d18,
	0x5d4c, 0x5d52, 0x5d4e, 0x5d4b, 0x5d6c, 0x5d73, 0x5d76, 0x5d87,
	0x5d84, 0x5d82, 0x5da2, 0x5d9d, 0x5dac, 0x5dae, 0x5dbd, 0x5d90,
	0x5db7, 0x5dbc, 0x5dc9, 0x5dcd, 0x5dd3, 0x5dd2, 0x5dd6, 0x5ddb,
	0x5deb, 0x5df2, 0x5df5, 0x5e0b, 0x5e1a, 0x5e19, 0x5e11, 0x5e1b,
	0x5e36, 0x5e37, 0x5e44, 0x5e43, 0x5e40, 0x5e4e, 0x5e57, 0x5e54,
	0x5e5f, 0x5e62, 0x5e64, 0x5e47, 0x5e75, 0x5e76, 0x5e7a, 0x9ebc,
	0x5e7f, 0x5ea0, 0x5ec1, 0x5ec2, 0x5ec8, 0x5ed0, 0x5ecf,
	/* 0x5721 - 0x577e */
		0x5ed6, 0x5ee3, 0x5edd, 0x5eda, 0x5edb, 0x5ee2, 0x5ee1,
	0x5ee8, 0x5ee9, 0x5eec, 0x5ef1, 0x5ef3, 0x5ef0, 0x5ef4, 0x5ef8,
	0x5efe, 0x5f03, 0x5f09, 0x5f5d, 0x5f5c, 0x5f0b, 0x5f11, 0x5f16,
	0x5f29, 0x5f2d, 0x5f38, 0x5f41, 0x5f48, 0x5f4c, 0x5f4e, 0x5f2f,
	0x5f51, 0x5f56, 0x5f57, 0x5f59, 0x5f61, 0x5f6d, 0x5f73, 0x5f77,
	0x5f83, 0x5f82, 0x5f7f, 0x5f8a, 0x5f88, 0x5f91, 0x5f87, 0x5f9e,
	0x5f99, 0x5f98, 0x5fa0, 0x5fa8, 0x5fad, 0x5fbc, 0x5fd6, 0x5ffb,
	0x5fe4, 0x5ff8, 0x5ff1, 0x5fdd, 0x60b3, 0x5fff, 0x6021, 0x6060,
	0x6019, 0x6010, 0x6029, 0x600e, 0x6031, 0x601b, 0x6015, 0x602b,
	0x6026, 0x600f, 0x603a, 0x605a, 0x6041, 0x606a, 0x6077, 0x605f,
	0x604a, 0x6046, 0x604d, 0x6063, 0x6043, 0x6064, 0x6042, 0x606c,
	0x606b, 0x6059, 0x6081, 0x608d, 0x60e7, 0x6083, 0x609a,
	/* 0x5821 - 0x587e */
		0x6084, 0x609b, 0x6096, 0x6097, 0x6092, 0x60a7, 0x608b,
	0x60e1, 0x60b8, 0x60e0, 0x60d3, 0x60b4, 0x5ff0, 0x60bd, 0x60c6,
	0x60b5, 0x60d8, 0x614d, 0x6115, 0x6106, 0x60f6, 0x60f7, 0x6100,
	0x60f4, 0x60fa, 0x6103, 0x6121, 0x60fb, 0x60f1, 0x610d, 0x610e,
	0x6147, 0x613e, 0x6128, 0x6127, 0x614a, 0x613f, 0x613c, 0x612c,
	0x6134, 0x613d, 0x6142, 0x6144, 0x6173, 0x6177, 0x6158, 0x6159,
	0x615a, 0x616b, 0x6174, 0x616f, 0x6165, 0x6171, 0x615f, 0x615d,
	0x6153, 0x6175, 0x6199, 0x6196, 0x6187, 0x61ac, 0x6194, 0x619a,
	0x618a, 0x6191, 0x61ab, 0x61ae, 0x61cc, 0x61ca, 0x61c9, 0x61f7,
	0x61c8, 0x61c3, 0x61c6, 0x61ba, 0x61cb, 0x7f79, 0x61cd, 0x61e6,
	0x61e3, 0x61f6, 0x61fa, 0x61f4, 0x61ff, 0x61fd, 0x61fc, 0x61fe,
	0x6200, 0x6208, 0x6209, 0x620d, 0x620c, 0x6214, 0x621b,
	/* 0x5921 - 0x597e */
		0x621e, 0x6221, 0x622a, 0x622e, 0x6230, 0x6232, 0x6233,
	0x6241, 0x624e, 0x625e, 0x6263, 0x625b, 0x6260, 0x6268, 0x627c,
	0x6282, 0x6289, 0x627e, 0x6292, 0x6293, 0x6296, 0x62d4, 0x6283,
	0x6294, 0x62d7, 0x62d1, 0x62bb, 0x62cf, 0x62ff, 0x62c6, 0x64d4,
	0x62c8, 0x62dc, 0x62cc, 0x62ca, 0x62c2, 0x62c7, 0x629b, 0x62c9,
	0x630c, 0x62ee, 0x62f1, 0x6327, 0x6302, 0x6308, 0x62ef, 0x62f5,
	0x6350, 0x633e, 0x634d, 0x641c, 0x634f, 0x6396, 0x638e, 0x6380,
	0x63ab, 0x6376, 0x63a3, 0x638f, 0x6389, 0x639f, 0x63b5, 0x636b,
	0x6369, 0x63be, 0x63e9, 0x63c0, 0x63c6, 0x63e3, 0x63c9, 0x63d2,
	0x63f6, 0x63c4, 0x6416, 0x6434, 0x6406, 0x6413, 0x6426, 0x6436,
	0x651d, 0x6417, 0x6428, 0x640f, 0x6467, 0x646f, 0x6476, 0x644e,
	0x652a, 0x6495, 0x6493, 0x64a5, 0x64a9, 0x6488, 0x64bc,
	/* 0x5a21 - 0x5a7e */
		0x64da, 0x64d2, 0x64c5, 0x64c7, 0x64bb, 0x64d8, 0x64c2,
	0x64f1, 0x64e7, 0x8209, 0x64e0, 0x64e1, 0x62ac, 0x64e3, 0x64ef,
	0x652c, 0x64f6, 0x64f4, 0x64f2, 0x64fa, 0x6500, 0x64fd, 0x6518,
	0x651c, 0x6505, 0x6524, 0x6523, 0x652b, 0x6534, 0x6535, 0x6537,
	0x6536, 0x6538, 0x754b, 0x6548, 0x6556, 0x6555, 0x654d, 0x6558,
	0x655e, 0x655d, 0x6572, 0x6578, 0x6582, 0x6583, 0x8b8a, 0x659b,
	0x659f, 0x65ab, 0x65b7, 0x65c3, 0x65c6, 0x65c1, 0x65c4, 0x65cc,
	0x65d2, 0x65db, 0x65d9, 0x65e0, 0x65e1, 0x65f1, 0x6772, 0x660a,
	0x6603, 0x65fb, 0x6773, 0x6635, 0x6636, 0x6634, 0x661c, 0x664f,
	0x6644, 0x6649, 0x6641, 0x665e, 0x665d, 0x6664, 0x6667, 0x6668,
	0x665f, 0x6662, 0x6670, 0x6683, 0x6688, 0x668e, 0x6689, 0x6684,
	0x6698, 0x669d, 0x66c1, 0x66b9, 0x66c9, 0x66be, 0x66bc,
	/* 0x5b21 - 0x5b7e */
		0x66c4, 0x66b8, 0x66d6, 0x66da, 0x66e0, 0x663f, 0x66e6,
	0x66e9, 0x66f0, 0x66f5, 0x66f7, 0x670f, 0x6716, 0x671e, 0x6726,
	0x6727, 0x9738, 0x672e, 0x673f, 0x6736, 0x6741, 0x6738, 0x6737,
	0x6746, 0x675e, 0x6760, 0x6759, 0x6763, 0x6764, 0x6789, 0x6770,
	0x67a9, 0x677c, 0x676a, 0x678c, 0x678b, 0x67a6, 0x67a1, 0x6785,
	0x67b7, 0x67ef, 0x67b4, 0x67ec, 0x67b3, 0x67e9, 0x67b8, 0x67e4,
	0x67de, 0x67dd, 0x67e2, 0x67ee, 0x67b9, 0x67ce, 0x67c6, 0x67e7,
	0x6a9c, 0x681e, 0x6846, 0x6829, 0x6840, 0x684d, 0x6832, 0x684e,
	0x68b3, 0x682b, 0x6859, 0x6863, 0x6877, 0x687f, 0x689f, 0x688f,
	0x68ad, 0x6894, 0x689d, 0x689b, 0x6883, 0x6aae, 0x68b9, 0x6874,
	0x68b5, 0x68a0, 0x68ba, 0x690f, 0x688d, 0x687e, 0x6901, 0x68ca,
	0x6908, 0x68d8, 0x6922, 0x6926, 0x68e1, 0x690c, 0x68cd,
	/* 0x5c21 - 0x5c7e */
		0x68d4, 0x68e7, 0x68d5, 0x6936, 0x6912, 0x6904, 0x68d7,
	0x68e3, 0x6925, 0x68f9, 0x68e0, 0x68ef, 0x6928, 0x692a, 0x691a,
	0x6923, 0x6921, 0x68c6, 0x6979, 0x6977, 0x695c, 0x6978, 0x696b,
	0x6954, 0x697e, 0x696e, 0x6939, 0x6974, 0x693d, 0x6959, 0x6930,
	0x6961, 0x695e, 0x695d, 0x6981, 0x696a, 0x69b2, 0x69ae, 0x69d0,
	0x69bf, 0x69c1, 0x69d3, 0x69be, 0x69ce, 0x5be8, 0x69ca, 0x69dd,
	0x69bb, 0x69c3, 0x69a7, 0x6a2e, 0x6991, 0x69a0, 0x699c, 0x6995,
	0x69b4, 0x69de, 0x69e8, 0x6a02, 0x6a1b, 0x69ff, 0x6b0a, 0x69f9,
	0x69f2, 0x69e7, 0x6a05, 0x69b1, 0x6a1e, 0x69ed, 0x6a14, 0x69eb,
	0x6a0a, 0x6a12, 0x6ac1, 0x6a23, 0x6a13, 0x6a44, 0x6a0c, 0x6a72,
	0x6a36, 0x6a78, 0x6a47, 0x6a62, 0x6a59, 0x6a66, 0x6a48, 0x6a38,
	0x6a22, 0x6a90, 0x6a8d, 0x6aa0, 0x6a84, 0x6aa2, 0x6aa3,
	/* 0x5d21 - 0x5d7e */
		0x6a97, 0x8617, 0x6abb, 0x6ac3, 0x6ac2, 0x6ab8, 0x6ab3,
	0x6aac, 0x6ade, 0x6ad1, 0x6adf, 0x6aaa, 0x6ada, 0x6aea, 0x6afb,
	0x6b05, 0x8616, 0x6afa, 0x6b12, 0x6b16, 0x9b31, 0x6b1f, 0x6b38,
	0x6b37, 0x76dc, 0x6b39, 0x98ee, 0x6b47, 0x6b43, 0x6b49, 0x6b50,
	0x6b59, 0x6b54, 0x6b5b, 0x6b5f, 0x6b61, 0x6b78, 0x6b79, 0x6b7f,
	0x6b80, 0x6b84, 0x6b83, 0x6b8d, 0x6b98, 0x6b95, 0x6b9e, 0x6ba4,
	0x6baa, 0x6bab, 0x6baf, 0x6bb2, 0x6bb1, 0x6bb3, 0x6bb7, 0x6bbc,
	0x6bc6, 0x6bcb, 0x6bd3, 0x6bdf, 0x6bec, 0x6beb, 0x6bf3, 0x6bef,
	0x9ebe, 0x6c08, 0x6c13, 0x6c14, 0x6c1b, 0x6c24, 0x6c23, 0x6c5e,
	0x6c55, 0x6c62, 0x6c6a, 0x6c82, 0x6c8d, 0x6c9a, 0x6c81, 0x6c9b,
	0x6c7e, 0x6c68, 0x6c73, 0x6c92, 0x6c90, 0x6cc4, 0x6cf1, 0x6cd3,
	0x6cbd, 0x6cd7, 0x6cc5, 0x6cdd, 0x6cae, 0x6cb1, 0x6cbe,
	/* 0x5e21 - 0x5e7e */
		0x6cba, 0x6cdb, 0x6cef, 0x6cd9, 0x6cea, 0x6d1f, 0x884d,
	0x6d36, 0x6d2b, 0x6d3d, 0x6d38, 0x6d19, 0x6d35, 0x6d33, 0x6d12,
	0x6d0c, 0x6d63, 0x6d93, 0x6d64, 0x6d5a, 0x6d79, 0x6d59, 0x6d8e,
	0x6d95, 0x6fe4, 0x6d85, 0x6df9, 0x6e15, 0x6e0a, 0x6db5, 0x6dc7,
	0x6de6, 0x6db8, 0x6dc6, 0x6dec, 0x6dde, 0x6dcc, 0x6de8, 0x6dd2,
	0x6dc5, 0x6dfa, 0x6dd9, 0x6de4, 0x6dd5, 0x6dea, 0x6dee, 0x6e2d,
	0x6e6e, 0x6e2e, 0x6e19, 0x6e72, 0x6e5f, 0x6e3e, 0x6e23, 0x6e6b,
	0x6e2b, 0x6e76, 0x6e4d, 0x6e1f, 0x6e43, 0x6e3a, 0x6e4e, 0x6e24,
	0x6eff, 0x6e1d, 0x6e38, 0x6e82, 0x6eaa, 0x6e98, 0x6ec9, 0x6eb7,
	0x6ed3, 0x6ebd, 0x6eaf, 0x6ec4, 0x6eb2, 0x6ed4, 0x6ed5, 0x6e8f,
	0x6ea5, 0x6ec2, 0x6e9f, 0x6f41, 0x6f11, 0x704c, 0x6eec, 0x6ef8,
	0x6efe, 0x6f3f, 0x6ef2, 0x6f31, 0x6eef, 0x6f32, 0x6ecc,
	/* 0x5f21 - 0x5f7e */
		0x6f3e, 0x6f13, 0x6ef7, 0x6f86, 0x6f7a, 0x6f78, 0x6f81,
	0x6f80, 0x6f6f, 0x6f5b, 0x6ff3, 0x6f6d, 0x6f82, 0x6f7c, 0x6f58,
	0x6f8e, 0x6f91, 0x6fc2, 0x6f66, 0x6fb3, 0x6fa3, 0x6fa1, 0x6fa4,
	0x6fb9, 0x6fc6, 0x6faa, 0x6fdf, 0x6fd5, 0x6fec, 0x6fd4, 0x6fd8,
	0x6ff1, 0x6fee, 0x6fdb, 0x7009, 0x700b, 0x6ffa, 0x7011, 0x7001,
	0x700f, 0x6ffe, 0x701b, 0x701a, 0x6f74, 0x701d, 0x7018, 0x701f,
	0x7030, 0x703e, 0x7032, 0x7051, 0x7063, 0x7099, 0x7092, 0x70af,
	0x70f1, 0x70ac, 0x70b8, 0x70b3, 0x70ae, 0x70df, 0x70cb, 0x70dd,
	0x70d9, 0x7109, 0x70fd, 0x711c, 0x7119, 0x7165, 0x7155, 0x7188,
	0x7166, 0x7162, 0x714c, 0x7156, 0x716c, 0x718f, 0x71fb, 0x7184,
	0x7195, 0x71a8, 0x71ac, 0x71d7, 0x71b9, 0x71be, 0x71d2, 0x71c9,
	0x71d4, 0x71ce, 0x71e0, 0x71ec, 0x71e7, 0x71f5, 0x71fc,
	/* 0x6021 - 0x607e */
		0x71f9, 0x71ff, 0x720d, 0x7210, 0x721b, 0x7228, 0x722d,
	0x722c, 0x7230, 0x7232, 0x723b, 0x723c, 0x723f, 0x7240, 0x7246,
	0x724b, 0x7258, 0x7274, 0x727e, 0x7282, 0x7281, 0x7287, 0x7292,
	0x7296, 0x72a2, 0x72a7, 0x72b9, 0x72b2, 0x72c3, 0x72c6, 0x72c4,
	0x72ce, 0x72d2, 0x72e2, 0x72e0, 0x72e1, 0x72f9, 0x72f7, 0x500f,
	0x7317, 0x730a, 0x731c, 0x7316, 0x731d, 0x7334, 0x732f, 0x7329,
	0x7325, 0x733e, 0x734e, 0x734f, 0x9ed8, 0x7357, 0x736a, 0x7368,
	0x7370, 0x7378, 0x7375, 0x737b, 0x737a, 0x73c8, 0x73b3, 0x73ce,
	0x73bb, 0x73c0, 0x73e5, 0x73ee, 0x73de, 0x74a2, 0x7405, 0x746f,
	0x7425, 0x73f8, 0x7432, 0x743a, 0x7455, 0x743f, 0x745f, 0x7459,
	0x7441, 0x745c, 0x7469, 0x7470, 0x7463, 0x746a, 0x7476, 0x747e,
	0x748b, 0x749e, 0x74a7, 0x74ca, 0x74cf, 0x74d4, 0x73f1,
	/* 0x6121 - 0x617e */
		0x74e0, 0x74e3, 0x74e7, 0x74e9, 0x74ee, 0x74f2, 0x74f0,
	0x74f1, 0x74f8, 0x74f7, 0x7504, 0x7503, 0x7505, 0x750c, 0x750e,
	0x750d, 0x7515, 0x7513, 0x751e, 0x7526, 0x752c, 0x753c, 0x7544,
	0x754d, 0x754a, 0x7549, 0x755b, 0x7546, 0x755a, 0x7569, 0x7564,
	0x7567, 0x756b, 0x756d, 0x7578, 0x7576, 0x7586, 0x7587, 0x7574,
	0x758a, 0x7589, 0x7582, 0x7594, 0x759a, 0x759d, 0x75a5, 0x75a3,
	0x75c2, 0x75b3, 0x75c3, 0x75b5, 0x75bd, 0x75b8, 0x75bc, 0x75b1,
	0x75cd, 0x75ca, 0x75d2, 0x75d9, 0x75e3, 0x75de, 0x75fe, 0x75ff,
	0x75fc, 0x7601, 0x75f0, 0x75fa, 0x75f2, 0x75f3, 0x760b, 0x760d,
	0x7609, 0x761f, 0x7627, 0x7620, 0x7621, 0x7622, 0x7624, 0x7634,
	0x7630, 0x763b, 0x7647, 0x7648, 0x7646, 0x765c, 0x7658, 0x7661,
	0x7662, 0x7668, 0x7669, 0x766a, 0x7667, 0x766c, 0x7670,
	/* 0x6221 - 0x627e */
		0x7672, 0x7676, 0x7678, 0x767c, 0x7680, 0x7683, 0x7688,
	0x768b, 0x768e, 0x7696, 0x7693, 0x7699, 0x769a, 0x76b0, 0x76b4,
	0x76b8, 0x76b9, 0x76ba, 0x76c2, 0x76cd, 0x76d6, 0x76d2, 0x76de,
	0x76e1, 0x76e5, 0x76e7, 0x76ea, 0x862f, 0x76fb, 0x7708, 0x7707,
	0x7704, 0x7729, 0x7724, 0x771e, 0x7725, 0x7726, 0x771b, 0x7737,
	0x7738, 0x7747, 0x775a, 0x7768, 0x776b, 0x775b, 0x7765, 0x777f,
	0x777e, 0x7779, 0x778e, 0x778b, 0x7791, 0x77a0, 0x779e, 0x77b0,
	0x77b6, 0x77b9, 0x77bf, 0x77bc, 0x77bd, 0x77bb, 0x77c7, 0x77cd,
	0x77d7, 0x77da, 0x77dc, 0x77e3, 0x77ee, 0x77fc, 0x780c, 0x7812,
	0x7926, 0x7820, 0x792a, 0x7845, 0x788e, 0x7874, 0x7886, 0x787c,
	0x789a, 0x788c, 0x78a3, 0x78b5, 0x78aa, 0x78af, 0x78d1, 0x78c6,
	0x78cb, 0x78d4, 0x78be, 0x78bc, 0x78c5, 0x78ca, 0x78ec,
	/* 0x6321 - 0x637e */
		0x78e7, 0x78da, 0x78fd, 0x78f4, 0x7907, 0x7912, 0x7911,
	0x7919, 0x792c, 0x792b, 0x7940, 0x7960, 0x7957, 0x795f, 0x795a,
	0x7955, 0x7953, 0x797a, 0x797f, 0x798a, 0x799d, 0x79a7, 0x9f4b,
	0x79aa, 0x79ae, 0x79b3, 0x79b9, 0x79ba, 0x79c9, 0x79d5, 0x79e7,
	0x79ec, 0x79e1, 0x79e3, 0x7a08, 0x7a0d, 0x7a18, 0x7a19, 0x7a20,
	0x7a1f, 0x7980, 0x7a31, 0x7a3b, 0x7a3e, 0x7a37, 0x7a43, 0x7a57,
	0x7a49, 0x7a61, 0x7a62, 0x7a69, 0x9f9d, 0x7a70, 0x7a79, 0x7a7d,
	0x7a88, 0x7a97, 0x7a95, 0x7a98, 0x7a96, 0x7aa9, 0x7ac8, 0x7ab0,
	0x7ab6, 0x7ac5, 0x7ac4, 0x7abf, 0x9083, 0x7ac7, 0x7aca, 0x7acd,
	0x7acf, 0x7ad5, 0x7ad3, 0x7ad9, 0x7ada, 0x7add, 0x7ae1, 0x7ae2,
	0x7ae6, 0x7aed, 0x7af0, 0x7b02, 0x7b0f, 0x7b0a, 0x7b06, 0x7b33,
	0x7b18, 0x7b19, 0x7b1e, 0x7b35, 0x7b28, 0x7b36, 0x7b50,
	/* 0x6421 - 0x647e */
		0x7b7a, 0x7b04, 0x7b4d, 0x7b0b, 0x7b4c, 0x7b45, 0x7b75,
	0x7b65, 0x7b74, 0x7b67, 0x7b70, 0x7b71, 0x7b6c, 0x7b6e, 0x7b9d,
	0x7b98, 0x7b9f, 0x7b8d, 0x7b9c, 0x7b9a, 0x7b8b, 0x7b92, 0x7b8f,
	0x7b5d, 0x7b99, 0x7bcb, 0x7bc1, 0x7bcc, 0x7bcf, 0x7bb4, 0x7bc6,
	0x7bdd, 0x7be9, 0x7c11, 0x7c14, 0x7be6, 0x7be5, 0x7c60, 0x7c00,
	0x7c07, 0x7c13, 0x7bf3, 0x7bf7, 0x7c17, 0x7c0d, 0x7bf6, 0x7c23,
	0x7c27, 0x7c2a, 0x7c1f, 0x7c37, 0x7c2b, 0x7c3d, 0x7c4c, 0x7c43,
	0x7c54, 0x7c4f, 0x7c40, 0x7c50, 0x7c58, 0x7c5f, 0x7c64, 0x7c56,
	0x7c65, 0x7c6c, 0x7c75, 0x7c83, 0x7c90, 0x7ca4, 0x7cad, 0x7ca2,
	0x7cab, 0x7ca1, 0x7ca8, 0x7cb3, 0x7cb2, 0x7cb1, 0x7cae, 0x7cb9,
	0x7cbd, 0x7cc0, 0x7cc5, 0x7cc2, 0x7cd8, 0x7cd2, 0x7cdc, 0x7ce2,
	0x9b3b, 0x7cef, 0x7cf2, 0x7cf4, 0x7cf6, 0x7cfa, 0x7d06,
	/* 0x6521 - 0x657e */
		0x7d02, 0x7d1c, 0x7d15, 0x7d0a, 0x7d45, 0x7d4b, 0x7d2e,
	0x7d32, 0x7d3f, 0x7d35, 0x7d46, 0x7d73, 0x7d56, 0x7d4e, 0x7d72,
	0x7d68, 0x7d6e, 0x7d4f, 0x7d63, 0x7d93, 0x7d89, 0x7d5b, 0x7d8f,
	0x7d7d, 0x7d9b, 0x7dba, 0x7dae, 0x7da3, 0x7db5, 0x7dc7, 0x7dbd,
	0x7dab, 0x7e3d, 0x7da2, 0x7daf, 0x7ddc, 0x7db8, 0x7d9f, 0x7db0,
	0x7dd8, 0x7ddd, 0x7de4, 0x7dde, 0x7dfb, 0x7df2, 0x7de1, 0x7e05,
	0x7e0a, 0x7e23, 0x7e21, 0x7e12, 0x7e31, 0x7e1f, 0x7e09, 0x7e0b,
	0x7e22, 0x7e46, 0x7e66, 0x7e3b, 0x7e35, 0x7e39, 0x7e43, 0x7e37,
	0x7e32, 0x7e3a, 0x7e67, 0x7e5d, 0x7e56, 0x7e5e, 0x7e59, 0x7e5a,
	0x7e79, 0x7e6a, 0x7e69, 0x7e7c, 0x7e7b, 0x7e83, 0x7dd5, 0x7e7d,
	0x8fae, 0x7e7f, 0x7e88, 0x7e89, 0x7e8c, 0x7e92, 0x7e90, 0x7e93,
	0x7e94, 0x7e96, 0x7e8e, 0x7e9b, 0x7e9c, 0x7f38, 0x7f3a,
	/* 0x6621 - 0x667e */
		0x7f45, 0x7f4c, 0x7f4d, 0x7f4e, 0x7f50, 0x7f51, 0x7f55,
	0x7f54, 0x7f58, 0x7f5f, 0x7f60, 0x7f68, 0x7f69, 0x7f67, 0x7f78,
	0x7f82, 0x7f86, 0x7f83, 0x7f88, 0x7f87, 0x7f8c, 0x7f94, 0x7f9e,
	0x7f9d, 0x7f9a, 0x7fa3, 0x7faf, 0x7fb2, 0x7fb9, 0x7fae, 0x7fb6,
	0x7fb8, 0x8b71, 0x7fc5, 0x7fc6, 0x7fca, 0x7fd5, 0x7fd4, 0x7fe1,
	0x7fe6, 0x7fe9, 0x7ff3, 0x7ff9, 0x98dc, 0x8006, 0x8004, 0x800b,
	0x8012, 0x8018, 0x8019, 0x801c, 0x8021, 0x8028, 0x803f, 0x803b,
	0x804a, 0x8046, 0x8052, 0x8058, 0x805a, 0x805f, 0x8062, 0x8068,
	0x8073, 0x8072, 0x8070, 0x8076, 0x8079, 0x807d, 0x807f, 0x8084,
	0x8086, 0x8085, 0x809b, 0x8093, 0x809a, 0x80ad, 0x5190, 0x80ac,
	0x80db, 0x80e5, 0x80d9, 0x80dd, 0x80c4, 0x80da, 0x80d6, 0x8109,
	0x80ef, 0x80f1, 0x811b, 0x8129, 0x8123, 0x812f, 0x814b,
	/* 0x6721 - 0x677e */
		0x968b, 0x8146, 0x813e, 0x8153, 0x8151, 0x80fc, 0x8171,
	0x816e, 0x8165, 0x8166, 0x8174, 0x8183, 0x8188, 0x818a, 0x8180,
	0x8182, 0x81a0, 0x8195, 0x81a4, 0x81a3, 0x815f, 0x8193, 0x81a9,
	0x81b0, 0x81b5, 0x81be, 0x81b8, 0x81bd, 0x81c0, 0x81c2, 0x81ba,
	0x81c9, 0x81cd, 0x81d1, 0x81d9, 0x81d8, 0x81c8, 0x81da, 0x81df,
	0x81e0, 0x81e7, 0x81fa, 0x81fb, 0x81fe, 0x8201, 0x8202, 0x8205,
	0x8207, 0x820a, 0x820d, 0x8210, 0x8216, 0x8229, 0x822b, 0x8238,
	0x8233, 0x8240, 0x8259, 0x8258, 0x825d, 0x825a, 0x825f, 0x8264,
	0x8262, 0x8268, 0x826a, 0x826b, 0x822e, 0x8271, 0x8277, 0x8278,
	0x827e, 0x828d, 0x8292, 0x82ab, 0x829f, 0x82bb, 0x82ac, 0x82e1,
	0x82e3, 0x82df, 0x82d2, 0x82f4, 0x82f3, 0x82fa, 0x8393, 0x8303,
	0x82fb, 0x82f9, 0x82de, 0x8306, 0x82dc, 0x8309, 0x82d9,
	/* 0x6821 - 0x687e */
		0x8335, 0x8334, 0x8316, 0x8332, 0x8331, 0x8340, 0x8339,
	0x8350, 0x8345, 0x832f, 0x832b, 0x8317, 0x8318, 0x8385, 0x839a,
	0x83aa, 0x839f, 0x83a2, 0x8396, 0x8323, 0x838e, 0x8387, 0x838a,
	0x837c, 0x83b5, 0x8373, 0x8375, 0x83a0, 0x8389, 0x83a8, 0x83f4,
	0x8413, 0x83eb, 0x83ce, 0x83fd, 0x8403, 0x83d8, 0x840b, 0x83c1,
	0x83f7, 0x8407, 0x83e0, 0x83f2, 0x840d, 0x8422, 0x8420, 0x83bd,
	0x8438, 0x8506, 0x83fb, 0x846d, 0x842a, 0x843c, 0x855a, 0x8484,
	0x8477, 0x846b, 0x84ad, 0x846e, 0x8482, 0x8469, 0x8446, 0x842c,
	0x846f, 0x8479, 0x8435, 0x84ca, 0x8462, 0x84b9, 0x84bf, 0x849f,
	0x84d9, 0x84cd, 0x84bb, 0x84da, 0x84d0, 0x84c1, 0x84c6, 0x84d6,
	0x84a1, 0x8521, 0x84ff, 0x84f4, 0x8517, 0x8518, 0x852c, 0x851f,
	0x8515, 0x8514, 0x84fc, 0x8540, 0x8563, 0x8558, 0x8548,
	/* 0x6921 - 0x697e */
		0x8541, 0x8602, 0x854b, 0x8555, 0x8580, 0x85a4, 0x8588,
	0x8591, 0x858a, 0x85a8, 0x856d, 0x8594, 0x859b, 0x85ea, 0x8587,
	0x859c, 0x8577, 0x857e, 0x8590, 0x85c9, 0x85ba, 0x85cf, 0x85b9,
	0x85d0, 0x85d5, 0x85dd, 0x85e5, 0x85dc, 0x85f9, 0x860a, 0x8613,
	0x860b, 0x85fe, 0x85fa, 0x8606, 0x8622, 0x861a, 0x8630, 0x863f,
	0x864d, 0x4e55, 0x8654, 0x865f, 0x8667, 0x8671, 0x8693, 0x86a3,
	0x86a9, 0x86aa, 0x868b, 0x868c, 0x86b6, 0x86af, 0x86c4, 0x86c6,
	0x86b0, 0x86c9, 0x8823, 0x86ab, 0x86d4, 0x86de, 0x86e9, 0x86ec,
	0x86df, 0x86db, 0x86ef, 0x8712, 0x8706, 0x8708, 0x8700, 0x8703,
	0x86fb, 0x8711, 0x8709, 0x870d, 0x86f9, 0x870a, 0x8734, 0x873f,
	0x8737, 0x873b, 0x8725, 0x8729, 0x871a, 0x8760, 0x875f, 0x8778,
	0x874c, 0x874e, 0x8774, 0x8757, 0x8768, 0x876e, 0x8759,
	/* 0x6a21 - 0x6a7e */
		0x8753, 0x8763, 0x876a, 0x8805, 0x87a2, 0x879f, 0x8782,
	0x87af, 0x87cb, 0x87bd, 0x87c0, 0x87d0, 0x96d6, 0x87ab, 0x87c4,
	0x87b3, 0x87c7, 0x87c6, 0x87bb, 0x87ef, 0x87f2, 0x87e0, 0x880f,
	0x880d, 0x87fe, 0x87f6, 0x87f7, 0x880e, 0x87d2, 0x8811, 0x8816,
	0x8815, 0x8822, 0x8821, 0x8831, 0x8836, 0x8839, 0x8827, 0x883b,
	0x8844, 0x8842, 0x8852, 0x8859, 0x885e, 0x8862, 0x886b, 0x8881,
	0x887e, 0x889e, 0x8875, 0x887d, 0x88b5, 0x8872, 0x8882, 0x8897,
	0x8892, 0x88ae, 0x8899, 0x88a2, 0x888d, 0x88a4, 0x88b0, 0x88bf,
	0x88b1, 0x88c3, 0x88c4, 0x88d4, 0x88d8, 0x88d9, 0x88dd, 0x88f9,
	0x8902, 0x88fc, 0x88f4, 0x88e8, 0x88f2, 0x8904, 0x890c, 0x890a,
	0x8913, 0x8943, 0x891e, 0x8925, 0x892a, 0x892b, 0x8941, 0x8944,
	0x893b, 0x8936, 0x8938, 0x894c, 0x891d, 0x8960, 0x895e,
	/* 0x6b21 - 0x6b7e */
		0x8966, 0x8964, 0x896d, 0x896a, 0x896f, 0x8974, 0x8977,
	0x897e, 0x8983, 0x8988, 0x898a, 0x8993, 0x8998, 0x89a1, 0x89a9,
	0x89a6, 0x89ac, 0x89af, 0x89b2, 0x89ba, 0x89bd, 0x89bf, 0x89c0,
	0x89da, 0x89dc, 0x89dd, 0x89e7, 0x89f4, 0x89f8, 0x8a03, 0x8a16,
	0x8a10, 0x8a0c, 0x8a1b, 0x8a1d, 0x8a25, 0x8a36, 0x8a41, 0x8a5b,
	0x8a52, 0x8a46, 0x8a48, 0x8a7c, 0x8a6d, 0x8a6c, 0x8a62, 0x8a85,
	0x8a82, 0x8a84, 0x8aa8, 0x8aa1, 0x8a91, 0x8aa5, 0x8aa6, 0x8a9a,
	0x8aa3, 0x8ac4, 0x8acd, 0x8ac2, 0x8ada, 0x8aeb, 0x8af3, 0x8ae7,
	0x8ae4, 0x8af1, 0x8b14, 0x8ae0, 0x8ae2, 0x8af7, 0x8ade, 0x8adb,
	0x8b0c, 0x8b07, 0x8b1a, 0x8ae1, 0x8b16, 0x8b10, 0x8b17, 0x8b20,
	0x8b33, 0x97ab, 0x8b26, 0x8b2b, 0x8b3e, 0x8b28, 0x8b41, 0x8b4c,
	0x8b4f, 0x8b4e, 0x8b49, 0x8b56, 0x8b5b, 0x8b5a, 0x8b6b,
	/* 0x6c21 - 0x6c7e */
		0x8b5f, 0x8b6c, 0x8b6f, 0x8b74, 0x8b7d, 0x8b80, 0x8b8c,
	0x8b8e, 0x8b92, 0x8b93, 0x8b96, 0x8b99, 0x8b9a, 0x8c3a, 0x8c41,
	0x8c3f, 0x8c48, 0x8c4c, 0x8c4e, 0x8c50, 0x8c55, 0x8c62, 0x8c6c,
	0x8c78, 0x8c7a, 0x8c82, 0x8c89, 0x8c85, 0x8c8a, 0x8c8d, 0x8c8e,
	0x8c94, 0x8c7c, 0x8c98, 0x621d, 0x8cad, 0x8caa, 0x8cbd, 0x8cb2,
	0x8cb3, 0x8cae, 0x8cb6, 0x8cc8, 0x8cc1, 0x8ce4, 0x8ce3, 0x8cda,
	0x8cfd, 0x8cfa, 0x8cfb, 0x8d04, 0x8d05, 0x8d0a, 0x8d07, 0x8d0f,
	0x8d0d, 0x8d10, 0x9f4e, 0x8d13, 0x8ccd, 0x8d14, 0x8d16, 0x8d67,
	0x8d6d, 0x8d71, 0x8d73, 0x8d81, 0x8d99, 0x8dc2, 0x8dbe, 0x8dba,
	0x8dcf, 0x8dda, 0x8dd6, 0x8dcc, 0x8ddb, 0x8dcb, 0x8dea, 0x8deb,
	0x8ddf, 0x8de3, 0x8dfc, 0x8e08, 0x8e09, 0x8dff, 0x8e1d, 0x8e1e,
	0x8e10, 0x8e1f, 0x8e42, 0x8e35, 0x8e30, 0x8e34, 0x8e4a,
	/* 0x6d21 - 0x6d7e */
		0x8e47, 0x8e49, 0x8e4c, 0x8e50, 0x8e48, 0x8e59, 0x8e64,
	0x8e60, 0x8e2a, 0x8e63, 0x8e55, 0x8e76, 0x8e72, 0x8e7c, 0x8e81,
	0x8e87, 0x8e85, 0x8e84, 0x8e8b, 0x8e8a, 0x8e93, 0x8e91, 0x8e94,
	0x8e99, 0x8eaa, 0x8ea1, 0x8eac, 0x8eb0, 0x8ec6, 0x8eb1, 0x8ebe,
	0x8ec5, 0x8ec8, 0x8ecb, 0x8edb, 0x8ee3, 0x8efc, 0x8efb, 0x8eeb,
	0x8efe, 0x8f0a, 0x8f05, 0x8f15, 0x8f12, 0x8f19, 0x8f13, 0x8f1c,
	0x8f1f, 0x8f1b, 0x8f0c, 0x8f26, 0x8f33, 0x8f3b, 0x8f39, 0x8f45,
	0x8f42, 0x8f3e, 0x8f4c, 0x8f49, 0x8f46, 0x8f4e, 0x8f57, 0x8f5c,
	0x8f62, 0x8f63, 0x8f64, 0x8f9c, 0x8f9f, 0x8fa3, 0x8fad, 0x8faf,
	0x8fb7, 0x8fda, 0x8fe5, 0x8fe2, 0x8fea, 0x8fef, 0x9087, 0x8ff4,
	0x9005, 0x8ff9, 0x8ffa, 0x9011, 0x9015, 0x9021, 0x900d, 0x901e,
	0x9016, 0x900b, 0x9027, 0x9036, 0x9035, 0x9039, 0x8ff8,
	/* 0x6e21 - 0x6e7e */
		0x904f, 0x9050, 0x9051, 0x9052, 0x900e, 0x9049, 0x903e,
	0x9056, 0x9058, 0x905e, 0x9068, 0x906f, 0x9076, 0x96a8, 0x9072,
	0x9082, 0x907d, 0x9081, 0x9080, 0x908a, 0x9089, 0x908f, 0x90a8,
	0x90af, 0x90b1, 0x90b5, 0x90e2, 0x90e4, 0x6248, 0x90db, 0x9102,
	0x9112, 0x9119, 0x9132, 0x9130, 0x914a, 0x9156, 0x9158, 0x9163,
	0x9165, 0x9169, 0x9173, 0x9172, 0x918b, 0x9189, 0x9182, 0x91a2,
	0x91ab, 0x91af, 0x91aa, 0x91b5, 0x91b4, 0x91ba, 0x91c0, 0x91c1,
	0x91c9, 0x91cb, 0x91d0, 0x91d6, 0x91df, 0x91e1, 0x91db, 0x91fc,
	0x91f5, 0x91f6, 0x921e, 0x91ff, 0x9214, 0x922c, 0x9215, 0x9211,
	0x925e, 0x9257, 0x9245, 0x9249, 0x9264, 0x9248, 0x9295, 0x923f,
	0x924b, 0x9250, 0x929c, 0x9296, 0x9293, 0x929b, 0x925a, 0x92cf,
	0x92b9, 0x92b7, 0x92e9, 0x930f, 0x92fa, 0x9344, 0x932e,
	/* 0x6f21 - 0x6f7e */
		0x9319, 0x9322, 0x931a, 0x9323, 0x933a, 0x9335, 0x933b,
	0x935c, 0x9360, 0x937c, 0x936e, 0x9356, 0x93b0, 0x93ac, 0x93ad,
	0x9394, 0x93b9, 0x93d6, 0x93d7, 0x93e8, 0x93e5, 0x93d8, 0x93c3,
	0x93dd, 0x93d0, 0x93c8, 0x93e4, 0x941a, 0x9414, 0x9413, 0x9403,
	0x9407, 0x9410, 0x9436, 0x942b, 0x9435, 0x9421, 0x943a, 0x9441,
	0x9452, 0x9444, 0x945b, 0x9460, 0x9462, 0x945e, 0x946a, 0x9229,
	0x9470, 0x9475, 0x9477, 0x947d, 0x945a, 0x947c, 0x947e, 0x9481,
	0x947f, 0x9582, 0x9587, 0x958a, 0x9594, 0x9596, 0x9598, 0x9599,
	0x95a0, 0x95a8, 0x95a7, 0x95ad, 0x95bc, 0x95bb, 0x95b9, 0x95be,
	0x95ca, 0x6ff6, 0x95c3, 0x95cd, 0x95cc, 0x95d5, 0x95d4, 0x95d6,
	0x95dc, 0x95e1, 0x95e5, 0x95e2, 0x9621, 0x9628, 0x962e, 0x962f,
	0x9642, 0x964c, 0x964f, 0x964b, 0x9677, 0x965c, 0x965e,
	/* 0x7021 - 0x707e */
		0x965d, 0x965f, 0x9666, 0x9672, 0x966c, 0x968d, 0x9698,
	0x9695, 0x9697, 0x96aa, 0x96a7, 0x96b1, 0x96b2, 0x96b0, 0x96b4,
	0x96b6, 0x96b8, 0x96b9, 0x96ce, 0x96cb, 0x96c9, 0x96cd, 0x894d,
	0x96dc, 0x970d, 0x96d5, 0x96f9, 0x9704, 0x9706, 0x9708, 0x9713,
	0x970e, 0x9711, 0x970f, 0x9716, 0x9719, 0x9724, 0x972a, 0x9730,
	0x9739, 0x973d, 0x973e, 0x9744, 0x9746, 0x9748, 0x9742, 0x9749,
	0x975c, 0x9760, 0x9764, 0x9766, 0x9768, 0x52d2, 0x976b, 0x9771,
	0x9779, 0x9785, 0x977c, 0x9781, 0x977a, 0x9786, 0x978b, 0x978f,
	0x9790, 0x979c, 0x97a8, 0x97a6, 0x97a3, 0x97b3, 0x97b4, 0x97c3,
	0x97c6, 0x97c8, 0x97cb, 0x97dc, 0x97ed, 0x9f4f, 0x97f2, 0x7adf,
	0x97f6, 0x97f5, 0x980f, 0x980c, 0x9838, 0x9824, 0x9821, 0x9837,
	0x983d, 0x9846, 0x984f, 0x984b, 0x986b, 0x986f, 0x9870,
	/* 0x7121 - 0x717e */
		0x9871, 0x9874, 0x9873, 0x98aa, 0x98af, 0x98b1, 0x98b6,
	0x98c4, 0x98c3, 0x98c6, 0x98e9, 0x98eb, 0x9903, 0x9909, 0x9912,
	0x9914, 0x9918, 0x9921, 0x991d, 0x991e, 0x9924, 0x9920, 0x992c,
	0x992e, 0x993d, 0x993e, 0x9942, 0x9949, 0x9945, 0x9950, 0x994b,
	0x9951, 0x9952, 0x994c, 0x9955, 0x9997, 0x9998, 0x99a5, 0x99ad,
	0x99ae, 0x99bc, 0x99df, 0x99db, 0x99dd, 0x99d8, 0x99d1, 0x99ed,
	0x99ee, 0x99f1, 0x99f2, 0x99fb, 0x99f8, 0x9a01, 0x9a0f, 0x9a05,
	0x99e2, 0x9a19, 0x9a2b, 0x9a37, 0x9a45, 0x9a42, 0x9a40, 0x9a43,
	0x9a3e, 0x9a55, 0x9a4d, 0x9a5b, 0x9a57, 0x9a5f, 0x9a62, 0x9a65,
	0x9a64, 0x9a69, 0x9a6b, 0x9a6a, 0x9aad, 0x9ab0, 0x9abc, 0x9ac0,
	0x9acf, 0x9ad1, 0x9ad3, 0x9ad4, 0x9ade, 0x9adf, 0x9ae2, 0x9ae3,
	0x9ae6, 0x9aef, 0x9aeb, 0x9aee, 0x9af4, 0x9af1, 0x9af7,
	/* 0x7221 - 0x727e */
		0x9afb, 0x9b06, 0x9b18, 0x9b1a, 0x9b1f, 0x9b22, 0x9b23,
	0x9b25, 0x9b27, 0x9b28, 0x9b29, 0x9b2a, 0x9b2e, 0x9b2f, 0x9b32,
	0x9b44, 0x9b43, 0x9b4f, 0x9b4d, 0x9b4e, 0x9b51, 0x9b58, 0x9b74,
	0x9b93, 0x9b83, 0x9b91, 0x9b96, 0x9b97, 0x9b9f, 0x9ba0, 0x9ba8,
	0x9bb4, 0x9bc0, 0x9bca, 0x9bb9, 0x9bc6, 0x9bcf, 0x9bd1, 0x9bd2,
	0x9be3, 0x9be2, 0x9be4, 0x9bd4, 0x9be1, 0x9c3a, 0x9bf2, 0x9bf1,
	0x9bf0, 0x9c15, 0x9c14, 0x9c09, 0x9c13, 0x9c0c, 0x9c06, 0x9c08,
	0x9c12, 0x9c0a, 0x9c04, 0x9c2e, 0x9c1b, 0x9c25, 0x9c24, 0x9c21,
	0x9c30, 0x9c47, 0x9c32, 0x9c46, 0x9c3e, 0x9c5a, 0x9c60, 0x9c67,
	0x9c76, 0x9c78, 0x9ce7, 0x9cec, 0x9cf0, 0x9d09, 0x9d08, 0x9ceb,
	0x9d03, 0x9d06, 0x9d2a, 0x9d26, 0x9daf, 0x9d23, 0x9d1f, 0x9d44,
	0x9d15, 0x9d12, 0x9d41, 0x9d3f, 0x9d3e, 0x9d46, 0x9d48,
	/* 0x7321 - 0x737e */
		0x9d5d, 0x9d5e, 0x9d64, 0x9d51, 0x9d50, 0x9d59, 0x9d72,
	0x9d89, 0x9d87, 0x9dab, 0x9d6f, 0x9d7a, 0x9d9a, 0x9da4, 0x9da9,
	0x9db2, 0x9dc4, 0x9dc1, 0x9dbb, 0x9db8, 0x9dba, 0x9dc6, 0x9dcf,
	0x9dc2, 0x9dd9, 0x9dd3, 0x9df8, 0x9de6, 0x9ded, 0x9def, 0x9dfd,
	0x9e1a, 0x9e1b, 0x9e1e, 0x9e75, 0x9e79, 0x9e7d, 0x9e81, 0x9e88,
	0x9e8b, 0x9e8c, 0x9e92, 0x9e95, 0x9e91, 0x9e9d, 0x9ea5, 0x9ea9,
	0x9eb8, 0x9eaa, 0x9ead, 0x9761, 0x9ecc, 0x9ece, 0x9ecf, 0x9ed0,
	0x9ed4, 0x9edc, 0x9ede, 0x9edd, 0x9ee0, 0x9ee5, 0x9ee8, 0x9eef,
	0x9ef4, 0x9ef6, 0x9ef7, 0x9ef9, 0x9efb, 0x9efc, 0x9efd, 0x9f07,
	0x9f08, 0x76b7, 0x9f15, 0x9f21, 0x9f2c, 0x9f3e, 0x9f4a, 0x9f52,
	0x9f54, 0x9f63, 0x9f5f, 0x9f60, 0x9f61, 0x9f66, 0x9f67, 0x9f6c,
	0x9f6a, 0x9f77, 0x9f72, 0x9f76, 0x9f95, 0x9f9c, 0x9fa0,
	/* 0x7421 - 0x747e */
		0x582f, 0x69c7, 0x9059, 0x7464, 0x51dc, 0x7199, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7521 - 0x757e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7621 - 0x767e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7721 - 0x777e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7821 - 0x787e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7921 - 0x797e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7a21 - 0x7a7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7b21 - 0x7b7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7c21 - 0x7c7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7d21 - 0x7d7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7e21 - 0x7e7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static uint Jisx0208ToUnicode11(uint h, uint l)
{
    if ((0x0021 <= h) && (h <= 0x007e) && (0x0021 <= l) && (l <= 0x007e)) {
	return jisx0208_to_unicode[(h - 0x0021) * 0x005e + (l - 0x0021)];
    }
    return 0x0000;
}

/*
 * This data is derived from Unicode 1.1,
 * JIS X 0208 (1990) to Unicode mapping table version 0.9 .
 * (In addition NEC Vender Defined Char included)
 */
static unsigned short const unicode_to_jisx0208_00[] = {
	/* 0x0000 - 0x00ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2140, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2171, 0x2172, 0x0000, 0x0000, 0x0000, 0x2178,
	0x212f, 0x0000, 0x0000, 0x0000, 0x224c, 0x0000, 0x0000, 0x0000,
	0x216b, 0x215e, 0x0000, 0x0000, 0x212d, 0x0000, 0x2279, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x215f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2160,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_03[] = {
	/* 0x0300 - 0x03ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2621, 0x2622, 0x2623, 0x2624, 0x2625, 0x2626, 0x2627,
	0x2628, 0x2629, 0x262a, 0x262b, 0x262c, 0x262d, 0x262e, 0x262f,
	0x2630, 0x2631, 0x0000, 0x2632, 0x2633, 0x2634, 0x2635, 0x2636,
	0x2637, 0x2638, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2641, 0x2642, 0x2643, 0x2644, 0x2645, 0x2646, 0x2647,
	0x2648, 0x2649, 0x264a, 0x264b, 0x264c, 0x264d, 0x264e, 0x264f,
	0x2650, 0x2651, 0x0000, 0x2652, 0x2653, 0x2654, 0x2655, 0x2656,
	0x2657, 0x2658, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_04[] = {
	/* 0x0400 - 0x04ff */
	0x0000, 0x2727, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2721, 0x2722, 0x2723, 0x2724, 0x2725, 0x2726, 0x2728, 0x2729,
	0x272a, 0x272b, 0x272c, 0x272d, 0x272e, 0x272f, 0x2730, 0x2731,
	0x2732, 0x2733, 0x2734, 0x2735, 0x2736, 0x2737, 0x2738, 0x2739,
	0x273a, 0x273b, 0x273c, 0x273d, 0x273e, 0x273f, 0x2740, 0x2741,
	0x2751, 0x2752, 0x2753, 0x2754, 0x2755, 0x2756, 0x2758, 0x2759,
	0x275a, 0x275b, 0x275c, 0x275d, 0x275e, 0x275f, 0x2760, 0x2761,
	0x2762, 0x2763, 0x2764, 0x2765, 0x2766, 0x2767, 0x2768, 0x2769,
	0x276a, 0x276b, 0x276c, 0x276d, 0x276e, 0x276f, 0x2770, 0x2771,
	0x0000, 0x2757, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_20[] = {
	/* 0x2000 - 0x20ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x213e, 0x0000, 0x0000, 0x0000, 0x0000, 0x213d, 0x2142, 0x0000,
	0x2146, 0x2147, 0x0000, 0x0000, 0x2148, 0x2149, 0x0000, 0x0000,
	0x2277, 0x2278, 0x0000, 0x0000, 0x0000, 0x2145, 0x2144, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2273, 0x0000, 0x216c, 0x216d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2228, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_21[] = {
	/* 0x2100 - 0x21ff */
	0x0000, 0x0000, 0x0000, 0x216e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d62, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2d64, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2272, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2d35, 0x2d36, 0x2d37, 0x2d38, 0x2d39, 0x2d3a, 0x2d3b, 0x2d3c,
	0x2d3d, 0x2d3e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x222b, 0x222c, 0x222a, 0x222d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x224d, 0x0000, 0x224e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_22[] = {
	/* 0x2200 - 0x22ff */
	0x224f, 0x0000, 0x225f, 0x2250, 0x0000, 0x0000, 0x0000, 0x2260,
	0x223a, 0x0000, 0x0000, 0x223b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2d74, 0x215d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2265, 0x0000, 0x0000, 0x2267, 0x2167, 0x2d78,
	0x225c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x224a,
	0x224b, 0x2241, 0x2240, 0x2269, 0x226a, 0x0000, 0x2d73, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2168, 0x2268, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2266, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2262, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2162, 0x2261, 0x0000, 0x0000, 0x0000, 0x0000, 0x2165, 0x2166,
	0x0000, 0x0000, 0x2263, 0x2264, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x223e, 0x223f, 0x0000, 0x0000, 0x223c, 0x223d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x225d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d79,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_23[] = {
	/* 0x2300 - 0x23ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x225e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_24[] = {
	/* 0x2400 - 0x24ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2d21, 0x2d22, 0x2d23, 0x2d24, 0x2d25, 0x2d26, 0x2d27, 0x2d28,
	0x2d29, 0x2d2a, 0x2d2b, 0x2d2c, 0x2d2d, 0x2d2e, 0x2d2f, 0x2d30,
	0x2d31, 0x2d32, 0x2d33, 0x2d34, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_25[] = {
	/* 0x2500 - 0x25ff */
	0x2821, 0x282c, 0x2822, 0x282d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2823, 0x0000, 0x0000, 0x282e,
	0x2824, 0x0000, 0x0000, 0x282f, 0x2826, 0x0000, 0x0000, 0x2831,
	0x2825, 0x0000, 0x0000, 0x2830, 0x2827, 0x283c, 0x0000, 0x0000,
	0x2837, 0x0000, 0x0000, 0x2832, 0x2829, 0x283e, 0x0000, 0x0000,
	0x2839, 0x0000, 0x0000, 0x2834, 0x2828, 0x0000, 0x0000, 0x2838,
	0x283d, 0x0000, 0x0000, 0x2833, 0x282a, 0x0000, 0x0000, 0x283a,
	0x283f, 0x0000, 0x0000, 0x2835, 0x282b, 0x0000, 0x0000, 0x283b,
	0x0000, 0x0000, 0x2840, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2836, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2223, 0x2222, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2225, 0x2224, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2227, 0x2226, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2221, 0x217e,
	0x0000, 0x0000, 0x0000, 0x217b, 0x0000, 0x0000, 0x217d, 0x217c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x227e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_26[] = {
	/* 0x2600 - 0x26ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x217a, 0x2179, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x216a, 0x0000, 0x2169, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2276, 0x0000, 0x0000, 0x2275, 0x0000, 0x2274,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_30[] = {
	/* 0x3000 - 0x30ff */
	0x2121, 0x2122, 0x2123, 0x2137, 0x0000, 0x2139, 0x213a, 0x213b,
	0x2152, 0x2153, 0x2154, 0x2155, 0x2156, 0x2157, 0x2158, 0x2159,
	0x215a, 0x215b, 0x2229, 0x222e, 0x214c, 0x214d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2141, 0x2d60, 0x0000, 0x2d61,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2421, 0x2422, 0x2423, 0x2424, 0x2425, 0x2426, 0x2427,
	0x2428, 0x2429, 0x242a, 0x242b, 0x242c, 0x242d, 0x242e, 0x242f,
	0x2430, 0x2431, 0x2432, 0x2433, 0x2434, 0x2435, 0x2436, 0x2437,
	0x2438, 0x2439, 0x243a, 0x243b, 0x243c, 0x243d, 0x243e, 0x243f,
	0x2440, 0x2441, 0x2442, 0x2443, 0x2444, 0x2445, 0x2446, 0x2447,
	0x2448, 0x2449, 0x244a, 0x244b, 0x244c, 0x244d, 0x244e, 0x244f,
	0x2450, 0x2451, 0x2452, 0x2453, 0x2454, 0x2455, 0x2456, 0x2457,
	0x2458, 0x2459, 0x245a, 0x245b, 0x245c, 0x245d, 0x245e, 0x245f,
	0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466, 0x2467,
	0x2468, 0x2469, 0x246a, 0x246b, 0x246c, 0x246d, 0x246e, 0x246f,
	0x2470, 0x2471, 0x2472, 0x2473, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x212b, 0x212c, 0x2135, 0x2136, 0x0000,
	0x0000, 0x2521, 0x2522, 0x2523, 0x2524, 0x2525, 0x2526, 0x2527,
	0x2528, 0x2529, 0x252a, 0x252b, 0x252c, 0x252d, 0x252e, 0x252f,
	0x2530, 0x2531, 0x2532, 0x2533, 0x2534, 0x2535, 0x2536, 0x2537,
	0x2538, 0x2539, 0x253a, 0x253b, 0x253c, 0x253d, 0x253e, 0x253f,
	0x2540, 0x2541, 0x2542, 0x2543, 0x2544, 0x2545, 0x2546, 0x2547,
	0x2548, 0x2549, 0x254a, 0x254b, 0x254c, 0x254d, 0x254e, 0x254f,
	0x2550, 0x2551, 0x2552, 0x2553, 0x2554, 0x2555, 0x2556, 0x2557,
	0x2558, 0x2559, 0x255a, 0x255b, 0x255c, 0x255d, 0x255e, 0x255f,
	0x2560, 0x2561, 0x2562, 0x2563, 0x2564, 0x2565, 0x2566, 0x2567,
	0x2568, 0x2569, 0x256a, 0x256b, 0x256c, 0x256d, 0x256e, 0x256f,
	0x2570, 0x2571, 0x2572, 0x2573, 0x2574, 0x2575, 0x2576, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2126, 0x213c, 0x2133, 0x2134, 0x0000,
};

static unsigned short const unicode_to_jisx0208_32[] = {
	/* 0x3200 - 0x32ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2d6a, 0x2d6b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2d6c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2d65, 0x2d66, 0x2d67, 0x2d68,
	0x2d69, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_33[] = {
	/* 0x3300 - 0x33ff */
	0x0000, 0x0000, 0x0000, 0x2d46, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d4a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2d41, 0x0000, 0x0000, 0x0000,
	0x2d44, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2d42, 0x2d4c, 0x0000, 0x0000, 0x2d4b, 0x2d45,
	0x0000, 0x0000, 0x0000, 0x2d4d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d47, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2d4f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2d40, 0x2d4e, 0x0000, 0x0000, 0x2d43, 0x0000, 0x0000,
	0x0000, 0x2d48, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d49,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2d5f, 0x2d6f, 0x2d6e, 0x2d6d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d53, 0x2d54,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2d50, 0x2d51, 0x2d52, 0x0000,
	0x0000, 0x2d56, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2d55, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2d63, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_4e[] = {
	/* 0x4e00 - 0x4eff */
	0x306c, 0x437a, 0x0000, 0x3c37, 0x0000, 0x0000, 0x0000, 0x4b7c,
	0x3e66, 0x3b30, 0x3e65, 0x323c, 0x0000, 0x4954, 0x4d3f, 0x0000,
	0x5022, 0x312f, 0x0000, 0x0000, 0x336e, 0x5023, 0x4024, 0x5242,
	0x3556, 0x4a3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e67, 0x0000,
	0x0000, 0x4e3e, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a42, 0x0000,
	0x0000, 0x0000, 0x5024, 0x0000, 0x0000, 0x4366, 0x0000, 0x0000,
	0x0000, 0x5025, 0x367a, 0x0000, 0x0000, 0x0000, 0x5026, 0x0000,
	0x345d, 0x4330, 0x0000, 0x3c67, 0x5027, 0x0000, 0x0000, 0x5028,
	0x0000, 0x0000, 0x5029, 0x4735, 0x0000, 0x3557, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4737, 0x0000, 0x4663, 0x3843, 0x4b33,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6949, 0x502a, 0x3e68,
	0x502b, 0x3235, 0x0000, 0x0000, 0x0000, 0x3665, 0x3870, 0x4c69,
	0x0000, 0x0000, 0x5626, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4d70, 0x0000, 0x467d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3425, 0x0000,
	0x3535, 0x0000, 0x502c, 0x0000, 0x0000, 0x502d, 0x4e3b, 0x0000,
	0x4d3d, 0x4168, 0x502f, 0x3b76, 0x4673, 0x0000, 0x5032, 0x0000,
	0x0000, 0x313e, 0x385f, 0x0000, 0x385e, 0x3066, 0x0000, 0x0000,
	0x4f4b, 0x4f4a, 0x0000, 0x3a33, 0x3021, 0x0000, 0x5033, 0x5034,
	0x5035, 0x4b34, 0x5036, 0x0000, 0x3872, 0x3067, 0x4b72, 0x0000,
	0x357c, 0x0000, 0x0000, 0x357d, 0x357e, 0x4462, 0x4e3c, 0x0000,
	0x5037, 0x0000, 0x0000, 0x5038, 0x0000, 0x0000, 0x5039, 0x0000,
	0x0000, 0x0000, 0x3f4d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3d3a, 0x3f4e, 0x503e, 0x0000, 0x503c, 0x0000, 0x503d, 0x3558,
	0x0000, 0x0000, 0x3a23, 0x3270, 0x0000, 0x503b, 0x503a, 0x4a29,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3b46, 0x3b45, 0x423e, 0x503f,
	0x4955, 0x4067, 0x0000, 0x0000, 0x0000, 0x2138, 0x5040, 0x5042,
	0x0000, 0x0000, 0x0000, 0x4265, 0x4e61, 0x304a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5041, 0x323e, 0x0000,
	0x3644, 0x0000, 0x4367, 0x0000, 0x0000, 0x0000, 0x376f, 0x5043,
	0x0000, 0x0000, 0x0000, 0x4724, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_4f[] = {
	/* 0x4f00 - 0x4fff */
	0x0000, 0x346b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5044, 0x304b, 0x0000, 0x0000, 0x3860, 0x346c, 0x497a,
	0x4832, 0x3559, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3271, 0x0000, 0x5067, 0x4541, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x476c,
	0x5046, 0x0000, 0x0000, 0x0000, 0x483c, 0x0000, 0x4e62, 0x0000,
	0x3f2d, 0x0000, 0x3b47, 0x0000, 0x3b77, 0x3240, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4451, 0x0000, 0x0000, 0x4322, 0x504a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x304c, 0x4463, 0x3d3b,
	0x3a34, 0x4d24, 0x0000, 0x424e, 0x0000, 0x323f, 0x0000, 0x5049,
	0x0000, 0x4d3e, 0x5045, 0x5047, 0x3a6e, 0x5048, 0x5524, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5050, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5053,
	0x5051, 0x0000, 0x0000, 0x3242, 0x0000, 0x4a3b, 0x504b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x504f, 0x3873, 0x0000, 0x0000, 0x3b48,
	0x0000, 0x0000, 0x0000, 0x3426, 0x0000, 0x0000, 0x5054, 0x0000,
	0x504c, 0x0000, 0x0000, 0x4e63, 0x0000, 0x3b78, 0x0000, 0x504d,
	0x0000, 0x5052, 0x0000, 0x0000, 0x0000, 0x0000, 0x5055, 0x0000,
	0x504e, 0x0000, 0x0000, 0x3621, 0x0000, 0x304d, 0x0000, 0x0000,
	0x3622, 0x3241, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5525, 0x0000, 0x4b79, 0x496e, 0x3874,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f2f, 0x4e37, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a58,
	0x0000, 0x0000, 0x3738, 0x4225, 0x3264, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3d53, 0x0000, 0x0000, 0x0000, 0x5059, 0x0000,
	0x505e, 0x505c, 0x0000, 0x0000, 0x5057, 0x0000, 0x0000, 0x422f,
	0x505a, 0x0000, 0x505d, 0x505b, 0x0000, 0x4a5d, 0x0000, 0x5058,
	0x0000, 0x3f2e, 0x0000, 0x4b73, 0x505f, 0x5060, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d24, 0x506d,
	0x0000, 0x0000, 0x0000, 0x4750, 0x0000, 0x4936, 0x5068, 0x0000,
	0x4a70, 0x0000, 0x3236, 0x0000, 0x0000, 0x0000, 0x506c, 0x0000,
};

static unsigned short const unicode_to_jisx0208_50[] = {
	/* 0x5000 - 0x50ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5066, 0x506f, 0x0000,
	0x0000, 0x4152, 0x0000, 0x3844, 0x0000, 0x475c, 0x0000, 0x6047,
	0x0000, 0x506e, 0x455d, 0x0000, 0x5063, 0x0000, 0x3876, 0x0000,
	0x0000, 0x3875, 0x5061, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c5a,
	0x0000, 0x5069, 0x0000, 0x4a6f, 0x434d, 0x5065, 0x3771, 0x0000,
	0x5062, 0x506a, 0x5064, 0x4e51, 0x506b, 0x4f41, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3666, 0x0000,
	0x0000, 0x3770, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5070, 0x0000, 0x0000, 0x0000, 0x5071,
	0x5075, 0x304e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a50,
	0x5074, 0x0000, 0x0000, 0x0000, 0x0000, 0x5073, 0x5077, 0x0000,
	0x0000, 0x0000, 0x5076, 0x0000, 0x4464, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3772, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5078, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3c45, 0x0000, 0x4226, 0x4465, 0x3676, 0x0000,
	0x5079, 0x0000, 0x0000, 0x0000, 0x0000, 0x3536, 0x0000, 0x0000,
	0x507a, 0x0000, 0x0000, 0x0000, 0x0000, 0x507c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b35, 0x0000, 0x0000,
	0x0000, 0x3766, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3b31, 0x4877, 0x507b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3a45, 0x4d43, 0x0000, 0x0000,
	0x0000, 0x0000, 0x507e, 0x5123, 0x507d, 0x3a44, 0x0000, 0x3d7d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3739, 0x0000,
	0x0000, 0x0000, 0x5124, 0x0000, 0x0000, 0x364f, 0x0000, 0x0000,
	0x0000, 0x5121, 0x5122, 0x0000, 0x0000, 0x462f, 0x0000, 0x417c,
	0x0000, 0x3623, 0x0000, 0x0000, 0x0000, 0x4b4d, 0x5125, 0x0000,
	0x0000, 0x0000, 0x4e3d, 0x0000, 0x0000, 0x0000, 0x5126, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5129, 0x0000, 0x5127, 0x0000, 0x414e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5128, 0x512a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x512c, 0x0000, 0x0000,
	0x0000, 0x512b, 0x0000, 0x4a48, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_51[] = {
	/* 0x5100 - 0x51ff */
	0x3537, 0x512e, 0x512f, 0x0000, 0x322f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x512d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3c74, 0x0000, 0x5132, 0x5131, 0x5130, 0x0000,
	0x5056, 0x0000, 0x5133, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d7e,
	0x0000, 0x5134, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c59, 0x0000, 0x0000, 0x0000, 0x0000, 0x5136,
	0x0000, 0x0000, 0x5135, 0x5138, 0x5137, 0x0000, 0x0000, 0x5139,
	0x513a, 0x3074, 0x0000, 0x3835, 0x373b, 0x3d3c, 0x437b, 0x3624,
	0x4068, 0x3877, 0x0000, 0x396e, 0x513c, 0x4c48, 0x4546, 0x0000,
	0x3b79, 0x0000, 0x513b, 0x0000, 0x513d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x455e, 0x0000, 0x3375, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x513e, 0x0000, 0x0000, 0x467e, 0x0000, 0x0000,
	0x4134, 0x5140, 0x5141, 0x482c, 0x3878, 0x4f3b, 0x5142, 0x0000,
	0x0000, 0x3626, 0x0000, 0x0000, 0x0000, 0x4a3c, 0x4236, 0x3671,
	0x4535, 0x0000, 0x0000, 0x0000, 0x3773, 0x0000, 0x0000, 0x0000,
	0x5143, 0x0000, 0x5144, 0x0000, 0x0000, 0x4662, 0x315f, 0x0000,
	0x0000, 0x5147, 0x3a7d, 0x0000, 0x5146, 0x3a46, 0x0000, 0x5148,
	0x666e, 0x5149, 0x4b41, 0x514a, 0x0000, 0x514b, 0x514c, 0x3e69,
	0x0000, 0x3c4c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3427, 0x0000, 0x514f, 0x0000, 0x514d, 0x4c3d, 0x514e, 0x0000,
	0x495a, 0x5150, 0x5151, 0x5152, 0x455f, 0x0000, 0x0000, 0x0000,
	0x5156, 0x5154, 0x5155, 0x5153, 0x3a63, 0x5157, 0x4c6a, 0x4e64,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5158, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4028, 0x5159, 0x3d5a, 0x0000,
	0x0000, 0x515a, 0x0000, 0x437c, 0x4e3f, 0x4560, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5245, 0x0000,
	0x0000, 0x0000, 0x0000, 0x515b, 0x7425, 0x3645, 0x0000, 0x0000,
	0x515c, 0x4b5e, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d68, 0x427c,
	0x0000, 0x515e, 0x4664, 0x0000, 0x0000, 0x515f, 0x0000, 0x0000,
	0x5160, 0x332e, 0x0000, 0x0000, 0x0000, 0x5161, 0x3627, 0x0000,
	0x464c, 0x317a, 0x3d50, 0x0000, 0x0000, 0x4821, 0x5162, 0x0000,
};

static unsigned short const unicode_to_jisx0208_52[] = {
	/* 0x5200 - 0x52ff */
	0x4561, 0x0000, 0x0000, 0x3f4f, 0x5163, 0x0000, 0x4a2c, 0x405a,
	0x3422, 0x0000, 0x3429, 0x5164, 0x0000, 0x0000, 0x5166, 0x0000,
	0x0000, 0x373a, 0x0000, 0x0000, 0x5165, 0x0000, 0x0000, 0x4e73,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d69, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x483d, 0x4a4c, 0x0000, 0x5167,
	0x0000, 0x4d78, 0x5168, 0x0000, 0x0000, 0x0000, 0x5169, 0x0000,
	0x457e, 0x0000, 0x0000, 0x516a, 0x0000, 0x0000, 0x4029, 0x3a7e,
	0x3774, 0x516b, 0x3b49, 0x396f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4466, 0x516d, 0x0000, 0x0000, 0x4227,
	0x0000, 0x0000, 0x3a6f, 0x516e, 0x516f, 0x4130, 0x0000, 0x516c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5171, 0x0000, 0x4b36, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3964, 0x0000, 0x0000, 0x5170, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3775, 0x3a5e, 0x476d, 0x0000, 0x0000,
	0x0000, 0x5174, 0x5172, 0x0000, 0x0000, 0x0000, 0x0000, 0x497b,
	0x3e6a, 0x517b, 0x3364, 0x5175, 0x5173, 0x414f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5177, 0x0000, 0x5176,
	0x0000, 0x0000, 0x0000, 0x3344, 0x0000, 0x0000, 0x0000, 0x3760,
	0x517c, 0x4e2d, 0x0000, 0x0000, 0x0000, 0x5178, 0x0000, 0x0000,
	0x0000, 0x517d, 0x517a, 0x0000, 0x5179, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4e4f, 0x0000, 0x0000, 0x0000, 0x3879,
	0x3243, 0x0000, 0x0000, 0x4e74, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3d75, 0x4558, 0x3965, 0x5222, 0x5223, 0x0000, 0x0000,
	0x0000, 0x4e65, 0x0000, 0x0000, 0x4f2b, 0x5225, 0x0000, 0x0000,
	0x0000, 0x387a, 0x0000, 0x0000, 0x5224, 0x0000, 0x332f, 0x0000,
	0x0000, 0x5226, 0x0000, 0x4b56, 0x0000, 0x443c, 0x0000, 0x4d26,
	0x0000, 0x4a59, 0x0000, 0x0000, 0x0000, 0x5227, 0x0000, 0x0000,
	0x0000, 0x0000, 0x7055, 0x0000, 0x0000, 0x4630, 0x0000, 0x5228,
	0x342a, 0x4c33, 0x0000, 0x0000, 0x0000, 0x3e21, 0x5229, 0x4a67,
	0x522d, 0x0000, 0x402a, 0x522a, 0x3650, 0x0000, 0x522b, 0x342b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x372e, 0x522e, 0x0000, 0x522f, 0x0000, 0x0000,
	0x5230, 0x5231, 0x3c5b, 0x0000, 0x0000, 0x0000, 0x387b, 0x4c5e,
};

static unsigned short const unicode_to_jisx0208_53[] = {
	/* 0x5300 - 0x53ff */
	0x0000, 0x4c68, 0x4677, 0x0000, 0x0000, 0x4a71, 0x5232, 0x0000,
	0x5233, 0x0000, 0x0000, 0x0000, 0x0000, 0x5235, 0x0000, 0x5237,
	0x5236, 0x0000, 0x0000, 0x0000, 0x0000, 0x5238, 0x323d, 0x4b4c,
	0x0000, 0x3a7c, 0x5239, 0x0000, 0x0000, 0x4159, 0x0000, 0x0000,
	0x3e22, 0x3629, 0x0000, 0x523a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x485b, 0x0000, 0x0000, 0x0000, 0x0000, 0x523b,
	0x0000, 0x523c, 0x0000, 0x523d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x523e, 0x4924, 0x3668, 0x3065, 0x0000, 0x0000, 0x0000, 0x463f,
	0x523f, 0x3d3d, 0x0000, 0x4069, 0x0000, 0x5241, 0x5240, 0x3e23,
	0x3861, 0x5243, 0x483e, 0x0000, 0x0000, 0x5244, 0x0000, 0x0000,
	0x0000, 0x485c, 0x4234, 0x426e, 0x3628, 0x0000, 0x0000, 0x466e,
	0x4331, 0x0000, 0x476e, 0x0000, 0x4b4e, 0x0000, 0x5246, 0x0000,
	0x406a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3735, 0x0000,
	0x0000, 0x5247, 0x0000, 0x0000, 0x0000, 0x0000, 0x5248, 0x312c,
	0x3075, 0x346d, 0x0000, 0x4228, 0x3551, 0x4d71, 0x0000, 0x524b,
	0x3237, 0x0000, 0x0000, 0x524a, 0x0000, 0x0000, 0x0000, 0x362a,
	0x0000, 0x0000, 0x524c, 0x0000, 0x4c71, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x524d, 0x0000,
	0x4e52, 0x0000, 0x387c, 0x0000, 0x0000, 0x0000, 0x0000, 0x3836,
	0x524e, 0x0000, 0x0000, 0x0000, 0x0000, 0x5250, 0x524f, 0x0000,
	0x3f5f, 0x3139, 0x0000, 0x0000, 0x0000, 0x315e, 0x5251, 0x0000,
	0x5252, 0x0000, 0x0000, 0x3837, 0x0000, 0x0000, 0x5253, 0x0000,
	0x0000, 0x0000, 0x0000, 0x356e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b32, 0x5254, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4b74, 0x3a35, 0x355a, 0x4d27, 0x4150, 0x483f, 0x3c7d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d47, 0x0000, 0x3c68, 0x3c75,
	0x0000, 0x3d76, 0x0000, 0x4840, 0x0000, 0x0000, 0x0000, 0x5257,
	0x0000, 0x3143, 0x4151, 0x387d, 0x3845, 0x3667, 0x0000, 0x0000,
	0x525b, 0x4321, 0x427e, 0x362b, 0x3e24, 0x525c, 0x525a, 0x3244,
	0x4266, 0x3c38, 0x3b4b, 0x3126, 0x0000, 0x0000, 0x3370, 0x3966,
	0x3b4a, 0x0000, 0x525d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_54[] = {
	/* 0x5400 - 0x54ff */
	0x0000, 0x525e, 0x0000, 0x3549, 0x3346, 0x0000, 0x0000, 0x0000,
	0x3967, 0x3548, 0x445f, 0x3125, 0x4631, 0x4c3e, 0x3921, 0x4d79,
	0x4547, 0x387e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x372f, 0x0000, 0x5267, 0x0000, 0x3663,
	0x4b4a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x485d, 0x0000,
	0x0000, 0x5266, 0x0000, 0x345e, 0x5261, 0x5262, 0x5264, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5265, 0x0000,
	0x355b, 0x3f61, 0x0000, 0x4a2d, 0x5263, 0x525f, 0x3863, 0x0000,
	0x5260, 0x0000, 0x4f24, 0x0000, 0x0000, 0x0000, 0x4a72, 0x0000,
	0x4468, 0x3862, 0x3970, 0x0000, 0x0000, 0x0000, 0x5268, 0x0000,
	0x0000, 0x465d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x526c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c7e, 0x0000, 0x3c76, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x526f, 0x526d, 0x0000, 0x4c23, 0x0000, 0x526a, 0x5273, 0x526e,
	0x0000, 0x0000, 0x0000, 0x5271, 0x3846, 0x4c3f, 0x0000, 0x0000,
	0x5272, 0x0000, 0x0000, 0x0000, 0x5274, 0x0000, 0x5276, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3a70, 0x4f42, 0x0000, 0x526b, 0x5269,
	0x5275, 0x0000, 0x5270, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5278, 0x0000, 0x5323, 0x527a, 0x0000, 0x0000,
	0x527e, 0x0000, 0x0000, 0x5321, 0x527b, 0x0000, 0x0000, 0x533e,
	0x0000, 0x0000, 0x3a69, 0x3331, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5279, 0x0000, 0x0000, 0x0000, 0x5325, 0x3076, 0x5324, 0x0000,
	0x3025, 0x494a, 0x5322, 0x0000, 0x527c, 0x0000, 0x0000, 0x5277,
	0x527d, 0x3a48, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5326, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3077, 0x532f, 0x0000, 0x0000, 0x5327, 0x5328, 0x0000,
	0x3e25, 0x4b69, 0x0000, 0x0000, 0x0000, 0x532d, 0x532c, 0x0000,
	0x0000, 0x0000, 0x452f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x532e, 0x0000, 0x0000, 0x532b, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_55[] = {
	/* 0x5500 - 0x55ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x3134, 0x0000, 0x3a36, 0x3f30,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5329,
	0x4562, 0x0000, 0x0000, 0x0000, 0x532a, 0x0000, 0x3022, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5334, 0x4d23,
	0x0000, 0x3e27, 0x0000, 0x533a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5339, 0x5330, 0x0000, 0x0000, 0x0000, 0x0000, 0x4243, 0x0000,
	0x5331, 0x0000, 0x0000, 0x0000, 0x426f, 0x5336, 0x3e26, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5333, 0x0000, 0x0000, 0x4c64,
	0x0000, 0x0000, 0x0000, 0x373c, 0x0000, 0x0000, 0x5337, 0x5338,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5335, 0x533b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5332, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5341, 0x5346, 0x0000, 0x5342, 0x0000,
	0x533d, 0x0000, 0x0000, 0x5347, 0x4131, 0x0000, 0x0000, 0x5349,
	0x0000, 0x3922, 0x533f, 0x437d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5343, 0x533c, 0x342d, 0x0000, 0x346e, 0x3365, 0x5344, 0x5340,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3776,
	0x534a, 0x5348, 0x4153, 0x354a, 0x362c, 0x0000, 0x5345, 0x0000,
	0x3674, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3144, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x534e, 0x534c, 0x0000, 0x5427,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5351, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x534b, 0x0000, 0x534f, 0x0000, 0x0000, 0x534d,
	0x0000, 0x0000, 0x0000, 0x3b4c, 0x5350, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5353,
	0x0000, 0x5358, 0x0000, 0x0000, 0x0000, 0x5356, 0x5355, 0x0000,
};

static unsigned short const unicode_to_jisx0208_56[] = {
	/* 0x5600 - 0x56ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4332, 0x0000,
	0x0000, 0x3245, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5352, 0x0000, 0x5354, 0x3e28,
	0x3133, 0x0000, 0x0000, 0x5357, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x325e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5362,
	0x0000, 0x3e7c, 0x535e, 0x0000, 0x535c, 0x0000, 0x535d, 0x0000,
	0x535f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x313d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4139, 0x0000, 0x5359, 0x0000,
	0x535a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x337a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5361, 0x0000, 0x0000, 0x0000,
	0x346f, 0x0000, 0x5364, 0x5360, 0x5363, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4a2e, 0x0000, 0x0000, 0x0000,
	0x4655, 0x0000, 0x4838, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5366, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5365, 0x3345,
	0x0000, 0x0000, 0x5367, 0x0000, 0x0000, 0x0000, 0x0000, 0x536a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5369, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5368, 0x0000, 0x4739, 0x0000, 0x0000, 0x536b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x536c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x536e, 0x0000, 0x536d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5370, 0x0000, 0x0000, 0x0000,
	0x5373, 0x5371, 0x536f, 0x5372, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5374, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5375, 0x0000,
	0x0000, 0x5376, 0x0000, 0x5377, 0x0000, 0x0000, 0x0000, 0x5378,
	0x5145, 0x0000, 0x3c7c, 0x3b4d, 0x0000, 0x0000, 0x3273, 0x0000,
	0x3078, 0x0000, 0x0000, 0x4344, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5379, 0x0000,
	0x3a24, 0x0000, 0x304f, 0x3f5e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x537a, 0x3847, 0x0000, 0x0000, 0x3971, 0x0000, 0x537c,
};

static unsigned short const unicode_to_jisx0208_57[] = {
	/* 0x5700 - 0x57ff */
	0x537b, 0x0000, 0x0000, 0x4a60, 0x537d, 0x0000, 0x0000, 0x0000,
	0x5421, 0x537e, 0x0000, 0x5422, 0x0000, 0x5423, 0x0000, 0x3777,
	0x0000, 0x0000, 0x3160, 0x5424, 0x0000, 0x0000, 0x5426, 0x0000,
	0x5425, 0x0000, 0x0000, 0x0000, 0x5428, 0x0000, 0x0000, 0x455a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5429, 0x3035,
	0x3a5f, 0x0000, 0x0000, 0x0000, 0x0000, 0x373d, 0x0000, 0x0000,
	0x434f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x542a,
	0x542b, 0x0000, 0x0000, 0x542d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x542e, 0x0000, 0x3a64, 0x0000, 0x0000, 0x0000, 0x0000, 0x3651,
	0x0000, 0x0000, 0x4b37, 0x0000, 0x0000, 0x0000, 0x542c, 0x542f,
	0x3a41, 0x3923, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5433, 0x0000, 0x0000, 0x3a25, 0x0000, 0x4333, 0x0000,
	0x0000, 0x5430, 0x445a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5434,
	0x0000, 0x0000, 0x3f62, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5432, 0x5435, 0x0000, 0x373f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5436, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5437, 0x0000, 0x3924, 0x3340, 0x5439, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x543a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x543b, 0x0000, 0x0000, 0x5438, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5431, 0x0000, 0x0000, 0x543c, 0x0000, 0x0000, 0x543d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4b64, 0x0000, 0x0000, 0x3e6b, 0x0000,
	0x0000, 0x0000, 0x543f, 0x5440, 0x543e, 0x0000, 0x5442, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4738, 0x0000, 0x0000, 0x3068,
	0x4956, 0x0000, 0x0000, 0x5443, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3e7d, 0x0000, 0x0000, 0x3c39,
	0x0000, 0x475d, 0x3470, 0x0000, 0x3a6b, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_58[] = {
	/* 0x5800 - 0x58ff */
	0x4b59, 0x0000, 0x4632, 0x0000, 0x0000, 0x3778, 0x424f, 0x0000,
	0x0000, 0x0000, 0x5441, 0x5444, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4244, 0x0000, 0x0000,
	0x0000, 0x5445, 0x0000, 0x0000, 0x0000, 0x5446, 0x0000, 0x0000,
	0x0000, 0x5448, 0x0000, 0x0000, 0x4469, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x342e, 0x0000, 0x0000, 0x0000, 0x0000, 0x7421,
	0x3161, 0x4a73, 0x0000, 0x0000, 0x3e6c, 0x4548, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a66, 0x0000, 0x0000, 0x544e, 0x0000, 0x0000,
	0x4a3d, 0x4e5d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3274, 0x544a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x413a, 0x544d, 0x0000, 0x4563, 0x0000, 0x0000, 0x4549,
	0x4564, 0x4839, 0x444d, 0x0000, 0x0000, 0x0000, 0x3a49, 0x0000,
	0x0000, 0x0000, 0x5449, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3176, 0x0000, 0x4536, 0x0000, 0x0000, 0x0000, 0x0000,
	0x544b, 0x0000, 0x5447, 0x0000, 0x0000, 0x3f50, 0x0000, 0x0000,
	0x0000, 0x544f, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d4e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x362d, 0x0000, 0x5450, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a68, 0x0000, 0x0000, 0x0000, 0x417d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4446, 0x0000, 0x0000, 0x5452,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4b4f, 0x0000, 0x0000, 0x5453, 0x0000, 0x0000, 0x5458, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a2f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5457, 0x5451, 0x5454, 0x5456, 0x0000, 0x0000, 0x3a26, 0x0000,
	0x0000, 0x4a49, 0x0000, 0x0000, 0x0000, 0x5459, 0x0000, 0x4345,
	0x0000, 0x0000, 0x3275, 0x0000, 0x3e6d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x545b, 0x0000, 0x545a, 0x0000, 0x3968, 0x0000, 0x545c,
	0x545e, 0x545d, 0x0000, 0x0000, 0x5460, 0x0000, 0x5455, 0x5462,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5461, 0x545f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b4e, 0x3f51, 0x0000, 0x4154, 0x5463,
	0x403c, 0x306d, 0x4764, 0x0000, 0x0000, 0x0000, 0x0000, 0x445b,
	0x0000, 0x5465, 0x5464, 0x5466, 0x5467, 0x5468, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_59[] = {
	/* 0x5900 - 0x59ff */
	0x0000, 0x0000, 0x5469, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a51, 0x546a, 0x0000, 0x0000, 0x0000, 0x0000, 0x3246,
	0x546b, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d3c, 0x3330, 0x0000,
	0x5249, 0x3d48, 0x423f, 0x546c, 0x4c6b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c34, 0x0000, 0x0000, 0x546e, 0x0000, 0x4267,
	0x0000, 0x4537, 0x4240, 0x4957, 0x546f, 0x5470, 0x317b, 0x0000,
	0x0000, 0x3c3a, 0x5471, 0x0000, 0x0000, 0x0000, 0x0000, 0x3050,
	0x5472, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5473, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3162, 0x0000, 0x0000, 0x3471,
	0x4660, 0x4a74, 0x0000, 0x0000, 0x0000, 0x0000, 0x5477, 0x4155,
	0x5476, 0x3740, 0x0000, 0x0000, 0x4b5b, 0x5475, 0x0000, 0x4565,
	0x5479, 0x0000, 0x5478, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x547b, 0x0000, 0x547a, 0x0000, 0x0000, 0x317c, 0x0000, 0x547c,
	0x3e29, 0x547e, 0x4325, 0x0000, 0x547d, 0x0000, 0x4a33, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3d77, 0x455b, 0x0000, 0x0000, 0x0000,
	0x5521, 0x0000, 0x0000, 0x0000, 0x0000, 0x3925, 0x0000, 0x0000,
	0x0000, 0x5522, 0x4721, 0x485e, 0x4c51, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4725, 0x0000, 0x0000, 0x552b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3538, 0x0000, 0x0000, 0x4d45, 0x0000,
	0x0000, 0x4c2f, 0x0000, 0x562c, 0x0000, 0x5523, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5526, 0x0000, 0x4245, 0x0000, 0x0000,
	0x4b38, 0x0000, 0x0000, 0x0000, 0x454a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5527, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4b65, 0x0000, 0x3a4a, 0x0000, 0x0000, 0x3e2a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5528, 0x0000,
	0x0000, 0x3b50, 0x0000, 0x3b4f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3039, 0x3848, 0x0000, 0x402b, 0x3051, 0x0000, 0x0000, 0x0000,
	0x0000, 0x552c, 0x552d, 0x0000, 0x552a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3138, 0x342f, 0x0000,
	0x5529, 0x0000, 0x4c45, 0x4931, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3028, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3079, 0x0000, 0x0000, 0x0000, 0x3b51,
};

static unsigned short const unicode_to_jisx0208_5a[] = {
	/* 0x5a00 - 0x5aff */
	0x0000, 0x3052, 0x0000, 0x3023, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5532, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5530, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c3c, 0x0000, 0x5533, 0x0000, 0x5531, 0x0000, 0x0000, 0x552f,
	0x3f31, 0x0000, 0x0000, 0x0000, 0x0000, 0x552e, 0x0000, 0x0000,
	0x0000, 0x4a5a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3864,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5537, 0x5538, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3e2b, 0x0000, 0x0000, 0x0000,
	0x5534, 0x4f2c, 0x0000, 0x0000, 0x0000, 0x0000, 0x474c, 0x0000,
	0x0000, 0x5536, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a27, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5539, 0x0000, 0x0000, 0x0000, 0x4958, 0x0000,
	0x0000, 0x0000, 0x553a, 0x0000, 0x5535, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c3b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x475e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x553b, 0x4932, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x553c, 0x5540, 0x553d, 0x0000,
	0x0000, 0x3247, 0x553f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c3b, 0x0000, 0x553e, 0x3779, 0x0000, 0x0000, 0x0000,
	0x554c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5545, 0x5542,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4364, 0x0000, 0x5541, 0x0000, 0x0000, 0x5543, 0x0000,
	0x0000, 0x5544, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5546, 0x5547, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_5b[] = {
	/* 0x5b00 - 0x5bff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3472, 0x0000, 0x5549, 0x5548, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x554a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3e6e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x554d, 0x0000, 0x445c, 0x0000, 0x0000, 0x0000,
	0x3145, 0x0000, 0x554b, 0x0000, 0x0000, 0x0000, 0x554e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x554f, 0x0000,
	0x5552, 0x0000, 0x0000, 0x5550, 0x0000, 0x5551, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3b52, 0x5553, 0x0000, 0x0000, 0x3926, 0x5554, 0x0000, 0x3b7a,
	0x4238, 0x0000, 0x5555, 0x5556, 0x3b5a, 0x3927, 0x0000, 0x4c52,
	0x0000, 0x0000, 0x0000, 0x3528, 0x3849, 0x5557, 0x3358, 0x0000,
	0x0000, 0x5558, 0x0000, 0x4239, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5559, 0x5623, 0x0000, 0x555a, 0x0000, 0x555b, 0x0000, 0x0000,
	0x555c, 0x0000, 0x555e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x555f, 0x0000, 0x0000, 0x5560, 0x0000, 0x4270, 0x0000, 0x3127,
	0x3c69, 0x3042, 0x0000, 0x4157, 0x3430, 0x3c35, 0x0000, 0x3928,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4566, 0x0000, 0x3d21,
	0x3431, 0x4368, 0x446a, 0x3038, 0x3539, 0x4a75, 0x0000, 0x3c42,
	0x0000, 0x0000, 0x3552, 0x406b, 0x3c3c, 0x4d28, 0x5561, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x355c, 0x0000,
	0x3a4b, 0x0000, 0x0000, 0x3332, 0x3163, 0x3e2c, 0x3248, 0x0000,
	0x5562, 0x4d46, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d49,
	0x0000, 0x0000, 0x3c64, 0x5563, 0x3473, 0x4652, 0x4c29, 0x5564,
	0x0000, 0x5565, 0x0000, 0x0000, 0x4959, 0x0000, 0x0000, 0x0000,
	0x5567, 0x0000, 0x3428, 0x3677, 0x5566, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3432, 0x0000, 0x3f32, 0x556b, 0x3b21,
	0x0000, 0x3249, 0x556a, 0x0000, 0x5568, 0x556c, 0x5569, 0x472b,
	0x5c4d, 0x3f33, 0x0000, 0x556d, 0x0000, 0x0000, 0x4e40, 0x0000,
	0x556e, 0x0000, 0x0000, 0x5570, 0x0000, 0x437e, 0x556f, 0x0000,
	0x4023, 0x0000, 0x3b7b, 0x0000, 0x0000, 0x0000, 0x4250, 0x3c77,
};

static unsigned short const unicode_to_jisx0208_5c[] = {
	/* 0x5c00 - 0x5cff */
	0x0000, 0x4975, 0x406c, 0x0000, 0x3c4d, 0x5571, 0x3e2d, 0x5572,
	0x5573, 0x3053, 0x423a, 0x3f52, 0x0000, 0x5574, 0x4633, 0x3e2e,
	0x0000, 0x3e2f, 0x0000, 0x5575, 0x0000, 0x0000, 0x406d, 0x0000,
	0x0000, 0x0000, 0x3e30, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5576, 0x0000, 0x5577, 0x0000, 0x4c60, 0x0000, 0x0000, 0x0000,
	0x5578, 0x0000, 0x0000, 0x0000, 0x0000, 0x3646, 0x0000, 0x0000,
	0x0000, 0x3d22, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5579, 0x557a, 0x3c5c, 0x3f2c, 0x4674, 0x3f54, 0x4878, 0x4722,
	0x3649, 0x557b, 0x0000, 0x0000, 0x0000, 0x356f, 0x557c, 0x0000,
	0x367e, 0x0000, 0x464f, 0x3230, 0x0000, 0x3b53, 0x557d, 0x5622,
	0x5621, 0x367d, 0x0000, 0x557e, 0x0000, 0x4538, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4230, 0x0000,
	0x454b, 0x3c48, 0x0000, 0x0000, 0x4158, 0x4d7a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5624, 0x0000, 0x5625, 0x4656,
	0x0000, 0x3b33, 0x0000, 0x0000, 0x0000, 0x0000, 0x5627, 0x0000,
	0x0000, 0x5628, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5629, 0x0000, 0x0000, 0x0000,
	0x3474, 0x562a, 0x0000, 0x0000, 0x562b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x322c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x413b, 0x3464, 0x0000, 0x562d, 0x4c28, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4252, 0x0000, 0x3359, 0x0000, 0x0000, 0x562f, 0x5631,
	0x345f, 0x0000, 0x0000, 0x562e, 0x5630, 0x0000, 0x5633, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5632, 0x0000, 0x5634,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5635, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x463d, 0x362e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3265, 0x5636, 0x563b, 0x0000, 0x0000, 0x5639, 0x0000, 0x4a77,
	0x4a76, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4567, 0x0000,
	0x0000, 0x0000, 0x5638, 0x3d54, 0x0000, 0x5637, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_5d[] = {
	/* 0x5d00 - 0x5dff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f72,
	0x0000, 0x0000, 0x0000, 0x563c, 0x0000, 0x0000, 0x3a6a, 0x0000,
	0x0000, 0x5642, 0x0000, 0x0000, 0x5643, 0x563d, 0x3333, 0x563e,
	0x5647, 0x5646, 0x5645, 0x5641, 0x0000, 0x0000, 0x0000, 0x5640,
	0x0000, 0x0000, 0x5644, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a78, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x564b, 0x5648, 0x0000, 0x564a, 0x0000,
	0x4d72, 0x0000, 0x5649, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x563f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3f73, 0x0000, 0x0000, 0x564c, 0x0000, 0x0000, 0x3a37,
	0x0000, 0x0000, 0x0000, 0x564d, 0x0000, 0x0000, 0x564e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5651, 0x0000, 0x5650, 0x0000, 0x0000, 0x564f,
	0x0000, 0x0000, 0x0000, 0x4568, 0x563a, 0x0000, 0x0000, 0x0000,
	0x5657, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5653, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5652, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5654, 0x0000, 0x5655, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5658,
	0x0000, 0x0000, 0x4e66, 0x0000, 0x5659, 0x5656, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x565a, 0x0000, 0x0000, 0x3460, 0x565b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x565d, 0x565c, 0x0000, 0x0000, 0x565e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x565f, 0x0000, 0x406e, 0x3d23, 0x0000,
	0x0000, 0x3d64, 0x0000, 0x4163, 0x0000, 0x3929, 0x3a38, 0x392a,
	0x3570, 0x0000, 0x0000, 0x5660, 0x0000, 0x0000, 0x3a39, 0x0000,
	0x0000, 0x384a, 0x5661, 0x4c26, 0x4743, 0x5662, 0x0000, 0x392b,
	0x0000, 0x0000, 0x0000, 0x342c, 0x0000, 0x4327, 0x3652, 0x0000,
};

static unsigned short const unicode_to_jisx0208_5e[] = {
	/* 0x5e00 - 0x5eff */
	0x0000, 0x0000, 0x3b54, 0x495b, 0x0000, 0x0000, 0x4841, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5663, 0x3475, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5666, 0x0000, 0x0000, 0x0000, 0x0000, 0x4421, 0x0000,
	0x0000, 0x5665, 0x5664, 0x5667, 0x0000, 0x446b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f63, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b55, 0x0000, 0x404a, 0x0000, 0x4253,
	0x3522, 0x0000, 0x0000, 0x4422, 0x0000, 0x0000, 0x5668, 0x5669,
	0x3e6f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b39, 0x0000, 0x0000,
	0x566c, 0x0000, 0x0000, 0x566b, 0x566a, 0x497d, 0x0000, 0x5673,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4b5a, 0x0000, 0x566d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x566f, 0x4b6b, 0x0000, 0x566e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5670,
	0x0000, 0x4828, 0x5671, 0x4a3e, 0x5672, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3433, 0x4a3f, 0x472f, 0x5674, 0x5675, 0x0000,
	0x392c, 0x3434, 0x5676, 0x3838, 0x4d44, 0x4d29, 0x3476, 0x5678,
	0x0000, 0x4423, 0x0000, 0x392d, 0x3e31, 0x0000, 0x0000, 0x485f,
	0x0000, 0x0000, 0x3e32, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d78,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x446c, 0x4a79, 0x4539,
	0x0000, 0x0000, 0x392e, 0x0000, 0x495c, 0x0000, 0x0000, 0x0000,
	0x5679, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4559, 0x3a42,
	0x0000, 0x0000, 0x0000, 0x384b, 0x0000, 0x446d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3043, 0x3d6e, 0x392f,
	0x4d47, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x567a, 0x567b, 0x4751, 0x0000, 0x0000, 0x0000, 0x0000,
	0x567c, 0x4e77, 0x4f2d, 0x0000, 0x0000, 0x0000, 0x0000, 0x567e,
	0x567d, 0x0000, 0x0000, 0x3347, 0x0000, 0x0000, 0x5721, 0x0000,
	0x0000, 0x0000, 0x5724, 0x5725, 0x0000, 0x5723, 0x0000, 0x4940,
	0x3e33, 0x5727, 0x5726, 0x5722, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5728, 0x5729, 0x0000, 0x0000, 0x572a, 0x0000, 0x0000, 0x0000,
	0x572d, 0x572b, 0x0000, 0x572c, 0x572e, 0x0000, 0x3164, 0x446e,
	0x572f, 0x0000, 0x377a, 0x3276, 0x4736, 0x0000, 0x5730, 0x467b,
};

static unsigned short const unicode_to_jisx0208_5f[] = {
	/* 0x5f00 - 0x5fff */
	0x0000, 0x4a5b, 0x0000, 0x5731, 0x4f2e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5732, 0x4a40, 0x5735, 0x5021, 0x5031, 0x0000, 0x3c30,
	0x4675, 0x5736, 0x0000, 0x355d, 0x4424, 0x307a, 0x5737, 0x4a26,
	0x3930, 0x0000, 0x0000, 0x4350, 0x0000, 0x0000, 0x0000, 0x446f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c6f, 0x3839, 0x384c,
	0x0000, 0x5738, 0x0000, 0x0000, 0x0000, 0x5739, 0x0000, 0x573f,
	0x0000, 0x3c65, 0x0000, 0x0000, 0x0000, 0x4425, 0x0000, 0x362f,
	0x573a, 0x0000, 0x0000, 0x0000, 0x492b, 0x0000, 0x4346, 0x0000,
	0x0000, 0x573b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x573c, 0x0000, 0x3630, 0x0000, 0x573d, 0x0000, 0x573e, 0x0000,
	0x0000, 0x5740, 0x0000, 0x4576, 0x0000, 0x0000, 0x5741, 0x5742,
	0x0000, 0x5743, 0x0000, 0x0000, 0x5734, 0x5733, 0x0000, 0x0000,
	0x0000, 0x5744, 0x3741, 0x0000, 0x0000, 0x0000, 0x4927, 0x0000,
	0x0000, 0x3a4c, 0x4937, 0x4426, 0x494b, 0x5745, 0x0000, 0x0000,
	0x3e34, 0x3146, 0x0000, 0x5746, 0x0000, 0x0000, 0x0000, 0x5747,
	0x0000, 0x4c72, 0x0000, 0x0000, 0x4860, 0x0000, 0x0000, 0x574a,
	0x317d, 0x402c, 0x5749, 0x5748, 0x3742, 0x4254, 0x0000, 0x574e,
	0x574c, 0x0000, 0x574b, 0x4e27, 0x3865, 0x0000, 0x0000, 0x0000,
	0x3d79, 0x574d, 0x454c, 0x3d3e, 0x0000, 0x0000, 0x0000, 0x4640,
	0x5751, 0x5750, 0x0000, 0x0000, 0x0000, 0x0000, 0x574f, 0x0000,
	0x5752, 0x3866, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5753, 0x497c, 0x3d5b, 0x0000, 0x0000, 0x5754, 0x4879, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4641, 0x4427, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4530, 0x0000, 0x0000, 0x5755, 0x352b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3f34, 0x0000, 0x492c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3477, 0x4726, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5756, 0x3b56,
	0x4b3a, 0x4b3b, 0x0000, 0x0000, 0x317e, 0x575b, 0x0000, 0x0000,
	0x4369, 0x0000, 0x0000, 0x0000, 0x5758, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3277, 0x0000, 0x0000, 0x0000, 0x0000,
	0x582d, 0x575a, 0x0000, 0x0000, 0x0000, 0x4730, 0x0000, 0x0000,
	0x5759, 0x0000, 0x0000, 0x5757, 0x0000, 0x397a, 0x0000, 0x575d,
};

static unsigned short const unicode_to_jisx0208_60[] = {
	/* 0x6000 - 0x60ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5763, 0x5769,
	0x5761, 0x0000, 0x455c, 0x0000, 0x0000, 0x5766, 0x495d, 0x0000,
	0x0000, 0x5760, 0x0000, 0x5765, 0x4e67, 0x3b57, 0x0000, 0x0000,
	0x4255, 0x575e, 0x0000, 0x0000, 0x0000, 0x355e, 0x5768, 0x402d,
	0x3165, 0x5762, 0x3278, 0x5767, 0x0000, 0x0000, 0x0000, 0x3631,
	0x0000, 0x5764, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x576a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x576c, 0x5776, 0x5774, 0x0000, 0x0000, 0x5771, 0x0000,
	0x0000, 0x0000, 0x5770, 0x4e78, 0x0000, 0x5772, 0x0000, 0x0000,
	0x3632, 0x0000, 0x3931, 0x0000, 0x0000, 0x3d7a, 0x0000, 0x0000,
	0x0000, 0x5779, 0x576b, 0x0000, 0x0000, 0x0000, 0x0000, 0x576f,
	0x575f, 0x0000, 0x327a, 0x5773, 0x5775, 0x4351, 0x0000, 0x0000,
	0x3a28, 0x3238, 0x576d, 0x5778, 0x5777, 0x3633, 0x0000, 0x4229,
	0x3366, 0x0000, 0x0000, 0x0000, 0x0000, 0x3743, 0x0000, 0x576e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x577a, 0x0000, 0x577d, 0x5821, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c3d, 0x0000, 0x5827, 0x4470, 0x577b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5825, 0x0000, 0x3279, 0x0000, 0x5823, 0x5824,
	0x0000, 0x0000, 0x577e, 0x5822, 0x0000, 0x0000, 0x0000, 0x3867,
	0x4d2a, 0x0000, 0x0000, 0x3435, 0x0000, 0x0000, 0x3159, 0x5826,
	0x0000, 0x473a, 0x302d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4861, 0x575c, 0x582c, 0x5830, 0x4c65, 0x0000,
	0x5829, 0x0000, 0x0000, 0x0000, 0x4569, 0x582e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e70, 0x582f, 0x4657,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4f47, 0x0000, 0x582b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5831, 0x0000, 0x397b, 0x0000, 0x404b, 0x0000, 0x0000, 0x3054,
	0x582a, 0x5828, 0x0000, 0x415a, 0x0000, 0x0000, 0x0000, 0x577c,
	0x3b34, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4246, 0x583d, 0x0000, 0x415b, 0x5838, 0x0000, 0x5835, 0x5836,
	0x0000, 0x3c66, 0x5839, 0x583c, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_61[] = {
	/* 0x6100 - 0x61ff */
	0x5837, 0x3d25, 0x0000, 0x583a, 0x0000, 0x0000, 0x5834, 0x0000,
	0x4c7c, 0x4c7b, 0x0000, 0x0000, 0x0000, 0x583e, 0x583f, 0x3055,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5833, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3672, 0x3026, 0x0000, 0x0000, 0x0000, 0x3436,
	0x0000, 0x583b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5843,
	0x5842, 0x0000, 0x0000, 0x0000, 0x5847, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5848, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5846, 0x5849, 0x5841, 0x5845,
	0x0000, 0x0000, 0x584a, 0x0000, 0x584b, 0x0000, 0x0000, 0x5840,
	0x3b7c, 0x0000, 0x5844, 0x4256, 0x3932, 0x5832, 0x3f35, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5858, 0x0000, 0x4a69, 0x0000, 0x0000,
	0x584e, 0x584f, 0x5850, 0x0000, 0x0000, 0x5857, 0x0000, 0x5856,
	0x0000, 0x0000, 0x4b7d, 0x3437, 0x0000, 0x5854, 0x0000, 0x3745,
	0x3334, 0x0000, 0x0000, 0x5851, 0x0000, 0x0000, 0x4e38, 0x5853,
	0x3056, 0x5855, 0x0000, 0x584c, 0x5852, 0x5859, 0x3744, 0x584d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d5d, 0x0000,
	0x0000, 0x0000, 0x4d2b, 0x0000, 0x0000, 0x0000, 0x0000, 0x585c,
	0x0000, 0x0000, 0x5860, 0x0000, 0x0000, 0x0000, 0x417e, 0x0000,
	0x4e79, 0x5861, 0x0000, 0x0000, 0x585e, 0x0000, 0x585b, 0x0000,
	0x0000, 0x585a, 0x585f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4a30, 0x0000, 0x0000, 0x4634,
	0x0000, 0x3746, 0x0000, 0x5862, 0x585d, 0x0000, 0x5863, 0x0000,
	0x0000, 0x0000, 0x377b, 0x0000, 0x0000, 0x0000, 0x3231, 0x0000,
	0x0000, 0x0000, 0x586b, 0x0000, 0x0000, 0x0000, 0x3438, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5869, 0x0000, 0x0000, 0x586a, 0x3a29,
	0x5868, 0x5866, 0x5865, 0x586c, 0x5864, 0x586e, 0x0000, 0x0000,
	0x327b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5870, 0x0000, 0x0000, 0x586f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4428, 0x0000, 0x5873, 0x0000, 0x5871, 0x5867,
	0x377c, 0x0000, 0x5872, 0x0000, 0x5876, 0x5875, 0x5877, 0x5874,
};

static unsigned short const unicode_to_jisx0208_62[] = {
	/* 0x6200 - 0x62ff */
	0x5878, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5879, 0x587a, 0x4a6a, 0x0000, 0x587c, 0x587b, 0x3d3f, 0x0000,
	0x402e, 0x3266, 0x327c, 0x0000, 0x587d, 0x0000, 0x303f, 0x0000,
	0x0000, 0x0000, 0x404c, 0x587e, 0x0000, 0x6c43, 0x5921, 0x3761,
	0x0000, 0x5922, 0x0000, 0x0000, 0x0000, 0x0000, 0x406f, 0x0000,
	0x0000, 0x0000, 0x5923, 0x0000, 0x0000, 0x0000, 0x5924, 0x353a,
	0x5925, 0x0000, 0x5926, 0x5927, 0x4257, 0x0000, 0x0000, 0x0000,
	0x384d, 0x0000, 0x0000, 0x4c61, 0x0000, 0x0000, 0x0000, 0x4b3c,
	0x3d6a, 0x5928, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4070,
	0x6e3d, 0x4862, 0x0000, 0x3c6a, 0x0000, 0x3a4d, 0x5929, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4247, 0x0000, 0x4a27, 0x0000, 0x0000,
	0x4271, 0x0000, 0x0000, 0x592c, 0x0000, 0x0000, 0x592a, 0x0000,
	0x592d, 0x0000, 0x0000, 0x592b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x592e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a31, 0x0000,
	0x0000, 0x3037, 0x0000, 0x0000, 0x0000, 0x0000, 0x495e, 0x0000,
	0x0000, 0x4863, 0x0000, 0x0000, 0x592f, 0x0000, 0x5932, 0x3e35,
	0x353b, 0x0000, 0x5930, 0x5937, 0x3e36, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5931, 0x4744, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4d5e, 0x5933, 0x5934, 0x5938, 0x456a, 0x5935, 0x3933,
	0x405e, 0x0000, 0x0000, 0x5946, 0x4834, 0x0000, 0x4272, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4864, 0x5a2d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a7a, 0x0000, 0x0000, 0x0000, 0x4471, 0x0000, 0x0000,
	0x0000, 0x4b75, 0x0000, 0x593b, 0x3221, 0x436a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5944, 0x0000, 0x0000, 0x4334, 0x593e, 0x5945,
	0x5940, 0x5947, 0x5943, 0x0000, 0x5942, 0x476f, 0x0000, 0x593c,
	0x327d, 0x593a, 0x3571, 0x4273, 0x5936, 0x0000, 0x0000, 0x5939,
	0x3934, 0x405b, 0x0000, 0x3e37, 0x5941, 0x4752, 0x0000, 0x0000,
	0x3572, 0x3348, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3367, 0x3f21, 0x5949, 0x594e,
	0x0000, 0x594a, 0x0000, 0x377d, 0x0000, 0x594f, 0x3b22, 0x3969,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d26, 0x593d,
};

static unsigned short const unicode_to_jisx0208_63[] = {
	/* 0x6300 - 0x63ff */
	0x0000, 0x3b7d, 0x594c, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b58,
	0x594d, 0x3044, 0x0000, 0x0000, 0x5948, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4429, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3573, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3634,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x594b,
	0x3027, 0x0000, 0x0000, 0x3a43, 0x0000, 0x0000, 0x0000, 0x3f36,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4472, 0x0000, 0x0000, 0x4854, 0x5951, 0x415e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x422a, 0x0000, 0x0000, 0x3b2b, 0x5952, 0x0000, 0x5954,
	0x5950, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a61, 0x0000, 0x443d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x415c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a7b,
	0x3c4e, 0x5960, 0x0000, 0x595f, 0x0000, 0x0000, 0x3f78, 0x0000,
	0x0000, 0x0000, 0x377e, 0x0000, 0x0000, 0x0000, 0x5959, 0x3e39,
	0x0000, 0x0000, 0x4668, 0x4731, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5957, 0x0000, 0x0000, 0x415d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c78, 0x595c, 0x0000, 0x0000, 0x3e38, 0x0000, 0x5956, 0x595b,
	0x0000, 0x0000, 0x4753, 0x0000, 0x0000, 0x0000, 0x5955, 0x0000,
	0x3721, 0x0000, 0x0000, 0x335d, 0x0000, 0x0000, 0x0000, 0x595d,
	0x4e2b, 0x3a4e, 0x4335, 0x595a, 0x0000, 0x405c, 0x0000, 0x3935,
	0x3f64, 0x3166, 0x413c, 0x5958, 0x3545, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3747, 0x0000, 0x444f, 0x595e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x415f, 0x0000, 0x0000, 0x5961, 0x0000,
	0x5963, 0x0000, 0x0000, 0x4237, 0x5969, 0x0000, 0x5964, 0x0000,
	0x0000, 0x5966, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4941,
	0x4473, 0x0000, 0x5967, 0x0000, 0x0000, 0x0000, 0x4d2c, 0x0000,
	0x0000, 0x0000, 0x4d48, 0x3439, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x302e, 0x0000, 0x5965, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5962, 0x0000, 0x0000, 0x0000, 0x0000, 0x3478, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3167, 0x0000, 0x5968, 0x0000,
	0x0000, 0x0000, 0x4d49, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_64[] = {
	/* 0x6400 - 0x64ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x596c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x423b, 0x0000, 0x5973,
	0x0000, 0x0000, 0x0000, 0x596d, 0x0000, 0x0000, 0x596a, 0x5971,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5953, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x596e, 0x0000,
	0x5972, 0x0000, 0x0000, 0x0000, 0x4842, 0x456b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x596b, 0x0000, 0x596f, 0x0000,
	0x0000, 0x0000, 0x3748, 0x0000, 0x0000, 0x0000, 0x3a71, 0x0000,
	0x0000, 0x0000, 0x405d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5977, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4526, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5974,
	0x0000, 0x4b60, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5975,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5976, 0x0000,
	0x4c4e, 0x0000, 0x4022, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3762, 0x0000, 0x0000, 0x0000, 0x0000,
	0x597d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b35, 0x597a, 0x0000, 0x5979, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4732, 0x0000, 0x0000, 0x0000, 0x4635, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4531, 0x597b, 0x0000, 0x0000,
	0x0000, 0x597c, 0x0000, 0x496f, 0x0000, 0x4745, 0x3b23, 0x0000,
	0x4071, 0x0000, 0x4b50, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3349, 0x0000, 0x5a25, 0x597e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4d4a, 0x5a27, 0x0000, 0x0000, 0x5a23, 0x0000, 0x5a24,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4160, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5a22, 0x0000, 0x593f, 0x0000, 0x0000, 0x0000,
	0x5a26, 0x0000, 0x5a21, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5a2b, 0x5a2c, 0x4527, 0x5a2e, 0x0000, 0x0000, 0x3b24, 0x5a29,
	0x0000, 0x0000, 0x0000, 0x0000, 0x353c, 0x0000, 0x0000, 0x5a2f,
	0x0000, 0x5a28, 0x5a33, 0x0000, 0x5a32, 0x0000, 0x5a31, 0x0000,
	0x0000, 0x0000, 0x5a34, 0x0000, 0x0000, 0x5a36, 0x3e71, 0x0000,
};

static unsigned short const unicode_to_jisx0208_65[] = {
	/* 0x6500 - 0x65ff */
	0x5a35, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a39, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5a37, 0x0000, 0x0000, 0x0000, 0x5a38, 0x5970, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5a3b, 0x5a3a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5978, 0x5a3c, 0x5a30, 0x0000, 0x0000, 0x3b59,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5a3d, 0x5a3e, 0x5a40, 0x5a3f,
	0x5a41, 0x327e, 0x0000, 0x3936, 0x0000, 0x0000, 0x4a7c, 0x402f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x384e, 0x0000, 0x0000,
	0x5a43, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a46, 0x0000, 0x4952,
	0x0000, 0x355f, 0x0000, 0x0000, 0x0000, 0x5a45, 0x5a44, 0x4754,
	0x5a47, 0x3635, 0x0000, 0x0000, 0x0000, 0x5a49, 0x5a48, 0x0000,
	0x0000, 0x0000, 0x343a, 0x3b36, 0x0000, 0x0000, 0x4658, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3749, 0x0000, 0x0000, 0x0000,
	0x3f74, 0x0000, 0x5a4a, 0x0000, 0x4030, 0x4528, 0x0000, 0x495f,
	0x5a4b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5a4c, 0x5a4d, 0x0000, 0x0000, 0x0000, 0x4a38,
	0x555d, 0x4046, 0x0000, 0x0000, 0x494c, 0x0000, 0x3a58, 0x0000,
	0x4865, 0x4843, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x454d,
	0x0000, 0x4e41, 0x0000, 0x5a4f, 0x3c50, 0x0000, 0x0000, 0x5a50,
	0x0000, 0x3036, 0x0000, 0x0000, 0x3654, 0x404d, 0x0000, 0x4960,
	0x0000, 0x0000, 0x0000, 0x5a51, 0x3b42, 0x4347, 0x0000, 0x3b5b,
	0x3f37, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a52,
	0x0000, 0x4a7d, 0x0000, 0x0000, 0x3177, 0x3b5c, 0x0000, 0x0000,
	0x0000, 0x5a55, 0x0000, 0x5a53, 0x5a56, 0x4e39, 0x5a54, 0x0000,
	0x0000, 0x0000, 0x0000, 0x407b, 0x5a57, 0x0000, 0x0000, 0x4232,
	0x0000, 0x0000, 0x5a58, 0x0000, 0x0000, 0x0000, 0x0000, 0x347a,
	0x0000, 0x5a5a, 0x0000, 0x5a59, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5a5b, 0x5a5c, 0x347b, 0x0000, 0x0000, 0x467c, 0x4336, 0x356c,
	0x3b5d, 0x4161, 0x0000, 0x0000, 0x3d5c, 0x3030, 0x0000, 0x0000,
	0x0000, 0x5a5d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3222, 0x5a61, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_66[] = {
	/* 0x6600 - 0x66ff */
	0x0000, 0x0000, 0x3937, 0x5a60, 0x0000, 0x0000, 0x3a2b, 0x3e3a,
	0x0000, 0x0000, 0x5a5f, 0x0000, 0x3e3b, 0x0000, 0x4c40, 0x3a2a,
	0x0000, 0x0000, 0x0000, 0x3057, 0x404e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5a66, 0x0000, 0x0000, 0x4031,
	0x3147, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d55, 0x0000, 0x4b66,
	0x3a72, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e3c, 0x0000, 0x4027,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5a65, 0x5a63, 0x5a64, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x436b, 0x0000, 0x0000, 0x5b26,
	0x0000, 0x5a6a, 0x3b7e, 0x3938, 0x5a68, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5a69, 0x0000, 0x3f38, 0x0000, 0x0000, 0x0000, 0x5a67,
	0x0000, 0x0000, 0x3b2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a6c, 0x5a6b, 0x5a70,
	0x0000, 0x0000, 0x5a71, 0x0000, 0x5a6d, 0x0000, 0x3322, 0x5a6e,
	0x5a6f, 0x4855, 0x0000, 0x0000, 0x0000, 0x0000, 0x4961, 0x374a,
	0x5a72, 0x0000, 0x0000, 0x0000, 0x4032, 0x0000, 0x3e3d, 0x0000,
	0x0000, 0x0000, 0x4352, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3647, 0x0000, 0x5a73, 0x5a77, 0x0000, 0x0000, 0x324b,
	0x5a74, 0x5a76, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a75, 0x0000,
	0x0000, 0x3d6b, 0x0000, 0x0000, 0x0000, 0x0000, 0x4348, 0x3045,
	0x5a78, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a79, 0x0000, 0x0000,
	0x0000, 0x0000, 0x442a, 0x0000, 0x0000, 0x0000, 0x4e71, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b43, 0x0000, 0x0000, 0x4a6b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4b3d, 0x0000, 0x0000, 0x0000,
	0x5b22, 0x5a7b, 0x0000, 0x0000, 0x5a7e, 0x0000, 0x5a7d, 0x0000,
	0x0000, 0x5a7a, 0x0000, 0x0000, 0x5b21, 0x0000, 0x0000, 0x465e,
	0x0000, 0x5a7c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b23, 0x0000,
	0x0000, 0x3d6c, 0x5b24, 0x0000, 0x4d4b, 0x4778, 0x0000, 0x0000,
	0x5b25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b27, 0x0000,
	0x0000, 0x5b28, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b29, 0x0000, 0x364a, 0x3148, 0x3939, 0x5b2a, 0x0000, 0x5b2b,
	0x3d71, 0x4162, 0x0000, 0x0000, 0x5258, 0x413e, 0x413d, 0x4258,
};

static unsigned short const unicode_to_jisx0208_67[] = {
	/* 0x6700 - 0x67ff */
	0x3a47, 0x0000, 0x0000, 0x5072, 0x0000, 0x0000, 0x0000, 0x0000,
	0x376e, 0x4d2d, 0x0000, 0x4a7e, 0x0000, 0x497e, 0x0000, 0x5b2c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3a73, 0x443f, 0x5b2d, 0x4f2f,
	0x0000, 0x0000, 0x0000, 0x4b3e, 0x0000, 0x442b, 0x5b2e, 0x347c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b2f, 0x5b30,
	0x4c5a, 0x0000, 0x4c24, 0x4b76, 0x4b5c, 0x3b25, 0x5b32, 0x0000,
	0x0000, 0x3c6b, 0x0000, 0x0000, 0x4b51, 0x0000, 0x5b34, 0x5b37,
	0x5b36, 0x0000, 0x3479, 0x0000, 0x0000, 0x3560, 0x0000, 0x5b33,
	0x0000, 0x5b35, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b38, 0x0000,
	0x0000, 0x3f79, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d7b, 0x3049,
	0x3a60, 0x423c, 0x0000, 0x3c5d, 0x0000, 0x0000, 0x3e73, 0x0000,
	0x0000, 0x5b3b, 0x0000, 0x0000, 0x454e, 0x0000, 0x5b39, 0x422b,
	0x5b3a, 0x3e72, 0x4c5d, 0x5b3c, 0x5b3d, 0x4d68, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5b42, 0x0000, 0x0000, 0x393a, 0x0000, 0x4755,
	0x5b3f, 0x456c, 0x5a5e, 0x5a62, 0x0000, 0x354f, 0x0000, 0x4747,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5b41, 0x0000, 0x3e3e, 0x4844,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b47, 0x0000, 0x487a,
	0x0000, 0x5b3e, 0x0000, 0x5b44, 0x5b43, 0x0000, 0x0000, 0x0000,
	0x404f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b6d, 0x0000, 0x4e53,
	0x0000, 0x0000, 0x4b67, 0x0000, 0x324c, 0x3b5e, 0x0000, 0x0000,
	0x4f48, 0x5b46, 0x3f75, 0x0000, 0x0000, 0x0000, 0x5b45, 0x0000,
	0x0000, 0x5b40, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x384f,
	0x0000, 0x0000, 0x0000, 0x5b4c, 0x5b4a, 0x0000, 0x324d, 0x5b48,
	0x5b4e, 0x5b54, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4248, 0x0000, 0x0000, 0x4a41, 0x0000, 0x5b56, 0x0000,
	0x0000, 0x0000, 0x4922, 0x0000, 0x0000, 0x0000, 0x5b55, 0x4770,
	0x4b3f, 0x343b, 0x0000, 0x4077, 0x3d40, 0x0000, 0x0000, 0x0000,
	0x4453, 0x0000, 0x4d2e, 0x0000, 0x0000, 0x5b51, 0x5b50, 0x0000,
	0x0000, 0x0000, 0x5b52, 0x0000, 0x5b4f, 0x0000, 0x0000, 0x5b57,
	0x0000, 0x5b4d, 0x0000, 0x0000, 0x5b4b, 0x0000, 0x5b53, 0x5b49,
	0x0000, 0x436c, 0x0000, 0x4c78, 0x3c46, 0x3a74, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3a3a, 0x0000, 0x0000, 0x4b6f, 0x3341,
};

static unsigned short const unicode_to_jisx0208_68[] = {
	/* 0x6800 - 0x68ff */
	0x0000, 0x0000, 0x444e, 0x464a, 0x3149, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4072, 0x0000, 0x0000, 0x4034, 0x372a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b59, 0x0000,
	0x0000, 0x393b, 0x337c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5b5b, 0x3374, 0x5b61, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5b5e, 0x0000, 0x4073, 0x0000, 0x0000, 0x0000,
	0x334b, 0x3a2c, 0x0000, 0x0000, 0x334a, 0x3a4f, 0x0000, 0x0000,
	0x5b5c, 0x3765, 0x374b, 0x456d, 0x0000, 0x0000, 0x5b5a, 0x0000,
	0x3046, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b5d, 0x5b5f, 0x0000,
	0x364d, 0x372c, 0x0000, 0x343c, 0x354b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5b62, 0x0000, 0x0000, 0x3a79, 0x4b71, 0x0000, 0x3b37,
	0x0000, 0x0000, 0x0000, 0x5b63, 0x0000, 0x0000, 0x0000, 0x4930,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5b6f, 0x0000, 0x3233, 0x5b64,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b75, 0x5b65,
	0x0000, 0x4e42, 0x0000, 0x5b6c, 0x0000, 0x475f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b74, 0x0000, 0x5b67,
	0x0000, 0x0000, 0x0000, 0x3034, 0x5b69, 0x0000, 0x0000, 0x393c,
	0x0000, 0x0000, 0x0000, 0x5b6b, 0x0000, 0x5b6a, 0x0000, 0x5b66,
	0x5b71, 0x0000, 0x3e3f, 0x0000, 0x0000, 0x0000, 0x546d, 0x3868,
	0x4d7c, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b68, 0x0000, 0x4474,
	0x3323, 0x3a2d, 0x0000, 0x5b60, 0x0000, 0x5b70, 0x3361, 0x0000,
	0x0000, 0x5b6e, 0x5b72, 0x0000, 0x456e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x347e, 0x0000, 0x5c32, 0x0000,
	0x0000, 0x4c49, 0x5b77, 0x347d, 0x0000, 0x5b7e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4b40, 0x0000, 0x5c21, 0x5c23, 0x0000, 0x5c27,
	0x5b79, 0x0000, 0x432a, 0x0000, 0x0000, 0x0000, 0x0000, 0x456f,
	0x5c2b, 0x5b7c, 0x0000, 0x5c28, 0x0000, 0x0000, 0x0000, 0x5c22,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f39, 0x5c2c,
	0x0000, 0x0000, 0x4033, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c2a, 0x343d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_69[] = {
	/* 0x6900 - 0x69ff */
	0x4f50, 0x5b76, 0x0000, 0x0000, 0x5c26, 0x3058, 0x0000, 0x0000,
	0x5b78, 0x0000, 0x0000, 0x4c3a, 0x5b7d, 0x3f22, 0x4447, 0x5b73,
	0x0000, 0x0000, 0x5c25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3f7a, 0x5c2f, 0x3371, 0x3821, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c31, 0x5b7a, 0x5c30, 0x0000, 0x5c29, 0x5b7b, 0x0000,
	0x5c2d, 0x0000, 0x5c2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5c3f, 0x0000, 0x0000, 0x0000, 0x464e, 0x0000, 0x5c24, 0x0000,
	0x0000, 0x5c3b, 0x0000, 0x0000, 0x0000, 0x5c3d, 0x0000, 0x4458,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d4c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4976, 0x5c38, 0x424a, 0x0000, 0x0000,
	0x0000, 0x5c3e, 0x413f, 0x0000, 0x5c35, 0x5c42, 0x5c41, 0x0000,
	0x466f, 0x5c40, 0x466a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c44, 0x5c37, 0x0000, 0x3648, 0x5c3a, 0x3d5d,
	0x0000, 0x0000, 0x0000, 0x4760, 0x5c3c, 0x364b, 0x0000, 0x5c34,
	0x5c36, 0x5c33, 0x0000, 0x0000, 0x4f30, 0x335a, 0x5c39, 0x0000,
	0x0000, 0x5c43, 0x3335, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a67, 0x0000, 0x0000, 0x0000, 0x315d, 0x0000,
	0x0000, 0x5c54, 0x0000, 0x0000, 0x4f31, 0x5c57, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3f3a, 0x5c56, 0x0000, 0x0000, 0x0000,
	0x5c55, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c52,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c46, 0x0000,
	0x0000, 0x5c63, 0x5c45, 0x0000, 0x5c58, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5c50, 0x0000, 0x0000, 0x5c4b, 0x5c48,
	0x0000, 0x5c49, 0x0000, 0x5c51, 0x0000, 0x0000, 0x0000, 0x7422,
	0x0000, 0x0000, 0x5c4e, 0x393d, 0x4448, 0x4164, 0x5c4c, 0x0000,
	0x5c47, 0x0000, 0x0000, 0x5c4a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4d4d, 0x4b6a, 0x0000, 0x0000, 0x0000, 0x5c4f, 0x5c59, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c61,
	0x5c5a, 0x0000, 0x0000, 0x5c67, 0x0000, 0x5c65, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c60, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c5f, 0x0000, 0x4450, 0x0000, 0x4165, 0x0000, 0x5c5d,
};

static unsigned short const unicode_to_jisx0208_6a[] = {
	/* 0x6a00 - 0x6aff */
	0x0000, 0x0000, 0x5c5b, 0x0000, 0x0000, 0x5c62, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c68, 0x4875, 0x5c6e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c69, 0x5c6c, 0x5c66, 0x0000, 0x0000, 0x4374,
	0x0000, 0x4938, 0x0000, 0x5c5c, 0x0000, 0x0000, 0x5c64, 0x3e40,
	0x0000, 0x4c4f, 0x5c78, 0x5c6b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3822, 0x3223, 0x335f, 0x0000, 0x0000, 0x5c53, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e41, 0x5c70, 0x0000,
	0x5c77, 0x3c79, 0x3372, 0x0000, 0x0000, 0x432e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5c6d, 0x0000, 0x0000, 0x5c72,
	0x5c76, 0x0000, 0x0000, 0x3636, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x354c, 0x5c74, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3521,
	0x0000, 0x464b, 0x5c73, 0x0000, 0x0000, 0x0000, 0x5c75, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c6f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5c71, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3360,
	0x4349, 0x0000, 0x0000, 0x0000, 0x5c7c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c7a, 0x3869, 0x0000,
	0x5c79, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d21,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5b58, 0x0000, 0x0000, 0x0000,
	0x5c7b, 0x0000, 0x5c7d, 0x5c7e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5d2c, 0x0000, 0x5d28, 0x0000, 0x5b6d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d27, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5d26, 0x0000, 0x0000, 0x5d23, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c6a, 0x5d25, 0x5d24, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5d2a, 0x0000, 0x4f26, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5d2d, 0x367b, 0x0000, 0x0000, 0x5d29, 0x5d2b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4827, 0x0000, 0x5d2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5d32, 0x5d2f, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_6b[] = {
	/* 0x6b00 - 0x6bff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d73, 0x5d30, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c5e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5d33, 0x0000, 0x0000, 0x0000, 0x5d34, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3135, 0x0000, 0x5d36,
	0x3767, 0x3c21, 0x0000, 0x3655, 0x0000, 0x0000, 0x0000, 0x3224,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d5f, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d38,
	0x5d37, 0x5d3a, 0x353d, 0x0000, 0x0000, 0x3656, 0x343e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d3d, 0x0000, 0x0000, 0x0000, 0x5d3c,
	0x0000, 0x5d3e, 0x0000, 0x0000, 0x324e, 0x0000, 0x4337, 0x0000,
	0x5d3f, 0x0000, 0x0000, 0x343f, 0x5d41, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5d40, 0x0000, 0x5d42, 0x0000, 0x0000, 0x0000, 0x5d43,
	0x0000, 0x5d44, 0x3b5f, 0x4035, 0x3a21, 0x0000, 0x4970, 0x0000,
	0x0000, 0x4a62, 0x4f44, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b75,
	0x0000, 0x0000, 0x0000, 0x3a50, 0x4e72, 0x0000, 0x0000, 0x0000,
	0x5d45, 0x5d46, 0x0000, 0x3b60, 0x0000, 0x0000, 0x0000, 0x5d47,
	0x5d48, 0x0000, 0x0000, 0x5d4a, 0x5d49, 0x0000, 0x4b58, 0x0000,
	0x0000, 0x3d5e, 0x3c6c, 0x3b44, 0x0000, 0x5d4b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d4d, 0x3f23, 0x0000,
	0x5d4c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d4e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5d4f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5d50, 0x5d51, 0x0000, 0x0000, 0x0000, 0x5d52,
	0x0000, 0x5d54, 0x5d53, 0x5d55, 0x3225, 0x434a, 0x0000, 0x5d56,
	0x0000, 0x0000, 0x3b26, 0x334c, 0x5d57, 0x0000, 0x0000, 0x4542,
	0x544c, 0x0000, 0x0000, 0x0000, 0x0000, 0x3523, 0x5d58, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d59, 0x0000, 0x4a6c, 0x4b68, 0x0000,
	0x0000, 0x0000, 0x4647, 0x5d5a, 0x4866, 0x0000, 0x0000, 0x0000,
	0x487b, 0x0000, 0x0000, 0x4c53, 0x0000, 0x0000, 0x0000, 0x5d5b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d5d, 0x5d5c, 0x0000, 0x0000, 0x5d5f,
	0x0000, 0x0000, 0x0000, 0x5d5e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_6c[] = {
	/* 0x6c00 - 0x6cff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5d61, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b61,
	0x0000, 0x4c31, 0x0000, 0x5d62, 0x5d63, 0x0000, 0x0000, 0x3524,
	0x0000, 0x0000, 0x0000, 0x5d64, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d66, 0x5d65, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f65, 0x0000, 0x0000, 0x4939,
	0x314a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4845, 0x0000,
	0x4475, 0x3d41, 0x3561, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4846, 0x0000,
	0x3c2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d68, 0x0000, 0x3440,
	0x0000, 0x0000, 0x3178, 0x0000, 0x0000, 0x4672, 0x5d67, 0x393e,
	0x4353, 0x0000, 0x5d69, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5d71, 0x0000, 0x5d6a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4241, 0x0000, 0x3562, 0x5d72, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3768, 0x0000, 0x0000, 0x3525, 0x5d70, 0x0000,
	0x0000, 0x5d6e, 0x5d6b, 0x4d60, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4440, 0x0000, 0x0000, 0x0000, 0x4659, 0x5d6c, 0x0000, 0x0000,
	0x5d74, 0x0000, 0x5d73, 0x3723, 0x0000, 0x0000, 0x322d, 0x0000,
	0x0000, 0x3a3b, 0x5d6d, 0x5d6f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4b57, 0x4274, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4b77, 0x0000, 0x0000, 0x5d7c, 0x0000,
	0x0000, 0x5d7d, 0x0000, 0x324f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a28, 0x4c7d, 0x5e21, 0x3c23, 0x3e42, 0x5d78, 0x5d7e, 0x3168,
	0x0000, 0x3637, 0x0000, 0x0000, 0x5d75, 0x5d7a, 0x0000, 0x0000,
	0x0000, 0x4074, 0x4771, 0x0000, 0x4867, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d77, 0x0000, 0x4b21, 0x0000, 0x5d79,
	0x0000, 0x5e24, 0x0000, 0x5e22, 0x0000, 0x5d7b, 0x0000, 0x0000,
	0x0000, 0x4b22, 0x4748, 0x3563, 0x0000, 0x4525, 0x0000, 0x0000,
	0x436d, 0x0000, 0x5e25, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e23,
	0x4259, 0x5d76, 0x0000, 0x314b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_6d[] = {
	/* 0x6d00 - 0x6dff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4d4e, 0x5e30, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5e2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4076,
	0x0000, 0x5e2c, 0x0000, 0x4d6c, 0x0000, 0x0000, 0x4636, 0x5e26,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4445, 0x0000, 0x0000,
	0x0000, 0x314c, 0x393f, 0x5e29, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3d27, 0x5e2e, 0x0000, 0x5e2d, 0x5e28, 0x0000,
	0x5e2b, 0x0000, 0x0000, 0x3368, 0x0000, 0x5e2a, 0x4749, 0x0000,
	0x0000, 0x4e2e, 0x0000, 0x0000, 0x3e74, 0x4075, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5e36, 0x5e34, 0x0000, 0x494d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e31, 0x5e33, 0x0000, 0x313a, 0x0000,
	0x0000, 0x3940, 0x4f32, 0x0000, 0x333d, 0x0000, 0x4962, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d61, 0x0000, 0x0000, 0x3324,
	0x3f3b, 0x5e35, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e3a, 0x0000, 0x0000,
	0x3e43, 0x0000, 0x0000, 0x0000, 0x4d30, 0x0000, 0x5e37, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e32, 0x0000, 0x5e38, 0x0000, 0x0000,
	0x0000, 0x4e5e, 0x0000, 0x4573, 0x4642, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3336,
	0x0000, 0x0000, 0x3155, 0x0000, 0x0000, 0x5e3e, 0x0000, 0x0000,
	0x5e41, 0x0000, 0x0000, 0x0000, 0x4e43, 0x0000, 0x0000, 0x0000,
	0x4d64, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e48, 0x5e42, 0x5e3f,
	0x0000, 0x0000, 0x0000, 0x4e54, 0x5e45, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3d4a, 0x5e47, 0x0000, 0x0000, 0x5e4c, 0x0000, 0x0000,
	0x4571, 0x5e4a, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e44, 0x0000,
	0x0000, 0x4338, 0x0000, 0x0000, 0x5e4b, 0x0000, 0x5e40, 0x0000,
	0x5e46, 0x0000, 0x5e4d, 0x307c, 0x5e43, 0x0000, 0x5e4e, 0x0000,
	0x0000, 0x3f3c, 0x0000, 0x3d5f, 0x0000, 0x4a25, 0x0000, 0x3a2e,
	0x0000, 0x5e3b, 0x5e49, 0x453a, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_6e[] = {
	/* 0x6e00 - 0x6eff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4036, 0x0000, 0x3369,
	0x3a51, 0x3e44, 0x5e3d, 0x3d42, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x374c, 0x0000, 0x5e3c, 0x0000, 0x0000,
	0x0000, 0x5e52, 0x3d6d, 0x383a, 0x0000, 0x5e61, 0x0000, 0x5e5b,
	0x3574, 0x454f, 0x0000, 0x5e56, 0x5e5f, 0x302f, 0x3132, 0x0000,
	0x0000, 0x3239, 0x0000, 0x5e58, 0x422c, 0x5e4f, 0x5e51, 0x3941,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5e62, 0x0000, 0x5e5d, 0x0000, 0x0000, 0x0000, 0x5e55, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e5c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c2b, 0x0000, 0x0000, 0x5e5a, 0x5e5e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3850, 0x0000,
	0x3e45, 0x0000, 0x0000, 0x4339, 0x0000, 0x0000, 0x0000, 0x5e54,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d2f,
	0x0000, 0x0000, 0x0000, 0x5e57, 0x0000, 0x0000, 0x5e50, 0x4572,
	0x0000, 0x0000, 0x5e53, 0x0000, 0x0000, 0x0000, 0x5e59, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f51, 0x3c3e,
	0x4b7e, 0x0000, 0x5e63, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x482e, 0x0000, 0x0000, 0x5e6f,
	0x383b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d60, 0x0000,
	0x5e65, 0x0000, 0x0000, 0x0000, 0x4e2f, 0x3942, 0x0000, 0x5e72,
	0x0000, 0x0000, 0x306e, 0x0000, 0x0000, 0x5e70, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5e64, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e6a,
	0x0000, 0x0000, 0x5e6c, 0x0000, 0x0000, 0x0000, 0x4d4f, 0x5e67,
	0x0000, 0x0000, 0x452e, 0x0000, 0x0000, 0x5e69, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5e71, 0x0000, 0x5e6b, 0x4c47, 0x0000, 0x0000,
	0x0000, 0x5e66, 0x0000, 0x3c22, 0x5e7e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x336a, 0x0000, 0x5e68, 0x5e6d, 0x5e6e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x426c, 0x425a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e76, 0x0000, 0x0000, 0x5e7c,
	0x0000, 0x0000, 0x5e7a, 0x0000, 0x4529, 0x0000, 0x0000, 0x5f23,
	0x5e77, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e78, 0x5e60,
};

static unsigned short const unicode_to_jisx0208_6f[] = {
	/* 0x6f00 - 0x6fff */
	0x0000, 0x3579, 0x493a, 0x0000, 0x0000, 0x0000, 0x3c3f, 0x0000,
	0x0000, 0x3977, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f33,
	0x0000, 0x5e74, 0x0000, 0x5f22, 0x3169, 0x4166, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4779, 0x0000, 0x3441, 0x4e7a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4c21, 0x4452, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5e7b, 0x5e7d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4132, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f21, 0x5e79,
	0x0000, 0x5e73, 0x0000, 0x0000, 0x0000, 0x3443, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3769, 0x0000, 0x0000, 0x0000,
	0x5f2f, 0x0000, 0x0000, 0x5f2a, 0x4078, 0x0000, 0x0000, 0x3363,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d61, 0x0000, 0x5f33, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f2c, 0x442c, 0x5f29,
	0x4459, 0x0000, 0x0000, 0x0000, 0x5f4c, 0x0000, 0x0000, 0x0000,
	0x5f26, 0x0000, 0x5f25, 0x0000, 0x5f2e, 0x0000, 0x0000, 0x0000,
	0x5f28, 0x5f27, 0x5f2d, 0x0000, 0x4021, 0x0000, 0x5f24, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f30, 0x0000,
	0x0000, 0x5f31, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3442,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f36, 0x0000, 0x5f35, 0x5f37, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5f3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4543, 0x0000, 0x5f34, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f38, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3763, 0x4279, 0x5f32, 0x473b, 0x0000, 0x0000, 0x5f39, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f3e, 0x5f3c, 0x0000, 0x0000,
	0x5f3f, 0x0000, 0x0000, 0x5f42, 0x0000, 0x0000, 0x0000, 0x5f3b,
	0x396a, 0x4728, 0x0000, 0x0000, 0x5e39, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4d74, 0x5f3d, 0x0000, 0x5f41, 0x4275,
	0x0000, 0x5f40, 0x0000, 0x5f2b, 0x0000, 0x0000, 0x6f69, 0x0000,
	0x0000, 0x0000, 0x5f45, 0x0000, 0x0000, 0x0000, 0x5f49, 0x0000,
};

static unsigned short const unicode_to_jisx0208_70[] = {
	/* 0x7000 - 0x70ff */
	0x0000, 0x5f47, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f43, 0x0000, 0x5f44, 0x0000, 0x0000, 0x0000, 0x5f48,
	0x0000, 0x5f46, 0x0000, 0x0000, 0x0000, 0x494e, 0x0000, 0x0000,
	0x5f4e, 0x0000, 0x5f4b, 0x5f4a, 0x0000, 0x5f4d, 0x4654, 0x5f4f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4375, 0x426d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4025, 0x0000, 0x0000, 0x0000,
	0x5f50, 0x0000, 0x5f52, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f51, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e75, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f53, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4667, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5f54, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3250, 0x0000, 0x0000, 0x0000, 0x4574,
	0x3325, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3564, 0x0000, 0x0000, 0x0000, 0x3c5e, 0x3a52, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4f27, 0x3f66, 0x0000, 0x0000, 0x0000, 0x316a, 0x0000,
	0x0000, 0x0000, 0x5f56, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f55, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f59, 0x433a, 0x5f5c, 0x5f57,
	0x0000, 0x0000, 0x0000, 0x5f5b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5f5a, 0x4540, 0x3059, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4e75, 0x0000, 0x0000, 0x5f5e, 0x0000, 0x0000, 0x0000, 0x3128,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f60, 0x0000, 0x0000, 0x0000, 0x5f5f, 0x0000, 0x5f5d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f58, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4b23, 0x0000, 0x0000, 0x0000, 0x5f62, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_71[] = {
	/* 0x7100 - 0x71ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f61, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x316b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f64, 0x4a32, 0x0000, 0x5f63, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4c35, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e47, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4133, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3e46, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4e7b, 0x0000, 0x0000, 0x5f6a, 0x0000, 0x4079, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f66, 0x5f6b, 0x0000,
	0x0000, 0x316c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5f69, 0x0000, 0x4761, 0x5f65, 0x5f68, 0x3e48,
	0x0000, 0x4851, 0x0000, 0x0000, 0x5f6c, 0x0000, 0x3c51, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x407a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f6f, 0x0000, 0x0000, 0x0000,
	0x5f67, 0x0000, 0x3727, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f6d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d50, 0x5f70, 0x0000, 0x0000,
	0x0000, 0x7426, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d4f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5f71, 0x0000, 0x0000, 0x0000, 0x5f72, 0x0000, 0x0000, 0x0000,
	0x0000, 0x472e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f74, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f75, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4733, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4575, 0x5f77, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f79, 0x0000,
	0x4e55, 0x0000, 0x5f76, 0x0000, 0x5f78, 0x316d, 0x0000, 0x5f73,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x535b,
	0x5f7a, 0x0000, 0x0000, 0x0000, 0x0000, 0x4167, 0x3b38, 0x5f7c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f7b, 0x3f24, 0x5259, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f7d, 0x0000, 0x0000,
	0x0000, 0x6021, 0x0000, 0x5f6e, 0x5f7e, 0x0000, 0x0000, 0x6022,
};

static unsigned short const unicode_to_jisx0208_72[] = {
	/* 0x7200 - 0x72ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x477a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6023, 0x0000, 0x0000,
	0x6024, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6025, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6026, 0x0000, 0x445e, 0x0000, 0x6028, 0x6027, 0x0000, 0x0000,
	0x6029, 0x0000, 0x602a, 0x0000, 0x0000, 0x3c5f, 0x4963, 0x0000,
	0x0000, 0x0000, 0x4c6c, 0x602b, 0x602c, 0x4156, 0x3c24, 0x602d,
	0x602e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x602f, 0x4a52,
	0x4847, 0x0000, 0x0000, 0x6030, 0x4757, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x442d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6031, 0x3267, 0x0000, 0x356d, 0x0000, 0x4c46, 0x0000, 0x4c36,
	0x0000, 0x3234, 0x4f34, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b52,
	0x0000, 0x4a2a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4037, 0x0000, 0x6032, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4643, 0x0000, 0x0000, 0x0000, 0x3823, 0x6033, 0x0000,
	0x3a54, 0x6035, 0x6034, 0x0000, 0x0000, 0x0000, 0x0000, 0x6036,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6037, 0x0000, 0x0000, 0x0000, 0x6038, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x353e, 0x0000, 0x6039, 0x0000, 0x0000, 0x0000, 0x0000, 0x603a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3824, 0x0000, 0x0000, 0x4848,
	0x0000, 0x0000, 0x603c, 0x0000, 0x0000, 0x0000, 0x3e75, 0x0000,
	0x0000, 0x603b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3638, 0x603d, 0x603f, 0x0000, 0x603e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6040, 0x0000,
	0x3851, 0x0000, 0x6041, 0x0000, 0x0000, 0x0000, 0x0000, 0x3669,
	0x0000, 0x4140, 0x0000, 0x397d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6043, 0x6044, 0x6042, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c6d, 0x0000, 0x0000, 0x4648, 0x3639, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6046,
	0x432c, 0x6045, 0x0000, 0x0000, 0x4f35, 0x4762, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_73[] = {
	/* 0x7300 - 0x73ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x604b, 0x6048,
	0x0000, 0x0000, 0x0000, 0x4c54, 0x604a, 0x604c, 0x0000, 0x4e44,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6050, 0x0000, 0x0000,
	0x0000, 0x604f, 0x4376, 0x472d, 0x0000, 0x0000, 0x3825, 0x604e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x604d, 0x0000, 0x4d31, 0x4d32,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6051, 0x316e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3976, 0x3b62, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6052, 0x6053,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6055,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3d43, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6057, 0x0000, 0x6056, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6058, 0x0000, 0x334d, 0x0000, 0x0000, 0x605a, 0x0000, 0x0000,
	0x6059, 0x0000, 0x605c, 0x605b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x383c, 0x0000, 0x0000, 0x4e28,
	0x0000, 0x364c, 0x0000, 0x3226, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x366a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3461, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4e68, 0x605e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6060, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6061, 0x0000, 0x3251, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x605d, 0x0000, 0x3b39, 0x0000, 0x0000, 0x4441, 0x605f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6064, 0x0000,
	0x3c6e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6062, 0x0000, 0x0000,
	0x0000, 0x0000, 0x373e, 0x0000, 0x0000, 0x4849, 0x6063, 0x0000,
	0x0000, 0x607e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6069, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x383d, 0x0000,
};

static unsigned short const unicode_to_jisx0208_74[] = {
	/* 0x7400 - 0x74ff */
	0x0000, 0x0000, 0x0000, 0x3565, 0x0000, 0x6066, 0x4d7d, 0x0000,
	0x0000, 0x4e30, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4276, 0x0000, 0x0000, 0x6068, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x606a, 0x4e56, 0x3657, 0x487c, 0x474a, 0x0000,
	0x0000, 0x0000, 0x606b, 0x0000, 0x0000, 0x0000, 0x0000, 0x606d,
	0x0000, 0x6070, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x606c, 0x0000, 0x0000,
	0x0000, 0x606f, 0x386a, 0x314d, 0x6071, 0x0000, 0x3f70, 0x606e,
	0x4e5c, 0x0000, 0x0000, 0x6074, 0x7424, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6072, 0x6075, 0x0000, 0x0000, 0x0000, 0x0000, 0x6067,
	0x6073, 0x0000, 0x0000, 0x3a3c, 0x0000, 0x0000, 0x6076, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6077, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4d7e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6078, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6079, 0x0000,
	0x0000, 0x0000, 0x6065, 0x0000, 0x0000, 0x0000, 0x0000, 0x607a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3444, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c25, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x607b, 0x0000, 0x0000, 0x0000, 0x0000, 0x607c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x607d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x313b, 0x0000, 0x0000, 0x0000,
	0x6121, 0x0000, 0x493b, 0x6122, 0x0000, 0x0000, 0x3424, 0x6123,
	0x0000, 0x6124, 0x0000, 0x0000, 0x0000, 0x0000, 0x6125, 0x0000,
	0x6127, 0x6128, 0x6126, 0x0000, 0x0000, 0x0000, 0x4953, 0x612a,
	0x6129, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_75[] = {
	/* 0x7500 - 0x75ff */
	0x0000, 0x0000, 0x0000, 0x612c, 0x612b, 0x612d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x612e, 0x6130, 0x612f, 0x0000,
	0x0000, 0x3979, 0x0000, 0x6132, 0x0000, 0x6131, 0x0000, 0x0000,
	0x3445, 0x0000, 0x3f53, 0x0000, 0x453c, 0x0000, 0x6133, 0x4038,
	0x0000, 0x0000, 0x0000, 0x3b3a, 0x0000, 0x3179, 0x6134, 0x0000,
	0x4d51, 0x0000, 0x0000, 0x4a63, 0x6135, 0x0000, 0x0000, 0x0000,
	0x4544, 0x4d33, 0x3943, 0x3f3d, 0x0000, 0x0000, 0x0000, 0x434b,
	0x5234, 0x0000, 0x442e, 0x3268, 0x6136, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6137, 0x0000, 0x613c, 0x0000,
	0x0000, 0x613a, 0x6139, 0x5a42, 0x3326, 0x6138, 0x0000, 0x305a,
	0x0000, 0x482a, 0x0000, 0x0000, 0x484a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4e31, 0x613d, 0x613b, 0x435c, 0x4026, 0x0000, 0x0000,
	0x482b, 0x0000, 0x492d, 0x0000, 0x613f, 0x4e2c, 0x374d, 0x6140,
	0x0000, 0x613e, 0x4856, 0x6141, 0x0000, 0x6142, 0x0000, 0x0000,
	0x305b, 0x0000, 0x0000, 0x3e76, 0x6147, 0x0000, 0x6144, 0x466d,
	0x6143, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3526,
	0x0000, 0x0000, 0x614a, 0x0000, 0x0000, 0x0000, 0x6145, 0x6146,
	0x0000, 0x6149, 0x6148, 0x4925, 0x0000, 0x0000, 0x4142, 0x4141,
	0x0000, 0x353f, 0x0000, 0x0000, 0x614b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x614c, 0x0000, 0x0000, 0x614d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x614f, 0x0000, 0x614e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3156, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6157, 0x4868, 0x6151, 0x0000, 0x6153, 0x0000, 0x0000,
	0x6155, 0x3f3e, 0x0000, 0x0000, 0x6156, 0x6154, 0x3c40, 0x0000,
	0x0000, 0x0000, 0x6150, 0x6152, 0x0000, 0x4942, 0x0000, 0x3e49,
	0x0000, 0x0000, 0x6159, 0x0000, 0x0000, 0x6158, 0x0000, 0x0000,
	0x0000, 0x0000, 0x615a, 0x0000, 0x3c26, 0x3a2f, 0x0000, 0x0000,
	0x4577, 0x615b, 0x0000, 0x444b, 0x0000, 0x0000, 0x615d, 0x0000,
	0x0000, 0x0000, 0x4e21, 0x615c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4169, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6162, 0x0000, 0x6164, 0x6165, 0x4354, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6163, 0x0000, 0x6160, 0x0000, 0x615e, 0x615f,
};

static unsigned short const unicode_to_jisx0208_76[] = {
	/* 0x7600 - 0x76ff */
	0x0000, 0x6161, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6168, 0x0000, 0x6166, 0x0000, 0x6167, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6169,
	0x616b, 0x616c, 0x616d, 0x0000, 0x616e, 0x0000, 0x0000, 0x616a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6170, 0x0000, 0x0000, 0x0000, 0x616f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6171, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4e45, 0x0000, 0x0000, 0x0000, 0x6174, 0x6172,
	0x6173, 0x0000, 0x0000, 0x0000, 0x3462, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c7e, 0x0000, 0x0000, 0x0000, 0x4a4a, 0x0000,
	0x6176, 0x0000, 0x0000, 0x0000, 0x6175, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6177, 0x6178, 0x0000, 0x0000, 0x0000, 0x0000, 0x617c,
	0x6179, 0x617a, 0x617b, 0x0000, 0x617d, 0x0000, 0x0000, 0x0000,
	0x617e, 0x0000, 0x6221, 0x0000, 0x0000, 0x0000, 0x6222, 0x0000,
	0x6223, 0x0000, 0x482f, 0x4550, 0x6224, 0x4772, 0x4934, 0x0000,
	0x6225, 0x0000, 0x0000, 0x6226, 0x452a, 0x0000, 0x3327, 0x3944,
	0x6227, 0x0000, 0x0000, 0x6228, 0x0000, 0x0000, 0x6229, 0x0000,
	0x3b29, 0x0000, 0x0000, 0x622b, 0x0000, 0x0000, 0x622a, 0x0000,
	0x0000, 0x622c, 0x622d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4869, 0x0000,
	0x622e, 0x0000, 0x0000, 0x0000, 0x622f, 0x0000, 0x0000, 0x7369,
	0x6230, 0x6231, 0x6232, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b2e,
	0x0000, 0x0000, 0x6233, 0x4756, 0x0000, 0x0000, 0x4b5f, 0x0000,
	0x314e, 0x0000, 0x3157, 0x0000, 0x0000, 0x6234, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6236, 0x0000, 0x0000, 0x0000, 0x6235, 0x4570,
	0x0000, 0x0000, 0x0000, 0x4039, 0x5d39, 0x0000, 0x6237, 0x4c41,
	0x0000, 0x6238, 0x0000, 0x3446, 0x4857, 0x6239, 0x0000, 0x623a,
	0x0000, 0x0000, 0x623b, 0x0000, 0x0000, 0x0000, 0x4c5c, 0x0000,
	0x0000, 0x0000, 0x4c55, 0x0000, 0x443e, 0x0000, 0x0000, 0x0000,
	0x416a, 0x0000, 0x0000, 0x623d, 0x0000, 0x0000, 0x3d62, 0x0000,
};

static unsigned short const unicode_to_jisx0208_77[] = {
	/* 0x7700 - 0x77ff */
	0x0000, 0x3e4a, 0x0000, 0x0000, 0x6240, 0x0000, 0x0000, 0x623f,
	0x623e, 0x487d, 0x0000, 0x3447, 0x3829, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6246, 0x0000, 0x0000, 0x6243, 0x3f3f,
	0x4c32, 0x0000, 0x0000, 0x0000, 0x6242, 0x6244, 0x6245, 0x0000,
	0x0000, 0x6241, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6247,
	0x6248, 0x0000, 0x442f, 0x0000, 0x3463, 0x0000, 0x0000, 0x0000,
	0x4365, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6249,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x624a, 0x624d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3f67, 0x0000, 0x4644, 0x0000, 0x624e, 0x4b53, 0x0000,
	0x624b, 0x0000, 0x0000, 0x624c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6251, 0x0000, 0x0000, 0x0000, 0x0000, 0x6250, 0x624f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6253, 0x0000, 0x0000, 0x6252, 0x0000,
	0x0000, 0x6254, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6256, 0x0000,
	0x6255, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a4d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d56, 0x4e46, 0x0000, 0x0000,
	0x6257, 0x0000, 0x0000, 0x4637, 0x0000, 0x0000, 0x6258, 0x0000,
	0x0000, 0x6259, 0x0000, 0x625d, 0x625b, 0x625c, 0x0000, 0x625a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x625e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x625f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6260,
	0x0000, 0x0000, 0x6261, 0x4c37, 0x6262, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c70, 0x6263, 0x0000, 0x434e, 0x0000, 0x476a,
	0x0000, 0x366b, 0x0000, 0x0000, 0x0000, 0x433b, 0x6264, 0x363a,
	0x0000, 0x0000, 0x0000, 0x4050, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6265, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_78[] = {
	/* 0x7800 - 0x78ff */
	0x0000, 0x0000, 0x3a3d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6266, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6267, 0x0000, 0x3826, 0x3a55, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6269, 0x0000, 0x0000, 0x0000, 0x0000, 0x4556, 0x3a56, 0x354e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4b24, 0x0000, 0x474b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4557, 0x0000, 0x0000, 0x0000, 0x0000, 0x395c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x626b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e4b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4e32, 0x3945, 0x0000, 0x0000, 0x3827,
	0x0000, 0x0000, 0x4823, 0x0000, 0x626d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x626f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x386b, 0x0000, 0x0000, 0x0000, 0x0000, 0x626e, 0x4476,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6271, 0x3337, 0x626c, 0x0000,
	0x0000, 0x486a, 0x0000, 0x3130, 0x0000, 0x3a6c, 0x0000, 0x4f52,
	0x0000, 0x0000, 0x6270, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6272, 0x0000, 0x0000, 0x0000, 0x4a4b,
	0x0000, 0x4059, 0x6274, 0x0000, 0x0000, 0x0000, 0x0000, 0x6275,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6273, 0x0000, 0x0000,
	0x0000, 0x0000, 0x334e, 0x0000, 0x627b, 0x0000, 0x627a, 0x0000,
	0x0000, 0x3c27, 0x0000, 0x0000, 0x0000, 0x627c, 0x6277, 0x0000,
	0x0000, 0x0000, 0x627d, 0x6278, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4858, 0x6276, 0x0000, 0x0000, 0x6279, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6322, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6321,
	0x4b61, 0x0000, 0x0000, 0x0000, 0x627e, 0x0000, 0x0000, 0x306b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6324, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6323, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_79[] = {
	/* 0x7900 - 0x79ff */
	0x0000, 0x3e4c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6325,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4143, 0x0000,
	0x0000, 0x6327, 0x6326, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6328, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6268, 0x0000,
	0x0000, 0x0000, 0x626a, 0x632a, 0x6329, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3c28, 0x0000, 0x4e69, 0x0000, 0x3c52, 0x0000,
	0x632b, 0x3737, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3540,
	0x3527, 0x3b63, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4d34, 0x0000, 0x0000, 0x6331, 0x0000, 0x6330, 0x4144, 0x632d,
	0x0000, 0x0000, 0x632f, 0x0000, 0x0000, 0x3d4b, 0x3f40, 0x632e,
	0x632c, 0x0000, 0x472a, 0x0000, 0x0000, 0x3e4d, 0x0000, 0x0000,
	0x493c, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a57, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4578,
	0x0000, 0x0000, 0x6332, 0x0000, 0x0000, 0x0000, 0x0000, 0x6333,
	0x6349, 0x3658, 0x0000, 0x0000, 0x4f3d, 0x4135, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6334, 0x0000, 0x0000, 0x3252, 0x4477, 0x4a21,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6335, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x357a, 0x6336,
	0x0000, 0x0000, 0x6338, 0x0000, 0x0000, 0x0000, 0x6339, 0x0000,
	0x4729, 0x0000, 0x0000, 0x633a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x633b, 0x633c, 0x0000, 0x0000, 0x3659, 0x3253, 0x4645,
	0x3d28, 0x3b64, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x633d, 0x0000, 0x3d29, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x324a, 0x4943, 0x0000, 0x0000, 0x633e, 0x0000, 0x0000,
	0x486b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4145,
	0x0000, 0x6341, 0x0000, 0x6342, 0x4769, 0x0000, 0x3f41, 0x633f,
	0x0000, 0x4361, 0x0000, 0x0000, 0x6340, 0x0000, 0x0000, 0x0000,
	0x3e4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x305c, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_7a[] = {
	/* 0x7a00 - 0x7aff */
	0x3529, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6343, 0x0000, 0x0000, 0x4478, 0x0000, 0x6344, 0x4047, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4c2d, 0x0000, 0x0000, 0x4923,
	0x6345, 0x6346, 0x4355, 0x0000, 0x4e47, 0x0000, 0x0000, 0x6348,
	0x6347, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c6f, 0x0000,
	0x0000, 0x634a, 0x3070, 0x0000, 0x0000, 0x0000, 0x0000, 0x634d,
	0x0000, 0x0000, 0x0000, 0x634b, 0x3254, 0x374e, 0x634c, 0x3946,
	0x3972, 0x0000, 0x4a66, 0x634e, 0x0000, 0x0000, 0x4b54, 0x0000,
	0x0000, 0x6350, 0x0000, 0x0000, 0x0000, 0x4051, 0x314f, 0x323a,
	0x302c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x634f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6351, 0x6352, 0x3e77, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6353, 0x0000, 0x334f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6355, 0x0000, 0x0000, 0x0000, 0x376a, 0x0000, 0x3566, 0x0000,
	0x0000, 0x6356, 0x3675, 0x0000, 0x0000, 0x6357, 0x0000, 0x407c,
	0x0000, 0x464d, 0x0000, 0x4060, 0x3a75, 0x0000, 0x0000, 0x0000,
	0x6358, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4362, 0x416b, 0x0000, 0x635a, 0x635c, 0x6359,
	0x635b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3722,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x635d, 0x3726, 0x0000, 0x0000, 0x0000, 0x3567, 0x4d52,
	0x635f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6360, 0x0000,
	0x0000, 0x0000, 0x312e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6363,
	0x0000, 0x0000, 0x0000, 0x3376, 0x6362, 0x6361, 0x0000, 0x6365,
	0x635e, 0x0000, 0x6366, 0x4e29, 0x0000, 0x6367, 0x0000, 0x6368,
	0x0000, 0x0000, 0x5474, 0x636a, 0x0000, 0x6369, 0x0000, 0x0000,
	0x0000, 0x636b, 0x636c, 0x0000, 0x4e35, 0x636d, 0x0000, 0x706f,
	0x3e4f, 0x636e, 0x636f, 0x3d57, 0x0000, 0x4638, 0x6370, 0x0000,
	0x0000, 0x0000, 0x4328, 0x0000, 0x0000, 0x6371, 0x0000, 0x433c,
	0x6372, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3625, 0x0000,
	0x513f, 0x435d, 0x3c33, 0x0000, 0x0000, 0x0000, 0x0000, 0x3448,
};

static unsigned short const unicode_to_jisx0208_7b[] = {
	/* 0x7b00 - 0x7bff */
	0x0000, 0x0000, 0x6373, 0x0000, 0x6422, 0x0000, 0x6376, 0x0000,
	0x3568, 0x0000, 0x6375, 0x6424, 0x0000, 0x0000, 0x0000, 0x6374,
	0x0000, 0x3e50, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6378, 0x6379, 0x0000, 0x452b, 0x0000, 0x0000, 0x637a, 0x0000,
	0x335e, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f5a, 0x4964, 0x0000,
	0x637c, 0x0000, 0x0000, 0x0000, 0x4268, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6377, 0x0000, 0x637b, 0x637d, 0x0000,
	0x0000, 0x3a7b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6426, 0x492e, 0x0000,
	0x4826, 0x4579, 0x0000, 0x365a, 0x6425, 0x6423, 0x0000, 0x4835,
	0x637e, 0x435e, 0x457b, 0x0000, 0x457a, 0x0000, 0x3a76, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6438, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6428, 0x0000, 0x642a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x642d, 0x0000, 0x642e, 0x0000,
	0x642b, 0x642c, 0x0000, 0x0000, 0x6429, 0x6427, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6421, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a4f, 0x3255,
	0x0000, 0x0000, 0x0000, 0x6435, 0x0000, 0x6432, 0x0000, 0x6437,
	0x0000, 0x0000, 0x6436, 0x0000, 0x4773, 0x4c27, 0x0000, 0x3b3b,
	0x6430, 0x6439, 0x6434, 0x0000, 0x6433, 0x642f, 0x0000, 0x6431,
	0x0000, 0x3449, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x433d, 0x0000, 0x0000, 0x407d, 0x0000, 0x0000,
	0x0000, 0x4822, 0x0000, 0x0000, 0x643e, 0x0000, 0x0000, 0x0000,
	0x4824, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4061, 0x643b, 0x0000, 0x0000, 0x484f, 0x0000, 0x643f, 0x4a53,
	0x0000, 0x435b, 0x0000, 0x643a, 0x643c, 0x0000, 0x0000, 0x643d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6440, 0x0000, 0x0000,
	0x3c44, 0x0000, 0x0000, 0x0000, 0x4646, 0x6445, 0x6444, 0x0000,
	0x0000, 0x6441, 0x0000, 0x0000, 0x0000, 0x4f36, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x644a, 0x0000, 0x0000, 0x644e, 0x644b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_7c[] = {
	/* 0x7c00 - 0x7cff */
	0x6447, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6448,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x644d, 0x0000, 0x0000,
	0x0000, 0x6442, 0x5255, 0x6449, 0x6443, 0x0000, 0x0000, 0x644c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6452,
	0x0000, 0x344a, 0x0000, 0x644f, 0x0000, 0x0000, 0x0000, 0x6450,
	0x0000, 0x0000, 0x6451, 0x6454, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6453,
	0x4876, 0x0000, 0x0000, 0x0000, 0x0000, 0x6455, 0x4e7c, 0x4a6d,
	0x645a, 0x0000, 0x0000, 0x6457, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6456, 0x4052, 0x0000, 0x6459,
	0x645b, 0x0000, 0x0000, 0x0000, 0x6458, 0x0000, 0x645f, 0x0000,
	0x645c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x645d,
	0x6446, 0x0000, 0x0000, 0x0000, 0x645e, 0x6460, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6461, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a46, 0x0000, 0x6462, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c62, 0x0000,
	0x0000, 0x364e, 0x3729, 0x6463, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a34, 0x0000, 0x3f68, 0x0000, 0x4c30, 0x0000, 0x0000,
	0x6464, 0x0000, 0x4e33, 0x0000, 0x0000, 0x4774, 0x0000, 0x4146,
	0x4734, 0x0000, 0x0000, 0x3d4d, 0x0000, 0x0000, 0x0000, 0x3040,
	0x0000, 0x6469, 0x6467, 0x0000, 0x6465, 0x3421, 0x0000, 0x3e51,
	0x646a, 0x0000, 0x0000, 0x6468, 0x0000, 0x6466, 0x646e, 0x0000,
	0x0000, 0x646d, 0x646c, 0x646b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x646f, 0x0000, 0x0000, 0x0000, 0x6470, 0x403a, 0x0000,
	0x6471, 0x0000, 0x6473, 0x0000, 0x0000, 0x6472, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3852, 0x0000, 0x0000, 0x0000, 0x4138, 0x0000,
	0x0000, 0x0000, 0x6475, 0x0000, 0x0000, 0x0000, 0x457c, 0x0000,
	0x6474, 0x0000, 0x0000, 0x0000, 0x6476, 0x0000, 0x4a35, 0x416c,
	0x3947, 0x0000, 0x6477, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e48,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6479,
	0x0000, 0x0000, 0x647a, 0x0000, 0x647b, 0x0000, 0x647c, 0x0000,
	0x3b65, 0x0000, 0x647d, 0x374f, 0x0000, 0x0000, 0x356a, 0x0000,
};

static unsigned short const unicode_to_jisx0208_7d[] = {
	/* 0x7d00 - 0x7dff */
	0x352a, 0x0000, 0x6521, 0x0000, 0x4c73, 0x3948, 0x647e, 0x0000,
	0x0000, 0x0000, 0x6524, 0x4c66, 0x0000, 0x473c, 0x0000, 0x0000,
	0x4933, 0x0000, 0x0000, 0x0000, 0x3d63, 0x6523, 0x0000, 0x3c53,
	0x3949, 0x3b66, 0x3569, 0x4a36, 0x6522, 0x0000, 0x0000, 0x0000,
	0x4147, 0x4b42, 0x3a77, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b67, 0x445d, 0x0000, 0x6527, 0x4e5f,
	0x3a59, 0x0000, 0x6528, 0x3f42, 0x0000, 0x652a, 0x0000, 0x0000,
	0x0000, 0x3e52, 0x3a30, 0x0000, 0x0000, 0x0000, 0x0000, 0x6529,
	0x0000, 0x0000, 0x3d2a, 0x383e, 0x4148, 0x6525, 0x652b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6526, 0x3750, 0x0000, 0x652e, 0x6532,
	0x376b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x652d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6536, 0x0000, 0x0000, 0x394a, 0x0000,
	0x0000, 0x4d6d, 0x303c, 0x6533, 0x0000, 0x0000, 0x356b, 0x0000,
	0x6530, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6531, 0x0000,
	0x0000, 0x457d, 0x652f, 0x652c, 0x0000, 0x3328, 0x4064, 0x0000,
	0x0000, 0x3828, 0x0000, 0x0000, 0x0000, 0x6538, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6535, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6537,
	0x0000, 0x0000, 0x0000, 0x6534, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3751, 0x4233, 0x6539, 0x416e, 0x0000, 0x0000, 0x6546,
	0x0000, 0x0000, 0x6542, 0x653c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6540, 0x3c7a, 0x305d, 0x653b, 0x6543,
	0x6547, 0x394b, 0x4c56, 0x0000, 0x4456, 0x653d, 0x0000, 0x0000,
	0x6545, 0x0000, 0x653a, 0x433e, 0x0000, 0x653f, 0x303d, 0x4c4a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x653e,
	0x0000, 0x0000, 0x365b, 0x486c, 0x0000, 0x0000, 0x0000, 0x416d,
	0x0000, 0x4e50, 0x3d6f, 0x0000, 0x0000, 0x656e, 0x0000, 0x0000,
	0x6548, 0x0000, 0x407e, 0x0000, 0x6544, 0x6549, 0x654b, 0x0000,
	0x4479, 0x654e, 0x0000, 0x0000, 0x654a, 0x0000, 0x0000, 0x0000,
	0x4a54, 0x344b, 0x0000, 0x0000, 0x4c4b, 0x0000, 0x0000, 0x305e,
	0x0000, 0x0000, 0x654d, 0x0000, 0x4e7d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x654c, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_7e[] = {
	/* 0x7e00 - 0x7eff */
	0x0000, 0x316f, 0x0000, 0x0000, 0x466c, 0x654f, 0x0000, 0x0000,
	0x0000, 0x6556, 0x6550, 0x6557, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6553, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x477b, 0x0000, 0x0000, 0x3c4a, 0x6555,
	0x0000, 0x6552, 0x6558, 0x6551, 0x0000, 0x0000, 0x3d44, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4b25, 0x0000, 0x0000, 0x3d4c, 0x0000,
	0x0000, 0x6554, 0x6560, 0x0000, 0x0000, 0x655c, 0x0000, 0x655f,
	0x0000, 0x655d, 0x6561, 0x655b, 0x0000, 0x6541, 0x4053, 0x0000,
	0x0000, 0x484b, 0x0000, 0x655e, 0x0000, 0x0000, 0x6559, 0x0000,
	0x0000, 0x0000, 0x4121, 0x3752, 0x0000, 0x3d2b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f25, 0x4136, 0x6564, 0x0000,
	0x0000, 0x6566, 0x6567, 0x0000, 0x0000, 0x6563, 0x6565, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x655a, 0x6562,
	0x0000, 0x656a, 0x6569, 0x0000, 0x0000, 0x4b7a, 0x0000, 0x0000,
	0x372b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6568, 0x0000, 0x656c, 0x656b, 0x656f, 0x0000, 0x6571,
	0x0000, 0x0000, 0x3b3c, 0x656d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6572, 0x6573, 0x0000, 0x0000, 0x6574, 0x0000, 0x657a, 0x453b,
	0x6576, 0x0000, 0x6575, 0x6577, 0x6578, 0x0000, 0x6579, 0x0000,
	0x0000, 0x0000, 0x0000, 0x657b, 0x657c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_7f[] = {
	/* 0x7f00 - 0x7fff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x344c, 0x0000,
	0x657d, 0x0000, 0x657e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6621, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6622, 0x6623, 0x6624, 0x0000,
	0x6625, 0x6626, 0x0000, 0x0000, 0x6628, 0x6627, 0x0000, 0x0000,
	0x6629, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x662a,
	0x662b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x662e,
	0x662c, 0x662d, 0x3a61, 0x3753, 0x0000, 0x0000, 0x4356, 0x0000,
	0x4833, 0x0000, 0x3d70, 0x0000, 0x0000, 0x474d, 0x0000, 0x486d,
	0x662f, 0x586d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6630, 0x6632, 0x0000, 0x4d65, 0x6631, 0x6634,
	0x6633, 0x0000, 0x4d53, 0x0000, 0x6635, 0x0000, 0x487e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6636, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6639, 0x0000, 0x0000, 0x6638, 0x6637, 0x0000,
	0x0000, 0x0000, 0x0000, 0x663a, 0x3732, 0x0000, 0x0000, 0x0000,
	0x4122, 0x3541, 0x0000, 0x0000, 0x0000, 0x0000, 0x663e, 0x663b,
	0x0000, 0x0000, 0x663c, 0x0000, 0x0000, 0x0000, 0x663f, 0x0000,
	0x6640, 0x663d, 0x0000, 0x0000, 0x0000, 0x3129, 0x0000, 0x0000,
	0x0000, 0x3227, 0x0000, 0x0000, 0x0000, 0x6642, 0x6643, 0x0000,
	0x0000, 0x0000, 0x6644, 0x0000, 0x4d62, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3d2c, 0x0000, 0x6646, 0x6645, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3f69, 0x6647, 0x0000, 0x0000, 0x0000, 0x0000, 0x6648, 0x0000,
	0x0000, 0x6649, 0x0000, 0x3465, 0x0000, 0x0000, 0x0000, 0x0000,
	0x344d, 0x0000, 0x0000, 0x664a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x664b, 0x0000, 0x4b5d, 0x4d63, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_80[] = {
	/* 0x8000 - 0x80ff */
	0x4d54, 0x4f37, 0x0000, 0x394d, 0x664e, 0x3c54, 0x664d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x664f, 0x3c29, 0x0000, 0x0000, 0x0000,
	0x4251, 0x0000, 0x6650, 0x0000, 0x0000, 0x394c, 0x0000, 0x4c57,
	0x6651, 0x6652, 0x0000, 0x0000, 0x6653, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6654, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6655, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c2a, 0x0000, 0x0000, 0x4c6d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6657, 0x0000, 0x433f, 0x0000, 0x6656,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6659, 0x0000,
	0x0000, 0x0000, 0x6658, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x665a, 0x0000, 0x0000, 0x0000, 0x403b, 0x0000,
	0x665b, 0x0000, 0x665c, 0x0000, 0x0000, 0x0000, 0x4a39, 0x665d,
	0x0000, 0x416f, 0x665e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x665f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e7e,
	0x6662, 0x0000, 0x6661, 0x6660, 0x4430, 0x0000, 0x6663, 0x3f26,
	0x0000, 0x6664, 0x0000, 0x0000, 0x0000, 0x6665, 0x4f38, 0x6666,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6667, 0x6669, 0x6668, 0x4825,
	0x0000, 0x4679, 0x0000, 0x4f3e, 0x4829, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x666b, 0x0000, 0x0000, 0x3e53, 0x0000,
	0x492a, 0x0000, 0x666c, 0x666a, 0x0000, 0x344e, 0x0000, 0x0000,
	0x0000, 0x3854, 0x3b68, 0x0000, 0x0000, 0x486e, 0x0000, 0x0000,
	0x0000, 0x382a, 0x4b43, 0x0000, 0x666f, 0x666d, 0x0000, 0x394e,
	0x0000, 0x394f, 0x3069, 0x0000, 0x3a68, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4759, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x305f, 0x6674, 0x0000, 0x4340, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4758, 0x0000, 0x425b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6676, 0x0000,
	0x0000, 0x6672, 0x6675, 0x6670, 0x0000, 0x6673, 0x4b26, 0x0000,
	0x0000, 0x3855, 0x0000, 0x0000, 0x307d, 0x6671, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6678,
	0x0000, 0x6679, 0x0000, 0x0000, 0x4639, 0x0000, 0x0000, 0x0000,
	0x363b, 0x0000, 0x0000, 0x0000, 0x6726, 0x473d, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_81[] = {
	/* 0x8100 - 0x81ff */
	0x0000, 0x0000, 0x3b69, 0x0000, 0x0000, 0x363c, 0x4048, 0x4f46,
	0x4c2e, 0x6677, 0x4054, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3553, 0x667a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x667c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x667b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x667d,
	0x0000, 0x4326, 0x0000, 0x473e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4431, 0x0000, 0x0000, 0x0000, 0x0000, 0x6723, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6722, 0x0000,
	0x0000, 0x0000, 0x0000, 0x667e, 0x0000, 0x0000, 0x3f55, 0x0000,
	0x4965, 0x6725, 0x0000, 0x6724, 0x3950, 0x4f53, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6735,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6729, 0x672a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c70, 0x0000, 0x0000, 0x6728, 0x0000,
	0x3978, 0x6727, 0x0000, 0x0000, 0x672b, 0x0000, 0x0000, 0x0000,
	0x4432, 0x4a22, 0x4123, 0x0000, 0x0000, 0x0000, 0x0000, 0x425c,
	0x672f, 0x0000, 0x6730, 0x672c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x672d, 0x0000, 0x672e, 0x0000, 0x0000, 0x0000, 0x0000, 0x3951,
	0x0000, 0x0000, 0x0000, 0x6736, 0x0000, 0x6732, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4966, 0x0000, 0x4b6c, 0x4928, 0x0000, 0x0000,
	0x6731, 0x0000, 0x0000, 0x6734, 0x6733, 0x0000, 0x0000, 0x0000,
	0x4b44, 0x6737, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6738, 0x0000, 0x0000, 0x4137, 0x0000, 0x6739, 0x0000, 0x0000,
	0x673b, 0x0000, 0x673f, 0x0000, 0x0000, 0x673c, 0x673a, 0x473f,
	0x673d, 0x0000, 0x673e, 0x0000, 0x0000, 0x0000, 0x3232, 0x0000,
	0x6745, 0x6740, 0x0000, 0x0000, 0x0000, 0x6741, 0x0000, 0x0000,
	0x0000, 0x6742, 0x0000, 0x4221, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6744, 0x6743, 0x6746, 0x0000, 0x0000, 0x0000, 0x0000, 0x6747,
	0x6748, 0x0000, 0x0000, 0x3f43, 0x0000, 0x3269, 0x0000, 0x6749,
	0x4e57, 0x0000, 0x3c2b, 0x0000, 0x0000, 0x3d2d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b6a, 0x4357, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x674a, 0x674b, 0x3131, 0x0000, 0x674c, 0x0000,
};

static unsigned short const unicode_to_jisx0208_82[] = {
	/* 0x8200 - 0x82ff */
	0x0000, 0x674d, 0x674e, 0x0000, 0x0000, 0x674f, 0x0000, 0x6750,
	0x363d, 0x5a2a, 0x6751, 0x0000, 0x4065, 0x6752, 0x3c4b, 0x0000,
	0x6753, 0x0000, 0x5030, 0x0000, 0x0000, 0x0000, 0x6754, 0x4a5e,
	0x345c, 0x0000, 0x0000, 0x4124, 0x3d58, 0x0000, 0x4971, 0x3d2e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6755, 0x3952, 0x6756, 0x484c, 0x0000, 0x6764, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6758, 0x0000, 0x4249, 0x4775, 0x383f,
	0x6757, 0x4125, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6759, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x447a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x675b, 0x675a, 0x675d, 0x0000, 0x0000, 0x675c, 0x0000, 0x675e,
	0x0000, 0x0000, 0x6760, 0x0000, 0x675f, 0x0000, 0x344f, 0x0000,
	0x6761, 0x0000, 0x6762, 0x6763, 0x0000, 0x0000, 0x3a31, 0x4e49,
	0x0000, 0x6765, 0x3f27, 0x0000, 0x0000, 0x0000, 0x3170, 0x6766,
	0x6767, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6768, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3072, 0x0000, 0x6769, 0x0000, 0x0000,
	0x0000, 0x0000, 0x676a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4967, 0x0000, 0x0000, 0x0000, 0x3c47, 0x0000, 0x676c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3329, 0x3032, 0x0000,
	0x0000, 0x0000, 0x0000, 0x676b, 0x676e, 0x474e, 0x0000, 0x3f44,
	0x0000, 0x3256, 0x0000, 0x4b27, 0x0000, 0x0000, 0x0000, 0x0000,
	0x375d, 0x365c, 0x0000, 0x676d, 0x0000, 0x326a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3423, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3171, 0x6772, 0x4e6a, 0x425d, 0x0000, 0x0000, 0x4944,
	0x0000, 0x677e, 0x0000, 0x3257, 0x677c, 0x0000, 0x677a, 0x6771,
	0x0000, 0x676f, 0x0000, 0x6770, 0x0000, 0x3c63, 0x366c, 0x4377,
	0x0000, 0x0000, 0x0000, 0x4651, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3151, 0x0000, 0x6774, 0x6773, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6779, 0x6775, 0x6778, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_83[] = {
	/* 0x8300 - 0x83ff */
	0x0000, 0x0000, 0x4c50, 0x6777, 0x3258, 0x337d, 0x677b, 0x0000,
	0x0000, 0x677d, 0x0000, 0x0000, 0x0000, 0x0000, 0x3754, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6823, 0x682c,
	0x682d, 0x0000, 0x0000, 0x0000, 0x302b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6834, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3071, 0x0000, 0x0000, 0x682b, 0x0000, 0x0000, 0x0000, 0x682a,
	0x0000, 0x6825, 0x6824, 0x0000, 0x6822, 0x6821, 0x4363, 0x0000,
	0x427b, 0x6827, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6826, 0x0000, 0x0000, 0x0000, 0x0000, 0x6829, 0x0000, 0x0000,
	0x0000, 0x4170, 0x3755, 0x0000, 0x0000, 0x0000, 0x0000, 0x3141,
	0x6828, 0x0000, 0x3953, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4171, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x683a, 0x0000, 0x683b, 0x0000, 0x3259,
	0x0000, 0x0000, 0x0000, 0x322e, 0x6838, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x682e, 0x0000, 0x6836,
	0x0000, 0x683d, 0x6837, 0x0000, 0x0000, 0x0000, 0x6835, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6776, 0x0000, 0x0000, 0x6833, 0x0000,
	0x0000, 0x0000, 0x682f, 0x0000, 0x0000, 0x0000, 0x3450, 0x6831,
	0x683c, 0x0000, 0x6832, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x683e, 0x0000, 0x6830, 0x477c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4d69, 0x0000, 0x0000, 0x0000, 0x6839, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x684f, 0x0000, 0x0000,
	0x0000, 0x6847, 0x0000, 0x0000, 0x0000, 0x3f7b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3546, 0x0000, 0x365d, 0x0000, 0x6842, 0x0000,
	0x0000, 0x0000, 0x0000, 0x325b, 0x0000, 0x0000, 0x3e54, 0x0000,
	0x6845, 0x0000, 0x0000, 0x0000, 0x3a5a, 0x0000, 0x0000, 0x4551,
	0x684a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a6e, 0x0000, 0x6841, 0x0000, 0x0000, 0x0000, 0x325a,
	0x3856, 0x4929, 0x684b, 0x0000, 0x683f, 0x0000, 0x0000, 0x6848,
	0x0000, 0x0000, 0x0000, 0x6852, 0x0000, 0x6843, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_84[] = {
	/* 0x8400 - 0x84ff */
	0x0000, 0x0000, 0x0000, 0x6844, 0x463a, 0x0000, 0x0000, 0x6849,
	0x0000, 0x0000, 0x0000, 0x6846, 0x4b28, 0x684c, 0x3060, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6840, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x684e, 0x0000, 0x684d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x476b, 0x6854, 0x0000, 0x685f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x337e, 0x0000, 0x0000, 0x0000, 0x6862, 0x0000, 0x0000,
	0x6850, 0x0000, 0x0000, 0x0000, 0x6855, 0x4d6e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x685e, 0x0000,
	0x0000, 0x4d55, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e2a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4378,
	0x0000, 0x0000, 0x0000, 0x336b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4972, 0x6864, 0x4621, 0x0000, 0x0000, 0x3031, 0x0000,
	0x0000, 0x685d, 0x0000, 0x6859, 0x4172, 0x6853, 0x685b, 0x6860,
	0x0000, 0x472c, 0x0000, 0x0000, 0x0000, 0x302a, 0x0000, 0x6858,
	0x0000, 0x6861, 0x4978, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x685c, 0x0000, 0x6857, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3e55, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3d2f, 0x0000, 0x0000, 0x0000, 0x3c2c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4c58, 0x0000, 0x0000, 0x4947, 0x0000, 0x0000, 0x6867,
	0x0000, 0x6870, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x685a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3377, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3e78, 0x6865, 0x0000, 0x686a, 0x4173, 0x0000, 0x0000, 0x6866,
	0x0000, 0x686d, 0x0000, 0x0000, 0x435f, 0x0000, 0x686e, 0x0000,
	0x0000, 0x4d56, 0x6863, 0x3338, 0x0000, 0x6869, 0x0000, 0x0000,
	0x686c, 0x4c2c, 0x0000, 0x0000, 0x0000, 0x0000, 0x686f, 0x0000,
	0x0000, 0x6868, 0x686b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4b29, 0x0000, 0x4f21, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6873, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x687a, 0x0000, 0x0000, 0x6872,
};

static unsigned short const unicode_to_jisx0208_85[] = {
	/* 0x8500 - 0x85ff */
	0x3c43, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6851, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a4e, 0x0000, 0x4c22, 0x6879, 0x6878, 0x0000, 0x6874,
	0x6875, 0x0000, 0x3136, 0x0000, 0x0000, 0x0000, 0x0000, 0x6877,
	0x0000, 0x6871, 0x0000, 0x0000, 0x0000, 0x0000, 0x4455, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6876, 0x307e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4222, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a43, 0x0000, 0x0000,
	0x687b, 0x6921, 0x0000, 0x4859, 0x0000, 0x0000, 0x0000, 0x0000,
	0x687e, 0x3e56, 0x3c49, 0x6923, 0x0000, 0x0000, 0x363e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6924, 0x0000, 0x4979,
	0x687d, 0x0000, 0x6856, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x687c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4f4f, 0x4622, 0x4973, 0x0000, 0x0000, 0x692b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6931,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6932, 0x0000,
	0x6925, 0x0000, 0x0000, 0x0000, 0x4776, 0x0000, 0x0000, 0x692f,
	0x6927, 0x0000, 0x6929, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6933, 0x6928, 0x0000, 0x0000, 0x692c, 0x0000, 0x0000, 0x3172,
	0x0000, 0x4665, 0x0000, 0x692d, 0x6930, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6926, 0x0000, 0x4126, 0x0000,
	0x692a, 0x3b27, 0x3f45, 0x3730, 0x4c74, 0x0000, 0x4c79, 0x3d72,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6937, 0x6935, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4f4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6934, 0x0000, 0x0000, 0x0000, 0x4d75, 0x0000, 0x6936,
	0x6938, 0x0000, 0x0000, 0x0000, 0x0000, 0x6939, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x693c, 0x693a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4623, 0x693b, 0x0000, 0x0000,
	0x0000, 0x484d, 0x692e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d73,
	0x0000, 0x693d, 0x6942, 0x4174, 0x0000, 0x0000, 0x6941, 0x0000,
};

static unsigned short const unicode_to_jisx0208_86[] = {
	/* 0x8600 - 0x86ff */
	0x0000, 0x0000, 0x6922, 0x0000, 0x0000, 0x0000, 0x6943, 0x4149,
	0x0000, 0x0000, 0x693e, 0x6940, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x693f, 0x0000, 0x0000, 0x5d31, 0x5d22,
	0x0000, 0x0000, 0x6945, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6944, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d76, 0x0000, 0x623c,
	0x6946, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6947,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6948, 0x3857, 0x0000,
	0x3554, 0x0000, 0x0000, 0x0000, 0x694a, 0x515d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3575, 0x0000, 0x4e3a, 0x0000, 0x3673, 0x694b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x694c,
	0x0000, 0x0000, 0x0000, 0x436e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x694d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x467a, 0x0000, 0x303a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3263, 0x6952, 0x6953, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x694e, 0x0000, 0x3b3d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x694f, 0x4742, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6950, 0x6951, 0x695b, 0x0000, 0x0000, 0x0000, 0x6955,
	0x6958, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6954, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6956, 0x0000, 0x6957, 0x3c58,
	0x0000, 0x6959, 0x0000, 0x4341, 0x0000, 0x3756, 0x3342, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x695c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x333f, 0x0000, 0x6961, 0x0000, 0x0000, 0x695d, 0x6960,
	0x0000, 0x0000, 0x0000, 0x0000, 0x483a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x695e, 0x0000, 0x0000, 0x695f, 0x4948, 0x485a, 0x6962,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x427d, 0x696c, 0x0000, 0x6968, 0x0000, 0x0000, 0x326b, 0x0000,
};

static unsigned short const unicode_to_jisx0208_87[] = {
	/* 0x8700 - 0x87ff */
	0x6966, 0x0000, 0x4b2a, 0x6967, 0x0000, 0x0000, 0x6964, 0x0000,
	0x6965, 0x696a, 0x696d, 0x0000, 0x0000, 0x696b, 0x0000, 0x0000,
	0x0000, 0x6969, 0x6963, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4358, 0x0000, 0x6974, 0x0000, 0x4c2a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6972, 0x0000, 0x0000,
	0x0000, 0x6973, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x696e, 0x0000, 0x0000, 0x6970,
	0x0000, 0x0000, 0x0000, 0x6971, 0x0000, 0x0000, 0x0000, 0x696f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4066, 0x0000, 0x4f39, 0x6978, 0x0000, 0x6979, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6a21, 0x0000, 0x3f2a, 0x0000, 0x697b,
	0x0000, 0x697e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6976,
	0x6975, 0x0000, 0x0000, 0x6a22, 0x0000, 0x0000, 0x325c, 0x0000,
	0x697c, 0x0000, 0x6a23, 0x0000, 0x0000, 0x0000, 0x697d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x697a, 0x0000, 0x4433, 0x0000,
	0x6977, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4768,
	0x0000, 0x0000, 0x6a27, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d3b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a26,
	0x0000, 0x0000, 0x6a25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6a2e, 0x0000, 0x0000, 0x0000, 0x6a28,
	0x0000, 0x0000, 0x0000, 0x6a30, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d66, 0x6a33, 0x0000, 0x6a2a, 0x0000, 0x0000,
	0x6a2b, 0x0000, 0x0000, 0x0000, 0x6a2f, 0x0000, 0x6a32, 0x6a31,
	0x0000, 0x0000, 0x0000, 0x6a29, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6a2c, 0x0000, 0x6a3d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6a36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a34,
	0x0000, 0x0000, 0x6a35, 0x0000, 0x0000, 0x0000, 0x6a3a, 0x6a3b,
	0x0000, 0x332a, 0x0000, 0x3542, 0x0000, 0x0000, 0x6a39, 0x0000,
};

static unsigned short const unicode_to_jisx0208_88[] = {
	/* 0x8800 - 0x88ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a24, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a38, 0x6a3c, 0x6a37,
	0x0000, 0x6a3e, 0x0000, 0x0000, 0x0000, 0x6a40, 0x6a3f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6a42, 0x6a41, 0x695a, 0x0000, 0x0000, 0x0000, 0x6a46,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6a43, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a44, 0x0000,
	0x0000, 0x6a45, 0x0000, 0x6a47, 0x0000, 0x0000, 0x0000, 0x0000,
	0x376c, 0x0000, 0x6a49, 0x0000, 0x6a48, 0x0000, 0x3d30, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3954, 0x5e27, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6a4a, 0x3d51, 0x0000, 0x0000, 0x0000, 0x3339,
	0x0000, 0x6a4b, 0x0000, 0x3152, 0x0000, 0x3e57, 0x6a4c, 0x0000,
	0x0000, 0x3955, 0x6a4d, 0x3061, 0x0000, 0x0000, 0x0000, 0x0000,
	0x493d, 0x0000, 0x0000, 0x6a4e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3f6a, 0x0000, 0x6a55, 0x0000, 0x0000, 0x6a52, 0x0000, 0x436f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a53, 0x6a50, 0x365e,
	0x0000, 0x6a4f, 0x6a56, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3736, 0x0000, 0x0000, 0x425e, 0x0000, 0x6a5c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6a58, 0x0000, 0x0000, 0x0000, 0x4235, 0x6a57,
	0x0000, 0x6a5a, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a51, 0x0000,
	0x0000, 0x0000, 0x6a5b, 0x0000, 0x6a5d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x486f, 0x0000, 0x0000, 0x6a59, 0x0000,
	0x6a5e, 0x6a60, 0x0000, 0x0000, 0x3853, 0x6a54, 0x0000, 0x3041,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a5f,
	0x0000, 0x3a5b, 0x4e76, 0x6a61, 0x6a62, 0x4175, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e22,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6a63, 0x4d35, 0x0000, 0x0000,
	0x6a64, 0x6a65, 0x0000, 0x0000, 0x4a64, 0x6a66, 0x0000, 0x3a40,
	0x0000, 0x4e23, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6a6b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6a6c, 0x3e58, 0x6a6a, 0x0000, 0x0000, 0x0000,
	0x4d67, 0x6a67, 0x0000, 0x0000, 0x6a69, 0x403d, 0x3f7e, 0x0000,
};

static unsigned short const unicode_to_jisx0208_89[] = {
	/* 0x8900 - 0x89ff */
	0x0000, 0x0000, 0x6a68, 0x0000, 0x6a6d, 0x0000, 0x0000, 0x4a23,
	0x0000, 0x0000, 0x6a6f, 0x0000, 0x6a6e, 0x0000, 0x0000, 0x0000,
	0x336c, 0x0000, 0x4b2b, 0x6a70, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a7c, 0x6a72, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a73, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6a74, 0x6a75, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a79, 0x0000,
	0x6a7a, 0x0000, 0x0000, 0x6a78, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6a76, 0x0000, 0x6a71, 0x6a77, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6a7b, 0x7037, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3228, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a7e, 0x365f,
	0x6a7d, 0x0000, 0x0000, 0x0000, 0x6b22, 0x0000, 0x6b21, 0x0000,
	0x0000, 0x0000, 0x6b24, 0x0000, 0x0000, 0x6b23, 0x0000, 0x6b25,
	0x0000, 0x0000, 0x3d31, 0x0000, 0x6b26, 0x0000, 0x0000, 0x6b27,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b28, 0x403e,
	0x0000, 0x4d57, 0x0000, 0x6b29, 0x0000, 0x0000, 0x4a24, 0x4746,
	0x6b2a, 0x0000, 0x6b2b, 0x382b, 0x0000, 0x0000, 0x0000, 0x352c,
	0x0000, 0x0000, 0x0000, 0x6b2c, 0x0000, 0x0000, 0x3b6b, 0x4741,
	0x6b2d, 0x0000, 0x3350, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6b2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b30, 0x4d77,
	0x0000, 0x6b2f, 0x3f46, 0x0000, 0x6b31, 0x0000, 0x0000, 0x6b32,
	0x0000, 0x0000, 0x6b33, 0x3451, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6b34, 0x0000, 0x0000, 0x6b35, 0x0000, 0x6b36,
	0x6b37, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3351, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6b38, 0x0000, 0x6b39, 0x6b3a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3272, 0x0000, 0x0000, 0x3f28, 0x6b3b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6b3c, 0x0000, 0x0000, 0x0000,
	0x6b3d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_8a[] = {
	/* 0x8a00 - 0x8aff */
	0x3840, 0x0000, 0x447b, 0x6b3e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3757, 0x0000, 0x3f56, 0x0000, 0x6b41, 0x0000, 0x4624, 0x0000,
	0x6b40, 0x0000, 0x0000, 0x3731, 0x0000, 0x0000, 0x6b3f, 0x4277,
	0x352d, 0x0000, 0x0000, 0x6b42, 0x0000, 0x6b43, 0x0000, 0x3e59,
	0x0000, 0x0000, 0x0000, 0x376d, 0x0000, 0x6b44, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4b2c, 0x0000, 0x0000, 0x405f, 0x0000, 0x0000,
	0x0000, 0x3576, 0x0000, 0x4c75, 0x414a, 0x0000, 0x6b45, 0x0000,
	0x0000, 0x0000, 0x3f47, 0x4370, 0x3e5a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6b46, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b49, 0x0000,
	0x6b4a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3a3e, 0x4242, 0x6b48, 0x0000, 0x3e5b, 0x493e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6b47, 0x0000, 0x0000, 0x3b6c, 0x0000,
	0x3153, 0x0000, 0x6b4e, 0x3758, 0x0000, 0x0000, 0x3b6e, 0x0000,
	0x0000, 0x3b6d, 0x0000, 0x4f4d, 0x6b4d, 0x6b4c, 0x4127, 0x0000,
	0x354d, 0x4f43, 0x333a, 0x3e5c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6b4b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6b50, 0x0000, 0x6b51, 0x6b4f, 0x0000, 0x3858,
	0x0000, 0x4d40, 0x0000, 0x0000, 0x3b6f, 0x4727, 0x0000, 0x0000,
	0x0000, 0x6b54, 0x0000, 0x4040, 0x0000, 0x4342, 0x0000, 0x0000,
	0x4d36, 0x0000, 0x6b57, 0x0000, 0x0000, 0x0000, 0x386c, 0x0000,
	0x403f, 0x6b53, 0x0000, 0x6b58, 0x386d, 0x6b55, 0x6b56, 0x0000,
	0x6b52, 0x0000, 0x0000, 0x0000, 0x4062, 0x4649, 0x0000, 0x0000,
	0x432f, 0x0000, 0x325d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4870, 0x0000, 0x0000, 0x3543, 0x0000, 0x0000, 0x4434,
	0x0000, 0x0000, 0x6b5b, 0x0000, 0x6b59, 0x0000, 0x0000, 0x434c,
	0x0000, 0x0000, 0x0000, 0x4041, 0x3452, 0x6b5a, 0x0000, 0x3f5b,
	0x0000, 0x0000, 0x4e4a, 0x0000, 0x0000, 0x0000, 0x4f40, 0x0000,
	0x0000, 0x0000, 0x6b5c, 0x6b67, 0x4435, 0x0000, 0x6b66, 0x0000,
	0x6b63, 0x6b6b, 0x6b64, 0x0000, 0x6b60, 0x0000, 0x447c, 0x6b5f,
	0x0000, 0x0000, 0x0000, 0x6b5d, 0x0000, 0x4d21, 0x3b70, 0x0000,
	0x0000, 0x6b61, 0x0000, 0x6b5e, 0x0000, 0x0000, 0x0000, 0x6b65,
	0x3d74, 0x0000, 0x3841, 0x0000, 0x0000, 0x0000, 0x427a, 0x0000,
};

static unsigned short const unicode_to_jisx0208_8b[] = {
	/* 0x8b00 - 0x8bff */
	0x4b45, 0x315a, 0x3062, 0x0000, 0x4625, 0x0000, 0x0000, 0x6b69,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6b68, 0x0000, 0x4666, 0x0000,
	0x6b6d, 0x0000, 0x0000, 0x0000, 0x6b62, 0x0000, 0x6b6c, 0x6b6e,
	0x0000, 0x382c, 0x6b6a, 0x3956, 0x0000, 0x3c55, 0x0000, 0x0000,
	0x6b6f, 0x4d58, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b72, 0x0000,
	0x6b75, 0x0000, 0x0000, 0x6b73, 0x4935, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6b70, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3660, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b74, 0x0000,
	0x0000, 0x6b76, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6b7a, 0x0000, 0x0000, 0x6b77, 0x0000, 0x6b79, 0x6b78,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b7b, 0x0000,
	0x3c31, 0x0000, 0x6b7d, 0x6b7c, 0x4968, 0x0000, 0x0000, 0x6c21,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3759, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6b7e, 0x6c22, 0x0000, 0x0000, 0x6c23,
	0x3544, 0x6641, 0x3e79, 0x0000, 0x6c24, 0x0000, 0x0000, 0x386e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c25, 0x0000, 0x0000,
	0x6c26, 0x0000, 0x0000, 0x3b3e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5a4e, 0x0000, 0x6c27, 0x0000, 0x6c28, 0x0000,
	0x3d32, 0x0000, 0x6c29, 0x6c2a, 0x0000, 0x0000, 0x6c2b, 0x0000,
	0x0000, 0x6c2c, 0x6c2d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_8c[] = {
	/* 0x8c00 - 0x8cff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x432b,
	0x0000, 0x0000, 0x6c2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c30,
	0x0000, 0x6c2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4626, 0x0000,
	0x6c31, 0x0000, 0x4b2d, 0x0000, 0x6c32, 0x0000, 0x6c33, 0x0000,
	0x6c34, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c35, 0x0000, 0x0000,
	0x0000, 0x0000, 0x465a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3e5d, 0x6c36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x396b, 0x502e, 0x6c37, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6c38, 0x493f, 0x6c39, 0x0000, 0x6c41, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c3a, 0x0000, 0x0000, 0x6c3c, 0x0000, 0x0000,
	0x0000, 0x6c3b, 0x6c3d, 0x0000, 0x4b46, 0x6c3e, 0x6c3f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c40, 0x0000, 0x0000, 0x0000,
	0x6c42, 0x0000, 0x0000, 0x0000, 0x0000, 0x332d, 0x4467, 0x0000,
	0x4969, 0x3a62, 0x3957, 0x0000, 0x0000, 0x0000, 0x0000, 0x494f,
	0x325f, 0x484e, 0x6c45, 0x3453, 0x4055, 0x6c44, 0x6c49, 0x4379,
	0x4c63, 0x0000, 0x6c47, 0x6c48, 0x352e, 0x0000, 0x6c4a, 0x4763,
	0x425f, 0x0000, 0x0000, 0x4871, 0x453d, 0x6c46, 0x0000, 0x4b47,
	0x326c, 0x6c4c, 0x4f28, 0x4442, 0x4f45, 0x0000, 0x0000, 0x3b71,
	0x6c4b, 0x0000, 0x4231, 0x0000, 0x0000, 0x6c5c, 0x4128, 0x0000,
	0x0000, 0x4678, 0x0000, 0x4950, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c4f, 0x3b3f, 0x3b72, 0x0000, 0x3e5e, 0x0000,
	0x4765, 0x0000, 0x382d, 0x6c4e, 0x6c4d, 0x0000, 0x496a, 0x0000,
	0x0000, 0x0000, 0x3c41, 0x0000, 0x0000, 0x4552, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c51, 0x6c52, 0x3958, 0x6c50, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_8d[] = {
	/* 0x8d00 - 0x8dff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c53, 0x6c54, 0x0000, 0x6c56,
	0x4223, 0x0000, 0x6c55, 0x3466, 0x0000, 0x6c58, 0x0000, 0x6c57,
	0x6c59, 0x0000, 0x0000, 0x6c5b, 0x6c5d, 0x0000, 0x6c5e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4056, 0x0000, 0x3c4f, 0x6c5f,
	0x0000, 0x0000, 0x0000, 0x3352, 0x0000, 0x6c60, 0x0000, 0x0000,
	0x4176, 0x6c61, 0x0000, 0x6c62, 0x496b, 0x0000, 0x0000, 0x352f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6c63, 0x0000, 0x0000, 0x0000, 0x4436, 0x0000, 0x0000,
	0x0000, 0x0000, 0x315b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6c64, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c71, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3f76, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x422d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c67, 0x0000, 0x0000, 0x0000, 0x6c66, 0x0000,
	0x0000, 0x0000, 0x6c65, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6c6d, 0x6c6b, 0x0000, 0x0000, 0x6c68,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c6a, 0x0000,
	0x0000, 0x0000, 0x6c69, 0x6c6c, 0x0000, 0x3577, 0x0000, 0x6c70,
	0x0000, 0x4057, 0x0000, 0x6c71, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3859, 0x0000, 0x6c6e, 0x6c6f, 0x0000, 0x0000, 0x0000, 0x4f29,
	0x0000, 0x0000, 0x0000, 0x4437, 0x0000, 0x4129, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c72, 0x0000, 0x0000, 0x6c75,
};

static unsigned short const unicode_to_jisx0208_8e[] = {
	/* 0x8e00 - 0x8eff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6c73, 0x6c74, 0x4d59, 0x0000, 0x0000, 0x0000, 0x0000, 0x4627,
	0x6c78, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c76, 0x6c77, 0x6c79,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d29, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6c7c, 0x0000, 0x0000, 0x0000, 0x6c7d, 0x6c7b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c7a, 0x0000, 0x447d, 0x0000, 0x0000, 0x6d21,
	0x6d25, 0x6d22, 0x6c7e, 0x0000, 0x6d23, 0x0000, 0x0000, 0x0000,
	0x6d24, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d2b, 0x0000, 0x0000,
	0x0000, 0x6d26, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4058,
	0x6d28, 0x0000, 0x0000, 0x6d2a, 0x6d27, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d2d, 0x0000, 0x3d33, 0x0000, 0x6d2c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6d2e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6d2f, 0x0000, 0x0000, 0x6d32, 0x6d31, 0x0000, 0x6d30,
	0x0000, 0x0000, 0x6d34, 0x6d33, 0x0000, 0x4c76, 0x0000, 0x0000,
	0x0000, 0x6d36, 0x0000, 0x6d35, 0x6d37, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6d38, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6d3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d39, 0x3f48, 0x6d3b, 0x0000, 0x0000, 0x366d,
	0x6d3c, 0x6d3e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d3f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d40, 0x6d3d, 0x0000,
	0x6d41, 0x0000, 0x3c56, 0x6d42, 0x3530, 0x3733, 0x0000, 0x0000,
	0x0000, 0x0000, 0x382e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6d43, 0x0000, 0x0000, 0x0000, 0x4670,
	0x0000, 0x0000, 0x453e, 0x6d44, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6d47, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c34, 0x0000, 0x0000, 0x6d46, 0x6d45, 0x375a, 0x6d48, 0x0000,
};

static unsigned short const unicode_to_jisx0208_8f[] = {
	/* 0x8f00 - 0x8fff */
	0x0000, 0x0000, 0x0000, 0x3353, 0x0000, 0x6d4a, 0x0000, 0x0000,
	0x0000, 0x3a5c, 0x6d49, 0x0000, 0x6d52, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d4c, 0x6d4e, 0x4a65, 0x6d4b, 0x0000, 0x0000,
	0x0000, 0x6d4d, 0x0000, 0x6d51, 0x6d4f, 0x3531, 0x0000, 0x6d50,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d53, 0x0000,
	0x0000, 0x475a, 0x4e58, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d34,
	0x0000, 0x0000, 0x0000, 0x6d54, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4d22, 0x6d56, 0x0000, 0x6d55, 0x0000, 0x0000, 0x6d59, 0x4d41,
	0x0000, 0x0000, 0x6d58, 0x0000, 0x336d, 0x6d57, 0x6d5c, 0x0000,
	0x0000, 0x6d5b, 0x0000, 0x0000, 0x6d5a, 0x4532, 0x6d5d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d5e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6d5f, 0x0000, 0x0000, 0x396c,
	0x0000, 0x3725, 0x6d60, 0x6d61, 0x6d62, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3f49, 0x6d63, 0x0000, 0x3c2d, 0x6d64,
	0x0000, 0x0000, 0x0000, 0x6d65, 0x0000, 0x0000, 0x0000, 0x5221,
	0x517e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d66, 0x6570, 0x6d67,
	0x4324, 0x3f2b, 0x4740, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d68,
	0x0000, 0x0000, 0x4a55, 0x4454, 0x397e, 0x0000, 0x0000, 0x4329,
	0x0000, 0x0000, 0x312a, 0x0000, 0x4b78, 0x3f57, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x375e, 0x0000,
	0x0000, 0x3661, 0x0000, 0x0000, 0x4a56, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d69, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d6b, 0x0000, 0x0000, 0x6d6a, 0x3260, 0x0000,
	0x0000, 0x4676, 0x6d6c, 0x4777, 0x0000, 0x4533, 0x0000, 0x6d6d,
	0x3d52, 0x0000, 0x0000, 0x0000, 0x6d6f, 0x0000, 0x0000, 0x4c42,
	0x6d7e, 0x6d71, 0x6d72, 0x0000, 0x0000, 0x4449, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_90[] = {
	/* 0x9000 - 0x90ff */
	0x4260, 0x4177, 0x0000, 0x4628, 0x0000, 0x6d70, 0x3555, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6d79, 0x0000, 0x6d76, 0x6e25, 0x4629,
	0x4360, 0x6d73, 0x0000, 0x447e, 0x4553, 0x6d74, 0x6d78, 0x3f60,
	0x0000, 0x4767, 0x444c, 0x0000, 0x0000, 0x4042, 0x6d77, 0x422e,
	0x4224, 0x6d75, 0x3029, 0x4f22, 0x0000, 0x0000, 0x0000, 0x6d7a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4261, 0x0000,
	0x0000, 0x3d35, 0x3f4a, 0x0000, 0x0000, 0x6d7c, 0x6d7b, 0x0000,
	0x306f, 0x6d7d, 0x0000, 0x0000, 0x492f, 0x0000, 0x6e27, 0x0000,
	0x0000, 0x465b, 0x3f6b, 0x0000, 0x0000, 0x4359, 0x0000, 0x3678,
	0x0000, 0x6e26, 0x4d37, 0x313f, 0x0000, 0x4a57, 0x3261, 0x6e21,
	0x6e22, 0x6e23, 0x6e24, 0x463b, 0x4323, 0x3063, 0x6e28, 0x0000,
	0x6e29, 0x7423, 0x0000, 0x0000, 0x423d, 0x0000, 0x6e2a, 0x0000,
	0x3173, 0x414c, 0x0000, 0x382f, 0x0000, 0x4d5a, 0x0000, 0x0000,
	0x6e2b, 0x452c, 0x0000, 0x0000, 0x0000, 0x4178, 0x3c57, 0x6e2c,
	0x0000, 0x0000, 0x6e2f, 0x0000, 0x0000, 0x3d65, 0x6e2d, 0x412b,
	0x412a, 0x0000, 0x3064, 0x0000, 0x4e4b, 0x6e31, 0x0000, 0x4872,
	0x6e33, 0x6e32, 0x6e30, 0x6364, 0x3454, 0x0000, 0x0000, 0x6d6e,
	0x0000, 0x6e35, 0x6e34, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e36,
	0x0000, 0x4d38, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4661, 0x0000, 0x0000, 0x4b2e, 0x0000,
	0x6e37, 0x0000, 0x3c59, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e38,
	0x0000, 0x6e39, 0x0000, 0x0000, 0x0000, 0x6e3a, 0x0000, 0x0000,
	0x4521, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x306a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3959, 0x0000, 0x0000, 0x0000, 0x4f3a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6e3e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3734, 0x6e3b, 0x0000, 0x6e3c, 0x0000, 0x0000, 0x0000,
	0x4974, 0x0000, 0x0000, 0x0000, 0x0000, 0x3354, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d39, 0x0000, 0x363f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4554, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_91[] = {
	/* 0x9100 - 0x91ff */
	0x0000, 0x0000, 0x6e3f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6e40, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6e41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4522, 0x0000, 0x0000,
	0x6e43, 0x0000, 0x6e42, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4653, 0x6e44, 0x3d36, 0x3c60, 0x475b, 0x4371, 0x0000,
	0x0000, 0x0000, 0x3c72, 0x0000, 0x3f6c, 0x0000, 0x6e45, 0x0000,
	0x6e46, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3f5d, 0x6e47, 0x0000, 0x6e48, 0x0000, 0x0000,
	0x0000, 0x6e49, 0x4d6f, 0x0000, 0x3d37, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6e4b, 0x6e4a, 0x0000, 0x395a, 0x0000, 0x3973,
	0x3b40, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6e4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d66,
	0x0000, 0x6e4d, 0x0000, 0x6e4c, 0x0000, 0x4269, 0x0000, 0x0000,
	0x386f, 0x0000, 0x4043, 0x0000, 0x0000, 0x0000, 0x0000, 0x4830,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d39, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6e4f, 0x0000, 0x3e5f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6e52, 0x6e50, 0x0000, 0x0000, 0x0000, 0x6e51,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6e54, 0x6e53, 0x0000, 0x0000,
	0x3e7a, 0x0000, 0x6e55, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6e56, 0x6e57, 0x0000, 0x0000, 0x0000, 0x0000, 0x4850, 0x3a53,
	0x3c61, 0x6e58, 0x0000, 0x6e59, 0x4e24, 0x3d45, 0x4c6e, 0x4e4c,
	0x6e5a, 0x3662, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e5b, 0x0000,
	0x4523, 0x0000, 0x0000, 0x6e5e, 0x3378, 0x3f4b, 0x0000, 0x6e5c,
	0x0000, 0x6e5d, 0x0000, 0x4460, 0x0000, 0x0000, 0x4b55, 0x367c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e60, 0x6e61, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6e5f, 0x0000, 0x0000, 0x6e63,
};

static unsigned short const unicode_to_jisx0208_92[] = {
	/* 0x9200 - 0x92ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x465f, 0x3343, 0x0000,
	0x0000, 0x6e67, 0x0000, 0x0000, 0x6e64, 0x6e66, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e62, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f4f, 0x0000, 0x0000, 0x6e65, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4e6b, 0x0000, 0x0000, 0x385a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e6f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4534, 0x6e6a, 0x0000, 0x0000,
	0x6e6d, 0x6e6b, 0x0000, 0x6e70, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6e71, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e69,
	0x0000, 0x0000, 0x6e76, 0x3174, 0x0000, 0x0000, 0x6e68, 0x0000,
	0x0000, 0x0000, 0x482d, 0x0000, 0x6e6c, 0x0000, 0x3e60, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x395b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b48, 0x0000,
	0x3664, 0x0000, 0x0000, 0x3d46, 0x0000, 0x463c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x412d, 0x0000, 0x6e74, 0x0000, 0x6e6e, 0x6e73, 0x0000,
	0x4c43, 0x0000, 0x4438, 0x6e75, 0x6e72, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x412c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e79,
	0x0000, 0x6e78, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e77,
	0x0000, 0x0000, 0x4b2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d7b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6e7a, 0x4a5f, 0x0000, 0x0000, 0x3154, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4946, 0x4372, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3578, 0x0000, 0x6e7c, 0x0000, 0x395d, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_93[] = {
	/* 0x9300 - 0x93ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b2c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6e7b,
	0x3f6d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3f6e, 0x6f21, 0x6f23, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3e7b, 0x0000, 0x6f22, 0x6f24, 0x0000, 0x0000, 0x3653, 0x0000,
	0x4945, 0x0000, 0x0000, 0x3c62, 0x4f23, 0x0000, 0x6e7e, 0x3a78,
	0x0000, 0x0000, 0x4f3f, 0x0000, 0x0000, 0x6f26, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6f25, 0x6f27, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6e7d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4669, 0x0000, 0x4555, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4457, 0x0000, 0x6f2c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4343, 0x6f28, 0x0000, 0x0000, 0x0000,
	0x6f29, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x372d, 0x0000, 0x6f2b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3830, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6f2a, 0x0000, 0x3e61, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3379, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6f30, 0x0000, 0x3a3f, 0x4179,
	0x0000, 0x0000, 0x444a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x333b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6f2e, 0x6f2f, 0x4443, 0x0000,
	0x6f2d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f31, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6f37, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6f3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6f39, 0x452d, 0x0000, 0x0000, 0x0000, 0x0000, 0x6f32, 0x6f33,
	0x6f36, 0x0000, 0x0000, 0x0000, 0x0000, 0x6f38, 0x0000, 0x0000,
	0x0000, 0x3640, 0x0000, 0x0000, 0x6f3b, 0x6f35, 0x0000, 0x0000,
	0x6f34, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_94[] = {
	/* 0x9400 - 0x94ff */
	0x0000, 0x0000, 0x0000, 0x6f3f, 0x0000, 0x0000, 0x0000, 0x6f40,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6f41, 0x0000, 0x0000, 0x6f3e, 0x6f3d, 0x0000, 0x0000, 0x0000,
	0x3e62, 0x462a, 0x6f3c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f45, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6f43, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6f44, 0x6f42, 0x0000,
	0x4278, 0x0000, 0x6f46, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f47, 0x0000, 0x0000, 0x6f49, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3455, 0x6f48, 0x4c7a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6f54, 0x6f4a, 0x0000, 0x0000, 0x6f4d, 0x0000,
	0x6f4b, 0x0000, 0x6f4c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6f4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6f50, 0x0000, 0x0000, 0x0000, 0x0000, 0x6f51, 0x0000, 0x6f52,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6f55, 0x6f53, 0x6f56, 0x6f58,
	0x0000, 0x6f57, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_95[] = {
	/* 0x9500 - 0x95ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4439,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c67, 0x0000, 0x6f59, 0x412e, 0x0000, 0x0000, 0x0000, 0x6f5a,
	0x0000, 0x4a44, 0x6f5b, 0x332b, 0x0000, 0x0000, 0x0000, 0x313c,
	0x0000, 0x3457, 0x0000, 0x3456, 0x6f5c, 0x0000, 0x6f5d, 0x0000,
	0x6f5e, 0x6f5f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6f60, 0x0000, 0x3458, 0x3355, 0x395e, 0x4836, 0x0000, 0x6f62,
	0x6f61, 0x0000, 0x0000, 0x0000, 0x0000, 0x6f63, 0x0000, 0x0000,
	0x0000, 0x0000, 0x315c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f66, 0x0000, 0x6f65, 0x6f64, 0x0000, 0x6f67, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6f6a, 0x0000, 0x0000, 0x0000, 0x3047,
	0x0000, 0x0000, 0x6f68, 0x0000, 0x6f6c, 0x6f6b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6f6e, 0x6f6d, 0x6f6f, 0x0000,
	0x462e, 0x0000, 0x0000, 0x0000, 0x6f70, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f71, 0x6f73, 0x0000, 0x0000, 0x6f72, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_96[] = {
	/* 0x9600 - 0x96ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x496c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6f74, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6f75, 0x0000, 0x3a65, 0x0000, 0x0000, 0x0000, 0x6f76, 0x6f77,
	0x0000, 0x0000, 0x4b49, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x414b, 0x0000, 0x0000, 0x0000, 0x3024,
	0x424b, 0x0000, 0x6f78, 0x0000, 0x496d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6f7b, 0x6f79, 0x395f, 0x0000, 0x6f7a,
	0x3842, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a45, 0x6f7d, 0x7021, 0x6f7e, 0x7022,
	0x0000, 0x0000, 0x3121, 0x3f58, 0x3d7c, 0x3459, 0x7023, 0x0000,
	0x0000, 0x0000, 0x4766, 0x0000, 0x7025, 0x0000, 0x0000, 0x0000,
	0x3122, 0x0000, 0x7024, 0x4444, 0x0000, 0x4e4d, 0x462b, 0x6f7c,
	0x4e26, 0x0000, 0x3831, 0x0000, 0x0000, 0x4d5b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3679, 0x4e34, 0x0000,
	0x3728, 0x0000, 0x4262, 0x6721, 0x0000, 0x7026, 0x332c, 0x3f6f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3356, 0x7028, 0x0000, 0x7029,
	0x7027, 0x3764, 0x0000, 0x3a5d, 0x3e63, 0x0000, 0x0000, 0x0000,
	0x3123, 0x0000, 0x0000, 0x4e59, 0x0000, 0x0000, 0x0000, 0x702b,
	0x6e2e, 0x0000, 0x702a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x702e, 0x702c, 0x702d, 0x0000, 0x702f, 0x0000, 0x7030, 0x4e6c,
	0x7031, 0x7032, 0x0000, 0x4049, 0x483b, 0x0000, 0x0000, 0x0000,
	0x3f7d, 0x3467, 0x0000, 0x0000, 0x4d3a, 0x326d, 0x3d38, 0x385b,
	0x0000, 0x7035, 0x0000, 0x7034, 0x3b73, 0x7036, 0x7033, 0x0000,
	0x0000, 0x3b28, 0x0000, 0x0000, 0x0000, 0x703a, 0x6a2d, 0x0000,
	0x0000, 0x5256, 0x0000, 0x3f77, 0x7038, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4e25, 0x4671, 0x0000, 0x0000, 0x0000, 0x0000,
	0x312b, 0x0000, 0x4063, 0x3c36, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a37, 0x0000, 0x3140, 0x0000, 0x0000, 0x0000, 0x4e6d, 0x4d6b,
	0x0000, 0x703b, 0x0000, 0x4545, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_97[] = {
	/* 0x9700 - 0x97ff */
	0x3c7b, 0x0000, 0x0000, 0x0000, 0x703c, 0x0000, 0x703d, 0x3f4c,
	0x703e, 0x0000, 0x4e6e, 0x0000, 0x0000, 0x7039, 0x7040, 0x7042,
	0x0000, 0x7041, 0x0000, 0x703f, 0x0000, 0x0000, 0x7043, 0x0000,
	0x0000, 0x7044, 0x0000, 0x0000, 0x417a, 0x0000, 0x3262, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7045, 0x0000, 0x0000, 0x4c38,
	0x0000, 0x0000, 0x7046, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7047, 0x0000, 0x4f2a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b31, 0x7048, 0x0000, 0x0000, 0x0000, 0x7049, 0x704a, 0x0000,
	0x0000, 0x0000, 0x704e, 0x0000, 0x704b, 0x0000, 0x704c, 0x0000,
	0x704d, 0x704f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4044, 0x0000, 0x0000, 0x0000, 0x4c77, 0x0000,
	0x0000, 0x4045, 0x0000, 0x0000, 0x7050, 0x0000, 0x4873, 0x0000,
	0x7051, 0x7353, 0x4c4c, 0x0000, 0x7052, 0x0000, 0x7053, 0x0000,
	0x7054, 0x3357, 0x0000, 0x7056, 0x0000, 0x3f59, 0x0000, 0x0000,
	0x0000, 0x7057, 0x0000, 0x0000, 0x3724, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7058, 0x705c, 0x0000, 0x705a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x705b, 0x0000, 0x0000, 0x3373, 0x7059, 0x705d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x705e, 0x0000, 0x3048, 0x0000, 0x705f,
	0x7060, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3e64, 0x0000, 0x0000, 0x0000, 0x7061, 0x0000, 0x0000, 0x0000,
	0x3547, 0x0000, 0x0000, 0x7064, 0x0000, 0x0000, 0x7063, 0x0000,
	0x7062, 0x0000, 0x0000, 0x6b71, 0x0000, 0x4a5c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x7065, 0x7066, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x7067, 0x0000, 0x0000, 0x7068, 0x0000,
	0x7069, 0x0000, 0x0000, 0x706a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x345a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x706b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x706c, 0x4723, 0x0000,
	0x0000, 0x0000, 0x706e, 0x323b, 0x0000, 0x7071, 0x7070, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3124, 0x0000, 0x0000, 0x0000, 0x3641,
};

static unsigned short const unicode_to_jisx0208_98[] = {
	/* 0x9800 - 0x98ff */
	0x0000, 0x4a47, 0x443a, 0x3a22, 0x0000, 0x3960, 0x3d67, 0x0000,
	0x3f5c, 0x0000, 0x0000, 0x0000, 0x7073, 0x0000, 0x0000, 0x7072,
	0x4d42, 0x3468, 0x4852, 0x465c, 0x0000, 0x0000, 0x0000, 0x3f7c,
	0x4e4e, 0x0000, 0x375b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7076, 0x0000, 0x0000, 0x7075, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4b4b, 0x462c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3150, 0x0000, 0x0000, 0x7077,
	0x7074, 0x0000, 0x0000, 0x4951, 0x4d6a, 0x7078, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7079, 0x0000,
	0x0000, 0x0000, 0x0000, 0x707b, 0x426a, 0x335b, 0x335c, 0x707a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3469, 0x3832, 0x0000, 0x0000,
	0x346a, 0x0000, 0x0000, 0x453f, 0x0000, 0x0000, 0x4e60, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x385c,
	0x0000, 0x0000, 0x0000, 0x707c, 0x0000, 0x0000, 0x0000, 0x707d,
	0x707e, 0x7121, 0x0000, 0x7123, 0x7122, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4977, 0x0000, 0x7124, 0x0000, 0x0000, 0x0000, 0x0000, 0x7125,
	0x0000, 0x7126, 0x0000, 0x0000, 0x0000, 0x0000, 0x7127, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x7129, 0x7128, 0x0000, 0x712a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4874, 0x664c, 0x0000, 0x0000, 0x3f29,
	0x0000, 0x0000, 0x3532, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x712b, 0x0000, 0x712c, 0x0000, 0x522c, 0x5d3b, 0x4853,
	0x0000, 0x0000, 0x307b, 0x0000, 0x303b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3b74, 0x4b30, 0x3e7e, 0x0000,
};

static unsigned short const unicode_to_jisx0208_99[] = {
	/* 0x9900 - 0x99ff */
	0x0000, 0x0000, 0x0000, 0x712d, 0x0000, 0x4c5f, 0x0000, 0x0000,
	0x0000, 0x712e, 0x4d5c, 0x0000, 0x3142, 0x0000, 0x0000, 0x0000,
	0x3b41, 0x0000, 0x712f, 0x326e, 0x7130, 0x0000, 0x0000, 0x0000,
	0x7131, 0x0000, 0x0000, 0x0000, 0x0000, 0x7133, 0x7134, 0x0000,
	0x7136, 0x7132, 0x0000, 0x0000, 0x7135, 0x0000, 0x0000, 0x0000,
	0x345b, 0x0000, 0x0000, 0x0000, 0x7137, 0x0000, 0x7138, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7139, 0x713a, 0x0000,
	0x0000, 0x0000, 0x713b, 0x0000, 0x0000, 0x713d, 0x0000, 0x0000,
	0x0000, 0x713c, 0x0000, 0x713f, 0x7142, 0x0000, 0x0000, 0x0000,
	0x713e, 0x7140, 0x7141, 0x0000, 0x0000, 0x7143, 0x0000, 0x3642,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c73, 0x7144,
	0x7145, 0x3961, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7146, 0x0000, 0x0000,
	0x333e, 0x0000, 0x0000, 0x0000, 0x474f, 0x7147, 0x7148, 0x0000,
	0x0000, 0x0000, 0x0000, 0x435a, 0x466b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7149, 0x0000, 0x0000, 0x0000,
	0x0000, 0x477d, 0x0000, 0x0000, 0x424c, 0x3158, 0x366e, 0x0000,
	0x366f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4373, 0x714e, 0x3670, 0x0000, 0x0000, 0x326f, 0x0000, 0x0000,
	0x714d, 0x0000, 0x0000, 0x714b, 0x0000, 0x714c, 0x0000, 0x714a,
	0x0000, 0x0000, 0x7158, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x714f, 0x7150, 0x0000,
	0x0000, 0x7151, 0x7152, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7154, 0x0000, 0x0000, 0x7153, 0x0000, 0x0000, 0x0000, 0x3d59,
};

static unsigned short const unicode_to_jisx0208_9a[] = {
	/* 0x9a00 - 0x9aff */
	0x0000, 0x7155, 0x0000, 0x0000, 0x0000, 0x7157, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3533, 0x7156,
	0x0000, 0x0000, 0x417b, 0x3833, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7159, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x424d, 0x0000, 0x0000, 0x715a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x462d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x715b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7160, 0x0000,
	0x715e, 0x0000, 0x715d, 0x715f, 0x0000, 0x715c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7162, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7161, 0x0000, 0x7164,
	0x0000, 0x0000, 0x3643, 0x7163, 0x0000, 0x0000, 0x0000, 0x7165,
	0x0000, 0x0000, 0x7166, 0x0000, 0x7168, 0x7167, 0x0000, 0x0000,
	0x0000, 0x7169, 0x716b, 0x716a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x397c, 0x0000, 0x0000, 0x0000, 0x0000, 0x716c, 0x0000, 0x0000,
	0x716d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x333c, 0x0000, 0x0000, 0x0000, 0x716e, 0x0000, 0x0000, 0x0000,
	0x716f, 0x0000, 0x0000, 0x0000, 0x3f71, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7170,
	0x0000, 0x7171, 0x0000, 0x7172, 0x7173, 0x0000, 0x0000, 0x0000,
	0x3962, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7174, 0x7175,
	0x0000, 0x0000, 0x7176, 0x7177, 0x0000, 0x0000, 0x7178, 0x0000,
	0x0000, 0x0000, 0x4831, 0x717a, 0x0000, 0x4926, 0x717b, 0x7179,
	0x0000, 0x717d, 0x0000, 0x0000, 0x717c, 0x0000, 0x0000, 0x717e,
	0x0000, 0x0000, 0x0000, 0x7221, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_9b[] = {
	/* 0x9b00 - 0x9bff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7222, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7223, 0x0000, 0x7224, 0x0000, 0x0000, 0x0000, 0x0000, 0x7225,
	0x0000, 0x0000, 0x7226, 0x7227, 0x0000, 0x7228, 0x0000, 0x7229,
	0x722a, 0x722b, 0x722c, 0x0000, 0x0000, 0x0000, 0x722d, 0x722e,
	0x0000, 0x5d35, 0x722f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6478, 0x3534, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3321, 0x3a32, 0x7231, 0x7230, 0x4c25, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7233, 0x7234, 0x7232,
	0x0000, 0x7235, 0x0000, 0x0000, 0x4b62, 0x0000, 0x0000, 0x0000,
	0x7236, 0x0000, 0x357b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f25,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7237, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x7239, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x303e, 0x0000,
	0x0000, 0x723a, 0x4a2b, 0x7238, 0x0000, 0x0000, 0x723b, 0x723c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x723d,
	0x723e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x723f, 0x0000, 0x4b6e, 0x3b2d, 0x0000, 0x3a7a, 0x412f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7240, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7243, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7241, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7244, 0x0000,
	0x0000, 0x3871, 0x7242, 0x0000, 0x0000, 0x0000, 0x0000, 0x7245,
	0x0000, 0x7246, 0x7247, 0x0000, 0x724b, 0x0000, 0x3b2a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4264, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x724c, 0x7249, 0x7248, 0x724a, 0x0000, 0x0000, 0x0000,
	0x375f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7250, 0x724f, 0x724e, 0x0000, 0x0000, 0x3033, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_9c[] = {
	/* 0x9c00 - 0x9cff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x725a, 0x0000, 0x7256, 0x0000,
	0x7257, 0x7253, 0x7259, 0x0000, 0x7255, 0x3362, 0x0000, 0x0000,
	0x4f4c, 0x0000, 0x7258, 0x7254, 0x7252, 0x7251, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x725c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x725f, 0x0000, 0x0000, 0x725e, 0x725d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4949, 0x725b, 0x3073,
	0x7260, 0x0000, 0x7262, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x336f, 0x724d, 0x3137, 0x0000, 0x0000, 0x7264, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7263, 0x7261,
	0x432d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4b70, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e5a,
	0x0000, 0x0000, 0x7265, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7266, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7267,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7268, 0x0000,
	0x7269, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x443b, 0x0000, 0x726a,
	0x0000, 0x4837, 0x0000, 0x726f, 0x726b, 0x0000, 0x0000, 0x0000,
	0x726c, 0x0000, 0x0000, 0x4b31, 0x4c44, 0x0000, 0x4650, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_9d[] = {
	/* 0x9d00 - 0x9dff */
	0x0000, 0x0000, 0x0000, 0x7270, 0x0000, 0x0000, 0x7271, 0x463e,
	0x726e, 0x726d, 0x0000, 0x0000, 0x0000, 0x0000, 0x322a, 0x0000,
	0x0000, 0x0000, 0x7279, 0x0000, 0x0000, 0x7278, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3175, 0x0000, 0x0000, 0x0000, 0x7276,
	0x0000, 0x0000, 0x0000, 0x7275, 0x0000, 0x0000, 0x7273, 0x0000,
	0x337b, 0x0000, 0x7272, 0x3c32, 0x3229, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3963, 0x0000, 0x0000, 0x727c, 0x727b,
	0x0000, 0x727a, 0x0000, 0x0000, 0x7277, 0x0000, 0x727d, 0x0000,
	0x727e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7325, 0x7324, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7326, 0x0000, 0x0000, 0x312d, 0x7321, 0x7322, 0x0000,
	0x3974, 0x4c39, 0x0000, 0x0000, 0x7323, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4b32, 0x0000, 0x0000, 0x732b,
	0x0000, 0x0000, 0x7327, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x732c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7329,
	0x0000, 0x7328, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x375c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x732d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x732e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x732f, 0x0000, 0x732a, 0x0000, 0x0000, 0x0000, 0x7274,
	0x0000, 0x0000, 0x7330, 0x0000, 0x4461, 0x0000, 0x0000, 0x0000,
	0x7334, 0x0000, 0x7335, 0x7333, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7332, 0x7338, 0x0000, 0x7331, 0x0000, 0x7336, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7337,
	0x0000, 0x0000, 0x0000, 0x733a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7339, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x733c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x733d, 0x0000, 0x733e,
	0x0000, 0x0000, 0x4f49, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x733b, 0x426b, 0x3a6d, 0x0000, 0x0000, 0x733f, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_9e[] = {
	/* 0x9e00 - 0x9eff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x7340, 0x7341, 0x0000, 0x0000, 0x7342, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7343, 0x0000, 0x0000,
	0x3834, 0x7344, 0x0000, 0x0000, 0x0000, 0x7345, 0x0000, 0x3c2f,
	0x0000, 0x7346, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7347, 0x0000, 0x0000, 0x7348, 0x7349, 0x0000, 0x0000, 0x0000,
	0x0000, 0x734c, 0x734a, 0x4f3c, 0x0000, 0x734b, 0x0000, 0x4e6f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x734d, 0x0000, 0x4e5b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x734e, 0x477e, 0x0000,
	0x0000, 0x734f, 0x7351, 0x0000, 0x0000, 0x7352, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7350, 0x396d, 0x4c4d, 0x4b63, 0x5677, 0x0000, 0x5d60, 0x4b7b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x322b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7354, 0x3550, 0x7355, 0x7356,
	0x7357, 0x0000, 0x3975, 0x0000, 0x7358, 0x0000, 0x0000, 0x0000,
	0x6054, 0x4c5b, 0x0000, 0x4263, 0x7359, 0x735b, 0x735a, 0x0000,
	0x735c, 0x0000, 0x0000, 0x0000, 0x0000, 0x735d, 0x0000, 0x0000,
	0x735e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x735f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7360, 0x0000, 0x7361, 0x7362,
	0x0000, 0x7363, 0x0000, 0x7364, 0x7365, 0x7366, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_9f[] = {
	/* 0x9f00 - 0x9fff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7367,
	0x7368, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4524, 0x0000,
	0x0000, 0x0000, 0x0000, 0x385d, 0x0000, 0x736a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x414d, 0x736b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x736c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4921, 0x0000, 0x0000, 0x736d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x736e, 0x6337, 0x0000, 0x0000, 0x6c5a, 0x706d,
	0x0000, 0x0000, 0x736f, 0x0000, 0x7370, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7372,
	0x7373, 0x7374, 0x4e70, 0x7371, 0x0000, 0x0000, 0x7375, 0x7376,
	0x0000, 0x0000, 0x7378, 0x0000, 0x7377, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x737a, 0x0000, 0x0000, 0x0000, 0x737b, 0x7379,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e36, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x737c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x737d, 0x6354, 0x0000, 0x0000,
	0x737e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0208_ff[] = {
	/* 0xff00 - 0xffff */
	0x0000, 0x212a, 0x0000, 0x2174, 0x2170, 0x2173, 0x2175, 0x0000,
	0x214a, 0x214b, 0x2176, 0x215c, 0x2124, 0x0000, 0x2125, 0x213f,
	0x2330, 0x2331, 0x2332, 0x2333, 0x2334, 0x2335, 0x2336, 0x2337,
	0x2338, 0x2339, 0x2127, 0x2128, 0x2163, 0x2161, 0x2164, 0x2129,
	0x2177, 0x2341, 0x2342, 0x2343, 0x2344, 0x2345, 0x2346, 0x2347,
	0x2348, 0x2349, 0x234a, 0x234b, 0x234c, 0x234d, 0x234e, 0x234f,
	0x2350, 0x2351, 0x2352, 0x2353, 0x2354, 0x2355, 0x2356, 0x2357,
	0x2358, 0x2359, 0x235a, 0x214e, 0x0000, 0x214f, 0x2130, 0x2132,
	0x212e, 0x2361, 0x2362, 0x2363, 0x2364, 0x2365, 0x2366, 0x2367,
	0x2368, 0x2369, 0x236a, 0x236b, 0x236c, 0x236d, 0x236e, 0x236f,
	0x2370, 0x2371, 0x2372, 0x2373, 0x2374, 0x2375, 0x2376, 0x2377,
	0x2378, 0x2379, 0x237a, 0x2150, 0x2143, 0x2151, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2131, 0x0000, 0x216f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const *unicode_to_jisx0208_map[0x100] = {
	/* 0x00XX - 0x0fXX */
	unicode_to_jisx0208_00,
	0, 0,
	unicode_to_jisx0208_03,
	unicode_to_jisx0208_04,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x10XX - 0x1fXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x20XX - 0x2fXX */
	unicode_to_jisx0208_20,
	unicode_to_jisx0208_21,
	unicode_to_jisx0208_22,
	unicode_to_jisx0208_23,
	unicode_to_jisx0208_24,
	unicode_to_jisx0208_25,
	unicode_to_jisx0208_26,
	0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x30XX - 0x3fXX */
	unicode_to_jisx0208_30,
	0,
	unicode_to_jisx0208_32,
	unicode_to_jisx0208_33,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x40XX - 0x4fXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	unicode_to_jisx0208_4e,
	unicode_to_jisx0208_4f,
	/* 0x50XX - 0x5fXX */
	unicode_to_jisx0208_50,
	unicode_to_jisx0208_51,
	unicode_to_jisx0208_52,
	unicode_to_jisx0208_53,
	unicode_to_jisx0208_54,
	unicode_to_jisx0208_55,
	unicode_to_jisx0208_56,
	unicode_to_jisx0208_57,
	unicode_to_jisx0208_58,
	unicode_to_jisx0208_59,
	unicode_to_jisx0208_5a,
	unicode_to_jisx0208_5b,
	unicode_to_jisx0208_5c,
	unicode_to_jisx0208_5d,
	unicode_to_jisx0208_5e,
	unicode_to_jisx0208_5f,
	/* 0x60XX - 0x6fXX */
	unicode_to_jisx0208_60,
	unicode_to_jisx0208_61,
	unicode_to_jisx0208_62,
	unicode_to_jisx0208_63,
	unicode_to_jisx0208_64,
	unicode_to_jisx0208_65,
	unicode_to_jisx0208_66,
	unicode_to_jisx0208_67,
	unicode_to_jisx0208_68,
	unicode_to_jisx0208_69,
	unicode_to_jisx0208_6a,
	unicode_to_jisx0208_6b,
	unicode_to_jisx0208_6c,
	unicode_to_jisx0208_6d,
	unicode_to_jisx0208_6e,
	unicode_to_jisx0208_6f,
	/* 0x70XX - 0x7fXX */
	unicode_to_jisx0208_70,
	unicode_to_jisx0208_71,
	unicode_to_jisx0208_72,
	unicode_to_jisx0208_73,
	unicode_to_jisx0208_74,
	unicode_to_jisx0208_75,
	unicode_to_jisx0208_76,
	unicode_to_jisx0208_77,
	unicode_to_jisx0208_78,
	unicode_to_jisx0208_79,
	unicode_to_jisx0208_7a,
	unicode_to_jisx0208_7b,
	unicode_to_jisx0208_7c,
	unicode_to_jisx0208_7d,
	unicode_to_jisx0208_7e,
	unicode_to_jisx0208_7f,
	/* 0x80XX - 0x8fXX */
	unicode_to_jisx0208_80,
	unicode_to_jisx0208_81,
	unicode_to_jisx0208_82,
	unicode_to_jisx0208_83,
	unicode_to_jisx0208_84,
	unicode_to_jisx0208_85,
	unicode_to_jisx0208_86,
	unicode_to_jisx0208_87,
	unicode_to_jisx0208_88,
	unicode_to_jisx0208_89,
	unicode_to_jisx0208_8a,
	unicode_to_jisx0208_8b,
	unicode_to_jisx0208_8c,
	unicode_to_jisx0208_8d,
	unicode_to_jisx0208_8e,
	unicode_to_jisx0208_8f,
	/* 0x90XX - 0x9fXX */
	unicode_to_jisx0208_90,
	unicode_to_jisx0208_91,
	unicode_to_jisx0208_92,
	unicode_to_jisx0208_93,
	unicode_to_jisx0208_94,
	unicode_to_jisx0208_95,
	unicode_to_jisx0208_96,
	unicode_to_jisx0208_97,
	unicode_to_jisx0208_98,
	unicode_to_jisx0208_99,
	unicode_to_jisx0208_9a,
	unicode_to_jisx0208_9b,
	unicode_to_jisx0208_9c,
	unicode_to_jisx0208_9d,
	unicode_to_jisx0208_9e,
	unicode_to_jisx0208_9f,
	/* 0xa0XX - 0xafXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xb0XX - 0xbfXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xc0XX - 0xcfXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xd0XX - 0xdfXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xe0XX - 0xefXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xf0XX - 0xffXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	unicode_to_jisx0208_ff,
};

static uint Unicode11ToJisx0208(uint h, uint l)
{
    unsigned short const *table;

    table = unicode_to_jisx0208_map[h];
    if (table != 0) {
	return table[l];
    }
    return 0x0000;
}

#ifdef USE_JISX0212

/*
 * This data is derived from Unicode 1.1,
 * JIS X 0212 (1990) to Unicode mapping table version 0.9 .
 * (In addition IBM Vender Defined Char included)
 */
static unsigned short const jisx0212_to_unicode[] = {
	/* 0x2121 - 0x217e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2221 - 0x227e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x02d8,
	0x02c7, 0x00b8, 0x02d9, 0x02dd, 0x00af, 0x02db, 0x02da, 0x007e,
	0x0384, 0x0385, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x00a1, 0x00a6, 0x00bf, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x00ba, 0x00aa, 0x00a9, 0x00ae, 0x2122,
	0x00a4, 0x2116, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2321 - 0x237e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2421 - 0x247e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2521 - 0x257e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2621 - 0x267e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0386, 0x0388, 0x0389, 0x038a, 0x03aa, 0x0000, 0x038c,
	0x0000, 0x038e, 0x03ab, 0x0000, 0x038f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x03ac, 0x03ad, 0x03ae, 0x03af, 0x03ca, 0x0390, 0x03cc,
	0x03c2, 0x03cd, 0x03cb, 0x03b0, 0x03ce, 0x0000, 0x0000,
	/* 0x2721 - 0x277e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407,
	0x0408, 0x0409, 0x040a, 0x040b, 0x040c, 0x040e, 0x040f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457,
	0x0458, 0x0459, 0x045a, 0x045b, 0x045c, 0x045e, 0x045f,
	/* 0x2821 - 0x287e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2921 - 0x297e */
		0x00c6, 0x0110, 0x0000, 0x0126, 0x0000, 0x0132, 0x0000,
	0x0141, 0x013f, 0x0000, 0x014a, 0x00d8, 0x0152, 0x0000, 0x0166,
	0x00de, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x00e6, 0x0111, 0x00f0, 0x0127, 0x0131, 0x0133, 0x0138,
	0x0142, 0x0140, 0x0149, 0x014b, 0x00f8, 0x0153, 0x00df, 0x0167,
	0x00fe, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2a21 - 0x2a7e */
		0x00c1, 0x00c0, 0x00c4, 0x00c2, 0x0102, 0x01cd, 0x0100,
	0x0104, 0x00c5, 0x00c3, 0x0106, 0x0108, 0x010c, 0x00c7, 0x010a,
	0x010e, 0x00c9, 0x00c8, 0x00cb, 0x00ca, 0x011a, 0x0116, 0x0112,
	0x0118, 0x0000, 0x011c, 0x011e, 0x0122, 0x0120, 0x0124, 0x00cd,
	0x00cc, 0x00cf, 0x00ce, 0x01cf, 0x0130, 0x012a, 0x012e, 0x0128,
	0x0134, 0x0136, 0x0139, 0x013d, 0x013b, 0x0143, 0x0147, 0x0145,
	0x00d1, 0x00d3, 0x00d2, 0x00d6, 0x00d4, 0x01d1, 0x0150, 0x014c,
	0x00d5, 0x0154, 0x0158, 0x0156, 0x015a, 0x015c, 0x0160, 0x015e,
	0x0164, 0x0162, 0x00da, 0x00d9, 0x00dc, 0x00db, 0x016c, 0x01d3,
	0x0170, 0x016a, 0x0172, 0x016e, 0x0168, 0x01d7, 0x01db, 0x01d9,
	0x01d5, 0x0174, 0x00dd, 0x0178, 0x0176, 0x0179, 0x017d, 0x017b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2b21 - 0x2b7e */
		0x00e1, 0x00e0, 0x00e4, 0x00e2, 0x0103, 0x01ce, 0x0101,
	0x0105, 0x00e5, 0x00e3, 0x0107, 0x0109, 0x010d, 0x00e7, 0x010b,
	0x010f, 0x00e9, 0x00e8, 0x00eb, 0x00ea, 0x011b, 0x0117, 0x0113,
	0x0119, 0x01f5, 0x011d, 0x011f, 0x0000, 0x0121, 0x0125, 0x00ed,
	0x00ec, 0x00ef, 0x00ee, 0x01d0, 0x0000, 0x012b, 0x012f, 0x0129,
	0x0135, 0x0137, 0x013a, 0x013e, 0x013c, 0x0144, 0x0148, 0x0146,
	0x00f1, 0x00f3, 0x00f2, 0x00f6, 0x00f4, 0x01d2, 0x0151, 0x014d,
	0x00f5, 0x0155, 0x0159, 0x0157, 0x015b, 0x015d, 0x0161, 0x015f,
	0x0165, 0x0163, 0x00fa, 0x00f9, 0x00fc, 0x00fb, 0x016d, 0x01d4,
	0x0171, 0x016b, 0x0173, 0x016f, 0x0169, 0x01d8, 0x01dc, 0x01da,
	0x01d6, 0x0175, 0x00fd, 0x00ff, 0x0177, 0x017a, 0x017e, 0x017c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2c21 - 0x2c7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2d21 - 0x2d7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2e21 - 0x2e7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2f21 - 0x2f7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x3021 - 0x307e */
		0x4e02, 0x4e04, 0x4e05, 0x4e0c, 0x4e12, 0x4e1f, 0x4e23,
	0x4e24, 0x4e28, 0x4e2b, 0x4e2e, 0x4e2f, 0x4e30, 0x4e35, 0x4e40,
	0x4e41, 0x4e44, 0x4e47, 0x4e51, 0x4e5a, 0x4e5c, 0x4e63, 0x4e68,
	0x4e69, 0x4e74, 0x4e75, 0x4e79, 0x4e7f, 0x4e8d, 0x4e96, 0x4e97,
	0x4e9d, 0x4eaf, 0x4eb9, 0x4ec3, 0x4ed0, 0x4eda, 0x4edb, 0x4ee0,
	0x4ee1, 0x4ee2, 0x4ee8, 0x4eef, 0x4ef1, 0x4ef3, 0x4ef5, 0x4efd,
	0x4efe, 0x4eff, 0x4f00, 0x4f02, 0x4f03, 0x4f08, 0x4f0b, 0x4f0c,
	0x4f12, 0x4f15, 0x4f16, 0x4f17, 0x4f19, 0x4f2e, 0x4f31, 0x4f60,
	0x4f33, 0x4f35, 0x4f37, 0x4f39, 0x4f3b, 0x4f3e, 0x4f40, 0x4f42,
	0x4f48, 0x4f49, 0x4f4b, 0x4f4c, 0x4f52, 0x4f54, 0x4f56, 0x4f58,
	0x4f5f, 0x4f63, 0x4f6a, 0x4f6c, 0x4f6e, 0x4f71, 0x4f77, 0x4f78,
	0x4f79, 0x4f7a, 0x4f7d, 0x4f7e, 0x4f81, 0x4f82, 0x4f84,
	/* 0x3121 - 0x317e */
		0x4f85, 0x4f89, 0x4f8a, 0x4f8c, 0x4f8e, 0x4f90, 0x4f92,
	0x4f93, 0x4f94, 0x4f97, 0x4f99, 0x4f9a, 0x4f9e, 0x4f9f, 0x4fb2,
	0x4fb7, 0x4fb9, 0x4fbb, 0x4fbc, 0x4fbd, 0x4fbe, 0x4fc0, 0x4fc1,
	0x4fc5, 0x4fc6, 0x4fc8, 0x4fc9, 0x4fcb, 0x4fcc, 0x4fcd, 0x4fcf,
	0x4fd2, 0x4fdc, 0x4fe0, 0x4fe2, 0x4ff0, 0x4ff2, 0x4ffc, 0x4ffd,
	0x4fff, 0x5000, 0x5001, 0x5004, 0x5007, 0x500a, 0x500c, 0x500e,
	0x5010, 0x5013, 0x5017, 0x5018, 0x501b, 0x501c, 0x501d, 0x501e,
	0x5022, 0x5027, 0x502e, 0x5030, 0x5032, 0x5033, 0x5035, 0x5040,
	0x5041, 0x5042, 0x5045, 0x5046, 0x504a, 0x504c, 0x504e, 0x5051,
	0x5052, 0x5053, 0x5057, 0x5059, 0x505f, 0x5060, 0x5062, 0x5063,
	0x5066, 0x5067, 0x506a, 0x506d, 0x5070, 0x5071, 0x503b, 0x5081,
	0x5083, 0x5084, 0x5086, 0x508a, 0x508e, 0x508f, 0x5090,
	/* 0x3221 - 0x327e */
		0x5092, 0x5093, 0x5094, 0x5096, 0x509b, 0x509c, 0x509e,
	0x509f, 0x50a0, 0x50a1, 0x50a2, 0x50aa, 0x50af, 0x50b0, 0x50b9,
	0x50ba, 0x50bd, 0x50c0, 0x50c3, 0x50c4, 0x50c7, 0x50cc, 0x50ce,
	0x50d0, 0x50d3, 0x50d4, 0x50d8, 0x50dc, 0x50dd, 0x50df, 0x50e2,
	0x50e4, 0x50e6, 0x50e8, 0x50e9, 0x50ef, 0x50f1, 0x50f6, 0x50fa,
	0x50fe, 0x5103, 0x5106, 0x5107, 0x5108, 0x510b, 0x510c, 0x510d,
	0x510e, 0x50f2, 0x5110, 0x5117, 0x5119, 0x511b, 0x511c, 0x511d,
	0x511e, 0x5123, 0x5127, 0x5128, 0x512c, 0x512d, 0x512f, 0x5131,
	0x5133, 0x5134, 0x5135, 0x5138, 0x5139, 0x5142, 0x514a, 0x514f,
	0x5153, 0x5155, 0x5157, 0x5158, 0x515f, 0x5164, 0x5166, 0x517e,
	0x5183, 0x5184, 0x518b, 0x518e, 0x5198, 0x519d, 0x51a1, 0x51a3,
	0x51ad, 0x51b8, 0x51ba, 0x51bc, 0x51be, 0x51bf, 0x51c2,
	/* 0x3321 - 0x337e */
		0x51c8, 0x51cf, 0x51d1, 0x51d2, 0x51d3, 0x51d5, 0x51d8,
	0x51de, 0x51e2, 0x51e5, 0x51ee, 0x51f2, 0x51f3, 0x51f4, 0x51f7,
	0x5201, 0x5202, 0x5205, 0x5212, 0x5213, 0x5215, 0x5216, 0x5218,
	0x5222, 0x5228, 0x5231, 0x5232, 0x5235, 0x523c, 0x5245, 0x5249,
	0x5255, 0x5257, 0x5258, 0x525a, 0x525c, 0x525f, 0x5260, 0x5261,
	0x5266, 0x526e, 0x5277, 0x5278, 0x5279, 0x5280, 0x5282, 0x5285,
	0x528a, 0x528c, 0x5293, 0x5295, 0x5296, 0x5297, 0x5298, 0x529a,
	0x529c, 0x52a4, 0x52a5, 0x52a6, 0x52a7, 0x52af, 0x52b0, 0x52b6,
	0x52b7, 0x52b8, 0x52ba, 0x52bb, 0x52bd, 0x52c0, 0x52c4, 0x52c6,
	0x52c8, 0x52cc, 0x52cf, 0x52d1, 0x52d4, 0x52d6, 0x52db, 0x52dc,
	0x52e1, 0x52e5, 0x52e8, 0x52e9, 0x52ea, 0x52ec, 0x52f0, 0x52f1,
	0x52f4, 0x52f6, 0x52f7, 0x5300, 0x5303, 0x530a, 0x530b,
	/* 0x3421 - 0x347e */
		0x530c, 0x5311, 0x5313, 0x5318, 0x531b, 0x531c, 0x531e,
	0x531f, 0x5325, 0x5327, 0x5328, 0x5329, 0x532b, 0x532c, 0x532d,
	0x5330, 0x5332, 0x5335, 0x533c, 0x533d, 0x533e, 0x5342, 0x534c,
	0x534b, 0x5359, 0x535b, 0x5361, 0x5363, 0x5365, 0x536c, 0x536d,
	0x5372, 0x5379, 0x537e, 0x5383, 0x5387, 0x5388, 0x538e, 0x5393,
	0x5394, 0x5399, 0x539d, 0x53a1, 0x53a4, 0x53aa, 0x53ab, 0x53af,
	0x53b2, 0x53b4, 0x53b5, 0x53b7, 0x53b8, 0x53ba, 0x53bd, 0x53c0,
	0x53c5, 0x53cf, 0x53d2, 0x53d3, 0x53d5, 0x53da, 0x53dd, 0x53de,
	0x53e0, 0x53e6, 0x53e7, 0x53f5, 0x5402, 0x5413, 0x541a, 0x5421,
	0x5427, 0x5428, 0x542a, 0x542f, 0x5431, 0x5434, 0x5435, 0x5443,
	0x5444, 0x5447, 0x544d, 0x544f, 0x545e, 0x5462, 0x5464, 0x5466,
	0x5467, 0x5469, 0x546b, 0x546d, 0x546e, 0x5474, 0x547f,
	/* 0x3521 - 0x357e */
		0x5481, 0x5483, 0x5485, 0x5488, 0x5489, 0x548d, 0x5491,
	0x5495, 0x5496, 0x549c, 0x549f, 0x54a1, 0x54a6, 0x54a7, 0x54a9,
	0x54aa, 0x54ad, 0x54ae, 0x54b1, 0x54b7, 0x54b9, 0x54ba, 0x54bb,
	0x54bf, 0x54c6, 0x54ca, 0x54cd, 0x54ce, 0x54e0, 0x54ea, 0x54ec,
	0x54ef, 0x54f6, 0x54fc, 0x54fe, 0x54ff, 0x5500, 0x5501, 0x5505,
	0x5508, 0x5509, 0x550c, 0x550d, 0x550e, 0x5515, 0x552a, 0x552b,
	0x5532, 0x5535, 0x5536, 0x553b, 0x553c, 0x553d, 0x5541, 0x5547,
	0x5549, 0x554a, 0x554d, 0x5550, 0x5551, 0x5558, 0x555a, 0x555b,
	0x555e, 0x5560, 0x5561, 0x5564, 0x5566, 0x557f, 0x5581, 0x5582,
	0x5586, 0x5588, 0x558e, 0x558f, 0x5591, 0x5592, 0x5593, 0x5594,
	0x5597, 0x55a3, 0x55a4, 0x55ad, 0x55b2, 0x55bf, 0x55c1, 0x55c3,
	0x55c6, 0x55c9, 0x55cb, 0x55cc, 0x55ce, 0x55d1, 0x55d2,
	/* 0x3621 - 0x367e */
		0x55d3, 0x55d7, 0x55d8, 0x55db, 0x55de, 0x55e2, 0x55e9,
	0x55f6, 0x55ff, 0x5605, 0x5608, 0x560a, 0x560d, 0x560e, 0x560f,
	0x5610, 0x5611, 0x5612, 0x5619, 0x562c, 0x5630, 0x5633, 0x5635,
	0x5637, 0x5639, 0x563b, 0x563c, 0x563d, 0x563f, 0x5640, 0x5641,
	0x5643, 0x5644, 0x5646, 0x5649, 0x564b, 0x564d, 0x564f, 0x5654,
	0x565e, 0x5660, 0x5661, 0x5662, 0x5663, 0x5666, 0x5669, 0x566d,
	0x566f, 0x5671, 0x5672, 0x5675, 0x5684, 0x5685, 0x5688, 0x568b,
	0x568c, 0x5695, 0x5699, 0x569a, 0x569d, 0x569e, 0x569f, 0x56a6,
	0x56a7, 0x56a8, 0x56a9, 0x56ab, 0x56ac, 0x56ad, 0x56b1, 0x56b3,
	0x56b7, 0x56be, 0x56c5, 0x56c9, 0x56ca, 0x56cb, 0x56cf, 0x56d0,
	0x56cc, 0x56cd, 0x56d9, 0x56dc, 0x56dd, 0x56df, 0x56e1, 0x56e4,
	0x56e5, 0x56e6, 0x56e7, 0x56e8, 0x56f1, 0x56eb, 0x56ed,
	/* 0x3721 - 0x377e */
		0x56f6, 0x56f7, 0x5701, 0x5702, 0x5707, 0x570a, 0x570c,
	0x5711, 0x5715, 0x571a, 0x571b, 0x571d, 0x5720, 0x5722, 0x5723,
	0x5724, 0x5725, 0x5729, 0x572a, 0x572c, 0x572e, 0x572f, 0x5733,
	0x5734, 0x573d, 0x573e, 0x573f, 0x5745, 0x5746, 0x574c, 0x574d,
	0x5752, 0x5762, 0x5765, 0x5767, 0x5768, 0x576b, 0x576d, 0x576e,
	0x576f, 0x5770, 0x5771, 0x5773, 0x5774, 0x5775, 0x5777, 0x5779,
	0x577a, 0x577b, 0x577c, 0x577e, 0x5781, 0x5783, 0x578c, 0x5794,
	0x5797, 0x5799, 0x579a, 0x579c, 0x579d, 0x579e, 0x579f, 0x57a1,
	0x5795, 0x57a7, 0x57a8, 0x57a9, 0x57ac, 0x57b8, 0x57bd, 0x57c7,
	0x57c8, 0x57cc, 0x57cf, 0x57d5, 0x57dd, 0x57de, 0x57e4, 0x57e6,
	0x57e7, 0x57e9, 0x57ed, 0x57f0, 0x57f5, 0x57f6, 0x57f8, 0x57fd,
	0x57fe, 0x57ff, 0x5803, 0x5804, 0x5808, 0x5809, 0x57e1,
	/* 0x3821 - 0x387e */
		0x580c, 0x580d, 0x581b, 0x581e, 0x581f, 0x5820, 0x5826,
	0x5827, 0x582d, 0x5832, 0x5839, 0x583f, 0x5849, 0x584c, 0x584d,
	0x584f, 0x5850, 0x5855, 0x585f, 0x5861, 0x5864, 0x5867, 0x5868,
	0x5878, 0x587c, 0x587f, 0x5880, 0x5881, 0x5887, 0x5888, 0x5889,
	0x588a, 0x588c, 0x588d, 0x588f, 0x5890, 0x5894, 0x5896, 0x589d,
	0x58a0, 0x58a1, 0x58a2, 0x58a6, 0x58a9, 0x58b1, 0x58b2, 0x58c4,
	0x58bc, 0x58c2, 0x58c8, 0x58cd, 0x58ce, 0x58d0, 0x58d2, 0x58d4,
	0x58d6, 0x58da, 0x58dd, 0x58e1, 0x58e2, 0x58e9, 0x58f3, 0x5905,
	0x5906, 0x590b, 0x590c, 0x5912, 0x5913, 0x5914, 0x8641, 0x591d,
	0x5921, 0x5923, 0x5924, 0x5928, 0x592f, 0x5930, 0x5933, 0x5935,
	0x5936, 0x593f, 0x5943, 0x5946, 0x5952, 0x5953, 0x5959, 0x595b,
	0x595d, 0x595e, 0x595f, 0x5961, 0x5963, 0x596b, 0x596d,
	/* 0x3921 - 0x397e */
		0x596f, 0x5972, 0x5975, 0x5976, 0x5979, 0x597b, 0x597c,
	0x598b, 0x598c, 0x598e, 0x5992, 0x5995, 0x5997, 0x599f, 0x59a4,
	0x59a7, 0x59ad, 0x59ae, 0x59af, 0x59b0, 0x59b3, 0x59b7, 0x59ba,
	0x59bc, 0x59c1, 0x59c3, 0x59c4, 0x59c8, 0x59ca, 0x59cd, 0x59d2,
	0x59dd, 0x59de, 0x59df, 0x59e3, 0x59e4, 0x59e7, 0x59ee, 0x59ef,
	0x59f1, 0x59f2, 0x59f4, 0x59f7, 0x5a00, 0x5a04, 0x5a0c, 0x5a0d,
	0x5a0e, 0x5a12, 0x5a13, 0x5a1e, 0x5a23, 0x5a24, 0x5a27, 0x5a28,
	0x5a2a, 0x5a2d, 0x5a30, 0x5a44, 0x5a45, 0x5a47, 0x5a48, 0x5a4c,
	0x5a50, 0x5a55, 0x5a5e, 0x5a63, 0x5a65, 0x5a67, 0x5a6d, 0x5a77,
	0x5a7a, 0x5a7b, 0x5a7e, 0x5a8b, 0x5a90, 0x5a93, 0x5a96, 0x5a99,
	0x5a9c, 0x5a9e, 0x5a9f, 0x5aa0, 0x5aa2, 0x5aa7, 0x5aac, 0x5ab1,
	0x5ab2, 0x5ab3, 0x5ab5, 0x5ab8, 0x5aba, 0x5abb, 0x5abf,
	/* 0x3a21 - 0x3a7e */
		0x5ac4, 0x5ac6, 0x5ac8, 0x5acf, 0x5ada, 0x5adc, 0x5ae0,
	0x5ae5, 0x5aea, 0x5aee, 0x5af5, 0x5af6, 0x5afd, 0x5b00, 0x5b01,
	0x5b08, 0x5b17, 0x5b34, 0x5b19, 0x5b1b, 0x5b1d, 0x5b21, 0x5b25,
	0x5b2d, 0x5b38, 0x5b41, 0x5b4b, 0x5b4c, 0x5b52, 0x5b56, 0x5b5e,
	0x5b68, 0x5b6e, 0x5b6f, 0x5b7c, 0x5b7d, 0x5b7e, 0x5b7f, 0x5b81,
	0x5b84, 0x5b86, 0x5b8a, 0x5b8e, 0x5b90, 0x5b91, 0x5b93, 0x5b94,
	0x5b96, 0x5ba8, 0x5ba9, 0x5bac, 0x5bad, 0x5baf, 0x5bb1, 0x5bb2,
	0x5bb7, 0x5bba, 0x5bbc, 0x5bc0, 0x5bc1, 0x5bcd, 0x5bcf, 0x5bd6,
	0x5bd7, 0x5bd8, 0x5bd9, 0x5bda, 0x5be0, 0x5bef, 0x5bf1, 0x5bf4,
	0x5bfd, 0x5c0c, 0x5c17, 0x5c1e, 0x5c1f, 0x5c23, 0x5c26, 0x5c29,
	0x5c2b, 0x5c2c, 0x5c2e, 0x5c30, 0x5c32, 0x5c35, 0x5c36, 0x5c59,
	0x5c5a, 0x5c5c, 0x5c62, 0x5c63, 0x5c67, 0x5c68, 0x5c69,
	/* 0x3b21 - 0x3b7e */
		0x5c6d, 0x5c70, 0x5c74, 0x5c75, 0x5c7a, 0x5c7b, 0x5c7c,
	0x5c7d, 0x5c87, 0x5c88, 0x5c8a, 0x5c8f, 0x5c92, 0x5c9d, 0x5c9f,
	0x5ca0, 0x5ca2, 0x5ca3, 0x5ca6, 0x5caa, 0x5cb2, 0x5cb4, 0x5cb5,
	0x5cba, 0x5cc9, 0x5ccb, 0x5cd2, 0x5cdd, 0x5cd7, 0x5cee, 0x5cf1,
	0x5cf2, 0x5cf4, 0x5d01, 0x5d06, 0x5d0d, 0x5d12, 0x5d2b, 0x5d23,
	0x5d24, 0x5d26, 0x5d27, 0x5d31, 0x5d34, 0x5d39, 0x5d3d, 0x5d3f,
	0x5d42, 0x5d43, 0x5d46, 0x5d48, 0x5d55, 0x5d51, 0x5d59, 0x5d4a,
	0x5d5f, 0x5d60, 0x5d61, 0x5d62, 0x5d64, 0x5d6a, 0x5d6d, 0x5d70,
	0x5d79, 0x5d7a, 0x5d7e, 0x5d7f, 0x5d81, 0x5d83, 0x5d88, 0x5d8a,
	0x5d92, 0x5d93, 0x5d94, 0x5d95, 0x5d99, 0x5d9b, 0x5d9f, 0x5da0,
	0x5da7, 0x5dab, 0x5db0, 0x5db4, 0x5db8, 0x5db9, 0x5dc3, 0x5dc7,
	0x5dcb, 0x5dd0, 0x5dce, 0x5dd8, 0x5dd9, 0x5de0, 0x5de4,
	/* 0x3c21 - 0x3c7e */
		0x5de9, 0x5df8, 0x5df9, 0x5e00, 0x5e07, 0x5e0d, 0x5e12,
	0x5e14, 0x5e15, 0x5e18, 0x5e1f, 0x5e20, 0x5e2e, 0x5e28, 0x5e32,
	0x5e35, 0x5e3e, 0x5e4b, 0x5e50, 0x5e49, 0x5e51, 0x5e56, 0x5e58,
	0x5e5b, 0x5e5c, 0x5e5e, 0x5e68, 0x5e6a, 0x5e6b, 0x5e6c, 0x5e6d,
	0x5e6e, 0x5e70, 0x5e80, 0x5e8b, 0x5e8e, 0x5ea2, 0x5ea4, 0x5ea5,
	0x5ea8, 0x5eaa, 0x5eac, 0x5eb1, 0x5eb3, 0x5ebd, 0x5ebe, 0x5ebf,
	0x5ec6, 0x5ecc, 0x5ecb, 0x5ece, 0x5ed1, 0x5ed2, 0x5ed4, 0x5ed5,
	0x5edc, 0x5ede, 0x5ee5, 0x5eeb, 0x5f02, 0x5f06, 0x5f07, 0x5f08,
	0x5f0e, 0x5f19, 0x5f1c, 0x5f1d, 0x5f21, 0x5f22, 0x5f23, 0x5f24,
	0x5f28, 0x5f2b, 0x5f2c, 0x5f2e, 0x5f30, 0x5f34, 0x5f36, 0x5f3b,
	0x5f3d, 0x5f3f, 0x5f40, 0x5f44, 0x5f45, 0x5f47, 0x5f4d, 0x5f50,
	0x5f54, 0x5f58, 0x5f5b, 0x5f60, 0x5f63, 0x5f64, 0x5f67,
	/* 0x3d21 - 0x3d7e */
		0x5f6f, 0x5f72, 0x5f74, 0x5f75, 0x5f78, 0x5f7a, 0x5f7d,
	0x5f7e, 0x5f89, 0x5f8d, 0x5f8f, 0x5f96, 0x5f9c, 0x5f9d, 0x5fa2,
	0x5fa7, 0x5fab, 0x5fa4, 0x5fac, 0x5faf, 0x5fb0, 0x5fb1, 0x5fb8,
	0x5fc4, 0x5fc7, 0x5fc8, 0x5fc9, 0x5fcb, 0x5fd0, 0x5fd1, 0x5fd2,
	0x5fd3, 0x5fd4, 0x5fde, 0x5fe1, 0x5fe2, 0x5fe8, 0x5fe9, 0x5fea,
	0x5fec, 0x5fed, 0x5fee, 0x5fef, 0x5ff2, 0x5ff3, 0x5ff6, 0x5ffa,
	0x5ffc, 0x6007, 0x600a, 0x600d, 0x6013, 0x6014, 0x6017, 0x6018,
	0x601a, 0x601f, 0x6024, 0x602d, 0x6033, 0x6035, 0x6040, 0x6047,
	0x6048, 0x6049, 0x604c, 0x6051, 0x6054, 0x6056, 0x6057, 0x605d,
	0x6061, 0x6067, 0x6071, 0x607e, 0x607f, 0x6082, 0x6086, 0x6088,
	0x608a, 0x608e, 0x6091, 0x6093, 0x6095, 0x6098, 0x609d, 0x609e,
	0x60a2, 0x60a4, 0x60a5, 0x60a8, 0x60b0, 0x60b1, 0x60b7,
	/* 0x3e21 - 0x3e7e */
		0x60bb, 0x60be, 0x60c2, 0x60c4, 0x60c8, 0x60c9, 0x60ca,
	0x60cb, 0x60ce, 0x60cf, 0x60d4, 0x60d5, 0x60d9, 0x60db, 0x60dd,
	0x60de, 0x60e2, 0x60e5, 0x60f2, 0x60f5, 0x60f8, 0x60fc, 0x60fd,
	0x6102, 0x6107, 0x610a, 0x610c, 0x6110, 0x6111, 0x6112, 0x6113,
	0x6114, 0x6116, 0x6117, 0x6119, 0x611c, 0x611e, 0x6122, 0x612a,
	0x612b, 0x6130, 0x6131, 0x6135, 0x6136, 0x6137, 0x6139, 0x6141,
	0x6145, 0x6146, 0x6149, 0x615e, 0x6160, 0x616c, 0x6172, 0x6178,
	0x617b, 0x617c, 0x617f, 0x6180, 0x6181, 0x6183, 0x6184, 0x618b,
	0x618d, 0x6192, 0x6193, 0x6197, 0x6198, 0x619c, 0x619d, 0x619f,
	0x61a0, 0x61a5, 0x61a8, 0x61aa, 0x61ad, 0x61b8, 0x61b9, 0x61bc,
	0x61c0, 0x61c1, 0x61c2, 0x61ce, 0x61cf, 0x61d5, 0x61dc, 0x61dd,
	0x61de, 0x61df, 0x61e1, 0x61e2, 0x61e7, 0x61e9, 0x61e5,
	/* 0x3f21 - 0x3f7e */
		0x61ec, 0x61ed, 0x61ef, 0x6201, 0x6203, 0x6204, 0x6207,
	0x6213, 0x6215, 0x621c, 0x6220, 0x6222, 0x6223, 0x6227, 0x6229,
	0x622b, 0x6239, 0x623d, 0x6242, 0x6243, 0x6244, 0x6246, 0x624c,
	0x6250, 0x6251, 0x6252, 0x6254, 0x6256, 0x625a, 0x625c, 0x6264,
	0x626d, 0x626f, 0x6273, 0x627a, 0x627d, 0x628d, 0x628e, 0x628f,
	0x6290, 0x62a6, 0x62a8, 0x62b3, 0x62b6, 0x62b7, 0x62ba, 0x62be,
	0x62bf, 0x62c4, 0x62ce, 0x62d5, 0x62d6, 0x62da, 0x62ea, 0x62f2,
	0x62f4, 0x62fc, 0x62fd, 0x6303, 0x6304, 0x630a, 0x630b, 0x630d,
	0x6310, 0x6313, 0x6316, 0x6318, 0x6329, 0x632a, 0x632d, 0x6335,
	0x6336, 0x6339, 0x633c, 0x6341, 0x6342, 0x6343, 0x6344, 0x6346,
	0x634a, 0x634b, 0x634e, 0x6352, 0x6353, 0x6354, 0x6358, 0x635b,
	0x6365, 0x6366, 0x636c, 0x636d, 0x6371, 0x6374, 0x6375,
	/* 0x4021 - 0x407e */
		0x6378, 0x637c, 0x637d, 0x637f, 0x6382, 0x6384, 0x6387,
	0x638a, 0x6390, 0x6394, 0x6395, 0x6399, 0x639a, 0x639e, 0x63a4,
	0x63a6, 0x63ad, 0x63ae, 0x63af, 0x63bd, 0x63c1, 0x63c5, 0x63c8,
	0x63ce, 0x63d1, 0x63d3, 0x63d4, 0x63d5, 0x63dc, 0x63e0, 0x63e5,
	0x63ea, 0x63ec, 0x63f2, 0x63f3, 0x63f5, 0x63f8, 0x63f9, 0x6409,
	0x640a, 0x6410, 0x6412, 0x6414, 0x6418, 0x641e, 0x6420, 0x6422,
	0x6424, 0x6425, 0x6429, 0x642a, 0x642f, 0x6430, 0x6435, 0x643d,
	0x643f, 0x644b, 0x644f, 0x6451, 0x6452, 0x6453, 0x6454, 0x645a,
	0x645b, 0x645c, 0x645d, 0x645f, 0x6460, 0x6461, 0x6463, 0x646d,
	0x6473, 0x6474, 0x647b, 0x647d, 0x6485, 0x6487, 0x648f, 0x6490,
	0x6491, 0x6498, 0x6499, 0x649b, 0x649d, 0x649f, 0x64a1, 0x64a3,
	0x64a6, 0x64a8, 0x64ac, 0x64b3, 0x64bd, 0x64be, 0x64bf,
	/* 0x4121 - 0x417e */
		0x64c4, 0x64c9, 0x64ca, 0x64cb, 0x64cc, 0x64ce, 0x64d0,
	0x64d1, 0x64d5, 0x64d7, 0x64e4, 0x64e5, 0x64e9, 0x64ea, 0x64ed,
	0x64f0, 0x64f5, 0x64f7, 0x64fb, 0x64ff, 0x6501, 0x6504, 0x6508,
	0x6509, 0x650a, 0x650f, 0x6513, 0x6514, 0x6516, 0x6519, 0x651b,
	0x651e, 0x651f, 0x6522, 0x6526, 0x6529, 0x652e, 0x6531, 0x653a,
	0x653c, 0x653d, 0x6543, 0x6547, 0x6549, 0x6550, 0x6552, 0x6554,
	0x655f, 0x6560, 0x6567, 0x656b, 0x657a, 0x657d, 0x6581, 0x6585,
	0x658a, 0x6592, 0x6595, 0x6598, 0x659d, 0x65a0, 0x65a3, 0x65a6,
	0x65ae, 0x65b2, 0x65b3, 0x65b4, 0x65bf, 0x65c2, 0x65c8, 0x65c9,
	0x65ce, 0x65d0, 0x65d4, 0x65d6, 0x65d8, 0x65df, 0x65f0, 0x65f2,
	0x65f4, 0x65f5, 0x65f9, 0x65fe, 0x65ff, 0x6600, 0x6604, 0x6608,
	0x6609, 0x660d, 0x6611, 0x6612, 0x6615, 0x6616, 0x661d,
	/* 0x4221 - 0x427e */
		0x661e, 0x6621, 0x6622, 0x6623, 0x6624, 0x6626, 0x6629,
	0x662a, 0x662b, 0x662c, 0x662e, 0x6630, 0x6631, 0x6633, 0x6639,
	0x6637, 0x6640, 0x6645, 0x6646, 0x664a, 0x664c, 0x6651, 0x664e,
	0x6657, 0x6658, 0x6659, 0x665b, 0x665c, 0x6660, 0x6661, 0x66fb,
	0x666a, 0x666b, 0x666c, 0x667e, 0x6673, 0x6675, 0x667f, 0x6677,
	0x6678, 0x6679, 0x667b, 0x6680, 0x667c, 0x668b, 0x668c, 0x668d,
	0x6690, 0x6692, 0x6699, 0x669a, 0x669b, 0x669c, 0x669f, 0x66a0,
	0x66a4, 0x66ad, 0x66b1, 0x66b2, 0x66b5, 0x66bb, 0x66bf, 0x66c0,
	0x66c2, 0x66c3, 0x66c8, 0x66cc, 0x66ce, 0x66cf, 0x66d4, 0x66db,
	0x66df, 0x66e8, 0x66eb, 0x66ec, 0x66ee, 0x66fa, 0x6705, 0x6707,
	0x670e, 0x6713, 0x6719, 0x671c, 0x6720, 0x6722, 0x6733, 0x673e,
	0x6745, 0x6747, 0x6748, 0x674c, 0x6754, 0x6755, 0x675d,
	/* 0x4321 - 0x437e */
		0x6766, 0x676c, 0x676e, 0x6774, 0x6776, 0x677b, 0x6781,
	0x6784, 0x678e, 0x678f, 0x6791, 0x6793, 0x6796, 0x6798, 0x6799,
	0x679b, 0x67b0, 0x67b1, 0x67b2, 0x67b5, 0x67bb, 0x67bc, 0x67bd,
	0x67f9, 0x67c0, 0x67c2, 0x67c3, 0x67c5, 0x67c8, 0x67c9, 0x67d2,
	0x67d7, 0x67d9, 0x67dc, 0x67e1, 0x67e6, 0x67f0, 0x67f2, 0x67f6,
	0x67f7, 0x6852, 0x6814, 0x6819, 0x681d, 0x681f, 0x6828, 0x6827,
	0x682c, 0x682d, 0x682f, 0x6830, 0x6831, 0x6833, 0x683b, 0x683f,
	0x6844, 0x6845, 0x684a, 0x684c, 0x6855, 0x6857, 0x6858, 0x685b,
	0x686b, 0x686e, 0x686f, 0x6870, 0x6871, 0x6872, 0x6875, 0x6879,
	0x687a, 0x687b, 0x687c, 0x6882, 0x6884, 0x6886, 0x6888, 0x6896,
	0x6898, 0x689a, 0x689c, 0x68a1, 0x68a3, 0x68a5, 0x68a9, 0x68aa,
	0x68ae, 0x68b2, 0x68bb, 0x68c5, 0x68c8, 0x68cc, 0x68cf,
	/* 0x4421 - 0x447e */
		0x68d0, 0x68d1, 0x68d3, 0x68d6, 0x68d9, 0x68dc, 0x68dd,
	0x68e5, 0x68e8, 0x68ea, 0x68eb, 0x68ec, 0x68ed, 0x68f0, 0x68f1,
	0x68f5, 0x68f6, 0x68fb, 0x68fc, 0x68fd, 0x6906, 0x6909, 0x690a,
	0x6910, 0x6911, 0x6913, 0x6916, 0x6917, 0x6931, 0x6933, 0x6935,
	0x6938, 0x693b, 0x6942, 0x6945, 0x6949, 0x694e, 0x6957, 0x695b,
	0x6963, 0x6964, 0x6965, 0x6966, 0x6968, 0x6969, 0x696c, 0x6970,
	0x6971, 0x6972, 0x697a, 0x697b, 0x697f, 0x6980, 0x698d, 0x6992,
	0x6996, 0x6998, 0x69a1, 0x69a5, 0x69a6, 0x69a8, 0x69ab, 0x69ad,
	0x69af, 0x69b7, 0x69b8, 0x69ba, 0x69bc, 0x69c5, 0x69c8, 0x69d1,
	0x69d6, 0x69d7, 0x69e2, 0x69e5, 0x69ee, 0x69ef, 0x69f1, 0x69f3,
	0x69f5, 0x69fe, 0x6a00, 0x6a01, 0x6a03, 0x6a0f, 0x6a11, 0x6a15,
	0x6a1a, 0x6a1d, 0x6a20, 0x6a24, 0x6a28, 0x6a30, 0x6a32,
	/* 0x4521 - 0x457e */
		0x6a34, 0x6a37, 0x6a3b, 0x6a3e, 0x6a3f, 0x6a45, 0x6a46,
	0x6a49, 0x6a4a, 0x6a4e, 0x6a50, 0x6a51, 0x6a52, 0x6a55, 0x6a56,
	0x6a5b, 0x6a64, 0x6a67, 0x6a6a, 0x6a71, 0x6a73, 0x6a7e, 0x6a81,
	0x6a83, 0x6a86, 0x6a87, 0x6a89, 0x6a8b, 0x6a91, 0x6a9b, 0x6a9d,
	0x6a9e, 0x6a9f, 0x6aa5, 0x6aab, 0x6aaf, 0x6ab0, 0x6ab1, 0x6ab4,
	0x6abd, 0x6abe, 0x6abf, 0x6ac6, 0x6ac9, 0x6ac8, 0x6acc, 0x6ad0,
	0x6ad4, 0x6ad5, 0x6ad6, 0x6adc, 0x6add, 0x6ae4, 0x6ae7, 0x6aec,
	0x6af0, 0x6af1, 0x6af2, 0x6afc, 0x6afd, 0x6b02, 0x6b03, 0x6b06,
	0x6b07, 0x6b09, 0x6b0f, 0x6b10, 0x6b11, 0x6b17, 0x6b1b, 0x6b1e,
	0x6b24, 0x6b28, 0x6b2b, 0x6b2c, 0x6b2f, 0x6b35, 0x6b36, 0x6b3b,
	0x6b3f, 0x6b46, 0x6b4a, 0x6b4d, 0x6b52, 0x6b56, 0x6b58, 0x6b5d,
	0x6b60, 0x6b67, 0x6b6b, 0x6b6e, 0x6b70, 0x6b75, 0x6b7d,
	/* 0x4621 - 0x467e */
		0x6b7e, 0x6b82, 0x6b85, 0x6b97, 0x6b9b, 0x6b9f, 0x6ba0,
	0x6ba2, 0x6ba3, 0x6ba8, 0x6ba9, 0x6bac, 0x6bad, 0x6bae, 0x6bb0,
	0x6bb8, 0x6bb9, 0x6bbd, 0x6bbe, 0x6bc3, 0x6bc4, 0x6bc9, 0x6bcc,
	0x6bd6, 0x6bda, 0x6be1, 0x6be3, 0x6be6, 0x6be7, 0x6bee, 0x6bf1,
	0x6bf7, 0x6bf9, 0x6bff, 0x6c02, 0x6c04, 0x6c05, 0x6c09, 0x6c0d,
	0x6c0e, 0x6c10, 0x6c12, 0x6c19, 0x6c1f, 0x6c26, 0x6c27, 0x6c28,
	0x6c2c, 0x6c2e, 0x6c33, 0x6c35, 0x6c36, 0x6c3a, 0x6c3b, 0x6c3f,
	0x6c4a, 0x6c4b, 0x6c4d, 0x6c4f, 0x6c52, 0x6c54, 0x6c59, 0x6c5b,
	0x6c5c, 0x6c6b, 0x6c6d, 0x6c6f, 0x6c74, 0x6c76, 0x6c78, 0x6c79,
	0x6c7b, 0x6c85, 0x6c86, 0x6c87, 0x6c89, 0x6c94, 0x6c95, 0x6c97,
	0x6c98, 0x6c9c, 0x6c9f, 0x6cb0, 0x6cb2, 0x6cb4, 0x6cc2, 0x6cc6,
	0x6ccd, 0x6ccf, 0x6cd0, 0x6cd1, 0x6cd2, 0x6cd4, 0x6cd6,
	/* 0x4721 - 0x477e */
		0x6cda, 0x6cdc, 0x6ce0, 0x6ce7, 0x6ce9, 0x6ceb, 0x6cec,
	0x6cee, 0x6cf2, 0x6cf4, 0x6d04, 0x6d07, 0x6d0a, 0x6d0e, 0x6d0f,
	0x6d11, 0x6d13, 0x6d1a, 0x6d26, 0x6d27, 0x6d28, 0x6c67, 0x6d2e,
	0x6d2f, 0x6d31, 0x6d39, 0x6d3c, 0x6d3f, 0x6d57, 0x6d5e, 0x6d5f,
	0x6d61, 0x6d65, 0x6d67, 0x6d6f, 0x6d70, 0x6d7c, 0x6d82, 0x6d87,
	0x6d91, 0x6d92, 0x6d94, 0x6d96, 0x6d97, 0x6d98, 0x6daa, 0x6dac,
	0x6db4, 0x6db7, 0x6db9, 0x6dbd, 0x6dbf, 0x6dc4, 0x6dc8, 0x6dca,
	0x6dce, 0x6dcf, 0x6dd6, 0x6ddb, 0x6ddd, 0x6ddf, 0x6de0, 0x6de2,
	0x6de5, 0x6de9, 0x6def, 0x6df0, 0x6df4, 0x6df6, 0x6dfc, 0x6e00,
	0x6e04, 0x6e1e, 0x6e22, 0x6e27, 0x6e32, 0x6e36, 0x6e39, 0x6e3b,
	0x6e3c, 0x6e44, 0x6e45, 0x6e48, 0x6e49, 0x6e4b, 0x6e4f, 0x6e51,
	0x6e52, 0x6e53, 0x6e54, 0x6e57, 0x6e5c, 0x6e5d, 0x6e5e,
	/* 0x4821 - 0x487e */
		0x6e62, 0x6e63, 0x6e68, 0x6e73, 0x6e7b, 0x6e7d, 0x6e8d,
	0x6e93, 0x6e99, 0x6ea0, 0x6ea7, 0x6ead, 0x6eae, 0x6eb1, 0x6eb3,
	0x6ebb, 0x6ebf, 0x6ec0, 0x6ec1, 0x6ec3, 0x6ec7, 0x6ec8, 0x6eca,
	0x6ecd, 0x6ece, 0x6ecf, 0x6eeb, 0x6eed, 0x6eee, 0x6ef9, 0x6efb,
	0x6efd, 0x6f04, 0x6f08, 0x6f0a, 0x6f0c, 0x6f0d, 0x6f16, 0x6f18,
	0x6f1a, 0x6f1b, 0x6f26, 0x6f29, 0x6f2a, 0x6f2f, 0x6f30, 0x6f33,
	0x6f36, 0x6f3b, 0x6f3c, 0x6f2d, 0x6f4f, 0x6f51, 0x6f52, 0x6f53,
	0x6f57, 0x6f59, 0x6f5a, 0x6f5d, 0x6f5e, 0x6f61, 0x6f62, 0x6f68,
	0x6f6c, 0x6f7d, 0x6f7e, 0x6f83, 0x6f87, 0x6f88, 0x6f8b, 0x6f8c,
	0x6f8d, 0x6f90, 0x6f92, 0x6f93, 0x6f94, 0x6f96, 0x6f9a, 0x6f9f,
	0x6fa0, 0x6fa5, 0x6fa6, 0x6fa7, 0x6fa8, 0x6fae, 0x6faf, 0x6fb0,
	0x6fb5, 0x6fb6, 0x6fbc, 0x6fc5, 0x6fc7, 0x6fc8, 0x6fca,
	/* 0x4921 - 0x497e */
		0x6fda, 0x6fde, 0x6fe8, 0x6fe9, 0x6ff0, 0x6ff5, 0x6ff9,
	0x6ffc, 0x6ffd, 0x7000, 0x7005, 0x7006, 0x7007, 0x700d, 0x7017,
	0x7020, 0x7023, 0x702f, 0x7034, 0x7037, 0x7039, 0x703c, 0x7043,
	0x7044, 0x7048, 0x7049, 0x704a, 0x704b, 0x7054, 0x7055, 0x705d,
	0x705e, 0x704e, 0x7064, 0x7065, 0x706c, 0x706e, 0x7075, 0x7076,
	0x707e, 0x7081, 0x7085, 0x7086, 0x7094, 0x7095, 0x7096, 0x7097,
	0x7098, 0x709b, 0x70a4, 0x70ab, 0x70b0, 0x70b1, 0x70b4, 0x70b7,
	0x70ca, 0x70d1, 0x70d3, 0x70d4, 0x70d5, 0x70d6, 0x70d8, 0x70dc,
	0x70e4, 0x70fa, 0x7103, 0x7104, 0x7105, 0x7106, 0x7107, 0x710b,
	0x710c, 0x710f, 0x711e, 0x7120, 0x712b, 0x712d, 0x712f, 0x7130,
	0x7131, 0x7138, 0x7141, 0x7145, 0x7146, 0x7147, 0x714a, 0x714b,
	0x7150, 0x7152, 0x7157, 0x715a, 0x715c, 0x715e, 0x7160,
	/* 0x4a21 - 0x4a7e */
		0x7168, 0x7179, 0x7180, 0x7185, 0x7187, 0x718c, 0x7192,
	0x719a, 0x719b, 0x71a0, 0x71a2, 0x71af, 0x71b0, 0x71b2, 0x71b3,
	0x71ba, 0x71bf, 0x71c0, 0x71c1, 0x71c4, 0x71cb, 0x71cc, 0x71d3,
	0x71d6, 0x71d9, 0x71da, 0x71dc, 0x71f8, 0x71fe, 0x7200, 0x7207,
	0x7208, 0x7209, 0x7213, 0x7217, 0x721a, 0x721d, 0x721f, 0x7224,
	0x722b, 0x722f, 0x7234, 0x7238, 0x7239, 0x7241, 0x7242, 0x7243,
	0x7245, 0x724e, 0x724f, 0x7250, 0x7253, 0x7255, 0x7256, 0x725a,
	0x725c, 0x725e, 0x7260, 0x7263, 0x7268, 0x726b, 0x726e, 0x726f,
	0x7271, 0x7277, 0x7278, 0x727b, 0x727c, 0x727f, 0x7284, 0x7289,
	0x728d, 0x728e, 0x7293, 0x729b, 0x72a8, 0x72ad, 0x72ae, 0x72b1,
	0x72b4, 0x72be, 0x72c1, 0x72c7, 0x72c9, 0x72cc, 0x72d5, 0x72d6,
	0x72d8, 0x72df, 0x72e5, 0x72f3, 0x72f4, 0x72fa, 0x72fb,
	/* 0x4b21 - 0x4b7e */
		0x72fe, 0x7302, 0x7304, 0x7305, 0x7307, 0x730b, 0x730d,
	0x7312, 0x7313, 0x7318, 0x7319, 0x731e, 0x7322, 0x7324, 0x7327,
	0x7328, 0x732c, 0x7331, 0x7332, 0x7335, 0x733a, 0x733b, 0x733d,
	0x7343, 0x734d, 0x7350, 0x7352, 0x7356, 0x7358, 0x735d, 0x735e,
	0x735f, 0x7360, 0x7366, 0x7367, 0x7369, 0x736b, 0x736c, 0x736e,
	0x736f, 0x7371, 0x7377, 0x7379, 0x737c, 0x7380, 0x7381, 0x7383,
	0x7385, 0x7386, 0x738e, 0x7390, 0x7393, 0x7395, 0x7397, 0x7398,
	0x739c, 0x739e, 0x739f, 0x73a0, 0x73a2, 0x73a5, 0x73a6, 0x73aa,
	0x73ab, 0x73ad, 0x73b5, 0x73b7, 0x73b9, 0x73bc, 0x73bd, 0x73bf,
	0x73c5, 0x73c6, 0x73c9, 0x73cb, 0x73cc, 0x73cf, 0x73d2, 0x73d3,
	0x73d6, 0x73d9, 0x73dd, 0x73e1, 0x73e3, 0x73e6, 0x73e7, 0x73e9,
	0x73f4, 0x73f5, 0x73f7, 0x73f9, 0x73fa, 0x73fb, 0x73fd,
	/* 0x4c21 - 0x4c7e */
		0x73ff, 0x7400, 0x7401, 0x7404, 0x7407, 0x740a, 0x7411,
	0x741a, 0x741b, 0x7424, 0x7426, 0x7428, 0x7429, 0x742a, 0x742b,
	0x742c, 0x742d, 0x742e, 0x742f, 0x7430, 0x7431, 0x7439, 0x7440,
	0x7443, 0x7444, 0x7446, 0x7447, 0x744b, 0x744d, 0x7451, 0x7452,
	0x7457, 0x745d, 0x7462, 0x7466, 0x7467, 0x7468, 0x746b, 0x746d,
	0x746e, 0x7471, 0x7472, 0x7480, 0x7481, 0x7485, 0x7486, 0x7487,
	0x7489, 0x748f, 0x7490, 0x7491, 0x7492, 0x7498, 0x7499, 0x749a,
	0x749c, 0x749f, 0x74a0, 0x74a1, 0x74a3, 0x74a6, 0x74a8, 0x74a9,
	0x74aa, 0x74ab, 0x74ae, 0x74af, 0x74b1, 0x74b2, 0x74b5, 0x74b9,
	0x74bb, 0x74bf, 0x74c8, 0x74c9, 0x74cc, 0x74d0, 0x74d3, 0x74d8,
	0x74da, 0x74db, 0x74de, 0x74df, 0x74e4, 0x74e8, 0x74ea, 0x74eb,
	0x74ef, 0x74f4, 0x74fa, 0x74fb, 0x74fc, 0x74ff, 0x7506,
	/* 0x4d21 - 0x4d7e */
		0x7512, 0x7516, 0x7517, 0x7520, 0x7521, 0x7524, 0x7527,
	0x7529, 0x752a, 0x752f, 0x7536, 0x7539, 0x753d, 0x753e, 0x753f,
	0x7540, 0x7543, 0x7547, 0x7548, 0x754e, 0x7550, 0x7552, 0x7557,
	0x755e, 0x755f, 0x7561, 0x756f, 0x7571, 0x7579, 0x757a, 0x757b,
	0x757c, 0x757d, 0x757e, 0x7581, 0x7585, 0x7590, 0x7592, 0x7593,
	0x7595, 0x7599, 0x759c, 0x75a2, 0x75a4, 0x75b4, 0x75ba, 0x75bf,
	0x75c0, 0x75c1, 0x75c4, 0x75c6, 0x75cc, 0x75ce, 0x75cf, 0x75d7,
	0x75dc, 0x75df, 0x75e0, 0x75e1, 0x75e4, 0x75e7, 0x75ec, 0x75ee,
	0x75ef, 0x75f1, 0x75f9, 0x7600, 0x7602, 0x7603, 0x7604, 0x7607,
	0x7608, 0x760a, 0x760c, 0x760f, 0x7612, 0x7613, 0x7615, 0x7616,
	0x7619, 0x761b, 0x761c, 0x761d, 0x761e, 0x7623, 0x7625, 0x7626,
	0x7629, 0x762d, 0x7632, 0x7633, 0x7635, 0x7638, 0x7639,
	/* 0x4e21 - 0x4e7e */
		0x763a, 0x763c, 0x764a, 0x7640, 0x7641, 0x7643, 0x7644,
	0x7645, 0x7649, 0x764b, 0x7655, 0x7659, 0x765f, 0x7664, 0x7665,
	0x766d, 0x766e, 0x766f, 0x7671, 0x7674, 0x7681, 0x7685, 0x768c,
	0x768d, 0x7695, 0x769b, 0x769c, 0x769d, 0x769f, 0x76a0, 0x76a2,
	0x76a3, 0x76a4, 0x76a5, 0x76a6, 0x76a7, 0x76a8, 0x76aa, 0x76ad,
	0x76bd, 0x76c1, 0x76c5, 0x76c9, 0x76cb, 0x76cc, 0x76ce, 0x76d4,
	0x76d9, 0x76e0, 0x76e6, 0x76e8, 0x76ec, 0x76f0, 0x76f1, 0x76f6,
	0x76f9, 0x76fc, 0x7700, 0x7706, 0x770a, 0x770e, 0x7712, 0x7714,
	0x7715, 0x7717, 0x7719, 0x771a, 0x771c, 0x7722, 0x7728, 0x772d,
	0x772e, 0x772f, 0x7734, 0x7735, 0x7736, 0x7739, 0x773d, 0x773e,
	0x7742, 0x7745, 0x7746, 0x774a, 0x774d, 0x774e, 0x774f, 0x7752,
	0x7756, 0x7757, 0x775c, 0x775e, 0x775f, 0x7760, 0x7762,
	/* 0x4f21 - 0x4f7e */
		0x7764, 0x7767, 0x776a, 0x776c, 0x7770, 0x7772, 0x7773,
	0x7774, 0x777a, 0x777d, 0x7780, 0x7784, 0x778c, 0x778d, 0x7794,
	0x7795, 0x7796, 0x779a, 0x779f, 0x77a2, 0x77a7, 0x77aa, 0x77ae,
	0x77af, 0x77b1, 0x77b5, 0x77be, 0x77c3, 0x77c9, 0x77d1, 0x77d2,
	0x77d5, 0x77d9, 0x77de, 0x77df, 0x77e0, 0x77e4, 0x77e6, 0x77ea,
	0x77ec, 0x77f0, 0x77f1, 0x77f4, 0x77f8, 0x77fb, 0x7805, 0x7806,
	0x7809, 0x780d, 0x780e, 0x7811, 0x781d, 0x7821, 0x7822, 0x7823,
	0x782d, 0x782e, 0x7830, 0x7835, 0x7837, 0x7843, 0x7844, 0x7847,
	0x7848, 0x784c, 0x784e, 0x7852, 0x785c, 0x785e, 0x7860, 0x7861,
	0x7863, 0x7864, 0x7868, 0x786a, 0x786e, 0x787a, 0x787e, 0x788a,
	0x788f, 0x7894, 0x7898, 0x78a1, 0x789d, 0x789e, 0x789f, 0x78a4,
	0x78a8, 0x78ac, 0x78ad, 0x78b0, 0x78b1, 0x78b2, 0x78b3,
	/* 0x5021 - 0x507e */
		0x78bb, 0x78bd, 0x78bf, 0x78c7, 0x78c8, 0x78c9, 0x78cc,
	0x78ce, 0x78d2, 0x78d3, 0x78d5, 0x78d6, 0x78e4, 0x78db, 0x78df,
	0x78e0, 0x78e1, 0x78e6, 0x78ea, 0x78f2, 0x78f3, 0x7900, 0x78f6,
	0x78f7, 0x78fa, 0x78fb, 0x78ff, 0x7906, 0x790c, 0x7910, 0x791a,
	0x791c, 0x791e, 0x791f, 0x7920, 0x7925, 0x7927, 0x7929, 0x792d,
	0x7931, 0x7934, 0x7935, 0x793b, 0x793d, 0x793f, 0x7944, 0x7945,
	0x7946, 0x794a, 0x794b, 0x794f, 0x7951, 0x7954, 0x7958, 0x795b,
	0x795c, 0x7967, 0x7969, 0x796b, 0x7972, 0x7979, 0x797b, 0x797c,
	0x797e, 0x798b, 0x798c, 0x7991, 0x7993, 0x7994, 0x7995, 0x7996,
	0x7998, 0x799b, 0x799c, 0x79a1, 0x79a8, 0x79a9, 0x79ab, 0x79af,
	0x79b1, 0x79b4, 0x79b8, 0x79bb, 0x79c2, 0x79c4, 0x79c7, 0x79c8,
	0x79ca, 0x79cf, 0x79d4, 0x79d6, 0x79da, 0x79dd, 0x79de,
	/* 0x5121 - 0x517e */
		0x79e0, 0x79e2, 0x79e5, 0x79ea, 0x79eb, 0x79ed, 0x79f1,
	0x79f8, 0x79fc, 0x7a02, 0x7a03, 0x7a07, 0x7a09, 0x7a0a, 0x7a0c,
	0x7a11, 0x7a15, 0x7a1b, 0x7a1e, 0x7a21, 0x7a27, 0x7a2b, 0x7a2d,
	0x7a2f, 0x7a30, 0x7a34, 0x7a35, 0x7a38, 0x7a39, 0x7a3a, 0x7a44,
	0x7a45, 0x7a47, 0x7a48, 0x7a4c, 0x7a55, 0x7a56, 0x7a59, 0x7a5c,
	0x7a5d, 0x7a5f, 0x7a60, 0x7a65, 0x7a67, 0x7a6a, 0x7a6d, 0x7a75,
	0x7a78, 0x7a7e, 0x7a80, 0x7a82, 0x7a85, 0x7a86, 0x7a8a, 0x7a8b,
	0x7a90, 0x7a91, 0x7a94, 0x7a9e, 0x7aa0, 0x7aa3, 0x7aac, 0x7ab3,
	0x7ab5, 0x7ab9, 0x7abb, 0x7abc, 0x7ac6, 0x7ac9, 0x7acc, 0x7ace,
	0x7ad1, 0x7adb, 0x7ae8, 0x7ae9, 0x7aeb, 0x7aec, 0x7af1, 0x7af4,
	0x7afb, 0x7afd, 0x7afe, 0x7b07, 0x7b14, 0x7b1f, 0x7b23, 0x7b27,
	0x7b29, 0x7b2a, 0x7b2b, 0x7b2d, 0x7b2e, 0x7b2f, 0x7b30,
	/* 0x5221 - 0x527e */
		0x7b31, 0x7b34, 0x7b3d, 0x7b3f, 0x7b40, 0x7b41, 0x7b47,
	0x7b4e, 0x7b55, 0x7b60, 0x7b64, 0x7b66, 0x7b69, 0x7b6a, 0x7b6d,
	0x7b6f, 0x7b72, 0x7b73, 0x7b77, 0x7b84, 0x7b89, 0x7b8e, 0x7b90,
	0x7b91, 0x7b96, 0x7b9b, 0x7b9e, 0x7ba0, 0x7ba5, 0x7bac, 0x7baf,
	0x7bb0, 0x7bb2, 0x7bb5, 0x7bb6, 0x7bba, 0x7bbb, 0x7bbc, 0x7bbd,
	0x7bc2, 0x7bc5, 0x7bc8, 0x7bca, 0x7bd4, 0x7bd6, 0x7bd7, 0x7bd9,
	0x7bda, 0x7bdb, 0x7be8, 0x7bea, 0x7bf2, 0x7bf4, 0x7bf5, 0x7bf8,
	0x7bf9, 0x7bfa, 0x7bfc, 0x7bfe, 0x7c01, 0x7c02, 0x7c03, 0x7c04,
	0x7c06, 0x7c09, 0x7c0b, 0x7c0c, 0x7c0e, 0x7c0f, 0x7c19, 0x7c1b,
	0x7c20, 0x7c25, 0x7c26, 0x7c28, 0x7c2c, 0x7c31, 0x7c33, 0x7c34,
	0x7c36, 0x7c39, 0x7c3a, 0x7c46, 0x7c4a, 0x7c55, 0x7c51, 0x7c52,
	0x7c53, 0x7c59, 0x7c5a, 0x7c5b, 0x7c5c, 0x7c5d, 0x7c5e,
	/* 0x5321 - 0x537e */
		0x7c61, 0x7c63, 0x7c67, 0x7c69, 0x7c6d, 0x7c6e, 0x7c70,
	0x7c72, 0x7c79, 0x7c7c, 0x7c7d, 0x7c86, 0x7c87, 0x7c8f, 0x7c94,
	0x7c9e, 0x7ca0, 0x7ca6, 0x7cb0, 0x7cb6, 0x7cb7, 0x7cba, 0x7cbb,
	0x7cbc, 0x7cbf, 0x7cc4, 0x7cc7, 0x7cc8, 0x7cc9, 0x7ccd, 0x7ccf,
	0x7cd3, 0x7cd4, 0x7cd5, 0x7cd7, 0x7cd9, 0x7cda, 0x7cdd, 0x7ce6,
	0x7ce9, 0x7ceb, 0x7cf5, 0x7d03, 0x7d07, 0x7d08, 0x7d09, 0x7d0f,
	0x7d11, 0x7d12, 0x7d13, 0x7d16, 0x7d1d, 0x7d1e, 0x7d23, 0x7d26,
	0x7d2a, 0x7d2d, 0x7d31, 0x7d3c, 0x7d3d, 0x7d3e, 0x7d40, 0x7d41,
	0x7d47, 0x7d48, 0x7d4d, 0x7d51, 0x7d53, 0x7d57, 0x7d59, 0x7d5a,
	0x7d5c, 0x7d5d, 0x7d65, 0x7d67, 0x7d6a, 0x7d70, 0x7d78, 0x7d7a,
	0x7d7b, 0x7d7f, 0x7d81, 0x7d82, 0x7d83, 0x7d85, 0x7d86, 0x7d88,
	0x7d8b, 0x7d8c, 0x7d8d, 0x7d91, 0x7d96, 0x7d97, 0x7d9d,
	/* 0x5421 - 0x547e */
		0x7d9e, 0x7da6, 0x7da7, 0x7daa, 0x7db3, 0x7db6, 0x7db7,
	0x7db9, 0x7dc2, 0x7dc3, 0x7dc4, 0x7dc5, 0x7dc6, 0x7dcc, 0x7dcd,
	0x7dce, 0x7dd7, 0x7dd9, 0x7e00, 0x7de2, 0x7de5, 0x7de6, 0x7dea,
	0x7deb, 0x7ded, 0x7df1, 0x7df5, 0x7df6, 0x7df9, 0x7dfa, 0x7e08,
	0x7e10, 0x7e11, 0x7e15, 0x7e17, 0x7e1c, 0x7e1d, 0x7e20, 0x7e27,
	0x7e28, 0x7e2c, 0x7e2d, 0x7e2f, 0x7e33, 0x7e36, 0x7e3f, 0x7e44,
	0x7e45, 0x7e47, 0x7e4e, 0x7e50, 0x7e52, 0x7e58, 0x7e5f, 0x7e61,
	0x7e62, 0x7e65, 0x7e6b, 0x7e6e, 0x7e6f, 0x7e73, 0x7e78, 0x7e7e,
	0x7e81, 0x7e86, 0x7e87, 0x7e8a, 0x7e8d, 0x7e91, 0x7e95, 0x7e98,
	0x7e9a, 0x7e9d, 0x7e9e, 0x7f3c, 0x7f3b, 0x7f3d, 0x7f3e, 0x7f3f,
	0x7f43, 0x7f44, 0x7f47, 0x7f4f, 0x7f52, 0x7f53, 0x7f5b, 0x7f5c,
	0x7f5d, 0x7f61, 0x7f63, 0x7f64, 0x7f65, 0x7f66, 0x7f6d,
	/* 0x5521 - 0x557e */
		0x7f71, 0x7f7d, 0x7f7e, 0x7f7f, 0x7f80, 0x7f8b, 0x7f8d,
	0x7f8f, 0x7f90, 0x7f91, 0x7f96, 0x7f97, 0x7f9c, 0x7fa1, 0x7fa2,
	0x7fa6, 0x7faa, 0x7fad, 0x7fb4, 0x7fbc, 0x7fbf, 0x7fc0, 0x7fc3,
	0x7fc8, 0x7fce, 0x7fcf, 0x7fdb, 0x7fdf, 0x7fe3, 0x7fe5, 0x7fe8,
	0x7fec, 0x7fee, 0x7fef, 0x7ff2, 0x7ffa, 0x7ffd, 0x7ffe, 0x7fff,
	0x8007, 0x8008, 0x800a, 0x800d, 0x800e, 0x800f, 0x8011, 0x8013,
	0x8014, 0x8016, 0x801d, 0x801e, 0x801f, 0x8020, 0x8024, 0x8026,
	0x802c, 0x802e, 0x8030, 0x8034, 0x8035, 0x8037, 0x8039, 0x803a,
	0x803c, 0x803e, 0x8040, 0x8044, 0x8060, 0x8064, 0x8066, 0x806d,
	0x8071, 0x8075, 0x8081, 0x8088, 0x808e, 0x809c, 0x809e, 0x80a6,
	0x80a7, 0x80ab, 0x80b8, 0x80b9, 0x80c8, 0x80cd, 0x80cf, 0x80d2,
	0x80d4, 0x80d5, 0x80d7, 0x80d8, 0x80e0, 0x80ed, 0x80ee,
	/* 0x5621 - 0x567e */
		0x80f0, 0x80f2, 0x80f3, 0x80f6, 0x80f9, 0x80fa, 0x80fe,
	0x8103, 0x810b, 0x8116, 0x8117, 0x8118, 0x811c, 0x811e, 0x8120,
	0x8124, 0x8127, 0x812c, 0x8130, 0x8135, 0x813a, 0x813c, 0x8145,
	0x8147, 0x814a, 0x814c, 0x8152, 0x8157, 0x8160, 0x8161, 0x8167,
	0x8168, 0x8169, 0x816d, 0x816f, 0x8177, 0x8181, 0x8190, 0x8184,
	0x8185, 0x8186, 0x818b, 0x818e, 0x8196, 0x8198, 0x819b, 0x819e,
	0x81a2, 0x81ae, 0x81b2, 0x81b4, 0x81bb, 0x81cb, 0x81c3, 0x81c5,
	0x81ca, 0x81ce, 0x81cf, 0x81d5, 0x81d7, 0x81db, 0x81dd, 0x81de,
	0x81e1, 0x81e4, 0x81eb, 0x81ec, 0x81f0, 0x81f1, 0x81f2, 0x81f5,
	0x81f6, 0x81f8, 0x81f9, 0x81fd, 0x81ff, 0x8200, 0x8203, 0x820f,
	0x8213, 0x8214, 0x8219, 0x821a, 0x821d, 0x8221, 0x8222, 0x8228,
	0x8232, 0x8234, 0x823a, 0x8243, 0x8244, 0x8245, 0x8246,
	/* 0x5721 - 0x577e */
		0x824b, 0x824e, 0x824f, 0x8251, 0x8256, 0x825c, 0x8260,
	0x8263, 0x8267, 0x826d, 0x8274, 0x827b, 0x827d, 0x827f, 0x8280,
	0x8281, 0x8283, 0x8284, 0x8287, 0x8289, 0x828a, 0x828e, 0x8291,
	0x8294, 0x8296, 0x8298, 0x829a, 0x829b, 0x82a0, 0x82a1, 0x82a3,
	0x82a4, 0x82a7, 0x82a8, 0x82a9, 0x82aa, 0x82ae, 0x82b0, 0x82b2,
	0x82b4, 0x82b7, 0x82ba, 0x82bc, 0x82be, 0x82bf, 0x82c6, 0x82d0,
	0x82d5, 0x82da, 0x82e0, 0x82e2, 0x82e4, 0x82e8, 0x82ea, 0x82ed,
	0x82ef, 0x82f6, 0x82f7, 0x82fd, 0x82fe, 0x8300, 0x8301, 0x8307,
	0x8308, 0x830a, 0x830b, 0x8354, 0x831b, 0x831d, 0x831e, 0x831f,
	0x8321, 0x8322, 0x832c, 0x832d, 0x832e, 0x8330, 0x8333, 0x8337,
	0x833a, 0x833c, 0x833d, 0x8342, 0x8343, 0x8344, 0x8347, 0x834d,
	0x834e, 0x8351, 0x8355, 0x8356, 0x8357, 0x8370, 0x8378,
	/* 0x5821 - 0x587e */
		0x837d, 0x837f, 0x8380, 0x8382, 0x8384, 0x8386, 0x838d,
	0x8392, 0x8394, 0x8395, 0x8398, 0x8399, 0x839b, 0x839c, 0x839d,
	0x83a6, 0x83a7, 0x83a9, 0x83ac, 0x83be, 0x83bf, 0x83c0, 0x83c7,
	0x83c9, 0x83cf, 0x83d0, 0x83d1, 0x83d4, 0x83dd, 0x8353, 0x83e8,
	0x83ea, 0x83f6, 0x83f8, 0x83f9, 0x83fc, 0x8401, 0x8406, 0x840a,
	0x840f, 0x8411, 0x8415, 0x8419, 0x83ad, 0x842f, 0x8439, 0x8445,
	0x8447, 0x8448, 0x844a, 0x844d, 0x844f, 0x8451, 0x8452, 0x8456,
	0x8458, 0x8459, 0x845a, 0x845c, 0x8460, 0x8464, 0x8465, 0x8467,
	0x846a, 0x8470, 0x8473, 0x8474, 0x8476, 0x8478, 0x847c, 0x847d,
	0x8481, 0x8485, 0x8492, 0x8493, 0x8495, 0x849e, 0x84a6, 0x84a8,
	0x84a9, 0x84aa, 0x84af, 0x84b1, 0x84b4, 0x84ba, 0x84bd, 0x84be,
	0x84c0, 0x84c2, 0x84c7, 0x84c8, 0x84cc, 0x84cf, 0x84d3,
	/* 0x5921 - 0x597e */
		0x84dc, 0x84e7, 0x84ea, 0x84ef, 0x84f0, 0x84f1, 0x84f2,
	0x84f7, 0x8532, 0x84fa, 0x84fb, 0x84fd, 0x8502, 0x8503, 0x8507,
	0x850c, 0x850e, 0x8510, 0x851c, 0x851e, 0x8522, 0x8523, 0x8524,
	0x8525, 0x8527, 0x852a, 0x852b, 0x852f, 0x8533, 0x8534, 0x8536,
	0x853f, 0x8546, 0x854f, 0x8550, 0x8551, 0x8552, 0x8553, 0x8556,
	0x8559, 0x855c, 0x855d, 0x855e, 0x855f, 0x8560, 0x8561, 0x8562,
	0x8564, 0x856b, 0x856f, 0x8579, 0x857a, 0x857b, 0x857d, 0x857f,
	0x8581, 0x8585, 0x8586, 0x8589, 0x858b, 0x858c, 0x858f, 0x8593,
	0x8598, 0x859d, 0x859f, 0x85a0, 0x85a2, 0x85a5, 0x85a7, 0x85b4,
	0x85b6, 0x85b7, 0x85b8, 0x85bc, 0x85bd, 0x85be, 0x85bf, 0x85c2,
	0x85c7, 0x85ca, 0x85cb, 0x85ce, 0x85ad, 0x85d8, 0x85da, 0x85df,
	0x85e0, 0x85e6, 0x85e8, 0x85ed, 0x85f3, 0x85f6, 0x85fc,
	/* 0x5a21 - 0x5a7e */
		0x85ff, 0x8600, 0x8604, 0x8605, 0x860d, 0x860e, 0x8610,
	0x8611, 0x8612, 0x8618, 0x8619, 0x861b, 0x861e, 0x8621, 0x8627,
	0x8629, 0x8636, 0x8638, 0x863a, 0x863c, 0x863d, 0x8640, 0x8642,
	0x8646, 0x8652, 0x8653, 0x8656, 0x8657, 0x8658, 0x8659, 0x865d,
	0x8660, 0x8661, 0x8662, 0x8663, 0x8664, 0x8669, 0x866c, 0x866f,
	0x8675, 0x8676, 0x8677, 0x867a, 0x868d, 0x8691, 0x8696, 0x8698,
	0x869a, 0x869c, 0x86a1, 0x86a6, 0x86a7, 0x86a8, 0x86ad, 0x86b1,
	0x86b3, 0x86b4, 0x86b5, 0x86b7, 0x86b8, 0x86b9, 0x86bf, 0x86c0,
	0x86c1, 0x86c3, 0x86c5, 0x86d1, 0x86d2, 0x86d5, 0x86d7, 0x86da,
	0x86dc, 0x86e0, 0x86e3, 0x86e5, 0x86e7, 0x8688, 0x86fa, 0x86fc,
	0x86fd, 0x8704, 0x8705, 0x8707, 0x870b, 0x870e, 0x870f, 0x8710,
	0x8713, 0x8714, 0x8719, 0x871e, 0x871f, 0x8721, 0x8723,
	/* 0x5b21 - 0x5b7e */
		0x8728, 0x872e, 0x872f, 0x8731, 0x8732, 0x8739, 0x873a,
	0x873c, 0x873d, 0x873e, 0x8740, 0x8743, 0x8745, 0x874d, 0x8758,
	0x875d, 0x8761, 0x8764, 0x8765, 0x876f, 0x8771, 0x8772, 0x877b,
	0x8783, 0x8784, 0x8785, 0x8786, 0x8787, 0x8788, 0x8789, 0x878b,
	0x878c, 0x8790, 0x8793, 0x8795, 0x8797, 0x8798, 0x8799, 0x879e,
	0x87a0, 0x87a3, 0x87a7, 0x87ac, 0x87ad, 0x87ae, 0x87b1, 0x87b5,
	0x87be, 0x87bf, 0x87c1, 0x87c8, 0x87c9, 0x87ca, 0x87ce, 0x87d5,
	0x87d6, 0x87d9, 0x87da, 0x87dc, 0x87df, 0x87e2, 0x87e3, 0x87e4,
	0x87ea, 0x87eb, 0x87ed, 0x87f1, 0x87f3, 0x87f8, 0x87fa, 0x87ff,
	0x8801, 0x8803, 0x8806, 0x8809, 0x880a, 0x880b, 0x8810, 0x8819,
	0x8812, 0x8813, 0x8814, 0x8818, 0x881a, 0x881b, 0x881c, 0x881e,
	0x881f, 0x8828, 0x882d, 0x882e, 0x8830, 0x8832, 0x8835,
	/* 0x5c21 - 0x5c7e */
		0x883a, 0x883c, 0x8841, 0x8843, 0x8845, 0x8848, 0x8849,
	0x884a, 0x884b, 0x884e, 0x8851, 0x8855, 0x8856, 0x8858, 0x885a,
	0x885c, 0x885f, 0x8860, 0x8864, 0x8869, 0x8871, 0x8879, 0x887b,
	0x8880, 0x8898, 0x889a, 0x889b, 0x889c, 0x889f, 0x88a0, 0x88a8,
	0x88aa, 0x88ba, 0x88bd, 0x88be, 0x88c0, 0x88ca, 0x88cb, 0x88cc,
	0x88cd, 0x88ce, 0x88d1, 0x88d2, 0x88d3, 0x88db, 0x88de, 0x88e7,
	0x88ef, 0x88f0, 0x88f1, 0x88f5, 0x88f7, 0x8901, 0x8906, 0x890d,
	0x890e, 0x890f, 0x8915, 0x8916, 0x8918, 0x8919, 0x891a, 0x891c,
	0x8920, 0x8926, 0x8927, 0x8928, 0x8930, 0x8931, 0x8932, 0x8935,
	0x8939, 0x893a, 0x893e, 0x8940, 0x8942, 0x8945, 0x8946, 0x8949,
	0x894f, 0x8952, 0x8957, 0x895a, 0x895b, 0x895c, 0x8961, 0x8962,
	0x8963, 0x896b, 0x896e, 0x8970, 0x8973, 0x8975, 0x897a,
	/* 0x5d21 - 0x5d7e */
		0x897b, 0x897c, 0x897d, 0x8989, 0x898d, 0x8990, 0x8994,
	0x8995, 0x899b, 0x899c, 0x899f, 0x89a0, 0x89a5, 0x89b0, 0x89b4,
	0x89b5, 0x89b6, 0x89b7, 0x89bc, 0x89d4, 0x89d5, 0x89d6, 0x89d7,
	0x89d8, 0x89e5, 0x89e9, 0x89eb, 0x89ed, 0x89f1, 0x89f3, 0x89f6,
	0x89f9, 0x89fd, 0x89ff, 0x8a04, 0x8a05, 0x8a07, 0x8a0f, 0x8a11,
	0x8a12, 0x8a14, 0x8a15, 0x8a1e, 0x8a20, 0x8a22, 0x8a24, 0x8a26,
	0x8a2b, 0x8a2c, 0x8a2f, 0x8a35, 0x8a37, 0x8a3d, 0x8a3e, 0x8a40,
	0x8a43, 0x8a45, 0x8a47, 0x8a49, 0x8a4d, 0x8a4e, 0x8a53, 0x8a56,
	0x8a57, 0x8a58, 0x8a5c, 0x8a5d, 0x8a61, 0x8a65, 0x8a67, 0x8a75,
	0x8a76, 0x8a77, 0x8a79, 0x8a7a, 0x8a7b, 0x8a7e, 0x8a7f, 0x8a80,
	0x8a83, 0x8a86, 0x8a8b, 0x8a8f, 0x8a90, 0x8a92, 0x8a96, 0x8a97,
	0x8a99, 0x8a9f, 0x8aa7, 0x8aa9, 0x8aae, 0x8aaf, 0x8ab3,
	/* 0x5e21 - 0x5e7e */
		0x8ab6, 0x8ab7, 0x8abb, 0x8abe, 0x8ac3, 0x8ac6, 0x8ac8,
	0x8ac9, 0x8aca, 0x8ad1, 0x8ad3, 0x8ad4, 0x8ad5, 0x8ad7, 0x8add,
	0x8adf, 0x8aec, 0x8af0, 0x8af4, 0x8af5, 0x8af6, 0x8afc, 0x8aff,
	0x8b05, 0x8b06, 0x8b0b, 0x8b11, 0x8b1c, 0x8b1e, 0x8b1f, 0x8b0a,
	0x8b2d, 0x8b30, 0x8b37, 0x8b3c, 0x8b42, 0x8b43, 0x8b44, 0x8b45,
	0x8b46, 0x8b48, 0x8b52, 0x8b53, 0x8b54, 0x8b59, 0x8b4d, 0x8b5e,
	0x8b63, 0x8b6d, 0x8b76, 0x8b78, 0x8b79, 0x8b7c, 0x8b7e, 0x8b81,
	0x8b84, 0x8b85, 0x8b8b, 0x8b8d, 0x8b8f, 0x8b94, 0x8b95, 0x8b9c,
	0x8b9e, 0x8b9f, 0x8c38, 0x8c39, 0x8c3d, 0x8c3e, 0x8c45, 0x8c47,
	0x8c49, 0x8c4b, 0x8c4f, 0x8c51, 0x8c53, 0x8c54, 0x8c57, 0x8c58,
	0x8c5b, 0x8c5d, 0x8c59, 0x8c63, 0x8c64, 0x8c66, 0x8c68, 0x8c69,
	0x8c6d, 0x8c73, 0x8c75, 0x8c76, 0x8c7b, 0x8c7e, 0x8c86,
	/* 0x5f21 - 0x5f7e */
		0x8c87, 0x8c8b, 0x8c90, 0x8c92, 0x8c93, 0x8c99, 0x8c9b,
	0x8c9c, 0x8ca4, 0x8cb9, 0x8cba, 0x8cc5, 0x8cc6, 0x8cc9, 0x8ccb,
	0x8ccf, 0x8cd6, 0x8cd5, 0x8cd9, 0x8cdd, 0x8ce1, 0x8ce8, 0x8cec,
	0x8cef, 0x8cf0, 0x8cf2, 0x8cf5, 0x8cf7, 0x8cf8, 0x8cfe, 0x8cff,
	0x8d01, 0x8d03, 0x8d09, 0x8d12, 0x8d17, 0x8d1b, 0x8d65, 0x8d69,
	0x8d6c, 0x8d6e, 0x8d7f, 0x8d82, 0x8d84, 0x8d88, 0x8d8d, 0x8d90,
	0x8d91, 0x8d95, 0x8d9e, 0x8d9f, 0x8da0, 0x8da6, 0x8dab, 0x8dac,
	0x8daf, 0x8db2, 0x8db5, 0x8db7, 0x8db9, 0x8dbb, 0x8dc0, 0x8dc5,
	0x8dc6, 0x8dc7, 0x8dc8, 0x8dca, 0x8dce, 0x8dd1, 0x8dd4, 0x8dd5,
	0x8dd7, 0x8dd9, 0x8de4, 0x8de5, 0x8de7, 0x8dec, 0x8df0, 0x8dbc,
	0x8df1, 0x8df2, 0x8df4, 0x8dfd, 0x8e01, 0x8e04, 0x8e05, 0x8e06,
	0x8e0b, 0x8e11, 0x8e14, 0x8e16, 0x8e20, 0x8e21, 0x8e22,
	/* 0x6021 - 0x607e */
		0x8e23, 0x8e26, 0x8e27, 0x8e31, 0x8e33, 0x8e36, 0x8e37,
	0x8e38, 0x8e39, 0x8e3d, 0x8e40, 0x8e41, 0x8e4b, 0x8e4d, 0x8e4e,
	0x8e4f, 0x8e54, 0x8e5b, 0x8e5c, 0x8e5d, 0x8e5e, 0x8e61, 0x8e62,
	0x8e69, 0x8e6c, 0x8e6d, 0x8e6f, 0x8e70, 0x8e71, 0x8e79, 0x8e7a,
	0x8e7b, 0x8e82, 0x8e83, 0x8e89, 0x8e90, 0x8e92, 0x8e95, 0x8e9a,
	0x8e9b, 0x8e9d, 0x8e9e, 0x8ea2, 0x8ea7, 0x8ea9, 0x8ead, 0x8eae,
	0x8eb3, 0x8eb5, 0x8eba, 0x8ebb, 0x8ec0, 0x8ec1, 0x8ec3, 0x8ec4,
	0x8ec7, 0x8ecf, 0x8ed1, 0x8ed4, 0x8edc, 0x8ee8, 0x8eee, 0x8ef0,
	0x8ef1, 0x8ef7, 0x8ef9, 0x8efa, 0x8eed, 0x8f00, 0x8f02, 0x8f07,
	0x8f08, 0x8f0f, 0x8f10, 0x8f16, 0x8f17, 0x8f18, 0x8f1e, 0x8f20,
	0x8f21, 0x8f23, 0x8f25, 0x8f27, 0x8f28, 0x8f2c, 0x8f2d, 0x8f2e,
	0x8f34, 0x8f35, 0x8f36, 0x8f37, 0x8f3a, 0x8f40, 0x8f41,
	/* 0x6121 - 0x617e */
		0x8f43, 0x8f47, 0x8f4f, 0x8f51, 0x8f52, 0x8f53, 0x8f54,
	0x8f55, 0x8f58, 0x8f5d, 0x8f5e, 0x8f65, 0x8f9d, 0x8fa0, 0x8fa1,
	0x8fa4, 0x8fa5, 0x8fa6, 0x8fb5, 0x8fb6, 0x8fb8, 0x8fbe, 0x8fc0,
	0x8fc1, 0x8fc6, 0x8fca, 0x8fcb, 0x8fcd, 0x8fd0, 0x8fd2, 0x8fd3,
	0x8fd5, 0x8fe0, 0x8fe3, 0x8fe4, 0x8fe8, 0x8fee, 0x8ff1, 0x8ff5,
	0x8ff6, 0x8ffb, 0x8ffe, 0x9002, 0x9004, 0x9008, 0x900c, 0x9018,
	0x901b, 0x9028, 0x9029, 0x902f, 0x902a, 0x902c, 0x902d, 0x9033,
	0x9034, 0x9037, 0x903f, 0x9043, 0x9044, 0x904c, 0x905b, 0x905d,
	0x9062, 0x9066, 0x9067, 0x906c, 0x9070, 0x9074, 0x9079, 0x9085,
	0x9088, 0x908b, 0x908c, 0x908e, 0x9090, 0x9095, 0x9097, 0x9098,
	0x9099, 0x909b, 0x90a0, 0x90a1, 0x90a2, 0x90a5, 0x90b0, 0x90b2,
	0x90b3, 0x90b4, 0x90b6, 0x90bd, 0x90cc, 0x90be, 0x90c3,
	/* 0x6221 - 0x627e */
		0x90c4, 0x90c5, 0x90c7, 0x90c8, 0x90d5, 0x90d7, 0x90d8,
	0x90d9, 0x90dc, 0x90dd, 0x90df, 0x90e5, 0x90d2, 0x90f6, 0x90eb,
	0x90ef, 0x90f0, 0x90f4, 0x90fe, 0x90ff, 0x9100, 0x9104, 0x9105,
	0x9106, 0x9108, 0x910d, 0x9110, 0x9114, 0x9116, 0x9117, 0x9118,
	0x911a, 0x911c, 0x911e, 0x9120, 0x9125, 0x9122, 0x9123, 0x9127,
	0x9129, 0x912e, 0x912f, 0x9131, 0x9134, 0x9136, 0x9137, 0x9139,
	0x913a, 0x913c, 0x913d, 0x9143, 0x9147, 0x9148, 0x914f, 0x9153,
	0x9157, 0x9159, 0x915a, 0x915b, 0x9161, 0x9164, 0x9167, 0x916d,
	0x9174, 0x9179, 0x917a, 0x917b, 0x9181, 0x9183, 0x9185, 0x9186,
	0x918a, 0x918e, 0x9191, 0x9193, 0x9194, 0x9195, 0x9198, 0x919e,
	0x91a1, 0x91a6, 0x91a8, 0x91ac, 0x91ad, 0x91ae, 0x91b0, 0x91b1,
	0x91b2, 0x91b3, 0x91b6, 0x91bb, 0x91bc, 0x91bd, 0x91bf,
	/* 0x6321 - 0x637e */
		0x91c2, 0x91c3, 0x91c5, 0x91d3, 0x91d4, 0x91d7, 0x91d9,
	0x91da, 0x91de, 0x91e4, 0x91e5, 0x91e9, 0x91ea, 0x91ec, 0x91ed,
	0x91ee, 0x91ef, 0x91f0, 0x91f1, 0x91f7, 0x91f9, 0x91fb, 0x91fd,
	0x9200, 0x9201, 0x9204, 0x9205, 0x9206, 0x9207, 0x9209, 0x920a,
	0x920c, 0x9210, 0x9212, 0x9213, 0x9216, 0x9218, 0x921c, 0x921d,
	0x9223, 0x9224, 0x9225, 0x9226, 0x9228, 0x922e, 0x922f, 0x9230,
	0x9233, 0x9235, 0x9236, 0x9238, 0x9239, 0x923a, 0x923c, 0x923e,
	0x9240, 0x9242, 0x9243, 0x9246, 0x9247, 0x924a, 0x924d, 0x924e,
	0x924f, 0x9251, 0x9258, 0x9259, 0x925c, 0x925d, 0x9260, 0x9261,
	0x9265, 0x9267, 0x9268, 0x9269, 0x926e, 0x926f, 0x9270, 0x9275,
	0x9276, 0x9277, 0x9278, 0x9279, 0x927b, 0x927c, 0x927d, 0x927f,
	0x9288, 0x9289, 0x928a, 0x928d, 0x928e, 0x9292, 0x9297,
	/* 0x6421 - 0x647e */
		0x9299, 0x929f, 0x92a0, 0x92a4, 0x92a5, 0x92a7, 0x92a8,
	0x92ab, 0x92af, 0x92b2, 0x92b6, 0x92b8, 0x92ba, 0x92bb, 0x92bc,
	0x92bd, 0x92bf, 0x92c0, 0x92c1, 0x92c2, 0x92c3, 0x92c5, 0x92c6,
	0x92c7, 0x92c8, 0x92cb, 0x92cc, 0x92cd, 0x92ce, 0x92d0, 0x92d3,
	0x92d5, 0x92d7, 0x92d8, 0x92d9, 0x92dc, 0x92dd, 0x92df, 0x92e0,
	0x92e1, 0x92e3, 0x92e5, 0x92e7, 0x92e8, 0x92ec, 0x92ee, 0x92f0,
	0x92f9, 0x92fb, 0x92ff, 0x9300, 0x9302, 0x9308, 0x930d, 0x9311,
	0x9314, 0x9315, 0x931c, 0x931d, 0x931e, 0x931f, 0x9321, 0x9324,
	0x9325, 0x9327, 0x9329, 0x932a, 0x9333, 0x9334, 0x9336, 0x9337,
	0x9347, 0x9348, 0x9349, 0x9350, 0x9351, 0x9352, 0x9355, 0x9357,
	0x9358, 0x935a, 0x935e, 0x9364, 0x9365, 0x9367, 0x9369, 0x936a,
	0x936d, 0x936f, 0x9370, 0x9371, 0x9373, 0x9374, 0x9376,
	/* 0x6521 - 0x657e */
		0x937a, 0x937d, 0x937f, 0x9380, 0x9381, 0x9382, 0x9388,
	0x938a, 0x938b, 0x938d, 0x938f, 0x9392, 0x9395, 0x9398, 0x939b,
	0x939e, 0x93a1, 0x93a3, 0x93a4, 0x93a6, 0x93a8, 0x93ab, 0x93b4,
	0x93b5, 0x93b6, 0x93ba, 0x93a9, 0x93c1, 0x93c4, 0x93c5, 0x93c6,
	0x93c7, 0x93c9, 0x93ca, 0x93cb, 0x93cc, 0x93cd, 0x93d3, 0x93d9,
	0x93dc, 0x93de, 0x93df, 0x93e2, 0x93e6, 0x93e7, 0x93f9, 0x93f7,
	0x93f8, 0x93fa, 0x93fb, 0x93fd, 0x9401, 0x9402, 0x9404, 0x9408,
	0x9409, 0x940d, 0x940e, 0x940f, 0x9415, 0x9416, 0x9417, 0x941f,
	0x942e, 0x942f, 0x9431, 0x9432, 0x9433, 0x9434, 0x943b, 0x943f,
	0x943d, 0x9443, 0x9445, 0x9448, 0x944a, 0x944c, 0x9455, 0x9459,
	0x945c, 0x945f, 0x9461, 0x9463, 0x9468, 0x946b, 0x946d, 0x946e,
	0x946f, 0x9471, 0x9472, 0x9484, 0x9483, 0x9578, 0x9579,
	/* 0x6621 - 0x667e */
		0x957e, 0x9584, 0x9588, 0x958c, 0x958d, 0x958e, 0x959d,
	0x959e, 0x959f, 0x95a1, 0x95a6, 0x95a9, 0x95ab, 0x95ac, 0x95b4,
	0x95b6, 0x95ba, 0x95bd, 0x95bf, 0x95c6, 0x95c8, 0x95c9, 0x95cb,
	0x95d0, 0x95d1, 0x95d2, 0x95d3, 0x95d9, 0x95da, 0x95dd, 0x95de,
	0x95df, 0x95e0, 0x95e4, 0x95e6, 0x961d, 0x961e, 0x9622, 0x9624,
	0x9625, 0x9626, 0x962c, 0x9631, 0x9633, 0x9637, 0x9638, 0x9639,
	0x963a, 0x963c, 0x963d, 0x9641, 0x9652, 0x9654, 0x9656, 0x9657,
	0x9658, 0x9661, 0x966e, 0x9674, 0x967b, 0x967c, 0x967e, 0x967f,
	0x9681, 0x9682, 0x9683, 0x9684, 0x9689, 0x9691, 0x9696, 0x969a,
	0x969d, 0x969f, 0x96a4, 0x96a5, 0x96a6, 0x96a9, 0x96ae, 0x96af,
	0x96b3, 0x96ba, 0x96ca, 0x96d2, 0x5db2, 0x96d8, 0x96da, 0x96dd,
	0x96de, 0x96df, 0x96e9, 0x96ef, 0x96f1, 0x96fa, 0x9702,
	/* 0x6721 - 0x677e */
		0x9703, 0x9705, 0x9709, 0x971a, 0x971b, 0x971d, 0x9721,
	0x9722, 0x9723, 0x9728, 0x9731, 0x9733, 0x9741, 0x9743, 0x974a,
	0x974e, 0x974f, 0x9755, 0x9757, 0x9758, 0x975a, 0x975b, 0x9763,
	0x9767, 0x976a, 0x976e, 0x9773, 0x9776, 0x9777, 0x9778, 0x977b,
	0x977d, 0x977f, 0x9780, 0x9789, 0x9795, 0x9796, 0x9797, 0x9799,
	0x979a, 0x979e, 0x979f, 0x97a2, 0x97ac, 0x97ae, 0x97b1, 0x97b2,
	0x97b5, 0x97b6, 0x97b8, 0x97b9, 0x97ba, 0x97bc, 0x97be, 0x97bf,
	0x97c1, 0x97c4, 0x97c5, 0x97c7, 0x97c9, 0x97ca, 0x97cc, 0x97cd,
	0x97ce, 0x97d0, 0x97d1, 0x97d4, 0x97d7, 0x97d8, 0x97d9, 0x97dd,
	0x97de, 0x97e0, 0x97db, 0x97e1, 0x97e4, 0x97ef, 0x97f1, 0x97f4,
	0x97f7, 0x97f8, 0x97fa, 0x9807, 0x980a, 0x9819, 0x980d, 0x980e,
	0x9814, 0x9816, 0x981c, 0x981e, 0x9820, 0x9823, 0x9826,
	/* 0x6821 - 0x687e */
		0x982b, 0x982e, 0x982f, 0x9830, 0x9832, 0x9833, 0x9835,
	0x9825, 0x983e, 0x9844, 0x9847, 0x984a, 0x9851, 0x9852, 0x9853,
	0x9856, 0x9857, 0x9859, 0x985a, 0x9862, 0x9863, 0x9865, 0x9866,
	0x986a, 0x986c, 0x98ab, 0x98ad, 0x98ae, 0x98b0, 0x98b4, 0x98b7,
	0x98b8, 0x98ba, 0x98bb, 0x98bf, 0x98c2, 0x98c5, 0x98c8, 0x98cc,
	0x98e1, 0x98e3, 0x98e5, 0x98e6, 0x98e7, 0x98ea, 0x98f3, 0x98f6,
	0x9902, 0x9907, 0x9908, 0x9911, 0x9915, 0x9916, 0x9917, 0x991a,
	0x991b, 0x991c, 0x991f, 0x9922, 0x9926, 0x9927, 0x992b, 0x9931,
	0x9932, 0x9933, 0x9934, 0x9935, 0x9939, 0x993a, 0x993b, 0x993c,
	0x9940, 0x9941, 0x9946, 0x9947, 0x9948, 0x994d, 0x994e, 0x9954,
	0x9958, 0x9959, 0x995b, 0x995c, 0x995e, 0x995f, 0x9960, 0x999b,
	0x999d, 0x999f, 0x99a6, 0x99b0, 0x99b1, 0x99b2, 0x99b5,
	/* 0x6921 - 0x697e */
		0x99b9, 0x99ba, 0x99bd, 0x99bf, 0x99c3, 0x99c9, 0x99d3,
	0x99d4, 0x99d9, 0x99da, 0x99dc, 0x99de, 0x99e7, 0x99ea, 0x99eb,
	0x99ec, 0x99f0, 0x99f4, 0x99f5, 0x99f9, 0x99fd, 0x99fe, 0x9a02,
	0x9a03, 0x9a04, 0x9a0b, 0x9a0c, 0x9a10, 0x9a11, 0x9a16, 0x9a1e,
	0x9a20, 0x9a22, 0x9a23, 0x9a24, 0x9a27, 0x9a2d, 0x9a2e, 0x9a33,
	0x9a35, 0x9a36, 0x9a38, 0x9a47, 0x9a41, 0x9a44, 0x9a4a, 0x9a4b,
	0x9a4c, 0x9a4e, 0x9a51, 0x9a54, 0x9a56, 0x9a5d, 0x9aaa, 0x9aac,
	0x9aae, 0x9aaf, 0x9ab2, 0x9ab4, 0x9ab5, 0x9ab6, 0x9ab9, 0x9abb,
	0x9abe, 0x9abf, 0x9ac1, 0x9ac3, 0x9ac6, 0x9ac8, 0x9ace, 0x9ad0,
	0x9ad2, 0x9ad5, 0x9ad6, 0x9ad7, 0x9adb, 0x9adc, 0x9ae0, 0x9ae4,
	0x9ae5, 0x9ae7, 0x9ae9, 0x9aec, 0x9af2, 0x9af3, 0x9af5, 0x9af9,
	0x9afa, 0x9afd, 0x9aff, 0x9b00, 0x9b01, 0x9b02, 0x9b03,
	/* 0x6a21 - 0x6a7e */
		0x9b04, 0x9b05, 0x9b08, 0x9b09, 0x9b0b, 0x9b0c, 0x9b0d,
	0x9b0e, 0x9b10, 0x9b12, 0x9b16, 0x9b19, 0x9b1b, 0x9b1c, 0x9b20,
	0x9b26, 0x9b2b, 0x9b2d, 0x9b33, 0x9b34, 0x9b35, 0x9b37, 0x9b39,
	0x9b3a, 0x9b3d, 0x9b48, 0x9b4b, 0x9b4c, 0x9b55, 0x9b56, 0x9b57,
	0x9b5b, 0x9b5e, 0x9b61, 0x9b63, 0x9b65, 0x9b66, 0x9b68, 0x9b6a,
	0x9b6b, 0x9b6c, 0x9b6d, 0x9b6e, 0x9b73, 0x9b75, 0x9b77, 0x9b78,
	0x9b79, 0x9b7f, 0x9b80, 0x9b84, 0x9b85, 0x9b86, 0x9b87, 0x9b89,
	0x9b8a, 0x9b8b, 0x9b8d, 0x9b8f, 0x9b90, 0x9b94, 0x9b9a, 0x9b9d,
	0x9b9e, 0x9ba6, 0x9ba7, 0x9ba9, 0x9bac, 0x9bb0, 0x9bb1, 0x9bb2,
	0x9bb7, 0x9bb8, 0x9bbb, 0x9bbc, 0x9bbe, 0x9bbf, 0x9bc1, 0x9bc7,
	0x9bc8, 0x9bce, 0x9bd0, 0x9bd7, 0x9bd8, 0x9bdd, 0x9bdf, 0x9be5,
	0x9be7, 0x9bea, 0x9beb, 0x9bef, 0x9bf3, 0x9bf7, 0x9bf8,
	/* 0x6b21 - 0x6b7e */
		0x9bf9, 0x9bfa, 0x9bfd, 0x9bff, 0x9c00, 0x9c02, 0x9c0b,
	0x9c0f, 0x9c11, 0x9c16, 0x9c18, 0x9c19, 0x9c1a, 0x9c1c, 0x9c1e,
	0x9c22, 0x9c23, 0x9c26, 0x9c27, 0x9c28, 0x9c29, 0x9c2a, 0x9c31,
	0x9c35, 0x9c36, 0x9c37, 0x9c3d, 0x9c41, 0x9c43, 0x9c44, 0x9c45,
	0x9c49, 0x9c4a, 0x9c4e, 0x9c4f, 0x9c50, 0x9c53, 0x9c54, 0x9c56,
	0x9c58, 0x9c5b, 0x9c5d, 0x9c5e, 0x9c5f, 0x9c63, 0x9c69, 0x9c6a,
	0x9c5c, 0x9c6b, 0x9c68, 0x9c6e, 0x9c70, 0x9c72, 0x9c75, 0x9c77,
	0x9c7b, 0x9ce6, 0x9cf2, 0x9cf7, 0x9cf9, 0x9d0b, 0x9d02, 0x9d11,
	0x9d17, 0x9d18, 0x9d1c, 0x9d1d, 0x9d1e, 0x9d2f, 0x9d30, 0x9d32,
	0x9d33, 0x9d34, 0x9d3a, 0x9d3c, 0x9d45, 0x9d3d, 0x9d42, 0x9d43,
	0x9d47, 0x9d4a, 0x9d53, 0x9d54, 0x9d5f, 0x9d63, 0x9d62, 0x9d65,
	0x9d69, 0x9d6a, 0x9d6b, 0x9d70, 0x9d76, 0x9d77, 0x9d7b,
	/* 0x6c21 - 0x6c7e */
		0x9d7c, 0x9d7e, 0x9d83, 0x9d84, 0x9d86, 0x9d8a, 0x9d8d,
	0x9d8e, 0x9d92, 0x9d93, 0x9d95, 0x9d96, 0x9d97, 0x9d98, 0x9da1,
	0x9daa, 0x9dac, 0x9dae, 0x9db1, 0x9db5, 0x9db9, 0x9dbc, 0x9dbf,
	0x9dc3, 0x9dc7, 0x9dc9, 0x9dca, 0x9dd4, 0x9dd5, 0x9dd6, 0x9dd7,
	0x9dda, 0x9dde, 0x9ddf, 0x9de0, 0x9de5, 0x9de7, 0x9de9, 0x9deb,
	0x9dee, 0x9df0, 0x9df3, 0x9df4, 0x9dfe, 0x9e0a, 0x9e02, 0x9e07,
	0x9e0e, 0x9e10, 0x9e11, 0x9e12, 0x9e15, 0x9e16, 0x9e19, 0x9e1c,
	0x9e1d, 0x9e7a, 0x9e7b, 0x9e7c, 0x9e80, 0x9e82, 0x9e83, 0x9e84,
	0x9e85, 0x9e87, 0x9e8e, 0x9e8f, 0x9e96, 0x9e98, 0x9e9b, 0x9e9e,
	0x9ea4, 0x9ea8, 0x9eac, 0x9eae, 0x9eaf, 0x9eb0, 0x9eb3, 0x9eb4,
	0x9eb5, 0x9ec6, 0x9ec8, 0x9ecb, 0x9ed5, 0x9edf, 0x9ee4, 0x9ee7,
	0x9eec, 0x9eed, 0x9eee, 0x9ef0, 0x9ef1, 0x9ef2, 0x9ef5,
	/* 0x6d21 - 0x6d7e */
		0x9ef8, 0x9eff, 0x9f02, 0x9f03, 0x9f09, 0x9f0f, 0x9f10,
	0x9f11, 0x9f12, 0x9f14, 0x9f16, 0x9f17, 0x9f19, 0x9f1a, 0x9f1b,
	0x9f1f, 0x9f22, 0x9f26, 0x9f2a, 0x9f2b, 0x9f2f, 0x9f31, 0x9f32,
	0x9f34, 0x9f37, 0x9f39, 0x9f3a, 0x9f3c, 0x9f3d, 0x9f3f, 0x9f41,
	0x9f43, 0x9f44, 0x9f45, 0x9f46, 0x9f47, 0x9f53, 0x9f55, 0x9f56,
	0x9f57, 0x9f58, 0x9f5a, 0x9f5d, 0x9f5e, 0x9f68, 0x9f69, 0x9f6d,
	0x9f6e, 0x9f6f, 0x9f70, 0x9f71, 0x9f73, 0x9f75, 0x9f7a, 0x9f7d,
	0x9f8f, 0x9f90, 0x9f91, 0x9f92, 0x9f94, 0x9f96, 0x9f97, 0x9f9e,
	0x9fa1, 0x9fa2, 0x9fa3, 0x9fa5, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x6e21 - 0x6e7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x6f21 - 0x6f7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7021 - 0x707e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7121 - 0x717e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7221 - 0x727e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7321 - 0x737e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x2170, 0x2171, 0x2172, 0x2173, 0x2174,
	0x2175, 0x2176, 0x2177, 0x2178, 0x2179, 0x2160, 0x2161,
	/* 0x7421 - 0x747e */
		0x2162, 0x2163, 0x2164, 0x2165, 0x2166, 0x2167, 0x2168,
	0x2169, 0xff07, 0xff02, 0x3231, 0x2116, 0x2121, 0x70bb, 0x4efc,
	0x50f4, 0x51ec, 0x5307, 0x5324, 0xfa0e, 0x548a, 0x5759, 0xfa0f,
	0xfa10, 0x589e, 0x5bec, 0x5cf5, 0x5d53, 0xfa11, 0x5fb7, 0x6085,
	0x6120, 0x654e, 0x663b, 0x6665, 0xfa12, 0xf929, 0x6801, 0xfa13,
	0xfa14, 0x6a6b, 0x6ae2, 0x6df8, 0x6df2, 0x7028, 0xfa15, 0xfa16,
	0x7501, 0x7682, 0x769e, 0xfa17, 0x7930, 0xfa18, 0xfa19, 0xfa1a,
	0xfa1b, 0x7ae7, 0xfa1c, 0xfa1d, 0x7da0, 0x7dd6, 0xfa1e, 0x8362,
	0xfa1f, 0x85b0, 0xfa20, 0xfa21, 0x8807, 0xfa22, 0x8b7f, 0x8cf4,
	0x8d76, 0xfa23, 0xfa24, 0xfa25, 0x90de, 0xfa26, 0x9115, 0xfa27,
	0xfa28, 0x9592, 0xf9dc, 0xfa29, 0x973b, 0x974d, 0x9751, 0xfa2a,
	0xfa2b, 0xfa2c, 0x999e, 0x9ad9, 0x9b72, 0xfa2d, 0x9ed1,
	/* 0x7521 - 0x757e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7621 - 0x767e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7721 - 0x777e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7821 - 0x787e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7921 - 0x797e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7a21 - 0x7a7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7b21 - 0x7b7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7c21 - 0x7c7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7d21 - 0x7d7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x7e21 - 0x7e7e */
		0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

#endif

#ifdef USE_JISX0212
static uint Jisx0212ToUnicode11(uint h, uint l)
{
    if ((0x0021 <= h) && (h <= 0x007e) && (0x0021 <= l) && (l <= 0x007e)) {
	return jisx0212_to_unicode[(h - 0x0021) * 0x005e + (l - 0x0021)];
    }
    return 0x0000;
}
#else
static uint Jisx0212ToUnicode11(uint h, uint l)
{
    return 0x0000;
}
#endif

#ifdef USE_JISX0212

/*
 * This data is derived from Unicode 1.1,
 * JIS X 0212 (1990) to Unicode mapping table version 0.9 .
 * (In addition IBM Vender Defined Char included)
 */
static unsigned short const unicode_to_jisx0212_00[] = {
	/* 0x0000 - 0x00ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2237, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x2242, 0x0000, 0x0000, 0x2270, 0x0000, 0x2243, 0x0000,
	0x0000, 0x226d, 0x226c, 0x0000, 0x0000, 0x0000, 0x226e, 0x2234,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x2231, 0x0000, 0x226b, 0x0000, 0x0000, 0x0000, 0x0000, 0x2244,
	0x2a22, 0x2a21, 0x2a24, 0x2a2a, 0x2a23, 0x2a29, 0x2921, 0x2a2e,
	0x2a32, 0x2a31, 0x2a34, 0x2a33, 0x2a40, 0x2a3f, 0x2a42, 0x2a41,
	0x0000, 0x2a50, 0x2a52, 0x2a51, 0x2a54, 0x2a58, 0x2a53, 0x0000,
	0x292c, 0x2a63, 0x2a62, 0x2a65, 0x2a64, 0x2a72, 0x2930, 0x294e,
	0x2b22, 0x2b21, 0x2b24, 0x2b2a, 0x2b23, 0x2b29, 0x2941, 0x2b2e,
	0x2b32, 0x2b31, 0x2b34, 0x2b33, 0x2b40, 0x2b3f, 0x2b42, 0x2b41,
	0x2943, 0x2b50, 0x2b52, 0x2b51, 0x2b54, 0x2b58, 0x2b53, 0x0000,
	0x294c, 0x2b63, 0x2b62, 0x2b65, 0x2b64, 0x2b72, 0x2950, 0x2b73,
};

static unsigned short const unicode_to_jisx0212_01[] = {
	/* 0x0100 - 0x01ff */
	0x2a27, 0x2b27, 0x2a25, 0x2b25, 0x2a28, 0x2b28, 0x2a2b, 0x2b2b,
	0x2a2c, 0x2b2c, 0x2a2f, 0x2b2f, 0x2a2d, 0x2b2d, 0x2a30, 0x2b30,
	0x2922, 0x2942, 0x2a37, 0x2b37, 0x0000, 0x0000, 0x2a36, 0x2b36,
	0x2a38, 0x2b38, 0x2a35, 0x2b35, 0x2a3a, 0x2b3a, 0x2a3b, 0x2b3b,
	0x2a3d, 0x2b3d, 0x2a3c, 0x0000, 0x2a3e, 0x2b3e, 0x2924, 0x2944,
	0x2a47, 0x2b47, 0x2a45, 0x2b45, 0x0000, 0x0000, 0x2a46, 0x2b46,
	0x2a44, 0x2945, 0x2926, 0x2946, 0x2a48, 0x2b48, 0x2a49, 0x2b49,
	0x2947, 0x2a4a, 0x2b4a, 0x2a4c, 0x2b4c, 0x2a4b, 0x2b4b, 0x2929,
	0x2949, 0x2928, 0x2948, 0x2a4d, 0x2b4d, 0x2a4f, 0x2b4f, 0x2a4e,
	0x2b4e, 0x294a, 0x292b, 0x294b, 0x2a57, 0x2b57, 0x0000, 0x0000,
	0x2a56, 0x2b56, 0x292d, 0x294d, 0x2a59, 0x2b59, 0x2a5b, 0x2b5b,
	0x2a5a, 0x2b5a, 0x2a5c, 0x2b5c, 0x2a5d, 0x2b5d, 0x2a5f, 0x2b5f,
	0x2a5e, 0x2b5e, 0x2a61, 0x2b61, 0x2a60, 0x2b60, 0x292f, 0x294f,
	0x2a6c, 0x2b6c, 0x2a69, 0x2b69, 0x2a66, 0x2b66, 0x2a6b, 0x2b6b,
	0x2a68, 0x2b68, 0x2a6a, 0x2b6a, 0x2a71, 0x2b71, 0x2a74, 0x2b74,
	0x2a73, 0x2a75, 0x2b75, 0x2a77, 0x2b77, 0x2a76, 0x2b76, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2a26, 0x2b26, 0x2a43,
	0x2b43, 0x2a55, 0x2b55, 0x2a67, 0x2b67, 0x2a70, 0x2b70, 0x2a6d,
	0x2b6d, 0x2a6f, 0x2b6f, 0x2a6e, 0x2b6e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2b39, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_02[] = {
	/* 0x0200 - 0x02ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2230,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x222f, 0x2232, 0x2236, 0x2235, 0x0000, 0x2233, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_03[] = {
	/* 0x0300 - 0x03ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x2238, 0x2239, 0x2661, 0x0000,
	0x2662, 0x2663, 0x2664, 0x0000, 0x2667, 0x0000, 0x2669, 0x266c,
	0x2676, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2665, 0x266a, 0x2671, 0x2672, 0x2673, 0x2674,
	0x267b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2678, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2675, 0x267a, 0x2677, 0x2679, 0x267c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_04[] = {
	/* 0x0400 - 0x04ff */
	0x0000, 0x0000, 0x2742, 0x2743, 0x2744, 0x2745, 0x2746, 0x2747,
	0x2748, 0x2749, 0x274a, 0x274b, 0x274c, 0x0000, 0x274d, 0x274e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x2772, 0x2773, 0x2774, 0x2775, 0x2776, 0x2777,
	0x2778, 0x2779, 0x277a, 0x277b, 0x277c, 0x0000, 0x277d, 0x277e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_21[] = {
	/* 0x2100 - 0x21ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2271, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x742d, 0x226f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x737d, 0x737e, 0x7421, 0x7422, 0x7423, 0x7424, 0x7425, 0x7426,
	0x7427, 0x7428, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x7373, 0x7374, 0x7375, 0x7376, 0x7377, 0x7378, 0x7379, 0x737a,
	0x737b, 0x737c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_32[] = {
	/* 0x3200 - 0x32ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x742b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_4e[] = {
	/* 0x4e00 - 0x4eff */
	0x0000, 0x0000, 0x3021, 0x0000, 0x3022, 0x3023, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3024, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3025, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3026,
	0x0000, 0x0000, 0x0000, 0x3027, 0x3028, 0x0000, 0x0000, 0x0000,
	0x3029, 0x0000, 0x0000, 0x302a, 0x0000, 0x0000, 0x302b, 0x302c,
	0x302d, 0x0000, 0x0000, 0x0000, 0x0000, 0x302e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x302f, 0x3030, 0x0000, 0x0000, 0x3031, 0x0000, 0x0000, 0x3032,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3033, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3034, 0x0000, 0x3035, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3036, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3037, 0x3038, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3039, 0x303a, 0x0000, 0x0000,
	0x0000, 0x303b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x303c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x303d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x303e, 0x303f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3040, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3041,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3042, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3043, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3044, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3045, 0x3046, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3047, 0x3048, 0x3049, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x304a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x304b,
	0x0000, 0x304c, 0x0000, 0x304d, 0x0000, 0x304e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x742f, 0x304f, 0x3050, 0x3051,
};

static unsigned short const unicode_to_jisx0212_4f[] = {
	/* 0x4f00 - 0x4fff */
	0x3052, 0x0000, 0x3053, 0x3054, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3055, 0x0000, 0x0000, 0x3056, 0x3057, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3058, 0x0000, 0x0000, 0x3059, 0x305a, 0x305b,
	0x0000, 0x305c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x305d, 0x0000,
	0x0000, 0x305e, 0x0000, 0x3060, 0x0000, 0x3061, 0x0000, 0x3062,
	0x0000, 0x3063, 0x0000, 0x3064, 0x0000, 0x0000, 0x3065, 0x0000,
	0x3066, 0x0000, 0x3067, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3068, 0x3069, 0x0000, 0x306a, 0x306b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x306c, 0x0000, 0x306d, 0x0000, 0x306e, 0x0000,
	0x306f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3070,
	0x305f, 0x0000, 0x0000, 0x3071, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3072, 0x0000, 0x3073, 0x0000, 0x3074, 0x0000,
	0x0000, 0x3075, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3076,
	0x3077, 0x3078, 0x3079, 0x0000, 0x0000, 0x307a, 0x307b, 0x0000,
	0x0000, 0x307c, 0x307d, 0x0000, 0x307e, 0x3121, 0x0000, 0x0000,
	0x0000, 0x3122, 0x3123, 0x0000, 0x3124, 0x0000, 0x3125, 0x0000,
	0x3126, 0x0000, 0x3127, 0x3128, 0x3129, 0x0000, 0x0000, 0x312a,
	0x0000, 0x312b, 0x312c, 0x0000, 0x0000, 0x0000, 0x312d, 0x312e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x312f, 0x0000, 0x0000, 0x0000, 0x0000, 0x3130,
	0x0000, 0x3131, 0x0000, 0x3132, 0x3133, 0x3134, 0x3135, 0x0000,
	0x3136, 0x3137, 0x0000, 0x0000, 0x0000, 0x3138, 0x3139, 0x0000,
	0x313a, 0x313b, 0x0000, 0x313c, 0x313d, 0x313e, 0x0000, 0x313f,
	0x0000, 0x0000, 0x3140, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3141, 0x0000, 0x0000, 0x0000,
	0x3142, 0x0000, 0x3143, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3144, 0x0000, 0x3145, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3146, 0x3147, 0x0000, 0x3148,
};

static unsigned short const unicode_to_jisx0212_50[] = {
	/* 0x5000 - 0x50ff */
	0x3149, 0x314a, 0x0000, 0x0000, 0x314b, 0x0000, 0x0000, 0x314c,
	0x0000, 0x0000, 0x314d, 0x0000, 0x314e, 0x0000, 0x314f, 0x0000,
	0x3150, 0x0000, 0x0000, 0x3151, 0x0000, 0x0000, 0x0000, 0x3152,
	0x3153, 0x0000, 0x0000, 0x3154, 0x3155, 0x3156, 0x3157, 0x0000,
	0x0000, 0x0000, 0x3158, 0x0000, 0x0000, 0x0000, 0x0000, 0x3159,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x315a, 0x0000,
	0x315b, 0x0000, 0x315c, 0x315d, 0x0000, 0x315e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3176, 0x0000, 0x0000, 0x0000, 0x0000,
	0x315f, 0x3160, 0x3161, 0x0000, 0x0000, 0x3162, 0x3163, 0x0000,
	0x0000, 0x0000, 0x3164, 0x0000, 0x3165, 0x0000, 0x3166, 0x0000,
	0x0000, 0x3167, 0x3168, 0x3169, 0x0000, 0x0000, 0x0000, 0x316a,
	0x0000, 0x316b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x316c,
	0x316d, 0x0000, 0x316e, 0x316f, 0x0000, 0x0000, 0x3170, 0x3171,
	0x0000, 0x0000, 0x3172, 0x0000, 0x0000, 0x3173, 0x0000, 0x0000,
	0x3174, 0x3175, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3177, 0x0000, 0x3178, 0x3179, 0x0000, 0x317a, 0x0000,
	0x0000, 0x0000, 0x317b, 0x0000, 0x0000, 0x0000, 0x317c, 0x317d,
	0x317e, 0x0000, 0x3221, 0x3222, 0x3223, 0x0000, 0x3224, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3225, 0x3226, 0x0000, 0x3227, 0x3228,
	0x3229, 0x322a, 0x322b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x322c, 0x0000, 0x0000, 0x0000, 0x0000, 0x322d,
	0x322e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x322f, 0x3230, 0x0000, 0x0000, 0x3231, 0x0000, 0x0000,
	0x3232, 0x0000, 0x0000, 0x3233, 0x3234, 0x0000, 0x0000, 0x3235,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3236, 0x0000, 0x3237, 0x0000,
	0x3238, 0x0000, 0x0000, 0x3239, 0x323a, 0x0000, 0x0000, 0x0000,
	0x323b, 0x0000, 0x0000, 0x0000, 0x323c, 0x323d, 0x0000, 0x323e,
	0x0000, 0x0000, 0x323f, 0x0000, 0x3240, 0x0000, 0x3241, 0x0000,
	0x3242, 0x3243, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3244,
	0x0000, 0x3245, 0x3251, 0x0000, 0x7430, 0x0000, 0x3246, 0x0000,
	0x0000, 0x0000, 0x3247, 0x0000, 0x0000, 0x0000, 0x3248, 0x0000,
};

static unsigned short const unicode_to_jisx0212_51[] = {
	/* 0x5100 - 0x51ff */
	0x0000, 0x0000, 0x0000, 0x3249, 0x0000, 0x0000, 0x324a, 0x324b,
	0x324c, 0x0000, 0x0000, 0x324d, 0x324e, 0x324f, 0x3250, 0x0000,
	0x3252, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3253,
	0x0000, 0x3254, 0x0000, 0x3255, 0x3256, 0x3257, 0x3258, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3259, 0x0000, 0x0000, 0x0000, 0x325a,
	0x325b, 0x0000, 0x0000, 0x0000, 0x325c, 0x325d, 0x0000, 0x325e,
	0x0000, 0x325f, 0x0000, 0x3260, 0x3261, 0x3262, 0x0000, 0x0000,
	0x3263, 0x3264, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3265, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3266, 0x0000, 0x0000, 0x0000, 0x0000, 0x3267,
	0x0000, 0x0000, 0x0000, 0x3268, 0x0000, 0x3269, 0x0000, 0x326a,
	0x326b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x326c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x326d, 0x0000, 0x326e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x326f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3270, 0x3271, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3272, 0x0000, 0x0000, 0x3273, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3274, 0x0000, 0x0000, 0x0000, 0x0000, 0x3275, 0x0000, 0x0000,
	0x0000, 0x3276, 0x0000, 0x3277, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3278, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3279, 0x0000, 0x327a, 0x0000, 0x327b, 0x0000, 0x327c, 0x327d,
	0x0000, 0x0000, 0x327e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3321, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3322,
	0x0000, 0x3323, 0x3324, 0x3325, 0x0000, 0x3326, 0x0000, 0x0000,
	0x3327, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3328, 0x0000,
	0x0000, 0x0000, 0x3329, 0x0000, 0x0000, 0x332a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7431, 0x0000, 0x332b, 0x0000,
	0x0000, 0x0000, 0x332c, 0x332d, 0x332e, 0x0000, 0x0000, 0x332f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_52[] = {
	/* 0x5200 - 0x52ff */
	0x0000, 0x3330, 0x3331, 0x0000, 0x0000, 0x3332, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3333, 0x3334, 0x0000, 0x3335, 0x3336, 0x0000,
	0x3337, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3338, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3339, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x333a, 0x333b, 0x0000, 0x0000, 0x333c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x333d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x333e, 0x0000, 0x0000,
	0x0000, 0x333f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3340, 0x0000, 0x3341,
	0x3342, 0x0000, 0x3343, 0x0000, 0x3344, 0x0000, 0x0000, 0x3345,
	0x3346, 0x3347, 0x0000, 0x0000, 0x0000, 0x0000, 0x3348, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3349, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x334a,
	0x334b, 0x334c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x334d, 0x0000, 0x334e, 0x0000, 0x0000, 0x334f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3350, 0x0000, 0x3351, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3352, 0x0000, 0x3353, 0x3354, 0x3355,
	0x3356, 0x0000, 0x3357, 0x0000, 0x3358, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3359, 0x335a, 0x335b, 0x335c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x335d,
	0x335e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x335f, 0x3360,
	0x3361, 0x0000, 0x3362, 0x3363, 0x0000, 0x3364, 0x0000, 0x0000,
	0x3365, 0x0000, 0x0000, 0x0000, 0x3366, 0x0000, 0x3367, 0x0000,
	0x3368, 0x0000, 0x0000, 0x0000, 0x3369, 0x0000, 0x0000, 0x336a,
	0x0000, 0x336b, 0x0000, 0x0000, 0x336c, 0x0000, 0x336d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x336e, 0x336f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3370, 0x0000, 0x0000, 0x0000, 0x3371, 0x0000, 0x0000,
	0x3372, 0x3373, 0x3374, 0x0000, 0x3375, 0x0000, 0x0000, 0x0000,
	0x3376, 0x3377, 0x0000, 0x0000, 0x3378, 0x0000, 0x3379, 0x337a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_53[] = {
	/* 0x5300 - 0x53ff */
	0x337b, 0x0000, 0x0000, 0x337c, 0x0000, 0x0000, 0x0000, 0x7432,
	0x0000, 0x0000, 0x337d, 0x337e, 0x3421, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3422, 0x0000, 0x3423, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3424, 0x0000, 0x0000, 0x3425, 0x3426, 0x0000, 0x3427, 0x3428,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7433, 0x3429, 0x0000, 0x342a,
	0x342b, 0x342c, 0x0000, 0x342d, 0x342e, 0x342f, 0x0000, 0x0000,
	0x3430, 0x0000, 0x3431, 0x0000, 0x0000, 0x3432, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3433, 0x3434, 0x3435, 0x0000,
	0x0000, 0x0000, 0x3436, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3438, 0x3437, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3439, 0x0000, 0x343a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x343b, 0x0000, 0x343c, 0x0000, 0x343d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x343e, 0x343f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3440, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3441, 0x0000, 0x0000, 0x0000, 0x0000, 0x3442, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3443, 0x0000, 0x0000, 0x0000, 0x3444,
	0x3445, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3446, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3447, 0x3448, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3449, 0x0000, 0x0000, 0x0000, 0x344a, 0x0000, 0x0000,
	0x0000, 0x344b, 0x0000, 0x0000, 0x344c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x344d, 0x344e, 0x0000, 0x0000, 0x0000, 0x344f,
	0x0000, 0x0000, 0x3450, 0x0000, 0x3451, 0x3452, 0x0000, 0x3453,
	0x3454, 0x0000, 0x3455, 0x0000, 0x0000, 0x3456, 0x0000, 0x0000,
	0x3457, 0x0000, 0x0000, 0x0000, 0x0000, 0x3458, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3459,
	0x0000, 0x0000, 0x345a, 0x345b, 0x0000, 0x345c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x345d, 0x0000, 0x0000, 0x345e, 0x345f, 0x0000,
	0x3460, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3461, 0x3462,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3463, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_54[] = {
	/* 0x5400 - 0x54ff */
	0x0000, 0x0000, 0x3464, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3465, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3466, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3467, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3468,
	0x3469, 0x0000, 0x346a, 0x0000, 0x0000, 0x0000, 0x0000, 0x346b,
	0x0000, 0x346c, 0x0000, 0x0000, 0x346d, 0x346e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x346f, 0x3470, 0x0000, 0x0000, 0x3471,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3472, 0x0000, 0x3473,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3474, 0x0000,
	0x0000, 0x0000, 0x3475, 0x0000, 0x3476, 0x0000, 0x3477, 0x3478,
	0x0000, 0x3479, 0x0000, 0x347a, 0x0000, 0x347b, 0x347c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x347d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x347e,
	0x0000, 0x3521, 0x0000, 0x3522, 0x0000, 0x3523, 0x0000, 0x0000,
	0x3524, 0x3525, 0x7435, 0x0000, 0x0000, 0x3526, 0x0000, 0x0000,
	0x0000, 0x3527, 0x0000, 0x0000, 0x0000, 0x3528, 0x3529, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x352a, 0x0000, 0x0000, 0x352b,
	0x0000, 0x352c, 0x0000, 0x0000, 0x0000, 0x0000, 0x352d, 0x352e,
	0x0000, 0x352f, 0x3530, 0x0000, 0x0000, 0x3531, 0x3532, 0x0000,
	0x0000, 0x3533, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3534,
	0x0000, 0x3535, 0x3536, 0x3537, 0x0000, 0x0000, 0x0000, 0x3538,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3539, 0x0000,
	0x0000, 0x0000, 0x353a, 0x0000, 0x0000, 0x353b, 0x353c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x353d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x353e, 0x0000, 0x353f, 0x0000, 0x0000, 0x3540,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3541, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3542, 0x0000, 0x3543, 0x3544,
};

static unsigned short const unicode_to_jisx0212_55[] = {
	/* 0x5500 - 0x55ff */
	0x3545, 0x3546, 0x0000, 0x0000, 0x0000, 0x3547, 0x0000, 0x0000,
	0x3548, 0x3549, 0x0000, 0x0000, 0x354a, 0x354b, 0x354c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x354d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x354e, 0x354f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3550, 0x0000, 0x0000, 0x3551, 0x3552, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3553, 0x3554, 0x3555, 0x0000, 0x0000,
	0x0000, 0x3556, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3557,
	0x0000, 0x3558, 0x3559, 0x0000, 0x0000, 0x355a, 0x0000, 0x0000,
	0x355b, 0x355c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x355d, 0x0000, 0x355e, 0x355f, 0x0000, 0x0000, 0x3560, 0x0000,
	0x3561, 0x3562, 0x0000, 0x0000, 0x3563, 0x0000, 0x3564, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3565,
	0x0000, 0x3566, 0x3567, 0x0000, 0x0000, 0x0000, 0x3568, 0x0000,
	0x3569, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x356a, 0x356b,
	0x0000, 0x356c, 0x356d, 0x356e, 0x356f, 0x0000, 0x0000, 0x3570,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3571, 0x3572, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3573, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3574, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3575,
	0x0000, 0x3576, 0x0000, 0x3577, 0x0000, 0x0000, 0x3578, 0x0000,
	0x0000, 0x3579, 0x0000, 0x357a, 0x357b, 0x0000, 0x357c, 0x0000,
	0x0000, 0x357d, 0x357e, 0x3621, 0x0000, 0x0000, 0x0000, 0x3622,
	0x3623, 0x0000, 0x0000, 0x3624, 0x0000, 0x0000, 0x3625, 0x0000,
	0x0000, 0x0000, 0x3626, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3627, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3628, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3629,
};

static unsigned short const unicode_to_jisx0212_56[] = {
	/* 0x5600 - 0x56ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x362a, 0x0000, 0x0000,
	0x362b, 0x0000, 0x362c, 0x0000, 0x0000, 0x362d, 0x362e, 0x362f,
	0x3630, 0x3631, 0x3632, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3633, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3634, 0x0000, 0x0000, 0x0000,
	0x3635, 0x0000, 0x0000, 0x3636, 0x0000, 0x3637, 0x0000, 0x3638,
	0x0000, 0x3639, 0x0000, 0x363a, 0x363b, 0x363c, 0x0000, 0x363d,
	0x363e, 0x363f, 0x0000, 0x3640, 0x3641, 0x0000, 0x3642, 0x0000,
	0x0000, 0x3643, 0x0000, 0x3644, 0x0000, 0x3645, 0x0000, 0x3646,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3647, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3648, 0x0000,
	0x3649, 0x364a, 0x364b, 0x364c, 0x0000, 0x0000, 0x364d, 0x0000,
	0x0000, 0x364e, 0x0000, 0x0000, 0x0000, 0x364f, 0x0000, 0x3650,
	0x0000, 0x3651, 0x3652, 0x0000, 0x0000, 0x3653, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3654, 0x3655, 0x0000, 0x0000,
	0x3656, 0x0000, 0x0000, 0x3657, 0x3658, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3659, 0x0000, 0x0000,
	0x0000, 0x365a, 0x365b, 0x0000, 0x0000, 0x365c, 0x365d, 0x365e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x365f, 0x3660,
	0x3661, 0x3662, 0x0000, 0x3663, 0x3664, 0x3665, 0x0000, 0x0000,
	0x0000, 0x3666, 0x0000, 0x3667, 0x0000, 0x0000, 0x0000, 0x3668,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3669, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x366a, 0x0000, 0x0000,
	0x0000, 0x366b, 0x366c, 0x366d, 0x3670, 0x3671, 0x0000, 0x366e,
	0x366f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3672, 0x0000, 0x0000, 0x3673, 0x3674, 0x0000, 0x3675,
	0x0000, 0x3676, 0x0000, 0x0000, 0x3677, 0x3678, 0x3679, 0x367a,
	0x367b, 0x0000, 0x0000, 0x367d, 0x0000, 0x367e, 0x0000, 0x0000,
	0x0000, 0x367c, 0x0000, 0x0000, 0x0000, 0x0000, 0x3721, 0x3722,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_57[] = {
	/* 0x5700 - 0x57ff */
	0x0000, 0x3723, 0x3724, 0x0000, 0x0000, 0x0000, 0x0000, 0x3725,
	0x0000, 0x0000, 0x3726, 0x0000, 0x3727, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3728, 0x0000, 0x0000, 0x0000, 0x3729, 0x0000, 0x0000,
	0x0000, 0x0000, 0x372a, 0x372b, 0x0000, 0x372c, 0x0000, 0x0000,
	0x372d, 0x0000, 0x372e, 0x372f, 0x3730, 0x3731, 0x0000, 0x0000,
	0x0000, 0x3732, 0x3733, 0x0000, 0x3734, 0x0000, 0x3735, 0x3736,
	0x0000, 0x0000, 0x0000, 0x3737, 0x3738, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3739, 0x373a, 0x373b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x373c, 0x373d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x373e, 0x373f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3740, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7436, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3741, 0x0000, 0x0000, 0x3742, 0x0000, 0x3743,
	0x3744, 0x0000, 0x0000, 0x3745, 0x0000, 0x3746, 0x3747, 0x3748,
	0x3749, 0x374a, 0x0000, 0x374b, 0x374c, 0x374d, 0x0000, 0x374e,
	0x0000, 0x374f, 0x3750, 0x3751, 0x3752, 0x0000, 0x3753, 0x0000,
	0x0000, 0x3754, 0x0000, 0x3755, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3756, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3757, 0x3760, 0x0000, 0x3758,
	0x0000, 0x3759, 0x375a, 0x0000, 0x375b, 0x375c, 0x375d, 0x375e,
	0x0000, 0x375f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3761,
	0x3762, 0x3763, 0x0000, 0x0000, 0x3764, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3765, 0x0000, 0x0000, 0x0000, 0x0000, 0x3766, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3767,
	0x3768, 0x0000, 0x0000, 0x0000, 0x3769, 0x0000, 0x0000, 0x376a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x376b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x376c, 0x376d, 0x0000,
	0x0000, 0x377e, 0x0000, 0x0000, 0x376e, 0x0000, 0x376f, 0x3770,
	0x0000, 0x3771, 0x0000, 0x0000, 0x0000, 0x3772, 0x0000, 0x0000,
	0x3773, 0x0000, 0x0000, 0x0000, 0x0000, 0x3774, 0x3775, 0x0000,
	0x3776, 0x0000, 0x0000, 0x0000, 0x0000, 0x3777, 0x3778, 0x3779,
};

static unsigned short const unicode_to_jisx0212_58[] = {
	/* 0x5800 - 0x58ff */
	0x0000, 0x0000, 0x0000, 0x377a, 0x377b, 0x0000, 0x0000, 0x0000,
	0x377c, 0x377d, 0x0000, 0x0000, 0x3821, 0x3822, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3823, 0x0000, 0x0000, 0x3824, 0x3825,
	0x3826, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3827, 0x3828,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3829, 0x0000, 0x0000,
	0x0000, 0x0000, 0x382a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x382b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x382c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x382d, 0x0000, 0x0000, 0x382e, 0x382f, 0x0000, 0x3830,
	0x3831, 0x0000, 0x0000, 0x0000, 0x0000, 0x3832, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3833,
	0x0000, 0x3834, 0x0000, 0x0000, 0x3835, 0x0000, 0x0000, 0x3836,
	0x3837, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3838, 0x0000, 0x0000, 0x0000, 0x3839, 0x0000, 0x0000, 0x383a,
	0x383b, 0x383c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x383d,
	0x383e, 0x383f, 0x3840, 0x0000, 0x3841, 0x3842, 0x0000, 0x3843,
	0x3844, 0x0000, 0x0000, 0x0000, 0x3845, 0x0000, 0x3846, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3847, 0x7439, 0x0000,
	0x3848, 0x3849, 0x384a, 0x0000, 0x0000, 0x0000, 0x384b, 0x0000,
	0x0000, 0x384c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x384d, 0x384e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3850, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3851, 0x0000, 0x384f, 0x0000, 0x0000, 0x0000,
	0x3852, 0x0000, 0x0000, 0x0000, 0x0000, 0x3853, 0x3854, 0x0000,
	0x3855, 0x0000, 0x3856, 0x0000, 0x3857, 0x0000, 0x3858, 0x0000,
	0x0000, 0x0000, 0x3859, 0x0000, 0x0000, 0x385a, 0x0000, 0x0000,
	0x0000, 0x385b, 0x385c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x385d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x385e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_59[] = {
	/* 0x5900 - 0x59ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x385f, 0x3860, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3861, 0x3862, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3863, 0x3864, 0x3865, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3867, 0x0000, 0x0000,
	0x0000, 0x3868, 0x0000, 0x3869, 0x386a, 0x0000, 0x0000, 0x0000,
	0x386b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x386c,
	0x386d, 0x0000, 0x0000, 0x386e, 0x0000, 0x386f, 0x3870, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3871,
	0x0000, 0x0000, 0x0000, 0x3872, 0x0000, 0x0000, 0x3873, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3874, 0x3875, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3876, 0x0000, 0x3877, 0x0000, 0x3878, 0x3879, 0x387a,
	0x0000, 0x387b, 0x0000, 0x387c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x387d, 0x0000, 0x387e, 0x0000, 0x3921,
	0x0000, 0x0000, 0x3922, 0x0000, 0x0000, 0x3923, 0x3924, 0x0000,
	0x0000, 0x3925, 0x0000, 0x3926, 0x3927, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3928, 0x3929, 0x0000, 0x392a, 0x0000,
	0x0000, 0x0000, 0x392b, 0x0000, 0x0000, 0x392c, 0x0000, 0x392d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x392e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x392f, 0x0000, 0x0000, 0x3930,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3931, 0x3932, 0x3933,
	0x3934, 0x0000, 0x0000, 0x3935, 0x0000, 0x0000, 0x0000, 0x3936,
	0x0000, 0x0000, 0x3937, 0x0000, 0x3938, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3939, 0x0000, 0x393a, 0x393b, 0x0000, 0x0000, 0x0000,
	0x393c, 0x0000, 0x393d, 0x0000, 0x0000, 0x393e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x393f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3940, 0x3941, 0x3942,
	0x0000, 0x0000, 0x0000, 0x3943, 0x3944, 0x0000, 0x0000, 0x3945,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3946, 0x3947,
	0x0000, 0x3948, 0x3949, 0x0000, 0x394a, 0x0000, 0x0000, 0x394b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_5a[] = {
	/* 0x5a00 - 0x5aff */
	0x394c, 0x0000, 0x0000, 0x0000, 0x394d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x394e, 0x394f, 0x3950, 0x0000,
	0x0000, 0x0000, 0x3951, 0x3952, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3953, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3954, 0x3955, 0x0000, 0x0000, 0x3956,
	0x3957, 0x0000, 0x3958, 0x0000, 0x0000, 0x3959, 0x0000, 0x0000,
	0x395a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x395b, 0x395c, 0x0000, 0x395d,
	0x395e, 0x0000, 0x0000, 0x0000, 0x395f, 0x0000, 0x0000, 0x0000,
	0x3960, 0x0000, 0x0000, 0x0000, 0x0000, 0x3961, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3962, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3963, 0x0000, 0x3964, 0x0000, 0x3965,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3966, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3967,
	0x0000, 0x0000, 0x3968, 0x3969, 0x0000, 0x0000, 0x396a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x396b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x396c, 0x0000, 0x0000, 0x396d, 0x0000, 0x0000, 0x396e, 0x0000,
	0x0000, 0x396f, 0x0000, 0x0000, 0x3970, 0x0000, 0x3971, 0x3972,
	0x3973, 0x0000, 0x3974, 0x0000, 0x0000, 0x0000, 0x0000, 0x3975,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3976, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3977, 0x3978, 0x3979, 0x0000, 0x397a, 0x0000, 0x0000,
	0x397b, 0x0000, 0x397c, 0x397d, 0x0000, 0x0000, 0x0000, 0x397e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3a21, 0x0000, 0x3a22, 0x0000,
	0x3a23, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a24,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a25, 0x0000, 0x3a26, 0x0000, 0x0000, 0x0000,
	0x3a27, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a28, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a29, 0x0000, 0x0000, 0x0000, 0x3a2a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a2b, 0x3a2c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a2d, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_5b[] = {
	/* 0x5b00 - 0x5bff */
	0x3a2e, 0x3a2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3a30, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a31,
	0x0000, 0x3a33, 0x0000, 0x3a34, 0x0000, 0x3a35, 0x0000, 0x0000,
	0x0000, 0x3a36, 0x0000, 0x0000, 0x0000, 0x3a37, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a38, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3a32, 0x0000, 0x0000, 0x0000,
	0x3a39, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3a3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3a3b, 0x3a3c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a3d, 0x0000, 0x0000, 0x0000, 0x3a3e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a3f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3a40, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a41, 0x3a42,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3a43, 0x3a44, 0x3a45, 0x3a46,
	0x0000, 0x3a47, 0x0000, 0x0000, 0x3a48, 0x0000, 0x3a49, 0x0000,
	0x0000, 0x0000, 0x3a4a, 0x0000, 0x0000, 0x0000, 0x3a4b, 0x0000,
	0x3a4c, 0x3a4d, 0x0000, 0x3a4e, 0x3a4f, 0x0000, 0x3a50, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3a51, 0x3a52, 0x0000, 0x0000, 0x3a53, 0x3a54, 0x0000, 0x3a55,
	0x0000, 0x3a56, 0x3a57, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a58,
	0x0000, 0x0000, 0x3a59, 0x0000, 0x3a5a, 0x0000, 0x0000, 0x0000,
	0x3a5b, 0x3a5c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a5d, 0x0000, 0x3a5e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a5f, 0x3a60,
	0x3a61, 0x3a62, 0x3a63, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3a64, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x743a, 0x0000, 0x0000, 0x3a65,
	0x0000, 0x3a66, 0x0000, 0x0000, 0x3a67, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a68, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_5c[] = {
	/* 0x5c00 - 0x5cff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3a69, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a6a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3a6b, 0x3a6c,
	0x0000, 0x0000, 0x0000, 0x3a6d, 0x0000, 0x0000, 0x3a6e, 0x0000,
	0x0000, 0x3a6f, 0x0000, 0x3a70, 0x3a71, 0x0000, 0x3a72, 0x0000,
	0x3a73, 0x0000, 0x3a74, 0x0000, 0x0000, 0x3a75, 0x3a76, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3a77, 0x3a78, 0x0000, 0x3a79, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3a7a, 0x3a7b, 0x0000, 0x0000, 0x0000, 0x3a7c,
	0x3a7d, 0x3a7e, 0x0000, 0x0000, 0x0000, 0x3b21, 0x0000, 0x0000,
	0x3b22, 0x0000, 0x0000, 0x0000, 0x3b23, 0x3b24, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b25, 0x3b26, 0x3b27, 0x3b28, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b29,
	0x3b2a, 0x0000, 0x3b2b, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b2c,
	0x0000, 0x0000, 0x3b2d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b2e, 0x0000, 0x3b2f,
	0x3b30, 0x0000, 0x3b31, 0x3b32, 0x0000, 0x0000, 0x3b33, 0x0000,
	0x0000, 0x0000, 0x3b34, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b35, 0x0000, 0x3b36, 0x3b37, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b38, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3b39, 0x0000, 0x3b3a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b3b, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b3d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b3c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b3e, 0x0000,
	0x0000, 0x3b3f, 0x3b40, 0x0000, 0x3b41, 0x743b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_5d[] = {
	/* 0x5d00 - 0x5dff */
	0x0000, 0x3b42, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b43, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b44, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b45, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b47, 0x3b48, 0x0000, 0x3b49, 0x3b4a,
	0x0000, 0x0000, 0x0000, 0x3b46, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3b4b, 0x0000, 0x0000, 0x3b4c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3b4d, 0x0000, 0x0000, 0x0000, 0x3b4e, 0x0000, 0x3b4f,
	0x0000, 0x0000, 0x3b50, 0x3b51, 0x0000, 0x0000, 0x3b52, 0x0000,
	0x3b53, 0x0000, 0x3b57, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3b55, 0x0000, 0x743c, 0x0000, 0x3b54, 0x0000, 0x0000,
	0x0000, 0x3b56, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b58,
	0x3b59, 0x3b5a, 0x3b5b, 0x0000, 0x3b5c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b5d, 0x0000, 0x0000, 0x3b5e, 0x0000, 0x0000,
	0x3b5f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3b60, 0x3b61, 0x0000, 0x0000, 0x0000, 0x3b62, 0x3b63,
	0x0000, 0x3b64, 0x0000, 0x3b65, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3b66, 0x0000, 0x3b67, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3b68, 0x3b69, 0x3b6a, 0x3b6b, 0x0000, 0x0000,
	0x0000, 0x3b6c, 0x0000, 0x3b6d, 0x0000, 0x0000, 0x0000, 0x3b6e,
	0x3b6f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3b70,
	0x0000, 0x0000, 0x0000, 0x3b71, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3b72, 0x0000, 0x6674, 0x0000, 0x3b73, 0x0000, 0x0000, 0x0000,
	0x3b74, 0x3b75, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3b76, 0x0000, 0x0000, 0x0000, 0x3b77,
	0x0000, 0x0000, 0x0000, 0x3b78, 0x0000, 0x0000, 0x3b7a, 0x0000,
	0x3b79, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3b7b, 0x3b7c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3b7d, 0x0000, 0x0000, 0x0000, 0x3b7e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c21, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c22, 0x3c23, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_5e[] = {
	/* 0x5e00 - 0x5eff */
	0x3c24, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c25,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c26, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3c27, 0x0000, 0x3c28, 0x3c29, 0x0000, 0x0000,
	0x3c2a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c2b,
	0x3c2c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c2d, 0x0000,
	0x0000, 0x0000, 0x3c2f, 0x0000, 0x0000, 0x3c30, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c31, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c34, 0x0000, 0x3c32, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c33, 0x3c35, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c36, 0x0000,
	0x3c37, 0x0000, 0x0000, 0x3c38, 0x3c39, 0x0000, 0x3c3a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c3b, 0x0000, 0x3c3c, 0x3c3d, 0x3c3e, 0x3c3f, 0x3c40, 0x0000,
	0x3c41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c42, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c43, 0x0000, 0x0000, 0x3c44, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3c45, 0x0000, 0x3c46, 0x3c47, 0x0000, 0x0000,
	0x3c48, 0x0000, 0x3c49, 0x0000, 0x3c4a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c4b, 0x0000, 0x3c4c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c4d, 0x3c4e, 0x3c4f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c50, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c52, 0x3c51, 0x0000, 0x3c53, 0x0000,
	0x0000, 0x3c54, 0x3c55, 0x0000, 0x3c56, 0x3c57, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3c58, 0x0000, 0x3c59, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c5a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c5b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_5f[] = {
	/* 0x5f00 - 0x5fff */
	0x0000, 0x0000, 0x3c5c, 0x0000, 0x0000, 0x0000, 0x3c5d, 0x3c5e,
	0x3c5f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c60, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3c61, 0x0000, 0x0000, 0x3c62, 0x3c63, 0x0000, 0x0000,
	0x0000, 0x3c64, 0x3c65, 0x3c66, 0x3c67, 0x0000, 0x0000, 0x0000,
	0x3c68, 0x0000, 0x0000, 0x3c69, 0x3c6a, 0x0000, 0x3c6b, 0x0000,
	0x3c6c, 0x0000, 0x0000, 0x0000, 0x3c6d, 0x0000, 0x3c6e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3c6f, 0x0000, 0x3c70, 0x0000, 0x3c71,
	0x3c72, 0x0000, 0x0000, 0x0000, 0x3c73, 0x3c74, 0x0000, 0x3c75,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3c76, 0x0000, 0x0000,
	0x3c77, 0x0000, 0x0000, 0x0000, 0x3c78, 0x0000, 0x0000, 0x0000,
	0x3c79, 0x0000, 0x0000, 0x3c7a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3c7b, 0x0000, 0x0000, 0x3c7c, 0x3c7d, 0x0000, 0x0000, 0x3c7e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d21,
	0x0000, 0x0000, 0x3d22, 0x0000, 0x3d23, 0x3d24, 0x0000, 0x0000,
	0x3d25, 0x0000, 0x3d26, 0x0000, 0x0000, 0x3d27, 0x3d28, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3d29, 0x0000, 0x0000, 0x0000, 0x3d2a, 0x0000, 0x3d2b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d2c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d2d, 0x3d2e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3d2f, 0x0000, 0x3d32, 0x0000, 0x0000, 0x3d30,
	0x0000, 0x0000, 0x0000, 0x3d31, 0x3d33, 0x0000, 0x0000, 0x3d34,
	0x3d35, 0x3d36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x743e,
	0x3d37, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d38, 0x0000, 0x0000, 0x3d39,
	0x3d3a, 0x3d3b, 0x0000, 0x3d3c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3d3d, 0x3d3e, 0x3d3f, 0x3d40, 0x3d41, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d42, 0x0000,
	0x0000, 0x3d43, 0x3d44, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3d45, 0x3d46, 0x3d47, 0x0000, 0x3d48, 0x3d49, 0x3d4a, 0x3d4b,
	0x0000, 0x0000, 0x3d4c, 0x3d4d, 0x0000, 0x0000, 0x3d4e, 0x0000,
	0x0000, 0x0000, 0x3d4f, 0x0000, 0x3d50, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_60[] = {
	/* 0x6000 - 0x60ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d51,
	0x0000, 0x0000, 0x3d52, 0x0000, 0x0000, 0x3d53, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3d54, 0x3d55, 0x0000, 0x0000, 0x3d56,
	0x3d57, 0x0000, 0x3d58, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d59,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3d5a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d5b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3d5c, 0x0000, 0x3d5d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3d5e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d5f,
	0x3d60, 0x3d61, 0x0000, 0x0000, 0x3d62, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3d63, 0x0000, 0x0000, 0x3d64, 0x0000, 0x3d65, 0x3d66,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d67, 0x0000, 0x0000,
	0x0000, 0x3d68, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d69,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3d6a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d6b, 0x3d6c,
	0x0000, 0x0000, 0x3d6d, 0x0000, 0x0000, 0x743f, 0x3d6e, 0x0000,
	0x3d6f, 0x0000, 0x3d70, 0x0000, 0x0000, 0x0000, 0x3d71, 0x0000,
	0x0000, 0x3d72, 0x0000, 0x3d73, 0x0000, 0x3d74, 0x0000, 0x0000,
	0x3d75, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d76, 0x3d77, 0x0000,
	0x0000, 0x0000, 0x3d78, 0x0000, 0x3d79, 0x3d7a, 0x0000, 0x0000,
	0x3d7b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3d7c, 0x3d7d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3d7e,
	0x0000, 0x0000, 0x0000, 0x3e21, 0x0000, 0x0000, 0x3e22, 0x0000,
	0x0000, 0x0000, 0x3e23, 0x0000, 0x3e24, 0x0000, 0x0000, 0x0000,
	0x3e25, 0x3e26, 0x3e27, 0x3e28, 0x0000, 0x0000, 0x3e29, 0x3e2a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3e2b, 0x3e2c, 0x0000, 0x0000,
	0x0000, 0x3e2d, 0x0000, 0x3e2e, 0x0000, 0x3e2f, 0x3e30, 0x0000,
	0x0000, 0x0000, 0x3e31, 0x0000, 0x0000, 0x3e32, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3e33, 0x0000, 0x0000, 0x3e34, 0x0000, 0x0000,
	0x3e35, 0x0000, 0x0000, 0x0000, 0x3e36, 0x3e37, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_61[] = {
	/* 0x6100 - 0x61ff */
	0x0000, 0x0000, 0x3e38, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e39,
	0x0000, 0x0000, 0x3e3a, 0x0000, 0x3e3b, 0x0000, 0x0000, 0x0000,
	0x3e3c, 0x3e3d, 0x3e3e, 0x3e3f, 0x3e40, 0x0000, 0x3e41, 0x3e42,
	0x0000, 0x3e43, 0x0000, 0x0000, 0x3e44, 0x0000, 0x3e45, 0x0000,
	0x7440, 0x0000, 0x3e46, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3e47, 0x3e48, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3e49, 0x3e4a, 0x0000, 0x0000, 0x0000, 0x3e4b, 0x3e4c, 0x3e4d,
	0x0000, 0x3e4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3e4f, 0x0000, 0x0000, 0x0000, 0x3e50, 0x3e51, 0x0000,
	0x0000, 0x3e52, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e53, 0x0000,
	0x3e54, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3e55, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3e56, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3e57, 0x0000, 0x0000, 0x3e58, 0x3e59, 0x0000, 0x0000, 0x3e5a,
	0x3e5b, 0x3e5c, 0x0000, 0x3e5d, 0x3e5e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3e5f, 0x0000, 0x3e60, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3e61, 0x3e62, 0x0000, 0x0000, 0x0000, 0x3e63,
	0x3e64, 0x0000, 0x0000, 0x0000, 0x3e65, 0x3e66, 0x0000, 0x3e67,
	0x3e68, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e69, 0x0000, 0x0000,
	0x3e6a, 0x0000, 0x3e6b, 0x0000, 0x0000, 0x3e6c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x3e6d, 0x3e6e, 0x0000, 0x0000, 0x3e6f, 0x0000, 0x0000, 0x0000,
	0x3e70, 0x3e71, 0x3e72, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e73, 0x3e74,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3e75, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3e76, 0x3e77, 0x3e78, 0x3e79,
	0x0000, 0x3e7a, 0x3e7b, 0x0000, 0x0000, 0x3e7e, 0x0000, 0x3e7c,
	0x0000, 0x3e7d, 0x0000, 0x0000, 0x3f21, 0x3f22, 0x0000, 0x3f23,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_62[] = {
	/* 0x6200 - 0x62ff */
	0x0000, 0x3f24, 0x0000, 0x3f25, 0x3f26, 0x0000, 0x0000, 0x3f27,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3f28, 0x0000, 0x3f29, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f2a, 0x0000, 0x0000, 0x0000,
	0x3f2b, 0x0000, 0x3f2c, 0x3f2d, 0x0000, 0x0000, 0x0000, 0x3f2e,
	0x0000, 0x3f2f, 0x0000, 0x3f30, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3f31, 0x0000, 0x0000, 0x0000, 0x3f32, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3f33, 0x3f34, 0x3f35, 0x0000, 0x3f36, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f37, 0x0000, 0x0000, 0x0000,
	0x3f38, 0x3f39, 0x3f3a, 0x0000, 0x3f3b, 0x0000, 0x3f3c, 0x0000,
	0x0000, 0x0000, 0x3f3d, 0x0000, 0x3f3e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f3f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f40, 0x0000, 0x3f41,
	0x0000, 0x0000, 0x0000, 0x3f42, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3f43, 0x0000, 0x0000, 0x3f44, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f45, 0x3f46, 0x3f47,
	0x3f48, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f49, 0x0000,
	0x3f4a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x3f4b, 0x0000, 0x0000, 0x3f4c, 0x3f4d,
	0x0000, 0x0000, 0x3f4e, 0x0000, 0x0000, 0x0000, 0x3f4f, 0x3f50,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f51, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f52, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f53, 0x3f54, 0x0000,
	0x0000, 0x0000, 0x3f55, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3f56, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3f57, 0x0000, 0x3f58, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f59, 0x3f5a, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_63[] = {
	/* 0x6300 - 0x63ff */
	0x0000, 0x0000, 0x0000, 0x3f5b, 0x3f5c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x3f5d, 0x3f5e, 0x0000, 0x3f5f, 0x0000, 0x0000,
	0x3f60, 0x0000, 0x0000, 0x3f61, 0x0000, 0x0000, 0x3f62, 0x0000,
	0x3f63, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3f64, 0x3f65, 0x0000, 0x0000, 0x3f66, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f67, 0x3f68, 0x0000,
	0x0000, 0x3f69, 0x0000, 0x0000, 0x3f6a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x3f6b, 0x3f6c, 0x3f6d, 0x3f6e, 0x0000, 0x3f6f, 0x0000,
	0x0000, 0x0000, 0x3f70, 0x3f71, 0x0000, 0x0000, 0x3f72, 0x0000,
	0x0000, 0x0000, 0x3f73, 0x3f74, 0x3f75, 0x0000, 0x0000, 0x0000,
	0x3f76, 0x0000, 0x0000, 0x3f77, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3f78, 0x3f79, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x3f7a, 0x3f7b, 0x0000, 0x0000,
	0x0000, 0x3f7c, 0x0000, 0x0000, 0x3f7d, 0x3f7e, 0x0000, 0x0000,
	0x4021, 0x0000, 0x0000, 0x0000, 0x4022, 0x4023, 0x0000, 0x4024,
	0x0000, 0x0000, 0x4025, 0x0000, 0x4026, 0x0000, 0x0000, 0x4027,
	0x0000, 0x0000, 0x4028, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4029, 0x0000, 0x0000, 0x0000, 0x402a, 0x402b, 0x0000, 0x0000,
	0x0000, 0x402c, 0x402d, 0x0000, 0x0000, 0x0000, 0x402e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x402f, 0x0000, 0x4030, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4031, 0x4032, 0x4033,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4034, 0x0000, 0x0000,
	0x0000, 0x4035, 0x0000, 0x0000, 0x0000, 0x4036, 0x0000, 0x0000,
	0x4037, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4038, 0x0000,
	0x0000, 0x4039, 0x0000, 0x403a, 0x403b, 0x403c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x403d, 0x0000, 0x0000, 0x0000,
	0x403e, 0x0000, 0x0000, 0x0000, 0x0000, 0x403f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4040, 0x0000, 0x4041, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4042, 0x4043, 0x0000, 0x4044, 0x0000, 0x0000,
	0x4045, 0x4046, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_64[] = {
	/* 0x6400 - 0x64ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4047, 0x4048, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4049, 0x0000, 0x404a, 0x0000, 0x404b, 0x0000, 0x0000, 0x0000,
	0x404c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x404d, 0x0000,
	0x404e, 0x0000, 0x404f, 0x0000, 0x4050, 0x4051, 0x0000, 0x0000,
	0x0000, 0x4052, 0x4053, 0x0000, 0x0000, 0x0000, 0x0000, 0x4054,
	0x4055, 0x0000, 0x0000, 0x0000, 0x0000, 0x4056, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4057, 0x0000, 0x4058,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4059, 0x0000, 0x0000, 0x0000, 0x405a,
	0x0000, 0x405b, 0x405c, 0x405d, 0x405e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x405f, 0x4060, 0x4061, 0x4062, 0x0000, 0x4063,
	0x4064, 0x4065, 0x0000, 0x4066, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4067, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4068, 0x4069, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x406a, 0x0000, 0x406b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x406c, 0x0000, 0x406d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x406e,
	0x406f, 0x4070, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4071, 0x4072, 0x0000, 0x4073, 0x0000, 0x4074, 0x0000, 0x4075,
	0x0000, 0x4076, 0x0000, 0x4077, 0x0000, 0x0000, 0x4078, 0x0000,
	0x4079, 0x0000, 0x0000, 0x0000, 0x407a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x407b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x407c, 0x407d, 0x407e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4121, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4122, 0x4123, 0x4124, 0x4125, 0x0000, 0x4126, 0x0000,
	0x4127, 0x4128, 0x0000, 0x0000, 0x0000, 0x4129, 0x0000, 0x412a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x412b, 0x412c, 0x0000, 0x0000,
	0x0000, 0x412d, 0x412e, 0x0000, 0x0000, 0x412f, 0x0000, 0x0000,
	0x4130, 0x0000, 0x0000, 0x0000, 0x0000, 0x4131, 0x0000, 0x4132,
	0x0000, 0x0000, 0x0000, 0x4133, 0x0000, 0x0000, 0x0000, 0x4134,
};

static unsigned short const unicode_to_jisx0212_65[] = {
	/* 0x6500 - 0x65ff */
	0x0000, 0x4135, 0x0000, 0x0000, 0x4136, 0x0000, 0x0000, 0x0000,
	0x4137, 0x4138, 0x4139, 0x0000, 0x0000, 0x0000, 0x0000, 0x413a,
	0x0000, 0x0000, 0x0000, 0x413b, 0x413c, 0x0000, 0x413d, 0x0000,
	0x0000, 0x413e, 0x0000, 0x413f, 0x0000, 0x0000, 0x4140, 0x4141,
	0x0000, 0x0000, 0x4142, 0x0000, 0x0000, 0x0000, 0x4143, 0x0000,
	0x0000, 0x4144, 0x0000, 0x0000, 0x0000, 0x0000, 0x4145, 0x0000,
	0x0000, 0x4146, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4147, 0x0000, 0x4148, 0x4149, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x414a, 0x0000, 0x0000, 0x0000, 0x414b,
	0x0000, 0x414c, 0x0000, 0x0000, 0x0000, 0x0000, 0x7441, 0x0000,
	0x414d, 0x0000, 0x414e, 0x0000, 0x414f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4150,
	0x4151, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4152,
	0x0000, 0x0000, 0x0000, 0x4153, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4154, 0x0000, 0x0000, 0x4155, 0x0000, 0x0000,
	0x0000, 0x4156, 0x0000, 0x0000, 0x0000, 0x4157, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4158, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4159, 0x0000, 0x0000, 0x415a, 0x0000, 0x0000,
	0x415b, 0x0000, 0x0000, 0x0000, 0x0000, 0x415c, 0x0000, 0x0000,
	0x415d, 0x0000, 0x0000, 0x415e, 0x0000, 0x0000, 0x415f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4160, 0x0000,
	0x0000, 0x0000, 0x4161, 0x4162, 0x4163, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4164,
	0x0000, 0x0000, 0x4165, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4166, 0x4167, 0x0000, 0x0000, 0x0000, 0x0000, 0x4168, 0x0000,
	0x4169, 0x0000, 0x0000, 0x0000, 0x416a, 0x0000, 0x416b, 0x0000,
	0x416c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x416d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x416e, 0x0000, 0x416f, 0x0000, 0x4170, 0x4171, 0x0000, 0x0000,
	0x0000, 0x4172, 0x0000, 0x0000, 0x0000, 0x0000, 0x4173, 0x4174,
};

static unsigned short const unicode_to_jisx0212_66[] = {
	/* 0x6600 - 0x66ff */
	0x4175, 0x0000, 0x0000, 0x0000, 0x4176, 0x0000, 0x0000, 0x0000,
	0x4177, 0x4178, 0x0000, 0x0000, 0x0000, 0x4179, 0x0000, 0x0000,
	0x0000, 0x417a, 0x417b, 0x0000, 0x0000, 0x417c, 0x417d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x417e, 0x4221, 0x0000,
	0x0000, 0x4222, 0x4223, 0x4224, 0x4225, 0x0000, 0x4226, 0x0000,
	0x0000, 0x4227, 0x4228, 0x4229, 0x422a, 0x0000, 0x422b, 0x0000,
	0x422c, 0x422d, 0x0000, 0x422e, 0x0000, 0x0000, 0x0000, 0x4230,
	0x0000, 0x422f, 0x0000, 0x7442, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4231, 0x0000, 0x0000, 0x0000, 0x0000, 0x4232, 0x4233, 0x0000,
	0x0000, 0x0000, 0x4234, 0x0000, 0x4235, 0x0000, 0x4237, 0x0000,
	0x0000, 0x4236, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4238,
	0x4239, 0x423a, 0x0000, 0x423b, 0x423c, 0x0000, 0x0000, 0x0000,
	0x423d, 0x423e, 0x0000, 0x0000, 0x0000, 0x7443, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4240, 0x4241, 0x4242, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4244, 0x0000, 0x4245, 0x0000, 0x4247,
	0x4248, 0x4249, 0x0000, 0x424a, 0x424c, 0x0000, 0x4243, 0x4246,
	0x424b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x424d, 0x424e, 0x424f, 0x0000, 0x0000,
	0x4250, 0x0000, 0x4251, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4252, 0x4253, 0x4254, 0x4255, 0x0000, 0x0000, 0x4256,
	0x4257, 0x0000, 0x0000, 0x0000, 0x4258, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4259, 0x0000, 0x0000,
	0x0000, 0x425a, 0x425b, 0x0000, 0x0000, 0x425c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x425d, 0x0000, 0x0000, 0x0000, 0x425e,
	0x425f, 0x0000, 0x4260, 0x4261, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4262, 0x0000, 0x0000, 0x0000, 0x4263, 0x0000, 0x4264, 0x4265,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4266, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4267, 0x0000, 0x0000, 0x0000, 0x4268,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4269, 0x0000, 0x0000, 0x426a, 0x426b, 0x0000, 0x426c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x426d, 0x423f, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_67[] = {
	/* 0x6700 - 0x67ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x426e, 0x0000, 0x426f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4270, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4271, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4272, 0x0000, 0x0000, 0x4273, 0x0000, 0x0000, 0x0000,
	0x4274, 0x0000, 0x4275, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4276, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4277, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4278, 0x0000, 0x4279,
	0x427a, 0x0000, 0x0000, 0x0000, 0x427b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x427c, 0x427d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x427e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4321, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4322, 0x0000, 0x4323, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4324, 0x0000, 0x4325, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4326, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4327, 0x0000, 0x0000, 0x4328, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4329, 0x432a,
	0x0000, 0x432b, 0x0000, 0x432c, 0x0000, 0x0000, 0x432d, 0x0000,
	0x432e, 0x432f, 0x0000, 0x4330, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4331, 0x4332, 0x4333, 0x0000, 0x0000, 0x4334, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4335, 0x4336, 0x4337, 0x0000, 0x0000,
	0x4339, 0x0000, 0x433a, 0x433b, 0x0000, 0x433c, 0x0000, 0x0000,
	0x433d, 0x433e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x433f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4340,
	0x0000, 0x4341, 0x0000, 0x0000, 0x4342, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4343, 0x0000, 0x0000, 0x0000, 0x0000, 0x4344, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4345, 0x0000, 0x4346, 0x0000, 0x0000, 0x0000, 0x4347, 0x4348,
	0x0000, 0x4338, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_68[] = {
	/* 0x6800 - 0x68ff */
	0x0000, 0x7446, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x434a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x434b, 0x0000, 0x0000, 0x0000, 0x434c, 0x0000, 0x434d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x434f,
	0x434e, 0x0000, 0x0000, 0x0000, 0x4350, 0x4351, 0x0000, 0x4352,
	0x4353, 0x4354, 0x0000, 0x4355, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4356, 0x0000, 0x0000, 0x0000, 0x4357,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4358, 0x4359, 0x0000, 0x0000,
	0x0000, 0x0000, 0x435a, 0x0000, 0x435b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4349, 0x0000, 0x0000, 0x435c, 0x0000, 0x435d,
	0x435e, 0x0000, 0x0000, 0x435f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4360, 0x0000, 0x0000, 0x4361, 0x4362,
	0x4363, 0x4364, 0x4365, 0x0000, 0x0000, 0x4366, 0x0000, 0x0000,
	0x0000, 0x4367, 0x4368, 0x4369, 0x436a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x436b, 0x0000, 0x436c, 0x0000, 0x436d, 0x0000,
	0x436e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x436f, 0x0000,
	0x4370, 0x0000, 0x4371, 0x0000, 0x4372, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4373, 0x0000, 0x4374, 0x0000, 0x4375, 0x0000, 0x0000,
	0x0000, 0x4376, 0x4377, 0x0000, 0x0000, 0x0000, 0x4378, 0x0000,
	0x0000, 0x0000, 0x4379, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x437a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x437b, 0x0000, 0x0000,
	0x437c, 0x0000, 0x0000, 0x0000, 0x437d, 0x0000, 0x0000, 0x437e,
	0x4421, 0x4422, 0x0000, 0x4423, 0x0000, 0x0000, 0x4424, 0x0000,
	0x0000, 0x4425, 0x0000, 0x0000, 0x4426, 0x4427, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4428, 0x0000, 0x0000,
	0x4429, 0x0000, 0x442a, 0x442b, 0x442c, 0x442d, 0x0000, 0x0000,
	0x442e, 0x442f, 0x0000, 0x0000, 0x0000, 0x4430, 0x4431, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4432, 0x4433, 0x4434, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_69[] = {
	/* 0x6900 - 0x69ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4435, 0x0000,
	0x0000, 0x4436, 0x4437, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4438, 0x4439, 0x0000, 0x443a, 0x0000, 0x0000, 0x443b, 0x443c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x443d, 0x0000, 0x443e, 0x0000, 0x443f, 0x0000, 0x0000,
	0x4440, 0x0000, 0x0000, 0x4441, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4442, 0x0000, 0x0000, 0x4443, 0x0000, 0x0000,
	0x0000, 0x4444, 0x0000, 0x0000, 0x0000, 0x0000, 0x4445, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4446,
	0x0000, 0x0000, 0x0000, 0x4447, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4448, 0x4449, 0x444a, 0x444b, 0x0000,
	0x444c, 0x444d, 0x0000, 0x0000, 0x444e, 0x0000, 0x0000, 0x0000,
	0x444f, 0x4450, 0x4451, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4452, 0x4453, 0x0000, 0x0000, 0x0000, 0x4454,
	0x4455, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4456, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4457, 0x0000, 0x0000, 0x0000, 0x4458, 0x0000,
	0x4459, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x445a, 0x0000, 0x0000, 0x0000, 0x445b, 0x445c, 0x0000,
	0x445d, 0x0000, 0x0000, 0x445e, 0x0000, 0x445f, 0x0000, 0x4460,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4461,
	0x4462, 0x0000, 0x4463, 0x0000, 0x4464, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4465, 0x0000, 0x0000,
	0x4466, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4467, 0x0000, 0x0000, 0x0000, 0x0000, 0x4468, 0x4469,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x446a, 0x0000, 0x0000, 0x446b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x446c, 0x446d,
	0x0000, 0x446e, 0x0000, 0x446f, 0x0000, 0x4470, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4471, 0x0000,
};

static unsigned short const unicode_to_jisx0212_6a[] = {
	/* 0x6a00 - 0x6aff */
	0x4472, 0x4473, 0x0000, 0x4474, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4475,
	0x0000, 0x4476, 0x0000, 0x0000, 0x0000, 0x4477, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4478, 0x0000, 0x0000, 0x4479, 0x0000, 0x0000,
	0x447a, 0x0000, 0x0000, 0x0000, 0x447b, 0x0000, 0x0000, 0x0000,
	0x447c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x447d, 0x0000, 0x447e, 0x0000, 0x4521, 0x0000, 0x0000, 0x4522,
	0x0000, 0x0000, 0x0000, 0x4523, 0x0000, 0x0000, 0x4524, 0x4525,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4526, 0x4527, 0x0000,
	0x0000, 0x4528, 0x4529, 0x0000, 0x0000, 0x0000, 0x452a, 0x0000,
	0x452b, 0x452c, 0x452d, 0x0000, 0x0000, 0x452e, 0x452f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4530, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4531, 0x0000, 0x0000, 0x4532,
	0x0000, 0x0000, 0x4533, 0x7449, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4534, 0x0000, 0x4535, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4536, 0x0000,
	0x0000, 0x4537, 0x0000, 0x4538, 0x0000, 0x0000, 0x4539, 0x453a,
	0x0000, 0x453b, 0x0000, 0x453c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x453d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x453e, 0x0000, 0x453f, 0x4540, 0x4541,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4542, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4543, 0x0000, 0x0000, 0x0000, 0x4544,
	0x4545, 0x4546, 0x0000, 0x0000, 0x4547, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4548, 0x4549, 0x454a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x454b, 0x0000,
	0x454d, 0x454c, 0x0000, 0x0000, 0x454e, 0x0000, 0x0000, 0x0000,
	0x454f, 0x0000, 0x0000, 0x0000, 0x4550, 0x4551, 0x4552, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4553, 0x4554, 0x0000, 0x0000,
	0x0000, 0x0000, 0x744a, 0x0000, 0x4555, 0x0000, 0x0000, 0x4556,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4557, 0x0000, 0x0000, 0x0000,
	0x4558, 0x4559, 0x455a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x455b, 0x455c, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_6b[] = {
	/* 0x6b00 - 0x6bff */
	0x0000, 0x0000, 0x455d, 0x455e, 0x0000, 0x0000, 0x455f, 0x4560,
	0x0000, 0x4561, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4562,
	0x4563, 0x4564, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4565,
	0x0000, 0x0000, 0x0000, 0x4566, 0x0000, 0x0000, 0x4567, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4568, 0x0000, 0x0000, 0x0000,
	0x4569, 0x0000, 0x0000, 0x456a, 0x456b, 0x0000, 0x0000, 0x456c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x456d, 0x456e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x456f, 0x0000, 0x0000, 0x0000, 0x4570,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4571, 0x0000,
	0x0000, 0x0000, 0x4572, 0x0000, 0x0000, 0x4573, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4574, 0x0000, 0x0000, 0x0000, 0x4575, 0x0000,
	0x4576, 0x0000, 0x0000, 0x0000, 0x0000, 0x4577, 0x0000, 0x0000,
	0x4578, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4579,
	0x0000, 0x0000, 0x0000, 0x457a, 0x0000, 0x0000, 0x457b, 0x0000,
	0x457c, 0x0000, 0x0000, 0x0000, 0x0000, 0x457d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x457e, 0x4621, 0x0000,
	0x0000, 0x0000, 0x4622, 0x0000, 0x0000, 0x4623, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4624,
	0x0000, 0x0000, 0x0000, 0x4625, 0x0000, 0x0000, 0x0000, 0x4626,
	0x4627, 0x0000, 0x4628, 0x4629, 0x0000, 0x0000, 0x0000, 0x0000,
	0x462a, 0x462b, 0x0000, 0x0000, 0x462c, 0x462d, 0x462e, 0x0000,
	0x462f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4630, 0x4631, 0x0000, 0x0000, 0x0000, 0x4632, 0x4633, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4634, 0x4635, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4636, 0x0000, 0x0000, 0x4637, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4638, 0x0000,
	0x0000, 0x0000, 0x4639, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x463a, 0x0000, 0x463b, 0x0000, 0x0000, 0x463c, 0x463d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x463e, 0x0000,
	0x0000, 0x463f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4640,
	0x0000, 0x4641, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4642,
};

static unsigned short const unicode_to_jisx0212_6c[] = {
	/* 0x6c00 - 0x6cff */
	0x0000, 0x0000, 0x4643, 0x0000, 0x4644, 0x4645, 0x0000, 0x0000,
	0x0000, 0x4646, 0x0000, 0x0000, 0x0000, 0x4647, 0x4648, 0x0000,
	0x4649, 0x0000, 0x464a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x464b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x464c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x464d, 0x464e,
	0x464f, 0x0000, 0x0000, 0x0000, 0x4650, 0x0000, 0x4651, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4652, 0x0000, 0x4653, 0x4654, 0x0000,
	0x0000, 0x0000, 0x4655, 0x4656, 0x0000, 0x0000, 0x0000, 0x4657,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4658, 0x4659, 0x0000, 0x465a, 0x0000, 0x465b,
	0x0000, 0x0000, 0x465c, 0x0000, 0x465d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x465e, 0x0000, 0x465f, 0x4660, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4736,
	0x0000, 0x0000, 0x0000, 0x4661, 0x0000, 0x4662, 0x0000, 0x4663,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4664, 0x0000, 0x4665, 0x0000,
	0x4666, 0x4667, 0x0000, 0x4668, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4669, 0x466a, 0x466b,
	0x0000, 0x466c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x466d, 0x466e, 0x0000, 0x466f,
	0x4670, 0x0000, 0x0000, 0x0000, 0x4671, 0x0000, 0x0000, 0x4672,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4673, 0x0000, 0x4674, 0x0000, 0x4675, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4676, 0x0000, 0x0000, 0x0000, 0x4677, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4678, 0x0000, 0x4679,
	0x467a, 0x467b, 0x467c, 0x0000, 0x467d, 0x0000, 0x467e, 0x0000,
	0x0000, 0x0000, 0x4721, 0x0000, 0x4722, 0x0000, 0x0000, 0x0000,
	0x4723, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4724,
	0x0000, 0x4725, 0x0000, 0x4726, 0x4727, 0x0000, 0x4728, 0x0000,
	0x0000, 0x0000, 0x4729, 0x0000, 0x472a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_6d[] = {
	/* 0x6d00 - 0x6dff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x472b, 0x0000, 0x0000, 0x472c,
	0x0000, 0x0000, 0x472d, 0x0000, 0x0000, 0x0000, 0x472e, 0x472f,
	0x0000, 0x4730, 0x0000, 0x4731, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4732, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4733, 0x4734,
	0x4735, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4737, 0x4738,
	0x0000, 0x4739, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x473a, 0x0000, 0x0000, 0x473b, 0x0000, 0x0000, 0x473c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x473d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x473e, 0x473f,
	0x0000, 0x4740, 0x0000, 0x0000, 0x0000, 0x4741, 0x0000, 0x4742,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4743,
	0x4744, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4745, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4746, 0x0000, 0x0000, 0x0000, 0x0000, 0x4747,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4748, 0x4749, 0x0000, 0x474a, 0x0000, 0x474b, 0x474c,
	0x474d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x474e, 0x0000, 0x474f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4750, 0x0000, 0x0000, 0x4751,
	0x0000, 0x4752, 0x0000, 0x0000, 0x0000, 0x4753, 0x0000, 0x4754,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4755, 0x0000, 0x0000, 0x0000,
	0x4756, 0x0000, 0x4757, 0x0000, 0x0000, 0x0000, 0x4758, 0x4759,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x475a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x475b, 0x0000, 0x475c, 0x0000, 0x475d,
	0x475e, 0x0000, 0x475f, 0x0000, 0x0000, 0x4760, 0x0000, 0x0000,
	0x0000, 0x4761, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4762,
	0x4763, 0x0000, 0x744c, 0x0000, 0x4764, 0x0000, 0x4765, 0x0000,
	0x744b, 0x0000, 0x0000, 0x0000, 0x4766, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_6e[] = {
	/* 0x6e00 - 0x6eff */
	0x4767, 0x0000, 0x0000, 0x0000, 0x4768, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4769, 0x0000,
	0x0000, 0x0000, 0x476a, 0x0000, 0x0000, 0x0000, 0x0000, 0x476b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x476c, 0x0000, 0x0000, 0x0000, 0x476d, 0x0000,
	0x0000, 0x476e, 0x0000, 0x476f, 0x4770, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4771, 0x4772, 0x0000, 0x0000,
	0x4773, 0x4774, 0x0000, 0x4775, 0x0000, 0x0000, 0x0000, 0x4776,
	0x0000, 0x4777, 0x4778, 0x4779, 0x477a, 0x0000, 0x0000, 0x477b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x477c, 0x477d, 0x477e, 0x0000,
	0x0000, 0x0000, 0x4821, 0x4822, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4823, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4824, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4825, 0x0000, 0x4826, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4827, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4828, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4829, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x482a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x482b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x482c, 0x482d, 0x0000,
	0x0000, 0x482e, 0x0000, 0x482f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4830, 0x0000, 0x0000, 0x0000, 0x4831,
	0x4832, 0x4833, 0x0000, 0x4834, 0x0000, 0x0000, 0x0000, 0x4835,
	0x4836, 0x0000, 0x4837, 0x0000, 0x0000, 0x4838, 0x4839, 0x483a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x483b, 0x0000, 0x483c, 0x483d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x483e, 0x0000, 0x483f, 0x0000, 0x4840, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_6f[] = {
	/* 0x6f00 - 0x6fff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x4841, 0x0000, 0x0000, 0x0000,
	0x4842, 0x0000, 0x4843, 0x0000, 0x4844, 0x4845, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4846, 0x0000,
	0x4847, 0x0000, 0x4848, 0x4849, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x484a, 0x0000,
	0x0000, 0x484b, 0x484c, 0x0000, 0x0000, 0x4853, 0x0000, 0x484d,
	0x484e, 0x0000, 0x0000, 0x484f, 0x0000, 0x0000, 0x4850, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4851, 0x4852, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4854,
	0x0000, 0x4855, 0x4856, 0x4857, 0x0000, 0x0000, 0x0000, 0x4858,
	0x0000, 0x4859, 0x485a, 0x0000, 0x0000, 0x485b, 0x485c, 0x0000,
	0x0000, 0x485d, 0x485e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x485f, 0x0000, 0x0000, 0x0000, 0x4860, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4861, 0x4862, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4863, 0x0000, 0x0000, 0x0000, 0x4864,
	0x4865, 0x0000, 0x0000, 0x4866, 0x4867, 0x4868, 0x0000, 0x0000,
	0x4869, 0x0000, 0x486a, 0x486b, 0x486c, 0x0000, 0x486d, 0x0000,
	0x0000, 0x0000, 0x486e, 0x0000, 0x0000, 0x0000, 0x0000, 0x486f,
	0x4870, 0x0000, 0x0000, 0x0000, 0x0000, 0x4871, 0x4872, 0x4873,
	0x4874, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4875, 0x4876,
	0x4877, 0x0000, 0x0000, 0x0000, 0x0000, 0x4878, 0x4879, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x487a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x487b, 0x0000, 0x487c,
	0x487d, 0x0000, 0x487e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4921, 0x0000, 0x0000, 0x0000, 0x4922, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4923, 0x4924, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4925, 0x0000, 0x0000, 0x0000, 0x0000, 0x4926, 0x0000, 0x0000,
	0x0000, 0x4927, 0x0000, 0x0000, 0x4928, 0x4929, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_70[] = {
	/* 0x7000 - 0x70ff */
	0x492a, 0x0000, 0x0000, 0x0000, 0x0000, 0x492b, 0x492c, 0x492d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x492e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x492f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4930, 0x0000, 0x0000, 0x4931, 0x0000, 0x0000, 0x0000, 0x0000,
	0x744d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4932,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4933, 0x0000, 0x0000, 0x4934,
	0x0000, 0x4935, 0x0000, 0x0000, 0x4936, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4937, 0x4938, 0x0000, 0x0000, 0x0000,
	0x4939, 0x493a, 0x493b, 0x493c, 0x0000, 0x0000, 0x4941, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x493d, 0x493e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x493f, 0x4940, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4942, 0x4943, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4944, 0x0000, 0x4945, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4946, 0x4947, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4948, 0x0000,
	0x0000, 0x4949, 0x0000, 0x0000, 0x0000, 0x494a, 0x494b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x494c, 0x494d, 0x494e, 0x494f,
	0x4950, 0x0000, 0x0000, 0x4951, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4952, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4953, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4954, 0x4955, 0x0000, 0x0000, 0x4956, 0x0000, 0x0000, 0x4957,
	0x0000, 0x0000, 0x0000, 0x742e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4958, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4959, 0x0000, 0x495a, 0x495b, 0x495c, 0x495d, 0x0000,
	0x495e, 0x0000, 0x0000, 0x0000, 0x495f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4960, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4961, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_71[] = {
	/* 0x7100 - 0x71ff */
	0x0000, 0x0000, 0x0000, 0x4962, 0x4963, 0x4964, 0x4965, 0x4966,
	0x0000, 0x0000, 0x0000, 0x4967, 0x4968, 0x0000, 0x0000, 0x4969,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x496a, 0x0000,
	0x496b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x496c, 0x0000, 0x496d, 0x0000, 0x496e,
	0x496f, 0x4970, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4971, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4972, 0x0000, 0x0000, 0x0000, 0x4973, 0x4974, 0x4975,
	0x0000, 0x0000, 0x4976, 0x4977, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4978, 0x0000, 0x4979, 0x0000, 0x0000, 0x0000, 0x0000, 0x497a,
	0x0000, 0x0000, 0x497b, 0x0000, 0x497c, 0x0000, 0x497d, 0x0000,
	0x497e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a21, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a22, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a23, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a24, 0x0000, 0x4a25,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4a26, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4a27, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4a28, 0x4a29, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a2a, 0x0000, 0x4a2b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a2c,
	0x4a2d, 0x0000, 0x4a2e, 0x4a2f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4a30, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a31,
	0x4a32, 0x4a33, 0x0000, 0x0000, 0x4a34, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a35, 0x4a36, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a37, 0x0000, 0x0000, 0x4a38, 0x0000,
	0x0000, 0x4a39, 0x4a3a, 0x0000, 0x4a3b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a3c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a3d, 0x0000,
};

static unsigned short const unicode_to_jisx0212_72[] = {
	/* 0x7200 - 0x72ff */
	0x4a3e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a3f,
	0x4a40, 0x4a41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a42, 0x0000, 0x0000, 0x0000, 0x4a43,
	0x0000, 0x0000, 0x4a44, 0x0000, 0x0000, 0x4a45, 0x0000, 0x4a46,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4a47, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a48, 0x0000, 0x0000, 0x0000, 0x4a49,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4a4a, 0x0000, 0x0000, 0x0000,
	0x4a4b, 0x4a4c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a4d, 0x4a4e, 0x4a4f, 0x0000, 0x4a50, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a51, 0x4a52,
	0x4a53, 0x0000, 0x0000, 0x4a54, 0x0000, 0x4a55, 0x4a56, 0x0000,
	0x0000, 0x0000, 0x4a57, 0x0000, 0x4a58, 0x0000, 0x4a59, 0x0000,
	0x4a5a, 0x0000, 0x0000, 0x4a5b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a5c, 0x0000, 0x0000, 0x4a5d, 0x0000, 0x0000, 0x4a5e, 0x4a5f,
	0x0000, 0x4a60, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a61,
	0x4a62, 0x0000, 0x0000, 0x4a63, 0x4a64, 0x0000, 0x0000, 0x4a65,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4a66, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4a67, 0x0000, 0x0000, 0x0000, 0x4a68, 0x4a69, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a6a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a6b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4a6c, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a6d, 0x4a6e, 0x0000,
	0x0000, 0x4a6f, 0x0000, 0x0000, 0x4a70, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a71, 0x0000,
	0x0000, 0x4a72, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a73,
	0x0000, 0x4a74, 0x0000, 0x0000, 0x4a75, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a76, 0x4a77, 0x0000,
	0x4a78, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a79,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4a7a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4a7b, 0x4a7c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4a7d, 0x4a7e, 0x0000, 0x0000, 0x4b21, 0x0000,
};

static unsigned short const unicode_to_jisx0212_73[] = {
	/* 0x7300 - 0x73ff */
	0x0000, 0x0000, 0x4b22, 0x0000, 0x4b23, 0x4b24, 0x0000, 0x4b25,
	0x0000, 0x0000, 0x0000, 0x4b26, 0x0000, 0x4b27, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4b28, 0x4b29, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4b2a, 0x4b2b, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b2c, 0x0000,
	0x0000, 0x0000, 0x4b2d, 0x0000, 0x4b2e, 0x0000, 0x0000, 0x4b2f,
	0x4b30, 0x0000, 0x0000, 0x0000, 0x4b31, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4b32, 0x4b33, 0x0000, 0x0000, 0x4b34, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4b35, 0x4b36, 0x0000, 0x4b37, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4b38, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b39, 0x0000, 0x0000,
	0x4b3a, 0x0000, 0x4b3b, 0x0000, 0x0000, 0x0000, 0x4b3c, 0x0000,
	0x4b3d, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b3e, 0x4b3f, 0x4b40,
	0x4b41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b42, 0x4b43,
	0x0000, 0x4b44, 0x0000, 0x4b45, 0x4b46, 0x0000, 0x4b47, 0x4b48,
	0x0000, 0x4b49, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b4a,
	0x0000, 0x4b4b, 0x0000, 0x0000, 0x4b4c, 0x0000, 0x0000, 0x0000,
	0x4b4d, 0x4b4e, 0x0000, 0x4b4f, 0x0000, 0x4b50, 0x4b51, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b52, 0x0000,
	0x4b53, 0x0000, 0x0000, 0x4b54, 0x0000, 0x4b55, 0x0000, 0x4b56,
	0x4b57, 0x0000, 0x0000, 0x0000, 0x4b58, 0x0000, 0x4b59, 0x4b5a,
	0x4b5b, 0x0000, 0x4b5c, 0x0000, 0x0000, 0x4b5d, 0x4b5e, 0x0000,
	0x0000, 0x0000, 0x4b5f, 0x4b60, 0x0000, 0x4b61, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b62, 0x0000, 0x4b63,
	0x0000, 0x4b64, 0x0000, 0x0000, 0x4b65, 0x4b66, 0x0000, 0x4b67,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4b68, 0x4b69, 0x0000,
	0x0000, 0x4b6a, 0x0000, 0x4b6b, 0x4b6c, 0x0000, 0x0000, 0x4b6d,
	0x0000, 0x0000, 0x4b6e, 0x4b6f, 0x0000, 0x0000, 0x4b70, 0x0000,
	0x0000, 0x4b71, 0x0000, 0x0000, 0x0000, 0x4b72, 0x0000, 0x0000,
	0x0000, 0x4b73, 0x0000, 0x4b74, 0x0000, 0x0000, 0x4b75, 0x4b76,
	0x0000, 0x4b77, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4b78, 0x4b79, 0x0000, 0x4b7a,
	0x0000, 0x4b7b, 0x4b7c, 0x4b7d, 0x0000, 0x4b7e, 0x0000, 0x4c21,
};

static unsigned short const unicode_to_jisx0212_74[] = {
	/* 0x7400 - 0x74ff */
	0x4c22, 0x4c23, 0x0000, 0x0000, 0x4c24, 0x0000, 0x0000, 0x4c25,
	0x0000, 0x0000, 0x4c26, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4c27, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c28, 0x4c29, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4c2a, 0x0000, 0x4c2b, 0x0000,
	0x4c2c, 0x4c2d, 0x4c2e, 0x4c2f, 0x4c30, 0x4c31, 0x4c32, 0x4c33,
	0x4c34, 0x4c35, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4c36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c37, 0x0000, 0x0000, 0x4c38, 0x4c39, 0x0000, 0x4c3a, 0x4c3b,
	0x0000, 0x0000, 0x0000, 0x4c3c, 0x0000, 0x4c3d, 0x0000, 0x0000,
	0x0000, 0x4c3e, 0x4c3f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c40,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c41, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c42, 0x0000, 0x0000, 0x0000, 0x4c43, 0x4c44,
	0x4c45, 0x0000, 0x0000, 0x4c46, 0x0000, 0x4c47, 0x4c48, 0x0000,
	0x0000, 0x4c49, 0x4c4a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c4b, 0x4c4c, 0x0000, 0x0000, 0x0000, 0x4c4d, 0x4c4e, 0x4c4f,
	0x0000, 0x4c50, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c51,
	0x4c52, 0x4c53, 0x4c54, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c55, 0x4c56, 0x4c57, 0x0000, 0x4c58, 0x0000, 0x0000, 0x4c59,
	0x4c5a, 0x4c5b, 0x0000, 0x4c5c, 0x0000, 0x0000, 0x4c5d, 0x0000,
	0x4c5e, 0x4c5f, 0x4c60, 0x4c61, 0x0000, 0x0000, 0x4c62, 0x4c63,
	0x0000, 0x4c64, 0x4c65, 0x0000, 0x0000, 0x4c66, 0x0000, 0x0000,
	0x0000, 0x4c67, 0x0000, 0x4c68, 0x0000, 0x0000, 0x0000, 0x4c69,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c6a, 0x4c6b, 0x0000, 0x0000, 0x4c6c, 0x0000, 0x0000, 0x0000,
	0x4c6d, 0x0000, 0x0000, 0x4c6e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4c6f, 0x0000, 0x4c70, 0x4c71, 0x0000, 0x0000, 0x4c72, 0x4c73,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4c74, 0x0000, 0x0000, 0x0000,
	0x4c75, 0x0000, 0x4c76, 0x4c77, 0x0000, 0x0000, 0x0000, 0x4c78,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4c79, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4c7a, 0x4c7b, 0x4c7c, 0x0000, 0x0000, 0x4c7d,
};

static unsigned short const unicode_to_jisx0212_75[] = {
	/* 0x7500 - 0x75ff */
	0x0000, 0x7450, 0x0000, 0x0000, 0x0000, 0x0000, 0x4c7e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d21, 0x0000, 0x0000, 0x0000, 0x4d22, 0x4d23,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4d24, 0x4d25, 0x0000, 0x0000, 0x4d26, 0x0000, 0x0000, 0x4d27,
	0x0000, 0x4d28, 0x4d29, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d2a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d2b, 0x0000,
	0x0000, 0x4d2c, 0x0000, 0x0000, 0x0000, 0x4d2d, 0x4d2e, 0x4d2f,
	0x4d30, 0x0000, 0x0000, 0x4d31, 0x0000, 0x0000, 0x0000, 0x4d32,
	0x4d33, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d34, 0x0000,
	0x4d35, 0x0000, 0x4d36, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d37,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d38, 0x4d39,
	0x0000, 0x4d3a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d3b,
	0x0000, 0x4d3c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4d3d, 0x4d3e, 0x4d3f, 0x4d40, 0x4d41, 0x4d42, 0x0000,
	0x0000, 0x4d43, 0x0000, 0x0000, 0x0000, 0x4d44, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4d45, 0x0000, 0x4d46, 0x4d47, 0x0000, 0x4d48, 0x0000, 0x0000,
	0x0000, 0x4d49, 0x0000, 0x0000, 0x4d4a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d4b, 0x0000, 0x4d4c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d4d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d4f,
	0x4d50, 0x4d51, 0x0000, 0x0000, 0x4d52, 0x0000, 0x4d53, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d54, 0x0000, 0x4d55, 0x4d56,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4d57,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d58, 0x0000, 0x0000, 0x4d59,
	0x4d5a, 0x4d5b, 0x0000, 0x0000, 0x4d5c, 0x0000, 0x0000, 0x4d5d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4d5e, 0x0000, 0x4d5f, 0x4d60,
	0x0000, 0x4d61, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4d62, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_76[] = {
	/* 0x7600 - 0x76ff */
	0x4d63, 0x0000, 0x4d64, 0x4d65, 0x4d66, 0x0000, 0x0000, 0x4d67,
	0x4d68, 0x0000, 0x4d69, 0x0000, 0x4d6a, 0x0000, 0x0000, 0x4d6b,
	0x0000, 0x0000, 0x4d6c, 0x4d6d, 0x0000, 0x4d6e, 0x4d6f, 0x0000,
	0x0000, 0x4d70, 0x0000, 0x4d71, 0x4d72, 0x4d73, 0x4d74, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4d75, 0x0000, 0x4d76, 0x4d77, 0x0000,
	0x0000, 0x4d78, 0x0000, 0x0000, 0x0000, 0x4d79, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4d7a, 0x4d7b, 0x0000, 0x4d7c, 0x0000, 0x0000,
	0x4d7d, 0x4d7e, 0x4e21, 0x0000, 0x4e22, 0x0000, 0x0000, 0x0000,
	0x4e24, 0x4e25, 0x0000, 0x4e26, 0x4e27, 0x4e28, 0x0000, 0x0000,
	0x0000, 0x4e29, 0x4e23, 0x4e2a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e2b, 0x0000, 0x0000,
	0x0000, 0x4e2c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e2d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4e2e, 0x4e2f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e30, 0x4e31, 0x4e32,
	0x0000, 0x4e33, 0x0000, 0x0000, 0x4e34, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4e35, 0x7451, 0x0000, 0x0000, 0x4e36, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4e37, 0x4e38, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e39, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4e3a, 0x4e3b, 0x4e3c, 0x7452, 0x4e3d,
	0x4e3e, 0x0000, 0x4e3f, 0x4e40, 0x4e41, 0x4e42, 0x4e43, 0x4e44,
	0x4e45, 0x0000, 0x4e46, 0x0000, 0x0000, 0x4e47, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e48, 0x0000, 0x0000,
	0x0000, 0x4e49, 0x0000, 0x0000, 0x0000, 0x4e4a, 0x0000, 0x0000,
	0x0000, 0x4e4b, 0x0000, 0x4e4c, 0x4e4d, 0x0000, 0x4e4e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4e4f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4e50, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4e51, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e52, 0x0000,
	0x4e53, 0x0000, 0x0000, 0x0000, 0x4e54, 0x0000, 0x0000, 0x0000,
	0x4e55, 0x4e56, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e57, 0x0000,
	0x0000, 0x4e58, 0x0000, 0x0000, 0x4e59, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_77[] = {
	/* 0x7700 - 0x77ff */
	0x4e5a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e5b, 0x0000,
	0x0000, 0x0000, 0x4e5c, 0x0000, 0x0000, 0x0000, 0x4e5d, 0x0000,
	0x0000, 0x0000, 0x4e5e, 0x0000, 0x4e5f, 0x4e60, 0x0000, 0x4e61,
	0x0000, 0x4e62, 0x4e63, 0x0000, 0x4e64, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4e65, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x4e66, 0x0000, 0x0000, 0x0000, 0x0000, 0x4e67, 0x4e68, 0x4e69,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4e6a, 0x4e6b, 0x4e6c, 0x0000,
	0x0000, 0x4e6d, 0x0000, 0x0000, 0x0000, 0x4e6e, 0x4e6f, 0x0000,
	0x0000, 0x0000, 0x4e70, 0x0000, 0x0000, 0x4e71, 0x4e72, 0x0000,
	0x0000, 0x0000, 0x4e73, 0x0000, 0x0000, 0x4e74, 0x4e75, 0x4e76,
	0x0000, 0x0000, 0x4e77, 0x0000, 0x0000, 0x0000, 0x4e78, 0x4e79,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4e7a, 0x0000, 0x4e7b, 0x4e7c,
	0x4e7d, 0x0000, 0x4e7e, 0x0000, 0x4f21, 0x0000, 0x0000, 0x4f22,
	0x0000, 0x0000, 0x4f23, 0x0000, 0x4f24, 0x0000, 0x0000, 0x0000,
	0x4f25, 0x0000, 0x4f26, 0x4f27, 0x4f28, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4f29, 0x0000, 0x0000, 0x4f2a, 0x0000, 0x0000,
	0x4f2b, 0x0000, 0x0000, 0x0000, 0x4f2c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4f2d, 0x4f2e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4f2f, 0x4f30, 0x4f31, 0x0000,
	0x0000, 0x0000, 0x4f32, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f33,
	0x0000, 0x0000, 0x4f34, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f35,
	0x0000, 0x0000, 0x4f36, 0x0000, 0x0000, 0x0000, 0x4f37, 0x4f38,
	0x0000, 0x4f39, 0x0000, 0x0000, 0x0000, 0x4f3a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f3b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4f3c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4f3d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x4f3e, 0x4f3f, 0x0000, 0x0000, 0x4f40, 0x0000, 0x0000,
	0x0000, 0x4f41, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f42, 0x4f43,
	0x4f44, 0x0000, 0x0000, 0x0000, 0x4f45, 0x0000, 0x4f46, 0x0000,
	0x0000, 0x0000, 0x4f47, 0x0000, 0x4f48, 0x0000, 0x0000, 0x0000,
	0x4f49, 0x4f4a, 0x0000, 0x0000, 0x4f4b, 0x0000, 0x0000, 0x0000,
	0x4f4c, 0x0000, 0x0000, 0x4f4d, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_78[] = {
	/* 0x7800 - 0x78ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f4e, 0x4f4f, 0x0000,
	0x0000, 0x4f50, 0x0000, 0x0000, 0x0000, 0x4f51, 0x4f52, 0x0000,
	0x0000, 0x4f53, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f54, 0x0000, 0x0000,
	0x0000, 0x4f55, 0x4f56, 0x4f57, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f58, 0x4f59, 0x0000,
	0x4f5a, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f5b, 0x0000, 0x4f5c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x4f5d, 0x4f5e, 0x0000, 0x0000, 0x4f5f,
	0x4f60, 0x0000, 0x0000, 0x0000, 0x4f61, 0x0000, 0x4f62, 0x0000,
	0x0000, 0x0000, 0x4f63, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4f64, 0x0000, 0x4f65, 0x0000,
	0x4f66, 0x4f67, 0x0000, 0x4f68, 0x4f69, 0x0000, 0x0000, 0x0000,
	0x4f6a, 0x0000, 0x4f6b, 0x0000, 0x0000, 0x0000, 0x4f6c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4f6d, 0x0000, 0x0000, 0x0000, 0x4f6e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x4f6f, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f70,
	0x0000, 0x0000, 0x0000, 0x0000, 0x4f71, 0x0000, 0x0000, 0x0000,
	0x4f72, 0x0000, 0x0000, 0x0000, 0x0000, 0x4f74, 0x4f75, 0x4f76,
	0x0000, 0x4f73, 0x0000, 0x0000, 0x4f77, 0x0000, 0x0000, 0x0000,
	0x4f78, 0x0000, 0x0000, 0x0000, 0x4f79, 0x4f7a, 0x0000, 0x0000,
	0x4f7b, 0x4f7c, 0x4f7d, 0x4f7e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5021, 0x0000, 0x5022, 0x0000, 0x5023,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5024,
	0x5025, 0x5026, 0x0000, 0x0000, 0x5027, 0x0000, 0x5028, 0x0000,
	0x0000, 0x0000, 0x5029, 0x502a, 0x0000, 0x502b, 0x502c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x502e, 0x0000, 0x0000, 0x0000, 0x502f,
	0x5030, 0x5031, 0x0000, 0x0000, 0x502d, 0x0000, 0x5032, 0x0000,
	0x0000, 0x0000, 0x5033, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5034, 0x5035, 0x0000, 0x0000, 0x5037, 0x5038,
	0x0000, 0x0000, 0x5039, 0x503a, 0x0000, 0x0000, 0x0000, 0x503b,
};

static unsigned short const unicode_to_jisx0212_79[] = {
	/* 0x7900 - 0x79ff */
	0x5036, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x503c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x503d, 0x0000, 0x0000, 0x0000,
	0x503e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x503f, 0x0000, 0x5040, 0x0000, 0x5041, 0x5042,
	0x5043, 0x0000, 0x0000, 0x0000, 0x0000, 0x5044, 0x0000, 0x5045,
	0x0000, 0x5046, 0x0000, 0x0000, 0x0000, 0x5047, 0x0000, 0x0000,
	0x7454, 0x5048, 0x0000, 0x0000, 0x5049, 0x504a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x504b, 0x0000, 0x504c, 0x0000, 0x504d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x504e, 0x504f, 0x5050, 0x0000,
	0x0000, 0x0000, 0x5051, 0x5052, 0x0000, 0x0000, 0x0000, 0x5053,
	0x0000, 0x5054, 0x0000, 0x0000, 0x5055, 0x0000, 0x0000, 0x0000,
	0x5056, 0x0000, 0x0000, 0x5057, 0x5058, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5059,
	0x0000, 0x505a, 0x0000, 0x505b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x505c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x505d, 0x0000, 0x505e, 0x505f, 0x0000, 0x5060, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5061, 0x5062, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5063, 0x0000, 0x5064, 0x5065, 0x5066, 0x5067, 0x0000,
	0x5068, 0x0000, 0x0000, 0x5069, 0x506a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x506b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x506c, 0x506d, 0x0000, 0x506e, 0x0000, 0x0000, 0x0000, 0x506f,
	0x0000, 0x5070, 0x0000, 0x0000, 0x5071, 0x0000, 0x0000, 0x0000,
	0x5072, 0x0000, 0x0000, 0x5073, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5074, 0x0000, 0x5075, 0x0000, 0x0000, 0x5076,
	0x5077, 0x0000, 0x5078, 0x0000, 0x0000, 0x0000, 0x0000, 0x5079,
	0x0000, 0x0000, 0x0000, 0x0000, 0x507a, 0x0000, 0x507b, 0x0000,
	0x0000, 0x0000, 0x507c, 0x0000, 0x0000, 0x507d, 0x507e, 0x0000,
	0x5121, 0x0000, 0x5122, 0x0000, 0x0000, 0x5123, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5124, 0x5125, 0x0000, 0x5126, 0x0000, 0x0000,
	0x0000, 0x5127, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5128, 0x0000, 0x0000, 0x0000, 0x5129, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_7a[] = {
	/* 0x7a00 - 0x7aff */
	0x0000, 0x0000, 0x512a, 0x512b, 0x0000, 0x0000, 0x0000, 0x512c,
	0x0000, 0x512d, 0x512e, 0x0000, 0x512f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5130, 0x0000, 0x0000, 0x0000, 0x5131, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5132, 0x0000, 0x0000, 0x5133, 0x0000,
	0x0000, 0x5134, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5135,
	0x0000, 0x0000, 0x0000, 0x5136, 0x0000, 0x5137, 0x0000, 0x5138,
	0x5139, 0x0000, 0x0000, 0x0000, 0x513a, 0x513b, 0x0000, 0x0000,
	0x513c, 0x513d, 0x513e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x513f, 0x5140, 0x0000, 0x5141,
	0x5142, 0x0000, 0x0000, 0x0000, 0x5143, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5144, 0x5145, 0x0000,
	0x0000, 0x5146, 0x0000, 0x0000, 0x5147, 0x5148, 0x0000, 0x5149,
	0x514a, 0x0000, 0x0000, 0x0000, 0x0000, 0x514b, 0x0000, 0x514c,
	0x0000, 0x0000, 0x514d, 0x0000, 0x0000, 0x514e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x514f, 0x0000, 0x0000,
	0x5150, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5151, 0x0000,
	0x5152, 0x0000, 0x5153, 0x0000, 0x0000, 0x5154, 0x5155, 0x0000,
	0x0000, 0x0000, 0x5156, 0x5157, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5158, 0x5159, 0x0000, 0x0000, 0x515a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x515b, 0x0000,
	0x515c, 0x0000, 0x0000, 0x515d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x515e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x515f, 0x0000, 0x5160, 0x0000, 0x0000,
	0x0000, 0x5161, 0x0000, 0x5162, 0x5163, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5164, 0x0000,
	0x0000, 0x5165, 0x0000, 0x0000, 0x5166, 0x0000, 0x5167, 0x0000,
	0x0000, 0x5168, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5169, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7459,
	0x516a, 0x516b, 0x0000, 0x516c, 0x516d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x516e, 0x0000, 0x0000, 0x516f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5170, 0x0000, 0x5171, 0x5172, 0x0000,
};

static unsigned short const unicode_to_jisx0212_7b[] = {
	/* 0x7b00 - 0x7bff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5173,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5174, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5175,
	0x0000, 0x0000, 0x0000, 0x5176, 0x0000, 0x0000, 0x0000, 0x5177,
	0x0000, 0x5178, 0x5179, 0x517a, 0x0000, 0x517b, 0x517c, 0x517d,
	0x517e, 0x5221, 0x0000, 0x0000, 0x5222, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5223, 0x0000, 0x5224,
	0x5225, 0x5226, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5227,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5228, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5229, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x522a, 0x0000, 0x0000, 0x0000, 0x522b, 0x0000, 0x522c, 0x0000,
	0x0000, 0x522d, 0x522e, 0x0000, 0x0000, 0x522f, 0x0000, 0x5230,
	0x0000, 0x0000, 0x5231, 0x5232, 0x0000, 0x0000, 0x0000, 0x5233,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5234, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5235, 0x0000, 0x0000, 0x0000, 0x0000, 0x5236, 0x0000,
	0x5237, 0x5238, 0x0000, 0x0000, 0x0000, 0x0000, 0x5239, 0x0000,
	0x0000, 0x0000, 0x0000, 0x523a, 0x0000, 0x0000, 0x523b, 0x0000,
	0x523c, 0x0000, 0x0000, 0x0000, 0x0000, 0x523d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x523e, 0x0000, 0x0000, 0x523f,
	0x5240, 0x0000, 0x5241, 0x0000, 0x0000, 0x5242, 0x5243, 0x0000,
	0x0000, 0x0000, 0x5244, 0x5245, 0x5246, 0x5247, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5248, 0x0000, 0x0000, 0x5249, 0x0000, 0x0000,
	0x524a, 0x0000, 0x524b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x524c, 0x0000, 0x524d, 0x524e,
	0x0000, 0x524f, 0x5250, 0x5251, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5252, 0x0000, 0x5253, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5254, 0x0000, 0x5255, 0x5256, 0x0000, 0x0000,
	0x5257, 0x5258, 0x5259, 0x0000, 0x525a, 0x0000, 0x525b, 0x0000,
};

static unsigned short const unicode_to_jisx0212_7c[] = {
	/* 0x7c00 - 0x7cff */
	0x0000, 0x525c, 0x525d, 0x525e, 0x525f, 0x0000, 0x5260, 0x0000,
	0x0000, 0x5261, 0x0000, 0x5262, 0x5263, 0x0000, 0x5264, 0x5265,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5266, 0x0000, 0x5267, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5268, 0x0000, 0x0000, 0x0000, 0x0000, 0x5269, 0x526a, 0x0000,
	0x526b, 0x0000, 0x0000, 0x0000, 0x526c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x526d, 0x0000, 0x526e, 0x526f, 0x0000, 0x5270, 0x0000,
	0x0000, 0x5271, 0x5272, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5273, 0x0000,
	0x0000, 0x0000, 0x5274, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5276, 0x5277, 0x5278, 0x0000, 0x5275, 0x0000, 0x0000,
	0x0000, 0x5279, 0x527a, 0x527b, 0x527c, 0x527d, 0x527e, 0x0000,
	0x0000, 0x5321, 0x0000, 0x5322, 0x0000, 0x0000, 0x0000, 0x5323,
	0x0000, 0x5324, 0x0000, 0x0000, 0x0000, 0x5325, 0x5326, 0x0000,
	0x5327, 0x0000, 0x5328, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5329, 0x0000, 0x0000, 0x532a, 0x532b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x532c, 0x532d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x532e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x532f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5330, 0x0000,
	0x5331, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5332, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5333, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5334, 0x5335,
	0x0000, 0x0000, 0x5336, 0x5337, 0x5338, 0x0000, 0x0000, 0x5339,
	0x0000, 0x0000, 0x0000, 0x0000, 0x533a, 0x0000, 0x0000, 0x533b,
	0x533c, 0x533d, 0x0000, 0x0000, 0x0000, 0x533e, 0x0000, 0x533f,
	0x0000, 0x0000, 0x0000, 0x5340, 0x5341, 0x5342, 0x0000, 0x5343,
	0x0000, 0x5344, 0x5345, 0x0000, 0x0000, 0x5346, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5347, 0x0000,
	0x0000, 0x5348, 0x0000, 0x5349, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x534a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_7d[] = {
	/* 0x7d00 - 0x7dff */
	0x0000, 0x0000, 0x0000, 0x534b, 0x0000, 0x0000, 0x0000, 0x534c,
	0x534d, 0x534e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x534f,
	0x0000, 0x5350, 0x5351, 0x5352, 0x0000, 0x0000, 0x5353, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5354, 0x5355, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5356, 0x0000, 0x0000, 0x5357, 0x0000,
	0x0000, 0x0000, 0x5358, 0x0000, 0x0000, 0x5359, 0x0000, 0x0000,
	0x0000, 0x535a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x535b, 0x535c, 0x535d, 0x0000,
	0x535e, 0x535f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5360,
	0x5361, 0x0000, 0x0000, 0x0000, 0x0000, 0x5362, 0x0000, 0x0000,
	0x0000, 0x5363, 0x0000, 0x5364, 0x0000, 0x0000, 0x0000, 0x5365,
	0x0000, 0x5366, 0x5367, 0x0000, 0x5368, 0x5369, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x536a, 0x0000, 0x536b,
	0x0000, 0x0000, 0x536c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x536d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x536e, 0x0000, 0x536f, 0x5370, 0x0000, 0x0000, 0x0000, 0x5371,
	0x0000, 0x5372, 0x5373, 0x5374, 0x0000, 0x5375, 0x5376, 0x0000,
	0x5377, 0x0000, 0x0000, 0x5378, 0x5379, 0x537a, 0x0000, 0x0000,
	0x0000, 0x537b, 0x0000, 0x0000, 0x0000, 0x0000, 0x537c, 0x537d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x537e, 0x5421, 0x0000,
	0x745c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5422, 0x5423,
	0x0000, 0x0000, 0x5424, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5425, 0x0000, 0x0000, 0x5426, 0x5427,
	0x0000, 0x5428, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5429, 0x542a, 0x542b, 0x542c, 0x542d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x542e, 0x542f, 0x5430, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x745d, 0x5431,
	0x0000, 0x5432, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5434, 0x0000, 0x0000, 0x5435, 0x5436, 0x0000,
	0x0000, 0x0000, 0x5437, 0x5438, 0x0000, 0x5439, 0x0000, 0x0000,
	0x0000, 0x543a, 0x0000, 0x0000, 0x0000, 0x543b, 0x543c, 0x0000,
	0x0000, 0x543d, 0x543e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_7e[] = {
	/* 0x7e00 - 0x7eff */
	0x5433, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x543f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5440, 0x5441, 0x0000, 0x0000, 0x0000, 0x5442, 0x0000, 0x5443,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5444, 0x5445, 0x0000, 0x0000,
	0x5446, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5447,
	0x5448, 0x0000, 0x0000, 0x0000, 0x5449, 0x544a, 0x0000, 0x544b,
	0x0000, 0x0000, 0x0000, 0x544c, 0x0000, 0x0000, 0x544d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x544e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x544f, 0x5450, 0x0000, 0x5451,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5452, 0x0000,
	0x5453, 0x0000, 0x5454, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5455, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5456,
	0x0000, 0x5457, 0x5458, 0x0000, 0x0000, 0x5459, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x545a, 0x0000, 0x0000, 0x545b, 0x545c,
	0x0000, 0x0000, 0x0000, 0x545d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x545e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x545f, 0x0000,
	0x0000, 0x5460, 0x0000, 0x0000, 0x0000, 0x0000, 0x5461, 0x5462,
	0x0000, 0x0000, 0x5463, 0x0000, 0x0000, 0x5464, 0x0000, 0x0000,
	0x0000, 0x5465, 0x0000, 0x0000, 0x0000, 0x5466, 0x0000, 0x0000,
	0x5467, 0x0000, 0x5468, 0x0000, 0x0000, 0x5469, 0x546a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_7f[] = {
	/* 0x7f00 - 0x7fff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x546c, 0x546b, 0x546d, 0x546e, 0x546f,
	0x0000, 0x0000, 0x0000, 0x5470, 0x5471, 0x0000, 0x0000, 0x5472,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5473,
	0x0000, 0x0000, 0x5474, 0x5475, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5476, 0x5477, 0x5478, 0x0000, 0x0000,
	0x0000, 0x5479, 0x0000, 0x547a, 0x547b, 0x547c, 0x547d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x547e, 0x0000, 0x0000,
	0x0000, 0x5521, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5522, 0x5523, 0x5524,
	0x5525, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5526, 0x0000, 0x5527, 0x0000, 0x5528,
	0x5529, 0x552a, 0x0000, 0x0000, 0x0000, 0x0000, 0x552b, 0x552c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x552d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x552e, 0x552f, 0x0000, 0x0000, 0x0000, 0x5530, 0x0000,
	0x0000, 0x0000, 0x5531, 0x0000, 0x0000, 0x5532, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5533, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5534, 0x0000, 0x0000, 0x5535,
	0x5536, 0x0000, 0x0000, 0x5537, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5538, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5539, 0x553a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x553b, 0x0000, 0x0000, 0x0000, 0x553c,
	0x0000, 0x0000, 0x0000, 0x553d, 0x0000, 0x553e, 0x0000, 0x0000,
	0x553f, 0x0000, 0x0000, 0x0000, 0x5540, 0x0000, 0x5541, 0x5542,
	0x0000, 0x0000, 0x5543, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5544, 0x0000, 0x0000, 0x5545, 0x5546, 0x5547,
};

static unsigned short const unicode_to_jisx0212_80[] = {
	/* 0x8000 - 0x80ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5548,
	0x5549, 0x0000, 0x554a, 0x0000, 0x0000, 0x554b, 0x554c, 0x554d,
	0x0000, 0x554e, 0x0000, 0x554f, 0x5550, 0x0000, 0x5551, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5552, 0x5553, 0x5554,
	0x5555, 0x0000, 0x0000, 0x0000, 0x5556, 0x0000, 0x5557, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5558, 0x0000, 0x5559, 0x0000,
	0x555a, 0x0000, 0x0000, 0x0000, 0x555b, 0x555c, 0x0000, 0x555d,
	0x0000, 0x555e, 0x555f, 0x0000, 0x5560, 0x0000, 0x5561, 0x0000,
	0x5562, 0x0000, 0x0000, 0x0000, 0x5563, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5564, 0x0000, 0x0000, 0x0000, 0x5565, 0x0000, 0x5566, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5567, 0x0000, 0x0000,
	0x0000, 0x5568, 0x0000, 0x0000, 0x0000, 0x5569, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x556a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x556b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x556c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x556d, 0x0000, 0x556e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x556f, 0x5570,
	0x0000, 0x0000, 0x0000, 0x5571, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5572, 0x5573, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5574, 0x0000, 0x0000, 0x0000, 0x0000, 0x5575, 0x0000, 0x5576,
	0x0000, 0x0000, 0x5577, 0x0000, 0x5578, 0x5579, 0x0000, 0x557a,
	0x557b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x557c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x557d, 0x557e, 0x0000,
	0x5621, 0x0000, 0x5622, 0x5623, 0x0000, 0x0000, 0x5624, 0x0000,
	0x0000, 0x5625, 0x5626, 0x0000, 0x0000, 0x0000, 0x5627, 0x0000,
};

static unsigned short const unicode_to_jisx0212_81[] = {
	/* 0x8100 - 0x81ff */
	0x0000, 0x0000, 0x0000, 0x5628, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5629, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x562a, 0x562b,
	0x562c, 0x0000, 0x0000, 0x0000, 0x562d, 0x0000, 0x562e, 0x0000,
	0x562f, 0x0000, 0x0000, 0x0000, 0x5630, 0x0000, 0x0000, 0x5631,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5632, 0x0000, 0x0000, 0x0000,
	0x5633, 0x0000, 0x0000, 0x0000, 0x0000, 0x5634, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5635, 0x0000, 0x5636, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5637, 0x0000, 0x5638,
	0x0000, 0x0000, 0x5639, 0x0000, 0x563a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x563b, 0x0000, 0x0000, 0x0000, 0x0000, 0x563c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x563d, 0x563e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x563f,
	0x5640, 0x5641, 0x0000, 0x0000, 0x0000, 0x5642, 0x0000, 0x5643,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5644,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5645, 0x0000, 0x0000, 0x5647, 0x5648, 0x5649, 0x0000,
	0x0000, 0x0000, 0x0000, 0x564a, 0x0000, 0x0000, 0x564b, 0x0000,
	0x5646, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x564c, 0x0000,
	0x564d, 0x0000, 0x0000, 0x564e, 0x0000, 0x0000, 0x564f, 0x0000,
	0x0000, 0x0000, 0x5650, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5651, 0x0000,
	0x0000, 0x0000, 0x5652, 0x0000, 0x5653, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5654, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5656, 0x0000, 0x5657, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5658, 0x5655, 0x0000, 0x0000, 0x5659, 0x565a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x565b, 0x0000, 0x565c,
	0x0000, 0x0000, 0x0000, 0x565d, 0x0000, 0x565e, 0x565f, 0x0000,
	0x0000, 0x5660, 0x0000, 0x0000, 0x5661, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5662, 0x5663, 0x0000, 0x0000, 0x0000,
	0x5664, 0x5665, 0x5666, 0x0000, 0x0000, 0x5667, 0x5668, 0x0000,
	0x5669, 0x566a, 0x0000, 0x0000, 0x0000, 0x566b, 0x0000, 0x566c,
};

static unsigned short const unicode_to_jisx0212_82[] = {
	/* 0x8200 - 0x82ff */
	0x566d, 0x0000, 0x0000, 0x566e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x566f,
	0x0000, 0x0000, 0x0000, 0x5670, 0x5671, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5672, 0x5673, 0x0000, 0x0000, 0x5674, 0x0000, 0x0000,
	0x0000, 0x5675, 0x5676, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5677, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5678, 0x0000, 0x5679, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x567a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x567b, 0x567c, 0x567d, 0x567e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5721, 0x0000, 0x0000, 0x5722, 0x5723,
	0x0000, 0x5724, 0x0000, 0x0000, 0x0000, 0x0000, 0x5725, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5726, 0x0000, 0x0000, 0x0000,
	0x5727, 0x0000, 0x0000, 0x5728, 0x0000, 0x0000, 0x0000, 0x5729,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x572a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x572b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x572c, 0x0000, 0x572d, 0x0000, 0x572e,
	0x572f, 0x5730, 0x0000, 0x5731, 0x5732, 0x0000, 0x0000, 0x5733,
	0x0000, 0x5734, 0x5735, 0x0000, 0x0000, 0x0000, 0x5736, 0x0000,
	0x0000, 0x5737, 0x0000, 0x0000, 0x5738, 0x0000, 0x5739, 0x0000,
	0x573a, 0x0000, 0x573b, 0x573c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x573d, 0x573e, 0x0000, 0x573f, 0x5740, 0x0000, 0x0000, 0x5741,
	0x5742, 0x5743, 0x5744, 0x0000, 0x0000, 0x0000, 0x5745, 0x0000,
	0x5746, 0x0000, 0x5747, 0x0000, 0x5748, 0x0000, 0x0000, 0x5749,
	0x0000, 0x0000, 0x574a, 0x0000, 0x574b, 0x0000, 0x574c, 0x574d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x574e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x574f, 0x0000, 0x0000, 0x0000, 0x0000, 0x5750, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5751, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5752, 0x0000, 0x5753, 0x0000, 0x5754, 0x0000, 0x0000, 0x0000,
	0x5755, 0x0000, 0x5756, 0x0000, 0x0000, 0x5757, 0x0000, 0x5758,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5759, 0x575a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x575b, 0x575c, 0x0000,
};

static unsigned short const unicode_to_jisx0212_83[] = {
	/* 0x8300 - 0x83ff */
	0x575d, 0x575e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x575f,
	0x5760, 0x0000, 0x5761, 0x5762, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5764, 0x0000, 0x5765, 0x5766, 0x5767,
	0x0000, 0x5768, 0x5769, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x576a, 0x576b, 0x576c, 0x0000,
	0x576d, 0x0000, 0x0000, 0x576e, 0x0000, 0x0000, 0x0000, 0x576f,
	0x0000, 0x0000, 0x5770, 0x0000, 0x5771, 0x5772, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5773, 0x5774, 0x5775, 0x0000, 0x0000, 0x5776,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5777, 0x5778, 0x0000,
	0x0000, 0x5779, 0x0000, 0x583e, 0x5763, 0x577a, 0x577b, 0x577c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x745f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x577d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x577e, 0x0000, 0x0000, 0x0000, 0x0000, 0x5821, 0x0000, 0x5822,
	0x5823, 0x0000, 0x5824, 0x0000, 0x5825, 0x0000, 0x5826, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5827, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5828, 0x0000, 0x5829, 0x582a, 0x0000, 0x0000,
	0x582b, 0x582c, 0x0000, 0x582d, 0x582e, 0x582f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5830, 0x5831,
	0x0000, 0x5832, 0x0000, 0x0000, 0x5833, 0x584c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5834, 0x5835,
	0x5836, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5837,
	0x0000, 0x5838, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5839,
	0x583a, 0x583b, 0x0000, 0x0000, 0x583c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x583d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x583f, 0x0000, 0x5840, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5841, 0x0000,
	0x5842, 0x5843, 0x0000, 0x0000, 0x5844, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_84[] = {
	/* 0x8400 - 0x84ff */
	0x0000, 0x5845, 0x0000, 0x0000, 0x0000, 0x0000, 0x5846, 0x0000,
	0x0000, 0x0000, 0x5847, 0x0000, 0x0000, 0x0000, 0x0000, 0x5848,
	0x0000, 0x5849, 0x0000, 0x0000, 0x0000, 0x584a, 0x0000, 0x0000,
	0x0000, 0x584b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x584d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x584e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x584f, 0x0000, 0x5850,
	0x5851, 0x0000, 0x5852, 0x0000, 0x0000, 0x5853, 0x0000, 0x5854,
	0x0000, 0x5855, 0x5856, 0x0000, 0x0000, 0x0000, 0x5857, 0x0000,
	0x5858, 0x5859, 0x585a, 0x0000, 0x585b, 0x0000, 0x0000, 0x0000,
	0x585c, 0x0000, 0x0000, 0x0000, 0x585d, 0x585e, 0x0000, 0x585f,
	0x0000, 0x0000, 0x5860, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5861, 0x0000, 0x0000, 0x5862, 0x5863, 0x0000, 0x5864, 0x0000,
	0x5865, 0x0000, 0x0000, 0x0000, 0x5866, 0x5867, 0x0000, 0x0000,
	0x0000, 0x5868, 0x0000, 0x0000, 0x0000, 0x5869, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x586a, 0x586b, 0x0000, 0x586c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x586d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x586e, 0x0000,
	0x586f, 0x5870, 0x5871, 0x0000, 0x0000, 0x0000, 0x0000, 0x5872,
	0x0000, 0x5873, 0x0000, 0x0000, 0x5874, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5875, 0x0000, 0x0000, 0x5876, 0x5877, 0x0000,
	0x5878, 0x0000, 0x5879, 0x0000, 0x0000, 0x0000, 0x0000, 0x587a,
	0x587b, 0x0000, 0x0000, 0x0000, 0x587c, 0x0000, 0x0000, 0x587d,
	0x0000, 0x0000, 0x0000, 0x587e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5921, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5922,
	0x0000, 0x0000, 0x5923, 0x0000, 0x0000, 0x0000, 0x0000, 0x5924,
	0x5925, 0x5926, 0x5927, 0x0000, 0x0000, 0x0000, 0x0000, 0x5928,
	0x0000, 0x0000, 0x592a, 0x592b, 0x0000, 0x592c, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_85[] = {
	/* 0x8500 - 0x85ff */
	0x0000, 0x0000, 0x592d, 0x592e, 0x0000, 0x0000, 0x0000, 0x592f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5930, 0x0000, 0x5931, 0x0000,
	0x5932, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5933, 0x0000, 0x5934, 0x0000,
	0x0000, 0x0000, 0x5935, 0x5936, 0x5937, 0x5938, 0x0000, 0x5939,
	0x0000, 0x0000, 0x593a, 0x593b, 0x0000, 0x0000, 0x0000, 0x593c,
	0x0000, 0x0000, 0x5929, 0x593d, 0x593e, 0x0000, 0x593f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5940,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5941, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5942,
	0x5943, 0x5944, 0x5945, 0x5946, 0x0000, 0x0000, 0x5947, 0x0000,
	0x0000, 0x5948, 0x0000, 0x0000, 0x5949, 0x594a, 0x594b, 0x594c,
	0x594d, 0x594e, 0x594f, 0x0000, 0x5950, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5951, 0x0000, 0x0000, 0x0000, 0x5952,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5953, 0x5954, 0x5955, 0x0000, 0x5956, 0x0000, 0x5957,
	0x0000, 0x5958, 0x0000, 0x0000, 0x0000, 0x5959, 0x595a, 0x0000,
	0x0000, 0x595b, 0x0000, 0x595c, 0x595d, 0x0000, 0x0000, 0x595e,
	0x0000, 0x0000, 0x0000, 0x595f, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5960, 0x0000, 0x0000, 0x0000, 0x0000, 0x5961, 0x0000, 0x5962,
	0x5963, 0x0000, 0x5964, 0x0000, 0x0000, 0x5965, 0x0000, 0x5966,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5974, 0x0000, 0x0000,
	0x7461, 0x0000, 0x0000, 0x0000, 0x5967, 0x0000, 0x5968, 0x5969,
	0x596a, 0x0000, 0x0000, 0x0000, 0x596b, 0x596c, 0x596d, 0x596e,
	0x0000, 0x0000, 0x596f, 0x0000, 0x0000, 0x0000, 0x0000, 0x5970,
	0x0000, 0x0000, 0x5971, 0x5972, 0x0000, 0x0000, 0x5973, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5975, 0x0000, 0x5976, 0x0000, 0x0000, 0x0000, 0x0000, 0x5977,
	0x5978, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5979, 0x0000,
	0x597a, 0x0000, 0x0000, 0x0000, 0x0000, 0x597b, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x597c, 0x0000, 0x0000, 0x597d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x597e, 0x0000, 0x0000, 0x5a21,
};

static unsigned short const unicode_to_jisx0212_86[] = {
	/* 0x8600 - 0x86ff */
	0x5a22, 0x0000, 0x0000, 0x0000, 0x5a23, 0x5a24, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a25, 0x5a26, 0x0000,
	0x5a27, 0x5a28, 0x5a29, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5a2a, 0x5a2b, 0x0000, 0x5a2c, 0x0000, 0x0000, 0x5a2d, 0x0000,
	0x0000, 0x5a2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a2f,
	0x0000, 0x5a30, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a31, 0x0000,
	0x5a32, 0x0000, 0x5a33, 0x0000, 0x5a34, 0x5a35, 0x0000, 0x0000,
	0x5a36, 0x3866, 0x5a37, 0x0000, 0x0000, 0x0000, 0x5a38, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5a39, 0x5a3a, 0x0000, 0x0000, 0x5a3b, 0x5a3c,
	0x5a3d, 0x5a3e, 0x0000, 0x0000, 0x0000, 0x5a3f, 0x0000, 0x0000,
	0x5a40, 0x5a41, 0x5a42, 0x5a43, 0x5a44, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5a45, 0x0000, 0x0000, 0x5a46, 0x0000, 0x0000, 0x5a47,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a48, 0x5a49, 0x5a4a,
	0x0000, 0x0000, 0x5a4b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5a6d, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a4c, 0x0000, 0x0000,
	0x0000, 0x5a4d, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a4e, 0x0000,
	0x5a4f, 0x0000, 0x5a50, 0x0000, 0x5a51, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5a52, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a53, 0x5a54,
	0x5a55, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a56, 0x0000, 0x0000,
	0x0000, 0x5a57, 0x0000, 0x5a58, 0x5a59, 0x5a5a, 0x0000, 0x5a5b,
	0x5a5c, 0x5a5d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a5e,
	0x5a5f, 0x5a60, 0x0000, 0x5a61, 0x0000, 0x5a62, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5a63, 0x5a64, 0x0000, 0x0000, 0x5a65, 0x0000, 0x5a66,
	0x0000, 0x0000, 0x5a67, 0x0000, 0x5a68, 0x0000, 0x0000, 0x0000,
	0x5a69, 0x0000, 0x0000, 0x5a6a, 0x0000, 0x5a6b, 0x0000, 0x5a6c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5a6e, 0x0000, 0x5a6f, 0x5a70, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_87[] = {
	/* 0x8700 - 0x87ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x5a71, 0x5a72, 0x0000, 0x5a73,
	0x0000, 0x0000, 0x0000, 0x5a74, 0x0000, 0x0000, 0x5a75, 0x5a76,
	0x5a77, 0x0000, 0x0000, 0x5a78, 0x5a79, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5a7a, 0x0000, 0x0000, 0x0000, 0x0000, 0x5a7b, 0x5a7c,
	0x0000, 0x5a7d, 0x0000, 0x5a7e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b21, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b22, 0x5b23,
	0x0000, 0x5b24, 0x5b25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5b26, 0x5b27, 0x0000, 0x5b28, 0x5b29, 0x5b2a, 0x0000,
	0x5b2b, 0x0000, 0x0000, 0x5b2c, 0x0000, 0x5b2d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b2e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b30, 0x0000, 0x0000,
	0x0000, 0x5b31, 0x0000, 0x0000, 0x5b32, 0x5b33, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b34,
	0x0000, 0x5b35, 0x5b36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5b37, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5b38, 0x5b39, 0x5b3a, 0x5b3b, 0x5b3c,
	0x5b3d, 0x5b3e, 0x0000, 0x5b3f, 0x5b40, 0x0000, 0x0000, 0x0000,
	0x5b41, 0x0000, 0x0000, 0x5b42, 0x0000, 0x5b43, 0x0000, 0x5b44,
	0x5b45, 0x5b46, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b47, 0x0000,
	0x5b48, 0x0000, 0x0000, 0x5b49, 0x0000, 0x0000, 0x0000, 0x5b4a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5b4b, 0x5b4c, 0x5b4d, 0x0000,
	0x0000, 0x5b4e, 0x0000, 0x0000, 0x0000, 0x5b4f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b50, 0x5b51,
	0x0000, 0x5b52, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b53, 0x5b54, 0x5b55, 0x0000, 0x0000, 0x0000, 0x5b56, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b57, 0x5b58, 0x0000,
	0x0000, 0x5b59, 0x5b5a, 0x0000, 0x5b5b, 0x0000, 0x0000, 0x5b5c,
	0x0000, 0x0000, 0x5b5d, 0x5b5e, 0x5b5f, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5b60, 0x5b61, 0x0000, 0x5b62, 0x0000, 0x0000,
	0x0000, 0x5b63, 0x0000, 0x5b64, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b65, 0x0000, 0x5b66, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b67,
};

static unsigned short const unicode_to_jisx0212_88[] = {
	/* 0x8800 - 0x88ff */
	0x0000, 0x5b68, 0x0000, 0x5b69, 0x0000, 0x0000, 0x5b6a, 0x7464,
	0x0000, 0x5b6b, 0x5b6c, 0x5b6d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b6e, 0x0000, 0x5b70, 0x5b71, 0x5b72, 0x0000, 0x0000, 0x0000,
	0x5b73, 0x5b6f, 0x5b74, 0x5b75, 0x5b76, 0x0000, 0x5b77, 0x5b78,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5b79, 0x0000, 0x0000, 0x0000, 0x0000, 0x5b7a, 0x5b7b, 0x0000,
	0x5b7c, 0x0000, 0x5b7d, 0x0000, 0x0000, 0x5b7e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c21, 0x0000, 0x5c22, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c23, 0x0000, 0x5c24, 0x0000, 0x5c25, 0x0000, 0x0000,
	0x5c26, 0x5c27, 0x5c28, 0x5c29, 0x0000, 0x0000, 0x5c2a, 0x0000,
	0x0000, 0x5c2b, 0x0000, 0x0000, 0x0000, 0x5c2c, 0x5c2d, 0x0000,
	0x5c2e, 0x0000, 0x5c2f, 0x0000, 0x5c30, 0x0000, 0x0000, 0x5c31,
	0x5c32, 0x0000, 0x0000, 0x0000, 0x5c33, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c34, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c35, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c36, 0x0000, 0x5c37, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5c38, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5c39, 0x0000, 0x5c3a, 0x5c3b, 0x5c3c, 0x0000, 0x0000, 0x5c3d,
	0x5c3e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5c3f, 0x0000, 0x5c40, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c41, 0x0000, 0x0000, 0x5c42, 0x5c43, 0x0000,
	0x5c44, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c45, 0x5c46, 0x5c47, 0x5c48, 0x5c49, 0x0000,
	0x0000, 0x5c4a, 0x5c4b, 0x5c4c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5c4d, 0x0000, 0x0000, 0x5c4e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c4f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c50,
	0x5c51, 0x5c52, 0x0000, 0x0000, 0x0000, 0x5c53, 0x0000, 0x5c54,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_89[] = {
	/* 0x8900 - 0x89ff */
	0x0000, 0x5c55, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c56, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c57, 0x5c58, 0x5c59,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c5a, 0x5c5b, 0x0000,
	0x5c5c, 0x5c5d, 0x5c5e, 0x0000, 0x5c5f, 0x0000, 0x0000, 0x0000,
	0x5c60, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c61, 0x5c62,
	0x5c63, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5c64, 0x5c65, 0x5c66, 0x0000, 0x0000, 0x5c67, 0x0000, 0x0000,
	0x0000, 0x5c68, 0x5c69, 0x0000, 0x0000, 0x0000, 0x5c6a, 0x0000,
	0x5c6b, 0x0000, 0x5c6c, 0x0000, 0x0000, 0x5c6d, 0x5c6e, 0x0000,
	0x0000, 0x5c6f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c70,
	0x0000, 0x0000, 0x5c71, 0x0000, 0x0000, 0x0000, 0x0000, 0x5c72,
	0x0000, 0x0000, 0x5c73, 0x5c74, 0x5c75, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5c76, 0x5c77, 0x5c78, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5c79, 0x0000, 0x0000, 0x5c7a, 0x0000,
	0x5c7b, 0x0000, 0x0000, 0x5c7c, 0x0000, 0x5c7d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5c7e, 0x5d21, 0x5d22, 0x5d23, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5d24, 0x0000, 0x0000, 0x0000, 0x5d25, 0x0000, 0x0000,
	0x5d26, 0x0000, 0x0000, 0x0000, 0x5d27, 0x5d28, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d29, 0x5d2a, 0x0000, 0x0000, 0x5d2b,
	0x5d2c, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d2d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5d2e, 0x0000, 0x0000, 0x0000, 0x5d2f, 0x5d30, 0x5d31, 0x5d32,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5d33, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5d34, 0x5d35, 0x5d36, 0x5d37,
	0x5d38, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d39, 0x0000, 0x0000,
	0x0000, 0x5d3a, 0x0000, 0x5d3b, 0x0000, 0x5d3c, 0x0000, 0x0000,
	0x0000, 0x5d3d, 0x0000, 0x5d3e, 0x0000, 0x0000, 0x5d3f, 0x0000,
	0x0000, 0x5d40, 0x0000, 0x0000, 0x0000, 0x5d41, 0x0000, 0x5d42,
};

static unsigned short const unicode_to_jisx0212_8a[] = {
	/* 0x8a00 - 0x8aff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x5d43, 0x5d44, 0x0000, 0x5d45,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d46,
	0x0000, 0x5d47, 0x5d48, 0x0000, 0x5d49, 0x5d4a, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d4b, 0x0000,
	0x5d4c, 0x0000, 0x5d4d, 0x0000, 0x5d4e, 0x0000, 0x5d4f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d50, 0x5d51, 0x0000, 0x0000, 0x5d52,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d53, 0x0000, 0x5d54,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d55, 0x5d56, 0x0000,
	0x5d57, 0x0000, 0x0000, 0x5d58, 0x0000, 0x5d59, 0x0000, 0x5d5a,
	0x0000, 0x5d5b, 0x0000, 0x0000, 0x0000, 0x5d5c, 0x5d5d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d5e, 0x0000, 0x0000, 0x5d5f, 0x5d60,
	0x5d61, 0x0000, 0x0000, 0x0000, 0x5d62, 0x5d63, 0x0000, 0x0000,
	0x0000, 0x5d64, 0x0000, 0x0000, 0x0000, 0x5d65, 0x0000, 0x5d66,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d67, 0x5d68, 0x5d69,
	0x0000, 0x5d6a, 0x5d6b, 0x5d6c, 0x0000, 0x0000, 0x5d6d, 0x5d6e,
	0x5d6f, 0x0000, 0x0000, 0x5d70, 0x0000, 0x0000, 0x5d71, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5d72, 0x0000, 0x0000, 0x0000, 0x5d73,
	0x5d74, 0x0000, 0x5d75, 0x0000, 0x0000, 0x0000, 0x5d76, 0x5d77,
	0x0000, 0x5d78, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d79,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d7a,
	0x0000, 0x5d7b, 0x0000, 0x0000, 0x0000, 0x0000, 0x5d7c, 0x5d7d,
	0x0000, 0x0000, 0x0000, 0x5d7e, 0x0000, 0x0000, 0x5e21, 0x5e22,
	0x0000, 0x0000, 0x0000, 0x5e23, 0x0000, 0x0000, 0x5e24, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e25, 0x0000, 0x0000, 0x5e26, 0x0000,
	0x5e27, 0x5e28, 0x5e29, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5e2a, 0x0000, 0x5e2b, 0x5e2c, 0x5e2d, 0x0000, 0x5e2e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e2f, 0x0000, 0x5e30,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e31, 0x0000, 0x0000, 0x0000,
	0x5e32, 0x0000, 0x0000, 0x0000, 0x5e33, 0x5e34, 0x5e35, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e36, 0x0000, 0x0000, 0x5e37,
};

static unsigned short const unicode_to_jisx0212_8b[] = {
	/* 0x8b00 - 0x8bff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e38, 0x5e39, 0x0000,
	0x0000, 0x0000, 0x5e3f, 0x5e3a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5e3b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e3c, 0x0000, 0x5e3d, 0x5e3e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e40, 0x0000, 0x0000,
	0x5e41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e42,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e43, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5e44, 0x5e45, 0x5e46, 0x5e47, 0x5e48, 0x0000,
	0x5e49, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e4e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5e4a, 0x5e4b, 0x5e4c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5e4d, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e4f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e50, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e51, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e52, 0x0000,
	0x5e53, 0x5e54, 0x0000, 0x0000, 0x5e55, 0x0000, 0x5e56, 0x7466,
	0x0000, 0x5e57, 0x0000, 0x0000, 0x5e58, 0x5e59, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e5a, 0x0000, 0x5e5b, 0x0000, 0x5e5c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e5d, 0x5e5e, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5e5f, 0x0000, 0x5e60, 0x5e61,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_8c[] = {
	/* 0x8c00 - 0x8cff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5e62, 0x5e63, 0x0000, 0x0000, 0x0000, 0x5e64, 0x5e65, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e66, 0x0000, 0x5e67,
	0x0000, 0x5e68, 0x0000, 0x5e69, 0x0000, 0x0000, 0x0000, 0x5e6a,
	0x0000, 0x5e6b, 0x0000, 0x5e6c, 0x5e6d, 0x0000, 0x0000, 0x5e6e,
	0x5e6f, 0x5e72, 0x0000, 0x5e70, 0x0000, 0x5e71, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e73, 0x5e74, 0x0000, 0x5e75, 0x0000,
	0x5e76, 0x5e77, 0x0000, 0x0000, 0x0000, 0x5e78, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e79, 0x0000, 0x5e7a, 0x5e7b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5e7c, 0x0000, 0x0000, 0x5e7d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5e7e, 0x5f21,
	0x0000, 0x0000, 0x0000, 0x5f22, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5f23, 0x0000, 0x5f24, 0x5f25, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f26, 0x0000, 0x5f27, 0x5f28, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f29, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f2a, 0x5f2b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f2c, 0x5f2d, 0x0000,
	0x0000, 0x5f2e, 0x0000, 0x5f2f, 0x0000, 0x0000, 0x0000, 0x5f30,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f32, 0x5f31, 0x0000,
	0x0000, 0x5f33, 0x0000, 0x0000, 0x0000, 0x5f34, 0x0000, 0x0000,
	0x0000, 0x5f35, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5f36, 0x0000, 0x0000, 0x0000, 0x5f37, 0x0000, 0x0000, 0x5f38,
	0x5f39, 0x0000, 0x5f3a, 0x0000, 0x7467, 0x5f3b, 0x0000, 0x5f3c,
	0x5f3d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f3e, 0x5f3f,
};

static unsigned short const unicode_to_jisx0212_8d[] = {
	/* 0x8d00 - 0x8dff */
	0x0000, 0x5f40, 0x0000, 0x5f41, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f42, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x5f43, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f44,
	0x0000, 0x0000, 0x0000, 0x5f45, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f46, 0x0000, 0x0000,
	0x0000, 0x5f47, 0x0000, 0x0000, 0x5f48, 0x0000, 0x5f49, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7468, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f4a,
	0x0000, 0x0000, 0x5f4b, 0x0000, 0x5f4c, 0x0000, 0x0000, 0x0000,
	0x5f4d, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f4e, 0x0000, 0x0000,
	0x5f4f, 0x5f50, 0x0000, 0x0000, 0x0000, 0x5f51, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f52, 0x5f53,
	0x5f54, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f55, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5f56, 0x5f57, 0x0000, 0x0000, 0x5f58,
	0x0000, 0x0000, 0x5f59, 0x0000, 0x0000, 0x5f5a, 0x0000, 0x5f5b,
	0x0000, 0x5f5c, 0x0000, 0x5f5d, 0x5f6f, 0x0000, 0x0000, 0x0000,
	0x5f5e, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f5f, 0x5f60, 0x5f61,
	0x5f62, 0x0000, 0x5f63, 0x0000, 0x0000, 0x0000, 0x5f64, 0x0000,
	0x0000, 0x5f65, 0x0000, 0x0000, 0x5f66, 0x5f67, 0x0000, 0x5f68,
	0x0000, 0x5f69, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f6a, 0x5f6b, 0x0000, 0x5f6c,
	0x0000, 0x0000, 0x0000, 0x0000, 0x5f6d, 0x0000, 0x0000, 0x0000,
	0x5f6e, 0x5f70, 0x5f71, 0x0000, 0x5f72, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x5f73, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_8e[] = {
	/* 0x8e00 - 0x8eff */
	0x0000, 0x5f74, 0x0000, 0x0000, 0x5f75, 0x5f76, 0x5f77, 0x0000,
	0x0000, 0x0000, 0x0000, 0x5f78, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x5f79, 0x0000, 0x0000, 0x5f7a, 0x0000, 0x5f7b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x5f7c, 0x5f7d, 0x5f7e, 0x6021, 0x0000, 0x0000, 0x6022, 0x6023,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6024, 0x0000, 0x6025, 0x0000, 0x0000, 0x6026, 0x6027,
	0x6028, 0x6029, 0x0000, 0x0000, 0x0000, 0x602a, 0x0000, 0x0000,
	0x602b, 0x602c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x602d, 0x0000, 0x602e, 0x602f, 0x6030,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6031, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6032, 0x6033, 0x6034, 0x6035, 0x0000,
	0x0000, 0x6036, 0x6037, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6038, 0x0000, 0x0000, 0x6039, 0x603a, 0x0000, 0x603b,
	0x603c, 0x603d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x603e, 0x603f, 0x6040, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6041, 0x6042, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6043, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6044, 0x0000, 0x6045, 0x0000, 0x0000, 0x6046, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6047, 0x6048, 0x0000, 0x6049, 0x604a, 0x0000,
	0x0000, 0x0000, 0x604b, 0x0000, 0x0000, 0x0000, 0x0000, 0x604c,
	0x0000, 0x604d, 0x0000, 0x0000, 0x0000, 0x604e, 0x604f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6050, 0x0000, 0x6051, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6052, 0x6053, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6054, 0x6055, 0x0000, 0x6056, 0x6057, 0x0000, 0x0000, 0x6058,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6059,
	0x0000, 0x605a, 0x0000, 0x0000, 0x605b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x605c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x605d, 0x0000, 0x0000, 0x0000, 0x0000, 0x6064, 0x605e, 0x0000,
	0x605f, 0x6060, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6061,
	0x0000, 0x6062, 0x6063, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_8f[] = {
	/* 0x8f00 - 0x8fff */
	0x6065, 0x0000, 0x6066, 0x0000, 0x0000, 0x0000, 0x0000, 0x6067,
	0x6068, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6069,
	0x606a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x606b, 0x606c,
	0x606d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x606e, 0x0000,
	0x606f, 0x6070, 0x0000, 0x6071, 0x0000, 0x6072, 0x0000, 0x6073,
	0x6074, 0x0000, 0x0000, 0x0000, 0x6075, 0x6076, 0x6077, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6078, 0x6079, 0x607a, 0x607b,
	0x0000, 0x0000, 0x607c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x607d, 0x607e, 0x0000, 0x6121, 0x0000, 0x0000, 0x0000, 0x6122,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6123,
	0x0000, 0x6124, 0x6125, 0x6126, 0x6127, 0x6128, 0x0000, 0x0000,
	0x6129, 0x0000, 0x0000, 0x0000, 0x0000, 0x612a, 0x612b, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x612c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x612d, 0x0000, 0x0000,
	0x612e, 0x612f, 0x0000, 0x0000, 0x6130, 0x6131, 0x6132, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6133, 0x6134, 0x0000,
	0x6135, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6136, 0x0000,
	0x6137, 0x6138, 0x0000, 0x0000, 0x0000, 0x0000, 0x6139, 0x0000,
	0x0000, 0x0000, 0x613a, 0x613b, 0x0000, 0x613c, 0x0000, 0x0000,
	0x613d, 0x0000, 0x613e, 0x613f, 0x0000, 0x6140, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6141, 0x0000, 0x0000, 0x6142, 0x6143, 0x0000, 0x0000, 0x0000,
	0x6144, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6145, 0x0000,
	0x0000, 0x6146, 0x0000, 0x0000, 0x0000, 0x6147, 0x6148, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6149, 0x0000, 0x0000, 0x614a, 0x0000,
};

static unsigned short const unicode_to_jisx0212_90[] = {
	/* 0x9000 - 0x90ff */
	0x0000, 0x0000, 0x614b, 0x0000, 0x614c, 0x0000, 0x0000, 0x0000,
	0x614d, 0x0000, 0x0000, 0x0000, 0x614e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x614f, 0x0000, 0x0000, 0x6150, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6151, 0x6152, 0x6154, 0x0000, 0x6155, 0x6156, 0x0000, 0x6153,
	0x0000, 0x0000, 0x0000, 0x6157, 0x6158, 0x0000, 0x0000, 0x6159,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x615a,
	0x0000, 0x0000, 0x0000, 0x615b, 0x615c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x615d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x615e, 0x0000, 0x615f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6160, 0x0000, 0x0000, 0x0000, 0x6161, 0x6162,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6163, 0x0000, 0x0000, 0x0000,
	0x6164, 0x0000, 0x0000, 0x0000, 0x6165, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6166, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6167, 0x0000, 0x0000,
	0x6168, 0x0000, 0x0000, 0x6169, 0x616a, 0x0000, 0x616b, 0x0000,
	0x616c, 0x0000, 0x0000, 0x0000, 0x0000, 0x616d, 0x0000, 0x616e,
	0x616f, 0x6170, 0x0000, 0x6171, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6172, 0x6173, 0x6174, 0x0000, 0x0000, 0x6175, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6176, 0x0000, 0x6177, 0x6178, 0x6179, 0x0000, 0x617a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x617b, 0x617d, 0x0000,
	0x0000, 0x0000, 0x0000, 0x617e, 0x6221, 0x6222, 0x0000, 0x6223,
	0x6224, 0x0000, 0x0000, 0x0000, 0x617c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x622d, 0x0000, 0x0000, 0x6225, 0x0000, 0x6226,
	0x6227, 0x6228, 0x0000, 0x0000, 0x6229, 0x622a, 0x746c, 0x622b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x622c, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x622f, 0x0000, 0x0000, 0x0000, 0x6230,
	0x6231, 0x0000, 0x0000, 0x0000, 0x6232, 0x0000, 0x622e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6233, 0x6234,
};

static unsigned short const unicode_to_jisx0212_91[] = {
	/* 0x9100 - 0x91ff */
	0x6235, 0x0000, 0x0000, 0x0000, 0x6236, 0x6237, 0x6238, 0x0000,
	0x6239, 0x0000, 0x0000, 0x0000, 0x0000, 0x623a, 0x0000, 0x0000,
	0x623b, 0x0000, 0x0000, 0x0000, 0x623c, 0x746e, 0x623d, 0x623e,
	0x623f, 0x0000, 0x6240, 0x0000, 0x6241, 0x0000, 0x6242, 0x0000,
	0x6243, 0x0000, 0x6245, 0x6246, 0x0000, 0x6244, 0x0000, 0x6247,
	0x0000, 0x6248, 0x0000, 0x0000, 0x0000, 0x0000, 0x6249, 0x624a,
	0x0000, 0x624b, 0x0000, 0x0000, 0x624c, 0x0000, 0x624d, 0x624e,
	0x0000, 0x624f, 0x6250, 0x0000, 0x6251, 0x6252, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6253, 0x0000, 0x0000, 0x0000, 0x6254,
	0x6255, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6256,
	0x0000, 0x0000, 0x0000, 0x6257, 0x0000, 0x0000, 0x0000, 0x6258,
	0x0000, 0x6259, 0x625a, 0x625b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x625c, 0x0000, 0x0000, 0x625d, 0x0000, 0x0000, 0x625e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x625f, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6260, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6261, 0x6262, 0x6263, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6264, 0x0000, 0x6265, 0x0000, 0x6266, 0x6267, 0x0000,
	0x0000, 0x0000, 0x6268, 0x0000, 0x0000, 0x0000, 0x6269, 0x0000,
	0x0000, 0x626a, 0x0000, 0x626b, 0x626c, 0x626d, 0x0000, 0x0000,
	0x626e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x626f, 0x0000,
	0x0000, 0x6270, 0x0000, 0x0000, 0x0000, 0x0000, 0x6271, 0x0000,
	0x6272, 0x0000, 0x0000, 0x0000, 0x6273, 0x6274, 0x6275, 0x0000,
	0x6276, 0x6277, 0x6278, 0x6279, 0x0000, 0x0000, 0x627a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x627b, 0x627c, 0x627d, 0x0000, 0x627e,
	0x0000, 0x0000, 0x6321, 0x6322, 0x0000, 0x6323, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6324, 0x6325, 0x0000, 0x0000, 0x6326,
	0x0000, 0x6327, 0x6328, 0x0000, 0x0000, 0x0000, 0x6329, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x632a, 0x632b, 0x0000, 0x0000,
	0x0000, 0x632c, 0x632d, 0x0000, 0x632e, 0x632f, 0x6330, 0x6331,
	0x6332, 0x6333, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6334,
	0x0000, 0x6335, 0x0000, 0x6336, 0x0000, 0x6337, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_92[] = {
	/* 0x9200 - 0x92ff */
	0x6338, 0x6339, 0x0000, 0x0000, 0x633a, 0x633b, 0x633c, 0x633d,
	0x0000, 0x633e, 0x633f, 0x0000, 0x6340, 0x0000, 0x0000, 0x0000,
	0x6341, 0x0000, 0x6342, 0x6343, 0x0000, 0x0000, 0x6344, 0x0000,
	0x6345, 0x0000, 0x0000, 0x0000, 0x6346, 0x6347, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6348, 0x6349, 0x634a, 0x634b, 0x0000,
	0x634c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x634d, 0x634e,
	0x634f, 0x0000, 0x0000, 0x6350, 0x0000, 0x6351, 0x6352, 0x0000,
	0x6353, 0x6354, 0x6355, 0x0000, 0x6356, 0x0000, 0x6357, 0x0000,
	0x6358, 0x0000, 0x6359, 0x635a, 0x0000, 0x0000, 0x635b, 0x635c,
	0x0000, 0x0000, 0x635d, 0x0000, 0x0000, 0x635e, 0x635f, 0x6360,
	0x0000, 0x6361, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6362, 0x6363, 0x0000, 0x0000, 0x6364, 0x6365, 0x0000, 0x0000,
	0x6366, 0x6367, 0x0000, 0x0000, 0x0000, 0x6368, 0x0000, 0x6369,
	0x636a, 0x636b, 0x0000, 0x0000, 0x0000, 0x0000, 0x636c, 0x636d,
	0x636e, 0x0000, 0x0000, 0x0000, 0x0000, 0x636f, 0x6370, 0x6371,
	0x6372, 0x6373, 0x0000, 0x6374, 0x6375, 0x6376, 0x0000, 0x6377,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6378, 0x6379, 0x637a, 0x0000, 0x0000, 0x637b, 0x637c, 0x0000,
	0x0000, 0x0000, 0x637d, 0x0000, 0x0000, 0x0000, 0x0000, 0x637e,
	0x0000, 0x6421, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6422,
	0x6423, 0x0000, 0x0000, 0x0000, 0x6424, 0x6425, 0x0000, 0x6426,
	0x6427, 0x0000, 0x0000, 0x6428, 0x0000, 0x0000, 0x0000, 0x6429,
	0x0000, 0x0000, 0x642a, 0x0000, 0x0000, 0x0000, 0x642b, 0x0000,
	0x642c, 0x0000, 0x642d, 0x642e, 0x642f, 0x6430, 0x0000, 0x6431,
	0x6432, 0x6433, 0x6434, 0x6435, 0x0000, 0x6436, 0x6437, 0x6438,
	0x6439, 0x0000, 0x0000, 0x643a, 0x643b, 0x643c, 0x643d, 0x0000,
	0x643e, 0x0000, 0x0000, 0x643f, 0x0000, 0x6440, 0x0000, 0x6441,
	0x6442, 0x6443, 0x0000, 0x0000, 0x6444, 0x6445, 0x0000, 0x6446,
	0x6447, 0x6448, 0x0000, 0x6449, 0x0000, 0x644a, 0x0000, 0x644b,
	0x644c, 0x0000, 0x0000, 0x0000, 0x644d, 0x0000, 0x644e, 0x0000,
	0x644f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6450, 0x0000, 0x6451, 0x0000, 0x0000, 0x0000, 0x6452,
};

static unsigned short const unicode_to_jisx0212_93[] = {
	/* 0x9300 - 0x93ff */
	0x6453, 0x0000, 0x6454, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6455, 0x0000, 0x0000, 0x0000, 0x0000, 0x6456, 0x0000, 0x0000,
	0x0000, 0x6457, 0x0000, 0x0000, 0x6458, 0x6459, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x645a, 0x645b, 0x645c, 0x645d,
	0x0000, 0x645e, 0x0000, 0x0000, 0x645f, 0x6460, 0x0000, 0x6461,
	0x0000, 0x6462, 0x6463, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6464, 0x6465, 0x0000, 0x6466, 0x6467,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6468,
	0x6469, 0x646a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x646b, 0x646c, 0x646d, 0x0000, 0x0000, 0x646e, 0x0000, 0x646f,
	0x6470, 0x0000, 0x6471, 0x0000, 0x0000, 0x0000, 0x6472, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6473, 0x6474, 0x0000, 0x6475,
	0x0000, 0x6476, 0x6477, 0x0000, 0x0000, 0x6478, 0x0000, 0x6479,
	0x647a, 0x647b, 0x0000, 0x647c, 0x647d, 0x0000, 0x647e, 0x0000,
	0x0000, 0x0000, 0x6521, 0x0000, 0x0000, 0x6522, 0x0000, 0x6523,
	0x6524, 0x6525, 0x6526, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6527, 0x0000, 0x6528, 0x6529, 0x0000, 0x652a, 0x0000, 0x652b,
	0x0000, 0x0000, 0x652c, 0x0000, 0x0000, 0x652d, 0x0000, 0x0000,
	0x652e, 0x0000, 0x0000, 0x652f, 0x0000, 0x0000, 0x6530, 0x0000,
	0x0000, 0x6531, 0x0000, 0x6532, 0x6533, 0x0000, 0x6534, 0x0000,
	0x6535, 0x653b, 0x0000, 0x6536, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6537, 0x6538, 0x6539, 0x0000,
	0x0000, 0x0000, 0x653a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x653c, 0x0000, 0x0000, 0x653d, 0x653e, 0x653f, 0x6540,
	0x0000, 0x6541, 0x6542, 0x6543, 0x6544, 0x6545, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6546, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6547, 0x0000, 0x0000, 0x6548, 0x0000, 0x6549, 0x654a,
	0x0000, 0x0000, 0x654b, 0x0000, 0x0000, 0x0000, 0x654c, 0x654d,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x654f,
	0x6550, 0x654e, 0x6551, 0x6552, 0x0000, 0x6553, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_94[] = {
	/* 0x9400 - 0x94ff */
	0x0000, 0x6554, 0x6555, 0x0000, 0x6556, 0x0000, 0x0000, 0x0000,
	0x6557, 0x6558, 0x0000, 0x0000, 0x0000, 0x6559, 0x655a, 0x655b,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x655c, 0x655d, 0x655e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x655f,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6560, 0x6561,
	0x0000, 0x6562, 0x6563, 0x6564, 0x6565, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6566, 0x0000, 0x6568, 0x0000, 0x6567,
	0x0000, 0x0000, 0x0000, 0x6569, 0x0000, 0x656a, 0x0000, 0x0000,
	0x656b, 0x0000, 0x656c, 0x0000, 0x656d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x656e, 0x0000, 0x0000,
	0x0000, 0x656f, 0x0000, 0x0000, 0x6570, 0x0000, 0x0000, 0x6571,
	0x0000, 0x6572, 0x0000, 0x6573, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6574, 0x0000, 0x0000, 0x6575, 0x0000, 0x6576, 0x6577, 0x6578,
	0x0000, 0x6579, 0x657a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x657c, 0x657b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_95[] = {
	/* 0x9500 - 0x95ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x657d, 0x657e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6621, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6622, 0x0000, 0x0000, 0x0000,
	0x6623, 0x0000, 0x0000, 0x0000, 0x6624, 0x6625, 0x6626, 0x0000,
	0x0000, 0x0000, 0x7471, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6627, 0x6628, 0x6629,
	0x0000, 0x662a, 0x0000, 0x0000, 0x0000, 0x0000, 0x662b, 0x0000,
	0x0000, 0x662c, 0x0000, 0x662d, 0x662e, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x662f, 0x0000, 0x6630, 0x0000,
	0x0000, 0x0000, 0x6631, 0x0000, 0x0000, 0x6632, 0x0000, 0x6633,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6634, 0x0000,
	0x6635, 0x6636, 0x0000, 0x6637, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6638, 0x6639, 0x663a, 0x663b, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x663c, 0x663d, 0x0000, 0x0000, 0x663e, 0x663f, 0x6640,
	0x6641, 0x0000, 0x0000, 0x0000, 0x6642, 0x0000, 0x6643, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_96[] = {
	/* 0x9600 - 0x96ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6644, 0x6645, 0x0000,
	0x0000, 0x0000, 0x6646, 0x0000, 0x6647, 0x6648, 0x6649, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x664a, 0x0000, 0x0000, 0x0000,
	0x0000, 0x664b, 0x0000, 0x664c, 0x0000, 0x0000, 0x0000, 0x664d,
	0x664e, 0x664f, 0x6650, 0x0000, 0x6651, 0x6652, 0x0000, 0x0000,
	0x0000, 0x6653, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6654, 0x0000, 0x6655, 0x0000, 0x6656, 0x6657,
	0x6658, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6659, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x665a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x665b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x665c, 0x665d, 0x0000, 0x665e, 0x665f,
	0x0000, 0x6660, 0x6661, 0x6662, 0x6663, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6664, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6665, 0x0000, 0x0000, 0x0000, 0x0000, 0x6666, 0x0000,
	0x0000, 0x0000, 0x6667, 0x0000, 0x0000, 0x6668, 0x0000, 0x6669,
	0x0000, 0x0000, 0x0000, 0x0000, 0x666a, 0x666b, 0x666c, 0x0000,
	0x0000, 0x666d, 0x0000, 0x0000, 0x0000, 0x0000, 0x666e, 0x666f,
	0x0000, 0x0000, 0x0000, 0x6670, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6671, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6672, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6673, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6675, 0x0000, 0x6676, 0x0000, 0x0000, 0x6677, 0x6678, 0x6679,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x667a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x667b,
	0x0000, 0x667c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x667d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_97[] = {
	/* 0x9700 - 0x97ff */
	0x0000, 0x0000, 0x667e, 0x6721, 0x0000, 0x6722, 0x0000, 0x0000,
	0x0000, 0x6723, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6724, 0x6725, 0x0000, 0x6726, 0x0000, 0x0000,
	0x0000, 0x6727, 0x6728, 0x6729, 0x0000, 0x0000, 0x0000, 0x0000,
	0x672a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x672b, 0x0000, 0x672c, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x7474, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x672d, 0x0000, 0x672e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x672f, 0x0000, 0x0000, 0x7475, 0x6730, 0x6731,
	0x0000, 0x7476, 0x0000, 0x0000, 0x0000, 0x6732, 0x0000, 0x6733,
	0x6734, 0x0000, 0x6735, 0x6736, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6737, 0x0000, 0x0000, 0x0000, 0x6738,
	0x0000, 0x0000, 0x6739, 0x0000, 0x0000, 0x0000, 0x673a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x673b, 0x0000, 0x0000, 0x673c, 0x673d,
	0x673e, 0x0000, 0x0000, 0x673f, 0x0000, 0x6740, 0x0000, 0x6741,
	0x6742, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6743, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6744, 0x6745, 0x6746,
	0x0000, 0x6747, 0x6748, 0x0000, 0x0000, 0x0000, 0x6749, 0x674a,
	0x0000, 0x0000, 0x674b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x674c, 0x0000, 0x674d, 0x0000,
	0x0000, 0x674e, 0x674f, 0x0000, 0x0000, 0x6750, 0x6751, 0x0000,
	0x6752, 0x6753, 0x6754, 0x0000, 0x6755, 0x0000, 0x6756, 0x6757,
	0x0000, 0x6758, 0x0000, 0x0000, 0x6759, 0x675a, 0x0000, 0x675b,
	0x0000, 0x675c, 0x675d, 0x0000, 0x675e, 0x675f, 0x6760, 0x0000,
	0x6761, 0x6762, 0x0000, 0x0000, 0x6763, 0x0000, 0x0000, 0x6764,
	0x6765, 0x6766, 0x0000, 0x676a, 0x0000, 0x6767, 0x6768, 0x0000,
	0x6769, 0x676b, 0x0000, 0x0000, 0x676c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x676d,
	0x0000, 0x676e, 0x0000, 0x0000, 0x676f, 0x0000, 0x0000, 0x6770,
	0x6771, 0x0000, 0x6772, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_98[] = {
	/* 0x9800 - 0x98ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6773,
	0x0000, 0x0000, 0x6774, 0x0000, 0x0000, 0x6776, 0x6777, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6778, 0x0000, 0x6779, 0x0000,
	0x0000, 0x6775, 0x0000, 0x0000, 0x677a, 0x0000, 0x677b, 0x0000,
	0x677c, 0x0000, 0x0000, 0x677d, 0x0000, 0x6828, 0x677e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6821, 0x0000, 0x0000, 0x6822, 0x6823,
	0x6824, 0x0000, 0x6825, 0x6826, 0x0000, 0x6827, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6829, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x682a, 0x0000, 0x0000, 0x682b,
	0x0000, 0x0000, 0x682c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x682d, 0x682e, 0x682f, 0x0000, 0x0000, 0x6830, 0x6831,
	0x0000, 0x6832, 0x6833, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6834, 0x6835, 0x0000, 0x6836, 0x6837, 0x0000,
	0x0000, 0x0000, 0x6838, 0x0000, 0x6839, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x683a, 0x0000, 0x683b, 0x683c, 0x0000,
	0x683d, 0x0000, 0x0000, 0x0000, 0x683e, 0x0000, 0x0000, 0x683f,
	0x6840, 0x0000, 0x6841, 0x6842, 0x0000, 0x0000, 0x0000, 0x6843,
	0x0000, 0x0000, 0x6844, 0x0000, 0x0000, 0x6845, 0x0000, 0x0000,
	0x6846, 0x0000, 0x0000, 0x0000, 0x6847, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6848, 0x0000, 0x6849, 0x0000, 0x684a, 0x684b, 0x684c,
	0x0000, 0x0000, 0x684d, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x684e, 0x0000, 0x0000, 0x684f, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_99[] = {
	/* 0x9900 - 0x99ff */
	0x0000, 0x0000, 0x6850, 0x0000, 0x0000, 0x0000, 0x0000, 0x6851,
	0x6852, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6853, 0x0000, 0x0000, 0x0000, 0x6854, 0x6855, 0x6856,
	0x0000, 0x0000, 0x6857, 0x6858, 0x6859, 0x0000, 0x0000, 0x685a,
	0x0000, 0x0000, 0x685b, 0x0000, 0x0000, 0x0000, 0x685c, 0x685d,
	0x0000, 0x0000, 0x0000, 0x685e, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x685f, 0x6860, 0x6861, 0x6862, 0x6863, 0x0000, 0x0000,
	0x0000, 0x6864, 0x6865, 0x6866, 0x6867, 0x0000, 0x0000, 0x0000,
	0x6868, 0x6869, 0x0000, 0x0000, 0x0000, 0x0000, 0x686a, 0x686b,
	0x686c, 0x0000, 0x0000, 0x0000, 0x0000, 0x686d, 0x686e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x686f, 0x0000, 0x0000, 0x0000,
	0x6870, 0x6871, 0x0000, 0x6872, 0x6873, 0x0000, 0x6874, 0x6875,
	0x6876, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6877, 0x0000, 0x6878, 0x747a, 0x6879,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x687a, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x687b, 0x687c, 0x687d, 0x0000, 0x0000, 0x687e, 0x0000, 0x0000,
	0x0000, 0x6921, 0x6922, 0x0000, 0x0000, 0x6923, 0x0000, 0x6924,
	0x0000, 0x0000, 0x0000, 0x6925, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6926, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6927, 0x6928, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6929, 0x692a, 0x0000, 0x692b, 0x0000, 0x692c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x692d,
	0x0000, 0x0000, 0x692e, 0x692f, 0x6930, 0x0000, 0x0000, 0x0000,
	0x6931, 0x0000, 0x0000, 0x0000, 0x6932, 0x6933, 0x0000, 0x0000,
	0x0000, 0x6934, 0x0000, 0x0000, 0x0000, 0x6935, 0x6936, 0x0000,
};

static unsigned short const unicode_to_jisx0212_9a[] = {
	/* 0x9a00 - 0x9aff */
	0x0000, 0x0000, 0x6937, 0x6938, 0x6939, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x693a, 0x693b, 0x0000, 0x0000, 0x0000,
	0x693c, 0x693d, 0x0000, 0x0000, 0x0000, 0x0000, 0x693e, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x693f, 0x0000,
	0x6940, 0x0000, 0x6941, 0x6942, 0x6943, 0x0000, 0x0000, 0x6944,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6945, 0x6946, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6947, 0x0000, 0x6948, 0x6949, 0x0000,
	0x694a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x694c, 0x0000, 0x0000, 0x694d, 0x0000, 0x0000, 0x694b,
	0x0000, 0x0000, 0x694e, 0x694f, 0x6950, 0x0000, 0x6951, 0x0000,
	0x0000, 0x6952, 0x0000, 0x0000, 0x6953, 0x0000, 0x6954, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6955, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6956, 0x0000, 0x6957, 0x0000, 0x6958, 0x6959,
	0x0000, 0x0000, 0x695a, 0x0000, 0x695b, 0x695c, 0x695d, 0x0000,
	0x0000, 0x695e, 0x0000, 0x695f, 0x0000, 0x0000, 0x6960, 0x6961,
	0x0000, 0x6962, 0x0000, 0x6963, 0x0000, 0x0000, 0x6964, 0x0000,
	0x6965, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6966, 0x0000,
	0x6967, 0x0000, 0x6968, 0x0000, 0x0000, 0x6969, 0x696a, 0x696b,
	0x0000, 0x747b, 0x0000, 0x696c, 0x696d, 0x0000, 0x0000, 0x0000,
	0x696e, 0x0000, 0x0000, 0x0000, 0x696f, 0x6970, 0x0000, 0x6971,
	0x0000, 0x6972, 0x0000, 0x0000, 0x6973, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6974, 0x6975, 0x0000, 0x6976, 0x0000, 0x0000,
	0x0000, 0x6977, 0x6978, 0x0000, 0x0000, 0x6979, 0x0000, 0x697a,
};

static unsigned short const unicode_to_jisx0212_9b[] = {
	/* 0x9b00 - 0x9bff */
	0x697b, 0x697c, 0x697d, 0x697e, 0x6a21, 0x6a22, 0x0000, 0x0000,
	0x6a23, 0x6a24, 0x0000, 0x6a25, 0x6a26, 0x6a27, 0x6a28, 0x0000,
	0x6a29, 0x0000, 0x6a2a, 0x0000, 0x0000, 0x0000, 0x6a2b, 0x0000,
	0x0000, 0x6a2c, 0x0000, 0x6a2d, 0x6a2e, 0x0000, 0x0000, 0x0000,
	0x6a2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a30, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6a31, 0x0000, 0x6a32, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6a33, 0x6a34, 0x6a35, 0x0000, 0x6a36,
	0x0000, 0x6a37, 0x6a38, 0x0000, 0x0000, 0x6a39, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6a3a, 0x0000, 0x0000, 0x6a3b, 0x6a3c, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a3d, 0x6a3e, 0x6a3f,
	0x0000, 0x0000, 0x0000, 0x6a40, 0x0000, 0x0000, 0x6a41, 0x0000,
	0x0000, 0x6a42, 0x0000, 0x6a43, 0x0000, 0x6a44, 0x6a45, 0x0000,
	0x6a46, 0x0000, 0x6a47, 0x6a48, 0x6a49, 0x6a4a, 0x6a4b, 0x0000,
	0x0000, 0x0000, 0x747c, 0x6a4c, 0x0000, 0x6a4d, 0x0000, 0x6a4e,
	0x6a4f, 0x6a50, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a51,
	0x6a52, 0x0000, 0x0000, 0x0000, 0x6a53, 0x6a54, 0x6a55, 0x6a56,
	0x0000, 0x6a57, 0x6a58, 0x6a59, 0x0000, 0x6a5a, 0x0000, 0x6a5b,
	0x6a5c, 0x0000, 0x0000, 0x0000, 0x6a5d, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6a5e, 0x0000, 0x0000, 0x6a5f, 0x6a60, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a61, 0x6a62,
	0x0000, 0x6a63, 0x0000, 0x0000, 0x6a64, 0x0000, 0x0000, 0x0000,
	0x6a65, 0x6a66, 0x6a67, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a68,
	0x6a69, 0x0000, 0x0000, 0x6a6a, 0x6a6b, 0x0000, 0x6a6c, 0x6a6d,
	0x0000, 0x6a6e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a6f,
	0x6a70, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a71, 0x0000,
	0x6a72, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a73,
	0x6a74, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a75, 0x0000, 0x6a76,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6a77, 0x0000, 0x6a78,
	0x0000, 0x0000, 0x6a79, 0x6a7a, 0x0000, 0x0000, 0x0000, 0x6a7b,
	0x0000, 0x0000, 0x0000, 0x6a7c, 0x0000, 0x0000, 0x0000, 0x6a7d,
	0x6a7e, 0x6b21, 0x6b22, 0x0000, 0x0000, 0x6b23, 0x0000, 0x6b24,
};

static unsigned short const unicode_to_jisx0212_9c[] = {
	/* 0x9c00 - 0x9cff */
	0x6b25, 0x0000, 0x6b26, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6b27, 0x0000, 0x0000, 0x0000, 0x6b28,
	0x0000, 0x6b29, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b2a, 0x0000,
	0x6b2b, 0x6b2c, 0x6b2d, 0x0000, 0x6b2e, 0x0000, 0x6b2f, 0x0000,
	0x0000, 0x0000, 0x6b30, 0x6b31, 0x0000, 0x0000, 0x6b32, 0x6b33,
	0x6b34, 0x6b35, 0x6b36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6b37, 0x0000, 0x0000, 0x0000, 0x6b38, 0x6b39, 0x6b3a,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b3b, 0x0000, 0x0000,
	0x0000, 0x6b3c, 0x0000, 0x6b3d, 0x6b3e, 0x6b3f, 0x0000, 0x0000,
	0x0000, 0x6b40, 0x6b41, 0x0000, 0x0000, 0x0000, 0x6b42, 0x6b43,
	0x6b44, 0x0000, 0x0000, 0x6b45, 0x6b46, 0x0000, 0x6b47, 0x0000,
	0x6b48, 0x0000, 0x0000, 0x6b49, 0x6b50, 0x6b4a, 0x6b4b, 0x6b4c,
	0x0000, 0x0000, 0x0000, 0x6b4d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6b52, 0x6b4e, 0x6b4f, 0x6b51, 0x0000, 0x0000, 0x6b53, 0x0000,
	0x6b54, 0x0000, 0x6b55, 0x0000, 0x0000, 0x6b56, 0x0000, 0x6b57,
	0x0000, 0x0000, 0x0000, 0x6b58, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b59, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6b5a, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b5b,
	0x0000, 0x6b5c, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_9d[] = {
	/* 0x9d00 - 0x9dff */
	0x0000, 0x0000, 0x6b5e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6b5d, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6b5f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b60,
	0x6b61, 0x0000, 0x0000, 0x0000, 0x6b62, 0x6b63, 0x6b64, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b65,
	0x6b66, 0x0000, 0x6b67, 0x6b68, 0x6b69, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6b6a, 0x0000, 0x6b6b, 0x6b6d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6b6e, 0x6b6f, 0x0000, 0x6b6c, 0x0000, 0x6b70,
	0x0000, 0x0000, 0x6b71, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6b72, 0x6b73, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b74,
	0x0000, 0x0000, 0x6b76, 0x6b75, 0x0000, 0x6b77, 0x0000, 0x0000,
	0x0000, 0x6b78, 0x6b79, 0x6b7a, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6b7b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6b7c, 0x6b7d,
	0x0000, 0x0000, 0x0000, 0x6b7e, 0x6c21, 0x0000, 0x6c22, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6c23, 0x6c24, 0x0000, 0x6c25, 0x0000,
	0x0000, 0x0000, 0x6c26, 0x0000, 0x0000, 0x6c27, 0x6c28, 0x0000,
	0x0000, 0x0000, 0x6c29, 0x6c2a, 0x0000, 0x6c2b, 0x6c2c, 0x6c2d,
	0x6c2e, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6c2f, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c30, 0x0000, 0x6c31, 0x0000, 0x6c32, 0x0000,
	0x0000, 0x6c33, 0x0000, 0x0000, 0x0000, 0x6c34, 0x0000, 0x0000,
	0x0000, 0x6c35, 0x0000, 0x0000, 0x6c36, 0x0000, 0x0000, 0x6c37,
	0x0000, 0x0000, 0x0000, 0x6c38, 0x0000, 0x0000, 0x0000, 0x6c39,
	0x0000, 0x6c3a, 0x6c3b, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c3c, 0x6c3d, 0x6c3e, 0x6c3f,
	0x0000, 0x0000, 0x6c40, 0x0000, 0x0000, 0x0000, 0x6c41, 0x6c42,
	0x6c43, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c44, 0x0000, 0x6c45,
	0x0000, 0x6c46, 0x0000, 0x6c47, 0x0000, 0x0000, 0x6c48, 0x0000,
	0x6c49, 0x0000, 0x0000, 0x6c4a, 0x6c4b, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c4c, 0x0000,
};

static unsigned short const unicode_to_jisx0212_9e[] = {
	/* 0x9e00 - 0x9eff */
	0x0000, 0x0000, 0x6c4e, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c4f,
	0x0000, 0x0000, 0x6c4d, 0x0000, 0x0000, 0x0000, 0x6c50, 0x0000,
	0x6c51, 0x6c52, 0x6c53, 0x0000, 0x0000, 0x6c54, 0x6c55, 0x0000,
	0x0000, 0x6c56, 0x0000, 0x0000, 0x6c57, 0x6c58, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6c59, 0x6c5a, 0x6c5b, 0x0000, 0x0000, 0x0000,
	0x6c5c, 0x0000, 0x6c5d, 0x6c5e, 0x6c5f, 0x6c60, 0x0000, 0x6c61,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c62, 0x6c63,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c64, 0x0000,
	0x6c65, 0x0000, 0x0000, 0x6c66, 0x0000, 0x0000, 0x6c67, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c68, 0x0000, 0x0000, 0x0000,
	0x6c69, 0x0000, 0x0000, 0x0000, 0x6c6a, 0x0000, 0x6c6b, 0x6c6c,
	0x6c6d, 0x0000, 0x0000, 0x6c6e, 0x6c6f, 0x6c70, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c71, 0x0000,
	0x6c72, 0x0000, 0x0000, 0x6c73, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x747e, 0x0000, 0x0000, 0x0000, 0x6c74, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6c75,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c76, 0x0000, 0x0000, 0x6c77,
	0x0000, 0x0000, 0x0000, 0x0000, 0x6c78, 0x6c79, 0x6c7a, 0x0000,
	0x6c7b, 0x6c7c, 0x6c7d, 0x0000, 0x0000, 0x6c7e, 0x0000, 0x0000,
	0x6d21, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d22,
};

static unsigned short const unicode_to_jisx0212_9f[] = {
	/* 0x9f00 - 0x9fff */
	0x0000, 0x0000, 0x6d23, 0x6d24, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x6d25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d26,
	0x6d27, 0x6d28, 0x6d29, 0x0000, 0x6d2a, 0x0000, 0x6d2b, 0x6d2c,
	0x0000, 0x6d2d, 0x6d2e, 0x6d2f, 0x0000, 0x0000, 0x0000, 0x6d30,
	0x0000, 0x0000, 0x6d31, 0x0000, 0x0000, 0x0000, 0x6d32, 0x0000,
	0x0000, 0x0000, 0x6d33, 0x6d34, 0x0000, 0x0000, 0x0000, 0x6d35,
	0x0000, 0x6d36, 0x6d37, 0x0000, 0x6d38, 0x0000, 0x0000, 0x6d39,
	0x0000, 0x6d3a, 0x6d3b, 0x0000, 0x6d3c, 0x6d3d, 0x0000, 0x6d3e,
	0x0000, 0x6d3f, 0x0000, 0x6d40, 0x6d41, 0x6d42, 0x6d43, 0x6d44,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x6d45, 0x0000, 0x6d46, 0x6d47, 0x6d48,
	0x6d49, 0x0000, 0x6d4a, 0x0000, 0x0000, 0x6d4b, 0x6d4c, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x6d4d, 0x6d4e, 0x0000, 0x0000, 0x0000, 0x6d4f, 0x6d50, 0x6d51,
	0x6d52, 0x6d53, 0x0000, 0x6d54, 0x0000, 0x6d55, 0x0000, 0x0000,
	0x0000, 0x0000, 0x6d56, 0x0000, 0x0000, 0x6d57, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d58,
	0x6d59, 0x6d5a, 0x6d5b, 0x0000, 0x6d5c, 0x0000, 0x6d5d, 0x6d5e,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6d5f, 0x0000,
	0x0000, 0x6d60, 0x6d61, 0x6d62, 0x0000, 0x6d63, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_f9[] = {
	/* 0xf900 - 0xf9ff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x7445, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7472, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_fa[] = {
	/* 0xfa00 - 0xfaff */
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x7434, 0x7437,
	0x7438, 0x743d, 0x7444, 0x7447, 0x7448, 0x744e, 0x744f, 0x7453,
	0x7455, 0x7456, 0x7457, 0x7458, 0x745a, 0x745b, 0x745e, 0x7460,
	0x7462, 0x7463, 0x7465, 0x7469, 0x746a, 0x746b, 0x746d, 0x746f,
	0x7470, 0x7473, 0x7477, 0x7478, 0x7479, 0x747d, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const unicode_to_jisx0212_ff[] = {
	/* 0xff00 - 0xffff */
	0x0000, 0x0000, 0x742a, 0x0000, 0x0000, 0x0000, 0x0000, 0x7429,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
};

static unsigned short const *unicode_to_jisx0212_map[0x100] = {
	/* 0x00XX - 0x0fXX */
	unicode_to_jisx0212_00,
	unicode_to_jisx0212_01,
	unicode_to_jisx0212_02,
	unicode_to_jisx0212_03,
	unicode_to_jisx0212_04,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x10XX - 0x1fXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x20XX - 0x2fXX */
	0,
	unicode_to_jisx0212_21,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x30XX - 0x3fXX */
	0, 0,
	unicode_to_jisx0212_32,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0x40XX - 0x4fXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	unicode_to_jisx0212_4e,
	unicode_to_jisx0212_4f,
	/* 0x50XX - 0x5fXX */
	unicode_to_jisx0212_50,
	unicode_to_jisx0212_51,
	unicode_to_jisx0212_52,
	unicode_to_jisx0212_53,
	unicode_to_jisx0212_54,
	unicode_to_jisx0212_55,
	unicode_to_jisx0212_56,
	unicode_to_jisx0212_57,
	unicode_to_jisx0212_58,
	unicode_to_jisx0212_59,
	unicode_to_jisx0212_5a,
	unicode_to_jisx0212_5b,
	unicode_to_jisx0212_5c,
	unicode_to_jisx0212_5d,
	unicode_to_jisx0212_5e,
	unicode_to_jisx0212_5f,
	/* 0x60XX - 0x6fXX */
	unicode_to_jisx0212_60,
	unicode_to_jisx0212_61,
	unicode_to_jisx0212_62,
	unicode_to_jisx0212_63,
	unicode_to_jisx0212_64,
	unicode_to_jisx0212_65,
	unicode_to_jisx0212_66,
	unicode_to_jisx0212_67,
	unicode_to_jisx0212_68,
	unicode_to_jisx0212_69,
	unicode_to_jisx0212_6a,
	unicode_to_jisx0212_6b,
	unicode_to_jisx0212_6c,
	unicode_to_jisx0212_6d,
	unicode_to_jisx0212_6e,
	unicode_to_jisx0212_6f,
	/* 0x70XX - 0x7fXX */
	unicode_to_jisx0212_70,
	unicode_to_jisx0212_71,
	unicode_to_jisx0212_72,
	unicode_to_jisx0212_73,
	unicode_to_jisx0212_74,
	unicode_to_jisx0212_75,
	unicode_to_jisx0212_76,
	unicode_to_jisx0212_77,
	unicode_to_jisx0212_78,
	unicode_to_jisx0212_79,
	unicode_to_jisx0212_7a,
	unicode_to_jisx0212_7b,
	unicode_to_jisx0212_7c,
	unicode_to_jisx0212_7d,
	unicode_to_jisx0212_7e,
	unicode_to_jisx0212_7f,
	/* 0x80XX - 0x8fXX */
	unicode_to_jisx0212_80,
	unicode_to_jisx0212_81,
	unicode_to_jisx0212_82,
	unicode_to_jisx0212_83,
	unicode_to_jisx0212_84,
	unicode_to_jisx0212_85,
	unicode_to_jisx0212_86,
	unicode_to_jisx0212_87,
	unicode_to_jisx0212_88,
	unicode_to_jisx0212_89,
	unicode_to_jisx0212_8a,
	unicode_to_jisx0212_8b,
	unicode_to_jisx0212_8c,
	unicode_to_jisx0212_8d,
	unicode_to_jisx0212_8e,
	unicode_to_jisx0212_8f,
	/* 0x90XX - 0x9fXX */
	unicode_to_jisx0212_90,
	unicode_to_jisx0212_91,
	unicode_to_jisx0212_92,
	unicode_to_jisx0212_93,
	unicode_to_jisx0212_94,
	unicode_to_jisx0212_95,
	unicode_to_jisx0212_96,
	unicode_to_jisx0212_97,
	unicode_to_jisx0212_98,
	unicode_to_jisx0212_99,
	unicode_to_jisx0212_9a,
	unicode_to_jisx0212_9b,
	unicode_to_jisx0212_9c,
	unicode_to_jisx0212_9d,
	unicode_to_jisx0212_9e,
	unicode_to_jisx0212_9f,
	/* 0xa0XX - 0xafXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xb0XX - 0xbfXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xc0XX - 0xcfXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xd0XX - 0xdfXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xe0XX - 0xefXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	/* 0xf0XX - 0xffXX */
	0, 0, 0, 0, 0, 0, 0, 0, 0,
	unicode_to_jisx0212_f9,
	unicode_to_jisx0212_fa,
	0, 0, 0, 0,
	unicode_to_jisx0212_ff,
};

#endif

#ifdef USE_JISX0212
static uint Unicode11ToJisx0212(uint h, uint l)
{
    unsigned short const *table;

    table = unicode_to_jisx0212_map[h];
    if (table != 0) {
	return table[l];
    }
    return 0x0000;
}
#else
static uint Unicode11ToJisx0212(uint h, uint l)
{
    return 0x0000;
}
#endif

// and now for the inlines:

/*! \fn uint  QJpUnicodeConv::AsciiToUnicode (uint ascii) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::Jisx0201ToUnicode (uint jis) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::Jisx0201LatinToUnicode (uint jis) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::Jisx0201KanaToUnicode (uint jis) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::Jisx0208ToUnicode (uint jis) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::Jisx0212ToUnicode (uint jis) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToAscii (uint unicode) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToJisx0201 (uint unicode) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToJisx0201Latin (uint unicode) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToJisx0201Kana (uint unicode) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToJisx0208 (uint unicode) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToJisx0212 (uint unicode) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::SjisToUnicode (uint sjis) const

\internal
*/

/*! \fn uint  QJpUnicodeConv::UnicodeToSjis (uint unicode) const

\internal
*/

#endif
