/*****************************************************************************\
*                                                                             *
* 			   KLONE OBJECT MODEL FOR C                           *
* 				implemetation                                 *
*                                                                             *
\*****************************************************************************/
/*****************************************************************************\
*                                                                             *
* Copyright (C) 1989-94 GROUPE BULL                                           *
*                                                                             *
* Permission is hereby granted, free of charge, to any person obtaining a     *
* copy of this software and associated documentation files (the "Software"),  *
* to deal in the Software without restriction, including without limitation   *
* the rights to use, copy, modify, merge, publish, distribute, sublicense,    *
* andor sell copies of the Software, and to permit persons to whom the        *
* Software is furnished to do so, subject to the following conditions:        *
*                                                                             *
* The above copyright notice and this permission notice shall be included in  *
* all copies or substantial portions of the Software.                         *
*                                                                             *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  *
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    *
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL     *
* GROUPE BULL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN *
* AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN        *
* CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.  *
*                                                                             *
* Except as contained in this notice, the name of GROUPE BULL shall not be    *
* used in advertising or otherwise to promote the sale, use or other dealings *
* in this Software without prior written authorization from GROUPE BULL.      *
*                                                                             *
\*****************************************************************************/

/* Author:
 * Colas NAHABOO        Koala Project, User Interfaces, BULL Research FRANCE
 * email: colas@sophia.inria.fr
 * surface mail:
 * INRIA
 * BP 93
 * 06902 Sophia Antipolis cedex
 * FRANCE
 * Tel: (33) 93 65 77 70
 * Fax: (33) 93 65 77 65
 * WWW: http://zenon.inria.fr:8003/koala/colas.html
 */

#ifdef EXT
#undef EXT
#endif
#define EXT
#ifdef INIT
#undef INIT
#endif
#define INIT(x) = x
#define DO_INIT
#define PRIVATE_DEFINITIONS

#include "KlLib.h"

/*****************************************************************************\
*                                                                             *
* 			     Internal Klone model                             *
*                                                                             *
\*****************************************************************************/

/************************************************************** traits names */

int
KlDeclareNewTrait(name)
    char *name;
{
    KlTraitNames = (char **) Realloc(KlTraitNames, (KlLastTrait + 1)
				     * sizeof(char *));
    KlTraitNames[KlLastTrait] = name;
    return 1 << (KlLastTrait++);
}

KlDeclareBuiltInTraits()
{
    char **sp = KlTraitNames = (char **) Malloc(KlLastTrait * sizeof(char *));

    *sp++ = "list";
}

/*****************************************************************************\
* 			    internal object layer                             *
\*****************************************************************************/
/* a type is an array of methods. first methods are in fact class variables
 * which are:
 * klone header: type and refcount
 * - the name of type (atom)
 * - a pointer to next type. This maintains a list of all known types
 * pointed to by KlTypes
 * - the traits: inheritance flags (32 1-bit flags in a 32 bit word) telling 
 * which "trait" this type has
 * - the trait this type may define
 * - then methods, first one must be "coerce"
 * pseudo-types stop here, normal types go on with the rest of the methods 
 * array
 */


KlMethod *KlTypeTemplate = 0;

/* KlDeclareTypeSlot
 * Declares an application-dependent type slot before the methods array
 * name is stored in the method array
 */

int
KlDeclareTypeSlot(initial_value, name)
    int initial_value;
    char *name;
{
    int position = KlDeclareSelector(0, name);

    KlTypeTemplate[position] = (KlMethod) initial_value;
    KlTypeReservedSlots = KlSelectorsSize;
    return position;
}

/* KlDeclareSelector
 * declare a new selector 
 * returns allocated position.
 * name must not freed afterwards, pointed to by structure
 * the arity (0 <= arity < KlSelectorsPropsSize(6 by default)) can be given
 * arity is 0 for n-ary methods (argc, argv), but NARY (-1) is accepted too
 */

int
KlDeclareSelector(arity, name)
    int arity;
    char *name;
{
    int position;

    if (arity == NARY)
	arity = 0;

    if (arity < 0 || arity >= KlSelectorsPropsSize)
	KlFatalError(3, name);

    position = KlSelectorsSize++;
    KlSelectors = (struct _KlSelector *)
	Realloc(KlSelectors, sizeof(struct _KlSelector) * KlSelectorsSize);
    KlSelectors[position].arity = arity;
    KlSelectors[position].name = name;

    KlTypeTemplate = (KlMethod *)
	Realloc(KlTypeTemplate, sizeof(KlMethod) * KlSelectorsSize);
    KlTypeTemplate[position] = KlSelectorUndefmethod(position);

    return position;
}

/* KlDeclareSelectorArity
 * Declares an Undefined Method and other things for a given selector arity
 * number of params may grow in the future
 * last param MUST be 0 in order to check for version mismatches
 */

KlDeclareSelectorArity(arity, undefmethod, hooker, bypass_once, zero)
    int arity;
    KlMethod undefmethod;
    KlMethod hooker;
    KlMethod bypass_once;
    int zero;
{
    if (zero)
	KlFatalError(13, "obsolete form of calling");
    if (arity != KlSelectorsPropsSize) { /* leave no holes! */
	char tmp[80];
	sprintf(tmp, "declared %d, expected %d!", arity, KlSelectorsPropsSize);
	KlFatalError(13, tmp);
    }

    KlSelectorsPropsSize++;
    KlSelectorsProps = (struct _KlSelectorProps *)
	Realloc(KlSelectorsProps,
		KlSelectorsPropsSize * sizeof (struct _KlSelectorProps));

    KlSelectorsProps[arity].undefmethod = undefmethod;
    KlSelectorsProps[arity].hooker = hooker;
    KlSelectorsProps[arity].bypass_once = bypass_once;
}


/* KlGetSelectorByName
 * searches a selector by its name, returns its position or 0 if not found
 */

int
KlGetSelectorByName(name)
    char *name;
{
    int i;

    for (i = KlTypeReservedSlots; i < KlSelectorsSize; i++) {
	if (!strcmp(name, KlSelectors[i].name)) {
	    return i;
	}
    }
    return 0;
}

/* KlInitPredefinedSelectors
 * used to pre-declare built-in selectors
 */

KlInitPredefinedSelectors()
{
    /* fill private selector slots with 0 */
    KlSelectors = (struct _KlSelector *)
	Calloc(sizeof(struct _KlSelector), KlSelectorsSize);

    KlSelectorsPropsSize = 6;
    KlSelectorsProps = (struct _KlSelectorProps *)
	Malloc(KlSelectorsPropsSize * sizeof(struct _KlSelectorProps));

    KlSelectorsProps[0].undefmethod = KlUndefinedMethodNary;
    KlSelectorsProps[1].undefmethod = KlUndefinedMethod1;
    KlSelectorsProps[2].undefmethod = KlUndefinedMethod2;
    KlSelectorsProps[3].undefmethod = KlUndefinedMethod3;
    KlSelectorsProps[4].undefmethod = KlUndefinedMethod4;
    KlSelectorsProps[5].undefmethod = KlUndefinedMethod5;

    KlTypeTemplate = (KlMethod *)
	Calloc(sizeof(KlMethod), KlTypeReservedSlots);
    KlSelFree = KlDeclareSelector(1, "free");
    KlSelEqual = KlDeclareSelector(2, "=");
    KlSelMake = KlDeclareSelector(1, "init");
    KlSelCopy = KlDeclareSelector(1, "copy");

    /* default methods */

    KlTypeTemplate[KlSelEqual] = KlEq;	/* equal is eq */
    KlTypeTemplate[KlSelFree] = KlKlOFree; /* free chunk itself */
}

/* KlDeclareMethod
 * declares an internal method, and propagates to the subtypes
 */

KlDeclareMethod(type, selector, method)
    KlType type;
    int selector;
    KlMethod method;
{
    KlType t;
    KlMethod oldmethod = type[selector];

    KlDeclareMethod1(type, selector, method);
    if (type == KlTypes) 		/* last type defined ==> no children */
	return;
    /* propagate */
    for (t = KlTypes; t; t = KlTypeNextGet(t)) {
	/* finds all direct sons of type with same old method */
	if (KlTypeFatherGet(t) == type
	    && t[selector] == oldmethod) {
	    KlDeclareMethod(t, selector, method); /* recurse */
	}
    }
}

CFAPPLY_ERROR()
{
    fprintf(stderr, "\nINTERNAL ERROR: applying a NULL C Function!\n");
    stop_if_in_dbx("NULL C FUNCTION POINTER");
}
/*****************************************************************************\
* 				type functions                                *
\*****************************************************************************/

/* KlTypesInit
 * bootstrap: builds the Type Type (first type)
 * its name field will have to be filled by KlAtomInit
 */

KlTypesInit()
{
    KlAnyType = (KlType) Malloc(sizeof(KlMethod) * KlSelectorsSize);
    KlTypeType = (KlType) Malloc(sizeof(KlMethod) * KlSelectorsSize);

    KlTypeTypeSet(KlTypeTemplate, KlTypeType);
    bcopy(KlTypeTemplate, KlAnyType, sizeof(KlMethod) * KlSelectorsSize);
    bcopy(KlTypeTemplate, KlTypeType, sizeof(KlMethod) * KlSelectorsSize);
    if (KlTypes) {
	KlFatalError(9, KlTypeCName(KlTypes));
    }
    KlTypes = KlTypeType;
    KlTypeNextSet(KlTypeType, KlAnyType);
    KlTypeFatherSet(KlTypeType, KlAnyType);

    /* no free method! */
    KlTypeNumCurrent += 2;
    
    KlCoerceTable = (KlMethod **) Calloc(KlTypeNumCurrent, sizeof(KlMethod *));
    KlCoerceTableSizes = (int *) Malloc(sizeof(int *) * 15);
    KlCoerceTableSizes[0] = 0;
}

/* KlDeclareSubType
 * to declare a type inheriting traits and methods from another
 * to be called after defining all the public methods of main type
 * copies all traits and methods of parent type
 * size is the size in bytes or 0 if variable-type size
 *
 * KlDeclareType is now a macro calling KlDeclareSubType with parent = 0
 */

KlDeclareSubType(typep, name, parent, size)
    KlType *typep;
    char *name;
    KlType parent;
    int size;
{
    KlType type = *typep = (KlType) Malloc(sizeof(KlMethod) * KlSelectorsSize);

    if (parent) {
	bcopy(parent, type, sizeof(KlMethod) * KlSelectorsSize);
	KlTypeTraitReset(type);
	KlTypeFatherSet(type, parent);
    } else {
	bcopy(KlTypeTemplate, type, sizeof(KlMethod) * KlSelectorsSize);
	KlTypeFatherSet(type, KlAnyType);
    }

    KlTypeSizeSet(type, KlMallocBucketOfSize(size));
    KlTypeNameSet(type, name);
    KlTypeNextSet(type, KlTypes);
    KlTypes = type;
}

/* KlTypeIsSonOf
 * walks up the primitive type hierarchy
 * Note: does not checks if type is == to type
 */

int
KlTypeIsSonOf(son, father)
    KlType son;
    KlType father;
{
    while (KlTypeFatherGet(son)) {
	if ((son = KlTypeFatherGet(son)) == father) {
	    return 1;
	}
    }
    return 0;
}

/**************************************************************************\
* 				  Extensions                               *
\**************************************************************************/

int
KlExtensionIsDeclared(name)
    char *name;
{
    int i;
    for (i = 0; i < KlExtensionsSize; i++) {
	if (!strcmp(KlExtensions[i].name, name)) {
	    return i+1;
	}
    }
    return 0;
}

KlDeclareExtension(name, selectors_initfunc, types_initfunc, profile_initfunc)
    char *name;				/* name of extension */
    int (*selectors_initfunc)();	/* init of selectors */
    int (*types_initfunc)();		/* init of types */
    int (*profile_initfunc)();		/* klone level inits */
{
    int i = KlExtensionIsDeclared(name);
    if (i) {
	i--;
    } else {
	i = KlExtensionsSize;
	if (KlExtensionsSize) {
	    KlExtensionsSize++;
	    KlExtensions = (KlExtension)
		Realloc(KlExtensions, KlExtensionsSize *
			sizeof(struct _KlExtension));
	} else {
	    KlExtensionsSize++;
	    KlExtensions = (KlExtension) Malloc(sizeof(struct _KlExtension));
	}
    }
    KlExtensions[i].name = name;
    KlExtensions[i].selectors = selectors_initfunc;
    KlExtensions[i].types = types_initfunc;
}

/*****************************************************************************\
* 				   methods                                    *
\*****************************************************************************/
KlO
KlEq(o1, o2)
    KlO o1, o2;
{
    if (o1 == o2)
	return TRU;
    else
	return NIL;
}

KlO
KlQuote(obj)
    KlO obj;
{
    return obj;
}

KlO
KlKlOFree(obj)
    KlO obj;
{
    Free(obj);
    return (KlO) obj;
}

/*****************************************************************************\
*                                                                             *
* 				    errors                                    *
*                                                                             *
\*****************************************************************************/

KlO
KlErrorUndefinedMethod(object)
    KlO object;
{
    fprintf(stderr, 
	    "ERROR: \"%s\" not defined for object 0x%lx (of type %s)",
	    KlSelectors[KlCurrentMessage].name, object, KlTypeCName(object));
    return object;
}

KlO
KlUndefinedMethodNary(argc, argv)
    int argc;
    KlO argv[];
{
    return KlErrorUndefinedMethod(argv[0]);
}

KlO
KlUndefinedMethod1(object)
    KlO object;
{
    return KlErrorUndefinedMethod(object);
}

/*ARGSUSED*/
KlO
KlUndefinedMethod2(object, param1)
    KlO object;
    KlO param1;
{
    return KlErrorUndefinedMethod(object);
}

/*ARGSUSED*/
KlO
KlUndefinedMethod3(object, param1, param2)
    KlO object;
    KlO param1;
    KlO param2;
{
    return KlErrorUndefinedMethod(object);
}

/*ARGSUSED*/
KlO
KlUndefinedMethod4(object, param1, param2, param3)
    KlO object;
    KlO param1;
    KlO param2;
    KlO param3;
{
    return KlErrorUndefinedMethod(object);
}

/*ARGSUSED*/
KlO
KlUndefinedMethod5(object, param1, param2, param3, param4)
    KlO object;
    KlO param1;
    KlO param2;
    KlO param3;
    KlO param4;
{
    return KlErrorUndefinedMethod(object);
}


/* bad argument call
 */

KlO
KlBadArgument(argument, position, expecting)
    KlO argument;
    int position;
    char *expecting;
{
    fprintf(stderr, 
	    "ERROR: Bad argument: 0x%lx (of type %s, at position %d), expecting a %s",
	    argument, KlTypeCName(argument), position, expecting);
    return argument;
}

char *KlFatalErrorMessages[] = {
    "",					/* 0 */
    "",					/* 1 */
    "",					/* 2 */
    "KlDeclareSelector(%s): bad arity, use KlDeclareSelectorArity",/* 3 */
    "",					/* 4 */
    "",					/* 5 */
    "",					/* 6 */
    "",					/* 7 */
    "",					/* 8 */
    "KlTypesInit(): a type (%s) is already defined", /* 9 */
    "",					/* 10 */
    "",					/* 11 */
    "",					/* 12 */	
    "KlDeclareSelectorArity error: %s!", /* 13 */
    ""					/* 14 */
};


int
KlFatalError(code, data)
int code;
char *data;
{
    fprintf(stderr, "FATAL Klone error #%d, aborting: ", code);
    fprintf(stderr, KlFatalErrorMessages[code], data);
    putc('\n', stderr);
    fflush(stderr);
    stop_if_in_dbx(KlFatalErrorMessages[code]);
    if (code == 3)			/* me MUST not flush buffers */
	_exit(code);
    else
	exit(code);
}


/*****************************************************************************\
*                                                                             *
* 				      GC                                      *
*                                                                             *
\*****************************************************************************/
/*
 * The memory management of KLONE is implemented via a differed reference
 * count. That, each time an object's reference count attains 0, it is put in
 * the KlZrt, which is polled at regular intervals
 */

/*************************************\
*                                     *
* Zero_reference table module (KlZrt) *
*                                     *
\*************************************/

/*
 * The KlZrt (Zero Reference Table) global structure  is used to mark ALL wobs
 * that have at any moment be of KlRef 0, that is either being created or via
 * KlDecRef.  Then you can call KlZrtGc at strategic moments, (ie in
 * no enclosing KLONE function) to free all the zero-referenced objects in the
 * KlZrt.
 */

KlZrtInit()
{
    KlZrtSizeLimit = 63;			/* pow(2,n)/4 -1 */
    KlZrt = (KlO *) Malloc(KlZrtSizeLimit * sizeof(KlO));
    KlZrtLast = KlZrt;
    KlZrtLimit = KlZrt + KlZrtSizeLimit;
    KlZrtFrom = 0;
}

/* disposes really of objects stacked in KlZrt. 
 * Be warned that a KlSelFree might trigger KlZrtPut during the KlZrtGc !
 *
 * KlZrtPut may trigger reallocation of Zrt, so KlZrtFrom and KlZrtLast 
 * are globals so they can be updated transparently
 */

KlZrtGc(from)
    Int from;
{
    if (KlZrtFrom) {
	/* we are already GCing. We cannot thus us KlZrtFrom which is global,
	 * but we must take care of possible reallocs of the Zrt, moving it
	 */
	while (KlZrtLast > KlZrt + from) {
	    if ((*(--KlZrtLast))->reference_count) {
		/* somebody claimed it, ok, graduate to normal object */
		(*KlZrtLast)->reference_count |= 1;
	    } else {
		/* last call! Nobody? Ok, smithe it */
		KlSend_free(*KlZrtLast);
	    }
	}
    } else {				/* toplevel GC */
	KlZrtFrom = KlZrt + from;
	while (KlZrtLast > KlZrtFrom) {
	    if ((*(--KlZrtLast))->reference_count) {
		/* somebody claimed it, ok, graduate to normal object */
		(*KlZrtLast)->reference_count |= 1;
	    } else {
		/* last call! Nobody? Ok, smithe it */
		KlSend_free(*KlZrtLast);
	    }
	}
	KlZrtFrom = 0;			/* to indicate we are done */
    }
}

/*
 * Never call KlZrtPut if obj was already in it (should not happen)
 */

KlZrtPut(obj)
    KlO obj;
{
    if (KlZrtLast >= KlZrtLimit) {
	KlO *oldZrt = KlZrt;
	KlZrtSizeLimit = (KlZrtSizeLimit + 1) * 2 - 1;
	KlZrt = (KlO *) Realloc(KlZrt, KlZrtSizeLimit * sizeof(KlO));
	KlZrtLast = KlZrt + (KlZrtLast - oldZrt);
	if (KlZrtFrom)
	    KlZrtFrom = KlZrt + (KlZrtFrom - oldZrt);
	KlZrtLimit = KlZrt + KlZrtSizeLimit;
    }
    (*(KlZrtLast++) = obj)->reference_count = 0;
}


/***********************\
* 		        *
* reference management  *
* 		        *
\***********************/

/* KlIncRef is a macro (KlRef(x)++) */

#ifdef DEBUGREF				/* macro otherwise */
KlIncRef(obj)
    KlO obj;				/* obj may be KlUndef */
{
    KlRef(obj) += 2;
}

KlDecRef(obj)
    KlO obj;				/* obj may be KlUndef */
{
    if (obj) {
	KlDecRefNonNull(obj);
    }
}

KlDecRefNonNull(obj)
    KlO obj;				/* obj must be non-nil */
{
    if (((obj->reference_count) -= 2) == 1)
	KlZrtPut(obj);
    else if (obj->reference_count < 0) {
	printf("INTERNAL ERROR: negative reference_count: %d, obj 0x%x\n",
	       obj->reference_count, obj);
	stop_if_in_dbx("negative reference_count!");
	/* if we are not in dbx, try to fix things and continue */
	if (KlIsActuallyInZrt(obj)) {
	    obj->reference_count = 0;
	} else {
	    KlZrtPut(obj);
	}
    }
}

int
KlIsActuallyInZrt(obj)
    KlO obj;
{
    KlO *zrt_ptr = KlZrt;
    KlO *zrtlast = KlZrtLast;

    while (zrt_ptr < zrtlast) {
	if (*zrt_ptr == obj) {
	    return 1;
	}
	zrt_ptr++;
    }
    return 0;
}


#endif					/* DEBUGREF */

/*
 * KlDecRefList:
 * decrease reference count of all the elements of the list.
 * but doesn't free the list.
 */

KlDecRefList(count, list)
    int count;
    KlO *list;
{
    KlO *last = list + count;

    while (list < last) {
	KlDecRef(*list);
	list++;
    }
}

/*
 * duplicate an array of objects, increasing the reference count,
 * and mallocing
 */

KlDuplicateNObjects(source, dest, n)
    KlO *source;			/* source is the array */
    KlO **dest;				/* while dest is a POINTER to the
					 * array */
    int n;				/* how many to copy */
{
    KlO *p = source, *q, *last = source + n;

    q = *dest = (KlO *) Malloc(sizeof(KlO) * n);
    while (p < last)
	KlIncRef(*q++ = *p++);
}

/*
 * duplicate an array of objects, increasing the reference count,
 * without mallocing (dest already points to an malloced aera)
 */

KlCopyNObjects(source, dest, n)
    KlO *source;			/* source is the array */
    KlO *dest;				/* dest is  the array */
    int n;				/* how many to copy */
{
    KlO *p = source, *q = dest, *last = source + n;

    while (p < last)
	KlIncRef(*q++ = *p++);
}

/* KlSetField
 * sets a Klone object to a memeory location, managing the reference counts
 */

void
KlSetField(ptr, value)
    KlO *ptr;
    KlO value;
{
    KlDecRef(*ptr);
    KlIncRef(*ptr = value);
}


/*****************************************************************************\
*                                                                             *
* 				    Lists                                     *
*                                                                             *
\*****************************************************************************/
/* makes a list with size slots, to be filled by the C programmer */

KlList
KlListNMake(size)
    int size;
{
    KlList kl_list;

    kl_list = (KlList) KlOMake(KlListType);

    kl_list->list = (size > 0 ? (KlO *) Malloc(size * sizeof(KlO)) :
	(KlO *) (size = 0));

    kl_list->size = size;
    return kl_list;
}

/*
 * freeing a list decreases the reference count of all the elements before
 * freeing itself!
 */

KlO
KlListFree(obj)
    KlList obj;
{
    if (obj->list) {
	KlO *last = obj->list + obj->size, *list = obj->list;

	while (list < last) {
	    KlDecRef(*list);
	    list++;
	}
	Free(obj->list);
    }
    Free(obj);
    return (KlO) obj;
}

/*
 * equality of two lists is equality of their elements
 */

KlO
KlListEqual(l1, l2)
    KlList l1, l2;
{
    int i;

    if (!KlIsAList(l2)			/* we know l1 is a list or vector */
	|| (l2->size != l1->size))
	return NIL;
    for (i = 0; i < l1->size; i++) {
	if (KlSend_equal(l1->list[i], l2->list[i]) == NIL)
	    return NIL;
    }
    return (KlO) (l1->size ? (KlO) l1 : TRU);
}

KlO
KlListDeleteC(list, i)
    KlList list;
    unsigned int i;
{
    KlO *p, *end;

    if (i < list->size) {
	KlDecRef(list->list[i]);
	list->size--;
	p = list->list + i;
	end = list->list + list->size;
	while (p < end) {
	    *p = *(p+1);
	    p++;
	}
    }
    return (KlO) list;
}

KlO
KlListAppend(list, obj)
    KlList list;
    KlO obj;
{
    list->size++;
    list->list = (KlO *) Realloc(list->list, KLSO * list->size);
    KlIncRef(list->list[list->size - 1] = obj);
    return (KlO) list;
}

KlListInit()
{
    KlDeclareType(&KlListType, "List", sizeof(struct _KlList));
    KlDeclareIsTrait(KlListType, KlTrait_list);

    KlDeclareMethod1(KlListType, KlSelFree, KlListFree);
    KlDeclareMethod1(KlListType, KlSelEqual, KlListEqual);

    /* create NIL */
    NIL = (KlO) KlOMake(KlListType);
    ((KlList) NIL)->size = 0;
    ((KlList) NIL)->list = 0;
    KlIncRef(NIL);
    TRU = (KlO) KlListNMake(1);
    KlIncRef(((KlList) TRU)->list[0] = NIL);
    KlIncRef(TRU);
}

/*****************************************************************************\
*                                                                             *
* 				    Inits                                     *
*                                                                             *
\*****************************************************************************/

int
KlInit()
{
    int i;

    KlDeclareBuiltInTraits();
    KlInitPredefinedSelectors();	/* built-ins */
    for (i = 0; i < KlExtensionsSize; i++) /* extensions */
	if (KlExtensions[i].selectors)
	    CFAPPLY((KlExtensions[i].selectors), ());
    KlZrtInit();
    KlTypesInit();
    KlListInit();
    for (i = 0; i < KlExtensionsSize; i++) /* extensions */
	if (KlExtensions[i].types)
	    CFAPPLY((KlExtensions[i].types), ());
    if (KlExtensionsSize)
	Free(KlExtensions);
    KlZrtGc(0);
    return 0;
}
