/*
 * cmd_loop.c - Primary read-eval-print loop (heart of system)
 *
 * Copyright (C) 1995, 1996 by Scott C. Gray
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, write to the Free Software
 * Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * You may contact the author :
 *   e-mail:  gray@voicenet.com
 *            sgray@sybase.com
 *            gray@xenotropic.com
 */
#include <stdio.h>
#include "sqsh_config.h"
#include "sqsh_global.h"
#include "sqsh_varbuf.h"
#include "sqsh_env.h"
#include "sqsh_error.h"
#include "sqsh_cmd.h"
#include "sqsh_job.h"
#include "sqsh_getopt.h"
#include "sqsh_readline.h"
#include "cmd.h"
#include "cmd_misc.h"
#include "cmd_input.h"

/*-- Current Version --*/
#if !defined(lint) && !defined(__LINT__)
static char RCS_Id[] = "$Id: cmd_loop.c,v 1.21 1999/01/06 02:21:27 gray Exp $" ;
USE(RCS_Id)
#endif /* !defined(lint) */

/*
 * cmd_loop():
 */
int cmd_loop( argc, argv )
	int    argc ;
	char  *argv[] ;
{
	char        *interactive ;        /* Value of $interactive */
	FILE        *input_file ;         /* Where input is coming from */
	int          old_interactive ;
	int          new_interactive ;
	int          read_flags ;
	int          ret ;
	int          exit_status ;

	/*-- Variables settable by command line options --*/
	char        *file_name     = NULL ;
	varbuf_t    *sql_buf       = NULL ;
	int          ignore_nofile = False ;
	int          do_connect    = True ;
	int          have_error    = False ;

	/*-- Variables required by sqsh_getopt --*/
	int          c ;
	extern int   sqsh_optind ;
	extern char *sqsh_optarg ;

	/*
	 * Parse the command line options.  Currently there aren't many
	 * of them, just one.
	 */
	while ((c = sqsh_getopt( argc, argv, "ine:" )) != EOF)
	{
		switch (c)
		{
			case 'i' :
				ignore_nofile = True;
				break;
			case 'n' :
				do_connect = False;
				break;
			case 'e' :
				do_connect = False;

				if (sql_buf == NULL)
				{
					if ((sql_buf = varbuf_create( strlen(sqsh_optarg) + 6 )) == NULL)
					{
						fprintf( stderr, "varbuf_create: %s\n", sqsh_get_errstr() );
						return CMD_FAIL;
					}
				}

				varbuf_strcat(sql_buf, sqsh_optarg);
				varbuf_strcat(sql_buf, "\n");

				break;
			default :
				fprintf(stderr, "\\loop: %s\n", sqsh_get_errstr() );
				have_error = True;
		}
	}

	/*
	 * If -e was supplied, then automagically stick a \go
	 * on the end of the whole mess.
	 */
	if (sql_buf != NULL)
	{
		varbuf_strcat( sql_buf, "\\go\n" );
	}

	/*
	 * If there are any arguments left on the command line, then 
	 * we have an error.
	 */
	if( (argc - sqsh_optind) > 1 || have_error ) {
		fprintf( stderr, "Use: \\loop [-i] [-n] [-e cmd | file]\n" ) ;
		return CMD_FAIL ;
	}

	if( argc != sqsh_optind )
	{
		if (sql_buf)
		{
			fprintf( stderr, "\\loop: Cannot specify -e option and a file name.\n" ) ;
			return CMD_FAIL ;
		}
		file_name = argv[sqsh_optind] ;
	}

	env_get( g_env, "interactive", &interactive ) ;
	if( interactive == NULL || *interactive == '0' )
		new_interactive = old_interactive = 0 ;
	else
		new_interactive = old_interactive = 1 ;
	
	/*
	 * If we are currently marked as being in interactive mode
	 * then we need to check for the various things that could
	 * kick us out of this mode, such as if our stdin was redirected
	 * or if we are to read from a file.
	 */
	if( old_interactive ) {
		/*
		 * As a little setup we need to determine if we are in interactive
		 * mode or not.  This is particularly useful if anyone ever does
		 * something like "\loop < file.sql".
		 */
		if( file_name != NULL || !isatty(fileno(stdin)) || sql_buf != NULL) {
			env_set( g_env, "interactive", "0" ) ;
			new_interactive = 0 ;
		}
	}

	/*
	 * Unless we are asked explicitly not to connect to the database,
	 * we run the connect command to do so.
	 */
	if( do_connect ) {
		if( (jobset_run( g_jobset, "\\connect", &exit_status )) == -1 ) {
			fprintf( stderr, "\\loop: %s\n", sqsh_get_errstr() ) ;
			return CMD_FAIL ;
		}
		if( exit_status == CMD_FAIL )
			return CMD_FAIL ;
	}

	/*
	 * If the user requests that we read from a file, then we need
	 * to open the file to be read from and set the interactive_flag
	 * to false to make sure that we don't print the prompt.
	 */
	if( file_name != NULL ) {
		if( (input_file = fopen( (char*)file_name, "r" )) == NULL ) {
			if( ignore_nofile )
				return CMD_LEAVEBUF ;
			fprintf( stderr, "\\loop: %s: %s\n", (char*)file_name,
						strerror( errno ) ) ;
			return CMD_FAIL ;
		}
	} else {
		input_file = NULL ;
	}

	/*-- Test for job completion --*/
	read_flags = CMD_F_JOBS ;

	/*-- If interactive mode, prompt user --*/
	if( new_interactive )
		read_flags |= CMD_F_PROMPT ;
	
	/*
	 * Ok, now read input from the user(?), fortunately, this takes care
	 * of dealing with interrupts for us, so we don't need to worry about
	 * them.
	 */
	ret = cmd_input( input_file, varbuf_getstr(sql_buf), read_flags ) ;

	/*
	 * Destroy the SQL buffer used by the -e command
	 */
	if ( sql_buf != NULL )
		varbuf_destroy(sql_buf);

	/*
	 * If we opened a file to read, then close it. There is a brief period
	 * here where an interrupt could trash us, but lets hope for the best.
	 */
	if( file_name != NULL )
		fclose( input_file ) ;

	/*
	 * Restore the interactive indicator.  If the indicator has been
	 * changed by this call it is because it was turned off.
	 */
	if( old_interactive != new_interactive )
		env_set( g_env, "interactive", "1" ) ;
	
	return ret ;
}
