/* $Id: proto_fics.cc,v 1.77 2002/04/28 21:04:34 bergo Exp $ */

/*

    eboard - chess client
    http://eboard.sourceforge.net
    Copyright (C) 2000-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "protocol.h"
#include "global.h"
#include "chess.h"
#include "status.h"
#include "seekgraph.h"

// <b1> game 45 white [PPPPPNBBR] black [PPNN]

FicsProtocolParser::FicsProtocolParser() {
  SetPat *sp;

  global.debug("FicsProtocolParser","FicsProtocolParser");

  GActions.push_back(new string("All Observers"));
  GActions.push_back(new string("Game Info"));

  PActions.push_back(new string("Finger"));
  PActions.push_back(new string("Stats"));
  PActions.push_back(new string("History"));
  PActions.push_back(new string("Ping"));
  PActions.push_back(new string("Log"));

  AuthenticationGone=0;
  LectureBotSpecial=false;
  LastWasPrompt=0;
  LastWasCTell=-1;

  RetrievingMoves=0;
  RetrievingGameList=0;
  RetrievingAdList=0;
  PendingStatus=0;
  LastWasStyle12=0;
  LastColor=global.Colors.TextBright;
  lastGLC=0;
  lastALC=0;

  PartnerGame = -1;

  xxnext=0;
  xxplayer[0][0]=0;
  xxplayer[1][0]=0;
  xxrating[0][0]=0;
  xxrating[1][0]=0;

  Login.set   ("*login:*");
  Password.set("*password:*");
  Prompt.set("*ics%%*");

  WelcomeMessage.append(new ExactStringPat("**** Starting "));
  WelcomeMessage.append(new KleeneStarPat());
  WelcomeMessage.append(new ExactStringPat("session as "));
  WelcomeMessage.append(sp=new PercSSetPat());
  WelcomeMessage.append(new KleeneStarPat());

  CreateGame.set("{Game %n (%s vs. %s) Creating %s %r *}*");
  CreateGame0.set("Creating: %s (*) %s (*)*");
  ContinueGame.set("{Game %n (%s vs. %s) Continuing %s %r *}*");

  BeginObserve.set("You are now observing game %n.");
  EndObserve.set("*Removing game %n from observation list*");

  EndExamine.set("You are no longer examining game %n*");

  Style12.set("<12> *");
  Style12House.set("<b1> game %n white [*] black [*]*");
  WhiteWon.set    ("{Game %n (%s vs. %s) *} 1-0*");
  BlackWon.set    ("{Game %n (%s vs. %s) *} 0-1*");
  Drawn.set       ("{Game %n (%s vs. %s) *} 1/2-1/2*");
  Interrupted.set ("{Game %n (%s vs. %s) *} %**");

  DrawOffer.set("%s offers you a draw.*");

  PrivateTell.set ("%a tells you:*");
  //Index of new news items
  News.set("Index of new *news items:*");

  // Ns: 0=game 1=rat.white 2=rat.black
  // Ss: 0=whitep 1=blackp
  // *s: 1=game string
  GameStatus.set("*%n%b%n%b%s%b%n%b%s%b[*]*");

  // 2 (Exam. 2413 Species     1234 Pulga     ) [ br  3   1] W:  1
  // amazingly enough, the indexes are the same as in GameStatus
  ExaGameStatus.set("*%n (Exam.%b%n %s%b%n %s%b)%b[*]*");

  GameListEntry.set("*%n *[*]*:*");
  EndOfGameList.set("*%n games displayed.*");

  AdListEntry.set("*%n%b%n%b%r%b%n%b%n%b%r*");
  EndOfAdList.set("*%n ad* displayed.*");

  //IceBox (1156) vs. damilasa (UNR) --- Sun Apr 22, 09:07 ??? 2001
  MovesHeader.set("%s (*) vs. %s (*) --- *");
  //Unrated blitz match, initial time: 2 minutes, increment: 12 seconds.
  MovesHeader2.set("%s %r match, initial time: *");
  //Move  IceBox             damilasa
  MovesHeader3.set("Move%b%s%b%s*");
  //----  ----------------   ----------------
  MovesHeader4.set("----  ---*");
  //  1.  d4      (0:00)     d5      (0:00)
  // n0   r0      r1         r2      r3
  MovesBody.set("*%n.%b%r%b%r%b%r%b%r*");
  // same thing, line without black move
  MovesBody2.set("*%n.%b%r%b%r*");
  //      {Still in progress} *
  MovesEnd.set("%b{*}*");
  
  Kibitz.set  ("%a*[%n]%bkibitzes:*");
  Whisper.set ("%a*[%n]%bwhispers:*");
  Say.set     ("%a[%n] says:*");
  Say2.set     ("%a says:*");
  ChannelTell.set("%a%n):*");
  Seek.set("*(*) seeking*(*to respond)");
  Shout.set ("%a shouts: *");
  cShout.set ("%a c-shouts: *");
  It.set ("-->*");
  Notify.set("Notification:*");

  SgClear.set("<sc>*");
  SgRemove.set("<sr>*");

  // <s> 8 w=visar ti=02 rt=2194  t=4 i=0 r=r tp=suicide c=? rr=0-9999 a=t f=f
  //    n0   s0       r0    n1  *0  n2  n3  s1    r1       r2    r3      s2  s3
  SgAdd.set("<s>%b%n%bw=%s%bti=%r%brt=%n*%bt=%n%bi=%n%br=%s%btp=%r%bc=%r%brr=%r%ba=%s%bf=%s*");
  SgAdd2.set("%b<s>%b%n%bw=%s%bti=%r%brt=%n*%bt=%n%bi=%n%br=%s%btp=%r%bc=%r%brr=%r%ba=%s%bf=%s*");

  // bughouse stuff
  PTell.set("%a (your partner) tells you: *");

  // sample output: Your partner is playing game 75 (aglup vs. cglup).
  GotPartnerGame.set("Your partner is playing game %n (*");

  // challenge
  Challenge.set("Challenge: *");

  // FICS LectureBot fake channel tells
  LectureBotXTell.set(":LectureBot(TD)(67):*");
  LectureBotXTellContinued.set(":   *");

  memset(style12table,0,256*sizeof(piece));
  style12table['-']=EMPTY;
  style12table['P']=PAWN|WHITE;
  style12table['R']=ROOK|WHITE;
  style12table['N']=KNIGHT|WHITE;
  style12table['B']=BISHOP|WHITE;
  style12table['Q']=QUEEN|WHITE;
  style12table['K']=KING|WHITE;
  style12table['p']=PAWN|BLACK;
  style12table['r']=ROOK|BLACK;
  style12table['n']=KNIGHT|BLACK;
  style12table['b']=BISHOP|BLACK;
  style12table['q']=QUEEN|BLACK;
  style12table['k']=KING|BLACK;

  Binder.add(&WelcomeMessage,&UserName,&Login,&Password,&Prompt,
	     &CreateGame,&CreateGame0,&ContinueGame,&BeginObserve,
	     &EndObserve,&NotObserve,&EndExamine,&GameStatus,
	     &ExaGameStatus,&GameListEntry,&EndOfGameList,
	     &AdListEntry,&EndOfAdList,&Style12,&Style12House,
	     &WhiteWon,&BlackWon,&Drawn,&Interrupted,&DrawOffer,
	     &PrivateTell,&News,&MovesHeader,&MovesHeader2,
	     &MovesHeader3,&MovesHeader4,&MovesBody,&MovesBody2,
	     &MovesEnd,0);
  Binder.add(&Kibitz,&Whisper,&Say,&Shout,&cShout,&It,&ChannelTell,
	     &Seek,&Notify,&SgClear,&SgAdd,&SgAdd2,&SgRemove,
	     &PTell,&Say2,&Challenge,&LectureBotXTell,
	     &LectureBotXTellContinued,&GotPartnerGame,0);
}

FicsProtocolParser::~FicsProtocolParser() {
  global.debug("FicsProtocolParser","~FicsProtocolParser");
  clearMoveBuffer();
}

void FicsProtocolParser::finalize() {
  global.debug("FicsProtocolParser","finalize");
  if (global.network)
    global.network->writeLine("quit");  
}

void FicsProtocolParser::receiveString(char *netstring) {
    int msgcolor=global.Colors.TextBright, ispersonal=0;
    char z[128];
    char pstring[1024];

    global.debug("FicsProtocolParser","receiveString",netstring);

    // eat timeseal annoying blank lines
    if ((netstring[0]==0)&&(LastWasStyle12)) {
      LastWasStyle12=0;
      return;
    }

    LastWasStyle12=0;
    Binder.prepare(netstring);

    while (Prompt.match()) {
      // typing 'help iv_defprompt' without this is --funny-- to
      // say the least
      if (strlen(Prompt.getStarToken(0))>9)
	break;
      // this is to cope with the free (GPL) 1.6.2 version of FICS
      if (!AuthenticationGone) { 
	AuthenticationGone=1;
	global.input->setPasswordMode(0);
      }
	
      if (strlen(Prompt.getStarToken(1))>3) {
	memset(pstring,0,1024);
	strncpy(pstring,Prompt.getStarToken(1),1023);
	netstring=pstring;
	if (netstring[0]==' ') netstring++;
	Binder.prepare(netstring);
      } else {
	LastWasPrompt=1;
	return;
      }
    } // while

    if ((netstring[0]==0)&&(LastWasPrompt))
      return;
    LastWasPrompt=0;
    
    if (!AuthenticationGone) {

      if (Password.match())
	global.input->setPasswordMode(1);

      if (Login.match())
	global.input->setPasswordMode(0);

      if (WelcomeMessage.match()) {
	global.input->setPasswordMode(0);
	AuthenticationGone=1;
	strcpy(nick,WelcomeMessage.getToken(3));	
	UserName.append(new KleeneStarPat());
	UserName.append(new ExactStringPat(nick));
	UserName.append(new KleeneStarPat());

	sprintf(z,"set interface eboard %s/%s\niset startpos 1\niset defprompt 1",
		global.Version,global.SystemType);
	global.network->writeLine(z);
	if (global.IcsSeekGraph)
	  global.network->writeLine("iset seekinfo 1\niset seekremove 1\nset seek 1");
	global.network->writeLine("iset lock 1\nstyle 12");
      }
    }

    // game events
    if (Style12.match()) {
      parseStyle12(netstring);
      LastWasStyle12=1;
      return;
    }
    if (Style12House.match()) {
      attachHouseStock();
      return;
    }

    if ((global.HideSeeks)&&(global.IcsSeekGraph))
      if (Seek.match())
	return;

    if (global.IcsSeekGraph) {
      if (SgClear.match()) {
	ensureSeekGraph();
	global.skgraph->clear();
	return;
      }

      if (SgAdd.match()) {
	seekAdd(SgAdd);
	return;
      }

      if (SgAdd2.match()) {
	seekAdd(SgAdd2);
	return;
      }

      if (SgRemove.match()) {
	seekRemove(SgRemove);
	return;
      }
    }

    // stateful retrieval

    if (RetrievingAdList) {
      if (AdListEntry.match()) {
	sendAdListEntry(netstring);
	return;
      }
      if (EndOfAdList.match()) {
	RetrievingAdList=0;
	lastALC->endOfList();
	lastALC=0;
	return;
      }
    }

    if (RetrievingGameList) {
      if (GameListEntry.match()) {
	sendGameListEntry();
	return;
      }
      if (EndOfGameList.match()) {
	RetrievingGameList=0;
	lastGLC->endOfList();
	lastGLC=0;
	return;
      }
    }

    if (RetrievingMoves) {

      switch(RetrievingMoves) {
      case 1:
	if (MovesHeader.match()) {
	  RetrievingMoves++;
	  strncpy(xxplayer[xxnext],MovesHeader.getSToken(0),64);
	  strncpy(xxrating[xxnext],MovesHeader.getStarToken(0),64);
	  xxnext=(++xxnext)%2;
	  strncpy(xxplayer[xxnext],MovesHeader.getSToken(1),64);
	  strncpy(xxrating[xxnext],MovesHeader.getStarToken(1),64);
	  xxnext=(++xxnext)%2;
	  return;
	}
	break;
      case 2:
	if (MovesHeader2.match()) { RetrievingMoves++; return; }
	break;
      case 3:
	if (MovesHeader3.match()) { RetrievingMoves++; return; }
	break;
      case 4:
	if (MovesHeader4.match()) { 
	  RetrievingMoves++;
	  clearMoveBuffer();
	  return;
	}
	break;
      case 5:
	if (MovesBody.match()) { retrieve1(MovesBody); return; }
	if (MovesBody2.match()) { retrieve2(MovesBody2); return; }
	if (MovesEnd.match()) {
	  retrievef();
	  RetrievingMoves=0;
	  return;
	}
	break;
      }
    }

    if (CreateGame0.match()) {
      strncpy(xxplayer[xxnext],CreateGame0.getSToken(0),64);
      strncpy(xxrating[xxnext],CreateGame0.getStarToken(0),64);
      xxnext=(++xxnext)%2;
      strncpy(xxplayer[xxnext],CreateGame0.getSToken(1),64);
      strncpy(xxrating[xxnext],CreateGame0.getStarToken(1),64);
      xxnext=(++xxnext)%2;
      goto just_output;
    }
    if (CreateGame.match()) {
      createGame(CreateGame);
      return;
    }
    if (ContinueGame.match()) {
      createGame(ContinueGame);
      // for some odd reason, FICS sends the move list
      // twice. The second one is plainly ignored (and goes to
      // the console)
      retrieveMoves(ContinueGame);
      return;
    }
    if (BeginObserve.match()) {
      addGame(BeginObserve);
      return;
    }
    if (EndObserve.match()) {
      removeGame(EndObserve);
      return;
    }
    if (EndExamine.match()) {
      removeGame(EndExamine);
      return;
    }
    if ((PendingStatus)&&(ExaGameStatus.match())) {
      updateGame(ExaGameStatus);
      PendingStatus--;
      goto just_output;
    }
    if ((PendingStatus)&&(GameStatus.match())) {
      updateGame(GameStatus);
      PendingStatus--;
      goto just_output;
    }
    if (WhiteWon.match()) {
      gameOver(WhiteWon,WHITE_WIN);
      msgcolor=global.Colors.TextBright;
      goto just_output;
    }
    if (BlackWon.match()) {
      gameOver(BlackWon,BLACK_WIN);
      msgcolor=global.Colors.TextBright;
      goto just_output;
    }
    if (Drawn.match()) {
      gameOver(Drawn,DRAW);
      msgcolor=global.Colors.TextBright;
      goto just_output;
    }
    if (Interrupted.match()) {
      gameOver(Interrupted,UNDEF);
      msgcolor=global.Colors.TextBright;
      goto just_output;
    }

    // bughouse partner game
    if (GotPartnerGame.match())
      prepareBugPane();

    msgcolor=global.Colors.TextDefault;
    
    if (netstring[0]=='\\')
      msgcolor=LastColor;
    else
      LastWasCTell=-1;

    if (Challenge.match()) {
      global.challenged();
    }

    if (UserName.match())    {
      msgcolor=global.Colors.PrivateTell; ispersonal=1;
      if (ChannelTell.match() || LectureBotSpecial || LectureBotXTell.match() )
	goto it_is_a_chtell_too;
      else
	goto just_output;
    }

    if (PTell.match()) {
      msgcolor=global.Colors.PrivateTell; ispersonal=1;
      global.privatelyTold();
      global.bugpane->addBugText(PTell.getStarToken(0));
      goto just_output;
    }

    if (PrivateTell.match()||Say.match()||Say2.match()) {
      msgcolor=global.Colors.PrivateTell; ispersonal=1;
      global.privatelyTold();
    }

    if (DrawOffer.match()) {
      msgcolor=global.Colors.PrivateTell; ispersonal=1;
      global.drawOffered();
    }

    if (News.match()||Notify.match()) {
      msgcolor=global.Colors.NewsNotify; ispersonal=0;
    }

    if (Kibitz.match()||Whisper.match()) {
      msgcolor= global.Colors.KibitzWhisper;
      goto just_output;
    }

    if (Shout.match()||cShout.match()||It.match()) {
      msgcolor= global.Colors.Shouts;
      goto just_output;
    }

    if (Seek.match()) {
      msgcolor= global.Colors.Seeks;
      goto just_output;
    }

 it_is_a_chtell_too:

    if (netstring[0]==':')  {
      if (!ispersonal)
	msgcolor= global.Colors.Mamer; // mamer messages
      
      if (global.SplitChannels) {

	// I am just pasting the continuation of a LectureBot fake chtell
	if (LectureBotSpecial) {

	  if (LectureBotXTellContinued.match()) {
	    global.appendToChannel(67,netstring,msgcolor,
				   (ispersonal)?IM_PERSONAL:IM_NORMAL);
	    if (!global.ChannelsToConsoleToo)
	      goto dont_even_output;
	  } else
	    LectureBotSpecial = false;

	} else {

	  // LectureBot is starting a fake chtell
	  if (LectureBotXTell.match()) {
	    global.appendToChannel(67,netstring,msgcolor,
				   (ispersonal)?IM_PERSONAL:IM_NORMAL);
	    LectureBotSpecial=true;
	    if (!global.ChannelsToConsoleToo)
	      goto dont_even_output;
	  }


	}
      } else // this would never happen without split channels
	LectureBotSpecial=false;

    } // mamer et al.

    if ((global.SplitChannels)&&(LastWasCTell>=0)&&(netstring[0]=='\\')) {      
      global.appendToChannel(LastWasCTell,netstring,msgcolor,
			     (ispersonal)?IM_PERSONAL:IM_NORMAL);
      if (!global.ChannelsToConsoleToo)
	goto dont_even_output;
    }

    if (ChannelTell.match()) {
      if (!ispersonal)
	msgcolor = global.Colors.ChannelTell;

      if (global.SplitChannels) {
	int ch;
	ch=atoi(ChannelTell.getNToken(0));
	LastWasCTell=ch;
	global.appendToChannel(ch,netstring,msgcolor,(ispersonal)?IM_PERSONAL:IM_NORMAL);
	if (!global.ChannelsToConsoleToo)
	  goto dont_even_output;
      }

      goto just_output;
    }
    
 just_output:
    global.output->append(netstring,msgcolor,
			  (ispersonal)?IM_PERSONAL:IM_NORMAL);
 dont_even_output:
    LastColor=msgcolor;
}

void FicsProtocolParser::createExaminedGame(int gameid,char *whitep,char *blackp) {
  ChessGame *cg;
  Board *b;
  char *p,z[64];

  global.debug("FicsProtocolParser","createExaminedGame");

  cg=new ChessGame();
  cg->clock_regressive=1;
  cg->GameNumber=gameid;
  cg->StopClock=1;
  cg->source = GS_ICS;
  strncpy(cg->PlayerName[0],whitep,64);
  strncpy(cg->PlayerName[1],blackp,64);
  cg->MyColor=WHITE|BLACK;
  cg->protodata[0]=258;

  global.prependGame(cg);
  b=new Board();
  b->reset();
  cg->setBoard(b);
  b->setGame(cg);
  b->setCanMove(true);
  b->setSensitive(true);

  sprintf(z,"Exam.Game #%d",cg->GameNumber);
  global.ebook->addPage(b->widget,z,cg->GameNumber);
  b->setNotebook(global.ebook,cg->GameNumber);

  if (global.PopupSecondaryGames) {
    b->pop();
    b->repaint();
  }

  cg->acknowledgeInfo();

  sprintf(z,"game %d",cg->GameNumber);
  global.network->writeLine(z);
  PendingStatus++;
}

// creates a non-moveable, non-editable position for sposition
void FicsProtocolParser::createScratchPos(int gameid,char *whitep,char *blackp)
{
  ChessGame *cg;
  Board *b;
  char *p,z[64];

  global.debug("FicsProtocolParser","createScratchPos");

  cg=new ChessGame();
  cg->clock_regressive=1;
  cg->GameNumber=gameid;
  cg->StopClock=1;
  cg->source=GS_ICS;
  strncpy(cg->PlayerName[0],whitep,64);
  strncpy(cg->PlayerName[1],blackp,64);
  cg->MyColor=WHITE|BLACK;
  cg->protodata[0]=259;

  global.prependGame(cg);
  b=new Board();
  b->reset();
  cg->setBoard(b);
  b->setGame(cg);
  b->setCanMove(false);
  b->setSensitive(false);

  sprintf(z,"Pos: %s vs. %s",whitep,blackp);
  global.ebook->addPage(b->widget,z,cg->GameNumber);
  b->setNotebook(global.ebook,cg->GameNumber);

  cg->acknowledgeInfo();
}

void FicsProtocolParser::createGame(ExtPatternMatcher &pm) {
  ChessGame *cg,*rep;
  char *p;

  global.debug("FicsProtocolParser","createGame");

  cg=new ChessGame();

  cg->source=GS_ICS;
  cg->clock_regressive=1;
  cg->GameNumber=atoi(pm.getNToken(0));
  strncpy(cg->PlayerName[0],pm.getSToken(0),64);
  strncpy(cg->PlayerName[1],pm.getSToken(1),64);

  // FICS sends two "Creating" strings when resuming an adjourned
  // game. Weird.
  rep=global.getGame(cg->GameNumber);
  if (rep) {
    if ( (!strcmp(cg->PlayerName[0],rep->PlayerName[0]))&&
	 (!strcmp(cg->PlayerName[1],rep->PlayerName[1]))&&
	 (rep->isFresh()) ) {
      delete cg;
      return;
    }
  }

  p=knowRating(cg->PlayerName[0]);
  if (p) strncpy(cg->Rating[0],p,32);
  p=knowRating(cg->PlayerName[1]);
  if (p) strncpy(cg->Rating[1],p,32);
  clearRatingBuffer();

  if (!strncmp(cg->PlayerName[0],nick,strlen(cg->PlayerName[0])))
    cg->MyColor=WHITE;
  if (!strncmp(cg->PlayerName[1],nick,strlen(cg->PlayerName[1])))
    cg->MyColor=BLACK;
  
  if (!strcmp(pm.getSToken(2),"rated"))
    cg->Rated=1;
  else
    cg->Rated=0;

  cg->Variant=REGULAR;
  if (!strcmp(pm.getRToken(0),"crazyhouse"))
    cg->Variant=CRAZYHOUSE;
  if (!strcmp(pm.getRToken(0),"bughouse"))
    cg->Variant=BUGHOUSE;
  if (!strcmp(pm.getRToken(0),"suicide"))
    cg->Variant=SUICIDE;
  if (!strcmp(pm.getRToken(0),"losers"))
    cg->Variant=LOSERS;
  if (!strncmp(pm.getRToken(0),"wild",4))
    cg->Variant=WILD;

  global.prependGame(cg);
  global.BoardList.front()->reset();

  cg->setBoard(global.BoardList.front());
  global.BoardList.front()->setGame(cg);
  global.BoardList.front()->pop();
  global.BoardList.front()->setCanMove(true);
  global.BoardList.front()->repaint();

  cg->acknowledgeInfo();
  global.status->setText("Game started!");
}

void FicsProtocolParser::addGame(ExtPatternMatcher &pm) {
  ChessGame *cg;
  Board *b;
  bool is_partner_game=false;
  char tab[96];

  global.debug("FicsProtocolParser","addGame");

  cg=new ChessGame();

  cg->clock_regressive=1;
  cg->GameNumber=atoi(pm.getNToken(0));

  cg->STime=999;
  cg->Inc=999;
  cg->Rated=0;
  cg->Variant=REGULAR;
  cg->source = GS_ICS;

  global.prependGame(cg);
  b=new Board();
  b->reset();
  cg->setBoard(b);
  b->setGame(cg);
  b->setCanMove(false);
  b->setSensitive(false);

  if (cg->GameNumber == PartnerGame) {
    is_partner_game=true;
    PartnerGame = -1;
    cg->protodata[2] = 1;
  }

  sprintf(tab,"Game #%d",cg->GameNumber);
  global.ebook->addPage(b->widget,tab,cg->GameNumber);
  b->setNotebook(global.ebook,cg->GameNumber);

  if (!is_partner_game)
    if (global.PopupSecondaryGames) {
      b->pop();
      b->repaint();
    }

  cg->acknowledgeInfo();

  // to gather info about time, variant, player names...
  sprintf(tab,"game %d",cg->GameNumber);
  global.network->writeLine(tab);
  PendingStatus++;
}

void FicsProtocolParser::discardGame(int gameid) {
  ChessGame *cg;
  Board *b;
  char z[64];

  global.debug("FicsProtocolParser","discardGame");

  cg=global.getGame(gameid);
  if (!cg)
    return;

  switch(cg->protodata[0]) {
  case 257: // observed    
    sprintf(z,"unobserve %d",cg->GameNumber);
    break;
  case 258: // examined
    strcpy(z,"unexamine");
    break;
  case 259: // isolated position
    z[0]=0;
    break;
  default:
    cerr << "protodata for game " << gameid << " is " << cg->protodata[0] << endl;
    return;
  }

  if (! cg->isOver())
    if (strlen(z))
      global.network->writeLine(z);

  b=cg->getBoard();
  if (b==0) return;
  global.ebook->removePage(cg->GameNumber);
  global.removeBoard(b);
  delete(b);
  cg->GameNumber=global.nextFreeGameId(10000);
  cg->setBoard(0);

}

void FicsProtocolParser::removeGame(ExtPatternMatcher &pm) {
  int gm;
  global.debug("FicsProtocolParser","removeGame");

  gm=atoi(pm.getNToken(0));
  innerRemoveGame(gm);
}

void FicsProtocolParser::innerRemoveGame(int gameid) {
  Board *b;
  ChessGame *cg;

  cg=global.getGame(gameid);
  if (!cg)
    return;

  // bughouse partner game ?
  if (cg->protodata[2])
    global.bugpane->stopClock();

  b=cg->getBoard();
  if (b==0) return;
  global.ebook->removePage(gameid);
  global.removeBoard(b);  
  delete(b);
  cg->GameNumber=global.nextFreeGameId(10000);
  cg->setBoard(0);
}

void FicsProtocolParser::sendDrop(piece p,int x,int y) {
  piece k;
  char drop[5];

  global.debug("FicsProtocolParser","sendDrop");

  k=p&PIECE_MASK;
  drop[4]=0;
  switch(k) {
  case PAWN:   drop[0]='P'; break;
  case ROOK:   drop[0]='R'; break;
  case KNIGHT: drop[0]='N'; break;
  case BISHOP: drop[0]='B'; break;
  case QUEEN:  drop[0]='Q'; break;
  default: return;
  }
  drop[1]='@';
  drop[2]='a'+x;
  drop[3]='1'+y;
  global.network->writeLine(drop);
}

void FicsProtocolParser::sendMove(int x1,int y1,int x2,int y2,int prom) {
  char move[7];
  piece pp;

  global.debug("FicsProtocolParser","sendMove");

  move[4]=0;
  move[5]=0;
  move[6]=0;
  move[0]='a'+x1;
  move[1]='1'+y1;
  move[2]='a'+x2;
  move[3]='1'+y2;

  if (prom) {
    pp=global.promotion->getPiece();
    move[4]='=';
    switch(pp) {
    case QUEEN:   move[5]='Q'; break;
    case ROOK:    move[5]='R'; break;
    case BISHOP:  move[5]='B'; break;
    case KNIGHT:  move[5]='N'; break;
    case KING:    move[5]='K'; break;
    }
  }
  global.network->writeLine(move);
}

// attach stock info to bug/zhouse games
void FicsProtocolParser::attachHouseStock() {
  ChessGame *cg;
  int gid;
  char *p;
  Position *pos;

  global.debug("FicsProtocolParser","attachHouseStock");

  //  <b1> game %n white [*] black [*]*

  gid=atoi(Style12House.getNToken(0));

  cg=global.getGame(gid);
  if (!cg)
    return;

  pos=&(cg->getLastPosition());

  pos->clearStock();
  p=Style12House.getStarToken(0);

  //  cerr << "[" << p << "]\n";

  for(;*p;p++)
    pos->incStockCount(style12table[*p]);
  
  p=Style12House.getStarToken(1);

  // cerr << "[" << p << "]\n";

  for(;*p;p++)
    pos->incStockCount(style12table[tolower(*p)]);

  cg->updateStock();
}

void FicsProtocolParser::parseStyle12(char *data) {
  static char *sep=" \t\n";
  tstring t;
  int i,j,n;
  string *sp;
  Position pos;
  int blacktomove;
  int game;
  int rel;
  int itime,inc,flip;
  int str[2],movenum,clk[2];
  ChessGame *refgame;
  char stra[64],strb[64],strc[64];
  char plyr[2][64];
  int ackp=0;
  bool spawnExamination;

  global.debug("FicsProtocolParser","parseStyle12");

  t.set(data);
  t.token(sep); // <12>
  
  // the position itself
  for(i=7;i>=0;i--) {
    sp=t.token(sep);
    for(j=0;j<8;j++)
      pos.setPiece(j,i,style12table[sp->at(j)]);
  }

  sp=t.token(sep); // B / W (color to move)
  blacktomove=(sp->at(0)=='B');

  pos.ep[0]=t.tokenvalue(sep); // double pawn push
  if (blacktomove)
    pos.ep[1]=2;
  else
    pos.ep[1]=5;

  // white castle short
  pos.maycastle[0]=t.tokenvalue(sep)?true:false;
  
  // white castle long
  pos.maycastle[2]=t.tokenvalue(sep)?true:false;

  // black castle short
  pos.maycastle[1]=t.tokenvalue(sep)?true:false;

  // black castle long
  pos.maycastle[3]=t.tokenvalue(sep)?true:false;

  t.token(sep); // mv count for draw

  game=t.tokenvalue(sep); // game number

  memset(plyr[0],0,64);
  memset(plyr[1],0,64);
  t.token(sep)->copy(plyr[0],63); // white's name
  t.token(sep)->copy(plyr[1],63); // black's name

  rel=t.tokenvalue(sep); // relationship

  switch(rel) {
  case 2: // I am the examiner of this game
    spawnExamination=false;
    refgame=global.getGame(game);

    if (!refgame)
      spawnExamination=true;      
    else // v_examine=1 on FICS: played game still on main board but inactive
      if ( (refgame->protodata[0]!=258) && (refgame->isOver()) )
	spawnExamination=true;

    if (spawnExamination)
      createExaminedGame(game,plyr[0],plyr[1]);
    break;

  case -3: // isolated position (e.g.: sposition command)
    game=global.nextFreeGameId(10000);
    createScratchPos(game,plyr[0],plyr[1]);
    break;
  case -2: // observing examined game
  case -1: // I am playing, it's my opponent's turn
  case 1: // I am playing, it's my turn
  case 0: // live observation
    break;
  }
  
  itime=t.tokenvalue(sep); // initial time (minutes)
  inc=t.tokenvalue(sep);   // increment (secs)
  
  str[0]=t.tokenvalue(sep);  // white strength
  str[1]=t.tokenvalue(sep);  // black strength

  clk[0]=t.tokenvalue(sep);  // white rem time
  clk[1]=t.tokenvalue(sep);  // black rem time

  movenum=t.tokenvalue(sep); // move # to be made

  t.token(sep);               // verbose for previous move

  memset(stra,0,64);
  memset(strb,0,64);
  t.token(sep)->copy(stra,63); // time taken for previous move
  t.token(sep)->copy(strb,63); // pretty print for previous move

  flip=t.tokenvalue(sep);    // flip board ?
 
  refgame=global.getGame(game);
  if (!refgame) {
    cerr << "no such game: " << game << endl;
    return;
  }

  switch(rel) {
  case  0: // observed
  case -2: // observing examined game
    refgame->protodata[0]=257;
    refgame->AmPlaying=false;
    break;
  case 1:
  case -1:
    refgame->protodata[0]=0;
    refgame->MyColor=(flip)?BLACK:WHITE;
    refgame->AmPlaying=true;
    if ((global.IcsSeekGraph)&&(global.skgraph))
      global.skgraph->clear();
    break;
  case 2: // examiner mode
    refgame->protodata[0]=258;
    refgame->setFree();
    refgame->AmPlaying=false;
    break;
  case -3: // isolated position
    refgame->protodata[0]=259;
    refgame->AmPlaying=false;
    break;
  }

  if (strcmp(refgame->PlayerName[0],plyr[0])) {
    strcpy(refgame->PlayerName[0],plyr[0]);
    ackp=1;
  }

  if (strcmp(refgame->PlayerName[1],plyr[1])) {
    strcpy(refgame->PlayerName[1],plyr[1]);
    ackp=1;
  }

  if ( (refgame->STime != itime*60)||(refgame->Inc != inc) ) {
    refgame->STime=itime*60;
    refgame->Inc=inc;
    ackp=1;
  }

  if (ackp)
    refgame->acknowledgeInfo();

  strcat(strb," ");
  strcat(strb,stra);
  sprintf(strc,"%d.%s%s",
	  (blacktomove?movenum:movenum-1),
	  (blacktomove?" ":" ... "),
	  strb);
  if (strc[0]=='0')
    strcpy(strc,"0. startpos");

  if ((rel==-1)&&(global.AnimateMoves))
    refgame->getBoard()->supressNextAnimation();

  if (rel==-2) // don't let it tick when observing examined games
    refgame->StopClock=1;

  refgame->updatePosition(pos, (movenum-(blacktomove?0:1)), blacktomove,
			  clk[0], clk[1], strc, (rel==1));

  /*
  if ( ((rel==-1)&&(blacktomove)) || ((rel==1)&&(!blacktomove)) ) {
    if (flip) cerr << "I'm white and <12> told me to flip!\n";    
  }

  if ( ((rel==1)&&(blacktomove)) || ((rel==-1)&&(!blacktomove)) ) {
    if (!flip) cerr << "I'm black and <12> told me not to flip!\n";    
  }
  */

  if (rel==1)
    global.opponentMoved();

  if (rel!=0) refgame->flipHint(flip);
  if (rel==1) refgame->enableMoving(true);
  if (rel==-1) refgame->enableMoving(false);
}

void FicsProtocolParser::updateGame(ExtPatternMatcher &pm) {
  ChessGame *refgame;
  char *p,z[32];
  int gm;

  global.debug("FicsProtocolParser","updateGame");

  // Ns: 0=game 1=rat.white 2=rat.black
  // Ss: 0=whitep 1=blackp
  // *s: 1=game string
  //  GameStatus.set("*%n%b%n%b%s%b%n%b%s%b[*]*");
  //  ExaGameStatus.set("*%n (Exam.%b%n %s%b%n %s%b)%b[*]*");

  gm=atoi(pm.getNToken(0));

  refgame=global.getGame(gm);
  if (!refgame) {
    PendingStatus++;
    return;
  }

  strncpy(refgame->PlayerName[0],pm.getSToken(0),64);
  strncpy(refgame->PlayerName[1],pm.getSToken(1),64);

  p=pm.getStarToken(1);

  switch(p[1]) {
  case 's':
  case 'b':
  case 'l':
    refgame->Variant=REGULAR;
    break;
  case 'z':
    refgame->Variant=CRAZYHOUSE;
    break;
  case 'S':
    refgame->Variant=SUICIDE;
    break;
  case 'w':
    refgame->Variant=WILD;
    break;
  case 'B':
    refgame->Variant=BUGHOUSE;
    break;
  case 'L':
    refgame->Variant=LOSERS;
    break;
  }

  refgame->Rated=(p[2]=='r');

  memset(z,0,8);
  memcpy(z,&p[3],3);
  refgame->STime=60*atoi(z);
  refgame->Inc=atoi(&p[7]);

  refgame->acknowledgeInfo();

  if ((refgame->Variant!=WILD)&&(refgame->protodata[0]!=258)) {
    MoveRetrievals.push_front(gm);
    sprintf(z,"moves %d",gm);
    global.network->writeLine(z);
    RetrievingMoves=1;
  }

  // bughouse partner game, set up names in the bugpane
  if (refgame->protodata[2]) {
    global.bugpane->setPlayerNames(refgame->PlayerName[0],
				   refgame->PlayerName[1]);
  }
}

void FicsProtocolParser::gameOver(ExtPatternMatcher &pm, GameResult result) {
  int game;
  char reason[64];
  ChessGame *refgame;

  global.debug("FicsProtocolParser","gameOver");
  
  game=atoi(pm.getNToken(0));
  reason[63]=0;
  strncpy(reason,pm.getStarToken(0),64);

  refgame=global.getGame(game);
  if (!refgame) {
    cerr << "no such game: " << game << endl;
    return;
  }

  refgame->endGame(reason,result);

  if (((refgame->protodata[0]==0)&&(global.AppendPlayed))
      ||
      ((refgame->protodata[0]==257)&&(global.AppendObserved)))
    {
      if  (refgame->savePGN(global.AppendFile,true)) {
	char z[128];
	sprintf(z,"Game appended to %s",global.AppendFile);
	global.status->setText(z);
      }
    }

  // refresh seek graph
  if ((global.IcsSeekGraph)&&(refgame->protodata[0]==0))
	refreshSeeks();

  // remove board ?
  // if game relation is observation and user has the Smart Discard setting on:
  //  discard now if user is not looking to the board or schedule removal to
  //  next pane switch if user _is_ looking to the board
  if ((refgame->protodata[0]==257)&&(global.SmartDiscard)) {
    if ( global.ebook->getCurrentPageId() == refgame->GameNumber )
      global.ebook->setNaughty(refgame->GameNumber,true);
    else
      innerRemoveGame(refgame->GameNumber);
  }
}

void FicsProtocolParser::refreshSeeks() {
	if ((global.IcsSeekGraph)&&(global.skgraph)) {
		global.skgraph->clear();
		global.network->writeLine("iset seekinfo 1");
	}
}

int  FicsProtocolParser::hasAuthenticationPrompts() {
  return 1;
}

void FicsProtocolParser::resign() {
  global.network->writeLine("resign");
  global.status->setText("Resigned.");
}

void FicsProtocolParser::draw() {
  global.network->writeLine("draw");
  global.status->setText("<Fics Protocol> draw request sent");
}

void FicsProtocolParser::adjourn() {
  global.network->writeLine("adjourn");
  global.status->setText("<Fics Protocol> adjourn request sent");
}

void FicsProtocolParser::abort() {
  global.network->writeLine("abort");
  global.status->setText("<Fics Protocol> abort request sent");
}

void FicsProtocolParser::clearMoveBuffer() {
  list<Position *>::iterator li;

  global.debug("FicsProtocolParser","clearMoveBuffer");
  MoveBuf.clear();
}

void FicsProtocolParser::retrieve1(ExtPatternMatcher &pm) {
  char z[32],mv[16];
  char tmp[2][32];
  int itmp;
  Position p,q;

  global.debug("FicsProtocolParser","retrieve1");

  if (!MoveBuf.empty())
    p=MoveBuf.back();
  else
    MoveBuf.push_back(Position());

  itmp=atoi(pm.getNToken(0));
  strncpy(tmp[0],pm.getRToken(0),32);
  strncpy(tmp[1],pm.getRToken(1),32);
  sprintf(z,"%.2d. %s %s",itmp,tmp[0],tmp[1]);
  strcpy(mv,tmp[0]);

  p.moveStdNotation(mv,WHITE);
  p.setLastMove(z);

  MoveBuf.push_back(p);

  q=p;

  itmp=atoi(pm.getNToken(0));
  strncpy(tmp[0],pm.getRToken(2),32);
  strncpy(tmp[1],pm.getRToken(3),32);
  sprintf(z,"%.2d. ... %s %s",itmp,tmp[0],tmp[1]);
  strcpy(mv,tmp[0]);

  q.moveStdNotation(mv,BLACK);
  q.setLastMove(z);

  MoveBuf.push_back(q);
}

void FicsProtocolParser::retrieve2(ExtPatternMatcher &pm) {
  Position p;
  char z[32],mv[16];
  char tmp[2][32];
  int itmp;

  global.debug("FicsProtocolParser","retrieve2");

  if (!MoveBuf.empty())
    p=MoveBuf.back();

  itmp=atoi(pm.getNToken(0));
  strncpy(tmp[0],pm.getRToken(0),32);
  strncpy(tmp[1],pm.getRToken(1),32);
  sprintf(z,"%.2d. %s %s",itmp,tmp[0],tmp[1]);
  strcpy(mv,tmp[0]);

  p.moveStdNotation(mv,WHITE);
  p.setLastMove(z);

  MoveBuf.push_back(p);
}

void FicsProtocolParser::retrievef() {
  int gm;
  ChessGame *cg;
  char *p;

  global.debug("FicsProtocolParser","retrievef");

  if (MoveRetrievals.empty())
    return;

  gm=MoveRetrievals.back();

  cg=global.getGame(gm);
  if (!cg) {
    cerr << "no such game: " << gm << endl;
    return;
  }
  MoveRetrievals.pop_back();

  p=knowRating(cg->PlayerName[0]);
  if (p) strncpy(cg->Rating[0],p,32);
  p=knowRating(cg->PlayerName[1]);
  if (p) strncpy(cg->Rating[1],p,32);
  clearRatingBuffer();

  cg->updateGame(MoveBuf);
  cg->acknowledgeInfo();
  clearMoveBuffer();
}

void FicsProtocolParser::queryGameList(GameListConsumer *glc) {
  if (RetrievingGameList) {
    if (glc!=lastGLC)
      glc->endOfList();
    return;
  }

  RetrievingGameList=1;
  lastGLC=glc;
  global.network->writeLine("games");  
}

void FicsProtocolParser::queryAdList(GameListConsumer *glc) {
  if (RetrievingAdList) {
    if (glc!=lastALC)
      glc->endOfList();
    return;
  }
  RetrievingAdList=1;
  lastALC=glc;
  global.network->writeLine("sough");
}

void FicsProtocolParser::sendGameListEntry() {
  char frank[256];
  int gn;
  //  GameListEntry.set("*%n *[*]*:*");

  gn=atoi(GameListEntry.getNToken(0));
  
  strcpy(frank,GameListEntry.getStarToken(1));
  strcat(frank,"[");
  strcat(frank,GameListEntry.getStarToken(2));
  strcat(frank,"]");
  strcat(frank,GameListEntry.getStarToken(3));
  strcat(frank,":");
  strcat(frank,GameListEntry.getStarToken(4));

  lastGLC->appendGame(gn,frank);
}

void FicsProtocolParser::sendAdListEntry(char *pat) {
  ExtPatternMatcher rpat;
  int gn;

  rpat.set("*%n*");
  if (!rpat.match(pat))
    return;

  gn=atoi(rpat.getNToken(0));
  lastALC->appendGame(gn,rpat.getStarToken(1));
}

void FicsProtocolParser::observe(int gameid) {
  char z[64];
  sprintf(z,"observe %d",gameid);
  global.network->writeLine(z);
}

void FicsProtocolParser::answerAd(int adid) {
  char z[64];
  sprintf(z,"play %d",adid);
  global.network->writeLine(z);
}

char * FicsProtocolParser::knowRating(char *player) {
  int i;
  for(i=0;i<2;i++)
    if (strlen(xxplayer[i]))
      if ( !strncmp(xxplayer[i],player,strlen(player)) )
	return(xxrating[i]);
  return 0;
}

void FicsProtocolParser::clearRatingBuffer() {
  xxplayer[0][0]=0;
  xxplayer[1][0]=0;
}

void FicsProtocolParser::exaForward(int n) {
  char z[64];
  sprintf(z,"forward %d",n);
  global.network->writeLine(z);
}

void FicsProtocolParser::exaBackward(int n) {
  char z[64];
  sprintf(z,"backward %d",n);
  global.network->writeLine(z);
}

void FicsProtocolParser::retrieveMoves(ExtPatternMatcher &pm) {
  ChessGame *refgame;
  int gm;
  char z[64];
  global.debug("FicsProtocolParser","retrieveMoves");

  gm=atoi(pm.getNToken(0));  
  refgame=global.getGame(gm);
  if (!refgame)
    return;

  if ((refgame->Variant!=WILD)&&(refgame->protodata[0]!=258)) {
    MoveRetrievals.push_front(gm);
    sprintf(z,"moves %d",gm);
    global.network->writeLine(z);
    RetrievingMoves=1;
  }
}

// --- seek graph

void FicsProtocolParser::ensureSeekGraph() {
  if (!global.skgraph) {
    global.skgraph=new SeekGraph();
    global.skgraph->show();
    global.ebook->addPage(global.skgraph->widget,"Seek Table",-3);
    global.skgraph->setNotebook(global.ebook,-3);
  }
}

void FicsProtocolParser::seekAdd(ExtPatternMatcher &pm) {
  SeekAd *ad;
  char *p;
  int flags;

  global.debug("FicsProtocolParser","seekAdd");

  // <s> 8 w=visar ti=02 rt=2194  t=4 i=0 r=r tp=suicide c=? rr=0-9999 a=t f=f
  //    n0   s0       r0    n1  *0  n2  n3  s1    r1       r2    r3      s2  s3

  ensureSeekGraph();
  ad=new SeekAd();

  // %n
  ad->id      = atoi(pm.getNToken(0));
  ad->rating  = pm.getNToken(1);
  ad->clock   = atoi(pm.getNToken(2));
  ad->incr    = atoi(pm.getNToken(3));

  p=pm.getStarToken(0);
  if ((p[0]=='P')&&(atoi(ad->rating.c_str())==0)) ad->rating="++++";

  // %s
  ad->player  = pm.getSToken(0);

  p=pm.getSToken(1);
  if (p[0]=='r') ad->rated=true;

  ad->kind    = pm.getRToken(1);

  // %r
  flags = strtol(pm.getRToken(0),NULL,16);

  ad->flags=" ";

  if (flags&0x01) ad->flags += "(U)";
  if (flags&0x02) ad->flags += "(C)";
  if (flags&0x04) ad->flags += "(GM)";
  if (flags&0x08) ad->flags += "(IM)";
  if (flags&0x10) ad->flags += "(FM)";
  if (flags&0x20) ad->flags += "(WGM)";
  if (flags&0x40) ad->flags += "(WIM)";
  if (flags&0x80) ad->flags += "(WFM)";

  p=pm.getRToken(2);
  switch(p[0]) {
  case '?': ad->color=" "; break;
  case 'W': ad->color="white"; break;
  case 'B': ad->color="black"; break;
  }

  ad->range=pm.getRToken(3);

  p=pm.getSToken(2); if (p[0]=='t') ad->automatic=true;
  p=pm.getSToken(3); if (p[0]=='t') ad->formula=true;

  global.skgraph->add(ad);
}

void FicsProtocolParser::seekRemove(ExtPatternMatcher &pm) {
  tstring t;
  int i;
  t.setFail(-1);
  t.set(pm.getStarToken(0));
  ensureSeekGraph();
  while( (i=t.tokenvalue(" \t\n")) >= 0 )
    global.skgraph->remove(i);
}

vector<string *> * FicsProtocolParser::getPlayerActions() {
  return(&PActions);
}

vector<string *> * FicsProtocolParser::getGameActions() {
  return(&GActions);
}

void FicsProtocolParser::callPlayerAction(char *player, string *action) {
  for(int i=0;i<PActions.size();i++)
    if ( (*PActions[i]) == (*action) ) { doPlayerAction(player, i); break; }
}

void FicsProtocolParser::callGameAction(int gameid, string *action) {
  for(int i=0;i<GActions.size();i++)
    if ( (*GActions[i]) == (*action) ) { doGameAction(gameid, i); break; }
}

void FicsProtocolParser::doPlayerAction(char *player, int id) {
  char z[256],w[256];
  bool ok=true;

  switch(id) {
  case 0: sprintf(z,"finger %s",player); break;
  case 1: sprintf(z,"stat %s",player); break;
  case 2: sprintf(z,"history %s",player); break;
  case 3: sprintf(z,"ping %s",player); break;
  case 4: sprintf(z,"log %s",player); break;
  default: ok=false;
  }

  if (ok) {
    sprintf(w,"> [issued from menu] %s",z);
    global.output->append(w,global.SelfInputColor);
    global.network->writeLine(z);
  }
}

void FicsProtocolParser::doGameAction(int gameid, int id) {
  char z[256], w[256];
  bool ok=true;

  switch(id) {
  case 0: sprintf(z,"allob %d",gameid); break;
  case 1: sprintf(z,"ginfo %d",gameid); break;
  default: ok=false;
  }

  if (ok) {
    sprintf(w,"> [issued from menu] %s",z);
    global.output->append(w,global.SelfInputColor);
    global.network->writeLine(z);
  }
}

void FicsProtocolParser::prepareBugPane() {
  char z[64];
  PartnerGame = atoi(GotPartnerGame.getNToken(0));
  sprintf(z,"observe %d",PartnerGame);
  global.network->writeLine(z);
}
