/* Hiawatha | target.c
 *
 * All the routines for sending the requested item back to the client.
 */
#include "config.h"
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SENDFILE
#include <sys/sendfile.h>
#endif
#include "global.h"
#include "alternative.h"
#include "libstr.h"
#include "libfs.h"
#include "target.h"
#include "log.h"
#include "cgi.h"
#include "send.h"
#ifdef HAVE_CACHE
#include "cache.h"
#endif
#ifdef HAVE_COMMAND
#include "command.h"
#endif

#define MAX_VOLATILE_SIZE    1048576
#define FILE_BUFFER_SIZE       32768
#define MAX_CGI_HEADER          8192
#define MAX_TRACE_HEADER        1024
#define VALUE_SIZE                64

#define FILESIZE_BUFFER_SIZE      30
#define CGI_BUFFER_SIZE        32768

#define rs_QUIT       -1
#define rs_DISCONNECT -2
#define rs_FORCE_QUIT -3

char *chunked  = "Transfer-Encoding: chunked\r\n";
char *hs_allow = "Allow: GET, HEAD, OPTIONS";
char *hs_post  = ", POST";
char *hs_trace = ", TRACE";

extern char *fb_filesystem;
extern char *fb_symlink;
extern char *hs_eol;
extern char *hs_conlen;
extern char *hs_contyp;

/* Read a file from disk and send it to the client.
 */
int send_file(t_session *session) {
	char *referer, *buffer = NULL, value[VALUE_SIZE + 1], *dot, *date, *range, *range_start, *range_end;
	long bytes_read, total_bytes, size, speed;
	off_t file_size, send_start, send_end, send_size;
	int  retval, handle, http_code;
	bool invalid_referer = false, prot_oke;
	struct stat status;
	struct tm *fdate;
#ifdef HAVE_CACHE
	t_cached_object *cached_object;
#endif

#ifdef HAVE_COMMAND
	increment_counter(COUNTER_FILE);
#endif
	http_code = 200;

	session->mimetype = get_mimetype(session->file_on_disk, session->config->mimetype);

	handle = -1;
	if (session->accept_gzip) {
		size = strlen(session->file_on_disk);
		memcpy(session->file_on_disk + size, ".gz\0", 4);
		if ((handle = open(session->file_on_disk, O_RDONLY)) != -1) {
			session->encode_gzip = true;
		} else {
			*(session->file_on_disk + size) = '\0';
		}
	}
	if (handle == -1) {
		if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
			if (errno == EACCES) {
				log_error(session, fb_filesystem);
				return 403;
			}
			return 404;
		}
	}

	/* Check the referer
	 */
	if (session->host->image_referer.size > 0) {
		if (strncmp(session->mimetype, "image/", 6) == 0) {
			invalid_referer = true;
			if ((referer = get_headerfield("Referer:", session->headerfields)) != NULL) {
				if (strncmp(referer, "http://", 7) == 0) {
					prot_oke = true;
					referer += 7;
				} else if (strncmp(referer, "https://", 8) == 0) {
					prot_oke = true;
					referer += 8;
				} else {
					prot_oke = false;
				}

				if (prot_oke) {
					if ((dot = strchr(referer, '/')) != NULL) {
						*dot = '\0';
					}
					for (size = 0; size < session->host->image_referer.size; size++) {
						if (strstr(referer, *(session->host->image_referer.item + size)) != NULL) {
							invalid_referer = false;
							break;
						}
					}
					if (dot != NULL) {
						*dot = '/';
					}
				}
			}
		}
	}
	if (invalid_referer) {
		free(session->file_on_disk);
		if ((session->file_on_disk = (char*)malloc(strlen(session->host->imgref_replacement) + 4)) != NULL) { // + 3 for ".gz" (gzip encoding)
			strcpy(session->file_on_disk, session->host->imgref_replacement);
			session->mimetype = get_mimetype(session->file_on_disk, session->config->mimetype);
		} else {
			close(handle);
			return 500;
		}
	}
	
	/* Symlink check
	 */
	if (session->host->follow_symlinks == false) {
		switch (contains_not_allowed_symlink(session->file_on_disk, session->host->website_root)) {
			case error:
				close(handle);
				log_error(session, "error while scanning file for symlinks");
				return 500;
			case not_found:
				close(handle);
				return 404;
			case no_access:
			case yes:
				close(handle);
				log_error(session, fb_symlink);
				return 403;
			case no:
				break;
		}
	}

	/* Modified-Since
	 */
	if (session->handling_error == false) {
		if ((date = get_headerfield("If-Modified-Since:", session->headerfields)) != NULL) {
			if (if_modified_since(handle, date) == 0) {
				close(handle);
				return 304;
			}
		} else if ((date = get_headerfield("If-Unmodified-Since:", session->headerfields)) != NULL) {
			if (if_modified_since(handle, date) == 1) {
				close(handle);
				return 412;
			}
		}
	}

	/* Set throttlespeed
	 */
	dot = session->uri + strlen(session->uri);
	while ((*dot != '.') && (dot != session->uri)) {
		dot--;
	}
	if (*dot == '.') {
		if ((speed = get_throttlespeed(dot, session->config->throttle)) != 0) {
			if ((session->throttle == 0) || (speed < session->throttle)) {
				session->throttle = speed;
			}
		}
		if ((speed = get_throttlespeed(session->mimetype, session->config->throttle)) != 0) {
			if ((session->throttle == 0) || (speed < session->throttle)) {
				session->throttle = speed;
			}
		}
	}

	if ((file_size = filesize(session->file_on_disk)) == -1) {
		close(handle);
		log_error(session, "error while determining filesize");
		return 500;
	}
	send_start = 0;
	send_end = file_size - 1;
	send_size = file_size;

	/* Range
	 */
	if (*(session->http_version + 7) == '1') {
		if ((range = get_headerfield("Range:", session->headerfields)) != NULL) {
			if (strncmp(range, "bytes=", 6) == 0) {
				range += 6;
				if (split_string(range, &range_start, &range_end, '-') == 0) {

					if (*range_start != '\0') {
						if ((send_start = str2int(range_start)) >= 0) {
							if (*range_end != '\0') {
								if ((send_end = str2int(range_end)) >= 0) {
									// bytes=XX-XX
									http_code = 206;
								}
							} else {
								// bytes=XX-
								http_code = 206;
							}
						}
					} else {
						if ((send_start = str2int(range_end)) >= 0) {
							// bytes=-XX
							send_start = file_size - send_start;
							http_code = 206;
						}
					}

					if (http_code == 206) {
						if (send_start >= file_size) {
							close(handle);
							return 416;
						}
						if (send_start < 0) {
							send_start = 0;
						}
						if (send_end >= file_size) {
							send_end = file_size - 1;
						}
						if (send_start <= send_end) {
							send_size = send_end - send_start + 1;
						} else {
							close(handle);
							return 416;
						}
					}

					/* Change filepointer offset
					 */
					if (send_start > 0) {
						if (lseek(handle, send_start, SEEK_SET) == -1) {
							http_code = 200;
						}
					}

					if (http_code == 200) {
						send_start = 0;
						send_end = file_size - 1;
						send_size = file_size;
					}
				}
			}
		}
	}

	do {
		retval = -1;
		if (send_header(session, http_code) == -1) {
			break;
		}

		value[VALUE_SIZE] = '\0';

		/* Last-Modified
		 */
		if (fstat(handle, &status) == -1) {
			break;
		} else if ((fdate = gmtime(&(status.st_mtime))) == NULL) {
			break;
		} else if (send_buffer(session, "Last-Modified: ", 15) == -1) {
			break;
		} else if (strftime(value, VALUE_SIZE, "%a, %d %b %Y %X GMT\r\n", fdate) == 0) {
			break;
		} else if (send_buffer(session, value, strlen(value)) == -1) {
			break;
		}

		/* Content-Range
		 */
		if (http_code == 206) {
			if (send_buffer(session, "Content-Range: bytes ", 21) == -1) {
				break;
			} else if (snprintf(value, VALUE_SIZE, "%lld-%lld/%lld\r\n", (long long)send_start, (long long)send_end, file_size) == -1) {
				break;
			} else if (send_buffer(session, value, strlen(value)) == -1) {
				break;
			}
		}

		if (send_buffer(session, hs_conlen, 16) == -1) {
			break;
		} else if (snprintf(value, VALUE_SIZE, "%lld\r\n\r\n", (long long)send_size) == -1) {
			break;
		} else if (send_buffer(session, value, strlen(value)) == -1) {
			break;
		}

		retval = 200;
		if (session->head_request == false) {
			if (is_volatile_object(session) && (file_size <= MAX_VOLATILE_SIZE)) {
				/* volatile object
				 */
				if ((buffer = (char*)malloc(send_size)) != NULL) {
					total_bytes = 0;
					do {
						if ((bytes_read = read(handle, buffer + total_bytes, send_size - total_bytes)) == -1) {
							if (errno == EINTR) {
								bytes_read = 0;
							}
						} else {
							total_bytes += bytes_read;
						}
					} while ((bytes_read != -1) && (total_bytes < send_size));
					if (bytes_read != -1) {
						if (send_buffer(session, buffer, send_size) == -1) {
							retval = -1;
						}
					} else {
						retval = -1;
					}
				} else {
					retval = -1;
				}
			} else {
				/* Normal file
				 */
#ifdef HAVE_SENDFILE
				/* This is for sendfile() under Linux. It turned out that it's
				 * slower (about 5%) then Hiawatha's 'sendfile' code.
				 */
				off_t offset;

				if ((session->throttle == 0) 
#ifdef HAVE_SSL
						&& (session->binding->use_ssl == false)
#endif
						) {
					send_buffer(session, NULL, 0);
					offset = send_start;
					if (sendfile(session->client_socket, handle, &offset, send_size) == -1) {
						retval = -1;
					}
				} else 
#endif

#ifdef HAVE_CACHE
				if ((cached_object = search_cache(session, session->file_on_disk)) != NULL) {
					send_buffer(session, cached_object->data, cached_object->size);
					done_with_cached_object(cached_object);
				} else if ((cached_object = add_to_cache(session, session->file_on_disk)) != NULL) {
					send_buffer(session, cached_object->data, cached_object->size);
					done_with_cached_object(cached_object);
				} else
#endif
				if ((buffer = (char*)malloc(FILE_BUFFER_SIZE)) != NULL) {
					while ((send_size > 0) && (retval == 200)) {
						switch ((bytes_read = read(handle, buffer, FILE_BUFFER_SIZE))) {
							case -1:
								if (errno != EINTR) {
									retval = -1;
								}
								break;
							case 0:
								send_size = 0;
								break;
							default:
								if (bytes_read > send_size) {
									bytes_read = send_size;
								}
								if (send_buffer(session, buffer, bytes_read) == -1) {
									retval = -1;
								}
								send_size -= bytes_read;
						}
					}
				} else {
					retval = -1;
				}
			}
			if (buffer != NULL) {
				free(buffer);
			}
		}
	} while (false);
	close(handle);

	return retval;
}

/* Run a CGI program and send output to the client.
 */
int execute_cgi(t_session *session) {
	int retval = 200, result, handle;
	char *line, *new_line, *contyp_start, *contyp_end, *code;
	bool in_body = false, send_in_chunks = true, wrap_cgi;
	t_connect_to *connect_to;
	t_cgi_info cgi_info;
	pid_t cgi_pid = -1;

#ifdef HAVE_COMMAND
	increment_counter(COUNTER_CGI);
#endif

	if (session->cgi_type != fastcgi) {
		wrap_cgi = (session->config->cgi_wrapper != NULL) && ((session->host->wrap_cgi != NULL) ||
		           ((session->local_user != NULL) && session->config->wrap_user_cgi));
	} else {
		wrap_cgi = false;
	}

	/* HTTP/1.0 does not support chunked Transfer-Encoding.
	 */
	if (*(session->http_version + 7) == '0') {
		session->keep_alive = false;
	}

	if ((wrap_cgi == false) && (session->cgi_type != fastcgi)) {
		if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
			if (errno == EACCES) {
				log_error(session, fb_filesystem);
				return 403;
			}
			return 404;
		} else {
			close(handle);
		}
	}

	if (session->host->execute_cgi == false) {
		log_error(session, "CGI execution not allowed");
		return 403;
	}

	if ((wrap_cgi == false) && (session->cgi_type != fastcgi)) {
		if (session->cgi_type == program) {
			switch (can_execute(session->file_on_disk, session->config->server_uid, session->config->server_gid, &(session->config->groups))) {
				case error:
					log_error(session, "error during CGI preprocess");
					return 500;
				case not_found:
					return 404;
				case no_access:
				case no:
					log_error(session, fb_filesystem);
					return 403;
				case yes:
					break;
			}
		}

		if (session->host->follow_symlinks == false) {
			switch (contains_not_allowed_symlink(session->file_on_disk, session->host->website_root)) {
				case error:
					log_error(session, "error while searching for symlinks in CGI path");
					return 500;
				case not_found:
					return 404;
				case no_access:
				case yes:
					log_error(session, fb_symlink);
					return 403;
				case no:
					break;
			}
		}
	}

	if (session->host->prevent_xss) {
		/* Prevent Cross-site scripting
		 */
		prevent_XSS(session->vars);
	}

	if (session->host->prevent_sqli) {
		/* Prevent SQL injection
		 */
		if (session->vars != NULL) {
			switch (prevent_SQLi(session->vars, strlen(session->vars) + 1, &line)) {
				case -1:
					return 500;
				case 0:
					break;
				default:
					if (session->config->ban_on_sqli > 0) {
						free(line);
						return rr_SQL_INJECTION;
					}
					session->vars = line;
					session->vars_copied = true;
			}
		}

		if (session->body != NULL) {
			session->content_length = parse_special_chars(session->body, session->content_length);
			switch (result = prevent_SQLi(session->body, session->content_length, &line)) {
				case -1:
					return 500;
				case 0:
					break;
				default:
					if (session->config->ban_on_sqli > 0) {
						free(line);
						return rr_SQL_INJECTION;
					}
					session->body = line;
					session->content_length = result;
					session->body_copied = true;
			}
		}

		if ((session->cookie = get_headerfield("Cookie:", session->headerfields)) != NULL) {
			parse_special_chars(session->cookie, strlen(session->cookie));
			switch (prevent_SQLi(session->cookie, strlen(session->cookie), &line)) {
				case -1:
					return 500;
				case 0:
					break;
				default:
					if (session->config->ban_on_sqli > 0) {
						free(line);
						return rr_SQL_INJECTION;
					}
					session->cookie = line;
					session->cookie_copied = true;
			}
		}
	}

	if (session->host->prevent_cmdi) {
		/* Prevent Command injection
		 */
		if ((prevent_CMDi(session->vars) > 0) && (session->config->ban_on_cmdi > 0)) {
			return rr_CMD_INJECTION;
		}
		if ((prevent_CMDi(session->body) > 0) && (session->config->ban_on_cmdi > 0)) {
			return rr_CMD_INJECTION;
		}
		if ((prevent_CMDi(session->cookie) > 0) && (session->config->ban_on_cmdi > 0)) {
			return rr_CMD_INJECTION;
		}
	}

	cgi_info.input_buffer_size = cgi_info.error_buffer_size = CGI_BUFFER_SIZE;
	cgi_info.input_len = cgi_info.error_len = 0;

	if (session->cgi_type == fastcgi) {
		cgi_info.read_header = true;
		if ((connect_to = select_connect_to(session->fcgi_server, session->ip_address)) == NULL) {
			return 503;
		} else if ((cgi_info.from_cgi = connect_to_fcgi_server(connect_to)) == -1) {
			connect_to->available = false;
			log_string(session->config->system_logfile, "can't connect to FastCGI server");
			return 503;
		} else {
			connect_to->available = true;
			if (send_fcgi_request(session, cgi_info.from_cgi) == -1) {
				log_error(session, "error while sending data to FastCGI server");
				return 500;
			}
		}
	} else {
		cgi_info.wrap_cgi = wrap_cgi;
		if ((cgi_pid = fork_cgi_process(session, &cgi_info)) == -1) {
			log_error(session, "error while forking CGI process");
			return 500;
		}
	}

	if ((cgi_info.input_buffer = (char*)malloc(cgi_info.input_buffer_size + 1)) == NULL) {
		retval = 500;
	} else if ((cgi_info.error_buffer = (char*)malloc(cgi_info.error_buffer_size + 1)) == NULL) {
		free(cgi_info.input_buffer);
		retval = 500;
	}

	if (retval != 200) {
		if (session->cgi_type == fastcgi) {
			close(cgi_info.from_cgi);
		} else {
			close(cgi_info.to_cgi);
			close(cgi_info.from_cgi);
			close(cgi_info.cgi_error);
		}
		return retval;
	}

	cgi_info.select_timeout.tv_sec = 1;
	cgi_info.select_timeout.tv_usec = 0;
	cgi_info.timer = session->config->time_for_cgi;

	do {
		if (session->cgi_type == fastcgi) {
			result = read_from_fcgi_server(session, &cgi_info);
		} else {
			result = read_from_cgi_process(session, &cgi_info);
		}

		switch (result) {
			case cgi_ERROR:
				log_file_error(session, session->file_on_disk, "error while executing CGI");
				retval = 500;
				break;
			case cgi_TIMEOUT:
				log_file_error(session, session->file_on_disk, "CGI timeout");
				if (in_body) {
					retval = rs_DISCONNECT;
				} else {
					retval = 500;
				}
				if (session->config->kill_timedout_cgi && (session->cgi_type != fastcgi)) {
					if (kill(cgi_pid, 15) != -1) {
						sleep(1);
						kill(cgi_pid, 9);
					}
				}
				break;
			case cgi_FORCE_QUIT:
				retval = rs_FORCE_QUIT;
				break;
			case cgi_OKE:
				if (cgi_info.error_len > 0) {
					/* Error received from CGI
					 */
					*(cgi_info.error_buffer + cgi_info.error_len) = '\0';
					log_file_error(session, session->file_on_disk, cgi_info.error_buffer);
					cgi_info.error_len = 0;
				}

				if (cgi_info.input_len > 0) {
					/* Data received from CGI
					 */
					if (in_body) {
						/* Read content
						 */
						if (session->head_request == false) {
							if (send_in_chunks) {
								if (send_chunk(session, cgi_info.input_buffer, cgi_info.input_len) == -1) {
									retval = rs_DISCONNECT;
								}
							} else {
								if (send_buffer(session, cgi_info.input_buffer, cgi_info.input_len) == -1) {
									retval = rs_DISCONNECT;
								}
							}
						}
						cgi_info.input_len = 0;
					} else {
						/* Read HTTP header
						 */
						*(cgi_info.input_buffer + cgi_info.input_len) = '\0';

						if ((new_line = strstr(cgi_info.input_buffer, "\r\n\r\n")) != NULL) {
							if (session->throttle == 0) {
								if ((contyp_start = strcasestr(cgi_info.input_buffer, hs_contyp)) != NULL) {
									if ((contyp_end = strchr(contyp_start, '\r')) != NULL) {
										contyp_start += 14;
										*contyp_end = '\0';
										session->throttle = get_throttlespeed(contyp_start, session->config->throttle);
										*contyp_end = '\r';
									}
								}
							}

							if (strcasestr(cgi_info.input_buffer, "Location:") != NULL) {
								if (send_header(session, 302) == -1) {
									retval = rs_DISCONNECT;
									break;
								}
							} else if ((code = strcasestr(cgi_info.input_buffer, "Status: ")) != NULL) {
								line = code += 8;
								while (*line != '\0') {
									if (*line == '\r') {
										*line = '\0';
										break;
									}
									line++;
								}

								if ((retval = str2int(code)) <= 0) {
									log_file_error(session, session->file_on_disk, "invalid status code received from CGI");
									retval = 500;
								}
								break;
							} else if (send_header(session, session->return_code) == -1) {
								retval = rs_DISCONNECT;
								break;
							}
							if (session->return_code == 401) {
								if (session->host->auth_method == basic) {
									send_basic_auth(session);
								} else {
									send_digest_auth(session);
								}
							}
							if ((strcasestr(cgi_info.input_buffer, hs_conlen) != NULL) || (session->keep_alive == false)) {
								send_in_chunks = false;
							} else if (send_buffer(session, chunked, 28) == -1) {
								retval = rs_DISCONNECT;
								break;
							}

							/* Send the header.
							 */
							if (send_in_chunks || session->head_request) {
								new_line += 4;
								result = new_line - cgi_info.input_buffer;
								if (send_buffer(session, cgi_info.input_buffer, result) == -1) {
									retval = rs_DISCONNECT;
									break;
								}
								if (send_buffer(session, NULL, 0) == -1) {
									retval = rs_DISCONNECT;
									break;
								}
								if (session->head_request == false) {
									result = cgi_info.input_len - result;
									if (result > 0) {
										if (send_chunk(session, new_line, result) == -1) {
											retval = rs_DISCONNECT;
											break;
										}
									}
								} else {
									//retval = rs_QUIT;
									//break;
								}
							} else {
								if (send_buffer(session, cgi_info.input_buffer, cgi_info.input_len) == -1) {
									retval = rs_DISCONNECT;
									break;
								}
							}

							in_body = true;
							cgi_info.input_len = 0;
						} else if (cgi_info.input_len > MAX_CGI_HEADER) {
							retval = 413;
							break;
						}
					}
				}
				break;
			case cgi_END_OF_DATA:
				if (in_body) {
					retval = rs_QUIT;
					if (send_in_chunks) {
						if (send_chunk(session, NULL, 0) == -1) {
							retval = rs_DISCONNECT;
						}
					}
				} else {
					if (cgi_info.input_len == 0) {
						log_file_error(session, session->file_on_disk, "no output");
					} else if (strstr(cgi_info.input_buffer, "\r\n\r\n") == NULL) {
						log_file_error(session, session->file_on_disk, "non-RFC HTTP header ('\\n' instead of '\\r\\n')");
					} else {
						log_error(session, "CGI only printed a HTTP header, no content");
					}
					retval = 500;
				}
		} // switch
	} while (retval == 200);

	if (session->cgi_type == fastcgi) {
		close(cgi_info.from_cgi);
	} else {
		close(cgi_info.to_cgi);
		close(cgi_info.from_cgi);
		close(cgi_info.cgi_error);
	}

	switch (retval) {
		case rs_DISCONNECT:
		case rs_FORCE_QUIT:
			session->keep_alive = false;
		case rs_QUIT:
			retval = 200;
	}

	free(cgi_info.input_buffer);
	free(cgi_info.error_buffer);

	return retval;
}

/* Converts a filesize to a string.
 */
static int filesize2str(char *buffer, off_t fsize) {
	int result = 0;

	buffer[FILESIZE_BUFFER_SIZE - 1] = '\0';
	if (fsize < KILOBYTE) {
		result = snprintf(buffer, FILESIZE_BUFFER_SIZE - 1, "%llu byte", (long long)fsize);
	} else if (fsize < MEGABYTE) {
		result = snprintf(buffer, FILESIZE_BUFFER_SIZE - 1, "%0.1f kB", ((double)(fsize >> 6)) / 16);
	} else if (fsize < GIGABYTE) {
		result = snprintf(buffer, FILESIZE_BUFFER_SIZE - 1, "%0.1f MB", ((double)(fsize >> 16)) / 16);
	} else {
		result = snprintf(buffer, FILESIZE_BUFFER_SIZE - 1, "%0.1f GB", ((double)(fsize >> 26)) / 16);
	}

	return (result < 0) ? 0 : result;
}

/* Read the content of a directory and send it in a formatted list to the client.
 */
int show_index(t_session *session) {
	int total_files = 0, retval = -1, len;
	off_t total_fsize = 0;
	char size[FILESIZE_BUFFER_SIZE], color = '0', timestr[33], *slash;
	struct tm s;
	t_filelist *filelist = NULL, *file;
	t_keyvalue *alias;
	bool root_dir;

#ifdef HAVE_COMMAND
	increment_counter(COUNTER_INDEX);
#endif

	session->mimetype = NULL;

	if ((slash = strrchr(session->file_on_disk, '/')) == NULL) {
		return 500;
	}
	*(slash + 1) = '\0';

	switch (is_directory(session->file_on_disk)) {
		case error:
		case no:
			return 500;
		case not_found:
			return 404;
		case no_access:
			log_error(session, fb_filesystem);
			return 403;
			break;
		case yes:
			break;
	}

	if (session->host->follow_symlinks == false) {
		switch (contains_not_allowed_symlink(session->file_on_disk, session->host->website_root)) {
			case error:
				return 500;
			case not_found:
				return 404;
			case no_access:
			case yes:
				log_error(session, fb_symlink);
				return 403;
			case no:
				break;
		}
	}

    /* HTTP/1.0 has no knowledge about chunked Transfer-Encoding.
	 */
	if (*(session->http_version + 7) == '0') {
		session->keep_alive = false;
	}

	if (send_header(session, 200) == -1) {
		return -1;
	}

	if (session->head_request) {
		send_buffer(session, "\r\n", 2);
		return 200;
	}

	/* Read directory content
	 */
	if ((filelist = read_filelist(session->file_on_disk)) == NULL) {
		return 500;
	}

	/* Add aliasses to directory list
	 */
	alias = session->host->alias;
	while (alias != NULL) {
		if ((file = (t_filelist*)malloc(sizeof(t_filelist))) != NULL) {
			if ((file->name = strdup(alias->key + 1)) == NULL) {
				remove_filelist(filelist);
				return 500;
			}
			file->size = 0;
			file->time = time(NULL);
			file->is_dir = true;
			file->next = filelist;
			filelist = file;
		} else {
			remove_filelist(filelist);
			return 500;
		}
		alias = alias->next;
	}

	file = filelist = sort_filelist(filelist);

	do {
		if (session->keep_alive) {
			if (send_buffer(session, chunked, 28) == -1) {
				break;
			}
		} 
		if (send_buffer(session, "Content-Type: text/html\r\n\r\n", 27) == -1) {
			break;
		} else if (send_buffer(session, NULL, 0) == -1) {
			break;
		} else if (send_chunk(session, "<html><head><title>Index of ", 28) == -1) {
			break;
		} else if (send_chunk(session, session->uri, strlen(session->uri)) == -1) {
			break;
		} else if (send_chunk(session,	"</title>\n", 9) == -1) {
			break;
		}

		if (session->host->index_style != NULL) {
			if (send_chunk(session, "<link rel=\"stylesheet\" type=\"text/css\" href=\"", 45) == -1) {
				break;
			} else if (send_chunk(session, session->host->index_style, strlen(session->host->index_style)) == -1) {
				break;
			} else if (send_chunk(session, "\">\n", 3) == -1) {
				break;
			}
		} else {
			if (send_chunk(session, "<style type=\"text/css\">BODY { margin:50px 100px 50px 100px ; background-color:#000008 ; font-family:sans-serif } TABLE.list { width:100% ; border-left:5px solid #8080ff ; border-right:5px solid #8080ff ; border-collapse:collapse } A { color:#900000 ; text-decoration:none } A:visited { color:#404040 } A:hover { color:#ff4040 ; text-decoration:underline } DIV.path { color:#ffffff ; font-size:20px ; text-align:right ; font-weight:bold ; margin-bottom:10px ; margin-right:25px} TR { font-size:12px } TH { background-color:#8080ff } TH.left { width:40px } TH.filename { text-align:left } TH.date { width:150px } TH.size { width:80px ; text-align:right} TH.right { width:30px } TD.left { padding-left:5px } TD.size { text-align:right } TR.line0 { background-color:#e0e0ff } TR.line1 { background-color:#e8e8ff } TR.bottom { background-color:#8080ff ; font-weight:bold } TD.totalsize { text-align:right }</style>\n", 910) == -1) {
				break;
			}
		}

		if (send_chunk(session,"<body><div class=\"path\">", 24) == -1) { 
			break;
		} else if (send_chunk(session, *(session->host->hostname.item), strlen(*(session->host->hostname.item))) == -1) {
			break;
		} else if (send_chunk(session, " : ", 3) == -1) {
			break;
		} else if (send_chunk(session, session->uri, strlen(session->uri)) == -1) {
			break;
		} else if (send_chunk(session, "</div>\n<table class=\"list\">\n<tr><th class=\"left\"></th><th class=\"filename\">Filename</th><th class=\"date\">Date</th><th class=\"size\">Size</th><th class=\"right\"></th></tr>\n", 169) == -1) {
			break;
		} 

		root_dir = (strcmp(session->uri, "/") == 0);

		while (file != NULL) {
			if (file->is_dir && root_dir) {
				if (strcmp(file->name, "..") == 0) {
					file = file->next;
					continue;
				}
			}

			if (send_chunk(session, "<tr class=\"line", 15) == -1) {
				break;
			} else if (send_chunk(session, &color, 1) == -1) {
				break;
			} else if (send_chunk(session, "\"><td class=\"left\">", 19) == -1) {
				break;
			}
			if (file->is_dir) {
				if (send_chunk(session, "[dir]", 5) == -1) {
					break;
				}
			}
			if (send_chunk(session, "</td><td class=\"filename\"><a href=\"", 35) == -1) {
				break;
			} else if (send_chunk(session, file->name, strlen(file->name)) == -1) {
				break;
			}
			if (file->is_dir) {
				if (send_chunk(session, "/", 1) == -1) {
					break;
				}
			}
			if (send_chunk(session, "\">", 2) == -1) {
				break;
			} else if (send_chunk(session, file->name, strlen(file->name)) == -1) {
				break;
			}
			if (file->is_dir) {
				if (send_chunk(session, "/", 1) == -1) {
					break;
				}
			}
			if (send_chunk(session, "</a></td><td class=\"date\">", 26) == -1) {
				break;
			}
			s = *localtime(&(file->time));
			*(timestr + 32) = '\0';
			strftime(timestr, 32, "%d %b %Y, %X", &s);
			if (send_chunk(session, timestr, strlen(timestr)) == -1) {
				break;
			} else if (send_chunk(session, "</td><td class=\"size\">", 22) == -1) {
				break;
			}
			if (file->is_dir == false) {
				len = filesize2str(size, file->size);
				if (send_chunk(session, size, len) == -1) {
					break;
				}
			}
			if (send_chunk(session, "</td><td class=\"right\"></td></tr>\n", 34) == -1) {
				break;
			}
			color = '0' + (1 - (color - '0'));

			total_files++;
			if (file->is_dir == false) {
				total_fsize += file->size;
			}
			file = file->next;
		}
		if (file != NULL) {
			break;
		}

		if (send_chunk(session,	"<tr class=\"bottom\"><td></td><td class=\"totalfiles\">", 51) == -1) {
			break;
		}
		snprintf(size, FILESIZE_BUFFER_SIZE - 1, "%d", total_files);
		if (send_chunk(session, size, strlen(size)) == -1) {
			break;
		} else if (send_chunk(session, " files</td><td></td><td class=\"totalsize\">", 42) == -1) {
			break;
		}
		len = filesize2str(size, total_fsize);
		if (send_chunk(session, size, len) == -1) {
			break;
		} else if (send_chunk(session, "</td><td></td></tr>\n</table></body></html>\n", 43) == -1) {
			break;
		} else  if (session->keep_alive) {
			if (send_chunk(session, NULL, 0) == -1) {
				break;
			}
		}

		retval = 200;
	} while (false);

	remove_filelist(filelist);

	return retval;
}

/* Send the result of an OPTIONS request method to the client.
 */
int handle_options_request(t_session *session) {
	int code = 200, handle;

	if ((handle = open(session->file_on_disk, O_RDONLY)) == -1) {
		if (session->uri_is_dir) {
			code = 501;
		} else {
			code = 404;
		}
		if (errno == EACCES) {
			log_error(session, fb_filesystem);
			code = 403;
		}
	} else {
		close(handle);
	}
	if (session->host->follow_symlinks == false) {
		switch (contains_not_allowed_symlink(session->file_on_disk, session->host->website_root)) {
			case error:
				code = 500;
				break;
			case not_found:
				code = 404;
				break;
			case no_access:
			case yes:
				log_error(session, fb_symlink);
				code = 403;
				break;
			case no:
				break;
		}
	}

	if (send_header(session, code) == -1) {
		return -1;
	} else if (send_buffer(session, hs_allow, 25) == -1) {
		return -1;
	} else if (session->cgi_type != none) {
		if (send_buffer(session, hs_post, 6) == -1) {
			return -1;
		}
	}
	if (session->binding->enable_trace) {
		if (send_buffer(session, hs_trace, 7) == -1) {
			return -1;
		}
	}
	if (send_buffer(session, hs_eol, 2) == -1) {
		return -1;
	} else if (send_buffer(session, hs_conlen, 16) == -1) {
		return -1;
	} else if (send_buffer(session, "0\r\n\r\n", 5) == -1) {
		return -1;
	}

	return 200;
}

int handle_trace_request(t_session *session) {
	int result = -1, code, body_size;
	size_t len;
	char buffer[MAX_TRACE_HEADER + 1];
	t_headerfield *header;

	body_size = 3;
	body_size += strlen(session->method) + strlen(session->uri);
	if (session->vars != NULL) {
		body_size += 1 + strlen(session->vars);
	}
	body_size += strlen(session->http_version);

	header = session->headerfields;
	while (header != NULL) {
		body_size += strlen(header->data) + 1;
		header = header->next;
	}

	buffer[MAX_TRACE_HEADER] = '\0';

	do {
		// Header
		if (snprintf(buffer, MAX_TRACE_HEADER, "%d\r\nContent-Type: message/http\r\n\r\n", body_size) < 0) {
			break;
		} else if (send_header(session, 200) == -1) {
			break;
		} else if (send_buffer(session, hs_conlen, 16) == -1) {
			break;
		} else if (send_buffer(session, buffer, strlen(buffer)) == -1) {
			break;
		}

		// Body
		if ((code = snprintf(buffer, MAX_TRACE_HEADER, "%s %s", session->method, session->uri)) < 0) {
			break;
		} else if (code >= MAX_TRACE_HEADER) {
			break;
		} else if (session->vars != NULL) {
			len = strlen(buffer);
			if ((code = snprintf(buffer + len, MAX_TRACE_HEADER - len, "?%s", session->vars)) < 0) {
				break;
			} else if (code >= MAX_TRACE_HEADER) {
				break;
			}
		}
		len = strlen(buffer);
		if ((code = snprintf(buffer + len, MAX_TRACE_HEADER - len, " %s\r\n", session->http_version)) < 0) {
			break;
		} else if (send_buffer(session, buffer, strlen(buffer)) == -1) {
			break;
		}

		header = session->headerfields;
		while (header != NULL) {
			len = strlen(header->data);
			*(header->data + len) = '\n';
			if (send_buffer(session, header->data, len + 1) == -1) {
				*(header->data + len) = '\0';
				result = -2;
				break;
			}
			*(header->data + len) = '\0';
			header = header->next;
		}
		if (result == -2) {
			break;
		}
	
		result = 200;
	} while (false);

	return result;
}
