/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <pthread.h>
#include "sudoku.h"

/*
 ******************************************************************************/
void copy_sudoku(t_sudoku sudokuSrc, t_sudoku *sudokuDest) {
	*sudokuDest = sudokuSrc;
}

/* sets values to zero, candidates all != 0
 ******************************************************************************/
void reset_sudoku(t_sudoku *sudoku) {

	int i, j, k;

	sudoku->size = SIZE2;
	sudoku->rating = 0;
	
	for (i=0; i<SIZE2; i++) {
		for (j=0; j<SIZE2; j++) {
			sudoku->grid[i][j].value = 0;
			for (k=0; k<SIZE2; k++) sudoku->grid[i][j].cand[k] = k+1;
			sudoku->grid[i][j].ncand = SIZE2;
			sudoku->grid[i][j].stat = stat_free;
		}
	}
}

/*
 ******************************************************************************/
void transpose_sudoku(t_sudoku *sudoku) {
	
	int r, c;
	int size = sudoku->size;
	t_sudoku sudoku0;
	
	copy_sudoku(*sudoku, &sudoku0);
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			sudoku->grid[r][c] = sudoku0.grid[c][r];
		}
	}
}

/* brcMap must contain each number from 0 to SIZE-1 exactly once
 ******************************************************************************/
void exchange_block_rc(t_sudoku *sudoku, char ent, int brcMap[SIZE]) {
	
	int br, bc, r, c, rNew, rOld, cc;
	t_sudoku sudoku0;
	
	copy_sudoku(*sudoku,&sudoku0);
	if (ent == 'c') transpose_sudoku(&sudoku0);
	for (br = 0; br < SIZE; br++) {
		for (r = 0; r < SIZE; r++) {
			rNew = brcMap[br] * SIZE + r;
			rOld = br         * SIZE + r;
			for (bc = 0; bc < SIZE; bc++) {
				for (c = 0; c < SIZE; c++) {
					cc = bc * SIZE + c;
					sudoku->grid[rNew][cc] = sudoku0.grid[rOld][cc];
				}
			}
		}
	}
	if (ent == 'c') transpose_sudoku(sudoku);
}

/* rcMap must contain each number from 0 to SIZE2-1 exactly once. To maintain
 * sudoku properties, lines and columns may be swapped only within block rows
 * and block columns. No checking is performed here.
 ******************************************************************************/
void exchange_rc(t_sudoku *sudoku, char ent, int rcMap[SIZE2]) {
	
	int r, c, rNew;
	t_sudoku sudoku0;
	
	copy_sudoku(*sudoku,&sudoku0);
	if (ent == 'c') transpose_sudoku(&sudoku0);
	for (r = 0; r < SIZE2; r++) {
		rNew = rcMap[r];
		for (c = 0; c < SIZE2; c++) {
			sudoku->grid[rNew][c] = sudoku0.grid[r][c];
		}
	}
	if (ent == 'c') transpose_sudoku(sudoku);
}

/* valMap must contain each number from 1 to SIZE2 exactly once
 ******************************************************************************/
void exchange_value(t_sudoku *sudoku, int valMap[SIZE2]) {
	
	int r, c, cc;
	t_field f;
	
	for (r = 0; r < SIZE2; r++) {
		for (c = 0; c < SIZE2; c++) {
			if (sudoku->grid[r][c].value > 0)
				sudoku->grid[r][c].value = valMap[sudoku->grid[r][c].value - 1];
			f = sudoku->grid[r][c];
			for (cc = 0; cc < SIZE2; cc++) {
				if (f.cand[cc] > 0)
					sudoku->grid[r][c].cand[valMap[cc]-1] = valMap[cc];
				else
					sudoku->grid[r][c].cand[valMap[cc]-1] = 0;
			}
		}
	}
}

/*
 ******************************************************************************/
int get_num_cell_with_stat1(t_sudoku sudoku, status stat) {

	int size = sudoku.size;
	int r, c, nstat=0;

	// count cells with status stat
	if (stat == stat_undef) {
		nstat = size*size;
	} else {
		for (r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				if (sudoku.grid[r][c].stat == stat) nstat++;
			}
		}
	}
	return(nstat);
}

/*
 ******************************************************************************/
int check_sudoku_symmetry(t_sudoku sudoku) {
	
	int size = sudoku.size;
	int r, c, hsym=1, vsym=1;

	// make if conditions easier to program
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if (sudoku.grid[r][c].stat == stat_ggiven) {
				sudoku.grid[r][c].stat = stat_given;
			} else if (sudoku.grid[r][c].stat != stat_given) {
				sudoku.grid[r][c].stat = stat_free;
			}
		}
	}
	
	// check for symmetry axes
	for (r=0; r<size/2; r++) {
		for (c=0; c<size; c++) {
			if ((vsym == 1) && (sudoku.grid[r][c].stat != sudoku.grid[size-1-r][c].stat)) vsym = 0;
			if ((hsym == 1) && (sudoku.grid[c][r].stat != sudoku.grid[c][size-1-r].stat)) hsym = 0;
		}
	}

	// return
	if      ((hsym == 1) && (vsym == 1)) return(3);
	else if ((hsym == 1) && (vsym == 0)) return(2);
	else if ((hsym == 0) && (vsym == 1)) return(1);
	return(0);
}

/* use random number from rand()
 ******************************************************************************/
int get_random_cell(t_field grid[SIZE*SIZE][SIZE*SIZE], int size, status stat) {
	int r, c, nstat=0, istat;

	// count cells with status stat
	if (stat == stat_undef) {
		nstat = size*size;
	} else {
		for (r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				if (grid[r][c].stat == stat) nstat++;
			}
		}
	}

	// pick random cell
	istat = rand() % nstat;

	// find row / column
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if ((stat == stat_undef) || (grid[r][c].stat == stat)) {
				if (istat > 0) istat--;
				else return(r*size+c);
			}
		}
	}

	// return error
	return(-1);
}

/* use random number from /dev/urandom
 ******************************************************************************/
int get_random_cell2(t_field grid[SIZE*SIZE][SIZE*SIZE], int size, status stat) {
	int r, c, nstat=0;
	unsigned int istat;
	extern FILE *devurandom;
	
	// count cells with status stat
	if (stat == stat_undef) {
		nstat = size*size;
	} else {
		for (r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				if (grid[r][c].stat == stat) nstat++;
			}
		}
	}
		
	// pick random cell
	if (devurandom != NULL) {
		if (fread(&istat,sizeof(unsigned int),1,devurandom) != 1) {
			fprintf(stderr,"error while trying to read from /dev/urandom\n");
		}
	} else {
		fprintf(stderr,"error while trying to open /dev/urandom\n");
	}

	// if reading from /dev/urandom fails, just use uninitialized istat
	istat = istat % nstat;

	// find row / column
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if ((stat == stat_undef) || (grid[r][c].stat == stat)) {
				if (istat > 0) istat--;
				else return(r*size+c);
			}
		}
	}
	// return error
	return(-1);
}

/*
 ******************************************************************************/
int read_sudoku_from_file(FILE *f, t_sudoku *sudoku) {
	
	int i, j, c, *a=NULL, num=0;
	
	while (((c=getc(f)) != EOF) && (num < SIZE2*SIZE2)) {
		if ((c > 47) && (c < 58)) {
			c -= 48;
			a = (int *) realloc(a,(++num)*sizeof(int));
			a[num-1]=c;
		}	else if (! isspace(c)) {
			c=0; // all non-digits and non-space chars are interpreted as '0'
			a = (int *) realloc(a,(++num)*sizeof(int));
			a[num-1]=c;
		}
	}
	if (num != SIZE2*SIZE2) return(-1);
	for (i=0; i<SIZE2; i++) {
		for (j=0; j<SIZE2; j++) {
			sudoku->grid[i][j].value = a[i * SIZE2 + j];
			if (sudoku->grid[i][j].value == 0)
				sudoku->grid[i][j].stat = stat_free;
			else
				sudoku->grid[i][j].stat = stat_given;
		}
	}
	free(a);

	sudoku->size = SIZE2;
	sudoku->rating = 0;
	return(0);
}

/*
 ******************************************************************************/
int read_sudoku_and_candidates_from_file(FILE *f, t_sudoku *sudoku) {
	
	int i, j, k, c, *a=NULL, num;
	int size = SIZE2;
	
	reset_sudoku(sudoku);
	
	if (DEBUG) printf("Terminate program with CTRL-c, send EOF with CTRL-d.\nWaiting for input from stdin ...");
	num = 0;
	while (((c=getc(f)) != EOF) && (num < size*size*(size+1))) {
		if ((c > 47) && (c < 58)) {
			c -= 48;
			a = (int *) realloc(a,(++num)*sizeof(int));
			a[num-1]=c;
		}	else if (! isspace(c)) {
			c=0; // all non-digits and non-space chars are interpreted as '0'
			a = (int *) realloc(a,(++num)*sizeof(int));
			a[num-1]=c;
		}
	}
	if (num != size*size*(size+1)) {
		free(a);
		return(-1);
	}
	for (i=0; i<size; i++) {
		for (j=0; j<size; j++) {
			sudoku->grid[i][j].value = a[i * size + j];
			if (sudoku->grid[i][j].value == 0)
				sudoku->grid[i][j].stat = stat_free;
			else
				sudoku->grid[i][j].stat = stat_given;
		}
	}
	for (i=0; i<size; i++) {
		for (j=0; j<size; j++) {
			for (k=0; k<size; k++) {
				sudoku->grid[i][j].cand[k] = a[(i+1) * size*size + j * size + k];
			}
		}
	}
	free(a);

	return(0);
}

/*
 ******************************************************************************/
void set_stat_and_ncand_from_values_and_cands(t_sudoku *sudoku) {
	
	int size = sudoku->size;
	int r, c, cc;
	
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			sudoku->grid[r][c].ncand = 0;
			if (sudoku->grid[r][c].value > 0) {
				sudoku->grid[r][c].stat = stat_given;
			} else {
				sudoku->grid[r][c].stat = stat_free;
				for (cc=0; cc<size; cc++)
					if (sudoku->grid[r][c].cand[cc] > 0)
						sudoku->grid[r][c].ncand++;
			}
		}
	}
}

/*
 ******************************************************************************/
int check_sudoku(t_sudoku sudoku) {
	
	int r, c, v, nv, ret=0, nc=0, br, bc;
	int size = sudoku.size;

	// if there are free cells without candidates return nc < 0;
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			nc += sudoku.grid[r][c].ncand;
			if ((sudoku.grid[r][c].value == 0) && (sudoku.grid[r][c].ncand < 1)) {
				if (DEBUG) printf("nothing fits at (%d,%d) any more\n",r+1,c+1);
				ret--;
			}
		}
	}
	if (ret != 0) return(ret);
	
	// if there are rows, columns or blocks with multiple occurences of the
	// same number return nc < 0
	for (r=0; r<size; r++) {
		for (v=0; v<size; v++) {
			for (nv=c=0; c<size; c++) {
				if (sudoku.grid[r][c].value == (v+1)) nv++;
			}
			if (nv > 1) {
				ret--;
				if (DEBUG) printf("%d appears %d times in row %d\n",v+1,nv,r+1);
			}
			for (nv=c=0; c<size; c++) {
				if (sudoku.grid[c][r].value == (v+1)) nv++;
			}
			if (nv > 1) {
				ret--;
				if (DEBUG) printf("%d appears %d times in column %d\n",v+1,nv,r+1);
			}
		}
	}	
	for (br=0; br<SIZE; br++) {
		for (bc=0; bc<SIZE; bc++) {
			for (v=0; v<size; v++) {
				nv = 0;
				for (r=br*SIZE; r<(br+1)*SIZE; r++) {
					for (c=bc*SIZE; c<(bc+1)*SIZE; c++) {
						if (sudoku.grid[r][c].value == (v+1)) nv++;
					}
				}
				if (nv > 1) {
					if (DEBUG) printf("%d appears %d times in block (%d,%d)\n",v+1,nv,br+1,bc+1);
					ret--;
				}
			}
		}
	}
	if (ret != 0) return(ret);

	// else return number of candidates	
	return(nc);
}

/*
 ******************************************************************************/
int get_basic_rating(t_sudoku sudoku, int elim[NUM_ET]) {
	
	int size = sudoku.size;
	int r, c, rating;

	// basic rating depends on most difficult technique
	if ((elim[et_fc]+elim[et_xw]+elim[et_sf]+elim[et_jf]+elim[et_g]) > 0)
		rating = 100;
	else if ((elim[et_b]+elim[et_l]+elim[et_d]+elim[et_t]+elim[et_ht]+elim[et_q]+elim[et_hq]) > 0)
		rating = 10;        // candidates to be determined
	else
		rating = 1;         // only hidden singles, singles and hidden doubles
	
	// remove 1 from rating for each cell to be solved, so a sudoku with just
	// hidden singles gets a rating of 1
	for (r=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if ((sudoku.grid[r][c].stat != stat_given) && (sudoku.grid[r][c].stat != stat_ggiven)) rating--;
		}
	}
	return(rating);
}

