/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <limits.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include "sudoku.h"

// initialize external identifiers
char progname[100];
int action = 0;
int max_rating = INT_MAX;
int num_sudokus = 4;
int verbosity = 0;
int nthread = 0;
int symmgrid = 0;
char *fn_out = NULL;
FILE *inFile = NULL;

pthread_mutex_t mut_input  = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t mut_stdout = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t mut_nsud   = PTHREAD_MUTEX_INITIALIZER;

int dfcs[NUM_ET] = {
	et_wg, //  0
	et_hs, //  1
	et_s,  //  2
	et_hd, //  3
	et_d,  //  4
	et_b,  //  5
	et_l,  //  6
	et_ht, //  7
	et_t,  //  8
	et_hq, //  9
	et_q,  // 10
	et_xw, // 11
	et_sf, // 12
	et_jf, // 13
	et_fc, // 14
	et_g   // 15
};

char fn_wsudognu[MAX_STRLEN+1] = "./wsudognu.cgi";
FILE *explFile;
FILE *devurandom;

/*
 ******************************************************************************/
int getoptions(int argc, char *argv[]) {
	
	int i, iet;
	char *fn;
	
	opterr = 0;
	
	while ((i = getopt (argc, argv, ":chpVwsStf:i:n:o:r:v:P:")) != -1) {
		switch (i) {
			case 'c':
				action = 1;
				break;
			case 'p':
				action = 2;
				break;
			case 's':
				action = 3;
				break;
			case 'S':
				symmgrid = 1;
				break;
			case 't':
				nthread = sysconf(_SC_NPROCESSORS_ONLN);
				break;
			case 'w':
				action = 4;
				break;
			case 'f':
				if (strlen(optarg) > MAX_STRLEN) {
					fprintf(stderr,"ERROR: filename too long.\n");
					return(1);
				}
				if (sscanf(optarg,"%s",fn_wsudognu) != 1) {
					fprintf(stderr,"ERROR: cannot read file name from argument list\n");
					return(1);
				}
				break;
			case 'h':
				print_help(stdout);
				exit(0);
			case 'n':
				if ((sscanf(optarg,"%d",&num_sudokus) != 1) || (num_sudokus < 1)) {
					return(1);
				}
				break;
			case 'i':
				fn = optarg;
				if ((inFile = fopen(fn,"r")) == NULL) {
					fprintf(stderr,"error opening input file %s",fn);
					return(1);
				}
				break;
			case 'o':
				fn_out = optarg;
				break;
			case 'r':
/*				if ((sscanf(optarg,"%d",&max_rating) != 1) || (max_rating < 1)) {
					return(1);
				}*/
				fprintf(stderr,"WARNING: ignoring option -r\n");
				break;
			case 'v':
				if ((sscanf(optarg,"%d",&verbosity) != 1) || (verbosity < 0)) {
					return(1);
				}
				break;
			case 'V':
				print_version(stdout);
				exit(0);
				break;
			case 'P':
				if(strlen(optarg) == (NUM_ET-2)) {
//					for (iet = 0; iet < NUM_ET; iet++) printf("%d ",dfcs[iet]);
					for (iet = 1; iet < NUM_ET-1; iet++) dfcs[iet] = -1;
					for (iet = 1; iet < NUM_ET-1; iet++) {
						switch(optarg[iet-1]) {
							case '.':	dfcs[iet] = et_hs; break;
							case '-':	dfcs[iet] = et_s;  break;
							case 'd':	dfcs[iet] = et_hd; break;
							case 'D':	dfcs[iet] = et_d;  break;
							case 'b':	dfcs[iet] = et_b;  break;
							case 'B':	dfcs[iet] = et_l;  break;
							case 't':	dfcs[iet] = et_ht; break;
							case 'T':	dfcs[iet] = et_t;  break;
							case 'q':	dfcs[iet] = et_hq; break;
							case 'Q':	dfcs[iet] = et_q;  break;
							case 'X':	dfcs[iet] = et_xw; break;
							case 'S':	dfcs[iet] = et_sf; break;
							case 'J':	dfcs[iet] = et_jf; break;
							case 'F':	dfcs[iet] = et_fc; break;
							default: return(1);
						}
					}
//					for (iet = 0; iet < NUM_ET; iet++) printf("%d ",dfcs[iet]);
					for (iet = 1; iet < NUM_ET-2; iet++)
						if (dfcs[iet] == -1)
							return(1);
				} else {
					return(1);
				}
				break;
			case ':':
				if (isprint(optopt)) fprintf (stderr,"%s: Missing parameter for option `-%c'.\n",progname,optopt);
				else                 fprintf (stderr,"%s: Missing parameter for option character `\\x%x'.\n",progname,optopt);
					return(1);
			case '?':
				if (isprint(optopt)) fprintf (stderr,"%s: Unknown option `-%c'.\n",progname,optopt);
				else                 fprintf (stderr,"%s: Unknown option character `\\x%x'.\n",progname,optopt);
					return(1);
			default:
				abort ();
		}
	}
	for (i = optind; i < argc; i++)	printf ("Non-option argument %s\n", argv[i]);

	if (action == 0) {
		return(1);
	}
	
	return(0);
}

/*
 ******************************************************************************/
void print_version(FILE *f) {
	fprintf(f,"%s version %s\n",progname,VERSION);
	fprintf(f,"\
Copyright (c) 2007 Jens Baaran (http://baaran.de)\n\n\
This is free software. You may redistribute copies of this software\n\
under the terms of the GNU General Public License Version 2, June 1991\n\
(http://www.gnu.org/licenses/gpl.html). There is NO WARRANTY, to the\n\
extent permitted by law.\n\
");
}

/*
 ******************************************************************************/
void print_help(FILE *f) {

	print_usage(f);
	
	fprintf(f,"\
Sudoku format:\n\t\n\
The input and output format for sudokus uses one string per sudoku:\n\
concatenate the 81 numbers line by line from top left to bottom\n\
right without white space between the numbers. Empty fields are to be\n\
denoted by \"0\" (zero).\n\n\
");

	fprintf(f,"\
Sudoku solving:\n\t\n\
Various elimination techniques for solving sudokus are implemented, starting\n\
with easy techniques such as singles and hidden singles. Complex techniques\n\
for candidate elimination (tuple / hidden tuple, line-block interaction,\n\
forcing chain, X-wing, swordfish, jellyfish) are applied only if necessary,\n\
reverting to backtracking at last resort. Solutions are checked for uniqueness.\n\
\n\
%s can display the solution process. When you set verbosity to 1, %s\n\
will display a string of characters, each one stands for one solution step:\n\
 .\t hidden single\n\
 -\t single\n\
 d\t hidden double\n\
 D\t double\n\
 b\t line-block interaction\n\
 B\t block-line interaction\n\
 t\t hidden triple\n\
 T\t triple\n\
 q\t quadruple\n\
 Q\t hidden quadruple\n\
 X\t X-wing\n\
 S\t swordfish\n\
 J\t jellyfish\n\
 F\t forcing chain to a common value with length of forcing chain\n\
 f\t forcing chain to a contradiction for one of the branches\n\
 e\t forcing chain to the end of the sudoku\n\
 g\t guess during backtracking process\n\
 w\t last guess turned out to be wrong.\n\
\n\
The default order for applying the solution steps is to start with hidden\n\
singles and singles before trying to eliminate candidates with hidden tuples\n\
and tuples from doubles to quadruples. Line-block interactions are applied\n\
between doubles and triples. Then the more complex techniques X-wing,\n\
swordfish and jellyfish are used and if all these fail, forcing chains are\n\
tried, before reverting to guessing and starting a backtracking process.\n\
\n\
You can influence the order, in which the techniques above are applied, by\n\
means of the -P option. As described above, the default order is\n\
\".-dDbBtTqQXSJF\". If you want the steps to be applied in a different order, just\n\
shuffle these characters and pass them to %s with the -P flag. Be sure to\n\
specify all of the 14 characters exactly once, otherwise %s will abort.\n\
For performance reasons not all techniques are applied during backtracking and\n\
while searching for forcing chains, only the single and hidden single\n\
techniques are applied. Forcing chains are only tried for cells with two\n\
candidates. All of such cells are checked and %s will apply the shortest\n\
forcing chain it finds (which is, by the way, not in all cases the shortest\n\
forcing chain there is).\n\n\
",progname,progname,progname,progname,progname);

	fprintf(f,"\
Sudoku rating:\n\t\n\
At first the sudoku is assigned a basic rating of 1, 10 or 100, depending on\n\
the most difficult technique required for its solution:\n\n\
 a basic rating of 1 is assigned for the techniques .-d\n\n\
 a basic rating of 10 is assigned for the techniques DbBtTqQ\n\n\
 a basic rating of 100 is assigned for the techniques XSJFfeg.\n\n\
Next, each solution step is assigned a difficulty rating, starting with 1 for\n\
hidden singles up to %d for guesses during the backtracking process. In each\n\
step the three easiest applicable techniques are taken into account. The\n\
individual ratings for each solution step are added to the basic rating.\n\
finally, 1 is subtracted from the sudoku rating for each cell to be solved,\n\
so the smallest possible rating of 1 is assigned to all sudokus completely\n\
solvable with the hidden singles technique.\n\n\
",NUM_ET-1);
	
	print_contact(f);
}

/*
 ******************************************************************************/
void print_usage(FILE *f) {
	fprintf(f,"\
Usage: %s <command> [options]\n\
%s can solve and create sudokus. It can print 4 sudokus to a pdf or an svg\n\
page and it comes with example cgi scripts for integration into www sites.\n\n\
",progname,progname);

	fprintf(f,"\
Examples:\n\
 %s -s -v 1 < sud.in > sol.out  \tsolve sudokus\n\
 %s -c -n 20 > sud.out     \tcreate 20 sudokus\n\
 %s -p -f sud.pdf < sud.in \tprint 4 sudokus to sud.pdf & sud.pdf.svg\n\n\
",progname,progname,progname);

	fprintf(f,"\
Commands (only one of these may be present):\n\
 -c\t create sudokus and print four of them to pdf/svg file\n\
 -p\t read sudokus from stdin and print four of them to pdf/svg file\n\
 -s\t read sudokus from stdin and solve them\n\
 -w\t read one sudoku and candidates from stdin and generate a html table\n\
 -h\t display usage information\n\
 -V\t display version and copyright information\n\n\
");
	
	fprintf(f,"\
Options (not all options work for every command):\n\
 -n num_sudokus\t number of sudokus to create\n\
 -i file\t read sudokus from file and not from stdin\n\
 -o file\t html or pdf file name for formatted output\n\
 -P sol_techn\t apply solution techniques according to sol_techn\n\
 \t The sol_techn string must consist of the fourteen characters\n\
 \t \".-dDbBtTqQXSJF\" in arbitrary order. \".-dDbltTqQXSJF\" is the default,\n\
 \t when -P is not present. For further explanation of the sol_techn\n\
 \t string use the -h command (or see \"Sudoku solving\" below, if you are\n\
 \t looking at the documentation).\n\
 -v verbosity\t for explaining solution techniques (set it to 0, 1, 10 or 11)\n\
 -f wsudognu\t location of wsudognu.cgi\n\
 \t wsudognu.cgi is a script necessary for processing the sudoku urls\n\
 \t created for the -s command with a verbosity of 11 (default value for\n\
 \t wsudognu is ./wsudognu.cgi)\n\
 -t\t use one thread for each available CPU (no pdf/svg file will be created)\n\
 -S\t create symmetrical sudokus (slow)\n\n\
");
}

/*
 ******************************************************************************/
void print_contact(FILE *f) {
	fprintf(f,"Report bugs to <%s_at_baaran_dot_de>\n\n",progname);
	fprintf(f,"Written by Jens Baaran\n\n");
}
