/*
 * sisctrl - SiS Display Control Panel
 * for the SiS X.org/XFree86 driver
 *
 * GUI part (gtk)
 *
 * (C) 2003-2005 Thomas Winischhofer <thomas@winischhofer.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "sisc.h"

#include "sisctrl_gui.h"

#ifdef USE_STRAY
static void gui_stray_build_menu(void);
static void gui_stray_crt1_set_type_menu_index(int index);
static void gui_stray_crt2_set_type_menu_index(int index, int index2);
static void gui_stray_video_set_sensitive(int enable);
static void gui_stray_video_set_active(int num);
extern gboolean stray_init(char *straytooltipt);
extern gboolean stray_uninit(void);
extern void     stray_position_menu(GtkMenu *menu, int *x, int *y, gboolean *push_in, gpointer user_data);
extern gboolean stray_is_embedded(void);
#endif

sisctrl_gui		gui;

extern gui_look		gl;

/**************************************************
 *                Clean up and exit               *
 **************************************************/

void
gui_clean_ups(void)
{
   int i;

   /* Put GUI related clean-ups here */

   for(i = 0; i < DM_GI_NUM; i++) {
      if(gui.dm_icon[i]) g_object_unref(gui.dm_icon[i]);
      gui.dm_icon[i] = NULL;
      if(gui.dm_icon90[i]) g_object_unref(gui.dm_icon90[i]);
      gui.dm_icon90[i] = NULL;
      if(gui.dm_icon180[i]) g_object_unref(gui.dm_icon180[i]);
      gui.dm_icon180[i] = NULL;
      if(gui.dm_icon270[i]) g_object_unref(gui.dm_icon270[i]);
      gui.dm_icon270[i] = NULL;
   }

   if(gui.dm_icon_clone) g_object_unref(gui.dm_icon_clone);
   gui.dm_icon_clone = NULL;

   if(gui.dm_mfb_context) g_object_unref(gui.dm_mfb_context);
   gui.dm_mfb_context = NULL;

   if(gui.pixbuf_vga) g_object_unref(gui.pixbuf_vga);
   gui.pixbuf_vga = NULL;
   if(gui.pixbuf_lcd) g_object_unref(gui.pixbuf_lcd);
   gui.pixbuf_lcd = NULL;
   if(gui.pixbuf_lcda) g_object_unref(gui.pixbuf_lcda);
   gui.pixbuf_lcda = NULL;
   if(gui.pixbuf_tvpal) g_object_unref(gui.pixbuf_tvpal);
   gui.pixbuf_tvpal = NULL;
   if(gui.pixbuf_tvpalm)g_object_unref(gui.pixbuf_tvpalm);
   gui.pixbuf_tvpalm = NULL;
   if(gui.pixbuf_tvpaln) g_object_unref(gui.pixbuf_tvpaln);
   gui.pixbuf_tvpaln = NULL;
   if(gui.pixbuf_tvntsc) g_object_unref(gui.pixbuf_tvntsc);
   gui.pixbuf_tvntsc = NULL;
   if(gui.pixbuf_tv525) g_object_unref(gui.pixbuf_tv525);
   gui.pixbuf_tv525 = NULL;
   if(gui.pixbuf_tv525i) g_object_unref(gui.pixbuf_tv525i);
   gui.pixbuf_tv525i = NULL;
   if(gui.pixbuf_tv525p) g_object_unref(gui.pixbuf_tv525p);
   gui.pixbuf_tv525p = NULL;
   if(gui.pixbuf_tv625) g_object_unref(gui.pixbuf_tv625);
   gui.pixbuf_tv625 = NULL;
   if(gui.pixbuf_tv625i) g_object_unref(gui.pixbuf_tv625i);
   gui.pixbuf_tv625i = NULL;
   if(gui.pixbuf_tv625p) g_object_unref(gui.pixbuf_tv625p);
   gui.pixbuf_tv625p = NULL;
   if(gui.pixbuf_tv750p) g_object_unref(gui.pixbuf_tv750p);
   gui.pixbuf_tv750p = NULL;
   if(gui.pixbuf_tv1080i) g_object_unref(gui.pixbuf_tv1080i);
   gui.pixbuf_tv1080i = NULL;
   if(gui.pixbuf_vga2) g_object_unref(gui.pixbuf_vga2);
   gui.pixbuf_vga2 = NULL;

   if(gui.ga_context) g_object_unref(gui.ga_context);
   gui.ga_context = NULL;
   if(gui.ga_stipple) g_object_unref(gui.ga_stipple);
   gui.ga_stipple = NULL;
   if(gui.ga_stipple2) g_object_unref(gui.ga_stipple2);
   gui.ga_stipple2 = NULL;

   if(gui.tv_context) g_object_unref(gui.tv_context);
   gui.tv_context = NULL;

#if 0
   if(gui.vi_context) g_object_unref(gui.vi_context);
   gui.vi_context = NULL;
#endif

   if(gui.ab_context) g_object_unref(gui.ab_context);
   gui.ab_context = NULL;
   if(gui.ab_iconul) g_object_unref(gui.ab_iconul);
   gui.ab_iconul = NULL;
   if(gui.ab_icon) g_object_unref(gui.ab_icon);
   gui.ab_icon = NULL;
}

void
gui_main_quit(void)
{
#ifdef USE_STRAY
   /* Remove sytem tray icon */
   stray_uninit();
   /* Destroy system tray menu */
   if(gui.st_menu) gtk_widget_destroy(gui.st_menu);
#endif

   if(gui.window) gtk_main_quit();
}

/**************************************************
 *               Show error pop-ups               *
 **************************************************/

void
gui_showerror(char *message)
{
    GtkWidget *dialog, *label;

    dialog = gtk_dialog_new_with_buttons("Error", GTK_WINDOW(gui.window),
			GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
			GTK_STOCK_OK,
			GTK_RESPONSE_NONE,
			NULL);

    label = gtk_label_new((gchar *)message);

    g_signal_connect_swapped(GTK_OBJECT(dialog), "response",
			G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);

    gtk_widget_show_all(dialog);
}

Bool
gui_showerror_pregui(char *message, Bool yesno, Bool warning)
{
   GtkWidget *dialog;
   gint result;

   dialog = gtk_message_dialog_new(NULL,
              		   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			   warning ? GTK_MESSAGE_WARNING : GTK_MESSAGE_ERROR,
			   yesno ? GTK_BUTTONS_YES_NO : GTK_BUTTONS_OK,
	      		   (gchar *)message);

   result = gtk_dialog_run(GTK_DIALOG(dialog));

   gtk_widget_destroy(dialog);

   switch(result) {
   case GTK_RESPONSE_YES:
      return TRUE;
   default:
      return FALSE;
   }
}

Bool
gui_showquestion(char *message)
{
   GtkWidget *dialog;
   gint result;

   dialog = gtk_message_dialog_new((gui.window && GTK_WIDGET_VISIBLE(gui.window)) ?
						GTK_WINDOW(gui.window) : NULL,
			   GTK_DIALOG_MODAL,
			   GTK_MESSAGE_QUESTION,
			   GTK_BUTTONS_YES_NO,
			   (gchar *)message);

   result = gtk_dialog_run(GTK_DIALOG(dialog));

   gtk_widget_destroy(dialog);

   switch(result) {
   case GTK_RESPONSE_YES:
      return TRUE;
   default:
      return FALSE;
   }
}

static GtkWidget *
xpm_label_box(char **xpmdata, gchar *labeltext, GtkWidget *toactivate, int space,
				gchar *tooltip, gchar *tooltip2)
{
   GtkWidget *box, *label = NULL, *image, *align, *event;
   GdkPixmap *icon = NULL;
   GdkBitmap *mask;

   box = gtk_hbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(box), 1);

   align = gtk_alignment_new(0.5, 0.5, 0, 0);

#if (GTK_MAJOR_VERSION > 2) || ((GTK_MAJOR_VERSION == 2) && (GTK_MINOR_VERSION >= 4))
   if((tooltip != NULL) && (strlen(tooltip) > 0)) {
      event = gtk_event_box_new();
      gtk_container_set_border_width(GTK_CONTAINER(event), 0);
      gtk_container_add(GTK_CONTAINER(event), box);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), event, tooltip, tooltip2);
      gtk_event_box_set_visible_window(GTK_EVENT_BOX(event), FALSE);
      gtk_widget_show(event);
      gtk_container_add(GTK_CONTAINER(align), event);
   } else {
#endif
      gtk_container_add(GTK_CONTAINER(align), box);
#if (GTK_MAJOR_VERSION > 2) || ((GTK_MAJOR_VERSION == 2) && (GTK_MINOR_VERSION >= 4))
   }
#endif

   icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, xpmdata);
   image = gtk_image_new_from_pixmap(icon, mask);

   g_object_unref(G_OBJECT(icon));
   g_object_unref(G_OBJECT(mask));

   gtk_box_pack_start(GTK_BOX(box), image, FALSE, FALSE, space);

   if(labeltext) {
      label = gtk_label_new_with_mnemonic(labeltext);
      gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
      gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, space);
      gtk_widget_show(label);
   }

   gtk_widget_show(box);
   gtk_widget_show(image);

   if((label) && (toactivate)) {
      gtk_label_set_mnemonic_widget(GTK_LABEL(label), toactivate);
   }

   return align;
}

/**************************************************
 *             Xinerama screen chooser            *
 **************************************************/

Bool
gui_xinerama_chooser(int maxscreens, int *newscreen)
{
   char *format = "Screen %d [%s %s%s]";
   char *crtnoformat = ", CRT%d";
   int i, crtnum;
   char *chip;
   char buffer[64];
   char buffer2[16];
   Bool myret, IsSiS;
   gint ret;
   GtkWidget *dialog, *frame, *vbox, *label;
   GtkWidget *combobox;

   frame = gtk_frame_new("Xinerama");
   gtk_container_set_border_width(GTK_CONTAINER(frame), 10);

   vbox = gtk_vbox_new(FALSE, 8);
   gtk_container_add(GTK_CONTAINER(frame), vbox);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

   label = gtk_label_new("Xinerama detected.\n\nPlease choose the screen to configure:");
   gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

   combobox = gtk_combo_box_new_text();

   for(i = 0; i < maxscreens; i++) {
      if(sisctrl_xinerama_query_screen(i, &chip, &crtnum, &IsSiS)) {
         buffer2[0] = 0;
	 if(crtnum != -1) sprintf(buffer2, crtnoformat, crtnum);
         sprintf(buffer, format, i, IsSiS ? "SiS" : "XGI", chip, buffer2);
         gtk_combo_box_append_text(GTK_COMBO_BOX(combobox), buffer);
      }
   }

   gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);

   dialog = gtk_dialog_new_with_buttons("SiSCtrl",
				NULL,
				GTK_DIALOG_MODAL,
				GTK_STOCK_OK,
				GTK_RESPONSE_OK,
				GTK_STOCK_CANCEL,
				GTK_RESPONSE_CANCEL,
				NULL);

   gtk_container_add(GTK_CONTAINER(vbox), label);
   gtk_container_add(GTK_CONTAINER(vbox), combobox);

   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), frame);

   gtk_widget_show_all(dialog);

   ret = gtk_dialog_run(GTK_DIALOG(dialog));

   switch(ret) {
   case GTK_RESPONSE_NONE:
   case GTK_RESPONSE_CANCEL:
   case GTK_RESPONSE_DELETE_EVENT:
      myret = FALSE;
      break;
   default:
      *newscreen = gtk_combo_box_get_active(GTK_COMBO_BOX(combobox));
      myret = TRUE;
   }

   gtk_widget_destroy(dialog);

   return myret;
}

/**************************************************
 *               Confirmation pop-up              *
 **************************************************/

static gboolean
gui_pbar_timeout_callback(void)
{
    char buffer[8];

    gui.pbar_prog--;

    if(gui.pbar_prog < 0) {

       if(gui.confdialog) {
	  gtk_dialog_response(GTK_DIALOG(gui.confdialog), GTK_RESPONSE_NONE);
       }

       gui.pbar_timeout = 0;
       return FALSE;

    } else {

       if(gui.pbar) {
          gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(gui.pbar), (gdouble)(1.0 / 7.0) * gui.pbar_prog);
          sprintf(buffer, "%d", gui.pbar_prog);
          gtk_progress_bar_set_text(GTK_PROGRESS_BAR(gui.pbar), buffer);
       }
       return TRUE;

    }
}

Bool
gui_confirmation_popup(void)
{
   Bool myret;
   gint ret;
   GtkWidget *frame, *vbox, *label, *bar, *omenu, *menuitem;

   frame = gtk_frame_new("Confirmation");
   gtk_container_set_border_width(GTK_CONTAINER(frame), 10);

   vbox = gtk_vbox_new(FALSE, 8);
   gtk_container_add(GTK_CONTAINER(frame), vbox);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

   label = gtk_label_new("Please confirm the change within the time-out");
   gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

   gui.pbar = gtk_progress_bar_new();
   gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(gui.pbar), 1.0);
   gtk_progress_bar_set_text(GTK_PROGRESS_BAR(gui.pbar), "7");
   gui.pbar_prog = 7;

   gui.confdialog = gtk_dialog_new_with_buttons("SiSCtrl",
				(gui.window && GTK_WIDGET_VISIBLE(gui.window)) ?
						GTK_WINDOW(gui.window) : NULL,
				GTK_DIALOG_MODAL,
				GTK_STOCK_OK,
				GTK_RESPONSE_OK,
				GTK_STOCK_CANCEL,
				GTK_RESPONSE_CANCEL,
				NULL);

   gtk_container_add(GTK_CONTAINER(vbox), label);
   gtk_container_add(GTK_CONTAINER(vbox), gui.pbar);

   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(gui.confdialog)->vbox), frame);

   gtk_widget_show_all(gui.confdialog);

   gui.pbar_timeout = g_timeout_add(1000, (GSourceFunc)gui_pbar_timeout_callback, NULL);

   ret = gtk_dialog_run(GTK_DIALOG(gui.confdialog));

   if(gui.pbar_timeout) {
      g_source_remove(gui.pbar_timeout);
      gui.pbar_timeout = 0;
   }

   gtk_widget_destroy(gui.confdialog);
   gui.confdialog = NULL;
   gui.pbar = NULL;

   switch(ret) {
   case GTK_RESPONSE_NONE:
   case GTK_RESPONSE_CANCEL:
   case GTK_RESPONSE_DELETE_EVENT:
      myret = FALSE;
      break;
   default:
      myret = TRUE;
   }

   return myret;
}

/**************************************************
 *                Displaymode page                *
 **************************************************/

int
gui_dm_get_current_selected_mode(void)
{
   int i;

   if(!gui.dm_select) return -1;

   if(!gtk_tree_selection_get_selected(gui.dm_select, NULL, NULL))
      return -1;

   for(i = 0; i < SISCTRL_MAXMODES; i++) {
      if(gtk_tree_selection_iter_is_selected(gui.dm_select, &gui.dm_iter[i])) {
	 return i;
      }
   }

   return -1;
}

#ifdef HAVE_RANDR
int
gui_dm_get_current_randr_index(void)
{
   int index = -1;
   if(gl.do_randr && gui.dm_comboboxrr) {
      index = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.dm_comboboxrr));
   }

   return index;
}

void
gui_dm_set_current_randr_index(int index)
{
   if(gl.do_randr && gui.dm_comboboxrr) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrr), index);
   }
}

int
gui_dm_get_current_randr_rot_index(void)
{
   int index = -1;

   if(gl.do_randr && gui.dm_comboboxrot) {
      index = gui.dm_rotindices[gtk_combo_box_get_active(GTK_COMBO_BOX(gui.dm_comboboxrot))];
   }

   return index;
}

void
gui_dm_set_current_randr_rot_index(int index)
{
   if(gl.do_randr && gui.dm_comboboxrot) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrot), gui.dm_rrotindices[index]);
   }
}
#endif

void
gui_dm_create_gfx(void)
{
   if(gl.do_mergedfb_gfx) {
      if(!gui.pixbuf_vga) {
         gui.pixbuf_vga = gdk_pixbuf_new_from_xpm_data((const char **)icon_vga1);
      }
   }

   if(!gui.pixbuf_lcd) {
      if(gl.islaptop)
         gui.pixbuf_lcd = gdk_pixbuf_new_from_xpm_data((const char **)icon_lcdnew);
      else if(gl.supportdvi)
         gui.pixbuf_lcd = gdk_pixbuf_new_from_xpm_data((const char **)icon_dvi);
      else
         gui.pixbuf_lcd = gdk_pixbuf_new_from_xpm_data((const char **)icon_lcd);
   }

   if(!gui.pixbuf_lcda) {
      if(gl.islaptop)
         gui.pixbuf_lcda = gdk_pixbuf_new_from_xpm_data((const char **)icon_lcdanew);
      else if(gl.supportdvi)
         gui.pixbuf_lcda = gdk_pixbuf_new_from_xpm_data((const char **)icon_dvi1);
      else
         gui.pixbuf_lcda = gdk_pixbuf_new_from_xpm_data((const char **)icon_lcda);
   }

   if(!gui.pixbuf_tvpal)
      gui.pixbuf_tvpal = gdk_pixbuf_new_from_xpm_data((const char **)icon_tvpal);

   if(!gui.pixbuf_tvpalm)
      gui.pixbuf_tvpalm = gdk_pixbuf_new_from_xpm_data((const char **)icon_tvpalm);

   if(!gui.pixbuf_tvpaln)
      gui.pixbuf_tvpaln = gdk_pixbuf_new_from_xpm_data((const char **)icon_tvpaln);

   if(!gui.pixbuf_tvntsc)
      gui.pixbuf_tvntsc = gdk_pixbuf_new_from_xpm_data((const char **)icon_tvntsc);

   if(gl.do_mergedfb_gfx) {
      if(!gui.pixbuf_tv525) {
         gui.pixbuf_tv525 = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv525);
      }
   }

   if(!gui.pixbuf_tv525i)
      gui.pixbuf_tv525i = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv525i);

   if(!gui.pixbuf_tv525p)
      gui.pixbuf_tv525p = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv525p);

   if(gl.do_mergedfb_gfx) {
      if(!gui.pixbuf_tv625) {
         gui.pixbuf_tv625 = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv625);
      }
   }

   if(!gui.pixbuf_tv625i)
      gui.pixbuf_tv625i = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv625i);

   if(!gui.pixbuf_tv625p)
      gui.pixbuf_tv625p = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv625p);

   if(!gui.pixbuf_tv750p)
      gui.pixbuf_tv750p = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv750p);

   if(!gui.pixbuf_tv1080i) {
      if(gl.supportypbpr) {
         gui.pixbuf_tv1080i = gdk_pixbuf_new_from_xpm_data((const char **)icon_tv1080i);
      } else {
         gui.pixbuf_tv1080i = gdk_pixbuf_new_from_xpm_data((const char **)icon_tvhivi);
      }
   }

   if(!gui.pixbuf_vga2)
      gui.pixbuf_vga2 = gdk_pixbuf_new_from_xpm_data((const char **)icon_vga2);
}

void
gui_dm_set_mode_list_attributes(int listindex, unsigned short field)
{

   if(gl.do_mergedfb_gfx)
      return;

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 1,
		(field & SiS_CRT2_CRT1LCDA) ? gui.pixbuf_lcda : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 2,
		(field & SiS_CRT2_LCD) ? gui.pixbuf_lcd : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 3,
		(field & SiS_CRT2_TVPAL) ? gui.pixbuf_tvpal : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 4,
		(field & SiS_CRT2_TVNTSC) ? gui.pixbuf_tvntsc : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 5,
		(field & SiS_CRT2_TVPALM) ? gui.pixbuf_tvpalm : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 6,
		(field & SiS_CRT2_TVPALN) ? gui.pixbuf_tvpaln : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 7,
   		(field & SiS_CRT2_YPBPR525I) ? gui.pixbuf_tv525i : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 8,
   		(field & SiS_CRT2_YPBPR525P) ? gui.pixbuf_tv525p : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 9,
   		(field & SiS_CRT2_YPBPR625I) ? gui.pixbuf_tv625i : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 10,
   		(field & SiS_CRT2_YPBPR625P) ? gui.pixbuf_tv625p : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 11,
   		(field & SiS_CRT2_YPBPR750P) ? gui.pixbuf_tv750p : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 12,
   		(field & (SiS_CRT2_YPBPR1080I | SiS_CRT2_HIVISION)) ? gui.pixbuf_tv1080i : NULL, -1);

   gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[listindex], 13,
   		(field & SiS_CRT2_VGA2) ? gui.pixbuf_vga2 : NULL, -1);
}

void
gui_dm_set_current_entry(int index)
{
   gtk_tree_selection_unselect_all(gui.dm_select);
   gtk_tree_selection_select_iter(gui.dm_select, &gui.dm_iter[index]);
}

#ifdef HAVE_RANDR
void
gui_dm_update_leaveunchanged(char *newtext, char *newrottext)
{
   gint  current_menu;

   if(!gui.dm_comboboxrr) return;

   current_menu = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.dm_comboboxrr));
   if(current_menu == 0) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrr), 1);
   }
   gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.dm_comboboxrr), 0);
   gtk_combo_box_prepend_text(GTK_COMBO_BOX(gui.dm_comboboxrr), (gchar *)newtext);
   if(current_menu == 0) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrr), 0);
   }

   if(!gui.dm_comboboxrot) return;

   current_menu = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.dm_comboboxrot));
   if(current_menu == 0) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrot), 1);
   }
   gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.dm_comboboxrot), 0);
   gtk_combo_box_prepend_text(GTK_COMBO_BOX(gui.dm_comboboxrot), (gchar *)newrottext);
   if(current_menu == 0) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrot), 0);
   }
}
#endif

static void
mfbdrawlabel(GtkWidget *label, PangoLayout *playout, int x, int y, Bool inverse)
{
   const GdkColor white = { 0, 65535, 65535, 65535 };
   const GdkColor black = { 0, 0, 0, 0 };
   GdkGCValues gcvals;
   int lh, lw;

   if(x <= 0) x = 1;
   if(y <= 0) y = 1;

   gdk_gc_get_values(label->style->fg_gc[GTK_WIDGET_STATE(label)],
		&gcvals);

   gdk_gc_set_rgb_fg_color(label->style->fg_gc[GTK_WIDGET_STATE(label)],
		&black);

   if(inverse) {
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x - 1, y, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x - 1, y - 1, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x, y - 1, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x + 1, y - 1, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x + 1, y, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x + 1, y + 1, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x, y + 1, playout);
      gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x - 1, y + 1, playout);

      gdk_gc_set_rgb_fg_color(label->style->fg_gc[GTK_WIDGET_STATE(label)],
		&white);
   }

   gdk_draw_layout(gui.dm_mfbdraw->window,
		label->style->fg_gc[GTK_WIDGET_STATE(label)],
		x, y, playout);

   gdk_gc_set_foreground(label->style->fg_gc[GTK_WIDGET_STATE(label)],
   		&gcvals.foreground);
}

static void
gui_draw_dev_icon(GdkPixbuf *pixbuf, int x, int y)
{
   gdk_draw_pixbuf(gui.dm_mfbdraw->window,
		 gui.dm_mfb_context,
		 pixbuf,
		 0, 0,
		 x, y,
		 gdk_pixbuf_get_width(pixbuf), gdk_pixbuf_get_height(pixbuf),
		 GDK_RGB_DITHER_NORMAL, 0, 0);
}

static void
gui_dm_fresh_mfb_gfx(void)
{
   int lwidth, lheight, lhsize, lvsize, rhsize, rvsize;
   int leftdev, rightdev, topdev, bottomdev, index, rotindex = 0, rot, x, y, lw, lh;
   int gfxareawidth, gfxareaheight, totalw, totalh;
   int lastystep1, lastystep2, lastystep3;
   int xbase1, ybase1, xwidth1, yheight1;
   int xbase2, ybase2, xwidth2, yheight2;
   unsigned short field;
   PangoLayout *playout;
   GdkPixbuf **pixbuf;
   char ***pixbufsrc;
   Bool cloned = FALSE;
   int xoffs = 2, yoffs = 2;

   if( (!gui.dm_mfbdraw) ||
       (!GTK_WIDGET_REALIZED(gui.dm_mfbdraw)) ||
       (!(gui.dm_mfbdraw->window)) ||
       (!gl.do_mergedfb_details) ) {
      return;
   }

   lwidth = gui.dm_mfbdraw->allocation.width;
   lheight = gui.dm_mfbdraw->allocation.height;
   if(lwidth <= 0 || !lheight) return;

   if(lwidth < gui.dm_mfbsizex) xoffs = 1;
   if(lheight < gui.dm_mfbsizey) yoffs = 1;

   /*fprintf(stderr, "real size %d %d\n", lwidth, lheight);*/

   if(!gui.dm_mfb_context) {
      gui.dm_mfb_context = gdk_gc_new(gui.dm_mfbdraw->window);
   }

   gdk_gc_set_fill(gui.dm_mfb_context, GDK_SOLID);
   gdk_draw_rectangle(gui.dm_mfbdraw->window,
		      gui.dm_mfbdraw->style->bg_gc[GTK_STATE_NORMAL],
		      TRUE,
		      0, 0,
		      lwidth, lheight);

   if((index = gui_dm_get_current_selected_mode()) == -1) {
      return;
   }

#ifdef HAVE_RANDR
   if(gl.supportedrots & 0x0e) {
      rotindex = gui_dm_get_current_randr_rot_index();
      if(rotindex == -1) rotindex = 0;
   }
#endif

   if(!sisctrl_get_mfb_devices(index, rotindex,
		&leftdev, &rightdev, &topdev, &bottomdev, &rot, &cloned)) {
      return;
   }

   switch(rot) {
#ifdef HAVE_RANDR
   case 1:
      pixbuf = gui.dm_icon90;
      pixbufsrc = gui.dm_icon90source;
      break;
   case 2:
      pixbuf = gui.dm_icon180;
      pixbufsrc = gui.dm_icon180source;
      break;
   case 3:
      pixbuf = gui.dm_icon270;
      pixbufsrc = gui.dm_icon270source;
      break;
#endif
   default:
      pixbuf = gui.dm_icon;
      pixbufsrc = NULL;
   }

   if(leftdev >= 0 || rightdev >= 0) {

      gfxareawidth = lwidth;
      gfxareaheight = lheight / 2;

      if(leftdev >= 0) {
	 if(pixbuf[leftdev] == NULL) {
	    if(pixbufsrc[leftdev]) {
	       pixbuf[leftdev] = gdk_pixbuf_new_from_xpm_data((const char **)pixbufsrc[leftdev]);
	    } else {
	       leftdev = -1;
	    }
	 }
	 if(leftdev >= 0) {
            lhsize = gdk_pixbuf_get_width(pixbuf[leftdev]);
            lvsize = gdk_pixbuf_get_height(pixbuf[leftdev]);
            x = ((gfxareawidth / 2) - lhsize) / 2;
            if(x < 0) x = 0;
            y = ((gfxareaheight - lvsize) / 2);
            if(y < 0) y = 0;
            gui_draw_dev_icon(pixbuf[leftdev], x, y);
         }
      }

      if(rightdev >= 0) {
         if(pixbuf[rightdev] == NULL) {
	    if(pixbufsrc[rightdev]) {
	       pixbuf[rightdev] = gdk_pixbuf_new_from_xpm_data((const char **)pixbufsrc[rightdev]);
	    } else {
	       rightdev = -1;
	    }
	 }
	 if(rightdev >= 0) {
            rhsize = gdk_pixbuf_get_width(pixbuf[rightdev]);
            rvsize = gdk_pixbuf_get_height(pixbuf[rightdev]);
            x = ((gfxareawidth / 2) - rhsize) / 2;
            if(x < 0) x = 0;
            x += (gfxareawidth / 2);
            y = ((gfxareaheight - rvsize) / 2);
            if(y < 0) y = 0;
            gui_draw_dev_icon(pixbuf[rightdev], x, y);
         }
      }

      if(cloned) {
         x = (gfxareawidth / 2) - (gdk_pixbuf_get_width(gui.dm_icon_clone) / 2);
         y = (gfxareaheight / 2) - (gdk_pixbuf_get_height(gui.dm_icon_clone) / 2);
         gui_draw_dev_icon(gui.dm_icon_clone, x, y);
      }

      playout = gtk_label_get_layout(GTK_LABEL(gui.dm_mmdl_label));
      pango_layout_get_pixel_size(playout, &lw, &lh);

      x = ((gfxareawidth / 2) - lw) / 2;
      if(x < 0) x = 0;
      y = (gfxareaheight - lh) / 2;
#if 0
      if((leftdev >= 0) && ((lvsize - 16) > lh)) {
         y = ((gfxareaheight - lvsize) / 2) + 8;
      }
#endif
      if(y < 0) y = 0;

      mfbdrawlabel(gui.dm_mmdl_label, playout, x, y, (leftdev >= 0));

      playout = gtk_label_get_layout(GTK_LABEL(gui.dm_mmdr_label));
      pango_layout_get_pixel_size(playout, &lw, &lh);

      x = ((gfxareawidth / 2) - lw) / 2;
      if(x < 0) x = 0;
      x += (gfxareawidth / 2);
      y = ((gfxareaheight - lh) / 2);
#if 0
      if((rightdev >= 0) && ((rvsize - 16) > lh)) {
         y = ((gfxareaheight - rvsize) / 2) + 8;
      }
#endif
      if(y < 0) y = 0;

      mfbdrawlabel(gui.dm_mmdr_label, playout, x, y, (rightdev >= 0));

      xbase1 = (gui.leftcrtnum == 1) ? 0 : lwidth / 2;
      xwidth1 = lwidth / 2;

      ybase1 = lheight / 2;
      yheight1 = lheight / 2;

      xbase2 = (gui.leftcrtnum == 2) ? 0 : lwidth / 2;
      xwidth2 = lwidth / 2;

      ybase2 = ybase1;
      yheight2 = yheight1;

   } else if(topdev >= 0 || bottomdev >= 0) {

      gfxareawidth = lwidth / 2;
      gfxareaheight = lheight;

      if(topdev >= 0) {
         if(pixbuf[topdev] == NULL) {
	    if(pixbufsrc[topdev]) {
	       pixbuf[topdev] = gdk_pixbuf_new_from_xpm_data((const char **)pixbufsrc[topdev]);
	    } else {
	       topdev = -1;
	    }
	 }
	 if(topdev >= 0) {
            lhsize = gdk_pixbuf_get_width(pixbuf[topdev]);
            lvsize = gdk_pixbuf_get_height(pixbuf[topdev]);
            x = (gfxareawidth - lhsize) / 2;
            if(x < 0) x = 0;
            y = ((gfxareaheight / 2) - lvsize) / 2;
            if(y < 0) y = 0;
            gui_draw_dev_icon(pixbuf[topdev], x, y);
         }
      }
      if(bottomdev >= 0) {
         if(pixbuf[bottomdev] == NULL) {
	    if(pixbufsrc[bottomdev]) {
	       pixbuf[bottomdev] = gdk_pixbuf_new_from_xpm_data((const char **)pixbufsrc[bottomdev]);
	    } else {
	       bottomdev = -1;
	    }
	 }
	 if(bottomdev >= 0) {
            rhsize = gdk_pixbuf_get_width(pixbuf[bottomdev]);
            rvsize = gdk_pixbuf_get_height(pixbuf[bottomdev]);
            x = (gfxareawidth - rhsize) / 2;
            if(x < 0) x = 0;
            y = ((gfxareaheight / 2) - rvsize) / 2;
            if(y < 0) y = 0;
            y += (gfxareaheight / 2);
            gui_draw_dev_icon(pixbuf[bottomdev], x, y);
         }
      }

      if(cloned) {
         x = (gfxareawidth / 2) - (gdk_pixbuf_get_width(gui.dm_icon_clone) / 2);
         y = (gfxareaheight / 2) - (gdk_pixbuf_get_height(gui.dm_icon_clone) / 2);
         gui_draw_dev_icon(gui.dm_icon_clone, x, y);
      }

      playout = gtk_label_get_layout(GTK_LABEL(gui.dm_mmdl_label));
      pango_layout_get_pixel_size(playout, &lw, &lh);

      x = ((gfxareawidth - lw) / 2);
      if(x < 0) x = 0;

      y = ((gfxareaheight / 2) - lh) / 2;
      if((topdev >= 0) && ((lvsize - 16) < lh)) {
         y = (((gfxareaheight / 2) - lvsize) / 2) + 8;
      }
      if(y < 0) y = 0;

      mfbdrawlabel(gui.dm_mmdl_label, playout, x, y, (topdev >= 0));

      playout = gtk_label_get_layout(GTK_LABEL(gui.dm_mmdr_label));
      pango_layout_get_pixel_size(playout, &lw, &lh);

      x = ((gfxareawidth - lw) / 2);
      if(x < 0) x = 0;

      y = ((gfxareaheight / 2) - lh) / 2;
      if((bottomdev >= 0) && (rvsize < lh)) {
         y = (((gfxareaheight / 2) - rvsize) / 2) + 8;
      }
      if(y < 0) y = 0;
      y += (gfxareaheight / 2);

      mfbdrawlabel(gui.dm_mmdr_label, playout, x, y, (bottomdev >= 0));

      xbase1 = lwidth / 2;
      xwidth1 = lwidth / 2;

      ybase1 = (gui.leftcrtnum == 1) ? 0 : lheight / 2;
      yheight1 = lheight / 2;

      xbase2 = xbase1;
      xwidth2 = xwidth1;

      ybase2 = (gui.leftcrtnum == 2) ? 0 : lheight / 2;
      yheight2 = yheight1;
   }

   field = sisctrl_dm_get_modefield(index);

   /* CRT1 icons */

   totalh = gdk_pixbuf_get_height(gui.pixbuf_vga) + yoffs;
   if(field & SiS_CRT2_CRT1LCDA) {
     totalh += gdk_pixbuf_get_height(gui.pixbuf_lcda);
   }

   x = (xbase1 + (xwidth1 / 2)) - (gdk_pixbuf_get_width(gui.pixbuf_vga) / 2);
   y = (ybase1 + (yheight1 / 2)) - (totalh / 2);

   gui_draw_dev_icon(gui.pixbuf_vga, x, y);

   if(field & SiS_CRT2_CRT1LCDA) {
      x = (xbase1 + (xwidth1 / 2)) - (gdk_pixbuf_get_width(gui.pixbuf_lcda) / 2);
      y = (ybase1 + (yheight1 / 2)) - (totalh / 2) + gdk_pixbuf_get_height(gui.pixbuf_vga) + yoffs;
      gui_draw_dev_icon(gui.pixbuf_lcda, x, y);
   }

   /* CRT2 icons */

   totalh = 0;
   if(field & SiS_CRT2_LCD) {
      totalh += gdk_pixbuf_get_height(gui.pixbuf_lcd) + yoffs;
   }
   if(field & (SiS_CRT2_TVPAL | SiS_CRT2_TVNTSC | SiS_CRT2_TVPALM | SiS_CRT2_TVPALN)) {
      totalh += gdk_pixbuf_get_height(gui.pixbuf_tvpal) + yoffs;
   }
   if(field & (SiS_CRT2_YPBPR525I | SiS_CRT2_YPBPR525P |
	       SiS_CRT2_YPBPR625I | SiS_CRT2_YPBPR625P |
	       SiS_CRT2_YPBPR750P | SiS_CRT2_YPBPR1080I | SiS_CRT2_HIVISION)) {
      totalh += gdk_pixbuf_get_height(gui.pixbuf_tv525) + yoffs;
   }
   if(field & SiS_CRT2_VGA2) {
      totalh += gdk_pixbuf_get_height(gui.pixbuf_vga2) + yoffs;
   }

   lastystep1 = lastystep2 = lastystep3 = 0;

   if(field & SiS_CRT2_LCD) {
      x = (xbase2 + (xwidth2 / 2)) - (gdk_pixbuf_get_width(gui.pixbuf_lcd) / 2);
      y = (ybase2 + (yheight2 / 2)) - (totalh / 2);
      gui_draw_dev_icon(gui.pixbuf_lcd, x, y);
      lastystep1 = gdk_pixbuf_get_height(gui.pixbuf_lcd) + 2;
   }

   if(field & (SiS_CRT2_TVPAL | SiS_CRT2_TVNTSC | SiS_CRT2_TVPALM | SiS_CRT2_TVPALN)) {
      int xstep = gdk_pixbuf_get_width(gui.pixbuf_tvpal);
      int xmult = 0;
      if(field & SiS_CRT2_TVPAL) xmult++;
      if(field & SiS_CRT2_TVNTSC) xmult++;
      if(field & SiS_CRT2_TVPALM) xmult++;
      if(field & SiS_CRT2_TVPALN) xmult++;
      x = (xbase2 + (xwidth2 / 2)) - (((xstep * xmult) + (xoffs * (xmult - 1))) / 2);
      y = (ybase2 + (yheight2 / 2)) - (totalh / 2) + lastystep1;
      if(field & SiS_CRT2_TVPAL) {
         gui_draw_dev_icon(gui.pixbuf_tvpal, x, y);
         x += xstep + xoffs;
      }
      if(field & SiS_CRT2_TVNTSC) {
         gui_draw_dev_icon(gui.pixbuf_tvntsc, x, y);
         x += xstep + xoffs;
      }
      if(field & SiS_CRT2_TVPALM) {
         gui_draw_dev_icon(gui.pixbuf_tvpalm, x, y);
         x += xstep + xoffs;
      }
      if(field & SiS_CRT2_TVPALN) {
         gui_draw_dev_icon(gui.pixbuf_tvpaln, x, y);
      }
      lastystep2 = gdk_pixbuf_get_height(gui.pixbuf_tvpal) + yoffs;
   }

   if(field & (SiS_CRT2_YPBPR525I | SiS_CRT2_YPBPR525P |
	       SiS_CRT2_YPBPR625I | SiS_CRT2_YPBPR625P |
	       SiS_CRT2_YPBPR750P | SiS_CRT2_YPBPR1080I | SiS_CRT2_HIVISION)) {
      int xstep = gdk_pixbuf_get_width(gui.pixbuf_tv525);
      int xmult = 0;
      if(field & (SiS_CRT2_YPBPR525I | SiS_CRT2_YPBPR525P)) xmult++;
      if(field & (SiS_CRT2_YPBPR625I | SiS_CRT2_YPBPR625P)) xmult++;
      if(field & SiS_CRT2_YPBPR750P) xmult++;
      if(field & (SiS_CRT2_YPBPR1080I | SiS_CRT2_HIVISION)) xmult++;
      x = (xbase2 + (xwidth2 / 2)) - (((xstep * xmult) + (xoffs * (xmult - 1))) / 2);
      y = (ybase2 + (yheight2 / 2)) - (totalh / 2) + lastystep1 + lastystep2;
      if((field & (SiS_CRT2_YPBPR525I | SiS_CRT2_YPBPR525P)) == (SiS_CRT2_YPBPR525I | SiS_CRT2_YPBPR525P)) {
         gui_draw_dev_icon(gui.pixbuf_tv525, x, y);
         x += xstep + xoffs;
      } else if(field & SiS_CRT2_YPBPR525I) {
         gui_draw_dev_icon(gui.pixbuf_tv525i, x, y);
         x += xstep + xoffs;
      } else if(field & SiS_CRT2_YPBPR525P) {
         gui_draw_dev_icon(gui.pixbuf_tv525p, x, y);
         x += xstep + xoffs;
      }
      if((field & (SiS_CRT2_YPBPR625I | SiS_CRT2_YPBPR625P)) == (SiS_CRT2_YPBPR625I | SiS_CRT2_YPBPR625P)) {
         gui_draw_dev_icon(gui.pixbuf_tv625, x, y);
         x += xstep + xoffs;
      } else if(field & SiS_CRT2_YPBPR625I) {
         gui_draw_dev_icon(gui.pixbuf_tv625i, x, y);
         x += xstep + xoffs;
      } else if(field & SiS_CRT2_YPBPR625P) {
         gui_draw_dev_icon(gui.pixbuf_tv625p, x, y);
         x += xstep + xoffs;
      }
      if(field & SiS_CRT2_YPBPR750P) {
         gui_draw_dev_icon(gui.pixbuf_tv750p, x, y);
         x += xstep + xoffs;
      }
      if(field & (SiS_CRT2_YPBPR1080I | SiS_CRT2_HIVISION)) {
         gui_draw_dev_icon(gui.pixbuf_tv1080i, x, y);
      }
      lastystep3 = gdk_pixbuf_get_height(gui.pixbuf_tv525) + yoffs;
   }

   if(field & SiS_CRT2_VGA2) {
      x = (xbase2 + (xwidth2 / 2)) - (gdk_pixbuf_get_width(gui.pixbuf_vga2) / 2);
      y = (ybase2 + (yheight2 / 2)) - (totalh / 2) + lastystep1 + lastystep2 + lastystep3;
      gui_draw_dev_icon(gui.pixbuf_vga2, x, y);
   }

   /* Hack: NOW that labels are realized, hide them */
   gtk_widget_hide(gui.dm_mmdl_label);
   gtk_widget_hide(gui.dm_mmdr_label);
}

static void
dm_update_selection(void)
{
   char leftstring[256];
   char rightstring[256];
   char leftdev[16], rightdev[16], leftcrt[16], rightcrt[16], leftcrtnum[4], rightcrtnum[4];
   char leftres[16], rightres[16], lefthz[16], righthz[16], lefthz2[16], righthz2[16];
   int i, index = -1, rotindex;
   Bool cloned;
   GtkWidget *crt1label = gui.dm_mmdl_label;
   GtkWidget *crt2label = gui.dm_mmdr_label;
   PangoLayout *playout;
   const char *longtext = "<span font_desc=\"8\">%s</span>\n<span font_desc=\"8\" weight=\"bold\">%s\n%s</span>";
   const char *shorttext = "<span font_desc=\"8\">%s</span>\n<span font_desc=\"8\" weight=\"bold\">%s</span>\n<span font_desc=\"7\">[VGA:%s]</span>";
   const char *normaltext = "%s(%s) <span weight=\"bold\">%s</span> %s%s";

   if(!gl.do_mergedfb_details) return;

   index = gui_dm_get_current_selected_mode();

   if(index == -1) return;

   rotindex = -1;
#ifdef HAVE_RANDR
   if(gl.do_mergedfb_gfx) {
      if(gl.supportedrots & 0x0e) {
         rotindex = gui_dm_get_current_randr_rot_index();
         if(rotindex == -1) rotindex = 0;
      }
   }
#endif

   sisctrl_dm_selection_changed(index, rotindex, leftdev, leftcrt, leftcrtnum, leftres, lefthz,
				rightdev, rightcrt, rightcrtnum, rightres, righthz,
				&gui.leftcrtnum, &gui.rightcrtnum, lefthz2, righthz2,
				&cloned);

   if(!gl.do_mergedfb_gfx) {
      sprintf(leftstring, normaltext,
		leftdev, leftcrtnum, leftres, (*lefthz == 0) ? lefthz2 : lefthz, cloned ? " (C)" : "");
      sprintf(rightstring, normaltext,
		rightdev, rightcrtnum, rightres, (*righthz == 0) ? righthz2 : righthz, cloned ? " (C)" : "");
   } else {
      if(*lefthz != 0) {
         sprintf(leftstring, longtext,
		leftcrt, leftres, lefthz);
      } else {
         sprintf(leftstring, shorttext,
		leftcrt, leftres, lefthz2);
      }
      if(*righthz != 0) {
         sprintf(rightstring, longtext,
		rightcrt, rightres, righthz);
      } else {
         sprintf(rightstring, shorttext,
		rightcrt, rightres, righthz2);
      }
   }

   gtk_label_set_justify(GTK_LABEL(gui.dm_mmdl_label), GTK_JUSTIFY_CENTER);
   gtk_label_set_justify(GTK_LABEL(gui.dm_mmdr_label), GTK_JUSTIFY_CENTER);

   gtk_label_set_markup(GTK_LABEL(gui.dm_mmdl_label), (const gchar *)leftstring);
   gtk_label_set_markup(GTK_LABEL(gui.dm_mmdr_label), (const gchar *)rightstring);

   gui_dm_fresh_mfb_gfx();
}

static void
gui_rebuild_modelist(void)
{
    int i;

    for(i = 0; i < SISCTRL_MAXMODES; i++) {
       if(gl.modetextptr[i]) {
          gtk_list_store_append(gui.dm_liststore, &gui.dm_iter[i]);
          gtk_list_store_set(gui.dm_liststore, &gui.dm_iter[i], 0, gl.modetextptr[i], -1);
       }
    }
}

#ifdef HAVE_RANDR
static void
gui_rebuild_sizes_menu(void)
{
    int i;

    for(i = SISCTRL_MAXMODES - 1; i >= 0; i--) {
       if(gui.dm_haverandrsize[i]) {
          gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.dm_comboboxrr), (i+2));
          gui.dm_haverandrsize[i] = FALSE;
       }
    }
    for(i = 0; i < gl.numrandrsizes; i++) {
       gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrr), gl.sizetextptr[i]);
       gui.dm_haverandrsize[i] = TRUE;
    }
}
#endif

void
gui_dm_remove_mode_list_and_randr_sizes(void)
{
    gtk_list_store_clear(gui.dm_liststore);
    /* FIXME: Does this imply freeing all memory? */

#ifdef HAVE_RANDR
    if(gl.do_randr) {
       /* Set menu index 0, leave rest alone for now */
       /* (this list will be changed in "rebuild" above) */
       gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrr), gl.mergedfb ? 1 : 0);
    }
#endif
}

void
gui_dm_rebuild_mode_list_and_randr_sizes(void)
{
    gui_rebuild_modelist();

    sisctrl_set_crt2_types_for_modelist();

#ifdef HAVE_RANDR
    if(gl.do_randr) {
       gui_rebuild_sizes_menu();
    }
#endif
}

/* Signal handler for displaymode list "selection changed" */

static void
dm_selection_changed(GtkTreeSelection *selection, gpointer data)
{
   dm_update_selection();
}

static gboolean
dm_mfddraw_exposed(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   gui_dm_fresh_mfb_gfx();
   return FALSE;
}

#ifdef HAVE_RANDR
static void
dm_rot_combobox_changed(GtkWidget *widget, gpointer data)
{
   dm_update_selection();
   gui_dm_fresh_mfb_gfx();
}
#endif

/* Build page */

static void
build_display_mode_page(void)
{
    int i, rows = 6, randrrow = 5, cols = 5;
    GValue wer = { 0, };
    GdkBitmap *mask;
    GtkWidget *eventbox;
    static const char *colnames[] = {
       "LCDA",
       "LCD",
       "TVPAL",
       "TVNTSC",
       "TVPALM",
       "TVPALN",
       "TVY525I",
       "TVY525P",
       "TVY625I",
       "TVY625P",
       "TVY750P",
       "TVY1080I",
       "VGA2"
    };

    gui.dm_frame = gtk_frame_new((gl.do_randr) ? "Display mode and desktop size" : "Display mode");
    gtk_container_set_border_width(GTK_CONTAINER(gui.dm_frame), 8);
    gtk_widget_show(gui.dm_frame);

    if(gl.do_randr || gl.do_mergedfb_details || gl.do_mergedfb_gfx) {
       rows = 5;
       randrrow = 5;
       if(gl.do_randr) {
          rows++;
       }
       if(gl.do_mergedfb_details && !gl.do_mergedfb_gfx) {
          rows++;
	  randrrow++;
       } else if(gl.do_mergedfb_gfx) {
          cols = 6;
       }
       gui.dm_table = gtk_table_new(rows, cols, TRUE);
       gtk_table_set_row_spacings(GTK_TABLE(gui.dm_table), gl.do_mergedfb_gfx ? 3 : 0);
       gtk_table_set_col_spacings(GTK_TABLE(gui.dm_table), gl.do_mergedfb_details ? 0 : 4);
       gtk_container_set_border_width(GTK_CONTAINER(gui.dm_table), 8);
       gtk_container_add(GTK_CONTAINER(gui.dm_frame), gui.dm_table);
       gtk_widget_show(gui.dm_table);
    }

#ifdef HAVE_RANDR
    if(gl.do_randr) {

       int rrtemp = 6;

       if(gl.supportedrots & 0x0e) rrtemp = 3;

       gui.dm_rr_table = gtk_table_new(1, 6, FALSE);
       gtk_table_set_row_spacings(GTK_TABLE(gui.dm_rr_table), 0);
       gtk_table_set_col_spacings(GTK_TABLE(gui.dm_rr_table), 4);
       gtk_container_set_border_width(GTK_CONTAINER(gui.dm_rr_table), 0);
       gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_rr_table, 0, cols, randrrow, randrrow+1);
       gtk_widget_show(gui.dm_rr_table);

       gui.dm_size_label = gtk_label_new("Desktop size");
       gtk_table_attach_defaults(GTK_TABLE(gui.dm_rr_table), gui.dm_size_label, 0, 1, 0, 1);
       gtk_widget_show(gui.dm_size_label);

       gui.dm_comboboxrr = gtk_combo_box_new_text();

       gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrr), gl.leaveunchanged_text);
       gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrr), "As display mode");

       for(i = 0; i < SISCTRL_MAXMODES; i++) {
          gui.dm_haverandrsize[i] = FALSE;
       }

       gui_rebuild_sizes_menu();

       /* Set menu to resize by default in MergedFB mode; this is better
        * for pseudo-xinerama than simple mode changes.
        */
       gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrr), gl.mergedfb ? 1 : 0);

       eventbox = gtk_event_box_new();
       gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
       gtk_widget_show(eventbox);
       gtk_container_add(GTK_CONTAINER(eventbox), gui.dm_comboboxrr);

       gtk_table_attach(GTK_TABLE(gui.dm_rr_table), eventbox, 1, rrtemp, 0, 1, GTK_EXPAND|GTK_FILL, 0, 0, 0);

       gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
       		"Select desktop size",
		"Select to resize your desktop to a certain size. This requires the RandR extension "
		"and a RandR-capable window manager, otherwise you end up with inaccessible menus "
		"or windows.");

       if(gl.supportedrots & 0x0e) {

          gui.dm_rot_label = gtk_label_new("Rotate");
          gtk_table_attach_defaults(GTK_TABLE(gui.dm_rr_table), gui.dm_rot_label, rrtemp, rrtemp+1, 0, 1);
          gtk_widget_show(gui.dm_rot_label);

	  gui.dm_comboboxrot = gtk_combo_box_new_text();

	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrot), gl.keeprot_text);

	  i = 0;
	  gui.dm_rrotindices[0] = i;
	  gui.dm_rotindices[i++] = 0;
	  if(gl.supportedrots & 0x01) {
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrot), "0\302\260");
	     gui.dm_rrotindices[1] = i;
	     gui.dm_rotindices[i++] = 1;
	  }
	  if(gl.supportedrots & 0x02) {
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrot), "90\302\260");
	     gui.dm_rrotindices[2] = i;
	     gui.dm_rotindices[i++] = 2;
	  }
	  if(gl.supportedrots & 0x04) {
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrot), "180\302\260");
	     gui.dm_rrotindices[3] = i;
	     gui.dm_rotindices[i++] = 3;
	  }
	  if(gl.supportedrots & 0x08) {
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.dm_comboboxrot), "270\302\260");
	     gui.dm_rrotindices[4] = i;
	     gui.dm_rotindices[i++] = 4;
	  }

	  gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrot), 0);

	  eventbox = gtk_event_box_new();
	  gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
	  gtk_widget_show(eventbox);
	  gtk_container_add(GTK_CONTAINER(eventbox), gui.dm_comboboxrot);

          gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
		"Select display rotation",
		"");

	  gtk_table_attach(GTK_TABLE(gui.dm_rr_table), eventbox, rrtemp+1, 6, 0, 1, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	  if(gl.do_mergedfb_gfx) {
	     g_signal_connect(G_OBJECT(gui.dm_comboboxrot), "changed", G_CALLBACK(dm_rot_combobox_changed), NULL);
	  }
       }

       sisctrl_dm_update_leaveunchanged();

       gtk_widget_show(gui.dm_comboboxrr);
       if(gl.supportedrots & 0x0e) {
          gtk_widget_show(gui.dm_comboboxrot);
       }
    }
#endif

    if(gl.do_mergedfb_gfx) {
       gui.dm_liststore = gtk_list_store_new(1, G_TYPE_STRING);
    } else {

       gui.dm_liststore = gtk_list_store_new(14, G_TYPE_STRING,   GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF,
					         GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF,
					         GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF,
					         GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF,
					         GDK_TYPE_PIXBUF, GDK_TYPE_PIXBUF);
    }

    gui_rebuild_modelist();

    sisctrl_set_crt2_types_for_modelist();

    gui.dm_list = gtk_tree_view_new_with_model(GTK_TREE_MODEL(gui.dm_liststore));

    g_value_init(&wer, G_TYPE_FLOAT);
    g_value_set_float(&wer, 0.0);

    gui.dm_renderer = gtk_cell_renderer_text_new();
    g_object_set_property(G_OBJECT(gui.dm_renderer), "xalign", &wer);
    gui.dm_column = gtk_tree_view_column_new_with_attributes("Display mode", gui.dm_renderer, "text", 0, NULL);
    gtk_tree_view_column_set_spacing(gui.dm_column, 0);
    gtk_tree_view_append_column(GTK_TREE_VIEW(gui.dm_list), gui.dm_column);

    if(!gl.do_mergedfb_gfx) {
       for(i = 0; i < 13; i++) {
          gui.dm_renderer = gtk_cell_renderer_pixbuf_new();
          g_object_set_property(G_OBJECT(gui.dm_renderer), "xalign", &wer);
          gui.dm_column = gtk_tree_view_column_new_with_attributes(colnames[i], gui.dm_renderer, "pixbuf", i+1, NULL);
          gtk_tree_view_column_set_spacing(gui.dm_column, 0);
          gtk_tree_view_append_column(GTK_TREE_VIEW(gui.dm_list), gui.dm_column);
       }
    }

    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(gui.dm_list), FALSE);

    gui.dm_select = gtk_tree_view_get_selection(GTK_TREE_VIEW(gui.dm_list));
    gtk_tree_selection_set_mode(gui.dm_select, GTK_SELECTION_SINGLE);

    if(gl.do_mergedfb_details) {
       gui.dm_mmdl_label = gtk_label_new("n/a");
       gtk_widget_show(gui.dm_mmdl_label);
       gui.dm_mmdr_label = gtk_label_new("n/a");
       gtk_widget_show(gui.dm_mmdr_label);

       if(!gl.do_mergedfb_gfx) {
          gui.dm_mmdt_label = gtk_label_new("Mode details:");
          gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_mmdt_label, 0, 1, 5, 6);
          gtk_widget_show(gui.dm_mmdt_label);
          gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_mmdl_label, 1, 3, 5, 6);
          gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_mmdr_label, 3, 5, 5, 6);
       }

       g_signal_connect(G_OBJECT(gui.dm_select), "changed", G_CALLBACK(dm_selection_changed), NULL);
    }

    gui_dm_set_current_entry(gl.startmodeindex);

    gui.dm_list_window = gtk_scrolled_window_new(NULL, NULL);
    if(gl.do_randr || gl.do_mergedfb_details || gl.do_mergedfb_gfx) {
       gtk_container_set_border_width(GTK_CONTAINER(gui.dm_list_window), 0);
    } else {
       gtk_container_set_border_width(GTK_CONTAINER(gui.dm_list_window), 10);
    }
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(gui.dm_list_window), GTK_POLICY_AUTOMATIC,
									   GTK_POLICY_AUTOMATIC);

    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(gui.dm_list_window), GTK_SHADOW_IN);
    gtk_container_add(GTK_CONTAINER(gui.dm_list_window), gui.dm_list);

    if(gl.do_mergedfb_gfx) {

       int temp, temp2, sizex, sizey, mult, maxsize;
       int crt1icon_w, crt1icon_h, crt2icon_w, crt2icon_h;

       gui.dm_icon[DM_GI_CRT] = gdk_pixbuf_new_from_xpm_data((const char **)crt2vga_xpm);
       if(gl.islaptop) {
	  gui.dm_icon[DM_GI_LCD] = gdk_pixbuf_new_from_xpm_data((const char **)crt2lcdl_xpm);
       } else {
	  gui.dm_icon[DM_GI_LCD] = gdk_pixbuf_new_from_xpm_data((const char **)crt2lcd_xpm);
       }
       gui.dm_icon[DM_GI_TV] = gdk_pixbuf_new_from_xpm_data((const char **)crt2tv_xpm);

       gui.dm_icon_clone = gdk_pixbuf_new_from_xpm_data((const char **)icon_clone);

       for(i = 0; i < DM_GI_NUM; i++) {
          gui.dm_icon90[i] = NULL;
          gui.dm_icon90source[i] = NULL;
          gui.dm_icon180[i] = NULL;
          gui.dm_icon180source[i] = NULL;
          gui.dm_icon270[i] = NULL;
          gui.dm_icon270source[i] = NULL;
       }

#ifdef HAVE_RANDR
       if(gl.supportedrots & 0x02) {
          gui.dm_icon90source[DM_GI_CRT] = crt2vga270_xpm;
          if(gl.islaptop) {
             gui.dm_icon90source[DM_GI_LCD] = crt2lcdl270_xpm;
          } else {
             gui.dm_icon90source[DM_GI_LCD] = crt2lcd270_xpm;
          }
          gui.dm_icon90source[DM_GI_TV] = crt2tv270_xpm;
       }
       if(gl.supportedrots & 0x04) {
          gui.dm_icon180source[DM_GI_CRT] = crt2vga180_xpm;
          if(gl.islaptop) {
             gui.dm_icon180source[DM_GI_LCD] = crt2lcdl180_xpm;
          } else {
             gui.dm_icon180source[DM_GI_LCD] = crt2lcd180_xpm;
          }
          gui.dm_icon180source[DM_GI_TV] = crt2tv180_xpm;
       }
       if(gl.supportedrots & 0x08) {
          gui.dm_icon270source[DM_GI_CRT] = crt2vga90_xpm;
          if(gl.islaptop) {
             gui.dm_icon270source[DM_GI_LCD] = crt2lcdl90_xpm;
          } else {
             gui.dm_icon270source[DM_GI_LCD] = crt2lcd90_xpm;
          }
          gui.dm_icon270source[DM_GI_TV] = crt2tv90_xpm;
       }
#endif

       /* Calculate required drawable size */
       maxsize = gdk_pixbuf_get_width(gui.dm_icon[DM_GI_CRT]);
       temp = gdk_pixbuf_get_height(gui.dm_icon[DM_GI_CRT]);
       if(temp > maxsize) maxsize = temp;
       temp = gdk_pixbuf_get_width(gui.dm_icon[DM_GI_LCD]);
       if(temp > maxsize) maxsize = temp;
       temp = gdk_pixbuf_get_height(gui.dm_icon[DM_GI_LCD]);
       if(temp > maxsize) maxsize = temp;
       if(gl.tv_page_type == TV_PAGE_TYPE_NORMAL) {
          temp = gdk_pixbuf_get_width(gui.dm_icon[DM_GI_TV]);
          if(temp > maxsize) maxsize = temp;
          temp = gdk_pixbuf_get_height(gui.dm_icon[DM_GI_TV]);
          if(temp > maxsize) maxsize = temp;
       }

       crt1icon_w = gdk_pixbuf_get_width(gui.pixbuf_vga) + 2;
       crt1icon_h = gdk_pixbuf_get_height(gui.pixbuf_vga) + 2;
       if(gl.supportlcda) {
          temp = gdk_pixbuf_get_width(gui.pixbuf_lcda) + 2;
          if(temp > crt1icon_w) crt1icon_w = temp;
          crt1icon_h += gdk_pixbuf_get_height(gui.pixbuf_lcda) + 2;
       }

       crt2icon_w = gdk_pixbuf_get_width(gui.pixbuf_lcd);
       crt2icon_h = gdk_pixbuf_get_height(gui.pixbuf_lcd) + 2;
       if(gl.tv_page_type == TV_PAGE_TYPE_NORMAL) {
          mult = gl.supportpalmn ? 4 : 2;
          temp = (gdk_pixbuf_get_width(gui.pixbuf_tvpal) * mult) + (2 * (mult - 1));
          if(temp > crt2icon_w) crt2icon_w = temp;
          crt2icon_h += gdk_pixbuf_get_height(gui.pixbuf_tvpal) + 2;
          if(gl.supporthivision || gl.supportypbpr) {
             mult = gl.supporthivision ? 1 : ((gl.supportypbpr625i || gl.supportypbpr625p) ? 4 : 3);
             temp = (gdk_pixbuf_get_width(gui.pixbuf_tv525) * mult) + (2 * (mult - 1));
             if(temp > crt2icon_w) crt2icon_w = temp;
             crt2icon_h += gdk_pixbuf_get_height(gui.pixbuf_tv525) + 2;
          }
       }
       if(gl.supportvga2) {
          temp = gdk_pixbuf_get_width(gui.pixbuf_vga2);
          if(temp > crt2icon_w) crt2icon_w = temp;
          crt2icon_h += gdk_pixbuf_get_height(gui.pixbuf_tv525) + 2;
       }

       sizex = (maxsize + 2) * 2;
       temp = crt1icon_w + 2 + crt2icon_w + 2;
       if(temp > sizex) sizex = temp;
       temp = (crt1icon_w + 2) * 2;
       if(temp > sizex) sizex = temp;
       temp = (crt2icon_w + 2) * 2;
       if(temp > sizex) sizex = temp;
       temp = maxsize + 2 + crt1icon_w + 2;
       if(temp > sizex) sizex = temp;
       temp = maxsize + 2 + crt2icon_w + 2;
       if(temp > sizex) sizex = temp;

       sizey = (maxsize + 2) * 2;
       temp = crt1icon_h + 2 + crt2icon_h + 2;
       if(temp > sizey) sizey = temp;
       temp = (crt1icon_h + 2) * 2;
       if(temp > sizey) sizey = temp;
       temp = (crt2icon_h + 2) * 2;
       if(temp > sizey) sizey = temp;
       temp = maxsize + 2 + crt1icon_h + 2;
       if(temp > sizey) sizey = temp;
       temp = maxsize + 2 + crt2icon_h + 2;
       if(temp > sizey) sizey = temp;

       gui.dm_mfbdraw = gtk_drawing_area_new();

       /*gtk_widget_set_size_request(gui.dm_mfbdraw, sizex - (213 - 184), sizey);*/

       /* GTK does not honor our size request. It allocates a bigger area
        * despite our (smaller) values. So reduce them in order to not cause
        * a "grow" of the window
        */
       gtk_widget_set_size_request(gui.dm_mfbdraw, sizex * 93 / 100, sizey);

       gui.dm_mfbsizex = sizex;
       gui.dm_mfbsizey = sizey;

       /*fprintf(stderr, "size request %d %d\n", sizex, sizey);*/

       gui.dm_mfbtable = gtk_table_new(2, 2, FALSE);

       gtk_table_set_row_spacings(GTK_TABLE(gui.dm_mfbtable), 0);
       gtk_table_set_col_spacings(GTK_TABLE(gui.dm_mfbtable), 0);
       gtk_container_set_border_width(GTK_CONTAINER(gui.dm_mfbtable), 0);

       gtk_table_attach_defaults(GTK_TABLE(gui.dm_mfbtable), gui.dm_mfbdraw, 0, 2, 0, 2);

       gtk_table_attach_defaults(GTK_TABLE(gui.dm_mfbtable), gui.dm_mmdl_label, 0, 2, 0, 2);
       gtk_table_attach_defaults(GTK_TABLE(gui.dm_mfbtable), gui.dm_mmdr_label, 0, 2, 0, 2);

       gtk_widget_show(gui.dm_mfbtable);
       gtk_widget_show(gui.dm_mfbdraw);

       gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_mfbtable, 3, cols, 0, 5);

       g_signal_connect(G_OBJECT(gui.dm_mfbdraw), "expose-event", G_CALLBACK(dm_mfddraw_exposed), NULL);

    }

    if(gl.do_randr || gl.do_mergedfb_details) {
       if(gl.do_mergedfb_gfx) {
          gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_list_window, 0, 3, 0, 5);
       } else {
          gtk_table_attach_defaults(GTK_TABLE(gui.dm_table), gui.dm_list_window, 0, 5, 0, 5);
       }
    } else {
       gtk_container_add(GTK_CONTAINER(gui.dm_frame), gui.dm_list_window);
    }

    gtk_widget_show(gui.dm_list);
    gtk_widget_show(gui.dm_list_window);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.dm_list,
       		"Select display mode (display resolution)", "");

    gui.dm_label = xpm_label_box(icon_mode, "Mode", NULL, 1,
        (gl.do_randr) ? "Select display mode and desktop size" :
			 "Select display mode", "");

    gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.dm_frame, gui.dm_label);
}

/**************************************************
 *                   CRT1 page                    *
 **************************************************/

/* CRT1 type menu and graphics */

void
gui_crt1_set_type_menu_index(int index)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.c1_combobox), gui.c1_rmenuindices[index]);

#ifdef USE_STRAY
   gui_stray_crt1_set_type_menu_index(index);
#endif
}

int
gui_crt1_get_type_menu_index(void)
{
   int index = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.c1_combobox));
   if(index < 0) return index;
   return gui.c1_menuindices[index];
}

void
gui_crt1_show_variant(int variant)
{
   int i, backup;

   if(!gui.c1_label_onoff)
      return;

   /* C1_MI_XX = menuindices[real menuindex] */
   /* rmenuindices[C1_MI_XX] = real menuindex */

   backup = gui_crt1_get_type_menu_index();

   for(i = 0; i < C1_MI_NUM; i++) {
      gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.c1_combobox), 0);
   }

   i = 0;

   if(variant) {
      gtk_label_set_text(GTK_LABEL(gui.c1_label_onoff), "Device type");
      if(gl.supportvga2) {
         gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_combobox), "VGA (analog output; primary plug)");
      } else {
         gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_combobox), "VGA (analog output)");
      }
   } else {
      gtk_label_set_text(GTK_LABEL(gui.c1_label_onoff), "Switch output");
      gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_combobox), "ON");
   }

   gui.c1_menuindices[i] = C1_MI_VGA;
   gui.c1_rmenuindices[C1_MI_VGA] = i++;
   if(variant) {
      gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_combobox), gl.supportdvi ?
				"DVI-D/LCD (digital output)" : "LCD");
      gui.c1_menuindices[i] = C1_MI_LCDA;
      gui.c1_rmenuindices[C1_MI_LCDA] = i++;
   }
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_combobox), "OFF");
   gui.c1_menuindices[i] = C1_MI_OFF;
   gui.c1_rmenuindices[C1_MI_OFF] = i;

   if(backup < 0) backup = 0;
   gui_crt1_set_type_menu_index(backup);
}

static void
gui_crt1_select_gfx(int gfxnum, int nomode)
{
   gtk_widget_hide(gui.c1_imageon);
   gtk_widget_hide(gui.c1_imagelcda);

   gtk_widget_hide(gui.c1_imagenomode);

   switch(gfxnum) {
   case C1_GFX_OFF:
      break;
   case C1_GFX_VGA:
      gtk_widget_show(gui.c1_imageon);
      break;
   case C1_GFX_LCDA:
      gtk_widget_show(gui.c1_imagelcda);
   }

   if(nomode) gtk_widget_show(gui.c1_imagenomode);
}

void
gui_crt1_update_gfx(void)
{
   int c1_index = gui_crt1_get_type_menu_index();
   int c1_gfx, nomode = 0;
   int mindex;

   switch(c1_index) {
   case C1_MI_OFF:
       c1_gfx = C1_GFX_OFF;
       break;
   case C1_MI_VGA:
       c1_gfx = C1_GFX_VGA;
       break;
   case C1_MI_LCDA:
       if(gl.supportlcda) {
          c1_gfx = C1_GFX_LCDA;
	  mindex = sisctrl_dm_get_current_mode_index();
	  if(!(sisctrl_dm_get_modefield(mindex) & SiS_CRT2_CRT1LCDA)) nomode = 1;
       } else {
          c1_gfx = C1_GFX_VGA;
       }
       break;
   default:
       c1_gfx = C1_GFX_OFF;
   }

   gui_crt1_select_gfx(c1_gfx, nomode);
}

/* LCD scaling menu */

void
gui_crt1_set_scaling_menu_index(int index)
{
   if(gui.c1_comboboxls) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.c1_comboboxls), (gint)index);
   }
}

int
gui_crt1_get_scaling_menu_index(int retdefault)
{
   if(!gui.c1_comboboxls) return retdefault;
   return(int)(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.c1_comboboxls)));
}

void
gui_crt1_show_scaling_menu_and_label(void)
{
   gtk_widget_show(gui.c1_eventboxls);
   gtk_widget_show(gui.c1_label_lcds);
}

void
gui_crt1_hide_scaling_menu_and_label(void)
{
   gtk_widget_hide(gui.c1_eventboxls);
   gtk_widget_hide(gui.c1_label_lcds);
}

/* LCD centering menu */

void
gui_crt1_set_centering_menu_index(int index)
{
   if(gui.c1_comboboxcc) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.c1_comboboxcc), (gint)index);
   }
}

int
gui_crt1_get_centering_menu_index(int retdefault)
{
   if(!gui.c1_comboboxcc) return retdefault;
   return(int)(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.c1_comboboxcc)));
}

void
gui_crt1_show_center_menu(void)
{
   gtk_widget_show(gui.c1_eventboxcc);
}

void
gui_crt1_hide_center_menu(void)
{
   gtk_widget_hide(gui.c1_eventboxcc);
}

/* Signal handlers */

static void
c1_combobox_changed(GtkWidget *widget, gpointer data)
{
    gui_crt1_update_gfx();
    gui_show_lcd_menus(1);
}

static void
c1_comboboxls_changed(GtkWidget *widget, gpointer data)
{
    int index = gui_crt1_get_scaling_menu_index(CO_MI_SC_DEFAULT);

    if(!gl.supportcenter) return;

    if(index == CO_MI_SC_ON) {
       gui_crt1_hide_center_menu();
    } else {
       gui_crt1_show_center_menu();
    }
}

static void
c1_redetect(GtkWidget *widget, gpointer data)
{
    gdk_window_set_cursor(GTK_WIDGET(gui.window)->window, gui.waitcursor);
    gtk_main_iteration_do(FALSE);
    sisctrl_crt1_redetect_devices();
    gdk_window_set_cursor(GTK_WIDGET(gui.window)->window, NULL);
}

static void
gui_crt1_makeredetectbutton(void)
{
    gui.c1_redetect = gtk_button_new();
    gui.c1_redetectbox = xpm_label_box(icon_redetect, "Re_detect", gui.c1_redetect, 3, "", "");
    gtk_container_add(GTK_CONTAINER(gui.c1_redetect), gui.c1_redetectbox);
    gtk_widget_show(gui.c1_redetectbox);
    gtk_widget_show(gui.c1_redetect);
}

/* Build page */

static void
build_crt1_page(void)
{
    int i, j, k, l;
    GdkPixmap *icon;
    GdkBitmap *mask;
    GtkWidget *eventbox;

    gui.c1_frame = gtk_frame_new("CRT1 settings");
    gtk_container_set_border_width(GTK_CONTAINER(gui.c1_frame), 8);
    gtk_widget_show(gui.c1_frame);

    i = 4; j = 6; k = 15; l = 4;
    if(gl.crt1_page_type == CRT1_PAGE_TYPE_NORMAL) {
       if(gl.supportlcda && gl.supportscale) {
          i = 5; j = 4; k = 8; l++;
       }
       if(gl.supportcrt1redetect) {
          i++;
          if(gl.supportlcda && gl.supportscale) {
             j = 2;
          }
       }
    }

    gui.c1_table = gtk_table_new(i, 5, TRUE);
    gtk_table_set_row_spacings(GTK_TABLE(gui.c1_table), j);
    gtk_table_set_col_spacings(GTK_TABLE(gui.c1_table), 10);

    gtk_container_set_border_width(GTK_CONTAINER(gui.c1_table), k);

    gtk_container_add(GTK_CONTAINER(gui.c1_frame), gui.c1_table);
    gtk_widget_show(gui.c1_table);

    switch(gl.crt1_page_type) {

    case CRT1_PAGE_TYPE_NORMAL:
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2vga_xpm);
       gui.c1_imageon = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(icon);
       g_object_unref(mask);
       if(gl.islaptop) {
	  icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt1lcdal_xpm);
       } else {
	  icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt1lcda_xpm);
       }
       gui.c1_imagelcda = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(icon);
       g_object_unref(mask);
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2nomode_xpm);
       gui.c1_imagenomode = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(icon);
       g_object_unref(mask);

       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_imagenomode, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_imageon, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_imagelcda, 0, 5, 0, 3);
       gtk_widget_hide(gui.c1_imagenomode);
       gtk_widget_hide(gui.c1_imageon);
       gtk_widget_hide(gui.c1_imagelcda);

       gui.c1_label_onoff = gtk_label_new("Switch output");

       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_label_onoff, 0, 1, 3, 4);
       gtk_widget_show(gui.c1_label_onoff);

       gui.c1_combobox = gtk_combo_box_new_text();

       eventbox = gtk_event_box_new();
       gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
       gtk_widget_show(eventbox);
       gtk_container_add(GTK_CONTAINER(eventbox), gui.c1_combobox);

       gtk_table_attach(GTK_TABLE(gui.c1_table), eventbox, 1, 5, 3, 4, GTK_EXPAND|GTK_FILL, 0, 0, 0);

       gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
		gl.supportlcda ? "Select first output (CRT1) device type" :
				 "Enable/disable first output (CRT1)", "");

       if(gl.supportlcda && gl.supportscale) {

	  gui.c1_label_lcds = gtk_label_new("LCD scaling");
	  gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_label_lcds, 0, 1, 4, 5);

	  gui.c1_comboboxls = gtk_combo_box_new_text();
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_comboboxls), "Default");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_comboboxls), "On");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_comboboxls), "Off");

	  gtk_widget_show(gui.c1_comboboxls);

	  gui.c1_eventboxls = gtk_event_box_new();
	  gtk_container_set_border_width(GTK_CONTAINER(gui.c1_eventboxls), 0);
	  gtk_container_add(GTK_CONTAINER(gui.c1_eventboxls), gui.c1_comboboxls);

          gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.c1_eventboxls,
		"Scale low resolution modes to panel's native resolution", "");

	  if(gl.supportcenter) {

	     gtk_table_attach(GTK_TABLE(gui.c1_table), gui.c1_eventboxls, 1, 3, 4, 5, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	     gui.c1_comboboxcc = gtk_combo_box_new_text();
             gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_comboboxcc), "Default");
             gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_comboboxcc), "Don't center");
             gtk_combo_box_append_text(GTK_COMBO_BOX(gui.c1_comboboxcc), "Center");

	     gtk_widget_show(gui.c1_comboboxcc);

	     gui.c1_eventboxcc = gtk_event_box_new();
	     gtk_container_set_border_width(GTK_CONTAINER(gui.c1_eventboxcc), 0);
	     gtk_container_add(GTK_CONTAINER(gui.c1_eventboxcc), gui.c1_comboboxcc);

	     gtk_table_attach(GTK_TABLE(gui.c1_table), gui.c1_eventboxcc, 3, 5, 4, 5, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	     gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.c1_eventboxcc,
		   "Center unscaled output or pass native mode timing to the panel", "");

	  } else {

	     gtk_table_attach(GTK_TABLE(gui.c1_table), gui.c1_eventboxls, 1, 5, 4, 5, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	  }
       }

       gui_crt1_show_variant(gl.supportlcda ? 1 : 0);

       gui_crt1_set_type_menu_index(sisctrl_get_current_crt1type_index());

       gui_crt1_update_gfx();

       if(gl.supportlcda && gl.supportscale) {

	  gui_crt1_set_scaling_menu_index(sisctrl_get_lcd_scaling_history());

	  if(gl.supportcenter) {
	     gui_crt1_set_centering_menu_index(sisctrl_get_lcd_center_history());
	  }

	  gui_show_lcd_menus(1);

       }

       if(gl.supportcrt1redetect) {
          gui_crt1_makeredetectbutton();

	  gtk_table_attach(GTK_TABLE(gui.c1_table), gui.c1_redetect, 3, 5, l, l+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	  gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.c1_redetect,
		 "Re-detect VGA monitor and evaluate its frequency capabilities. The "
		 "result of the re-detection will be effective upon clicking Apply if "
		 "CRT1 is currently on/VGA, or otherwise as soon as CRT1 is switched "
		 "on/to VGA. It is recommended to either switch CRT1 off or to a "
		 "low-resolution mode with a low refresh rate before hotplugging and "
		 "re-detecting monitors. (If there are VertRefresh and/or HorizSync "
		 "ranges specified in the config file, redetection is ineffective.)", "");
       }

       g_signal_connect(G_OBJECT(gui.c1_combobox), "changed", G_CALLBACK(c1_combobox_changed), NULL);
       gtk_widget_show(gui.c1_combobox);

       if(gl.supportlcda && gl.supportscale && gl.supportcenter) {
	  g_signal_connect(G_OBJECT(gui.c1_comboboxls), "changed", G_CALLBACK(c1_comboboxls_changed), NULL);
       }

       if(gl.supportcrt1redetect) {
          g_signal_connect(G_OBJECT(gui.c1_redetect), "clicked", G_CALLBACK(c1_redetect), NULL);
       }

       break;

    case CRT1_PAGE_TYPE_REDETECT_ONLY:
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2vga_xpm);
       gui.c1_imageon = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(icon);
       g_object_unref(mask);
       gtk_widget_show(gui.c1_imageon);
       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_imageon, 0, 5, 0, 3);

       gui_crt1_makeredetectbutton();

       gtk_table_attach(GTK_TABLE(gui.c1_table), gui.c1_redetect, 0, 5, 3, 4, GTK_FILL|GTK_EXPAND, 0, 0, 0);

       gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.c1_redetect,
		 "Re-detect VGA monitor and evaluate its frequency capabilities. The "
		 "result of the re-detection will be effective upon clicking Apply. "
		 "It is recommended to switch to a low-resolution mode with a low refresh "
		 "rate before hotplugging and re-detecting monitors. (If there are "
		 "VertRefresh and/or HorizSync ranges specified in the config file, "
		 "redetection is ineffective.)", "");

       g_signal_connect(G_OBJECT(gui.c1_redetect), "clicked", G_CALLBACK(c1_redetect), NULL);
       break;

    case CRT1_PAGE_TYPE_NA_NOVB:
       gui.c1_label_nodh = gtk_label_new("No changable settings available");
       gtk_widget_show(gui.c1_label_nodh);
       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_label_nodh, 0, 5, 0, 4);
       break;

    case CRT1_PAGE_TYPE_NA_DHM:
       gui.c1_label_nodh = gtk_label_new("Device setup not supported in Dual Head mode");
       gtk_widget_show(gui.c1_label_nodh);
       gtk_table_attach_defaults(GTK_TABLE(gui.c1_table), gui.c1_label_nodh, 0, 5, 0, 4);
       break;
    }

    gui.c1_label = xpm_label_box(icon_crt1, "CRT1", NULL, 1, "Configure first output (CRT1)", "");
    gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.c1_frame, gui.c1_label);
}


/**************************************************
 *                   CRT2 page                    *
 **************************************************/

/* CRT2 type menu and graphics */

void
gui_crt2_set_type_menu_index(int index)
{
#ifdef USE_STRAY
   int index2 = 0;
#endif

   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_combobox), gui.om_rmenuindices[index]);

#ifdef USE_STRAY
   if(index == C2_MI_TV_YPBPR) {
      index2 = gui_crt2_get_signal_menu_index();
   }
   gui_stray_crt2_set_type_menu_index(index, index2);
#endif
}

int
gui_crt2_get_type_menu_index(void)
{
   int index = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_combobox));
   if(index < 0) return index;
   return gui.om_menuindices[index];
}

void
gui_crt2_hide_crt2_menu_items(void)
{
   int i;

   for(i = 0; i <= C2_MI_MAX; i++) {
      gui.om_mi_visible[i] = FALSE;
   }
}

void
gui_crt2_show_crt2_menu_item(int index)
{
   if(index < 0 || index > C2_MI_MAX)
      return;

   gui.om_mi_visible[index] = TRUE;
}

void
gui_crt2_finished_show_crt2_menu_items(void)
{
    int i, j, backup;
    static char *devnames[] = {  /* Keep order of C2_MI_* */
		"LCD",
		"TV (S-Video)",
		"TV (Composite)",
		"TV (Composite + S-Video)",
		"TV",
		"HDTV (HiVision 1080i)",
		"HDTV (YPbPr)",
		"DVI-A/VGA (analog output; secondary plug)",
		"TV (SCART RGB PAL)",
		"OFF"
    };

    backup = gui_crt2_get_type_menu_index();

    for(i = 0; i < C2_MI_NUM; i++) {
       gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.om_combobox), 0);
    }

    if(gl.supportdvi) {
	devnames[0] = "DVI-D/LCD (digital output)";
    }

    for(i = 0, j = 0; i < C2_MI_NUM; i++) {
       if(gui.om_mi_visible[i]) {
          gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_combobox), devnames[i]);
          gui.om_menuindices[j] = i;		/* C2_MI_XX = menuindices[real menuindex] */
          gui.om_rmenuindices[i] = j++;		/* rmenuindices[C2_MI_XX] = real menuindex */
       }
    }

    if(backup < 0) backup = 0;
    gui_crt2_set_type_menu_index(backup);
}

void
gui_crt2_show_hide_nomode(int nomode)
{
   if(nomode) gtk_widget_show(gui.om_nomode);
   else       gtk_widget_hide(gui.om_nomode);
}

static void
gui_crt2_select_gfx(int gfxnum, int nomode)
{
   gtk_widget_hide(gui.om_lcd);
   gtk_widget_hide(gui.om_tv);
   gtk_widget_hide(gui.om_tvs);
   gtk_widget_hide(gui.om_tvc);
   gtk_widget_hide(gui.om_tvalign);
   gtk_widget_hide(gui.om_ypbpr);
   gtk_widget_hide(gui.om_vga);

   switch(gfxnum) {
   case C2_GFX_TV_LCD:
      gtk_widget_show(gui.om_lcd);
      break;
   case C2_GFX_TV_TV:
      gtk_widget_show(gui.om_tv);
      break;
   case C2_GFX_TV_SVHS:
      gtk_widget_show(gui.om_tvs);
      break;
   case C2_GFX_TV_CVBS:
      gtk_widget_show(gui.om_tvc);
      break;
   case C2_GFX_TV_CVSV:
      gtk_widget_show(gui.om_tvalign);
      break;
   case C2_GFX_TV_HDTV:
      gtk_widget_show(gui.om_ypbpr);
      break;
   case C2_GFX_TV_VGA:
      gtk_widget_show(gui.om_vga);
      break;
   }

   gui_crt2_show_hide_nomode(nomode);
}

/* Update CRT2 gfx according to current selection in menu */
void
gui_crt2_update_gfx(void)
{
   int c2_index;
   int c2_gfx = C2_GFX_NONE;
   int nomode = 0;
   int mindex = sisctrl_dm_get_current_mode_index();
   int hindex = 0;
   Bool havetype, havesignal;

   if(!gui.om_combobox) return;
   havetype = (gui.om_comboboxt) ? TRUE : FALSE;
   havesignal = (gui.om_comboboxst) ? TRUE : FALSE;

   c2_index = gui_crt2_get_type_menu_index();

   switch(c2_index) {
   case C2_MI_TV_TV:
   case C2_MI_TV_SVHS:
   case C2_MI_TV_CVBS:
   case C2_MI_TV_CVSV:
      if(!havetype) hindex = C2_MI_TV_PAL;
      else hindex = gui_crt2_get_tvstd_menu_index();
      break;
   case C2_MI_TV_YPBPR:
      if(!havesignal) hindex = C2_MI_ST_525I;
      else hindex = gui_crt2_get_signal_menu_index();
      break;
   }

   if(!(sisctrl_checkmodeforcrtx(1, c2_index, mindex, hindex, hindex))) {
      nomode = 1;
   }

   switch(c2_index) {
   case C2_MI_TV_LCD:
      c2_gfx = C2_GFX_TV_LCD;
      break;
   case C2_MI_TV_TV:
      c2_gfx = C2_GFX_TV_TV;
      break;
   case C2_MI_TV_SCART:
      c2_gfx = C2_GFX_TV_TV;
      break;
   case C2_MI_TV_SVHS:
      c2_gfx = C2_GFX_TV_SVHS;
      break;
   case C2_MI_TV_CVBS:
      c2_gfx = C2_GFX_TV_CVBS;
      break;
   case C2_MI_TV_CVSV:
      c2_gfx = C2_GFX_TV_CVSV;
      break;
   case C2_MI_TV_HIVI:
      c2_gfx = C2_GFX_TV_HDTV;
      break;
   case C2_MI_TV_YPBPR:
      c2_gfx = C2_GFX_TV_HDTV;
      break;
   case C2_MI_TV_VGA:
      c2_gfx = C2_GFX_TV_VGA;
      break;
   }

   gui_crt2_select_gfx(c2_gfx, nomode);
}

/* LCD scaling menu */

void
gui_crt2_set_scaling_menu_index(int index)
{
   /* CO_SC_* are identical with our real menu indices */
   if(gui.om_comboboxls) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_comboboxls), (gint)index);
   }
}

int
gui_crt2_get_scaling_menu_index(int retdefault)
{
   /* CO_SC_* are identical with our real menu indices */
   if(!gui.om_comboboxls) return retdefault;
   return(int)(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_comboboxls)));
}

void
gui_crt2_show_scaling_menu_and_label(void)
{
   gtk_widget_show(gui.om_eventboxls);
   gtk_widget_show(gui.om_label_lcds);
}

void
gui_crt2_hide_scaling_menu_and_label(void)
{
   gtk_widget_hide(gui.om_eventboxls);
   gtk_widget_hide(gui.om_label_lcds);
}

/* LCD centering menu */

void
gui_crt2_set_centering_menu_index(int index)
{
   /* CO_CE_* are identical with our real menu indices */
   if(gui.om_comboboxcc) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_comboboxcc), (gint)index);
   }
}

int
gui_crt2_get_centering_menu_index(int retdefault)
{
   /* CO_CE_* are identical with our real menu indices */
   if(!gui.om_comboboxcc) return retdefault;
   return(int)(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_comboboxcc)));
}

void
gui_crt2_show_center_menu(void)
{
   gtk_widget_show(gui.om_eventboxcc);
}

void
gui_crt2_hide_center_menu(void)
{
   gtk_widget_hide(gui.om_eventboxcc);
}

/* TV standard menu */

void
gui_crt2_set_tvstd_menu_index(int index)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_comboboxt), gui.om_rmenuindices_t[index]);
}

int
gui_crt2_get_tvstd_menu_index(void)
{
   return (int)(gui.om_menuindices_t[gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_comboboxt))]);
}

void
gui_crt2_show_tvstd_menu_and_label(void)
{
   gtk_widget_show(gui.om_eventboxt);
   gtk_widget_show(gui.om_label_tvtype);
}

void
gui_crt2_hide_tvstd_menu_and_label(void)
{
   gtk_widget_hide(gui.om_eventboxt);
   gtk_widget_hide(gui.om_label_tvtype);
}

/* Signal type menu */

void
gui_crt2_set_signal_menu_index(int index)
{
#ifdef USE_STRAY
   int index2;
#endif

   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_comboboxst), gui.om_rmenuindices_st[index]);

#ifdef USE_STRAY
   index2 = gui_crt2_get_type_menu_index();
   if(index2 == C2_MI_TV_YPBPR) {
      gui_stray_crt2_set_type_menu_index(index2, index);
   }
#endif
}

int
gui_crt2_get_signal_menu_index(void)
{
   if(!gui.om_comboboxst) return 0;
   return (int)(gui.om_menuindices_st[gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_comboboxst))]);
}

void
gui_crt2_show_signal_menu_and_label(void)
{
   gtk_widget_show(gui.om_eventboxst);
   gtk_widget_show(gui.om_label_tvst);
}

void
gui_crt2_hide_signal_menu_and_label(void)
{
   gtk_widget_hide(gui.om_eventboxst);
   gtk_widget_hide(gui.om_label_tvst);
}

/* Aspect ratio menu */

void
gui_crt2_set_aspect_menu_index(int index)
{
   /* C2_MI_AS_* are identical with our real menu indices */
   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_comboboxas), index);
}

int
gui_crt2_get_aspect_menu_index(void)
{
   /* C2_MI_AS_* are identical with our real menu indices */
   return (int)(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_comboboxas)));
}

void
gui_crt2_show_aspect_menu_and_label(void)
{
   gtk_widget_show(gui.om_eventboxas);
   gtk_widget_show(gui.om_label_tvas);
}

void
gui_crt2_hide_aspect_menu_and_label(void)
{
   gtk_widget_hide(gui.om_eventboxas);
   gtk_widget_hide(gui.om_label_tvas);
}

/* Chrontel overscan menu */

void
gui_crt2_set_overscan_menu_index(int index)
{
   /* C2_MI_OV_* are identical with our real menu indices */
   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.om_comboboxo), (gint)index);
}

int
gui_crt2_get_overscan_menu_index(void)
{
   /* C2_MI_OV_* are identical with our real menu indices */
   return (int)(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.om_comboboxo)));
}

void
gui_crt2_show_overscan_superover_menu_item(void)
{
   if(gui.soverscanshown) return;
   /* Thanks god, it's the last one... */
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxo), "S-Overscan");
   gui.soverscanshown = TRUE;
}

void
gui_crt2_hide_overscan_superover_menu_item(void)
{
   if(!gui.soverscanshown) return;
   gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.om_comboboxo), C2_MI_OV_SUPEROVERSCAN);
   gui.soverscanshown = FALSE;
}

/* Show detected devices toggle button */

int
gui_crt2_show_detected_devices_checked(void)
{
   return (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.om_check))) ? 1 : 0;
}

/* Show/hide overscan menu */

void
gui_crt2_show_hide_overscan_menu(void)
{
   gint index1, index2;

   if(gl.crt2_page_type != CRT2_PAGE_TYPE_NORMAL) return;
   if(!gl.supportoverscan) return;

   index1 = gui_crt2_get_type_menu_index();
   index2 = gui_crt2_get_tvstd_menu_index();

   if(index1 >= C2_MI_TV_SVHS && index1 <= C2_MI_TV_TV) {
      gtk_widget_show(gui.om_eventboxo);
      if((index2 == C2_MI_TV_PAL) && gl.supportsoverscan) {
         gui_crt2_show_overscan_superover_menu_item();
      } else {
	 if(gui_crt2_get_overscan_menu_index() == C2_MI_OV_SUPEROVERSCAN) {
	    gui_crt2_set_overscan_menu_index(C2_MI_OV_OVERSCAN);
	 }
	 gui_crt2_hide_overscan_superover_menu_item();
      }
   } else {
      gtk_widget_hide(gui.om_eventboxo);
   }
}

void
gui_crt2_show_tv_std_and_overscan_menus(void)
{
   int index;

   if(gl.crt2_page_type != CRT2_PAGE_TYPE_NORMAL) return;

   index = gui_crt2_get_type_menu_index();

   if(index >= C2_MI_TV_SVHS && index <= C2_MI_TV_TV) {
      gui_crt2_show_tvstd_menu_and_label();
      gui_crt2_show_hide_overscan_menu();
   } else {
      gui_crt2_hide_tvstd_menu_and_label();
      gui_crt2_show_hide_overscan_menu();
   }
}

void
gui_crt2_show_hide_signal_menu(void)
{
   int index;

   if(gl.crt2_page_type != CRT2_PAGE_TYPE_NORMAL) return;
   if(!gl.supportypbpr) return;

   index = gui_crt2_get_type_menu_index();

   if(index == C2_MI_TV_YPBPR) {
      gui_crt2_show_signal_menu_and_label();
   } else {
      gui_crt2_hide_signal_menu_and_label();
   }
}

void
gui_crt2_show_hide_aspect_menu(void)
{
   int index;

   if(gl.crt2_page_type != CRT2_PAGE_TYPE_NORMAL) return;
   if(!gl.supportypbprar) return;

   index = gui_crt2_get_type_menu_index();

   if(index == C2_MI_TV_YPBPR) {
      gui_crt2_show_aspect_menu_and_label();
   } else {
      gui_crt2_hide_aspect_menu_and_label();
   }
}

/* signal handlers */

static void
om_menu_changed(GtkWidget *widget, gpointer data)
{
   gui_crt2_show_tv_std_and_overscan_menus();
   gui_crt2_show_hide_signal_menu();
   gui_crt2_show_hide_aspect_menu();
   gui_show_lcd_menus(2);
   gui_crt2_update_gfx();
}

static void
om_menut_changed(GtkWidget *widget, gpointer data)
{
   gui_crt2_show_hide_overscan_menu();
   gui_crt2_update_gfx();
}

static void
om_menust_changed(GtkWidget *widget, gpointer data)
{
   gui_crt2_update_gfx();
}

static void
om_omenuls_changed(GtkWidget *widget, gpointer data)
{
    int index = gui_crt2_get_scaling_menu_index(CO_MI_SC_DEFAULT);

    if(!gl.supportcenter) return;

    if(index == CO_MI_SC_ON) {
       gui_crt2_hide_center_menu();
    } else {
       gui_crt2_show_center_menu();
    }
}

static void
om_check_toggled(GtkWidget *widget, gpointer data)
{
    GTK_CHECK_MENU_ITEM(gui.stm2_det)->active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
    sisctrl_crt2_show_detected_only_changed();
}

static void
om_redetect(GtkWidget *widget, gpointer data)
{
    gdk_window_set_cursor(GTK_WIDGET(gui.window)->window, gui.waitcursor);
    gtk_main_iteration_do(FALSE);
    sisctrl_crt2_redetect_devices();
    gdk_window_set_cursor(GTK_WIDGET(gui.window)->window, NULL);
}

/* Build page */

static void
build_crt2_page(void)
{
    int i;
    GdkPixmap *icon;
    GdkBitmap *mask;
    GtkWidget *eventbox;

    gui.om_frame = gtk_frame_new("CRT2 settings");
    gtk_container_set_border_width(GTK_CONTAINER(gui.om_frame), 8);
    gtk_widget_show(gui.om_frame);

    gui.om_table = gtk_table_new(6, 5, TRUE);
    gtk_table_set_row_spacings(GTK_TABLE(gui.om_table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(gui.om_table), 10);

    gtk_container_set_border_width(GTK_CONTAINER(gui.om_table), 8);

    gtk_container_add(GTK_CONTAINER(gui.om_frame), gui.om_table);
    gtk_widget_show(gui.om_table);

    switch(gl.crt2_page_type) {

    case CRT2_PAGE_TYPE_NORMAL:
       /* Build device type menu and graphics */
       gui.om_label_type = gtk_label_new("Device type");
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_type, 0, 1, 3, 4);
       gtk_widget_show(gui.om_label_type);

       if(gl.islaptop) {
          icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2lcdl_xpm);
       } else {
          icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2lcd_xpm);
       }
       gui.om_lcd = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2tv_xpm);
       gui.om_tv = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2ypbpr_xpm);
       gui.om_ypbpr = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2svideo_xpm);
       gui.om_tvs = gtk_image_new_from_pixmap(icon, mask);
       gui.om_tvs2 = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2cvbs_xpm);
       gui.om_tvc = gtk_image_new_from_pixmap(icon, mask);
       gui.om_tvc2 = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));
       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2vga_xpm);
       gui.om_vga = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));

       icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, crt2nomode_xpm);
       gui.om_nomode = gtk_image_new_from_pixmap(icon, mask);
       g_object_unref(G_OBJECT(icon));
       g_object_unref(G_OBJECT(mask));

       gui.om_tvbox = gtk_hbox_new(FALSE, 0);
       gtk_container_set_border_width(GTK_CONTAINER(gui.om_tvbox), 1);
       gtk_box_pack_start(GTK_BOX(gui.om_tvbox), gui.om_tvc2, FALSE, FALSE, 5);
       gtk_box_pack_start(GTK_BOX(gui.om_tvbox), gui.om_tvs2, FALSE, FALSE, 0);
       gtk_widget_show(gui.om_tvc2);
       gtk_widget_show(gui.om_tvs2);
       gtk_widget_show(gui.om_tvbox);
       gui.om_tvalign = gtk_alignment_new(0.5, 0.5, 0, 0);
       gtk_container_add(GTK_CONTAINER(gui.om_tvalign), gui.om_tvbox);

       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_nomode, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_lcd, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_tv, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_tvs, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_tvc, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_tvalign, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_ypbpr, 0, 5, 0, 3);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_vga, 0, 5, 0, 3);
       gtk_widget_hide(gui.om_lcd);
       gtk_widget_hide(gui.om_tv);
       gtk_widget_hide(gui.om_tvs);
       gtk_widget_hide(gui.om_tvc);
       gtk_widget_hide(gui.om_tvalign);
       gtk_widget_hide(gui.om_ypbpr);
       gtk_widget_hide(gui.om_vga);
       gtk_widget_hide(gui.om_nomode);

       for(i = 0; i < C2_MI_NUM; i++) {
          gui.om_mi_visible[i] = FALSE;
       }

       gui.om_combobox = gtk_combo_box_new_text();

       gtk_widget_show(gui.om_combobox);

       eventbox = gtk_event_box_new();
       gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
       gtk_widget_show(eventbox);
       gtk_container_add(GTK_CONTAINER(eventbox), gui.om_combobox);

       gtk_table_attach(GTK_TABLE(gui.om_table), eventbox, 1, 5, 3, 4, GTK_FILL|GTK_EXPAND, 0, 0, 0);

       gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
		"Select second output (CRT2) device type", "");

       sisctrl_crt2_show_initial_crt2_type_menu();

       sisctrl_crt2_set_initial_crt2_type_history();

       /* Build "TV standard" menu */
       gui.om_label_tvtype = gtk_label_new("TV standard");
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_tvtype, 0, 1, 4, 5);

       gui.om_comboboxt = gtk_combo_box_new_text();

       i = 0;
       gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxt), "PAL");
       gui.om_rmenuindices_t[C2_MI_TV_PAL] = i;
       gui.om_menuindices_t[i++] = C2_MI_TV_PAL;
       gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxt), "NTSC");
       gui.om_rmenuindices_t[C2_MI_TV_NTSC] = i;
       gui.om_menuindices_t[i++] = C2_MI_TV_NTSC;
       if(gl.supportpalmn) {
          gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxt), "PAL-M");
          gui.om_rmenuindices_t[C2_MI_TV_PALM] = i;
          gui.om_menuindices_t[i++] = C2_MI_TV_PALM;
          gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxt), "PAL-N");
          gui.om_rmenuindices_t[C2_MI_TV_PALN] = i;
          gui.om_menuindices_t[i++] = C2_MI_TV_PALN;
       }
       if(gl.supportntscj) {
          gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxt), "NTSC-J");
          gui.om_rmenuindices_t[C2_MI_TV_NTSCJ] = i;
          gui.om_menuindices_t[i++] = C2_MI_TV_NTSCJ;
       }

       gtk_widget_show(gui.om_comboboxt);

       gui.om_eventboxt = gtk_event_box_new();
       gtk_container_set_border_width(GTK_CONTAINER(gui.om_eventboxt), 0);
       gtk_container_add(GTK_CONTAINER(gui.om_eventboxt), gui.om_comboboxt);

       gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxt, 1,
		(gl.supportoverscan) ? 3 : 5, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);

       gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_eventboxt,
		   "Select TV standard", "");

       /* Build "signal type" menu */
       if(gl.supportypbpr) {
          int where;
          gui.om_label_tvst = gtk_label_new("Signal type");
          gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_tvst, 0, 1, 4, 5);

          where = 5;
	  if(gl.supportypbprar) where = 3;

	  gui.om_comboboxst = gtk_combo_box_new_text();
	  i = 0;
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxst), "480i/525i");
	  gui.om_rmenuindices_st[C2_MI_ST_525I] = i;
	  gui.om_menuindices_st[i++] = C2_MI_ST_525I;
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxst), "480p/525p");
	  gui.om_rmenuindices_st[C2_MI_ST_525P] = i;
	  gui.om_menuindices_st[i++] = C2_MI_ST_525P;
	  if(gl.supportypbpr625i) {
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxst), "576i/625i");
	     gui.om_rmenuindices_st[C2_MI_ST_625I] = i;
	     gui.om_menuindices_st[i++] = C2_MI_ST_625I;
	  }
	  if(gl.supportypbpr625p) {
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxst), "576p/625p");
	     gui.om_rmenuindices_st[C2_MI_ST_625P] = i;
	     gui.om_menuindices_st[i++] = C2_MI_ST_625P;
	  }
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxst), "720p/750p");
	  gui.om_rmenuindices_st[C2_MI_ST_750P] = i;
	  gui.om_menuindices_st[i++] = C2_MI_ST_750P;
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxst), "1080i/1125i");
	  gui.om_rmenuindices_st[C2_MI_ST_1080I] = i;
	  gui.om_menuindices_st[i++] = C2_MI_ST_1080I;

	  gtk_widget_show(gui.om_comboboxst);

	  gui.om_eventboxst = gtk_event_box_new();
	  gtk_container_set_border_width(GTK_CONTAINER(gui.om_eventboxst), 0);
	  gtk_container_add(GTK_CONTAINER(gui.om_eventboxst), gui.om_comboboxst);

	  gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxst, 1, where, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	  gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_eventboxst,
		   "Select YPbPr signal type", "");

       }

       /* Build "aspect ratio" menu */
       if(gl.supportypbprar) {
          int where;

	  where = 0;
	  if(gl.supportypbpr) where = 3;

	  gui.om_label_tvas = gtk_label_new("Aspect ratio");
          gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_tvas, where, where+1, 4, 5);

	  gui.om_comboboxas = gtk_combo_box_new_text();
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxas), "4:3");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxas), "4:3 LB");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxas), "16:9");

	  gtk_widget_show(gui.om_comboboxas);

	  gui.om_eventboxas = gtk_event_box_new();
	  gtk_container_set_border_width(GTK_CONTAINER(gui.om_eventboxas), 0);
	  gtk_container_add(GTK_CONTAINER(gui.om_eventboxas), gui.om_comboboxas);

	  gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxas, where+1, 5, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	  gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_eventboxas,
		   "Select aspect ratio", "");
       }

       /* Build "overscan" menu */
       if(gl.supportoverscan) {
	  gui.om_comboboxo = gtk_combo_box_new_text();
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxo), "BIOS");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxo), "Underscan");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxo), "Overscan");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxo), "S-Overscan");
	  gui.soverscanshown = TRUE;

	  gtk_widget_show(gui.om_comboboxo);

	  gui.om_eventboxo = gtk_event_box_new();
	  gtk_container_set_border_width(GTK_CONTAINER(gui.om_eventboxo), 0);
	  gtk_container_add(GTK_CONTAINER(gui.om_eventboxo), gui.om_comboboxo);

	  gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxo, 3, 5, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	  gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_eventboxo,
		   "Enable overscan output", "");
       }

       /* Build "LCD scaling" and "LCD center" menus */
       if(gl.supportscale) {
	  gui.om_label_lcds = gtk_label_new("LCD scaling");
          gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_lcds, 0, 1, 4, 5);

	  gui.om_comboboxls = gtk_combo_box_new_text();
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxls), "Default");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxls), "On");
	  gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxls), "Off");

	  gtk_widget_show(gui.om_comboboxls);

	  gui.om_eventboxls = gtk_event_box_new();
	  gtk_container_set_border_width(GTK_CONTAINER(gui.om_eventboxls), 0);
	  gtk_container_add(GTK_CONTAINER(gui.om_eventboxls), gui.om_comboboxls);

	  gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_eventboxls,
		"Scale low resolution modes to panel's native resolution", "");

	  if(gl.supportcenter) {
	     gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxls, 1, 3, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	     gui.om_comboboxcc = gtk_combo_box_new_text();
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxcc), "Default");
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxcc), "Don't center");
	     gtk_combo_box_append_text(GTK_COMBO_BOX(gui.om_comboboxcc), "Center");

	     gtk_widget_show(gui.om_comboboxcc);

	     gui.om_eventboxcc = gtk_event_box_new();
	     gtk_container_set_border_width(GTK_CONTAINER(gui.om_eventboxcc), 0);
	     gtk_container_add(GTK_CONTAINER(gui.om_eventboxcc), gui.om_comboboxcc);

	     gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxcc, 3, 5, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	     gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_eventboxcc,
		 	"Center unscaled output or pass mode timing to the panel", "");
	  } else {
	     gtk_table_attach(GTK_TABLE(gui.om_table), gui.om_eventboxls, 1, 5, 4, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	  }
       }

       /* Show/hide menus and set menu histories (=currently selected items) */
       gui_crt2_show_tv_std_and_overscan_menus();

       sisctrl_crt2_set_initial_tv_std();

       if(gl.supportoverscan) {
          gui_crt2_set_overscan_menu_index(sisctrl_get_overscan_history());
       }

       if(gl.supportypbpr) {
	  gui_crt2_show_hide_signal_menu();
	  sisctrl_crt2_set_initial_signal_type();
       }

       if(gl.supportypbprar) {
	  gui_crt2_show_hide_aspect_menu();
	  sisctrl_crt2_set_initial_aspect_ratio();
       }

       if(gl.supportscale) {
          gui_crt2_set_scaling_menu_index(sisctrl_get_lcd_scaling_history());
	  if(gl.supportcenter) {
	     gui_crt2_set_centering_menu_index(sisctrl_get_lcd_center_history());
	  }
	  gui_show_lcd_menus(2);
       }

       /* Build "show detected only" check box and "redetect" button */
       gui.om_check = gtk_check_button_new_with_label("List detected devices only");
       if(gl.supportcrt2redetect) {
	  gui.om_table2 = gtk_table_new(1, 2, FALSE);
          gtk_table_set_row_spacings(GTK_TABLE(gui.om_table2), 0);
          gtk_table_set_col_spacings(GTK_TABLE(gui.om_table2), 0);
          gtk_container_set_border_width(GTK_CONTAINER(gui.om_table2), 0);
	  gtk_table_attach_defaults(GTK_TABLE(gui.om_table2), gui.om_check, 0, 1, 0, 1);
       } else {
	  gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_check, 1, 5, 5, 6);
       }
       gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_check,
		 	"List all device types or detected ones only", "");
       gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.om_check), TRUE);
       gtk_widget_show(gui.om_check);
       if(gl.supportcrt2redetect) {
	  gui.om_redetect = gtk_button_new();
	  gui.om_redetectbox = xpm_label_box(icon_redetect, "Re_detect", gui.om_redetect, 3, "", "");
	  gtk_container_add(GTK_CONTAINER(gui.om_redetect), gui.om_redetectbox);
	  gtk_table_attach(GTK_TABLE(gui.om_table2), gui.om_redetect, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	  gtk_widget_show(gui.om_redetectbox);
	  gtk_widget_show(gui.om_redetect);
          gtk_widget_show(gui.om_table2);
	  gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_table2, 1, 5, 5, 6);
	  gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.om_redetect,
		 	"Re-detect connected output devices", "");
       }

       /* Attach signal handler for "device type" menu */
       g_signal_connect(G_OBJECT(gui.om_combobox),  "changed", G_CALLBACK(om_menu_changed), NULL);

       /* Attach signal handler for "lcd scaling" menu */
       if(gl.supportscale && gl.supportcenter)  {
	  g_signal_connect(G_OBJECT(gui.om_comboboxls), "changed", G_CALLBACK(om_omenuls_changed), NULL);
       }

       /* Attach signal handler for "tv standard" menu */
       g_signal_connect(G_OBJECT(gui.om_comboboxt),   "changed", G_CALLBACK(om_menut_changed), NULL);

       if(gl.supportypbpr) {
	  g_signal_connect(G_OBJECT(gui.om_comboboxst),   "changed", G_CALLBACK(om_menust_changed), NULL);
       }

       /* Attach signal handler for "show detected only" check box */
       g_signal_connect(G_OBJECT(gui.om_check), "toggled", G_CALLBACK(om_check_toggled), NULL);

       /* Attach signal handler for "redetect" button */
       if(gl.supportcrt2redetect) {
          g_signal_connect(G_OBJECT(gui.om_redetect), "clicked", G_CALLBACK(om_redetect), NULL);
       }

       break;

    case CRT2_PAGE_TYPE_NA_NOVB:
       gui.om_label_nodh = gtk_label_new("No videobridge present");
       gtk_widget_show(gui.om_label_nodh);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_nodh, 0, 5, 2, 4);
       break;

    case CRT2_PAGE_TYPE_NA_DHM:
       gui.om_label_nodh = gtk_label_new("Device setup not supported in Dual Head mode");
       gtk_widget_show(gui.om_label_nodh);
       gtk_table_attach_defaults(GTK_TABLE(gui.om_table), gui.om_label_nodh, 0, 5, 2, 4);
       break;

    }

    gui.om_label = xpm_label_box(icon_crt2, "CRT2", NULL, 1, "Configure second output (CRT2)", "");
    gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.om_frame, gui.om_label);

}

/**************************************************
 *         Common for CRT1 and CRT2 pages         *
 **************************************************/

/* Show/hide LCD scaling/centering menu(s) for CRT1 (1) or CRT2 (2) pages */
void
gui_show_lcd_menus(int crtnum)
{
   int index1, index2, index3;

   if(crtnum == 1) {
      if(gl.crt1_page_type != CRT1_PAGE_TYPE_NORMAL) return;
   } else {
      if(gl.crt2_page_type != CRT2_PAGE_TYPE_NORMAL) return;
   }

   if(!gl.supportscale) return;

   if(crtnum == 2) {
      index1 = gui_crt2_get_type_menu_index();
      index2 = C2_MI_TV_LCD;
      index3 = gui_crt2_get_scaling_menu_index(CO_MI_SC_DEFAULT);
   } else {
      if(!gl.supportlcda) return;
      index1 = gui_crt1_get_type_menu_index();
      index2 = C1_MI_LCDA;
      index3 = gui_crt1_get_scaling_menu_index(CO_MI_SC_DEFAULT);
   }

   if(index1 == index2) {
      if(crtnum == 2) {
	 gui_crt2_show_scaling_menu_and_label();
      } else {
         gui_crt1_show_scaling_menu_and_label();
      }
      if(gl.supportcenter) {
         if(index3 == CO_MI_SC_ON) {
	    if(crtnum == 2) gui_crt2_hide_center_menu();
	    else	    gui_crt1_hide_center_menu();
	 } else {
            if(crtnum == 2) gui_crt2_show_center_menu();
	    else	    gui_crt1_show_center_menu();
	 }
      }
   } else {
      if(crtnum == 2) {
         gui_crt2_hide_scaling_menu_and_label();
      } else {
         gui_crt1_hide_scaling_menu_and_label();
      }
      if(gl.supportcenter) {
         if(crtnum == 2) gui_crt2_hide_center_menu();
	 else	         gui_crt1_hide_center_menu();
      }
   }
}

/**************************************************
 *                     TV page                    *
 **************************************************/

static void
tv_update_ahr(void)
{
   int lwidth, lheight, widthheight;

   if(!gui.tv_layout) return;

   if(!GTK_WIDGET_REALIZED(gui.tv_layout)) return;

   if(!(GTK_LAYOUT(gui.tv_layout)->bin_window)) return;

#if 0	/* Does nothing */
   if(GTK_WIDGET_REALIZED(gui.tp_table)) {
      gtk_widget_set_size_request(gui.tv_layout,
      		gui.tp_table->allocation.width,
      		gui.tp_table->allocation.height);
   }
#endif

   lwidth = gui.tv_layout->allocation.width;
   lheight = gui.tv_layout->allocation.height;
   if(!lwidth || !lheight) return;

   if(!gui.tv_context) {
      gui.tv_context = gdk_gc_new(GTK_LAYOUT(gui.tv_layout)->bin_window);
   }

   gdk_gc_set_rgb_fg_color(gui.tv_context, &ga_colorblack);
   gdk_gc_set_fill(gui.tv_context, GDK_SOLID);
   gdk_draw_rectangle(GTK_LAYOUT(gui.tv_layout)->bin_window,
		      gui.tv_context,
		      TRUE,
		      0, 0,
		      lwidth, lheight);

   gdk_gc_set_rgb_fg_color(gui.tv_context, &ga_colorred);
   gdk_gc_set_line_attributes(gui.tv_context, 8, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);


   widthheight = ((lheight > lwidth) ? lwidth : lheight) / 5 * 4;

   if(widthheight > 0) {
      gdk_draw_rectangle(GTK_LAYOUT(gui.tv_layout)->bin_window,
		      gui.tv_context,
		      FALSE,
		      (lwidth - widthheight) / 2, (lheight - widthheight) / 2,
		      widthheight, widthheight);

   } else {
     /* Hm... */
   }
}

void
gui_tv_antiflicker_enable_and_set_menu_index(int enable, int index)
{
   if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;

   gtk_widget_set_sensitive(gui.tp_combobox, enable ? TRUE : FALSE);
   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.tp_combobox), (gint)index);
}

void
gui_tv_colorcalib_c_enable_and_set_value(int enable, double value)
{
   GtkAdjustment *myadj;

   if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;

   myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_coc));

   if(!enable) {
      gtk_widget_set_sensitive(gui.tp_range_coc, FALSE);
   } else {
      gtk_widget_set_sensitive(gui.tp_range_coc, TRUE);
      gtk_adjustment_set_value(myadj, value);
      g_signal_emit_by_name(G_OBJECT(myadj), "changed");
   }

}

void
gui_tv_colorcalib_f_enable_and_set_value(int enable, double value)
{
   GtkAdjustment *myadj;

   if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;

   myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_cof));

   if(!enable) {
      gtk_widget_set_sensitive(gui.tp_range_cof, FALSE);
   } else {
      gtk_widget_set_sensitive(gui.tp_range_cof, TRUE);
      gtk_adjustment_set_value(myadj, value);
      g_signal_emit_by_name(G_OBJECT(myadj), "changed");
   }
}

void
gui_tv_saturation_enable_and_set_value(int enable, double value)
{
   GtkAdjustment *myadj;

   if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;
   if(!gl.supporttvsaturation) return;

   myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_sa));

   if(!enable) {
      gtk_widget_set_sensitive(gui.tp_range_sa, FALSE);
   } else {
      gtk_widget_set_sensitive(gui.tp_range_sa, TRUE);
      gtk_adjustment_set_value(myadj, value);
      g_signal_emit_by_name(G_OBJECT(myadj), "changed");
   }
}

void
gui_tv_edgeenhance_enable_and_set_value(int enable, double value)
{
   GtkAdjustment *myadj;

   if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;
   if(!gl.supporttvsaturation) return;

   myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_ee));

   if(!enable) {
      gtk_widget_set_sensitive(gui.tp_range_ee, FALSE);
   } else {
      gtk_widget_set_sensitive(gui.tp_range_ee, TRUE);
      gtk_adjustment_set_value(myadj, value);
      g_signal_emit_by_name(G_OBJECT(myadj), "changed");
   }
}

void
gui_tv_filter_c_enable_and_set_value(int enable, int value)
{
   if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;

   if(!enable) {
      gtk_widget_set_sensitive(gui.tp_check_cfi, FALSE);
   } else {
      gtk_widget_set_sensitive(gui.tp_check_cfi, TRUE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.tp_check_cfi), (value ? TRUE : FALSE));
   }
}

void
gui_tv_filter_y_enable_and_set_value(int enable, int index)
{
    if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_SIS) return;

    if(!enable) {
       gtk_widget_set_sensitive(gui.tp_comboboxy, FALSE);
       gtk_combo_box_set_active(GTK_COMBO_BOX(gui.tp_comboboxy), 1);
    } else {
       gtk_widget_set_sensitive(gui.tp_comboboxy, TRUE);
       gtk_combo_box_set_active(GTK_COMBO_BOX(gui.tp_comboboxy), index);
    }
}

void
gui_tv_chrontel_slider_set_value(int slider, int enable, double value)
{
    GtkWidget *widget;
    GtkAdjustment *myadj;

    if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_CHRONTEL) return;

    switch(slider) {
    case TV_CH_SLI_CONTRAST:
       widget = gui.tp_range_co;
       break;
    case TV_CH_SLI_TEXTENHANCE:
       widget = gui.tp_range_te;
       break;
    case TV_CH_SLI_CHROMAFF:
       widget = gui.tp_range_cf;
       break;
    case TV_CH_SLI_LUMAFF:
       widget = gui.tp_range_lf;
       break;
    default:
       return;
    }

    myadj = gtk_range_get_adjustment(GTK_RANGE(widget));

    if(!enable) {
       gtk_widget_set_sensitive(widget, FALSE);
    } else {
       gtk_widget_set_sensitive(widget, TRUE);
       gtk_adjustment_set_value(myadj, value);
       g_signal_emit_by_name(G_OBJECT(myadj), "changed");
    }
}

void
gui_tv_chrontel_cvbscolor_set_value(int enable, int val)
{
    if(gl.tv_page_subtype != TV_PAGE_SUBTYPE_CHRONTEL) return;

    if(!enable) {
      gtk_widget_set_sensitive(gui.tp_check_col, FALSE);
   } else {
      gtk_widget_set_sensitive(gui.tp_check_col, TRUE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.tp_check_col), val ? TRUE : FALSE);
   }
}

/* Signal handlers */

static void
tv_de_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_de_clicked();
}

static void
tv_up_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_up_clicked();
}

static void
tv_dn_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_dn_clicked();
}

static void
tv_le_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_le_clicked();
}

static void
tv_ri_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_ri_clicked();
}

static void
tv_xm_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_xm_clicked();
}

static void
tv_xp_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_xp_clicked();
}

static void
tv_ym_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_ym_clicked();
}

static void
tv_yp_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_tv_yp_clicked();
}

static void
tv_arh_toggled(GtkWidget *widget, gpointer data)
{
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
#if 0
      if(GTK_WIDGET_REALIZED(gui.tp_frame)) {
         gtk_widget_set_size_request(gui.tp_framearh,
      		gui.tp_frame->allocation.width - 28,
      		gui.tp_frame->allocation.height);
      }
#endif
      gtk_widget_hide(gui.tp_frame);
      gtk_widget_show(gui.tp_framearh);
   } else {
      gtk_widget_hide(gui.tp_framearh);
      gtk_widget_show(gui.tp_frame);
   }
}

static gboolean
tv_arh_exposed(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   tv_update_ahr();
   return FALSE;
}

static void
tp_af_changed(GtkWidget *widget, gpointer data)
{
   int index = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.tp_combobox));

   sisctrl_tp_af_changed(index);
}

static void
tp_sa_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_tp_sa_changed(myval);
}

static void
tp_ee_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_tp_ee_changed(myval);
}

static void
tp_check_cfi_toggled(GtkWidget *widget, gpointer data)
{
    int myval = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? 1 : 0;

    sisctrl_tp_check_cfi_toggled(myval);
}

static void
tp_yfi_changed(GtkWidget *widget, gpointer data)
{
    int index = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.tp_comboboxy));

    sisctrl_tp_yfi_changed(index);
}

static void
tp_color_calib_changed(void)
{
    GtkAdjustment *myadj1 = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_coc));
    double myval1 = gtk_adjustment_get_value(myadj1);
    GtkAdjustment *myadj2 = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_cof));
    double myval2 = gtk_adjustment_get_value(myadj2);

    sisctrl_tp_col_calib_changed(myval1, myval2);
}

static void
tp_coc_changed(GtkWidget *widget, gpointer data)
{
   tp_color_calib_changed();
}

static void
tp_cof_changed(GtkWidget *widget, gpointer data)
{
   tp_color_calib_changed();
}

static void
tp_co_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_tp_co_changed(myval);
}

static void
tp_te_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_tp_te_changed(myval);
}

static void
tp_cf_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_tp_cf_changed(myval);
}

static void
tp_lf_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_tp_lf_changed(myval);
}

static void
tp_check_col_toggled(GtkWidget *widget, gpointer data)
{
   int myval = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? 1 : 0;

   sisctrl_tp_check_col_toggled(myval);
}

/* Build page */

static void
build_tv_page(void)
{
   int rows, eepos, colcpos, colfpos, filtpos;
   GtkWidget *eventbox;

   switch(gl.tv_page_type) {

   case TV_PAGE_TYPE_NORMAL:

      gui.tp_box = gtk_table_new(1, 3, TRUE);
      gtk_table_set_row_spacings(GTK_TABLE(gui.tp_box), 0);
      gtk_table_set_col_spacings(GTK_TABLE(gui.tp_box), 0);
      gtk_container_set_border_width(GTK_CONTAINER(gui.tp_box), 0);
      gtk_widget_show(gui.tp_box);

      gui.tp_frame = gtk_frame_new("TV settings");
      gtk_container_set_border_width(GTK_CONTAINER(gui.tp_frame), 8);
      gtk_widget_show(gui.tp_frame);

      gtk_table_attach_defaults(GTK_TABLE(gui.tp_box), gui.tp_frame, 0, 2, 0, 1);

      switch(gl.tv_page_subtype) {

      case TV_PAGE_SUBTYPE_SIS:
         rows = 4; eepos = 1; colcpos = 1; colfpos = 2; filtpos = 3;

	 if(gl.supporttvsaturation) {
	    rows++; eepos++; colcpos++; colfpos++; filtpos++;
	 }

	 if(gl.supporttvedgeenhance) {
	    rows++; colcpos++; colfpos++; filtpos++;
	 }

         gui.tp_table = gtk_table_new(rows, 3, TRUE);
         gtk_table_set_row_spacings(GTK_TABLE(gui.tp_table), 0);
         gtk_table_set_col_spacings(GTK_TABLE(gui.tp_table), 13);

         gtk_container_set_border_width(GTK_CONTAINER(gui.tp_table), 4);

	 gtk_container_add(GTK_CONTAINER(gui.tp_frame), gui.tp_table);
         gtk_widget_show(gui.tp_table);

         gui.tp_label_af = gtk_label_new("Antiflicker");
         gtk_widget_show(gui.tp_label_af);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_af, 0, 1, 0, 1);

	 gui.tp_combobox = gtk_combo_box_new_text();
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_combobox), "Off");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_combobox), "Low");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_combobox), "Medium");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_combobox), "High");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_combobox), "Adaptive");

	 gtk_widget_show(gui.tp_combobox);

	 eventbox = gtk_event_box_new();
	 gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
	 gtk_widget_show(eventbox);
	 gtk_container_add(GTK_CONTAINER(eventbox), gui.tp_combobox);

	 gtk_table_attach(GTK_TABLE(gui.tp_table), eventbox, 1, 3, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
		 	"Adjust anti-flicker facility for interlaced TV output (ineffective in progressive scan modes)", "");

         sisctrl_tv_antiflicker_set_history();

	 g_signal_connect(G_OBJECT(gui.tp_combobox), "changed", G_CALLBACK(tp_af_changed), NULL);

	 if(gl.supporttvsaturation) {
            gui.tp_label_sa = gtk_label_new("Saturation");
            gtk_widget_show(gui.tp_label_sa);
            gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_sa, 0, 1, 1, 2);

            gui.tp_range_sa = gtk_hscale_new_with_range(0, 15, 1);
	    gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_sa), GTK_POS_RIGHT);
            gtk_widget_show(gui.tp_range_sa);
            sisctrl_tv_saturation_set_value();
            gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_sa, 1, 3, 1, 2);

	    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_sa,
		 	"Adjust color saturation", "");

	    {
	    GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_sa));
            g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_sa_changed), NULL);
	    }
         }

	 if(gl.supporttvedgeenhance) {
            gui.tp_label_ee = gtk_label_new("Edge enhance");
            gtk_widget_show(gui.tp_label_ee);
            gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_ee, 0, 1, eepos, eepos+1);

            gui.tp_range_ee = gtk_hscale_new_with_range(0, 15, 1);
	    gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_ee), GTK_POS_RIGHT);
            gtk_widget_show(gui.tp_range_ee);
            sisctrl_tv_edgeenhance_set_value();
            gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_ee, 1, 3, eepos, eepos+1);

	    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_ee,
		 	"Adjust edge enhancement for interlaced TV output (ineffective in progressive scan modes)", "");

	    {
            GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_ee));
            g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_ee_changed), NULL);
	    }
         }

	 gui.tp_label_coc = gtk_label_new("Color (coarse)");
         gtk_widget_show(gui.tp_label_coc);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_coc, 0, 1, colcpos, colcpos+1);

         gui.tp_range_coc = gtk_hscale_new_with_range(-120, 120, 1);
	 gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_coc), GTK_POS_RIGHT);
         gtk_widget_show(gui.tp_range_coc);

         sisctrl_tv_col_calib_set_c_value();

         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_coc, 1, 3, colcpos, colcpos+1);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_coc,
		 	"Adjust color carrier frequency for PAL/NTSC output (coarse)", "");

         {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_coc));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_coc_changed), NULL);
         }

	 gui.tp_label_cof = gtk_label_new("Color (fine)");
         gtk_widget_show(gui.tp_label_cof);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_cof, 0, 1, colfpos, colfpos+1);

         gui.tp_range_cof = gtk_hscale_new_with_range(-128, 127, 1);
	 gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_cof), GTK_POS_RIGHT);
         gtk_widget_show(gui.tp_range_cof);

         sisctrl_tv_col_calib_set_f_value();

         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_cof, 1, 3, colfpos, colfpos+1);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_cof,
		 	"Adjust color carrier frequency for PAL/NTSC output (fine)", "");

         {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_cof));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_cof_changed), NULL);
         }

	 gui.tp_check_cfi = gtk_check_button_new_with_label("C-Filter");
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_check_cfi, 2, 3, filtpos, filtpos+1);
         gtk_widget_show(gui.tp_check_cfi);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_check_cfi,
		 	"Enable chroma (C) filter for composite TV output", "");

         sisctrl_tv_c_filter_set_value();

	 g_signal_connect(G_OBJECT(gui.tp_check_cfi), "toggled", G_CALLBACK(tp_check_cfi_toggled), NULL);

	 gui.tp_label_yfi = gtk_label_new("Y-Filter");
	 gtk_label_set_justify(GTK_LABEL(gui.tp_label_yfi), GTK_JUSTIFY_RIGHT);
         gtk_widget_show(gui.tp_label_yfi);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_yfi, 0, 1, filtpos, filtpos+1);

	 gui.tp_comboboxy = gtk_combo_box_new_text();
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Off");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Default");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 1");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 2");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 3");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 4");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 5");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 6");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.tp_comboboxy), "Filter 7");

	 gtk_widget_show(gui.tp_comboboxy);

	 eventbox = gtk_event_box_new();
	 gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
	 gtk_widget_show(eventbox);
	 gtk_container_add(GTK_CONTAINER(eventbox), gui.tp_comboboxy);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
		 	"Select luma (Y) filter for composite TV output", "");

         gtk_table_attach(GTK_TABLE(gui.tp_table), eventbox, 1, 2, filtpos, filtpos+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

	 sisctrl_tv_y_filter_set_value();

	 g_signal_connect(G_OBJECT(gui.tp_comboboxy), "changed", G_CALLBACK(tp_yfi_changed), NULL);
         break;

      case TV_PAGE_SUBTYPE_CHRONTEL:
         gui.tp_table = gtk_table_new(5, 3, TRUE);
         gtk_table_set_row_spacings(GTK_TABLE(gui.tp_table), 2);
         gtk_table_set_col_spacings(GTK_TABLE(gui.tp_table), 1);

         gtk_container_set_border_width(GTK_CONTAINER(gui.tp_table), 4);

         gtk_container_add(GTK_CONTAINER(gui.tp_frame), gui.tp_table);
         gtk_widget_show(gui.tp_table);

         gui.tp_label_co = gtk_label_new("Contrast");
         gtk_widget_show(gui.tp_label_co);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_co, 0, 1, 0, 1);

         gui.tp_range_co = gtk_hscale_new_with_range(0, 15, 1);
         gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_co), GTK_POS_RIGHT);
         gtk_widget_show(gui.tp_range_co);
         sisctrl_tv_ch_contrast_set_value();
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_co, 1, 3, 0, 1);

         {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_co));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_co_changed), NULL);
         }

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_co,
		 	"Adjust contrast", "");

         gui.tp_label_te = gtk_label_new("Text enhance");
         gtk_widget_show(gui.tp_label_te);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_te, 0, 1, 1, 2);

         gui.tp_range_te = gtk_hscale_new_with_range(0, 15, 1);
         gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_te), GTK_POS_RIGHT);
         gtk_widget_show(gui.tp_range_te);
         sisctrl_tv_ch_textenhance_set_value();
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_te, 1, 3, 1, 2);

         {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_te));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_te_changed), NULL);
         }

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_te,
		 	"Adjust text enhancement", "");

         gui.tp_label_cf = gtk_label_new("C-flickerfilter");
         gtk_widget_show(gui.tp_label_cf);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_cf, 0, 1, 2, 3);

         gui.tp_range_cf = gtk_hscale_new_with_range(0, 15, 1);
         gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_cf), GTK_POS_RIGHT);
         gtk_widget_show(gui.tp_range_cf);
         sisctrl_tv_ch_chromaff_set_value();
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_cf, 1, 3, 2, 3);

         {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_cf));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_cf_changed), NULL);
         }

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_cf,
		 	"Adjust chroma flicker filter", "");

         gui.tp_label_lf = gtk_label_new("L-flickerfilter");
         gtk_widget_show(gui.tp_label_lf);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_label_lf, 0, 1, 3, 4);

	 gui.tp_range_lf = gtk_hscale_new_with_range(0, 15, 1);
         gtk_scale_set_value_pos(GTK_SCALE(gui.tp_range_lf), GTK_POS_RIGHT);
         gtk_widget_show(gui.tp_range_lf);
         sisctrl_tv_ch_lumaff_set_value();
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_range_lf, 1, 3, 3, 4);

         {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.tp_range_lf));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(tp_lf_changed), NULL);
         }

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_range_lf,
		 	"Adjust luma flicker filter", "");

         gui.tp_check_col = gtk_check_button_new_with_label("Color (CVBS)");
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_table), gui.tp_check_col, 0, 3, 4, 5);
         gtk_widget_show(gui.tp_check_col);
         sisctrl_tv_ch_cvbscol_set_value();
         g_signal_connect(G_OBJECT(gui.tp_check_col), "toggled", G_CALLBACK (tp_check_col_toggled), NULL);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tp_check_col,
		 	"Enable color for composite output", "");

         break;
      }

      if(gl.supporttvsize) {
         gui.tv_frame = gtk_frame_new("TV position & size");
      } else {
         gui.tv_frame = gtk_frame_new("TV position");
      }
      gtk_container_set_border_width(GTK_CONTAINER(gui.tv_frame), 8);
      gtk_widget_show(gui.tv_frame);

      gtk_table_attach_defaults(GTK_TABLE(gui.tp_box), gui.tv_frame, 2, 3, 0, 1);

      gui.tv_table = gtk_table_new(3, 3, TRUE);
      gtk_table_set_row_spacings(GTK_TABLE(gui.tv_table), 8);

      gtk_table_set_col_spacings(GTK_TABLE(gui.tv_table), 4);

      gtk_container_set_border_width(GTK_CONTAINER(gui.tv_table), 4);

      if(gl.supporttvsize) {
         gui.tv_table2 = gtk_table_new(7, 1, TRUE);
         gtk_container_set_border_width(GTK_CONTAINER(gui.tv_table2), 0);
         gtk_table_set_row_spacings(GTK_TABLE(gui.tv_table2), 4);
         gtk_table_attach_defaults(GTK_TABLE(gui.tv_table2), gui.tv_table, 0, 1, 0, 6);
         gtk_container_add(GTK_CONTAINER(gui.tv_frame), gui.tv_table2);
         gtk_widget_show(gui.tv_table2);
      } else {
         gtk_container_add(GTK_CONTAINER(gui.tv_frame), gui.tv_table);
      }

      gtk_widget_show(gui.tv_table);

      gui.tv_arrow_le = xpm_label_box(icon_tvleft, NULL, NULL, 1, "", "");
      gui.tv_arrow_ri = xpm_label_box(icon_tvright, NULL, NULL, 1, "", "");
      gui.tv_arrow_up = xpm_label_box(icon_tvup, NULL, NULL, 1, "", "");
      gui.tv_arrow_dn = xpm_label_box(icon_tvdown, NULL, NULL, 1, "", "");
      gui.tv_arrow_def = xpm_label_box(icon_tvdef, NULL, NULL, 1, "", "");

      gui.tv_button_up = gtk_button_new();
      gui.tv_button_dn = gtk_button_new();
      gui.tv_button_ri = gtk_button_new();
      gui.tv_button_le = gtk_button_new();
      gui.tv_button_de = gtk_button_new();

      gtk_container_add(GTK_CONTAINER(gui.tv_button_up), gui.tv_arrow_up);
      gtk_container_add(GTK_CONTAINER(gui.tv_button_dn), gui.tv_arrow_dn);
      gtk_container_add(GTK_CONTAINER(gui.tv_button_le), gui.tv_arrow_le);
      gtk_container_add(GTK_CONTAINER(gui.tv_button_ri), gui.tv_arrow_ri);
      gtk_container_add(GTK_CONTAINER(gui.tv_button_de), gui.tv_arrow_def);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_up,
		 	"Move TV output upwards", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_dn,
		 	"Move TV output downwards", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_ri,
		 	"Move TV output right", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_le,
		 	"Move TV output left", "");

      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_de,
			gl.supporttvsize ? "Reset TV position and size" :
					   "Reset TV position", "");

      gtk_widget_show(gui.tv_button_up);
      gtk_widget_show(gui.tv_button_ri);
      gtk_widget_show(gui.tv_button_le);
      gtk_widget_show(gui.tv_button_dn);
      gtk_widget_show(gui.tv_button_de);

      gtk_widget_show(gui.tv_arrow_le);
      gtk_widget_show(gui.tv_arrow_ri);
      gtk_widget_show(gui.tv_arrow_dn);
      gtk_widget_show(gui.tv_arrow_up);
      gtk_widget_show(gui.tv_arrow_def);

      gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_up, 1, 2, 0, 1);
      gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_le, 0, 1, 1, 2);
      gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_de, 1, 2, 1, 2);
      gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_ri, 2, 3, 1, 2);
      gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_dn, 1, 2, 2, 3);

      if(!gl.supporttvpos) {
         gtk_widget_set_sensitive(gui.tv_button_up, FALSE);
         gtk_widget_set_sensitive(gui.tv_button_dn, FALSE);
         gtk_widget_set_sensitive(gui.tv_button_le, FALSE);
         gtk_widget_set_sensitive(gui.tv_button_ri, FALSE);
         gtk_widget_set_sensitive(gui.tv_button_de, FALSE);
      }

      g_signal_connect(G_OBJECT (gui.tv_button_up), "clicked", G_CALLBACK (tv_up_clicked), NULL);
      g_signal_connect(G_OBJECT (gui.tv_button_dn), "clicked", G_CALLBACK (tv_dn_clicked), NULL);
      g_signal_connect(G_OBJECT (gui.tv_button_le), "clicked", G_CALLBACK (tv_le_clicked), NULL);
      g_signal_connect(G_OBJECT (gui.tv_button_ri), "clicked", G_CALLBACK (tv_ri_clicked), NULL);
      g_signal_connect(G_OBJECT (gui.tv_button_de), "clicked", G_CALLBACK (tv_de_clicked), NULL);

      if(gl.supporttvsize) {
         gui.tv_arrow_xp = xpm_label_box(icon_tvxp, NULL, NULL, 1, "", "");
	 gui.tv_arrow_xm = xpm_label_box(icon_tvxm, NULL, NULL, 1, "", "");
	 gui.tv_arrow_yp = xpm_label_box(icon_tvyp, NULL, NULL, 1, "", "");
	 gui.tv_arrow_ym = xpm_label_box(icon_tvym, NULL, NULL, 1, "", "");

	 gui.tv_button_xp = gtk_button_new();
         gui.tv_button_xm = gtk_button_new();
	 gui.tv_button_yp = gtk_button_new();
         gui.tv_button_ym = gtk_button_new();

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_xp,
       		 	"Enlarge TV output horizontally", "");
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_xm,
       		 	"Shrink TV output horizontally", "");
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_yp,
       		 	"Enlarge TV output vertically", "");
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_ym,
       		 	"Shrink TV output vertically", "");

	 gtk_container_add(GTK_CONTAINER(gui.tv_button_xp), gui.tv_arrow_xp);
         gtk_container_add(GTK_CONTAINER(gui.tv_button_xm), gui.tv_arrow_xm);
	 gtk_container_add(GTK_CONTAINER(gui.tv_button_yp), gui.tv_arrow_yp);
         gtk_container_add(GTK_CONTAINER(gui.tv_button_ym), gui.tv_arrow_ym);

	 gtk_widget_show(gui.tv_button_xp);
         gtk_widget_show(gui.tv_button_xm);
	 gtk_widget_show(gui.tv_button_yp);
         gtk_widget_show(gui.tv_button_ym);

	 gtk_widget_show(gui.tv_arrow_xp);
	 gtk_widget_show(gui.tv_arrow_xm);
         gtk_widget_show(gui.tv_arrow_yp);
	 gtk_widget_show(gui.tv_arrow_ym);

	 gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_xm, 0, 1, 2, 3);
         gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_xp, 2, 3, 2, 3);
	 gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_ym, 0, 1, 0, 1);
         gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_button_yp, 2, 3, 0, 1);

	 g_signal_connect(G_OBJECT (gui.tv_button_xp), "clicked", G_CALLBACK (tv_xp_clicked), NULL);
         g_signal_connect(G_OBJECT (gui.tv_button_xm), "clicked", G_CALLBACK (tv_xm_clicked), NULL);
	 g_signal_connect(G_OBJECT (gui.tv_button_yp), "clicked", G_CALLBACK (tv_yp_clicked), NULL);
         g_signal_connect(G_OBJECT (gui.tv_button_ym), "clicked", G_CALLBACK (tv_ym_clicked), NULL);
      }

      if(gl.supporttvsize) {
         gui.tv_button_arh = gtk_check_button_new_with_label("Aspect ratio helper");
         gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.tv_button_arh), FALSE);
	 gtk_table_attach_defaults(GTK_TABLE(gui.tv_table2), gui.tv_button_arh, 0, 1, 6, 7);
	 gtk_widget_show(gui.tv_button_arh);

         g_signal_connect(G_OBJECT(gui.tv_button_arh), "toggled", G_CALLBACK (tv_arh_toggled), NULL);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_button_arh,
			"Enable the aspect ratio helper", "");

	 gui.tp_framearh = gtk_frame_new("TV aspect ratio helper");
         gtk_container_set_border_width(GTK_CONTAINER(gui.tp_framearh), 8);
         gtk_widget_hide(gui.tp_framearh);
         gtk_table_attach_defaults(GTK_TABLE(gui.tp_box), gui.tp_framearh, 0, 2, 0, 1);

         gui.tv_layout = gtk_layout_new(NULL, NULL);
         gtk_container_set_border_width(GTK_CONTAINER(gui.tv_layout), 0);

         gui.tv_eventbox = gtk_event_box_new();
         gtk_container_set_border_width(GTK_CONTAINER(gui.tv_eventbox), 4);

         gtk_container_add(GTK_CONTAINER(gui.tv_eventbox), gui.tv_layout);
         gtk_container_add(GTK_CONTAINER(gui.tp_framearh), gui.tv_eventbox);
         gtk_widget_show(gui.tv_eventbox);
         gtk_widget_show(gui.tv_layout);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.tv_eventbox,
			"If the red rectangle is a square, the aspect ratio is correct", "");

         g_signal_connect(G_OBJECT(gui.tv_layout), "expose-event", GTK_SIGNAL_FUNC(tv_arh_exposed), NULL);
      }

      gui.tp_label = xpm_label_box(icon_tvsettings, "TV", NULL, 1, "TV settings", "");
      gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.tp_box, gui.tp_label);

      break;

   case TV_PAGE_TYPE_NA_NOTV:
      gui.tp_frame = gtk_frame_new("TV settings");
      gtk_container_set_border_width(GTK_CONTAINER(gui.tp_frame), 8);
      gtk_widget_show(gui.tp_frame);

      gui.tv_table = gtk_table_new(3, 3, TRUE);
      gtk_table_set_row_spacings(GTK_TABLE(gui.tv_table), 10);
      gtk_table_set_col_spacings(GTK_TABLE(gui.tv_table), 4);

      gtk_container_set_border_width(GTK_CONTAINER(gui.tv_table), 8);

      gtk_container_add(GTK_CONTAINER(gui.tp_frame), gui.tv_table);
      gtk_widget_show(gui.tv_table);

      gui.tv_label_notv = gtk_label_new("No TV encoder present");
      gtk_widget_show(gui.tv_label_notv);
      gtk_table_attach_defaults(GTK_TABLE(gui.tv_table), gui.tv_label_notv, 0, 3, 1, 2);

      gui.tp_label = xpm_label_box(icon_tvsettings, "TV", NULL, 1, "TV settings", "");
      gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.tp_frame, gui.tp_label);
      break;
   }
}

/**************************************************
 *                   Gamma page                   *
 **************************************************/

void
gui_gamma_set_all_spin_values(double s1, double s2, double s3,
			      double s4, double s5, double s6,
			      double s7, double s8, double s9)
{
   if(!gui.ga_spin1) return;
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin1), s1);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin2), s2);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin3), s3);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin4), s4);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin5), s5);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin6), s6);
   if(gl.usenewgamma) {
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin7), s7);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin8), s8);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.ga_spin9), s9);
   }
}

void
gui_gamma_set_gamma_enabled_check_box_stati(int b1, int b2)
{
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.ga_check1), b1 ? TRUE : FALSE);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.ga_check2), b2 ? TRUE : FALSE);
}

void
gui_gamma_show_separate_gamma_menu(void)
{
   gtk_widget_show(gui.ga_eventboxsep);
   gtk_widget_show(gui.ga_label_adjust);
}

void
gui_gamma_hide_separate_gamma_menu(void)
{
   gtk_widget_hide(gui.ga_eventboxsep);
   gtk_widget_hide(gui.ga_label_adjust);
}

int
gui_gamma_get_active_separate_menu_item(void)
{
   return (int)gtk_combo_box_get_active(GTK_COMBO_BOX(gui.ga_comboboxsep));
}

void
gui_gamma_set_active_separate_menu_item(int index)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.ga_comboboxsep), index);
}

void
gui_gamma_drawggfx(void)
{
   int i, j, k;

   for(i = 0; i < GACW*3; i++) gui.rgbbuf[i] = 0x44;
   for(i = 1; i < GACH-1; i++) {
      gui.rgbbuf[i * GACW * 3] = 0x44;
      gui.rgbbuf[(i * GACW * 3) + 1] = 0x44;
      gui.rgbbuf[(i * GACW * 3) + 2] = 0x44;
      for(j = 3; j < (GACW-1) * 3; j++) gui.rgbbuf[(i * GACW * 3) + j] = 0;
      gui.rgbbuf[(i * GACW * 3) + ((GACW - 1) * 3)] = 0xff;
      gui.rgbbuf[(i * GACW * 3) + ((GACW - 1) * 3) + 1] = 0xff;
      gui.rgbbuf[(i * GACW * 3) + ((GACW - 1) * 3) + 2] = 0xff;
   }
   for(i = 0; i < GACW*3; i++) gui.rgbbuf[((GACH-1) * GACW * 3) + i] = 0xff;

   /* curve */
   for(i = 0; i < 3; i++) {
      for(k = 1; k < GACW-1; k++) {
         j = (k - 1) * (gl.nramp / (GACW-2));
         gui.rgbbuf[(((GACH-3) - ((gl.ramp[i][j] >> 8) / (256 / (GACH-3)))) * (GACW * 3)) + (GACW * 3) + (k * 3) + i] = 0xff;
      }
   }
}

/* Update the gamma curve gfx */
void
gui_gamma_update_gamma_gfx(void)
{
   if((gui.ga_drawable) && (gui.notebook_page == SIS_PAGE_COLOR)) {
      gdk_draw_rgb_image(gui.ga_drawable->window, gui.ga_drawable->style->fg_gc[GTK_STATE_NORMAL],
   	                 0, 0, GACW, GACH, GDK_RGB_DITHER_MAX, gui.rgbbuf, GACW * 3);
   }
}

/* Signal handlers */

/* CRT1-gamma toggled */
static void
ga_check1_toggled(GtkWidget *widget, gpointer data)
{
    sisctrl_gamma_enable_1(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.ga_check1)) ? 1 : 0);
}

/* CRT2-gamma toggled */
static void
ga_check2_toggled(GtkWidget *widget, gpointer data)
{
    sisctrl_gamma_enable_2(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.ga_check2)) ? 1 : 0);
}

static void
ga_sepmenu_changed_handler(int index)
{
    switch(index) {
    case 0: /* Both */
       sisctrl_gamma_separate_changed(0);
       break;
    case 1: /* CRT1 */
       sisctrl_gamma_separate_changed(1);
       sisctrl_gamma_separate_toggled(1);
       break;
    case 2: /* CRT2 */
       sisctrl_gamma_separate_changed(1);
       sisctrl_gamma_separate_toggled(0);
       break;
    }
}

static void
ga_comboboxsep_changed(GtkWidget *widget, gpointer data)
{
    ga_sepmenu_changed_handler(gtk_combo_box_get_active(GTK_COMBO_BOX(gui.ga_comboboxsep)));
}

static void
ga_radio_rgb_changed(GtkWidget *widget, gpointer data)
{
   gui.ga_current_adj = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? 0 : 1;

   switch(gui.ga_current_adj) {
   case 0:	/* R/G/B */
      gtk_widget_show(gui.ga_spin2);
      gtk_widget_show(gui.ga_spin3);
      gtk_widget_show(gui.ga_spin5);
      gtk_widget_show(gui.ga_spin6);
      gtk_widget_show(gui.ga_label_red);
      gtk_widget_show(gui.ga_label_green);
      gtk_widget_show(gui.ga_label_blue);
      gtk_widget_show(gui.ga_label_red1);
      gtk_widget_show(gui.ga_label_green1);
      gtk_widget_show(gui.ga_label_blue1);
      if(gl.usenewgamma) {
         gtk_widget_show(gui.ga_spin8);
         gtk_widget_show(gui.ga_spin9);
         gtk_widget_show(gui.ga_label_red2);
         gtk_widget_show(gui.ga_label_green2);
         gtk_widget_show(gui.ga_label_blue2);
      }
      sisctrl_gamma_spin_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin1)));
      sisctrl_gamma_spin_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin2)));
      sisctrl_gamma_spin_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin3)));
      sisctrl_gamma_spin_bri_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin4)));
      sisctrl_gamma_spin_bri_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin5)));
      sisctrl_gamma_spin_bri_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin6)));
      if(gl.usenewgamma) {
         sisctrl_gamma_spin_con_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin7)));
         sisctrl_gamma_spin_con_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin8)));
         sisctrl_gamma_spin_con_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin9)));
      }
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin1,
		 	"Adjust red channel gamma", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin4,
		 	"Adjust red channel gamma brightness", "");
      if(gl.usenewgamma) {
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin7,
		 	"Adjust red channel gamma contrast", "");
      }
      break;

   default:	/* All */
      gtk_widget_hide(gui.ga_spin2);
      gtk_widget_hide(gui.ga_spin3);
      gtk_widget_hide(gui.ga_spin5);
      gtk_widget_hide(gui.ga_spin6);
      gtk_widget_hide(gui.ga_label_red);
      gtk_widget_hide(gui.ga_label_green);
      gtk_widget_hide(gui.ga_label_blue);
      gtk_widget_hide(gui.ga_label_red1);
      gtk_widget_hide(gui.ga_label_green1);
      gtk_widget_hide(gui.ga_label_blue1);
      if(gl.usenewgamma) {
         gtk_widget_hide(gui.ga_spin8);
         gtk_widget_hide(gui.ga_spin9);
         gtk_widget_hide(gui.ga_label_red2);
         gtk_widget_hide(gui.ga_label_green2);
         gtk_widget_hide(gui.ga_label_blue2);
      }
      sisctrl_gamma_spin_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin1)));
      sisctrl_gamma_spin_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin1)));
      sisctrl_gamma_spin_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin1)));
      sisctrl_gamma_spin_bri_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin4)));
      sisctrl_gamma_spin_bri_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin4)));
      sisctrl_gamma_spin_bri_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin4)));
      if(gl.usenewgamma) {
         sisctrl_gamma_spin_con_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin7)));
         sisctrl_gamma_spin_con_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin7)));
         sisctrl_gamma_spin_con_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin7)));
      }
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin1,
		 	"Adjust gamma", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin4,
		 	"Adjust gamma brightness", "");
      if(gl.usenewgamma) {
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin7,
		 	"Adjust gamma contrast", "");
      }
      break;
   }

}

static void
ga_spin1_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) {
      sisctrl_gamma_spin_all_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin1)));
   } else {
      sisctrl_gamma_spin_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin1)));
   }
}

static void
ga_spin2_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) return;
   sisctrl_gamma_spin_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin2)));
}

static void
ga_spin3_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) return;
   sisctrl_gamma_spin_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin3)));
}

static void
ga_spin4_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) {
      sisctrl_gamma_spin_bri_all_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin4)));
   } else {
      sisctrl_gamma_spin_bri_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin4)));
   }
}

static void
ga_spin5_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) return;
   sisctrl_gamma_spin_bri_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin5)));
}

static void
ga_spin6_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) return;
   sisctrl_gamma_spin_bri_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin6)));
}

static void
ga_spin7_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) {
      sisctrl_gamma_spin_con_all_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin7)));
   } else {
      sisctrl_gamma_spin_con_red_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin7)));
   }
}

static void
ga_spin8_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) return;
   sisctrl_gamma_spin_con_green_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin8)));
}

static void
ga_spin9_changed(GtkWidget *widget, gpointer data)
{
   if(gui.ga_current_adj) return;
   sisctrl_gamma_spin_con_blue_changed((double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin8)));
}

static void
ga_sat1_changed(GtkWidget *widget, gpointer data)
{
   sisctrl_gamma_set_sat_crt1((int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin_sat1)));
}

static void
ga_sat2_changed(GtkWidget *widget, gpointer data)
{
   sisctrl_gamma_set_sat_crt2((int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_spin_sat2)));
}

static gboolean
ga_picture_exposed(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   gdk_draw_rgb_image(widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
   	0, 0, GACW, GACH, GDK_RGB_DITHER_MAX, gui.rgbbuf, GACW * 3);
   return TRUE;
}

static void
ga_draw_big_boxes(int stepw, int cello, int cellh, int cellvo)
{
	 gdk_gc_set_rgb_bg_color(gui.ga_context, &ga_colorblack);
	 gdk_gc_set_stipple(gui.ga_context, gui.ga_stipple2);
	 gdk_gc_set_ts_origin(gui.ga_context, 0, 0);
	 gdk_gc_set_fill(gui.ga_context, GDK_OPAQUE_STIPPLED);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorred);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 0) + cello, cellvo,
		      stepw * 2,           cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorgreen);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 3) + cello, cellvo,
		      stepw  * 2,          cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorblue);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 6) + cello, cellvo,
		      stepw * 2,           cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorwhite);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 9) + cello, cellvo,
		      stepw * 2,           cellh);
}

static void
ga_draw_small_boxes(int stepw, int cello, int cellh, int cellvo)
{
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorredh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 0) + cello + (stepw / 2), cellvo + (cellh / 4),
		      stepw,                             cellh / 2 - 1);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorgreenh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 3) + cello + (stepw / 2), cellvo + (cellh / 4),
		      stepw,                             cellh / 2 - 1);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorblueh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 6) + cello + (stepw / 2), cellvo + (cellh / 4),
		      stepw,                             cellh / 2 - 1);


	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorwhiteh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 9) + cello + (stepw / 2), cellvo + (cellh / 4),
		      stepw,                             cellh / 2 - 1);
}

static void
ga_update_test_pattern(void)
{
   gint lwidth;
   gint lheight;
   int  i, cellw, stepw, cello, cellh, steph, cellvo;
   float ga;

   if(!gui.ga_layout) return;

   if(!GTK_WIDGET_REALIZED(gui.ga_layout)) return;

   if(!(GTK_LAYOUT(gui.ga_layout)->bin_window)) return;

   lwidth = gui.ga_layout->allocation.width - 4;
   lheight = gui.ga_layout->allocation.height;
   if(lwidth <= 0 || !lheight) return;

   if(!gui.ga_context) {
      gui.ga_context = gdk_gc_new(GTK_LAYOUT(gui.ga_layout)->bin_window);
   }

   gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_bgcolors[gui.ga_current_gfx]);
   gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
   gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context,
		      TRUE,
		      4, 0,
		      lwidth, lheight);

   switch(gui.ga_current_gfx) {
      case 0:
      case 1:
         stepw = (((lwidth * 95) / 100) / 11);
	 cello = (lwidth - (stepw * 11)) / 2 + 4;
         cellh = ((lheight * 95) / 100);
	 cellvo = (lheight - cellh) / 2;

	 ga_draw_big_boxes(stepw, cello, cellh, cellvo);

	 ga = (gui.ga_current_gfx == 0) ? gl.crt1mongamma : gl.crt2mongamma;
	 ga_colorredh.red =
	    ga_colorgreenh.green =
	    ga_colorblueh.blue =
	    ga_colorwhiteh.red =
	    ga_colorwhiteh.green =
	    ga_colorwhiteh.blue = (guint16)((float)(pow(0.5, 1.0 / ga) * 65536.0));

	 ga_draw_small_boxes(stepw, cello, cellh, cellvo);

	 break;

      case 2:
      case 3:
      case 4:
         stepw = (((lwidth * 95) / 100) / 11);
	 cello = (lwidth - (stepw * 11)) / 2 + 4;
         cellh = ((lheight * 95) / 100);
	 cellvo = (lheight - cellh) / 2;

	 ga_draw_big_boxes(stepw, cello, cellh, cellvo);

	 switch(gui.ga_current_gfx) {
	 case 2:  ga = 2.2; break;
	 case 3:  ga = 1.8; break;
	 default: ga = (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_sping));
	 }
	 ga_colorredh.red =
	    ga_colorgreenh.green =
	    ga_colorblueh.blue =
	    ga_colorwhiteh.red =
	    ga_colorwhiteh.green =
	    ga_colorwhiteh.blue = (guint16)((float)(pow(0.5, 1.0 / ga) * 65536.0));

	 ga_draw_small_boxes(stepw, cello, cellh, cellvo);

	 break;

      case 5:
      case 6:
      case 7:
         stepw = (((lwidth * 95) / 100) / 11);
	 cello = (lwidth - (stepw * 11)) / 2 + 4;
         cellh = ((lheight * 95) / 100);
	 cellvo = (lheight - cellh) / 2;

	 gdk_gc_set_rgb_bg_color(gui.ga_context, &ga_colorblack);
	 gdk_gc_set_stipple(gui.ga_context, gui.ga_stipple2);
	 gdk_gc_set_ts_origin(gui.ga_context, 0, 0);
	 gdk_gc_set_fill(gui.ga_context, GDK_OPAQUE_STIPPLED);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorred);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 1) + cello, cellvo,
		      stepw,               cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorgreen);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 4) + cello, cellvo,
		      stepw,               cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorblue);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 7) + cello, cellvo,
		      stepw,               cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorwhite);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 10) + cello, cellvo,
		      stepw,               cellh);

	 switch(gui.ga_current_gfx) {
	 case 5:  ga = 2.2; break;
	 case 6:  ga = 1.8; break;
	 default: ga = (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.ga_sping));
	 }

	 ga_colorredh.red =
	    ga_colorgreenh.green =
	    ga_colorblueh.blue =
	    ga_colorwhiteh.red =
	    ga_colorwhiteh.green =
	    ga_colorwhiteh.blue = (guint16)((float)(pow(0.5, 1.0 / ga) * 65536.0));

	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorredh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 0) + cello, cellvo,
		      stepw,               cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorgreenh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 3) + cello, cellvo,
		      stepw,               cellh);

	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorblueh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 6) + cello, cellvo,
		      stepw,               cellh);


	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorwhiteh);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * 9) + cello, cellvo,
		      stepw,               cellh);

	 break;

      case 8: /* Brightness */
         cellw = (lwidth * 66) / 100;
	 cello = (lwidth - cellw) / 2 + 4;
	 cellh = (lheight * 85) / 100;
	 cellvo = (lheight - cellh) / 2;

	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colorblack);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      cello, cellvo,
		      cellw, cellh);

	 cellw = (lwidth * 33) / 100;
	 cello = (lwidth - cellw) / 2 + 4;
	 cellh = (lheight * 65) / 100;
	 cellvo = (lheight - cellh) / 2;
	 gdk_gc_set_rgb_fg_color(gui.ga_context, &ga_colordgrey);
	 gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      cello, cellvo,
		      cellw, cellh);
         break;

      case 9: /* RGB */
         stepw = (((lwidth * 90) / 100) / 16);
	 cello = (lwidth - (stepw * 16)) / 2 + 4;
         cellh = ((lheight * 70) / 100) / 3;
	 cellvo = (lheight - (cellh * 3)) / 4;
	 steph = cellh + cellvo;
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
         for(i = 0; i < 16; i++) {
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorrgb[i]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 0) + cellvo,
		      stepw,               cellh);
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorrgb[i+16]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 1) + cellvo,
		      stepw,               cellh);
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorrgb[i+(16*2)]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 2) + cellvo,
		      stepw,               cellh);
         }
         break;

      case 10: /* Grey */
         stepw = (((lwidth * 90) / 100) / 10);
	 cello = (lwidth - (stepw * 10)) / 2 + 4;
         cellh = ((lheight * 80) / 100) / 2;
	 cellvo = (lheight - (cellh * 2)) / 3;
	 steph = cellh + cellvo;
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
         for(i = 0; i < 10; i++) {
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorgrey[i]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 0) + cellvo,
		      stepw,               cellh);
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorgrey[i+10]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 1) + cellvo,
		      stepw,               cellh);
         }
         break;

      case 11: /* CMY */
         stepw = (((lwidth * 90) / 100) / 11);
	 cello = (lwidth - (stepw * 11)) / 2 + 4;
         cellh = ((lheight * 80) / 100) / 3;
	 cellvo = (lheight - (cellh * 3)) / 4;
	 steph = cellh + cellvo;
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
         for(i = 0; i < 11; i++) {
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorcmy[i]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 0) + cellvo,
		      stepw,               cellh);
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorcmy[i+11]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 1) + cellvo,
		      stepw,               cellh);
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorcmy[i+(11*2)]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, (steph * 2) + cellvo,
		      stepw,               cellh);
         }
         break;

      case 12: /* Dark */
         cellw = (((lwidth * 95) / 100) / 19);
         stepw = cellw * 2;
	 cello = (lwidth - (cellw * 19)) / 2 + 4;
         cellh = ((lheight * 80) / 100);
	 cellvo = (lheight - cellh) / 2;
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
         for(i = 0; i < 10; i++) {
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colordgrey[i]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, cellvo,
		      cellw,               cellh);
         }
         break;

      case 13: /* Med */
         cellw = (((lwidth * 95) / 100) / 21);
         stepw = cellw * 2;
	 cello = (lwidth - (cellw * 21)) / 2 + 4;
         cellh = ((lheight * 80) / 100);
	 cellvo = (lheight - cellh) / 2;
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
         for(i = 0; i < 11; i++) {
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colormgrey[i]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, cellvo,
		      cellw,               cellh);
         }
         break;

      case 14: /* Light */
         cellw = (((lwidth * 95) / 100) / 19);
         stepw = cellw * 2;
	 cello = (lwidth - (cellw * 19)) / 2 + 4;
         cellh = ((lheight * 80) / 100);
	 cellvo = (lheight - cellh) / 2;
	 gdk_gc_set_fill(gui.ga_context, GDK_SOLID);
         for(i = 0; i < 10; i++) {
	    gdk_gc_set_rgb_fg_color(gui.ga_context, &gui.ga_colorlgrey[i]);
            gdk_draw_rectangle(GTK_LAYOUT(gui.ga_layout)->bin_window,
		      gui.ga_context, TRUE,
		      (stepw * i) + cello, cellvo,
		      cellw,               cellh);
         }

         break;
   }
}

static gboolean
ga_scale_exposed(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   ga_update_test_pattern();
   return FALSE;
}

static void
ga_menu_changed(GtkWidget *widget, gpointer data)
{
   gui.ga_current_gfx = gui.ga_menuindices[gtk_combo_box_get_active(GTK_COMBO_BOX(gui.ga_combobox))];

   ga_update_test_pattern();

   switch(gui.ga_current_gfx) {

   case 0:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
		 	"For gamma correction: Currently, the reference gamma is set "
			"to your current CRT1 device's level as read from DDC. "
			"Reset contrast and brightness to 0.0 and adjust the "
			"gamma values so that when you defocus your eyes, "
			"the boxes in the middle vanish.", "");
      break;
   case 1:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
		 	"For gamma correction: Currently, the reference gamma is set "
			"to your current CRT2 device's level as read from DDC. Reset "
			"contrast and brightness to 0.0 and adjust the "
			"gamma values so that when you defocus your eyes, "
			"the boxes in the middle vanish.", "");
      break;
   case 2:
   case 5:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
			"For gamma correction: Currently, the reference gamma is set "
			"to 2.2. This is the standard for most devices and the basis "
			"of the Internet sRGB colorspace. Reset contrast and brightness "
			"to 0.0 and adjust the gamma values so that when you defocus "
			"your eyes, the boxes in the middle vanish, or the bars have the "
			"same color.", "");
      break;
   case 4:
   case 7:
      gtk_widget_show(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
			"For gamma correction: Currently, the reference gamma is set "
			"to the value to the left. Reset contrast and brightness to 0.0 "
			"and adjust the gamma values so that when you defocus your eyes, "
			"the boxes in the middle vanish, or the bars have the same "
			"color.", "");
      break;
   case 3:
   case 6:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
		 	"For gamma correction: Currently, the reference gamma is set "
			"to 1.8. Reset contrast and brightness to 0.0 and adjust the "
			"gamma correction so that when you defocus your eyes, "
			"the boxes in the middle vanish, or the bars have the same "
			"color.", "");
      break;
   case 8:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
		 	"For monitor brightness: Set your monitor's contrast control "
			"to its maximum, and then adjust the monitor's brightness "
			"so that you still see the grey box in the middle while the "
			"white is still bright white. Reset the brightness and contrast "
			"setting above to 0.0 before doing this adjustment.", "");
      break;
   case 9:
   case 10:
   case 11:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
      			"Color gradients for gamma/brightness/contrast correction", "");
      break;
   case 12:
   case 13:
   case 14:
      gtk_widget_hide(gui.ga_sping);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventbox,
      			"For monitor brightness adjustment. Reset brightness and "
      			"contrast both to 0.0 before doing adjustments on the "
      			"monitor.", "");
      break;
   }
}

static void
ga_sping_changed(GtkWidget *widget, gpointer data)
{
   ga_update_test_pattern();
}

void
gui_gamma_update_gamma_pattern_menu(void)
{
   char buffer[16];
   gint backup;
   int i, j;

   backup = gui.ga_menuindices[gtk_combo_box_get_active(GTK_COMBO_BOX(gui.ga_combobox))];

   sisctrl_gamma_get_monitor_gamma(0);
   sisctrl_gamma_get_monitor_gamma(1);

   if(gui.ga_menuindices[0] == GAMMA_GFX_CRT1 || gui.ga_menuindices[0] == GAMMA_GFX_CRT2)
      gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.ga_combobox), 0);
   if(gui.ga_menuindices[1] == GAMMA_GFX_CRT2)
      gtk_combo_box_remove_text(GTK_COMBO_BOX(gui.ga_combobox), 0);

   for(i = 0, j = GAMMA_GFX_BOX22; i < GAMMA_GFX_NUM; i++, j++) {
      gui.ga_menuindices[i] = j;
      gui.ga_rmenuindices[GAMMA_GFX_BOX22 + i] = i;
   }

   if(gl.crt2mongamma != 0.0) {
      sprintf(buffer, "CRT2 %.2f", gl.crt2mongamma);
      gtk_combo_box_prepend_text(GTK_COMBO_BOX(gui.ga_combobox), buffer);
      for(i = GAMMA_GFX_NUM - 1; i > 0; i--) {
         gui.ga_menuindices[i] = gui.ga_menuindices[i-1];
      }
      gui.ga_menuindices[0] = GAMMA_GFX_CRT2;
      for(i = GAMMA_GFX_BOX22; i < GAMMA_GFX_NUM; i++) {
         gui.ga_rmenuindices[i]++;
      }
      gui.ga_rmenuindices[GAMMA_GFX_CRT2] = 0;
   }

   if(gl.crt1mongamma != 0.0) {
      sprintf(buffer, "CRT1 %.2f", gl.crt1mongamma);
      gtk_combo_box_prepend_text(GTK_COMBO_BOX(gui.ga_combobox), buffer);
      for(i = GAMMA_GFX_NUM - 1; i > 0; i--) {
         gui.ga_menuindices[i] = gui.ga_menuindices[i-1];
      }
      gui.ga_menuindices[0] = GAMMA_GFX_CRT1;
      for(i = GAMMA_GFX_BOX22; i < GAMMA_GFX_NUM; i++) {
         gui.ga_rmenuindices[i]++;
      }
      gui.ga_rmenuindices[GAMMA_GFX_CRT1] = 0;
      gui.ga_rmenuindices[GAMMA_GFX_CRT2]++;
   }

   if( ((backup == GAMMA_GFX_CRT1) && (gl.crt1mongamma != 0.0)) ||
       ((backup == GAMMA_GFX_CRT2) && (gl.crt2mongamma != 0.0)) ||
       (backup >= GAMMA_GFX_BOX22) ) {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.ga_combobox), gui.ga_rmenuindices[backup]);
   } else {
      gtk_combo_box_set_active(GTK_COMBO_BOX(gui.ga_combobox), 0);
   }
}

/* Build page */

static void
build_gamma_page(void)
{
   int temp, numposoffs = 0, satpos = 2, i, j;
   GtkWidget *eventbox;

   gui.ga_frame = gtk_frame_new("Color settings");
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_frame), 8);
   gtk_widget_show(gui.ga_frame);

   gui.ga_table = gtk_table_new(7, 5, TRUE);
   gtk_table_set_row_spacings(GTK_TABLE(gui.ga_table), 0);
   gtk_table_set_col_spacings(GTK_TABLE(gui.ga_table), 2);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_table), 10);

   gtk_container_add(GTK_CONTAINER(gui.ga_frame), gui.ga_table);
   gtk_widget_show(gui.ga_table);

   /* Subtable 0: Top check buttons: cols 1-3, rows 1-2 */

   gui.ga_subtable = gtk_table_new(1 /* 2 */, 5 /* 4 */, TRUE /*FALSE*/);
   gtk_table_set_row_spacings(GTK_TABLE(gui.ga_subtable), 0);
   gtk_table_set_col_spacings(GTK_TABLE(gui.ga_subtable), 5);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_subtable), 0);

   switch(gl.gamma_page_type) {

   case GAMMA_PAGE_TYPE_XINERAMA:
      {
         char buffer[64];

	 sprintf(buffer, "Setting for Screen %d", gl.screennum);
         gui.ga_label_enable = gtk_label_new(buffer);

         gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable), gui.ga_label_enable, 0, 5, 0, 1);
         gtk_widget_show(gui.ga_label_enable);
      }
      break;

   default:
      gui.ga_label_enable = gtk_label_new("Enable for");
      gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable), gui.ga_label_enable, 0, 1, 0, 1);
      gtk_widget_show(gui.ga_label_enable);

      gui.ga_check1 = gtk_check_button_new_with_label("CRT1");
      gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable), gui.ga_check1, 1, 2, 0, 1);
      gtk_widget_show(gui.ga_check1);

      gui.ga_check2 = gtk_check_button_new_with_label("CRT2");
      gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable), gui.ga_check2, 2, 3, 0, 1);
      gtk_widget_show(gui.ga_check2);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_check1,
       		 	"Enable color adjustments for first output (CRT1)", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_check2,
       		 	"Enable color adjustments for second output (CRT2)", "");

      if(gl.havesepgamma2) {

         gui.ga_label_adjust = gtk_label_new("Adjust for");
         gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable), gui.ga_label_adjust, 3, 4, 0, 1);
         gtk_widget_show(gui.ga_label_adjust);

	 gui.ga_comboboxsep = gtk_combo_box_new_text();
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_comboboxsep), "Both");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_comboboxsep), "CRT1");
	 gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_comboboxsep), "CRT2");
	 gui_gamma_set_active_separate_menu_item(0);

	 gtk_widget_show(gui.ga_comboboxsep);

	 gui.ga_eventboxsep = gtk_event_box_new();
	 gtk_container_set_border_width(GTK_CONTAINER(gui.ga_eventboxsep), 0);
	 gtk_container_add(GTK_CONTAINER(gui.ga_eventboxsep), gui.ga_comboboxsep);

	 gtk_table_attach(GTK_TABLE(gui.ga_subtable), gui.ga_eventboxsep, 4, 5, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_eventboxsep,
		 	"Select whether the adjustments below apply to first (CRT1) or second (CRT2) output, "
		 	"or both outputs simultaniously", "");
      }

      sisctrl_gamma_set_status();

      if(gl.gammaenablecrt1) {
         gtk_widget_set_sensitive(gui.ga_check1, TRUE);
	 g_signal_connect(G_OBJECT(gui.ga_check1), "toggled", G_CALLBACK (ga_check1_toggled), NULL);
      } else {
         gtk_widget_set_sensitive(gui.ga_check1, FALSE);
      }

      if(gl.gammaenablecrt2) {
         gtk_widget_set_sensitive(gui.ga_check2, TRUE);
	 g_signal_connect(G_OBJECT(gui.ga_check2), "toggled", G_CALLBACK (ga_check2_toggled), NULL);
      } else {
         gtk_widget_set_sensitive(gui.ga_check2, FALSE);
      }

      if(gl.havesepgamma2) {
	 g_signal_connect(G_OBJECT(gui.ga_comboboxsep), "changed", G_CALLBACK(ga_comboboxsep_changed), NULL);
      }

      if(gl.supportsatcrt1 || gl.supportsatcrt2) {
         if(gl.gammaenablecrt1 || gl.gammaenablecrt2) {
            numposoffs = 1;
         }
      }

      break;
   }

   gtk_table_attach_defaults(GTK_TABLE(gui.ga_table), gui.ga_subtable, 0, 4, 0, (numposoffs == 0) ? 2 : 1);
   gtk_widget_show(gui.ga_subtable);

   /* Sub table 1: All / R/G/B: col 5, rows 4-5*/

   gui.ga_subtable1 = gtk_table_new(2, 1, TRUE);
   gtk_table_set_row_spacings(GTK_TABLE(gui.ga_subtable1), 0);
   gtk_table_set_col_spacings(GTK_TABLE(gui.ga_subtable1), 0);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_subtable1), 5);

   gui.ga_radio_rgb = gtk_radio_button_new_with_label(NULL, "R/G/B");
   gui.ga_group_rgb = gtk_radio_button_get_group(GTK_RADIO_BUTTON(gui.ga_radio_rgb));
   gui.ga_radio_all = gtk_radio_button_new_with_label(gui.ga_group_rgb, "All");

   gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable1), gui.ga_radio_rgb, 0, 1, 0, 1);
   gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable1), gui.ga_radio_all, 0, 1, 1, 2);

   gtk_widget_show(gui.ga_radio_rgb);
   gtk_widget_show(gui.ga_radio_all);

   gui.ga_current_adj = 0;
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.ga_radio_rgb), TRUE);

   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_radio_rgb,
	 	"Adjust color settings for red, green and blue channels", "");
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_radio_all,
	 	"Adjust color settings for all color channels simultaniously", "");

   g_signal_connect(G_OBJECT(gui.ga_radio_rgb), "toggled", G_CALLBACK(ga_radio_rgb_changed), NULL);

   gtk_table_attach_defaults(GTK_TABLE(gui.ga_table), gui.ga_subtable1, 4, 5, 3, 5);
   gtk_widget_show(gui.ga_subtable1);

   /* Subtable 2: Number input: cols 1-4, rows 3-5 */

   temp = 2;
   if(gl.usenewgamma) { temp++; satpos++; }
   if(numposoffs)     temp++;
   gui.ga_subtable2 = gtk_table_new(temp, 7, FALSE);
   gtk_table_set_row_spacings(GTK_TABLE(gui.ga_subtable2), 2);
   gtk_table_set_col_spacings(GTK_TABLE(gui.ga_subtable2), 0);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_subtable2), 0);

   gui.ga_label_gamma = gtk_label_new("Gamma");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_gamma, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_gamma);
   gui.ga_label_red = gtk_label_new("R");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_red, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_red);
   gui.ga_label_green = gtk_label_new("G");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_green, 3, 4, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_green);
   gui.ga_label_blue = gtk_label_new("B");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_blue, 5, 6, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_blue);

   gui.ga_label_bri = gtk_label_new("Brightness");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_bri, 0, 1, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_bri);
   gui.ga_label_red1 = gtk_label_new("R");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_red1, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_red1);
   gui.ga_label_green1 = gtk_label_new("G");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_green1, 3, 4, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_green1);
   gui.ga_label_blue1 = gtk_label_new("B");
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_blue1, 5, 6, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_label_blue1);

   if(gl.usenewgamma) {
      gui.ga_label_con = gtk_label_new("Contrast");
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_con, 0, 1, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_widget_show(gui.ga_label_con);
      gui.ga_label_red2 = gtk_label_new("R");
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_red2, 1, 2, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_widget_show(gui.ga_label_red2);
      gui.ga_label_green2 = gtk_label_new("G");
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_green2, 3, 4, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_widget_show(gui.ga_label_green2);
      gui.ga_label_blue2 = gtk_label_new("B");
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_blue2, 5, 6, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_widget_show(gui.ga_label_blue2);
   }

   gui.ga_spin1_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.1, 0.1, 0.0);
   gui.ga_spin2_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.1, 0.1, 0.0);
   gui.ga_spin3_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.1, 0.1, 0.0);

   if(gl.usenewgamma) {
      gui.ga_spin4_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -1.0, 1.0, 0.01, 0.01, 0.0);
      gui.ga_spin5_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -1.0, 1.0, 0.01, 0.01, 0.0);
      gui.ga_spin6_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -1.0, 1.0, 0.01, 0.01, 0.0);
      gui.ga_spin7_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -1.0, 1.0, 0.01, 0.01, 0.0);
      gui.ga_spin8_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -1.0, 1.0, 0.01, 0.01, 0.0);
      gui.ga_spin9_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, -1.0, 1.0, 0.01, 0.01, 0.0);
   } else {
      gui.ga_spin4_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.01, 0.1, 0.0);
      gui.ga_spin5_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.01, 0.1, 0.0);
      gui.ga_spin6_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.01, 0.1, 0.0);
   }

   gui.ga_spin1 = gtk_spin_button_new(gui.ga_spin1_adj, 0.1, 2);
   gui.ga_spin2 = gtk_spin_button_new(gui.ga_spin2_adj, 0.1, 2);
   gui.ga_spin3 = gtk_spin_button_new(gui.ga_spin3_adj, 0.1, 2);
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin1,
		 	"Adjust red channel gamma", "");
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin2,
		 	"Adjust green channel gamma", "");
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin3,
		 	"Adjust blue channel gamma", "");

   gui.ga_spin4 = gtk_spin_button_new(gui.ga_spin4_adj, 0.1, 2);
   gui.ga_spin5 = gtk_spin_button_new(gui.ga_spin5_adj, 0.1, 2);
   gui.ga_spin6 = gtk_spin_button_new(gui.ga_spin6_adj, 0.1, 2);
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin4,
		 	"Adjust red channel brightness", "");
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin5,
		 	"Adjust green channel brightness", "");
   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin6,
		 	"Adjust blue channel brightness", "");

   if(gl.usenewgamma) {
      gui.ga_spin7 = gtk_spin_button_new(gui.ga_spin7_adj, 0.1, 2);
      gui.ga_spin8 = gtk_spin_button_new(gui.ga_spin8_adj, 0.1, 2);
      gui.ga_spin9 = gtk_spin_button_new(gui.ga_spin9_adj, 0.1, 2);
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin7,
		 	"Adjust red channel contrast", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin8,
		 	"Adjust green channel contrast", "");
      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin9,
		 	"Adjust blue channel contrast", "");
   }

   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin1, 2, 3, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin2, 4, 5, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin3, 6, 7, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin4, 2, 3, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin5, 4, 5, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin6, 6, 7, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   if(gl.usenewgamma) {
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin7, 2, 3, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin8, 4, 5, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_spin9, 6, 7, 2, 3, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   }

   sisctrl_gamma_set_all_spinvalues();

   gtk_widget_show(gui.ga_spin1);
   gtk_widget_show(gui.ga_spin2);
   gtk_widget_show(gui.ga_spin3);
   gtk_widget_show(gui.ga_spin4);
   gtk_widget_show(gui.ga_spin5);
   gtk_widget_show(gui.ga_spin6);
   if(gl.usenewgamma) {
      gtk_widget_show(gui.ga_spin7);
      gtk_widget_show(gui.ga_spin8);
      gtk_widget_show(gui.ga_spin9);
   }

   g_signal_connect(G_OBJECT(gui.ga_spin1_adj), "value_changed", G_CALLBACK(ga_spin1_changed), NULL);
   g_signal_connect(G_OBJECT(gui.ga_spin2_adj), "value_changed", G_CALLBACK(ga_spin2_changed), NULL);
   g_signal_connect(G_OBJECT(gui.ga_spin3_adj), "value_changed", G_CALLBACK(ga_spin3_changed), NULL);
   g_signal_connect(G_OBJECT(gui.ga_spin4_adj), "value_changed", G_CALLBACK(ga_spin4_changed), NULL);
   g_signal_connect(G_OBJECT(gui.ga_spin5_adj), "value_changed", G_CALLBACK(ga_spin5_changed), NULL);
   g_signal_connect(G_OBJECT(gui.ga_spin6_adj), "value_changed", G_CALLBACK(ga_spin6_changed), NULL);
   if(gl.usenewgamma) {
      g_signal_connect(G_OBJECT(gui.ga_spin7_adj), "value_changed", G_CALLBACK(ga_spin7_changed), NULL);
      g_signal_connect(G_OBJECT(gui.ga_spin8_adj), "value_changed", G_CALLBACK(ga_spin8_changed), NULL);
      g_signal_connect(G_OBJECT(gui.ga_spin9_adj), "value_changed", G_CALLBACK(ga_spin9_changed), NULL);
   }

   if(numposoffs) {  /* = we have saturation support */
      gui.ga_label_sat = gtk_label_new("Saturation");
      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_label_sat, 0, 1, satpos, satpos+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_widget_show(gui.ga_label_sat);

      gui.ga_subsubtable2 = gtk_table_new(1, 4, TRUE);
      gtk_table_set_row_spacings(GTK_TABLE(gui.ga_subsubtable2), 2);
      gtk_table_set_col_spacings(GTK_TABLE(gui.ga_subsubtable2), 0);
      gtk_container_set_border_width(GTK_CONTAINER(gui.ga_subsubtable2), 0);

      if(gl.supportsatcrt1) {
         gui.ga_label_sat1 = gtk_label_new("CRT1");
         gtk_table_attach(GTK_TABLE(gui.ga_subsubtable2), gui.ga_label_sat1, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
         gtk_widget_show(gui.ga_label_sat1);

         gui.ga_sat1_adj = (GtkAdjustment *)gtk_adjustment_new((gdouble)(sisctrl_gamma_get_sat_crt1()), 0.0, 7.0, 1.0, 1.0, 0.0);
         gui.ga_spin_sat1 = gtk_spin_button_new(gui.ga_sat1_adj, 1.0, 0);
         gtk_table_attach(GTK_TABLE(gui.ga_subsubtable2), gui.ga_spin_sat1, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
         gtk_widget_show(gui.ga_spin_sat1);

         g_signal_connect(G_OBJECT(gui.ga_sat1_adj), "value_changed", G_CALLBACK(ga_sat1_changed), NULL);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin_sat1,
		 	"Adjust CRT1 color saturation", "");
      }

      if(gl.supportsatcrt2) {
         gui.ga_label_sat2 = gtk_label_new("CRT2");
         gtk_table_attach(GTK_TABLE(gui.ga_subsubtable2), gui.ga_label_sat2, 2, 3, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
         gtk_widget_show(gui.ga_label_sat2);

         gui.ga_sat2_adj = (GtkAdjustment *)gtk_adjustment_new((gdouble)(sisctrl_gamma_get_sat_crt2()), 0.0, 7.0, 1.0, 1.0, 0.0);
         gui.ga_spin_sat2 = gtk_spin_button_new(gui.ga_sat2_adj, 1.0, 0);
         gtk_table_attach(GTK_TABLE(gui.ga_subsubtable2), gui.ga_spin_sat2, 3, 4, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
         gtk_widget_show(gui.ga_spin_sat2);

         g_signal_connect(G_OBJECT(gui.ga_sat2_adj), "value_changed", G_CALLBACK(ga_sat2_changed), NULL);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.ga_spin_sat2,
		 	"Adjust CRT2 color saturation", "");
      }

      gtk_table_attach(GTK_TABLE(gui.ga_subtable2), gui.ga_subsubtable2, 1, 7, satpos, satpos+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
      gtk_widget_show(gui.ga_subsubtable2);
   }

   gtk_table_attach(GTK_TABLE(gui.ga_table), gui.ga_subtable2, 0, 4, 2 - numposoffs, 5, GTK_FILL|GTK_EXPAND, 0, 0, 0);
   gtk_widget_show(gui.ga_subtable2);

   /* Gamma curve: In main table col 5, rows 1-5 (although 5 overlaps; otherwise window to tall) */

   gui.ga_drawable = gtk_drawing_area_new();
   gtk_widget_set_size_request(gui.ga_drawable, GACW, GACH);
   gui.ga_vbox = gtk_vbox_new(FALSE, 0);
   gui.ga_hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(gui.ga_vbox), gui.ga_hbox, FALSE, FALSE, 0);
   gtk_box_pack_end(GTK_BOX(gui.ga_hbox), gui.ga_drawable, FALSE, FALSE, 0);
   gtk_table_attach_defaults(GTK_TABLE(gui.ga_table), gui.ga_vbox, 4, 5, 0, 5); /* Why 5?! */
   gtk_widget_show(gui.ga_drawable);
   gtk_widget_show(gui.ga_hbox);
   gtk_widget_show(gui.ga_vbox);
   g_signal_connect(G_OBJECT(gui.ga_drawable), "expose-event", GTK_SIGNAL_FUNC(ga_picture_exposed), NULL);

   /* Subtable 3: Test patterns: cols 1-5, rows 6-7 */

   gui.ga_subtable3 = gtk_table_new(2, 16, TRUE);
   gtk_table_set_row_spacings(GTK_TABLE(gui.ga_subtable3), 0);
   gtk_table_set_col_spacings(GTK_TABLE(gui.ga_subtable3), 0);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_subtable3), 0);

   /* RGB */
   for(i = 0, j = 0; i < 16; i++, j+= (65536 / 16)) {
      gui.ga_colorrgb[0*16 + i].red = j;
      gui.ga_colorrgb[0*16 + i].green = gui.ga_colorrgb[0*16 + i].blue = 0;
      gui.ga_colorrgb[1*16 + i].green = j;
      gui.ga_colorrgb[1*16 + i].red = gui.ga_colorrgb[1*16 + i].blue = 0;
      gui.ga_colorrgb[2*16 + i].blue = j;
      gui.ga_colorrgb[2*16 + i].red = gui.ga_colorrgb[2*16 + i].green = 0;
   }

   /* CMY */
   for(i = 0; i < 11; i++) {
      gui.ga_colorcmy[i].red   = cmyc[(i*3) + 0] << 8;
      gui.ga_colorcmy[i].green = cmyc[(i*3) + 1] << 8;
      gui.ga_colorcmy[i].blue  = cmyc[(i*3) + 2] << 8;
      gui.ga_colorcmy[i + 11].red   = cmym[(i*3) + 0] << 8;
      gui.ga_colorcmy[i + 11].green = cmym[(i*3) + 1] << 8;
      gui.ga_colorcmy[i + 11].blue  = cmym[(i*3) + 2] << 8;
      gui.ga_colorcmy[i + (2*11)].red   = cmyy[(i*3) + 0] << 8;
      gui.ga_colorcmy[i + (2*11)].green = cmyy[(i*3) + 1] << 8;
      gui.ga_colorcmy[i + (2*11)].blue  = cmyy[(i*3) + 2] << 8;
   }

   /* Grey */
   for(i = 0; i < 20; i++) {
      gui.ga_colorgrey[i].red =
        gui.ga_colorgrey[i].green =
	  gui.ga_colorgrey[i].blue = (int)((float)((float)(256.0 / 20.0) * (float)(i))) << 8;
   }

   /* DGrey, LGrey */
   for(i = 0; i < 10; i++) {
      gui.ga_colordgrey[i].red =
        gui.ga_colordgrey[i].green =
	  gui.ga_colordgrey[i].blue = dgrey[i] << 8;
      gui.ga_colorlgrey[i].red =
        gui.ga_colorlgrey[i].green =
	  gui.ga_colorlgrey[i].blue = lgrey[i] << 8;
   }

   /* MGrey */
   for(i = 0; i < 11; i++) {
      gui.ga_colormgrey[i].red =
        gui.ga_colormgrey[i].green =
	  gui.ga_colormgrey[i].blue = mgrey[i] << 8;
   }

   {
      GdkPixmap *pseudo;
      pseudo = gdk_pixmap_create_from_xpm_d(gui.window->window, &gui.ga_stipple, NULL, gammastipple);
      g_object_unref(pseudo);
      pseudo = gdk_pixmap_create_from_xpm_d(gui.window->window, &gui.ga_stipple2, NULL, gammastipple2);
      g_object_unref(pseudo);
   }

   gui.ga_current_gfx = 0;

   gui.ga_layout = gtk_layout_new(NULL, NULL);
   gtk_widget_set_size_request(gui.ga_layout, 0, 48);
   gui.ga_eventbox = gtk_event_box_new();
   gtk_widget_set_size_request(gui.ga_eventbox, 0, 48);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ga_eventbox), 0);
   gtk_container_add(GTK_CONTAINER(gui.ga_eventbox), gui.ga_layout);
   gtk_table_attach_defaults(GTK_TABLE(gui.ga_subtable3), gui.ga_eventbox, 5, 16, 0, 2);
   gtk_widget_show(gui.ga_eventbox);
   gtk_widget_show(gui.ga_layout);

   gui.ga_combobox = gtk_combo_box_new_text();

   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "CRT1");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "CRT2");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Box 2.2");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Box 1.8");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Box custom");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Bar 2.2");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Bar 1.8");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Bar custom");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Brightness");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "RGB");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Grey");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "CMY");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Dark");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Medium");
   gtk_combo_box_append_text(GTK_COMBO_BOX(gui.ga_combobox), "Light");

   gui.ga_menuindices[0] = GAMMA_GFX_CRT1;
   gui.ga_menuindices[1] = GAMMA_GFX_CRT2;
   gui.ga_rmenuindices[GAMMA_GFX_CRT1] = 0;
   gui.ga_rmenuindices[GAMMA_GFX_CRT2] = 1;

   gtk_combo_box_set_active(GTK_COMBO_BOX(gui.ga_combobox), 0);

   gtk_widget_show(gui.ga_combobox);

   eventbox = gtk_event_box_new();
   gtk_container_set_border_width(GTK_CONTAINER(eventbox), 0);
   gtk_widget_show(eventbox);
   gtk_container_add(GTK_CONTAINER(eventbox), gui.ga_combobox);

   gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), eventbox,
		 	"Select test pattern", "");

   gtk_table_attach(GTK_TABLE(gui.ga_subtable3), eventbox, 0, 5, 0, 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

   gui.ga_sping_adj = (GtkAdjustment *)gtk_adjustment_new(2.2, 1.0, 3.55, 0.01, 0.1, 0.0);
   gui.ga_sping = gtk_spin_button_new(gui.ga_sping_adj, 0.1, 2);
   gtk_table_attach(GTK_TABLE(gui.ga_subtable3), gui.ga_sping, 0, 5, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);

   g_signal_connect(G_OBJECT(gui.ga_combobox), "changed", G_CALLBACK(ga_menu_changed), NULL);
   g_signal_connect(G_OBJECT(gui.ga_sping_adj), "value_changed", G_CALLBACK(ga_sping_changed), NULL);

   gtk_table_attach_defaults(GTK_TABLE(gui.ga_table), gui.ga_subtable3, 0, 5, 5, 7);
   /*gtk_table_attach(GTK_TABLE(gui.ga_table), gui.ga_subtable3, 0, 5, 5, 7, GTK_FILL|GTK_EXPAND, 0, 0, 0);*/
   gtk_widget_show(gui.ga_subtable3);

   gui_gamma_update_gamma_pattern_menu();

   g_signal_connect(G_OBJECT(gui.ga_layout), "expose-event", GTK_SIGNAL_FUNC(ga_scale_exposed), NULL);

   /* Finally... */

   gui.ga_label = xpm_label_box(icon_gamma, "Colors", NULL, 1, "Gamma correction and color settings", "");
   gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.ga_frame, gui.ga_label);
}

/**************************************************
 *                    Video page                  *
 **************************************************/

void
gui_vi_slider_set_value(int slider, double value)
{
   GtkAdjustment *myadj;

   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;

   switch(slider) {
   case VI_SLI_CONTRAST:
      myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_co));
      break;
   case VI_SLI_BRIGHTNESS:
      myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_br));
      break;
   case VI_SLI_HUE:
      myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_hu));
      break;
   case VI_SLI_SATURATION:
      myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_sa));
      break;
   default:
      return;
   }

   gtk_adjustment_set_value(myadj, value);
   g_signal_emit_by_name(G_OBJECT(myadj), "changed");
}

void
gui_vi_enable_and_set_xv_gamma_check(int enable, int active)
{
   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;

   gtk_widget_set_sensitive(gui.vi_check_gamma, enable ? TRUE : FALSE);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_check_gamma), active ? TRUE : FALSE);
}

void
gui_vi_enable_xvgamma_spins(int enable)
{
   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;
   if(!gl.supportxvgamma1) return;

   gtk_widget_set_sensitive(gui.vi_spin1, enable ? TRUE : FALSE);
   gtk_widget_set_sensitive(gui.vi_spin2, enable ? TRUE : FALSE);
   gtk_widget_set_sensitive(gui.vi_spin3, enable ? TRUE : FALSE);
}

void
gui_vi_set_xvgamma_spins(double s1, double s2, double s3)
{
   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;

   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.vi_spin1), s1);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.vi_spin2), s2);
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(gui.vi_spin3), s3);
}

void
gui_vi_enable_switchcrt(int enable)
{
   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;

   gtk_widget_set_sensitive(gui.vi_radio_crt1, enable ? TRUE : FALSE);
   gtk_widget_set_sensitive(gui.vi_radio_crt2, enable ? TRUE : FALSE);
#ifdef USE_STRAY
   gui_stray_video_set_sensitive(enable);
#endif
}

void
gui_vi_set_switchcrt_active_num(int num)
{
   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;

   switch(num) {
   case 1:
      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt1)) == FALSE) {
         gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt1), TRUE);
      }
      break;
   case 2:
   default:
      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt2)) == FALSE) {
         gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt2), TRUE);
      }
   }
#ifdef USE_STRAY
   gui_stray_video_set_active(num);
#endif
}

void
gui_vi_set_video_only_value(int active)
{
   if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) return;

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_check_video_only), active ? TRUE : FALSE);
}

/* Xv demo stuff */

#ifdef USEXV
static int
myxvwinerrorhandler(Display *dpy, XErrorEvent *error)
{
   gui.xerror = 1;
   return 0;
}

static Bool
testxvwindow(void)
{
   XErrorHandler oldErrHandler;

   if(gl.noxvdemo) return FALSE;

   /* We now create a child window of our GUI window. If that
    * fails, we don't use the xv demo image.
    * We use our (eventually user-provided) dpy here on purpose.
    * This is some sort of hack-ish way to determine if we run
    * on a remote display (where the window creation will fail
    * because one can't make child windows on different displays).
    */

   gui.xerror = 0;
   oldErrHandler = XSetErrorHandler(myxvwinerrorhandler);

   gui.xvwin = XCreateSimpleWindow(gl.dpy,
				   GDK_WINDOW_XWINDOW(gui.window->window),
				   0, 0,
				   1, 1,
				   1, 0, 0);
   XSync(gl.dpy, False);
   XSetErrorHandler(oldErrHandler);

   if(!gui.xvwin)
      return FALSE;

   if(gui.xerror) {
      /* No, don't call Destroy() here ... BadWindow! */
      gui.xvwin = 0;
      return FALSE;
   }

   XDestroyWindow(gl.dpy, gui.xvwin);
   gui.xvwin = 0;

   return TRUE;
}

static void
destroyxvimage(void)
{
   if(gui.xvshown) {
      XvStopVideo(gl.dpy, gl.xv_port_nr, gui.xvwin);
      gui.xvshown = FALSE;
   }

   if(gui.xvattached) {
      XShmDetach(gl.dpy, &gui.xvshminfo);
      shmdt(gui.xvshminfo.shmaddr);
      shmctl(gui.xvshminfo.shmid, IPC_RMID, 0);
      XFree(gui.myxvimage);
      gui.myxvimage = NULL;
      gui.xvattached = FALSE;
   }

   if(gui.xvcontext) {
      XFreeGC(gl.dpy, gui.xvcontext);
      gui.xvcontext = NULL;
   }

   if(gui.xvwin) {
      XDestroyWindow(gl.dpy, gui.xvwin);
      gui.xvwin = 0;
   }
}

static Bool
makexvwindow(void)
{
   char *ptr;
   int i, j;
   XErrorHandler oldErrorHandler;
#if 0
   unsigned int colorkey = sisctrl_vi_get_colorkey();
   GdkColor bgcol;
#endif

   if(gl.noxvdemo) return FALSE;

   if(gui.xvwin) return TRUE;

   gui.xerror = 0;
   oldErrorHandler = XSetErrorHandler(myxvwinerrorhandler);

   gui.xvwin = XCreateSimpleWindow(gl.dpy,
			GDK_WINDOW_XWINDOW(gui.window->window),
			0, 0,
			1, 1,
			1, 0, 0);
   XSync(gl.dpy, FALSE);

   XSetErrorHandler(oldErrorHandler);

   if(!gui.xvwin)
      return FALSE;

   if(gui.xerror) {
      gui.xvwin = 0;
      return FALSE;
   }

   XMapWindow(gl.dpy, gui.xvwin);
   XFlush(gl.dpy);
   gui.xvraised = TRUE;

   if(!gui.xvcontext) {
      gui.xvcontext = XCreateGC(gl.dpy, gui.xvwin, 0, 0);
   }

   if(!gui.xvcontext) {
      destroyxvimage();
      return FALSE;
   }

#if 0
   gui.xvwin = GDK_WINDOW_XWINDOW(gui.vi_layout->window);

   gui.vi_context = gdk_gc_new(gui.vi_layout->window);

   gui.xvcontext = GDK_GC_XGC(gui.vi_context);
#endif

#if 0
   bgcol.pixel = 0;
   bgcol.red = (colorkey & 0xff0000) >> 8;
   bgcol.green = (colorkey & 0xff00);
   bgcol.blue = (colorkey & 0xff) << 8;
   gdk_gc_set_rgb_bg_color(gui.vi_context, &bgcol);

   gdk_gc_set_fill(gui.vi_context, GDK_SOLID);
#endif

   if(!gui.myxvimage) {
      gui.myxvimage = XvShmCreateImage(gl.dpy, gl.xv_port_nr,
             		0x59565955, 0, XVWIDTH, XVHEIGHT,
			&gui.xvshminfo);
   }

   if(!gui.myxvimage) {
      fprintf(stderr, "XvShmCreateImage failed\n");
      destroyxvimage();
      return FALSE;
   }

   gui.xvshminfo.shmid = shmget(IPC_PRIVATE, gui.myxvimage->data_size, IPC_CREAT|0777);
   gui.xvshminfo.shmaddr = gui.myxvimage->data = shmat(gui.xvshminfo.shmid, 0, 0);
   gui.xvshminfo.readOnly = False;
   gui.xvattached = XShmAttach(gl.dpy, &gui.xvshminfo);
   if(gui.xvattached) {
      /*XvSelectVideoNotify(gl.dpy, gui.xvwin, True);  --- does not work */
      ptr = gui.myxvimage->data;
      if(!ptr) {
         fprintf(stderr, "XvImage->data is NULL\n");
         destroyxvimage();
         return FALSE;
      } else {
         for(i = 0; i < XVHEIGHT; i++, ptr += gui.myxvimage->pitches[0]) {
            for(j = 0; j < XVWIDTH; j++) {
	       ptr[(j << 1)] = xviconbuffer[(i * XVWIDTH * 2) + (j*2)];
	       ptr[(j << 1) + 1] = xviconbuffer[(i * XVWIDTH * 2) + (j*2) + 1];
            }
	 }
      }
      return TRUE;
   } else {
      fprintf(stderr, "XShmAttach failed\n");
      destroyxvimage();
      return FALSE;
   }
}

void
gui_destroy_xv_image(void)
{
   if(!gl.noxvdemo) {
      destroyxvimage();
   }
}

static void
resizexvwindow(void)
{
   XWindowAttributes wa;

   if(!gui.xvinit || !gui.xvwin)
      return;

   if(!GTK_WIDGET_REALIZED(gui.vi_layout)) {

      gui.xvx = gui.xvy = 0;
      gui.xvwidth = gui.xvheight = 1;

   } else {

      gui.xvx = gui.vi_layout->allocation.x;
      gui.xvy = gui.vi_layout->allocation.y;
      gui.xvwidth = gui.vi_layout->allocation.width;
      gui.xvheight = gui.vi_layout->allocation.height;

      XGetWindowAttributes(gl.dpy, gui.xvwin, &wa);

      if( (gui.xvx != wa.x) ||
	  (gui.xvy != wa.y) ||
	  (gui.xvwidth != wa.width) ||
	  (gui.xvheight != wa.height) ) {
	 XMoveResizeWindow(gl.dpy, gui.xvwin,
		gui.xvx, gui.xvy, gui.xvwidth, gui.xvheight);
      }
   }
}

static void
hidexvimage(void)
{
   /*fprintf(stderr, "hide!\n");*/

   if(gui.xvinit && gui.xvshown) {
      sisctrl_restore_disable_gfx();
      XvStopVideo(gl.dpy, gl.xv_port_nr, gui.xvwin);
      XSync(gl.dpy, False);
      gui.xvshown = FALSE;
   }

   if(gui.xvinit && gui.xvraised) {
      XUnmapWindow(gl.dpy, gui.xvwin);
      XFlush(gl.dpy);
      gui.xvraised = FALSE;
   }
}

static void
showxvimage()
{
   unsigned int colorkey;
   GdkColor bgcol;

   gui.xvshown = FALSE;

   /*fprintf(stderr, "show\n");*/

   if(!gui.xvinit ||
      !gui.xvwin  ||
      !gui.xvattached ||
      (gui.xvwidth == 0) || (gui.xvheight == 0) ||
      !gui.notebook ||
      gui.xvwasbusy)
      return;

   if(gtk_notebook_get_current_page(GTK_NOTEBOOK(gui.notebook)) != SIS_PAGE_VIDEO)
      return;

#if 0
   colorkey = sisctrl_vi_get_colorkey();

   bgcol.pixel = 0;
   bgcol.red = (colorkey & 0xff0000) >> 8;
   bgcol.green = (colorkey & 0xff00);
   bgcol.blue = (colorkey & 0xff) << 8;
   gdk_gc_set_rgb_fg_color(gui.vi_context, &bgcol);
   gdk_gc_set_fill(gui.vi_context, GDK_SOLID);

   gdk_draw_rectangle(gui.vi_layout->window,
   	gui.vi_context,
   	TRUE,
   	0, 0,
   	gui.xvwidth, gui.xvheight);
#endif

   /*if(!gui.xvraised) { */
      XMapWindow(gl.dpy, gui.xvwin);
      XRaiseWindow(gl.dpy, gui.xvwin);
      XFlush(gl.dpy);
      resizexvwindow();
      XSync(gl.dpy, False);
      gui.xvraised = TRUE;
   /* } */

   sisctrl_set_disable_gfx(0, 0);

   XvShmPutImage(gl.dpy, gl.xv_port_nr,
		 gui.xvwin,
		 gui.xvcontext,
		 gui.myxvimage,
		 0, 0,
		 XVWIDTH, XVHEIGHT,
		 0, 0,
		 gui.xvwidth, gui.xvheight,
		 False);

   XSync(gl.dpy, False);

   gui.xvshown = TRUE;
}

int
gui_vi_get_xv_shown(void)
{
   return gui.xvshown ? 1 : 0;
}
#endif

/* Signal handlers */

static void
vi_co_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_vi_co_changed(myval);

#ifdef USEXV
   if(!gl.noxvdemo) showxvimage();
#endif
}

static void
vi_br_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_vi_br_changed(myval);

#ifdef USEXV
   if(!gl.noxvdemo) showxvimage();
#endif
}

static void
vi_hu_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_vi_hu_changed(myval);

#ifdef USEXV
   if(!gl.noxvdemo) showxvimage();
#endif
}

static void
vi_sa_changed(GtkWidget *widget, gpointer data)
{
   double myval = gtk_adjustment_get_value(GTK_ADJUSTMENT(widget));

   sisctrl_vi_sa_changed(myval);

#ifdef USEXV
   if(!gl.noxvdemo) showxvimage();
#endif
}

static void
vi_radio_changed(GtkWidget *widget, gpointer data)
{
   sisctrl_vi_radio_changed(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? 0 : 1);
#ifdef USEXV
   if(!gl.noxvdemo) showxvimage();
#endif
}

static void
vi_check_gamma_toggled(GtkWidget *widget, gpointer data)
{
   sisctrl_vi_check_gamma_toggled(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? 1 : 0);
#ifdef USEXV
   if(!gl.noxvdemo) showxvimage();
#endif
}

static void
vi_spin1_changed(GtkWidget *widget, gpointer data)
{
   double val = gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.vi_spin1));
   sisctrl_vi_spin1_changed(val);
}

static void
vi_spin2_changed(GtkWidget *widget, gpointer data)
{
   double val = gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.vi_spin2));
   sisctrl_vi_spin2_changed(val);
}

static void
vi_spin3_changed(GtkWidget *widget, gpointer data)
{
   double val = gtk_spin_button_get_value(GTK_SPIN_BUTTON(gui.vi_spin3));
   sisctrl_vi_spin3_changed(val);
}

static void
vi_check_video_only_toggled(GtkWidget *widget, gpointer data)
{
   int val = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.vi_check_video_only)) ? 1 : 0;
   sisctrl_vi_check_video_only_toggled(val);
   sisctrl_restore_disable_gfx_lr();
}

static void
vi_radio_gl_changed(GtkWidget *widget, gpointer data)
{
   if(!gui.haveglobal) return;

   gui.isglobal = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ? TRUE : FALSE;

   sisctrl_vi_global_changed(gui.isglobal ? 1 : 0);

   sisctrl_vi_set_contrast_value();
   sisctrl_vi_set_brightness_value();
   if(gl.supporthuesat) {
      sisctrl_vi_set_hue_value();
      sisctrl_vi_set_saturation_value();
   }
}

#ifdef USEXV
static gboolean
vi_layout_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   /*fprintf(stderr, "expo\n"); */
   if(gl.noxvdemo) return TRUE;
   if(gui.notebook_page != SIS_PAGE_VIDEO) return TRUE;
   if(!gui.xvwin) return TRUE;
   if(!gui.xvinit) return TRUE;
   resizexvwindow();
   showxvimage();
   return TRUE;
}

#if 0
static gboolean
vi_layout_expose2(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   fprintf(stderr, "expo2\n");
   if(gl.noxvdemo) return TRUE;
   if(gui.notebook_page != SIS_PAGE_VIDEO) return TRUE;
   if(!gui.xvwin) return TRUE;
   if(!gui.xvinit) return TRUE;
   resizexvwindow();
   showxvimage();
   return FALSE;
}
#endif

static gboolean
vi_layout_configure(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   /*fprintf(stderr, "conf\n");*/
   if(gl.noxvdemo) return TRUE;
   if(gui.notebook_page != SIS_PAGE_VIDEO) return;
   if(!gui.xvwin) return;
   if(!gui.xvinit) return TRUE;
   resizexvwindow();
   showxvimage();
   return TRUE;
}

static gboolean
vi_layout_configure2(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   /*fprintf(stderr, "conf 2\n");*/
   if(gl.noxvdemo) return FALSE;
   if(gui.notebook_page != SIS_PAGE_VIDEO) return FALSE;
   if(!gui.xvwin) return FALSE;
   if(!gui.xvinit) return FALSE;
   resizexvwindow();
   showxvimage();
   return FALSE;
}

static gboolean
vi_layout_visibility(GtkWidget *widget, GdkEventVisibility *event, gpointer data)
{
   /*fprintf(stderr, "visi\n");*/

   if(event->type != GDK_VISIBILITY_NOTIFY)
      return TRUE;

   if(gl.noxvdemo) return TRUE;
   if(!gui.xvwin) return TRUE;
   if(!gui.xvinit) return TRUE;
   if(!gui.xvshown) return TRUE;

   if(event->state == GDK_VISIBILITY_FULLY_OBSCURED) {
      hidexvimage();
   }

   return TRUE;
}

#if 0	/* Does not work */
static GdkFilterReturn
vi_xv_event_filter(GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
   fprintf(stderr, "Eventfilter type %d (base %d, %d)\n",
   	((XvVideoNotifyEvent *)xevent)->type, gl.xv_event_base, XvVideoNotify);

   if(((XvVideoNotifyEvent *)xevent)->type - gl.xv_event_base != XvVideoNotify)
      return GDK_FILTER_CONTINUE;

   fprintf(stderr, "!!!!!! Got Xv Event: %d\n", ((XvVideoNotifyEvent *)xevent)->reason);

#if 0
   if(gl.noxvdemo || !gui.xvwin || !gui.xvinit)
      return GDK_FILTER_CONTINUE;
#endif

   gui.current_xv_result = ((XvVideoNotifyEvent *)xevent)->reason;
   return GDK_FILTER_REMOVE;
}
#endif

static gboolean
vi_layout_realize(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{

   if(gl.noxvdemo || gui.xvinit)
      return TRUE;

   if(makexvwindow()) {

      g_signal_connect(G_OBJECT(gui.vi_layout), "configure_event", G_CALLBACK(vi_layout_configure), NULL);
      g_signal_connect(G_OBJECT(gui.vi_layout), "expose_event", G_CALLBACK(vi_layout_expose), NULL);
      g_signal_connect(G_OBJECT(gui.vi_layout), "visibility-notify-event", G_CALLBACK(vi_layout_visibility), NULL);
      gtk_widget_add_events(gui.vi_layout, gtk_widget_get_events(gui.vi_layout) | GDK_VISIBILITY_NOTIFY_MASK);

      /*g_signal_connect(G_OBJECT(gui.window), "expose_event", G_CALLBACK(vi_layout_expose2), NULL);*/
      g_signal_connect(G_OBJECT(gui.window), "configure_event", G_CALLBACK(vi_layout_configure2), NULL);


#if 0  /* Does not work */
      gdk_x11_register_standard_event_type(gtk_widget_get_display(gui.vi_layout), gl.xv_event_base, XvNumEvents);
      gdk_window_add_filter(/*GTK_WIDGET(gui.vi_layout)->window*/ NULL, vi_xv_event_filter, NULL);
      /* XSelectInput(gl.dpy, gui.xvwin, NoEventMask); */
#endif

      gui.xvinit = TRUE;

   } else {

      gl.noxvdemo = TRUE;

   }

   return TRUE;
}


#endif

/* Build page */

static void
build_video_page(void)
{
   GtkWidget *vitable;
   int rows, colspacing, border, contrastpos, gammapos, switchpos, rows2;

   gui.vi_frame = gtk_frame_new("Video settings (Xv overlay)");
   gtk_container_set_border_width(GTK_CONTAINER(gui.vi_frame), 8);
   gtk_widget_show(gui.vi_frame);

   switch(gl.video_page_type) {

   case VIDEO_PAGE_TYPE_NORMAL:

      rows = 5; colspacing = 10; border = 15;
      rows2 = 4;
      contrastpos = 0;
      gammapos = 4; switchpos = 4;

      if(gl.video_page_subtype == VIDEO_PAGE_SUBTYPE_WITH_GLOBAL) {
	 rows++;
	 rows2++;
	 contrastpos++;
	 gammapos++;
	 switchpos++;
      }

      if(gl.supportxvgamma1) {
	 rows++;
	 switchpos++;
      }

#ifdef USEXV
      if(!gl.noxvdemo) {
         colspacing = 5; border = 10;
      }
#endif
      gui.vi_table = gtk_table_new(rows, 3, FALSE);
      gtk_table_set_row_spacings(GTK_TABLE(gui.vi_table), 2);
      gtk_table_set_col_spacings(GTK_TABLE(gui.vi_table), colspacing);
      gtk_container_set_border_width(GTK_CONTAINER(gui.vi_table), border);
      vitable = gui.vi_table;

      gtk_container_add(GTK_CONTAINER(gui.vi_frame), gui.vi_table);
      gtk_widget_show(gui.vi_table);

#ifdef USEXV
      if(!gl.noxvdemo) {
         gui.vi_table2 = gtk_table_new(rows2, 3, FALSE);
         gtk_table_set_row_spacings(GTK_TABLE(gui.vi_table), 2);
         gtk_table_set_col_spacings(GTK_TABLE(gui.vi_table), 2);
         gtk_container_set_border_width(GTK_CONTAINER(gui.vi_table2), 0);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_table2, 1, 3, 0, rows2);
         gtk_widget_show(gui.vi_table2);
         vitable = gui.vi_table2;
      }
#endif

      gui.haveglobal = FALSE;
      gui.isglobal = FALSE;

      if(gl.video_page_subtype == VIDEO_PAGE_SUBTYPE_WITH_GLOBAL) {

         char buffer[16];
	 sprintf(buffer, "for screen %d", gl.screennum);

	 gui.vi_label_gl = gtk_label_new("Adjust");
         gtk_widget_show(gui.vi_label_gl);
         gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_label_gl, 0, 1, 0, 1);

	 gui.vi_radio_gl = gtk_radio_button_new_with_label(NULL, "globally");
	 gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_radio_gl, 1, 2, 0, 1);
	 gtk_widget_show(gui.vi_radio_gl);

	 gui.vi_radio_th = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(gui.vi_radio_gl), buffer);
	 gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_radio_th, 2, 3, 0, 1);
	 gtk_widget_show(gui.vi_radio_th);

	 gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_radio_gl), TRUE);
	 gui.haveglobal = TRUE;
	 gui.isglobal = TRUE;

	 g_signal_connect(G_OBJECT(gui.vi_radio_gl), "toggled", G_CALLBACK(vi_radio_gl_changed), NULL);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_radio_gl,
		 	"Adjust contrast, brightness, hue and saturation for all screens", "");
	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_radio_th,
		 	"Adjust contrast, brightness, hue and saturation for that screen only", "");

	 sisctrl_vi_global_changed(gui.isglobal ? 1 : 0);
      }

      gui.vi_label_co = gtk_label_new("Contrast");
      gtk_widget_show(gui.vi_label_co);
      gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_label_co, 0, 1, contrastpos+0, contrastpos+1);

      gui.vi_range_co = gtk_hscale_new_with_range(0, 7, 1);
      gtk_scale_set_value_pos(GTK_SCALE(gui.vi_range_co), GTK_POS_RIGHT);
      gtk_widget_show(gui.vi_range_co);
      sisctrl_vi_set_contrast_value();
      gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_range_co, 1, 3, contrastpos+0, contrastpos+1);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_range_co,
		 	"Adjust video contrast", "");

      {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_co));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(vi_co_changed), NULL);
      }

      gui.vi_label_br = gtk_label_new("Brightness");
      gtk_widget_show(gui.vi_label_br);
      gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_label_br, 0, 1, contrastpos+1, contrastpos+2);

      gui.vi_range_br = gtk_hscale_new_with_range(-128, 127, 1);
      gtk_scale_set_value_pos(GTK_SCALE(gui.vi_range_br), GTK_POS_RIGHT);
      gtk_widget_show(gui.vi_range_br);
      sisctrl_vi_set_brightness_value();
      gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_range_br, 1, 3, contrastpos+1, contrastpos+2);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_range_br,
		 	"Adjust video brightness", "");

      {
         GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_br));
         g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(vi_br_changed), NULL);
      }

      if(gl.supporthuesat) {
         gui.vi_label_hu = gtk_label_new("Hue");
         gtk_widget_show(gui.vi_label_hu);
         gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_label_hu, 0, 1, contrastpos+2, contrastpos+3);

         gui.vi_range_hu = gtk_hscale_new_with_range(-8, 7, 1);
         gtk_scale_set_value_pos(GTK_SCALE(gui.vi_range_hu), GTK_POS_RIGHT);
         gtk_widget_show(gui.vi_range_hu);
         sisctrl_vi_set_hue_value();
         gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_range_hu, 1, 3, contrastpos+2, contrastpos+3);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_range_hu,
		 	"Adjust video hue", "");

         {
            GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_hu));
            g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(vi_hu_changed), NULL);
         }

         gui.vi_label_sa = gtk_label_new("Saturation");
         gtk_widget_show(gui.vi_label_sa);
         gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_label_sa, 0, 1, contrastpos+3, contrastpos+4);

         gui.vi_range_sa = gtk_hscale_new_with_range(-7, 7, 1);
         gtk_scale_set_value_pos(GTK_SCALE(gui.vi_range_sa), GTK_POS_RIGHT);
         gtk_widget_show(gui.vi_range_sa);
         sisctrl_vi_set_saturation_value();
         gtk_table_attach_defaults(GTK_TABLE(vitable), gui.vi_range_sa, 1, 3, contrastpos+3, contrastpos+4);

	 gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_range_sa,
		 	"Adjust video saturation", "");

         {
            GtkAdjustment *myadj = gtk_range_get_adjustment(GTK_RANGE(gui.vi_range_sa));
            g_signal_connect(G_OBJECT(myadj), "value_changed", G_CALLBACK(vi_sa_changed), NULL);
         }
      }

#ifdef USEXV
      if(!gl.noxvdemo) {
         gui.vi_label_busy = gtk_label_new("Xv busy");
	 gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_label_busy, 0, 1, 0, rows2);
	 gtk_widget_hide(gui.vi_label_busy);
	 gui.vi_layout = gtk_label_new(" "); /*gtk_drawing_area_new();*/
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_layout, 0, 1, 0, rows2);
         gtk_widget_show(gui.vi_layout);
      }
#endif

      if(gl.supportxvgamma1) {

         gui.vi_check_gamma = gtk_check_button_new_with_label("Gamma (CRT1 only)");
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_check_gamma, 0, 1, gammapos, gammapos+1);
         gtk_widget_show(gui.vi_check_gamma);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_check_gamma,
		 	"Enable gamma correction for the video overlay. This works on first output (CRT1) only.", "");

         gui.vi_ga_table = gtk_table_new(1, 6, FALSE);
         gtk_table_set_row_spacings(GTK_TABLE(gui.vi_ga_table), 1);
         gtk_table_set_col_spacings(GTK_TABLE(gui.vi_ga_table), 1);
         gtk_container_set_border_width(GTK_CONTAINER(gui.vi_ga_table), 1);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_ga_table, 1, 3, gammapos, gammapos+1);
         gtk_widget_show(gui.vi_ga_table);

         gui.vi_label_red = gtk_label_new("R");
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_ga_table), gui.vi_label_red, 0, 1, 0, 1);
         gtk_widget_show(gui.vi_label_red);
         gui.vi_label_green = gtk_label_new("G");
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_ga_table), gui.vi_label_green, 2, 3, 0, 1);
         gtk_widget_show(gui.vi_label_green);
         gui.vi_label_blue = gtk_label_new("B");
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_ga_table), gui.vi_label_blue, 4, 5, 0, 1);
         gtk_widget_show(gui.vi_label_blue);

         gui.vi_spin1_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.1, 1.0, 0.0);
         gui.vi_spin2_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.1, 1.0, 0.0);
         gui.vi_spin3_adj = (GtkAdjustment *)gtk_adjustment_new(1.0, 0.1, 10.0, 0.1, 1.0, 0.0);
         gui.vi_spin1 = gtk_spin_button_new(gui.vi_spin1_adj, 0.1, 2);
         gui.vi_spin2 = gtk_spin_button_new(gui.vi_spin2_adj, 0.1, 2);
         gui.vi_spin3 = gtk_spin_button_new(gui.vi_spin3_adj, 0.1, 2);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_ga_table), gui.vi_spin1, 1, 2, 0, 1);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_ga_table), gui.vi_spin2, 3, 4, 0, 1);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_ga_table), gui.vi_spin3, 5, 6, 0, 1);
         sisctrl_vi_set_xvgamma_spin_values();
         gtk_widget_show(gui.vi_spin1);
         gtk_widget_show(gui.vi_spin2);
         gtk_widget_show(gui.vi_spin3);

         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_spin1,
		 	"Adjust red channel gamma for the video overlay", "");
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_spin2,
		 	"Adjust green channel gamma for the video overlay", "");
         gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_spin3,
		 	"Adjust blue channel gamma for the video overlay", "");

         sisctrl_vi_set_xv_gamma_status();

         g_signal_connect(G_OBJECT(gui.vi_check_gamma), "toggled", G_CALLBACK(vi_check_gamma_toggled), NULL);
         g_signal_connect(G_OBJECT(gui.vi_spin1_adj), "value_changed", G_CALLBACK(vi_spin1_changed), NULL);
         g_signal_connect(G_OBJECT(gui.vi_spin2_adj), "value_changed", G_CALLBACK(vi_spin2_changed), NULL);
         g_signal_connect(G_OBJECT(gui.vi_spin3_adj), "value_changed", G_CALLBACK(vi_spin3_changed), NULL);
      }

      gui.vi_check_video_only = gtk_check_button_new_with_label("Video only");
      gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_check_video_only, 0, 1, switchpos, switchpos+1);
      gtk_widget_show(gui.vi_check_video_only);
      g_signal_connect(G_OBJECT(gui.vi_check_video_only), "toggled", G_CALLBACK (vi_check_video_only_toggled), NULL);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_check_video_only,
		"Selects to display the video image only and to disable the graphics image left and right of the "
		"video image. This saves bandwidth and helps on machines with low memory clocks and other hardware "
		"issues, such as the SiS760. Takes effect outside of sisctrl only.", "");

      if(gl.needswitchcrt) {

	 gui.vi_sw_table = gtk_table_new(1, 3, TRUE);
         gtk_table_set_row_spacings(GTK_TABLE(gui.vi_sw_table), 1);
         gtk_table_set_col_spacings(GTK_TABLE(gui.vi_sw_table), 1);
         gtk_container_set_border_width(GTK_CONTAINER(gui.vi_sw_table), 1);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_sw_table, 1, 3, switchpos, switchpos+1);
         gtk_widget_show(gui.vi_sw_table);

         gui.vi_label_sw = gtk_label_new("Video on");
         gtk_widget_show(gui.vi_label_sw);
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_sw_table), gui.vi_label_sw, 0, 1, 0, 1);

         gui.vi_radio_crt1 = gtk_radio_button_new_with_label(NULL, "CRT1");
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_sw_table), gui.vi_radio_crt1, 1, 2, 0, 1);
         gtk_widget_show(gui.vi_radio_crt1);

         gui.vi_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(gui.vi_radio_crt1));
         gui.vi_radio_crt2 = gtk_radio_button_new_with_label(gui.vi_group, "CRT2");
         gtk_table_attach_defaults(GTK_TABLE(gui.vi_sw_table), gui.vi_radio_crt2, 2, 3, 0, 1);
         gtk_widget_show(gui.vi_radio_crt2);

	 if(!gl.mergedfb) {
            gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_radio_crt1,
		 	"Show video overlay on first output (CRT1)", "");
            gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_radio_crt2,
		 	"Show video overlay on second output (CRT2)", "");
	 } else {
	    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_radio_crt1,
		 	"Show video overlay on first output (CRT1). Mainly effective for clone modes.", "");
            gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.vi_radio_crt2,
		 	"Show video overlay on second output (CRT2). Mainly effective for clone modes.", "");
	 }

         sisctrl_vi_set_switchcrt();

         g_signal_connect(G_OBJECT(gui.vi_radio_crt1), "toggled", G_CALLBACK(vi_radio_changed), NULL);
      }
      break;

    case VIDEO_PAGE_TYPE_NA_NOOVL:
       gui.vi_table = gtk_table_new(3, 3, TRUE);
       gtk_table_set_row_spacings(GTK_TABLE(gui.vi_table), 10);
       gtk_table_set_col_spacings(GTK_TABLE(gui.vi_table), 4);

       gtk_container_set_border_width(GTK_CONTAINER(gui.vi_table), 8);

       gtk_container_add(GTK_CONTAINER(gui.vi_frame), gui.vi_table);
       gtk_widget_show(gui.vi_table);

       gui.vi_label_sw = gtk_label_new("No video overlay available");
       gtk_widget_show(gui.vi_label_sw);
       gtk_table_attach_defaults(GTK_TABLE(gui.vi_table), gui.vi_label_sw, 0, 3, 1, 2);
       break;
    }

    gui.vi_label = xpm_label_box(icon_video, "Video", NULL, 1, "Video overlay (Xv) settings", "");
    gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.vi_frame, gui.vi_label);
}

/**************************************************
 *                 "Current" page                 *
 **************************************************/

void
gui_cf_clear_text(void)
{
   gtk_text_buffer_set_text(gui.cf_buffer, "", -1);
}

static GtkTextTag *
gui_cf_convert_int_to_tag(int texttag)
{
   GtkTextTag *tag = NULL;

   switch(texttag) {
      case CU_TT_BLACK:		tag = gui.black; break;
      case CU_TT_BLACK_BOLD:	tag = gui.blackb; break;
      case CU_TT_RED:		tag = gui.red; break;
      case CU_TT_RED_BOLD:	tag = gui.redb; break;
      case CU_TT_BLUE:		tag = gui.blue; break;
      case CU_TT_BLUE_BOLD:	tag = gui.blueb; break;
      default:
      case CU_TT_GREY: 		tag = gui.grey; break;
   }
   return tag;
}

void
gui_cf_insert_option(char *oname, char *ovalue, int texttag)
{
   GtkTextIter myiter, start, end;
   GtkTextTag *tag = gui_cf_convert_int_to_tag(texttag);
   char options[] = "\tOption \"";
   int curlen;

   curlen = gtk_text_buffer_get_char_count(gui.cf_buffer);

   gtk_text_buffer_get_end_iter(gui.cf_buffer, &myiter);
   gtk_text_buffer_insert(gui.cf_buffer, &myiter, options, -1);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &myiter);
   gtk_text_buffer_insert(gui.cf_buffer, &myiter, oname, -1);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &myiter);
   gtk_text_buffer_insert(gui.cf_buffer, &myiter, "\" \"", -1);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &myiter);
   gtk_text_buffer_insert(gui.cf_buffer, &myiter, ovalue, -1);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &myiter);
   gtk_text_buffer_insert(gui.cf_buffer, &myiter, "\"\n", -1);

   gtk_text_buffer_get_iter_at_offset(gui.cf_buffer, &start, curlen);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &end);
   if(tag) gtk_text_buffer_apply_tag(gui.cf_buffer, tag, &start, &end);
}

void
gui_cf_append_text(char *mytext, int texttag)
{
   GtkTextIter myiter, start, end;
   GtkTextTag *tag = gui_cf_convert_int_to_tag(texttag);
   int curlen;

   curlen = gtk_text_buffer_get_char_count(gui.cf_buffer);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &myiter);
   gtk_text_buffer_insert(gui.cf_buffer, &myiter, mytext, -1);
   gtk_text_buffer_get_iter_at_offset(gui.cf_buffer, &start, curlen);
   gtk_text_buffer_get_end_iter(gui.cf_buffer, &end);
   if(tag) gtk_text_buffer_apply_tag(gui.cf_buffer, tag, &start, &end);
}

/* Signal handlers */

static void
gui_find_hint_type(void)
{
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.cf_radio1))) {
      gui.showhints = CF_HINTS_GE;
   } else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.cf_radio2))) {
      gui.showhints = CF_HINTS_MFB;
   } else {
      gui.showhints = CF_HINTS_DHM;
   }
}

static void
cf_check_toggled(GtkWidget *widget, gpointer data)
{
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
      gtk_widget_set_sensitive(gui.cf_radio1, TRUE);
      if( (gl.dualhead) || (gl.mergedfb) || (!gl.havevb) ) {
         gtk_widget_set_sensitive(gui.cf_radio2, FALSE);
         gtk_widget_set_sensitive(gui.cf_radio3, FALSE);
         gui.showhints = CF_HINTS_GE;
      } else {
         gtk_widget_set_sensitive(gui.cf_radio2, TRUE);
         gtk_widget_set_sensitive(gui.cf_radio3, TRUE);
         gui_find_hint_type();
      }
   } else {
      gui.showhints = CF_HINTS_NO;
      gtk_widget_set_sensitive(gui.cf_radio1, FALSE);
      gtk_widget_set_sensitive(gui.cf_radio2, FALSE);
      gtk_widget_set_sensitive(gui.cf_radio3, FALSE);
   }
   sisctrl_gl_current_page_selected(gui.showhints);
}

static void
cf_radio_changed(GtkWidget *widget, gpointer data)
{
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
      gui_find_hint_type();
      sisctrl_gl_current_page_selected(gui.showhints);
   }
}

/* Build page */

static void
build_current_page(void)
{
   int myspace = 6;

   if(gl.config_page_type != CONFIG_PAGE_TYPE_NO_HINTS) myspace = 0;

   gui.cf_frame = gtk_frame_new("Current configuration");
   gtk_container_set_border_width(GTK_CONTAINER(gui.cf_frame), 8);
   gtk_widget_show(gui.cf_frame);

   /* Text window, text buffer */
   gui.cf_text_window = gtk_scrolled_window_new(NULL, NULL);
   gtk_container_set_border_width(GTK_CONTAINER(gui.cf_text_window), myspace);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(gui.cf_text_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

   gui.cf_tview = gtk_text_view_new();
   gui.cf_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(gui.cf_tview));

   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(gui.cf_tview), GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(gui.cf_tview), FALSE);
   gtk_container_set_border_width(GTK_CONTAINER(gui.cf_tview), 1);
   gtk_text_buffer_set_text(gui.cf_buffer, "", -1);

   gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(gui.cf_text_window), GTK_SHADOW_IN);

   gtk_container_add(GTK_CONTAINER(gui.cf_text_window), gui.cf_tview);

   gtk_widget_show(gui.cf_tview);
   gtk_widget_show(gui.cf_text_window);

   if(gl.config_page_type == CONFIG_PAGE_TYPE_NO_HINTS) {

      gtk_container_add(GTK_CONTAINER(gui.cf_frame), gui.cf_text_window);

   } else {

      gui.cf_table = gtk_table_new(6, 4, FALSE);
      gtk_table_set_row_spacings(GTK_TABLE(gui.cf_table), 0);
      gtk_table_set_col_spacings(GTK_TABLE(gui.cf_table), 0);
      gtk_container_set_border_width(GTK_CONTAINER(gui.cf_table), 8);
      gtk_widget_show(gui.cf_table);

      gtk_table_attach_defaults(GTK_TABLE(gui.cf_table), gui.cf_text_window, 0, 4, 0, 5);

      gui.cf_check = gtk_check_button_new_with_label("Show hints");
      gtk_table_attach_defaults(GTK_TABLE(gui.cf_table), gui.cf_check, 0, 1, 5, 6);
      gtk_widget_show(gui.cf_check);

      gui.cf_radio1 = gtk_radio_button_new_with_label(NULL, "Generic");
      gtk_table_attach_defaults(GTK_TABLE(gui.cf_table), gui.cf_radio1, 1, 2, 5, 6);
      gtk_widget_show(gui.cf_radio1);

      /* gui.cf_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(gui.cf_radio1)); */

      gui.cf_radio2 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(gui.cf_radio1), "MergedFB");
      gtk_table_attach_defaults(GTK_TABLE(gui.cf_table), gui.cf_radio2, 2, 3, 5, 6);
      gtk_widget_show(gui.cf_radio2);

      gui.cf_radio3 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(gui.cf_radio1), "Dual Head");
      gtk_table_attach_defaults(GTK_TABLE(gui.cf_table), gui.cf_radio3, 3, 4, 5, 6);
      gtk_widget_show(gui.cf_radio3);

      gui.showhints = CF_HINTS_NO;
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.cf_check), FALSE);

      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.cf_radio1), TRUE);

      gtk_widget_set_sensitive(gui.cf_radio1, FALSE);
      gtk_widget_set_sensitive(gui.cf_radio2, FALSE);
      gtk_widget_set_sensitive(gui.cf_radio3, FALSE);

      g_signal_connect(G_OBJECT(gui.cf_check), "toggled", G_CALLBACK(cf_check_toggled), NULL);

      g_signal_connect(G_OBJECT(gui.cf_radio1), "toggled", G_CALLBACK(cf_radio_changed), NULL);
      g_signal_connect(G_OBJECT(gui.cf_radio2), "toggled", G_CALLBACK(cf_radio_changed), NULL);
      g_signal_connect(G_OBJECT(gui.cf_radio3), "toggled", G_CALLBACK(cf_radio_changed), NULL);

      gtk_container_add(GTK_CONTAINER(gui.cf_frame), gui.cf_table);
   }

   gui.black = gtk_text_buffer_create_tag(gui.cf_buffer, "black_foreground", "foreground", "black", NULL);
   gui.grey = gtk_text_buffer_create_tag(gui.cf_buffer, "grey_foreground", "foreground", "grey32", NULL);
   gui.red = gtk_text_buffer_create_tag(gui.cf_buffer, "red_foreground", "foreground", "red", NULL);
   gui.blackb = gtk_text_buffer_create_tag(gui.cf_buffer, "black_foreground_bold", "foreground", "black", "weight", "bold", NULL);
   gui.redb = gtk_text_buffer_create_tag(gui.cf_buffer, "red_foreground_bold", "foreground", "red", "weight", "bold", NULL);
   gui.blue = gtk_text_buffer_create_tag(gui.cf_buffer, "blue_foreground", "foreground", "blue", NULL);
   gui.blueb = gtk_text_buffer_create_tag(gui.cf_buffer, "blue_foreground_bold", "foreground", "blue", "weight", "bold", NULL);

   gui.cf_label = xpm_label_box(icon_config, "Current", NULL, 1, "Show current configuration", "");
   gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.cf_frame, gui.cf_label);
}

/**************************************************
 *                   Info page                    *
 **************************************************/

void
gui_ab_clear_text(void)
{
   gtk_text_buffer_set_text(gui.ab_buffer, "", -1);
}

void
gui_ab_append_text(char *mytext, int texttag)
{
   GtkTextIter myiter, start, end;
   GtkTextTag *tag = NULL;
   int curlen;

   switch(texttag) {
   case AB_TT_BLACK:
      tag = gui.abblack;
      break;
   default:
   case AB_TT_BLACK_BOLD:
      tag = gui.abblackb;
      break;
   }

   curlen = gtk_text_buffer_get_char_count(gui.ab_buffer);
   gtk_text_buffer_get_end_iter(gui.ab_buffer, &myiter);
   gtk_text_buffer_insert(gui.ab_buffer, &myiter, mytext, -1);
   gtk_text_buffer_get_iter_at_offset(gui.ab_buffer, &start, curlen);
   gtk_text_buffer_get_end_iter(gui.ab_buffer, &end);
   if(tag) gtk_text_buffer_apply_tag(gui.ab_buffer, tag, &start, &end);
}

static gboolean
ab_layout_expose(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
   int x, y, w, h, i, j, wmax, lastpos, hstepwidth, vstepwidth;
   int dynamicx = 0;
   gint lwidth = widget->allocation.width;
   gint lheight = widget->allocation.height;
   PangoLayout *playout;
   GtkAllocation alloc;
   GtkWidget *labels[] = { gui.ab_label_ab1,
			   gui.ab_label_ab2,
			   gui.ab_label_ab4 };

#define BW 2   /* border width */
#define BWT 8  /* border width for text area */

   if(!gui.ab_context) {
      gui.ab_context = gdk_gc_new(GTK_LAYOUT(gui.ab_layout)->bin_window);
   }

   gdk_gc_set_rgb_fg_color(gui.ab_context, &gui.ab_bgcolorb);
   gdk_draw_line(GTK_LAYOUT(widget)->bin_window,
		 gui.ab_context,
		 BW-1, BW-1, BW - 1 + lwidth - (BW * 2), BW - 1);
   gdk_draw_line(GTK_LAYOUT(widget)->bin_window,
		 gui.ab_context,
		 BW-1, BW-1, BW-1, BW - 1 + lheight - (BW * 2));

   gdk_gc_set_rgb_fg_color(gui.ab_context, &gui.ab_bgcolorw);
   gdk_draw_line(GTK_LAYOUT(widget)->bin_window,
		 gui.ab_context,
		 BW - 1,
		 BW - 1 + lheight - (BW * 2) + 1,
		 BW - 1 + lwidth - (BW * 2),
		 BW - 1 + lheight - (BW * 2) + 1);
   gdk_draw_line(GTK_LAYOUT(widget)->bin_window,
		 gui.ab_context,
		 BW - 1 + lwidth - (BW * 2) + 1,
		 BW - 1,
		 BW - 1 + lwidth - (BW * 2) + 1,
		 BW - 1 + lheight - (BW * 2) + 1);

   gdk_gc_set_rgb_fg_color(gui.ab_context, &gui.ab_bgcolor);
   gdk_draw_rectangle(GTK_LAYOUT(widget)->bin_window,
		      gui.ab_context,
		      TRUE,
		      BW, BW,
		      lwidth - (BW * 2), lheight - (BW * 2));

   gdk_draw_drawable(GTK_LAYOUT(widget)->bin_window,
		     gui.ab_context,
		     gui.ab_iconul,
		     0, 0,
		     BW, BW,
		     180, 130);

   hstepwidth = ((lwidth - (BWT * 2)) / 2);
   vstepwidth = ((lheight - 8 - (BWT * 2)) / 4);

   /* x = hstepwidth - (81 / 2) + BWT; */
   x = 180 + BWT;
   if((x + 81) > lwidth) x = hstepwidth - (81 / 2) + BWT;
   y = ((vstepwidth - (23 / 2)) / 2) + BWT;
   gdk_draw_drawable(GTK_LAYOUT(widget)->bin_window,
		     widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		     gui.ab_icon,
		     0, 0, x, y, 81, 23);

   wmax = 0;
   for(i = 0; i < 3; i++) {
      playout = gtk_label_get_layout(GTK_LABEL(labels[i]));
      pango_layout_get_pixel_size(playout, &w, &h);
      if(wmax < w) wmax = w;
   }

   x = 180 + 20 + BWT;
   if((x + wmax) > lwidth) x = lwidth - wmax;
   if(x < 0) dynamicx = 1;

   for(i = 0; i < 3; i++) {
      playout = gtk_label_get_layout(GTK_LABEL(labels[i]));
      pango_layout_get_pixel_size(playout, &w, &h);
      lastpos = 8 + (vstepwidth * (i + 1)) + ((vstepwidth - (h / 2)) / 2);
      if(dynamicx) x = hstepwidth - (w / 2) + BWT;
      gdk_draw_layout(GTK_LAYOUT(widget)->bin_window,
                   labels[i]->style->fg_gc[GTK_WIDGET_STATE(labels[i])],
		   x, lastpos + BWT, playout);
   }

   return TRUE;
}

static void
build_info_page(void)
{
   GdkBitmap *mask;

   gui.ab_bgcolor.red = 193 << 8;
   gui.ab_bgcolor.green = 193 << 8;
   gui.ab_bgcolor.blue = 217 << 8;

   gui.ab_bgcolorw.red = 65535;
   gui.ab_bgcolorw.green = 65535;
   gui.ab_bgcolorw.blue = 65535;

   gui.ab_bgcolorb.red = 0;
   gui.ab_bgcolorb.green = 0;
   gui.ab_bgcolorb.blue = 0;

   gui.ab_frame = gtk_frame_new("Info");
   gtk_container_set_border_width(GTK_CONTAINER(gui.ab_frame), 8);
   gtk_widget_show(gui.ab_frame);

   gui.ab_table = gtk_table_new(3, 1, FALSE);
   gtk_table_set_row_spacings(GTK_TABLE(gui.ab_table), 1);
   gtk_table_set_col_spacings(GTK_TABLE(gui.ab_table), 0);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ab_table), 1);

   gui.ab_layout = gtk_layout_new(NULL, NULL);
   gtk_widget_set_size_request(gui.ab_layout, 0, 138);
   gtk_table_attach_defaults(GTK_TABLE(gui.ab_table), gui.ab_layout, 0, 1, 0, 2);
   gtk_widget_show(gui.ab_layout);

   gui.ab_iconul = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, sisctrl_logo_xpm);
   g_object_unref(G_OBJECT(mask));
   gui.ab_icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, sisctrl2_xpm);
   g_object_unref(G_OBJECT(mask));

   gui.ab_label_ab1 = gtk_label_new("SiS/XGI Display Control Panel");
   gui.ab_label_ab2 = gtk_label_new("(C) 2003-2005 Thomas Winischhofer");
   gui.ab_label_ab4 = gtk_label_new("Licensed under the terms of the GPLv2");
   gtk_layout_put(GTK_LAYOUT(gui.ab_layout), gui.ab_label_ab1, 0, 0);
   gtk_layout_put(GTK_LAYOUT(gui.ab_layout), gui.ab_label_ab2, 0, 10);
   gtk_layout_put(GTK_LAYOUT(gui.ab_layout), gui.ab_label_ab4, 0, 20);
   gtk_widget_show(gui.ab_label_ab1);
   gtk_widget_show(gui.ab_label_ab2);
   gtk_widget_show(gui.ab_label_ab4);

   gui.ab_text_window = gtk_scrolled_window_new(NULL, NULL);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ab_text_window), 1);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(gui.ab_text_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

   gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(gui.ab_text_window), GTK_SHADOW_IN);

   gui.ab_tview = gtk_text_view_new();
   gui.ab_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(gui.ab_tview));

   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(gui.ab_tview), GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(gui.ab_tview), FALSE);
   gtk_container_set_border_width(GTK_CONTAINER(gui.ab_tview), 1);
   gtk_text_buffer_set_text(gui.ab_buffer, "", -1);

   gtk_container_add(GTK_CONTAINER(gui.ab_text_window), gui.ab_tview);
   gtk_table_attach_defaults(GTK_TABLE(gui.ab_table), gui.ab_text_window, 0, 1, 2, 3);

   gtk_widget_show(gui.ab_tview);
   gtk_widget_show(gui.ab_text_window);

   gui.abblack = gtk_text_buffer_create_tag(gui.ab_buffer, "black_foreground", "foreground", "black", NULL);
   gui.abblackb = gtk_text_buffer_create_tag(gui.ab_buffer, "black_bold", "foreground", "black",
							"weight", PANGO_WEIGHT_BOLD, NULL);

   sisctrl_ab_generate_current();

   gtk_container_add(GTK_CONTAINER(gui.ab_frame), gui.ab_table);

   gtk_widget_show(gui.ab_table);

   g_signal_connect(G_OBJECT(gui.ab_layout), "expose_event", G_CALLBACK(ab_layout_expose), NULL);

   gui.ab_label = xpm_label_box(icon_about, "Info", NULL, 1, "About SiSCtrl", "");
   gtk_notebook_append_page(GTK_NOTEBOOK(gui.notebook), gui.ab_frame, gui.ab_label);
}


/**************************************************
 *                  Global stuff                  *
 **************************************************/

#ifdef HAVE_RANDR
static void
xrandreventhandler(GdkScreen *screen, gpointer data)
{
   sisctrl_xrandreventhandler();
}
#endif

static gint
delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    sisctrl_delete_event();
    return FALSE;
}

/* destroy callback */
static void
destroy(GtkWidget *widget, gpointer data)
{
    sisctrl_destroy_event();
    gtk_main_quit();
}

static void
apply_clicked(GtkWidget *widget, gpointer data)
{
   int nb_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gui.notebook));

   sisctrl_do_apply(nb_page);
}

static void
ok_clicked(GtkWidget *widget, gpointer data)
{
   int nb_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gui.notebook));

   sisctrl_ok_clicked(nb_page);
}

static void
close_clicked(GtkWidget *widget, gpointer data)
{
   sisctrl_close_clicked();
}

static void
revert_clicked(GtkWidget *widget, gpointer data)
{
   int nb_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gui.notebook));

   sisctrl_revert_clicked(nb_page);
}

/* Helper for enabling/disabling "Revert" button */

void
gui_enable_revert_button(int enable)
{
   gtk_widget_set_sensitive(gui.button4, enable ? TRUE : FALSE);
}

/* Notebook signal handler */

static gboolean
notebook_changed(GtkNotebook *nb, GtkNotebookPage *page, guint page_num, gpointer data)
{
   int i;
   gint temp;
   int oldpage = gui.notebook_page;

   gui.notebook_page = page_num;

#ifdef USEXV
   if(!gl.noxvdemo) {
      if(page_num == SIS_PAGE_VIDEO) {
         if(XvGrabPort(gl.dpy, gl.xv_port_nr, CurrentTime) == Success) {
            XvUngrabPort(gl.dpy, gl.xv_port_nr, CurrentTime);
            if(gui.xvwasbusy) {
               gtk_widget_hide(gui.vi_label_busy);
               gtk_widget_show(gui.vi_layout);
               gui.xvwasbusy = FALSE;
            }
            showxvimage();
         } else {
	    hidexvimage();
	    if(!gui.xvwasbusy) {
	       gui.xvwasbusy = TRUE;
	       gtk_widget_hide(gui.vi_layout);
	       gtk_widget_show(gui.vi_label_busy);
	    }
	 }
      } else {
         hidexvimage();
      }
   }
#endif

   gtk_widget_set_sensitive(gui.button2, (page_num <= SIS_PAGE_CRT2));

   switch(oldpage) {
   case SIS_PAGE_CRT1:
      sisctrl_gl_crt1_page_deselected();
      break;
   case SIS_PAGE_CRT2:
      sisctrl_gl_crt2_page_deselected();
      break;
   }

   switch(page_num) {
   case SIS_PAGE_DM:
      sisctrl_gl_display_mode_page_selected();
      gui_enable_revert_button(0);
#ifdef HAVE_RANDR
      if(gl.do_randr) {
	 temp = gtk_combo_box_get_active(GTK_COMBO_BOX(gui.dm_comboboxrr));
	 if(temp >= 2) {
	    gtk_combo_box_set_active(GTK_COMBO_BOX(gui.dm_comboboxrr), 0);
	 }
      }
#endif
      break;
   case SIS_PAGE_CRT1:
      sisctrl_gl_crt1_page_selected();
      gui_enable_revert_button(0);
      break;
   case SIS_PAGE_CRT2:
      sisctrl_gl_crt2_page_selected();
      gui_enable_revert_button(0);
      break;
   case SIS_PAGE_TV:
      sisctrl_gl_tv_page_selected();
      gui_enable_revert_button(0);
      break;
   case SIS_PAGE_COLOR:
      sisctrl_gl_gamma_page_selected();
      gui_enable_revert_button(1);
      break;
   case SIS_PAGE_VIDEO:
      sisctrl_gl_video_page_selected();
      gui_enable_revert_button(0);
      break;
   case SIS_PAGE_CONFIG:
      sisctrl_gl_current_page_selected(gui.showhints);
      gui_enable_revert_button(0);
      break;
   default:
      gui_enable_revert_button(0);
   }

   return FALSE;
}

#if 0  /* For testing Xinerama events */
#if defined(HAVE_XINERAMA) && defined(HAVE_XINERAMA_1_2)
static GdkFilterFunc
eventfilter(GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
   XEvent *xev = (XEvent *)xevent;

   if(xev->xany.type == gl.xineramaeb + XineramaLayoutChangeNotify) {
      fprintf(stderr, "Xinerama changed!\n");
   }
   return GDK_FILTER_CONTINUE;
}
#endif
#endif

void
gui_init(int *argc, char ***argv)
{
   /* Clear gui data */
   memset(&gui, 0, sizeof(gui));

   /* Init gtk */
   gtk_init(argc, argv);
}

static void
gui_show_init_window(void)
{
    if(gui.init_done) return;

    /* Show the window */
    gtk_widget_show(gui.window);

#ifdef USEXV
    if(!gl.noxvdemo) {
       gdk_flush();
       g_signal_connect(G_OBJECT(gui.vi_layout), "realize", G_CALLBACK(vi_layout_realize), NULL);
    }
#endif

    gui.init_done = TRUE;
}

void
gui_build_gui(void)
{
    int i, j, k;

    /* create a new window */
    gui.window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_realize(gui.window); /* realize here for gdk calls */

    /* set window title */
    gtk_window_set_title(GTK_WINDOW(gui.window), gl.windowtitle);

    /* set the border width of the window. */
    gtk_container_set_border_width(GTK_CONTAINER(gui.window), 8);

    g_signal_connect(G_OBJECT(gui.window), "delete_event", G_CALLBACK(delete_event), NULL);

    gui.table = gtk_table_new(2, 4, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(gui.table), 10);
    gtk_table_set_col_spacings(GTK_TABLE(gui.table), 10);

    gtk_container_add(GTK_CONTAINER(gui.window), gui.table);

    g_signal_connect(G_OBJECT(gui.window), "destroy", G_CALLBACK(destroy), NULL);

    if(gl.do_randr) {
#ifdef HAVE_RANDR
       GdkScreen *screen = gtk_window_get_screen(GTK_WINDOW(gui.window));
       g_signal_connect(screen, "size-changed", G_CALLBACK(xrandreventhandler), NULL);
#endif
    }

    /* Create tooltips */
    gui.mytooltips = gtk_tooltips_new();

    /* Create and attach notebook */
    gui.notebook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(gui.notebook), GTK_POS_TOP);
    gtk_table_attach_defaults(GTK_TABLE(gui.table), gui.notebook, 0, 4, 0, 1);
    gtk_notebook_set_scrollable(GTK_NOTEBOOK(gui.notebook), FALSE);

    gtk_widget_show(gui.notebook);

    if(gl.video_page_type != VIDEO_PAGE_TYPE_NORMAL) gl.noxvdemo = TRUE;

    /* test if we can make a child window */
    if(!gl.noxvdemo) {
       /* Need to flush here so that we have a valid window for testing */
       gdk_flush();
       if(!testxvwindow()) gl.noxvdemo = TRUE;
    }

    /* Create and attach notebook pages */

    /* Keep order of SIS_PAGE_* */

    build_display_mode_page();

    build_crt1_page();

    build_crt2_page();

    build_tv_page();

    build_gamma_page();

    build_video_page();

    build_current_page();

    build_info_page();

    /* Set current page of notebook */
    gtk_notebook_set_current_page(GTK_NOTEBOOK(gui.notebook), SIS_PAGE_DM);

    g_signal_connect(G_OBJECT(gui.notebook), "switch-page", G_CALLBACK(notebook_changed), NULL);

    /* Create and attach buttons */
    gui.button1 = gtk_button_new();
    gui.button2 = gtk_button_new();
    gui.button3 = gtk_button_new();
    gui.button4 = gtk_button_new();

    gui.button1box = xpm_label_box(icon_ok, "_OK", gui.button1, 3, "", "");
    gui.button2box = xpm_label_box(icon_apply, "_Apply", gui.button2, 3, "", "");
    gui.button3box = xpm_label_box(icon_close, "_Close", gui.button3, 3, "", "");
    gui.button4box = xpm_label_box(icon_undo, "_Reset", gui.button4, 3, "", "");

    gtk_widget_show(gui.button1box);
    gtk_widget_show(gui.button2box);
    gtk_widget_show(gui.button3box);
    gtk_widget_show(gui.button4box);

    gtk_widget_set_sensitive(gui.button4, FALSE);

    gtk_container_add(GTK_CONTAINER(gui.button1), gui.button1box);
    gtk_container_add(GTK_CONTAINER(gui.button2), gui.button2box);
    gtk_container_add(GTK_CONTAINER(gui.button3), gui.button3box);
    gtk_container_add(GTK_CONTAINER(gui.button4), gui.button4box);

    g_signal_connect(G_OBJECT (gui.button1), "clicked", G_CALLBACK (ok_clicked), NULL);
    g_signal_connect(G_OBJECT (gui.button2), "clicked", G_CALLBACK (apply_clicked), NULL);
    g_signal_connect(G_OBJECT (gui.button3), "clicked", G_CALLBACK (close_clicked), NULL);
    g_signal_connect(G_OBJECT (gui.button4), "clicked", G_CALLBACK (revert_clicked), NULL);

    g_signal_connect_swapped(G_OBJECT(gui.button3), "clicked", G_CALLBACK (gtk_widget_destroy), gui.window);

    gtk_table_attach(GTK_TABLE(gui.table), gui.button1, 0, 1, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
    gtk_table_attach(GTK_TABLE(gui.table), gui.button2, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
    gtk_table_attach(GTK_TABLE(gui.table), gui.button3, 2, 3, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);
    gtk_table_attach(GTK_TABLE(gui.table), gui.button4, 3, 4, 1, 2, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    gtk_widget_show(gui.button1);
    gtk_widget_show(gui.button2);
    gtk_widget_show(gui.button3);
    gtk_widget_show(gui.button4);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.button1, "Apply changes and exit", "Apply the changes and quit SiSCtrl");
    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.button2, "Apply changes", "Apply changes, but do not quit SiSCtrl");
    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.button3, "Quit SiSCtrl", "Quit SiSCtrl without any changes");
    gtk_tooltips_set_tip(GTK_TOOLTIPS(gui.mytooltips), gui.button4, "Reset to initial settings", "Reset the controls to their initial setting");

    /* Show the table */
    gtk_widget_show(gui.table);

    /* Set window icons */
    {
       GdkPixbuf *icon16_pixbuf, *icon32_pixbuf, *icon48_pixbuf, *icon64_pixbuf;
       GList *icon_list = NULL;

       icon16_pixbuf = gdk_pixbuf_new_from_inline(-1, sisctrl_icon16, FALSE, NULL);
       icon32_pixbuf = gdk_pixbuf_new_from_inline(-1, sisctrl_icon32, FALSE, NULL);
       icon48_pixbuf = gdk_pixbuf_new_from_inline(-1, sisctrl_icon48, FALSE, NULL);
       icon64_pixbuf = gdk_pixbuf_new_from_inline(-1, sisctrl_icon64, FALSE, NULL);

       icon_list = g_list_append(icon_list, icon16_pixbuf);
       icon_list = g_list_append(icon_list, icon32_pixbuf);
       icon_list = g_list_append(icon_list, icon48_pixbuf);
       icon_list = g_list_append(icon_list, icon64_pixbuf);

       gtk_window_set_icon_list((GtkWindow *)gui.window, icon_list);
       gtk_window_set_default_icon_list(icon_list);

       g_list_free(icon_list);

       g_object_unref(icon16_pixbuf);
       g_object_unref(icon32_pixbuf);
       g_object_unref(icon48_pixbuf);
       g_object_unref(icon64_pixbuf);
    }

    gui.waitcursor = gdk_cursor_new(GDK_WATCH);

    /* Reset flag meaning window has never been shown */
    gui.init_done = FALSE;

    /* Initialize system tray icon */
#ifdef USE_STRAY
    gui.visible = TRUE;
    if(gl.iconify) gui.visible = FALSE;

    if(!gl.dontusesystemtray) {
       gui_stray_build_menu();
       stray_init(gl.straytooltip);
    }
#endif

    /* Show window (eventually - depending on iconify-setting) */
    if(!gl.iconify) {
       gui_show_init_window();
    } else {
#ifdef USE_STRAY
       if(gl.dontusesystemtray) {
#endif
	  /* Other way round makes Xv flicker */
	  gui_show_init_window();
          gtk_window_iconify(GTK_WINDOW(gui.window));
#ifdef USE_STRAY
       }
#endif
    }

#if 0 /* This is only for testing!!! It uses the wrong display!!! */
#if defined(HAVE_XINERAMA) && defined(HAVE_XINERAMA_1_2)
    gdk_window_add_filter(/*GTK_WIDGET(gui.window)->window*/NULL, (GdkFilterFunc)eventfilter, NULL);

    XineramaSelectInput(GDK_WINDOW_XDISPLAY(GTK_WIDGET(gui.window)->window),
        GDK_WINDOW_XID(GTK_WIDGET(gui.window)->window),
    	XineramaLayoutChangeNotifyMask);
#endif
#endif

}

void
gui_main(void)
{
   gtk_main();
}

/**************************************************
 *                  SYSTEM TRAY                   *
 **************************************************/

#ifdef USE_STRAY

/* Menu building & handling */

static void
gui_stray_crt1_set_sensitivity(gboolean sensible)
{
   gtk_widget_set_sensitive(gui.stm1_on, sensible);
   gtk_widget_set_sensitive(gui.stm1_lcda, sensible);
   gtk_widget_set_sensitive(gui.stm1_off, sensible);
}

static void
gui_stray_crt1_set_type_menu_index(int index)
{
   gboolean b1 = FALSE, b2 = FALSE, b3 = FALSE;

   if(!gui.stm1_on) return;

   gui_stray_crt1_set_sensitivity(FALSE);

   switch(index) {
   case C1_MI_VGA:
      b1 = TRUE;
      break;
   case C1_MI_LCDA:
      b2 = TRUE;
      break;
   case C1_MI_OFF:
      b3 = TRUE;
      break;
   }

   GTK_CHECK_MENU_ITEM(gui.stm1_on)->active = b1;
   GTK_CHECK_MENU_ITEM(gui.stm1_lcda)->active = b2;
   GTK_CHECK_MENU_ITEM(gui.stm1_off)->active = b3;

   gui_stray_crt1_set_sensitivity(TRUE);
}

static int
gui_stray_crt1_get_type_menu_index(void)
{
   int i;

   if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(gui.stm1_on))) i = C1_MI_VGA;
   else if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(gui.stm1_lcda))) i = C1_MI_LCDA;
   else i = C1_MI_OFF;

   return i;
}

static void
stm1_activated(GtkWidget *widget, gpointer data)
{
   /* Ignore de-activation events */
   if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)) == FALSE) return;

   /* "Remote-control" sisctrl here instead of reinventing the wheel: */
   /* 1. switch to notebook-page 1 */
   gtk_notebook_set_current_page(GTK_NOTEBOOK(gui.notebook), SIS_PAGE_CRT1);
   /* 2. set option menu to our index */
   gui_crt1_set_type_menu_index(gui_stray_crt1_get_type_menu_index());
   /* 3. hit apply */
   sisctrl_do_apply(SIS_PAGE_CRT1);
   /* 4. Reset our menu */
   /* no need, we are being called by the update function for the main window */
}

static void
gui_stray_crt2_showhide_menuitems(void)
{
   if(!gui.om_combobox) return;

   gtk_widget_hide(gui.stm2_lcd);
   gtk_widget_hide(gui.stm2_svhs);
   gtk_widget_hide(gui.stm2_cvbs);
   gtk_widget_hide(gui.stm2_cvsv);
   gtk_widget_hide(gui.stm2_tv);
   gtk_widget_hide(gui.stm2_hivi);
   gtk_widget_hide(gui.stm2_525i);
   gtk_widget_hide(gui.stm2_525p);
   gtk_widget_hide(gui.stm2_625i);
   gtk_widget_hide(gui.stm2_625p);
   gtk_widget_hide(gui.stm2_750p);
   gtk_widget_hide(gui.stm2_1080i);
   gtk_widget_hide(gui.stm2_vga);
   gtk_widget_hide(gui.stm2_scart);
   gtk_widget_hide(gui.stm2_off);

   if(gui.om_mi_visible[C2_MI_TV_LCD]) gtk_widget_show(gui.stm2_lcd);
   if(gui.om_mi_visible[C2_MI_TV_SVHS]) gtk_widget_show(gui.stm2_svhs);
   if(gui.om_mi_visible[C2_MI_TV_CVBS]) gtk_widget_show(gui.stm2_cvbs);
   if(gui.om_mi_visible[C2_MI_TV_CVSV]) gtk_widget_show(gui.stm2_cvsv);
   if(gui.om_mi_visible[C2_MI_TV_TV]) gtk_widget_show(gui.stm2_tv);
   if(gui.om_mi_visible[C2_MI_TV_HIVI]) gtk_widget_show(gui.stm2_hivi);
   if(gui.om_mi_visible[C2_MI_TV_YPBPR]) {
        		gtk_widget_show(gui.stm2_525i);
			gtk_widget_show(gui.stm2_525p);
			if(gl.supportypbpr625i) gtk_widget_show(gui.stm2_625i);
			if(gl.supportypbpr625p) gtk_widget_show(gui.stm2_625p);
			gtk_widget_show(gui.stm2_750p);
			gtk_widget_show(gui.stm2_1080i);
   }
   if(gui.om_mi_visible[C2_MI_TV_VGA]) gtk_widget_show(gui.stm2_vga);
   if(gui.om_mi_visible[C2_MI_TV_SCART]) gtk_widget_show(gui.stm2_scart);
   if(gui.om_mi_visible[C2_MI_TV_OFF]) gtk_widget_show(gui.stm2_off);
}

static void
gui_stray_crt2_set_sensitivity(gboolean sensible)
{
   gtk_widget_set_sensitive(gui.stm2_lcd, sensible);
   gtk_widget_set_sensitive(gui.stm2_svhs, sensible);
   gtk_widget_set_sensitive(gui.stm2_cvbs, sensible);
   gtk_widget_set_sensitive(gui.stm2_cvsv, sensible);
   gtk_widget_set_sensitive(gui.stm2_tv, sensible);
   gtk_widget_set_sensitive(gui.stm2_hivi, sensible);
   gtk_widget_set_sensitive(gui.stm2_525i, sensible);
   gtk_widget_set_sensitive(gui.stm2_525p, sensible);
   gtk_widget_set_sensitive(gui.stm2_625i, sensible);
   gtk_widget_set_sensitive(gui.stm2_625p, sensible);
   gtk_widget_set_sensitive(gui.stm2_750p, sensible);
   gtk_widget_set_sensitive(gui.stm2_1080i, sensible);
   gtk_widget_set_sensitive(gui.stm2_vga, sensible);
   gtk_widget_set_sensitive(gui.stm2_scart, sensible);
   gtk_widget_set_sensitive(gui.stm2_off, sensible);
}

static void
gui_stray_crt2_set_type_menu_index(int index, int index2)
{
   gboolean lcd = FALSE, tv = FALSE, svhs = FALSE, cvbs = FALSE, cvsv = FALSE;
   gboolean hivi = FALSE, y525i = FALSE, y525p = FALSE, y625i = FALSE, y625p = FALSE, y750p = FALSE;
   gboolean y1080i = FALSE, vga = FALSE, scart = FALSE, moff = FALSE;

   if(!gui.stm2_lcd) return;

   gui_stray_crt2_set_sensitivity(FALSE);

   switch(index) {
   case C2_MI_TV_LCD:  lcd = TRUE;  break;
   case C2_MI_TV_SVHS: svhs = TRUE; break;
   case C2_MI_TV_CVBS: cvbs = TRUE; break;
   case C2_MI_TV_CVSV: cvsv = TRUE; break;
   case C2_MI_TV_TV:   tv = TRUE; break;
   case C2_MI_TV_HIVI: hivi = TRUE; break;
   case C2_MI_TV_YPBPR:
      switch(index2) {
         case C2_MI_ST_525I:  y525i = TRUE; break;
         case C2_MI_ST_525P:  y525p = TRUE; break;
         case C2_MI_ST_625I:  y625i = TRUE; break;
         case C2_MI_ST_625P:  y625p = TRUE; break;
         case C2_MI_ST_750P:  y750p = TRUE; break;
         case C2_MI_ST_1080I: y1080i = TRUE; break;
      }
      break;
   case C2_MI_TV_VGA: vga = TRUE; break;
   case C2_MI_TV_SCART: scart = TRUE; break;
   case C2_MI_TV_OFF: moff = TRUE; break;
   }

   GTK_CHECK_MENU_ITEM(gui.stm2_lcd)->active = lcd;
   GTK_CHECK_MENU_ITEM(gui.stm2_svhs)->active = svhs;
   GTK_CHECK_MENU_ITEM(gui.stm2_cvbs)->active = cvbs;
   GTK_CHECK_MENU_ITEM(gui.stm2_cvsv)->active = cvsv;
   GTK_CHECK_MENU_ITEM(gui.stm2_tv)->active = tv;
   GTK_CHECK_MENU_ITEM(gui.stm2_hivi)->active = hivi;
   GTK_CHECK_MENU_ITEM(gui.stm2_525i)->active = y525i;
   GTK_CHECK_MENU_ITEM(gui.stm2_525p)->active = y525p;
   GTK_CHECK_MENU_ITEM(gui.stm2_625i)->active = y625i;
   GTK_CHECK_MENU_ITEM(gui.stm2_625p)->active = y625p;
   GTK_CHECK_MENU_ITEM(gui.stm2_750p)->active = y750p;
   GTK_CHECK_MENU_ITEM(gui.stm2_1080i)->active = y1080i;
   GTK_CHECK_MENU_ITEM(gui.stm2_vga)->active = vga;
   GTK_CHECK_MENU_ITEM(gui.stm2_scart)->active = scart;
   GTK_CHECK_MENU_ITEM(gui.stm2_off)->active = moff;

   gui_stray_crt2_set_sensitivity(TRUE);
}

static void
stm2_activated(GtkWidget *widget, int index, int index2)
{
   /* Ignore de-activation events */
   if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)) == FALSE) return;

   /* "Remote-control" sisctrl here instead of reinventing the wheel: */
   /* 1. switch to notebook-page 2 */
   gtk_notebook_set_current_page(GTK_NOTEBOOK(gui.notebook), SIS_PAGE_CRT2);
   /* 2. set option menu to our index */
   gui_crt2_set_type_menu_index(index);
   if(index == C2_MI_TV_YPBPR) {
      gui_crt2_set_signal_menu_index(index2);
   }
   /* 3. hit apply */
   sisctrl_do_apply(SIS_PAGE_CRT2);
   /* 4. Reset our menu */
   /* no need, we are being called by the update function for the main window */
}

static void
stm2_lcd_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_LCD, 0);
}

static void
stm2_svhs_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_SVHS, 0);
}

static void
stm2_cvbs_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_CVBS, 0);
}

static void
stm2_cvsv_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_CVSV, 0);
}

static void
stm2_tv_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_TV, 0);
}

static void
stm2_hivi_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_HIVI, 0);
}

static void
stm2_525i_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_YPBPR, C2_MI_ST_525I);
}

static void
stm2_525p_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_YPBPR, C2_MI_ST_525P);
}

static void
stm2_625i_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_YPBPR, C2_MI_ST_625I);
}

static void
stm2_625p_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_YPBPR, C2_MI_ST_625P);
}

static void
stm2_750p_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_YPBPR, C2_MI_ST_750P);
}

static void
stm2_1080i_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_YPBPR, C2_MI_ST_1080I);
}

static void
stm2_vga_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_VGA, 0);
}

static void
stm2_scart_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_SCART, 0);
}

static void
stm2_off_activated(GtkWidget *widget, gpointer data)
{
   stm2_activated(widget, C2_MI_TV_OFF, 0);
}

static void
stm2_det_toggled(GtkWidget *widget, gpointer data)
{
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.om_check),
   		gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)));
}

static void
gui_stray_video_set_active(int num)
{
   gboolean crt1 = (num == 1) ? TRUE : FALSE;
   gboolean crt2 = (num == 2) ? TRUE : FALSE;

   if(!gui.stmv_crt1) return;

   GTK_CHECK_MENU_ITEM(gui.stmv_crt1)->active = crt1;
   GTK_CHECK_MENU_ITEM(gui.stmv_crt2)->active = crt2;
}

static void
gui_stray_video_set_sensitive(int enable)
{
   gtk_widget_set_sensitive(gui.stmv_crt1, enable ? TRUE : FALSE);
   gtk_widget_set_sensitive(gui.stmv_crt2, enable ? TRUE : FALSE);
}

static void
stmv_activated(GtkWidget *widget, gpointer data)
{
   gboolean b1 = FALSE, b2 = FALSE;

   if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget))) {
      b1 = TRUE;
   } else {
      b2 = TRUE;
   }
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt1), b1);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt2), b2);
}

static void
gui_stray_make_image_menu_item(GtkWidget *menuentry, char **xpmdata)
{
   GdkPixmap *icon = NULL;
   GdkBitmap *mask;
   GtkWidget *image;

   icon = gdk_pixmap_create_from_xpm_d(gui.window->window, &mask, NULL, xpmdata);
   image = gtk_image_new_from_pixmap(icon, mask);
   g_object_unref(G_OBJECT(icon));
   g_object_unref(G_OBJECT(mask));
   gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuentry), image);
}

static void
gui_stray_build_menu(void)
{
   GtkWidget *entry;
   GSList *group = NULL;

   gui.st_menu = gtk_menu_new();
   gtk_menu_set_title(GTK_MENU(gui.st_menu), "SiSCtrl");

   entry = gtk_menu_item_new_with_label(gl.straytooltip);
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);
   gtk_widget_set_sensitive(entry, FALSE);

   entry = gtk_separator_menu_item_new();
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);

   /* CRT1 menu and submenu */
   entry = gtk_image_menu_item_new_with_label("CRT1");
   gui_stray_make_image_menu_item(entry, icon_crt1);
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);

   switch(gl.crt1_page_type) {

   case CRT1_PAGE_TYPE_NORMAL:
      gui.st_menu1 = gtk_menu_new();
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(entry), gui.st_menu1);

      gui.stm1_on = gtk_radio_menu_item_new_with_label(NULL, gl.supportlcda ? "VGA" : "On");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm1_on));
      gui.stm1_lcda = gtk_radio_menu_item_new_with_label(group, gl.supportdvi ? "DVI-D/LCD" : "LCD");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm1_lcda));
      gui.stm1_off = gtk_radio_menu_item_new_with_label(group, "Off");

      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu1), gui.stm1_on);
      gtk_widget_show(gui.stm1_on);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu1), gui.stm1_lcda);
      if(gl.supportlcda) gtk_widget_show(gui.stm1_lcda);
      else gtk_widget_hide(gui.stm1_lcda);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu1), gui.stm1_off);
      gtk_widget_show(gui.stm1_off);

      gui_stray_crt1_set_type_menu_index(sisctrl_get_current_crt1type_index());

      g_signal_connect(G_OBJECT(gui.stm1_on), "activate", G_CALLBACK(stm1_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm1_lcda), "activate", G_CALLBACK(stm1_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm1_off), "activate", G_CALLBACK(stm1_activated), NULL);
      break;

   default:
      gtk_widget_set_sensitive(entry, FALSE);
   }

   /* CRT2 menu and submenu */
   entry = gtk_image_menu_item_new_with_label("CRT2");
   gui_stray_make_image_menu_item(entry, icon_crt2);
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);

   switch(gl.crt2_page_type) {

   case CRT2_PAGE_TYPE_NORMAL:
      gui.st_menu2 = gtk_menu_new();
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(entry), gui.st_menu2);

      gui.stm2_lcd = gtk_radio_menu_item_new_with_label(NULL, gl.supportdvi ? "DVI-D/LCD" : "LCD");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_lcd));
      gui.stm2_svhs = gtk_radio_menu_item_new_with_label(group, "S-Video");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_svhs));
      gui.stm2_cvbs = gtk_radio_menu_item_new_with_label(group, "Composite");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_cvbs));
      gui.stm2_cvsv = gtk_radio_menu_item_new_with_label(group, "S-Video + Composite");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_cvsv));
      gui.stm2_tv = gtk_radio_menu_item_new_with_label(group, "TV");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_tv));
      gui.stm2_hivi = gtk_radio_menu_item_new_with_label(group, "HiVision 1080i");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_hivi));
      gui.stm2_525i = gtk_radio_menu_item_new_with_label(group, "YPbPr 480i/525i");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_525i));
      gui.stm2_525p = gtk_radio_menu_item_new_with_label(group, "YPbPr 480p/525p");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_525p));
      gui.stm2_625i = gtk_radio_menu_item_new_with_label(group, "YPbPr 576i/625i");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_625i));
      gui.stm2_625p = gtk_radio_menu_item_new_with_label(group, "YPbPr 576p/625p");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_625p));
      gui.stm2_750p = gtk_radio_menu_item_new_with_label(group, "YPbPr 720p/750p");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_750p));
      gui.stm2_1080i = gtk_radio_menu_item_new_with_label(group, "YPbPr 1080i");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_1080i));
      gui.stm2_vga = gtk_radio_menu_item_new_with_label(group, "DVI-A/VGA");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_vga));
      gui.stm2_scart = gtk_radio_menu_item_new_with_label(group, "SCART");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stm2_scart));
      gui.stm2_off = gtk_radio_menu_item_new_with_label(group, "Off");

      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_lcd);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_svhs);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_cvbs);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_cvsv);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_tv);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_hivi);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_525i);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_525p);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_625i);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_625p);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_750p);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_1080i);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_vga);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_scart);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_off);

      entry = gtk_separator_menu_item_new();
      gtk_widget_show(entry);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), entry);

      gui.stm2_det = gtk_check_menu_item_new_with_label("Detected only");
      gtk_widget_show(gui.stm2_det);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu2), gui.stm2_det);

      gui_stray_crt2_showhide_menuitems();

      gui_stray_crt2_set_type_menu_index(gui_crt2_get_type_menu_index(),
      				gui_crt2_get_signal_menu_index());

      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gui.stm2_det),
      				gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.om_check)));

      g_signal_connect(G_OBJECT(gui.stm2_lcd), "activate", G_CALLBACK(stm2_lcd_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_svhs), "activate", G_CALLBACK(stm2_svhs_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_cvbs), "activate", G_CALLBACK(stm2_cvbs_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_cvsv), "activate", G_CALLBACK(stm2_cvsv_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_tv), "activate", G_CALLBACK(stm2_tv_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_hivi), "activate", G_CALLBACK(stm2_hivi_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_525i), "activate", G_CALLBACK(stm2_525i_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_525p), "activate", G_CALLBACK(stm2_525p_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_625i), "activate", G_CALLBACK(stm2_625i_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_625p), "activate", G_CALLBACK(stm2_625p_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_750p), "activate", G_CALLBACK(stm2_750p_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_1080i), "activate", G_CALLBACK(stm2_1080i_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_vga), "activate", G_CALLBACK(stm2_vga_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_scart), "activate", G_CALLBACK(stm2_scart_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_off), "activate", G_CALLBACK(stm2_off_activated), NULL);
      g_signal_connect(G_OBJECT(gui.stm2_det), "toggled", G_CALLBACK(stm2_det_toggled), NULL);
      break;

   default:
      gtk_widget_set_sensitive(entry, FALSE);
   }

   /* Video menu and submenu */
   entry = gtk_image_menu_item_new_with_label("Video");
   gui_stray_make_image_menu_item(entry, icon_video);
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);

   if( (gl.video_page_type == VIDEO_PAGE_TYPE_NORMAL) &&
       (gl.needswitchcrt) ) {

      gui.st_menuv = gtk_menu_new();
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(entry), gui.st_menuv);

      gui.stmv_crt1 = gtk_radio_menu_item_new_with_label(NULL, "CRT1");
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gui.stmv_crt1));
      gui.stmv_crt2 = gtk_radio_menu_item_new_with_label(group, "CRT2");
      gtk_widget_show(gui.stmv_crt1);
      gtk_widget_show(gui.stmv_crt2);

      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menuv), gui.stmv_crt1);
      gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menuv), gui.stmv_crt2);

      gui_stray_video_set_active( (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.vi_radio_crt1))) ?
					1 : 2);
      gui_stray_video_set_sensitive(GTK_WIDGET_SENSITIVE(gui.vi_radio_crt1) ? 1 : 0);

      g_signal_connect(G_OBJECT(gui.stmv_crt1), "activate", G_CALLBACK(stmv_activated), NULL);

   } else {

      gtk_widget_set_sensitive(entry, FALSE);

   }

   /* Remaining menu items */
   entry = gtk_separator_menu_item_new();
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);

   entry = gtk_image_menu_item_new_with_label("Close");
   gui_stray_make_image_menu_item(entry, icon_close);
   gtk_menu_shell_append(GTK_MENU_SHELL(gui.st_menu), entry);
   gtk_widget_show(entry);
   g_signal_connect(G_OBJECT(entry), "activate", G_CALLBACK(close_clicked), NULL);
   g_signal_connect_swapped(G_OBJECT(entry), "activate", G_CALLBACK(gtk_widget_destroy), gui.window);
}

void
gui_embedding_failed(void)
{
   /* This is called if the system tray icon failed
    * to embed (eg if no system tray application is
    * available). Now what? If we were in iconify-mode
    * and our window has not been shown yet, do it
    * now, and iconify it.
    */
   if(gl.iconify && !gui.init_done) {
      gui_show_init_window();
      gtk_window_iconify(GTK_WINDOW(gui.window));
   }
}

void
gui_switch_window_visibility(gboolean visible)
{
   static gint x = -1, y = - 1;

   if(gui.window) {
      if(visible) {
         if((!GTK_WIDGET_VISIBLE(gui.window)) && x >= 0 && y >= 0) {
	    gtk_window_move(GTK_WINDOW(gui.window), x, y);
	 }
	 if(!gui.init_done) {
	    gui_show_init_window();
	 } else {
            gtk_window_present(GTK_WINDOW(gui.window));
	 }
      } else {
         gtk_window_get_position(GTK_WINDOW(gui.window), &x, &y);
         gtk_widget_hide(gui.window);
      }
   }
}

/* System tray callbacks */

/**************************************************
 *        SYSTEM TRAY: Return Pixbuf for icon     *
 **************************************************/

GdkPixbuf *
gui_stray_get_icon_pixbuf(void)
{
   GdkPixbuf *pixbuf;

   pixbuf = gdk_pixbuf_new_from_inline(-1, sisctrl_icon16, FALSE, NULL);

   return pixbuf;
}

/**************************************************
 *          SYSTEM TRAY: Being removed            *
 **************************************************/

void
gui_stray_being_removed(void)
{
    /* This might be called from external sources, if
     * something is wrong with the system tray. In this case,
     * check that our window is visible so that the
     * user can do something about it...
     */
   if(gui.visible == FALSE) {
      gui.visible = TRUE;
      gui_switch_window_visibility(gui.visible);
   }
}

/**************************************************
 *        SYSTEM TRAY: Left mouse button          *
 **************************************************/

void
gui_stray_left_mouse_button(void)
{
   gui.visible = !gui.visible;

   gui_switch_window_visibility(gui.visible);
}

/**************************************************
 *         SYSTEM TRAY: Right mouse button        *
 **************************************************/

void
gui_stray_right_mouse_button(void)
{

   /* Update CRT2 menu */
   gui_stray_crt2_showhide_menuitems();

   gtk_menu_popup(GTK_MENU(gui.st_menu), NULL, NULL,
#if GTK_CHECK_VERSION(2,2,0)
			stray_position_menu,
#else
			NULL,
#endif
			NULL, 0, gtk_get_current_event_time());
}

#endif









