/*--------------------------------------------------------------------
 *	$Id: psbasemap.c,v 1.25 2005/12/17 05:59:22 pwessel Exp $
 *
 *	Copyright (c) 1991-2006 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/*
 * psbasemap plots out a basemap for the given area using the specified map
 * projection.
 *
 * Author:	Paul Wessel
 * Date:	08-JUL-2000
 * Version:	4
 *
 */
 
#include "gmt.h"
#include "pslib.h"

int main (int argc, char **argv)
{
	int i;

	BOOLEAN error = FALSE, paint = FALSE, set_z = FALSE;

	double w = 0.0, e = 0.0, s = 0.0, n = 0.0, new_z_level = 0.0;

	struct GMT_FILL fill;

	struct MAP_SCALE ms;
	struct MAP_ROSE mr;

	GMT_init_fill (&fill, -1, -1, -1);
	ms.plot = mr.plot = FALSE;

	argc = GMT_begin (argc, argv);

	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-') {
			error = TRUE;
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Options must start with leading -. Your argument \"%s\" is unrecognized\n", GMT_program, argv[i]);
			continue;
		}

		switch (argv[i][1]) {

			/* Common parameters */

			case 'B':
			case 'J':
			case 'K':
			case 'O':
			case 'P':
			case 'R':
			case 'U':
			case 'V':
			case 'X':
			case 'x':
			case 'Y':
			case 'y':
			case 'c':
			case '\0':
				error += GMT_get_common_args (argv[i], &w, &e, &s, &n);
				break;

			/* Supplemental options */

			case 'E':
				sscanf (&argv[i][2], "%lf/%lf", &z_project.view_azimuth, &z_project.view_elevation);
				break;

			case 'G':
				if (GMT_getfill (&argv[i][2], &fill)) {
					GMT_fill_syntax ('G');
					error++;
				}
				paint = TRUE;
				break;

			case 'L':
				error += GMT_getscale (&argv[i][2], &ms);
				break;

			case 'T':
				error += GMT_getrose (&argv[i][2], &mr);
				break;

			case 'Z':
				if (argv[i][2]) {
					new_z_level = atof (&argv[i][2]);
					set_z = TRUE;
				}
				break;

			/* Illegal options */

			default:
				error = TRUE;
				GMT_default_error (argv[i][1]);
				break;
		}
	}

	if (GMT_quick || argc == 1) {
		fprintf (stderr,"psbasemap %s - To plot PostScript basemaps\n\n", GMT_VERSION);
		fprintf (stderr, "usage: psbasemap -B<tickinfo> -J<params> -R<west/east/south/north> [-Eaz/el] [-G<fill>]\n");
		fprintf (stderr, "\t[-K] [-L[f][x]<lon0>/<lat0>[/<slon>]/<slat>/<length>[m|n|k][:label:<just>][+p<pen>][+f<fill>]]\n");
		fprintf (stderr, "\t[-O] [-P] [-T[f|m][x]<lon0>/<lat0>/<size>[/<info>][:w,e,s,n:][+<gints>[/<mints>]]] [-U[dx/dy/][label]] [-V]\n");
		fprintf (stderr, "\t[-X<xshift>] [-Y<yshift>] [-Z<zlevel>] [-c<ncopies>]\n\n");

		if (GMT_quick) exit (EXIT_FAILURE);

		GMT_explain_option ('B');
		GMT_explain_option ('J');
		GMT_explain_option ('R');
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "\t-E set azimuth and elevation of viewpoint for 3-D perspective [180/90]\n");
		fprintf (stderr, "\t-G fill inside of basemap with the specified fill\n");
		GMT_explain_option ('K');
		fprintf (stderr, "\t-L draws a simple map scaLe centered on <lon0>/<lat0>.  Use -Lx to specify Cartesian\n");
		fprintf (stderr, "\t   coordinates instead.  Scale is calculated at latitude <slat>; optionally give longitude\n");
		fprintf (stderr, "\t   <slon> [Default is central longitude].  <length> is in km, or [nautical] miles if [n] m\n");
		fprintf (stderr, "\t   is appended.  -Lf draws a \"fancy\" scale [Default is plain]. By default, the label is set\n");
		fprintf (stderr, "\t   to the distance unit and placed on top [t].  Use the :label:<just> mechanism to specify another\n");
		fprintf (stderr, "\t   label (or - to keep the default) and placement (t,b,l,r, and u - to use the label as a unit.\n");
		fprintf (stderr, "\t   Append +p<pen> and/or +f<fill> to draw/paint a rectangle beneath the scale [no rectangle]\n");
		GMT_explain_option ('O');
		GMT_explain_option ('P');
		fprintf (stderr, "\t-T draws a north-poinTing rose centered on <lon0>/<lat0>.  Use -Tx to specify Cartesian\n");
		fprintf (stderr, "\t   coordinates instead.  -Tf draws a \"fancy\" rose [Default is plain].  Give rose diameter\n");
		fprintf (stderr, "\t   <size> and optionally the west, east, south, north labels desired [W,E,S,N].\n");
		fprintf (stderr, "\t   For fancy rose, specify as <info> the kind you want: 1 draws E-W, N-S directions [Default],\n");
		fprintf (stderr, "\t   2 adds NW-SE and NE-SW, while 3 adds WNW-ESE, NNW-SSE, NNE-SSW, and ENE-WSW.\n");
		fprintf (stderr, "\t   For Magnetic compass rose, specify -Tm.  Use the optional <info> = <dec>/<dlabel> (where <dec> is\n");
		fprintf (stderr, "\t   the magnetic declination and <dlabel> is a label for the magnetic compass needle) to plot\n");
		fprintf (stderr, "\t   directions to both magnetic and geographic north [Default is just geographic].\n");
		fprintf (stderr, "\t   If the North label = \'*\' then a north star is plotted instead of the label.\n");
		fprintf (stderr, "\t   Append +<gints>/<mints> to override default annotation/tick interval(s) [10/5/1/30/5/1].\n");
		GMT_explain_option ('U');
		GMT_explain_option ('V');
		GMT_explain_option ('X');
		fprintf (stderr, "\t-Z For 3-D plots: Set the z-level of map [0]\n");
		GMT_explain_option ('c');
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}

	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", GMT_program);
		error++;
	}
	if (!(frame_info.plot || ms.plot || mr.plot || paint)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify at least one of -B, -G, -L, -T\n", GMT_program);
		error++;
	}
	if (z_project.view_azimuth > 360.0 || z_project.view_elevation <= 0.0 || z_project.view_elevation > 90.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option:  Enter azimuth in 0-360 range, elevation in 0-90 range\n", GMT_program);
		error++;
	}

	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands4 */

	if (gmtdefs.verbose) fprintf (stderr, "psbasemap: Constructing basemap\n");

	GMT_map_setup (w, e, s, n);

	ps_plotinit (CNULL, gmtdefs.overlay, gmtdefs.page_orientation, gmtdefs.x_origin, gmtdefs.y_origin,
		gmtdefs.global_x_scale, gmtdefs.global_y_scale, gmtdefs.n_copies,
		gmtdefs.dpi, GMT_INCH, gmtdefs.paper_width, gmtdefs.page_rgb, gmtdefs.encoding.name, GMT_epsinfo (argv[0]));

	GMT_echo_command (argc, argv);
	if (gmtdefs.unix_time) GMT_timestamp (argc, argv);

	if (project_info.three_D) ps_transrotate (-z_project.xmin, -z_project.ymin, 0.0);

	if (paint) {
		double *x, *y;
		int np, donut;
		np = GMT_map_clip_path (&x, &y, &donut);
		GMT_fill (x, y, (1 + donut) * np, &fill, FALSE);
		GMT_free ((void *)x);
		GMT_free ((void *)y);
	}

	if (set_z) project_info.z_level = new_z_level;

	GMT_map_basemap ();

	if (ms.plot) GMT_draw_map_scale (&ms);

	if (mr.plot) GMT_draw_map_rose (&mr);

	if (project_info.three_D) ps_rotatetrans (z_project.xmin, z_project.ymin, 0.0);
	ps_plotend (gmtdefs.last_page);

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}
