/*--------------------------------------------------------------------
 *	$Id: project.c,v 1.29 2005/12/17 05:59:22 pwessel Exp $
 *
 *	Copyright (c) 1991-2006 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/*
 * project.c
 * reads (x,y,[z]) data and writes some combination of (x,y,z,p,q,u,v),
 * where p,q is the distance along,across the track of the projection of (x,y),
 * and u,v are the un-transformed (x,y) coordinates of the projected position.
 * Can also create (x,y) along track.
  
   Author: 	Walter H. F. Smith
   Date:	19 April, 1988.
   Modified:	4 December 1988, to be more flexible.
   Complete rebuild 22 June, 1989 to use vector products and do more things.
   version 2.0
   		23-FEB-1998	PW: Added support for multiple files, multi-segment formats
				and binary i/o.  Old -M renamed -Q.
		03-NOV-1998	PW: Can read any number of data columns; z in -Fz refers to
				all these columns in the output.
   Version:	3.4		PW: Fixed problem with small circle distances
   Version:	4
*/

#include "gmt.h"

struct DATA {
        double  a[6];
	double *z;
	char *t;
};

int	compare_distances(const void *point_1, const void *point_2);
int	do_flat_earth();
int	solve_right_spherical_triangle();
int	sphere_azim_dist();
double	oblique_setup(double plat, double plon, double *p, double clat, double clon, double *c, int c_given, int rads);
void	oblique_transform(double xlat, double xlon, double *x_t_lat, double *x_t_lon, double *p, double *c, int rads);
void	make_euler_matrix(double *p, double *e, double *theta, int rads);
void	matrix_3v(double *a, double *x, double *b);
void	matrix_2v(double *a, double *x, double *b);
void	sphere_project_setup(double alat, double alon, double *a, double blat, double blon, double *b, double *azim, double *p, double *c, int two_pts, int rads);
void	flat_project_setup(double alat, double alon, double blat, double blon, double plat, double plon, double *azim, double *e, int two_pts, BOOLEAN pole_set);
void	copy_text_from_col3 (char *line, char *z_cols);

int main (int argc, char **argv)
{
	int	i, j, k, n_definitions, n_outputs, n_used, n_total_read, n_alloc = GMT_CHUNK;
	int	nc = 0, ne = 0, np = 0, nl = 0, nw = 0, n_files = 0, fno, n_args, n_fields, n_expected_fields;
	int	n_total_used = 0, n_z = 0, n_items, output_choice[7];

	double	xx, yy, x_a, y_a, x_b, y_b, x_p, y_p, cos_theta, sin_theta, sin_lat_to_pole = 1.0;
	double	theta, d_inc = 0.0, d_along, *in, *out = (double *)NULL;

	double	azimuth = 0.0, l_min = 0.0, l_max = 0.0, w_min = 0.0, w_max = 0.0;
	double	a[3], b[3], x[3], xt[3], pole[3], center[3], e[9];

	BOOLEAN	check_length, check_width, convert_units, dateline, error, find_new_point, flat_earth, first = TRUE;
	BOOLEAN generate, greenwich, origin_set = FALSE, pole_set, rads, sort_output, stay_within, two_points;
	BOOLEAN nofile = TRUE, done = FALSE, want_z_output = FALSE, pure_ascii, skip;

	FILE *fp = NULL;

	char	modifier, record_str[BUFSIZ], heading[7][GMT_TEXT_LEN], txt_a[GMT_LONG_TEXT], txt_b[GMT_LONG_TEXT];

	struct DATA *p_data;

	argc = GMT_begin (argc, argv);

	check_length = check_width = dateline = error = find_new_point = flat_earth = generate = greenwich = FALSE;
	pole_set = sort_output = stay_within = two_points = convert_units = FALSE;
	rads = TRUE;
	x_a = x_b = x_p = y_a = y_b = y_p = 0.0;
	n_definitions = 0;
	n_outputs = 0;
	j = 1;
	for (i = 0; i < 7; i++) output_choice[i] = 0;

	/* New in GMT 4: Must process -N before the rest to ensure col_types are set properly (there is no -H here)  */
	for (i = 1; i < argc; i++) if (!strcmp (argv[i], "-N")) flat_earth = TRUE;
	if (!flat_earth) {
		GMT_io.in_col_type[0] = GMT_IS_LON;
		GMT_io.in_col_type[1] = GMT_IS_LAT;
	}

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {


				/* Common parameters */

				case 'H':
				case 'V':
				case ':':
				case 'b':
				case 'f':
				case '\0':
					error += GMT_get_common_args (argv[i], 0, 0, 0, 0);
					break;

				/* Supplemental parameters */

				case 'F':
					for (j = 2, k = 0; argv[i][j]; j++, k++) {
						switch (argv[i][j]) {
							case 'z':	/* Special flag, can mean any number of z columns */
								output_choice[k] = -1;
								want_z_output = TRUE;
								break;
							case 'x':
								output_choice[k] = 0;
								break;
							case 'y':
								output_choice[k] = 1;
								break;
							case 'p':
								output_choice[k] = 2;
								break;
							case 'q':
								output_choice[k] = 3;
								break;
							case 'r':
								output_choice[k] = 4;
								find_new_point = TRUE;
								break;
							case 's':
								output_choice[k] = 5;
								find_new_point = TRUE;
								break;
							default:
								fprintf (stderr, "%s: GMT SYNTAX ERROR -F option:  Unrecognized choice %c\n", GMT_program, argv[i][j]);
								error = TRUE;
						}
						n_outputs++;
					}
					break;
				case 'A':
					azimuth = atof(&argv[i][2]);
					n_definitions++;
					break;
				case 'C':
					nc = sscanf(&argv[i][2], "%[^/]/%s", txt_a, txt_b);
					error += GMT_verify_expectations (GMT_io.in_col_type[0], GMT_scanf_arg (txt_a, GMT_io.in_col_type[0], &x_a), txt_a);
					error += GMT_verify_expectations (GMT_io.in_col_type[1], GMT_scanf_arg (txt_b, GMT_io.in_col_type[1], &y_a), txt_b);
					if (error) fprintf (stderr, "%s: GMT SYNTAX ERROR -C option:  Undecipherable argument %s\n", GMT_program, &argv[i][2]);
					origin_set = TRUE;
					break;
				case 'D':
					modifier = argv[i][2];
					if (modifier == 'D' || modifier == 'd') {
						dateline = TRUE;
					}
					else if (modifier == 'g' || modifier == 'G') {
						greenwich = TRUE;
					}
					else if (modifier) {
						fprintf (stderr, "%s: GMT SYNTAX ERROR -D option:  Unrecognized modifier %c\n", GMT_program, modifier);
						error = TRUE;
					}
					break;
				case 'E':
					ne = sscanf(&argv[i][2], "%[^/]/%s", txt_a, txt_b);
					error += GMT_verify_expectations (GMT_io.in_col_type[0], GMT_scanf_arg (txt_a, GMT_io.in_col_type[0], &x_b), txt_a);
					error += GMT_verify_expectations (GMT_io.in_col_type[1], GMT_scanf_arg (txt_b, GMT_io.in_col_type[1], &y_b), txt_b);
					if (error) fprintf (stderr, "%s: GMT SYNTAX ERROR -E option:  Undecipherable argument %s\n", GMT_program, &argv[i][2]);
					two_points = TRUE;
					n_definitions++;
					break;
				case 'G':
					generate = TRUE;
					d_inc = atof(&argv[i][2]);
					break;
				case 'L':
					check_length = TRUE;
					modifier = argv[i][2]; 
					if (modifier == 'W' || modifier == 'w') {
						stay_within = TRUE;
					}
					else {
						nl = sscanf(&argv[i][2], "%lf/%lf", &l_min, &l_max);
					}
					break;
				case 'M':
					GMT_multisegment (&argv[i][2]);
					break;
				case 'N': /* Handled above but still in argv */
					break;
				case 'Q':
					convert_units = TRUE;
					break;
				case 'S':
					sort_output = TRUE;
					break;
				case 'T':
					np = sscanf(&argv[i][2], "%[^/]/%s", txt_a, txt_b);
					error += GMT_verify_expectations (GMT_io.in_col_type[0], GMT_scanf_arg (txt_a, GMT_io.in_col_type[0], &x_p), txt_a);
					error += GMT_verify_expectations (GMT_io.in_col_type[1], GMT_scanf_arg (txt_b, GMT_io.in_col_type[1], &y_p), txt_b);
					if (error) fprintf (stderr, "%s: GMT SYNTAX ERROR -T option:  Undecipherable argument %s\n", GMT_program, &argv[i][2]);
					pole_set = TRUE;
					n_definitions++;
					break;
				case 'W':
					nw = sscanf(&argv[i][2], "%lf/%lf", &w_min, &w_max);
					check_width = TRUE;
					break;
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else
			n_files++;
	}

	if (argc == 1 || GMT_quick) {
		fprintf (stderr, "project %s - project data onto line or great circle, generate track, or translate coordiantes\n\n", GMT_VERSION);
		fprintf(stderr,"usage:	project [files] -C<ox>/<oy> [-A<azimuth>] [-D<d_or_g>] [-E<bx>/<by>]\n");
		fprintf(stderr,"\t[-F<flags>] [-G<dist>] [-H[<nrec>]] [-L[w][<l_min>/<l_max>]]\n");
		fprintf(stderr,"\t[-M[<flag>]] [-N] [-Q] [-S] [-T<px>/<py>] [-V] [-W<w_min>/<w_max>]\n");
		fprintf(stderr,"\t[-:] [-bi[s][<n>]] [-bo[s][<n>]] [-f[i|o]<colinfo>]\n\n");

		if (GMT_quick) exit (EXIT_FAILURE);

		fprintf(stderr,"\tproject will read stdin or file, and does not want input if -G option.\n");
		fprintf(stderr,"\tThe projection may be defined in (only) one of three ways:\n");
		fprintf(stderr,"\t  (1) by a center -C and an azimuth -A,\n");
		fprintf(stderr,"\t  (2) by a center -C and end point of the path -E,\n");
		fprintf(stderr,"\t  (3) by a center -C and a roTation pole position -T.\n");
		fprintf(stderr,"\t  In a spherical projection [default], all cases place the central meridian\n");
		fprintf(stderr,"\t  of the transformed coordinates (p,q) through -C (p = 0 at -C).  The equator\n");
		fprintf(stderr,"\t  of the (p,q) system (line q = 0) passes through -C and makes an angle\n");
		fprintf(stderr,"\t  <azimuth> with North (case 1), or passes through -E (case 2), or is\n");
		fprintf(stderr,"\t  determined by the pole -T (case 3).  In (3), point -C need not be on equator.\n");
		fprintf(stderr,"\t  In a cartesian [-N option] projection, p = q = 0 at -O in all cases;\n");
		fprintf(stderr,"\t  (1) and (2) orient the p axis, while (3) orients the q axis.\n\n");
		fprintf(stderr,"\t-C<ox>/<oy> sets the location of the center.\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf(stderr,"\t-A<azimuth> sets the option (1) Azimuth, (degrees CW from North).\n");
		fprintf(stderr,"\t-D will force the location of the Discontinuity in the r coordinate;\n");
		fprintf(stderr,"\t  -Dd (dateline) means [-180 < r < 180], -Dg (greenwich) means [0 < r < 360].\n");
		fprintf(stderr,"\t  The default does not check; in spherical case this usually results in [-180,180].\n");
		fprintf(stderr,"\t-E<bx>/<by> sets the option (2) location of end point E.\n");
		fprintf(stderr,"\t-Fflags: Indicate what output you want as one or more of xyzpqrs in any order;\n");
		fprintf(stderr,"\t  where x,y,[z] refer to input data locations and optional values,\n");
		fprintf(stderr,"\t  p,q are the coordinates of x,y in the projection's coordinate system,\n");
		fprintf(stderr,"\t  r,s is the projected position of x,y (taking q = 0) in the (x,y) coordinate system.\n");
		fprintf(stderr,"\t  p,q may be scaled from degrees into kilometers by the -Q option.  See -L, -Q, -W.\n");
		fprintf(stderr,"\t  Note z refers to all input data columns beyond the required x,y\n");
		fprintf(stderr,"\t  [Default is all fields, i.e. -Fxyzpqrs]\n");
		fprintf(stderr,"\t  If -G is set, -F is not available and output defaults to rsp\n");
		fprintf(stderr,"\t-G means Generate (r,s,p) points along profile every <dist> units. (No input data used.)\n");
		fprintf(stderr,"\t   If E given, will generate from C to E; else must give -L<l_min>/<l_max> for length.\n");
		GMT_explain_option ('H');
		fprintf(stderr,"\t-L Check the Length along the projected track and use only certain points.\n");
		fprintf(stderr,"\t  -Lw will use only those points Within the span from C to E (Must have set -E).\n");
		fprintf(stderr,"\t  -L<l_min>/<l_max> will only use points whose p is [l_min <= p <= l_max].\n");
		fprintf(stderr,"\t  Default uses all points.  Note p = 0 at C and increases toward E in azim direction.\n");
		GMT_explain_option ('M');
		fprintf(stderr,"\t-N means Flat_earth; a cartesian projection is made.  Default is spherical.\n");
		fprintf(stderr,"\t-Q means convert to Map units, so x,y,r,s are degrees,\n");
		fprintf(stderr,"\t  while p,q,dist,l_min,l_max,w_min,w_max are km.\n");
		fprintf(stderr,"\t  If not set, then p,q,dist,l_min,l_max,w_min,w_max are assumed to be in same units as x,y,r,s.\n");
		fprintf(stderr,"\t-S means the output should be Sorted into increasing p value.\n");
		fprintf(stderr,"\t-T<px>/<py> sets the option (3) location of the roTation pole to the projection.\n");
		GMT_explain_option ('V');
		fprintf(stderr,"\t-W Check the width across the projected track and use only certain points.\n");
		fprintf(stderr,"\t  This will use only those points whose q is [w_min <= q <= w_max].\n");
		fprintf(stderr,"\t  Note that q is positive to your LEFT as you walk from C toward E in azim direction.\n");
		GMT_explain_option (':');
		GMT_explain_option ('i');
		GMT_explain_option ('n');
		fprintf(stderr,"\t  Default is 2 input columns (x, y)\n");
		GMT_explain_option ('o');
		GMT_explain_option ('n');
		GMT_explain_option ('f');
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}

	if ( !(origin_set && nc == 2) ) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -C option.  Correct syntax: -C<lon0>/<lat0>\n", GMT_program);
		error++;
	}
	if (two_points && ne != 2) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option.  Correct syntax: -E<lon1>/<lat1>\n", GMT_program);
		error++;
	}
	if (pole_set && np != 2) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -T option.  Correct syntax: -T<lonp>/<latp>\n", GMT_program);
		error++;
	}
	if (check_length && !stay_within && nl != 2) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -L option.  Correct syntax: -L[w | <min>/<max>]\n", GMT_program);
		error++;
	}
	if (check_width && (nw != 2 || w_min >= w_max)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -L option.  Correct syntax: -L[w | <min>/<max>]\n", GMT_program);
		error++;
	}
	if (azimuth < 0.0 || azimuth >= 360.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -A option.  Must specify azimuth in 0-360 degree range\n", GMT_program);
		error++;
	}
	if ( n_definitions != 1) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Specify only one of -A, -E, and -T\n", GMT_program);
		error++;
	}
	if ( two_points && (x_a == x_b) && (y_a == y_b) ) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option: Second point must differ from origin!\n", GMT_program);
		error++;
	}
	if ( generate && l_min == l_max && !(two_points)) {	/* We don't know how long to generate  */
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option: Must also specify -Lmin/max or use -E instead\n", GMT_program);
		error++;
	}
	if ( generate && n_outputs > 0) {	/* -F not allowed with -G  */
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option: -F not allowed [Defaults to rsp]\n", GMT_program);
		error++;
	}
	if ( generate && d_inc <= 0.0) {	/* No increment given  */
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option: Must specify a positive increment\n", GMT_program);
		error++;
	}
	if (stay_within && !(two_points) ) {	/* Same problem.  */
		fprintf (stderr, "%s: GMT SYNTAX ERROR -L option: Must specify -Lmin/max or use -E instead\n", GMT_program);
		error++;
	}
	if (n_outputs > 7) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -F option: Too many output columns selected (%d)\n", GMT_program, n_outputs);
		error++;
	}
	if (GMT_io.binary[GMT_IN] && gmtdefs.io_header[GMT_IN]) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", GMT_program);
		error++;
	}
        if (GMT_io.binary[GMT_IN] && GMT_io.ncol[GMT_IN] == 0) GMT_io.ncol[GMT_IN] = 2;
        if (GMT_io.binary[GMT_IN] && GMT_io.ncol[GMT_IN] < 2) {
                fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data (-bi) must have at least 2 columns\n", GMT_program);
		error++;
	}
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands4 */

	if (GMT_io.binary[GMT_IN] && gmtdefs.verbose) {
		char *type[2] = {"double", "single"};
		fprintf (stderr, "%s: Expects %d-column %s-precision binary data\n", GMT_program, GMT_io.ncol[GMT_IN], type[GMT_io.single_precision[GMT_IN]]);
	}

#ifdef SET_IO_MODE
	GMT_setmode (GMT_OUT);
#endif

	pure_ascii = !(GMT_io.binary[GMT_IN] || GMT_io.binary[GMT_OUT]);

	if (n_outputs == 0 && !(generate) ) {	/* Generate default -F setting (all) */
		n_outputs = 7;
		for (i = 0; i < 2; i++) output_choice[i] = i;
		output_choice[2] = -1;
		for (i = 3; i < n_outputs; i++) output_choice[i] = i-1;
		find_new_point = TRUE;
	}

	p_data = (struct DATA *) GMT_memory (VNULL, (size_t)n_alloc, sizeof (struct DATA), GMT_program);

	if (generate && two_points && (l_min == l_max) ) stay_within = TRUE;	/* Default generate from A to B  */

	/* Set up rotation matrix e for flat earth, or pole and center for spherical; get l_min, l_max if stay_within  */

	if (flat_earth) {
		flat_project_setup(y_a, x_a, y_b, x_b, y_p, x_p, &azimuth, e, two_points, pole_set);
		/* Azimuth is now changed to cartesian theta in radians */
		if (stay_within) {
			l_min = 0.0;
			xx = x_b - x_a;
			yy = y_b - y_a;
			l_max = d_sqrt(xx*xx + yy*yy);
			if (convert_units) l_max *= project_info.KM_PR_DEG;
		}
	}
	else {
		if (pole_set) {
			sin_lat_to_pole = oblique_setup(y_p, x_p, pole, y_a, x_a, center, pole_set, rads);
		}
		else {
			sphere_project_setup(y_a, x_a, a, y_b, x_b, b, &azimuth, pole, center, two_points, rads);
		}
		/* Azimuth is now changed to radians  */
		if (stay_within) {
			l_min = 0.0;
			l_max = GMT_dot3v(a,b);
			l_max = d_acos(l_max) * R2D;
			if (convert_units) l_max *= project_info.KM_PR_DEG;
		}
	}

	/* Now things are initialized.  We will work in degrees for awhile, so we convert things:  */

	if (convert_units) {
		d_inc /= project_info.KM_PR_DEG;
		l_min /= project_info.KM_PR_DEG;
		l_max /= project_info.KM_PR_DEG;
		w_min /= project_info.KM_PR_DEG;
		w_max /= project_info.KM_PR_DEG;
	}

	/*  Now we are ready to work  */

	n_used = 0;
	n_total_read = 0;

	if (generate) {	/* Not input data expected, just generate track from arguments given */
		n_outputs = 3;
		output_choice[0] = 4;
		output_choice[1] = 5;
		output_choice[2] = 2;
		out = (double *) GMT_memory (VNULL, (size_t)n_outputs, sizeof (double), GMT_program);

		d_along = l_min;
		while (d_along < l_max) {
			p_data[n_used].a[2] = d_along;
			n_used++;
			d_along = l_min + n_used * d_inc;
			if (n_used == (n_alloc-1)) {
				n_alloc += GMT_CHUNK;
				p_data = (struct DATA *) GMT_memory ((void *)p_data, (size_t)n_alloc, sizeof (struct DATA), GMT_program);
			}
		}
		p_data[n_used].a[2] = l_max;
		n_used ++;

		/* We need to find r,s  */

		if (flat_earth) {
			sincos (azimuth, &sin_theta, &cos_theta);
			for (i = 0; i < n_used; i++) {
				p_data[i].a[4] = x_a + p_data[i].a[2] * cos_theta;
				p_data[i].a[5] = y_a + p_data[i].a[2] * sin_theta;
				while (greenwich && p_data[i].a[4] < 0.0) p_data[i].a[4] += 360.0;
				while (dateline && p_data[i].a[4] > 180.0) p_data[i].a[4] -= 360.0;
			}
		}
		else {
			xx = x_a;
			yy = y_a;
			GMT_geo_to_cart(&yy, &xx, x, rads);
			for (i = 0; i < n_used; i++) {
				theta = p_data[i].a[2] / sin_lat_to_pole;
				make_euler_matrix(pole, e, &theta, rads);
				matrix_3v(e,x,xt);
				GMT_cart_to_geo(&yy, &xx, xt, rads);
				p_data[i].a[4] = xx;
				p_data[i].a[5] = yy;
				while (greenwich && p_data[i].a[4] < 0.0) p_data[i].a[4] += 360.0;
				while (dateline && p_data[i].a[4] > 180.0) p_data[i].a[4] -= 360.0;
			}
		}

		/* At this stage, all values are still in degrees.  */

		if (convert_units) {
			for (i = 0; i < n_used; i++) {
				p_data[i].a[2] *= project_info.KM_PR_DEG;
				p_data[i].a[3] *= project_info.KM_PR_DEG;
			}
		}

		/* Now output generated track */

		if (!GMT_io.binary[GMT_OUT]) {
			if (gmtdefs.io_header[GMT_OUT]) fprintf (GMT_stdout, "lon\tlat\tdist\n");

			for (i = 0; i < n_used; i++) {
				for (j = 0; j < n_outputs; j++) out[j] = p_data[i].a[output_choice[j]];
				GMT_output (GMT_stdout, n_outputs, out);
			}
		}
	}

	else {	/* Must read input file */

		n_expected_fields = (GMT_io.ncol[GMT_IN]) ? GMT_io.ncol[GMT_IN] : BUFSIZ;

		if (n_files > 0)
			nofile = FALSE;
		else
			n_files = 1;

		n_args = (argc > 1) ? argc : 2;

		for (fno = 1; !done && fno < n_args; fno++) {	/* Loop over input files, if any */
			if (!nofile && argv[fno][0] == '-') continue;

			if (nofile) {	/* Just read standard input */
				fp = GMT_stdin;
				done = TRUE;
#ifdef SET_IO_MODE
				GMT_setmode (GMT_IN);
#endif
			}
			else if ((fp = GMT_fopen (argv[fno], GMT_io.r_mode)) == NULL) {
				fprintf (stderr, "%s: Cannot open file %s\n", GMT_program, argv[fno]);
				continue;
			}

			if (!nofile && gmtdefs.verbose) fprintf (stderr, "%s: Working on file %s\n", GMT_program, argv[fno]);

			if (gmtdefs.io_header[GMT_IN]) {
				GMT_fgets (record_str, BUFSIZ, fp);
				sscanf (record_str, "%s %s %s", heading[0], heading[1], heading[6]);
				if (! (heading[6]) ) strcpy (heading[6],"Z");
				strcpy (heading[2],"p");
				strcpy (heading[3],"q");
				strcpy (heading[4],"r");
				strcpy (heading[5],"s");
				for (i = 1; i < gmtdefs.n_header_recs; i++) GMT_fgets (record_str, BUFSIZ, fp);
			}

			n_fields = GMT_input (fp, &n_expected_fields, &in);
			n_z = n_expected_fields - 2;
			if (n_z == 0 && want_z_output) {
				fprintf (stderr, "%s: No data columns, cannot use z flag in -F\n", GMT_program);
				exit (EXIT_FAILURE);
			}
			n_used = 0;

			while (! (GMT_io.status & GMT_IO_EOF)) {	/* Not yet EOF */

				while (GMT_io.status & GMT_IO_SEGMENT_HEADER) {
					GMT_write_segmentheader (GMT_stdout, n_expected_fields);
					n_fields = GMT_input (fp, &n_expected_fields, &in);
				}

				while (! (GMT_io.status & (GMT_IO_SEGMENT_HEADER | GMT_IO_EOF))) {	/* Keep going until FALSE or = 2 segment header */
					if (GMT_io.status & GMT_IO_MISMATCH) {
						fprintf (stderr, "%s: Mismatch between actual (%d) and expected (%d) fields near line %d\n", GMT_program, n_fields, n_expected_fields, n_total_read);
						exit (EXIT_FAILURE);
					}

					xx = in[0];
					yy = in[1];

					n_total_read ++;

					if (flat_earth) {
						x[0] = xx - x_a;
						x[1] = yy - y_a;
						matrix_2v (e,x,xt);
					}
					else {
						oblique_transform(yy, xx, &xt[1], &xt[0], pole, center, rads);
					}

					skip = ((check_length && (xt[0] < l_min || xt[0] > l_max)) || (check_width && (xt[1] < w_min || xt[1] > w_max)));

					if (skip) {
						n_fields = GMT_input (fp, &n_expected_fields, &in);
						continue;
					}

					p_data[n_used].a[0] = xx;
					p_data[n_used].a[1] = yy;
					p_data[n_used].a[2] = xt[0];
					p_data[n_used].a[3] = xt[1];
					if (n_z) {	/* Copy over z column(s) */
						if (pure_ascii) {	/* Must store all text beyond x,y columns */
							p_data[n_used].t = (char *) GMT_memory (VNULL, strlen (GMT_io.current_record), sizeof (char), GMT_program);
							copy_text_from_col3 (GMT_io.current_record, p_data[n_used].t);
						}
						else {
							p_data[n_used].z = (double *) GMT_memory (VNULL, (size_t)n_z, sizeof (double), GMT_program);
							memcpy ((void *)p_data[n_used].z, (void *)&in[2], (size_t)(n_z * sizeof(double)));
						}
					}
					n_used++;
					if (n_used == n_alloc) {
						n_alloc += GMT_CHUNK;
						p_data = (struct DATA *) GMT_memory ((void *)p_data, (size_t)n_alloc, sizeof (struct DATA), GMT_program);
					}

					n_fields = GMT_input (fp, &n_expected_fields, &in);
				}

				if (sort_output) qsort ((void *)p_data, (size_t)n_used, sizeof (struct DATA), compare_distances);

/*				Get here when all data are loaded with p,q and p is in increasing order if desired.  */

				if (find_new_point) {	/* We need to find r,s  */

					if (flat_earth) {
						sincos (azimuth, &sin_theta, &cos_theta);
						for (i = 0; i < n_used; i++) {
							p_data[i].a[4] = x_a + p_data[i].a[2] * cos_theta;
							p_data[i].a[5] = y_a + p_data[i].a[2] * sin_theta;
							while (greenwich && p_data[i].a[4] < 0.0) p_data[i].a[4] += 360.0;
							while (dateline && p_data[i].a[4] > 180.0) p_data[i].a[4] -= 360.0;
						}
					}
					else {
						xx = x_a;
						yy = y_a;
						GMT_geo_to_cart(&yy, &xx, x, rads);
						for (i = 0; i < n_used; i++) {
							theta = p_data[i].a[2];
							make_euler_matrix(pole, e, &theta, rads);
							matrix_3v(e,x,xt);
							GMT_cart_to_geo(&yy, &xx, xt, rads);
							p_data[i].a[4] = xx;
							p_data[i].a[5] = yy;
							while (greenwich && p_data[i].a[4] < 0.0) p_data[i].a[4] += 360.0;
							while (dateline && p_data[i].a[4] > 180.0) p_data[i].a[4] -= 360.0;
						}
					}
				}

				/* At this stage, all values are still in degrees.  */

				if (convert_units) {
					for (i = 0; i < n_used; i++) {
						p_data[i].a[2] *= project_info.KM_PR_DEG;
						p_data[i].a[3] *= project_info.KM_PR_DEG;
					}
				}

				/* Now output  */

				if (!GMT_io.binary[GMT_OUT]) {	/* First do header */
					if (first && gmtdefs.io_header[GMT_OUT]) {
						for (j = 0; j < n_outputs; j++) {
							if (output_choice[j] == -1)
								fprintf (GMT_stdout, "%s", heading[6]);
							else
								fprintf (GMT_stdout, "%s", heading[output_choice[j]]);
							(j == (n_outputs - 1)) ? fprintf (GMT_stdout, "\n") : fprintf (GMT_stdout, "\t");
						}
						first = FALSE;
					}
				}

				n_items = n_outputs + ((want_z_output && n_z) ? n_z - 1 : 0);
				if (!out) out = (double *) GMT_memory (VNULL, (size_t)n_items, sizeof (double), GMT_program);

				/* Special case for pure ascii since we may pass text */

				if (n_z && pure_ascii) {
					for (i = 0; i < n_used; i++) {
						for (j = 0; j < n_outputs; j++) {
							if (output_choice[j] == -1) {	/* Output all z columns as one string */
								fprintf (GMT_stdout, "%s", p_data[i].t);
								GMT_free ((void *)p_data[i].t);
							}
							else
								fprintf (GMT_stdout, gmtdefs.d_format, p_data[i].a[output_choice[j]]);
							(j == (n_outputs - 1)) ? fprintf (GMT_stdout, "\n") : fprintf (GMT_stdout, "\t");
						}
					}
				}
				else {	/* Any other i/o combination */
					for (i = 0; i < n_used; i++) {
						for (j = k = 0; j < n_outputs; j++) {
							if (output_choice[j] == -1) {	/* Copy over all z columns */
								memcpy ((void *)&out[k], (void *)p_data[i].z, (size_t)(n_z * sizeof (double)));
								GMT_free ((void *)p_data[i].z);
								k += n_z;
							}
							else
								out[k++] = p_data[i].a[output_choice[j]];
						}
						GMT_output (GMT_stdout, n_items, out);
					}
				}

				n_total_used += n_used;
				n_used = 0;

			}

			if (fp != GMT_stdin) GMT_fclose(fp);
		}
	}

	if (gmtdefs.verbose) fprintf(stderr, "%s: %d read, %d used\n", GMT_program, n_total_read, n_total_used);

	GMT_free ((void *)p_data);
	GMT_free ((void *)out);

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}

int	compare_distances(const void *point_1, const void *point_2)
{
	double	d_1, d_2;

	d_1 = ((struct DATA *)point_1)->a[2];
	d_2 = ((struct DATA *)point_2)->a[2];

	if (d_1 < d_2)
		return (-1);
	if (d_1 > d_2)
		return (1);
	else
		return (0);
}

double	oblique_setup(double plat, double plon, double *p, double clat, double clon, double *c, int c_given, int rads)
{
	/* routine sets up a unit 3-vector p, the pole of an 
	   oblique projection, given plat, plon, the position 
	   of this pole in the usual coordinate frame.
	   c_given = TRUE means that clat, clon are to be used
	   as the usual coordinates of a point through which the
	   user wants the central meridian of the oblique
	   projection to go.  If such a point is not given, then
	   the central meridian will go through p and the usual
	   N pole.  In either case, a unit 3-vector c is created
	   which is the directed normal to the plane of the central
	   meridian, pointing in the positive normal (east) sense.
	   rads = TRUE if we need to convert plat, plon, clat, clon
	   from degrees to radians.  */

	double	s[3];  /* s points to the south pole  */
	double cp, sin_lat_to_pole;

	s[0] = s[1] = 0.0;
	s[2] = -1.0;

	GMT_geo_to_cart(&plat, &plon, p, rads);

	if (c_given) {	/* s points to user's clat, clon  */
		GMT_geo_to_cart(&clat, &clon, s, rads);
	}
	GMT_cross3v(p, s, c);
	GMT_normalize3v(c);
	cp = GMT_dot3v (p, s);
	sin_lat_to_pole = d_sqrt (1.0 - cp * cp);
	return (sin_lat_to_pole);
}

void	oblique_transform(double xlat, double xlon, double *x_t_lat, double *x_t_lon, double *p, double *c, int rads)
{
	/* routine takes the point x at conventional (xlat, xlon) and
	   computes the transformed coordinates (x_t_lat, x_t_lon) in
	   an oblique reference frame specified by the unit 3-vectors
	   p (the pole) and c (the directed normal to the oblique
	   central meridian).  p and c have been computed earlier by
	   the routine oblique_setup().  rads = TRUE if lats and lons
	   are in degrees.  */

	double	x[3], p_cross_x[3], temp1, temp2;

	GMT_geo_to_cart(&xlat, &xlon, x, rads);

	temp1 = GMT_dot3v(x,p);
	*x_t_lat = d_asin(temp1);

	GMT_cross3v(p,x,p_cross_x);
	GMT_normalize3v(p_cross_x);

	temp1 = GMT_dot3v(p_cross_x, c);
	temp2 = GMT_dot3v(x, c);
	*x_t_lon = copysign( d_acos(temp1), temp2);

	if (rads) {
		*x_t_lat *= R2D;
		*x_t_lon *= R2D;
	}
}

void	make_euler_matrix(double *p, double *e, double *theta, int rads)
{
	/* Routine to fill an euler matrix e with the elements
	   needed to rotate a 3-vector about the pole p through
	   an angle theta.  p is a unit 3-vector.  If rads = TRUE,
	   we have to convert theta from degrees into radians before
	   we use it.  */

	double	cos_theta, sin_theta, one_minus_cos_theta;
	double	pxsin, pysin, pzsin, temp;

	if (rads) {
		*theta *= D2R;
	}
	cos_theta = cos(*theta);
	sin_theta = sin(*theta);
	one_minus_cos_theta = 1.0 - cos_theta;

	pxsin = p[0] * sin_theta;
	pysin = p[1] * sin_theta;
	pzsin = p[2] * sin_theta;

	temp = p[0] * one_minus_cos_theta;
	e[0] = temp * p[0] + cos_theta;
	e[1] = temp * p[1] - pzsin;
	e[2] = temp * p[2] + pysin;

	temp = p[1] * one_minus_cos_theta;
	e[3] = temp * p[0] + pzsin;
	e[4] = temp * p[1] + cos_theta;
	e[5] = temp * p[2] - pxsin;

	temp = p[2] * one_minus_cos_theta;
	e[6] = temp * p[0] - pysin;
	e[7] = temp * p[1] + pxsin;
	e[8] = temp * p[2] + cos_theta;
}

void	matrix_3v(double *a, double *x, double *b)
{
	/* routine to find b, where Ax = b, A is a 3 by 3 square matrix,
	   and x and b are 3-vectors.  A is stored row wise, that is:
	   
	   A = { a11, a12, a13, a21, a22, a23, a31, a32, a33 }  */

	b[0] = x[0]*a[0] + x[1]*a[1] + x[2]*a[2];
	b[1] = x[0]*a[3] + x[1]*a[4] + x[2]*a[5];
	b[2] = x[0]*a[6] + x[1]*a[7] + x[2]*a[8];
}

void	matrix_2v(double *a, double *x, double *b)
{
	/* routine to find b, where Ax = b, A is a 2 by 2 square matrix,
	   and x and b are 2-vectors.  A is stored row wise, that is:
	   
	   A = { a11, a12, a21, a22 }  */

	b[0] = x[0]*a[0] + x[1]*a[1];
	b[1] = x[0]*a[2] + x[1]*a[3];
}

void sphere_project_setup(double alat, double alon, double *a, double blat, double blon, double *b, double *azim, double *p, double *c, int two_pts, int rads)
{
	/* routine to initialize a pole vector, p, and a central meridian 
	   normal vector, c, for use in projecting points onto a great circle.
	   
	   The great circle is specified in either one of two ways:
	   if (two_pts), then the user has given two points, a and b,
	   which specify the great circle (directed from a to b);
	   if !(two_pts), then the user has given one point, a, and an azimuth,
	   azim, clockwise from north, which defines the projection.

	   The strategy is to use the oblique_transform operations above,
	   in such a way that the great circle of the projection is the
	   equator of an oblique transform, and the central meridian goes
	   through a.  Then the transformed longitude gives the distance
	   along the projection circle, and the transformed latitude gives
	   the distance normal to the projection circle.

	   If (two_pts), then p = normalized(a X b).  If not, we temporarily
	   create p_temp = normalized(a X n), where n is the north pole.
	   p_temp is then rotated about a through the angle azim to give p.
	   After p is found, then c = normalized(p X a).
	*/

	double	e[9];	/* Euler rotation matrix, if needed  */
	double neg_azim;

	/* First find p vector  */

	if (two_pts) {
		GMT_geo_to_cart(&alat, &alon, a, rads);
		GMT_geo_to_cart(&blat, &blon, b, rads);
		GMT_cross3v(a, b, p);
		GMT_normalize3v(p);
	}
	else {
		GMT_geo_to_cart(&alat, &alon, a, rads);
		b[0] = b[1] = 0.0;	/* set b to north pole  */
		b[2] = 1.0;
		GMT_cross3v(a, b, c);	/* use c for p_temp  */
		GMT_normalize3v(c);
/*		make_euler_matrix(a, e, azim, rads);	*/
		neg_azim = -(*azim);
		make_euler_matrix(a, e, &neg_azim, rads);
		if (rads) *azim *= D2R;
		matrix_3v(e, c, p);	/* c (p_temp) rotates to p  */
	}

	/* Now set c vector  */

	GMT_cross3v(p, a, c);
	GMT_normalize3v(c);
}

void	flat_project_setup(double alat, double alon, double blat, double blon, double plat, double plon, double *azim, double *e, int two_pts, BOOLEAN pole_set)
{
	/* Sets up stuff for rotation of cartesian 2-vectors, analogous
	   to the spherical three vector stuff above.  Also change azim
	   to the cartesian theta, counterclockwise from the x axis.  */

	if (two_pts) {
		*azim = d_atan2((blat - alat), (blon - alon));
	}
	else if (pole_set) {
		*azim = d_atan2((plat - alat), (plon - alon)) - 0.5 * M_PI;
	}
	else {
		*azim = D2R * (90.0 - *azim);
	}

	e[0] = e[3] = cos(*azim);
	e[2] = sin(*azim);
	e[1] = -e[2];
}

void copy_text_from_col3 (char *line, char *z_cols)
{	/* returns the input line starting at the 3rd column */

	int i;

	/* First replace any commas with spaces */

	for (i = 0; line[i]; i++) if (line[i] == ',') line[i] = ' ';

	sscanf (line, "%*s %*s %[^\n]", z_cols);
}
