/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "terminal.h"
#include "config.h"
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QTextCursor>
#include <QLineEdit>
#include <QIcon>
#include <QTextStream>
#include <QMap>


Terminal::Terminal(QWidget * parent):BaseWidget(parent)
{
	//Se crea la ventana del terminal
	setWindowTitle("Octave Terminal");
	setWindowIcon( QIcon(QString( ICON_PATH )+"/konsole.png"));
	
	if(get_config("lines_in_terminal").isEmpty())
	{
		QMap<QString,QString> c;
		c["lines_in_terminal"]="1000";
		set_config(c);
	}
	if(get_config("cols_in_terminal").isEmpty())
	{
		QMap<QString,QString> c;
		c["cols_in_terminal"]="80";
		set_config(c);
	}
	
	lines_in_terminal=get_config("lines_in_terminal").toInt();
	cols_in_terminal=get_config("cols_in_terminal").toInt();
	text=new QTextEdit(this);
	text->setLineWrapMode (QTextEdit::NoWrap);
	text->setReadOnly (true);
	if(get_config("terminal_font").isEmpty())
	{
		QMap<QString,QString> c;
		c["terminal_font"]="Courier 10 Pitch";
		set_config(c);
	}
	
	{
		QFont font;
		font.fromString(get_config("terminal_font"));
		text->setFont( font );
	}
	if(get_config("terminal_foreground_color").isEmpty())
	{
		QMap<QString,QString> c;
		c["terminal_foreground_color"]="Black";
		set_config(c);
	}
	
	if(get_config("terminal_background_color").isEmpty())
	{
		QMap<QString,QString> c;
		c["terminal_background_color"]="White";
		set_config(c);
	}
	
	if(get_config("terminal_error_color").isEmpty())
	{
		QMap<QString,QString> c;
		c["terminal_error_color"]="Red";
		set_config(c);
	}
	
	text->setTextColor(get_config("terminal_foreground_color"));
	{
		QPalette p=text->viewport ()->palette();
		p.setColor(QPalette::Base, QColor(get_config("terminal_background_color")) );
		text->viewport()->setPalette(p);
	}
	text->show();
	
	combo_box=new Autocomplete(this);
	combo_box->setEditable(true);
	combo_box->setAutoCompletion(false);
	combo_box->setDuplicatesEnabled(true);
	combo_box->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
	combo_box->load_from_file(CONFIG_PATH "/autocomplete.lst");
	combo_box->show();
	
	//Se captura el "intro" para pasar comandos al octave
	QLineEdit *line_edit=combo_box->lineEdit();
	QObject::connect(line_edit, SIGNAL(returnPressed()), this, SLOT(return_pressed()));
	
	line_edit->setText(tr("Insert your commands here"));
	line_edit->selectAll();
	
	/*
	completion_matches_button=new QPushButton(this);
	completion_matches_button->setIcon(QIcon(QString(ICON_PATH)+"/images/search.png"));
	completion_matches_button->setSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
	completion_matches_button->setToolTip(
		tr("<b>Completion matches:</b><br> Generates possible completions given in comand line.")
		);
	completion_matches_button->show();
	connect(completion_matches_button, SIGNAL(clicked()), this, SLOT(completion_matches_callback()));
	
	
	dynamic_help_button=new QPushButton(this);
	dynamic_help_button->setIcon(QIcon(QString(ICON_PATH)+"/images/help_index.png"));
	dynamic_help_button->setSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
	dynamic_help_button->setToolTip(
			tr("<b>Dynamic Help:</b><br> Shows help of completions matches.")
					     );
	dynamic_help_button->show();
	connect(dynamic_help_button, SIGNAL(clicked()), this, SLOT(dynamic_help_callback()));
	
	stop_process_button=new QPushButton(this);
	stop_process_button->setIcon(QIcon(QString(ICON_PATH)+"/images/button_cancel.png"));
	stop_process_button->setSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
	stop_process_button->setToolTip(
			tr("<b>Stops process:</b><br>Stops uncontroled process, infinite loops,...")
				       );
	stop_process_button->show();
	connect(stop_process_button, SIGNAL(clicked()), this, SLOT(stop_process_callback()));
	
	clear_button=new QPushButton(this);
	clear_button->setIcon(QIcon(QString(ICON_PATH)+"/images/eraser.png"));
	clear_button->setSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
	clear_button->setToolTip(
			tr("<b>Clear terminal:</b><br>Erase terminal screen.")
				       );
	clear_button->show();
	connect(clear_button, SIGNAL(clicked()), this, SLOT(clear_callback()));
	*/
	
	QVBoxLayout *layout = new QVBoxLayout;
	//QHBoxLayout *comand_layout = new QHBoxLayout;
	QHBoxLayout *line_layout = new QHBoxLayout;
	layout->addWidget(text);
	layout->addLayout(line_layout);
	/*
	layout->addLayout(comand_layout);
	comand_layout->addWidget(completion_matches_button);
	comand_layout->addWidget(dynamic_help_button);
	comand_layout->addWidget(stop_process_button);
	comand_layout->addWidget(clear_button);
	comand_layout->addWidget(new QWidget);
	*/
	line_layout->addWidget(combo_box);
	
	setLayout(layout);
	
	dynamic_help=NULL;
}

void Terminal::return_pressed()
{
	QLineEdit *line_edit=combo_box->lineEdit();
	QString command(line_edit->text());
	command_enter(command);
};

QTextEdit *Terminal::getTextEdit()
{
	return text;
}

void Terminal::setOctaveConnection(OctaveConnection *octave_connection)
{
	this->octave_connection=octave_connection;
	//octave_connection->setTextEdit(text);
	QObject::connect(octave_connection, SIGNAL(error_ready(QString)), this, SLOT(write_error(QString)));
	QObject::connect(octave_connection, SIGNAL(output_ready(QString)), this, SLOT(write_output(QString)));
	QObject::connect(octave_connection, SIGNAL(command_ready(QString)), this, SLOT(write_command(QString)));
}

OctaveConnection *Terminal::getOctaveConnection()
{
	return octave_connection;
}

void Terminal::command_enter (const QString &command)
{
	octave_connection->command_enter(command);
	combo_box->clearEditText();
}

#include <QTextBlock>

void Terminal::remove_lines(QTextCursor &cursor)
{
	//Se cortan las l�eas que sean demasiado largas
	cursor.movePosition(QTextCursor::Start);
	int lines=0;
	while(!cursor.atEnd())
	{
		QTextBlock block=cursor.block();
		if(block.length()>cols_in_terminal)
		{
			cursor.movePosition(QTextCursor::StartOfBlock);
			cursor.movePosition(QTextCursor::Right,QTextCursor::MoveAnchor,cols_in_terminal-1);
			cursor.insertText("\n");
		}
		else cursor.movePosition(QTextCursor::Down);
		cursor.movePosition(QTextCursor::EndOfBlock);
		lines++;
	}
	//Se borran las l�eas iniciales si hay m� de lines_in_terminal l�eas
	//printf("lines=%d %d\n",lines,lines_in_terminal);
	lines=text->toPlainText().count('\n');
	while(lines>lines_in_terminal)
	{
		cursor.movePosition(QTextCursor::Start);
		cursor.movePosition(QTextCursor::EndOfBlock,QTextCursor::KeepAnchor);
		//printf("line %d: %s\n",lines, cursor.selectedText().toLocal8Bit().data());
		if(cursor.hasSelection()) cursor.removeSelectedText();
		else cursor.deleteChar();
		cursor.deleteChar();
		lines--;
	}
	//printf("lines=%d %d\n",lines,lines_in_terminal);
}

void Terminal::write_output(QString output)
{
	QTextCursor cursor=text->textCursor();
	cursor.beginEditBlock();
	cursor.movePosition(QTextCursor::End);
	text->setTextCursor( cursor );
	text->setTextColor( get_config("terminal_foreground_color") );
	text->setFontWeight ( QFont::Normal );
	text->insertPlainText(output);
	remove_lines(cursor);
	cursor.movePosition(QTextCursor::End);
	cursor.endEditBlock();
	text->setTextCursor( cursor );
}

void Terminal::write_error(QString error)
{
	QTextCursor cursor=text->textCursor();
	cursor.beginEditBlock();
	cursor.movePosition(QTextCursor::End);
	text->setTextCursor( cursor );
	text->setTextColor( get_config("terminal_error_color") );
	text->setFontWeight ( QFont::Normal );
	text->insertPlainText(error);
	remove_lines(cursor);
	cursor.movePosition(QTextCursor::End);
	cursor.endEditBlock();
	text->setTextCursor( cursor );
}

void Terminal::write_command(QString command)
{
	QTextCursor cursor=text->textCursor();
	cursor.beginEditBlock();
	cursor.movePosition(QTextCursor::End);
	text->setTextCursor( cursor );
	text->setTextColor( get_config("terminal_foreground_color") );
	text->setFontWeight ( QFont::Bold );
	text->insertPlainText(command);
	text->setFontWeight ( QFont::Normal );
	remove_lines(cursor);
	cursor.movePosition(QTextCursor::End);
	cursor.endEditBlock();
	text->setTextCursor( cursor );
}

void Terminal::completion_matches_callback()
{
	QString command;
	QTextStream(&command) << "completion_matches(\"" << combo_box->lineEdit()->text() << "\")";
	
	octave_connection->command_enter(command,false);
}

void Terminal::dynamic_help_callback()
{
	/*
	if(dynamic_help!=NULL)
	{
		delete dynamic_help;
	}
	*/
	dynamic_help=new Dynamic_help(work_space);
	dynamic_help->setLineEdit(combo_box->lineEdit());
	work_space->addWindow(dynamic_help);
	dynamic_help->show();
}

#include <sys/types.h>
#include <signal.h>

void Terminal::stop_process_callback()
{
	kill(octave_connection->pid(),SIGINT);
}

void Terminal::clear_callback()
{
	text->clear();
	octave_connection->command_enter("",false);
}



