/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1999 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for interfacing with the DNS. */

#include "exim.h"

/* While IPv6 is yet young: the definition of T_AAAA may not be included in
arpa/nameser.h. Fudge it here. */

#if HAVE_IPV6
#ifndef T_AAAA
#define T_AAAA 28
#endif
#endif


/* Ancient systems (e.g. SunOS4) don't appear to have T_TXT defined in their
header files. */

#ifndef T_TXT
#define T_TXT 16
#endif


static unsigned char *dns_answer = NULL;
static int dns_answerlen = 0;

#if PACKETSZ < 1024
  #define MAXPACKET 1024
#else
  #define MAXPACKET PACKETSZ
#endif




/*************************************************
*        Initializer and configure resolver      *
*************************************************/

/* Initialize the resolver and the storage for holding DNS answers if this is
the first time we have been here, and set the resolver options.

Arguments:
  qualify_single    TRUE to set the RES_DEFNAMES option
  search_parents    TRUE to set the RES_DNSRCH option

Returns:            nothing
*/

void
dns_init(BOOL qualify_single, BOOL search_parents)
{
if (dns_answer == NULL)
  {
  dns_answer = store_malloc(MAXPACKET);
  res_init();
  DEBUG(11) _res.options |= RES_DEBUG;
  }
_res.options &= ~(RES_DNSRCH | RES_DEFNAMES);
_res.options |= (qualify_single? RES_DEFNAMES : 0) |
                (search_parents? RES_DNSRCH : 0);

if (dns_retrans > 0) _res.retrans = dns_retrans;
if (dns_retry > 0) _res.retry = dns_retry;
}



/*************************************************
*           Expand text in DNS reply             *
*************************************************/

/* This function exists because dns_answer and dns_answerlen are private to
this module.

Arguments:
  ptr         pointer into the data part of the RR
  buffer      where to put the expanded text
  length      length of the buffer

Returns:      length of the expanded text
*/

int
dns_expand(unsigned char *ptr, char *buffer, int length)
{
return dn_expand(dns_answer, dns_answer + dns_answerlen, ptr,
    (DN_EXPAND_ARG4_TYPE)buffer, length);
}



/*************************************************
*       Get next DNS record from answer block    *
*************************************************/

/* Call this with reset == RESET_ANSWERS to scan the answer block, reset ==
RESET_ADDITIONAL to scan the additional records, and reset == RESET_NEXT to
get the next record. The result is in static storage which must be copied if
it is to be preserved.

Argument:   option specifing what portion to scan, as described above
Returns:    next dns record, or NULL when no more
*/

dns_record *
dns_next_rr(int reset)
{
static int rrcount;
static dns_record rr;
static unsigned char *ptr;

int namelen;
HEADER *h = (HEADER *)dns_answer;


/* Reset the saved data when requested to, and skip to the first required RR */

if (reset != RESET_NEXT)
  {
  rrcount = ntohs(h->qdcount);
  ptr = dns_answer + sizeof(HEADER);

  /* Skip over questions; failure to expand the name just gives up */

  while (rrcount-- > 0)
    {
    namelen = dn_expand(dns_answer, dns_answer + dns_answerlen, ptr,
      (DN_EXPAND_ARG4_TYPE) &(rr.name), DNS_MAXNAME);
    if (namelen < 0) { rrcount = 0; return NULL; }
    ptr += namelen + 4;    /* skip name & type & class */
    }

  /* Get the number of answer records. */

  rrcount = ntohs(h->ancount);

  /* Skip over answers and NS records if wanting to look at the additional
  records. */

  if (reset == RESET_ADDITIONAL)
    {
    rrcount += ntohs(h->nscount);
    while (rrcount-- > 0)
      {
      namelen = dn_expand(dns_answer, dns_answer + dns_answerlen, ptr,
        (DN_EXPAND_ARG4_TYPE) &(rr.name), DNS_MAXNAME);
      if (namelen < 0) { rrcount = 0; return NULL; }
      ptr += namelen + 8;     /* skip name & type & class & TTL */
      GETSHORT(rr.size, ptr); /* size of data portion */
      ptr += rr.size;         /* skip over it */
      }
    rrcount = ntohs(h->arcount);
    }
  }


/* The variable "ptr" is now pointing at the next RR, and rrcount contains the
number of RR records left. */

if (rrcount-- <= 0) return NULL;

/* Expand the RR domain name into the slot in the static rr structure; if this
fails, behave as if no more records (something safe). */

namelen = dn_expand(dns_answer, dns_answer + dns_answerlen, ptr,
  (DN_EXPAND_ARG4_TYPE) &(rr.name), DNS_MAXNAME);
if (namelen < 0) { rrcount = 0; return NULL; }

/* Move the pointer past the name and fill in the rest of the data structure
from the following bytes. */

ptr += namelen;
GETSHORT(rr.type, ptr);        /* Record type */
ptr += 6;                      /* Don't want class or TTL */
GETSHORT(rr.size, ptr);        /* Size of data portion */
rr.data = ptr;                 /* The record's data follows */
ptr += rr.size;                /* Advance to next RR */

/* Return a pointer to the static structure. */

return &rr;
}




/*************************************************
*            Turn DNS type into text             *
*************************************************/

/* Turn the coded record type into a string for printing.

Argument:   record type
Returns:    pointer to string
*/

char *
dns_text_type(int t)
{
switch(t)
  {
  case T_A:    return "A";
  case T_MX:   return "MX";
  #if HAVE_IPV6
  case T_AAAA: return "AAAA";
  #endif
  case T_TXT:  return "TXT";
  default:     return "?";
  }
}



/*************************************************
*              Do basic DNS lookup               *
*************************************************/

/* Call the resolver to look up the given domain name, using the given type,
and check the result. The error code TRY_AGAIN is documented as meaning "non-
Authoritive Host not found, or SERVERFAIL". Sometimes there are badly set
up nameservers that produce this error continually, so there is the option of
providing a list of domains for which this is treated as a non-existent
host.

Arguments:
  name      name to look up
  type      type of DNS record required (T_A, T_MX, etc)

Returns:    DNS_SUCCEED   successful lookup
            DNS_NOMATCH   name not found, or no data found for the given type,
                          or name contains illegal characters (if checking)
            DNS_AGAIN     soft failure, try again later
            DNS_FAIL      DNS failure
*/

int
dns_basic_lookup(char *name, int type)
{

/* If configured, check the hygene of the name passed to lookup. Otherwise,
although DNS lookups may give REFUSED at the lower level, some resolvers
turn this into TRY_AGAIN, which is silly. Give a NOMATCH return, since such
domains cannot be in the DNS. The check is now done by a regular expression;
give it space for substring storage to save it having to get its own if the
regex has substrings that are used - the default uses a conditional. */

#ifndef STAND_ALONE   /* Omit this for stand-alone tests */

if (check_dns_names)
  {
  int ovector[3*(EXPAND_MAXN+1)];

  if (regex_check_dns_names == NULL)
    regex_check_dns_names =
      regex_must_compile(check_dns_names_pattern, FALSE, TRUE);

  if (pcre_exec(regex_check_dns_names, NULL, name, (int)strlen(name), 0,
      PCRE_EOPT, ovector, sizeof(ovector)/sizeof(int)) < 0)
    {
    DEBUG(3)
      debug_printf("DNS name syntax check failed: %s (%s)\n", name,
        dns_text_type(type));
    host_find_failed_syntax = TRUE;
    return DNS_NOMATCH;
    }
  }

#endif /* STAND_ALONE */

/* Call the resolver */

dns_answerlen = res_search(name, C_IN, type, dns_answer, MAXPACKET);

if (dns_answerlen < 0) switch (h_errno)
  {
  case HOST_NOT_FOUND:
  DEBUG(8) debug_printf("DNS lookup of %s (%s) gave HOST_NOT_FOUND\n",
    name, dns_text_type(type));
  return DNS_NOMATCH;

  case TRY_AGAIN:
  DEBUG(8) debug_printf("DNS lookup of %s (%s) gave TRY_AGAIN\n",
    name, dns_text_type(type));

  /* Cut this out for various test programs */
  #ifndef STAND_ALONE
  if (!match_isinlist(name, &dns_again_means_nonexist, FALSE, TRUE, NULL))
    {
    DEBUG(8) debug_printf("returning DNS_AGAIN\n");
    return DNS_AGAIN;
    }
  DEBUG(8) debug_printf("%s is in dns_again_means_nonexist: returning "
    "DNS_NOMATCH\n", name);
  #endif

  return DNS_NOMATCH;

  case NO_RECOVERY:
  DEBUG(8) debug_printf("DNS lookup of %s (%s) gave NO_RECOVERY\n",
    name, dns_text_type(type));
  return DNS_FAIL;

  case NO_DATA:
  DEBUG(8) debug_printf("DNS lookup of %s (%s) gave NO_DATA\n",
    name, dns_text_type(type));
  return DNS_NOMATCH;

  default:
  DEBUG(8) debug_printf("DNS lookup of %s (%s) gave unknown DNS error %d\n",
    name, dns_text_type(type), h_errno);
  return DNS_FAIL;
  }

DEBUG(8) debug_printf("DNS lookup of %s (%s) succeeded\n",
  name, dns_text_type(type));

return DNS_SUCCEED;
}




/************************************************
*        Do a DNS lookup and handle CNAMES      *
************************************************/

/* Look up the given domain name, using the given type. Follow CNAMEs if
necessary, but only so many times. There aren't supposed to be CNAME chains in
the DNS, but you are supposed to cope with them if you find them.

The assumption is made that if the resolver gives back records of the
requested type *and* a CNAME, we don't need to make another call to look up
the CNAME. I can't see how it could return only some of the right records. If
it's done a CNAME lookup in the past, it will have all of them; if not, it
won't return any.

If fully_qualified_name is not NULL, set it to point to the full name
returned by the resolver, if this is different to what it is given, unless
the returned name starts with "*" as some nameservers seem to be returning
wildcards in this form.

Arguments:
  name                  domain name to look up
  type                  DNS record type (T_A, T_MX, etc)
  fully_qualified_name  if not NULL, return the returned name here if its
                          contents are different (i.e. it must be preset)

Returns:                DNS_SUCCEED   successful lookup
                        DNS_NOMATCH   name not found, or no data found
                        DNS_AGAIN     soft failure, try again later
                        DNS_FAIL      DNS failure
*/

int
dns_lookup(char *name, int type, char **fully_qualified_name)
{
int i;
char *orig_name = name;

/* Loop to follow CNAME chains so far, but no further... */

for (i = 0; i < 10; i++)
  {
  char data[256];
  dns_record *rr, cname_rr, type_rr;
  int datalen, rc;

  /* DNS lookup failures get passed straight back. */

  if ((rc = dns_basic_lookup(name, type)) != DNS_SUCCEED) return rc;

  /* We should have either records of the required type, or a CNAME record,
  or both. We need to know whether both exist for getting the fully qualified
  name, but avoid scanning more than necessary. Note that we must copy the
  contents of any rr blocks returned by dns_next_rr() as they use the same static
  area. */

  cname_rr.data = type_rr.data = NULL;
  for (rr = dns_next_rr(RESET_ANSWERS); rr != NULL; rr = dns_next_rr(RESET_NEXT))
    {
    if (rr->type == type)
      {
      if (type_rr.data == NULL) type_rr = *rr;
      if (cname_rr.data != NULL) break;
      }
    else if (rr->type == T_CNAME) cname_rr = *rr;
    }

  /* If a CNAME was found, take the fully qualified name from it; otherwise
  from the first data record, if present. */

  if (fully_qualified_name != NULL)
    {
    if (cname_rr.data != NULL)
      {
      if (strcmp(cname_rr.name, *fully_qualified_name) != 0 &&
          cname_rr.name[0] != '*')
        *fully_qualified_name = string_copy(cname_rr.name);
      }
    else if (type_rr.data != NULL)
      {
      if (strcmp(type_rr.name, *fully_qualified_name) != 0 &&
          type_rr.name[0] != '*')
        *fully_qualified_name = string_copy(type_rr.name);
      }
    }

  /* If any data records of the correct type were found, we are done. */

  if (type_rr.data != NULL) return DNS_SUCCEED;

  /* If there are no data records, we need to re-scan the DNS using the
  domain given in the CNAME record, which should exist (otherwise we should
  have had a failure from dns_lookup). However code against the possibility of
  its not existing. */

  if (cname_rr.data == NULL) return DNS_FAIL;
  datalen = dn_expand(dns_answer, dns_answer + dns_answerlen,
    cname_rr.data, (DN_EXPAND_ARG4_TYPE)data, 256);
  if (datalen < 0) return DNS_FAIL;
  name = data;
  }

/*Control reaches here after 10 times round the CNAME loop. Something isn't
right... */

log_write(0, LOG_MAIN, "CNAME loop for %s encountered", orig_name);
return DNS_FAIL;
}



/*************************************************
*          Get address from DNS record           *
*************************************************/

/* The record type is either T_A for an IPv4 address or T_AAAA for an IPv6
address.

Argument:    the DNS record
Returns:     pointer to textual representation of address, in new store
*/

char *
dns_address_from_rr(dns_record *rr)
{
#if HAVE_IPV6
if (rr->type == T_AAAA)
  {
  char buffer[50];
  inet_ntop(AF_INET6, rr->data, buffer, 50);
  return string_copy(buffer);
  }
else
#endif

return string_sprintf("%d.%d.%d.%d",
  rr->data[0], rr->data[1], rr->data[2], rr->data[3]);
}

/* End of dns.c */
