/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: util.c,v 1.29 2002/11/01 10:00:13 martine Exp $
 */

#include "config.h" 

#include <gtk/gtk.h>
#include <glib.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>

#ifdef G_OS_WIN32
#include <direct.h> /* mkdir */
#endif

#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>

#include "md5.h"
#include "util.h"
#include "livejournal.xpm"
#include "icons.h"

gboolean
ljdate_to_tm(const char *ljtime, struct tm *ptm) {
	gboolean ret = TRUE;
	/* LiveJournal ignores seconds, apparently. */
	if (sscanf(ljtime, "%4d-%2d-%2d %2d:%2d", /*:%2d",*/
		&ptm->tm_year, &ptm->tm_mon, &ptm->tm_mday,
		&ptm->tm_hour, &ptm->tm_min) < 5)/*, &ptm->tm_sec) < 6)*/
		ret = FALSE;
	ptm->tm_year -= 1900;
	ptm->tm_mon -= 1;
	/* FIXME? convert to time_t and back, so the weekday and such get updated. 
	ttime = mktime(&tm);
	ptm = localtime(&ttime);*/
	return ret;
}
char*
tm_to_ljdate(struct tm *ptm) {
	/* LiveJournal ignores seconds, apparently. */
	return g_strdup_printf("%04d-%02d-%02d %02d:%02d", /*:%02d",*/
		ptm->tm_year+1900, ptm->tm_mon+1, ptm->tm_mday,
		ptm->tm_hour, ptm->tm_min); /*, ptm->tm_sec);*/
}

gint free_all_hash_cb(gpointer key, gpointer val, gpointer user_data) {
	g_free(key); g_free(val);
	return TRUE;
}

void hash_destroy(GHashTable* hash) {
	if (hash == NULL) return;
	g_hash_table_foreach_remove(hash, free_all_hash_cb, NULL);
	g_hash_table_destroy(hash);
}

GtkWidget* lj_table_new(int rows, int cols) {
	GtkWidget *table;
	table = gtk_table_new(rows, cols, FALSE);
	return table;
}
GtkWidget* lj_table_label(GtkTable *table, int row, const char *text) {
	GtkWidget *label = gtk_label_new_with_mnemonic(text);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.5f);
	gtk_table_attach(table, GTK_WIDGET(label), 
			0, 1, row, row+1, GTK_FILL, GTK_FILL, 2, 2);
	return label;
}
void lj_table_content(GtkTable *table, int row, GtkWidget *content) {
	gtk_table_attach(table, GTK_WIDGET(content), 
			1, 2, row, row+1, GTK_EXPAND|GTK_FILL, 0, 2, 2);
}
void lj_table_label_content_mne(GtkTable *table, int row, const char *text, GtkWidget *content, GtkWidget *mne) {
	GtkWidget *label;
	label = lj_table_label(table, row, text);
	lj_table_content(table, row, content);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), mne);
}
void lj_table_label_content(GtkTable *table, int row, const char *text, GtkWidget *content) {
	lj_table_label_content_mne(table, row, text, content, content);
}
void lj_table_fillrow(GtkTable *table, int row, GtkWidget *content) {
	gtk_table_attach(table, GTK_WIDGET(content), 
			0, 2, row, row+1, GTK_EXPAND|GTK_FILL, 0, 2, 2);
}

char* field_strdup_printf(char *fmt, char* fields[]) {
	GString *out;
	char *src = fmt;
	int index;
	char *ret;

	out = g_string_sized_new(256);

	while (*src != 0) {
		/* scan forward for a "%x" item */
		for ( ; *src != 0; src++) {
			if ((*src == '%') && (isdigit((int)*(src+1))))
				break;
			g_string_append_c(out, *src);
		}

		if (*src == 0) break;

		/* now read the %x item */
		src++;
		if (*src == 0) break; /* damnit, a % with no following char. */
		if (*src >= '0' && *src <= '9') {
			index = *src - '0';
			if (fields[index]) 
				g_string_append(out, fields[index-1]); /* 1-based index. */
		} else if (*src == '%') {
			/* %% -> % */
			g_string_append_c(out, *src);
		}
		src++;
	}
	ret = out->str;
	g_string_free(out, FALSE);
	return ret;
}

void
lj_dialog_init(GtkWidget *dlg, GtkWidget *parent, const gchar *title, int width, int height) {
	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(parent));
	if (title)
		gtk_window_set_title(GTK_WINDOW(dlg), title);
	if (width > 0) {
		if (height > 0) {
			gtk_window_set_default_size(GTK_WINDOW(dlg), width, height);
		} else {
			gtk_window_set_default_size(GTK_WINDOW(dlg), width, (int)(0.618*width));
		}
	}
	lj_win_set_icon(dlg);
}

GtkWidget*
lj_dialog_new(GtkWidget *parent, const gchar *title, int width, int height) {
	GtkWidget *dlg;
	dlg = gtk_dialog_new();
	lj_dialog_init(dlg, parent, title, width, height);
	return dlg;
}

GtkWidget*
lj_dialog_set_contents(GtkWidget *dlg, GtkWidget *contents) {
	GtkWidget *vbox;
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	gtk_box_pack_start(GTK_BOX(vbox), contents, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show_all(GTK_DIALOG(dlg)->vbox);
	return vbox;
}

void
lj_dialog_add_button(GtkWidget *dlg, GtkWidget *button) {
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->action_area), button, FALSE, FALSE, 0);
}

GtkWidget*
lj_dialog_buttonbox_new(void) {
	return gtk_hbox_new(FALSE, 5);
}
void
lj_dialog_buttonbox_add(GtkWidget *box, GtkWidget *button) {
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
}
GtkWidget*
lj_dialog_buttonbox_button_with_label(GtkWidget *box, const char *label) {
	GtkWidget *button;
	char buf[100];
	g_snprintf(buf, 100, "  %s  ", label);
	button = gtk_button_new_with_mnemonic(buf);
	lj_dialog_buttonbox_add(box, button);
	return button;
}

void
lj_dialog_set_contents_buttonbox(GtkWidget *dlg, GtkWidget *contents, GtkWidget *buttonbox) {
	GtkWidget *vbox, *hbox;;

	vbox = lj_dialog_set_contents(dlg, contents);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(hbox), buttonbox, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show_all(vbox);
}

void
lj_dialog_add_close(GtkWidget *dlg) {
	gtk_dialog_add_button(GTK_DIALOG(dlg), 
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
	
	g_signal_connect_object(G_OBJECT(dlg), "response",
			G_CALLBACK(gtk_widget_destroy), dlg,
			G_CONNECT_AFTER | G_CONNECT_SWAPPED);
}

GtkWidget*
lj_dialog_add_okcancel(GtkWidget *dlg, const char *okcaption) {
	GtkWidget *ok;
	gtk_dialog_add_button(GTK_DIALOG(dlg), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	ok = gtk_dialog_add_button(GTK_DIALOG(dlg), okcaption ? okcaption : GTK_STOCK_OK, GTK_RESPONSE_OK);
	return ok;
}
int lj_dialog_run(GtkWidget *dlg) {
	int res;
	res = gtk_dialog_run(GTK_DIALOG(dlg));
	if (res != GTK_RESPONSE_OK)
		gtk_widget_destroy(dlg);
	return res == GTK_RESPONSE_OK;
}
void lj_dialog_show(GtkWidget *dlg) {
	gtk_widget_show(dlg);
}

int lj_confirm(GtkWidget *parent, const char *title, const char *msg) {
	GtkWidget *dlg;
	int res;

	dlg = gtk_message_dialog_new(GTK_WINDOW(parent), 0,
			GTK_MESSAGE_QUESTION,
			GTK_BUTTONS_YES_NO,
			msg);
	lj_dialog_init(dlg, parent, title, -1, -1);
	res = (gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_YES);
	gtk_widget_destroy(dlg);
	return res;
}

/* give a message on stderr unless running in quiet mode */
void lj_cli_note(const char *msg, ...) {
	char buf[1024];
	va_list ap;

	if (app.quiet)
		return;

	va_start(ap, msg);
	g_vsnprintf(buf, 1024, msg, ap);
	va_end(ap);

	fprintf(stderr, buf);
}

void
lj_win_set_icon(GtkWidget *win) {
	static GdkPixmap *pix = NULL;
	static GdkBitmap *bit;

	gtk_widget_realize(win);
	if (pix == NULL) 
		pix = gdk_pixmap_create_from_xpm_d(win->window, 
				&bit, NULL, livejournal_xpm);
	gdk_window_set_icon(win->window, (GdkWindow*)NULL, pix, bit);
}

void
string_replace(char **dest, const char *src) {
	if (*dest) g_free(*dest);
	*dest = (char*)src; /* FIXME ick. can't really know 
	                       if dest is const or not. */
}

void
gdkcolor_to_hex(GdkColor *color, char* buf) {
	g_snprintf(buf, 8, "#%02X%02X%02X",
			color->red >> 8,
			color->green >> 8,
			color->blue >> 8);
}

GtkWidget*
scroll_wrap(GtkWidget *w) {
	GtkWidget *scroll;

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), 
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
			GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(scroll), w);
	return scroll;
}

void
geometry_save(GtkWidget *win, Geometry *geom) {
	if (win->window) {
		gtk_window_get_position(GTK_WINDOW(win), &geom->x, &geom->y);
		gtk_window_get_size(GTK_WINDOW(win), &geom->width, &geom->height);
	}
}

void
geometry_load(GtkWidget *win, Geometry *geom) {
	if (geom->width == 0)
		return;

	gtk_window_move(GTK_WINDOW(win), geom->x, geom->y);
	gtk_window_set_default_size(GTK_WINDOW(win), geom->width, geom->height);
}
static gboolean
geometry_save_event(GtkWidget *win, GdkEvent *e, Geometry *geom) {
	geometry_save(win, geom);
	return FALSE;
}
void geometry_tie(GtkWidget *win, GeometryType g) {
#ifdef G_OS_WIN32
	/* geometry is totally broken on win32--
	 * gtk_window_get_position() returns random values. */
	return;
#else
	Geometry *geom = &conf.geometries[g];

	/* load the existing geometry for this window */
	geometry_load(win, geom);
	/* and track the new geometry */
	g_signal_connect(G_OBJECT(win), "configure-event",
			G_CALLBACK(geometry_save_event), geom);
#endif
}

static inline gboolean
needsescape(char c) {
	return !isalnum(c) && c != '_';
}

char*
urlencode(unsigned char *string) {
	int escapecount = 0;
	unsigned char *src, *dest;
	unsigned char *newstr;
	
	char hextable[] = { '0', '1', '2', '3', '4', '5', '6', '7',
					'8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
	
	if (string == NULL) return NULL;

	for (src = string; *src != 0; src++) 
		if (needsescape(*src)) escapecount++;
	
	newstr = g_new(char, strlen(string) - escapecount + (escapecount * 3) + 1);
	
	src = string;
	dest = newstr;
	while (*src != 0) {
		if (needsescape(*src)) {
			*dest++ = '%';
			*dest++ = hextable[*src >> 4];
			*dest++ = hextable[*src & 0x0F];
			src++;
		} else {
			*dest++ = *src++;
		}
	}
	*dest = 0;

	return newstr;
}

unsigned char* 
urldecode(char *string) {
	int destlen = 0;
	unsigned char *src, *dest;
	unsigned char *newstr;
	
	if (string == NULL) return NULL;

	for (src = string; *src != 0; src++) {
		if (*src == '%') { src+=2; } /* FIXME: this isn't robust. should check
										the next two chars for 0 */
		destlen++;
	}
	
	newstr = g_new(char, destlen + 1);
	src = string;
	dest = newstr;
	while (*src != 0) {
		if (*src == '%') {
			char h = (char)toupper(src[1]);
			char l = (char)toupper(src[2]);
			char vh, vl;
			vh = isalpha(h) ? (10+(h-'A')) : (h-'0');
			vl = isalpha(l) ? (10+(l-'A')) : (l-'0');
			*dest++ = ((vh<<4)+vl);
			src += 3;
		} else if (*src == '+') {
			*dest++ = ' ';
			src++;
		} else {
			*dest++ = *src++;
		}
	}
	*dest = 0;

	return newstr;
}

const char*
util_skipline(const char *text) {
	while (*text != 0) {
		if (*text == '\n') {
			text++;
			break;
		}
		text++;
	}
	return text;
}

char*
util_getline(const char *text) {
	int len;
	char *str;

	for (len = 0; text[len] != 0; len++)
		if (text[len] == '\r' || text[len] == '\n') break;

	str = g_new(char, len+1);
	memcpy(str, text, len+1);
	str[len] = 0;
	return str;
}

/* GLib unfortunately doesn't make up its mind over whether TRUE or
 * FALSE should denote comparison success, so its provided g_str_equal
 * is useless as a custom plugin to g_list_find_custom. */
static gboolean _str_equal(gconstpointer v1, gconstpointer v2) {
	return !g_str_equal(v1, v2);
}

static void
forget_cb(GObject *obj, gboolean *state) {
	*state = !*state;
}

void
lj_message_va(GtkWidget *parent, MessageType type, gboolean forgettable,
		const char *title, const char *message, va_list ap) {
	GtkWidget *dlg;
	GtkWidget *forget_check;
	gint msgtype = 0, buttontype = 0;
	const gchar *mtitle = NULL;
	gint res;
	gboolean forget_state;
	
	gchar ourhash[120];
	
	char fullmsg[1024];

	if (app.cli)
		return;

	g_vsnprintf(fullmsg, 1024, message, ap);
	
	{ /* compute hash of this message */
		GString *id;
		id = g_string_new(title);
		g_string_append(id, message);
		md5_hash(id->str, ourhash);
		g_string_free(id, TRUE);
	}
	
	/* do nothing if the user has asked not to view this message again */
	if (g_list_find_custom(app.quiet_dlgs, ourhash, _str_equal))
		return;
	
	switch (type) {
		case LJ_MSG_INFO:
			msgtype = GTK_MESSAGE_INFO;
			buttontype = GTK_BUTTONS_OK;
			mtitle = (gchar*)title;
			break;
		case LJ_MSG_WARNING:
			msgtype = GTK_MESSAGE_WARNING;
			buttontype = GTK_BUTTONS_CLOSE;
			mtitle = title ? (gchar*)title : _("Warning");
			break;
	}
	
	/* TODO: switch to lj_dialogs, which are prettier */
	dlg = gtk_message_dialog_new(GTK_WINDOW(parent), 0, msgtype,
			buttontype,
			fullmsg);
	lj_dialog_init(dlg, parent, title, -1, -1);
	
	if (forgettable) {
		forget_state = FALSE;
		forget_check = gtk_check_button_new_with_label(_("Do not show again"));
		g_signal_connect(G_OBJECT(forget_check), "toggled",
				G_CALLBACK(forget_cb), &forget_state);
		gtk_box_set_homogeneous(GTK_BOX(GTK_DIALOG(dlg)->action_area),
				FALSE); /* XXX: this doesn't work :( */
		
		/* aggressively make this dialog less ugly */
		gtk_button_box_set_layout(GTK_BUTTON_BOX(GTK_DIALOG(dlg)->action_area),
				GTK_BUTTONBOX_EDGE);
		gtk_box_set_child_packing(GTK_BOX(GTK_DIALOG(dlg)->action_area),
				((GtkBoxChild*)GTK_BOX(GTK_DIALOG(dlg)->action_area)->
				 	children->data)->widget, FALSE, FALSE, 0, GTK_PACK_END);
				
		/* force our checkbox to *really* be first */
		gtk_container_add(GTK_CONTAINER((GTK_DIALOG(dlg)->action_area)),
				forget_check);
		gtk_box_reorder_child(GTK_BOX((GTK_DIALOG(dlg)->action_area)),
				forget_check, 0);
	
		gtk_widget_show_all(GTK_DIALOG(dlg)->action_area);
	}

	res = gtk_dialog_run(GTK_DIALOG(dlg));

	/* flag this dialog for oblivion if the user didn't like it */
	if (forgettable && forget_state)
		app.quiet_dlgs = g_list_append(app.quiet_dlgs, g_strdup(ourhash));

	gtk_widget_destroy(dlg);
}

void
lj_message(GtkWidget *parent, MessageType type, gboolean forgettable,
		const char *title, const char *message, ...) {
	va_list ap;
	va_start(ap, message);
	lj_message_va(parent, type, forgettable, title, message, ap);
	va_end(ap);
}

void lj_warning(GtkWidget *parent, const char *message, ...) {
	va_list ap;

	va_start(ap, message);
	lj_message_va(parent, LJ_MSG_WARNING, FALSE, NULL, message, ap);
	va_end(ap);
}

/* utility thunk function */
void lj_messagebox(GtkWidget *parent, const char *title, const char *message) {
	lj_message(parent, LJ_MSG_INFO, FALSE, title, message);
}

int
verify_dir(const char *path) {
#ifdef G_OS_WIN32
	if(mkdir(path) < 0 && errno != EEXIST) {
#else
	if (mkdir(path, 0700) < 0 && errno != EEXIST) {
#endif
		/* mode 0700 so other people can't peek at passwords! */
		perror("mkdir"); /* FIXME show error dialog. */
		return -1;
	}
	return 0;
}

void
md5_hash(const char *src, char *dest) {
	struct cvs_MD5Context context;
	unsigned char checksum[16];
	int i;

	cvs_MD5Init(&context);
	cvs_MD5Update(&context, (const unsigned char*)src, strlen(src));
	cvs_MD5Final(checksum, &context);
	for (i = 0; i < 16; i++)
		sprintf(&dest[i*2], "%02x", (unsigned int)checksum[i]);
}

void
strtoupper(char *str) {
	for (; *str; str++)
		*str = g_ascii_toupper(*str);
}

/* text sort utility function for GtkTreeModels */
gint
text_sort_func(GtkTreeModel *model, GtkTreeIter  *a, GtkTreeIter  *b,
		gpointer data) {
	gchar *ta, *tb;
	gint ret;
	gtk_tree_model_get(model, a, 0, &ta, -1);
	gtk_tree_model_get(model, b, 0, &tb, -1);
	ret = g_ascii_strcasecmp(ta, tb);
	g_free(ta);
	g_free(tb);
	return ret;
}

void
lj_widget_set_visible(GtkWidget *w, gboolean visible) {
	if (visible)
		gtk_widget_show(w);
	else
		gtk_widget_hide(w);
}
