/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: ljtypes.c,v 1.24 2002/11/28 03:09:28 martine Exp $
 */

#include "config.h"
#include <glib.h>

#ifndef G_OS_WIN32
#include <unistd.h>
#include <sys/types.h> /* for fork/wait (spawning editor) */
#include <sys/wait.h>
#else
#include <io.h> /* for open/close on Win32 */
#endif

#include <stdio.h>
#include <errno.h>
#include <libxml/tree.h>
#include <stdlib.h>  /* atoi. */
#include <string.h>  /* strchr. */
#include "util.h"
#include "network.h"
#include "security.h"
#include "ljtypes.h"
#include "xml_macros.h"

static Entry* entry_from_user_editor(const char *filename, GError **err);

void
friend_group_free(FriendGroup *fg) {
	conf_nid_free((NameIDHash*)fg);
}

Friend*
friend_new(void) {
	Friend *f = g_new0(Friend, 1);
	strcpy(f->foreground, "#000000");
	strcpy(f->background, "#FFFFFF");
	return f;
}

FriendType
friend_type_from_str(char *str) {
	if (str) {
		if (strcmp(str, "community") == 0) 
			return FRIEND_TYPE_COMMUNITY;
	} 
	return FRIEND_TYPE_USER;
}

void
friend_free(Friend *f) {
	g_free(f->username);
	g_free(f->fullname);
	g_free(f);
}

gint
friend_compare_username(gconstpointer a, gconstpointer b) {
	const Friend *fa = a;
	const Friend *fb = b;

	return strcmp(fa->username, fb->username);
}

/*
boxed friend type?
   
static Friend*
friend_copy(Friend *o) {
	Friend *f = g_new0(Friend, 1);
	f->username = g_strdup(o->username);
	f->fullname = g_strdup(o->fullname);
	f->foreground = o->foreground;
	f->background = o->background;
	f->conn = o->conn;
	f->type = o->type;
	f->groupmask = o->groupmask;
	return f;
}

GType friend_get_gtype(void) {
	static GType type = 0;
	if (type == 0)
		type = g_boxed_type_register_static("Friend",
				(GBoxedCopyFunc) friend_copy,
				(GBoxedFreeFunc) friend_free);
	return type;
}*/

void 
security_append_to_request(Security *security, NetRequest *request) {
	char *text = NULL;

	switch (security->type) {
		case SECURITY_PUBLIC:
			text = "public"; break;
		case SECURITY_PRIVATE:
			text = "private"; break;
		case SECURITY_FRIENDS:
		case SECURITY_CUSTOM:
			text = "usemask"; break;
	}
	net_request_copys(request, "security", text);

	if (security->type == SECURITY_FRIENDS) {
		net_request_seti(request, "allowmask", SECURITY_ALLOWMASK_FRIENDS);
	} else if (security->type == SECURITY_CUSTOM) {
		net_request_seti(request, "allowmask", security->allowmask);
	}
}

void
security_load_from_strings(Security *security, const char *sectext, const char *allowmask) {
	if (sectext) {
		if (g_ascii_strcasecmp(sectext, "public") == 0) {
			security->type = SECURITY_PUBLIC;
		} else if (g_ascii_strcasecmp(sectext, "private") == 0) {
			security->type = SECURITY_PRIVATE;
		} else if (g_ascii_strcasecmp(sectext, "usemask") == 0) {
			unsigned int am;
			am = atoi(allowmask);
			if (am == SECURITY_ALLOWMASK_FRIENDS) {
				security->type = SECURITY_FRIENDS;
			} else {
				security->type = SECURITY_CUSTOM;
				security->allowmask = am;
			}
		} else {
			g_warning("security: '%s' unhandled", sectext);
		}
	} else {
		security->type = SECURITY_PUBLIC;
	}
}

void 
security_load_from_result(Security *security, NetResult *result) {
	char *sectext, *allowmask;

	sectext = net_result_get(result, "security");
	if (sectext) {
		allowmask = net_result_get(result, "allowmask");
		security_load_from_strings(security, sectext, allowmask);
	}
}

Entry *
entry_new(void) {
	Entry *entry = g_new0(Entry, 1);
	entry->security = conf.defaultsecurity;
	return entry;
}
void
entry_free(Entry *e) {
	g_free(e->subject);
	g_free(e->event);
	g_free(e->mood);
	g_free(e->music);
	g_free(e->pickeyword);
	g_free(e);
}

char*
entry_make_summary(Entry *entry) {
	static char buf[100];
	if (entry->subject)
		return entry->subject;
	if (entry->event) {
		if (strlen(entry->event) < 100)
			return entry->event;
		g_strlcpy(buf, entry->event, 97);
		buf[96] = '.';
		buf[97] = '.';
		buf[98] = '.';
		buf[99] = 0;
		return buf;
	}
	return NULL;
}

void
entry_set_request_fields(Entry *entry, NetRequest *request) {
	struct tm *ptm = &entry->time;

	/* basic information */
	if (entry->itemid)
		net_request_seti(request, "itemid", entry->itemid);

	net_request_copys(request, "subject", entry->subject ? entry->subject : "");
	net_request_copys(request, "event",   entry->event);

	if (!ptm->tm_year) {
		time_t curtime_time_t = time(NULL);
		ptm = localtime(&curtime_time_t);
	}
	net_request_seti( request, "year", ptm->tm_year+1900);
	net_request_seti( request, "mon",  ptm->tm_mon+1);
	net_request_seti( request, "day",  ptm->tm_mday);
	net_request_seti( request, "hour", ptm->tm_hour);
	net_request_seti( request, "min",  ptm->tm_min);

	/* metadata */
/* http://www.livejournal.com/admin/schema/?mode=viewdata&table=logproplist */
	if (entry->mood) {
		int id = conf_mood_id_from_name(entry->mood);
		if (id >= 0) {
			net_request_seti(request, "prop_current_moodid", id);
		} else {
			net_request_copys(request, "prop_current_mood", entry->mood);
		}	
	} else {
		net_request_copys(request, "prop_current_mood", "");
	}

	net_request_copys(request, 
			"prop_current_music", entry->music ? entry->music : "");
	net_request_copys(request, 
			"prop_picture_keyword", entry->pickeyword ? entry->pickeyword : "");
	net_request_seti(request, "prop_opt_preformatted", entry->preformatted);
	net_request_seti(request, "prop_opt_nocomments", entry->comments == COMMENTS_DISABLE);
	net_request_seti(request, "prop_opt_noemail", entry->comments == COMMENTS_NOEMAIL);
	net_request_seti(request, "prop_opt_backdated", entry->backdated);

	security_append_to_request(&entry->security, request);
}

void
entry_load_metadata(Entry *entry, const char *key, const char *value) {
	g_return_if_fail(key != NULL);

	if (strcmp(key, "current_mood") == 0)
		entry->mood = g_strdup(value);
	else if (strcmp(key, "current_moodid") == 0) {
		int moodid;
		char *mood;
		moodid = atoi(value);
		mood = conf_nid_by_id(conf_cur_server()->moods, moodid);
		if (mood)
			entry->mood = g_strdup(mood);
		else
			lj_warning(NULL, _("entry_load_meta: unknown moodid '%s'."), value);
	} else if (strcmp(key, "current_music") == 0)
		entry->music = g_strdup(value);
	else if (strcmp(key, "picture_keyword") == 0)
		entry->pickeyword = g_strdup(value);
	else if (strcmp(key, "opt_preformatted") == 0)
		entry->preformatted = (value && value[0] == '1');
	else if (strcmp(key, "opt_nocomments") == 0)
		entry->comments = COMMENTS_DISABLE;
	else if (strcmp(key, "opt_noemail") == 0)
		entry->comments = COMMENTS_NOEMAIL;
	else if (strcmp(key, "opt_backdated") == 0)
		entry->backdated = (value && value[0] == '1');
#ifdef DEBUG
/* we currently don't have a DEBUG define,
 * but this code may be useful in the future. */
	else
		lj_warning(NULL, _("entry_load_meta: unknown key '%s'."), key);
#endif
}

static xmlNodePtr
addtextchildenc(xmlDocPtr doc, xmlNodePtr parent, char *name, char *val) {
	char *enc;
	xmlNodePtr node;
	enc = xmlEncodeEntitiesReentrant(doc, val);
	node = xmlNewTextChild(parent, NULL, name, val);
	xmlFree(enc);
	return node;
}

/* macros to simplify XML metadata access code
 * (a description of # and ## is in k&r2, A 12.3; 229-30)
 */

#define XML_ENTRY_META_GET(A)                                                \
    if ((!strcmp(cur->name, #A))) {	                                     \
        entry->##A = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1); \
    }

#define XML_ENTRY_META_SET(A)                            \
	if (entry->##A)                                 \
		addtextchildenc(doc, root, #A, entry->##A);

static xmlDocPtr
entry_to_xml(Entry *entry) {
	xmlDocPtr doc;
	xmlNodePtr root;

	doc = xmlNewDoc("1.0");
	root = xmlNewDocNode(doc, NULL, "entry", NULL);
	xmlDocSetRootElement(doc, root);

	if (entry->itemid) {
		char buf[10];
		g_snprintf(buf, 10, "%d", entry->itemid);
		xmlSetProp(root, "itemid", buf);
	}
	if (entry->time.tm_year) {
		char *ljdate = tm_to_ljdate(&entry->time);
		addtextchildenc(doc, root, "time", ljdate);
		g_free(ljdate);
	}
	XML_ENTRY_META_SET(subject);
	XML_ENTRY_META_SET(event);
	XML_ENTRY_META_SET(mood);
	XML_ENTRY_META_SET(music);
	XML_ENTRY_META_SET(pickeyword);

	return doc;
}

Entry*
entry_from_result(NetResult *result, int i) {
	Entry *entry;
	int propcount;
	char *pname, *pvalue;
	char buf[30];

	sprintf(buf, "events_%d_", i);

	entry = entry_new();
	entry->itemid = atoi(net_result_get_prefix(result, buf, "itemid"));
	ljdate_to_tm(net_result_get_prefix(result, buf, "eventtime"), &entry->time);
	entry->event = urldecode(net_result_get_prefix(result, buf, "event"));
	entry->subject = g_strdup(net_result_get_prefix(result, buf, "subject"));
	security_load_from_strings(&entry->security, 
			net_result_get_prefix(result, buf, "security"),
			net_result_get_prefix(result, buf, "allowmask"));

	/* FIXME: this is not quite the right way to do this...
	 * if we retrieved more than one entry, this is definitely wrong. */
	propcount = net_result_geti(result, "prop_count");
	for (i = 1; i <= propcount; i++) {
		sprintf(buf, "prop_%d_", i);
		pname = net_result_get_prefix(result,  buf, "name");
		pvalue = net_result_get_prefix(result, buf, "value");
		entry_load_metadata(entry, pname, pvalue);
	}

	return entry;
}

static gboolean
entry_load_from_xml(Entry *entry, const char *data, int len, GError **err) {
	xmlNodePtr cur;
	xmlDocPtr  doc = NULL;
	char *itemid;
	xmlParserCtxtPtr ctxt;

	ctxt = xmlCreatePushParserCtxt(NULL, NULL,
				data, 4,
				NULL /* XXX why does this want a filename? */);
	/* suppress error messages */
	ctxt->sax->warning = NULL;
	ctxt->sax->error   = NULL;

	xmlParseChunk(ctxt, data+4, len-4, 0);
	xmlParseChunk(ctxt, data, 0, 1);
	if (!ctxt->errNo)
		doc = ctxt->myDoc;

	xmlFreeParserCtxt(ctxt);

	if (!doc) {
		/* XXX better error message. */
		g_set_error(err,
				0,
				0,
				_("Error parsing XML"));
		return FALSE;
	}

	cur = xmlDocGetRootElement(doc);
	if ((itemid = xmlGetProp(cur, "itemid")) != NULL) {
		entry->itemid = atoi(itemid);
		xmlFree(itemid);
	}
	cur = cur->xmlChildrenNode;
	while (cur != NULL) {
		XML_ENTRY_META_GET(subject)
			else
		XML_ENTRY_META_GET(event)
			else
		XML_ENTRY_META_GET(mood)
			else
		XML_ENTRY_META_GET(music)
			else
		XML_ENTRY_META_GET(pickeyword)
			else
		if ((!strcmp(cur->name, "time"))) {
			char *date = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
			ljdate_to_tm(date, &entry->time);
			g_free(date);
		}

		cur = cur->next;
	}
	return TRUE;
}

static gboolean
rfc822_get_keyval(const char **bufp, char *key, char *val) {
	const char *buf = *bufp;
	const char *p = buf;

	/* move p up to the colon after the key */
	while (*p != ':' && (p-buf < 100)) {
		if (*p == 0) return FALSE;
		if (*p == '\n') return FALSE;
		p++;
	}
	memcpy(key, buf, p-buf);
	key[p-buf] = 0;

	/* move p up to the head of the value */
	p++;
	while (*p == ' ') {
		if (*p == 0) return FALSE;
		if (*p == 0) return FALSE;
		p++;
	}
	/* scoot buf up to the newline, so the
	 * value lies between p and buf */
	buf = p;
	while (*buf != 0 && *buf != '\n' && (buf-p < 1000))
		buf++;
	memcpy(val, p, buf-p);
	val[buf-p] = 0;

	if (*buf != 0)
		*bufp = buf + 1;
	else
		*bufp = buf;
	return TRUE;
}

static gboolean
rfc822_load_entry(const char *key, const char *val, Entry *entry) {
#define RFC822_GET(A)                 \
    if (g_ascii_strcasecmp(key, #A) == 0) { \
        entry->##A = g_strdup(val);   \
    }

	RFC822_GET(subject)
	else RFC822_GET(mood)
	else RFC822_GET(music)
	else RFC822_GET(pickeyword)
	else if (g_ascii_strcasecmp(key, "time") == 0) {
		ljdate_to_tm(val, &entry->time);
	}
	else return FALSE;

	return TRUE;
}

static gboolean
entry_load_from_rfc822(Entry *entry, const char *data, int len) {
	const char *buf = data;
	char key[1024], val[1024];

	/* if we don't get a keyval on the first line,
	 * we know we're not reading an rfc822 file. */
	if (!rfc822_get_keyval(&buf, key, val))
		return FALSE;
	/* if it's not a valid keyval,
	 * we know we're not reading an rfc822 file. */
	if (!rfc822_load_entry(key, val, entry))
		return FALSE;

	while (rfc822_get_keyval(&buf, key, val)) {
		rfc822_load_entry(key, val, entry);
	}

	/* buf is only advanced when we successfully get a keyval,
	 * so the first time we don't we're at the text of the entry. */
	while (*buf == '\n')
		buf++;
	entry->event = g_strdup(buf);

	return TRUE;
}

static gboolean
entry_load_from_plain(Entry *entry, const char *data, int len) {
	entry->event = g_strdup(data);
	return TRUE;
}

static gboolean
entry_load_from_autodetect(Entry *entry, const char *data, int len, GError **err) {
	if (entry_load_from_xml(entry, data, len, err))
		return TRUE;
	g_clear_error(err);
	if (entry_load_from_rfc822(entry, data, len))
		return TRUE;
	return entry_load_from_plain(entry, data, len);
}

gboolean
entry_load(Entry *entry, gchar *data, gsize len, EntryFileType type, GError **err) {
	switch (type) {
		case ENTRY_FILE_XML:
			return entry_load_from_xml(entry, data, len, err);
		case ENTRY_FILE_RFC822:
			return entry_load_from_rfc822(entry, data, len);
		case ENTRY_FILE_PLAIN:
			return entry_load_from_plain(entry, data, len);
		case ENTRY_FILE_AUTODETECT:
		default:
			return entry_load_from_autodetect(entry, data, len, err);
	}
}

Entry*
entry_new_from_filename(const char *filename, EntryFileType type, GError **err) {
	Entry *entry;
	gchar *data; gsize size;

	if (!g_file_get_contents(filename, &data, &size, err))
		return NULL;

	entry = entry_new();
	if (!entry_load(entry, data, size, type, err)) {
		g_free(data);
		entry_free(entry);
		return NULL;
	}

	g_free(data);
	return entry;
}

int
entry_to_xml_file(Entry *entry, const char *filename) {
	xmlDocPtr doc = entry_to_xml(entry);
	if (xmlSaveFormatFileEnc(filename, doc, "utf-8", TRUE) < 0) 
		return -1;
	xmlFreeDoc(doc);
	return 0;
}

/* the do/while kludge is to allow this macro to come in a blockless if */
#define _FILE_ERROR_THROW(err, message)                          \
	{                                                            \
		g_set_error(err, G_FILE_ERROR,                           \
				g_file_error_from_errno(errno), #message);       \
		return 0; /* works both as NULL and as FALSE, luckily */ \
	}

gboolean
entry_to_rfc822_file(Entry *entry, const char *filename, GError **err) {
	FILE *f = fopen(filename, "wb");
	if (!f)
		_FILE_ERROR_THROW(err, _("can't open rfc822 file"));

	/* XXX: add all other metadata fields */
	fprintf(f, "Subject: %s\n"
			"Mood: %s\n"
			"Music: %s\n"
			"PicKeyword: %s\n\n%s",
			entry->subject    ? entry->subject    : "", /* prevents */
			entry->mood       ? entry->mood       : "", /* "(null)" */
			entry->music      ? entry->music      : "",
			entry->pickeyword ? entry->pickeyword : "",
			entry->event      ? entry->event      : "");

	if (entry->time.tm_year) {
		char *ljdate;
		ljdate = tm_to_ljdate(&entry->time);
		fprintf(f, "Time: %s\n", ljdate);
		g_free(ljdate);
	}
	
	if (fclose(f) != 0)
		_FILE_ERROR_THROW(err, _("can't close rfc822 file"));
	
	return TRUE;
}

/* give the user a chance to edit an Entry with their favorite editor.
 * loads either an empty entry, one partially specified on the command
 * line, or whatever is in the draft, whichever is appropriate. Should
 * work correctly with -c (postcli) cmdline option.
 *
 * XXX update docs here
 * Note how unlike the entry_from_* functions, this acts on an out
 * variable containing an *existing* Entry. see entry_from_user_editor
 * for the lower-level service of forking off an editor on a file and
 * constructing an Entry as a result. */
gboolean
entry_edit_with_usereditor(Entry *entry, GError **err) {
	gchar   editfn[1024];
	gchar   draftpath[1024];
	Entry  *tmp_entry = NULL;
	gint    editfh;
	gchar   clean_headers[] =
		"Subject: \n"
		"Mood: \n"
		"Music: \n"
		"PicKeyword: \n\n";
	
	conf_make_path("draft", draftpath, 1024);

	conf_make_path("edit-XXXXXX", editfn, 1024);
	editfh = g_mkstemp(editfn);
	if (!editfh)
		_FILE_ERROR_THROW(err, _("can't make temp file"));
	
	if (conf.options.autosave && app.autoloaddraft_ok &&
			((tmp_entry = entry_new_from_filename(draftpath, ENTRY_FILE_XML, NULL)) != NULL)) {
		/* we're about to open the same file again */
		if (close(editfh) != 0)
			_FILE_ERROR_THROW(err, _("can't close temp file"));
		if (!entry_to_rfc822_file(tmp_entry, editfn, err))
			return FALSE; /* err has already been set */
		entry_free(tmp_entry); /* and _not_ g_free */
		
	} else if (entry->itemid != 0) { /* partially constructed (-s etc.) */
		/* we're about to open the same file again */
		if (close(editfh) != 0)
			_FILE_ERROR_THROW(err, _("can't close temp file"));
		if (!entry_to_rfc822_file(entry, editfn, err))
			return FALSE; /* err has already been set */
		
	} else {                         /* new entry */
		if (!write(editfh, clean_headers, strlen(clean_headers)))
			_FILE_ERROR_THROW(err, _("can't write to temp file"));
		if (close(editfh) != 0)
			_FILE_ERROR_THROW(err, _("can't close temp file"));
	}

	tmp_entry = entry_from_user_editor(editfn, err);
	memcpy(entry, tmp_entry, sizeof(Entry)); /* the ol' switcheroo */
	g_free(tmp_entry);                       /* and _not_ entry_free */

	if (unlink(editfn) != 0)
		_FILE_ERROR_THROW(err, _("can't unlink temp file"));

	return TRUE;
}

static Entry*
entry_from_user_editor(const char *filename, GError **err) {
#ifndef G_OS_WIN32
	gint   pid;
	Entry *entry;
	
	/* g_spawn* would do no good: it disassociates the tty. viva fork! */
	pid = fork();
	if (pid < 0) {                 /* fork error */
		g_set_error(err, G_SPAWN_ERROR, G_SPAWN_ERROR_FORK,
				g_strerror(errno));
		return NULL;
	}

	if (pid == 0) {                /* child */
		gchar *editor =
			(getenv("VISUAL") ? getenv("VISUAL") :
			 getenv("EDITOR") ? getenv("EDITOR") : "vi");
		execlp(editor, editor, filename, NULL);
		_exit(0);
	}

	/* parent */
	if (wait(NULL) != pid) {
		g_set_error(err, G_SPAWN_ERROR, G_SPAWN_ERROR_FAILED,
				g_strerror(errno));
		return NULL;
	}

	if (!(entry = entry_new_from_filename(filename, ENTRY_FILE_RFC822, err))) {
		return NULL; /* err has already been already set; propagate it */
	}
	
	app.autoloaddraft_ok = FALSE;

	return entry;
	
#else
	g_warning("external editor option not supported on Win32 yet. Sorry.\n");
	return NULL;
#endif /* ndef G_OS_WIN32 */
}
