/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: init.c,v 1.23 2002/11/26 09:39:16 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <glib.h>

#define _GNU_SOURCE
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifndef G_OS_WIN32
#include <unistd.h>
#endif

#include <stdlib.h>  /* for exit (and __argc and __argv on windows) */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <locale.h>

#include "network.h"
#include "conf.h"
#include "util.h"
#include "login.h"
#include "lj.h"
#include "spawn.h"
#include "icons.h"
#include "checkfriends.h"

enum {
	LJ_ARGS_OPTION,
	LJ_ARGS_USER_UNKNOWN,
	LJ_ARGS_NO_USER,
	LJ_ARGS_NO_POST,
	LJ_ARGS_CF_OPTION,

	LJ_ARGS_FAILED
};

#define LJ_ARGS lj_args_error_quark()

static GQuark
lj_args_error_quark (void) {
	static GQuark q = 0;
	if (q == 0)
		q = g_quark_from_static_string ("lj-args-error-quark");

	return q;
}

static void
print_header(char *name) {
	g_print("LogJam %s\nCopyright (C) 2000-2002 Evan Martin\n\n", VERSION);
}

static void
print_help(char *name) {
#include "helptext.h" /* autogenerated */
	print_header(name);
	g_print(_(help), name);
#ifndef HAVE_GETOPT_LONG
	g_print(_("(Long options aren't supported on this system.)\n"));
#endif
}

static int
post_cli(Entry *entry) {
	NetRequest *request;
	NetResult *result;

	time_t curtime_time_t = time(NULL);
	struct tm *curtime = localtime(&curtime_time_t);

	entry->time = *curtime;

	request = net_request_new("postevent");
	entry_set_request_fields(entry, request);
	result = net_request_run_cli(request);
	net_request_free(request);
	
	if (net_result_succeeded(result)) {
		g_print(_("Success.\n"));
	} else {
		net_result_free(result);
		return -1;
	}
	net_result_free(result);

	return 0;
}

void set_defaults() {
	if (conf.spawn_command == NULL)
		conf.spawn_command = g_strdup(spawn_commands[0]);

	conf_verify_a_server_exists();

	if (conf.servercur == NULL) {
		conf.servercur = conf.servers->data;
	}

	if (conf.cfuserinterval == 0)
		conf.cfuserinterval = 30;
	if (conf.cfmask == NULL)
		conf.cfmask = g_strdup("0");
	cf_threshold_normalize(&conf.cfthreshold);
}

#ifdef HAVE_GETOPT_H
/* may return false with no error, eg when -h is issued on the
 * command line. check err to see if an error occured. */
static int
parse_options(int argc, char* argv[], Entry *entry, GError **err) {
#include "cmdlineopts_short.h"
#ifdef HAVE_GETOPT_LONG
#include "cmdlineopts_long.h"
#endif
	int c = 0;
	char *username = NULL, *password = NULL, *postas = NULL, *cf_mode = NULL;
	gchar *opt_subject = NULL, *opt_pickeyword = NULL;
	int postcli = FALSE;
	gboolean cf_cli = FALSE;
	gboolean use_editor = FALSE;

	while (c != -1) {

#ifdef HAVE_GETOPT_LONG
		c = getopt_long(argc, argv, short_options, long_options, NULL);
#else
		c = getopt(argc, argv, short_options);
#endif
		
		switch (c) {
			case 'h':
				print_help(argv[0]);
				return -1;
			case 'v':
				print_header(argv[0]);
				return -1;
			case 'c':
				postcli = TRUE;
				break;
			case 'u':
				username = optarg;
				break;
			case 'a':
				postas = optarg;
				break;
			case 'p':
				password = optarg;
				break;
			case 's':
				opt_subject          = optarg;
				entry->subject       = optarg;
				app.autoloaddraft_ok = FALSE;
				entry->itemid        = -1;
				break;
			case 'k':
				opt_pickeyword       = optarg;
				entry-> pickeyword   = optarg;
				app.autoloaddraft_ok = FALSE;
				entry->itemid        = -1;
				break;
			case 'e':
				use_editor = TRUE;
				break;
			case 'r':
				cf_cli  = TRUE;
				cf_mode = optarg;
				break;
			case 'q':
				app.quiet = TRUE;
				break;
			case '?': /* unknown option character. */
			case ':': /* missing option character. */
				g_set_error(err, LJ_ARGS, LJ_ARGS_OPTION,
						_("unkown or missing option character"));
				return -1;
		}
	}

	if (optind < argc) { /* read entry from file (possibly override meta) */
		gchar *contents; gsize len;

		app.autoloaddraft_ok = FALSE;
		if (strcmp(argv[optind], "-") == 0) {
			GString *str = g_string_new(NULL);
			char buf[1024];
			while ((len = fread(buf, 1, 1024, stdin)) > 0)
				g_string_append_len(str, buf, len);
			contents = str->str;
			len = str->len;
			g_string_free(str, FALSE);
		} else {
			if (!g_file_get_contents(argv[optind], &contents, &len, err))
				return -1;
		}

		if (!entry_load(entry, contents, len, ENTRY_FILE_AUTODETECT, err)) {
			g_free(contents);
			return -1;
		}
		g_free(contents);

		if (opt_pickeyword)
			string_replace(&entry->pickeyword, opt_pickeyword);
		if (opt_subject)
			string_replace(&entry->subject, opt_subject);

		if (entry->itemid == 0) { /* signals external editor that the */
			entry->itemid = -1;   /* message is partially constructed */
		}
	}

	if (username) {
		User *u;

		u = conf_user_by_username(conf_cur_server(), username);
		if (!u) {
			g_set_error(err, LJ_ARGS, LJ_ARGS_USER_UNKNOWN,
					_("Unknown user: %s.\n"), username);
			return -1;
		}

		if (password) {
			string_replace(&u->password, g_strdup(password));
		} 
		if (!u->password) {
			if ((password = getpass(_("Password: "))) != NULL)
				u->password = g_strdup(password);
		}
		conf_cur_user() = u;
	}

	if (postas) 
		string_replace(&conf.usejournal, g_strdup(postas));

	if (use_editor && !entry_edit_with_usereditor(entry, err)) {
		return -1;
	}

	if (postcli) {
		if (!conf_cur_user()) {
			g_set_error(err, LJ_ARGS, LJ_ARGS_NO_USER, _("No user specified"));
			return -1;
		}
		if (entry->event == NULL) {
			g_set_error(err, LJ_ARGS, LJ_ARGS_NO_POST, _("No post specified"));
			g_print(_("No post specified!\n"));
			return -1;
		}

		if (post_cli(entry) == 0)
			/* if the post succeeded, don't start the gui. */
			exit(0);
	}

	if (cf_cli) {
		gchar *ljid = g_strdup_printf("%s@%s",
				conf_cur_user()->username, conf_cur_server()->name);
		app.cli = TRUE;

		if (cf_mode == NULL) {
			if (checkfriends_cli(ljid)) {
				g_print("1\n"); exit(0); /* NEW */
			} else {
				g_print("0\n"); exit(1); /* no new entries, or some error */
			}
		}
		if (!g_ascii_strncasecmp("purge", cf_mode,
				strlen("purge"))) {
			checkfriends_cli_purge(ljid);
			g_print(_("checkfriends information for %s purged.\n"), ljid);
			exit(0);
		}
		g_set_error(err, LJ_ARGS, LJ_ARGS_CF_OPTION,
				_("Unknown argument %s to --checkfriends\n"), cf_mode);
		return -1;
	}

	return 0;
}
#endif /* HAVE_GETOPT_H */

static void
init_app(int *argc, gchar *argv[]) {
	gint i, shift = 0;
	
	memset(&app, 0, sizeof(Application));
	app.programname      = argv[0]; /* not the same as PROGRAMNAME */
	app.autosave_ok      = TRUE;
	app.autoloaddraft_ok = TRUE;
	
	/* treatment of configuration dir; must be done before read_conf */
	/* XXX: I wish there were a more concise way to do this (Perl!:) */
	for (i = 1; i < *argc; i++) {                    /* look for arg */
		if (!strcmp(argv[i], "--conf-dir")) {
			app.conf_dir = argv[i+1];
			shift = 2;
			break;
		}
		if (g_strrstr(argv[i], "--conf-dir=")) {
			app.conf_dir = argv[i] + strlen("--conf-dir=");
			shift = 1;
			break;
		}
	}
	if (shift) {                           /* shift what comes after */
		*argc -= shift;
		for (; i < *argc; i++) {
			argv[i] = argv[i + shift];
		}
	}
}

int
main(int argc, char* argv[]) {
	gboolean has_gtk;
	Entry  *entry;
#ifdef HAVE_GETOPT_H
	GError *err   = NULL;
#endif

	setlocale(LC_ALL, "");
#ifdef ENABLE_NLS
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif

#ifdef G_OS_WIN32
	gtk_rc_add_default_file("gtkrc");
#endif
	has_gtk = gtk_init_check(&argc, &argv);
	g_type_init();

	init_app(&argc, argv); /* should be called before conf_read */
	conf_read(&conf);
	set_defaults();

#ifndef G_OS_WIN32
	signal(SIGCHLD, SIG_IGN);
#endif

	entry = entry_new();  /* should be called after conf_read,
	                         so we can get the default security. */
#ifdef HAVE_GETOPT_H
	if (parse_options(argc, argv, entry, &err) < 0) {
		if (err) {
			g_print(_("Error: %s\n"), err->message);
			g_error_free(err);
		}
		return -1;
	}
#endif

	if (!has_gtk) {
		g_print(_("GTK init failed -- are you in X?\n\n"));
		print_help(argv[0]);
		return -1;
	}

	icons_initialize();
	lj_run(entry);

	conf_write(&conf);

	return 0;
}

#ifdef G_OS_WIN32
int _stdcall WinMain(void *hInst, void *hPrev, char* lpCmdLine, int nCmdShow) {
	return main(__argc, __argv);
}
#endif

