/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: console.c,v 1.9 2002/10/02 02:04:27 martine Exp $
 */
#include <config.h>

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "util.h"
#include "network.h"

typedef struct {
	GtkWidget *win;
	GtkWidget *entry, *display;
	GtkTextMark *mark_end;
} ConsoleUI;

static void
console_print(ConsoleUI *cui, const char *type, const char *text, ...) {
	char buf[4096];
	va_list ap;
	GtkTextBuffer *buffer;
	GtkTextIter iter;

	va_start(ap, text);
	g_vsnprintf(buf, 4096, text, ap);
	va_end(ap);

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(cui->display));
	gtk_text_buffer_get_end_iter(buffer, &iter);
	if (type) {
		gtk_text_buffer_insert_with_tags_by_name(buffer, &iter,
				buf, -1, type, NULL);
	} else {
		gtk_text_buffer_insert(buffer, &iter, buf, -1);
	}
	gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(cui->display),
			cui->mark_end, 0.0, FALSE, 0, 0);
}

static gboolean
run_command(ConsoleUI *cui, const char *command) {
	NetRequest *request;
	NetResult *result;
	char *linetext, *linetype;
	int line, linecount;

	console_print(cui, NULL, _("Running command: '"));
	console_print(cui, "command", command);
	console_print(cui, NULL, "'.\n");

	request = net_request_new("consolecommand");
	net_request_copys(request, "command", command);
	result = net_request_run(cui->win, _("Running Command..."), request);
	net_request_free(request);

	if (!net_result_succeeded(result)) {
		console_print(cui, _("error"), _("Error running command.\n"));
		net_result_free(result);
		return FALSE;
	}

	linecount = net_result_geti(result, "cmd_line_count");

	for (line = 1; line <= linecount; line++) {
		char buf[30];
		g_snprintf(buf, 30, "cmd_line_%d", line);
		linetext = net_result_get(result, buf);
		linetype = net_result_get_prefix(result, buf, "_type");
		if (linetype == NULL)
			linetype = "info";

		console_print(cui, linetype, linetext);
		console_print(cui, linetype, "\n");
	}
	
	net_result_free(result);
	return TRUE;
}

static void
submit_cb(GtkWidget *w, ConsoleUI *cui) {
	char *command;

	command = gtk_editable_get_chars(GTK_EDITABLE(cui->entry), 0, -1);
	if (run_command(cui, command))
		gtk_editable_delete_text(GTK_EDITABLE(cui->entry), 0, -1);
	g_free(command);
}

static void
console_help(GtkWidget *w, ConsoleUI *cui) {
	run_command(cui, "help");
}

void
console_dialog_run(GtkWidget *parent) {
	ConsoleUI *cui;
	GtkWidget *vbox, *hbox, *button;
	GtkTextBuffer *buffer;
	GtkTextIter end;

	cui = g_new0(ConsoleUI, 1);
	cui->win = lj_dialog_new(parent, _("LiveJournal Console"), 300, 300);
	g_signal_connect_swapped(G_OBJECT(cui->win), "destroy",
			G_CALLBACK(g_free), cui);
	geometry_tie(cui->win, GEOM_CONSOLE);

	vbox = gtk_vbox_new(FALSE, 5);

	hbox = gtk_hbox_new(FALSE, 5);
	cui->entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), cui->entry, TRUE, TRUE, 0);

	button = gtk_button_new_with_mnemonic(_(" _Submit "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(cui->entry), "activate",
			G_CALLBACK(submit_cb), cui);
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(submit_cb), cui);

	button = gtk_button_new_from_stock(GTK_STOCK_HELP);
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(console_help), cui);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	cui->display = gtk_text_view_new();
	gtk_text_view_set_editable(GTK_TEXT_VIEW(cui->display), FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), scroll_wrap(cui->display), 
			TRUE, TRUE, 0);

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(cui->display));
	gtk_text_buffer_get_end_iter(buffer, &end);
	cui->mark_end = gtk_text_buffer_create_mark(buffer, NULL, &end, FALSE);
	gtk_text_buffer_create_tag(buffer, "command",
			"family", "monospace",
			 NULL);
	gtk_text_buffer_create_tag(buffer, "info",
			"family", "monospace",
			"foreground", "darkgreen",
			NULL);
	gtk_text_buffer_create_tag(buffer, "error",
			"family", "monospace",
			"weight", PANGO_WEIGHT_BOLD,
			"foreground", "red",
			NULL);

	lj_dialog_add_close(cui->win);

	lj_dialog_set_contents(cui->win, vbox);

	console_print(cui, NULL, _("LiveJournal console ready.\n"));

	lj_dialog_show(cui->win);
}
