/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: conf_xml.c,v 1.19 2002/09/24 07:57:50 martine Exp $
 */

#include "config.h"

#include <libxml/parser.h>
#include <libxml/tree.h>

#include <glib.h>

#include <stdlib.h> /* atoi */
#include "util.h"
#include "conf.h"
#include "conf_xml.h"
#include "xml_macros.h"

/* it seems this should be doable in a cleaner manner.
 * i wish i were writing this in python. :(
 */
static char*
xmlGetString(xmlDocPtr doc, xmlNodePtr node) {
	/* we can't guarantee that xmlFree and g_free do the same thing,
	 * so we must copy this string into a glib-allocated string. */
	xmlChar *str = xmlNodeListGetString(doc, node->xmlChildrenNode, TRUE);
	char *ret = g_strdup(str);
	g_free(str);
	return ret;
}
static int
xmlGetInt(xmlDocPtr doc, xmlNodePtr node) {
	int ret = -1;
	char *value = xmlNodeListGetString(doc, node->xmlChildrenNode, TRUE);
	if (value) {
		ret = atoi(value);
		xmlFree(value);
	}
	return ret;
}
static xmlNodePtr
xmlAddTN(xmlNodePtr node, char *name, char *val) {
	/* this was used to insert a newline after every node,
	 * but there's a flag to libxml to do it for us.
	 * instead, we simply create a normal child with no namespace.
	 */
	return xmlNewTextChild(node, NULL, name, val);
	/*xmlAddChild(node, xmlNewText("\n"));*/
}

/* we use a bunch of macro magic to make this simpler.
 * (see xml_macros.h) 
 */

/* this should match the enum in conf.h */
static char* geometry_names[] = {
	"main",
	"login",
	"friends",
	"friendgroups",
	"console",
	"manager",
	"cffloat"
};

static void
parsegeometry(Geometry *geom, xmlDocPtr doc, xmlNodePtr node) {
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_INT("x", geom->x)
		XML_GET_INT("y", geom->y)
		XML_GET_INT("width", geom->width)
		XML_GET_INT("height", geom->height)
		XML_GET_END("parsegeometry")
	}
}

static void
parsegeometries(Configuration *c, xmlDocPtr doc, xmlNodePtr node) {
	char *window;
	int i;
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (xmlStrcmp(node->name, "geometry") == 0) {
			window = xmlGetProp(node, "window");
			if (!window) continue;

			for (i = 0; i < GEOM_COUNT; i++) {
				if (g_ascii_strcasecmp(window, geometry_names[i]) == 0) {
					parsegeometry(&c->geometries[i], doc, node);
					break;
				}
			}
			xmlFree(window);
		}
	}
}

static GList*
parsepickws(xmlDocPtr doc, xmlNodePtr node) {
	GList *pickws = NULL;
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_LIST("pickw", pickws, xmlGetString)
		XML_GET_END("parsepickws")
	}
	return pickws;
}

static GList*
parsefriendgroups(xmlDocPtr doc, xmlNodePtr node) {
	GList *fgs = NULL;
	gchar *id, *ispublic;
	FriendGroup *fg;
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (xmlStrcmp(node->name, "friendgroup") == 0) {
			id = xmlGetProp(node, "id");
			if (!id) continue;
			fg = g_new0(FriendGroup, 1);
			
			fg->id = atoi(id);
			xmlFree(id);
			
			fg->name = xmlGetString(doc, node);
			
			ispublic = xmlGetProp(node, "ispublic");
			if (!ispublic) continue;
			fg->ispublic = atoi(ispublic);
			xmlFree(ispublic);

			fgs = g_list_append(fgs, fg);
		} else
		XML_GET_END("parsefriendgroups")
	}
	return fgs;
}

static User*
parseuser(xmlDocPtr doc, xmlNodePtr node) {
	User *user;
	user = g_new0(User, 1);
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_STR("username", user->username)
		XML_GET_STR("fullname", user->fullname)
		XML_GET_STR("password", user->password)
		XML_GET_BOOL("fastserver", user->fastserver)
		XML_GET_FUNC("pickws", user->pickws, parsepickws)
		XML_GET_FUNC("friendgroups", user->friendgroups, parsefriendgroups)
		XML_GET_END("parseuser")
	}
	user->remember_user = TRUE;
	if (user->password)
		user->remember_password = TRUE;
	return user;
}

static GList*
parseusers(xmlDocPtr doc, xmlNodePtr node) {
	GList *users = NULL;
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_LIST("user", users, parseuser)
		XML_GET_END("parseusers")
	}
	return users;
}

static GList*
parsemoods(xmlDocPtr doc, xmlNodePtr node) {
	GList *moods = NULL;
	NameIDHash *nid;

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		if (xmlStrcmp(node->name, "mood") == 0) {
			char *idstr;
			nid = g_new0(NameIDHash, 1);
			idstr = xmlGetProp(node, "id");
			nid->id = atoi(idstr);
			xmlFree(idstr);

			nid->name = xmlGetString(doc, node);

			moods = g_list_append(moods, nid);
		} else XML_GET_END("parsemoods")
	}
	return moods;
}

static Server*
parseserver(xmlDocPtr doc, xmlNodePtr node) {
	Server *server;
	char *username = NULL;

	server = g_new0(Server, 1);

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_STR("name", server->name)
		XML_GET_STR("url", server->url)
		XML_GET_FUNC("users", server->users, parseusers)
		XML_GET_STR("currentuser", username)
		XML_GET_FUNC("moods", server->moods, parsemoods)
		XML_GET_END("parseserver")
	}

	/* after we've parsed all of the users, scan for the current one. */
	if (username) {
		server->usercur = conf_user_by_username(server, username);
		g_free(username);
	}

	return server;
}

static GList*
parseservers(xmlDocPtr doc, xmlNodePtr node) {
	GList *servers = NULL;
	//char *servername = NULL;

	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_LIST("server", servers, parseserver)
		XML_GET_END("parseservers")
	}
	return servers;
}

static void
parseoptions(Options *options, xmlDocPtr doc, xmlNodePtr node) {
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_BOOL("revertusejournal", options->revertusejournal)
		XML_GET_BOOL("keepmetadata", options->keepmetadata)
		XML_GET_BOOL("autosave", options->autosave)
		XML_GET_BOOL("noautologin", options->noautologin)
		XML_GET_BOOL("netdump", options->netdump)
		XML_GET_BOOL("nofork", options->nofork)
		XML_GET_BOOL("useproxy", options->useproxy)
		XML_GET_BOOL("useproxyauth", options->useproxyauth)
#ifdef HAVE_GTKSPELL
		XML_GET_BOOL("usespellcheck", options->usespellcheck)
#endif
		XML_GET_BOOL("cfautostart", options->cfautostart)
		XML_GET_BOOL("cfusemask", options->cfusemask)
		XML_GET_BOOL("cfautofloat", options->cfautofloat)
		XML_GET_BOOL("cfautofloatraise", options->cfautofloatraise)
		XML_GET_BOOL("friends_statsvis", options->friends_statsvis)
		XML_GET_BOOL("friends_filtervis", options->friends_filtervis)
		XML_GET_END("parseoptions")
	}
}

static void
parsesecurity(Configuration *c, xmlDocPtr doc, xmlNodePtr node) {
	char *typestr = NULL;
	SecurityType type = SECURITY_PUBLIC;

	typestr = xmlGetProp(node, "type");

	if (typestr) {
		if (g_ascii_strcasecmp(typestr, "public") == 0) {
			type = SECURITY_PUBLIC;
		} else if (g_ascii_strcasecmp(typestr, "friends") == 0) {
			type = SECURITY_FRIENDS;
		} else if (g_ascii_strcasecmp(typestr, "private") == 0) {
			type = SECURITY_PRIVATE;
		}
		xmlFree(typestr);
	}

	c->defaultsecurity.type = type;
}

static void
parseproxyauth(Configuration *c, xmlDocPtr doc, xmlNodePtr node) {
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_STR("username", c->proxyuser)
		XML_GET_STR("password", c->proxypass)
		XML_GET_END("parseproxyauth")
	}
}

static GList*
parsequietdlgs(xmlDocPtr doc, xmlNodePtr node) {
	GList *quiet_dlgs = NULL;
	for (node = node->xmlChildrenNode; node != NULL; node = node->next) {
		XML_GET_LIST("quiet_dlg", quiet_dlgs, xmlGetString)
		XML_GET_END("parsequietdlgs")
	}
	return quiet_dlgs;
}

int
conf_xml_read(Configuration *c, char *path) {
	xmlDocPtr doc;
	xmlNodePtr node;
	char *servername = NULL;
	
	if (!g_file_test(path, G_FILE_TEST_EXISTS))
		return -1;

	doc = xmlParseFile(path);
	if (doc == NULL) {
		fprintf(stderr, _("error parsing configuration file.\n"));
		return -1;
	}

	/* we get the root element instead of the doc's first child
	 * because XML allows comments outside of the root element. */
	node = xmlDocGetRootElement(doc);
	if (node == NULL) {
		fprintf(stderr, _("empty document.\n"));
		xmlFreeDoc(doc);
		return -1;
	}

	node = node->xmlChildrenNode;
	for (; node != NULL; node = node->next) {
		XML_GET_FUNC("servers", c->servers, parseservers)
		XML_GET_STR("currentserver", servername)
		if (xmlStrcmp(node->name, "geometries") == 0) {
			parsegeometries(c, doc, node);
		} else if (xmlStrcmp(node->name, "options") == 0) {
			parseoptions(&c->options, doc, node);
		} else if (xmlStrcmp(node->name, "defaultsecurity") == 0) {
			parsesecurity(c, doc, node);
		} else if (xmlStrcmp(node->name, "proxyauth") == 0) {
			parseproxyauth(c, doc, node);
		} else 
		XML_GET_STR("uifont", c->uifont)
		XML_GET_STR("proxy", c->proxy)
		XML_GET_STR("spawncommand", c->spawn_command)
		XML_GET_STR("cfmask", c->cfmask)
		XML_GET_INT("cfuserinterval", c->cfuserinterval)
		XML_GET_INT("cfthreshold", c->cfthreshold)
		XML_GET_FUNC("quiet_dlgs", app.quiet_dlgs, parsequietdlgs)
		XML_GET_END("conf_xml_read")
	}

	if (servername) {
		c->servercur = conf_server_by_name(c, servername);
		g_free(servername);
	}

	xmlFreeDoc(doc);

	return 0;
}

static void
writeuser(User *user, xmlNodePtr node) {
	GList *l;
	xmlNodePtr xmlList, child;
	FriendGroup *fg;
	char buf[10];

	if (!user->remember_user)
		return;

	node = xmlNewChild(node, NULL, "user", NULL);
	xmlAddTN(node, "username", user->username);
	if (user->remember_password)
		xmlAddTN(node, "password", user->password);
	if (user->fullname)
		xmlAddTN(node, "fullname", user->fullname);
	if (user->fastserver)
		xmlNewChild(node, NULL, "fastserver", NULL);

	if (user->pickws) {
		xmlList = xmlNewChild(node, NULL, "pickws", NULL);
		for (l = user->pickws; l != NULL; l = l->next) {
			xmlAddTN(xmlList, "pickw", (char*)l->data);
		}
	}

	if (user->friendgroups) {
		xmlList = xmlNewChild(node, NULL, "friendgroups", NULL);
		for (l = user->friendgroups; l != NULL; l = l->next) {
			fg = (FriendGroup*)l->data;
			child = xmlAddTN(xmlList, "friendgroup", fg->name);
			g_snprintf(buf, 10, "%d", fg->id);
			xmlSetProp(child, "id", buf);
			g_snprintf(buf, 10, "%d", fg->ispublic);
			xmlSetProp(child, "ispublic", buf);
		}
	}
}

static void
writeserver(Server *server, xmlNodePtr node) {
	GList *l;
	xmlNodePtr servernode;

	servernode = xmlNewChild(node, NULL, "server", NULL);
	xmlAddTN(servernode, "name", server->name);
	xmlAddTN(servernode, "url", server->url);

	node = xmlNewChild(servernode, NULL, "users", NULL);
	for (l = server->users; l != NULL; l = l->next) 
		writeuser(l->data, node);

	if (server->usercur)
		xmlAddTN(servernode, "currentuser", server->usercur->username);

	node = xmlNewChild(servernode, NULL, "moods", NULL);
	for (l = server->moods; l != NULL; l = l->next) {
		Mood *mood = (Mood*)l->data;
		char buf[10];
		xmlNodePtr moodnode = xmlAddTN(node, "mood", mood->name);
		
		sprintf(buf, "%d", mood->id);
		xmlSetProp(moodnode, "id", buf);
	}
}

static void
writegeometry(Geometry *geometry, char *name, xmlNodePtr node) {
	char buf[10];

	node = xmlNewChild(node, NULL, "geometry", NULL);
	xmlSetProp(node, "window", name);

	sprintf(buf, "%d", geometry->x);
	xmlAddTN(node, "x", buf);

	sprintf(buf, "%d", geometry->y);
	xmlAddTN(node, "y", buf);

	sprintf(buf, "%d", geometry->width);
	xmlAddTN(node, "width", buf);

	sprintf(buf, "%d", geometry->height);
	xmlAddTN(node, "height", buf);
}

static void
writeoptions(Options *options, xmlNodePtr node) {
	if (options->revertusejournal)
		xmlNewChild(node, NULL, "revertusejournal", NULL);
	if (options->autosave)
		xmlNewChild(node, NULL, "autosave", NULL);
	if (options->noautologin)
		xmlNewChild(node, NULL, "noautologin", NULL);
	if (options->keepmetadata)
		xmlNewChild(node, NULL, "keepmetadata", NULL);
	if (options->netdump)
		xmlNewChild(node, NULL, "netdump", NULL);
	if (options->nofork)
		xmlNewChild(node, NULL, "nofork", NULL);
	if (options->useproxy)
		xmlNewChild(node, NULL, "useproxy", NULL);
	if (options->useproxyauth)
		xmlNewChild(node, NULL, "useproxyauth", NULL);
#ifdef HAVE_GTKSPELL
	if (options->usespellcheck)
		xmlNewChild(node, NULL, "usespellcheck", NULL);
#endif
	if (options->cfautostart)
		xmlNewChild(node, NULL, "cfautostart", NULL);
	if (options->cfusemask)
		xmlNewChild(node, NULL, "cfusemask", NULL);
	if (options->cfautofloatraise)
		xmlNewChild(node, NULL, "cfautofloatraise", NULL);
	if (options->cfautofloat)
		xmlNewChild(node, NULL, "cfautofloat", NULL);
	if (options->friends_statsvis)
		xmlNewChild(node, NULL, "friends_statsvis", NULL);
	if (options->friends_filtervis)
		xmlNewChild(node, NULL, "friends_filtervis", NULL);
}

int
conf_xml_write(Configuration *c, char *path) {
	xmlDocPtr doc;
	xmlNodePtr root, node;
	GList *l;
	int i;

	doc = xmlNewDoc("1.0");
	root = xmlNewDocNode(doc, NULL, "configuration", NULL);
	xmlDocSetRootElement(doc, root);
	xmlSetProp(root, "version", "cvs");

	node = xmlNewChild(root, NULL, "servers", NULL);
	for (l = c->servers; l != NULL; l = l->next) {
		writeserver(l->data, node);
	}

	if (c->servercur)
		xmlAddTN(root, "currentserver", c->servercur->name);

	node = xmlNewChild(root, NULL, "geometries", NULL);
	for (i = 0; i < GEOM_COUNT; i++) {
		if (c->geometries[i].width > 0) {
			writegeometry(&c->geometries[i], geometry_names[i], node);
		}
	}

	node = xmlNewChild(root, NULL, "options", NULL);
	writeoptions(&c->options, node);

	if (c->uifont)
		xmlAddTN(root, "uifont", c->uifont);

	if (c->proxy)
		xmlAddTN(root, "proxy", c->proxy);
	if (c->proxyuser) {
		node = xmlNewChild(root, NULL, "proxyauth", NULL);
		xmlAddTN(node, "username", c->proxyuser);
		xmlAddTN(node, "password", c->proxypass);
	}

	if (c->spawn_command)
		xmlAddTN(root, "spawncommand", c->spawn_command);

	if (c->cfmask)
		xmlAddTN(root, "cfmask", c->cfmask);

	if (c->cfuserinterval) {
		char buf[20];
		g_snprintf(buf, 20, "%d", c->cfuserinterval);
		xmlAddTN(root, "cfuserinterval", buf);
	}

	if (c->cfthreshold) {
		char buf[20];
		g_snprintf(buf, 20, "%d", c->cfthreshold);
		xmlAddTN(root, "cfthreshold", buf);
	}

	if (app.quiet_dlgs) {
		xmlNodePtr xmlList = xmlNewChild(root, NULL, "quiet_dlgs", NULL);
		for (l = app.quiet_dlgs; l != NULL; l = l->next) {
			xmlAddTN(xmlList, "quiet_dlg", (char*)l->data);
		}
	}

	if (c->defaultsecurity.type != SECURITY_PUBLIC) {
		char *type = NULL;
		switch (c->defaultsecurity.type) {
			case SECURITY_PRIVATE: type = "private"; break;
			case SECURITY_FRIENDS: type = "friends"; break;
			default: ; /* do nothing. */
		}
		if (type) {
			node = xmlNewChild(root, NULL, "defaultsecurity", NULL);
			xmlSetProp(node, "type", type);
		}
	}

	if (xmlSaveFormatFile(path, doc, TRUE) < 0) 
		fprintf(stderr, "Error saving to %s!?\n", path);
	xmlFreeDoc(doc);
	return 0;
}

