/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: checkfriends.c,v 1.17 2002/11/10 08:30:12 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h> /* atoi */
#ifndef G_OS_WIN32
#include <unistd.h> /* unlink */
#endif
#include <string.h> /* strerror */
#include <errno.h>

#include "checkfriends.h"
#include "conf.h"
#include "network.h"
#include "util.h"
#include "lj.h"
#include "spawn.h"
#include "menu.h"
#include "util.h"

static gboolean checkfriends(CFMgr *cfm);
static void checkfriends_schedule(CFMgr *cfm, gint interval);
static void cf_pref_cb(GtkMenuItem *mi, LJWin *ljw);
static void cf_float_toggle_cb(GtkMenuItem *mi, CFIndicator *cfi);
static void cf_toggle_cb(GtkMenuItem *mi, CFIndicator *cfi);
static gboolean cfindicator_clicked_cb(GtkWidget* w, GdkEventButton *ev, CFIndicator *cfi);
static time_t cf_cli_conf_read(CFMgr *cfm, gchar *ljid);
static void cf_cli_conf_write(CFMgr *cfm, gchar *ljid);

static const gchar *cf_help = N_(
	"The Check Friends feature is enabled by right-clicking on this "
	"button and selecting \"Check friends view for new entries\". Once "
	"you do so, you will be notified every time one of your friends "
	"updates their journal with a new entry.\n\n"

	"Once this happens, the indicator will turn red. You can then "
	"read your friends page (as a convenience, double-clicking on "
	"the indicator will take you there). Click on the indicator once "
	"you've read the new entries to resume monitoring for newer "
	"entries.\n\n");
	
CFMgr*
cfmgr_new(void) {
	CFMgr *cfm = g_new0(CFMgr, 1);

	cfm->indicators      = NULL;
	cfm->state           = CF_DISABLED;
	cfm->server_interval = 45; /* just a default quicky overridden */
	cfm->user_interval   = conf.cfuserinterval ? conf.cfuserinterval :
							   cfm->server_interval;
	cfm->mask            = g_strdup(conf.cfmask);
	cfm->lastupdate      = g_strdup("");
	cfm->newcount        = 0; /* approximate number of new friend entries */
	cfm->errors          = 0;
	cfm->timeout         = 0; /* will be installed upon first "CF_ON" */
	cfm->parent          = NULL; /* set by cfmgr_parent_set */

	/* TODO: when we go multi-LJWin, this should probably be replaced by a
	 * per-LJWin registration. */
	app.cfmgr_list = g_slist_append(app.cfmgr_list, cfm);

	return cfm;
}

void
cfmgr_parent_set(CFMgr *cfm, GtkWidget *parent) {
	/* We need a handle on ljw so that we can call the settings dialog,
	 * which in turn needs it so it can do completely unrelated stuff like
	 * control the display font. Unfortunately, this means circular reference.
	 *
	 * This is proof we need to rethink our object model :-( */
	cfm->parent          = parent;
}

void
cfmgr_mask_set(CFMgr *cfm, gchar *maskstr) {
	/* validate input */
	guint32 ui32_mask = atol(maskstr); /* what have you got, twenty oh two, */
	gchar   str_mask[20];              /* that makes so damn superior? */
	g_snprintf(str_mask, 20, "%u", ui32_mask);
	if (strcmp(maskstr, str_mask)) {
		g_warning("cfmgr_mask_set passed a non-guint32 arg");
		return;
	}
				
	if (strcmp(cfm->mask, maskstr)) {
		string_replace(&cfm->mask, g_strdup(maskstr)); /* install new mask */
		cfm->lastupdate = 0;             /* invalidate monitor information */
	}
}

CFIndicator*
cfindicator_new(CFMgr *cfm, CFIndicatorStyle style, GtkWidget *parent) {
	CFIndicator *ind  = g_new0(CFIndicator, 1);

	GtkWidget *box    = gtk_hbox_new(FALSE, 0);
	GtkWidget *button = gtk_button_new();
	GtkWidget *image  = gtk_image_new();

	if (!cfm)
		g_error("cfindicator_new() called on uninitialized CFMgr\n");

	if (!parent)
		g_warning("cfindicator_new() called with NULL parent\n");

	gtk_container_add(GTK_CONTAINER(box), button);
	gtk_container_add(GTK_CONTAINER(button), image);
	g_signal_connect(G_OBJECT(button), "button_press_event",
		G_CALLBACK(cfindicator_clicked_cb), ind);

	ind->box      = box;
	ind->button   = button;
	ind->image    = image;
	ind->parent   = parent;
	ind->cfmgr    = cfm;
	ind->style    = style;
	ind->floatwin = NULL; /* for CF_FLOATING style only */

	cfm->indicators = g_slist_append(cfm->indicators, ind);
	cfmgr_refresh_images(cfm);

	/* it's still the responsibility of the caller to pack the box! */
	gtk_widget_show_all(box);

	return ind;
}

void
cf_threshold_normalize(gint *threshold) {
	if (*threshold < 1) {
		*threshold = 1;
	}
	if (*threshold > CF_MAX_THRESHOLD) {
		*threshold = CF_MAX_THRESHOLD;
	}		
}

GtkWidget*
cfindicator_box_get(CFIndicator *ind) {
	return ind->box;
}

void
cfmgr_refresh_images(CFMgr* cfm) {
	gchar *image_name = NULL;
	GSList *ind;
	const gchar *tip_text = NULL;
	
	if (!cfm)
		g_error("cfindicator_refresh_images() called on uninitialized CFMgr");

	/* in cli mode, for example, there are no images to update */
	if (!cfm->indicators)
		return;

	switch (cfm->state) {
		case CF_DISABLED:
			image_name  = "logjam-cfriends-off";
			tip_text    = _("Check Friends disabled");
			break;
		case CF_ON:
			image_name = "logjam-cfriends-on";
			tip_text    = _("No new entries in your friends view");
			break;
		case CF_NEW:
			image_name = "logjam-cfriends-new";
			tip_text    = _("There are new entries in your friends view");
			break;
	}

	for (ind = cfm->indicators; ind; ind = g_slist_next(ind)) {
		CFIndicator *cfi = (CFIndicator*)ind->data;
		GtkWidget *image = cfi->image;
		
		/* (re)draw indicator */
		gtk_image_set_from_stock(GTK_IMAGE(image), image_name,
			GTK_ICON_SIZE_MENU);
		gtk_tooltips_set_tip(app.tooltips, ((CFIndicator*)ind->data)->button,
				tip_text, _(cf_help));

		/* autoraise floating indicator */
		if ((cfi->style == CF_FLOATING) && (cfm->state == CF_NEW) &&
				(conf.options.cfautofloatraise)) {
			gtk_window_present(GTK_WINDOW(cfi->floatwin));
		}
	}
}

void
cfmgr_set_state(CFMgr *cfm, CFState state) {

	cfm->state = state;

	switch (state) {
		case CF_DISABLED:
			if (cfm->timeout)
				gtk_timeout_remove(cfm->timeout);
			break;

		case CF_ON:
			/* every startup gets a clean slate */
			cfm->errors   = 0;
			cfm->newcount = 0;

			/* Start checking friends right away when we're turned on.
			 *
			 * We schedule an almost-immediate checkfriends instead
			 * of calling it directly to improve UI responsiveness */
			checkfriends_schedule(cfm, 0);
			break;

		case CF_NEW:
			break;
	}
	cfmgr_refresh_images(cfm);
}

/* install a timeout for the next checkfriends call according to the
 * most recent interval information */
static void
checkfriends_schedule(CFMgr *cfm, gint interval) {
	/* no indicators are available eg. when in cli mode. */
	if (cfm->indicators == NULL)
		return;
	
	if (interval == CF_AUTOSCHED)
		interval = MAX(cfm->server_interval, cfm->user_interval) * 1000;

	cfm->timeout = g_timeout_add(
			interval,
			(GSourceFunc)checkfriends,
			(gpointer)cfm);
}

/* periodically check for new entries in the user's friends list.
 * uses the LJ "checkfriends" protocol mode.
 *
 * NOTE: This function should always return FALSE. For the
 * (quite normal) case where checking friends should keep going,
 * we call checkfriends_schedule() just before returning, and that
 * schedules the next timeout correctly according to the most recent
 * interval information, which may have changed either from the server
 * or from the user (in the preferences).
 *
 * Micro-rant: I wish there were a GTK+ API for one-time timeouts.
 * Accidentally returning TRUE from here would mean a bug, since it
 * will then likeley imply two future checkfriends. Of course we could
 * write a wrapper function, but this is really framework stuff.
 */
gboolean
checkfriends(CFMgr *cfm) {
	NetRequest *request;
	NetResult  *result;
	gboolean new;

	request = net_request_new("checkfriends");

	net_request_copys(request, "lastupdate", cfm->lastupdate);
	if (cfm->mask)
		net_request_copys(request, "mask", cfm->mask);

	result = net_request_run_silent(request);
	net_request_free(request);

	/* if the request fails, we only stop polling after several attempts */
	if (!net_result_succeeded(result)) {
		net_request_free(result);
		if (++cfm->errors > CF_MAXERRORS) {
			lj_message(NULL, LJ_MSG_WARNING, TRUE, NULL,
					_("Too many network errors; checking friends disabled."));
			cfmgr_set_state(cfm, CF_DISABLED);
			return FALSE;
		}		
		checkfriends_schedule(cfm, CF_AUTOSCHED);
		return FALSE;
	}

	string_replace(&cfm->lastupdate, 
			g_strdup(net_result_get(result, "lastupdate")));
	cfm->server_interval = net_result_geti(result, "interval");
	new = (net_result_geti(result, "new") != 0);

	net_result_free(result);

	if (new && (++cfm->newcount >= conf.cfthreshold)) {
		cfmgr_set_state(cfm, CF_NEW);
		return FALSE; /* stop polling */
	} else {
		/* keep polling with new scheduling information */
		checkfriends_schedule(cfm, CF_AUTOSCHED);
		return FALSE;
	}
}

/* gui-less version of checkfriends.
 *
 * returns TRUE  when new friends entries have been detected
 *         FALSE when no such entries exist (or when something has prevented
 *                                       the check)
 *
 * keeps track of its own persistent information with cf_cli_conf_*(). */
gboolean
checkfriends_cli(gchar *ljid) {
	CFMgr *cfm = cfmgr_new();
	time_t now = time(NULL);
	time_t then = cf_cli_conf_read(cfm, ljid);

	/* don't even approach the server in some cases.
	 * report the reason to the user unless we're in quiet mode */

	if (cfm->errors > CF_MAXERRORS) {
		lj_cli_note(_("Maximum error count reached in contacting server.\n"
			"Run \"%s --checkfriends=purge\" and try again.\n"),
			app.programname);
		return FALSE;
	}
	if (now - then < cfm->server_interval) {
		lj_cli_note(_("Request rate exceeded. Slow down.\n"));
		return FALSE;
	}
	if (cfm->state == CF_NEW) {
		lj_cli_note(_("Read your friends page, then run\n"
				"\"%s --checkfriends=purge\"\n"), app.programname);
		return TRUE;
	}

	checkfriends(cfm);

	cf_cli_conf_write(cfm, ljid);

	return (cfm->state == CF_NEW);
}

void
checkfriends_cli_purge(gchar *ljid) {
	gchar path[1024];
	gchar id[1024];
	g_snprintf(id, 1024, "checkfriends-%s", ljid);
	conf_make_path(id, path, 1024);

	if (g_file_test(path, G_FILE_TEST_EXISTS)) {
		if (!unlink(path)) return;

		fprintf(stderr, _("can't unlink %s: %s"), id, strerror(errno));
	}
}

static time_t
cf_cli_conf_read(CFMgr *cfm, gchar *ljid) {
	time_t lasttry = 0;
	gchar *cfconfdata;
	gchar **parseddata;
	gchar path[1024];
	gchar id[1024];
	g_snprintf(id, 1024, "checkfriends-%s", ljid);
	conf_make_path(id, path, 1024);

	if (g_file_get_contents(path, &cfconfdata, NULL, NULL)) {
		parseddata = g_strsplit(cfconfdata, "|", 5);
		cfm->lastupdate      = parseddata[0];
		cfm->server_interval = atoi(parseddata[1]); g_free(parseddata[1]);
		cfm->errors          = atoi(parseddata[2]); g_free(parseddata[2]);
		cfm->state           = atoi(parseddata[3]); g_free(parseddata[3]);
		lasttry              = atoi(parseddata[4]); g_free(parseddata[4]);
	} else {
		cfm->state = CF_ON;
	}
	return lasttry;
}
				
static void
cf_cli_conf_write(CFMgr *cfm, gchar *ljid) {
	FILE *f;
	gchar confdata[1024];
	gchar path[1024];
	gchar id[1024];
	g_snprintf(id, 1024, "checkfriends-%s", ljid);
	conf_make_path(id, path, 1024);

	g_snprintf(confdata, 1024, "%s|%d|%d|%d|%ld",
			cfm->lastupdate,
			cfm->server_interval,
			cfm->errors,
			cfm->state,
			time(NULL));

	f = fopen(path, "w");
	if (f) {
		fprintf(f, confdata);
		fclose(f);
	} else {
		fprintf(stderr, _("error opening %s for write: %s"),
				path, strerror(errno));
		exit(2);
	}
}

static void
open_friends_list(GtkWidget *parent) {
	gchar url[2000];

	g_snprintf(url, 2000, "%s/users/%s/friends/", conf_cur_server()->url,
			conf_cur_user()->username);
	spawn_url(parent, url);
}

static gboolean
cfindicator_clicked_cb(GtkWidget* w, GdkEventButton *ev, CFIndicator *cfi) {
	CFMgr *cfm = cfi->cfmgr;

	/* right-clicks start context menu (note: this case is terminal) */
	if (ev->button == 3) {
		cf_context_menu(cfi, ev);
		return TRUE;
	}

	/* *all* left-clicks move CF_NEW to CF_ON */
	if (cfm->state == CF_NEW) {
		cfmgr_set_state(cfm, CF_ON);
	}
	/* and double-clicks open up the browser, too */
	if (ev->type==GDK_2BUTTON_PRESS) {
		open_friends_list(cfm->parent);
	
		return TRUE;
	}
	
	/* this help box will only be called once in a double-click,
	 * thankfully, because the above is terminal on double-clicks. */
	if (cfm->state == CF_DISABLED) {
		lj_message(cfi->parent, LJ_MSG_INFO, TRUE, _("Check Friends"), _(cf_help));
	}

	return TRUE;
}

static void
cf_toggle_cb(GtkMenuItem *mi, CFIndicator* cfi) {
	CFMgr *cfm = cfi->cfmgr;

	if ((cfm->state == CF_DISABLED) && conf.loginok) {
		cfmgr_set_state(cfm, CF_ON);
	} else if ((cfm->state == CF_ON) || (cfm->state == CF_NEW)) {
		cfmgr_set_state(cfm, CF_DISABLED);
	} else {
		g_error("inconsistent cfmgr state in cf_activate_cb()");
	}
}

static void
cf_float_decorate_toggle_cb(GtkMenuItem *mi, CFIndicator* cfi) {
	conf.options.cffloat_decorate = !conf.options.cffloat_decorate;

	cf_float_decorate_refresh();
}

void cf_float_decorate_refresh(void) {
	/* suck a bunch of events in. */
	while (gtk_events_pending())
		gtk_main_iteration();

	if (app.cfi_float) {
		gtk_window_set_decorated(GTK_WINDOW(app.cfi_float->floatwin),
				conf.options.cffloat_decorate);
		gtk_widget_hide(app.cfi_float->floatwin);
		gtk_widget_show(app.cfi_float->floatwin);
	}
}

static void
cf_floatraise_toggle_cb(GtkMenuItem *mi, CFIndicator* cfi) {
	conf.options.cfautofloatraise = !conf.options.cfautofloatraise;
}

static void
cf_float_toggle_cb(GtkMenuItem *mi, CFIndicator* cfi) {
	CFMgr *cfm = cfi->cfmgr;
	GtkWidget *parent = cfi->parent;

	if (app.cfi_float) {
		cf_float_destroy(app.cfi_float);
	} else {
		app.cfi_float = make_cf_float(cfm, parent);
	}
}

void
cf_float_destroy(CFIndicator *cfi) {
	CFMgr *cfm = cfi->cfmgr;

	if (cfi->style != CF_FLOATING)
		g_error("cf_float_destroy called on non-floating indicator");

	gtk_widget_destroy(cfi->floatwin);
	cfm->indicators = g_slist_remove(cfm->indicators, cfi);
	app.cfi_float  = NULL;
}

CFIndicator*
make_cf_float(CFMgr* cfm, GtkWidget* parent) {
	CFIndicator *ind = cfindicator_new(cfm, CF_FLOATING, parent);
	GtkWidget *flwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	ind->floatwin = flwin;
	geometry_tie(flwin, GEOM_CFFLOAT);

	g_signal_connect(G_OBJECT(ind->box), "button_press_event", 
		G_CALLBACK(cfindicator_clicked_cb), ind);

	gtk_window_set_decorated(GTK_WINDOW(flwin),
			conf.options.cffloat_decorate);
	gtk_container_add(GTK_CONTAINER(flwin), GTK_WIDGET(ind->box));
	gtk_widget_show_all(flwin);

	return ind;
}

static void
cf_pref_cb(GtkMenuItem *mi, LJWin* ljw) {
	settings_run(ljw, SETTINGS_PAGE_CF);
}

void
cf_context_menu(CFIndicator* cfi, GdkEventButton *ev) {
	CFMgr *cfm = cfi->cfmgr;

	GtkAccelGroup *accelgroup = gtk_accel_group_new();
	GtkWidget *cfmenu = gtk_menu_new();
	GtkWidget *item;

	{ /* Check friends on/off */
		item = gtk_check_menu_item_new_with_mnemonic(
			_("_Check friends view for new entries"));
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_widget_set_sensitive(item, conf.loginok);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item),
			(cfm->state != CF_DISABLED));
		g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(cf_toggle_cb), cfi);
		gtk_widget_show(item);
	}

	{ /* ---------- */
		item = gtk_separator_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_widget_show(item);
	}

	{ /* Preferences... */
		item = gtk_image_menu_item_new_from_stock(GTK_STOCK_PREFERENCES,
			accelgroup);
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(cf_pref_cb), cfm->parent);
		gtk_widget_show(item);
	}

	{ /* ---------- */
		item = gtk_separator_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_widget_show(item);
	}

	{ /* floating indicator on/off */
		item = gtk_check_menu_item_new_with_mnemonic(
			_("_Show floating indicator"));
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item),
			(app.cfi_float != NULL));
		g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(cf_float_toggle_cb), cfi);
		gtk_widget_show(item);
	}

	{ /* autoraise floating indicator */
		item = gtk_check_menu_item_new_with_mnemonic(
			_("_Raise floating indicator when friends post"));
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item),
			conf.options.cfautofloatraise);
		g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(cf_floatraise_toggle_cb), cfi);
		gtk_widget_show(item);
	}

	{ /* floating indicator decoration */
		item = gtk_check_menu_item_new_with_mnemonic(
			_("Show _titlebar on floating indicator"));
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item),
			conf.options.cffloat_decorate);
		g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(cf_float_decorate_toggle_cb), cfi);
		gtk_widget_show(item);
	}

	{ /* ---------- */
		item = gtk_separator_menu_item_new();
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		gtk_widget_show(item);
	}

	{ /* open friends list in browser */
		item = gtk_image_menu_item_new_with_mnemonic(
			_("_Open friends list in browser"));
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item),
			gtk_image_new_from_stock(GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_MENU));
		gtk_menu_shell_append(GTK_MENU_SHELL(cfmenu), item);
		g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(open_friends_list), NULL);
		gtk_widget_show_all(item);
	}

	gtk_menu_popup(GTK_MENU(cfmenu), NULL, NULL, NULL, NULL,
		ev->button, ev->time);

}
