/*
libutil -- handling xover records

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998, 1999.
Modified by Ralf Wildenhues <ralf.wildenhues@gmx.de>
Copyright of the modifications 2002.
Modified by Matthias Andree <matthias.andree@gmx.de>
Copyright of the modifications 2000 - 2003.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"
#include <fcntl.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "system.h"
#include "ln_log.h"

extern struct state _res;

static void
tabstospaces(char *t)
{
    while (*t) {
	if (*t == '\t')
	    *t = ' ';
	t++;
    }

}

static /*@null@*/ /*@only@*/
char *getxoverline(const char *filename)
{
    char *l;
    char *result;
    FILE *f;

    result = NULL;
    debug = 0;
    if ((f = fopen(filename, "r"))) {
	char *from, *subject, *date, *msgid, *references, *lines, *xref;
	unsigned long bytes, linecount;
	char **h;
	int body;

	from = subject = date = msgid = references = xref = lines = NULL;
	bytes = linecount = 0;
	h = NULL;
	body = 0;

	while (!feof(f) && ((l = getaline(f)) != NULL)) {
	    tabstospaces(l);
	    linecount++;
	    bytes += strlen(l) + 2; /* normalize CR LF -> add 2 per line */
	    if (body || !l) {
		/* do nothing */
	    } else if (!body && !*l) {
		linecount = 0;
		body = 1;
	    } else if (*l && isspace((unsigned char)*l)) {
		/* cater for folded headers */
		if (h) {
		    (*h) = critrealloc(*h, strlen(*h) + strlen(l) + 1,
				       "extending header");
		    strcat(*h, l);	/* RATS: ignore */
		}
	    } else if (!from && !strncasecmp("From: ", l, 6)) {
		l += 6;
		if (*l) {
		    from = critstrdup(l, "getxoverline");
		    h = &from;
		}
	    } else if (!subject && !strncasecmp("Subject: ", l, 9)) {
		l += 9;
		if (*l) {
		    subject = critstrdup(l, "getxoverline");
		    h = &subject;
		}
	    } else if (!date && !strncasecmp("Date: ", l, 6)) {
		l += 6;
		if (*l) {
		    date = critstrdup(l, "getxoverline");
		    h = &date;
		}
	    } else if (!msgid && !strncasecmp("Message-ID: ", l, 12)) {
		l += 12;
		if (*l) {
		    msgid = critstrdup(l, "getxoverline");
		    h = &msgid;
		}
	    } else if (!references && !strncasecmp("References: ", l, 12)) {
		l += 12;
		if (*l) {
		    references = critstrdup(l, "getxoverline");
		    h = &references;
		}
	    } else if (!lines && !strncasecmp("Lines: ", l, 7)) {
		l += 7;
		if (*l) {
		    lines = critstrdup(l, "getxoverline");
		    h = &lines;
		}
	    } else if (!xref && !strncasecmp("Xref: ", l, 6)) {
		l += 6;
		if (*l) {
		    xref = critstrdup(l, "getxoverline");
		    h = &xref;
		}
	    } else {
		h = NULL;
	    }
	}
	if (from != NULL && date != NULL && subject != NULL &&
	    msgid != NULL && bytes) {
	    result = critmalloc(strlen(filename) + strlen(subject) + strlen(from) +
				strlen(date) + strlen(msgid) +
				(references ? strlen(references) : 0) +
				100 + (xref ? strlen(xref) : 0),
				"computing overview line");
	    sprintf(result, "%s\t%s\t%s\t%s\t%s\t%s\t%lu\t%lu",	/* RATS: ignore */
		    filename, subject, from, date, msgid,
		    references ? references : "",
		    bytes, lines ? strtoul(lines, NULL, 10) : linecount);
	    if (xref) {
		strcat(result, "\tXref: ");	/* RATS: ignore */
		strcat(result, xref);	/* RATS: ignore */
	    }
	} else {
	    if (from == NULL) syslog(LOG_DEBUG, "getxoverline: %s lacks From:", filename);
	    else if (date == NULL) syslog(LOG_DEBUG, "getxoverline: %s lacks Date:", filename);
	    else if (subject == NULL) syslog(LOG_DEBUG, "getxoverline: %s lacks Subject:", filename);
	    else if (msgid == NULL) syslog(LOG_DEBUG, "getxoverline: %s lacks Message-ID:", filename);
	    else if (bytes == 0) syslog(LOG_DEBUG, "getxoverline: %s 0 bytes", filename);
	}
	(void)fclose(f);
	if (from)
	    free(from);
	if (date)
	    free(date);
	if (subject)
	    free(subject);
	if (msgid)
	    free(msgid);
	if (references)
	    free(references);
	if (lines)
	    free(lines);
	if (xref)
	    free(xref);
    } else {
	syslog(LOG_DEBUG, "getxoverline: cannot open %s: %m", filename);
    }
    debug = debugmode;
    return result;
}

/* returns malloc'd string containing current directory name */
static char *mgetcwd(void) {
    size_t size = PATH_MAX;
    char *buf;

    for(;;) {
	buf = critmalloc(size, "killcwd");
	if (getcwd(buf, size) == NULL) {
	    if (errno == ERANGE) {
		size_t osize = size;
		free(buf);
		size <<= 1;
		if (size < osize) {
		    syslog(LOG_ERR, "cannot allocate buffer large enough for getcwd, aborting");
		    return NULL;
		}
	    } else {
		syslog(LOG_ERR, "getcwd failed: %m");
		free(buf);
		return NULL;
	    }
	} else {
	    break;
	}
    }

    return buf;
}

static void killcwd(void) {
    char *t = mgetcwd();
    if (t) {
	if (chdir(spooldir)) {
	    syslog(LOG_ERR, "cannot chdir(%s): %m", spooldir);
	}
	if (rmdir(t) && errno != ENOTEMPTY) {
	    syslog(LOG_ERR, "cannot rmdir(%s): %m", t);
	}
	free(t);
    }
}

/* utility routine to pull the xover info into memory
   returns 0 if there's some error, non-zero else */
int
getxover(void)
{
    DIR *d;
    struct dirent *de;
    int fd;
    struct stat st;
    unsigned long art;
    static char *overview;
    int error;
    char *p, *q;
    char s[PATH_MAX+1];

    /* free any memory left over from last time */
    error = 0;

    if (xoverinfo) {
	for (art = xfirst; art <= xlast; art++) {
	    if (xoverinfo[art - xfirst].text) {
		free(xoverinfo[art - xfirst].text);
		xoverinfo[art - xfirst].text = NULL;
	    }
	}
	free(xoverinfo);
	xoverinfo = NULL;
    }

    if (overview) {
	free(overview);
	overview = NULL;
    }

    /* find article range */
    d = opendir(".");
    if (!d) {
	syslog(LOG_ERR, "opendir: %m");
	return 0;
    }

    xfirst = ULONG_MAX;
    xlast = 0;
    while ((de = readdir(d))) {
	if (!isdigit((unsigned char)de->d_name[0]))
	    continue; /* skip files that don't start with a digit */
	/* WARNING: strtoul will happily return the negated value when
	 * fed a string that starts with a minus character! */
	art = strtoul(de->d_name, &p, 10);
	if (art && p && !*p) {
	    if (art < xfirst)
		xfirst = art;
	    if (art > xlast)
		xlast = art;
	}
    }

    if (xlast < xfirst) {
	/* we did not find any article files (1, 17, 815 or the like) */
	closedir(d);
	(void)unlink(".overview");
	if (debugmode) {
	    char *t = mgetcwd();
	    syslog(LOG_DEBUG, "removed .overview file for %s", 
		   t ? t : "(unknown)");
	    if (t) free(t);
	}
	killcwd();
	return 0;
    }

    /* next, read .overview, correct it if it seems too different from
       what the directory implies, and write the result back */
    rewinddir(d);

    xoverinfo = (struct xoverinfo *)
	critmalloc(sizeof(struct xoverinfo) * (xlast + 1 - xfirst),
		    "allocating overview array");
    memset(xoverinfo, 0, sizeof(struct xoverinfo) * (xlast + 1 - xfirst));

    if ((fd = open(".overview", O_RDONLY)) >= 0 &&
	fstat(fd, &st) == 0 &&
	(overview = (char *)malloc(st.st_size + 1)) != NULL &&
	(off_t) read(fd, overview, st.st_size) == st.st_size) {

	close(fd);
	overview[st.st_size] = '\0';

	/* okay, we have the content, so let's parse it roughly */
	/* iterate line-wise */
	p = overview;
	while (p && *p) {
	    char *t;

	    while (p && isspace((unsigned char)*p))
		p++;
	    q = strchr(p, '\n');
	    if (q)
		*q++ = '\0';

	    art = strtoul(p, &t, 10);
	    if (t && *t == '\t') {
		if (art > xlast || art < xfirst) {
		    error++;
		} else if (xoverinfo[art - xfirst].text) {
		    error++;
		    syslog(LOG_ERR, "multiple lines for article %lu "
			   "in .overview for %s", art,
			   getcwd(s, sizeof(s)) ? s : "(unknown)");
		    free (xoverinfo[art - xfirst].text);
		    xoverinfo[art - xfirst].text = NULL;
		    xoverinfo[art - xfirst].exists = -1;
		} else if (xoverinfo[art - xfirst].exists == 0) {
		    xoverinfo[art - xfirst].text = critstrdup(p, "getxover");
		}
	    }

	    p = q;
	}
    }

    /* so, what was missing? */
    while ((de = readdir(d))) {
	if (de->d_name[0] == '.')
	    continue;
	art = strtoul(de->d_name, &p, 10);
	if (p && !*p && art >= xfirst && art <= xlast) {
	    if (!xoverinfo[art - xfirst].text) {
		xoverinfo[art - xfirst].exists = 0;
		if (debugmode)
		    syslog(LOG_DEBUG, "reading XOVER info from %s/%s",
			   getcwd(s, sizeof(s)), de->d_name);
		error++;
		if ((xoverinfo[art - xfirst].text =
		     getxoverline(de->d_name)) == NULL
		    && !delaybody) { /* FIXME: why ignore this for
				      * delaybody? */
		    (void)getcwd(s, sizeof(s));
		    syslog(LOG_NOTICE, "illegal article: %s/%s",
			   s, de->d_name);
		    if ((lstat(de->d_name, &st) == 0) && S_ISREG(st.st_mode)) {
			if (unlink(de->d_name))
			    syslog(LOG_WARNING, "failed to remove %s/%s: %m",
				   s, de->d_name);
		    } else {
			syslog(LOG_WARNING, "%s/%s is not a regular file",
			       s, de->d_name);
		    }
		}
	    }
	}

	if (art >= xfirst && art <= xlast && xoverinfo[art - xfirst].text) {
	    xoverinfo[art - xfirst].exists = 1;
	} else {
	    (void)getcwd(s, sizeof(s));
	    /* kill non-article files, like "core" */
	    if (art == 0
		&& strcmp(de->d_name, ".overview")
		&& strcmp(de->d_name, ".")
		&& strcmp(de->d_name, ".."))
	    {
		if (unlink(de->d_name)
		    && errno != EISDIR
		    && errno != EPERM
		    && verbose) {
		    ln_log(LNLOG_SERR, LNLOG_CTOP,
			    "attempted to delete %s/%s in case it's junk: %s\n",
			   s, de->d_name, strerror(errno));
		}
	    } else if (p && !*p) {
		if (art >= xfirst && art <= xlast) {
		    ln_log(LNLOG_SERR, LNLOG_CTOP,
			   "article %s/%lu contained illegal headers\n",
			   s, art);
		}
	    }
	}
    }

    /* count removed articles */
    for (art = xfirst; art <= xlast; art++) {
	if (xoverinfo[art - xfirst].text
	    && !xoverinfo[art - xfirst].exists) {
	    ++error;
	    free(xoverinfo[art - xfirst].text);
	    xoverinfo[art - xfirst].text = NULL;
	}
    }

    /* if something had to be fixed, write a better file to disk for
       next time - race conditions here, but none dangerous */
    if (error) {
	int wfd;
	char newfile[20]; /* RATS: ignore */

	if (debugmode)
	    syslog(LOG_DEBUG, "updated %d lines in %s/.overview\n",
		   error, getcwd(s, sizeof(s)));

	strcpy(newfile, ".overview.XXXXXX");
	if ((wfd = mkstemp(newfile)) != -1) {
	    int va;

	    va = 1;
	    for (art = xfirst; art <= xlast; art++) {
		if (xoverinfo[art - xfirst].exists &&
		    xoverinfo[art - xfirst].text) {
		    if (writes(wfd, xoverinfo[art - xfirst].text) == - 1
			|| writes(wfd, "\n") == -1) 
		    {
			syslog(LOG_ERR,
			       "write() for .overview failed: %m");
			va = 0;
			break;
		    }
		}
	    }
	    if (fchmod(wfd, 0664)) va = 0;
	    if (fsync(wfd)) va = 0;
	    if (close(wfd)) va = 0;
	    if (va) {
		if (rename(newfile, ".overview")) {
		    if (unlink(newfile))
			syslog(LOG_ERR,
			       "rename() and unlink() both failed: %m");
		    else
			syslog(LOG_ERR,
			       "rename(%s/%s, .overview) failed: %m",
			       getcwd(s, sizeof(s)), newfile);
		} else {
		    if (debugmode)
			syslog(LOG_DEBUG, "wrote %s/.overview\n",
			       getcwd(s, sizeof(s)));
		}
	    } else {
		unlink(newfile);
		/* the group must be newly empty: I want to keep the old
		   .overview file I think */
	    }
	} else {
	    syslog(LOG_ERR, "mkstemp of new .overview failed: %m");
	}
    }

    closedir(d);
    return 1;
}

void
fixxover(void)
{
    DIR *d;
    struct dirent *de;
    char s[SIZE_s + 1];

    xsnprintf(s, SIZE_s, "%s/interesting.groups", spooldir);
    d = opendir(s);
    if (!d) {
	syslog(LOG_ERR, "opendir %s: %m", s);
	return;
    }

    while ((de = readdir(d))) {
	if (isalnum((unsigned char)*(de->d_name)) && findgroup(de->d_name)) {
	    if (chdirgroup(de->d_name, FALSE))
		getxover();
	}
    }
    closedir(d);
}
