%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\chapter{SlurpDirectory}

\section{Introduction}

SlurpDirectory is intended to give a nice lazy way of traversing directory
trees.

\begin{code}
module SlurpDirectory ( Slurpy( ), FileContents, empty_slurpy,
                        slurp, co_slurp, slurp_name, is_file, is_dir,
                        get_filecontents, get_dircontents, get_mtime,
                        get_length, get_slurp,
                        slurp_write_dirty, slurp_write, launder_slurpy,
                        slurp_runfunc,
                        slurp_addfile, slurp_removefile,
                        slurp_adddir, slurp_removedir, slurp_move,
                        slurp_modfile, slurp_hasfile, slurp_hasdir,
                        slurp_has_anycase, wait_a_moment, undefined_time,
                        slurp_has,
                        readFileLinesPS
                      ) where

import IO
import System
import Directory
import IOExts--PrelIOBase
import List ( sort )
import Monad ( unless, liftM )
import Char ( toLower )
import Posix ( EpochTime, setFileTimes, getFileStatus,
               accessTime, modificationTime, epochTime,
               sleep,
               FileOffset, fileSize,
             )
import Maybe ( catMaybes )
import FastPackedString

data Slurpy = SlurpDir PackedString (IO ()) [Slurpy]
            | SlurpFile PackedString Bool (EpochTime,FileOffset) FileContents
type FileContents = ([PackedString],Maybe PackedString)              

slurp :: FilePath -> IO Slurpy
empty_slurpy :: Slurpy
empty_slurpy = SlurpDir (packString ".") (return ()) []
slurp_name :: Slurpy -> FilePath
is_file :: Slurpy -> Bool
is_dir :: Slurpy -> Bool

get_filecontents :: Slurpy -> FileContents
get_dircontents :: Slurpy -> [Slurpy]
get_mtime :: Slurpy -> EpochTime
get_length :: Slurpy -> FileOffset

instance Eq Slurpy where
    s1 == s2 = (slurp_name s1) == (slurp_name s2)
instance Ord Slurpy where
    s1 <= s2 = (slurp_name s1) <= (slurp_name s2)
\end{code}

Here are a few access functions.

\begin{code}
slurp_name (SlurpFile f _ _ _) = unpackPS f
slurp_name (SlurpDir d _ _) = unpackPS d
slurp_setname f (SlurpDir _ b c) = SlurpDir f b c
slurp_setname f (SlurpFile _ b m c) = SlurpFile f b m c

is_file (SlurpDir _ _ _) = False
is_file (SlurpFile _ _ _ _) = True

is_dir (SlurpDir _ _ _) = True
is_dir (SlurpFile _ _ _ _) = False

get_filecontents (SlurpFile _ _ _ c) = c

get_dircontents (SlurpDir _ _ c) = sort c

get_mtime (SlurpFile _ _ m _) = fst m
get_length (SlurpFile _ _ m _) = snd m

getModTime f = do stat <- getFileStatus f
                  return (modificationTime stat, fileSize stat)

nopatch = return ()

own_name :: PackedString -> PackedString
own_name f = case breakLastPS '/' f of Nothing -> f
                                       Just (_,f') -> f'
super_name :: PackedString -> PackedString
super_name f = case breakLastPS '/' f of Nothing -> packString "."
                                         Just (d,_) -> d
break_on_dir :: PackedString -> Maybe (PackedString,PackedString)
break_on_dir p = case breakFirstPS '/' p of
                 Nothing -> Nothing
                 Just (d,f) | d == packString "." -> break_on_dir f
                            | otherwise -> Just (d,f)
norm_path :: PackedString -> PackedString -- remove "./"
norm_path p = case breakFirstPS '/' p of
              Nothing -> p
              Just (d,f) | d == packString "." -> norm_path f
                         | otherwise -> p

undefined_time :: EpochTime
undefined_time = -1

wait_a_moment :: IO ()
wait_a_moment = do { sleep 1; return () }
    -- HACKERY: In ghc 6.1, sleep has the type signature IO Int; it
    -- returns an integer just like sleep(3) does. To stay compatible
    -- with older versions, though, we just ignore sleep's return
    -- value. Hackery, like I said.
\end{code}

slurp is how we get a slurpy in the first place...

\begin{code}
slurp dirname = do
    isdir <- doesDirectoryExist dirname
    if isdir
       then do
            former_dir <- getCurrentDirectory
            setCurrentDirectory dirname
            slurpy <- slurp_helper "."
            setCurrentDirectory former_dir
            return slurpy
       else do
            slurp_helper dirname

slurp_helper dirname = do
    isdir <- doesDirectoryExist dirname
    if isdir
       then do
            former_dir <- getCurrentDirectory
            fnames <- getDirectoryContents dirname
            setCurrentDirectory dirname
            sl <- sequence $ map slurp_helper $ filter not_hidden fnames
            setCurrentDirectory former_dir
            return $ SlurpDir (packString dirname) nopatch sl
       else do
            isfile <- doesFileExist dirname
            current_dir <- getCurrentDirectory
            if isfile
              then do
                   ls <- unsafeInterleaveIO $
                         readFileLinesPSetc $ current_dir ++ "/" ++ dirname
                   mtime <- getModTime $ current_dir++"/"++dirname
                   return $ SlurpFile (packString dirname) False mtime ls
              else error "bug in slurp_helper!"
not_hidden :: FilePath -> Bool
not_hidden ('.':_) = False
not_hidden ('_':_) = False
not_hidden _ = True

co_slurp :: Slurpy -> FilePath -> IO Slurpy
co_slurp guide dirname = do
    isdir <- doesDirectoryExist dirname
    if isdir
       then do
            former_dir <- getCurrentDirectory
            setCurrentDirectory dirname
            Just slurpy <- co_slurp_helper guide
            setCurrentDirectory former_dir
            return slurpy
       else error "Error coslurping!!! Please report this."

co_slurp_helper (SlurpDir d _ c) = do
    isdir <- doesDirectoryExist $ unpackPS d
    if not isdir
       then return Nothing
       else do former_dir <- getCurrentDirectory
               setCurrentDirectory $ unpackPS d
               sl <- sequence $ map co_slurp_helper c
               setCurrentDirectory former_dir
               return $ Just $ SlurpDir d nopatch $ catMaybes sl
co_slurp_helper (SlurpFile f d m c) = do
   isfile <- doesFileExist $ unpackPS f
   if isfile
     then do
          current_dir <- getCurrentDirectory
          ls <- unsafeInterleaveIO $
                readFileLinesPSetc $ current_dir ++ "/" ++ (unpackPS f)
          mtime <- getModTime $ current_dir++"/"++ (unpackPS f)
          return $ Just $ SlurpFile f False mtime ls
     else return Nothing
readFileLinesPSetc :: String -> IO FileContents
readFileLinesPSetc f = do ps <- readFilePS f
                          return (linesPS ps, Just ps)
readFileLinesPS :: String -> IO [PackedString]
readFileLinesPS f = linesPS `liftM` readFilePS f
\end{code}

It is important to be able to readily modify a slurpy.

\begin{code}
slurp_remove :: FilePath -> Slurpy -> Maybe Slurpy
slurp_remove f s@(SlurpDir d p c) =
    Just $ SlurpDir d p $ catMaybes $ map (sr (norm_path $ packString f)) c
    where sr f s@(SlurpDir d p c) =
              if f == d then Nothing
              else case break_on_dir f of
                   Just (dn,fn) -> if dn /= d then Just s
                                   else Just $ SlurpDir d p $ catMaybes $
                                        map (sr fn) c
                   Nothing -> Just s
          sr f s@(SlurpFile f' _ _ _) | f == f' = Nothing
                                      | otherwise =Just s

slurp_removefile :: FilePath -> Slurpy -> Maybe Slurpy
slurp_removefile f s =
  if slurp_hasfile f s
  then slurp_removefile_private (norm_path $ packString f) s
  else Nothing

slurp_removefile_private :: PackedString -> Slurpy -> Maybe Slurpy
slurp_removefile_private f (SlurpDir d p c) =
    case break_on_dir f of
    Just (dn,fn) -> remove_file_from_dir fn dn d p [] c
    Nothing -> Just $ SlurpDir d (p>>rm_file f) $
                               filter (\s-> unpackPS f/=slurp_name s) c
rm_file f = removeFile (unpackPS f) `catch` (\isDoesNotExistError-> return ())

remove_file_from_dir _ _ _ _ _ [] = Nothing
remove_file_from_dir f d superd superp oldss (SlurpFile f' dirty m c':ss) =
    remove_file_from_dir f d superd superp (SlurpFile f' dirty m c':oldss) ss
remove_file_from_dir f d superd superp oldss (SlurpDir d' p' c':ss)
    | d == d' =
       case slurp_removefile_private f (SlurpDir d' p' c') of
       Just goods -> Just $ SlurpDir superd superp (oldss++[goods]++ss)
       Nothing -> Nothing
    | otherwise = remove_file_from_dir f d superd superp (SlurpDir d' p' c':oldss) ss
\end{code}

\begin{code}
slurp_move :: FilePath -> FilePath -> Slurpy -> Maybe Slurpy
slurp_move f f' s =
    if slurp_hasfile f s
    then slurp_movefile f f' s
    else if slurp_hasdir f s
         then slurp_movedir f f' s
         else Just s

slurp_movefile :: FilePath -> FilePath -> Slurpy -> Maybe Slurpy
slurp_movefile f f' s@(SlurpDir _ _ _) =
    if not (slurp_hasfile f' s) &&
       not (slurp_hasdir f' s) &&
       slurp_hasdir (unpackPS $ super_name $ packString f') s
    then case get_slurp f s of
         Nothing -> Nothing
         Just sf ->
             case slurp_remove f s of
             Nothing -> Nothing
             Just (SlurpDir d p c) ->
                 Just $ addslurp f' (slurp_setname (own_name $ packString f') sf)
                           $ SlurpDir d (p >> mv_file f f') c
    else Nothing
mv_file f f' = renameFile f f' `catch` (\isDoesNotExistError-> return ())
mv_dir f f' = renameDirectory f f' `catch` (\isDoesNotExistError-> return ())

slurp_movedir :: FilePath -> FilePath -> Slurpy -> Maybe Slurpy
slurp_movedir f f' s@(SlurpDir _ _ _) =
    if not (slurp_hasfile f' s) &&
       not (slurp_hasdir f' s) &&
       slurp_hasdir (unpackPS $ super_name $ packString f') s
    then case get_slurp f s of
         Nothing -> Nothing
         Just sf ->
             case slurp_remove f s of
             Nothing -> Nothing
             Just (SlurpDir d p c) ->
                 Just $ addslurp f'
                          (slurp_setname (own_name $ packString f') sf)
                          $ SlurpDir d (p >> mv_dir f f') c
    else Nothing

addslurp :: FilePath -> Slurpy -> Slurpy -> Slurpy
addslurp f s s' =
    addslurp_private (packString ".") (norm_path $ packString f) s s'

addslurp_private d f s (SlurpFile a b m c) = SlurpFile a b m c
addslurp_private d f s (SlurpDir d' p c)
    | d /= d' = SlurpDir d' p c
    | otherwise =
        case break_on_dir f of
        Just (dn,fn) -> SlurpDir d p $ map (addslurp_private dn fn s) c
        Nothing -> SlurpDir d p (s:c)

get_slurp :: FilePath -> Slurpy -> Maybe Slurpy
get_slurp f (SlurpFile f' b m c) =
    if packString f == f' then Just $ SlurpFile (packString f) b m c
    else Nothing
get_slurp f (SlurpDir d b c)
  | packString f == d = Just $ SlurpDir d b c
  | otherwise =
       case breakFirstPS '/' $ packString f of
       Just (dn,fn) ->
           if dn == d
               then case filter (/=Nothing) $ map (get_slurp $ unpackPS fn) c of
                [] -> Nothing
                [msf] -> msf
               else Nothing
       _ -> Nothing
\end{code}

\begin{code}
slurp_addfile :: FilePath -> Slurpy -> Maybe Slurpy
slurp_addfile f s =
  if slurp_hasfile f s
  then slurp_modfile f (\_ -> Just ([],Nothing)) s
  else if slurp_hasdir (unpackPS $ super_name $ packString f) s
       then Just $ addslurp f
                (SlurpFile (own_name $ packString f) True
                 (undefined_time,0) ([],Nothing)) s
       else Nothing
\end{code}

\begin{code}
slurp_removedir :: FilePath -> Slurpy -> Maybe Slurpy
slurp_removedir f s =
  if slurp_hasdir f s
  then slurp_removedir_private (norm_path $ packString f) s
  else Nothing

slurp_removedir_private :: PackedString -> Slurpy -> Maybe Slurpy
slurp_removedir_private f (SlurpDir d p c) =
    case break_on_dir f of
    Just (dn,fn) -> remove_dir_from_dir fn dn d (p >> rm_dir f) [] c
    -- FIXME--really should also check first if dir is empty...
    Nothing -> Just $ SlurpDir d (p>>rm_dir f) $
                               filter (\s-> unpackPS f/=slurp_name s) c
rm_dir f = removeDirectory (unpackPS f) `catch` (\isDoesNotExistError-> return ())

remove_dir_from_dir _ _ _ _ _ [] = Nothing
remove_dir_from_dir f d superd superp oldss (SlurpFile f' dirty m c':ss) =
    remove_dir_from_dir f d superd superp (SlurpFile f' dirty m c':oldss) ss
remove_dir_from_dir f d superd superp oldss (SlurpDir d' p' c':ss)
    | d == d' =
       case slurp_removedir_private f (SlurpDir d' p' c') of
       Just goods -> Just $ SlurpDir superd superp (oldss++[goods]++ss)
       Nothing -> Nothing
    | otherwise = remove_dir_from_dir f d superd superp (SlurpDir d' p' c':oldss) ss
\end{code}

\begin{code}
slurp_adddir :: FilePath -> Slurpy -> Maybe Slurpy
slurp_adddir f s =
  if slurp_hasfile f s || slurp_hasdir f s
  then Nothing
  else slurp_adddir_private (norm_path $ packString f) s

slurp_adddir_private :: PackedString -> Slurpy -> Maybe Slurpy
slurp_adddir_private f (SlurpDir d p c) =
    case break_on_dir f of
    Just (dn,fn) -> add_dir_to_dir fn dn d p [] c
    Nothing -> Just $ SlurpDir d (p >> createDirectory (unpackPS f))
               (SlurpDir f nopatch []:c)

add_dir_to_dir _ _ _ _ _ [] = Nothing
add_dir_to_dir f d superd superp oldss (SlurpFile f' dirty m c':ss) =
    add_dir_to_dir f d superd superp (SlurpFile f' dirty m c':oldss) ss
add_dir_to_dir f d superd superp oldss (SlurpDir d' p' c':ss)
    | d == d' =
       case slurp_adddir_private f (SlurpDir d' p' c') of
       Just goods -> Just $ SlurpDir superd superp (oldss++[goods]++ss)
       Nothing -> Nothing
    | otherwise = add_dir_to_dir f d superd superp (SlurpDir d' p' c':oldss) ss
\end{code}

Code to modify a given file in a slurpy.

\begin{code}
slurp_modfile :: FilePath -> (FileContents -> Maybe FileContents)
              -> Slurpy -> Maybe Slurpy
slurp_modfile f mod s@(SlurpDir d p c) =
  if not $ slurp_hasfile f s
  then Nothing
  else case sequence $ map (sm $norm_path$packString f) c of
       Nothing -> Nothing
       Just c' -> Just $ SlurpDir d p c'
    where sm :: PackedString -> Slurpy -> Maybe Slurpy
          sm f s@(SlurpDir d p c) =
              case break_on_dir f of
              Nothing -> Just s
              Just (dn,fn) ->
                  if dn == d
                  then case sequence $ map (sm fn) c of
                       Nothing -> Nothing
                       Just c' -> Just $ SlurpDir d p c'
                  else Just s
          sm f s@(SlurpFile ff dirt m c)
              | f == ff = case mod c of
                          Nothing -> Nothing
                          Just c' -> Just $ SlurpFile ff True m c'
              | otherwise = Just s
\end{code}

\begin{code}
slurp_hasfile :: FilePath -> Slurpy -> Bool
slurp_hasfile f (SlurpDir d _ c) =
    or $ map (slurp_hasfile_private $ norm_path $ packString f) c

slurp_hasfile_private f (SlurpFile f' _ _ _) = f == f'
slurp_hasfile_private f (SlurpDir d _ c)
  | f == d = False
  | otherwise =
       case break_on_dir f of
       Just (dn,fn) ->
           if dn == d
           then or $ map (slurp_hasfile_private fn) c
           else False
       _ -> False

slurp_has :: FilePath -> Slurpy -> Bool
slurp_has f (SlurpDir d _ c) =
    or $ map (has_private $ norm_path $ packString f) c
    where has_private f (SlurpFile f' _ _ _) = f == f'
          has_private f (SlurpDir d _ c)
            | f == d = True
            | otherwise =
                case break_on_dir f of
                Just (dn,fn)
                    | dn == d -> or $ map (has_private fn) c
                    | otherwise -> False
                _ -> False

slurp_has_anycase :: FilePath -> Slurpy -> Bool
slurp_has_anycase f (SlurpDir d _ c) =
  or $ map (hasany_private $ norm_path $ packString f) c
  where hasany_private f (SlurpFile f' _ _ _) = tolower f == tolower f'
        hasany_private f (SlurpDir d _ c)
            | tolower f == tolower d = True
            | otherwise =
                case break_on_dir f of
                Just (dn,fn) -> if tolower dn == tolower d
                                then or $ map (hasany_private fn) c
                                else False
                _ -> False
tolower = mapPS toLower

slurp_hasdir :: FilePath -> Slurpy -> Bool
slurp_hasdir "." _ = True
slurp_hasdir f (SlurpDir d _ c) =
    or $ map (slurp_hasdir_private $ norm_path $ packString f) c

slurp_hasdir_private f (SlurpFile f' _ _ _) = False
slurp_hasdir_private f (SlurpDir d _ c)
  | f == d = True
  | otherwise =
       case break_on_dir f of
       Just (dn,fn) ->
           if dn == d
           then or $ map (slurp_hasdir_private fn) c
           else False
       _ -> False
\end{code}

Code to output the dirty files from a slurpy.

\begin{code}
slurp_write :: Slurpy -> IO ()
slurp_write (SlurpDir d p ss) = do
  former_dir <- getCurrentDirectory
  isdir <- doesDirectoryExist $ unpackPS d
  unless isdir $ createDirectory $ unpackPS d
  setCurrentDirectory $ unpackPS d
  p
  sequence_ $ map slurp_write ss
  setCurrentDirectory former_dir
slurp_write (SlurpFile f _ _ ls) = writeLinesPSetc f ls
\end{code}

\begin{code}
slurp_runfunc :: IO () -> Slurpy -> Maybe Slurpy
slurp_runfunc f (SlurpDir d p ss) =
    Just $ SlurpDir d (p >> f) ss
\end{code}

\begin{code}
slurp_write_dirty :: Slurpy -> IO ()
slurp_write_dirty (SlurpDir d p ss) = do
  former_dir <- getCurrentDirectory
  setCurrentDirectory $ unpackPS d
  p
  sequence_ $ map slurp_write_dirty ss
  setCurrentDirectory former_dir

slurp_write_dirty (SlurpFile f dirt _ ls)
    | dirt == False = return ()
    | otherwise = writeLinesPSetc f ls
writeLinesPSetc :: PackedString -> FileContents -> IO ()
writeLinesPSetc f (pss, Nothing) = do
  h <- openFile (unpackPS f) WriteMode
  sequence_ $ map (\ps -> fixedhPutPS h ps >> hPutChar h '\n') pss
  hClose h
writeLinesPSetc f (_, Just ps) = do
  h <- openFile (unpackPS f) WriteMode
  fixedhPutPS h ps
  hClose h
fixedhPutPS h ps = if lengthPS ps == 0 then return () else hPutPS h ps
\end{code}

Code to flag all files as clean

\begin{code}
launder_slurpy :: Slurpy -> Slurpy
launder_slurpy (SlurpDir d _ ss) = SlurpDir d nopatch (map launder_slurpy ss)
launder_slurpy (SlurpFile f _ m ls) = SlurpFile f False m ls

make_dirty :: Slurpy -> Slurpy
make_dirty (SlurpFile f _ m ls) = SlurpFile f True m ls
make_dirty (SlurpDir f p c) =
    SlurpDir f (p >> createDirectory (unpackPS f) `catch` (\_-> return ()))
                 (map make_dirty c)
\end{code}
