%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs replace}
\begin{code}
module Replace ( replace ) where

import Maybe ( catMaybes )
import Monad (unless, liftM)
import RegexString (matchRegex, mkRegex)

import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import SlurpDirectory
\end{code}

\begin{code}
replace_description =
 "Replace a token with a new value for that token."
\end{code}

\options{replace}

\haskell{replace_help} FIXME: Currently there is no way to specify a
different regexp (other than \verb![A-Za-z_0-9]!) to be used for a token
replace.  Actually, if one of your tokens contains a `-' or `.', you will
then get the ``filename'' regexp, which is \verb![A-Za-z_0-9\-\.]!.

\begin{code}
replace_help =
 "Replace allows you to change a specified token whereever it
occurs in the specified files.  Tokens here are defined by a
regexp specifying the characters which are allowed.  By default
a token corresponds to a C identifier.
"
\end{code}

\begin{code}
replace = DarcsCommand {command_name = "replace",
                   command_help = replace_help,
                   command_description = replace_description,
                   command_extra_args = -1,
                   command_command = replace_cmd,
                   command_prereq = am_in_repo,
                   command_get_arg_possibilities = list_registered_files,
                   command_argdefaults = nodefaults,
                   command_darcsoptions = [verbose]}
\end{code}

\begin{code}
default_toks = "A-Za-z_0-9"
filename_toks = "A-Za-z_0-9\\-\\."
is_tok toks s = matchRegex (mkRegex $ "^["++toks++"]+$") s /= Nothing
choose_toks a b = if is_tok default_toks a && is_tok default_toks b
                  then default_toks
                  else filename_toks
replace_cmd _ (old:new:fs) = do
  toks <- return $ choose_toks old new
  unless (is_tok toks old) $ do
    putStr $ "'"++old++"' is not a valid token!\n"
    exitWith $ ExitFailure 1
  unless (is_tok toks new) $ do
    putStr $ "'"++new++"' is not a valid token!\n"
    exitWith $ ExitFailure 1
  cur <- slurp_current "."
  ps <- catMaybes `liftM` sequence (map (repl toks cur) fs)
  work <- slurp "."
  case apply_to_slurpy (join_patches ps) work of
    Nothing -> do putStr $ "Can't do replace on working!\n"
                  putStr $ "Perhaps one of the files already contains '"
                                 ++new++"'?\n"
                  exitWith $ ExitFailure 1
    Just w' -> slurp_write_dirty w'
  pend <- read_pending
  case pend of
      Nothing -> write_pending $ join_patches ps
      Just op -> write_pending $ join_patches $ flatten op ++ ps
  where repl toks s f =
          if not $ slurp_hasfile f s
          then do putStr $ "Skipping file '"++f++"' which isn't in the repo.\n"
                  return Nothing
          else case apply_to_slurpy (tokreplace f toks old new) s of
            Nothing -> do putStr $ "Skipping file '"++f++"'\n"
                          putStr $ "Perhaps this file already contains '"
                                     ++new++"'?\n"
                          return Nothing
            _ -> return $ Just $ tokreplace f toks old new
        
replace_cmd _ _ = do
    putStr "Usage: darcs replace OLD NEW [FILES]\n"
    exitWith $ ExitFailure 1
\end{code}
